/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.io.rest.sitemap;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.eclipse.emf.common.util.BasicEList;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.events.Event;
import org.openhab.core.events.EventSubscriber;
import org.openhab.core.i18n.TimeZoneProvider;
import org.openhab.core.io.rest.sitemap.internal.SitemapEvent;
import org.openhab.core.io.rest.sitemap.internal.WidgetsChangeListener;
import org.openhab.core.items.GroupItem;
import org.openhab.core.items.Item;
import org.openhab.core.items.events.ItemStatePredictedEvent;
import org.openhab.core.model.core.EventType;
import org.openhab.core.model.core.ModelRepositoryChangeListener;
import org.openhab.core.model.sitemap.SitemapProvider;
import org.openhab.core.model.sitemap.sitemap.LinkableWidget;
import org.openhab.core.model.sitemap.sitemap.Sitemap;
import org.openhab.core.model.sitemap.sitemap.Widget;
import org.openhab.core.thing.events.ChannelDescriptionChangedEvent;
import org.openhab.core.ui.items.ItemUIRegistry;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.annotations.Activate;
import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Deactivate;
import org.osgi.service.component.annotations.Modified;
import org.osgi.service.component.annotations.Reference;
import org.osgi.service.component.annotations.ReferenceCardinality;
import org.osgi.service.component.annotations.ReferencePolicy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Component(service={SitemapSubscriptionService.class, EventSubscriber.class}, configurationPid={"org.openhab.sitemapsubscription"})
@NonNullByDefault
public class SitemapSubscriptionService
implements ModelRepositoryChangeListener,
EventSubscriber {
    private static final String SITEMAP_PAGE_SEPARATOR = "#";
    private static final String SITEMAP_SUFFIX = ".sitemap";
    private static final int DEFAULT_MAX_SUBSCRIPTIONS = 50;
    private static final Duration WAIT_AFTER_CREATE_SECONDS = Duration.ofSeconds(30L);
    private final Logger logger = LoggerFactory.getLogger(SitemapSubscriptionService.class);
    private final BundleContext bundleContext;
    private final ItemUIRegistry itemUIRegistry;
    private final TimeZoneProvider timeZoneProvider;
    private final List<SitemapProvider> sitemapProviders = new ArrayList<SitemapProvider>();
    private final Map<String, String> scopeOfSubscription = new ConcurrentHashMap<String, String>();
    private final Map<String, SitemapSubscriptionCallback> callbacks = new ConcurrentHashMap<String, SitemapSubscriptionCallback>();
    private final Map<String, Instant> creationInstants = new ConcurrentHashMap<String, Instant>();
    private final Map<String, ListenerRecord> pageChangeListeners = new ConcurrentHashMap<String, ListenerRecord>();
    private int maxSubscriptions = 50;

    @Activate
    public SitemapSubscriptionService(Map<String, Object> config, @Reference ItemUIRegistry itemUIRegistry, @Reference TimeZoneProvider timeZoneProvider, BundleContext bundleContext) {
        this.itemUIRegistry = itemUIRegistry;
        this.timeZoneProvider = timeZoneProvider;
        this.bundleContext = bundleContext;
        this.applyConfig(config);
    }

    @Deactivate
    protected void deactivate() {
        this.scopeOfSubscription.clear();
        this.callbacks.clear();
        this.creationInstants.clear();
        this.pageChangeListeners.values().forEach(l -> l.serviceRegistration.unregister());
        this.pageChangeListeners.clear();
    }

    @Modified
    protected void modified(Map<String, Object> config) {
        this.applyConfig(config);
    }

    private void applyConfig(Map<String, Object> config) {
        if (config == null) {
            return;
        }
        String max = Objects.toString(config.get("maxSubscriptions"), null);
        if (max != null) {
            try {
                this.maxSubscriptions = Integer.parseInt(max);
            }
            catch (NumberFormatException e) {
                this.logger.debug("Setting 'maxSubscriptions' must be a number; value '{}' ignored.", (Object)max);
            }
        }
    }

    @Reference(cardinality=ReferenceCardinality.MULTIPLE, policy=ReferencePolicy.DYNAMIC)
    public void addSitemapProvider(SitemapProvider provider) {
        this.sitemapProviders.add(provider);
        provider.addModelChangeListener((ModelRepositoryChangeListener)this);
    }

    public void removeSitemapProvider(SitemapProvider provider) {
        this.sitemapProviders.remove(provider);
        provider.removeModelChangeListener((ModelRepositoryChangeListener)this);
    }

    public @Nullable String createSubscription(SitemapSubscriptionCallback callback) {
        if (this.maxSubscriptions >= 0 && this.callbacks.size() >= this.maxSubscriptions) {
            this.logger.debug("No new subscription delivered as limit ({}) is already reached", (Object)this.maxSubscriptions);
            return null;
        }
        String subscriptionId = UUID.randomUUID().toString();
        this.callbacks.put(subscriptionId, callback);
        this.creationInstants.put(subscriptionId, Instant.now());
        this.logger.debug("Created new subscription with id {} ({} active subscriptions for a max of {})", new Object[]{subscriptionId, this.callbacks.size(), this.maxSubscriptions});
        return subscriptionId;
    }

    public void removeSubscription(String subscriptionId) {
        ListenerRecord listener;
        this.creationInstants.remove(subscriptionId);
        this.callbacks.remove(subscriptionId);
        String sitemapWithPageId = this.scopeOfSubscription.remove(subscriptionId);
        if (sitemapWithPageId != null && !this.scopeOfSubscription.containsValue(sitemapWithPageId) && (listener = this.pageChangeListeners.remove(sitemapWithPageId)) != null) {
            listener.serviceRegistration().unregister();
        }
        this.logger.debug("Removed subscription with id {} ({} active subscriptions)", (Object)subscriptionId, (Object)this.callbacks.size());
    }

    public boolean exists(String subscriptionId) {
        return this.callbacks.containsKey(subscriptionId);
    }

    public @Nullable String getPageId(String subscriptionId) {
        String sitemapWithPageId = this.scopeOfSubscription.get(subscriptionId);
        return sitemapWithPageId == null ? null : this.extractPageId(sitemapWithPageId);
    }

    public @Nullable String getSitemapName(String subscriptionId) {
        String sitemapWithPageId = this.scopeOfSubscription.get(subscriptionId);
        return sitemapWithPageId == null ? null : this.extractSitemapName(sitemapWithPageId);
    }

    private String extractSitemapName(String sitemapWithPageId) {
        return sitemapWithPageId.split(SITEMAP_PAGE_SEPARATOR)[0];
    }

    private boolean isPageListener(String sitemapWithPageId) {
        return sitemapWithPageId.contains(SITEMAP_PAGE_SEPARATOR);
    }

    private String extractPageId(String sitemapWithPageId) {
        return sitemapWithPageId.split(SITEMAP_PAGE_SEPARATOR)[1];
    }

    public void updateSubscriptionLocation(String subscriptionId, String sitemapName, @Nullable String pageId) {
        SitemapSubscriptionCallback callback = this.callbacks.get(subscriptionId);
        if (callback != null) {
            String oldSitemapWithPage = this.scopeOfSubscription.remove(subscriptionId);
            if (oldSitemapWithPage != null) {
                this.removeCallbackFromListener(oldSitemapWithPage, callback);
            }
        } else {
            throw new IllegalArgumentException("Subscription " + subscriptionId + " does not exist!");
        }
        this.addCallbackToListener(sitemapName, pageId, callback);
        String scopeIdentifier = this.getScopeIdentifier(sitemapName, pageId);
        this.scopeOfSubscription.put(subscriptionId, scopeIdentifier);
        this.logger.debug("Subscription {} changed to {} ({} active subscriptions}", new Object[]{subscriptionId, scopeIdentifier, this.callbacks.size()});
    }

    private void addCallbackToListener(String sitemapName, @Nullable String pageId, SitemapSubscriptionCallback callback) {
        String sitemapWithPageId = this.getScopeIdentifier(sitemapName, pageId);
        ListenerRecord listener = this.pageChangeListeners.computeIfAbsent(sitemapWithPageId, v -> {
            WidgetsChangeListener newListener = new WidgetsChangeListener(sitemapName, pageId, this.itemUIRegistry, this.timeZoneProvider, this.collectWidgets(sitemapName, pageId));
            ServiceRegistration registration = this.bundleContext.registerService(EventSubscriber.class.getName(), (Object)newListener, null);
            return new ListenerRecord(newListener, registration);
        });
        listener.widgetsChangeListener().addCallback(callback);
    }

    public EList<Widget> collectWidgets(String sitemapName, @Nullable String pageId) {
        BasicEList widgets = new BasicEList();
        Sitemap sitemap = this.getSitemap(sitemapName);
        if (sitemap == null) {
            return widgets;
        }
        if (pageId != null && !pageId.equals(sitemap.getName())) {
            Widget pageWidget = this.itemUIRegistry.getWidget(sitemap, pageId);
            if (pageWidget instanceof LinkableWidget) {
                LinkableWidget widget = (LinkableWidget)pageWidget;
                widgets.addAll((Collection)this.itemUIRegistry.getChildren(widget));
                widgets.add((Object)pageWidget);
            }
        } else {
            widgets.addAll((Collection)this.itemUIRegistry.getChildren(sitemap));
            if (pageId == null) {
                LinkedList childrenQueue = new LinkedList(widgets);
                while (!childrenQueue.isEmpty()) {
                    Widget child = (Widget)childrenQueue.removeFirst();
                    if (!(child instanceof LinkableWidget)) continue;
                    LinkableWidget widget = (LinkableWidget)child;
                    EList subWidgets = this.itemUIRegistry.getChildren(widget);
                    widgets.addAll((Collection)subWidgets);
                    childrenQueue.addAll(subWidgets);
                }
            }
        }
        this.logger.debug("Collected {} widgets for sitemap: {}, page id {}", new Object[]{widgets.size(), sitemapName, pageId});
        return widgets;
    }

    private void removeCallbackFromListener(String sitemapPage, SitemapSubscriptionCallback callback) {
        ListenerRecord oldListener = this.pageChangeListeners.get(sitemapPage);
        if (oldListener != null) {
            oldListener.widgetsChangeListener().removeCallback(callback);
            if (!this.scopeOfSubscription.containsValue(sitemapPage)) {
                oldListener.serviceRegistration().unregister();
                this.pageChangeListeners.remove(sitemapPage);
            }
        }
    }

    private String getScopeIdentifier(String sitemapName, @Nullable String pageId) {
        return pageId == null ? sitemapName : sitemapName + SITEMAP_PAGE_SEPARATOR + pageId;
    }

    private @Nullable Sitemap getSitemap(String sitemapName) {
        for (SitemapProvider provider : this.sitemapProviders) {
            Sitemap sitemap = provider.getSitemap(sitemapName);
            if (sitemap == null) continue;
            return sitemap;
        }
        return null;
    }

    public void modelChanged(String modelName, EventType type) {
        if (type != EventType.MODIFIED || !modelName.endsWith(SITEMAP_SUFFIX)) {
            return;
        }
        String changedSitemapName = modelName.substring(0, modelName.length() - SITEMAP_SUFFIX.length());
        for (Map.Entry<String, ListenerRecord> listenerEntry : this.pageChangeListeners.entrySet()) {
            EList<Widget> widgets;
            String sitemapWithPage = listenerEntry.getKey();
            String sitemapName = this.extractSitemapName(sitemapWithPage);
            if (!sitemapName.equals(changedSitemapName)) continue;
            if (this.isPageListener(sitemapWithPage)) {
                String pageId = this.extractPageId(sitemapWithPage);
                widgets = this.collectWidgets(sitemapName, pageId);
            } else {
                widgets = this.collectWidgets(sitemapName, null);
            }
            listenerEntry.getValue().widgetsChangeListener().sitemapContentChanged(widgets);
        }
    }

    public void checkAliveClients() {
        for (Map.Entry<String, Instant> creationEntry : this.creationInstants.entrySet()) {
            String subscriptionId = creationEntry.getKey();
            SitemapSubscriptionCallback callback = this.callbacks.get(subscriptionId);
            if (this.scopeOfSubscription.containsKey(subscriptionId) || callback == null || !creationEntry.getValue().plus(WAIT_AFTER_CREATE_SECONDS).isBefore(Instant.now())) continue;
            this.logger.debug("Release subscription {} as it was not queried within {} seconds", (Object)subscriptionId, (Object)WAIT_AFTER_CREATE_SECONDS);
            this.removeSubscription(subscriptionId);
            callback.onRelease(subscriptionId);
        }
        this.pageChangeListeners.values().forEach(l -> l.widgetsChangeListener().sendAliveEvent());
    }

    public Set<String> getSubscribedEventTypes() {
        return Set.of(ItemStatePredictedEvent.TYPE, ChannelDescriptionChangedEvent.TYPE);
    }

    public void receive(Event event) {
        if (event instanceof ItemStatePredictedEvent) {
            ItemStatePredictedEvent prediction = (ItemStatePredictedEvent)event;
            Item item = (Item)this.itemUIRegistry.get((Object)prediction.getItemName());
            if (item instanceof GroupItem) {
                return;
            }
            for (ListenerRecord listener : this.pageChangeListeners.values()) {
                if (prediction.isConfirmation()) {
                    listener.widgetsChangeListener().keepCurrentState(item);
                    continue;
                }
                listener.widgetsChangeListener().changeStateTo(item, prediction.getPredictedState());
            }
        } else if (event instanceof ChannelDescriptionChangedEvent) {
            ChannelDescriptionChangedEvent channelDescriptionChangedEvent = (ChannelDescriptionChangedEvent)event;
            channelDescriptionChangedEvent.getLinkedItemNames().forEach(itemName -> {
                for (ListenerRecord listener : this.pageChangeListeners.values()) {
                    listener.widgetsChangeListener().descriptionChanged((String)itemName);
                }
            });
        }
    }

    private record ListenerRecord(WidgetsChangeListener widgetsChangeListener, ServiceRegistration<?> serviceRegistration) {
    }

    public static interface SitemapSubscriptionCallback {
        public void onEvent(SitemapEvent var1);

        public void onRelease(String var1);
    }
}

