# unicode_data.tcl --
#
#	Declarations of Unicode character information tables.  This file is
#	automatically generated by the gen_unicode_data.tcl script.  Do not
#	modify this file by hand.
#
# Copyright (c) 1998 Scriptics Corporation.
# Copyright (c) 2007 Alexey Shchepin
# Copyright (c) 2007 Sergei Golovan
#
# See the file "license.terms" for information on usage and redistribution
# of this file, and for a DISCLAIMER OF ALL WARRANTIES.

#
# A 16-bit Unicode character is split into two parts in order to index
# into the following tables.  The lower CCLASS_OFFSET_BITS comprise an offset
# into a page of characters.  The upper bits comprise the page number.
#

package provide unicode::data 1.1.1

namespace eval ::unicode::data {

variable CCLASS_OFFSET_BITS 2

#
# The cclassPageMap is indexed by page number and returns an alternate page number
# that identifies a unique page of characters.  Many Unicode characters map
# to the same alternate page number.
#

variable cclassPageMap
array unset cclassPageMap
array set cclassPageMap [list \
    192 1 193 1 194 1 195 1 196 1 197 2 198 3 199 4 200 5 201 6 202 7 203 4 \
    204 4 205 8 206 9 207 10 208 1 209 11 210 12 211 13 212 14 213 15 214 16 \
    215 17 216 18 217 1 218 1 219 1 288 19 289 20 356 21 357 22 358 23 359 1 \
    360 24 361 4 362 22 363 25 364 26 365 27 366 28 367 29 368 30 369 31 \
    388 1 389 32 402 33 403 34 404 35 405 36 406 1 407 37 412 38 437 39 \
    438 1 439 40 440 14 441 40 442 41 443 42 452 43 460 44 461 45 462 46 \
    463 47 464 22 465 47 466 37 506 19 507 1 508 22 591 48 595 49 596 50 \
    597 51 623 48 627 49 655 48 659 49 687 48 691 49 719 48 723 49 755 49 \
    787 49 789 52 815 48 819 49 851 49 882 53 910 54 914 55 942 56 946 57 \
    966 58 973 59 974 60 988 61 989 62 990 63 991 64 992 65 993 66 1009 67 \
    1037 68 1038 49 1239 19 1477 69 1485 69 1524 53 1527 70 1578 71 1614 72 \
    1669 19 1670 73 1741 48 1745 69 1754 19 1755 10 1756 1 1904 22 1905 1 \
    1906 74 1919 75 2100 76 2101 1 2102 77 2103 51 2104 70 2105 78 2106 79 \
    2107 4 3082 80 3083 81 3110 82 10753 53 16071 83 16264 1 17027 84 17038 85 \
    17039 86 29785 87 29786 88 29787 89 29788 90 29790 91 29791 4 29792 92 \
    29793 93 29794 24 29802 39 29803 32 29840 39 29841 51]

variable CCLASS_COMMON_PAGE_MAP 0

#
# The cclassGroupMap is indexed by combining the alternate page number with
# the page offset and returns a combining class number.
#

variable cclassGroupMap [list \
    0 0 0 0 230 230 230 230 230 232 220 220 220 220 232 216 220 220 220 \
    220 220 202 202 220 220 220 220 202 202 220 220 220 1 1 1 1 1 220 220 \
    220 220 230 230 230 230 240 230 220 220 220 230 230 230 220 220 0 230 \
    230 230 220 220 220 220 230 232 220 220 230 233 234 234 233 234 234 \
    233 230 0 0 0 230 230 230 230 0 0 220 230 230 230 230 220 230 230 230 \
    222 220 230 230 220 220 230 222 228 230 10 11 12 13 14 15 16 17 18 \
    19 19 20 21 22 0 23 0 24 25 0 230 220 0 18 230 230 0 0 0 0 0 27 28 \
    29 30 31 32 33 34 230 230 220 220 230 220 230 230 0 35 0 0 0 0 0 230 \
    230 230 0 0 230 230 0 220 230 230 220 0 0 0 36 0 0 230 220 230 230 \
    220 230 230 220 220 220 230 220 220 230 220 230 7 0 0 0 0 9 0 0 0 230 \
    220 230 230 0 0 0 0 84 91 0 0 0 9 0 103 103 9 0 107 107 107 107 118 \
    118 0 0 122 122 122 122 220 220 0 0 0 220 0 220 0 216 0 0 0 129 130 \
    0 132 0 0 0 0 0 130 130 130 130 0 0 130 0 230 230 9 0 230 230 0 0 220 \
    0 0 0 0 7 9 0 0 0 0 230 0 0 0 228 0 0 0 222 230 220 220 0 0 0 230 230 \
    220 0 0 0 230 220 230 230 1 1 1 1 1 230 0 1 1 230 220 230 1 1 0 0 218 \
    228 232 222 224 224 0 8 8 0 0 0 26 0 0 220 0 230 230 1 220 0 0 0 0 \
    9 0 216 216 1 1 1 0 0 0 226 216 216 216 216 216 0 0 0 0 220 220 220 \
    220 0 0 230 230 230]

proc GetUniCharCClass {uc} {
    variable CCLASS_OFFSET_BITS
    variable CCLASS_COMMON_PAGE_MAP
    variable cclassPageMap
    variable cclassGroupMap

    set page [expr {($uc & 0x1fffff) >> $CCLASS_OFFSET_BITS}]
    if {[info exists cclassPageMap($page)]} {
	set apage $cclassPageMap($page)
    } else {
	set apage $CCLASS_COMMON_PAGE_MAP
    }

    lindex $cclassGroupMap \
	   [expr {($apage << $CCLASS_OFFSET_BITS) | \
		   ($uc & ((1 << $CCLASS_OFFSET_BITS) - 1))}]
}


variable DECOMP_OFFSET_BITS 3

#
# The pageMap is indexed by page number and returns an alternate page number
# that identifies a unique page of characters.  Many Unicode characters map
# to the same alternate page number.
#

variable decompPageMap
array unset decompPageMap
array set decompPageMap [list \
    20 1 21 2 22 3 23 4 24 5 25 6 26 7 27 8 28 9 29 10 30 11 31 12 32 13 \
    33 14 34 15 35 16 36 17 37 18 38 19 39 20 40 21 41 22 42 23 43 24 44 25 \
    45 26 46 27 47 28 52 29 53 30 54 31 56 32 57 33 58 34 59 35 60 36 61 37 \
    62 38 63 39 64 40 65 41 66 42 67 43 68 44 69 45 70 46 86 47 87 48 91 49 \
    92 50 104 51 110 52 111 53 112 54 113 55 114 56 117 57 118 58 121 59 \
    122 60 126 61 127 62 128 63 129 64 131 65 135 66 138 67 139 68 142 69 \
    152 70 154 71 155 72 156 73 157 74 158 75 159 76 176 77 196 78 206 79 \
    207 80 216 81 218 82 293 83 294 84 299 85 313 86 315 87 326 88 331 89 \
    361 90 363 91 370 92 377 93 393 94 408 95 409 96 425 97 443 98 454 99 \
    470 100 475 101 481 102 488 103 489 104 490 105 491 106 493 107 494 108 \
    495 109 496 110 498 111 499 112 500 113 501 114 503 115 516 116 543 117 \
    864 118 865 119 866 120 871 121 872 122 933 123 934 124 935 125 936 126 \
    937 127 938 128 939 129 940 130 941 131 943 132 947 133 948 134 949 135 \
    950 136 951 137 960 138 961 139 962 140 963 141 964 142 965 143 966 144 \
    967 145 968 146 969 147 970 148 971 149 972 150 973 151 974 152 975 153 \
    976 154 977 155 978 156 979 157 980 158 981 159 982 160 983 161 984 162 \
    985 163 986 164 987 165 988 166 989 167 990 168 991 169 992 170 993 171 \
    994 172 995 173 996 174 997 175 998 176 999 177 1000 178 1001 179 1002 180 \
    1003 181 1004 182 1005 183 1006 184 1007 185 1008 186 1009 187 1010 188 \
    1011 189 1012 190 1013 191 1014 192 1015 193 1016 194 1017 195 1018 196 \
    1019 197 1020 198 1021 199 1022 200 1023 201 1024 202 1025 203 1026 204 \
    1028 205 1029 206 1030 207 1031 208 1032 209 1033 210 1034 211 1035 206 \
    1038 212 1039 213 1040 214 1041 215 1042 216 1045 217 1056 218 1057 219 \
    1058 220 1059 221 1060 222 1061 223 1062 224 1063 225 1064 226 1065 227 \
    1066 228 1067 229 1068 230 1069 231 1070 232 1071 233 1075 234 1077 235 \
    1081 236 1088 237 1089 238 1092 239 1093 240 1094 241 1096 242 1097 243 \
    1100 244 1101 245 1102 246 1103 247 1104 248 1105 249 1109 250 1116 251 \
    1117 252 1125 253 1164 254 1165 255 1166 256 1167 257 1168 258 1169 259 \
    1170 260 1171 261 1172 262 1173 263 1174 264 1175 265 1176 266 1177 267 \
    1178 268 1179 269 1180 270 1181 271 1345 272 1358 273 1371 274 1453 275 \
    1491 276 1502 277 1504 278 1505 279 1506 280 1507 281 1508 282 1509 283 \
    1510 284 1511 285 1512 286 1513 287 1514 288 1515 289 1516 290 1517 291 \
    1518 292 1519 293 1520 294 1521 295 1522 296 1523 297 1524 298 1525 299 \
    1526 300 1527 301 1528 302 1529 303 1530 304 1536 1 1542 305 1543 306 \
    1545 307 1546 308 1547 309 1548 310 1549 311 1550 312 1551 313 1554 314 \
    1555 315 1557 316 1558 317 1559 318 1560 319 1561 320 1562 321 1563 322 \
    1566 323 1567 324 1574 325 1575 326 1576 327 1577 328 1578 329 1579 330 \
    1580 331 1581 332 1582 333 1583 334 1584 335 1585 336 1586 337 1587 338 \
    1600 339 1601 340 1602 341 1603 342 1604 343 1605 344 1606 345 1607 346 \
    1608 347 1610 348 1611 349 1612 350 1613 351 1614 352 1615 353 1616 354 \
    1617 355 1618 356 1619 357 1620 358 1621 359 1622 360 1623 361 1624 362 \
    1625 363 1626 364 1627 365 1628 366 1629 367 1630 368 1631 369 1632 370 \
    1633 371 1634 372 1635 373 1636 374 1637 375 1638 376 1639 377 1640 378 \
    1641 379 1642 380 1643 381 1644 382 1645 383 1646 384 1647 385 1648 386 \
    1649 387 1650 388 1651 389 1652 390 1653 391 1654 392 1655 393 1656 394 \
    1657 395 1658 396 1659 397 1660 398 1661 399 1662 400 1663 401 7968 402 \
    7969 403 7970 404 7971 405 7972 406 7973 407 7974 408 7975 409 7976 410 \
    7977 411 7978 412 7979 413 7980 414 7981 415 7982 416 7983 417 7984 418 \
    7985 419 7986 420 7987 421 7988 422 7989 423 7990 424 7991 425 7992 426 \
    7993 427 7994 428 7995 429 7996 430 7997 431 7998 432 7999 433 8000 434 \
    8001 435 8002 436 8003 437 8004 438 8005 439 8006 440 8007 441 8008 442 \
    8009 443 8010 444 8011 445 8012 446 8013 447 8014 448 8015 449 8016 450 \
    8017 451 8018 452 8019 453 8020 454 8021 455 8022 456 8023 457 8024 458 \
    8025 459 8026 460 8027 461 8032 462 8034 463 8035 464 8036 465 8037 466 \
    8038 467 8039 468 8040 469 8041 470 8042 471 8043 472 8044 473 8045 474 \
    8046 475 8047 476 8048 477 8049 478 8050 479 8051 480 8052 481 8053 482 \
    8054 483 8058 484 8059 485 8060 486 8061 487 8062 488 8063 489 8064 490 \
    8065 491 8066 492 8067 493 8068 494 8069 495 8070 496 8071 497 8072 498 \
    8073 499 8074 500 8075 501 8076 502 8077 503 8078 504 8079 505 8080 506 \
    8081 507 8082 508 8083 509 8084 510 8085 511 8086 512 8087 513 8088 514 \
    8089 515 8090 516 8091 517 8092 518 8093 519 8094 520 8095 521 8096 522 \
    8097 523 8098 524 8099 525 8100 526 8101 527 8102 528 8103 529 8106 530 \
    8107 531 8108 532 8109 533 8110 534 8111 535 8112 536 8113 537 8114 538 \
    8115 539 8116 540 8117 541 8118 542 8119 543 8120 544 8126 545 8127 546 \
    8130 547 8131 548 8134 549 8135 550 8136 551 8137 552 8138 553 8139 554 \
    8140 555 8141 556 8142 557 8143 558 8144 559 8145 560 8146 561 8147 562 \
    8148 563 8149 564 8150 565 8151 566 8152 567 8153 568 8154 569 8155 570 \
    8156 571 8157 572 8158 573 8159 574 8160 575 8161 576 8162 214 8163 577 \
    8164 578 8165 579 8166 580 8167 581 8168 582 8169 583 8170 584 8171 585 \
    8172 586 8173 587 8174 588 8175 589 8176 590 8177 591 8178 592 8179 593 \
    8180 594 8181 595 8182 596 8183 597 8184 598 8185 599 8186 600 8187 601 \
    8188 602 8189 603 14891 604 14892 605 14903 606 14904 607 14976 608 \
    14977 609 14978 610 14979 611 14980 612 14981 613 14982 614 14983 615 \
    14984 616 14985 617 14986 618 14987 619 14988 620 14989 608 14990 609 \
    14991 610 14992 611 14993 612 14994 613 14995 621 14996 622 14997 623 \
    14998 617 14999 624 15000 625 15001 620 15002 608 15003 609 15004 610 \
    15005 611 15006 612 15007 613 15008 626 15009 627 15010 628 15011 629 \
    15012 630 15013 619 15014 620 15015 631 15016 632 15017 633 15018 634 \
    15019 612 15020 613 15021 614 15022 615 15023 616 15024 617 15025 630 \
    15026 619 15027 620 15028 608 15029 609 15030 610 15031 611 15032 612 \
    15033 613 15034 614 15035 615 15036 616 15037 617 15038 630 15039 619 \
    15040 620 15041 608 15042 609 15043 610 15044 611 15045 612 15046 613 \
    15047 614 15048 615 15049 616 15050 617 15051 630 15052 619 15053 620 \
    15054 608 15055 609 15056 610 15057 611 15058 612 15059 613 15060 635 \
    15061 636 15062 637 15063 638 15064 639 15065 640 15066 641 15067 642 \
    15068 643 15069 644 15070 645 15071 646 15072 647 15073 648 15074 649 \
    15075 650 15076 651 15077 652 15078 653 15079 654 15080 655 15081 656 \
    15082 657 15083 658 15084 659 15085 660 15086 661 15087 662 15088 663 \
    15089 664 15090 636 15091 637 15092 638 15093 639 15094 640 15095 641 \
    15096 642 15097 665 15098 666 15099 214 15100 667 15101 668 15102 669 \
    15103 666 24320 670 24321 671 24322 672 24323 673 24324 674 24325 675 \
    24326 676 24327 677 24328 678 24329 679 24330 680 24331 681 24332 682 \
    24333 683 24334 684 24335 685 24336 686 24337 687 24338 688 24339 689 \
    24340 690 24341 691 24342 692 24343 693 24344 694 24345 695 24346 696 \
    24347 697 24348 698 24349 699 24350 700 24351 701 24352 702 24353 703 \
    24354 704 24355 705 24356 706 24357 707 24358 708 24359 709 24360 710 \
    24361 711 24362 712 24363 713 24364 714 24365 715 24366 716 24367 717 \
    24368 718 24369 719 24370 720 24371 721 24372 722 24373 723 24374 724 \
    24375 725 24376 726 24377 727 24378 728 24379 729 24380 730 24381 731 \
    24382 732 24383 733 24384 734 24385 735 24386 736 24387 737]

variable DECOMP_COMMON_PAGE_MAP 0

#
# The decompGroupMap is indexed by combining the alternate page number with
# the page offset and returns a group number that identifies a length and
# shift of decomposition sequence in decompList
#

variable decompGroupMap [list \
    -1 -1 -1 -1 -1 -1 -1 -1 196608 -1 -1 -1 -1 -1 -1 -1 327681 -1 196611 \
    -1 -1 -1 -1 327684 -1 -1 196614 196615 327688 196618 -1 -1 327691 196621 \
    196622 -1 458767 458770 458773 -1 262168 262170 262172 262174 262176 \
    262178 -1 262180 262182 262184 262186 262188 262190 262192 262194 262196 \
    -1 262198 262200 262202 262204 262206 262208 -1 -1 262210 262212 262214 \
    262216 262218 -1 -1 262220 262222 262224 262226 262228 262230 -1 262232 \
    262234 262236 262238 262240 262242 262244 262246 262248 -1 262250 262252 \
    262254 262256 262258 262260 -1 -1 262262 262264 262266 262268 262270 \
    -1 262272 262274 262276 262278 262280 262282 262284 262286 262288 262290 \
    262292 262294 262296 262298 262300 262302 262304 -1 -1 262306 262308 \
    262310 262312 262314 262316 262318 262320 262322 262324 262326 262328 \
    262330 262332 262334 262336 262338 262340 262342 262344 -1 -1 262346 \
    262348 262350 262352 262354 262356 262358 262360 262362 -1 327900 327902 \
    262368 262370 262372 262374 -1 262376 262378 262380 262382 262384 262386 \
    327924 327926 -1 -1 262392 262394 262396 262398 262400 262402 327940 \
    -1 -1 262406 262408 262410 262412 262414 262416 -1 -1 262418 262420 \
    262422 262424 262426 262428 262430 262432 262434 262436 262438 262440 \
    262442 262444 262446 262448 262450 262452 -1 -1 262454 262456 262458 \
    262460 262462 262464 262466 262468 262470 262472 262474 262476 262478 \
    262480 262482 262484 262486 262488 262490 262492 262494 262496 262498 \
    196964 262501 262503 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 262505 \
    262507 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 328045 328047 328049 328051 \
    328053 328055 328057 328059 328061 262527 262529 262531 262533 262535 \
    262537 262539 262541 262543 262545 262547 262549 262551 262553 262555 \
    262557 -1 262559 262561 262563 262565 262567 262569 -1 -1 262571 262573 \
    262575 262577 262579 262581 262583 262585 262587 262589 262591 328129 \
    328131 328133 262599 262601 -1 -1 262603 262605 262607 262609 262611 \
    262613 262615 262617 262619 262621 262623 262625 262627 262629 262631 \
    262633 262635 262637 262639 262641 262643 262645 262647 262649 262651 \
    262653 262655 262657 262659 262661 262663 262665 262667 262669 262671 \
    262673 -1 -1 262675 262677 -1 -1 -1 -1 -1 -1 262679 262681 262683 262685 \
    262687 262689 262691 262693 262695 262697 262699 262701 262703 262705 \
    -1 -1 -1 -1 197171 197172 197173 197174 197175 197176 197177 197178 \
    197179 -1 -1 -1 -1 -1 -1 -1 328252 328254 328256 328258 328260 328262 \
    -1 -1 197192 197193 196964 197194 197195 -1 -1 -1 131660 131661 -1 \
    131662 262735 -1 -1 -1 -1 -1 -1 -1 131665 -1 -1 -1 -1 -1 328274 -1 \
    -1 -1 131668 -1 -1 -1 -1 -1 327688 262741 262743 131673 262746 262748 \
    262750 -1 262752 -1 262754 262756 262758 -1 -1 -1 -1 -1 -1 -1 -1 -1 \
    262760 262762 262764 262766 262768 262770 262772 -1 -1 -1 -1 -1 -1 \
    -1 -1 -1 262774 262776 262778 262780 262782 -1 197248 197249 197250 \
    262787 262789 197255 197256 -1 197257 197258 197259 -1 197260 197261 \
    -1 -1 -1 197262 -1 -1 -1 -1 -1 -1 262799 262801 -1 262803 -1 -1 -1 \
    262805 -1 -1 -1 -1 262807 262809 262811 -1 -1 262813 -1 -1 -1 -1 -1 \
    -1 -1 262815 -1 -1 -1 -1 -1 -1 262817 262819 -1 262821 -1 -1 -1 262823 \
    -1 -1 -1 -1 262825 262827 262829 -1 -1 -1 -1 -1 -1 -1 262831 262833 \
    -1 262835 262837 -1 -1 -1 -1 -1 262839 262841 262843 262845 -1 -1 262847 \
    262849 -1 -1 262851 262853 262855 262857 262859 262861 -1 -1 262863 \
    262865 262867 262869 262871 262873 -1 -1 262875 262877 262879 262881 \
    262883 262885 262887 262889 262891 262893 262895 262897 -1 -1 262899 \
    262901 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 328439 -1 -1 262905 262907 \
    262909 262911 262913 -1 -1 -1 -1 -1 -1 328451 328453 328455 328457 \
    -1 -1 -1 -1 -1 -1 -1 262923 -1 262925 -1 -1 -1 -1 -1 -1 -1 -1 262927 \
    -1 -1 -1 -1 -1 262929 -1 -1 -1 -1 -1 -1 -1 262931 -1 -1 262933 -1 -1 \
    -1 262935 262937 262939 262941 262943 262945 262947 262949 -1 -1 -1 \
    262951 262953 -1 -1 -1 -1 -1 -1 -1 262955 262957 -1 262959 -1 -1 -1 \
    262961 -1 -1 262963 -1 -1 262965 262967 262969 -1 -1 262971 -1 262973 \
    -1 -1 262975 262977 -1 -1 -1 -1 -1 -1 -1 262979 262981 -1 -1 -1 -1 \
    -1 -1 262983 -1 -1 -1 -1 -1 262985 262987 262989 -1 -1 -1 262991 -1 \
    -1 -1 -1 -1 -1 -1 262993 -1 -1 -1 -1 -1 -1 262995 262997 -1 262999 \
    263001 -1 -1 -1 -1 -1 -1 263003 263005 263007 -1 -1 -1 -1 -1 263009 \
    -1 263011 263013 263015 -1 -1 -1 -1 328553 -1 -1 -1 -1 -1 -1 -1 328555 \
    -1 -1 -1 -1 -1 -1 -1 -1 328557 328559 -1 -1 -1 -1 -1 -1 197489 -1 -1 \
    -1 -1 -1 -1 263026 -1 -1 -1 -1 -1 -1 -1 -1 -1 263028 -1 -1 -1 -1 263030 \
    -1 -1 -1 -1 263032 -1 -1 -1 -1 263034 -1 -1 -1 -1 263036 -1 -1 -1 -1 \
    -1 -1 -1 -1 -1 263038 -1 263040 263042 328580 263046 328584 -1 -1 -1 \
    -1 -1 -1 -1 263050 -1 -1 -1 -1 -1 -1 -1 -1 -1 263052 -1 -1 -1 -1 -1 \
    -1 -1 -1 -1 263054 -1 -1 -1 -1 263056 -1 -1 -1 -1 263058 -1 -1 -1 -1 \
    263060 -1 -1 -1 -1 263062 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 263064 \
    -1 -1 -1 -1 -1 197530 -1 -1 -1 -1 -1 -1 -1 -1 -1 263067 -1 263069 -1 \
    263071 -1 263073 -1 263075 -1 -1 -1 263077 -1 -1 -1 -1 -1 -1 -1 -1 \
    263079 -1 263081 -1 -1 263083 263085 -1 263087 -1 -1 -1 -1 -1 -1 -1 \
    -1 197553 197554 197555 -1 197556 197557 197558 197559 197560 197561 \
    197562 197563 197564 197565 197566 -1 197567 197568 197569 197570 197571 \
    197572 197573 196611 197574 197575 197576 197577 197578 197579 197580 \
    197581 197582 197583 -1 197584 197585 197586 196622 197587 197588 197589 \
    197590 197591 197592 197593 197594 197595 197596 197248 197597 197598 \
    197255 197599 197600 197174 197592 197595 197248 197597 197258 197255 \
    197599 -1 -1 -1 -1 -1 197601 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 197602 197603 \
    197604 197605 197582 197606 197607 197608 197609 197610 197611 197612 \
    197613 197614 197615 197616 197617 197618 197619 197620 197621 197622 \
    197623 197624 197625 197626 197627 197628 197629 197630 197631 197632 \
    197633 197634 197635 197636 197249 263173 263175 263177 263179 263181 \
    263183 263185 263187 263189 263191 263193 263195 263197 263199 263201 \
    263203 263205 263207 263209 263211 263213 263215 263217 263219 263221 \
    263223 263225 263227 263229 263231 263233 263235 263237 263239 263241 \
    263243 263245 263247 263249 263251 263253 263255 263257 263259 263261 \
    263263 263265 263267 263269 263271 263273 263275 263277 263279 263281 \
    263283 263285 263287 263289 263291 263293 263295 263297 263299 263301 \
    263303 263305 263307 263309 263311 263313 263315 263317 263319 263321 \
    263323 263325 263327 263329 263331 263333 263335 263337 263339 263341 \
    263343 263345 263347 263349 263351 263353 263355 263357 263359 263361 \
    263363 263365 263367 263369 263371 263373 263375 263377 263379 263381 \
    263383 263385 263387 263389 263391 263393 263395 263397 263399 263401 \
    263403 263405 263407 263409 263411 263413 263415 263417 263419 263421 \
    263423 263425 263427 263429 263431 263433 263435 263437 263439 263441 \
    263443 263445 263447 263449 263451 263453 263455 263457 263459 263461 \
    263463 263465 263467 263469 263471 263473 263475 263477 263479 329017 \
    263483 -1 -1 -1 -1 263485 263487 263489 263491 263493 263495 263497 \
    263499 263501 263503 263505 263507 263509 263511 263513 263515 263517 \
    263519 263521 263523 263525 263527 263529 263531 263533 263535 263537 \
    263539 263541 263543 263545 263547 263549 263551 263553 263555 263557 \
    263559 263561 263563 263565 263567 263569 263571 263573 263575 263577 \
    263579 263581 263583 263585 263587 263589 263591 263593 263595 263597 \
    263599 263601 263603 263605 263607 263609 263611 263613 263615 263617 \
    263619 263621 263623 263625 263627 263629 263631 263633 263635 263637 \
    263639 263641 263643 263645 263647 263649 263651 263653 263655 263657 \
    263659 263661 263663 -1 -1 -1 -1 -1 -1 263665 263667 263669 263671 \
    263673 263675 263677 263679 263681 263683 263685 263687 263689 263691 \
    263693 263695 263697 263699 263701 263703 263705 263707 -1 -1 263709 \
    263711 263713 263715 263717 263719 -1 -1 263721 263723 263725 263727 \
    263729 263731 263733 263735 263737 263739 263741 263743 263745 263747 \
    263749 263751 263753 263755 263757 263759 263761 263763 263765 263767 \
    263769 263771 263773 263775 263777 263779 263781 263783 263785 263787 \
    263789 263791 263793 263795 -1 -1 263797 263799 263801 263803 263805 \
    263807 -1 -1 263809 263811 263813 263815 263817 263819 263821 263823 \
    -1 263825 -1 263827 -1 263829 -1 263831 263833 263835 263837 263839 \
    263841 263843 263845 263847 263849 263851 263853 263855 263857 263859 \
    263861 263863 263865 132795 263868 132798 263871 132801 263874 132804 \
    263877 132807 263880 132810 263883 132813 -1 -1 263886 263888 263890 \
    263892 263894 263896 263898 263900 263902 263904 263906 263908 263910 \
    263912 263914 263916 263918 263920 263922 263924 263926 263928 263930 \
    263932 263934 263936 263938 263940 263942 263944 263946 263948 263950 \
    263952 263954 263956 263958 263960 263962 263964 263966 263968 263970 \
    263972 263974 263976 263978 263980 263982 263984 263986 263988 263990 \
    -1 263992 263994 263996 263998 264000 132930 264003 329541 132935 329541 \
    329544 264010 264012 264014 264016 -1 264018 264020 264022 132952 264025 \
    132955 264028 264030 264032 264034 264036 264038 264040 132970 -1 -1 \
    264043 264045 264047 264049 264051 132981 -1 264054 264056 264058 264060 \
    264062 264064 132994 264067 264069 264071 264073 264075 264077 264079 \
    133009 264082 264084 133014 133015 -1 -1 264088 264090 264092 -1 264094 \
    264096 264098 133028 264101 133031 264104 133034 329643 -1 133037 133038 \
    196608 196608 196608 196608 196608 196608 196608 196608 196608 -1 -1 \
    -1 -1 -1 -1 198575 -1 -1 -1 -1 -1 329648 -1 -1 -1 -1 198578 329651 \
    460725 -1 -1 -1 -1 -1 -1 -1 -1 196608 -1 -1 -1 329656 460730 -1 329661 \
    460735 -1 -1 -1 -1 329666 -1 329668 -1 -1 -1 -1 -1 -1 -1 -1 329670 \
    329672 329674 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 591820 198608 \
    197600 -1 -1 198609 198610 198611 198612 198613 198614 198615 198616 \
    198617 198618 198619 198620 198608 196621 196614 196615 198609 198610 \
    198611 198612 198613 198614 198615 198616 198617 198618 198619 -1 196611 \
    197579 196622 197194 197580 -1 -1 -1 329693 -1 -1 -1 -1 -1 -1 -1 460767 \
    460770 198629 329702 -1 460776 460779 198638 -1 329711 197583 197560 \
    197560 197560 197171 198641 197561 197561 197564 197193 -1 197566 329714 \
    -1 -1 197569 198644 197570 197570 197570 -1 -1 329717 460791 329722 \
    -1 198652 -1 133117 -1 198652 -1 132027 133118 197555 198629 -1 197579 \
    197557 198655 -1 197565 196622 198656 198657 198658 198659 197600 -1 \
    460804 197256 197597 198663 198664 198665 -1 -1 -1 -1 197556 197578 \
    197579 197600 197173 -1 -1 -1 -1 -1 -1 -1 -1 -1 460810 460813 460816 \
    460819 460822 460825 460828 460831 460834 460837 460840 460843 329774 \
    197561 329776 460850 329781 198711 329784 460858 591933 329793 198723 \
    329796 460870 197564 198629 197556 197565 197600 329801 460875 329806 \
    197595 329808 460882 591957 329817 197194 329819 460893 197193 197603 \
    197578 197585 -1 -1 264288 264290 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 264292 \
    -1 -1 -1 -1 -1 -1 264294 264296 264298 -1 -1 -1 -1 264300 -1 -1 -1 \
    -1 264302 -1 -1 264304 -1 -1 -1 -1 -1 -1 -1 264306 -1 264308 -1 -1 \
    -1 -1 -1 329846 460920 -1 329851 460925 -1 -1 -1 -1 -1 -1 -1 -1 264320 \
    -1 -1 264322 -1 -1 264324 -1 264326 -1 -1 -1 -1 -1 -1 264328 -1 264330 \
    -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 264332 264334 264336 264338 264340 -1 \
    -1 264342 264344 -1 -1 264346 264348 -1 -1 -1 -1 -1 -1 264350 264352 \
    -1 -1 264354 264356 -1 -1 264358 264360 -1 -1 -1 -1 -1 -1 -1 -1 -1 \
    -1 264362 264364 264366 264368 264370 264372 264374 264376 -1 -1 -1 \
    -1 -1 -1 264378 264380 264382 264384 -1 -1 -1 133314 133315 -1 -1 -1 \
    -1 -1 196621 196614 196615 198609 198610 198611 198612 198613 198614 \
    329924 329926 329928 329930 329932 329934 329936 329938 329940 329942 \
    329944 461018 461021 461024 461027 461030 461033 461036 461039 461042 \
    592117 592121 592125 592129 592133 592137 592141 592145 592149 592153 \
    592157 330017 330019 330021 330023 330025 330027 330029 330031 330033 \
    461107 461110 461113 461116 461119 461122 461125 461128 461131 461134 \
    461137 461140 461143 461146 461149 461152 461155 461158 461161 461164 \
    461167 461170 461173 461176 461179 461182 461185 461188 461191 461194 \
    461197 461200 461203 461206 461209 461212 461215 197553 197555 198629 \
    197556 197557 198655 197559 197560 197561 197562 197563 197564 197565 \
    197566 197567 197569 198644 197570 199074 197571 197572 198711 197573 \
    198723 199075 198652 196611 197577 197603 197578 197579 197606 197583 \
    197171 197600 197173 197584 197193 197585 198620 196622 197590 199076 \
    197174 196964 197591 197592 197595 197178 197194 197179 197633 198608 \
    -1 -1 -1 -1 -1 -1 -1 -1 -1 592293 -1 -1 -1 -1 -1 -1 -1 461225 330156 \
    461230 -1 -1 -1 -1 -1 264625 -1 -1 -1 -1 -1 -1 -1 -1 -1 -1 199091 -1 \
    -1 -1 -1 -1 -1 -1 199092 -1 -1 -1 199093 -1 -1 -1 -1 199094 199095 \
    199096 199097 199098 199099 199100 199101 199102 199103 199104 199105 \
    199106 199107 199108 199109 199110 199111 199112 199113 199114 199115 \
    199116 199117 199118 199119 199120 199121 199122 199123 199124 199125 \
    199126 199127 199128 199129 199130 199131 199132 199133 199134 199135 \
    199136 199137 199138 199139 199140 199141 199142 199143 199144 199145 \
    199146 199147 199148 199149 199150 199151 199152 199153 199154 199155 \
    199156 199157 199158 199159 199160 199161 199162 199163 199164 199165 \
    199166 199167 199168 199169 199170 199171 199172 199173 199174 199175 \
    199176 199177 199178 199179 199180 199181 199182 199183 199184 199185 \
    199186 199187 199188 199189 199190 199191 199192 199193 199194 199195 \
    199196 199197 199198 199199 199200 199201 199202 199203 199204 199205 \
    199206 199207 199208 199209 199210 199211 199212 199213 199214 199215 \
    199216 199217 199218 199219 199220 199221 199222 199223 199224 199225 \
    199226 199227 199228 199229 199230 199231 199232 199233 199234 199235 \
    199236 199237 199238 199239 199240 199241 199242 199243 199244 199245 \
    199246 199247 199248 199249 199250 199251 199252 199253 199254 199255 \
    199256 199257 199258 199259 199260 199261 199262 199263 199264 199265 \
    199266 199267 199268 199269 199270 199271 199272 199273 199274 199275 \
    199276 199277 199278 199279 199280 199281 199282 199283 199284 199285 \
    199286 199287 199288 199289 199290 199291 199292 199293 199294 199295 \
    199296 199297 199298 199299 199300 199301 199302 199303 199304 199305 \
    199306 199307 -1 -1 -1 -1 -1 -1 -1 -1 199308 -1 199117 199309 199310 \
    -1 -1 -1 -1 -1 -1 -1 -1 -1 264847 -1 264849 -1 264851 -1 264853 -1 \
    264855 -1 264857 -1 264859 -1 264861 -1 264863 -1 264865 -1 264867 \
    -1 264869 -1 -1 264871 -1 264873 -1 264875 -1 -1 -1 -1 -1 -1 264877 \
    264879 -1 264881 264883 -1 264885 264887 -1 264889 264891 -1 264893 \
    264895 -1 -1 -1 -1 -1 -1 264897 -1 -1 -1 -1 -1 -1 330435 330437 -1 \
    264903 330441 -1 -1 -1 -1 264907 -1 264909 -1 264911 -1 264913 -1 264915 \
    -1 264917 -1 264919 -1 264921 -1 264923 -1 264925 -1 264927 -1 264929 \
    -1 -1 264931 -1 264933 -1 264935 -1 -1 -1 -1 -1 -1 264937 264939 -1 \
    264941 264943 -1 264945 264947 -1 264949 264951 -1 264953 264955 -1 \
    -1 -1 -1 -1 -1 264957 -1 -1 264959 264961 264963 264965 -1 -1 -1 264967 \
    330505 -1 199435 199436 199437 199438 199439 199440 199441 199442 199443 \
    199444 199445 199446 199447 199448 199449 199450 199451 199452 199453 \
    199454 199455 199456 199457 199458 199459 199460 199461 199462 199463 \
    199464 199465 199466 199467 199468 199469 199470 199471 199472 199473 \
    199474 199475 199476 199477 199478 199479 199480 199481 199482 199483 \
    199484 199485 199486 199487 199488 199489 199490 199491 199492 199493 \
    199494 199495 199496 199497 199498 199499 199500 199501 199502 199503 \
    199504 199505 199506 199507 199508 199509 199510 199511 199512 199513 \
    199514 199515 199516 199517 199518 199519 199520 199521 199522 199523 \
    199524 199525 199526 199527 199528 -1 -1 -1 199094 199100 199529 199530 \
    199531 199532 199533 199534 199098 199535 199536 199537 199538 199102 \
    461683 461686 461689 461692 461695 461698 461701 461704 461707 461710 \
    461713 461716 461719 461722 592797 592801 592805 592809 592813 592817 \
    592821 592825 592829 592833 592837 592841 592845 592849 592853 986073 \
    855008 -1 461798 461801 461804 461807 461810 461813 461816 461819 461822 \
    461825 461828 461831 461834 461837 461840 461843 461846 461849 461852 \
    461855 461858 461861 461864 461867 461870 461873 461876 461879 461882 \
    461885 461888 461891 461894 461897 461900 461903 -1 -1 -1 -1 461906 \
    330837 330839 330841 330843 330845 330847 330849 330851 330853 330855 \
    330857 330859 330861 330863 330865 199435 199438 199441 199443 199451 \
    199452 199455 199457 199458 199460 199461 199462 199463 199464 330867 \
    330869 330871 330873 330875 330877 330879 330881 330883 330885 330887 \
    330889 330891 330893 724111 593044 330904 -1 199094 199100 199529 199530 \
    199834 199835 199836 199105 199837 199117 199167 199179 199178 199168 \
    199260 199125 199165 199838 199839 199840 199841 199842 199843 199844 \
    199845 199846 199847 199131 199848 199849 199850 199851 199852 199853 \
    199854 199855 199531 199532 199533 199856 199857 199858 199859 199860 \
    199861 199862 199863 199864 199865 330938 330940 330942 330944 330946 \
    330948 330950 330952 330954 330956 330958 330960 330962 330964 330966 \
    330968 330970 330972 330974 330976 330978 330980 330982 330984 462058 \
    462061 462064 330995 462069 331000 462074 199933 199934 199935 199936 \
    199937 199938 199939 199940 199941 199942 199943 199944 199945 199946 \
    199947 199948 199949 199950 199951 199952 199953 199954 199955 199956 \
    199957 199958 199959 199960 199961 199962 199963 199964 199965 199966 \
    199967 199968 199969 199970 199971 199972 199973 199974 199975 199976 \
    199977 199978 199979 -1 593196 593200 593204 462136 593211 462143 462146 \
    724293 593226 462158 462161 462164 593239 593243 462175 462178 331109 \
    462183 593258 593262 331122 724340 855417 724351 462212 724359 724364 \
    593297 462229 462232 462235 593310 724386 593319 462251 462254 462257 \
    331188 331190 331192 331194 462268 462271 724418 462279 593354 724430 \
    462291 331222 331224 724442 593375 724451 462312 724459 331248 462322 \
    462325 462328 462331 462334 593409 462341 331272 462346 462349 462352 \
    593427 462359 462362 462365 724512 593445 331305 724523 331312 593458 \
    593462 462394 462397 462400 593475 331335 462409 593484 331344 724562 \
    462423 331354 331356 331358 331360 331362 331364 331366 331368 331370 \
    331372 462446 462449 462452 462455 462458 462461 462464 462467 462470 \
    462473 462476 462479 462482 462485 462488 462491 331422 331424 462498 \
    331429 331431 331433 462507 462510 331441 331443 331445 331447 331449 \
    593595 331455 331457 331459 331461 331463 331465 331467 331469 462543 \
    593618 331478 331480 331482 331484 331486 331488 331490 462564 462567 \
    462570 462573 331504 331506 331508 331510 331512 331514 331516 331518 \
    331520 331522 462596 462599 331530 462604 462607 462610 331541 462615 \
    462618 593693 331553 462627 462630 462633 462636 724783 855860 331578 \
    331580 331582 331584 331586 331588 331590 331592 331594 331596 331598 \
    331600 331602 331604 331606 331608 331610 331612 593758 331618 331620 \
    331622 593768 462700 331631 331633 331635 331637 331639 331641 331643 \
    331645 331647 331649 462723 331654 331656 462730 462733 331664 593810 \
    462742 331673 331675 331677 331679 462753 462756 331687 331689 331691 \
    331693 331695 331697 331699 331701 331703 462777 462780 462783 462786 \
    462789 462792 462795 462798 462801 462804 462807 462810 462813 462816 \
    462819 462822 462825 462828 462831 462834 462837 462840 462843 135166 \
    135167 133716 135168 135169 135170 135171 133770 133770 135172 133724 \
    135173 135174 135175 135176 135177 135178 135179 135180 135181 135182 \
    135183 135184 135185 135186 135187 135188 135189 135190 135191 135192 \
    135193 135194 135195 135196 135197 135198 135199 135200 135201 135202 \
    135203 135204 135205 135206 135207 135208 135209 135210 135211 135212 \
    135213 133682 135214 135215 135216 135217 135218 135219 135220 135221 \
    135222 135223 135224 133755 135225 135226 135227 135228 135229 135230 \
    135231 135232 135233 135234 135235 135236 135237 135238 135239 135240 \
    135241 135242 135243 135244 135245 135246 135247 135248 135249 135250 \
    135251 135182 135252 135253 135254 135255 135256 135257 135258 135259 \
    135260 135261 135262 135263 135264 135265 135266 135267 135268 135269 \
    135270 135271 133718 135272 135273 135274 135275 135276 135277 135278 \
    135279 135280 135281 135282 135283 135284 135285 135286 133595 135287 \
    135288 135289 135290 135291 135292 135293 135294 133576 135295 135296 \
    135297 135298 135299 135300 135301 135302 135303 135304 135305 135306 \
    135307 135308 135309 135310 135311 135312 135313 135314 135315 135316 \
    135270 135317 135318 135319 135320 135321 135322 135323 135324 135254 \
    135325 135326 135327 135328 135329 135330 135331 135332 135333 135334 \
    135335 135336 135337 135338 135339 135340 135341 135342 135343 135344 \
    135182 135345 135346 135347 135348 133769 135349 135350 135351 135352 \
    135353 135354 135355 135356 135357 135358 135359 135360 134299 135361 \
    135362 135363 135364 135365 135366 135367 135368 135369 135256 135370 \
    135371 135372 135373 135374 135375 135376 135377 135378 135379 135380 \
    135381 135382 133723 135383 135384 135385 135386 135387 135388 135389 \
    135390 135391 135392 135393 135394 135395 133674 135396 135397 135398 \
    135399 135400 135401 135402 135403 135404 135405 135406 135407 135408 \
    135409 135410 135411 133701 135412 133704 135413 135414 135415 -1 -1 \
    135416 -1 135417 -1 -1 135418 135419 135420 135421 135422 135423 135424 \
    135425 135426 133681 -1 135427 -1 135428 -1 -1 135429 135430 -1 -1 \
    -1 135431 135432 135433 135434 -1 -1 135435 135436 135437 135438 135439 \
    135440 135441 135442 135443 135444 135445 135446 133602 135447 135448 \
    135449 135450 135451 135452 135453 135454 135455 135456 135457 135458 \
    135459 135460 135461 134304 135462 135463 135464 135465 134308 135466 \
    135467 135468 135469 135470 135306 135471 135472 135473 135474 135475 \
    135476 135476 135477 135478 135479 135480 135481 135482 135483 135484 \
    135429 135485 135486 135487 -1 -1 -1 -1 -1 135488 135489 135490 135491 \
    135492 135493 135494 135495 135441 135496 135497 135498 135416 135499 \
    135500 135501 135502 135503 135504 135505 135506 135507 135508 135509 \
    135510 135449 135511 135450 135512 135513 135514 135515 135516 135417 \
    135203 135517 135518 133635 135271 135354 135519 135520 135457 135521 \
    135458 135522 135523 135524 135419 135525 135526 135527 135528 135529 \
    135420 135530 135531 135532 135533 135534 135535 135470 135536 135537 \
    135306 135538 135474 135539 135540 135541 135542 135543 135479 135544 \
    135428 135545 135480 135252 135546 135481 135547 135483 135548 135549 \
    135550 135551 135552 135485 135425 135553 135486 135554 135487 135555 \
    133770 135556 135557 135558 135559 135560 135561 135562 135563 135564 \
    135565 135566 -1 -1 -1 -1 -1 -1 332175 332177 332179 463253 463256 \
    332187 332189 -1 -1 -1 -1 332191 332193 332195 332197 332199 -1 -1 \
    -1 -1 -1 266665 -1 266667 201133 198656 198659 201134 201135 201136 \
    201137 201138 201139 198615 266676 266678 266680 266682 266684 266686 \
    266688 266690 266692 266694 266696 266698 266700 -1 266702 266704 266706 \
    266708 266710 -1 266712 -1 266714 266716 -1 266718 266720 -1 266722 \
    266724 266726 266728 266730 266732 266734 266736 266738 332276 201206 \
    201206 201207 201207 201207 201207 201208 201208 201208 201208 201209 \
    201209 201209 201209 201210 201210 201210 201210 201211 201211 201211 \
    201211 201212 201212 201212 201212 201213 201213 201213 201213 201214 \
    201214 201214 201214 201215 201215 201215 201215 201216 201216 201216 \
    201216 201217 201217 201217 201217 201218 201218 201218 201218 201219 \
    201219 201220 201220 201221 201221 201222 201222 201223 201223 201224 \
    201224 201225 201225 201225 201225 201226 201226 201226 201226 201227 \
    201227 201227 201227 201228 201228 201228 201228 201229 201229 201230 \
    201230 201230 201230 201231 201231 201232 201232 201232 201232 201233 \
    201233 201233 201233 201234 201234 201235 201235 -1 -1 -1 -1 -1 -1 \
    -1 -1 -1 201236 201236 201236 201236 201237 201237 201238 201238 201239 \
    201239 201240 201241 201241 201242 201242 201243 201243 201244 201244 \
    201244 201244 201245 201245 332318 332318 332320 332320 332322 332322 \
    332324 332324 332326 332326 332328 332328 332330 332330 332330 332332 \
    332332 332332 201262 201262 201262 201262 332335 332337 332339 332332 \
    332341 332343 332345 332347 332349 332351 332353 332355 332357 332359 \
    332361 332363 332365 332367 332369 332371 332373 332375 332377 332379 \
    332381 332383 332385 332387 332389 332391 332393 332395 332397 332399 \
    332401 332403 332405 332407 332409 332411 332413 332415 332417 332419 \
    332421 332423 332425 332427 332429 332431 332433 332435 332437 332439 \
    332441 332443 332445 332447 332449 332451 332453 332455 332457 332459 \
    332461 332463 332465 332467 332469 332471 332473 332475 332477 332479 \
    332481 332483 332485 332487 332489 332491 332493 332495 332497 332499 \
    332501 332503 332505 332507 332509 332511 332513 332515 332517 332519 \
    463593 463596 463599 463602 463605 463608 332539 332541 332339 332543 \
    332332 332341 332545 332547 332349 332549 332351 332353 332551 332553 \
    332361 332555 332363 332365 332557 332559 332369 332561 332371 332373 \
    332431 332433 332439 332441 332443 332451 332453 332455 332457 332465 \
    332467 332469 332563 332477 332565 332567 332489 332569 332491 332493 \
    332519 332571 332573 332509 332575 332511 332513 332335 332337 332577 \
    332339 332579 332343 332345 332347 332349 332581 332355 332357 332359 \
    332361 332583 332369 332375 332377 332379 332381 332383 332387 332389 \
    332391 332393 332395 332397 332585 332399 332401 332403 332405 332407 \
    332409 332413 332415 332417 332419 332421 332423 332425 332427 332429 \
    332435 332437 332445 332447 332449 332451 332453 332459 332461 332463 \
    332465 332587 332471 332473 332475 332477 332483 332485 332487 332489 \
    332589 332495 332497 332591 332503 332505 332507 332509 332593 332339 \
    332579 332349 332581 332361 332583 332369 332595 332395 332597 332599 \
    332601 332451 332453 332465 332489 332589 332509 332593 463675 463678 \
    463681 332612 332614 332616 332618 332620 332622 332624 332626 332628 \
    332630 332632 332634 332636 332638 332640 332642 332644 332646 332648 \
    332650 332652 332654 332656 332599 332658 332660 332662 332664 332612 \
    332614 332616 332618 332620 332622 332624 332626 332628 332630 332632 \
    332634 332636 332638 332640 332642 332644 332646 332648 332650 332652 \
    332654 332656 332599 332658 332660 332662 332664 332652 332654 332656 \
    332599 332597 332601 332411 332389 332391 332393 332652 332654 332656 \
    332411 332413 332666 332666 -1 -1 463740 463743 463743 463746 463749 \
    463752 463755 463758 463761 463761 463764 463767 463770 463773 463776 \
    463779 463779 463782 463785 463785 463788 463788 463791 463794 463794 \
    463797 463800 463800 463803 463803 463806 463809 463809 463812 463812 \
    463815 463818 463821 463824 463824 463827 463830 463833 463836 463839 \
    463839 463842 463845 463848 463851 463854 463857 463857 463860 463860 \
    463863 463863 463866 463869 463872 463875 463878 463881 463884 -1 -1 \
    463887 463890 463893 463896 463899 463902 463902 463905 463908 463911 \
    463914 463914 463917 463920 463923 463926 463929 463932 463935 463938 \
    463941 463944 463947 463950 463953 463956 463959 463962 463965 463968 \
    463971 463974 463977 463980 463842 463848 463983 463986 463989 463992 \
    463995 463998 463995 463989 464001 464004 464007 464010 464013 463998 \
    463821 463791 464016 464019 464022 464025 595100 595104 595108 595112 \
    595116 595120 595124 464056 2430139 1119437 595157 -1 -1 -1 201945 \
    201946 201947 201948 197204 201949 201950 201951 201952 201953 -1 -1 \
    -1 -1 -1 -1 201954 201955 201956 201957 201957 198618 198619 201958 \
    201959 201960 201961 201962 201963 201964 201965 198850 198851 201966 \
    201967 201968 201969 -1 -1 201970 201971 201972 201972 201972 201972 \
    201957 201957 201957 201945 201946 198578 -1 197204 201948 201950 201949 \
    201955 198618 198619 201958 201959 201960 201961 201973 201974 201975 \
    198615 201976 201977 201978 198617 -1 201979 201980 201981 201982 -1 \
    -1 -1 -1 333055 333057 333059 -1 333061 -1 333063 333065 333067 333069 \
    333071 333073 333075 333077 333079 333081 202011 202012 202012 202013 \
    202013 202014 202014 202015 202015 202016 202016 202016 202016 202017 \
    202017 202018 202018 202018 202018 202019 202019 202020 202020 202020 \
    202020 202021 202021 202021 202021 202022 202022 202022 202022 202023 \
    202023 202023 202023 202024 202024 202024 202024 202025 202025 202026 \
    202026 202027 202027 202028 202028 202029 202029 202029 202029 202030 \
    202030 202030 202030 202031 202031 202031 202031 202032 202032 202032 \
    202032 202033 202033 202033 202033 202034 202034 202034 202034 202035 \
    202035 202035 202035 202036 202036 202036 202036 202037 202037 202037 \
    202037 202038 202038 202038 202038 202039 202039 202039 202039 202040 \
    202040 202040 202040 202041 202041 202041 202041 202042 202042 202042 \
    202042 202043 202043 202043 202043 202044 202044 201245 201245 202045 \
    202045 202045 202045 333118 333118 333120 333120 333122 333122 333124 \
    333124 -1 -1 -1 -1 201949 202054 201973 201980 201981 201974 202055 \
    198618 198619 201975 198615 201945 201976 198578 202056 198613 198614 \
    201948 197204 201977 198617 201978 201950 201982 197553 197555 198629 \
    197556 197557 198655 197559 197560 197561 197562 197563 197564 197565 \
    197566 197567 197569 198644 197570 199074 197571 197572 198711 197573 \
    198723 199075 198652 201970 201979 201971 202057 201957 198551 196611 \
    197577 197603 197578 197579 197606 197583 197171 197600 197173 197584 \
    197193 197585 198620 196622 197590 199076 197174 196964 197591 197592 \
    197595 197178 197194 197179 197633 201958 202058 201959 202059 202060 \
    202061 201947 201966 201967 201946 202062 199979 202063 202064 202065 \
    202066 202067 202068 202069 202070 202071 202072 199933 199934 199935 \
    199936 199937 199938 199939 199940 199941 199942 199943 199944 199945 \
    199946 199947 199948 199949 199950 199951 199952 199953 199954 199955 \
    199956 199957 199958 199959 199960 199961 199962 199963 199964 199965 \
    199966 199967 199968 199969 199970 199971 199972 199973 199974 199975 \
    199976 202073 202074 202075 202076 202077 202078 202079 202080 202081 \
    202082 202083 202084 202085 202086 202087 202088 202089 202090 202091 \
    202092 202093 202094 202095 202096 202097 202098 202099 202100 202101 \
    202102 202103 202104 202105 202106 -1 -1 -1 202107 202108 202109 202110 \
    202111 202112 -1 -1 202113 202114 202115 202116 202117 202118 -1 -1 \
    202119 202120 202121 202122 202123 202124 -1 -1 202125 202126 202127 \
    -1 -1 -1 202128 202129 202130 202131 202132 202133 202134 -1 202135 \
    202136 202137 202138 202139 202140 202141 -1 -1 -1 -1 -1 -1 -1 267678 \
    267680 267682 267684 267686 267688 267690 -1 -1 -1 -1 -1 -1 267692 \
    267694 267696 267698 267700 267702 -1 -1 -1 -1 -1 -1 -1 197553 197555 \
    198629 197556 197557 198655 197559 197560 197561 197562 197563 197564 \
    197565 197566 197567 197569 198644 197570 199074 197571 197572 198711 \
    197573 198723 199075 198652 196611 197577 197603 197578 197579 197606 \
    197583 197171 197600 197173 197584 197193 197585 198620 196622 197590 \
    199076 197174 196964 197591 197592 197595 197178 197194 197179 197633 \
    197553 197555 198629 197556 197557 198655 197559 197560 197561 197562 \
    197563 197564 197565 197566 197567 197569 198644 197570 199074 197571 \
    197572 198711 197573 198723 199075 198652 196611 197577 197603 197578 \
    197579 197606 197583 -1 197600 197173 197584 197193 197585 198620 196622 \
    197590 199076 197174 196964 197591 197592 197595 197178 197194 197179 \
    197633 197178 197194 197179 197633 197553 -1 198629 197556 -1 -1 197559 \
    -1 -1 197562 197563 -1 -1 197566 197567 197569 198644 -1 199074 197571 \
    197603 197578 -1 197606 -1 197171 197600 197173 197584 197193 197585 \
    198620 -1 197590 199076 197174 197178 197194 197179 197633 197553 197555 \
    -1 197556 197557 198655 197559 -1 -1 197562 197563 197564 197565 197566 \
    197567 197569 198644 -1 199074 197571 197572 198711 197573 198723 199075 \
    -1 196611 197577 197603 197578 197579 197606 197583 197171 197600 197173 \
    197553 197555 -1 197556 197557 198655 197559 -1 197561 197562 197563 \
    197564 197565 -1 197567 -1 -1 -1 199074 197571 197572 198711 197573 \
    198723 199075 -1 196611 197577 197603 197578 197579 197606 197178 197194 \
    197179 197633 202168 202169 -1 -1 202170 202171 198663 202172 202173 \
    202174 202175 197260 202176 202177 202178 202179 202180 202181 202182 \
    198664 202183 202184 197262 202185 197250 202186 202187 202188 198653 \
    202189 202190 197248 197597 197598 197261 202191 202192 197249 198471 \
    197257 202193 196618 202194 202195 202196 197256 197258 197259 202197 \
    202198 202199 197255 197599 202200 202201 202202 202203 202204 202205 \
    202206 202207 202208 202170 202171 198663 202172 202173 202174 202175 \
    197260 202176 202177 202178 202179 202180 202181 202182 198664 202183 \
    202184 197262 202185 197250 202186 202187 202188 198653 202189 202190 \
    197248 197597 197598 197261 202191 202192 197249 198471 197257 202193 \
    196618 202194 202195 202196 197256 197258 197259 202197 202198 202199 \
    197255 197599 202200 202201 202202 202203 202204 202205 202206 202207 \
    202208 202170 202171 198663 202172 202173 202174 202175 197260 202176 \
    202177 202178 202179 202180 202181 202182 198664 202183 202184 197262 \
    202185 197250 202186 202187 202188 198653 202189 202190 197248 197597 \
    197598 197261 202191 202192 197249 198471 197257 202193 196618 202194 \
    202195 202196 197256 197258 197259 202197 202198 202199 197255 197599 \
    202200 202201 202202 202203 202204 202205 202206 202207 202208 202170 \
    202171 198663 202172 202173 202174 202175 197260 202176 202177 202178 \
    202179 202180 202181 202182 198664 202183 202184 197262 202185 197250 \
    202186 202187 202188 198653 202189 202190 197248 197597 197598 197261 \
    202191 202192 197249 198471 197257 202193 196618 202194 202195 202196 \
    197256 197258 197259 202197 202198 202199 197255 197599 202200 202201 \
    202202 202203 202204 202205 202206 202207 202208 202207 202208 202209 \
    202210 -1 -1 198608 196621 196614 196615 198609 198610 198611 198612 \
    198613 198614 198613 198614 198608 196621 196614 196615 198609 198610 \
    198611 198612 198613 198614 198608 196621 196614 196615 198609 198610 \
    198611 198612 198613 198614 198608 196621 136675 136676 136677 136678 \
    136679 135435 136680 136681 136682 136683 135436 136684 136685 136686 \
    135437 136687 136688 136689 136690 136691 136692 136693 136694 136695 \
    136696 136697 136698 135489 136699 133574 136700 136701 136702 136703 \
    136704 136705 136706 135494 135438 135439 135495 136707 136708 135258 \
    136709 135440 136710 136711 136712 136713 136713 136713 136714 136715 \
    136716 136717 136718 136719 136720 136721 136722 136723 136724 136725 \
    136726 136727 136728 136729 136730 136731 136731 135497 136732 136733 \
    136734 136735 135442 136736 136737 136738 135404 136739 136740 136741 \
    136742 136743 136744 136745 136746 136747 136748 136749 136750 136751 \
    136752 136753 136754 136755 136756 136757 136758 136759 136760 136761 \
    136762 136763 136764 136764 136765 136766 136767 135254 136768 136769 \
    136770 136771 136772 133600 136773 136774 133602 136775 136776 136777 \
    136778 136779 136780 136781 136782 136783 136784 136785 136786 136787 \
    136788 136789 136790 136791 136792 136793 136794 136795 135202 136796 \
    133612 136797 136797 136798 136799 136799 136800 136801 136802 136803 \
    136804 136805 136806 136807 136808 136809 136810 136811 136812 135447 \
    136813 136814 136815 136816 135509 136816 136817 135449 136818 136819 \
    136820 136821 135450 135175 136822 136823 136824 136825 136826 136827 \
    136828 136829 136830 136831 136832 136833 136834 136835 136836 136837 \
    136838 136839 136840 136841 136842 136843 135451 136844 136845 136846 \
    136847 136848 136849 135453 136850 136851 136852 136853 136854 136855 \
    136856 136857 135203 135517 136858 136859 136860 136861 136862 136863 \
    136864 136865 135454 136866 136867 136868 136869 135559 136870 136871 \
    136872 136873 136874 136875 136876 136877 136878 136879 136880 136881 \
    136882 135271 136883 136884 136885 136886 136887 136888 136889 136890 \
    136891 136892 136893 135455 135354 136894 136895 136896 136897 136898 \
    136899 136900 136901 135520 136902 136903 136904 136905 136906 136907 \
    136908 136909 135521 136910 136911 136912 136913 136914 136915 136916 \
    136917 136918 136919 136920 136921 135523 136922 136923 136924 136925 \
    136926 136927 136928 136929 136930 136931 136932 136932 136933 136934 \
    135525 136935 136936 136937 136938 136939 136940 136941 135257 136942 \
    136943 136944 136945 136946 136947 136948 135531 136949 136950 136951 \
    136952 136953 136954 136954 135532 135561 136955 136956 136957 136958 \
    136959 135220 135534 136960 136961 135465 136962 136963 135424 136964 \
    136965 135468 136966 136967 136968 136969 136969 136970 136971 136972 \
    136973 136974 136975 136976 136977 136978 136979 136980 136981 136982 \
    136983 136984 136985 136986 136987 136988 136989 136990 136991 136992 \
    136993 136994 136995 136996 135474 136997 136998 136999 137000 137001 \
    137002 137003 137004 137005 137006 137007 137008 137009 137010 137011 \
    137012 136798 137013 137014 137015 137016 137017 137018 137019 137020 \
    137021 137022 137023 137024 135274 137025 137026 137027 137028 137029 \
    137030 135477 137031 137032 137033 137034 137035 137036 137037 137038 \
    137039 137040 137041 137042 137043 137044 137045 137046 137047 137048 \
    137049 137050 135215 137051 137052 137053 137054 137055 137056 135541 \
    137057 137058 137059 137060 137061 137062 137063 137064 133702 137065 \
    137066 137067 137068 137069 137070 137071 137072 137073 137074 137075 \
    135546 135547 133709 137076 137077 137078 137079 137080 137081 137082 \
    137083 137084 137085 137086 137087 135548 137088 137089 137090 137091 \
    137092 137093 137094 137095 137096 137097 137098 137099 137100 137101 \
    137102 137103 137104 137105 137106 137107 137108 137109 137110 137111 \
    137112 137113 137114 137115 137116 137117 135554 135554 137118 137119 \
    137120 137121 137122 137123 137124 137125 137126 137127 135555 137128 \
    137129 137130 137131 137132 137133 137134 137135 137136 137137 133757 \
    137138 133761 137139 137140 137141 137142 133766 137143 -1 -1]

#
# List of decomposition sequences
#

variable decompList [list \
    32 32 776 97 32 772 50 51 32 769 956 32 807 49 111 49 8260 52 49 8260 \
    50 51 8260 52 65 768 65 769 65 770 65 771 65 776 65 778 67 807 69 768 \
    69 769 69 770 69 776 73 768 73 769 73 770 73 776 78 771 79 768 79 769 \
    79 770 79 771 79 776 85 768 85 769 85 770 85 776 89 769 97 768 97 769 \
    97 770 97 771 97 776 97 778 99 807 101 768 101 769 101 770 101 776 \
    105 768 105 769 105 770 105 776 110 771 111 768 111 769 111 770 111 \
    771 111 776 117 768 117 769 117 770 117 776 121 769 121 776 65 772 \
    97 772 65 774 97 774 65 808 97 808 67 769 99 769 67 770 99 770 67 775 \
    99 775 67 780 99 780 68 780 100 780 69 772 101 772 69 774 101 774 69 \
    775 101 775 69 808 101 808 69 780 101 780 71 770 103 770 71 774 103 \
    774 71 775 103 775 71 807 103 807 72 770 104 770 73 771 105 771 73 \
    772 105 772 73 774 105 774 73 808 105 808 73 775 73 74 105 106 74 770 \
    106 770 75 807 107 807 76 769 108 769 76 807 108 807 76 780 108 780 \
    76 183 108 183 78 769 110 769 78 807 110 807 78 780 110 780 700 110 \
    79 772 111 772 79 774 111 774 79 779 111 779 82 769 114 769 82 807 \
    114 807 82 780 114 780 83 769 115 769 83 770 115 770 83 807 115 807 \
    83 780 115 780 84 807 116 807 84 780 116 780 85 771 117 771 85 772 \
    117 772 85 774 117 774 85 778 117 778 85 779 117 779 85 808 117 808 \
    87 770 119 770 89 770 121 770 89 776 90 769 122 769 90 775 122 775 \
    90 780 122 780 115 79 795 111 795 85 795 117 795 68 381 68 382 100 \
    382 76 74 76 106 108 106 78 74 78 106 110 106 65 780 97 780 73 780 \
    105 780 79 780 111 780 85 780 117 780 220 772 252 772 220 769 252 769 \
    220 780 252 780 220 768 252 768 196 772 228 772 550 772 551 772 198 \
    772 230 772 71 780 103 780 75 780 107 780 79 808 111 808 490 772 491 \
    772 439 780 658 780 106 780 68 90 68 122 100 122 71 769 103 769 78 \
    768 110 768 197 769 229 769 198 769 230 769 216 769 248 769 65 783 \
    97 783 65 785 97 785 69 783 101 783 69 785 101 785 73 783 105 783 73 \
    785 105 785 79 783 111 783 79 785 111 785 82 783 114 783 82 785 114 \
    785 85 783 117 783 85 785 117 785 83 806 115 806 84 806 116 806 72 \
    780 104 780 65 775 97 775 69 807 101 807 214 772 246 772 213 772 245 \
    772 79 775 111 775 558 772 559 772 89 772 121 772 104 614 106 114 633 \
    635 641 119 121 32 774 32 775 32 778 32 808 32 771 32 779 611 108 120 \
    661 768 769 787 776 769 697 32 837 59 168 769 913 769 183 917 769 919 \
    769 921 769 927 769 933 769 937 769 970 769 921 776 933 776 945 769 \
    949 769 951 769 953 769 971 769 953 776 965 776 959 769 965 769 969 \
    769 946 952 933 978 769 978 776 966 960 954 961 962 920 949 931 1045 \
    768 1045 776 1043 769 1030 776 1050 769 1048 768 1059 774 1048 774 \
    1080 774 1077 768 1077 776 1075 769 1110 776 1082 769 1080 768 1091 \
    774 1140 783 1141 783 1046 774 1078 774 1040 774 1072 774 1040 776 \
    1072 776 1045 774 1077 774 1240 776 1241 776 1046 776 1078 776 1047 \
    776 1079 776 1048 772 1080 772 1048 776 1080 776 1054 776 1086 776 \
    1256 776 1257 776 1069 776 1101 776 1059 772 1091 772 1059 776 1091 \
    776 1059 779 1091 779 1063 776 1095 776 1067 776 1099 776 1381 1410 \
    1575 1619 1575 1620 1608 1620 1575 1621 1610 1620 1575 1652 1608 1652 \
    1735 1652 1610 1652 1749 1620 1729 1620 1746 1620 2344 2364 2352 2364 \
    2355 2364 2325 2364 2326 2364 2327 2364 2332 2364 2337 2364 2338 2364 \
    2347 2364 2351 2364 2503 2494 2503 2519 2465 2492 2466 2492 2479 2492 \
    2610 2620 2616 2620 2582 2620 2583 2620 2588 2620 2603 2620 2887 2902 \
    2887 2878 2887 2903 2849 2876 2850 2876 2962 3031 3014 3006 3015 3006 \
    3014 3031 3142 3158 3263 3285 3270 3285 3270 3286 3270 3266 3274 3285 \
    3398 3390 3399 3390 3398 3415 3545 3530 3545 3535 3548 3530 3545 3551 \
    3661 3634 3789 3762 3755 3737 3755 3745 3851 3906 4023 3916 4023 3921 \
    4023 3926 4023 3931 4023 3904 4021 3953 3954 3953 3956 4018 3968 4018 \
    3969 4019 3968 4019 3969 3953 3968 3986 4023 3996 4023 4001 4023 4006 \
    4023 4011 4023 3984 4021 4133 4142 4316 6917 6965 6919 6965 6921 6965 \
    6923 6965 6925 6965 6929 6965 6970 6965 6972 6965 6974 6965 6975 6965 \
    6978 6965 65 198 66 68 69 398 71 72 73 74 75 76 77 78 79 546 80 82 \
    84 85 87 592 593 7426 98 100 101 601 603 604 103 107 109 331 596 7446 \
    7447 112 116 117 7453 623 118 7461 947 948 967 105 1085 594 99 597 \
    240 102 607 609 613 616 617 618 7547 669 621 7557 671 625 624 626 627 \
    628 629 632 642 643 427 649 650 7452 651 652 122 656 657 658 65 805 \
    97 805 66 775 98 775 66 803 98 803 66 817 98 817 199 769 231 769 68 \
    775 100 775 68 803 100 803 68 817 100 817 68 807 100 807 68 813 100 \
    813 274 768 275 768 274 769 275 769 69 813 101 813 69 816 101 816 552 \
    774 553 774 70 775 102 775 71 772 103 772 72 775 104 775 72 803 104 \
    803 72 776 104 776 72 807 104 807 72 814 104 814 73 816 105 816 207 \
    769 239 769 75 769 107 769 75 803 107 803 75 817 107 817 76 803 108 \
    803 7734 772 7735 772 76 817 108 817 76 813 108 813 77 769 109 769 \
    77 775 109 775 77 803 109 803 78 775 110 775 78 803 110 803 78 817 \
    110 817 78 813 110 813 213 769 245 769 213 776 245 776 332 768 333 \
    768 332 769 333 769 80 769 112 769 80 775 112 775 82 775 114 775 82 \
    803 114 803 7770 772 7771 772 82 817 114 817 83 775 115 775 83 803 \
    115 803 346 775 347 775 352 775 353 775 7778 775 7779 775 84 775 116 \
    775 84 803 116 803 84 817 116 817 84 813 116 813 85 804 117 804 85 \
    816 117 816 85 813 117 813 360 769 361 769 362 776 363 776 86 771 118 \
    771 86 803 118 803 87 768 119 768 87 769 119 769 87 776 119 776 87 \
    775 119 775 87 803 119 803 88 775 120 775 88 776 120 776 89 775 121 \
    775 90 770 122 770 90 803 122 803 90 817 122 817 104 817 116 776 119 \
    778 121 778 97 702 383 775 65 803 97 803 65 777 97 777 194 769 226 \
    769 194 768 226 768 194 777 226 777 194 771 226 771 7840 770 7841 770 \
    258 769 259 769 258 768 259 768 258 777 259 777 258 771 259 771 7840 \
    774 7841 774 69 803 101 803 69 777 101 777 69 771 101 771 202 769 234 \
    769 202 768 234 768 202 777 234 777 202 771 234 771 7864 770 7865 770 \
    73 777 105 777 73 803 105 803 79 803 111 803 79 777 111 777 212 769 \
    244 769 212 768 244 768 212 777 244 777 212 771 244 771 7884 770 7885 \
    770 416 769 417 769 416 768 417 768 416 777 417 777 416 771 417 771 \
    416 803 417 803 85 803 117 803 85 777 117 777 431 769 432 769 431 768 \
    432 768 431 777 432 777 431 771 432 771 431 803 432 803 89 768 121 \
    768 89 803 121 803 89 777 121 777 89 771 121 771 945 787 945 788 7936 \
    768 7937 768 7936 769 7937 769 7936 834 7937 834 913 787 913 788 7944 \
    768 7945 768 7944 769 7945 769 7944 834 7945 834 949 787 949 788 7952 \
    768 7953 768 7952 769 7953 769 917 787 917 788 7960 768 7961 768 7960 \
    769 7961 769 951 787 951 788 7968 768 7969 768 7968 769 7969 769 7968 \
    834 7969 834 919 787 919 788 7976 768 7977 768 7976 769 7977 769 7976 \
    834 7977 834 953 787 953 788 7984 768 7985 768 7984 769 7985 769 7984 \
    834 7985 834 921 787 921 788 7992 768 7993 768 7992 769 7993 769 7992 \
    834 7993 834 959 787 959 788 8000 768 8001 768 8000 769 8001 769 927 \
    787 927 788 8008 768 8009 768 8008 769 8009 769 965 787 965 788 8016 \
    768 8017 768 8016 769 8017 769 8016 834 8017 834 933 788 8025 768 8025 \
    769 8025 834 969 787 969 788 8032 768 8033 768 8032 769 8033 769 8032 \
    834 8033 834 937 787 937 788 8040 768 8041 768 8040 769 8041 769 8040 \
    834 8041 834 945 768 940 949 768 941 951 768 942 953 768 943 959 768 \
    972 965 768 973 969 768 974 7936 837 7937 837 7938 837 7939 837 7940 \
    837 7941 837 7942 837 7943 837 7944 837 7945 837 7946 837 7947 837 \
    7948 837 7949 837 7950 837 7951 837 7968 837 7969 837 7970 837 7971 \
    837 7972 837 7973 837 7974 837 7975 837 7976 837 7977 837 7978 837 \
    7979 837 7980 837 7981 837 7982 837 7983 837 8032 837 8033 837 8034 \
    837 8035 837 8036 837 8037 837 8038 837 8039 837 8040 837 8041 837 \
    8042 837 8043 837 8044 837 8045 837 8046 837 8047 837 945 774 945 772 \
    8048 837 945 837 940 837 945 834 8118 837 913 774 913 772 913 768 902 \
    913 837 32 787 953 32 834 168 834 8052 837 951 837 942 837 951 834 \
    8134 837 917 768 904 919 768 905 919 837 8127 768 8127 769 8127 834 \
    953 774 953 772 970 768 912 953 834 970 834 921 774 921 772 921 768 \
    906 8190 768 8190 769 8190 834 965 774 965 772 971 768 944 961 787 \
    961 788 965 834 971 834 933 774 933 772 933 768 910 929 788 168 768 \
    901 96 8060 837 969 837 974 837 969 834 8182 837 927 768 908 937 768 \
    911 937 837 180 32 788 8194 8195 8208 32 819 46 46 46 46 46 46 8242 \
    8242 8242 8242 8242 8245 8245 8245 8245 8245 33 33 32 773 63 63 63 \
    33 33 63 8242 8242 8242 8242 48 52 53 54 55 56 57 43 8722 61 40 41 \
    110 82 115 97 47 99 97 47 115 67 176 67 99 47 111 99 47 117 400 176 \
    70 295 78 111 81 83 77 84 69 76 84 77 90 937 197 70 1488 1489 1490 \
    1491 70 65 88 915 928 8721 49 8260 51 50 8260 51 49 8260 53 50 8260 \
    53 51 8260 53 52 8260 53 49 8260 54 53 8260 54 49 8260 56 51 8260 56 \
    53 8260 56 55 8260 56 49 8260 73 73 73 73 73 73 86 86 86 73 86 73 73 \
    86 73 73 73 73 88 88 88 73 88 73 73 105 105 105 105 105 105 118 118 \
    105 118 105 105 118 105 105 105 105 120 120 105 120 105 105 8592 824 \
    8594 824 8596 824 8656 824 8660 824 8658 824 8707 824 8712 824 8715 \
    824 8739 824 8741 824 8747 8747 8747 8747 8747 8750 8750 8750 8750 \
    8750 8764 824 8771 824 8773 824 8776 824 61 824 8801 824 8781 824 60 \
    824 62 824 8804 824 8805 824 8818 824 8819 824 8822 824 8823 824 8826 \
    824 8827 824 8834 824 8835 824 8838 824 8839 824 8866 824 8872 824 \
    8873 824 8875 824 8828 824 8829 824 8849 824 8850 824 8882 824 8883 \
    824 8884 824 8885 824 12296 12297 49 48 49 49 49 50 49 51 49 52 49 \
    53 49 54 49 55 49 56 49 57 50 48 40 49 41 40 50 41 40 51 41 40 52 41 \
    40 53 41 40 54 41 40 55 41 40 56 41 40 57 41 40 49 48 41 40 49 49 41 \
    40 49 50 41 40 49 51 41 40 49 52 41 40 49 53 41 40 49 54 41 40 49 55 \
    41 40 49 56 41 40 49 57 41 40 50 48 41 49 46 50 46 51 46 52 46 53 46 \
    54 46 55 46 56 46 57 46 49 48 46 49 49 46 49 50 46 49 51 46 49 52 46 \
    49 53 46 49 54 46 49 55 46 49 56 46 49 57 46 50 48 46 40 97 41 40 98 \
    41 40 99 41 40 100 41 40 101 41 40 102 41 40 103 41 40 104 41 40 105 \
    41 40 106 41 40 107 41 40 108 41 40 109 41 40 110 41 40 111 41 40 112 \
    41 40 113 41 40 114 41 40 115 41 40 116 41 40 117 41 40 118 41 40 119 \
    41 40 120 41 40 121 41 40 122 41 83 89 113 8747 8747 8747 8747 58 58 \
    61 61 61 61 61 61 10973 824 11617 27597 40863 19968 20008 20022 20031 \
    20057 20101 20108 20128 20154 20799 20837 20843 20866 20886 20907 20960 \
    20981 20992 21147 21241 21269 21274 21304 21313 21340 21353 21378 21430 \
    21448 21475 22231 22303 22763 22786 22794 22805 22823 22899 23376 23424 \
    23544 23567 23586 23608 23662 23665 24027 24037 24049 24062 24178 24186 \
    24191 24308 24318 24331 24339 24400 24417 24435 24515 25096 25142 25163 \
    25903 25908 25991 26007 26020 26041 26080 26085 26352 26376 26408 27424 \
    27490 27513 27571 27595 27604 27611 27663 27668 27700 28779 29226 29238 \
    29243 29247 29255 29273 29275 29356 29572 29577 29916 29926 29976 29983 \
    29992 30000 30091 30098 30326 30333 30382 30399 30446 30683 30690 30707 \
    31034 31160 31166 31348 31435 31481 31859 31992 32566 32593 32650 32701 \
    32769 32780 32786 32819 32895 32905 33251 33258 33267 33276 33292 33307 \
    33311 33390 33394 33400 34381 34411 34880 34892 34915 35198 35211 35282 \
    35328 35895 35910 35925 35960 35997 36196 36208 36275 36523 36554 36763 \
    36784 36789 37009 37193 37318 37324 37329 38263 38272 38428 38582 38585 \
    38632 38737 38750 38754 38761 38859 38893 38899 38913 39080 39131 39135 \
    39318 39321 39340 39592 39640 39647 39717 39727 39730 39740 39770 40165 \
    40565 40575 40613 40635 40643 40653 40657 40697 40701 40718 40723 40736 \
    40763 40778 40786 40845 40860 40864 12306 21316 21317 12363 12441 12365 \
    12441 12367 12441 12369 12441 12371 12441 12373 12441 12375 12441 12377 \
    12441 12379 12441 12381 12441 12383 12441 12385 12441 12388 12441 12390 \
    12441 12392 12441 12399 12441 12399 12442 12402 12441 12402 12442 12405 \
    12441 12405 12442 12408 12441 12408 12442 12411 12441 12411 12442 12358 \
    12441 32 12441 32 12442 12445 12441 12424 12426 12459 12441 12461 12441 \
    12463 12441 12465 12441 12467 12441 12469 12441 12471 12441 12473 12441 \
    12475 12441 12477 12441 12479 12441 12481 12441 12484 12441 12486 12441 \
    12488 12441 12495 12441 12495 12442 12498 12441 12498 12442 12501 12441 \
    12501 12442 12504 12441 12504 12442 12507 12441 12507 12442 12454 12441 \
    12527 12441 12528 12441 12529 12441 12530 12441 12541 12441 12467 12488 \
    4352 4353 4522 4354 4524 4525 4355 4356 4357 4528 4529 4530 4531 4532 \
    4533 4378 4358 4359 4360 4385 4361 4362 4363 4364 4365 4366 4367 4368 \
    4369 4370 4449 4450 4451 4452 4453 4454 4455 4456 4457 4458 4459 4460 \
    4461 4462 4463 4464 4465 4466 4467 4468 4469 4448 4372 4373 4551 4552 \
    4556 4558 4563 4567 4569 4380 4573 4575 4381 4382 4384 4386 4387 4391 \
    4393 4395 4396 4397 4398 4399 4402 4406 4416 4423 4428 4593 4594 4439 \
    4440 4441 4484 4485 4488 4497 4498 4500 4510 4513 19977 22235 19978 \
    20013 19979 30002 19993 19969 22825 22320 40 4352 41 40 4354 41 40 \
    4355 41 40 4357 41 40 4358 41 40 4359 41 40 4361 41 40 4363 41 40 4364 \
    41 40 4366 41 40 4367 41 40 4368 41 40 4369 41 40 4370 41 40 4352 4449 \
    41 40 4354 4449 41 40 4355 4449 41 40 4357 4449 41 40 4358 4449 41 \
    40 4359 4449 41 40 4361 4449 41 40 4363 4449 41 40 4364 4449 41 40 \
    4366 4449 41 40 4367 4449 41 40 4368 4449 41 40 4369 4449 41 40 4370 \
    4449 41 40 4364 4462 41 40 4363 4457 4364 4453 4523 41 40 4363 4457 \
    4370 4462 41 40 19968 41 40 20108 41 40 19977 41 40 22235 41 40 20116 \
    41 40 20845 41 40 19971 41 40 20843 41 40 20061 41 40 21313 41 40 26376 \
    41 40 28779 41 40 27700 41 40 26408 41 40 37329 41 40 22303 41 40 26085 \
    41 40 26666 41 40 26377 41 40 31038 41 40 21517 41 40 29305 41 40 36001 \
    41 40 31069 41 40 21172 41 40 20195 41 40 21628 41 40 23398 41 40 30435 \
    41 40 20225 41 40 36039 41 40 21332 41 40 31085 41 40 20241 41 40 33258 \
    41 40 33267 41 80 84 69 50 49 50 50 50 51 50 52 50 53 50 54 50 55 50 \
    56 50 57 51 48 51 49 51 50 51 51 51 52 51 53 4352 4449 4354 4449 4355 \
    4449 4357 4449 4358 4449 4359 4449 4361 4449 4363 4449 4364 4449 4366 \
    4449 4367 4449 4368 4449 4369 4449 4370 4449 4366 4449 4535 4352 4457 \
    4364 4462 4363 4468 4363 4462 20116 20845 19971 20061 26666 26377 31038 \
    21517 29305 36001 31069 21172 31192 30007 36969 20778 21360 27880 38917 \
    20241 20889 27491 24038 21491 21307 23447 23398 30435 20225 36039 21332 \
    22812 51 54 51 55 51 56 51 57 52 48 52 49 52 50 52 51 52 52 52 53 52 \
    54 52 55 52 56 52 57 53 48 49 26376 50 26376 51 26376 52 26376 53 26376 \
    54 26376 55 26376 56 26376 57 26376 49 48 26376 49 49 26376 49 50 26376 \
    72 103 101 114 103 101 86 76 84 68 12450 12452 12454 12456 12458 12459 \
    12461 12463 12465 12467 12469 12471 12473 12475 12477 12479 12481 12484 \
    12486 12488 12490 12491 12492 12493 12494 12495 12498 12501 12504 12507 \
    12510 12511 12512 12513 12514 12516 12518 12520 12521 12522 12523 12524 \
    12525 12527 12528 12529 12530 12450 12497 12540 12488 12450 12523 12501 \
    12449 12450 12531 12506 12450 12450 12540 12523 12452 12491 12531 12464 \
    12452 12531 12481 12454 12457 12531 12456 12473 12463 12540 12489 12456 \
    12540 12459 12540 12458 12531 12473 12458 12540 12512 12459 12452 12522 \
    12459 12521 12483 12488 12459 12525 12522 12540 12460 12525 12531 12460 \
    12531 12510 12462 12460 12462 12491 12540 12461 12517 12522 12540 12462 \
    12523 12480 12540 12461 12525 12461 12525 12464 12521 12512 12461 12525 \
    12513 12540 12488 12523 12461 12525 12527 12483 12488 12464 12521 12512 \
    12464 12521 12512 12488 12531 12463 12523 12476 12452 12525 12463 12525 \
    12540 12493 12465 12540 12473 12467 12523 12490 12467 12540 12509 12469 \
    12452 12463 12523 12469 12531 12481 12540 12512 12471 12522 12531 12464 \
    12475 12531 12481 12475 12531 12488 12480 12540 12473 12487 12471 12489 \
    12523 12488 12531 12490 12494 12494 12483 12488 12495 12452 12484 12497 \
    12540 12475 12531 12488 12497 12540 12484 12496 12540 12524 12523 12500 \
    12450 12473 12488 12523 12500 12463 12523 12500 12467 12499 12523 12501 \
    12449 12521 12483 12489 12501 12451 12540 12488 12502 12483 12471 12455 \
    12523 12501 12521 12531 12504 12463 12479 12540 12523 12506 12477 12506 \
    12491 12498 12504 12523 12484 12506 12531 12473 12506 12540 12472 12505 \
    12540 12479 12509 12452 12531 12488 12508 12523 12488 12507 12531 12509 \
    12531 12489 12507 12540 12523 12507 12540 12531 12510 12452 12463 12525 \
    12510 12452 12523 12510 12483 12495 12510 12523 12463 12510 12531 12471 \
    12519 12531 12511 12463 12525 12531 12511 12522 12511 12522 12496 12540 \
    12523 12513 12460 12513 12460 12488 12531 12513 12540 12488 12523 12516 \
    12540 12489 12516 12540 12523 12518 12450 12531 12522 12483 12488 12523 \
    12522 12521 12523 12500 12540 12523 12540 12502 12523 12524 12512 12524 \
    12531 12488 12466 12531 12527 12483 12488 48 28857 49 28857 50 28857 \
    51 28857 52 28857 53 28857 54 28857 55 28857 56 28857 57 28857 49 48 \
    28857 49 49 28857 49 50 28857 49 51 28857 49 52 28857 49 53 28857 49 \
    54 28857 49 55 28857 49 56 28857 49 57 28857 50 48 28857 50 49 28857 \
    50 50 28857 50 51 28857 50 52 28857 104 80 97 100 97 65 85 98 97 114 \
    111 86 112 99 100 109 100 109 178 100 109 179 73 85 24179 25104 26157 \
    21644 22823 27491 26126 27835 26666 24335 20250 31038 112 65 110 65 \
    956 65 109 65 107 65 75 66 77 66 71 66 99 97 108 107 99 97 108 112 \
    70 110 70 956 70 956 103 109 103 107 103 72 122 107 72 122 77 72 122 \
    71 72 122 84 72 122 956 8467 109 8467 100 8467 107 8467 102 109 110 \
    109 956 109 109 109 99 109 107 109 109 109 178 99 109 178 109 178 107 \
    109 178 109 109 179 99 109 179 109 179 107 109 179 109 8725 115 109 \
    8725 115 178 80 97 107 80 97 77 80 97 71 80 97 114 97 100 114 97 100 \
    8725 115 114 97 100 8725 115 178 112 115 110 115 956 115 109 115 112 \
    86 110 86 956 86 109 86 107 86 77 86 112 87 110 87 956 87 109 87 107 \
    87 77 87 107 937 77 937 97 46 109 46 66 113 99 99 99 100 67 8725 107 \
    103 67 111 46 100 66 71 121 104 97 72 80 105 110 75 75 75 77 107 116 \
    108 109 108 110 108 111 103 108 120 109 98 109 105 108 109 111 108 \
    80 72 112 46 109 46 80 80 77 80 82 115 114 83 118 87 98 86 8725 109 \
    65 8725 109 49 26085 50 26085 51 26085 52 26085 53 26085 54 26085 55 \
    26085 56 26085 57 26085 49 48 26085 49 49 26085 49 50 26085 49 51 26085 \
    49 52 26085 49 53 26085 49 54 26085 49 55 26085 49 56 26085 49 57 26085 \
    50 48 26085 50 49 26085 50 50 26085 50 51 26085 50 52 26085 50 53 26085 \
    50 54 26085 50 55 26085 50 56 26085 50 57 26085 51 48 26085 51 49 26085 \
    103 97 108 35912 26356 36040 28369 20018 21477 22865 21895 22856 25078 \
    30313 32645 34367 34746 35064 37007 27138 27931 28889 29662 33853 37226 \
    39409 20098 21365 27396 29211 34349 40478 23888 28651 34253 35172 25289 \
    33240 34847 24266 26391 28010 29436 37070 20358 20919 21214 25796 27347 \
    29200 30439 34310 34396 36335 38706 39791 40442 30860 31103 32160 33737 \
    37636 35542 22751 24324 31840 32894 29282 30922 36034 38647 22744 23650 \
    27155 28122 28431 32047 32311 38475 21202 32907 20956 20940 31260 32190 \
    33777 38517 35712 25295 35582 20025 23527 24594 29575 30064 21271 30971 \
    20415 24489 19981 27852 25976 32034 21443 22622 30465 33865 35498 27578 \
    27784 25342 33509 25504 30053 20142 20841 20937 26753 31975 33391 35538 \
    37327 21237 21570 24300 26053 28670 31018 38317 39530 40599 40654 26310 \
    27511 36706 24180 24976 25088 25754 28451 29001 29833 31178 32244 32879 \
    36646 34030 36899 37706 21015 21155 21693 28872 35010 24265 24565 25467 \
    27566 31806 29557 20196 22265 23994 24604 29618 29801 32666 32838 37428 \
    38646 38728 38936 20363 31150 37300 38584 24801 20102 20698 23534 23615 \
    26009 29134 30274 34044 36988 26248 38446 21129 26491 26611 27969 28316 \
    29705 30041 30827 32016 39006 25134 38520 20523 23833 28138 36650 24459 \
    24900 26647 38534 21033 21519 23653 26131 26446 26792 27877 29702 30178 \
    32633 35023 35041 38626 21311 28346 21533 29136 29848 34298 38563 40023 \
    40607 26519 28107 33256 31520 31890 29376 28825 35672 20160 33590 21050 \
    20999 24230 25299 31958 23429 27934 26292 36667 38477 24275 20800 21952 \
    22618 26228 20958 29482 30410 31036 31070 31077 31119 38742 31934 34322 \
    35576 36920 37117 39151 39164 39208 40372 20398 20711 20813 21193 21220 \
    21329 21917 22022 22120 22592 22696 23652 24724 24936 24974 25074 25935 \
    26082 26257 26757 28023 28186 28450 29038 29227 29730 30865 31049 31048 \
    31056 31062 31117 31118 31296 31361 31680 32265 32321 32626 32773 33261 \
    33401 33879 35088 35222 35585 35641 36051 36104 36790 38627 38911 38971 \
    20006 20917 20840 20352 20805 20864 21191 21242 21845 21913 21986 22707 \
    22852 22868 23138 23336 24274 24281 24425 24493 24792 24910 24840 24928 \
    25140 25540 25628 25682 25942 26395 26454 28379 28363 28702 30631 29237 \
    29359 29809 29958 30011 30237 30239 30427 30452 30538 30528 30924 31409 \
    31867 32091 32574 33618 33775 34681 35137 35206 35519 35531 35565 35722 \
    36664 36978 37273 37494 38524 38875 38923 39698 141386 141380 144341 \
    15261 16408 16441 152137 154832 163539 40771 40846 102 102 102 105 \
    102 108 102 102 105 102 102 108 383 116 115 116 1396 1398 1396 1381 \
    1396 1387 1406 1398 1396 1389 1497 1460 1522 1463 1506 1492 1499 1500 \
    1501 1512 1514 1513 1473 1513 1474 64329 1473 64329 1474 1488 1463 \
    1488 1464 1488 1468 1489 1468 1490 1468 1491 1468 1492 1468 1493 1468 \
    1494 1468 1496 1468 1497 1468 1498 1468 1499 1468 1500 1468 1502 1468 \
    1504 1468 1505 1468 1507 1468 1508 1468 1510 1468 1511 1468 1512 1468 \
    1513 1468 1514 1468 1493 1465 1489 1471 1499 1471 1508 1471 1488 1500 \
    1649 1659 1662 1664 1658 1663 1657 1700 1702 1668 1667 1670 1671 1677 \
    1676 1678 1672 1688 1681 1705 1711 1715 1713 1722 1723 1728 1729 1726 \
    1746 1747 1709 1735 1734 1736 1655 1739 1733 1737 1744 1609 1574 1575 \
    1574 1749 1574 1608 1574 1735 1574 1734 1574 1736 1574 1744 1574 1609 \
    1740 1574 1580 1574 1581 1574 1605 1574 1610 1576 1580 1576 1581 1576 \
    1582 1576 1605 1576 1609 1576 1610 1578 1580 1578 1581 1578 1582 1578 \
    1605 1578 1609 1578 1610 1579 1580 1579 1605 1579 1609 1579 1610 1580 \
    1581 1580 1605 1581 1580 1581 1605 1582 1580 1582 1581 1582 1605 1587 \
    1580 1587 1581 1587 1582 1587 1605 1589 1581 1589 1605 1590 1580 1590 \
    1581 1590 1582 1590 1605 1591 1581 1591 1605 1592 1605 1593 1580 1593 \
    1605 1594 1580 1594 1605 1601 1580 1601 1581 1601 1582 1601 1605 1601 \
    1609 1601 1610 1602 1581 1602 1605 1602 1609 1602 1610 1603 1575 1603 \
    1580 1603 1581 1603 1582 1603 1604 1603 1605 1603 1609 1603 1610 1604 \
    1580 1604 1581 1604 1582 1604 1605 1604 1609 1604 1610 1605 1580 1605 \
    1581 1605 1582 1605 1605 1605 1609 1605 1610 1606 1580 1606 1581 1606 \
    1582 1606 1605 1606 1609 1606 1610 1607 1580 1607 1605 1607 1609 1607 \
    1610 1610 1580 1610 1581 1610 1582 1610 1605 1610 1609 1610 1610 1584 \
    1648 1585 1648 1609 1648 32 1612 1617 32 1613 1617 32 1614 1617 32 \
    1615 1617 32 1616 1617 32 1617 1648 1574 1585 1574 1586 1574 1606 1576 \
    1585 1576 1586 1576 1606 1578 1585 1578 1586 1578 1606 1579 1585 1579 \
    1586 1579 1606 1605 1575 1606 1585 1606 1586 1606 1606 1610 1585 1610 \
    1586 1610 1606 1574 1582 1574 1607 1576 1607 1578 1607 1589 1582 1604 \
    1607 1606 1607 1607 1648 1610 1607 1579 1607 1587 1607 1588 1605 1588 \
    1607 1600 1614 1617 1600 1615 1617 1600 1616 1617 1591 1609 1591 1610 \
    1593 1609 1593 1610 1594 1609 1594 1610 1587 1609 1587 1610 1588 1609 \
    1588 1610 1581 1609 1581 1610 1580 1609 1580 1610 1582 1609 1582 1610 \
    1589 1609 1589 1610 1590 1609 1590 1610 1588 1580 1588 1581 1588 1582 \
    1588 1585 1587 1585 1589 1585 1590 1585 1575 1611 1578 1580 1605 1578 \
    1581 1580 1578 1581 1605 1578 1582 1605 1578 1605 1580 1578 1605 1581 \
    1578 1605 1582 1580 1605 1581 1581 1605 1610 1581 1605 1609 1587 1581 \
    1580 1587 1580 1581 1587 1580 1609 1587 1605 1581 1587 1605 1580 1587 \
    1605 1605 1589 1581 1581 1589 1605 1605 1588 1581 1605 1588 1580 1610 \
    1588 1605 1582 1588 1605 1605 1590 1581 1609 1590 1582 1605 1591 1605 \
    1581 1591 1605 1605 1591 1605 1610 1593 1580 1605 1593 1605 1605 1593 \
    1605 1609 1594 1605 1605 1594 1605 1610 1594 1605 1609 1601 1582 1605 \
    1602 1605 1581 1602 1605 1605 1604 1581 1605 1604 1581 1610 1604 1581 \
    1609 1604 1580 1580 1604 1582 1605 1604 1605 1581 1605 1581 1580 1605 \
    1581 1605 1605 1581 1610 1605 1580 1581 1605 1580 1605 1605 1582 1580 \
    1605 1582 1605 1605 1580 1582 1607 1605 1580 1607 1605 1605 1606 1581 \
    1605 1606 1581 1609 1606 1580 1605 1606 1580 1609 1606 1605 1610 1606 \
    1605 1609 1610 1605 1605 1576 1582 1610 1578 1580 1610 1578 1580 1609 \
    1578 1582 1610 1578 1582 1609 1578 1605 1610 1578 1605 1609 1580 1605 \
    1610 1580 1581 1609 1580 1605 1609 1587 1582 1609 1589 1581 1610 1588 \
    1581 1610 1590 1581 1610 1604 1580 1610 1604 1605 1610 1610 1581 1610 \
    1610 1580 1610 1610 1605 1610 1605 1605 1610 1602 1605 1610 1606 1581 \
    1610 1593 1605 1610 1603 1605 1610 1606 1580 1581 1605 1582 1610 1604 \
    1580 1605 1603 1605 1605 1580 1581 1610 1581 1580 1610 1605 1580 1610 \
    1601 1605 1610 1576 1581 1610 1587 1582 1610 1606 1580 1610 1589 1604 \
    1746 1602 1604 1746 1575 1604 1604 1607 1575 1603 1576 1585 1605 1581 \
    1605 1583 1589 1604 1593 1605 1585 1587 1608 1604 1593 1604 1610 1607 \
    1608 1587 1604 1605 1589 1604 1609 1589 1604 1609 32 1575 1604 1604 \
    1607 32 1593 1604 1610 1607 32 1608 1587 1604 1605 1580 1604 32 1580 \
    1604 1575 1604 1607 1585 1740 1575 1604 44 12289 12290 58 33 63 12310 \
    12311 8230 8229 8212 8211 95 123 125 12308 12309 12304 12305 12298 \
    12299 12300 12301 12302 12303 91 93 8254 35 38 42 45 60 62 92 36 37 \
    64 32 1611 1600 1611 32 1612 32 1613 32 1614 1600 1614 32 1615 1600 \
    1615 32 1616 1600 1616 32 1617 1600 1617 32 1618 1600 1618 1569 1570 \
    1571 1572 1573 1574 1575 1576 1577 1578 1579 1580 1581 1582 1583 1584 \
    1585 1586 1587 1588 1589 1590 1591 1592 1593 1594 1601 1602 1603 1604 \
    1605 1606 1607 1608 1610 1604 1570 1604 1571 1604 1573 1604 1575 34 \
    39 47 94 124 126 10629 10630 12539 12449 12451 12453 12455 12457 12515 \
    12517 12519 12483 12540 12531 12441 12442 12644 12593 12594 12595 12596 \
    12597 12598 12599 12600 12601 12602 12603 12604 12605 12606 12607 12608 \
    12609 12610 12611 12612 12613 12614 12615 12616 12617 12618 12619 12620 \
    12621 12622 12623 12624 12625 12626 12627 12628 12629 12630 12631 12632 \
    12633 12634 12635 12636 12637 12638 12639 12640 12641 12642 12643 162 \
    163 172 175 166 165 8361 9474 8592 8593 8594 8595 9632 9675 119127 \
    119141 119128 119141 119135 119150 119135 119151 119135 119152 119135 \
    119153 119135 119154 119225 119141 119226 119141 119227 119150 119228 \
    119150 119227 119151 119228 119151 305 567 913 914 916 917 918 919 \
    921 922 923 924 925 926 927 929 1012 932 934 935 936 8711 945 950 951 \
    955 957 958 959 963 964 965 968 969 8706 1013 977 1008 981 1009 982 \
    988 989 20029 20024 20033 131362 20320 20411 20482 20602 20633 20687 \
    13470 132666 20820 20836 20855 132380 13497 20839 20877 132427 20887 \
    20900 20172 20908 168415 20995 13535 21051 21062 21106 21111 13589 \
    21253 21254 21321 21338 21363 21373 21375 133676 28784 21450 21471 \
    133987 21483 21489 21510 21662 21560 21576 21608 21666 21750 21776 \
    21843 21859 21892 21931 21939 21954 22294 22295 22097 22132 22766 22478 \
    22516 22541 22411 22578 22577 22700 136420 22770 22775 22790 22810 \
    22818 22882 136872 136938 23020 23067 23079 23000 23142 14062 14076 \
    23304 23358 137672 23491 23512 23539 138008 23551 23558 24403 14209 \
    23648 23744 23693 138724 23875 138726 23918 23915 23932 24033 24034 \
    14383 24061 24104 24125 24169 14434 139651 14460 24240 24243 24246 \
    172946 140081 33281 24354 14535 144056 156122 24418 24427 14563 24474 \
    24525 24535 24569 24705 14650 14620 141012 24775 24904 24908 24954 \
    25010 24996 25007 25054 25104 25115 25181 25265 25300 25424 142092 \
    25405 25340 25448 25475 25572 142321 25634 25541 25513 14894 25705 \
    25726 25757 25719 14956 25964 143370 26083 26360 26185 15129 15112 \
    15076 20882 20885 26368 26268 32941 17369 26401 26462 26451 144323 \
    15177 26618 26501 26706 144493 26766 26655 26900 26946 27043 27114 \
    27304 145059 27355 15384 27425 145575 27476 15438 27506 27551 27579 \
    146061 138507 146170 27726 146620 27839 27853 27751 27926 27966 28009 \
    28024 28037 146718 27956 28207 28270 15667 28359 147153 28153 28526 \
    147294 147342 28614 28729 28699 15766 28746 28797 28791 28845 132389 \
    28997 148067 29084 148395 29224 29264 149000 29312 29333 149301 149524 \
    29562 29579 16044 29605 16056 29767 29788 29829 29898 16155 29988 150582 \
    30014 150674 139679 30224 151457 151480 151620 16380 16392 151795 151794 \
    151833 151859 30494 30495 30603 16454 16534 152605 30798 16611 153126 \
    153242 153285 31211 16687 31306 31311 153980 154279 31470 16898 154539 \
    31686 31689 16935 154752 31954 17056 31976 31971 32000 155526 32099 \
    17153 32199 32258 32325 17204 156200 156231 17241 156377 32634 156478 \
    32661 32762 156890 156963 32864 157096 32880 144223 17365 32946 33027 \
    17419 33086 23221 157607 157621 144275 144284 33284 36766 17515 33425 \
    33419 33437 21171 33457 33459 33469 33510 158524 33565 33635 33709 \
    33571 33725 33767 33619 33738 33740 33756 158774 159083 158933 17707 \
    34033 34035 34070 160714 34148 159532 17757 17761 159665 159954 17771 \
    34384 34407 34409 34473 34440 34574 34530 34600 34667 34694 17879 34785 \
    34817 17913 34912 161383 35031 35038 17973 35066 13499 161966 162150 \
    18110 18119 35488 162984 36011 36033 36123 36215 163631 133124 36299 \
    36284 36336 133342 36564 165330 165357 37012 37105 37137 165678 37147 \
    37432 37591 37592 37500 37881 37909 166906 38283 18837 38327 167287 \
    18918 38595 23986 38691 168261 168474 19054 19062 38880 168970 19122 \
    169110 38953 169398 39138 19251 39209 39335 39362 39422 19406 170800 \
    40000 40189 19662 19693 40295 172238 19704 172293 172558 172689 19798 \
    40702 40709 40719 40726 173568]

variable DECOMP_COMPAT_MASK 65536
variable DECOMP_INFO_BITS 17

#
# This macro extracts the information about a character from the
# Unicode character tables.
#

proc GetUniCharDecompCompatInfo {uc} {
    variable DECOMP_OFFSET_BITS
    variable DECOMP_COMMON_PAGE_MAP
    variable decompPageMap
    variable decompGroupMap

    set page [expr {($uc & 0x1fffff) >> $DECOMP_OFFSET_BITS}]
    if {[info exists decompPageMap($page)]} {
	set apage $decompPageMap($page)
    } else {
	set apage $DECOMP_COMMON_PAGE_MAP
    }

    lindex $decompGroupMap \
	   [expr {($apage << $DECOMP_OFFSET_BITS) | \
		   ($uc & ((1 << $DECOMP_OFFSET_BITS) - 1))}]
}

proc GetUniCharDecompInfo {uc} {
    variable DECOMP_COMPAT_MASK

    set info [GetUniCharDecompCompatInfo $uc]
    if {$info & $DECOMP_COMPAT_MASK} {
	return -1
    } else {
	return $info
    }
}

proc GetDecompList {info} {
    variable DECOMP_INFO_BITS
    variable decompList

    set decomp_len [expr {$info >> $DECOMP_INFO_BITS}]
    set decomp_shift [expr {$info & ((1 << ($DECOMP_INFO_BITS - 1)) - 1)}]

    lrange $decompList $decomp_shift [expr {$decomp_shift + $decomp_len - 1}]
}

variable COMP_OFFSET_BITS 1

#
# The pageMap is indexed by page number and returns an alternate page number
# that identifies a unique page of characters.  Many Unicode characters map
# to the same alternate page number.
#

variable compPageMap
array unset compPageMap
array set compPageMap [list \
    30 1 31 2 32 3 33 4 34 5 35 6 36 7 37 8 38 9 39 10 40 11 41 12 42 13 \
    43 14 44 15 45 16 48 17 49 18 50 19 51 20 52 21 53 22 54 23 55 24 56 25 \
    57 26 58 27 59 28 60 29 61 30 84 31 97 32 98 33 99 34 101 35 103 36 \
    106 37 107 38 108 39 110 40 113 41 114 42 115 43 117 44 119 45 122 46 \
    123 47 124 48 126 49 129 50 137 51 166 52 173 53 176 54 180 55 181 56 \
    191 57 208 58 215 59 216 60 219 61 245 62 275 63 276 64 279 65 329 66 \
    384 67 385 68 386 69 387 70 388 71 389 72 390 73 391 74 392 75 393 76 \
    394 77 397 78 401 79 402 80 403 81 404 82 406 83 407 84 408 85 417 86 \
    418 87 456 88 458 89 459 90 460 91 463 92 464 93 466 94 468 95 470 96 \
    471 97 472 98 474 99 475 100 476 101 479 102 480 103 482 104 484 105 \
    485 106 487 107 489 108 515 109 520 110 521 111 522 112 523 113 524 114 \
    525 115 527 116 529 117 531 118 533 119 534 120 536 121 537 122 538 123 \
    539 124 540 125 541 126 543 127 545 128 547 129 549 130 550 131 555 132 \
    570 133 620 134 628 135 787 136 804 137 805 138 809 139 810 140 864 141 \
    873 142 874 143 1172 144 1176 145 1177 146 1247 147 1259 148 1439 149 \
    1451 150 1481 151 1503 152 1507 153 1515 154 1571 155 1631 156 1633 157 \
    1635 158 1637 159 1642 160 1643 161 1695 162 1699 163 1707 164 1765 165 \
    1767 166 1772 167 1774 168 1775 169 2066 170 3458 171 3459 172 3460 173 \
    3461 174 3462 175 3464 176 3485 177 3486 178 3487 179 3489 180 3867 181 \
    3885 182 3889 183 3920 184 3932 185 3942 186 3968 187 3969 188 3970 189 \
    3971 190 3972 191 3973 192 3974 193 3975 194 3976 195 3980 196 3984 197 \
    3985 198 3986 199 3987 200 3988 201 3989 202 3990 203 3991 204 3992 205 \
    3996 206 4000 207 4004 208 4008 209 4012 210 4016 211 4017 212 4018 213 \
    4019 214 4020 215 4021 216 4022 217 4023 218 4024 219 4026 220 4030 221 \
    4059 222 4063 223 4067 224 4091 225 4095 226 4296 227 4297 228 4298 229 \
    4328 230 4329 231 4330 232 4353 233 4356 234 4357 235 4369 236 4370 237 \
    4382 238 4385 239 4386 240 4388 241 4390 242 4400 243 4402 244 4409 245 \
    4411 246 4413 247 4414 248 4417 249 4419 250 4424 251 4425 252 4433 253 \
    4436 254 4437 255 4441 256 4442 257 6179 258 6181 259 6182 260 6183 261 \
    6184 262 6185 263 6186 264 6187 265 6188 266 6189 267 6190 268 6191 269 \
    6192 270 6194 271 6195 272 6196 273 6199 274 6201 275 6202 276 6204 277 \
    6205 278 6220 279 6221 280 6222 281 6227 282 6229 283 6230 284 6231 285 \
    6232 286 6233 287 6234 288 6235 289 6236 290 6237 291 6238 292 6239 293 \
    6240 294 6242 295 6243 296 6244 297 6247 298 6249 299 6250 300 6252 301 \
    6253 302 6263 303 6264 304 6265 305 6270 306]

variable COMP_COMMON_PAGE_MAP 0

#
# The groupMap is indexed by combining the alternate page number with
# the page offset and returns a group number that identifies a unique
# set of character attributes.
#

variable compGroupMap [list \
    -1 -1 65568 65648 65749 -1 -1 30 91 141 65 121 65712 38 94 1 65611 \
    124 44 100 12 76 77 -1 48 105 17 84 136 54 113 23 24 -1 -1 140 64 120 \
    71 123 65577 99 10 75 129 47 104 16 15 83 135 -1 110 22 86 137 59 117 \
    118 28 89 -1 102 -1 6 -1 65724 65579 101 65551 42 -1 -1 65580 130 50 \
    65686 -1 65636 -1 115 -1 32 -1 65753 65607 67 65715 5 -1 -1 65550 107 \
    20 65634 -1 65591 -1 87 -1 142 66 106 18 37 93 65632 65733 65640 65741 \
    65604 65709 65571 65657 -1 65739 81 131 -1 119 33 -1 -1 65546 65597 \
    65705 65683 65555 65633 65735 65742 65594 65740 -1 131073 131074 131075 \
    131077 131079 -1 131080 131082 131083 131084 131098 131102 131085 -1 \
    -1 131086 -1 131087 -1 131076 131078 -1 -1 131103 -1 131088 131104 \
    131101 131097 131092 131089 -1 -1 131094 131093 -1 131091 131095 131090 \
    -1 -1 131072 -1 34 -1 122 -1 96 -1 8 -1 80 -1 65590 -1 85 -1 26 65538 \
    -1 65718 -1 -1 2 -1 95 -1 7 -1 125 -1 51 -1 111 -1 56 -1 0 138 62 65672 \
    -1 39 -1 65747 -1 97 -1 -1 65725 -1 103 14 65630 132 -1 65585 -1 65738 \
    -1 -1 139 -1 65570 -1 65717 -1 65671 72 -1 -1 65678 -1 79 133 65586 \
    52 -1 65558 -1 65696 -1 -1 63 -1 65755 -1 65667 -1 65601 65666 -1 65669 \
    65545 65659 65542 65691 65563 -1 143 65744 -1 65665 -1 -1 196618 131108 \
    196619 -1 65626 65619 -1 -1 65618 65745 -1 65664 -1 -1 65663 196611 \
    -1 -1 196612 196613 -1 196614 196615 65581 -1 131105 -1 98 65588 -1 \
    131106 65582 -1 -1 65727 196616 -1 127 -1 65638 -1 -1 131107 196617 \
    -1 131096 -1 19 65704 -1 196610 131081 -1 -1 196608 -1 3 65539 -1 -1 \
    196609 -1 65624 -1 65710 -1 65562 -1 65708 -1 65559 -1 65621 -1 65556 \
    65547 -1 65602 -1 65687 65600 65599 -1 65688 65560 65694 65565 65754 \
    65608 114 25 65681 65554 65540 65612 82 134 65595 65701 65702 65569 \
    65649 65750 58 116 65573 65661 65544 65616 65617 65721 36 92 9 73 53 \
    108 65699 65567 65647 65748 65605 65711 27 88 65541 65613 65614 65720 \
    65576 65675 11 69 126 45 13 78 55 112 29 90 -1 70 128 46 65692 65564 \
    65643 65743 65596 65703 109 21 65650 65751 65606 65713 65572 65660 \
    65543 -1 65578 -1 65553 -1 65537 -1 -1 40 65689 -1 65574 -1 4 -1 65716 \
    -1 65668 -1 65622 -1 65707 -1 65652 -1 65752 -1 -1 65722 65627 -1 -1 \
    65684 -1 65676 -1 65549 65656 -1 -1 65548 -1 65726 65552 -1 -1 65690 \
    -1 65603 65655 65536 65732 65587 65561 65639 65697 65566 65645 65746 \
    65654 65756 65575 65670 -1 65584 65583 -1 65609 -1 65620 65723 -1 65677 \
    65557 65637 65736 65592 65729 -1 -1 65719 -1 65714 -1 65706 -1 65695 \
    -1 65693 -1 65685 -1 65679 -1 65673 -1 65658 -1 65653 -1 65646 -1 65641 \
    65631 -1 65628 -1 65623 -1 -1 60 74 -1 -1 68 61 -1 -1 57 -1 131099 \
    131100 -1 -1 65728 65700 -1 -1 65682 -1 65680 -1 65674 -1 65662 -1 \
    65651 -1 65644 -1 65642 -1 65635 -1 65629 -1 65625 -1 65615 -1 65610 \
    65598 -1 65593 -1 65589 -1 -1 41 31 -1 -1 49 43 -1 -1 35 -1 65737 65734 \
    65731 65730 -1 -1 65698]

#
# Lists of compositions for characters that appears only in one composition
#

variable compFirstList [list \
    {824 8817} {837 8119} {837 8116} {3530 3549} {770 7896} {837 8090} \
    {776 1243} {837 8114} {837 8076} {783 1143} {780 494} {6965 6971} \
    {824 8772} {824 8742} {769 7727} {769 7688} {824 8777} {837 8178} \
    {770 7879} {772 481} {6965 6930} {824 8938} {769 1116} {6965 6924} \
    {772 7737} {824 8824} {6965 6920} {776 1259} {837 8099} {772 7773} \
    {824 8833} {837 8083} {824 8814} {837 8069} {776 1268} {776 7802} \
    {837 8110} {837 8074} {837 8183} {824 8840} {837 8094} {775 7711} \
    {837 8130} {769 506} {769 7726} {3031 2964} {3158 3144} {824 8931} \
    {824 8930} {769 1036} {776 1247} {824 8821} {3006 3019} {12441 12489} \
    {788 8172} {769 511} {824 8941} {12441 12487} {772 561} {837 8066} \
    {837 8102} {772 492} {12441 12485} {6965 6979} {6965 6977} \
    {776 1261} {6965 6973} {824 8802} {769 7800} {837 8086} {837 8108} \
    {769 507} {775 7785} {824 8876} {12441 12482} {770 308} {770 7897} \
    {837 8091} {837 8092} {12441 12480} {837 8077} {837 8078} \
    {1620 1728} {1620 1747} {824 8877} {6965 6926} {824 8622} \
    {12441 12393} {4142 4134} {12441 12478} {1620 1730} {824 8713} \
    {12441 12391} {12441 12476} {776 1246} {12441 12389} {775 7780} \
    {774 7708} {772 555} {12441 12474} {769 510} {824 8939} {3285 3275} \
    {824 8825} {775 7782} {12441 12386} {12441 12472} {837 8100} \
    {12441 12470} {824 8928} {12441 12384} {837 8084} {824 8800} \
    {837 8070} {837 8106} {12441 12468} {824 8655} {12441 12382} \
    {824 8836} {824 8816} {824 8769} {776 7803} {12441 12380} \
    {776 1242} {837 8111} {837 8075} {12441 12466} {2364 2356} \
    {2364 2353} {1620 1574} {776 1111} {776 1273} {824 8603} \
    {783 1142} {824 8841} {776 1260} {837 8180} {12441 12378} \
    {12441 12464} {837 8095} {824 8740} {824 8879} {769 1107} \
    {12441 12376} {12441 12462} {770 7878} {12441 12460} {772 480} \
    {824 8716} {12441 12374} {772 554} {6965 6976} {772 7736} \
    {837 8135} {824 8813} {776 1258} {837 8098} {12441 12372} \
    {772 7772} {12441 12370} {776 1255} {824 8832} {12441 12542} \
    {837 8082} {12441 12532} {837 8067} {837 8068} {837 8103} \
    {3390 3403} {772 493} {12441 12368} {824 8653} {6965 6922} \
    {769 7801} {6965 6918} {837 8087} {775 7710} {837 8109} {12441 12366} \
    {769 7689} {824 8602} {776 1272} {837 8132} {12441 12364} \
    {837 8093} {837 8079} {824 8708} {824 8878} {772 478} {769 1027} \
    {824 8775} {3285 3264} {12441 12446} {12441 12436} {12441 12538} \
    {12441 12537} {824 8820} {775 7781} {12441 12536} {774 7709} \
    {824 8940} {12441 12535} {776 1254} {775 7835} {780 495} \
    {775 7783} {772 560} {837 8101} {1620 1572} {2364 2345} {824 8929} \
    {776 1031} {837 8085} {824 8815} {837 8071} {837 8107} {824 8654} \
    {772 479} {775 7784} {776 1269} {824 8837}]

variable compSecondList [list \
    {3545 3548} {3545 3550} {3398 3404} {2503 2507} {2503 2508} \
    {2887 2891} {2887 2888} {2887 2892} {3270 3274} {3270 3272} \
    {1575 1570} {1575 1573}]

#
# Compositions matrix
#

variable compBothMap
array unset compBothMap
array set compBothMap [list \
    0 8179 1 8060 2 974 4 8032 6 8033 18 8182 145 204 146 205 147 206 149 296 \
    151 298 152 300 154 304 155 207 156 7880 157 463 158 520 159 522 160 7882 \
    161 302 163 7724 288 8115 289 8048 290 940 292 7936 294 7937 295 8113 \
    296 8112 306 8118 441 3546 577 8157 578 8158 594 8159 721 7873 722 7871 \
    725 7877 732 7875 865 7846 866 7844 869 7850 876 7848 1008 8131 1009 8052 \
    1010 942 1012 7968 1014 7969 1026 8134 1153 8154 1154 906 1156 7992 \
    1158 7993 1159 8153 1160 8152 1163 938 1297 7962 1298 7964 1443 293 \
    1450 7715 1451 7719 1453 543 1456 7717 1460 7721 1461 7723 1463 7830 \
    1585 7986 1586 7988 1602 7990 1729 504 1730 323 1733 209 1738 7748 1741 327 \
    1744 7750 1748 325 1750 7754 1751 7752 1873 8002 1874 8004 2024 1217 \
    2027 1244 2161 505 2162 324 2165 241 2170 7749 2173 328 2176 7751 2180 326 \
    2182 7755 2183 7753 2306 7743 2314 7745 2320 7747 2458 7786 2461 356 \
    2464 7788 2468 354 2470 7792 2471 7790 2473 538 2593 7701 2594 7703 \
    2760 3402 2882 7757 2887 557 2891 7759 3024 8105 3025 8043 3026 8045 \
    3042 8047 3170 347 3171 349 3178 7777 3181 353 3184 7779 3188 351 3193 537 \
    3313 7922 3314 221 3315 374 3317 7928 3319 562 3322 7822 3323 376 3324 7926 \
    3328 7924 3458 377 3459 7824 3466 379 3469 381 3472 7826 3479 7828 3603 7853 \
    3608 7863 3744 8188 3745 8186 3746 911 3748 8040 3750 8041 3888 8088 \
    3889 7978 3890 7980 3906 7982 4033 7923 4034 253 4035 375 4037 7929 \
    4039 563 4042 7823 4043 255 4044 7927 4048 7925 4058 7833 4177 8018 \
    4178 8020 4194 8022 4321 192 4322 193 4323 194 4325 195 4327 256 4328 258 \
    4330 550 4331 196 4332 7842 4333 461 4334 512 4335 514 4336 7840 4337 260 \
    4346 197 4349 7680 4491 12499 4492 12500 4609 7847 4610 7845 4613 7851 \
    4620 7849 4753 7915 4754 7913 4757 7919 4764 7917 4768 7921 4896 8124 \
    4897 8122 4898 902 4900 7944 4902 7945 4903 8121 4904 8120 5067 12508 \
    5068 12509 5185 7954 5186 7956 5329 7760 5330 7762 5474 500 5475 284 \
    5479 7712 5480 286 5482 288 5485 486 5492 290 5618 979 5627 980 5761 8141 \
    5762 8142 5778 8143 5931 12496 5932 12497 6049 7872 6050 7870 6053 7876 \
    6060 7874 6219 12505 6220 12506 6338 313 6349 317 6352 7734 6356 315 \
    6358 7740 6359 7738 6481 7995 6482 7997 6498 7999 6624 8097 6625 8035 \
    6626 8037 6642 8039 6770 7729 6781 489 6784 7731 6788 311 6791 7733 \
    6914 340 6922 7768 6925 344 6926 528 6927 530 6928 7770 6932 342 6935 7774 \
    7083 12502 7084 12503 7202 7756 7207 556 7211 7758 7345 8056 7346 972 \
    7348 8000 7350 8001 7489 1117 7495 1251 7496 1081 7499 1253 7632 8080 \
    7633 7970 7634 7972 7650 7974 7777 7808 7778 7810 7779 372 7786 7814 \
    7787 7812 7792 7816 7921 8010 7922 8012 8065 8058 8066 973 8068 8016 \
    8070 8017 8071 8161 8072 8160 8075 971 8082 8166 8235 12412 8236 12413 \
    8352 8072 8353 7946 8354 7948 8370 7950 8501 7805 8512 7807 8667 12400 \
    8668 12401 8811 12409 8812 12410 8929 8162 8930 944 8946 8167 9079 1263 \
    9080 1118 9083 1265 9102 1267 9226 7683 9232 7685 9239 7687 9370 7690 \
    9373 270 9376 7692 9380 7696 9382 7698 9383 7694 9505 7857 9506 7855 \
    9509 7861 9516 7859 9650 509 9655 483 9819 12406 9820 12407 9937 7987 \
    9938 7989 9954 7991 10081 8027 10082 8029 10098 8031 10234 7691 10237 271 \
    10240 7693 10244 7697 10246 7699 10247 7695 10376 1233 10379 1235 10513 7963 \
    10514 7965 10683 12403 10684 12404 10801 236 10802 237 10803 238 10805 297 \
    10807 299 10808 301 10811 239 10812 7881 10813 464 10814 521 10815 523 \
    10816 7883 10817 303 10819 7725 10945 210 10946 211 10947 212 10949 213 \
    10951 332 10952 334 10954 558 10955 214 10956 7886 10957 465 10958 524 \
    10959 526 10960 7884 10961 490 10974 336 10975 416 11090 7764 11098 7766 \
    11233 8003 11234 8005 11377 1104 11384 1239 11387 1105 11521 8184 11522 908 \
    11524 8008 11526 8009 11665 7900 11666 7898 11669 7904 11676 7902 11680 7906 \
    11808 8064 11809 7938 11810 7940 11826 7942 11953 242 11954 243 11955 244 \
    11957 245 11959 333 11960 335 11962 559 11963 246 11964 7887 11965 466 \
    11966 525 11967 527 11968 7885 11969 491 11982 337 11983 417 12097 217 \
    12098 218 12099 219 12101 360 12103 362 12104 364 12107 220 12108 7910 \
    12109 467 12110 532 12111 534 12112 7908 12113 370 12115 7796 12118 7798 \
    12122 366 12126 368 12127 431 12128 7794 12241 8170 12242 910 12246 8025 \
    12247 8169 12248 8168 12251 939 12394 7787 12395 7831 12397 357 12400 7789 \
    12404 355 12406 7793 12407 7791 12409 539 12529 476 12530 472 12535 470 \
    12541 474 12672 8089 12673 7979 12674 7981 12690 7983 12818 378 12819 7825 \
    12826 380 12829 382 12832 7827 12839 7829 12961 8019 12962 8021 12978 8023 \
    13114 7682 13120 7684 13127 7686 13249 7955 13250 7957 13393 7761 13394 7763 \
    13539 292 13546 7714 13547 7718 13549 542 13552 7716 13556 7720 13557 7722 \
    13681 8050 13682 941 13684 7952 13686 7953 13824 8140 13825 8138 13826 905 \
    13828 7976 13830 7977 13976 1232 13979 1234 14145 3018 14146 3020 14258 501 \
    14259 285 14263 7713 14264 287 14266 289 14269 487 14276 291 14402 7742 \
    14410 7744 14416 7746 14546 508 14551 482 14689 8173 14690 901 14706 8129 \
    14833 1024 14840 1238 14843 1025 14978 314 14989 318 14992 7735 14996 316 \
    14998 7741 14999 7739 15122 346 15123 348 15130 7776 15133 352 15136 7778 \
    15140 350 15145 536 15265 7700 15266 7702 15409 7891 15410 7889 15413 7895 \
    15420 7893 15552 8081 15553 7971 15554 7973 15570 7975 15696 8104 15697 8042 \
    15698 8044 15714 8046 15842 341 15850 7769 15853 345 15854 529 15855 531 \
    15856 7771 15860 343 15863 7775 15988 8164 15990 8165 16129 8011 16130 8013 \
    16282 7818 16283 7820 16419 7852 16424 7862 16561 475 16562 471 16567 469 \
    16573 473 16704 8073 16705 7947 16706 7949 16722 7951 16849 7809 16850 7811 \
    16851 373 16858 7815 16859 7813 16864 7817 16874 7832 17002 7819 17003 7821 \
    17137 7914 17138 7912 17141 7918 17148 7916 17152 7920 17282 263 17283 265 \
    17290 267 17293 269 17300 231 17425 200 17426 201 17427 202 17429 7868 \
    17431 274 17432 276 17434 278 17435 203 17436 7866 17437 282 17438 516 \
    17439 518 17440 7864 17441 280 17443 7706 17444 552 17446 7704 17569 8136 \
    17570 904 17572 7960 17574 7961 17713 232 17714 233 17715 234 17717 7869 \
    17719 275 17720 277 17722 279 17723 235 17724 7867 17725 283 17726 517 \
    17727 519 17728 7865 17729 281 17731 7707 17732 553 17734 7705 17858 7728 \
    17869 488 17872 7730 17876 310 17879 7732 18001 8054 18002 943 18004 7984 \
    18006 7985 18007 8145 18008 8144 18011 970 18018 8150 18145 7994 18146 7996 \
    18162 7998 18323 3271 18432 8096 18433 8034 18434 8036 18450 8038 18579 309 \
    18589 496 18721 7890 18722 7888 18725 7894 18732 7892 18865 7901 18866 7899 \
    18869 7905 18876 7903 18880 7907 19009 1037 19015 1250 19016 1049 19019 1252 \
    19160 1218 19163 1245 19296 8065 19297 7939 19298 7941 19314 7943 19442 7765 \
    19450 7767 19589 7804 19600 7806 19729 249 19730 250 19731 251 19733 361 \
    19735 363 19736 365 19739 252 19740 7911 19741 468 19742 533 19743 535 \
    19744 7909 19745 371 19747 7797 19750 7799 19754 367 19758 369 19759 432 \
    19760 7795 19873 8146 19874 912 19890 8151 20023 1262 20024 1038 20027 1264 \
    20046 1266 20161 224 20162 225 20163 226 20165 227 20167 257 20168 259 \
    20170 551 20171 228 20172 7843 20173 462 20174 513 20175 515 20176 7841 \
    20177 261 20186 229 20189 7681 20306 262 20307 264 20314 266 20317 268 \
    20324 199 20449 7856 20450 7854 20453 7860 20460 7858 20628 1571]


proc GetUniCharCompInfo {uc} {
    variable COMP_OFFSET_BITS
    variable COMP_COMMON_PAGE_MAP
    variable compPageMap
    variable compGroupMap

    set page [expr {($uc & 0x1fffff) >> $COMP_OFFSET_BITS}]
    if {[info exists compPageMap($page)]} {
	set apage $compPageMap($page)
    } else {
	set apage $COMP_COMMON_PAGE_MAP
    }

    lindex $compGroupMap \
	   [expr {($apage << $COMP_OFFSET_BITS) | \
		   ($uc & ((1 << $COMP_OFFSET_BITS) - 1))}]
}

variable COMP_SINGLE_MASK 65536
variable COMP_SECOND_MASK 131072
variable COMP_MASK 65535
variable COMP_LENGTH1 144

proc GetCompFirst {uc info} {
    variable COMP_SINGLE_MASK
    variable COMP_SECOND_MASK
    variable COMP_MASK
    variable compFirstList

    if {$info == -1 || !($info & $COMP_SINGLE_MASK)} {
	return -1
    }
    if {!($info & $COMP_SECOND_MASK)} {
	set comp [lindex $compFirstList [expr {$info & $COMP_MASK}]]
	if {$uc == [lindex $comp 0]} {
	    return [lindex $comp 1]
	}
    }
    return 0
}

proc GetCompSecond {uc info} {
    variable COMP_SINGLE_MASK
    variable COMP_SECOND_MASK
    variable COMP_MASK
    variable compSecondList

    if {$info == -1 || !($info & $COMP_SINGLE_MASK)} {
	return -1
    }
    if {$info & $COMP_SECOND_MASK} {
	set comp [lindex $compSecondList [expr {$info & $COMP_MASK}]]
	if {$uc == [lindex $comp 0]} {
	    return [lindex $comp 1]
	}
    }
    return 0
}

proc GetCompBoth {info1 info2} {
    variable COMP_SECOND_MASK
    variable COMP_MASK
    variable COMP_LENGTH1
    variable compBothMap

    if {$info1 != -1 && $info2 != -1 &&  !($info1 & $COMP_SECOND_MASK) && ($info2 & $COMP_SECOND_MASK)} {
	set idx [expr {$COMP_LENGTH1 * $info1 + ($info2 & $COMP_MASK)}]
	if {[info exists compBothMap($idx)]} {
	    return $compBothMap($idx)
	} else {
	    return 0
	}
    } else {
	return 0
    }
}

} ; # namespace eval ::unicode::data

