/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.core.internal;

import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.LongAdder;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.CyclicTimeouts;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.Retainable;
import org.eclipse.jetty.io.RetainableByteBuffer;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.IteratingCallback;
import org.eclipse.jetty.util.NanoTime;
import org.eclipse.jetty.util.TypeUtil;
import org.eclipse.jetty.util.thread.AutoLock;
import org.eclipse.jetty.util.thread.Scheduler;
import org.eclipse.jetty.websocket.core.Behavior;
import org.eclipse.jetty.websocket.core.CloseStatus;
import org.eclipse.jetty.websocket.core.Frame;
import org.eclipse.jetty.websocket.core.OutgoingEntry;
import org.eclipse.jetty.websocket.core.exception.WebSocketException;
import org.eclipse.jetty.websocket.core.exception.WebSocketWriteTimeoutException;
import org.eclipse.jetty.websocket.core.internal.Generator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FrameFlusher
extends IteratingCallback {
    public static final Frame FLUSH_FRAME = new Frame(-1){

        @Override
        public boolean isControlFrame() {
            return true;
        }
    };
    private static final Logger LOG = LoggerFactory.getLogger(FrameFlusher.class);
    private final AutoLock _lock = new AutoLock();
    private final LongAdder _messagesOut = new LongAdder();
    private final LongAdder _bytesOut = new LongAdder();
    private final ByteBufferPool _bufferPool;
    private final EndPoint _endPoint;
    private final int _bufferSize;
    private final Generator _generator;
    private final int _maxGather;
    private final CyclicTimeouts<FlusherEntry> _cyclicTimeouts;
    private final Deque<FlusherEntry> _queue = new ArrayDeque<FlusherEntry>();
    private final List<FlusherEntry> _currentEntries;
    private final List<FlusherEntry> _completedEntries = new ArrayList<FlusherEntry>();
    private final List<RetainableByteBuffer> _releasableBuffers = new ArrayList<RetainableByteBuffer>();
    private final Behavior _behavior;
    private long _currentMessageExpiry = Long.MAX_VALUE;
    private RetainableByteBuffer _batchBuffer;
    private boolean _canEnqueue = true;
    private Throwable _closedCause;
    private boolean _useDirectByteBuffers;

    public FrameFlusher(ByteBufferPool bufferPool, Scheduler scheduler, Generator generator, EndPoint endPoint, int bufferSize, int maxGather, Behavior behavior) {
        this._behavior = behavior;
        this._bufferPool = bufferPool;
        this._endPoint = endPoint;
        this._bufferSize = bufferSize;
        this._generator = Objects.requireNonNull(generator);
        this._maxGather = maxGather;
        this._currentEntries = new ArrayList<FlusherEntry>(maxGather);
        this._cyclicTimeouts = new CyclicTimeouts<FlusherEntry>(scheduler){
            private boolean _expired;
            {
                this._expired = false;
            }

            protected boolean onExpired(FlusherEntry expirable) {
                this._expired = true;
                return false;
            }

            protected Iterator<FlusherEntry> iterator() {
                return TypeUtil.concat(FrameFlusher.this._currentEntries.iterator(), FrameFlusher.this._queue.iterator());
            }

            protected void iterate() {
                try (AutoLock ignored = FrameFlusher.this._lock.lock();){
                    super.iterate();
                }
                if (this._expired) {
                    FrameFlusher.this.abort(new WebSocketWriteTimeoutException("FrameFlusher Write Timeout"));
                }
            }
        };
    }

    public boolean isUseDirectByteBuffers() {
        return this._useDirectByteBuffers;
    }

    public void setUseDirectByteBuffers(boolean useDirectByteBuffers) {
        this._useDirectByteBuffers = useDirectByteBuffers;
    }

    public boolean enqueue(OutgoingEntry outgoingEntry) {
        FlusherEntry entry = new FlusherEntry(outgoingEntry);
        Frame frame = outgoingEntry.getFrame();
        byte opCode = frame.getOpCode();
        Throwable error = null;
        boolean abort = false;
        ArrayList<FlusherEntry> failedEntries = null;
        CloseStatus closeStatus = null;
        try (AutoLock ignored = this._lock.lock();){
            if (!this._canEnqueue || this._closedCause != null) {
                error = this._closedCause == null ? new ClosedChannelException() : this._closedCause;
            } else {
                switch (opCode) {
                    case 8: {
                        closeStatus = CloseStatus.getCloseStatus(frame);
                        if (closeStatus.isAbnormal()) {
                            failedEntries = new ArrayList<FlusherEntry>(this._queue);
                            this._queue.clear();
                        }
                        this._queue.offerLast(entry);
                        this._canEnqueue = false;
                        break;
                    }
                    case 9: 
                    case 10: {
                        this._queue.offerFirst(entry);
                        break;
                    }
                    default: {
                        if (entry.isExpired()) {
                            error = new WebSocketWriteTimeoutException("FrameFlusher Write Timeout");
                            abort = true;
                            break;
                        }
                        this._queue.offerLast(entry);
                    }
                }
                if (!abort) {
                    this._cyclicTimeouts.schedule((CyclicTimeouts.Expirable)entry);
                }
            }
        }
        if (failedEntries != null) {
            WebSocketException failure = new WebSocketException("Flusher received abnormal CloseFrame: " + CloseStatus.codeString(closeStatus.getCode()), closeStatus.getCause());
            for (FlusherEntry e : failedEntries) {
                this.notifyCallbackFailure(e.getCallback(), failure);
            }
        }
        if (error != null) {
            this.notifyCallbackFailure(outgoingEntry.getCallback(), error);
            if (abort) {
                this.abort(error);
            }
            return false;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Enqueued {} to {}", (Object)entry, (Object)this);
        }
        return true;
    }

    public void onClose(Throwable cause) {
        try (AutoLock ignored = this._lock.lock();){
            this._closedCause = cause == null ? new ClosedChannelException() : cause;
        }
        this.abort(this._closedCause);
    }

    protected IteratingCallback.Action process() throws Throwable {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Flushing {}", (Object)this);
        }
        boolean flush = false;
        try (AutoLock ignored = this._lock.lock();){
            if (this._closedCause != null) {
                throw this._closedCause;
            }
            while (!this._queue.isEmpty() && this._currentEntries.size() <= this._maxGather) {
                FlusherEntry entry = this._queue.poll();
                this._currentEntries.add(entry);
                if (entry.getFrame() != FLUSH_FRAME) continue;
                flush = true;
                break;
            }
        }
        boolean canBatch = true;
        ArrayList<ByteBuffer> buffers = new ArrayList<ByteBuffer>(this._maxGather * 2 + 1);
        if (this._batchBuffer != null) {
            buffers.add(this._batchBuffer.getByteBuffer());
        }
        for (FlusherEntry entry : this._currentEntries) {
            boolean batch;
            if (entry.getFrame() == FLUSH_FRAME) continue;
            this._messagesOut.increment();
            int batchSpace = this._batchBuffer == null ? this._bufferSize : BufferUtil.space((ByteBuffer)this._batchBuffer.getByteBuffer());
            boolean bl = batch = canBatch && entry.isBatch() && !entry.getFrame().isControlFrame() && entry.getFrame().getPayloadLength() < this._bufferSize / 4 && batchSpace - 28 >= entry.getFrame().getPayloadLength();
            if (batch) {
                if (this._batchBuffer == null) {
                    this._batchBuffer = this.acquireBuffer(this._bufferSize);
                    buffers.add(this._batchBuffer.getByteBuffer());
                }
                this._generator.generateWholeFrame(entry.getFrame(), this._batchBuffer.getByteBuffer());
                continue;
            }
            if (canBatch && this._batchBuffer != null && batchSpace >= 28) {
                this._generator.generateHeader(entry.getFrame(), this._batchBuffer.getByteBuffer());
            } else {
                RetainableByteBuffer headerBuffer = this.acquireBuffer(28);
                this._releasableBuffers.add(headerBuffer);
                this._generator.generateHeader(entry.getFrame(), headerBuffer.getByteBuffer());
                buffers.add(headerBuffer.getByteBuffer());
            }
            ByteBuffer payload = entry.getFrame().getPayload();
            if (BufferUtil.hasContent((ByteBuffer)payload)) {
                if (entry.getFrame().isMasked()) {
                    RetainableByteBuffer masked = this.acquireBuffer(entry.getFrame().getPayloadLength());
                    payload = masked.getByteBuffer();
                    this._releasableBuffers.add(masked);
                    this._generator.generatePayload(entry.getFrame(), payload);
                }
                buffers.add(payload);
            }
            canBatch = false;
            flush = true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} processed {} entries flush={}: {}", new Object[]{this, this._currentEntries.size(), flush, this._currentEntries});
        }
        if (flush) {
            if (this._batchBuffer != null) {
                this._releasableBuffers.add(this._batchBuffer);
                this._batchBuffer = null;
            }
            int i = 0;
            int bytes = 0;
            ByteBuffer[] bufferArray = new ByteBuffer[buffers.size()];
            for (ByteBuffer bb : buffers) {
                bytes += bb.limit() - bb.position();
                bufferArray[i++] = bb;
            }
            this._bytesOut.add(bytes);
            this._endPoint.write((Callback)this, bufferArray);
            buffers.clear();
        } else {
            if (this._currentEntries.isEmpty()) {
                this.releaseAggregateIfEmpty();
                return IteratingCallback.Action.IDLE;
            }
            this.succeeded();
        }
        return IteratingCallback.Action.SCHEDULED;
    }

    private RetainableByteBuffer acquireBuffer(int capacity) {
        return this._bufferPool.acquire(capacity, this.isUseDirectByteBuffers());
    }

    private int getQueueSize() {
        try (AutoLock ignored = this._lock.lock();){
            int n = this._queue.size();
            return n;
        }
    }

    protected void onSuccess() {
        try (AutoLock ignored = this._lock.lock();){
            assert (this._completedEntries.isEmpty());
            this._completedEntries.addAll(this._currentEntries);
            this._currentEntries.clear();
        }
        for (FlusherEntry entry : this._completedEntries) {
            if (entry.getFrame().getOpCode() == 8 && this._behavior == Behavior.SERVER) {
                this._endPoint.shutdownOutput();
            }
            this.notifyCallbackSuccess(entry.getCallback());
        }
        this._completedEntries.clear();
        this._releasableBuffers.forEach(Retainable::release);
        this._releasableBuffers.clear();
        super.onSuccess();
    }

    public void onCompleteFailure(Throwable failure) {
        if (this._batchBuffer != null) {
            this._batchBuffer.clear();
        }
        this.releaseAggregateIfEmpty();
        try (AutoLock ignored = this._lock.lock();){
            this._canEnqueue = false;
            if (this._closedCause == null) {
                this._closedCause = failure;
            } else if (this._closedCause != failure) {
                this._closedCause.addSuppressed(failure);
            }
            assert (this._completedEntries.isEmpty());
            this._completedEntries.addAll(this._queue);
            this._completedEntries.addAll(this._currentEntries);
            this._queue.clear();
            this._currentEntries.clear();
            this._cyclicTimeouts.destroy();
        }
        for (FlusherEntry entry : this._completedEntries) {
            this.notifyCallbackFailure(entry.getCallback(), failure);
        }
        this._completedEntries.clear();
        this._releasableBuffers.forEach(Retainable::release);
        this._releasableBuffers.clear();
        this._endPoint.close(this._closedCause);
    }

    private void releaseAggregateIfEmpty() {
        if (this._batchBuffer != null && !this._batchBuffer.hasRemaining()) {
            this._batchBuffer.release();
            this._batchBuffer = null;
        }
    }

    protected void notifyCallbackSuccess(Callback callback) {
        block3: {
            try {
                if (callback != null) {
                    callback.succeeded();
                }
            }
            catch (Throwable x) {
                if (!LOG.isDebugEnabled()) break block3;
                LOG.atDebug().setCause(x).log("Exception while notifying success of callback {}", (Object)callback);
            }
        }
    }

    protected void notifyCallbackFailure(Callback callback, Throwable failure) {
        block3: {
            try {
                if (callback != null) {
                    callback.failed(failure);
                }
            }
            catch (Throwable x) {
                if (!LOG.isDebugEnabled()) break block3;
                LOG.atDebug().setCause(x).log("Exception while notifying failure of callback {}", (Object)callback);
            }
        }
    }

    public long getMessagesOut() {
        return this._messagesOut.longValue();
    }

    public long getBytesOut() {
        return this._bytesOut.longValue();
    }

    public String toString() {
        return String.format("%s[queueSize=%d,aggregate=%s]", super.toString(), this.getQueueSize(), this._batchBuffer);
    }

    private class FlusherEntry
    implements CyclicTimeouts.Expirable {
        private final OutgoingEntry _outgoingEntry;
        private final long _expiry;

        public FlusherEntry(OutgoingEntry outgoingEntry) {
            this._outgoingEntry = outgoingEntry;
            long messageTimeout = outgoingEntry.getMessageTimeout();
            long frameExpiry = CyclicTimeouts.Expirable.calcExpireNanoTime((long)outgoingEntry.getFrameTimeout());
            Frame frame = outgoingEntry.getFrame();
            if (frame.isDataFrame()) {
                if (frame.getOpCode() != 0) {
                    FrameFlusher.this._currentMessageExpiry = CyclicTimeouts.Expirable.calcExpireNanoTime((long)messageTimeout);
                }
                if (FrameFlusher.this._currentMessageExpiry != Long.MAX_VALUE) {
                    frameExpiry = frameExpiry == Long.MAX_VALUE ? FrameFlusher.this._currentMessageExpiry : FlusherEntry.minNanoTime(frameExpiry, FrameFlusher.this._currentMessageExpiry);
                }
            } else {
                long messageExpiry = CyclicTimeouts.Expirable.calcExpireNanoTime((long)messageTimeout);
                if (messageTimeout != Long.MAX_VALUE) {
                    frameExpiry = frameExpiry == Long.MAX_VALUE ? messageExpiry : FlusherEntry.minNanoTime(frameExpiry, messageExpiry);
                }
            }
            this._expiry = frameExpiry;
        }

        public Frame getFrame() {
            return this._outgoingEntry.getFrame();
        }

        public Callback getCallback() {
            return this._outgoingEntry.getCallback();
        }

        public boolean isBatch() {
            return this._outgoingEntry.isBatch();
        }

        public long getExpireNanoTime() {
            return this._expiry;
        }

        public boolean isExpired() {
            return this._expiry != Long.MAX_VALUE && NanoTime.until((long)this._expiry) < 0L;
        }

        private static long minNanoTime(long nanoTime1, long nanoTime2) {
            return NanoTime.isBeforeOrSame((long)nanoTime1, (long)nanoTime2) ? nanoTime1 : nanoTime2;
        }

        public String toString() {
            return String.format("%s{entry=%b,expire=%s}", TypeUtil.toShortName(this.getClass()), this._outgoingEntry, NanoTime.millisUntil((long)this._expiry));
        }
    }
}

