/*************************************************************************
 *
 *  $RCSfile: imapurl.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:13:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _INET_IMAPCLNT_HXX
#include <inet/imapclnt.hxx>
#endif
#ifndef _INET_CONFIG_HXX
#include <inet/inetcfg.hxx>
#endif
#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef TOOLS_INETMIME_HXX
#include <tools/inetmime.hxx>
#endif
#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif

#ifndef _CNTIDS_HRC
#include <cntids.hrc>
#endif

#ifndef CHAOS_IMAPURL_HXX
#include <imapurl.hxx>
#endif

namespace unnamed_chaos_imapurl {} using namespace unnamed_chaos_imapurl;
	// unnamed namespaces don't work well yet...

using namespace chaos;

//============================================================================
namespace unnamed_chaos_imapurl {

class ModifiedUTF7Sink
{
	ByteString m_aBuffer;
	sal_uInt32 m_nValue;
	int m_nFilled;

	void write(bool bEndEncoding);

	void writeUTF16(sal_uInt32 nUTF16);

public:
	ModifiedUTF7Sink(): m_nFilled(-1) {}

	void writeUTF32(sal_uInt32 nUTF32);

	void writeVerbatim(sal_Char cChar);

	ByteString const & getBuffer();
};

inline void ModifiedUTF7Sink::writeVerbatim(sal_Char cChar)
{
	write(true);
	m_aBuffer += cChar;
}

inline ByteString const & ModifiedUTF7Sink::getBuffer()
{
	write(true);
	return m_aBuffer;
}

}

//============================================================================
//
//  CntIMAPURL
//
//============================================================================

/* Canonic pseudo IMAP URLs are of the form

	 canonic-pseudo-IMAP-URL = account-URL / mailbox-URL / message-URL

	 account-URL = "imap://" user [";AUTH=" auth] "@" hostport "/"
	 mailbox-URL = account-URL segment *("/" segment)
	 message-URL = mailbox-URL [";UIDValidity=" nz_number "/"]
	                   ";UID=" nz_number

	 user = 1*{RFC 2060 <CHAR8> using (escape / ALPHA / DIGIT / "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / "-" / "." / "=" / "_" / "~")}

	 auth = {RFC 2060 <atom> using *(escape / ALPHA / DIGIT / "!" / "$" / "&" / "'" / "+" / "," / "-" / "." / "=" / "_" / "~")}

	 hostport = host [":" port]
	 host = incomplete-hostname / hostname / IPv4address / IPv6reference
	 incomplete-hostname = *(domainlabel ".") domainlabel
	 hostname = *(domainlabel ".") toplabel ["."]
	 domainlabel = alphadigit [*(alphadigit / "-") alphadigit]
	 toplabel = ALPHA [*(alphadigit / "-") alphadigit]
	 alphadigit = ALPHA / DIGIT
	 IPv4address = 1*3DIGIT "." 1*3DIGIT "." 1*3DIGIT "." 1*3DIGIT
	 IPv6reference = "[" hexpart [":" IPv4address] "]"
	 hexpart = (hexseq ["::" [hexseq]]) / ("::" [hexseq])
	 hexseq = hex4 *(":" hex4)
	 hex4 = 1*4HEXDIG
	 port = *DIGIT

	 segment = 1*({RFC 2279 <non-null UTF-8 character> using (escape / ALPHA / DIGIT / "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / "-" / "." / "=" / "_" / "~" / ":" / "@")} / "%26")

	 nz_number = %x31-39 *DIGIT  ; 0 < n < 4294967296

	 escape = "%" HEXDIG HEXDIG

  These URLs are 'pseudo' IMAP URLs because they do not follow RFC 2192.
*/

//============================================================================
/** The number of numeric segments in IMAP numeric host addresses.
 */
#define CNT_IMAP_HOST_NUMERIC_SEGMENT_COUNT 4

//============================================================================
/** The IMAP default port 143.
 */
#define CNT_IMAP_URL_DEFAULT_PORT_LEN 3
#define CNT_IMAP_URL_DEFAULT_PORT_DIGIT1 '1'
#define CNT_IMAP_URL_DEFAULT_PORT_DIGIT2 '4'
#define CNT_IMAP_URL_DEFAULT_PORT_DIGIT3 '3'

//============================================================================
namespace unnamed_chaos_imapurl {

enum Encoding
{
	ENCODING_INVALID,
	ENCODING_NON_CANONIC,
	ENCODING_CANONIC
};

//============================================================================
/** The various parts of pseudo IMAP URLs.
 */
enum URLPart
{
	ENC_USER_AUTH_TYPE = 1,
	HOST = 2,
	HOST_PORT = 4,
	MBOX_SEGMENT = 8
};

//============================================================================
/** Return whether for a given part of a pseudo IMAP URL a given UCS-4
	character must be encoded.

	@param ePart  Some URL part.

	@param nUCS4  Some UCS-4 character.

	@return  True if the character must be encoded in the URL part.
 */
inline bool mustEncode(URLPart ePart, sal_uInt32 nUCS4)
{
	sal_uInt8 const aNotEncode[128]
		= {  0,  0,  0,  0,  0,  0,  0,  0,
			 0,  0,  0,  0,  0,  0,  0,  0,
			 0,  0,  0,  0,  0,  0,  0,  0,
			 0,  0,  0,  0,  0,  0,  0,  0,
			 0,  9,  0,  0,  9,  0,  9,  9, //  !"#$%&'
			 9,  9,  1,  9,  9, 15, 15,  0, // ()*+,-./
			15, 15, 15, 15, 15, 15, 15, 15, // 01234567
			15, 15, 14,  0,  0,  9,  0,  0, // 89:;<=>?
			 8, 15, 15, 15, 15, 15, 15, 15, // @ABCDEFG
			15, 15, 15, 15, 15, 15, 15, 15, // HIJKLMNO
			15, 15, 15, 15, 15, 15, 15, 15, // PQRSTUVW
			15, 15, 15,  0,  0,  0,  0,  9, // XYZ[\]^_
			 0, 15, 15, 15, 15, 15, 15, 15, // `abcdefg
			15, 15, 15, 15, 15, 15, 15, 15, // hijklmno
			15, 15, 15, 15, 15, 15, 15, 15, // pqrstuvw
			15, 15, 15,  0,  0,  0,  1,  0 }; // xyz{|}~
		// Due to a problem with CntNode::IsDummyURL_Impl(), asterisks ('*')
		// must	be encoded in mailbox segments, though RFC 2192 does not
		// require this.  The problem is that the 'factory pattern' for
		// CntIMAPMboxNode is a solidus ('/') followed by an asterisk ('*'),
		// and that CntNode::IsDummyURL_Impl() says a node is a 'dummy node'
		// if it contains the 'factory pattern'.  So, a mailbox containing an
		// asterisks ('*') as the first character in its name would have a URL
		// containing a solidus ('/') followed by an asterisks ('*'), and
		// would therefore be considered a 'dummy'.
		// And due to a problem in CntStorageNode::Own2FileURL(), which
		// translates ".cache:imap://user@host/~other" to
		// "file:///.../store/userhostimap.scs" without an appended
		// "#imap://user@host/~other", the tilde ('~') must also be encoded in
		// mailbox segments, though RFC 2192 does not require this.

	return !(INetMIME::isUSASCII(nUCS4) && aNotEncode[nUCS4] & ePart);
}

//============================================================================
bool isCanonicMboxSegment(String const & rText, xub_StrLen nOffset)
{
	if (nOffset >= rText.Len())
		return false;
	while (nOffset < rText.Len())
	{
		sal_Unicode cUTF16 = rText.GetChar(nOffset++);
		if (cUTF16 == '%' ?
			    nOffset + 1 >= rText.Len()
			    || !INetMIME::isCanonicHexDigit(rText.GetChar(nOffset++))
			    || !INetMIME::isCanonicHexDigit(rText.GetChar(nOffset++)) :
			    mustEncode(MBOX_SEGMENT, cUTF16))
			return false;
	}
	return true;
}

//============================================================================
ByteString translateSegmentFromIURI(sal_Unicode const * pBegin,
									sal_Unicode const * pEnd)
{
	ModifiedUTF7Sink aSink;
	while (pBegin != pEnd)
	{
		INetURLObject::EscapeType eEscapeType;
		sal_uInt32 nUTF32
			= INetURLObject::getUTF32(pBegin, pEnd, false, '%',
									  INetURLObject::WAS_ENCODED,
									  RTL_TEXTENCODING_UTF8, eEscapeType);
		// Special handling of "%26" to represent mailbox names with bad
		// occurences of "&" characters:
		if (nUTF32 == '&' && eEscapeType != INetURLObject::ESCAPE_UTF32)
			aSink.writeVerbatim('&');
		else
			aSink.writeUTF32(nUTF32);
	}
	return aSink.getBuffer();
}

}

//============================================================================
bool CntIMAPURL::makeCanonicURL(String const & rURL, String & rCanonicURL)
{
	INetURLObject aTheCanonicURL(rURL);
	aTheCanonicURL.makeAuthCanonic();
	aTheCanonicURL.SetHost(aTheCanonicURL.
						       GetHost(INetURLObject::DECODE_WITH_CHARSET).
						           ToLowerAscii(),
						   INetURLObject::ENCODE_ALL);
	aTheCanonicURL.makePortCanonic();

	// Handle "/segment/.../segment;UIDValidity=val/;UID=id" part:
	String aOldPath(aTheCanonicURL.GetURLPath());
	if (aOldPath.Len() == 0 || aOldPath.GetChar(0) != '/')
		return false;
	sal_Unicode const * p = aOldPath.GetBuffer();
	sal_Unicode const * pEnd = p + aOldPath.Len();
	String aCanonicPath;
	bool bSlash = true;
	bool bDone = false;

	// Handle "segment/.../segment" part (change escapes to upper case, escape
	// exactly what's neccessary, and turn a first segment of "INBOX" into
	// canonic form):
	for (++p;;)
	{
		if (p == pEnd)
		{
			bDone = true;
			break;
		}
		INetURLObject::EscapeType eEscapeType;
		sal_uInt32 nUTF32
			= INetURLObject::getUTF32(p, pEnd, false, '%',
									  INetURLObject::WAS_ENCODED,
									  RTL_TEXTENCODING_UTF8, eEscapeType);
		if (nUTF32 == '/' && eEscapeType == INetURLObject::ESCAPE_NO)
		{
			if (bSlash)
				return false;
			else if (p == pEnd)
			{
				// HACK: Because
				// CntNode::ConstructOwnURL_Impl(".cache:imap://host/a/b") for
				// storage node .cache:imap://host/a returns
				// ".cache:imap://host/a/", normalization should keep any
				// invalid trailing slashes and hope that nothing breaks
				// because of this!
//				DBG_ERROR("CntIMAPURL::makeCanonicURL():"
//						      " Trailing slash hack");
				aCanonicPath += '/';
				bDone = true;
				break;
			}
			else
				bSlash = true;
		}
		else if (nUTF32 == ';' && eEscapeType == INetURLObject::ESCAPE_NO)
			break;
		else
		{
			if (bSlash)
			{
				aCanonicPath += '/';
				bSlash = false;
			}

			// Special handling of "%26" to represent mailbox names with bad
			// occurences of "&" characters:
			if (mustEncode(MBOX_SEGMENT, nUTF32)
				|| nUTF32 == '&'
				   && eEscapeType != INetURLObject::ESCAPE_UTF32)
				INetURLObject::appendUCS4Escape(aCanonicPath, '%', nUTF32);
			else
				aCanonicPath += sal_Unicode(nUTF32);
		}
	}
	if (aCanonicPath.Len() == 0)
		aCanonicPath = '/';
	else if ((aCanonicPath.Len()
			          == RTL_CONSTASCII_LENGTH(
						     "/" INET_IMAP_CANONIC_INBOX_NAME)
			      || aCanonicPath.Len()
			             > RTL_CONSTASCII_LENGTH(
							   "/" INET_IMAP_CANONIC_INBOX_NAME)
			         && aCanonicPath.
			                    GetChar(RTL_CONSTASCII_LENGTH(
									        "/" INET_IMAP_CANONIC_INBOX_NAME))
			                == '/')
			 && aCanonicPath.
			        EqualsIgnoreCaseAscii(INET_IMAP_CANONIC_INBOX_NAME,
										  1,
										  RTL_CONSTASCII_LENGTH(
											  INET_IMAP_CANONIC_INBOX_NAME)))
		aCanonicPath.
			Replace(1, RTL_CONSTASCII_LENGTH(INET_IMAP_CANONIC_INBOX_NAME),
					String::CreateFromAscii(
						RTL_CONSTASCII_STRINGPARAM(
							INET_IMAP_CANONIC_INBOX_NAME)));

	// Handle ";UIDValidity=val/;UID=id" part:
	if (!bDone)
	{
		// Change "segment/.../segment/;UID=id" into the (by RFC 2192 wrong,
		// but canonic) form "segment/.../segment;UID=id":
		if (bSlash
			&& (pEnd - p < RTL_CONSTASCII_LENGTH("UID=")
				|| p[RTL_CONSTASCII_LENGTH("UID")] != '='))
			return false;

		// Skip ";UID" part:
		if (pEnd - p < RTL_CONSTASCII_LENGTH("UID")
			|| !INetMIME::equalIgnoreCase(p, p + RTL_CONSTASCII_LENGTH("UID"),
										  "UID"))
			return false;
		aCanonicPath.AppendAscii(RTL_CONSTASCII_STRINGPARAM(";UID"));
		p += RTL_CONSTASCII_LENGTH("UID");

		// Handle "Validity=val/;UID" part:
		if (pEnd - p >= RTL_CONSTASCII_LENGTH("Validity=")
			&& INetMIME::equalIgnoreCase(p,
										 p + RTL_CONSTASCII_LENGTH(
											     "Validity="),
										 "Validity="))
		{
			aCanonicPath.AppendAscii(RTL_CONSTASCII_STRINGPARAM("Validity="));
			p += RTL_CONSTASCII_LENGTH("Validity=");

			// Handle "val" part (skip leading zeroes):
			sal_uInt32 nValue;
			if (!INetMIME::scanUnsigned(p, pEnd, true, nValue) || nValue == 0)
				return false;
			aCanonicPath += String::CreateFromInt64(nValue);

			// Skip "/;UID" part:
			if (pEnd - p < RTL_CONSTASCII_LENGTH("/;UID")
				|| !INetMIME::equalIgnoreCase(p,
											  p + RTL_CONSTASCII_LENGTH(
												      "/;UID"),
											  "/;UID"))
				return false;
			aCanonicPath.AppendAscii(RTL_CONSTASCII_STRINGPARAM("/;UID"));
			p += RTL_CONSTASCII_LENGTH("/;UID");
		}

		// Skip "=" part:
		if (p == pEnd || *p++ != '=')
			return false;
		aCanonicPath += '=';

		// Handle "id" part (skip leading zeroes):
		sal_uInt32 nValue;
		if (!INetMIME::scanUnsigned(p, pEnd, true, nValue) || nValue == 0)
			return false;
		aCanonicPath += String::CreateFromInt64(nValue);

		if (p != pEnd)
			return false;
	}

	aTheCanonicURL.SetURLPath(aCanonicPath);

	rCanonicURL = aTheCanonicURL.GetMainURL(INetURLObject::NO_DECODE);
	return true;
}

//============================================================================
bool CntIMAPURL::makeCanonicMboxPath(String const & rMboxPath,
									 String & rCanonicMboxPath)
{
	sal_Unicode const * p = rMboxPath.GetBuffer();
	sal_Unicode const * pEnd = p + rMboxPath.Len();
	if (p != pEnd && *p == '/')
		++p;

	String aTheCanonicMboxPath;
	bool bSlash = true;
	while (p != pEnd)
	{
		INetURLObject::EscapeType eEscapeType;
		sal_uInt32 nUTF32
			= INetURLObject::getUTF32(p, pEnd, false, '%',
									  INetURLObject::WAS_ENCODED,
									  RTL_TEXTENCODING_UTF8, eEscapeType);
		if (nUTF32 == '/' && eEscapeType == INetURLObject::ESCAPE_NO)
		{
			if (bSlash)
				return false;
			else if (p == pEnd)
				break;
			else
				bSlash = true;
		}
		else
		{
			if (bSlash)
			{
				aTheCanonicMboxPath += '/';
				bSlash = false;
			}

			// Special handling of "%26" to represent mailbox names with bad
			// occurences of "&" characters:
			if (mustEncode(MBOX_SEGMENT, nUTF32)
				|| nUTF32 == '&'
				   && eEscapeType != INetURLObject::ESCAPE_UTF32)
				INetURLObject::appendUCS4Escape(aTheCanonicMboxPath, '%',
												nUTF32);
			else
				aTheCanonicMboxPath += sal_Unicode(nUTF32);
		}
	}

	if ((aTheCanonicMboxPath.Len()
		         == RTL_CONSTASCII_LENGTH(INET_IMAP_CANONIC_INBOX_NAME)
		     || aTheCanonicMboxPath.Len()
		            > RTL_CONSTASCII_LENGTH(INET_IMAP_CANONIC_INBOX_NAME)
		        && aTheCanonicMboxPath.GetChar(
					       RTL_CONSTASCII_LENGTH(
							   INET_IMAP_CANONIC_INBOX_NAME))
		               == '/')
		&& aTheCanonicMboxPath.
		       EqualsIgnoreCaseAscii(INET_IMAP_CANONIC_INBOX_NAME, 0,
									 RTL_CONSTASCII_LENGTH(
										 INET_IMAP_CANONIC_INBOX_NAME)))
		aTheCanonicMboxPath.
			Replace(0, RTL_CONSTASCII_LENGTH(INET_IMAP_CANONIC_INBOX_NAME),
					String::CreateFromAscii(
						RTL_CONSTASCII_STRINGPARAM(
							INET_IMAP_CANONIC_INBOX_NAME)));

	rCanonicMboxPath = aTheCanonicMboxPath;
	return true;
}

//============================================================================
ByteString CntIMAPURL::getRootLiteralName(String const & rMboxURL)
{
	xub_StrLen nBegin
		= rMboxURL.Search('/', RTL_CONSTASCII_LENGTH(CNT_IMAP_URL_PREFIX))
		      + 1;
	xub_StrLen nEnd = rMboxURL.Search('/', nBegin);
	if (nEnd == STRING_NOTFOUND)
		nEnd = rMboxURL.Len();
	return translateSegmentFromIURI(rMboxURL.GetBuffer() + nBegin,
									rMboxURL.GetBuffer() + nEnd);
}

//============================================================================
bool CntIMAPURL::getMboxLiteralFullName(String const & rMboxURL,
										sal_Char cHierarchySeparator,
										ByteString & rMboxLiteralFullName)
{
	ByteString aTheMboxLiteralFullName;
	for (xub_StrLen nBegin
			 = rMboxURL.Search('/',
							   RTL_CONSTASCII_LENGTH(CNT_IMAP_URL_PREFIX))
			       + 1;;)
	{
		xub_StrLen nEnd = rMboxURL.Search('/', nBegin);
		if (nEnd == STRING_NOTFOUND)
			nEnd = rMboxURL.Len();
		aTheMboxLiteralFullName
			+= translateSegmentFromIURI(rMboxURL.GetBuffer() + nBegin,
										rMboxURL.GetBuffer() + nEnd);
		if (nEnd == rMboxURL.Len())
			break;
		if (cHierarchySeparator == '\0')
			return false;
		aTheMboxLiteralFullName += cHierarchySeparator;
		nBegin = nEnd + 1;
	}
	rMboxLiteralFullName = aTheMboxLiteralFullName;
	return true;
}

//============================================================================
ByteString CntIMAPURL::getMboxLiteralLastName(String const & rMboxURL)
{
	return translateSegmentFromIURI(rMboxURL.GetBuffer()
									    + rMboxURL.SearchBackward('/') + 1,
									rMboxURL.GetBuffer() + rMboxURL.Len());
}

//============================================================================
bool CntIMAPURL::isAcntURL(String const & rURL)
{
	return rURL.Search('/', RTL_CONSTASCII_LENGTH(CNT_IMAP_URL_PREFIX))
		       == rURL.Len() - 1;
}

//============================================================================
bool CntIMAPURL::isMesgURL(String const & rAcntURL, String const & rURL,
						   String & rCanonicMesgURL)
{
	String aCanonicURL;
	if (!makeCanonicURL(rURL, aCanonicURL)
		|| rAcntURL.Match(aCanonicURL) != STRING_MATCH
		|| aCanonicURL.Search(';',
							  aCanonicURL.Search('/',
												 RTL_CONSTASCII_LENGTH(
													 CNT_IMAP_URL_PREFIX)))
		       == STRING_NOTFOUND)
		return false;
	rCanonicMesgURL = aCanonicURL;
	return true;
}

//============================================================================
bool CntIMAPURL::isRootMboxURL(String const & rAcntURL,
							   String const & rRootMboxURL)
{
	return rAcntURL.Match(rRootMboxURL) == STRING_MATCH
	       && isCanonicMboxSegment(rRootMboxURL, rAcntURL.Len());
}

//============================================================================
bool CntIMAPURL::isDirectSubMboxURL(String const & rParentMboxURL,
									String const & rSubMboxURL)
{
	return rParentMboxURL.Match(rSubMboxURL) == STRING_MATCH
	       && rSubMboxURL.Len() > rParentMboxURL.Len()
	       && rSubMboxURL.GetChar(rParentMboxURL.Len()) == '/'
	       && isCanonicMboxSegment(rSubMboxURL, rParentMboxURL.Len() + 1);
}

//============================================================================
bool CntIMAPURL::isInboxURL(String const & rMboxURL)
{
	return rMboxURL.EqualsAscii(INET_IMAP_CANONIC_INBOX_NAME,
								rMboxURL.Search('/',
												RTL_CONSTASCII_LENGTH(
													CNT_IMAP_URL_PREFIX))
								    + 1,
								STRING_LEN)
		       != false;
}

//============================================================================
void CntIMAPURL::decomposeMesgURL(String const & rMesgURL, String & rMboxURL,
								  sal_uInt32 & rUIDValidity,
								  sal_uInt32 & rMesgUID)
{
	rUIDValidity = 0;
	rMesgUID = 0;

	xub_StrLen nPos = rMesgURL.Len();
	for (sal_uInt32 nDigitWeight = 1;; nDigitWeight *= 10)
	{
		int nDigitValue = INetMIME::getWeight(rMesgURL.GetChar(--nPos));
		if (nDigitValue < 0)
			break;
		rMesgUID += nDigitWeight * nDigitValue;
	}

	nPos -= RTL_CONSTASCII_LENGTH(CNT_IMAP_URL_UID_PREFIX);

	if (rMesgURL.GetChar(nPos - 1) == '/')
	{
		--nPos;
		for (sal_uInt32 nDigitWeight = 1;; nDigitWeight *= 10)
		{
			int nDigitValue = INetMIME::getWeight(rMesgURL.GetChar(--nPos));
			if (nDigitValue < 0)
				break;
			rUIDValidity += nDigitWeight * nDigitValue;
		}

		nPos -= RTL_CONSTASCII_LENGTH(CNT_IMAP_URL_UIDVALIDITY_PREFIX);
	}

	rMboxURL = rMesgURL.Copy(0, nPos);
}

//============================================================================
ErrCode CntIMAPURL::makeCanonicHostPort(String const & rHostPort,
										String & rCanonicHostPort)
{
	xub_StrLen nPortSeparatorPos = STRING_NOTFOUND;
	int nDotCount = 0;
	bool bTrailingDot = false;
	bool bNonDigitChars = false;
	bool bLeadingZeroes = false;

	for (xub_StrLen i = 0; i < rHostPort.Len(); ++i)
	{
		sal_Unicode cChar = rHostPort.GetChar(i);
		if (cChar == ':')
		{
			nPortSeparatorPos = i;
			break;
		}
		else if (cChar == '.')
		{
			if (i != 0 && rHostPort.GetChar(i - 1) == '.')
				bNonDigitChars = true;
			else if (i + 1 == rHostPort.Len()
					 || rHostPort.GetChar(i + 1) == ':')
				bTrailingDot = true;
			else
				++nDotCount;
		}
		else if (cChar == '0')
		{
			if (i == 0 || rHostPort.GetChar(i - 1) == '.')
				bLeadingZeroes = true;
		}
		else if (!INetMIME::isDigit(cChar))
			bNonDigitChars = true;
	}

	xub_StrLen nPortPos = STRING_NOTFOUND;
	bool bDefaultPort = false;
	if (nPortSeparatorPos != STRING_NOTFOUND)
	{
		if (nPortSeparatorPos == rHostPort.Len() - 1)
			return ERRCODE_CHAOS_SERVER_PORT_SYNTAX;

		for (xub_StrLen i = nPortSeparatorPos + 1; i < rHostPort.Len(); ++i)
		{
			sal_Unicode cChar = rHostPort.GetChar(i);
			if (INetMIME::isDigit(cChar))
			{
				if (cChar != '0' && nPortPos == STRING_NOTFOUND)
					nPortPos = i;
			}
			else
				return ERRCODE_CHAOS_SERVER_PORT_SYNTAX;
		}

		bDefaultPort
			= nPortPos != STRING_NOTFOUND
			  && nPortPos + CNT_IMAP_URL_DEFAULT_PORT_LEN == rHostPort.Len()
			  && rHostPort.GetChar(nPortPos)
			         == CNT_IMAP_URL_DEFAULT_PORT_DIGIT1
			  && rHostPort.GetChar(nPortPos + 1)
			         == CNT_IMAP_URL_DEFAULT_PORT_DIGIT2
			  && rHostPort.GetChar(nPortPos + 2)
			         == CNT_IMAP_URL_DEFAULT_PORT_DIGIT3;
	}

	if (bNonDigitChars || nDotCount != CNT_IMAP_HOST_NUMERIC_SEGMENT_COUNT - 1
		|| !bLeadingZeroes || bTrailingDot)
	{
		if (nPortSeparatorPos == STRING_NOTFOUND)
			rCanonicHostPort = rHostPort;
		else if (bDefaultPort)
			rCanonicHostPort = rHostPort.Copy(0, nPortSeparatorPos);
		else if (nPortPos == STRING_NOTFOUND)
			rCanonicHostPort = rHostPort.Copy(0, nPortSeparatorPos + 2);
		else if (nPortPos == nPortSeparatorPos + 1)
			rCanonicHostPort = rHostPort;
		else
		{
			String aTheCanonicHostPort(rHostPort, 0, nPortSeparatorPos + 1);
			aTheCanonicHostPort += rHostPort.Copy(nPortPos);
			rCanonicHostPort = aTheCanonicHostPort;
		}
	}
	else
	{
		String aTheCanonicHostPort;

		for (xub_StrLen i = 0; i < rHostPort.Len(); ++i)
		{
			sal_Unicode cChar = rHostPort.GetChar(i);
			if (cChar == ':')
				break;
			else if (cChar != '0'
					 || aTheCanonicHostPort.Len()
					    && aTheCanonicHostPort.
					               GetChar(aTheCanonicHostPort.Len() - 1)
					           != '.'
					 || i + 1 == rHostPort.Len()
					 || rHostPort.GetChar(i + 1) == ':'
					 || rHostPort.GetChar(i + 1) == '.')
				aTheCanonicHostPort += cChar;
		}

		if (nPortSeparatorPos != STRING_NOTFOUND && !bDefaultPort)
		{
			aTheCanonicHostPort += ':';
			if (nPortPos == STRING_NOTFOUND)
				aTheCanonicHostPort += '0';
			else
				aTheCanonicHostPort += rCanonicHostPort.Copy(nPortPos);
		}

		rCanonicHostPort = aTheCanonicHostPort;
	}

	return ERRCODE_NONE;
}

//============================================================================
String CntIMAPURL::createHostAndPortURL(String const & rHost,
										sal_uInt16 nPort)
{
	String
		aHostAndPortURL(String::CreateFromAscii(
			                RTL_CONSTASCII_STRINGPARAM(CNT_IMAP_URL_PREFIX)));
	sal_Unicode const * p = rHost.GetBuffer();
	sal_Unicode const * pEnd = p + rHost.Len();
	while (p != pEnd)
	{
		sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
		if (mustEncode(HOST, nUTF32))
			INetURLObject::appendUCS4Escape(aHostAndPortURL, '%', nUTF32);
		else
			aHostAndPortURL += sal_Unicode(nUTF32);
	}
	if (nPort != inet::eINET_IMAP_PORT)
	{
		aHostAndPortURL += ':';
		aHostAndPortURL += nPort;
	}
	return aHostAndPortURL;
}

//============================================================================
String CntIMAPURL::createAcntURL(bool bHasIUserAuth, String const & rUserID,
								 String const & rAuthType,
								 String const & rHostPort)
{
	String aAcntURL(String::CreateFromAscii(
		                RTL_CONSTASCII_STRINGPARAM(CNT_IMAP_URL_PREFIX)));
	if (bHasIUserAuth)
	{
		sal_Unicode const * p = rUserID.GetBuffer();
		sal_Unicode const * pEnd = p + rUserID.Len();
		while (p != pEnd)
		{
			sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
			if (mustEncode(ENC_USER_AUTH_TYPE, nUTF32))
				INetURLObject::appendUCS4Escape(aAcntURL, '%', nUTF32);
			else
				aAcntURL += sal_Unicode(nUTF32);
		}
		if (rAuthType.Len() != 0)
		{
			aAcntURL.AppendAscii(RTL_CONSTASCII_STRINGPARAM(";AUTH="));
			p = rAuthType.GetBuffer();
			pEnd = p + rAuthType.Len();
			while (p != pEnd)
			{
				sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
				if (mustEncode(ENC_USER_AUTH_TYPE, nUTF32))
					INetURLObject::appendUCS4Escape(aAcntURL, '%', nUTF32);
				else
					aAcntURL += sal_Unicode(nUTF32);
			}
		}
		aAcntURL += '@';
	}
	sal_Unicode const * p = rHostPort.GetBuffer();
	sal_Unicode const * pEnd = p + rHostPort.Len();
	while (p != pEnd)
	{
		sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
		if (mustEncode(HOST_PORT, nUTF32))
			INetURLObject::appendUCS4Escape(aAcntURL, '%', nUTF32);
		else
			aAcntURL += sal_Unicode(nUTF32);
	}
	aAcntURL += '/';
	return aAcntURL;
}

//============================================================================
String CntIMAPURL::createMboxURL(String const & rAcntURL,
								 ByteString const &
								     rMboxCanonicLiteralFullName,
								 sal_Char cHierarchySeparator)
{
	String aMboxURL(rAcntURL);
	for (xub_StrLen i = 0;;)
	{
		xub_StrLen j
			= cHierarchySeparator == '\0' ?
			      STRING_NOTFOUND :
			      rMboxCanonicLiteralFullName.Search(cHierarchySeparator,
													 i == 0 ? 1 : i);
		if (j == STRING_NOTFOUND)
			j = rMboxCanonicLiteralFullName.Len();
		CntIMAPUTF7::translateFromUTF7(rMboxCanonicLiteralFullName.GetBuffer()
									       + i,
									   rMboxCanonicLiteralFullName.GetBuffer()
									       + j,
									   true, aMboxURL);
		if (j >= rMboxCanonicLiteralFullName.Len())
			break;
		aMboxURL += '/';
		i = j + 1;
		if (i >= rMboxCanonicLiteralFullName.Len())
			break;
	}
	return aMboxURL;
}

//============================================================================
String CntIMAPURL::createMboxURL(String const & rAcntURL,
								 String const & rMboxPath)
{
	String aMboxURL(rAcntURL);
	aMboxURL += rMboxPath;
	return aMboxURL;
}

//============================================================================
String CntIMAPURL::createSubMboxURL(String const & rParentURL,
									ByteString const &
									    rSubMboxLiteralLastName)
{
	bool bParentAcnt = rParentURL.GetChar(rParentURL.Len() - 1) == '/';

	String aSubMboxURL(rParentURL);
	if (!bParentAcnt)
		aSubMboxURL += '/';

	if (bParentAcnt
		&& INetMIME::equalIgnoreCase(rSubMboxLiteralLastName,
									 INET_IMAP_CANONIC_INBOX_NAME))
		aSubMboxURL.AppendAscii(RTL_CONSTASCII_STRINGPARAM(
			                        INET_IMAP_CANONIC_INBOX_NAME));
	else
		CntIMAPUTF7::translateFromUTF7(rSubMboxLiteralLastName.GetBuffer(),
									   rSubMboxLiteralLastName.GetBuffer()
									       + rSubMboxLiteralLastName.Len(),
									   true, aSubMboxURL);

	return aSubMboxURL;
}

//============================================================================
String CntIMAPURL::createMesgURL(String const & rMboxURL,
								 sal_uInt32 nUIDValidity, sal_uInt32 nMesgUID)
{
	String aMesgURL(rMboxURL);
	if (nUIDValidity != 0)
	{
		aMesgURL.AppendAscii(RTL_CONSTASCII_STRINGPARAM(
			                     ";" CNT_IMAP_URL_UIDVALIDITY_PREFIX));
		aMesgURL += String::CreateFromInt64(nUIDValidity);
		aMesgURL += '/';
	}
	aMesgURL.
		AppendAscii(RTL_CONSTASCII_STRINGPARAM(";" CNT_IMAP_URL_UID_PREFIX));
	aMesgURL += String::CreateFromInt64(nMesgUID);
	return aMesgURL;
}

//============================================================================
bool CntIMAPURL::createMesgURL(String const & rAcntURL,
							   String const & rMboxPath,
							   sal_uInt32 nMesgUID, String & rMesgURL)
{
	String aTheMesgURL(rAcntURL);

	sal_Unicode const * p = rMboxPath.GetBuffer();
	sal_Unicode const * pEnd = p + rMboxPath.Len();
	while (p != pEnd)
	{
		sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
		if (mustEncode(MBOX_SEGMENT, nUTF32) && nUTF32 != '/')
			INetURLObject::appendUCS4Escape(aTheMesgURL, '%', nUTF32);
		else
			aTheMesgURL += sal_Unicode(nUTF32);
	}

	aTheMesgURL.
		AppendAscii(RTL_CONSTASCII_STRINGPARAM(";" CNT_IMAP_URL_UID_PREFIX));
	aTheMesgURL += String::CreateFromInt64(nMesgUID);

	return makeCanonicURL(aTheMesgURL, rMesgURL);
}

//============================================================================
//
//  CntIMAPUTF7
//
//============================================================================

namespace unnamed_chaos_imapurl {

bool decode(sal_Char const *& rBegin, sal_Char const * pEnd, String * pTarget,
			bool bToURI)
{
	if (rBegin != pEnd && sal_uChar(*rBegin) == '-')
	{
		if (pTarget)
			*pTarget += '&';
		++rBegin;
		return true;
	}

	bool bSuccessful = false;
	sal_Unicode cUTF16Char;
	sal_Unicode cHighSurrogate = 0;
	int nFilled = 0;
	for (sal_Char const * p = rBegin; p != pEnd;)
	{
		static sal_uInt8 const aModifiedBase64[256]
			= { 65, 65, 65, 65, 65, 65, 65, 65,
				65, 65, 65, 65, 65, 65, 65, 65,
				65, 65, 65, 65, 65, 65, 65, 65,
				65, 65, 65, 65, 65, 65, 65, 65,
				65, 65, 65, 65, 65, 65, 65, 65, //  !"#$%&'
				65, 65, 65, 62, 63, 64, 65, 65, // ()*+,-./
				52, 53, 54, 55, 56, 57, 58, 59, // 01234567
				60, 61, 65, 65, 65, 65, 65, 65, // 89:;<=>?
				65,  0,  1,  2,  3,  4,  5,  6, // @ABCDEFG
				 7,  8,  9, 10, 11, 12, 13, 14, // HIJKLMNO
				15, 16, 17, 18, 19, 20, 21, 22, // PQRSTUVW
				23, 24, 25, 65, 65, 65, 65, 65, // XYZ[\]^_
				65, 26, 27, 28, 29, 30, 31, 32, // `abcdefg
				33, 34, 35, 36, 37, 38, 39, 40, // hijklmno
				41, 42, 43, 44, 45, 46, 47, 48, // pqrstuvw
				49, 50, 51, 65, 65, 65, 65, 65 }; // xyz{|}~

		sal_uInt32 nUCS4Char = sal_uChar(*p++);
		int nDigit = INetMIME::isUSASCII(nUCS4Char) ?
			             aModifiedBase64[nUCS4Char] : 65;
		switch (nDigit)
		{
			default: // valid character
				switch (nFilled)
				{
					case 2:
						cUTF16Char |= nDigit >> 2;
						if (INetMIME::isVisible(cUTF16Char))
							return false;
						goto write;

					case 5:
						cUTF16Char |= nDigit >> 4;
						if (INetMIME::isVisible(cUTF16Char))
							return false;
						goto write;

					case 7:
						cUTF16Char |= nDigit;
						if (INetMIME::isVisible(cUTF16Char))
							return false;
				write:
						if (pTarget)
							if (bToURI)
								if (cHighSurrogate == 0)
									if (INetMIME::isHighSurrogate(cUTF16Char))
										cHighSurrogate = cUTF16Char;
									else if (mustEncode(MBOX_SEGMENT,
														cUTF16Char))
										INetURLObject::appendUCS4Escape(
											*pTarget, '%', cUTF16Char);
									else
										*pTarget += cUTF16Char;
								else if (INetMIME::isLowSurrogate(cUTF16Char))
								{
									INetURLObject::appendUCS4Escape(
										*pTarget, '%',
										INetMIME::toUTF32(cHighSurrogate,
														  cUTF16Char));
									cHighSurrogate = 0;
								}
								else
								{
									INetURLObject::appendUCS4Escape(
										*pTarget, '%', cHighSurrogate);
									if (INetMIME::isHighSurrogate(cUTF16Char))
										cHighSurrogate = cUTF16Char;
									else
									{
										if (mustEncode(MBOX_SEGMENT,
													   cUTF16Char))
											INetURLObject::appendUCS4Escape(
												*pTarget, '%', cUTF16Char);
										else
											*pTarget += cUTF16Char;
										cHighSurrogate = 0;
									}
								}
							else
								*pTarget += cUTF16Char;
						break;
				}
				switch (nFilled)
				{
					case 0:
						cUTF16Char = nDigit << 10;
						++nFilled;
						break;

					case 1:
						cUTF16Char |= nDigit << 4;
						++nFilled;
						break;

					case 2:
						cUTF16Char = (nDigit & 3) << 14;
						++nFilled;
						break;

					case 3:
						cUTF16Char |= nDigit << 8;
						++nFilled;
						break;

					case 4:
						cUTF16Char |= nDigit << 2;
						++nFilled;
						break;

					case 5:
						cUTF16Char = (nDigit & 15) << 12;
						++nFilled;
						break;

					case 6:
						cUTF16Char |= nDigit << 6;
						++nFilled;
						break;

					case 7:
						nFilled = 0;
						break;
				}
				break;

			case 64: // terminating '-'
				switch (nFilled)
				{
					case 3:
					case 6:
						if (cUTF16Char != '\0')
							break;
					case 0:
						if (pTarget && cHighSurrogate != 0)
							INetURLObject::appendUCS4Escape(*pTarget, '%',
															cHighSurrogate);
						rBegin = p;
						return true;
				}
			case 65: // invalid character
				return false;
		}
	}
	return false;
}

}

//============================================================================
ByteString CntIMAPUTF7::translateToUTF7(String const & rSource)
{
	ModifiedUTF7Sink aSink;
	sal_Unicode const * p = rSource.GetBuffer();
	sal_Unicode const * pEnd = p + rSource.Len();
	while (p != pEnd)
	{
		sal_uInt32 nUTF32 = INetMIME::getUTF32Character(p, pEnd);
		aSink.writeUTF32(nUTF32);
	}
	return aSink.getBuffer();
}

//============================================================================
void CntIMAPUTF7::translateFromUTF7(sal_Char const * pBegin,
									sal_Char const * pEnd, bool bToURI,
									String & rTarget)
{
	// rSource is assumed to be in ISO-8859-1.  This implementation translates
	// to UTF-16 (malformed occurences of '&' are translated verbatim).

	while (pBegin != pEnd)
	{
		sal_Char c = *pBegin++;
		if (c == '&')
		{
			xub_StrLen nDecodedStart = rTarget.Len();
			if (!decode(pBegin, pEnd, &rTarget, bToURI))
			{
				rTarget.Erase(nDecodedStart);
				if (bToURI)
					rTarget.AppendAscii(RTL_CONSTASCII_STRINGPARAM("%26"));
				else
					rTarget += '&';
			}
		}
		else if (bToURI && mustEncode(MBOX_SEGMENT, sal_uChar(c)))
			INetURLObject::appendUCS4Escape(rTarget, '%', sal_uChar(c));
		else
			rTarget += sal_uChar(c);
	}
}

//============================================================================
//
//  ModifiedUTF7Sink
//
//============================================================================

void ModifiedUTF7Sink::write(bool bEndEncoding)
{
	static sal_Char const aModifiedBase64[64]
		= { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
			'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
			'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
			'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
			'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '+', ',' };

	switch (m_nFilled)
	{
		case -1:
			return;

		case 1:
			m_aBuffer += aModifiedBase64[m_nValue >> 18];
			m_aBuffer += aModifiedBase64[m_nValue >> 12 & 63];
			break;

		case 2:
			m_aBuffer += aModifiedBase64[m_nValue >> 18];
			m_aBuffer += aModifiedBase64[m_nValue >> 12 & 63];
			m_aBuffer += aModifiedBase64[m_nValue >> 6 & 63];
			break;

		case 3:
			m_aBuffer += aModifiedBase64[m_nValue >> 18];
			m_aBuffer += aModifiedBase64[m_nValue >> 12 & 63];
			m_aBuffer += aModifiedBase64[m_nValue >> 6 & 63];
			m_aBuffer += aModifiedBase64[m_nValue & 63];
			break;
	}
	if (bEndEncoding)
	{
		m_aBuffer += '-';
		m_nFilled = -1;
	}
	else
	{
		m_nFilled = 0;
		m_nValue = 0;
	}
}

//============================================================================
void ModifiedUTF7Sink::writeUTF16(sal_uInt32 nUTF16)
{
	switch (m_nFilled)
	{
		case 0:
			m_nValue = nUTF16 << 8;
			m_nFilled = 2;
			break;

		case 1:
			m_nValue |= nUTF16;
			m_nFilled = 3;
			write(false);
			break;

		case 2:
			m_nValue |= nUTF16 >> 8;
			m_nFilled = 3;
			write(false);
			m_nValue = (nUTF16 & 0xFF) << 16;
			m_nFilled = 1;
			break;
	}
}

//============================================================================
void ModifiedUTF7Sink::writeUTF32(sal_uInt32 nUTF32)
{
	if (INetMIME::isVisible(nUTF32))
	{
		writeVerbatim(sal_Char(nUTF32));
		if (nUTF32 == '&')
			m_aBuffer += '-';
	}
	else
	{
		if (m_nFilled == -1)
		{
			m_aBuffer += '&';
			m_nFilled = 0;
			m_nValue = 0;
		}
		if (nUTF32 <= 0xFFFF)
			writeUTF16(nUTF32);
		else
		{
			DBG_ASSERT(nUTF32 <= 0x10FFFF,
					   "ModifiedUTF7Sink::writeUTF32(): Bad char");
			nUTF32 -= 0x10000;
			writeUTF16(0xD800 | (nUTF32 >> 10));
			writeUTF16(0xDC00 | (nUTF32 & 0x3FF));
		}
	}
}

