/*************************************************************************
 *
 *  $RCSfile: treedata.cxx,v $
 *
 *  $Revision: 1.32 $
 *
 *  last change: $Author: jb $ $Date: 2001/11/14 17:06:13 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <stdio.h>

#include "treedata.hxx"

#ifndef _CONFIGMGR_TREECACHE_HXX_
#include "treecache.hxx"
#endif
#ifndef CONFIGMGR_LOADER_HXX
#include "loader.hxx"
#endif
#ifndef CONFIGMGR_TREE_NODEFACTORY_HXX
#include "treenodefactory.hxx"
#endif
#ifndef CONFIGMGR_TREE_CHANGEFACTORY_HXX
#include "treechangefactory.hxx"
#endif
#ifndef CONFIGMGR_CMTREEMODEL_HXX
#include "cmtreemodel.hxx"
#endif
#ifndef _CONFIGMGR_TREEACTIONS_HXX_
#include "treeactions.hxx"
#endif
#ifndef CONFIGMGR_UPDATEHELPER_HXX
#include "updatehelper.hxx"
#endif
#ifndef CONFIGMGR_DISPOSETIMER_HXX
#include "disposetimer.hxx"
#endif
#ifndef CONFIGMGR_NODEPART_HXX
#include "nodepart.hxx"
#endif
#ifndef INCLUDED_CONFIGMGR_MERGECHANGE_HXX
#include "mergechange.hxx"
#endif
#ifndef CONFIGMGR_CONFIGEXCEPT_HXX_
#include "configexcept.hxx"
#endif

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _RTL_STRING_HXX_
#include <rtl/string.hxx>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif

#ifndef _CONFIGMGR_TRACER_HXX_
#include "tracer.hxx"
#endif
#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagnose.h>
#endif

#ifndef INCLUDED_SET
#include <set>
#define INCLUDED_SET
#endif
#ifndef INCLUDED_DEQUE
#include <deque>
#define INCLUDED_DEQUE
#endif
#ifndef INCLUDED_VECTOR
#include <vector>
#define INCLUDED_VECTOR
#endif
#ifndef INCLUDED_IOSTREAM
#include <iostream>
#define INCLUDED_IOSTREAM
#endif
#ifndef INCLUDED_EXCEPTION
#include <exception>
#define INCLUDED_EXCEPTION
#endif

using namespace std;
using namespace rtl;
using namespace com::sun::star::uno;

namespace configmgr
{
    using namespace configuration;
	// =========================================================================
	struct ONodeIdCollector : public NodeAction
	{
		typedef std::set< rtl::OUString, comphelper::UStringLess > NodeIdSet;
		NodeIdSet ids;
		
		ONodeIdCollector() : ids() { }
		
		virtual void handle(ValueNode const& _rValueNode) { /* not interested in value nodes */ }
		virtual void handle(ISubtree const& _rSubtree) { ids.insert(_rSubtree.getId()); applyToChildren(_rSubtree); }
	};

	//==========================================================================
    static 
    inline 
    INode* findExistingChild(ISubtree* pCurrentParent, Path::Component const & _aName)
    {
        OSL_ASSERT(pCurrentParent);

	    INode * pChild = pCurrentParent->getChild(_aName.getName().toString());
			    
	    if (!pChild && !_aName.isSimpleName())
	    {
            pChild = pCurrentParent->getChild(_aName.toPathString());
            OSL_ENSURE(!pChild, "TreeData trouble: Existing node found only by composite name");
        }

        return pChild;
    }
	//==========================================================================
    static 
    inline 
    INode* findExistingChild(ISubtree* pCurrentParent, OUString const & _aName)
    {
        OSL_ASSERT(pCurrentParent);

	    INode * pChild = pCurrentParent->getChild(_aName );
			    
        return pChild;
    }
	//==========================================================================

    static sal_Int16 childLevel(sal_Int16 _nLevel)
    {
        OSL_ASSERT(0 > ITreeProvider::ALL_LEVELS);
        return (_nLevel > 0) ? _nLevel-1 : _nLevel;
    }

	//==========================================================================
    static bool deeper(sal_Int16 lhs, sal_Int16  rhs)
    {
        if (rhs == ITreeProvider::ALL_LEVELS) return false;
        if (lhs == ITreeProvider::ALL_LEVELS) return true;
        return lhs > rhs;
    }
	//==========================================================================
	//= OCompleteTree
	//==========================================================================
	/** completes the cache with missing subelements		
	*/
	class OCompleteTree : private NodeModification
	{
	public:
        typedef sal_Int16 Level;

		static bool complete(ISubtree& _rExistingSubtree, ISubtree& _rNewSubtree, 
                                Level _nNewDepth, Level _nDefaultDepth);
	private:
		OCompleteTree(ISubtree* pSubtree, Level _nParentLevel, Level _nParentDefaultLevel)
			:m_pCacheSubtree(pSubtree)
            ,m_nChildLevel( childLevel(_nParentLevel) )
            ,m_nDefaultLevel( childLevel(_nParentDefaultLevel) )
		{
			OSL_ENSURE(m_pCacheSubtree,"OCompleteTree: Need a tree to complete");

			OSL_ENSURE(m_pCacheSubtree->hasId(), "WARNING: Completing subtree without ID");
		}

		virtual void handle(ValueNode& _rNode);
		virtual void handle(ISubtree& _rSubtree);

	private:
		ISubtree* m_pCacheSubtree;
		sal_Int16 m_nChildLevel;
		sal_Int16 m_nDefaultLevel;
	};
	//----------------------------------------------------------------------
	void OCompleteTree::handle(ValueNode& _rNode) 
	{ 
		OSL_ASSERT(m_pCacheSubtree);

		OUString aNodeName = _rNode.getName();
		INode*   pChild	   = m_pCacheSubtree->getChild(aNodeName);
		// only not existing nodes are interesting other should be in the cache			
		if (!pChild)
		{
			std::auto_ptr<INode> pNewChild( _rNode.clone() );

			pChild = m_pCacheSubtree->addChild(pNewChild);				
		}
		else
		{
			OSL_ENSURE(pChild->ISA(ValueNode), "OCompleteTree: Node type mismatch");
            if (ValueNode* pValue = pChild->asValueNode())
            {
                if (m_nDefaultLevel != 0)
                    pValue->changeDefault(_rNode.getDefault());

                OSL_ENSURE(!m_nChildLevel || pValue->getValue() == _rNode.getValue(), 
                            "WARNING: Cached value differs from data being spliced");
            }
		}
	}
	//----------------------------------------------------------------------

	void OCompleteTree::handle(ISubtree& _rSubtree)
	{
		OSL_ASSERT(m_pCacheSubtree);

		OUString aNodeName = _rSubtree.getName();
		INode*   pChild	   = m_pCacheSubtree->getChild(aNodeName);

		// now we have different possibilites
		// a.) the node does not exist than clone the subtree and add it to the cache tree
		if (!pChild)
		{
			std::auto_ptr<INode> pNewChild( _rSubtree.clone() );

			ISubtree* pChildTree = pNewChild->asISubtree();

			OSL_ASSERT(pChildTree);
			OSL_ASSERT(!pChildTree->hasId()); // should be cleared by cloning a subtree
			
			OSL_ENSURE(_rSubtree.hasId() || m_pCacheSubtree->hasId(), "WARNING: OCompleteTree: completing a tree without ID");
			OUString const aTreeId = _rSubtree.hasId() ? _rSubtree.getId() : m_pCacheSubtree->getId();

			OIdPropagator::propagateIdToTree(aTreeId, *pChildTree);

			pChild = m_pCacheSubtree->addChild(pNewChild);	
			
			OSL_ASSERT(pChild && pChild->asISubtree());
			OSL_ENSURE(pChild->asISubtree()->getId()== aTreeId, "OCompleteTree: Tree lost ID while being added");
		}
		else
		{
			ISubtree* pSubTree = pChild->asISubtree();
			OSL_ENSURE(pSubTree, "OCompleteTree::handle : node must be a inner node!");

			// b.) the node does exist with level all or greater level -> nothing to do
			// c.) the node does exist but with smaller level
			if (pSubTree)
			{
				complete(*pSubTree, _rSubtree, m_nChildLevel, m_nDefaultLevel); 

				OSL_ENSURE(pSubTree->hasId(), "OCompleteTree: Completed Tree has no ID");
			}
			// else throw something (?)
		}
	}		
	//----------------------------------------------------------------------
	bool OCompleteTree::complete(ISubtree& _rExistingSubtree, ISubtree& _rNewSubtree, 
                                    Level _nNewDepth, Level _nDefaultDepth)
	{

		OSL_ENSURE(_rNewSubtree.hasId(),"WARNING: Completing subtree without providing an ID");

		Level const nExistingLevels     = _rExistingSubtree.getLevel();
		Level const nExistingDefaults   = _rExistingSubtree.getDefaultsLevel();
		if (!_rExistingSubtree.hasId()) 
		{
			// if it has levels, it surely was really loaded (and therefore has an ID)
			OSL_ENSURE(nExistingLevels   == 0,"ERROR: Dummy subtree has nonzero levels");
			OSL_ENSURE(nExistingDefaults == 0,"ERROR: Dummy subtree has nonzero default levels");
			OSL_ENSURE(_nNewDepth != 0,"ERROR: Merging default for data that isn't loaded");

			// the tree wasn't loaded before
			OSL_ENSURE(!_rExistingSubtree.getElementTemplateName().getLength(),
						"WARNING: Dummy tree has child template");
			OSL_ENSURE(!_rExistingSubtree.getElementTemplateName().getLength() ||  _rNewSubtree.getElementTemplateName().getLength(),
						"ERROR: Dummy tree with child template replaced by node without child template");
			OSL_ENSURE(_rExistingSubtree.getName() ==  _rNewSubtree.getName(),
						"ERROR: Dummy tree renamed by completion (?!)");

			// bad hack: abuse of assignment operator to update a slice of _rExistingSubtree
			_rExistingSubtree = _rNewSubtree;

			// id is copied by assignment (but not by copy ctor)
			OSL_ASSERT(_rExistingSubtree.getId() ==  _rNewSubtree.getId());
			// _rExistingSubtree.setId(_rNewSubtree.getId();
		}
		else
		{
			OSL_ENSURE(_rExistingSubtree.getName() == _rNewSubtree.getName(),
						"ERROR: Existing tree replaced by node with different name");

			OSL_ENSURE(_rExistingSubtree.getElementTemplateName() == _rNewSubtree.getElementTemplateName(),
						"ERROR: Existing tree replaced by node with different child template");
		}

		bool bAddingData        = deeper(_nNewDepth,nExistingLevels);
		bool bAddingDefaults    = deeper(_nDefaultDepth,nExistingDefaults);
        bool bAdding = bAddingData || bAddingDefaults;

		if (bAdding)
		{
			// now do it 
			OCompleteTree aCompleter(&_rExistingSubtree, _nNewDepth, _nDefaultDepth);
			aCompleter.applyToChildren(_rNewSubtree);
			//_rExistingSubtree.setLevel(_nNewDepth); // don't do it here, as this is recursive
		}
		// else nothing to do (well, we should check/refresh - at least in DEBUG)

		OSL_ENSURE(_rExistingSubtree.hasId(), "OCompleteTree: Completed Tree lost ID");

		return bAdding; // true only if we actually did something
	}
// ---------------------------- Client Acquire helper ----------------------------

	struct ModuleClientRef
	{
		ModuleTreeRef xModule;

		ModuleClientRef(ModuleTreeRef const& _xModule) 
		: xModule(_xModule) 
		{
			if (xModule.isValid()) 
				xModule->clientAcquire();
		}

		~ModuleClientRef() // release the client reference unless it was 'kept'
		{
			if (xModule.isValid()) 
				xModule->clientRelease();
		}

		void rebind(ModuleTreeRef const& _xModule) 
		{
			if (_xModule.isValid()) 
				_xModule->clientAcquire();
			if (xModule.isValid()) 
				xModule->clientRelease();
			xModule = _xModule;
		}

		/// release the contained module so that the client reference will be kept active
		void keep() 
		{
			xModule.unbind();
			OSL_ASSERT(!xModule.isValid());
		}

		/// return the contained module so that the client reference will be kept active
		ModuleTreeRef keepModule() 
		{
			ModuleTreeRef xRet = xModule;
			this->keep();
			return xRet;
		}
	private:
		ModuleClientRef(ModuleClientRef const& );
		ModuleClientRef& operator=(ModuleClientRef const& );

	};

// -----------------------------------------------------------------------------
	static inline ModuleTree::Name implExtractModuleName(AbsolutePath const& aConfigPath)
	{
		return aConfigPath.getModuleName();
	}


// -----------------------------------------------------------------------------
// class ModuleTree
// -----------------------------------------------------------------------------
	ModuleTree::ModuleTree(std::auto_ptr<ISubtree> pModuleRoot)
		: m_pModuleRoot(pModuleRoot)
	{
	}

// -----------------------------------------------------------------------------
	ModuleTree::Name ModuleTree::getModuleName() const
	{
		OSL_ENSURE( m_pModuleRoot.get(), "Cannot get a subtree for a dataless module");
        return makeName( m_pModuleRoot->getName(), Name::NoValidate() );
	}

// -----------------------------------------------------------------------------
	ISubtree* ModuleTree::getSubtree(const Location& aConfigName) const
	{
		ISubtree* pSubtree = m_pModuleRoot.get();
		OSL_ENSURE( pSubtree, "Cannot get a subtree for a dataless module");

		Location::Iterator it = aConfigName.begin();
		OSL_ENSURE( it != aConfigName.end(), "Empty Path can't match any module");

		OSL_ENSURE( pSubtree && pSubtree->getName() == it->toPathString(), "Module part in config path does not match selected ModuleTree");

		while(pSubtree != NULL && ++it != aConfigName.end())
		{
			INode* pNode = findExistingChild(pSubtree, *it);

			pSubtree = pNode ? pNode->asISubtree() : 0;
		}
		return pSubtree;
	}
// -----------------------------------------------------------------------------
	INode* ModuleTree::getNode(const Location& aConfigName) const
	{
		INode*		pNode	 = m_pModuleRoot.get();
		OSL_ENSURE( pNode, "Cannot get a node from a dataless module");

		Location::Iterator it = aConfigName.begin();
		OSL_ENSURE( it != aConfigName.end(), "Empty Path can't match any module");

		OSL_ENSURE( pNode && pNode->getName() == it->toPathString(), "Module part in config path does not match selected ModuleTree");

		while(pNode != NULL && ++it != aConfigName.end())
		{
			ISubtree* pSubtree = pNode->asISubtree();

			pNode = pSubtree ? findExistingChild(pSubtree, *it) : 0;
		}
		return pNode;
	}
// -----------------------------------------------------------------------------
	ModuleTreeRef ModuleTree::makeModuleTree(const Location& _rLocation, std::auto_ptr<ISubtree> _pSubtree, 
                                             sal_Int16 _nLevels, sal_Int16 _nDefaultLevels) CFG_UNO_THROW_RTE(  )
	{
		if (_rLocation.isRoot())
		{
			OSL_ENSURE(false, "Cannot make a module tree without a path");
			return NULL;
		}

		std::auto_ptr<ISubtree> pNewRoot;

		if (_pSubtree.get() && _rLocation.getDepth() == 1) // directly inserting the root
		{
			OSL_ENSURE(_rLocation.getLocalName().toPathString() == _pSubtree->getName(), "Path does not match data being inserted");
			OSL_ENSURE(_rLocation.getModuleName().toString() == _pSubtree->getName(), "Path does not match data being inserted");

			pNewRoot = _pSubtree;
			pNewRoot->setLevels(_nLevels,_nDefaultLevels);
			OSL_ASSERT( !_pSubtree.get() );	// already used
		}

		else // need a dummy at root, will add to that later
		{
			Name sModuleName = implExtractModuleName(_rLocation);
            pNewRoot = OTreeNodeFactory::createDummyTree(sModuleName, Name());	// module is no set
			OSL_ASSERT(0 == pNewRoot->getLevel());	// which means "we know nothing about any children yet" 
		}

		OSL_ASSERT( pNewRoot.get() );
		ModuleTreeRef xResult = new ModuleTree(pNewRoot);

		if (_pSubtree.get())
		{
			//OSL_ENSURE(_rLocation.localName() == _pSubtree->getName(), "Path does not match data being inserted");
			xResult->addSubtree(_rLocation,_pSubtree,_nLevels,_nDefaultLevels);
		}

		return xResult;
	}
// -----------------------------------------------------------------------------

	ISubtree* ModuleTree::addSubtree(const Location& _rLocation, std::auto_ptr<ISubtree> _pSubtree, 
                                     sal_Int16 _nLevels, sal_Int16 _nDefaultLevels) CFG_UNO_THROW_RTE(  )
	{
		OSL_ENSURE(_pSubtree.get(), "Tree::addSubtree : inserting NULL subtree !");
		OSL_ENSURE(_nLevels != 0 || _nDefaultLevels != 0, "Tree::addSubtree : invalid level count !");
		// there was a time where 0 meant "all levels", but we changed the according enum in ITReeProvider
		// so that it is -1 now. Since this time, 0 isn't valid as level depth anymore !

		// if we notice that a subtree already exists we have to verify 
		// if the subtree is already complete populated
		
		// do we already have the subtree in cache, but not completely populated?
		// so find the subtree
		ISubtree* pExistingEntry = m_pModuleRoot.get();
		OSL_ENSURE( pExistingEntry, "Cannot add a subtree to a dataless module");

		// It points to the next path component to resolve
		Location::Iterator it = _rLocation.begin();
		OSL_ENSURE( it != _rLocation.end(), "Empty Path can't match any module");

		// now find the closest existing entry
		OSL_ENSURE( pExistingEntry->getName() == it->getName().toString(), "Module part in config path does not match selected ModuleTree");

		while(++it != _rLocation.end())
		{
			INode* pNode = findExistingChild(pExistingEntry, *it);
			if (pNode == NULL) break;

			ISubtree* pSubtree = pNode->asISubtree();
			if (pSubtree == NULL) 
			{
				OSL_ENSURE( false, "Needed subtree already exists as non-Subtree !?");
				throw uno::RuntimeException(OUString::createFromAscii(
						"Inconsistent configuration data: Needed subtree already exists as non-Subtree !?"), 0);
				break;
			}

			pExistingEntry = pSubtree;
		}
		OSL_ASSERT(pExistingEntry);

		// resolved it all - subtree already exists ?
		if (it == _rLocation.end()) 
		{
			// we already have a tree, so we fill the subtree with the neccessary fragments (if it isn't deep enough already)
			if (OCompleteTree::complete(*pExistingEntry, *_pSubtree, _nLevels, _nDefaultLevels))
			{
				// really did something - now adjust the levels
				pExistingEntry->setLevels(_nLevels, _nDefaultLevels);
			}
			else
			{
				// did nothing - was there already
				OSL_ENSURE(false, "ModuleTree::addSubtree : node already deep enough in cache, why adding it again? !");
			}

			OSL_ENSURE(pExistingEntry->hasId(), "ModuleTree::addSubtree : tree has no ID after completing");

			// release the ownership and delete that unnecessary tree
			// _pSubtree.reset(); - done by destructor

			return pExistingEntry;			
		}
		
		// else insert the complete subtree because it is not part of the cache yet
		else 
		{
			Location::Iterator itFinal = _rLocation.end();
			--itFinal;

			OSL_ENSURE( itFinal->getName().toString() == _pSubtree->getName(), "Data being inserted does not match the path" );

			// First make the context for the insertion
			while (it != itFinal)
			{
				OSL_ASSERT( it+1 != _rLocation.end());	// how did we get to the next-to-last (itFinal)

                // get the properties of the node
				Name const aNodeName = it->getName();
                Name const aElementTypeName = (it+1)->getTypeName();

				// we need a new 'filler' node
				std::auto_ptr<ISubtree> pNewTree = OTreeNodeFactory::createDummyTree(aNodeName,aElementTypeName);
				std::auto_ptr<INode>    pNewChild( pNewTree.release() );

				ISubtree* pInsertedChild = pExistingEntry->addChild(pNewChild)->asISubtree();
				OSL_ASSERT( pInsertedChild );					// we just inserted one, so there it is
				OSL_ASSERT(0 == pInsertedChild->getLevel());	// which means "we know nothing about any children"

				// advance the existing part
				pExistingEntry = pInsertedChild;
				++it;

				OSL_ASSERT( it != _rLocation.end());			// how did we get past the next-to-last (itFinal)
			}

			// we need a new 'filler' node
			std::auto_ptr<INode> pNodeToInsert( _pSubtree.release() );

			ISubtree* pInsertedTree = pExistingEntry->addChild(pNodeToInsert)->asISubtree();
			OSL_ASSERT( pInsertedTree );					// we just inserted one, so there it is

			pInsertedTree->setLevels(_nLevels, _nDefaultLevels);
			OSL_ENSURE(pInsertedTree->hasId(), "ModuleTree::addSubtree : tree has no ID after inserting");

			return pInsertedTree;
		}
	}

	// -----------------------------------------------------------------------------

	void ModuleTree::addPending(const Location& _rLocation, const SubtreeChange* _pSubtreeChange) CFG_UNO_THROW_RTE(  )
	{
        OSL_PRECOND(!_rLocation.isRoot(),"Pending change cannot be located at root");
        OSL_PRECOND(_rLocation.getModuleName() == this->getModuleName(),"Pending change without location");

        OSL_PRECOND(_pSubtreeChange,"Pending change is NULL");
        OSL_PRECOND(_pSubtreeChange->getNodeName() == _rLocation.getLocalName().getName().toString(),
                    "Path to pending change does not match change name");

        // first make the _pSubtreeChange a full tree starting at the module root
		auto_ptr<SubtreeChange> pRootChange;
		SubtreeChange *pExistingEntry = NULL;

		Location::Iterator last = _rLocation.end();

        OSL_ASSERT(last != _rLocation.begin());
		--last;

		for (Location::Iterator it = _rLocation.begin();
			 it != last;
			 ++it)
		{
            OSL_ASSERT( it   != _rLocation.end());
            OSL_ASSERT( it+1 != _rLocation.end());
			// We need to create a new SubtreeChange
			Name const aChangeName      = it->getName();
            Name const aElementTypeName = (it+1)->getTypeName();

            auto_ptr<SubtreeChange> pNewChange =
                        OTreeChangeFactory::createDummyChange(aChangeName, aElementTypeName);

			if (pExistingEntry == NULL)
			{
                OSL_ASSERT(pRootChange.get() == NULL); 

				pRootChange     = pNewChange;
				pExistingEntry  = pRootChange.get();
			}
			else
			{
                OSL_ASSERT(pRootChange.get() != NULL); 

				pExistingEntry->addChange(base_ptr(pNewChange));

				Change* pChange = pExistingEntry->getChange(aChangeName.toString());
				pExistingEntry = static_cast<SubtreeChange*>(pChange);

                OSL_ENSURE(pChange && pChange->ISA(SubtreeChange), "ERROR: Cannot recover change just added");
			}
		}
		
		auto_ptr<SubtreeChange> pAddedChange( static_cast<SubtreeChange*>(_pSubtreeChange->clone()) );
		
		if (_rLocation.getDepth() > 1)
		{
            OSL_ASSERT(pRootChange.get() != NULL && pExistingEntry != NULL); 

			// the _pSubtreeChange did not start at root, so add its clone to the built dummies
			pExistingEntry->addChange(base_ptr(pAddedChange));
		}
		else
		{
            OSL_ASSERT(pRootChange.get() == NULL && pExistingEntry == NULL); 

			// the _pSubtreeChange starts at root, so just reset pRootChange
			pRootChange = pAddedChange;
		}
        OSL_ASSERT(pRootChange.get() != NULL); 

		if (m_pPending.get() == NULL)
		{
			// no merge is need, because the existing pending changes are empty
			m_pPending = pRootChange;
		}
		else
		{
            try
            {
			    // We need to merge the new rebased changes into the m_pPending 
			    combineUpdates(*pRootChange,*m_pPending);
            }
            catch (configuration::Exception& e)
            {
			    OUString sMessage(RTL_CONSTASCII_USTRINGPARAM("Update cache for module: Could not add pending changes at"));

			    sMessage += _rLocation.toString();

			    sMessage += OUString(RTL_CONSTASCII_USTRINGPARAM(". Internal Exception:")) + e.message();

			    throw uno::RuntimeException(sMessage,0);
            }
		}

        OSL_POSTCOND(m_pPending.get() != NULL, "Could not insert new pending changes");
	}
	
// ---------------------------- Tree implementation ----------------------------
	Tree::Tree()
	{
	}
	
	Tree::~Tree()
	{
		CFG_TRACE_INFO("Discarding Tree Data (Still holding %d module trees)", int (m_aModules.size()) );
	}
// -----------------------------------------------------------------------------

	ModuleTree::Name Tree::extractModuleName(AbsolutePath const& aPath)
	{
		return implExtractModuleName(aPath);
	}
// -----------------------------------------------------------------------------
	inline
    ModuleTreeRef Tree::implGetModuleTree(const ModuleTree::Name& aModuleName ) const
	{
		OSL_ASSERT(!aModuleName.isEmpty());

		ModuleList::const_iterator it = m_aModules.find(aModuleName);

		return it!=m_aModules.end() ? it->second : ModuleTreeRef();
	}
// -----------------------------------------------------------------------------
	inline
	ModuleTreeRef Tree::implGetModuleTree(const AbsolutePath&  aPath) const
	{
		ModuleTree::Name aModuleName = implExtractModuleName(aPath);
		return implGetModuleTree(aModuleName);
	}
// -----------------------------------------------------------------------------
	ISubtree* Tree::implGetSubtree(const AbsolutePath& aComponentName ) const
	{
		ModuleTreeRef xModule = implGetModuleTree(aComponentName);
		
		if ( !xModule.isValid() )
			return 0;

		ISubtree* pSubtree = xModule->getSubtree(aComponentName);

		OSL_ENSURE( pSubtree == NULL || xModule->clientReferences() != 0 || xModule->hasPending(), 
					"WARNING: returning subtree from module without clients\n" );
#ifdef CFG_ENABLE_TRACING
		if( pSubtree != NULL && xModule->clientReferences() == 0 && !xModule->hasPending()) 
		{
			CFG_TRACE_WARNING("Tree data: returning subtree %s from module without clients", OUSTRING2ASCII( aComponentName.toString() ) );
		}
#endif // CFG_ENABLE_TRACING

		return pSubtree;	
	}
// -----------------------------------------------------------------------------
	INode* Tree::implGetNode(const AbsolutePath& aComponentName ) const
	{
		ModuleTreeRef xModule = implGetModuleTree(aComponentName);
		
		if ( !xModule.isValid() )
			return 0;

		INode* pNode = xModule->getNode(aComponentName);

		OSL_ENSURE( pNode == NULL || xModule->clientReferences() != 0, 
					"WARNING: returning node from module without clients\n" );
	#ifdef CFG_ENABLE_TRACING
		if( pNode != NULL && xModule->clientReferences() == 0) 
		{
			CFG_TRACE_WARNING("Tree data: returning node %s from module without clients", OUSTRING2ASCII( aComponentName.toString() ) );
		}
#endif // CFG_ENABLE_TRACING

		return pNode;	
	}
// -----------------------------------------------------------------------------

	ModuleTreeRef Tree::implNewModuleTree(const AbsolutePath& _rLocation, std::auto_ptr<ISubtree> _pSubtree, sal_Int16 _nLevels, sal_Int16 _nDefaultLevels) CFG_UNO_THROW_RTE(  )
	{
		ModuleTreeRef aNewModule = ModuleTree::makeModuleTree(_rLocation,_pSubtree, _nLevels, _nDefaultLevels);

		Name aModuleName = implExtractModuleName(_rLocation);
		OSL_ENSURE(m_aModules.find(aModuleName) == m_aModules.end(), "Module already present in Tree");
		m_aModules[aModuleName] = aNewModule;
		
		CFG_TRACE_INFO("Tree Data: Created new module tree for module %s", OUSTRING2ASCII(aModuleName.toString()) );

		return aNewModule;
	}
// -----------------------------------------------------------------------------

	ISubtree* Tree::addSubtree(const AbsolutePath& _rLocation, std::auto_ptr<ISubtree> _pSubtree, sal_Int16 _nLevels, sal_Int16 _nDefaultLevels) CFG_UNO_THROW_RTE(  )
	{
		OSL_ENSURE(_nLevels != 0, "Tree::addSubtree : invalid level count !");
		// there was a time where 0 meant "all levels", but we changed the according enum in ITReeProvider
		// so that it is -1 now. Since this time, 0 isn't valid as level depth anymore !

		ISubtree* pResult = 0;

		// do we already have the module in cache ?
		ModuleTreeRef xModule( implGetModuleTree(_rLocation) );

		ModuleClientRef aClientRef( xModule );

		if (xModule.isEmpty())
		{
			xModule = implNewModuleTree(_rLocation, _pSubtree, _nLevels, _nDefaultLevels);
			OSL_ASSERT(xModule.isValid());
			aClientRef.rebind(xModule);

			pResult = xModule->getSubtree(_rLocation);
		}

		else
		{
			pResult = xModule->addSubtree(_rLocation, _pSubtree, _nLevels, _nDefaultLevels);
		}
		OSL_ASSERT(pResult);

		if (pResult) aClientRef.keep();

		return pResult;
	}
	// -----------------------------------------------------------------------------
	void Tree::addPending(const AbsolutePath& _rLocation, const SubtreeChange* _pSubtreeChange) CFG_UNO_THROW_RTE(  )
	{
		// do we already have the module in cache ?
		ModuleTreeRef xModule( implGetModuleTree(_rLocation) );
		if (!xModule.isEmpty())
		{
			xModule->addPending(_rLocation, _pSubtreeChange);
		}
		else
		{
			OSL_ENSURE(false, "We can only change Nodes if we already know something about it. So this must be a big bug.");
		}
	}
// -----------------------------------------------------------------------------
    auto_ptr<SubtreeChange> Tree::getPending(ModuleTree::Name const& _rLocation)
	{
		ModuleTreeRef xModule( implGetModuleTree(_rLocation) );
		if (!xModule.isEmpty())
		{
			if (xModule->hasPending())
				return xModule->releasePending();
			else
				return (auto_ptr<SubtreeChange>) NULL;
		}
		else
		{
			OSL_ENSURE(false, "We can only get Nodes if we already know something about it.");
		}
		return (auto_ptr<SubtreeChange>) NULL;
	}

// -----------------------------------------------------------------------------
	ISubtree* Tree::acquireSubtreeWithDepth( AbsolutePath const& aConfigName, 
                                            sal_Int16 _nMinLevels, sal_Int16 _nMinDefaultLevels )
	{
		// looking for the requested subtree: results are
		// a.) tree not found, tree isn't already cached
		// b.) tree found, but not complete
		// c.) tree found and complete

		// Build SearchName

		ModuleClientRef aClientRef(implGetModuleTree(aConfigName));

		ISubtree* pSubtree = implGetSubtree(aConfigName);

		// if the tree is not complete: ALL_LEVELS != pSubtree->getLevel()
		// or not fetched with all requested levels, we have to refetch
		if (pSubtree)
		{
			sal_Int16 nActualLevels         = pSubtree->getLevel();
			sal_Int16 nActualDefaultLevels  = pSubtree->getDefaultsLevel();

			bool bIncomplete = deeper(_nMinLevels,nActualLevels) || 			
			                   deeper(_nMinDefaultLevels,nActualDefaultLevels);			
			
            if (bIncomplete)
				pSubtree = NULL;
		}

		if (pSubtree) aClientRef.keep();

		return pSubtree;
	}
// -----------------------------------------------------------------------------
	vos::ORefCount::RefCount Tree::releaseSubtree( AbsolutePath const& aConfigName )
	{
		ModuleTreeRef xModule( implGetModuleTree(aConfigName) );

		const vos::ORefCount::RefCount c_nErrorCount = -1;
		
		OSL_ENSURE( xModule.isValid(), "ERROR: Releasing non-existent subtree");
		if ( xModule.isEmpty()) return c_nErrorCount;
		
		OSL_ENSURE( xModule->clientReferences() > 0, "ERROR: Releasing non-referenced subtree");
		return xModule->clientRelease();
	}

// -----------------------------------------------------------------------------

	void Tree::mergeChanges(const AbsolutePath& _rRootLocation, SubtreeChange& _aChangeRoot) CFG_UNO_THROW_RTE(  )
	{
		// request the subtree, atleast one level must exist!
		ISubtree *pSubtree = implGetSubtree(_rRootLocation);
		if (pSubtree)
		{
            applyUpdateToTree(_aChangeRoot,*pSubtree);
		}
		else
		{
			OSL_ENSURE(false, "Tree::updateTree called for non-existing tree");
			::rtl::OUString aStr(RTL_CONSTASCII_USTRINGPARAM("Tree: there is no Subtree for name:="));

			aStr += _rRootLocation.toString();

			throw uno::RuntimeException(aStr,0);
		}
	}
		// Better:
		// ISubtree *pCloneTree = m_pRoot->clone();
		// ISubtree *pSubtree = pCloneTree->requestSubtree(aTree.pathToRoot, ITreeProvider::ALL_LEVELS);
		// TreeUpdate aTreeUpdate(pSubtree);
		// aTreeUpdate.handle(aTree.root);
		// if (aTreeUpdate.isOk())
		// {
		//    delete m_pRoot;
		//    m_pRoot = pSubtree;
		// }

// -----------------------------------------------------------------------------
	ISubtree const* Tree::getSubtree( AbsolutePath const& aComponentName ) const
	{
		return implGetSubtree(aComponentName);
	}
// -----------------------------------------------------------------------------
	const INode* Tree::getNode(const AbsolutePath& _rPath)
	{
		return implGetNode(_rPath);
	}


//==========================================================================
//= TreeInfo
//==========================================================================

// -------------------------------------------------------------------------
// -------------------------------------------------------------------------

ISubtree * TreeInfo::addSubtree(const AbsolutePath& _rLoadedLocation, const AbsolutePath& _rRelLocation, std::auto_ptr<ISubtree> _pSubtree, 
                                sal_Int16 _nReliableLevels, sal_Int16 _nDefaultLevels) CFG_UNO_THROW_RTE(  )
{
	osl::MutexGuard aGuard( this->aMutex );

	CFG_TRACE_INFO("Tree Info: Adding subtree data for path '%s' (with depth = %d)", OUSTRING2ASCII(_rLoadedLocation.toString()), int(_nReliableLevels) );

	ISubtree* pSubtree = aTree.addSubtree(_rLoadedLocation, _pSubtree , _nReliableLevels, _nDefaultLevels);
	if (pSubtree)
	{
		m_aDeadModules.erase( Tree::extractModuleName(_rLoadedLocation) );
		CFG_TRACE_INFO_NI("- Data added successfully - returning Subtree");
	}
	else
		CFG_TRACE_INFO_NI("- Data not added - returning NULL");

	// the tree loaded isn't the desired subtree - it is one of the parents
	if (_rRelLocation.getDepth() != _rLoadedLocation.getDepth())
		pSubtree = acquireSubtreeWithDepth(_rRelLocation, _nReliableLevels, _nDefaultLevels);

    else
        OSL_ENSURE(matches(_rRelLocation,_rLoadedLocation),"ERROR: returning a different tree than requested");

	return pSubtree;
}
// -------------------------------------------------------------------------

ISubtree* TreeInfo::acquireSubtreeWithDepth( AbsolutePath const& rLocation, sal_Int16 _nMinLevels, sal_Int16 _nMinDefaultLevels )
{
	osl::MutexGuard aGuard( this->aMutex );

	CFG_TRACE_INFO("Tree Info: Requesting subtree data for path '%s' (required depth = %d, default-depth = %d)", OUSTRING2ASCII(rLocation.toString()), int(_nMinLevels), int(_nMinDefaultLevels) );

	ISubtree* pSubtree = aTree.acquireSubtreeWithDepth(rLocation, _nMinLevels, _nMinDefaultLevels);

	if (pSubtree)
	{
		m_aDeadModules.erase( Tree::extractModuleName(rLocation) );
		CFG_TRACE_INFO_NI("- Data is available - returning Subtree");
	}
	else
		CFG_TRACE_INFO_NI("- Data is not available - returning NULL");

	return pSubtree;
}
// -------------------------------------------------------------------------

vos::ORefCount::RefCount TreeInfo::releaseSubtree( AbsolutePath const& rLocation )
{
	osl::MutexGuard aGuard( this->aMutex );

	CFG_TRACE_INFO("Tree Info: Releasing subtree data for path '%s'", OUSTRING2ASCII(rLocation.toString()) );

	vos::ORefCount::RefCount nRet = aTree.releaseSubtree(rLocation);
	if (nRet == 0)
	{
		m_aDeadModules[ Tree::extractModuleName(rLocation) ] = m_rDisposer.getCleanupTime();
		CFG_TRACE_INFO_NI("- Last reference released - marking data for cleanup");
	}

	return nRet;
}
// -----------------------------------------------------------------------------

void TreeInfo::updateTree( TreeChangeList& aTreeChanges) CFG_UNO_THROW_RTE(  )
{
	osl::MutexGuard aGuard( this->aMutex ); // needed to protect the map access in the Tree

    AbsolutePath aRootLocation = aTreeChanges.getRootNodePath();

	CFG_TRACE_INFO("Tree Info: Merging changes into subtree '%s'", OUSTRING2ASCII(aRootLocation.toString()) );

	this->aTree.mergeChanges(aRootLocation, aTreeChanges.root);
}

// -----------------------------------------------------------------------------
vos::ORef< OTreeLoader > TreeInfo::getNewLoaderWithoutPending(AbsolutePath const& aComponentName, 
											 sal_Int16 nMinLevels, 
											 vos::ORef< OOptions > const& _xOptions, 
											 IConfigSession* pSession)
{
	vos::ORef< OTreeLoader > xLoader;
	osl::MutexGuard aGuard( aMutex ); 
	xLoader = new OTreeLoader(pSession, aComponentName, _xOptions, nMinLevels);
	return xLoader;
}
// -----------------------------------------------------------------------------
vos::ORef< OTreeLoader > TreeInfo::getLoader(AbsolutePath const& aComponentName, 
											 sal_Int16 nMinLevels, 
											 vos::ORef< OOptions > const& _xOptions, 
											 IConfigSession* pSession,
											 sal_Bool bPreload)
{
	vos::ORef< OTreeLoader > xLoader;
	osl::MutexGuard aGuard( aMutex ); 
	
	// do we already have a pending load?

	Name sModuleName = aComponentName.getModuleName();
	PendingLoadList::iterator i = m_aPendingLoads.find(sModuleName);
	if (i != m_aPendingLoads.end())
		xLoader = (*i).second;
	else
	{
		xLoader = new OTreeLoader(pSession, aComponentName, _xOptions, nMinLevels);
		if (bPreload)
			m_aPendingLoads[sModuleName] = xLoader;		
	}
	return xLoader;
}

// -----------------------------------------------------------------------------
/// release the Loader if necessary

void TreeInfo::releaseLoader(vos::ORef< OTreeLoader > const& _xLoader)
{
	osl::MutexGuard aGuard( aMutex ); 

	Name sModuleName = _xLoader->getComponentName();
	PendingLoadList::iterator i = m_aPendingLoads.find(sModuleName);
	if (i != m_aPendingLoads.end())
		m_aPendingLoads.erase(i);
}

// -----------------------------------------------------------------------------
void TreeInfo::addPending(TreeChangeList const & aTreeChanges) CFG_UNO_THROW_RTE(  )
{
	// NICE: m_pPending[_rLocation] += pSubtreeChange;
	
	// SubtreeMerger aMerger(m_pPending, _rLocation);
	// aMerger.applyToChange(pSubtreeChange->clone());
	osl::MutexGuard aGuard( this->aMutex ); // needed to protect the map access in the Tree
	CFG_TRACE_INFO("Tree Info: Adding pending changes for subtree '%s'", OUSTRING2ASCII(aTreeChanges.getRootNodePath().toString()) );
	
	aTree.addPending(aTreeChanges.getRootNodePath(), &aTreeChanges.root);
}

// -----------------------------------------------------------------------------
auto_ptr<SubtreeChange> TreeInfo::getPending(ModuleTree::Name const& aComponentName)
{
	osl::MutexGuard aGuard( this->aMutex ); // needed to protect the map access in the Tree
	CFG_TRACE_INFO("Tree Info: get changes from subtree '%s'", OUSTRING2ASCII(aComponentName.toString()) );
	return aTree.getPending(aComponentName);
}

// -------------------------------------------------------------------------

ISubtree const*	TreeInfo::getSubtree( AbsolutePath const& aComponentName )
{
	osl::MutexGuard aGuard( this->aMutex ); // needed to protect the map access in the Tree

	return this->aTree.getSubtree(aComponentName);
}
// -------------------------------------------------------------------------

void TreeInfo::clearTree(DisposeList& _rList) CFG_NOTHROW()
{
	osl::MutexGuard aGuard( this->aMutex );
	
	CFG_TRACE_INFO("Tree Info: Removing all module trees for cleanup" );

	typedef Tree::ModuleList ModuleList;

	ModuleList& rModules = this->aTree.accessModuleList();

	for(ModuleList::iterator it = rModules.begin();
		it != rModules.end();
		++it)
	{
		if (it->second.isValid()) 
			_rList.push_back(it->second);
	}

	rModules.clear();
	m_aDeadModules.clear();
}
// -------------------------------------------------------------------------

bool TreeInfo::isEmpty()
{
	osl::MutexGuard aGuard( this->aMutex );
	
	Tree::ModuleList& rModules = this->aTree.accessModuleList();

	bool bRet = rModules.empty();

	if (bRet) // while we are at it - clean up
		m_aDeadModules.empty(); 

	return bRet;
}
// -------------------------------------------------------------------------

TimeStamp TreeInfo::runDisposer(DisposeList& _rList, TimeStamp const& aLimitTime)
{
	TimeStamp aRetTime = TimeStamp::never();

	osl::MutexGuard aGuard( this->aMutex );

	CFG_TRACE_INFO("Tree Info: Collecting disposable module trees for cleanup" );

	typedef Tree::ModuleList ModuleList;

	ModuleList& rActiveModules = this->aTree.accessModuleList();

	DeadModuleList::iterator it = m_aDeadModules.begin();

	while (it != m_aDeadModules.end())
	{
		DeadModuleList::iterator current = it;
		// increment here, as we may later erase(current)
		++it;

#if defined _DEBUG || defined _DBG_UTIL || defined CFG_TRACE_ENABLE
		OUString sCurrentName( current->first.toString() );
#endif
		if (current->second <= aLimitTime)
		{
			ModuleList::iterator itModule = rActiveModules.find( current->first );

			if (itModule != rActiveModules.end())
			{
				ModuleTreeRef xModule = itModule->second;

				bool bHandled = false;

				if (xModule.isEmpty())
				{
					CFG_TRACE_ERROR_NI("- Unexpected: Module '%s' is NULL in active module list", OUSTRING2ASCII(sCurrentName) );
					bHandled = true;
				}
				else if (xModule->clientReferences() != 0)// at least in temporary use
				{
					OSL_ENSURE( false, "Referenced entry in dead module list");

					CFG_TRACE_WARNING_NI("- Module '%s' in (temporary ?) use - rescheduling", OUSTRING2ASCII(sCurrentName) );
					bHandled = false; // still remove from the lists
				}
				else if (xModule->hasPending())
				{
					CFG_TRACE_WARNING_NI("- Module '%s' has pending changes - rescheduling disposal", OUSTRING2ASCII(sCurrentName) );
					bHandled = false;
				}
				else // now this really can be disposed
				{
					CFG_TRACE_INFO_NI("- Removing module '%s' for disposal", OUSTRING2ASCII(sCurrentName) );

					// It really is ok to dispose this entry
					_rList.push_back(xModule);

					bHandled = true;
				}

				
				if (bHandled) 
				{
					// really remove
					rActiveModules.erase(itModule);			
					m_aDeadModules.erase(current);	
				}
				else
				{
					// reschedule
					TimeStamp aRetryTime = m_rDisposer.getCleanupTime(aLimitTime);
					OSL_ASSERT(aRetryTime > aLimitTime);

					current->second = aRetryTime;
					if (aRetryTime < aRetTime)
						aRetTime = aRetryTime;
				}
			}
			else
			{
				// obsolete dispose list entry - discard
				OSL_ENSURE( false, "Obsolete entry in dead module list");

				CFG_TRACE_WARNING_NI("- Module '%s' not found any more - obsolete entry in dead module list", OUSTRING2ASCII(sCurrentName) );

				m_aDeadModules.erase(current);
			}
		}
		else // consider for restart time
		{
			CFG_TRACE_INFO_NI("- Module '%s' has not expired yet - rescheduling", OUSTRING2ASCII(sCurrentName) );

			if (current->second < aRetTime)
				aRetTime = current->second;
		}
	}

	OSL_ASSERT(aRetTime > aLimitTime);
	return aRetTime;
}

// -------------------------------------------------------------------------
uno::Sequence< OUString > TreeInfo::collectNodeIds(DisposeList const& aDisposedList )
{
	Sequence< OUString > aCloseList;

	if (!aDisposedList.empty())
	{
		ONodeIdCollector aCollectIds;

		for(TreeInfo::DisposeList::const_iterator it = aDisposedList.begin();
			it != aDisposedList.end();
			++it)
		{
			if (it->isValid())
			{
				if (ISubtree* pRoottree = (*it)->getModuleRoot())
				{
					aCollectIds.applyToNode(*pRoottree);
				}
			}
		}

		aCollectIds.ids.erase( OUString() ); // don't need to close the empty (dummy) id

		if (!aCollectIds.ids.empty())
		{
			aCloseList.realloc( aCollectIds.ids.size() );
			std::copy(aCollectIds.ids.begin(),aCollectIds.ids.end(), aCloseList.getArray());
		}
	}

	return aCloseList;
}

// -------------------------------------------------------------------------
void TreeInfo::syncPending(vos::ORef< OOptions > const& _xOptions, TreeManager& _rTreeManager)  CFG_UNO_THROW_ALL(  )
{
	// write pending to session
	// return 0 if no errors treated

	typedef Tree::ModuleList ModuleList;
	ModuleList& rModules = this->aTree.accessModuleList();
	CFG_TRACE_INFO("write down pendings");
	
	for(ModuleList::iterator it = rModules.begin();
		it != rModules.end();
		++it)
	{
		ModuleTreeRef xModule = it->second;
		if (xModule.isValid())
		{
			if (xModule->hasPending())
			{
                AbsolutePath aCurrentModulePath = AbsolutePath::makeModulePath(it->first);
				CFG_TRACE_INFO_NI("write down pending %s", OUSTRING2ASCII(aCurrentModulePath.toString()));

				try
				{
					// We get a pointer from the pending
					SubtreeChange *pSubtreeChange = xModule->getPending();
					// _aList.push_back(PendingPair(sCurrentModuleName, pChange ) );
					_rTreeManager.sessionUpdate(_xOptions, aCurrentModulePath, pSubtreeChange);

					// remove Pending from Module, if it was written successfully
					xModule->releasePending();
				}
				catch (uno::Exception& ue)
				{
					CFG_TRACE_ERROR_NI("An Exception occurred. Can't write pending data from %s. Error: %s",
                                        OUSTRING2ASCII(aCurrentModulePath.toString()),OUSTRING2ASCII(ue.Message));

                    // remove Pending from Module,to avoid infinite retries
					xModule->releasePending();

                    // initiate refresh to resync with backend
                    _rTreeManager.invalidateTreeAsync(aCurrentModulePath, _xOptions);

                    // pass on the exception for others to handle
					throw;
				}	
			}
		}
	}
}

} // namespace configmgr


