/*************************************************************************
 *
 *  $RCSfile: binarybasereader.cxx,v $
 *
 *  $Revision: 1.7 $
 *
 *  last change: $Author: jb $ $Date: 2001/11/09 17:14:14 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "binarybasereader.hxx"

#ifndef _OSL_FILE_HXX_
#include <osl/file.hxx>
#endif
#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif

#ifndef _COM_SUN_STAR_IO_BUFFERSIZEEXCEEDEDEXCEPTION_HPP_
#include <com/sun/star/io/BufferSizeExceededException.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_UNEXPECTEDEOFEXCEPTION_HPP_
#include <com/sun/star/io/UnexpectedEOFException.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XDATAINPUTSTREAM_HPP_
#include <com/sun/star/io/XDataInputStream.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XMARKABLESTREAM_HPP_
#include <com/sun/star/io/XMarkableStream.hpp>
#endif

#ifndef _CPPUHELPER_WEAK_HXX_
#include <cppuhelper/weak.hxx>
#endif

#ifndef _CONFIGMGR_FILEHELPER_HXX_
#include "filehelper.hxx"
#endif
#ifndef _CONFIGMGR_TRACER_HXX_
#include "tracer.hxx"
#endif

#ifndef CONFIGMGR_BINARYREADER_HXX
#include "binaryreader.hxx"
#endif

#ifndef INCLUDED_MAP
#include <map>
#define INCLUDED_MAP
#endif

// --------------------------------------------------------------------------

namespace configmgr
{

// --------------------------------------------------------------------------

typedef std::map<sal_Int32, sal_uInt32, std::less<sal_Int32> > map_type;

inline rtl::OUString ErrorToMessage_Impl (osl::FileBase::RC eError)
{
	return FileHelper::createOSLErrorString (eError);
}

// --------------------------------------------------------------------------

class OBinaryBaseReader_Impl :
	public cppu::OWeakObject,
	public com::sun::star::io::XDataInputStream,
	public com::sun::star::io::XMarkableStream
{
public:
	/** Construction.
	 */
	explicit OBinaryBaseReader_Impl (rtl::OUString const & rFileUrl)
		SAL_THROW( (io::IOException) );

	/** XInterface.
	 */
	virtual uno::Any SAL_CALL queryInterface (const uno::Type & rType)
		throw (uno::RuntimeException);

	virtual void SAL_CALL acquire() throw();
	virtual void SAL_CALL release() throw();


	/** XInputStream.
	 */
	virtual sal_Int32 SAL_CALL readBytes (
		uno::Sequence<sal_Int8> & rData, sal_Int32 nBytesToRead)
		throw (
			io::NotConnectedException,
			io::BufferSizeExceededException,
			io::IOException, uno::RuntimeException);

	virtual sal_Int32 SAL_CALL readSomeBytes (
		uno::Sequence<sal_Int8> & rData, sal_Int32 nBytesToRead)
		throw (
			io::NotConnectedException,
			io::BufferSizeExceededException,
			io::IOException, uno::RuntimeException);

	virtual void SAL_CALL skipBytes (sal_Int32 nBytesToSkip)
		throw (
			io::NotConnectedException,
			io::BufferSizeExceededException,
			io::IOException, uno::RuntimeException);

	virtual sal_Int32 SAL_CALL available()
		throw (
			io::NotConnectedException,
			io::IOException, uno::RuntimeException);

	virtual void SAL_CALL closeInput()
		throw (
			io::NotConnectedException,
			io::IOException, uno::RuntimeException);


	/** XDataInputStream.
	 */
	virtual sal_Int8 SAL_CALL readBoolean()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Int8 SAL_CALL readByte()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Unicode SAL_CALL readChar()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Int16 SAL_CALL readShort()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Int32 SAL_CALL readLong()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Int64 SAL_CALL readHyper()
		throw (
			io::IOException, uno::RuntimeException);

	virtual float SAL_CALL readFloat()
		throw (
			io::IOException, uno::RuntimeException);

	virtual double SAL_CALL readDouble()
		throw (
			io::IOException, uno::RuntimeException);

	virtual rtl::OUString SAL_CALL readUTF()
		throw (
			io::IOException, uno::RuntimeException);

	/** XMarkableStream.
	 */
	virtual sal_Int32 SAL_CALL createMark()
		throw (
			io::IOException, uno::RuntimeException);

	virtual void SAL_CALL deleteMark (sal_Int32 nMark)
		throw (
			lang::IllegalArgumentException,
			io::IOException, uno::RuntimeException);

	virtual void SAL_CALL jumpToMark (sal_Int32 nMark)
		throw (
			lang::IllegalArgumentException,
			io::IOException, uno::RuntimeException);

	virtual void SAL_CALL jumpToFurthest()
		throw (
			io::IOException, uno::RuntimeException);

	virtual sal_Int32 SAL_CALL offsetToMark (sal_Int32 nMark)
		throw (
			lang::IllegalArgumentException,
			io::IOException, uno::RuntimeException);


protected:
	/** Destruction.
	 */
	virtual ~OBinaryBaseReader_Impl();

private:
	/** Representation.
	 */
	sal_uInt8 * m_pBuffer;
	sal_uInt32  m_nLength;
	sal_uInt32  m_nOffset;

	map_type    m_aMarks;
	sal_Int32   m_nMarks;
	sal_uInt32  m_nMaxOff;

	/** Not implemented.
	 */
	OBinaryBaseReader_Impl (const OBinaryBaseReader_Impl&);
	OBinaryBaseReader_Impl& operator= (const OBinaryBaseReader_Impl&);
};

// --------------------------------------------------------------------------

OBinaryBaseReader_Impl::OBinaryBaseReader_Impl (rtl::OUString const & rFileUrl)
	SAL_THROW( (io::IOException) )
	: m_pBuffer (0),
	  m_nLength (0),
	  m_nOffset (0),
	  m_nMarks  (0),
	  m_nMaxOff (0)
{
	osl::FileBase::RC result;
	osl::File aFile (rFileUrl);

	result = aFile.open (OpenFlag_Read);
	if (!(result == osl::FileBase::E_None))
	{
		throw io::IOException (ErrorToMessage_Impl (result), NULL);
	}

	result = aFile.setPos (Pos_End, 0);
	if (!(result == osl::FileBase::E_None))
	{
		throw io::IOException (ErrorToMessage_Impl (result), NULL);
	}

	sal_uInt64 nLength = 0;
	result = aFile.getPos (nLength);
	if (!(result == osl::FileBase::E_None))
	{
		throw io::IOException (ErrorToMessage_Impl (result), NULL);
	}
	if (!(nLength <= 0xffffffff))
	{
		throw io::BufferSizeExceededException();
	}
	m_nLength = sal_uInt32(nLength);

	result = aFile.setPos (Pos_Absolut, 0);
	if (!(result == osl::FileBase::E_None))
	{
		throw io::IOException (ErrorToMessage_Impl (result), NULL);
	}

	sal_uInt8 *pBuffer = (sal_uInt8*)(rtl_allocateMemory (m_nLength));
	if (!pBuffer)
	{
		throw io::BufferSizeExceededException();
	}

	sal_uInt64 nRead = 0;
	result = aFile.read (pBuffer, nLength, nRead);
	if (!(result == osl::FileBase::E_None))
	{
		rtl_freeMemory (pBuffer);
		throw io::IOException (ErrorToMessage_Impl (result), NULL);
	}
	if (!(nRead == nLength))
	{
		rtl_freeMemory (pBuffer);
		throw io::UnexpectedEOFException();
	}
	m_pBuffer = pBuffer;
}

// --------------------------------------------------------------------------

OBinaryBaseReader_Impl::~OBinaryBaseReader_Impl()
{
	rtl_freeMemory (m_pBuffer);
}

// --------------------------------------------------------------------------
// XInterface implementation.
// --------------------------------------------------------------------------

uno::Any SAL_CALL OBinaryBaseReader_Impl::queryInterface (
	const uno::Type & rType)
	throw (uno::RuntimeException)
{
	uno::Any result (cppu::queryInterface (
		rType,
		static_cast<io::XInputStream*>(this),
		static_cast<io::XDataInputStream*>(this),
		static_cast<io::XMarkableStream*>(this)));
	if (!(result.hasValue()))
		result = OWeakObject::queryInterface (rType);
	return (result);
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::acquire() throw()
{
	OWeakObject::acquire();
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::release() throw()
{
	OWeakObject::release();
}

// --------------------------------------------------------------------------
// XInputStream implementation.
// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::readBytes (
	uno::Sequence<sal_Int8> & rData, sal_Int32 nBytesToRead)
	throw (
		io::NotConnectedException,
		io::BufferSizeExceededException,
		io::IOException, uno::RuntimeException)
{
	sal_Int32 nAvail = sal_Int32(m_nLength - m_nOffset);
	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}

	nBytesToRead = std::min(nAvail, nBytesToRead);
	if (nBytesToRead > 0)
	{
		rData.realloc (nBytesToRead);
		memcpy (rData.getArray(), &(m_pBuffer[m_nOffset]), nBytesToRead);

		m_nOffset += nBytesToRead;
		m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);
	}
	return (nBytesToRead);
}

// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::readSomeBytes (
	uno::Sequence<sal_Int8> & rData, sal_Int32 nBytesToRead)
	throw (
		io::NotConnectedException,
		io::BufferSizeExceededException,
		io::IOException, uno::RuntimeException)
{
	sal_Int32 nAvail = sal_Int32(m_nLength - m_nOffset);
	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}

	nBytesToRead = std::min(nAvail, nBytesToRead);
	if (nBytesToRead > 0)
	{
		rData.realloc (nBytesToRead);
		memcpy (rData.getArray(), &(m_pBuffer[m_nOffset]), nBytesToRead);

		m_nOffset += nBytesToRead;
		m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);
	}
	return (nBytesToRead);
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::skipBytes (sal_Int32 nBytesToSkip)
	throw (
		io::NotConnectedException,
		io::BufferSizeExceededException,
		io::IOException, uno::RuntimeException)
{
	sal_Int32 nAvail = sal_Int32(m_nLength - m_nOffset);
	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}

	nBytesToSkip = std::min(nAvail, nBytesToSkip);
	if (nBytesToSkip > 0)
	{
		m_nOffset += nBytesToSkip;
		m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);
	}
}

// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::available()
	throw (
		io::NotConnectedException,
		io::IOException, uno::RuntimeException)
{
	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	return sal_Int32(m_nLength - m_nOffset);
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::closeInput()
	throw (
		io::NotConnectedException,
		io::IOException, uno::RuntimeException)
{
	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	rtl_freeMemory (m_pBuffer); m_pBuffer = 0;
}

// --------------------------------------------------------------------------
// XDataInputStream implementation.
// --------------------------------------------------------------------------

sal_Int8 SAL_CALL OBinaryBaseReader_Impl::readBoolean()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int8 result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Int8(m_pBuffer[m_nOffset]);

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

sal_Int8 SAL_CALL OBinaryBaseReader_Impl::readByte()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int8 result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Int8(m_pBuffer[m_nOffset]);

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

sal_Unicode SAL_CALL OBinaryBaseReader_Impl::readChar()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Unicode result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Unicode(
		(sal_uInt16(m_pBuffer[m_nOffset + 0]) << 8) |
		(sal_uInt16(m_pBuffer[m_nOffset + 1]) << 0)   );

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

sal_Int16 SAL_CALL OBinaryBaseReader_Impl::readShort()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int16 result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result) <= m_nLength)))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Int16(
		(sal_uInt16(m_pBuffer[m_nOffset + 0]) << 8) |
		(sal_uInt16(m_pBuffer[m_nOffset + 1]) << 0)   );

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::readLong()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int32 result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Int32(
		(sal_uInt32(m_pBuffer[m_nOffset + 0]) << 24) |
		(sal_uInt32(m_pBuffer[m_nOffset + 1]) << 16) |
		(sal_uInt32(m_pBuffer[m_nOffset + 2]) <<  8) |
		(sal_uInt32(m_pBuffer[m_nOffset + 3]) <<  0)   );

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

sal_Int64 SAL_CALL OBinaryBaseReader_Impl::readHyper()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int64 result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result = sal_Int64(
		(sal_uInt64(m_pBuffer[m_nOffset + 0]) << 56) |
		(sal_uInt64(m_pBuffer[m_nOffset + 1]) << 48) |
		(sal_uInt64(m_pBuffer[m_nOffset + 2]) << 40) |
		(sal_uInt64(m_pBuffer[m_nOffset + 3]) << 32) |
		(sal_uInt64(m_pBuffer[m_nOffset + 4]) << 24) |
		(sal_uInt64(m_pBuffer[m_nOffset + 5]) << 16) |
		(sal_uInt64(m_pBuffer[m_nOffset + 6]) <<  8) |
		(sal_uInt64(m_pBuffer[m_nOffset + 7]) <<  0)   );

	m_nOffset += sizeof(result);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result);
}

// --------------------------------------------------------------------------

float SAL_CALL OBinaryBaseReader_Impl::readFloat()
	throw (io::IOException, uno::RuntimeException)
{
	union { float f; sal_uInt32 n; } result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result.n)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result.n = sal_uInt32(
		(sal_uInt32(m_pBuffer[m_nOffset + 0]) << 24) |
		(sal_uInt32(m_pBuffer[m_nOffset + 1]) << 16) |
		(sal_uInt32(m_pBuffer[m_nOffset + 2]) <<  8) |
		(sal_uInt32(m_pBuffer[m_nOffset + 3]) <<  0)   );

	m_nOffset += sizeof(result.n);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result.f);
}

// --------------------------------------------------------------------------

double SAL_CALL OBinaryBaseReader_Impl::readDouble()
	throw (io::IOException, uno::RuntimeException)
{
	union { double d; sal_uInt64 n; } result;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(result.n)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	result.n = sal_uInt64(
		(sal_uInt64(m_pBuffer[m_nOffset + 0]) << 56) |
		(sal_uInt64(m_pBuffer[m_nOffset + 1]) << 48) |
		(sal_uInt64(m_pBuffer[m_nOffset + 2]) << 40) |
		(sal_uInt64(m_pBuffer[m_nOffset + 3]) << 32) |
		(sal_uInt64(m_pBuffer[m_nOffset + 4]) << 24) |
		(sal_uInt64(m_pBuffer[m_nOffset + 5]) << 16) |
		(sal_uInt64(m_pBuffer[m_nOffset + 6]) <<  8) |
		(sal_uInt64(m_pBuffer[m_nOffset + 7]) <<  0)   );

	m_nOffset += sizeof(result.n);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	return (result.d);
}

// --------------------------------------------------------------------------

rtl::OUString SAL_CALL OBinaryBaseReader_Impl::readUTF()
	throw (io::IOException, uno::RuntimeException)
{
	rtl::OUString result;
	sal_Int32     nLength;
	bool		  bIsAscii = false;

	if (!m_pBuffer)
	{
		throw io::NotConnectedException();
	}
	if (!((m_nOffset + sizeof(nLength)) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	nLength = sal_Int32(
		(sal_uInt32(m_pBuffer[m_nOffset + 0]) << 24) |
		(sal_uInt32(m_pBuffer[m_nOffset + 1]) << 16) |
		(sal_uInt32(m_pBuffer[m_nOffset + 2]) <<  8) |
		(sal_uInt32(m_pBuffer[m_nOffset + 3]) <<  0)   );

	bIsAscii = (nLength & STR_ASCII_MASK) == STR_ASCII_MASK;
	nLength &=~STR_ASCII_MASK;	

	if (!((m_nOffset + sizeof(nLength) + nLength) <= m_nLength))
	{
		throw io::UnexpectedEOFException();
	}

	m_nOffset += sizeof(nLength);
	m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);

	if (nLength > 0)
	{
		if (bIsAscii)
			rtl_string2UString (
			&(result.pData), (sal_Char*)&(m_pBuffer[m_nOffset]), nLength,
			RTL_TEXTENCODING_ASCII_US, OSTRING_TO_OUSTRING_CVTFLAGS);
		else
			rtl_string2UString (
			&(result.pData), (sal_Char*)&(m_pBuffer[m_nOffset]), nLength,
			RTL_TEXTENCODING_UTF8, OSTRING_TO_OUSTRING_CVTFLAGS);

		m_nOffset += nLength;
		m_nMaxOff  = std::max(m_nMaxOff, m_nOffset);
	}

	return (result);
}

// --------------------------------------------------------------------------
// XMarkableStream implementation.
// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::createMark()
	throw (io::IOException, uno::RuntimeException)
{
	sal_Int32 nMark = ++m_nMarks;
	m_aMarks[nMark] = m_nOffset;
	return (nMark);
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::deleteMark (sal_Int32 nMark)
	throw (
		lang::IllegalArgumentException,
		io::IOException, uno::RuntimeException)
{
	map_type::iterator it = m_aMarks.find (nMark);
	if (it == m_aMarks.end())
	{
		throw lang::IllegalArgumentException();
	}
	m_aMarks.erase (it);
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::jumpToMark (sal_Int32 nMark)
	throw (
		lang::IllegalArgumentException,
		io::IOException, uno::RuntimeException)
{
	map_type::const_iterator it = m_aMarks.find (nMark);
	if (it == m_aMarks.end())
	{
		throw lang::IllegalArgumentException();
	}
	m_nOffset = (*it).second;
}

// --------------------------------------------------------------------------

void SAL_CALL OBinaryBaseReader_Impl::jumpToFurthest()
	throw (io::IOException, uno::RuntimeException)
{
	m_nOffset = m_nMaxOff;
}

// --------------------------------------------------------------------------

sal_Int32 SAL_CALL OBinaryBaseReader_Impl::offsetToMark (sal_Int32 nMark)
	throw (
		lang::IllegalArgumentException,
		io::IOException, uno::RuntimeException)
{
	map_type::const_iterator it = m_aMarks.find (nMark);
	if (it == m_aMarks.end())
	{
		throw lang::IllegalArgumentException();
	}
	return (m_nOffset - (*it).second);
}

// --------------------------------------------------------------------------
// OBinaryBaseReader implementation.
// --------------------------------------------------------------------------

void OBinaryBaseReader::open()
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	uno::Reference<io::XDataInputStream> xImpl (
		new OBinaryBaseReader_Impl (m_sFileURL));
	if (!xImpl.is())
	{
		CFG_TRACE_ERROR("OBinaryBaseReader::open(): No DataInputStream");
		throw io::IOException();
	}

	uno::Reference<io::XMarkableStream> xMark (xImpl, uno::UNO_QUERY);
	if (!xMark.is())
	{
		CFG_TRACE_ERROR("OBinaryBaseReader::open(): No MarkableStream");
		throw io::IOException();
	}

	m_xDataInputStream = xImpl;
	m_xMarkableStream  = xMark;
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::dispose()
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	uno::Reference<io::XInputStream> xInputStream (
		m_xDataInputStream, uno::UNO_QUERY);
	if (xInputStream.is())
		xInputStream->closeInput();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Bool &_bValue) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_bValue = m_xDataInputStream->readBoolean();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Int8 &_nValue) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_nValue = m_xDataInputStream->readByte();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Int16 &_nValue) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_nValue = m_xDataInputStream->readShort();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Int32 &_nValue) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_nValue = m_xDataInputStream->readLong();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Int64 &_nValue)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_nValue = m_xDataInputStream->readHyper();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(double &_nValue)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_nValue = m_xDataInputStream->readDouble();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(sal_Unicode &_sValue) 
{
	_sValue = m_xDataInputStream->readChar();
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read (sal_Unicode *pStr, sal_Int32 _nCount)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	sal_uInt64 nLength = _nCount;
	sal_Unicode* pPtr = pStr;
	while(nLength--)
	{
		*pPtr++ = m_xDataInputStream->readChar();
	}
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read (uno::Sequence<sal_Int8> &_aBuf, sal_Int32 _nCount) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	m_xDataInputStream->readBytes(_aBuf, _nCount);
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read(rtl::OUString& _aStr) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_aStr = m_xDataInputStream->readUTF();
}

// -----------------------------------------------------------------------------
template <class Element>
static
void readSequence(OBinaryBaseReader& _rReader, uno::Sequence< Element > & aSequence)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	// PRE: the Sequence must exist
	sal_Int32 nLength;
	_rReader.read(nLength);

	if (aSequence.getLength() != nLength)		 // check size
	{
		aSequence.realloc(nLength);
	}

	Element* pElement = aSequence.getArray();	 // fill the hole array
	for(sal_Int32 i=0; i<nLength; ++i) 
	{
		Element aValue;
		_rReader.read(aValue);					 // read one element
		*pElement++ = aValue;					 // like pElement[i]
	}
}

// --------------------------------------------------------------------------

void OBinaryBaseReader::read (uno::Sequence<sal_Int8> &_aValue) 
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	readSequence(*this, _aValue);
}

// --------------------------------------------------------------------------

#define CASE_READ_SEQUENCE(TYPE_CLASS, DATA_TYPE)   \
    case TYPE_CLASS:                                \
    {                                               \
		OSL_ENSURE( ::getCppuType(static_cast<DATA_TYPE const*>(0)).getTypeClass() == (TYPE_CLASS), "Typeclass does not match element type" );    \
		uno::Sequence< DATA_TYPE > aData;                \
        readSequence(_rReader, aData);                   \
        _aValue <<= aData;                               \
    }   break                                             

void readSequenceValue (
	OBinaryBaseReader & _rReader,
	uno::Any          & _aValue,
	uno::Type const   & _aElementType)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	switch(_aElementType.getTypeClass())
	{
	CASE_READ_SEQUENCE( uno::TypeClass_BOOLEAN, sal_Bool );

	CASE_READ_SEQUENCE( uno::TypeClass_SHORT, sal_Int16 );

	CASE_READ_SEQUENCE( uno::TypeClass_LONG, sal_Int32 );

	CASE_READ_SEQUENCE( uno::TypeClass_HYPER, sal_Int64 );

	CASE_READ_SEQUENCE( uno::TypeClass_DOUBLE, double );

	CASE_READ_SEQUENCE( uno::TypeClass_STRING, rtl::OUString );

	CASE_READ_SEQUENCE( uno::TypeClass_SEQUENCE, uno::Sequence<sal_Int8> );

	CASE_READ_SEQUENCE( uno::TypeClass_BYTE, sal_Int8 ); /* Binary */

	default: 
		OSL_ENSURE(false, "Unexpected typeclass for sequence elements");
		break;
	}
}

#undef CASE_READ_SEQUENCE
 
// --------------------------------------------------------------------------
// OInputMark implementation.
// --------------------------------------------------------------------------

OInputMark::OInputMark(OBinaryBaseReader* _pReader)
	: m_pReader(_pReader)
{
	m_nMark = _pReader->getMarkableStream()->createMark();
	_pReader->read(m_nPosition);
}

void OInputMark::jumpToMark()
{
	m_pReader->getMarkableStream()->jumpToMark(m_nMark);
	m_pReader->getDataInputStream()->skipBytes(sal_Int32(m_nPosition));
	m_pReader->getMarkableStream()->deleteMark(m_nMark);
}

// --------------------------------------------------------------------------

} // namespace configmgr
