/*************************************************************************
 *
 *  $RCSfile: binarywriter.cxx,v $
 *
 *  $Revision: 1.14.4.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/11/07 08:14:54 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "binarywriter.hxx"

#ifndef _CONFIGMGR_TREE_VALUENODE_HXX
#include "valuenode.hxx"
#endif
#ifndef _CONFIGMGR_FILEHELPER_HXX_
#include "filehelper.hxx"
#endif
#ifndef _CONFIGMGR_OSLSTREAM_HXX_
#include "oslstream.hxx"
#endif
#ifndef CONFIGMGR_SIMPLETYPEHELPER_HXX
#include "simpletypehelper.hxx"
#endif
#ifndef _CONFIGMGR_SERVER_VERSION_HXX_
#include "server_version.hxx"
#endif
#ifndef CONFIGMGR_BINARYREADER_HXX
#include "binaryreader.hxx"
#endif

#ifndef _COM_SUN_STAR_UNO_TYPE_HXX_
#include <com/sun/star/uno/Type.hxx>
#endif
#ifndef _COM_SUN_STAR_UNO_ANY_HXX_
#include <com/sun/star/uno/Any.hxx>
#endif


#ifndef _COM_SUN_STAR_IO_IOEXCEPTION_HPP_
#include <com/sun/star/io/IOException.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XOUTPUTSTREAM_HPP_
#include <com/sun/star/io/XOutputStream.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XACTIVEDATASOURCE_HPP_
#include <com/sun/star/io/XActiveDataSource.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XDATAOUTPUTSTREAM_HPP_
#include <com/sun/star/io/XDataOutputStream.hpp>
#endif
#ifndef _COM_SUN_STAR_IO_XOBJECTOUTPUTSTREAM_HPP_
#include <com/sun/star/io/XObjectOutputStream.hpp>
#endif

#ifdef _USE_COMPRESSION
#ifndef CONFIGMGR_STREAMCOMPRESS_HXX_
#include "streamcompress.hxx"
#endif
#ifndef _COMPHELPER_SEQSTREAM_HXX
#include <comphelper/seqstream.hxx>
#endif
#endif

#ifndef _OSL_FILE_HXX_
#include <osl/file.hxx>
#endif

#ifndef INCLUDED_VECTOR
#include <vector>
#define INCLUDED_VECTOR
#endif

#define ASCII(x) rtl::OUString::createFromAscii(x)

namespace configmgr
{
	using namespace ::rtl;
	using namespace ::std;
	using namespace ::osl;
	using namespace ::com::sun::star;
	using namespace ::com::sun::star::uno;
	namespace uno = com::sun::star::uno;
	namespace io = com::sun::star::io;

// --------------------------------- Prototypes ---------------------------------
	uno::Type getSequenceElementType(uno::Type const& rSequenceType);
	uno::Type getBasicType(uno::Type const& rType, bool& bSequence);

// -----------------------------------------------------------------------------

// -----------------------------------------------------------------------------
	OBinaryBaseWriter::OBinaryBaseWriter(rtl::OUString const &_aFileURL, uno::Reference<lang::XMultiServiceFactory> const& _xServiceProvider)
			:m_pFileOut(NULL),
			 m_aBuffer(8192),
			 m_aFileURL(_aFileURL),
			 m_xServiceProvider(_xServiceProvider)			
	{}
	
	void OBinaryBaseWriter::open() SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		ensure_File();
	}
	
	void OBinaryBaseWriter::close() SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		if (m_pFileOut)
		{
            if (m_xDataOutputStream.is())
			    m_xDataOutputStream->closeOutput();
		
            m_xDataOutputStream.clear();
            m_xMarkableStream.clear();
#ifdef _USE_COMPRESSION
			try
			{
				// delegate the data to a new InputStream
				uno::Reference<io::XInputStream> xInput = new comphelper::SequenceInputStream(m_aBuffer);			
				uno::Reference<io::XOutputStream> xOutput = new OSLOutputStreamWrapper(*m_pFileOut);

				OStreamCompressor().compress(xInput,xOutput);
				xOutput->closeOutput();		
			}
			catch (io::IOException& )
			{
				delete m_pFileOut;
				m_pFileOut = NULL;

				// compression did not work, so do the best thing we can do, and try to remove the file				
				try {FileHelper::tryToRemoveFile(m_aFileURL);} catch (io::IOException) {}
				throw;
			}				
#endif
			delete m_pFileOut;
			m_pFileOut = NULL;
		}
	}	
	
	// -----------------------------------------------------------------------------
	OBinaryBaseWriter::~OBinaryBaseWriter()
	{
		try 
        {
            close();
		}
		catch (uno::Exception& e)
		{
			OSL_ENSURE(false, rtl::OUStringToOString(e.Message,RTL_TEXTENCODING_ASCII_US).getStr());
			delete m_pFileOut;
		}				
	}
	
	// -----------------------------------------------------------------------------
	void OBinaryBaseWriter::ensure_File() SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// POST: ensure that the file is open, if not, open it.
		if (m_pFileOut == NULL)
        try
		{
			FileHelper::removeFile(m_aFileURL);
		
			m_pFileOut = new File(m_aFileURL);
			OSL_ENSURE(m_pFileOut, "could not get the file handle.");
			FileBase::RC eError = m_pFileOut->open(OpenFlag_Write | OpenFlag_Create);
			if (eError != osl_File_E_None)
			{
				throw io::IOException(FileHelper::createOSLErrorString(eError), NULL);
			}

#ifdef _USE_COMPRESSION
			// first write to memory			
			uno::Reference< io::XOutputStream > xOutput(new comphelper::OSequenceOutputStream(m_aBuffer,1.3));
#else
			uno::Reference<io::XOutputStream> xOutput = new OSLOutputStreamWrapper(*m_pFileOut);
#endif
			uno::Reference< io::XMarkableStream > xMarkableStream(m_xServiceProvider->createInstance(ASCII("com.sun.star.io.MarkableOutputStream")), uno::UNO_QUERY);
			uno::Reference< io::XActiveDataSource > xMarkSource(xMarkableStream, uno::UNO_QUERY);
			xMarkSource->setOutputStream(xOutput);
			
			uno::Reference< io::XObjectOutputStream > xObjectOutputStream(m_xServiceProvider->createInstance(ASCII("com.sun.star.io.ObjectOutputStream")), uno::UNO_QUERY);
			uno::Reference< io::XActiveDataSource > xDataSource(xObjectOutputStream, uno::UNO_QUERY);
			uno::Reference< io::XOutputStream > xOutputStream(xMarkableStream, uno::UNO_QUERY);
			xDataSource->setOutputStream(xOutputStream);
			
			m_xMarkableStream = uno::Reference<io::XMarkableStream>(xObjectOutputStream, uno::UNO_QUERY);
			m_xDataOutputStream = uno::Reference<io::XDataOutputStream>(xObjectOutputStream, uno::UNO_QUERY);
			if (!m_xMarkableStream.is())
			{
				throw uno::RuntimeException(ASCII("There is no MarkableStream"), NULL);
			}
			if (!m_xDataOutputStream.is())
			{
				throw uno::RuntimeException(ASCII("There is no DataOutputStream"), NULL);
			}
		}
        catch ( uno::Exception & )
        {
            m_xDataOutputStream.clear();
            m_xMarkableStream.clear();

            delete m_pFileOut;
            m_pFileOut = NULL;

            throw;
        }
	}
	
	// -----------------------------------------------------------------------------
	
	void OBinaryBaseWriter::write(sal_Bool _aValue)
	    SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write one byte
		m_xDataOutputStream->writeBoolean(_aValue);
	}
	void OBinaryBaseWriter::write(sal_Int8 _aValue)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		// write one byte
		m_xDataOutputStream->writeByte(_aValue);
	}
	void OBinaryBaseWriter::write(sal_Int16 _aValue)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write two bytes
		m_xDataOutputStream->writeShort(_aValue);
	}
	void OBinaryBaseWriter::write(sal_Int32 _aValue)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write four byte
		m_xDataOutputStream->writeLong(_aValue);
	}
	void OBinaryBaseWriter::write(sal_Int64 _aValue)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write eight byte
		m_xDataOutputStream->writeHyper(_aValue);
	}
	void OBinaryBaseWriter::write(double _aValue)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write eight byte
		m_xDataOutputStream->writeDouble(_aValue);
	}
	void OBinaryBaseWriter::write(sal_Unicode _aValue)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// write two byte
		m_xDataOutputStream->writeChar(_aValue);
	}

	// -----------------------------------------------------------------------------
	bool isAsciiEncoding(rtl::OUString const& _aStr)
	{
		const sal_Unicode *pStr = _aStr.getStr();
		sal_Int32 nLen = _aStr.getLength();
		while (nLen--)
		{
			if (*pStr++ > 127)
				return false;
		}
		return true;
	}
	
	// -----------------------------------------------------------------------------
	void OBinaryBaseWriter::write(rtl::OUString const& _aStr)
	    SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		// @@@ OBinaryBaseReader_Impl::readUTF() @@@

		rtl::OString aUTF;
		// to fasten the conversion for ascii data, we mask the length
		bool bIsAscii = isAsciiEncoding(_aStr);
		if (bIsAscii)
			rtl_uString2String (
				&(aUTF.pData), _aStr.getStr(), _aStr.getLength(),
				RTL_TEXTENCODING_ASCII_US, OUSTRING_TO_OSTRING_CVTFLAGS);
		else
			rtl_uString2String (
				&(aUTF.pData), _aStr.getStr(), _aStr.getLength(),
				RTL_TEXTENCODING_UTF8, OUSTRING_TO_OSTRING_CVTFLAGS);

		sal_Int32 nLength = aUTF.getLength();		
		m_xDataOutputStream->writeLong (bIsAscii ? nLength | STR_ASCII_MASK : nLength);

		uno::Sequence<sal_Int8> aData (nLength);
		memcpy (aData.getArray(), aUTF.getStr(), nLength);
		m_xDataOutputStream->writeBytes (aData);
	}
	
// -----------------------------------------------------------------------------
template <class Element>
void writeSequence(OBinaryBaseWriter& _rWriter, Sequence< Element > const& aSequence)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	sal_Int32 const nLength = aSequence.getLength();
	_rWriter.write(nLength);

	const Element* pElement = aSequence.getConstArray();
	for(sal_Int32 i=0; i<nLength; ++i) 
	{
		writeSimpleValue(_rWriter, uno::makeAny(aSequence[i]), ::getCppuType(static_cast<Element const*>(0)));
	}
}

// -----------------------------------------------------------------------------

#define CASE_WRITE_SEQUENCE(TYPE_CLASS, DATA_TYPE)	\
	case TYPE_CLASS:								\
	{												\
		Sequence< DATA_TYPE > aData;				\
		OSL_ASSERT( ::getCppuType(static_cast<DATA_TYPE const*>(0)).getTypeClass() == (TYPE_CLASS) );	\
		OSL_VERIFY( _aValue >>= aData );										\
		writeSequence(_rWriter,aData);						\
	}	break																	\

void writeSequenceValue(OBinaryBaseWriter& _rWriter, Any const& _aValue, uno::Type const& aElementType)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	switch(aElementType.getTypeClass())
	{
	CASE_WRITE_SEQUENCE( TypeClass_BOOLEAN, sal_Bool );

	CASE_WRITE_SEQUENCE( TypeClass_SHORT, sal_Int16 );

	CASE_WRITE_SEQUENCE( TypeClass_LONG, sal_Int32 );

	CASE_WRITE_SEQUENCE( TypeClass_HYPER, sal_Int64 );

	CASE_WRITE_SEQUENCE( TypeClass_DOUBLE, double );

	CASE_WRITE_SEQUENCE( TypeClass_STRING, OUString );

	CASE_WRITE_SEQUENCE( TypeClass_SEQUENCE, Sequence<sal_Int8> );

	default: 
		OSL_ENSURE(false, "Unexpected typeclass for sequence elements");
		break;
	}
}

#undef CASE_WRITE_SEQUENCE
	// -----------------------------------------------------------------------------
	template <class T>
	inline /* make performance crew happy ;-) */
	void writeFromAny(OBinaryBaseWriter& _rWriter, uno::Any const& _aValue, T& _aVar)
	    SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		if (!(_aValue >>= _aVar))
			throw uno::RuntimeException(ASCII("Invalid Any for value"),NULL);
		_rWriter.write(_aVar);
	}
	
	

	// -----------------------------------------------------------------------------
	ValueType::Enum convertTypeToValueType(uno::Type const& _aType)
	{
		ValueType::Enum eType = ValueType::val_any; /* Default is AnyType */
		uno::TypeClass aClass = _aType.getTypeClass();
		switch(aClass)
		{
		case uno::TypeClass_ANY:
			eType = ValueType::val_any;
			OSL_ENSURE(false, "convertTypeToValueType(): Found not really sopported TypeClass.");
			break;
		case uno::TypeClass_BOOLEAN:
			eType = ValueType::val_boolean;
			break;
		case uno::TypeClass_BYTE:
			eType = ValueType::val_int8;
			break;
		case uno::TypeClass_SHORT:
			eType = ValueType::val_int16;
			break;
		case uno::TypeClass_LONG:
			eType = ValueType::val_int32;
			break;
		case uno::TypeClass_HYPER:
			eType = ValueType::val_int64;
			break;
		case uno::TypeClass_DOUBLE:
			eType = ValueType::val_double;
			break;
		case uno::TypeClass_STRING:
			eType = ValueType::val_string;
			break;
		case uno::TypeClass_SEQUENCE:
		{
			Type aType = configmgr::getSequenceElementType(_aType);
			eType = convertTypeToValueType(aType);
			eType = ValueType::Enum( eType | ValueType::sequence );
			break;
		}
		default:
		{
			::rtl::OString aStr("convertTypeToValueType(): Wrong typeclass! ");
			aStr += rtl::OString::valueOf((sal_Int32)aClass);
			OSL_ENSURE(0,aStr.getStr());
		}
		}
		return eType;
	}
	
// -----------------------------------------------------------------------------
	void writeType(OBinaryBaseWriter& _aWriter, uno::Type const& _aType)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		ValueType::Enum eType = convertTypeToValueType(_aType);
		sal_Int8 nValueType = sal_Int8(eType);
		_aWriter.write(nValueType);
	}

// -----------------------------------------------------------------------------
	void writeNullType(OBinaryBaseWriter& _aWriter, uno::Type const& _aType)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		ValueType::Enum eType = convertTypeToValueType(_aType);
		sal_Int8 nValueType = sal_Int8(eType) | ValueType::val_is_null;
		_aWriter.write(nValueType);
	}

// -----------------------------------------------------------------------------
	void writeNodeType(OBinaryBaseWriter& _aWriter, NodeType::Enum _eType)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		sal_Int8 nValue = _eType;
		_aWriter.write(nValue);
	}

// -----------------------------------------------------------------------------
	void writeType(OBinaryBaseWriter& _aWriter, ValueType::Enum _eType)
    	SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		sal_Int8 nValue = _eType;
		_aWriter.write(nValue);
	}
	
// -----------------------------------------------------------------------------
void writeAttributes(OBinaryBaseWriter& _aWriter, configuration::Attributes const& _aAttributes)
    SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	sal_Int8 nValue = _aAttributes.state();

    OSL_ASSERT(0 <= nValue && nValue <= 3);

    OSL_ENSURE(_aAttributes.bWritable || !_aAttributes.bFinalized,"Unexpected attribute mix: node is both read-only and finalized");

	nValue |= (_aAttributes.bWritable     ? 1 : 0) << 2;
	nValue |= (_aAttributes.bFinalized    ? 1 : 0) << 3;

	nValue |= (_aAttributes.bNullable	  ? 1 : 0) << 4;
	nValue |= (_aAttributes.bLocalized	  ? 1 : 0) << 5;

	nValue |= (_aAttributes.bNotified	  ? 1 : 0) << 6;
	nValue |= (_aAttributes.bConstrained  ? 1 : 0) << 7;
	
	_aWriter.write(nValue);
}

	// -----------------------------------------------------------------------------
	void writeHeader(OBinaryBaseWriter& _aWriter, rtl::OUString const& _aName, configuration::Attributes const& _aAttributes)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		_aWriter.write(_aName);
		writeAttributes(_aWriter, _aAttributes);
	}
	
	// -----------------------------------------------------------------------------
	void writeNullValue(OBinaryBaseWriter& _aWriter, uno::Type const& _aType)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		// PRE: Header must be written
		writeNullType(_aWriter, _aType);
	}
	// -----------------------------------------------------------------------------
	void writeSimpleValue(OBinaryBaseWriter& _aWriter,
		uno::Any const& _aValue, uno::Type const& _aType)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		// PRE: Header must be written
		uno::TypeClass aDestinationClass = _aType.getTypeClass();
		switch (aDestinationClass)
		{
		case uno::TypeClass_BOOLEAN:
		{
			sal_Bool bValue;
			writeFromAny(_aWriter, _aValue, bValue);
			break;
		}
		case uno::TypeClass_BYTE:
		{
			sal_Int8 nValue;
			writeFromAny(_aWriter, _aValue, nValue);
			break;
		}
		case uno::TypeClass_SHORT:
		{
			sal_Int16 nValue;
			writeFromAny(_aWriter, _aValue, nValue);
			break;
		}
		case uno::TypeClass_LONG:
		{
			sal_Int32 nValue;
			writeFromAny(_aWriter, _aValue, nValue);
			break;
		}
		case uno::TypeClass_HYPER:
		{
			sal_Int64 nValue;
			writeFromAny(_aWriter, _aValue, nValue);
			break;
		}
		case uno::TypeClass_DOUBLE:
		{
			double nValue;
			writeFromAny(_aWriter, _aValue, nValue);
			break;
		}
		case uno::TypeClass_STRING:
		{
			OUString aStr;
			// if (!(_aValue >>= aStr))
			// 	   //	throw uno::RuntimeException("Invalid Any for value",NULL, NULL);
			// 	   throw io::IOException(ASCII("Invalid Any for value"),NULL);
			// 
			// writeString(_aWriter, aStr);
			writeFromAny(_aWriter, _aValue, aStr);
			break;
		}
		case uno::TypeClass_SEQUENCE:
		{
			if (_aType == SimpleTypeHelper::getBinaryType())
			{
				Sequence<sal_Int8> aBinary;
				if (_aValue >>= aBinary)
				{
					writeSequence(_aWriter, aBinary);
				}
				else
				{
					OSL_ENSURE(false, "writeSimpleValue: can't get Sequence");
				}
			}
			else
			{
				OSL_ENSURE(false, "writeSimpleValue: Sequence has the wrong format, must Sequence<sal_Int8>");
			}
			break;
		}
		default:
		{
			::rtl::OString aStr("writeSimpleValue(): Wrong typeclass! ");
			aStr += rtl::OString::valueOf((sal_Int32)aDestinationClass);
			OSL_ENSURE(0,aStr.getStr());
		}
		}
	}

	// -----------------------------------------------------------------------------
	void writeValue(OBinaryBaseWriter& _aWriter, rtl::OUString const& _aName, configuration::Attributes const& _aAttributes,
					uno::Type const& _aType, uno::Any const& _aValue, uno::Any const& _aDefaultValue)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {

		if (_aType.getTypeClass() == TypeClass_ANY) OSL_ENSURE(false, "try to write an any.");
		writeHeader(_aWriter, _aName, _aAttributes);
		if (!_aValue.hasValue())
		{
			writeNullType(_aWriter, _aType);
		}
		else
		{
			bool bSeq;
			Type aTargetType = getBasicType(_aValue.getValueType(), bSeq);
			writeType(_aWriter, _aType);

			if (!bSeq)
			{
				writeSimpleValue(_aWriter, _aValue, _aType);
			}
			else
			{
				writeSequenceValue(_aWriter, _aValue, aTargetType);
			}
		}
	}
	// -----------------------------------------------------------------------------
	void writeGroup(OBinaryBaseWriter& _aWriter, rtl::OUString const& _aName, configuration::Attributes const& _aAttributes)
	    SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		writeHeader(_aWriter, _aName, _aAttributes);
	}
	
	// -----------------------------------------------------------------------------
	void writeSet(OBinaryBaseWriter& _aWriter, rtl::OUString const& _aName, configuration::Attributes const& _aAttributes,
		rtl::OUString const& _aTemplateName, rtl::OUString const& _aModuleName)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		writeHeader(_aWriter, _aName, _aAttributes);
		_aWriter.write(_aTemplateName);
		_aWriter.write(_aModuleName);
	}
	
	// -----------------------------------------------------------------------------
	void writeStop(OBinaryBaseWriter& _aWriter)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		sal_Int8 nStopValue = 0;
		_aWriter.write(nStopValue);
	}

	// -----------------------------------------------------------------------------

	void writeFileHeader(OBinaryBaseWriter& _aWriter, sal_Int32 _nMagic)
		SAL_THROW( (io::IOException, uno::RuntimeException) )
    {
		// Magic
		// static sal_Int32 nMagic = CFG_BINARY_MAGIC;
		_aWriter.write(_nMagic);

		// FileVersion or UPD
		// IMHO is an uncoupled version nummer better than UPD, because it's need only a
		// change, if the fileformat is changed.
		static sal_Int32 nVersion = CFG_BINARY_VERSION;
		_aWriter.write(nVersion);

		sal_Int32 nDataStamp = 0;
		_aWriter.write(nDataStamp);
	}
	
} // namespace configmgr
