/*************************************************************************
 *
 *  $RCSfile: socketpipe.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:13:43 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define _PRIVATE_TEST_

#include <stdio.h>
#include <string.h>

#ifndef _VOS_SOCKET_HXX_
#include <vos/socket.hxx>
#endif
#ifndef _VOS_THREAD_HXX_
#include <vos/thread.hxx>
#endif
#ifndef _VOS_CONDITN_HXX_
#include <vos/conditn.hxx>
#endif
#ifndef _OSL_MUTEX_HXX_
#include <osl/mutex.hxx>
#endif

using namespace ::vos;
using namespace ::osl;

#define ASCII_STRING(rtlOUString)	::rtl::OString((rtlOUString).getStr(), (rtlOUString).getLength(), RTL_TEXTENCODING_ASCII_US).getStr()
#define UNI_STRING(salCharPtr)		::rtl::OUString::createFromAscii(salCharPtr)

//=============================================================================
class OSocketConnector : public OThread
{
protected:
	OStreamSocket*		m_pSource;
	OStreamSocket*		m_pDest;
	sal_Bool			m_bVerbose;

	OCondition&			m_rFinishCondition;
	OCondition&			m_rBreakCondition;
	OCondition&			m_rDeleteCondition;

	ISocketTypes::TSocketError	m_eLastSourceError;
		// as the error code a sockets returnes is thread local, we have to determine this
		// error code within this thread, not within the main one
	::rtl::OUString				m_sLastSocketErrorDescription;

protected:
	~OSocketConnector() { }

public:
	OSocketConnector(OStreamSocket* _pSource, OStreamSocket* _pDest,
		OCondition& _rFinishCond, OCondition& _rBreakCond, OCondition& _rDeleteCondition, sal_Bool _bVerbose = sal_False);

	ISocketTypes::TSocketError	getLastSourceError() const { return m_eLastSourceError; }
	::rtl::OUString				getLastSourceErrorDescription() const { return m_sLastSocketErrorDescription; }

protected:
	virtual void SAL_CALL run();

	virtual void SAL_CALL onTerminated() { m_rDeleteCondition.wait(); delete this; }
};

//-----------------------------------------------------------------------------
OSocketConnector::OSocketConnector(
			OStreamSocket* _pSource, OStreamSocket* _pDest,
			OCondition& _rFinishCond, OCondition& _rBreakCond,
			OCondition& _rDeleteCondition, sal_Bool _bVerbose)
	:m_pSource(_pSource)
	,m_pDest(_pDest)
	,m_rFinishCondition(_rFinishCond)
	,m_rBreakCondition(_rBreakCond)
	,m_rDeleteCondition(_rDeleteCondition)
	,m_bVerbose(_bVerbose)
	,m_eLastSourceError(ISocketTypes::E_None)
{
}

//-----------------------------------------------------------------------------
void OSocketConnector::run()
{
	sal_Char* pBuffer = new sal_Char[1024];

//	TimeValue aTimeout;
//	aTimeout.Seconds = 0;
//	aTimeout.Nanosec = 1000;
//	m_pSource->setRecvTimeout(&aTimeout);
	m_pSource->setRecvTimeout(NULL);

	sal_Bool bClosed = sal_False;
	sal_Bool bTimedOut = sal_False;
	sal_Bool bError = sal_False;

	ISocketTypes::TSocketError eSourceError;
	while (!bClosed && !bError)
	{
		sal_Int32 nReallyRead = m_pSource->recv(pBuffer, 1024);

		eSourceError = m_pSource->getError();
//		bTimedOut = (ISocketTypes::E_TimedOut == eSourceError);
		bClosed = (0 == nReallyRead)/* && !bTimedOut*/;
		bTimedOut = bClosed;

		if (nReallyRead)
			m_pDest->send(pBuffer, nReallyRead);

		bError =	(	!bTimedOut
					&&	(ISocketTypes::E_None != eSourceError)
					)
				||	(ISocketTypes::E_None != m_pDest->getError());

		if (m_bVerbose)
		{
			sal_Char* pPrinterBuffer = pBuffer;
			while (nReallyRead-- > 0)
			{
				printf("%c", *pPrinterBuffer);
				++pPrinterBuffer;
			}
			fflush(stdout);
		}

		if (m_rBreakCondition.check())
		{
			m_eLastSourceError = m_pSource->getError();
			m_pSource->getError(m_sLastSocketErrorDescription);
			m_rFinishCondition.set();
			return;
		}
	}

	m_eLastSourceError = m_pSource->getError();
	m_pSource->getError(m_sLastSocketErrorDescription);

	// close the destination socket, which is the source socket for the other thread
	// this causes this other thread to fall out of the blocking read (probably) doing currently
	m_pDest->close();

	// and _after_that_ set our finish indicator
	m_rFinishCondition.set();
}
//=============================================================================
void explain(sal_Bool _bVerbose)
{
	printf("socketpipe - piping sockets streams with additional console output\r\n");
	printf("\r\nusage :\r\n");
	printf("socketpipe -l <listener_port> -c <connect_server>:<connect_port>\r\n");
	printf("          [-r] [-n] [-o] [-?] [-h]\r\n");
	printf("\r\nparameters (you may use '/' instead of '-'):\r\n");
	printf("  <listener_port>   - port where the program should listen for connections\r\n");
	printf("                      to accept\r\n");
	printf("  <connect_server>,\n\r");
	printf("  <connect_port>    - socket to which the program should connect to after\r\n");
	printf("                      an connection on the <listener_port> has been\r\n");
	printf("                      established.\r\n");
	printf("  -r                - if specified, the data got from the source socket\r\n");
	printf("                      is printed to stdout, else the responses from the\r\n");
	printf("                      destination socket.\r\n");
	printf("  -n                - suppress all stdout output except the stream data\r\n");
	printf("  -o                - Unless you use this switch, if one session is finished,\r\n");
	printf("                      (which means one of the both sockets the pipe is\r\n");
	printf("                      connecting has been closed) the program starts a new\r\n");
	printf("                      one\r\n");
	if (_bVerbose)
		printf("  -?, -h            - show this page");
	else
		printf("  -?, -h            - show this page, with some additional explanations.");
	if (_bVerbose)
	{
		printf("\r\n\r\n");
		printf("\r\nexample:\r\n\r\n");
		printf("    socketpipe -l 100 -c monet:120 -n -o\r\n\r\n");
		printf("  will wait for incoming connectings on the local machine on port 100.\r\n");
		printf("  After establishing it, a connection to a the machine called \"monet\"\r\n");
		printf("  will be made, using port 120. If this succeeds, too, the program will\r\n");
		printf("  constantly read from one and write to the other connection, and vice\r\n");
		printf("  versa. No changes are made to either data stream, they are simply\r\n");
		printf("  forwarded.\r\n");
		printf("  So if you call the program which connected to port 100 the client and\r\n");
		printf("  the one which accepted the call on monet on port 120 the server, then\r\n");
		printf("  from the client's point of view socketpipe behaves much like the server,\r\n");
		printf("  and as far as the server knows, it's serving the client.\r\n");
		printf("  In addition, all data got from the \"left\" side of the pipe - port 100 -\r\n");
		printf("  is dumped to the console. If you would have specified the -r switch,\r\n");
		printf("  the other direction would have been dumped. Because of the -n switch,\r\n");
		printf("  no other program output to stdout will occur.\r\n");
		printf("  If any of the both ends of the pipe is closed by it's counterpart\r\n");
		printf("  (the client or the server), the pipe closes its other end, too, and exits.\r\n");
		printf("  If you do not specifiy the -o switch, a next round will be started with\r\n");
		printf("  listening at port 100, again.\r\n");
	}

	printf("\r\n\r\n");
	fflush(stdout);
}

//=============================================================================
#if (defined UNX) || (defined OS2)
void main( int argc, char * argv[] )
#else
void _cdecl main( int argc, char * argv[] )
#endif
{
	sal_Char* pServer1 = NULL;
	sal_Char* pServer2 = NULL;
	sal_Char* pPort1 = NULL;
	sal_Char* pPort2 = NULL;
	sal_Bool bReverse = sal_False;
	sal_Bool bVerbose = sal_True;
	sal_Bool bLoop = sal_True;

	// collect some parameters
	sal_Char** pArgs = argv + 1;
	for (sal_Int32 i=1; i<argc; ++i, ++pArgs)
	{
		sal_Char* pCurArg = *pArgs;
		sal_Int32 nLen = strlen(pCurArg);
		sal_Bool bInvalidArg = sal_True;
		if (nLen && (('-' == *pCurArg) || ('/' == *pCurArg)))
		{	// it's a switch
			sal_Char* pSwitch = pCurArg + 1;
			switch (nLen)
			{
				case 2:
					switch (*pSwitch)
					{
						case '?':
						case 'h':
							explain(sal_True);
							return;
						case 'l':
							pPort1 = *++pArgs;
							++i;
							bInvalidArg = sal_False;
							break;
						case 'c':
						{
							sal_Char* pSeparator = strchr(*++pArgs, ':');
							++i;
							if (pSeparator && (0 != *(pSeparator + 1)))
							{
								pServer2 = *pArgs;
								*pSeparator = 0;
								pPort2 = pSeparator + 1;
								bInvalidArg = sal_False;
							}
						}
						break;
						case 'r':
							bReverse = sal_True;
							bInvalidArg = sal_False;
							break;
						case 'n':
							bVerbose = sal_False;
							bInvalidArg = sal_False;
							break;
						case 'o':
							bLoop = sal_False;
							bInvalidArg = sal_False;
							break;
					}
					break;
			}
		}
		else
		{
			if ((1 == nLen) && ('?' == *pCurArg))
			{
				explain(sal_True);
				return;
			}
		}
		if (bInvalidArg)
		{
			explain(sal_False);
			return;
		}
	}

	::rtl::OUString sServer1, sServer2;
	sal_Int32 nPort1, nPort2;
	if (!pServer2 || !pPort1 || !pPort2)
	{
		explain(sal_False);
		return;
	}
	osl_getLocalHostname(&sServer1.pData);
	sServer2 = UNI_STRING(pServer2);
	nPort1 = UNI_STRING(pPort1).toInt32();
	nPort2 = UNI_STRING(pPort2).toInt32();

	sal_Bool bSuccess;
	OInetSocketAddr aSourceSocketAddr(sServer1, nPort1);
	OInetSocketAddr aDestSocketAddr(sServer2, nPort2);

	TimeValue aTimeout;
	aTimeout.Seconds = 5;
	aTimeout.Nanosec = 0;

	OCondition aDeleteThreadCondition;
	do
	{
		OStreamSocket aSourceStream;
		// source
		OAcceptorSocket aAcceptor;

		bSuccess = aAcceptor.bind(aSourceSocketAddr);
		bSuccess = aAcceptor.listen();
		aAcceptor.setRecvTimeout(&aTimeout);

		if (bVerbose)
			printf("waiting for calls on %s:%i ...\r\n", ASCII_STRING(sServer1), nPort1);
		bSuccess = sal_False;
		while (!bSuccess)
		{
			if (bVerbose)
			{
				printf(".");
				fflush(stdout);
			}
			ISocketTypes::TResult eResult = aAcceptor.acceptConnection(aSourceStream);
			switch (eResult)
			{
				case ISocketTypes::TResult_Ok:
					bSuccess = sal_True;
					break;
				case ISocketTypes::TResult_Error:
					printf("\r\ngot an error (%i) ... exiting\r\n\r\n", aAcceptor.getError());
					aAcceptor.close();
					return;
					break;
				case ISocketTypes::TResult_TimedOut:
					break;
			}
		}

		// dest
		if (bVerbose)
		{
			printf("\r\nconnected ... connecting to %s:%i ...\r\n", ASCII_STRING(sServer2), nPort2);
			fflush(stdout);
		}
		OConnectorSocket aConnector;
		aTimeout.Seconds = 30;
		ISocketTypes::TResult eResult = aConnector.connect(aDestSocketAddr, &aTimeout);
		switch (eResult)
		{
			case ISocketTypes::TResult_Ok:
				break;
			case ISocketTypes::TResult_Error:
				printf("\r\ngot an error (%i) ... exiting\r\n\r\n", aConnector.getError());
				return;
			case ISocketTypes::TResult_TimedOut:
				printf("\r\ntimed out ... exiting\r\n\r\n");
				return;
		}

		if (bVerbose)
		{
			printf("\r\n\r\n");
			printf("forwarding %s (1) everything from %s:%i to %s:%i)\r\n", bReverse ? "   " : "(*)", ASCII_STRING(sServer1), nPort1, ASCII_STRING(sServer2), nPort2);
			printf("forwarding %s (2) everything from %s:%i to %s:%i)\r\n", bReverse ? "(*)" : "   ", ASCII_STRING(sServer2), nPort2, ASCII_STRING(sServer1), nPort1);
			printf("\r\n");
			printf("all data on the direction marked with (*) is dumped to the console.\n\r\n\r\n\r");
			printf("\r\n");
			fflush(stdout);
		}

		oslInterlockedCount nDoneCounter = 0;

		OCondition aSourceCond;
		OCondition aDestCont;

		aDeleteThreadCondition.reset();

		OSocketConnector* pSource2Dest = new OSocketConnector(&aSourceStream, &aConnector, aSourceCond, aDestCont, aDeleteThreadCondition, !bReverse);
		OSocketConnector* pDest2Source = new OSocketConnector(&aConnector, &aSourceStream, aDestCont, aSourceCond, aDeleteThreadCondition, bReverse);

		pSource2Dest->create();
		pDest2Source->create();

		aSourceCond.wait();
		aDestCont.wait();

		if (bVerbose)
		{
			printf("\r\n\r\n");
			printf("source status : %i (%s)\r\n", pSource2Dest->getLastSourceError(), ASCII_STRING(pSource2Dest->getLastSourceErrorDescription()));
			printf("dest   status : %i (%s)\r\n", pDest2Source->getLastSourceError(), ASCII_STRING(pDest2Source->getLastSourceErrorDescription()));
			fflush(stdout);
		}

		// reset this cond after we got the informations in from the thread, so they may delete themself
		aDeleteThreadCondition.set();

		aSourceStream.close();
		aAcceptor.close();

		if (bVerbose && bLoop)
		{
			printf("\r\n\r\n\r\n------------------------------------------------------------");
			printf("\r\nnext round ....\r\n");
			printf("------------------------------------------------------------\r\n\r\n\r\n");
			fflush(stdout);
		}
	} while (bLoop);
}


