/*************************************************************************
 *
 *  $RCSfile: flatany.cxx,v $
 *
 *  $Revision: 1.1.1.1.26.1 $
 *
 *  last change: $Author: nick $ $Date: 2002/08/08 02:49:50 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <assert.h>
#include <string.h>

#if defined(SOLARIS) || defined(IRIX)
#include <alloca.h>
#else
#include <malloc.h>
#endif

#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif

#include <flatany.hxx>

using namespace ::rtl;
using namespace ::com::sun::star::uno;

/***
 * Any serialization :
 *
 * Typecode + value
 *
 * - aligned values ( Every value is aligned on its size )
 * - network byte order ( Most significant byte first )
 *
 ***/

namespace persist_Impl
{

/***
 * Marshaler
 * ----------------------------------------------
 * Marshals uno-values into a sequence of bytes. The sequence is dynamically expanded
 * if necessary.
 *
 ***/
class Marshal
{
public:
	inline Marshal();

	void pack( void *pSource , const ::com::sun::star::uno::Type &rType );
	void pack( void *pSource , typelib_TypeDescription *pType );

	::com::sun::star::uno::Sequence < sal_Int8 > remove();
private:
	inline void align( sal_Int32 nToAlign );
	inline void ensureAdditionalMem( sal_Int32 nMemToAdd );
	void packTypecode( typelib_TypeDescription *pType );

	::com::sun::star::uno::Sequence < sal_Int8 > m_seqData;
	sal_Int8 *m_base;
	sal_Int8 *m_pos;
};

template < class C >
inline void operator <<= (Marshal & r,  C& value)
{
	r.pack( (void * ) &value, getCppuType( &value ) );
}



/****
 * Unmarshaler
 * ----------------------------------
 * Retrieves uno values out of a memory block.
 *
 ****/
class Unmarshal
{
public:
	Unmarshal( const sal_Int8 *pMem , sal_Int32 nLength );

	void unpackAndDestruct( void *pDest , const ::com::sun::star::uno::Type &rType );
	void unpackAndDestruct( void *pDest , typelib_TypeDescription *pType );
	void unpack( void *pDest , typelib_TypeDescription *pType );
private:
	void align( sal_Int32 nAlignTo );
	void unpackTypecode( typelib_TypeDescription **ppType );

	const sal_Int8 *m_pos;
	sal_Int32 m_nLength;
};

template < class C >
inline void operator >>= (Unmarshal & r, C& value)
{
	r.unpackAndDestruct( &value , getCppuType( &value ) ); //&value, pType );
}






/****
 * Marshaler impl
 ****/
Marshal::Marshal() :
	m_seqData( 512 ),
	m_pos( m_seqData.getArray() ),
	m_base( m_seqData.getArray() )
{}


inline void Marshal::align( sal_Int32 nToAlign )
{
	sal_Int32 n = ( (sal_uInt32)m_pos % nToAlign);
	if( n )
	{
		n = nToAlign -n;
	}
	m_pos += n;
}

inline void Marshal::ensureAdditionalMem( sal_Int32 nMemToAdd )
{
	sal_Int32 nDiff = m_pos - m_base;
	if( nDiff + nMemToAdd > m_seqData.getLength() )
	{
		m_seqData.realloc(
			m_seqData.getLength() *2 > nDiff+nMemToAdd ?
			m_seqData.getLength() *2 :
			nDiff + nMemToAdd );

		m_base = m_seqData.getArray();
		m_pos = m_base + nDiff;
	}
}

Sequence< sal_Int8 >  Marshal::remove()
{
	m_seqData.realloc( (sal_Int32) ( m_pos - m_base ) );

	Sequence < sal_Int8 > seq = m_seqData;
	m_seqData = Sequence< sal_Int8 > ();
	return seq;
}

void Marshal::pack( void *pSource , const ::com::sun::star::uno::Type &rType )
{
	typelib_TypeDescription * pDataTD = 0;
	TYPELIB_DANGER_GET( &pDataTD, rType.getTypeLibType() );
	pack( pSource, pDataTD );
	TYPELIB_DANGER_RELEASE( pDataTD );
}


/****
 * Packs a type description
 ***/
void Marshal::packTypecode( typelib_TypeDescription *pType )
{
	sal_Int32 i = ( sal_Int32 ) pType->eTypeClass;
	(*this) <<= i;

	switch( pType->eTypeClass )
	{
	case typelib_TypeClass_BYTE:
	case typelib_TypeClass_BOOLEAN:
	case typelib_TypeClass_CHAR:
	case typelib_TypeClass_SHORT:
	case typelib_TypeClass_UNSIGNED_SHORT:
	case typelib_TypeClass_FLOAT:
	case typelib_TypeClass_LONG:
	case typelib_TypeClass_UNSIGNED_LONG:
	case typelib_TypeClass_HYPER:
	case typelib_TypeClass_UNSIGNED_HYPER:
	case typelib_TypeClass_DOUBLE:
	case typelib_TypeClass_STRING:
	case typelib_TypeClass_ANY:
	case typelib_TypeClass_VOID:
	case typelib_TypeClass_TYPE:
	{
		break;
	}

	case typelib_TypeClass_TYPEDEF:
	{
		assert( 0 );
		break;
	}

	// complex types
	case typelib_TypeClass_ENUM:
	case typelib_TypeClass_UNION:
	case typelib_TypeClass_STRUCT:
	case typelib_TypeClass_EXCEPTION:
	case typelib_TypeClass_INTERFACE:
	case typelib_TypeClass_SEQUENCE:
	{
		(*this) <<= pType->pTypeName->length;
		ensureAdditionalMem( pType->pTypeName->length );
		memcpy( m_pos , pType->pTypeName->buffer , pType->pTypeName->length );
		m_pos += pType->pTypeName->length;
		break;
	}

	case typelib_TypeClass_ARRAY:
	case typelib_TypeClass_SERVICE:
	case typelib_TypeClass_MODULE:
	case typelib_TypeClass_INTERFACE_METHOD:
	case typelib_TypeClass_INTERFACE_ATTRIBUTE:
	case typelib_TypeClass_UNKNOWN:
	default:
		assert( 0 );
	}
}

void Marshal::pack( void *pSource , typelib_TypeDescription *pType )
{
	switch( pType->eTypeClass )
	{
	case typelib_TypeClass_BYTE:
	{
		ensureAdditionalMem( 1 );
		*m_pos = *((sal_Int8*) pSource );
		m_pos++;
		break;
	}
	case typelib_TypeClass_BOOLEAN:
	{
		ensureAdditionalMem( 1 );
		*m_pos = ( *((sal_Bool*) pSource ) ) ? 1 : 0;
		m_pos++;
		break;
	}

	case typelib_TypeClass_CHAR:
	{
		align( 2 );
		ensureAdditionalMem( 2 );
		sal_Unicode *p = ( sal_Unicode *  ) pSource;
		*m_pos = (sal_Int8) ( ( (*p) >> 8  ) & 0xff );
		m_pos++;
		*m_pos = (sal_Int8) (   (*p)         & 0xff );
		m_pos++;
		break;
	}
	case typelib_TypeClass_SHORT:
	case typelib_TypeClass_UNSIGNED_SHORT:
	{
		align( 2 );
		ensureAdditionalMem( 2 );
		sal_uInt16 *p = ( sal_uInt16 *  ) pSource;
		*m_pos = (sal_Int8) ( ((*p) >> 8) & 0xff );
		m_pos++;
		*m_pos = (sal_Int8) ( (*p)        & 0xff );
		m_pos++;
		break;
	}
	case typelib_TypeClass_FLOAT:
	case typelib_TypeClass_LONG:
	case typelib_TypeClass_UNSIGNED_LONG:
	{
		align( 4 );
		ensureAdditionalMem( 4 );
		sal_uInt32 *p = ( sal_uInt32 * ) pSource;

		*m_pos = (sal_uInt8) ( ( (*p) >> 24 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 16 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 8  ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) (   (*p)         & 0xff );
		m_pos++;
		break;
	}
	case typelib_TypeClass_HYPER:
	case typelib_TypeClass_UNSIGNED_HYPER:
	case typelib_TypeClass_DOUBLE:
	{
		align( 8 );
		ensureAdditionalMem( 8 );
		sal_uInt64 *p = ( sal_uInt64 * ) pSource;
		*m_pos = (sal_uInt8) ( ( (*p) >> 56 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 48 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 40 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 32 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 24 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 16 ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) ( ( (*p) >> 8  ) & 0xff );
		m_pos++;
		*m_pos = (sal_uInt8) (   (*p)         & 0xff );
		m_pos++;
		break;
	}
	case typelib_TypeClass_STRING:
	{
		rtl_wString *p = *( rtl_wString ** ) pSource;
		(*this) <<= ( p->length);

		ensureAdditionalMem( p->length * sizeof( sal_Unicode ) );

		for( sal_Int32 i = 0; i < p->length ; i ++ )
		{
			sal_Unicode c = (sal_Unicode) p->buffer[i];

			*m_pos = (sal_Int8) ( ( c >> 8  ) & 0xff );
			m_pos++;
			*m_pos = (sal_Int8) (   c         & 0xff );
			m_pos++;
		}
		break;
	}
	case typelib_TypeClass_ANY:
	{
		uno_Any *pAny = (uno_Any * ) pSource;

		packTypecode( pAny->pTypeDescription );
		pack( pAny->pData , pAny->pTypeDescription );
		break;
	}
	case typelib_TypeClass_ENUM:
	{
		sal_Int32 nValue = *(sal_Int32*)pSource;
		(*this) <<= nValue;
		break;
	}
	case typelib_TypeClass_TYPEDEF:
	{
		pack( pSource , ((typelib_IndirectTypeDescription *) pType )->pType );
		break;
	}
	case typelib_TypeClass_EXCEPTION:
	case typelib_TypeClass_STRUCT:
	{
		typelib_CompoundTypeDescription * pCompType =
			(typelib_CompoundTypeDescription *)pType;

		if (pCompType->pBaseTypeDescription)
		{
			pack( pSource , (typelib_TypeDescription * )pCompType->pBaseTypeDescription );
		}

		// then construct members
		typelib_TypeDescriptionReference ** ppTypeRefs = pCompType->ppTypeRefs;
		sal_Int32 * pMemberOffsets = pCompType->pMemberOffsets;
		sal_Int32 nDescr = pCompType->nMembers;

		for ( sal_Int32 nPos = 0; nPos < nDescr; ++nPos )
		{
			typelib_TypeDescription * pMemberType = 0;
			TYPELIB_DANGER_GET( &pMemberType, ppTypeRefs[nPos] );
			pack( (char*)pSource + pMemberOffsets[nPos] , pMemberType );
			TYPELIB_DANGER_RELEASE( pMemberType );
		}
		break;
	}
	case typelib_TypeClass_SEQUENCE:
	{
		typelib_TypeDescription * pElementType =
			((typelib_IndirectTypeDescription *)pType)->pType;

		sal_Int32 nElements		 = (*(uno_Sequence **)pSource)->nElements;
		sal_Int32 nElementSize	 = pElementType->nSize;

		char * pSourceElements	 = (char *)(*(uno_Sequence **)pSource)->pElements;
		(*this) <<= nElements;
		for ( sal_Int32 i = 0 ; i < nElements; i++ )
		{
			pack( pSourceElements + (nElementSize*i) , pElementType );
		}
		break;
	}
	case typelib_TypeClass_UNION:
	{
		// not supported
		assert( 0 );
		break;
	}

	case typelib_TypeClass_TYPE:
	{
		typelib_TypeDescriptionReference * p =
			*( typelib_TypeDescriptionReference **  ) pSource;

		typelib_TypeDescription *pType = 0;
		typelib_typedescriptionreference_getDescription( &pType , p );

		packTypecode( pType );

		typelib_typedescription_release( pType );
		break;
	}
	case typelib_TypeClass_INTERFACE:
	case typelib_TypeClass_VOID:
	{
		// nothing to do
		break;
	}
	case typelib_TypeClass_ARRAY:
	case typelib_TypeClass_SERVICE:
	case typelib_TypeClass_MODULE:
	case typelib_TypeClass_INTERFACE_METHOD:
	case typelib_TypeClass_INTERFACE_ATTRIBUTE:
	case typelib_TypeClass_UNKNOWN:
	default:
		assert( 0 );
	}
}




/*******
 * Unmarshaler impl
 ******/
Unmarshal::Unmarshal( const sal_Int8 *pMem ,
					  sal_Int32 nLength ) :
	m_pos( pMem ),
	m_nLength( nLength )
{

}


inline void Unmarshal::align( sal_Int32 nToAlign )
{
	sal_Int32 n = ( (sal_uInt32)m_pos % nToAlign);
	if( n )
	{
		n = nToAlign -n;
	}
	m_pos += n;
}


void Unmarshal::unpackAndDestruct( void *pDest, const ::com::sun::star::uno::Type &rType)
{
	typelib_TypeDescription * pDataTD = 0;
	TYPELIB_DANGER_GET( &pDataTD, rType.getTypeLibType() );
	uno_destructData( pDest , pDataTD , 0 );
	unpack( pDest, pDataTD );
	TYPELIB_DANGER_RELEASE( pDataTD );
}

void Unmarshal::unpackAndDestruct( void *pDest, typelib_TypeDescription *pType)
{
	uno_destructData( pDest , pType  , 0 );
	unpack( pDest, pType );
}




void Unmarshal::unpackTypecode( typelib_TypeDescription **ppType )
{
	sal_Int32 eTypeClass;
	(*this) >>= eTypeClass;

	switch( eTypeClass )
	{
	case typelib_TypeClass_BYTE:
	{
		getCppuType( (sal_Int8*)0).getDescription(ppType);
		break;
	}
	case typelib_TypeClass_BOOLEAN:
	{
		getCppuBooleanType().getDescription( ppType );
		break;
	}
	case typelib_TypeClass_CHAR:
	{
		getCppuCharType( ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_SHORT:
	{
		getCppuType( ( sal_Int16 * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_UNSIGNED_SHORT:
	{
		getCppuType( ( sal_uInt16 * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_FLOAT:
	{
		getCppuType( ( float * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_DOUBLE:
	{
		getCppuType( ( double * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_LONG:
	{
		getCppuType( ( sal_Int32 * ) 0 ).getDescription( ppType );
		break;
	}

	case typelib_TypeClass_UNSIGNED_LONG:
	{
		getCppuType( ( sal_uInt32 * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_HYPER:
	{
		getCppuType( ( sal_Int64 * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_UNSIGNED_HYPER:
	{
		getCppuType( ( sal_uInt64 * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_STRING:
	{
		getCppuType( ( OWString * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_ANY:
	{
		getCppuType( ( Any * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_VOID:
	{
		getCppuVoidType().getDescription( ppType );
		break;
	}
	case typelib_TypeClass_TYPE:
	{
		getCppuType( ( Type * ) 0 ).getDescription( ppType );
		break;
	}
	case typelib_TypeClass_TYPEDEF:
	{
		assert( 0 );
		break;
	}

	// complex types
	case typelib_TypeClass_UNION:
	case typelib_TypeClass_STRUCT:
	case typelib_TypeClass_EXCEPTION:
	case typelib_TypeClass_INTERFACE:
	case typelib_TypeClass_SEQUENCE:
	case typelib_TypeClass_ENUM:
	{
		// unpack string 8 !
		sal_Int32 nLen;
		(*this) >>= nLen;

		sal_Char *p = ( sal_Char * ) alloca( nLen + 1 );
		memcpy( p , m_pos , nLen );
		p[nLen] = 0;

		m_pos += nLen;

		typelib_typedescription_getByName( ppType , p );
		break;
	}
	default:
		assert( 0 );
	}
}



void Unmarshal::unpack( void *pDest , typelib_TypeDescription *pType )
{
	switch( pType->eTypeClass )
	{
	case typelib_TypeClass_VOID:
		// do nothing
		break;
	case typelib_TypeClass_BYTE:
	{
		*((sal_Int8*)pDest ) = *m_pos;
		m_pos++;
		break;
	}
	case typelib_TypeClass_BOOLEAN:
	{
		*((sal_Bool*)pDest) = (sal_Bool ) ( *m_pos);
		m_pos ++;
		break;
	}

	case typelib_TypeClass_CHAR:
	{
		align( 2 );
		sal_Unicode *p = ( sal_Unicode * ) pDest;
		*p = 0;
		*p |= ( (sal_Unicode) (*(sal_uInt8 *)m_pos) ) << 8;
		m_pos++;
		*p += ( (sal_Unicode) (*(sal_uInt8 *)m_pos) );
		m_pos++;
		break;
	}
	case typelib_TypeClass_SHORT:
	case typelib_TypeClass_UNSIGNED_SHORT:
	{
		align( 2 );
		sal_uInt16 *p = ( sal_uInt16 * ) pDest;
		*p = 0;
		*p |= (*(sal_uInt8 *)m_pos) << 8;
		m_pos++;
		*p += (*(sal_uInt8 *)m_pos);
		m_pos++;
		break;
	}
	case typelib_TypeClass_FLOAT:
	case typelib_TypeClass_LONG:
	case typelib_TypeClass_UNSIGNED_LONG:
	{
		align( 4 );
		sal_uInt32 *p = ( sal_uInt32 * ) pDest;
		*p = 0;

		*p |=  (*(sal_uInt8 *)m_pos) << 24;
		m_pos++;
		*p |=  (*(sal_uInt8 *)m_pos) << 16;
		m_pos++;
		*p |=  (*(sal_uInt8 *)m_pos) << 8;
		m_pos++;
		*p |=  (*(sal_uInt8 *)m_pos);
		m_pos++;
		break;
	}
	case typelib_TypeClass_HYPER:
	case typelib_TypeClass_UNSIGNED_HYPER:
	case typelib_TypeClass_DOUBLE:
	{
		align( 8 );
		sal_uInt64 *p = ( sal_uInt64 * ) pDest;
		*p = 0;

		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 56;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 48;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 40;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 32;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 24;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 16;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) ) << 8;
		m_pos++;
		*p |=  ( (sal_uInt64) (*(sal_uInt8 *)m_pos) );
		m_pos++;

		break;
	}
	case typelib_TypeClass_STRING:
	{
		sal_Int32 nLen;
		(*this) >>= nLen;

		sal_Unicode *pMem = (sal_Unicode*) alloca( nLen * sizeof( sal_Unicode )  );
		for( sal_Int32 i = 0; i < nLen ; i ++ )
		{
			pMem[i] = 0;

			pMem[i] |= ( ( sal_Unicode ) (*(sal_uInt8 *)m_pos) ) << 8;
			m_pos++;
			pMem[i] |= ( ( sal_Unicode ) (*(sal_uInt8 *)m_pos) );
			m_pos++;
		}

		*(rtl_wString ** ) pDest = 0;
		rtl_wString_newFromStr_WithLength( (rtl_wString **) pDest , pMem , nLen );
		break;
	}
	case typelib_TypeClass_ANY:
	{
		uno_Any *pAny = ( uno_Any * )pDest;

		pAny->pTypeDescription = 0;

		// Type is acquired with typelib_typedescription_acquire
		unpackTypecode( &(pAny->pTypeDescription) );

		pAny->pData = rtl_allocateMemory( pAny->pTypeDescription->nSize );
		unpack( pAny->pData , pAny->pTypeDescription );
		break;
	}

	case typelib_TypeClass_ENUM:
	{
		sal_Int32 nValue;
		(*this) >>= nValue;
		*(sal_Int32 *)pDest = nValue;
		break;
	}
	case typelib_TypeClass_TYPEDEF:
	{
		unpack( pDest , ((typelib_IndirectTypeDescription *) pType )->pType );
		break;
	}

	case typelib_TypeClass_STRUCT:
	case typelib_TypeClass_EXCEPTION:
	{
		typelib_CompoundTypeDescription * pCompType =
			(typelib_CompoundTypeDescription *)pType;

		if (pCompType->pBaseTypeDescription)
		{
			unpack( pDest , (typelib_TypeDescription * ) pCompType->pBaseTypeDescription );
		}

		// then construct members
		typelib_TypeDescriptionReference ** ppTypeRefs = pCompType->ppTypeRefs;
		sal_Int32 * pMemberOffsets = pCompType->pMemberOffsets;
		sal_Int32 nDescr = pCompType->nMembers;

		for ( sal_Int32 nPos = 0; nPos < nDescr; ++nPos )
		{
			typelib_TypeDescription * pMemberType = 0;
			TYPELIB_DANGER_GET( &pMemberType, ppTypeRefs[nPos] );
			unpack( (char*)pDest + pMemberOffsets[nPos] , pMemberType );
			TYPELIB_DANGER_RELEASE( pMemberType );
		}
		break;
	}
	case typelib_TypeClass_SEQUENCE:
	{
		sal_Int32 nLen;
		(*this) >>= nLen;

		uno_Sequence * pSequence = (uno_Sequence *)rtl_allocateMemory( sizeof(uno_Sequence) );
		pSequence->nRefCount = 1;
		pSequence->nElements = nLen;

		if (nLen)
		{
			typelib_TypeDescription * pET = ((typelib_IndirectTypeDescription *)pType)->pType;
			sal_Int32 nElementSize = pET->nSize;

			pSequence->pElements = rtl_allocateMemory( nElementSize * nLen );
			for( sal_Int32 i = 0 ; i < nLen ; i ++ )
			{
				unpack( ((char*)pSequence->pElements)+ i*nElementSize,pET );
			}
		}
		else
		{
			pSequence->pElements = (void *)0xdeadbeef;
		}

		*((uno_Sequence **)pDest) = pSequence;
		break;
	}
	case typelib_TypeClass_INTERFACE:
	{
		*(void **)pDest = 0;
		break;
	}

	case typelib_TypeClass_TYPE:
	{
		typelib_TypeDescription *pType = 0;

		unpackTypecode( &pType );

		assert( pType->pWeakRef );
		typelib_typedescriptionreference_acquire( pType->pWeakRef  );

		*( typelib_TypeDescriptionReference **  ) pDest =
			pType->pWeakRef;

		typelib_typedescription_release( pType );
		break;
	}

	case typelib_TypeClass_UNION:
	case typelib_TypeClass_ARRAY:
	case typelib_TypeClass_SERVICE:
	case typelib_TypeClass_MODULE:
	case typelib_TypeClass_INTERFACE_METHOD:
	case typelib_TypeClass_INTERFACE_ATTRIBUTE:
	case typelib_TypeClass_UNKNOWN:
	default:
		assert( 0 );
	}
}


} // end namespace persistimpl




/*****
 * The exported methods
 *****/
::com::sun::star::uno::Sequence < sal_Int8 > SAL_CALL
anySerialize(	const ::com::sun::star::uno::Any & any )
{
	::persist_Impl::Marshal m;
	m <<= any;
	return m.remove();
}

::com::sun::star::uno::Any  SAL_CALL
anyDeserialize( const ::com::sun::star::uno::Sequence < sal_Int8 > &s )
{
	::persist_Impl::Unmarshal u(s.getConstArray() , s.getLength() );
	Any any;
	u >>= any;
	return any;
}

