/*************************************************************************
 *
 *  $RCSfile: inetldap.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/03/19 12:12:56 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): Matthias Huetsch <matthias.huetsch@sun.com>
 *
 *
 ************************************************************************/

#define _INETCORELDAP_CXX "$Revision: 1.2 $"

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _OSL_INTERLCK_H_
#include <osl/interlck.h>
#endif

#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif
#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif
#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _VOS_MUTEX_HXX_
#include <vos/mutex.hxx>
#endif
#ifndef _VOS_REF_HXX_
#include <vos/ref.hxx>
#endif

#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif

#ifndef _INET_MACROS_HXX
#include <inet/macros.hxx>
#endif
#ifndef _INET_SOCKET_HXX
#include <inet/socket.hxx>
#endif

#ifndef _INET_CONFIG_HXX
#include <inetcfg.hxx>
#endif
#ifndef _INETDNS_HXX
#include <inetdns.hxx>
#endif

#ifndef _INETCORELDAP_HXX
#include <inetldap.hxx>
#endif
#ifndef _LDAPMSG_HXX
#include <ldapmsg.hxx>
#endif

#ifdef _USE_NAMESPACE
using namespace inet;
#endif

using rtl::OUString;

/*=======================================================================
 *
 * INetCoreLDAPConnection implementation.
 *
 * References:
 *   RFC 1777 - Lightweight Directory Access Protocol (Draft Standard).
 *   RFC 1823 - The LDAP Application Program Interface (Informational).
 *   RFC 1920 - Internet Official Protocol Standards (STD 1).
 *
 *=====================================================================*/
#define INETCORELDAP_SOCKET_DISPOSE(socket) \
if ((socket).isValid()) \
{ \
	(socket)->deregisterEventHandler(onSocketEvent); \
	(socket)->close(); \
	(socket).unbind(); \
}

#define INETCORELDAP_SOCKET_WOULDBLOCK (-osl_Socket_E_WouldBlock)

/*=======================================================================
 *
 * INetCoreLDAPRequestContext interface.
 *
 *=====================================================================*/
struct INetCoreLDAPRequestContext
{
	INetCoreLDAPRequestContext (const INetCoreLDAPMessage& rMsg);
	~INetCoreLDAPRequestContext (void);

	sal_uInt32  m_nBufSiz;
	sal_uInt8  *m_pBuffer;
	sal_uInt8  *m_pRead;
	sal_uInt8  *m_pWrite;

	sal_uInt16  m_nMessageID;
};

/*=======================================================================
 *
 * INetCoreLDAPConnectionContext interface.
 *
 *=====================================================================*/
struct INetCoreLDAPConnectionContext
{
	INetCoreLDAPConnectionContext  (sal_uInt32 nBufferSize = 16384);
	~INetCoreLDAPConnectionContext (void);

	/** Representation.
	 */
	INetCoreDNSResolver     *m_pResolver;
	NAMESPACE_VOS(ORef)<INetActiveTCPSocket> m_xSocket;

	sal_uInt32               m_nBufSiz;
	sal_uInt8               *m_pBuffer;
	sal_uInt8               *m_pRead;
	sal_uInt8               *m_pWrite;

	sal_Bool                 m_bIsOpen   : 1;
	sal_Bool                 m_bAborting : 1;

	INetCoreLDAPCallback    *m_pfnCB;
	void                    *m_pDataCB;

	NAMESPACE_VOS(OMutex)    m_aMutex;
	List                     m_aQueue;
	oslInterlockedCount      m_nRequest;

	/** create.
	 */
	void create (INetCoreDNSHostEntry &rDstAddr);

	/** Request handling.
	 */
	sal_uInt16 nextID (void)
	{
		oslInterlockedCount n = osl_incrementInterlockedCount (&m_nRequest);
		return (n & 0xffff);
	}

	void push_back (INetCoreLDAPRequestContext *pRequest)
	{
		NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
		m_aQueue.Insert (pRequest, LIST_APPEND);
	}

	void push_front (INetCoreLDAPRequestContext *pRequest)
	{
		NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
		m_aQueue.Insert (pRequest, (ULONG)0);
	}

	sal_Bool pop_front (INetCoreLDAPRequestContext *&rpRequest)
	{
		NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
		rpRequest = (INetCoreLDAPRequestContext*)(m_aQueue.Remove());
		return (!!rpRequest);
	}

	sal_Bool empty (void)
	{
		NAMESPACE_VOS(OGuard) aGuard (m_aMutex);
		return (m_aQueue.Count() == 0);
	}
};

/*=======================================================================
 *
 * INetCoreLDAPConnection implementation.
 *
 *=====================================================================*/
/*
 * INetCoreLDAPConnection.
 */
INetCoreLDAPConnection::INetCoreLDAPConnection (void)
	: m_pConCtx (new INetCoreLDAPConnectionContext())
{
	VOS_POSTCOND(m_pConCtx, "INetCoreLDAPConnection::ctor(): no context");
}

/*
 * ~INetCoreLDAPConnection.
 */
INetCoreLDAPConnection::~INetCoreLDAPConnection (void)
{
	VOS_PRECOND(m_pConCtx, "INetCoreLDAPConnection::dtor(): no context");
	if (m_pConCtx)
	{
		INETCORELDAP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
		delete m_pConCtx;
	}
}

/*
 * handleResolverEvent.
 */
sal_Bool INetCoreLDAPConnection::handleResolverEvent (
	sal_Int32 nStatus, INetCoreDNSHostEntry *pHostEntry)
{
	// Check connection context.
	if (m_pConCtx == NULL)
		return 0;

	// Check for abort during name resolution.
	if (m_pConCtx->m_bAborting)
		nStatus = INETCOREDNS_RESOLVER_ERROR;

	switch (nStatus)
	{
		case INETCOREDNS_RESOLVER_START:
			// Notify caller.
			if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
				this, INETCORELDAP_REPLY_RESOLVER_WAIT,
				NULL, 0, m_pConCtx->m_pDataCB);

			// Wait for next event.
			return 1;

		case INETCOREDNS_RESOLVER_SUCCESS:
		case INETCOREDNS_RESOLVER_EXPIRED:
			// Notify caller.
			if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
				this, INETCORELDAP_REPLY_RESOLVER_DONE,
				NULL, 0, m_pConCtx->m_pDataCB);

			// Initialize active socket to destination.
			m_pConCtx->create (*pHostEntry);
			m_pConCtx->m_xSocket->registerEventHandler (onSocketEvent, this);

			// Initiate connect.
			if (m_pConCtx->m_xSocket->connect (
				NAMESPACE_VOS(OInetSocketAddr)(
					pHostEntry->GetDottedDecimalName(),
					pHostEntry->GetPort())))
			{
				// Ok. Cleanup and notify caller.
				delete pHostEntry;

				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_CONNECT_WAIT,
					NULL, 0, m_pConCtx->m_pDataCB);

				// Wait for connect event.
				return 1;
			}
			else
			{
				// Failure. Cleanup and notify caller.
				m_pConCtx->m_xSocket.unbind();
				delete pHostEntry;

				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_CONNECT_ERROR,
					NULL, 0, m_pConCtx->m_pDataCB);
			}
			break;

		default:
			// Failure. Cleanup and notify caller.
			delete pHostEntry;

			if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
				this, INETCORELDAP_REPLY_RESOLVER_ERROR,
				NULL, 0, m_pConCtx->m_pDataCB);
			break;
	}

	// Done.
	return 0;
}

/*
 * handleSocketEvent.
 */
sal_Bool INetCoreLDAPConnection::handleSocketEvent (
	const NAMESPACE_VOS(ORef)<socket_type> &rxSocket, sal_Int32 nEvent)
{
	// Check connection context.
	if (m_pConCtx == NULL)
		return 0;

	// Check for abort during event wait.
	if (m_pConCtx->m_bAborting)
		return 0;

	// Process event.
	while (1)
	{
		if (nEvent & socket_type::EVENT_CONNECT)
		{
			// Determine connect status.
			m_pConCtx->m_bIsOpen = ((nEvent & socket_type::EVENT_WRITE) > 0);

			if (m_pConCtx->m_bIsOpen)
			{
				// Connected. Notify caller.
				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_CONNECT_DONE,
					NULL, 0, m_pConCtx->m_pDataCB);

				// Check for request(s) to send.
				if (!m_pConCtx->empty())
					nEvent = socket_type::EVENT_WRITE;
				else
					return 1;
			}
			else
			{
				// Connect failed.
				m_pConCtx->m_xSocket.unbind();

				// Notify caller.
				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_CONNECT_ERROR,
					NULL, 0, m_pConCtx->m_pDataCB);

				return 0;
			}
		}
		else if (nEvent & socket_type::EVENT_READ)
		{
			sal_uInt32 nSize = m_pConCtx->m_nBufSiz;
			nSize -= (m_pConCtx->m_pRead - m_pConCtx->m_pWrite);

			sal_Int32 nRead = rxSocket->recv (m_pConCtx->m_pRead, nSize);
			if (nRead > 0)
			{
				// Adjust read pointer.
				m_pConCtx->m_pRead += nRead;

				// Decode response message(s).
				while ((m_pConCtx->m_pRead - m_pConCtx->m_pWrite) > 0)
				{
					INetCoreLDAPMessage aMsg (
						&(m_pConCtx->m_pWrite),
						(m_pConCtx->m_pRead - m_pConCtx->m_pWrite));
					if (aMsg.IsValid())
					{
						sal_uInt16 nMsgID = aMsg.GetMessageID();
						if (aMsg.IsResult())
						{
							INetCoreLDAPResult *pResult = aMsg.GetResult();
							if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
								this, INETCORELDAP_REPLY_RESPONSE_DONE,
								pResult, nMsgID, m_pConCtx->m_pDataCB);
							delete pResult;
						}
						else if (aMsg.IsEntry())
						{
							INetCoreLDAPEntry *pEntry = aMsg.GetEntry();
							if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
								this, INETCORELDAP_REPLY_SEARCHENTRY,
								pEntry, nMsgID, m_pConCtx->m_pDataCB);
							delete pEntry;
						}
						else
						{
							// Oops, what is this ???
						}
					}
					else
					{
						// Move remainder to begin of buffer.
						nRead = m_pConCtx->m_pRead - m_pConCtx->m_pWrite;
						rtl_moveMemory (
							m_pConCtx->m_pBuffer, m_pConCtx->m_pWrite, nRead);

						m_pConCtx->m_pRead  = m_pConCtx->m_pBuffer + nRead;
						m_pConCtx->m_pWrite = m_pConCtx->m_pBuffer;

						// Read more.
						break;
					}
				}
				if (m_pConCtx->m_pWrite == m_pConCtx->m_pRead)
				{
					// Reset to empty buffer.
					m_pConCtx->m_pRead = m_pConCtx->m_pWrite =
						m_pConCtx->m_pBuffer;
				}
			}
			else if (nRead == INETCORELDAP_SOCKET_WOULDBLOCK)
			{
				// Notify caller.
				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_RESPONSE_WAIT,
					NULL, 0, m_pConCtx->m_pDataCB);

				// Wait for next event.
				return 1;
			}
			else
			{
				// Socket closed by peer or read error.
				rxSocket->close();

				// Notify caller.
				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_RESPONSE_ERROR,
					NULL, 0, m_pConCtx->m_pDataCB);

				// Wait for close event.
				return 1;
			}
		}
		else if (nEvent & socket_type::EVENT_WRITE)
		{
			INetCoreLDAPRequestContext *pCtx;
			if (m_pConCtx->pop_front (pCtx))
			{
				// Send request.
				sal_uInt16 nMsgID = pCtx->m_nMessageID;

				while ((pCtx->m_pRead - pCtx->m_pWrite) > 0)
				{
					sal_Int32 nWrite = rxSocket->send (
						pCtx->m_pWrite, (pCtx->m_pRead - pCtx->m_pWrite));
					if (nWrite > 0)
					{
						// Adjust write pointer.
						pCtx->m_pWrite += nWrite;
					}
					else if (nWrite == INETCORELDAP_SOCKET_WOULDBLOCK)
					{
						// Put request back.
						m_pConCtx->push_front (pCtx);

						// Notify caller.
						if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
							this, INETCORELDAP_REPLY_REQUEST_WAIT,
							NULL, nMsgID, m_pConCtx->m_pDataCB);

						// Wait for next event.
						return 1;
					}
					else
					{
						// Socket closed by peer or write error.
						rxSocket->close();

						// Cleanup and notify caller.
						delete pCtx;
						if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
							this, INETCORELDAP_REPLY_REQUEST_ERROR,
							NULL, nMsgID, m_pConCtx->m_pDataCB);

						// Wait for close event.
						return 1;
					}
				}

				// Request done. Cleanup and notify caller.
				delete pCtx;
				if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
					this, INETCORELDAP_REPLY_REQUEST_DONE,
					NULL, nMsgID, m_pConCtx->m_pDataCB);
			}
			else
			{
				// No more requests. Try reading a response.
				nEvent = socket_type::EVENT_READ;
			}
		}
		else if (nEvent & socket_type::EVENT_CLOSE)
		{
			m_pConCtx->m_xSocket.unbind();
			m_pConCtx->m_bIsOpen = sal_False;
			m_pConCtx->m_nRequest = 0;

			if (m_pConCtx->m_pfnCB) (m_pConCtx->m_pfnCB) (
				this, INETCORELDAP_REPLY_CONNECTION_TERMINATED,
				NULL, 0, m_pConCtx->m_pDataCB);

			return 0;
		}
		else
		{
			// Ignore event.
			return 1;
		}
	}
}

/*
 * startRequest.
 */
sal_uInt16 INetCoreLDAPConnection::startRequest (
	INetCoreLDAPMessage &rMessage)
{
	// Ensure clean destruction.
	NAMESPACE_VOS(ORef)<INetClientConnection_Impl> xThis (this);

	// Check connection context.
	if (m_pConCtx && m_pConCtx->m_bIsOpen && !m_pConCtx->m_bAborting)
	{
		// Enqueue request message.
		if (rMessage.SetMessageID (m_pConCtx->nextID()))
			m_pConCtx->push_back (new INetCoreLDAPRequestContext (rMessage));
		else
			return 0;

		// Start request.
		if (m_pConCtx->m_xSocket->postEvent (socket_type::EVENT_WRITE))
			return rMessage.GetMessageID();
		else
			return 0;
	}

	// Request failed.
	return 0;
}

/*
 * Open.
 * Initiate connection to LDAPD.
 */
sal_Bool INetCoreLDAPConnection::Open (
	const OUString &rHost, sal_uInt16 nPort,
	INetCoreLDAPCallback *pfnCallback, void *pData)
{
	// Check connection context.
	if (!m_pConCtx || m_pConCtx->m_bIsOpen || m_pConCtx->m_bAborting)
		return sal_False;

	m_pConCtx->m_bAborting = m_pConCtx->m_bIsOpen = sal_False;

	// Check arguments.
	if ((rHost.getLength() == 0) || (pfnCallback == NULL))
		return sal_False;
	if (nPort == 0)
		nPort = INETCORELDAP_DEF_PORT;

	// Setup callback.
	m_pConCtx->m_pfnCB   = pfnCallback;
	m_pConCtx->m_pDataCB = pData;

	// Start domain name resolution.
	INetCoreDNSHostEntry *he = new INetCoreDNSHostEntry (rHost, nPort);

	if (m_pConCtx->m_pResolver->GetHostByName (he, onResolverEvent, this))
	{
		// Ok. Wait for things to come.
		return sal_True;
	}
	else
	{
		// Cleanup and fail.
		delete he;

		m_pConCtx->m_pfnCB   = NULL;
		m_pConCtx->m_pDataCB = NULL;

		return sal_False;
	}
}

/*
 * IsOpen.
 */
sal_Bool INetCoreLDAPConnection::IsOpen (void)
{
	return (m_pConCtx ? m_pConCtx->m_bIsOpen : sal_False);
}

/*
 * Abort.
 */
void INetCoreLDAPConnection::Abort (void)
{
	if (m_pConCtx)
	{
		m_pConCtx->m_bIsOpen   = sal_False;
		m_pConCtx->m_bAborting = sal_True;

		m_pConCtx->m_pfnCB   = NULL;
		m_pConCtx->m_pDataCB = NULL;

		INETCORELDAP_SOCKET_DISPOSE (m_pConCtx->m_xSocket);
	}
}

/*
 * Bind.
 */
sal_uInt16 INetCoreLDAPConnection::Bind (
	const OUString &rUserDN, const OUString &rPassword)
{
	// Initialize bind request message.
	INetCoreLDAPBindRequestMessage aMsg;
	aMsg.SetLDAPVersion (2);

	if (rUserDN.getLength())
		aMsg.SetObjectName (rUserDN);
	if (rPassword.getLength())
		aMsg.SetObjectPassword (rPassword);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Unbind.
 */
sal_uInt16 INetCoreLDAPConnection::Unbind (void)
{
	// Initialize unbind request message.
	INetCoreLDAPUnbindRequestMessage aMsg;

	// Start request.
	return startRequest (aMsg);
}

/*
 * Add.
 */
sal_uInt16 INetCoreLDAPConnection::Add (const INetCoreLDAPEntry& rNewEntry)
{
	// Initialize add request message.
	INetCoreLDAPAddRequestMessage aMsg;
	aMsg.SetEntry (rNewEntry);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Compare.
 */
sal_uInt16 INetCoreLDAPConnection::Compare (
	const OUString                       &rObjectDN,
	const INetCoreLDAPAttributeValuePair &rAssertion)
{
	// Check arguments.
	if (rObjectDN.getLength() == 0)
		return 0;

	// Initialize compare request message.
	INetCoreLDAPCompareRequestMessage aMsg;
	aMsg.SetObjectName (rObjectDN);
	aMsg.SetAssertion  (rAssertion);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Delete.
 */
sal_uInt16 INetCoreLDAPConnection::Delete (const OUString &rObjectDN)
{
	// Check arguments.
	if (rObjectDN.getLength() == 0)
		return 0;

	// Initialize add request message.
	INetCoreLDAPDeleteRequestMessage aMsg;
	aMsg.SetObjectName (rObjectDN);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Modify.
 */
sal_uInt16 INetCoreLDAPConnection::Modify (
	const INetCoreLDAPModifyRequestMessage& rMsg)
{
	// Initialize modify request message.
	INetCoreLDAPModifyRequestMessage aMsg (rMsg);

	// Start request.
	return startRequest (aMsg);
}

/*
 * ModifyRDN.
 */
sal_uInt16 INetCoreLDAPConnection::ModifyRDN (
	const OUString &rObjectDN, const OUString &rNewRDN)
{
	// Check arguments.
	if ((rObjectDN.getLength() == 0) || (rNewRDN.getLength() == 0))
		return 0;

	// Initialize modifyRDN request message.
	INetCoreLDAPModifyRDNRequestMessage aMsg;
	aMsg.SetObjectName (rObjectDN);
	aMsg.SetNewRDN (rNewRDN);

	// Start request.
	return startRequest (aMsg);
}

/*
 * List.
 * (SearchRequest emulating X.500 "List" Operation):
 */
sal_uInt16 INetCoreLDAPConnection::List (
	const OUString &rBaseObject, sal_Bool bAttrsOnly)
{
	// Check arguments.
	if (rBaseObject.getLength() == 0)
		return 0;

	// Initialize search request message.
	INetCoreLDAPSearchRequestMessage aMsg;

	aMsg.SetBaseObject (rBaseObject);
	aMsg.SetScope      (INETCORELDAP_SEARCH_SCOPE_SINGLELEVEL);
	aMsg.SetDerefAlias (INETCORELDAP_SEARCH_DEREF_FINDING_BASEOBJECT);

	aMsg.SetSizeLimit (0);
	aMsg.SetTimeLimit (0);
	aMsg.SetAttributesOnly (bAttrsOnly);

	OUString aType (OUString::createFromAscii ("objectClass"));
	aMsg.SetAttributeType (aType);

	INetCoreLDAPPresentFilter aFilter;
	aFilter.SetAttributeType (aType);
	aMsg.SetFilter (aFilter);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Read.
 * (SearchRequest emulating X.500 "Read" Operation):
 */
sal_uInt16 INetCoreLDAPConnection::Read (
	const OUString &rBaseObject, sal_Bool bAttrsOnly)
{
	// Check arguments.
	if (rBaseObject.getLength() == 0)
		return 0;

	// Initialize search request message.
	INetCoreLDAPSearchRequestMessage aMsg;

	aMsg.SetBaseObject (rBaseObject);
	aMsg.SetScope      (INETCORELDAP_SEARCH_SCOPE_BASEOBJECT);
	aMsg.SetDerefAlias (INETCORELDAP_SEARCH_DEREF_FINDING_BASEOBJECT);

	aMsg.SetSizeLimit (0);
	aMsg.SetTimeLimit (0);
	aMsg.SetAttributesOnly (bAttrsOnly);

	INetCoreLDAPPresentFilter aFilter;
	aFilter.SetAttributeType (OUString::createFromAscii ("objectClass"));
	aMsg.SetFilter (aFilter);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Search.
 */
sal_uInt16 INetCoreLDAPConnection::Search (
	const INetCoreLDAPSearchRequestMessage& rMsg)
{
	// Initialize search request message.
	INetCoreLDAPSearchRequestMessage aMsg (rMsg);

	// Start request.
	return startRequest (aMsg);
}

/*
 * Abandon.
 */
sal_uInt16 INetCoreLDAPConnection::Abandon (sal_uInt16 nMessageID)
{
	// Initialize abandon request message.
	INetCoreLDAPAbandonRequestMessage aMsg;
	aMsg.SetAbandonMessageID (nMessageID);

	// Start request.
	return startRequest (aMsg);
}

/*=======================================================================
 *
 * INetCoreLDAPConnectionContext implementation.
 *
 *=====================================================================*/
/*
 * INetCoreLDAPConnectionContext.
 */
INetCoreLDAPConnectionContext::INetCoreLDAPConnectionContext (
	sal_uInt32 nBufferSize)
	: m_pResolver (new INetCoreDNSResolver),
	  m_xSocket   (NULL),
	  m_nBufSiz   (nBufferSize),
	  m_pBuffer   (NULL),
	  m_pRead     (NULL),
	  m_pWrite    (NULL),
	  m_bIsOpen   (sal_False),
	  m_bAborting (sal_False),
	  m_nRequest  (0),
	  m_pfnCB     (NULL),
	  m_pDataCB   (NULL)
{
	m_pBuffer = (sal_uInt8*)(rtl_allocateMemory (m_nBufSiz));
	m_pRead = m_pWrite = m_pBuffer;
}

/*
 * ~INetCoreLDAPConnectionContext.
 */
INetCoreLDAPConnectionContext::~INetCoreLDAPConnectionContext (void)
{
	if (m_xSocket.isValid())
	{
		m_xSocket->close();
		m_xSocket.unbind();
	}
	delete m_pResolver;

	rtl_freeMemory (m_pBuffer);

	INetCoreLDAPRequestContext *pCtx;
	while (pop_front (pCtx))
		delete pCtx;
}

/*
 * create.
 */
void INetCoreLDAPConnectionContext::create (INetCoreDNSHostEntry &rDstAddr)
{
	// Initialize active socket.
	m_xSocket = new INetActiveTCPSocket();

	// Check for SocksGateway.
	NAMESPACE_VOS(ORef)<INetConfig> xConfig;
	if (INetConfig::getOrCreate (xConfig))
	{
		NAMESPACE_VOS(ORef)<INetProxyPolicy> xProxyPolicy (
			xConfig->getProxyPolicy());
		if (xProxyPolicy.isValid())
		{
			rtl::OUStringBuffer aBuffer (
				OUString::createFromAscii ("ldap://"));
			if (rDstAddr.GetCanonicalName())
				aBuffer.append (rDstAddr.GetCanonicalName());
			else
				aBuffer.append (rDstAddr.GetDomainName());
			aBuffer.append (sal_Unicode (':'));
			aBuffer.append (rDstAddr.GetPort());
			aBuffer.append (sal_Unicode ('/'));

			OUString aUrl (aBuffer.makeStringAndClear());
			INetProxyConfig aProxyConfig;

			if (xProxyPolicy->shouldUseProxy (aUrl, aProxyConfig))
			{
				if (aProxyConfig.hasSocksProxy())
				{
					// Use SocksGateway.
					m_xSocket->setSocksGateway (
						NAMESPACE_VOS(OInetSocketAddr)(
							aProxyConfig.getSocksProxyName(),
							aProxyConfig.getSocksProxyPort()));
				}
			}
		}
	}
}

/*=======================================================================
 *
 * INetCoreLDAPRequestContext implementation.
 *
 *=====================================================================*/
/*
 * INetCoreLDAPRequestContext.
 */
INetCoreLDAPRequestContext::INetCoreLDAPRequestContext (
	const INetCoreLDAPMessage &rMessage)
{
	m_nMessageID = rMessage.GetMessageID();

	m_nBufSiz = rMessage.GetEncodedLength();
	m_pBuffer = (sal_uInt8*)(rtl_allocateMemory (m_nBufSiz));
	m_pRead = m_pWrite = m_pBuffer;

	m_pRead += rMessage.GetEncodedData (m_pBuffer, m_nBufSiz);
}

/*
 * ~INetCoreLDAPRequestContext.
 */
INetCoreLDAPRequestContext::~INetCoreLDAPRequestContext (void)
{
	rtl_freeMemory (m_pBuffer);
}

