/*************************************************************************
 *
 *  $RCSfile: UnoRuntime.java,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: jbu $ $Date: 2002/03/21 16:25:03 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

package com.sun.star.uno;


import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import java.util.Enumeration;
import java.util.Hashtable;


import com.sun.star.lib.uno.typedesc.TypeDescription;

import com.sun.star.lib.util.WeakTable;


/**
 * The <code>UnoRuntime</code> class is the central class needed for
 * implementing or using UNO components in Java.
 * <p>
 * The Methods <code>queryInterface</code> and <code>isSame</code> delegate
 * calls to the implementing objects and are used instead of casts
 * and identity comparisons.
 * <p>
 * The other methods are used for managing bridges, which enable a user to
 * map objects out to or into java.
 * <p>
 * @version 	$Revision: 1.10 $ $ $Date: 2002/03/21 16:25:03 $
 * @see         com.sun.star.uno.IEnvironment
 * @see         com.sun.star.uno.IQueryInterface
 * @see         com.sun.star.uno.IBridge
 * @since       UDK1.0
 */
public class UnoRuntime {
	static public final boolean DEBUG = false;

	static protected WeakTable __environments = new WeakTable();
	static protected WeakTable __bridges      = new WeakTable();

	private static int unique;
    private static long time;
    private static short count;
    
	private static int hostUnique = (new Object()).hashCode();
    private static long lastTime = System.currentTimeMillis();
    private static short lastCount = Short.MIN_VALUE;
    private static Object mutex = new Object();
    private static long  ONE_SECOND = 1000; // in milliseconds
    
    private static Hashtable m_currentContexts = new Hashtable();
    
	// debug interface begin
	static int getEnvironmentCount() {
		return __environments.size();
	}
	static int getBridgeCount() {
		return __bridges.size();
	}
	// debug interface end
	
 	static String __uid = null;

	static {
		__uid =getUniqueKey();//(new String()).valueOf((new Object()).hashCode());
	}

    /** returns a world wide unique string. It is guaranteed, that every invocation
        of this method generates a new id, which is unique within the VM.
        The quality of 'world wide unique' will depend on the actual implementation,
        you should look at the source to determine if it meets your requirements.
        @return a unique String
     */
  	public static String getUniqueKey() {
  	
  		// generate unique key
  		synchronized (mutex) {
	    if (lastCount == Short.MAX_VALUE) {
		boolean done = false;
		while (!done) {
		   
  			  	time =new Long(System.currentTimeMillis()).longValue();
		  
		  if (time < lastTime+ONE_SECOND) {
			// pause for a second to wait for time to change
			try {
			    Thread.currentThread().sleep(ONE_SECOND);
			} catch (java.lang.InterruptedException e) {
			}	// ignore exception
			continue;
		    } else {
			lastTime = time;
			lastCount = Short.MIN_VALUE;
			done = true;
		    }
		}
	    } else {
		time = lastTime;
	    }
	    unique = hostUnique;
	    count = lastCount++;
	}
  	
  	return  Integer.toString(unique,16) + ":" +Long.toString(time,16) + ":" +Integer.toString(count,16);
  }


	/**
	 * Generates a worldwide unique object identifier (oid) for the given object. It is
     * guaranteed, that subsequent calls to the method with the same object
     * will give the same id.
	 * <p>
	 * @return     the generated oid.
	 * @param      object     the object for which a Oid should be generated.
	 */
	static public synchronized String generateOid(Object object) {
  		String oId =  null;

		object = WeakTable.__getObject(object);

		if(object instanceof IQueryInterface)
			oId = ((IQueryInterface)object).getOid();
		if ( oId == null ) 
  			oId = __uid + object.hashCode(); 

  		if(DEBUG) System.err.println("#### UnoRuntime.generateOid:" + oId);

		return oId;
	}
    
    /** Gets the current context of current thread or null if no context has been set for
        the current thread. The current context is threadlocal, meaning that this method
        returns the current context, that was last set during this thread.

        @return current context of current thread or null if no context was set.
    */
    static public com.sun.star.uno.XCurrentContext getCurrentContext()
    {
        return (com.sun.star.uno.XCurrentContext)m_currentContexts.get( Thread.currentThread() );
    }
    /** Sets the current context of current thread.
        You have to manually call setCurrentContext( null ) before terminating the
        current thread. The current context is threadlocal, meaning that upper callstack
        layers can retrieve the current context, which has been set before. Every function,
        that sets the current context, should reset the context to the original value
        when exiting (e.g. within a finally block).
        
        @param context context to be set (if null, a previously set context will be removed)
    */
    static public void setCurrentContext( com.sun.star.uno.XCurrentContext context )
    {
        if (context == null)
        {
            m_currentContexts.remove( Thread.currentThread() );
        }
        else
        {
            m_currentContexts.put( Thread.currentThread(), context );
        }
    }
    
	/**
	 * queries the given object for the given type. 
	 * <p>
     * It returns null in case the object doesn't support this interface,
     * othewise a reference to the desired interface.
     *
	 * If the given object already implements the desired interface,
	 * it is passed back immediatly. Otherwise it is checked if the given
	 * object implments the delegator interface <code>IQueryInterface</code> and
	 * the request is delegated to <code>queryInterface</code> of <code>IQueryInterface</code>.
	 * <p>
	 * @return     the desired interface if available, otherwise null.
	 * @param      type the class of the desired interface.
	 * @param      object     the object to ask for it.
	 * @see        com.sun.star.uno.IQueryInterface
	 */
	static public Object queryInterface(Type type, Object object) {
		object = WeakTable.__getObject(object);

		if (object instanceof IQueryInterface) {  // If the object supports the IQueryInterface -> delegate
			object = ((IQueryInterface) object).queryInterface(type);

			// Map all non interface anys to "null"
			if(object instanceof Any && ((Any)object).getType().getTypeDescription().getTypeClass() != TypeClass.INTERFACE)
				object = null;
		}

		// ensure, that the object implements the desired interface
		try {
			if (!(TypeDescription.getTypeDescription(type).getZClass().isInstance(object))) 
				object = null;
		}
		catch(ClassNotFoundException classNotFoundException) {
		}

		return object;
	}

	/**
	 * queries the given object for the given java class (which must represent a UNO interface). 
	 * <p>
     * It returns null in case the object doesn't support this interface,
     * othewise a reference to the desired interface.
     *
	 * If the given object already implements the desired interface,
	 * it is passed back immediatly. Otherwise it is checked if the given
	 * object implments the delegator interface <code>IQueryInterface</code> and
	 * the request is delegated to <code>queryInterface</code> of <code>IQueryInterface</code>.
	 * <p>
	 * @return     the desired interface if available, otherwise null.
	 * @param      zInterface the class of the desired interface.
	 * @param      object     the object to ask for it.
	 * @see        com.sun.star.uno.IQueryInterface
	 */
	static public Object queryInterface(Class zInterface, Object object) {
		return queryInterface(new Type(zInterface), object);
	}

	/** 
	 * Tests if the given objects are interfaces of the same underlying object.
	 * <p>
     * For UNO objects, it is not suffient to check object identity with a simple
     * ==, because there may be multiple proxies for the same object (in general
     * implementing different interfaces.
	 * @return     true, if the underlying object is the same, otherwise false.
	 * @param      object1 an object representing an interface.
	 * @param      object2 an object representing an interface.
	 */
	static public boolean areSame(Object object1, Object object2) {

		object1 = WeakTable.__getObject(object1);
		object2 = WeakTable.__getObject(object2);

		boolean identical = false;

		if(object1 == object2)
			identical = true;
		else if(object1 instanceof IQueryInterface)
			identical = ((IQueryInterface) object1).isSame(object2);
		else if(object2 instanceof IQueryInterface)
			identical = ((IQueryInterface) object2).isSame(object1);

		return identical;
	}

    /**
	 * retrieves an environment of type <code>name</code> with context <code>context</code>.
	 * Environments are hold weak at the UnoRuntime. If the desired environment already
	 * exists, this methods gives it back.
	 * If the desired environement does not exist, this method looks for it under
	 * <code>com.sun.star.lib.uno.environments.&lt;name&gt;.&lt;name&gt;_environment</code>.
	 * <p>
     * @param      name      the name of the environment.
     * @param      context   the context of the environment.
     * @see        com.sun.star.uno.IEnvironment
     */
	static public synchronized IEnvironment getEnvironment(String name, Object context) throws java.lang.Exception {
		IEnvironment xEnvironment = (IEnvironment)__environments.get(name + context, IEnvironment.class);

		if(DEBUG) System.err.println("##### " + UnoRuntime.class.getName() + ".getEnvironment:" + name + " " + context + " got it:" + xEnvironment);
		if(xEnvironment == null) {
			String className = "com.sun.star.lib.uno.environments." + name + "." + name + "_environment"; 

  			Class zClass = Class.forName(className);
			Constructor constructor = zClass.getConstructor(new Class[]{Object.class});
			xEnvironment = (IEnvironment)constructor.newInstance(new Object[]{context});
 
			xEnvironment = (IEnvironment)__environments.put(name + context, xEnvironment, IEnvironment.class);
		}

		return  xEnvironment;
	}

    /**
	 * Gets a bridge from environment <code>from</code> to environment <code>to</code>.
	 * Creates a new bridge, if the desired bridge does not exist yet and handles the arguments
	 * to the bridge.
	 * <p>
	 * If the desired bridge does not exist, is it searched in <code>com.sun.star.lib.uno.bridges.&lt;from&gt;_&lt;to&gt;.&lt;</code> 
	 * and the root classpath as <code>from&gt;_&lt;to&gt;_bridge</code>.
	 * <p>
	 * @result     the desired bridge.
     * @param      from      the source environment.
     * @param      to        the target environment.
	 * @param      args      the initial arguments for the bridge
     * @see        com.sun.star.uno.IEnvironment
     * @see        com.sun.star.uno.IBridge
     * @see        com.sun.star.uno.UnoRuntime#getBridgeByName
     */
	static public synchronized IBridge getBridge(IEnvironment from, IEnvironment to, Object args[]) throws java.lang.Exception {
		String name = from.getName() + "_" + to.getName();
		String hashName = from.getName() + from.getContext() + "_" + to.getName() + to.getContext();
		
		IBridge bridge = (IBridge)__bridges.get(hashName, IBridge.class);
		if(bridge == null) {
			Class zClass = null;
			
			String className =  name + "_bridge";
			
			try {
				if(DEBUG) System.err.println("#### UnoRuntime.getBridge - looking for:" + className);
				zClass = Class.forName(className);
			}
			catch(ClassNotFoundException classNotFoundException) {
				className = "com.sun.star.lib.uno.bridges." + name + "." + className;

				if(DEBUG) System.err.println("#### UnoRuntime.getBridge - looking for:" + className);
				zClass = Class.forName(className);
			}
			
			Class signature[] = {IEnvironment.class, IEnvironment.class, args.getClass()};
			Constructor constructor = zClass.getConstructor(signature);
			Object iargs[] = {from, to, args};
			
			bridge = (IBridge)constructor.newInstance(iargs);
			bridge = (IBridge)__bridges.put(hashName, bridge, IBridge.class);
		}
		else
			if(DEBUG) System.err.println("#### UnoRuntime.getBridge - found existing bridge:" + bridge);

		return bridge;
	}

    /**
	 * Gets a bridge from environment <code>from</code> to environment <code>to</code>.
	 * Creates a new bridge, if the desired bridge does not exist yet and handles the arguments
	 * to the bridge.
	 * <p>
	 * If the desired bridge does not exist, is it searched in <code>com.sun.star.lib.uno.bridges.&lt;from&gt;_&lt;to&gt;.&lt;</code> 
	 * and the root classpath as <code>from&gt;_&lt;to&gt;_bridge</code>.
	 * The needed environments are retrieved through <code>UnoRuntime#getEnvironment</code>.
	 * <p>
	 * @result     the desired bridge.
     * @param      from         the name of the source environment.
	 * @param      from_context the context for the source environment
     * @param      to           the name of the target environment.
	 * @param      to_context   the context for the source environment
	 * @param      args         the initial arguments for the bridge
     * @see        com.sun.star.uno.IEnvironment
     * @see        com.sun.star.uno.IBridge
     * @see        com.sun.star.uno.UnoRuntime#getBridge
     * @see        com.sun.star.uno.UnoRuntime#getEnvironment
     */
	static public IBridge getBridgeByName(String from, Object from_context, String to, Object to_context, Object args[]) throws java.lang.Exception {
		return getBridge(getEnvironment(from, from_context), getEnvironment(to, to_context), args);
	}

	/**
	 * Gives an <code>array</code> of all active bridges.
	 * <p>
	 * @result     an array of <code>IBridge</code> objects.
	 * @see        com.sun.star.uno.IBridge
	 */
	static public IBridge []getBridges() {
		IBridge bridges[] = new IBridge[__bridges.size()];

		Enumeration keys = __bridges.keys();
		for(int i = 0; i < bridges.length; ++ i) {
			bridges[i] = (IBridge)__bridges.get((String)keys.nextElement(), IBridge.class);
		}

		return bridges;
	}

    /**
	 * Gets the desired mapping from <code>from</code> to <code>to</code>.
	 * <p>
	 * Mappings are like bridges, except that with mappings one can only map in one direction.
	 * Mappings are here for <code>api</code> compatibility reasons the binary UNO runtime.
	 * Mappings are implemented as wrappers around bridges.
	 * <p>
	 * @result     the desired bridge.
     * @param      from         the source environment.
     * @param      to           the target environment.
     * @see        com.sun.star.uno.Mapping
     * @see        com.sun.star.uno.UnoRuntime#getMapping
     * @see        com.sun.star.uno.UnoRuntime#getEnvironment
     * @see        com.sun.star.uno.IEnvironment
     */
	static public IMapping getMapping(IEnvironment from, IEnvironment to) throws java.lang.Exception {
  		IBridge bridge = null;

  		try {
  			bridge = getBridge(from, to, null);
  		}
  		catch(ClassNotFoundException classNotFoundException) {
  			bridge = new BridgeTurner(getBridge(to, from, null));
  		}

  		return new MappingWrapper(bridge);
	}

    /**
	 * Gets the desired mapping from <code>from</code> to <code>to</code>.
	 * Retrieves the environments via <code>UnoRuntime#getEnvironment</code>.
	 * <p>
	 * @result     the desired bridge.
     * @param      from         the name of the source environment.
     * @param      to           the name of the target environment.
     * @see        com.sun.star.uno.Mapping
     * @see        com.sun.star.uno.UnoRuntime#getMapping
     * @see        com.sun.star.uno.UnoRuntime#getEnvironment
     */
	static public IMapping getMappingByName(String from, String to) throws java.lang.Exception {
		return getMapping(getEnvironment(from, null), getEnvironment(to, null));
	}

    /**
	 * Resets the UnoRuntime to its initial state.
	 * Releases all references to bridges and environments.
	 * <p>
     * @see        com.sun.star.uno.UnoRuntime#getBridge
     * @see        com.sun.star.uno.UnoRuntime#getBridgeByName
     * @see        com.sun.star.uno.UnoRuntime#getEnvironment
     * @see        com.sun.star.uno.UnoRuntime#getMapping
     * @see        com.sun.star.uno.UnoRuntime#getMappingByName
     */
	static public boolean reset() {
		if(DEBUG) System.err.println("UnoRuntime.dispose");

		if(!__bridges.isEmpty()) {
			System.err.println("\tdisposing bridges:");
			__bridges.reset();
		}

		System.err.println();

		if(!__environments.isEmpty()) {
			System.err.println("\tdisposing environments:");
			__environments.reset();
		}

		boolean passed = __environments.isEmpty() && __bridges.isEmpty();

		if(!passed) {
			System.err.println();
			
			System.err.println("\tenvironments destroyed - passed? " + __environments.isEmpty());
			System.err.println("\bridges destroyed - passed? " + __bridges.isEmpty());
		}

		return passed;
	}
}

