/**
 * Title:        pretty printing for xml files
 * Description:  Validates an xml document against a dtd<p>
 * Copyright:    null<p>
 * Company:      null<p>
 * @author       Dirk Grobler
 * @version      1.4
 */
package org.openoffice.helper;

import java.io.*;
import org.xml.sax.*;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

public class PrettyPrinter extends HandlerBase
{
    public static void main (String argv [])
    {
        if (argv.length != 2) {
            System.err.println ("Usage: cmd filename outfile");
            System.exit (1);
        }

        // Use the validating parser
        SAXParserFactory factory = SAXParserFactory.newInstance();
        factory.setValidating(false);
        try {

			// Set the output file
			out = new java.io.BufferedWriter(new OutputStreamWriter(new FileOutputStream(argv[1]), "UTF8"));

            // Parse the input
            SAXParser saxParser = factory.newSAXParser();
            saxParser.parse( new File(argv [0]), new PrettyPrinter() );

			if (out != null)
				out.close();

        } catch (SAXParseException spe) {
           // Error generated by the parser
           System.out.println ("\n** Parsing error"
              + ", line " + spe.getLineNumber ()
              + ", uri " + spe.getSystemId ());
           System.out.println("   " + spe.getMessage() );

           // Use the contained exception, if any
           Exception  x = spe;
           if (spe.getException() != null)
               x = spe.getException();
           x.printStackTrace();

        } catch (SAXException sxe) {
           // Error generated by this application
           // (or a parser-initialization error)
           Exception  x = sxe;
           if (sxe.getException() != null)
               x = sxe.getException();
           x.printStackTrace();

        } catch (ParserConfigurationException pce) {
            // Parser with specified options can't be built
            pce.printStackTrace();

        } catch (IOException ioe) {
           // I/O error
           ioe.printStackTrace();
        }

		System.exit (0);
    }

    static private java.io.BufferedWriter out = null;
    private String indentString = "\t"; // Amount to indent
    private boolean bHasContentOrSubElements = true;
    private int indentLevel = 0;

	private static int NONE = 0;
	private static int START_ELEMENT = 1;
	private static int CONTENT = 2;
	private static int END_ELEMENT = 3;
	private int nStatus = NONE;

    //===========================================================
    // replacing XML conform
    //===========================================================
    private String transform(String aSource)
    {
        int nLen = aSource.length();
        StringBuffer aBuffer = new StringBuffer(nLen);
        for (int i = 0; i < nLen; i++)
        {
            char c = aSource.charAt(i);
            switch (c)
            {
                case '<':
                    aBuffer.append("&lt;");
                    break;
                case '>':
                    aBuffer.append("&gt;");
                    break;
                case '&':
                    aBuffer.append("&amp;");
                    break;
                case '"':
                    aBuffer.append("&quot;");
                    break;
                case '\'':
                    aBuffer.append("&apos;");
                    break;
                default:
                    aBuffer.append(c);
            }
        }
        return aBuffer.toString();
    }

    //===========================================================
    // SAX DocumentHandler methods
    //===========================================================

    public void setDocumentLocator (Locator l)
    {
    }

    public void startDocument ()
    throws SAXException
    {
        emit ("<?xml version='1.0' encoding='UTF-8'?>");
        nl();
    }

    public void endDocument ()
    throws SAXException
    {
        nl();
        try {
            nl();
            out.flush ();
        } catch (IOException e) {
            throw new SAXException ("I/O error", e);
        }
    }

    public void startElement (String name, AttributeList attrs)
    throws SAXException
    {
		if (indentLevel != 0 && (nStatus == START_ELEMENT))
          emit (">");

        nStatus = START_ELEMENT;

		nl(); emit ("<"+ name);
        if (attrs != null) {
            for (int i = 0; i < attrs.getLength (); i++) {
                emit (" ");
                emit (attrs.getName (i));
                emit ("=");
                emit ("\"");
                emit (transform(attrs.getValue (i)));
                emit ("\"");
            }
        }
        indentLevel++;
    }

    public void endElement (String name)
    throws SAXException
    {
        indentLevel--;
        if (nStatus == START_ELEMENT)
            emit ("/>");
        else
        {
			// treat the value special
            if (nStatus != CONTENT)
//			if (name != "value" && name != "defaultvalue")
    			nl();
			emit ("</"+name+">");
        }
        nStatus = END_ELEMENT;
    }

    public void characters (char buf [], int offset, int len)
    throws SAXException
    {
        String s = new String(buf, offset, len);
        if (!s.trim().equals(""))
        {
			if (nStatus == START_ELEMENT)
				emit (">");
			emit(transform(s));
			nStatus = CONTENT;
        }

		// otherwise ignore it
    }

    public void ignorableWhitespace (char buf [], int offset, int len)
    throws SAXException
    {
		String s = new String(buf, offset, len);
		if (nStatus == START_ELEMENT)
            emit (">");
        emit(transform(s));
		nStatus = CONTENT;
    }

    public void processingInstruction (String target, String data)
    throws SAXException
    {
    }

    //===========================================================
    // SAX ErrorHandler methods
    //===========================================================

    // treat validation errors as fatal
    public void error (SAXParseException e)
    throws SAXParseException
    {
        throw e;
    }

    // dump warnings too
    public void warning (SAXParseException err)
    throws SAXParseException
    {
        System.out.println ("** Warning"
            + ", line " + err.getLineNumber ()
            + ", uri " + err.getSystemId ());
        System.out.println("   " + err.getMessage ());
    }

    //===========================================================
    // Helpers ...
    //===========================================================

    // Wrap I/O exceptions in SAX exceptions, to
    // suit handler signature requirements
    private void emit (String s)
    throws SAXException
    {
        try {
            out.write (s);
        } catch (IOException e) {
            throw new SAXException ("I/O error", e);
        }
    }

    // Start a new line
    // and indent the next line appropriately
    private void nl ()
    throws SAXException
    {
        String lineEnd =  System.getProperty("line.separator");
        try {
            out.write (lineEnd);
            for (int i=0; i < indentLevel; i++) out.write(indentString);
        } catch (IOException e) {
            throw new SAXException ("I/O error", e);
        }
    }
}
