/*************************************************************************
 *
 *  $RCSfile: pipe.c,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 15:17:20 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "system.h"

#include <osl/pipe.h>
#include <osl/diagnose.h>
#include <osl/thread.h>
#include <osl/mutex.h>
#include <osl/semaphor.h>

#define PIPENAMEMASK	"OSL_PIPE_%s"
#define SECPIPENAMEMASK	"OSL_PIPE_%s_%s"

typedef enum {
	MSG_SYN,
	MSG_FIN,
	MSG_DATA,
	MSG_UNKNOWN
} MessageType;

typedef struct {
    HPIPE  hPipe;
    APIRET nLastError;

    oslSecurity  Security;
} PipeImpl;

/* default size for input/output buffer */
static const ULONG ulBufSize = 4096;

/* OS/2 path for pipes */
static const CHAR  pszPipePath[] = "\\PIPE\\";
static const UCHAR nPipePathLen  = sizeof (pszPipePath) - 1;

/* global last error value to be returned from oslGetLastPipeError */
static APIRET ngLastError;

/*****************************************************************************/
/* osl_createPipe  */
/*****************************************************************************/
oslPipe SAL_CALL osl_createPipe(const sal_Char *pszPipeName, oslPipeOptions Options,
                       oslSecurity Security)
{
    PipeImpl *pPipe;
    
    ULONG  ulAction;
    CHAR   pszPipeNameBuffer [CCHMAXPATHCOMP];

    /* check parameters */
    OSL_ASSERT( pszPipeName );
    OSL_ASSERT( Security == 0 );

    /* allocate impl-structure */
    pPipe = (PipeImpl*) malloc (sizeof(PipeImpl));
    if (!pPipe)
    {
        OSL_TRACE( "osl_createPipe failed allocating memory.\n" );
        return NULL;
    }

    /* create pipe name */
    strcpy  (pszPipeNameBuffer, pszPipePath);
    sprintf (pszPipeNameBuffer + nPipePathLen, PIPENAMEMASK, pszPipeName);

    switch( Options )
    {
    case osl_Pipe_OPEN:
        ngLastError =
            DosOpen( pszPipeNameBuffer,
                     &(pPipe->hPipe),
                     &ulAction,
                     0,
                     FILE_NORMAL,
                     FILE_OPEN,
                     OPEN_ACCESS_READWRITE |
                     OPEN_SHARE_DENYREADWRITE,
                     (PEAOP2) NULL
                   );
        break;
    case osl_Pipe_CREATE:
        ngLastError =
            DosCreateNPipe( pszPipeNameBuffer,
                            &(pPipe->hPipe),
                            NP_ACCESS_DUPLEX,    /* open pipe for read and write access */
                            0xFF,                /* allow unlimited number of instances */
                            ulBufSize,           /* output buffer size */
                            ulBufSize,           /* input buffer size */
                            0L                   /* use default time-out time */
                          );
        break;
    default:
        ngLastError = ERROR_INVALID_PARAMETER;
    }

    /* if failed, release allocated memory */
    if (ngLastError)
    {
        OSL_TRACE( "osl_createPipe failed %s the pipe %s, Error Code %d.\n",
                   Options == osl_Pipe_OPEN ? "opening" : "creating",
                   pszPipeNameBuffer,
                   ngLastError );
        free (pPipe);
        return NULL;
    }

    pPipe->Security   = Security;
    pPipe->nLastError = NO_ERROR;
    return (oslPipe)pPipe;
}

/*****************************************************************************/
/* osl_copyPipe  */
/*****************************************************************************/
oslPipe SAL_CALL osl_copyPipe(oslPipe Pipe)
{
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    PipeImpl* pNewPipe;

    
    /* check parameter */
    OSL_ASSERT (pPipe);

    /* allocate impl-structure */
    pNewPipe = (PipeImpl*) malloc (sizeof(PipeImpl));
    if (!pNewPipe) return NULL;

    /* create new handle */
    pNewPipe->hPipe = (HPIPE) -1;
    ngLastError  = DosDupHandle( pPipe->hPipe, &(pNewPipe->hPipe) );

    /* if failed, release allocated memory */
    if (ngLastError)
    {
        OSL_TRACE( "osl_copyPipe failed duplicating pipe handle, Error-Code: %d.\n",
                   ngLastError );
        free (pNewPipe);
        return NULL;
    }

    pNewPipe->nLastError = NO_ERROR;
	return (oslPipe)pNewPipe;
}


/*****************************************************************************/
/* osl_destroyPipe  */
/*****************************************************************************/
void SAL_CALL osl_destroyPipe(oslPipe Pipe)
{
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    /* check parameter */
    OSL_ASSERT (pPipe);

    /* disconnect client */
    DosDisConnectNPipe (pPipe->hPipe);
    
    /* close the pipe */
    DosClose (pPipe->hPipe);

    /* free pipe structure */
    free (pPipe);
}

/*****************************************************************************/
/* osl_acceptPipe  */
/*****************************************************************************/
oslPipe SAL_CALL osl_acceptPipe(oslPipe Pipe)
{
    
#define PINFO ((PIPEINFO *) &PipeInfoBuffer)
    
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    PipeImpl* pNewPipe;
    BYTE     PipeInfoBuffer[sizeof(PIPEINFO) + CCHMAXPATHCOMP];

    /* check parameter */
    OSL_ASSERT (pPipe);

    /* get pipe information */
    pPipe->nLastError = DosQueryNPipeInfo(pPipe->hPipe,
                                         1,
                                         (PVOID) &PipeInfoBuffer,
                                         sizeof(PipeInfoBuffer));
    
    if (pPipe->nLastError)
    {
        OSL_TRACE( "osl_acceptPipe failed for requesting pipe information.\n",
                   pPipe->nLastError );
        return NULL;
    }

    /* create a new instance of the pipe if possible */
    if (PINFO->cbMaxInst == -1 ||                   /* unlimited instances */
        PINFO->cbMaxInst > PINFO->cbCurInst)
    {
        HPIPE hPipe;
        
        pNewPipe = (PipeImpl*) malloc (sizeof(PipeImpl));

        if (!pNewPipe)
        {
            OSL_TRACE( "osl_acceptPipe failed creating new instance.\n", ngLastError );
            free(pNewPipe);
            return NULL;
        }

        pNewPipe->Security = pPipe->Security;
        
        pNewPipe->nLastError =
            DosCreateNPipe( PINFO->szName,
                            &(pNewPipe->hPipe),
                            NP_ACCESS_DUPLEX,    /* open pipe for read and write access */
                            0xFF,                /* allow unlimited number of instances */
                            ulBufSize,           /* output buffer size */
                            ulBufSize,           /* input buffer size */
                            0L                   /* use default time-out time */
                          );
        
        if (pNewPipe->nLastError)
        {
            OSL_TRACE( "osl_acceptPipe failed creating new named pipe, Error-Code: %d.\n",
                       pNewPipe->nLastError );
            free(pNewPipe);
            return NULL;
        }
        
        /* switch pipe handles */
        hPipe = pPipe->hPipe;
        pPipe->hPipe  = pNewPipe->hPipe;
        pNewPipe->hPipe = hPipe;
        
        /* connect new handle to client */
        pNewPipe->nLastError = DosConnectNPipe( pNewPipe->hPipe );
    
        /* if failed, release allocated memory */
        if (pNewPipe->nLastError)
        {
            OSL_TRACE( "osl_acceptPipe failed connecting pipe to client, Error-Code: %d.\n",
                       pNewPipe->nLastError );
            
            osl_destroyPipe((oslPipe)pNewPipe);
            return NULL;
        }
        return (oslPipe)pNewPipe;
    }
    else
    {
        /* connect original handle to client */
        pPipe->nLastError = DosConnectNPipe( pPipe->hPipe );
    
        if (pPipe->nLastError)
        {
            OSL_TRACE( "osl_acceptPipe failed connecting pipe to client, Error-Code: %d.\n",
                       pPipe->nLastError );
            return NULL;
        }
    
        return (oslPipe)pPipe;
    }
}

/*****************************************************************************/
/* osl_receivePipe  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_receivePipe(oslPipe Pipe,
					    void* pBuffer,
					    sal_uInt32 BytesToRead)
{
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    ULONG  ulActual;
    
    /* check parameter */
    OSL_ASSERT (pPipe);

    /* read data from pipe */
    pPipe->nLastError = DosRead( pPipe->hPipe, pBuffer, BytesToRead, &ulActual );

    /* return -1 if failed */
    if( pPipe->nLastError )
    {
        OSL_TRACE( "osl_receivePipe failed receiving from Pipe, Error-Code: %d.\n",
                   pPipe->nLastError );
        return -1;
    }

    return ulActual;
}


/*****************************************************************************/
/* osl_sendPipe  */
/*****************************************************************************/
sal_Int32 SAL_CALL osl_sendPipe(oslPipe Pipe,
				       const void* pBuffer,
				       sal_uInt32 BytesToSend)
{
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    ULONG  ulActual;
    
    /* check parameter */
    OSL_ASSERT (pPipe);

    /* read data from pipe */
    pPipe->nLastError = DosWrite( pPipe->hPipe, (PVOID) pBuffer, BytesToSend, &ulActual );

    /* return -1 if failed */
    if( pPipe->nLastError )
    {
        OSL_TRACE( "osl_receivePipe failed writing to Pipe, Error-Code: %d.\n",
                   pPipe->nLastError );
        return -1;
    }

    return ulActual;
}


/*****************************************************************************/
/* osl_getLastPipeError  */
/*****************************************************************************/

oslPipeError SAL_CALL osl_getLastPipeError(oslPipe Pipe)
{
    PipeImpl* pPipe = (PipeImpl*) Pipe;
    APIRET rc;

    /* return local error value if possible */
    if (pPipe)
        rc = pPipe->nLastError;
    else
        rc = ngLastError;
    
    /* map OS/2 error values */
    switch (rc)
    {
    case NO_ERROR:                return osl_Pipe_E_None;
    case ERROR_PATH_NOT_FOUND:    return osl_Pipe_E_NotFound;
    case ERROR_NOT_ENOUGH_MEMORY: return osl_Pipe_E_NoBufferSpace;
    default:                      return osl_Pipe_E_invalidError;
    }
}

/*****************************************************************************/

