/*************************************************************************
 *
 *  $RCSfile: XclExpExternsheet.hxx,v $
 *
 *  $Revision: 1.4.2.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/11/01 07:55:00 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


#ifndef _SC_XCLEXPETERNSHEET_HXX
#define _SC_XCLEXPETERNSHEET_HXX

#ifndef _LIST_HXX
#include <tools/list.hxx>
#endif

#ifndef _SC_FILTERTOOLS_HXX
#include "FilterTools.hxx"
#endif
#ifndef _SC_XCLEXPHELPER_HXX
#include "XclExpHelper.hxx"
#endif

#ifndef _EXCRECDS_HXX
#include "excrecds.hxx"
#endif

#include <vector>

//___________________________________________________________________
// 3D references, external references
//  --> records TABID, SUPBOOK, XCT, CRN, EXTERNNAME, EXTERNSHEET

//___________________________________________________________________

/** Stores the correct Excel sheet index for each Calc sheet. The class knows all
    sheets which will not exported (external link sheets, scenario sheets, ...) */
class XclExpTabNumBuffer : private ScfUInt16List
{
private:
    sal_uInt32*                 pBuffer;        /// Array of sheet indexes.

    sal_uInt16                  nScCnt;         /// Count of Calc sheets.
    sal_uInt16                  nExcCnt;        /// Count of Excel sheets to be exported.
    sal_uInt16                  nExtCnt;        /// Count of external link sheets (in Calc).
    sal_uInt16                  nCodeCnt;       /// Count of codepages.

    ::std::vector< sal_uInt16 > maFromSortedVec;    /// Sorted index -> real index.
    ::std::vector< sal_uInt16 > maToSortedVec;      /// Real index -> sorted index.

    sal_Bool                    bEnableLog;     /// sal_True = log all sheet indexes (for formula compiler).

                                /// Searches for sheets not to be exported.
    void                        ApplyBuffer();

    /** Sorts the names of all tables and stores the indexes of the sorted indexes. */
    void                        InitSortedIndexes( ScDocument& rDoc );

public:
                                XclExpTabNumBuffer( ScDocument& rDoc );
    virtual                     ~XclExpTabNumBuffer();

                                /** Tests whether a Calc sheet is used to store external cell contents.
                                    @return  sal_True = sheet is external. */
    sal_Bool                    IsExternal( sal_uInt16 nScTab ) const;
                                /** Tests whether a Calc sheet has to be exported.
                                    @return  sal_True = sheet has to be exported. */
    sal_Bool                    IsExportTable( sal_uInt16 nScTab ) const;

                                /** Finds the Excel sheet index for a given Calc sheet.
                                    @return  The Excel sheet index of a Calc sheet. */
    sal_uInt16                  GetExcTable( sal_uInt16 nScTab ) const;

    /** Returns the Calc sheet index of the nSortedTab-th entry in the sorted sheet names list. */
    sal_uInt16                  GetRealSheetIndex( sal_uInt16 nSortedTab ) const;
    /** Returns the index of the passed Calc sheet in the sorted sheet names list. */
    sal_uInt16                  GetSortedSheetIndex( sal_uInt16 nScTab ) const;

                                /// @return  Count of Calc sheets.
    inline sal_uInt16           GetScTabCount() const       { return nScCnt; }
                                /// @return  Count of Excel sheets to be exported.
    inline sal_uInt16           GetExcTabCount() const      { return nExcCnt; }
                                /// @return  Count of external link sheets (in Calc).
    inline sal_uInt16           GetExternTabCount() const   { return nExtCnt; }
                                /// @return  Count of codepages.
    inline sal_uInt16           GetCodenameCount() const    { return nCodeCnt; }
                                /// @return  Maximum count of Calc sheets and codepages.
    inline sal_uInt16           GetMaxScTabCount() const    { return Max( nScCnt, nCodeCnt ); }

// *** for change tracking ***
                                /// Appends sheet index pair (called by formula compiler).
    void                        AppendTabRef( sal_uInt16 nExcFirst, sal_uInt16 nExcLast );
                                /// Enables logging of Excel sheet indexes in each 3D-reference.
    inline void                 StartRefLog()       { ScfUInt16List::Clear(); bEnableLog = sal_True; }
                                /// Disables logging of Excel sheet indexes.
    inline void                 EndRefLog()         { bEnableLog = sal_False; }
                                /// @return  The Excel sheet index log list.
    inline const ScfUInt16List& GetRefLog() const   { return *this; }
};


//___________________________________________________________________

/// Record CRN: Base class to store the contents of one external cell.
class XclExpCrn : public ExcRecord
{
private:
    sal_uInt16                  nCol;           /// Column index of the external cell.
    sal_uInt16                  nRow;           /// Row index of the external cell.
    sal_uInt8                   nId;            /// Identifier for data type (EXC_CACHEDVAL_***).

protected:
                                /// Writes the CRN header and calls SaveDiff() for writing cell contents.
    virtual void                SaveCont( XclExpStream& rStrm );
                                /// Derived classes write cell contents here.
    virtual void                SaveDiff( XclExpStream& rStrm ) = 0;
                                /// @return  Derived classes return size of cell contents here.
    virtual sal_uInt32          GetDiffLen() const = 0;

public:
    inline                      XclExpCrn( sal_uInt16 _nCol, sal_uInt16 _nRow, sal_uInt8 _nId ) :
                                    nCol( _nCol ), nRow( _nRow ), nId( _nId ) {}

                                /// @return  sal_True, if this CRN has the given cell address.
    inline sal_Bool             IsAddress( sal_uInt16 _nCol, sal_uInt16 _nRow )
                                    { return (_nCol == nCol) && (_nRow == nRow); }

    virtual sal_uInt16          GetNum() const;
    virtual sal_uInt32          GetLen() const;
};


//___________________________________________________________________

/// External value cell.
class XclExpCrnDouble : public XclExpCrn
{
private:
    double                      fVal;           /// Value of the value cell.

                                /// Write the cell contents.
    virtual void                SaveDiff( XclExpStream& rStrm );
                                /// @return  The size of cell contents.
    virtual sal_uInt32          GetDiffLen() const;

public:
                                XclExpCrnDouble( sal_uInt16 _nCol, sal_uInt16 _nRow, double _fVal );
};


//___________________________________________________________________

/// External text cell.
class XclExpCrnString : public XclExpCrn
{
private:
    XclExpUniString             aText;          /// String of the text cell.

                                /// Write the cell contents.
    virtual void                SaveDiff( XclExpStream& rStrm );
                                /// @return  The size of cell contents.
    virtual sal_uInt32          GetDiffLen() const;

public:
                                XclExpCrnString( sal_uInt16 _nCol, sal_uInt16 _nRow, const String& rTxt );
};


//___________________________________________________________________

/// External boolean cell.
class XclExpCrnBool : public XclExpCrn
{
private:
    sal_uInt16                  nBool;          /// Boolean value of the cell.

                                /// Write the cell contents.
    virtual void                SaveDiff( XclExpStream& rStrm );
                                /// @return  The size of cell contents.
    virtual sal_uInt32          GetDiffLen() const;

public:
                                XclExpCrnBool( sal_uInt16 _nCol, sal_uInt16 _nRow, sal_Bool bBoolVal );
};


//___________________________________________________________________

/// Record XCT: Header record for CRN record list
class XclExpXct : public ExcRecord
{
private:
    ScfObjList< XclExpCrn >     aCrnList;       /// CRN records that follow this record.
    XclExpUniString             aTable;         /// Sheet name of the external sheet.
    sal_uInt16                  nTabNum;        /// Excel sheet index.

                                /// @return  sal_True, if the CRN list still contains an CRN with the given address.
    sal_Bool                    Exists( sal_uInt16 nCol, sal_uInt16 nRow );

                                /// Writes the XCT record.
    virtual void                SaveCont( XclExpStream& rStrm );

public:
    inline                      XclExpXct( const String& rTab ) :
                                    aTable( rTab )  {}

                                /// @return  The size the sheet name will take in stream.
    inline sal_uInt16           GetTableBytes() const   { return static_cast< sal_uInt16 >( aTable.GetByteCount() ); }
                                /// @return  The external sheet name.
    inline const XclExpUniString& GetTableName() const  { return aTable; }

                                /// Sets the Excel sheet index.
    inline void                 SetTableNum( sal_uInt16 nTab )  { nTabNum = nTab; }
                                /// Stores all cells in the given range in the CRN list.
    void                        StoreCellRange( RootData& rRootData, const ScRange& rRange );

                                /// Writes the XCT and all CRN records.
    virtual void                Save( XclExpStream& rStrm );

    virtual sal_uInt16          GetNum() const;
    virtual sal_uInt32          GetLen() const;
};


//___________________________________________________________________

/// Record EXTERNNAME: External name (base class for AddIn- and DDE EXTERNNAMEs).
class XclExpExternname : public ExcRecord
{
private:
    String                      aName;          /// Name of the external name.
    sal_uInt16                  nFlags;         /// Flags for record export.

protected:
                                /// Writes the EXTERNNAME header and calls SaveDiff() for writing record contents.
    virtual void                SaveCont( XclExpStream& rStrm );
                                /// Derived classes write record contents here.
    virtual void                SaveDiff( XclExpStream& rStrm ) = 0;
                                /// @return  Derived classes return size of record contents here.
    virtual sal_uInt32          GetDiffLen() const = 0;

public:
                                XclExpExternname( const String& rName, sal_uInt16 _nFlags );
    virtual                     ~XclExpExternname();

                                /// @return  The name of the external name.
    inline const String&        GetName() const     { return aName; }

    virtual sal_uInt16          GetNum() const;
    virtual sal_uInt32          GetLen() const;
};


//___________________________________________________________________

/// External name for an AddIn function.
class XclExpExternnameAddIn : public XclExpExternname
{
private:
                                /// Writes record contents.
    virtual void                SaveDiff( XclExpStream& rStrm );
                                /// @return  Size of record contents.
    virtual sal_uInt32          GetDiffLen() const;

public:
    inline                      XclExpExternnameAddIn( const String& rName ) :
                                    XclExpExternname( rName, 0x0000 ) {}
};


//___________________________________________________________________

/// External name for a DDE link.
class XclExpExternnameDDE : public XclExpExternname
{
private:
    XclExpCachedValueList*      pOpList;        /// Cached results of the DDE link.

                                /// Writes record contents.
    virtual void                SaveDiff( XclExpStream& rStrm );
                                /// @return  Size of record contents.
    virtual sal_uInt32          GetDiffLen() const;

public:
    inline                      XclExpExternnameDDE( const String& rName, sal_uInt16 _nFlags ) :
                                    XclExpExternname( rName, _nFlags ), pOpList( NULL ) {}
    virtual                     ~XclExpExternnameDDE();

                                /// Inserts result list of a DDE link.
    sal_Bool                    InsertDDE(
                                    RootData& rRootData,
                                    const String& rApplic,
                                    const String& rTopic,
                                    const String& rItem );
};


//___________________________________________________________________

/// List of all external names of a sheet.
class XclExpExternnameList : private ScfObjList< XclExpExternname >
{
private:
                                /// @return  The list index of the external name.
    sal_uInt16                  GetExtname( const String& rName );

                                /** Appends the external name.
                                    @return  The 1-based list index of the appended neme. */
    inline sal_uInt16           Append( XclExpExternname* pName );

public:
                                /** Inserts an Addin name.
                                    @return  The 1-based list index of the name. */
    sal_uInt16                  InsertAddin( const String& rName );
                                /** Inserts a DDE link.
                                    @return  The 1-based list index of the DDE link. */
    sal_uInt16                  InsertDDE(
                                    RootData& rRootData,
                                    const String& rApplic,
                                    const String& rTopic,
                                    const String& rItem );

                                /// Writes all EXTERNNAME records.
    void                        Save( XclExpStream& rStrm );
};

inline sal_uInt16 XclExpExternnameList::Append( XclExpExternname* pName )
{
    ScfObjList< XclExpExternname >::Append( pName );
    return static_cast< sal_uInt16 >( Count() );
}


//___________________________________________________________________

/// Record SUPBOOK: Data for an external document (doc name, sheet names, external values).
class XclExpSupbook : public ExcRecord
{
private:
    enum XclExpSBType           /// Type of the SUPBOOK.
    {
        xlSBAddin,              /// SUPBOOK contains AddIn functions.
        xlSBSelf,               /// SUPBOOK is used for internal references.
        xlSBExt                 /// SUPBOOK is used for external references.
    };

    ScfObjList< XclExpXct >     aXctList;       /// List of XCT records (which contain CRN records).
    String                      aDocName;       /// URL of the external document.
    XclExpUniString             aEncoded;       /// Document name encoded for Excel.
    XclExpExternnameList*       pExtnameList;   /// List of EXTERNNAME records.
    XclExpSBType                eType;          /// Type of this SUPBOOK.
    sal_uInt32                  nLen;           /// Current length of the record.
    sal_uInt16                  nTables;        /// Count of sheets.

                                /// Creates and returns the list of EXTERNNAME records.
    XclExpExternnameList&       GetExternnameList();

                                /// Writes the SUPBOOK record.
    virtual void                SaveCont( XclExpStream& rStrm );

public:
                                /// Creates a SUPBOOK for AddIn functions.
                                XclExpSupbook();
                                /// Creates a SUPBOOK for internal references.
                                XclExpSupbook( sal_uInt16 nTabs );
                                /** Creates a SUPBOOK for external references or DDE links.
                                    @param  bDDE
                                    sal_False = external reference, sal_True = DDE link. */
                                XclExpSupbook( const String& rDocName, sal_Bool bDDE );
    virtual                     ~XclExpSupbook();

                                /// @return  The original document URL.
    inline const String&        GetName() const         { return aDocName; }
                                /// @return  Document URL encoded for Excel.
    inline const XclExpUniString& GetEncName() const    { return aEncoded; }
                                /// @return  Sheet name inside of this SUPBOOK.
    const XclExpUniString*      GetTableName( sal_uInt16 nIndex ) const;

                                /// @return  sal_True = this SUPBOOK is for internal references.
    inline sal_Bool             IsSelf() const          { return eType == xlSBSelf; }
                                /// @return  sal_True = this SUPBOOK is for AddIn functions.
    inline sal_Bool             IsAddin() const         { return eType == xlSBAddin; }
                                /// @return  sal_True = this SUPBOOK is for external refs or DDE links.
    inline sal_Bool             IsExt() const           { return eType == xlSBExt; }

                                /// Stores all cells in the given range in the CRN list of the XCT with index nXct.
    void                        StoreCellRange( RootData& rRootData, const ScRange& rRange, sal_uInt16 nXct );

                                /** Inserts a new XCT record with the given sheet name.
                                    @return  The sheet index. */
    sal_uInt16                  InsertTable( const String& rTabName );
                                /** Finds or inserts an EXTERNNAME record for AddIns.
                                    @return  The 1-based EXTERNNAME record index. */
    sal_uInt16                  InsertAddin( const String& rName );
                                /** Finds or inserts an EXTERNNAME record for DDE links.
                                    @return  The 1-based EXTERNNAME record index. */
    sal_uInt16                  InsertDDE(
                                    RootData& rRootData,
                                    const String& rApplic,
                                    const String& rTopic,
                                    const String& rItem );

                                /// Writes the SUPBOOK and all EXTERNNAME, XCT and CRN records.
    virtual void                Save( XclExpStream& rStrm );

    virtual sal_uInt16          GetNum() const;
    virtual sal_uInt32          GetLen() const;
};


//___________________________________________________________________

/// Contains a list of all SUPBOOK records and index arrays of external sheets.
class XclExpSupbookBuffer : public ExcEmptyRec, public ExcRoot
{
private:
    ScfObjList< XclExpSupbook > aSupbookList;       /// List of all SUPBOOK records.
    sal_uInt16*                 pSBIndexBuffer;     /// SUPBOOK index for each Excel sheet.
    sal_uInt16*                 pTabIndexBuffer;    /// Sheet indexes inside of SUPBOOK records for each Excel sheet.
    sal_uInt16                  nRefdCnt;           /// Array size for p***IndexBuffer.
    sal_uInt16                  nAddinSupb;         /// Index to Addin SUPBOOK.

                                /// @return  The SUPBOOK record of an Excel sheet.
    inline XclExpSupbook*       GetSupbook( sal_uInt16 nExcTab ) const;
                                /** Searches for the SUPBOOK record with document URL rDocName.
                                    @param  rnIndex
                                    Return: The list index, if the SUPBOOK exists.
                                    @return  The SUPBOOK record or NULL, if not found. */
    XclExpSupbook*              GetSupbook( sal_uInt16& rnIndex, const String& rDocName );

                                /** Appends a new SUPBOOK to the list.
                                    @return  The list index of the SUPBOOK record. */
    sal_uInt16                  Append( XclExpSupbook* pBook );
                                /// Appends an external SUPBOOK from the Calc sheet nScTab.
    void                        AddExt( sal_uInt16 nScTab );

public:
                                XclExpSupbookBuffer( RootData& rRootData );
    virtual                     ~XclExpSupbookBuffer();

                                /// @return  The external document URL of the Excel sheet nExcTab.
    const XclExpUniString*      GetDocumentName( sal_uInt16 nExcTab ) const;
                                /// @return  The external sheet name of the Excel sheet nExcTab.
    const XclExpUniString*      GetTableName( sal_uInt16 nExcTab ) const;

                                /** Finds SUPBOOK index and SUPBOOK sheet range from given Excel sheet range.
                                    @param  rnSupb
                                    Return: The index of the SUPBOOK record containing the sheet range.
                                    @param  rnXtiFirst
                                    Return: The index of the first XTI structure.
                                    @param  rnXtiLast
                                    Return: The index of the last XTI structure. */
    void                        GetXtiRange(
                                    sal_uInt16& rnSupb, sal_uInt16& rnXtiFirst, sal_uInt16& rnXtiLast,
                                    sal_uInt16 nTabFirst, sal_uInt16 nTabLast ) const;

                                /// Stores all cells in the given range in a CRN record list.
    void                        StoreCellRange( const ScRange& rRange );

                                /** Finds or inserts an EXTERNNAME record for AddIns.
                                    @param  rnSupbook
                                    The index of the SUPBOOK record which contains the AddIn function name.
                                    @param  rnExtname
                                    The 1-based EXTERNNAME record index. */
    void                        InsertAddin(
                                    sal_uInt16& rnSupbook, sal_uInt16& rnExtname,
                                    const String& rName );
                                /** Finds or inserts an EXTERNNAME record for DDE links.
                                    @param  rnSupbook
                                    Return: The index of the SUPBOOK record which contains the DDE link.
                                    @param  rnExtname
                                    Return: The 1-based EXTERNNAME record index. */
    void                        InsertDDE(
                                    sal_uInt16& rnSupbook, sal_uInt16& rnExtname,
                                    const String& rApplic,
                                    const String& rTopic,
                                    const String& rItem );

                                /// Writes all SUPBOOK records with their sub records.
    virtual void                Save( XclExpStream& rStrm );
};

inline XclExpSupbook* XclExpSupbookBuffer::GetSupbook( sal_uInt16 nExcTab ) const
{
    DBG_ASSERT( nExcTab < nRefdCnt, "XclSupbookList::GetSupbook - out of range" );
    return aSupbookList.GetObject( pSBIndexBuffer[ nExcTab ] );
}


//___________________________________________________________________

/** This struct contains a sheet index range for 3D references. This reference
    consists of an index to a SUPBOOK record and indexes to SUPBOOK sheet names. */
struct XclExpXti
{
    sal_uInt16                  nSupb;          /// Index to SUPBOOK record.
    sal_uInt16                  nFirst;         /// Index to the first sheet of the range.
    sal_uInt16                  nLast;          /// Index to the last sheet of the range.

    inline                      XclExpXti( sal_uInt16 nSupbook, sal_uInt16 nTabFirst, sal_uInt16 nTabLast ) :
                                    nSupb( nSupbook ), nFirst( nTabFirst ), nLast( nTabLast ) {}

                                /// @return  sal_True = this XTI contains the given values.
    inline sal_Bool             Equals(  sal_uInt16 nSupbook, sal_uInt16 nTabFirst, sal_uInt16 nTabLast ) const
                                    {return (nSupb == nSupbook) && (nFirst == nTabFirst) && (nLast == nTabLast); }

                                /// Writes the XTI structure of the EXTERNSHEET record.
    inline void                 Save( XclExpStream& rStrm ) const
                                    { rStrm << nSupb << nFirst << nLast; }
};


//___________________________________________________________________

/** Stores all EXTERNSHEET and SUPBOOK record data. This is the central class
    for export of all external references.
    File contents in BIFF8:
    - Record EXTERNSHEET: Contains indexes to URLs of external workbooks (SUPBOOKs)
    and sheet indexes for each external reference used anywhere in the workbook.
    This record follows a list of SUPBOOK records.
    - Record SUPBOOK: Contains the name of an external workbook and the names of its sheets.
    This record is followed by EXTERNNAME, XCT and CRN records.
    - Record XCT: Contains the sheet index of the following CRN records.
    - Record CRN: Contains addresses (row and column) and values of external referenced cells.
    - Record NAME: Contains defined names of the own workbook. This record follows the
    EXTERNSHEET record.
    - Record EXTERNNAME: Contains external defined names or DDE links or OLE object links.
*/
class XclExpExtsheetBuffer : public ExcRecord, public ExcRoot
{
private:
    ScfObjList< XclExpXti >     aXtiBuffer;         /// List of XTI structures.
    XclExpSupbookBuffer         aSupbookBuffer;     /// List of all SUPBOOK records.

                                /** Appends an XTI structure to the list.
                                    @return  The list index of the XTI structure. */
    sal_uInt16                  AppendXti( XclExpXti* pXti );
                                /** Searches for or inserts a new XTI structure.
                                    @return  The list index of the XTI structure. */
    sal_uInt16                  InsertXti( sal_uInt16 nSupb, sal_uInt16 nFirst, sal_uInt16 nLast );

                                /** Excel is able to store a maximum of 0xFFFF XTI structures.
                                    @return  The given value or the maximum of 0xFFFF. */
    static inline sal_uInt16    GetVal16( sal_uInt32 nVal )
                                    { return static_cast< sal_uInt16 >( Min( nVal, 0xFFFFUL ) ); }

                                /// Writes the EXTERNSHEET record with its XTI structures.
    virtual void                SaveCont( XclExpStream& rStrm );

public:
                                XclExpExtsheetBuffer( RootData& rRootData );

                                /** Searches for XTI structure with the given Excel sheet range.
                                    Adds new XTI if not found.
                                    @return  The list index of the XTI structure. */
    sal_uInt16                  Find( sal_uInt16 nTabFirst, sal_uInt16 nTabLast );
                                /** Searches for XTI structure with the given SUPBOOK index.
                                    Adds new XTI if not found.
                                    @return  The list index of the XTI structure. */
    sal_uInt16                  FindSpecial( sal_uInt16 nSupb );

                                /// @return  The external document URL of the Excel sheet nExcTab.
    inline const XclExpUniString* GetDocumentName( sal_uInt16 nExcTab )
                                    { return aSupbookBuffer.GetDocumentName( nExcTab ); }
                                /// @return  The external sheet name of the Excel sheet nExcTab.
    inline const XclExpUniString* GetTableName( sal_uInt16 nExcTab )
                                    { return aSupbookBuffer.GetTableName( nExcTab ); }

                                /// Stores the cell with the given address in a CRN record list.
    void                        StoreCellCont( const SingleRefData& rRef );
                                /// Stores all cells in the given range in a CRN record list.
    void                        StoreCellRange( const SingleRefData& rRef1, const SingleRefData& rRef2 );

                                /** Finds or inserts an EXTERNNAME record for AddIns.
                                    @param  rnXti
                                    Return: The index of the XTI structure which contains the AddIn function name.
                                    @param  rnExtname
                                    Return: The 1-based EXTERNNAME record index. */
    void                        InsertAddin(
                                    sal_uInt16& rnXti, sal_uInt16& rnExtname,
                                    const String& rName );
                                /** Finds or inserts an EXTERNNAME record for DDE links.
                                    @param  rnXti
                                    Return: The index of the XTI structure which contains the DDE link.
                                    @param  rnExtname
                                    Return: The 1-based EXTERNNAME record index. */
    sal_Bool                    InsertDDE(
                                    sal_uInt16& rnXti, sal_uInt16& rnExtname,
                                    const String& rApplic,
                                    const String& rTopic,
                                    const String& rItem );

                                /// Writes the EXTERNSHEET record and all SUPBOOK records with their sub records.
    virtual void                Save( XclExpStream& rStrm );

    virtual sal_uInt16          GetNum() const;
    virtual sal_uInt32          GetLen() const;
};


//___________________________________________________________________

#endif

