/*************************************************************************
 *
 *  $RCSfile: XclExpStream.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: dr $ $Date: 2001/10/26 16:46:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _SC_XCLEXPSTREAM_HXX
#define _SC_XCLEXPSTREAM_HXX

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif

//___________________________________________________________________
// output stream class for Excel export
// - CONTINUE record handling
// - ByteString and UniString support

//___________________________________________________________________

/** This class is used to export Excel record streams. An instance is constructed
    with an SvStream and the maximum size of Excel record contents (in BIFF5: 2080 bytes,
    in BIFF8: 8224 bytes).
    To start writing a record call StartRecord(). Parameters are the record identifier
    and any calculated record size. This is for optimizing the write process: if the real
    written data has the same size as the calculated, the stream has not to seek back and
    update the record size field. But it is not mandatory to calculate a size. Each
    record must be closed by calling EndRecord(). This will check (and update) the record
    size field.
    If some data exceeds the record size limit, a CONTINUE record is started automatically
    and the new data will be written to this record.
    If specific data pieces must not be splitted, use SetSliceLen(). For instance:
    To write a sequence of sal_uInt16 values, where 4 values form a unit and cannot be
    splitted, call SetSliceLen( 8 ) first (4*2 bytes == 8).
    To write unicode character arrays, call WriteUnicodeBuffer(). It creates CONTINUE
    records and repeats the unicode string flag byte automatically. This function is used
    for instance from the class XclExpUniString which can write complete unicode strings.
*/
class XclExpStream
{
private:
    SvStream&                   rStrm;          /// reference to the output stream.

                                // length data
    sal_uInt32                  nMaxRecLen;     /// maximum size of record content.
    sal_uInt32                  nMaxContLen;    /// maximum size of CONTINUE content.
    sal_uInt32                  nMaxLen;        /// current maximum, either nMaxRecLen or nMaxContLen.
    sal_uInt32                  nMaxSliceLen;   /// maximum size of data slices (parts that cannot be splitted).
    sal_uInt32                  nCalcLen;       /// calculated size received from calling function.
    sal_uInt32                  nHeaderLen;     /// record length written in last record header.
    sal_uInt32                  nCurrLen;       /// count of bytes already written in current record.
    sal_uInt32                  nSliceLen;      /// count of bytes already written in current slice.

                                // stream position data
    sal_uInt32                  nLastLenPos;    /// stream position of length in current header.

    sal_Bool                    bInRec;         /// is sal_True if currently writing inside of a record.

                                /// Write header data, internal setup.
    void                        InitRecord( sal_uInt16 nRecNum );
                                /// Rewrite correct record length, if different from calculated.
    void                        UpdateRecLen();
                                /// Recalculate nCurrLen and nSliceLeft.
    void                        UpdateLenVars( sal_uInt32 nLen );
                                /// Write CONTINUE header, internal setup.
    void                        StartContinue();
                                /// Refresh counter vars, create CONTINUE records.
    void                        PrepareWrite( sal_uInt32 nLen );
                                /// Create CONTINUE record at end, return maximum data block size remaining.
    sal_uInt32                  PrepareWrite();

                                /// Write a sequence of zero bytes.
    void                        WriteRawZeroBytes( sal_uInt32 nBytes );

public:
                                /** @param  rOutStrm
                                    the output stream to write to.
                                    @param  _nMaxRecLen
                                    the maximum allowed size of record content (depending on BIFF type). */
                                XclExpStream( SvStream& rOutStrm, sal_uInt32 _nMaxRecLen );

                                ~XclExpStream();

                                /// Start a new record: write header data, store calculated record length.
    void                        StartRecord( sal_uInt16 nRecNum, sal_uInt32 nRecLen );
                                /// Check and correct real record length. Must be called everytime a record is finished.
    void                        EndRecord();

                                /// @return  the position inside of current record (starts by 0 in every CONTINUE).
    inline sal_uInt32           GetRecPos() const                   { return nCurrLen; }

                                /// @return  the maximum size of a record.
    inline sal_uInt32           GetMaxRecLen() const                { return nMaxRecLen; }
                                /// Set maximum record size (only for this record).
    inline void                 SetMaxRecLen( sal_uInt32 nMax )     { nMaxLen = nMax; }
                                /// Set maximum size of CONTINUEs (only for this record).
    inline void                 SetMaxContLen( sal_uInt32 nMax )    { nMaxContLen = nMax; }

                                /// Set data slice length, 0 -> no slices.
    void                        SetSliceLen( sal_uInt32 nLen );

    inline XclExpStream&        operator<<( sal_Int8 nValue );
    inline XclExpStream&        operator<<( sal_uInt8 nValue );
    inline XclExpStream&        operator<<( sal_Int16 nValue );
    inline XclExpStream&        operator<<( sal_uInt16 nValue );
    inline XclExpStream&        operator<<( sal_Int32 nValue );
    inline XclExpStream&        operator<<( sal_uInt32 nValue );
    inline XclExpStream&        operator<<( float fValue );
    inline XclExpStream&        operator<<( double fValue );

                                /// Write nBytes bytes from memory.
    sal_uInt32                  Write( const void* pData, sal_uInt32 nBytes );
                                /// Write a sequence of nBytes zero bytes.
    void                        WriteZeroBytes( sal_uInt32 nBytes );
                                /** Copy nBytes bytes from current position of the stream rInStrm.
                                    Omitting the second parameter means read to end of stream. */
    sal_uInt32                  CopyFromStream( SvStream& rInStrm, sal_uInt32 nBytes = STREAM_SEEK_TO_END );

    // *** unicode string export is realized with helper class XclExpUniString ***
    // (slice length setting has no effect here -> disabled automatically)

                                /// Write unicode buffer as 8/16 bit, repeat nFlags at start of a CONTINUE record.
    void                        WriteUnicodeBuffer( const sal_uInt16* pBuffer, sal_uInt32 nChars, UINT8 nFlags );

    // *** write 8-bit-strings *** (slice length setting has no effect here -> disabled automatically)

                                /// Write ByteString buffer.
    void                        WriteByteStringBuffer(
                                    const ByteString& rString,
                                    sal_uInt16 nMaxLen = 0x00FF );
                                /// Write 8/16 bit length and ByteString buffer.
    void                        WriteByteString(
                                    const ByteString& rString,
                                    sal_uInt16 nMaxLen = 0x00FF,
                                    sal_Bool b16BitCount = FALSE );

    // SvStream functions
                                /// Set position of file stream (only allowed outside of records).
    sal_uInt32                  SetStreamPos( sal_uInt32 nPos );
                                /// @return  the position of the file stream.
    inline sal_uInt32           GetStreamPos() const { return rStrm.Tell(); }
};


inline XclExpStream& XclExpStream::operator<<( sal_Int8 nValue )
{
    PrepareWrite( sizeof( sal_Int8 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( sal_uInt8 nValue )
{
    PrepareWrite( sizeof( sal_uInt8 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( sal_Int16 nValue )
{
    PrepareWrite( sizeof( sal_Int16 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( sal_uInt16 nValue )
{
    PrepareWrite( sizeof( sal_uInt16 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( sal_Int32 nValue )
{
    PrepareWrite( sizeof( sal_Int32 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( sal_uInt32 nValue )
{
    PrepareWrite( sizeof( sal_uInt32 ) );
    rStrm << nValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( float fValue )
{
    PrepareWrite( sizeof( float ) );
    rStrm << fValue;
    return *this;
}

inline XclExpStream& XclExpStream::operator<<( double fValue )
{
    PrepareWrite( sizeof( double ) );
    rStrm << fValue;
    return *this;
}


#endif

