/*************************************************************************
 *
 *  $RCSfile: XclImpStream.hxx,v $
 *
 *  $Revision: 1.8 $
 *
 *  last change: $Author: dr $ $Date: 2001/09/24 13:30:26 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _SC_XCLIMPSTREAM_HXX
#define _SC_XCLIMPSTREAM_HXX

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif
#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif
#ifndef _STACK_HXX
#include <tools/stack.hxx>
#endif

//___________________________________________________________________
// input stream class for Excel import
// - CONTINUE record handling
// - ByteString and UniString support

//___________________________________________________________________
// this struct contains the relevant data for a stream position

struct XclImpStreamPos
{
	ULONG						nPos;
	ULONG						nNextPos;
	ULONG						nCurrLen;
	ULONG						nRecLeft;
	UINT16						nRecLen;

								XclImpStreamPos( ULONG _nPos, ULONG _nNextPos, ULONG _nCurrLen, ULONG _nRecLeft, UINT16 _nRecLen );
	void						Set( ULONG _nPos, ULONG _nNextPos, ULONG _nCurrLen, ULONG _nRecLeft, UINT16 _nRecLen );
};

//___________________________________________________________________
// stores stream positions

class XclImpStreamPosStack : private Stack
{
public:
	inline						XclImpStreamPosStack() : Stack() {}
	virtual						~XclImpStreamPosStack();

	void						Clear();

	inline void					Push( XclImpStreamPos* pNewPos )
									{ if( pNewPos ) Stack::Push( pNewPos ); }
	inline XclImpStreamPos*		Pop()
									{ return (XclImpStreamPos*) Stack::Pop(); }
};

//___________________________________________________________________

class XclImpStream
{
private:
	SvStream&					rStrm;			// input stream
    const CharSet*              pChSet;         // string import encoding

    XclImpStreamPos             aFirstRec;      // position of first record (if CONTINUEs exist)

	XclImpStreamPosStack		aPosStack;		// stack for user positions
	XclImpStreamPos				aUserPos;		// user defined position
	UINT16						nUserRecNum;	// record id for user defined pos
	BOOL						bUserValidRec;	// was valid record
	BOOL						bHasUserPos;	// user position set

	ULONG						nStreamLen;		// size of <rStrm>
	ULONG						nNextRecPos;	// start of next rec header
	ULONG						nCurrRecLen;	// helper for record position
	ULONG						nComplRecLen;	// length of complete record data (with CONTINUE)
	BOOL						bHasComplRec;	// TRUE -> <nComplRecLen> is valid

	UINT16						nRecNum;		// current rec id
	UINT16						nRecLen;		// size of current rec content
	ULONG						nRecLeft;		// current rec bytes left

	BOOL						bCont;			// CONTINUE handling
	BOOL						bValidRec;		// read state: FALSE -> no record available
	BOOL						bValid;			// read state: FALSE -> overread
	BOOL						bWarnings;		// enable/disable assertions

								// internal start of a new record, doesn't change
								// <nNextRecPos> and <bValid>
	BOOL						GetNextRecord( UINT16& rNewRecNum, UINT16& rNewRecLen );
								// internal setup of a new record, expecting <nRecNum> and <nNextRecPos>
	void						SetupRecord();
								// check for and go to following CONTINUE record,
								// doesn't change <bValid>, update <nCurrRecLen>
	BOOL						GetContinue();

								// check <nRecLeft> and jump into next CONTINUE
								// record if necessary and <bCont>==TRUE
	inline BOOL					CheckDataLeft( ULONG nBytes );

								// go to start of the next CONTINUE record,
								// set <bValid>, <nNextRecPos> and <nRecLeft>
	inline void					StartContinue();
								// go to start of the next CONTINUE record,
								// set <bValid>, <nNextRecPos> and <nRecLeft>,
								// read additional unicode flag byte and set <rb16Bit>
	void						StartStringContinue( BOOL& rb16Bit );

								// reads 8/16 bit string length
	inline UINT16				ReadByteStrLen( BOOL b16BitLen )
									{ return b16BitLen ? ReaduInt16() : ReaduInt8(); }

								// restore stream position contained in <rPos>
	void						RestorePosition( const XclImpStreamPos& rPos );

public:
								// reset <rInStrm> to position 0
                                XclImpStream( SvStream& rInStrm, const CharSet* pCharSet, BOOL bContHandling = TRUE );

								~XclImpStream();

								// set stream pointer to the start of the next record content
								// return FALSE if no record found (end of stream)
	BOOL						StartNextRecord();
								// set stream pointer to begin of record content,
								// set CONTINUE handling mode
	void						InitializeRecord( BOOL bContHandling );
								// FALSE - no overread assertions
	inline void					SetWarningMode( BOOL bWarnMode ) { bWarnings = bWarnMode; }

								// put current position on user position stack
								// stack is emptied at start of a new record
	void						PushPosition();
								// seek to last position from user position stack
	void						PopPosition();
								// remove last position from user position stack, don't seek
	void						RejectPosition();

								// store current position, stream never deletes it
	void						StoreUserPosition();
								// seek to user position
	void						SeekUserPosition();
								// delete user position
	inline void					DeleteUserPosition() { bHasUserPos = FALSE; }

								// return read state: FALSE -> record overread
	inline BOOL					IsValid() const { return bValid; }
								// return record ID
	inline UINT16				GetRecNum() const { return nRecNum; }
								// return position inside of record content
	inline ULONG				GetRecPos() const { return IsValid() ? nCurrRecLen - nRecLeft : 0; }
								// return data size without rec headers
								// inclusive CONTINUE recs if enabled
	ULONG						GetRecLen();
								// return remaining data size without rec headers
								// inclusive CONTINUE recs if enabled
	inline ULONG				GetRecLeft() { return IsValid() ? GetRecLen() - GetRecPos() : 0; }

	inline XclImpStream&		operator>>( char& rValue );
	inline XclImpStream&		operator>>( BYTE& rValue );
	inline XclImpStream&		operator>>( short& rValue );
	inline XclImpStream&		operator>>( USHORT& rValue );
	inline XclImpStream&		operator>>( int& rValue );
	inline XclImpStream&		operator>>( unsigned int& rValue );
	inline XclImpStream&		operator>>( long& rValue );
	inline XclImpStream&		operator>>( ULONG& rValue );
	inline XclImpStream&		operator>>( float& rValue );
	inline XclImpStream&		operator>>( double& rValue );

	inline INT8					ReadInt8();
	inline UINT8				ReaduInt8();
	inline INT16				ReadInt16();
	inline UINT16				ReaduInt16();
	inline INT32				ReadInt32();
	inline UINT32				ReaduInt32();
	inline float				ReadFloat();
	inline double				ReadDouble();

								// read <nBytes> bytes
	ULONG						Read( void* pData, ULONG nBytes );
								// copy <nBytes> bytes to <rOutStrm>
	ULONG						CopyToStream( SvStream& rOutStrm, ULONG nBytes );
								// copy entire record to <rOutStrm>, restore stream position
	ULONG						CopyRecordToStream( SvStream& rOutStrm );

								// seek absolute in record content
	void						Seek( ULONG nPos );
								// seek forward
	void						Ignore( ULONG nBytes );


    // *** UNICODE STRINGS ***
    // structure of an Excel unicode string:
    // (1) 2 byte character count
    // (2) 1 byte flags (16-bit-characters, rich string, far east string)
    // (3) [2 byte rich string format run count]
    // (4) [4 byte far east data size]
    // (5) character array
    // (6) [4 * (rich string format run count) byte]
    // (7) [(far east data size) byte]
    // header = (1), (2)
    // ext. header = (3), (4)
    // ext. data = (6), (7)

	// *** special string functions - use with care ***
                                // read ext. header, detect 8/16 bit mode, set all ext. info, return ext. data len
    ULONG                       ReadUniStringExtHeader( BOOL& rb16Bit, BOOL& rbRich, BOOL& rbFareast, UINT16& rCrun, UINT32& rExtInf, UINT8 nFlags );
                                // seek to begin of char array, detect 8/16 bit mode, return ext. data len
    ULONG                       ReadUniStringExtHeader( BOOL& rb16Bit, UINT8 nFlags );
                                // skip extended data after char array
    inline void                 SkipUniStringExtData( ULONG nExtLen )   { Ignore( nExtLen ); }

	// *** read/ignore unicode strings ***
	// - look for CONTINUE records even if CONTINUE handling disabled
	//   (only if inside of a CONTINUE record - for TXO import)
	// - no overread assertions (for Applix wrong string length export bug)

                                // read/ignore char array only
    void                        AppendRawUniString( String& rString, UINT16 nChars, BOOL b16Bit );
    String                      ReadRawUniString( UINT16 nChars, BOOL b16Bit );
    String*                     ReadNewRawUniString( UINT16 nChars, BOOL b16Bit );
	void						IgnoreRawUniString( UINT16 nChars, BOOL b16Bit );

                                // read/ignore ext. header, char array, ext. data
    void                        AppendUniString( String& rString, UINT16 nChars, UINT8 nFlags );
    String                      ReadUniString( UINT16 nChars, UINT8 nFlags );
    String*                     ReadNewUniString( UINT16 nChars, UINT8 nFlags );
	void						IgnoreUniString( UINT16 nChars, UINT8 nFlags );

                                // read/ignore 8 bit flags, ext. header, char array, ext. data
    inline void                 AppendUniString( String& rString, UINT16 nChars );
    String                      ReadUniString( UINT16 nChars );
    String*                     ReadNewUniString( UINT16 nChars );
	inline void					IgnoreUniString( UINT16 nChars );

                                // read/ignore 16 bit char count, 8 bit flags, ext. header, char array, ext. data
    inline void                 AppendUniString( String& rString );
    String                      ReadUniString();
    String*                     ReadNewUniString();
	inline void					IgnoreUniString();

	// *** read/ignore 8-bit-strings, store in ByteString or String ***

                                // read/ignore string data only
    inline void                 AppendRawByteString( String& rString, UINT16 nChars );
    String                      ReadRawByteString( UINT16 nChars );
	inline void					IgnoreRawByteString( UINT16 nChars );

                                // read/ignore 8/16 bit string length, string data
    inline void                 AppendByteString( String& rString, BOOL b16BitLen );
    String                      ReadByteString( BOOL b16BitLen );
	inline void					IgnoreByteString( BOOL b16BitLen );

	// SvStream functions
	inline ULONG				GetStreamPos() const { return rStrm.Tell(); }
	inline ULONG				GetStreamLen() const { return nStreamLen; }
};

inline BOOL XclImpStream::CheckDataLeft( ULONG nBytes )
{
	BOOL bNewValid = bValid && ((nRecLeft >= nBytes) ? TRUE : (nRecLeft ? FALSE : GetContinue()));
	DBG_ASSERT( !bWarnings || bNewValid || !bValid, "XclImpStream - record overread" );	// one assertion per record
	return (bValid = bNewValid);
}

inline void XclImpStream::StartContinue()
{
	BOOL bNewValid = (bValid && !nRecLeft) ? GetContinue() : FALSE;
	DBG_ASSERT( !bWarnings || bNewValid || !bValid, "XclImpStream - record overread" );	// one assertion per record
	bValid = bNewValid;
}

inline XclImpStream& XclImpStream::operator>>( char& rValue )
{
	if( CheckDataLeft( sizeof(char) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(char);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( BYTE& rValue )
{
	if( CheckDataLeft( sizeof(BYTE) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(BYTE);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( short& rValue )
{
	if( CheckDataLeft( sizeof(short) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(short);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( USHORT& rValue )
{
	if( CheckDataLeft( sizeof(USHORT) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(USHORT);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( int& rValue )
{
	if( CheckDataLeft( sizeof(int) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(int);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( unsigned int& rValue )
{
	if( CheckDataLeft( sizeof(unsigned int) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(unsigned int);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( long& rValue )
{
	if( CheckDataLeft( sizeof(long) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(long);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( ULONG& rValue )
{
	if( CheckDataLeft( sizeof(ULONG) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(ULONG);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( float& rValue )
{
	if( CheckDataLeft( sizeof(float) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(float);
	}
	return *this;
}

inline XclImpStream& XclImpStream::operator>>( double& rValue )
{
	if( CheckDataLeft( sizeof(double) ) )
	{
		rStrm >> rValue;
		nRecLeft -= sizeof(double);
	}
	return *this;
}

inline INT8 XclImpStream::ReadInt8()
{
	INT8 nValue = 0;
	if( CheckDataLeft( sizeof(INT8) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(INT8);
	}
	return nValue;
}

inline UINT8 XclImpStream::ReaduInt8()
{
	UINT8 nValue = 0;
	if( CheckDataLeft( sizeof(UINT8) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(UINT8);
	}
	return nValue;
}

inline INT16 XclImpStream::ReadInt16()
{
	INT16 nValue = 0;
	if( CheckDataLeft( sizeof(INT16) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(INT16);
	}
	return nValue;
}

inline UINT16 XclImpStream::ReaduInt16()
{
	UINT16 nValue = 0;
	if( CheckDataLeft( sizeof(UINT16) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(UINT16);
	}
	return nValue;
}

inline INT32 XclImpStream::ReadInt32()
{
	INT32 nValue = 0;
	if( CheckDataLeft( sizeof(INT32) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(INT32);
	}
	return nValue;
}

inline UINT32 XclImpStream::ReaduInt32()
{
	UINT32 nValue = 0;
	if( CheckDataLeft( sizeof(UINT32) ) )
	{
		rStrm >> nValue;
		nRecLeft -= sizeof(UINT32);
	}
	return nValue;
}

inline float XclImpStream::ReadFloat()
{
	float fValue = 0.0;
	if( CheckDataLeft( sizeof(float) ) )
	{
		rStrm >> fValue;
		nRecLeft -= sizeof(float);
	}
	return fValue;
}

inline double XclImpStream::ReadDouble()
{
	double fValue = 0.0;
	if( CheckDataLeft( sizeof(double) ) )
	{
		rStrm >> fValue;
		nRecLeft -= sizeof(double);
	}
	return fValue;
}



inline void XclImpStream::AppendUniString( String& rString, UINT16 nChars )
{
    AppendUniString( rString, nChars, ReaduInt8() );
}

inline void XclImpStream::IgnoreUniString( UINT16 nChars )
{
	IgnoreUniString( nChars, ReaduInt8() );
}

inline void XclImpStream::AppendUniString( String& rString )
{
    AppendUniString( rString, ReaduInt16() );
}

inline void XclImpStream::IgnoreUniString()
{
	IgnoreUniString( ReaduInt16() );
}



inline void XclImpStream::AppendRawByteString( String& rString, UINT16 nChars )
{
    AppendRawUniString( rString, nChars, FALSE );
}

inline void XclImpStream::IgnoreRawByteString( UINT16 nChars )
{
	Ignore( nChars );
}

inline void XclImpStream::AppendByteString( String& rString, BOOL b16BitLen )
{
    AppendRawByteString( rString, ReadByteStrLen( b16BitLen ) );
}

inline void XclImpStream::IgnoreByteString( BOOL b16BitLen )
{
	IgnoreRawByteString( ReadByteStrLen( b16BitLen ) );
}

#endif

