/*************************************************************************
 *
 *  $RCSfile: XclExpPivotTables.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: dr $ $Date: 2001/11/23 19:34:23 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifdef PCH
#include "filt_pch.hxx"
#endif

#pragma hdrstop

//_________________________________________________________

#ifndef _SC_XCLEXPPIVOTTABLES_HXX
#include "XclExpPivotTables.hxx"
#endif

//_________________________________________________________

#ifndef _TOOLS_SOLMATH_HXX
#include <tools/solmath.hxx>
#endif
#ifndef _ZFORMAT_HXX
#include <svtools/zformat.hxx>
#endif

#ifndef _COM_SUN_STAR_CONTAINER_XNAMEACCESS_HPP_
#include <com/sun/star/container/XNameAccess.hpp>
#endif
#ifndef _COM_SUN_STAR_CONTAINER_XNAMED_HPP_
#include <com/sun/star/container/XNamed.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif
#ifndef _COM_SUN_STAR_SHEET_XDIMENSIONSSUPPLIER_HPP_
#include <com/sun/star/sheet/XDimensionsSupplier.hpp>
#endif
#ifndef _COM_SUN_STAR_SHEET_XHIERARCHIESSUPPLIER_HPP_
#include <com/sun/star/sheet/XHierarchiesSupplier.hpp>
#endif
#ifndef _COM_SUN_STAR_SHEET_XLEVELSSUPPLIER_HPP_
#include <com/sun/star/sheet/XLevelsSupplier.hpp>
#endif
#ifndef _COM_SUN_STAR_SHEET_DATAPILOTFIELDORIENTATION_HPP_
#include <com/sun/star/sheet/DataPilotFieldOrientation.hpp>
#endif
#ifndef _COM_SUN_STAR_SHEET_GENERALFUNCTION_HPP_
#include <com/sun/star/sheet/GeneralFunction.hpp>
#endif

#include <vos/xception.hxx>

#ifndef SC_SCGLOB_HXX
#include "global.hxx"
#endif
#ifndef SC_DOCUMENT_HXX
#include "document.hxx"
#endif
#ifndef SC_DPOBJECT_HXX
#include "dpobject.hxx"
#endif
#ifndef SC_DPSHTTAB_HXX
#include "dpshttab.hxx"
#endif
#ifndef SC_MISCUNO_HXX
#include "miscuno.hxx"
#endif
#ifndef SC_UNONAMES_HXX
#include "unonames.hxx"
#endif

#ifndef _SC_FILTERTOOLS_HXX
#include "FilterTools.hxx"
#endif
#ifndef _SC_FILTERAPIHELPER_HXX
#include "FilterApiHelper.hxx"
#endif


//_________________________________________________________

// global strings
#include "globstr.hrc"

//_________________________________________________________

extern const sal_Char*	pPivotCacheStorageName;

//_________________________________________________________

using namespace ::rtl;
using namespace ::com::sun::star;

//============================================================================
// local functions

//_________________________________________________________
// lcl_GetLevelProps() - subtotals, showempty

void lcl_GetLevelProps( XclPivotCacheField& rField,
						const uno::Reference< beans::XPropertySet >& xDimProp )
{
	uno::Reference< sheet::XHierarchiesSupplier > xHiersSupp( xDimProp, uno::UNO_QUERY );
	if( xHiersSupp.is() )
	{
		uno::Reference< container::XIndexAccess > xHiersAcc = new ScNameToIndexAccess( xHiersSupp->getHierarchies() );
        sal_Int32 nHierarchy;
        if( !::getPropValue( nHierarchy, xDimProp, PROPNAME( SC_UNO_USEDHIER ) ) || (nHierarchy >= xHiersAcc->getCount()) )
			nHierarchy = 0;

		uno::Reference< uno::XInterface > xHierInt = ScUnoHelpFunctions::AnyToInterface( xHiersAcc->getByIndex( nHierarchy ) );
		uno::Reference< sheet::XLevelsSupplier > xLevelsSupp( xHierInt, uno::UNO_QUERY );
		if( xLevelsSupp.is() )
		{
			uno::Reference< container::XIndexAccess > xLevAcc = new ScNameToIndexAccess( xLevelsSupp->getLevels() );
			uno::Reference< uno::XInterface > xLevInt = ScUnoHelpFunctions::AnyToInterface( xLevAcc->getByIndex( 0 ) );
			uno::Reference< beans::XPropertySet > xLevProp( xLevInt, uno::UNO_QUERY );
			if( xLevProp.is() )
			{
				uno::Sequence< sheet::GeneralFunction > aSeq;
                if( ::getPropValue( aSeq, xLevProp, PROPNAME( SC_UNO_SUBTOTAL ) ) )
				{
					const sheet::GeneralFunction* pArray = aSeq.getConstArray();

					for( long iSubT = 0; iSubT < aSeq.getLength(); iSubT++ )
						rField.AddSubTotal( (UINT16) pArray[iSubT] );
				}

                BOOL bShowAll = ::getPropBool( xLevProp, PROPNAME( SC_UNO_SHOWEMPT ) );
				rField.SetShowAll( bShowAll );
			}
		}
	}
}

//_________________________________________________________
// lcl_SetDataFunc() - stores data field function in a cache field

void lcl_GetDataFunc( XclPivotCacheField& rField,
						const uno::Reference< beans::XPropertySet >& xDimProp )
{
    sheet::GeneralFunction eFunc;
    if( !::getPropValue( eFunc, xDimProp, PROPNAME( SC_UNO_FUNCTION ) ) || (eFunc < sheet::GeneralFunction_SUM) )
        eFunc = sheet::GeneralFunction_SUM;
    rField.AddDataFunc( static_cast< sal_uInt16 >( eFunc - sheet::GeneralFunction_SUM ) );
}

//_________________________________________________________
// lcl_GetFuncName - stores name of xcl-function in rStr

void lcl_GetFuncName( String& rStr, UINT16 nFunc )
{
	USHORT nIndex;
	switch( nFunc )
	{
		case EXC_PIVOT_SUBT_SUM:		nIndex = STR_FUN_TEXT_SUM;		break;
		case EXC_PIVOT_SUBT_COUNT:		nIndex = STR_FUN_TEXT_COUNT;	break;
		case EXC_PIVOT_SUBT_AVERAGE:	nIndex = STR_FUN_TEXT_AVG;		break;
		case EXC_PIVOT_SUBT_MAX:		nIndex = STR_FUN_TEXT_MAX;		break;
		case EXC_PIVOT_SUBT_MIN:		nIndex = STR_FUN_TEXT_MIN;		break;
		case EXC_PIVOT_SUBT_PROD:		nIndex = STR_FUN_TEXT_PRODUCT;	break;
		case EXC_PIVOT_SUBT_COUNTNUM:	nIndex = STR_FUN_TEXT_COUNT;	break;
		case EXC_PIVOT_SUBT_STDDEV:		nIndex = STR_FUN_TEXT_STDDEV;	break;
		case EXC_PIVOT_SUBT_STDDEVP:	nIndex = STR_FUN_TEXT_STDDEV;	break;
		case EXC_PIVOT_SUBT_VAR:		nIndex = STR_FUN_TEXT_VAR;		break;
		case EXC_PIVOT_SUBT_VARP:		nIndex = STR_FUN_TEXT_VAR;		break;
		default:
			rStr.Erase();
			return;
	}
	rStr = ScGlobal::GetRscString( nIndex );
}



//============================================================================
// pivot cache and additional pivot table data

//_________________________________________________________
// XclPivotCacheItem

XclPivotCacheItem::~XclPivotCacheItem()
{	}

BOOL XclPivotCacheItem::IsDouble() const
{
	return FALSE;
}

BOOL XclPivotCacheItem::IsString() const
{
	return FALSE;
}

BOOL XclPivotCacheItem::IsDate() const
{
	return FALSE;
}



//_________________________________________________________
// XclPivotCacheDblItem

XclPivotCacheDblItem::XclPivotCacheDblItem( double fNewVal ) :
		fVal( fNewVal )
{
	nTypeFlags = (fNewVal - floor( fNewVal ) != 0.0) ? 0x0520 : 0x0560;
}

BOOL XclPivotCacheDblItem::IsDouble() const
{
	return TRUE;
}

BOOL XclPivotCacheDblItem::operator==( XclPivotCacheItem& rItem )
{
	if( rItem.IsDouble() )
		return (((XclPivotCacheDblItem&) rItem).GetCont() == fVal);
	return FALSE;
}

void XclPivotCacheDblItem::Write( XclExpStream& rStrm )
{
	rStrm.StartRecord( 0x00C9, 8 );
	rStrm << GetCont();
	rStrm.EndRecord();
}



//_________________________________________________________
// XclPivotCacheStrItem

BOOL XclPivotCacheStrItem::IsString() const
{
	return TRUE;
}

BOOL XclPivotCacheStrItem::operator==( XclPivotCacheItem& rItem )
{
	if( rItem.IsString() )
		return (((XclPivotCacheStrItem&) rItem).GetCont() == sText);
	return FALSE;
}

void XclPivotCacheStrItem::Write( XclExpStream& rStrm )
{
	XclExpUniString	sTemp( sText );
	rStrm.StartRecord( 0x00CD, sTemp.GetByteCount() );
	sTemp.Write( rStrm );				// item content
	rStrm.EndRecord();
}



//_________________________________________________________
// XclPivotCacheDatItem

XclPivotCacheDatItem::XclPivotCacheDatItem( double fNewDate, RootData* pExcRoot )
{
	nTypeFlags = 0x0900;

	double	fDays	= SolarMath::ApproxFloor( fNewDate );
	double	fTime	= fNewDate - fDays;

	if( fDays > 0.0 )
	{
		Date*	pNullDate	= pExcRoot->pFormTable->GetNullDate();
		Date	aDate		= *pNullDate;

		aDate += ( long ) fDays;
		nYear = (UINT16) aDate.GetYear();
		nMonth = (UINT16) aDate.GetMonth();
		nDay = (UINT8) aDate.GetDay();
	}
	else
		nYear = nMonth = nDay = 0;

	if( fTime > 0.0 )
	{
		fTime *= 24;
		nHour = (UINT8) fTime;
		fTime -= nHour;
		fTime *= 60;
		nMin = (UINT8) fTime;
		fTime -= nMin;
		fTime *= 60;
		nSec = (UINT8)(fTime + 0.001);
		if( nSec > 59 )
			nSec = 59;
	}
	else
		nHour = nMin = nSec = 0;

	fVal = nSec;
	fVal = (((fVal / 60) + nMin) / 60 + nHour) / 24 + fDays;
}


BOOL XclPivotCacheDatItem::IsDate() const
{
	return TRUE;
}

BOOL XclPivotCacheDatItem::operator==( XclPivotCacheItem& rItem )
{
	if( rItem.IsDate() )
		return (((XclPivotCacheDatItem&) rItem).GetCont() == fVal);
	return FALSE;
}

void XclPivotCacheDatItem::Write( XclExpStream& rStrm )
{
	rStrm.StartRecord( 0x00CE, 8 );
	rStrm << nYear << nMonth << nDay << nHour << nMin << nSec;
	rStrm.EndRecord();
}



//_________________________________________________________
// XclPivotCacheField

XclPivotCacheField::XclPivotCacheField( RootData* pRD, ScRange& rRange ) :
		ExcRoot( pRD ),
		nFlags( EXC_SXFIELD_COMMON ),
		pCurr( NULL ),
		nIndex( 0 ),
		bShowAllItems( FALSE )
{
	// pivot cache data
	ScDocument&			rDoc	= *pExcRoot->pDoc;
	SvNumberFormatter&	rFTab	= *pExcRoot->pFormTable;
	ScAddress			aAddr	= rRange.aStart;
	XclPivotCacheItem*	pItem	= NULL;

	rDoc.GetString( aAddr.Col(), aAddr.Row(), aAddr.Tab(), sName );

	for( USHORT nRow = rRange.aStart.Row() + 1; nRow <= rRange.aEnd.Row(); nRow++ )
	{
		aAddr.SetRow( nRow );

		if( rDoc.HasValueData( aAddr.Col(), aAddr.Row(), aAddr.Tab() ) )
		{
			double					fValue		= rDoc.GetValue( aAddr );
			ULONG					nNumForm	= rDoc.GetNumberFormat( aAddr );
			const SvNumberformat*	pEntry		= rFTab.GetEntry( nNumForm );

			BOOL bIsDate = pEntry ? (pEntry->GetType() & NUMBERFORMAT_DATETIME) : FALSE;

			if( bIsDate )
				pItem = new XclPivotCacheDatItem( fValue, pExcRoot );
			else
				pItem = new XclPivotCacheDblItem( fValue );
			Insert( pItem );
		}
		else
		{
			String sStr;
			rDoc.GetString( aAddr.Col(), aAddr.Row(), aAddr.Tab(), sStr );
			pItem = new XclPivotCacheStrItem( sStr );
			Insert( pItem );
		}
	}

	// pivot table data
	nAxis = nSubTotalMask = nPosition = 0;
}


XclPivotCacheField::~XclPivotCacheField()
{
	for( XclPivotCacheItem* pItem = _First(); pItem; pItem = _Next() )
		delete pItem;
}


UINT16 XclPivotCacheField::_Find( XclPivotCacheItem& rItem )
{
	if( pCurr )
		if( *pCurr == rItem )
			return nIndex;

	BOOL bFound = FALSE;

	pCurr	= _First();
	nIndex	= 0;

	while( !bFound && pCurr )
	{
		bFound = (*pCurr == rItem);
		if( !bFound )
		{
			pCurr = _Next();
			nIndex++;
		}
	}

	if( !bFound )
		pCurr = &rItem;
	return nIndex;
}


void XclPivotCacheField::Insert( XclPivotCacheItem*& rpItem )
{
	if( rpItem )
	{
		UINT16 nPos = _Find( *rpItem );

		if( nPos == List::Count() )			// not found
		{
			UINT16 nType = rpItem->GetTypeFlags();

			switch( nType )
			{
				case 0x0480:				// string
					nFlags |= nType;
					break;
				case 0x0520:				// double
				case 0x0560:				// double (int only)
					if( nFlags & 0x0800 )
						nFlags |= 0x0400;
					else
					{
						if( (nFlags & 0x0560) == 0x0520 )
							nType &= ~0x0040;
						nFlags &= ~0x0560;
						nFlags |= nType;
					}
					break;
				case 0x0900:				// date/time
					nFlags &= ~0x0060;
					nFlags |= nType;
					break;
			}
			List::Insert( rpItem, LIST_APPEND );

			if( List::Count() == 0x0100 )	// 16bit column in index list
				nFlags |= EXC_SXFIELD_16BIT;
		}
		else								// already in list, forget it
		{
			delete rpItem;
			rpItem = NULL;
		}

		aIndexList.Append( nPos );
	}
}


void XclPivotCacheField::AddSubTotal( UINT16 nSubT )
{
	aSubTotalList.Append( nSubT );
	if( nSubT )
		nSubTotalMask |= (0x0001 << (nSubT - 1));
}


void XclPivotCacheField::SetPosition( UINT16 nPos, UINT16 nAxisFlag )
{
	if( nAxisFlag == EXC_PIVOT_AXIS_DATA )
		aDataPosList.Append( nPos );
	else
		nPosition = nPos;
}


BOOL XclPivotCacheField::IsInDataPosList( UINT16 nPos ) const
{
	BOOL bFound = FALSE;
	for( UINT16 nInd = 0; (nInd < aDataPosList.Count()) && !bFound; nInd++ )
        bFound = (aDataPosList.GetValue( nInd ) == nPos);
	return bFound;
}


UINT16 XclPivotCacheField::GetFunction( UINT16 nPos ) const
{
	BOOL	bFound	= FALSE;
	UINT16	nInd	= 0;

	while( (nInd < aDataPosList.Count()) && !bFound )
	{
        bFound = (aDataPosList.GetValue( nInd ) == nPos);
		if( !bFound )
			nInd++;
	}

	if( bFound )
        return aDataFuncList.GetValue( nInd );
	return 0;
}


void XclPivotCacheField::WriteIndex( XclExpStream& rStrm, UINT16 nRow )
{
	if( GetIndexWidth() == 1 )
        rStrm << (UINT8) aIndexList.GetValue( nRow );
	else
        rStrm << aIndexList.GetValue( nRow );
}


void XclPivotCacheField::WriteFieldHeader( XclExpStream& rStrm )
{
	XclExpUniString	sTemp( sName );

	// field record
	rStrm.StartRecord( 0x00C7, 14 + sTemp.GetByteCount() );
	rStrm	<< nFlags							// type
			<< (UINT16) 0x0000					// undoc'ed
			<< (UINT16) 0x0000					// undoc'ed
			<< (UINT16) List::Count()			// num of items
			<< (UINT16) 0x0000					// undoc'ed
			<< (UINT16) 0x0000					// undoc'ed
			<< (UINT16) List::Count();			// num of items
	sTemp.Write( rStrm );						// field name
	rStrm.EndRecord();

	// SQL data type
	rStrm.StartRecord( 0x01BB, 2 );
	rStrm	<< (UINT16) 0x0000;					// SQL data type
	rStrm.EndRecord();
}


void XclPivotCacheField::Write( XclExpStream& rStrm )
{
	WriteFieldHeader( rStrm );

	for( XclPivotCacheItem* pItem = _First(); pItem; pItem = _Next() )
		pItem->Write( rStrm );
}



//_________________________________________________________
// XclPivotCache

XclPivotCache::XclPivotCache( RootData* pRD, ScDPObject& rObject, UINT16 nInd ) :
		ExcRoot( pRD ),
		nIndex( nInd ),
		aOutRange( rObject.GetOutRange() ),
		sTableName( rObject.GetName() ),
		sDataName( ScGlobal::GetRscString( STR_PIVOT_DATA ) ),
		nDataFAxis( EXC_PIVOT_AXIS_NONE ),
		bMultiData( FALSE )
{
	// remaining inits
	nColF = nRowF = nPageF = nDataF = nDataFPos = 0;

	// sheet source data only
	if( !rObject.GetSheetDesc() )
		return;

	// pivot cache data
	aSourceRange = rObject.GetSheetDesc()->aSourceRange;
	pExcRoot->pDoc->GetName( aSourceRange.aStart.Tab(), sSourceName );
	nSrcRecs = aSourceRange.aEnd.Row() - aSourceRange.aStart.Row();

	ScRange aColRange( aSourceRange );

	for( USHORT nCol = aSourceRange.aStart.Col(); nCol <= aSourceRange.aEnd.Col(); nCol++ )
	{
		aColRange.aStart.SetCol( nCol );
		aColRange.aEnd.SetCol( nCol );
		Append( new XclPivotCacheField( pRD, aColRange ) );
	}

	// pivot table data
//    aOutRange.aStart.IncRow( 2 );       //! skip filter button; works with sheet source only!
	bRowGrand = bColGrand = TRUE;

	uno::Reference< sheet::XDimensionsSupplier > xSource = rObject.GetSource();
	if( !xSource.is() )
		return;

	// row & col grands
	uno::Reference< beans::XPropertySet >	xSrcProp( xSource, uno::UNO_QUERY );
	if( xSrcProp.is() )
	{
        bRowGrand = bColGrand = sal_True;
        ::getPropBool( bRowGrand , xSrcProp, PROPNAME( SC_UNO_ROWGRAND ) );
        ::getPropBool( bColGrand , xSrcProp, PROPNAME( SC_UNO_COLGRAND ) );
	}

	uno::Reference< container::XIndexAccess >	xDimsIndex =
		new ScNameToIndexAccess( xSource->getDimensions() );
	if( !xDimsIndex.is() )
		return;

	// pivot field properties - index order to get DataLayoutDimension
	long nDimCount = xDimsIndex->getCount();
	for( long nDim = 0; nDim < nDimCount; nDim++ )
	{
		uno::Reference< uno::XInterface >		xIntDim =
			ScUnoHelpFunctions::AnyToInterface( xDimsIndex->getByIndex( nDim ) );

		uno::Reference< container::XNamed >		xDimName( xIntDim, uno::UNO_QUERY );
		uno::Reference< beans::XPropertySet >	xDimProp( xIntDim, uno::UNO_QUERY );

		if( xDimName.is() && xDimProp.is() )
		{
			// store orientation & position info
            UINT16 nAxisFlag = EXC_PIVOT_AXIS_NONE;
            sal_Bool bIsDataLayout = ::getPropBool( xDimProp, PROPNAME( SC_UNO_ISDATALA ) );
            sal_Int32 nPosition = 0;
            sheet::DataPilotFieldOrientation eOrientation = sheet::DataPilotFieldOrientation_HIDDEN;

            ::getPropValue( nPosition, xDimProp, PROPNAME( SC_UNO_POSITION ) );
            ::getPropValue( eOrientation, xDimProp, PROPNAME( SC_UNO_ORIENTAT ) );

			switch( eOrientation )
			{
				case sheet::DataPilotFieldOrientation_COLUMN:
					if( bIsDataLayout )
						nDataFAxis = EXC_PIVOT_AXIS_COL;
					nColF++;
					nAxisFlag = EXC_PIVOT_AXIS_COL;
					break;
				case sheet::DataPilotFieldOrientation_ROW:
					if( bIsDataLayout )
						nDataFAxis = EXC_PIVOT_AXIS_ROW;
					nRowF++;
					nAxisFlag = EXC_PIVOT_AXIS_ROW;
					break;
				case sheet::DataPilotFieldOrientation_PAGE:
//                    nPageF++;
//                    nAxisFlag = EXC_PIVOT_AXIS_PAGE;
					break;
				case sheet::DataPilotFieldOrientation_DATA:
					nDataF++;
					nAxisFlag = EXC_PIVOT_AXIS_DATA;
					break;

			}

			if( bIsDataLayout )
				nDataFPos = ( UINT16 ) nPosition;	// Daniel said, that the cast is ok
			else
			{
				// find correct cache entry
				String sName = xDimName->getName();

				XclPivotCacheField* pField = _GetByName( sName );
				if( !pField )
				{
                    uno::Reference< uno::XInterface > xIntOrig;
                    ::getPropValue( xIntOrig, xDimProp, PROPNAME( SC_UNO_ORIGINAL ) );
					uno::Reference< container::XNamed >	xNameOrig( xIntOrig, uno::UNO_QUERY );
					if ( xNameOrig.is() )
					{
                        sName   = xNameOrig->getName();
						pField	= _GetByName( sName );
					}
				}

				// set field and item properties
				if( pField )
				{
					pField->AddAxis( nAxisFlag );
					if( nAxisFlag == EXC_PIVOT_AXIS_DATA )
						lcl_GetDataFunc( *pField, xDimProp );
					else
						lcl_GetLevelProps( *pField, xDimProp );
					pField->SetPosition( ( UINT16 ) nPosition, nAxisFlag );
				}
			}
		}
	}
	bMultiData = (nDataF > 1);

	// find order of fields
	BuildPosList( aRowPosList, nRowF, EXC_PIVOT_AXIS_ROW );
	BuildPosList( aColPosList, nColF, EXC_PIVOT_AXIS_COL );
	BuildPosList( aDataPosList, nDataF, EXC_PIVOT_AXIS_DATA );
}


XclPivotCache::~XclPivotCache()
{
	for( XclPivotCacheField* pField = _First(); pField; pField = _Next() )
		delete pField;
}


XclPivotCacheField* XclPivotCache::_GetByName( const String& rName )
{
	XclPivotCacheField* pField = _First();
	BOOL				bFound = FALSE;

	while( pField && !bFound )
	{
		bFound = (pField->GetName() == rName);
		if( !bFound )
			pField = _Next();
	}

	return pField;
}


ULONG XclPivotCache::GetIndexRecLen()
{
	ULONG nRecLen = 0;

	for( XclPivotCacheField* pField = _First(); pField; pField = _Next() )
		nRecLen += pField->GetIndexWidth();

	return nRecLen;
}


void XclPivotCache::WriteCacheHeader( XclExpStream& rStrm )
{
	// SXDB
	rStrm.StartRecord( 0x00C6, 21 );
	rStrm	<< (UINT32) nSrcRecs				// number of source records
			<< nIndex							// SXIDSTM index
			<< (UINT16) 0x0021					// flags
			<< (UINT16) 0x1FFF					// num of DB recs (from xcl)
			<< (UINT16) List::Count()			// base fields
			<< (UINT16) List::Count()			// base, grouped, calc'ed fields
			<< (UINT16) 0x0000					// unused
			<< (UINT16) 0x0001					// source: Xcl worksheet
			<< (UINT16) 0x0000					// unicode user name
			<< (UINT8) 0x00;
	rStrm.EndRecord();

	// SXDBEX
	rStrm.StartRecord( 0x0122, 12 );
	rStrm	<< (double) 51901.029652778			// last change
			<< (UINT32) 0x00000000;				// num of SXFORMULA
	rStrm.EndRecord();
}


void XclPivotCache::WriteCacheEnd( XclExpStream& rStrm )
{
	rStrm.StartRecord( 0x000A, 0 );
	rStrm.EndRecord();
}


void XclPivotCache::WriteIndexList( XclExpStream& rStrm )
{
	ULONG nIndLen = GetIndexRecLen();

	for( UINT16 nRow = 0; nRow < nSrcRecs; nRow++ )
	{
		rStrm.StartRecord( 0x00C8, nIndLen );
		for( XclPivotCacheField* pField = _First(); pField; pField = _Next() )
			pField->WriteIndex( rStrm, nRow );
		rStrm.EndRecord();
	}
}


void XclPivotCache::Write()
{
	SvStorageStreamRef xSvStrm = pExcRoot->pPivotCacheStorage->OpenStream(
        ScfTools::GetHexStr( nIndex ), STREAM_READWRITE | STREAM_TRUNC );
	if( xSvStrm.Is() )
	{
		XclExpStream aXclStrm( *xSvStrm, EXC_MAXRECLEN_BIFF8 );

		WriteCacheHeader( aXclStrm );
		for( XclPivotCacheField* pField = _First(); pField; pField = _Next() )
			pField->Write( aXclStrm );
		WriteIndexList( aXclStrm );
		WriteCacheEnd( aXclStrm );

		xSvStrm->Commit();
	}
}


BOOL XclPivotCache::HasGrandTotal( UINT16 nAxisFlag ) const
{
	switch( nAxisFlag )
	{
		case EXC_PIVOT_AXIS_ROW:	return bRowGrand;
		case EXC_PIVOT_AXIS_COL:	return bColGrand;
		default:					return FALSE;
	}
}


const ScfUInt16List* XclPivotCache::GetPosList( UINT16 nAxisFlag ) const
{
	switch( nAxisFlag )
	{
		case EXC_PIVOT_AXIS_ROW:	return &aRowPosList;
		case EXC_PIVOT_AXIS_COL:	return &aColPosList;
		case EXC_PIVOT_AXIS_DATA:	return &aDataPosList;
		default:					return NULL;
	}
}


void XclPivotCache::BuildPosList( ScfUInt16List& rList, UINT16& rCount, UINT16 nAxisFlag )
{
	for( UINT16 nPos = 0; nPos < rCount; nPos++ )
	{
		BOOL bFound	= FALSE;
		BOOL bMulti	= FALSE;
		for( UINT16 nField = 0; (nField < Count()) && !bFound; nField++ )
		{
			XclPivotCacheField* pField = _Get( nField );
			if( pField )
			{
				if( nAxisFlag == EXC_PIVOT_AXIS_DATA )
					bFound = pField->IsInDataPosList( nPos );
				else
				{
					bFound = bMulti = (HasMultiData( nAxisFlag ) && (nDataFPos == nPos));
					if( !bFound )
						bFound = (pField->GetAxis() & nAxisFlag) && (pField->GetPosition() == nPos);
				}
			}
			if( bFound )
				rList.Append( bMulti ? EXC_SXIVD_IDDATA : nField );
		}
	}
	if( (nDataFAxis == nAxisFlag) && !HasMultiData( nAxisFlag ) )
		rCount--;
}



//_________________________________________________________
// XclPivotCacheList

XclPivotCacheList::XclPivotCacheList( RootData* pRD, ScDPCollection& rDPColl ) :
		ExcRoot( pRD )
{
	for( USHORT nObjCnt = 0; nObjCnt < rDPColl.GetCount(); nObjCnt++ )
	{
		ScDPObject* pDPObject = rDPColl[ nObjCnt ];
		if( pDPObject )
			Append( new XclPivotCache( pRD, *pDPObject, nObjCnt + 1 ) );
	}
}


XclPivotCacheList::~XclPivotCacheList()
{
	for( XclPivotCache* pCache = _First(); pCache; pCache = _Next() )
		delete pCache;
}


void XclPivotCacheList::Write()
{
	if( !pExcRoot->pPivotCacheStorage )
		pExcRoot->pPivotCacheStorage =
			pExcRoot->pRootStorage->OpenStorage(
				String::CreateFromAscii(pPivotCacheStorageName), STREAM_STD_WRITE );
	if( !pExcRoot->pPivotCacheStorage )
		return;

	for( XclPivotCache* pCache = _First(); pCache; pCache = _Next() )
		pCache->Write();
}



//============================================================================
// biff records

//_________________________________________________________
// XclSxIdStm

void XclSxIdStm::SaveCont( XclExpStream& rStrm )
{
	rStrm << nIndex;
}

UINT16 XclSxIdStm::GetNum() const
{
	return 0x00D5;
}

ULONG XclSxIdStm::GetLen() const
{
	return 2;
}



//_________________________________________________________
// XclSxVs

void XclSxVs::SaveCont( XclExpStream& rStrm )
{
	rStrm << nSource;
}

UINT16 XclSxVs::GetNum() const
{
	return 0x00E3;
}

ULONG XclSxVs::GetLen() const
{
	return 2;
}



//_________________________________________________________
// XclSxView

XclSxView::XclSxView( const XclPivotCache& rCache, UINT16 nCacheInd ) :
		nIndex( nCacheInd ),
		sTableName( rCache.GetTableName() ),
		sDataName( rCache.GetDataName() ),
		aOutRange( rCache.GetOutRange() ),
		nFields( ( UINT16 ) rCache.Count() ),
		nRowF( rCache.GetRowFCount() ),
		nColF( rCache.GetColFCount() ),
		nPageF( rCache.GetPageFCount() ),
		nDataF( rCache.GetDataFCount() )
{
	nDataFAxis = rCache.GetDataFAxis();
	if( nDataFAxis == EXC_PIVOT_AXIS_NONE )
		nDataFAxis = EXC_PIVOT_AXIS_ROW;
	nDataFPos = rCache.HasMultiData() ? rCache.GetDataFPos() : 0xFFFF;

	aFirstData = aOutRange.aStart;
	aFirstData.IncRow( nColF + 1 );
	aFirstData.IncCol( nRowF );

	nFirstHeading = aOutRange.aStart.Row();
	if( nColF )
		nFirstHeading++;

	if( nDataF )
	{
		nDataRows = (UINT16)(aOutRange.aEnd.Row() - aFirstData.Row() + 1);
		nDataCols = (UINT16)(aOutRange.aEnd.Col() - aFirstData.Col() + 1);
	}
	else
		nDataRows = nDataCols = 0;

	nFlags = EXC_SXVIEW_COMMON;
	if( rCache.HasGrandTotal( EXC_PIVOT_AXIS_ROW ) )
		nFlags |= EXC_SXVIEW_ROWGRAND;
	if( rCache.HasGrandTotal( EXC_PIVOT_AXIS_COL ) )
		nFlags |= EXC_SXVIEW_COLGRAND;
}


void XclSxView::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< (UINT16) aOutRange.aStart.Row()	// 1st row
			<< (UINT16) aOutRange.aEnd.Row()	// last row
			<< (UINT16) aOutRange.aStart.Col()	// 1st col
			<< (UINT16) aOutRange.aEnd.Col()	// last col
			<< nFirstHeading					// 1st heading row
			<< (UINT16) aFirstData.Row()		// 1st data row
			<< (UINT16) aFirstData.Col()		// 1st data col
			<< nIndex							// cache index
			<< (UINT16) 0x0000					// reserved
			<< nDataFAxis						// axis for data
			<< nDataFPos						// pos for data
			<< nFields							// num of fields
			<< nRowF							// num of row fields
			<< nColF							// num of col fields
			<< nPageF							// num of page fields
			<< nDataF							// num of data fields
			<< nDataRows						// num of data rows
			<< nDataCols						// num of data cols
			<< nFlags							// flags
			<< (UINT16) 0x0001					// index autoformat
			<< sTableName.GetLen()
			<< sDataName.GetLen();
	sTableName.WriteFlags( rStrm );
	sTableName.WriteBuffer( rStrm );
	sDataName.WriteFlags( rStrm	);
	sDataName.WriteBuffer( rStrm );
}


UINT16 XclSxView::GetNum() const
{
	return 0x00B0;
}

ULONG XclSxView::GetLen() const
{
	return 40 + sTableName.GetByteCount() + sDataName.GetByteCount();
}



//_________________________________________________________
// XclSxVi

void XclSxVi::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< nItemType						// item type flag
			<< (UINT16) 0x0000					// flags
			<< nCacheInd						// cache item index
			<< (UINT16) 0xFFFF;					// name in cache
}


UINT16 XclSxVi::GetNum() const
{
	return 0x00B2;
}

ULONG XclSxVi::GetLen() const
{
	return 8;
}



//_________________________________________________________
// XclSxViList

XclSxViList::XclSxViList( const XclPivotCacheField& rField )
{
	if( rField.GetAxis() & EXC_PIVOT_AXIS_RCP_MASK )
	{
		// data items
		for( UINT16 iItem = 0; iItem < rField.Count(); iItem++ )
			Append( new XclSxVi( 0x0000, iItem ) );
		// subtotal items
		for( UINT16 iSubT = 0; iSubT < rField.GetSubTotalCount(); iSubT++ )
			Append( new XclSxVi( rField.GetSubTotal( iSubT ), 0xFFFF ) );
	}
}


XclSxViList::~XclSxViList()
{
	for( XclSxVi* pSxVi = _First(); pSxVi; pSxVi = _Next() )
		delete pSxVi;
}


void XclSxViList::Save( XclExpStream& rStrm )
{
	for( XclSxVi* pSxVi = _First(); pSxVi; pSxVi = _Next() )
		pSxVi->Save( rStrm );
}



//_________________________________________________________
// XclSxVdex

XclSxVdex::XclSxVdex( const XclPivotCacheField& rField )
{
	nFlags = EXC_SXVDEX_COMMON;
	if( rField.IsShowAll() )
		nFlags |= EXC_SXVDEX_SHOWALL;
}


void XclSxVdex::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< nFlags							// flags
			<< (UINT16) 0xFFFF					// autoshow items
			<< (UINT16) 0xFFFF					// autosort field
			<< (UINT16) 0x0000					// autoshow field
			<< (UINT16) 0xFFFF					// number format
			<< (UINT32) 0x00000000				// unknown
			<< (UINT32) 0x00000000;				// unknown
}


UINT16 XclSxVdex::GetNum() const
{
	return 0x0100;
}

ULONG XclSxVdex::GetLen() const
{
	return 20;
}



//_________________________________________________________
// XclSxVd

XclSxVd::XclSxVd( const XclPivotCacheField& rField ) :
	nAxis( rField.GetAxis() ),
	nSubTotalCount( rField.GetSubTotalCount() ),
	nSubTotalMask( rField.GetSubTotalMask() ),
	aSxViList( rField ),
	aSxVdex( rField )
{
}


XclSxVd::~XclSxVd()
{
}


void XclSxVd::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< nAxis							// Orientation
			<< nSubTotalCount					// num of subtotals
			<< nSubTotalMask					// type flags
			<< (UINT16) aSxViList.Count()		// num of items
			<< (UINT16) 0xFFFF;					// name in cache
}


void XclSxVd::Save( XclExpStream& rStrm )
{
	ExcRecord::Save( rStrm );
	aSxViList.Save( rStrm );
	aSxVdex.Save( rStrm );
}


UINT16 XclSxVd::GetNum() const
{
	return 0x00B1;
}

ULONG XclSxVd::GetLen() const
{
	return 10;
}



//_________________________________________________________
// XclSxVdList

XclSxVdList::XclSxVdList( const XclPivotCache& rCache )
{
	const XclPivotCacheField* pField;

	for( UINT16 iField = 0; iField < rCache.Count(); iField++ )
	{
		pField = rCache.Get( iField );
		if( pField )
			Append( new XclSxVd( *pField ) );
	}
}


XclSxVdList::~XclSxVdList()
{
	for( XclSxVd* pSxVd = _First(); pSxVd; pSxVd = _Next() )
		delete pSxVd;
}


void XclSxVdList::Save( XclExpStream& rStrm )
{
	for( XclSxVd* pSxVd = _First(); pSxVd; pSxVd = _Next() )
		pSxVd->Save( rStrm );
}



//_________________________________________________________
// XclSxIvd

XclSxIvd::XclSxIvd( const XclPivotCache& rCache, UINT16 nAxis )
{
    const ScfUInt16List* pList = rCache.GetPosList( nAxis );

	if( pList )
        aIndexList = *pList;
}


void XclSxIvd::SaveCont( XclExpStream& rStrm )
{
	for( UINT16 iEntry = 0; iEntry < aIndexList.Count(); iEntry++ )
        rStrm << aIndexList.GetValue( iEntry );
}


void XclSxIvd::Save( XclExpStream& rStrm )
{
	if( aIndexList.Count() )
		ExcRecord::Save( rStrm );
}


UINT16 XclSxIvd::GetNum() const
{
	return 0x00B4;
}

ULONG XclSxIvd::GetLen() const
{
	return aIndexList.Count() * 2;
}



//_________________________________________________________
// XclSxDi

void XclSxDi::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< nIndex							// index to SXVD
			<< nFunction						// function index
			<< (UINT16) 0x0000					// data display
			<< (UINT16) 0x0000					// data disp SXVD ind
			<< (UINT16) 0x0000					// data disp SXVI ind
			<< (UINT16) 0x0000;					// format table ind
	sFieldName.Write( rStrm );
}

UINT16 XclSxDi::GetNum() const
{
	return 0x00C5;
}

ULONG XclSxDi::GetLen() const
{
	return 12 + sFieldName.GetByteCount();
}



//_________________________________________________________
// XclSxDiList

XclSxDiList::XclSxDiList( const XclPivotCache& rCache )
{
    const ScfUInt16List* pList = rCache.GetPosList( EXC_PIVOT_AXIS_DATA );
	if( pList )
	{
		for( UINT16 nPos = 0; nPos < pList->Count(); nPos++ )
		{
            UINT16                      nInd    = pList->GetValue( nPos );
			const XclPivotCacheField*	pField	= rCache.Get( nInd );

			if( pField )
			{
				String sName;
				UINT16 nFunc = pField->GetFunction( nPos );
				lcl_GetFuncName( sName, nFunc );
				sName.AppendAscii( RTL_CONSTASCII_STRINGPARAM( " - " ) );
				sName += pField->GetName();
				Append( new XclSxDi( nInd, nFunc, sName ) );
			}
		}
	}
}


XclSxDiList::~XclSxDiList()
{
	for( XclSxDi* pSxDi = _First(); pSxDi; pSxDi = _Next() )
		delete pSxDi;
}


void XclSxDiList::Save( XclExpStream& rStrm )
{
	for( XclSxDi* pSxDi = _First(); pSxDi; pSxDi = _Next() )
		pSxDi->Save( rStrm );
}


//_________________________________________________________
// XclSxLi

XclSxLi::XclSxLi( const XclPivotCache& rCache, UINT16 nAxisFlag, UINT16 nLines )
{
    const ScfUInt16List* pRCList = rCache.GetPosList( nAxisFlag );

	nLineCount	= rCache.GetDataFCount() ? nLines : 0;
	nIndCount	= pRCList ? ( UINT16 ) pRCList->Count() : 0;
}


void XclSxLi::SaveCont( XclExpStream& rStrm )
{
	if( nLineCount )
	{
		ULONG nItemCount = 8 + 2 * nIndCount;
		rStrm.SetSliceLen( nItemCount );
		rStrm.WriteZeroBytes( nItemCount * nLineCount );
	}
}


UINT16 XclSxLi::GetNum() const
{
	return 0x00B5;
}

ULONG XclSxLi::GetLen() const
{
	return nLineCount * 2 * (4 + nIndCount);
}



//_________________________________________________________
// XclSxEx

void XclSxEx::SaveCont( XclExpStream& rStrm )
{
	rStrm	<< (UINT16) 0x0000					// num of SXFORMULA
			<< (UINT16) 0xFFFF					// DisplayErrorString chars
			<< (UINT16) 0xFFFF					// DisplayNullString chars
			<< (UINT16) 0xFFFF					// tag string chars
			<< (UINT16) 0x0000					// num of RTSXSELECT
			<< (UINT16) 0x0000					// ?? page field p row
			<< (UINT16) 0x0000					// ?? page field p col
			<< (UINT16) 0x0200					// flags 1
			<< (UINT16) 0x004F					// flags 2
			<< (UINT16) 0xFFFF					// page field style str chars
			<< (UINT16) 0xFFFF					// table style string chars
			<< (UINT16) 0xFFFF;					// vacate style string chars
}

UINT16 XclSxEx::GetNum() const
{
	return 0x00F1;
}

ULONG XclSxEx::GetLen() const
{
	return 24;
}



//_________________________________________________________
// XclPivotTableRecs

XclPivotTableRecs::XclPivotTableRecs( const XclPivotCache& rCache, UINT16 nCacheInd ) :
		aSxView( rCache, nCacheInd ),
		aSxVdList( rCache ),
		aSxIvdRow( rCache, EXC_PIVOT_AXIS_ROW ),
		aSxIvdCol( rCache, EXC_PIVOT_AXIS_COL ),
		aSxDiList( rCache ),
		pSxLiRow( NULL ),
		pSxLiCol( NULL )
{
	if( rCache.GetDataFCount() )
	{
		pSxLiRow = new XclSxLi( rCache, EXC_PIVOT_AXIS_ROW, aSxView.GetDataRows() );
		pSxLiCol = new XclSxLi( rCache, EXC_PIVOT_AXIS_COL, aSxView.GetDataCols() );
	}
}


XclPivotTableRecs::~XclPivotTableRecs()
{
	if( pSxLiRow )
		delete pSxLiRow;
	if( pSxLiCol )
		delete pSxLiCol;
}


void XclPivotTableRecs::Save( XclExpStream& rStrm )
{
	aSxView.Save( rStrm );
	aSxVdList.Save( rStrm );
	aSxIvdRow.Save( rStrm );
	aSxIvdCol.Save( rStrm );
	aSxDiList.Save( rStrm );
	if( pSxLiRow )
		pSxLiRow->Save( rStrm );
	if( pSxLiCol )
		pSxLiCol->Save( rStrm );
	aSxEx.Save( rStrm );
}

