/*************************************************************************
 *
 *  $RCSfile: inethist.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: hr $ $Date: 2001/10/12 16:53:21 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _RTL_MEMORY_H_
#include <rtl/memory.h>
#endif

#ifndef _SOLAR_H
#include <tools/solar.h>
#endif

#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif

#ifndef _STREAM_HXX
#include <tools/stream.hxx>
#endif

#ifndef _URLOBJ_HXX
#include <tools/urlobj.hxx>
#endif

#ifndef _SHL_HXX
#include <tools/shl.hxx>
#endif

#ifndef _INETHIST_HXX
#include <inethist.hxx>
#endif
#include <algorithm>

/*========================================================================
 *
 * INetURLHistory internals.
 *
 *======================================================================*/
#define INETHIST_DEF_FTP_PORT    21
#define INETHIST_DEF_HTTP_PORT   80
#define INETHIST_DEF_HTTPS_PORT 443

#define INETHIST_DEF_FILENAME "inethist.dat"
#define INETHIST_SIZE_LIMIT   1024
#define INETHIST_MAGIC_HEAD   0x484D4849UL

/*
 * INetURLHistoryHint implementation.
 */
IMPL_PTRHINT (INetURLHistoryHint, const INetURLObject);

/*========================================================================
 *
 * INetURLHistory_Impl interface.
 *
 *======================================================================*/
class INetURLHistory_Impl
{
	/** head_entry.
	*/
	struct head_entry
	{
		/** Representation.
		*/
		UINT32 m_nMagic;
		UINT16 m_nNext;
		UINT16 m_nMBZ;

		/** Initialization.
		*/
		void initialize (void)
		{
			m_nMagic = INETHIST_MAGIC_HEAD;
			m_nNext  = 0;
			m_nMBZ   = 0;
		}

		/** swap (internal and external representation).
		*/
		void swap (void)
		{
#ifdef __BIGENDIAN
			m_nMagic = SWAPLONG(m_nMagic);
			m_nNext  = SWAPSHORT(m_nNext);
			m_nMBZ   = SWAPSHORT(m_nMBZ);
#endif /* __BIGENDIAN */
		}
	};

	/** hash_entry.
	*/
	struct hash_entry
	{
		/** Representation.
		*/
		UINT32 m_nHash;
		UINT16 m_nLru;
		UINT16 m_nMBZ;

		/** Initialization.
		*/
		void initialize (UINT16 nLru, UINT32 nHash = 0)
		{
			m_nHash = nHash;
			m_nLru  = nLru;
			m_nMBZ  = 0;
		}

		/** Comparison.
		*/
		BOOL operator== (const hash_entry &rOther) const
		{
			return (m_nHash == rOther.m_nHash);
		}
		BOOL operator< (const hash_entry &rOther) const
		{
			return (m_nHash < rOther.m_nHash);
		}

		BOOL operator== (UINT32 nHash) const
		{
			return (m_nHash == nHash);
		}
		BOOL operator< (UINT32 nHash) const
		{
			return (m_nHash < nHash);
		}

		/** swap (internal and external representation).
		*/
		void swap (void)
		{
#ifdef __BIGENDIAN
			m_nHash = SWAPLONG(m_nHash);
			m_nLru  = SWAPSHORT(m_nLru);
			m_nMBZ  = SWAPSHORT(m_nMBZ);
#endif /* __BIGENDIAN */
		}
	};

	/** lru_entry.
	*/
	struct lru_entry
	{
		/** Representation.
		*/
		UINT32 m_nHash;
		UINT16 m_nNext;
		UINT16 m_nPrev;

		/** Initialization.
		*/
		void initialize (UINT16 nThis, UINT32 nHash = 0)
		{
			m_nHash = nHash;
			m_nNext = nThis;
			m_nPrev = nThis;
		}

		/** swap (internal and external representation).
		*/
		void swap (void)
		{
#ifdef __BIGENDIAN
			m_nHash = SWAPLONG(m_nHash);
			m_nNext = SWAPSHORT(m_nNext);
			m_nPrev = SWAPSHORT(m_nPrev);
#endif /* __BIGENDIAN */
		}
	};

	/** CRC polynomial 0xEDB88320.
	*/
	static const UINT32 m_pTable[256];

	/** Representation.
	*/
	head_entry m_aHead;
	hash_entry m_pHash[INETHIST_SIZE_LIMIT];
	lru_entry  m_pList[INETHIST_SIZE_LIMIT];

	/** Initialization.
	*/
	void initialize (void);

	void downheap (hash_entry a[], UINT16 n, UINT16 k);
	void heapsort (hash_entry a[], UINT16 n);

	/** capacity.
	*/
	UINT16 capacity (void) const
	{
		return (UINT16)(INETHIST_SIZE_LIMIT);
	}

	/** updcrc32.
	*/
	UINT32 updcrc32 (UINT32 crc, BYTE octet) const
	{
		return m_pTable[((crc) ^ (octet)) & 0xff] ^ ((crc) >> 8);
	}

	/** crc32.
	*/
	UINT32 crc32 (UniString const & rData) const;

	/** find.
	*/
	UINT16 find (UINT32 nHash) const;

	/** move.
	*/
	void move (UINT16 nSI, UINT16 nDI);

	/** backlink.
	*/
	void backlink (UINT16 nThis, UINT16 nTail)
	{
		register lru_entry &rThis = m_pList[nThis];
		register lru_entry &rTail = m_pList[nTail];

		rTail.m_nNext = nThis;
		rTail.m_nPrev = rThis.m_nPrev;
		rThis.m_nPrev = nTail;
		m_pList[rTail.m_nPrev].m_nNext = nTail;
	}

	/** unlink.
	*/
	void unlink (UINT16 nThis)
	{
		register lru_entry &rThis = m_pList[nThis];

		m_pList[rThis.m_nPrev].m_nNext = rThis.m_nNext;
		m_pList[rThis.m_nNext].m_nPrev = rThis.m_nPrev;
		rThis.m_nNext = nThis;
		rThis.m_nPrev = nThis;
	}

	/** Not implemented.
	*/
	INetURLHistory_Impl (const INetURLHistory_Impl&);
	INetURLHistory_Impl& operator= (const INetURLHistory_Impl&);

public:
	INetURLHistory_Impl (void);
	~INetURLHistory_Impl (void);

	/** load/save.
	*/
	void load( const String& rFileURL );
	void save( const String& rFileURL );

	/** putUrl/queryUrl.
	*/
	void putUrl   (const String &rUrl);
	BOOL queryUrl (const String &rUrl);

	/** getHistory.
	*/
	static INetURLHistory*& getHistory (void);
};

/*========================================================================
 *
 * INetURLHistory_Impl implementation.
 *
 *======================================================================*/
/*
 * CRC polynomial 0xEDB88320.
 */
const UINT32 INetURLHistory_Impl::m_pTable[] =
{
	/* 0 */
	0x00000000, 0x77073096, 0xEE0E612C, 0x990951BA,
	0x076DC419, 0x706AF48F, 0xE963A535, 0x9E6495A3,
	0x0EDB8832, 0x79DCB8A4, 0xE0D5E91E, 0x97D2D988,
	0x09B64C2B, 0x7EB17CBD, 0xE7B82D07, 0x90BF1D91,

	/* 1 */
	0x1DB71064, 0x6AB020F2, 0xF3B97148, 0x84BE41DE,
	0x1ADAD47D, 0x6DDDE4EB, 0xF4D4B551, 0x83D385C7,
	0x136C9856, 0x646BA8C0, 0xFD62F97A, 0x8A65C9EC,
	0x14015C4F, 0x63066CD9, 0xFA0F3D63, 0x8D080DF5,

	/* 2 */
	0x3B6E20C8, 0x4C69105E, 0xD56041E4, 0xA2677172,
	0x3C03E4D1, 0x4B04D447, 0xD20D85FD, 0xA50AB56B,
	0x35B5A8FA, 0x42B2986C, 0xDBBBC9D6, 0xACBCF940,
	0x32D86CE3, 0x45DF5C75, 0xDCD60DCF, 0xABD13D59,

	/* 3 */
	0x26D930AC, 0x51DE003A, 0xC8D75180, 0xBFD06116,
	0x21B4F4B5, 0x56B3C423, 0xCFBA9599, 0xB8BDA50F,
	0x2802B89E, 0x5F058808, 0xC60CD9B2, 0xB10BE924,
	0x2F6F7C87, 0x58684C11, 0xC1611DAB, 0xB6662D3D,

	/* 4 */
	0x76DC4190, 0x01DB7106, 0x98D220BC, 0xEFD5102A,
	0x71B18589, 0x06B6B51F, 0x9FBFE4A5, 0xE8B8D433,
	0x7807C9A2, 0x0F00F934, 0x9609A88E, 0xE10E9818,
	0x7F6A0DBB, 0x086D3D2D, 0x91646C97, 0xE6635C01,

	/* 5 */
	0x6B6B51F4, 0x1C6C6162, 0x856530D8, 0xF262004E,
	0x6C0695ED, 0x1B01A57B, 0x8208F4C1, 0xF50FC457,
	0x65B0D9C6, 0x12B7E950, 0x8BBEB8EA, 0xFCB9887C,
	0x62DD1DDF, 0x15DA2D49, 0x8CD37CF3, 0xFBD44C65,

	/* 6 */
	0x4DB26158, 0x3AB551CE, 0xA3BC0074, 0xD4BB30E2,
	0x4ADFA541, 0x3DD895D7, 0xA4D1C46D, 0xD3D6F4FB,
	0x4369E96A, 0x346ED9FC, 0xAD678846, 0xDA60B8D0,
	0x44042D73, 0x33031DE5, 0xAA0A4C5F, 0xDD0D7CC9,

	/* 7 */
	0x5005713C, 0x270241AA, 0xBE0B1010, 0xC90C2086,
	0x5768B525, 0x206F85B3, 0xB966D409, 0xCE61E49F,
	0x5EDEF90E, 0x29D9C998, 0xB0D09822, 0xC7D7A8B4,
	0x59B33D17, 0x2EB40D81, 0xB7BD5C3B, 0xC0BA6CAD,

	/* 8 */
	0xEDB88320, 0x9ABFB3B6, 0x03B6E20C, 0x74B1D29A,
	0xEAD54739, 0x9DD277AF, 0x04DB2615, 0x73DC1683,
	0xE3630B12, 0x94643B84, 0x0D6D6A3E, 0x7A6A5AA8,
	0xE40ECF0B, 0x9309FF9D, 0x0A00AE27, 0x7D079EB1,

	/* 9 */
	0xF00F9344, 0x8708A3D2, 0x1E01F268, 0x6906C2FE,
	0xF762575D, 0x806567CB, 0x196C3671, 0x6E6B06E7,
	0xFED41B76, 0x89D32BE0, 0x10DA7A5A, 0x67DD4ACC,
	0xF9B9DF6F, 0x8EBEEFF9, 0x17B7BE43, 0x60B08ED5,

	/* A */
	0xD6D6A3E8, 0xA1D1937E, 0x38D8C2C4, 0x4FDFF252,
	0xD1BB67F1, 0xA6BC5767, 0x3FB506DD, 0x48B2364B,
	0xD80D2BDA, 0xAF0A1B4C, 0x36034AF6, 0x41047A60,
	0xDF60EFC3, 0xA867DF55, 0x316E8EEF, 0x4669BE79,

	/* B */
	0xCB61B38C, 0xBC66831A, 0x256FD2A0, 0x5268E236,
	0xCC0C7795, 0xBB0B4703, 0x220216B9, 0x5505262F,
	0xC5BA3BBE, 0xB2BD0B28, 0x2BB45A92, 0x5CB36A04,
	0xC2D7FFA7, 0xB5D0CF31, 0x2CD99E8B, 0x5BDEAE1D,

	/* C */
	0x9B64C2B0, 0xEC63F226, 0x756AA39C, 0x026D930A,
	0x9C0906A9, 0xEB0E363F, 0x72076785, 0x05005713,
	0x95BF4A82, 0xE2B87A14, 0x7BB12BAE, 0x0CB61B38,
	0x92D28E9B, 0xE5D5BE0D, 0x7CDCEFB7, 0x0BDBDF21,

	/* D */
	0x86D3D2D4, 0xF1D4E242, 0x68DDB3F8, 0x1FDA836E,
	0x81BE16CD, 0xF6B9265B, 0x6FB077E1, 0x18B74777,
	0x88085AE6, 0xFF0F6A70, 0x66063BCA, 0x11010B5C,
	0x8F659EFF, 0xF862AE69, 0x616BFFD3, 0x166CCF45,

	/* E */
	0xA00AE278, 0xD70DD2EE, 0x4E048354, 0x3903B3C2,
	0xA7672661, 0xD06016F7, 0x4969474D, 0x3E6E77DB,
	0xAED16A4A, 0xD9D65ADC, 0x40DF0B66, 0x37D83BF0,
	0xA9BCAE53, 0xDEBB9EC5, 0x47B2CF7F, 0x30B5FFE9,

	/* F */
	0xBDBDF21C, 0xCABAC28A, 0x53B39330, 0x24B4A3A6,
	0xBAD03605, 0xCDD70693, 0x54DE5729, 0x23D967BF,
	0xB3667A2E, 0xC4614AB8, 0x5D681B02, 0x2A6F2B94,
	0xB40BBE37, 0xC30C8EA1, 0x5A05DF1B, 0x2D02EF8D
};

/*
 * INetURLHistory_Impl.
 */
INetURLHistory_Impl::INetURLHistory_Impl (void)
{
	initialize();
}

/*
 * ~INetURLHistory_Impl.
 */
INetURLHistory_Impl::~INetURLHistory_Impl (void)
{
}

/*
 * initialize.
 */
void INetURLHistory_Impl::initialize (void)
{
	m_aHead.initialize();

	USHORT i, n = capacity();
	for (i = 0; i < n; i++)
		m_pHash[i].initialize(i);
	for (i = 0; i < n; i++)
		m_pList[i].initialize(i);
	for (i = 1; i < n; i++)
		backlink (m_aHead.m_nNext, i);
}

/*
 * downheap.
 */
void INetURLHistory_Impl::downheap (hash_entry a[], UINT16 n, UINT16 k)
{
	hash_entry h = a[k];
	while (k < n / 2)
	{
		UINT16 i = k + k + 1;
		if (((i + 1) < n) && (a[i] < a[i + 1])) i++;
		if (!(h < a[i])) break;
		a[k] = a[i];
		k = i;
	}
	a[k] = h;
}

/*
 * heapsort.
 */
void INetURLHistory_Impl::heapsort (hash_entry a[], UINT16 n)
{
	hash_entry h;

	for (UINT16 k = (n - 1) / 2 + 1; k > 0; k--)
		downheap (a, n, k - 1);

	while (n > 0)
	{
		h        = a[0    ];
		a[0    ] = a[n - 1];
		a[n - 1] = h;
		downheap (a, --n, 0);
	}
}

/*
 * load (called from INetURLHistory::SetLocation()).
 */
void INetURLHistory_Impl::load( const String &rFileURL )
{
	INetURLObject aObj( rFileURL );
	SvFileStream aStrm( aObj.PathToFileName(), STREAM_READ | STREAM_NOCREATE );
	if (aStrm.IsOpen())
	{
		// Load.
		aStrm.Read (&m_aHead, sizeof (m_aHead));
		aStrm.Read (m_pList, sizeof (m_pList));
		aStrm.Close();

#ifdef __BIGENDIAN
		// Swap to internal representation.
		m_aHead.swap();
#endif /* __BIGENDIAN */

		// Build hash table.
		UINT16 i, n = capacity();
		for (i = 0; i < n; i++)
		{
#ifdef __BIGENDIAN
			// Swap to internal representation.
			m_pList[i].swap();
#endif /* __BIGENDIAN */
			m_pHash[i].initialize (i, m_pList[i].m_nHash);
		}

		heapsort (m_pHash, n);
	}
}

/*
 * save (called from INetURLHistory::~INetURLHistory()).
 */
void INetURLHistory_Impl::save( const String& rFileURL )
{
	INetURLObject aObj( rFileURL );
	SvFileStream aStrm( aObj.PathToFileName(), STREAM_READWRITE | STREAM_TRUNC );
	if (aStrm.IsOpen() && aStrm.IsWritable())
	{
#ifdef __BIGENDIAN
		// Swap to external representation.
		m_aHead.swap();
		UINT16 i, n = capacity();
		for (i = 0; i < n; i++)
			m_pList[i].swap();
#endif /* __BIGENDIAN */

		// Save.
		aStrm.Write (&m_aHead, sizeof(m_aHead));
		aStrm.Write (m_pList, sizeof(m_pList));
		aStrm.Close();
	}
}

/*
 * crc32.
 */
UINT32 INetURLHistory_Impl::crc32 (UniString const & rData) const
{
	UINT32 nCRC32 = 0;

	register const sal_Unicode *p = rData.GetBuffer();
	register const sal_Unicode *q = p + rData.Len();

	nCRC32 = ~nCRC32;
	while (p < q)
		nCRC32 = updcrc32(nCRC32, BYTE(*p++));
	nCRC32 = ~nCRC32;

	return nCRC32;
}

/*
 * find.
 */
UINT16 INetURLHistory_Impl::find (UINT32 nHash) const
{
	UINT16 l = 0;
	UINT16 r = capacity() - 1;
	UINT16 c = capacity();

	while ((l < r) && (r < c))
	{
		UINT16 m = (l + r) / 2;
		if (m_pHash[m] == nHash)
			return m;

		if (m_pHash[m] < nHash)
			l = m + 1;
		else
			r = m - 1;
	}
	return l;
}

/*
 * move.
 */
void INetURLHistory_Impl::move (UINT16 nSI, UINT16 nDI)
{
	hash_entry e = m_pHash[nSI];
	if (nSI < nDI)
	{
		// shift left.
		rtl_moveMemory (
			&m_pHash[nSI    ],
			&m_pHash[nSI + 1],
			(nDI - nSI) * sizeof(hash_entry));
	}
	if (nSI > nDI)
	{
		// shift right.
		rtl_moveMemory (
			&m_pHash[nDI + 1],
			&m_pHash[nDI    ],
			(nSI - nDI) * sizeof(hash_entry));
	}
	m_pHash[nDI] = e;
}

/*
 * putUrl.
 */
void INetURLHistory_Impl::putUrl (const String &rUrl)
{
	UINT32 h = crc32 (rUrl);
	UINT16 k = find (h);
	if ((k < capacity()) && (m_pHash[k] == h))
	{
		// Cache hit.
		UINT16 nMRU = m_pHash[k].m_nLru;
		if (nMRU != m_aHead.m_nNext)
		{
			// Update LRU chain.
			unlink (nMRU);
			backlink (m_aHead.m_nNext, nMRU);

			// Rotate LRU chain.
			m_aHead.m_nNext = m_pList[m_aHead.m_nNext].m_nPrev;
		}
	}
	else
	{
		// Cache miss. Obtain least recently used.
		UINT16 nLRU = m_pList[m_aHead.m_nNext].m_nPrev;

		UINT16 nSI = find (m_pList[nLRU].m_nHash);
		if (!(nLRU == m_pHash[nSI].m_nLru))
		{
			// Update LRU chain.
			nLRU = m_pHash[nSI].m_nLru;
			unlink (nLRU);
			backlink (m_aHead.m_nNext, nLRU);
		}

		// Rotate LRU chain.
		m_aHead.m_nNext = m_pList[m_aHead.m_nNext].m_nPrev;

		// Check source and destination.
		UINT16 nDI = std::min (k, UINT16(capacity() - 1));
		if (nSI < nDI)
		{
			if (!(m_pHash[nDI] < h))
				nDI -= 1;
		}
		if (nDI < nSI)
		{
			if (m_pHash[nDI] < h)
				nDI += 1;
		}

		// Assign data.
		m_pList[m_aHead.m_nNext].m_nHash = m_pHash[nSI].m_nHash = h;
		move (nSI, nDI);
	}
}

/*
 * queryUrl.
 */
BOOL INetURLHistory_Impl::queryUrl (const String &rUrl)
{
	UINT32 h = crc32 (rUrl);
	UINT16 k = find (h);
	if ((k < capacity()) && (m_pHash[k] == h))
	{
		// Cache hit.
		return TRUE;
	}
	else
	{
		// Cache miss.
		return FALSE;
	}
}

/*
 * getHistory.
 */
INetURLHistory*& INetURLHistory_Impl::getHistory (void)
{
	void **ppAppData = GetAppData (SHL_HISTORY);
	return (INetURLHistory*&)(*ppAppData);
}

/*========================================================================
 *
 * INetURLHistory implementation.
 *
 *======================================================================*/
/*
 * INetURLHistory.
 */
INetURLHistory::INetURLHistory() : m_pImpl( new INetURLHistory_Impl )
{
}

/*
 * ~INetURLHistory.
 */
INetURLHistory::~INetURLHistory()
{
	if ( m_pImpl )
	{
		m_pImpl->save( m_aLocation );
		DELETEZ( m_pImpl );
	}
}

/*
 * GetOrCreate.
 */
INetURLHistory* INetURLHistory::GetOrCreate()
{
	INetURLHistory *&rpThis = INetURLHistory_Impl::getHistory();
	if (rpThis == NULL)
	{
		// Create single instance.
		rpThis = new INetURLHistory;
	}
	return rpThis;
}

/*
 * Delete.
 */
void INetURLHistory::Delete (void)
{
	INetURLHistory *&rpThis = INetURLHistory_Impl::getHistory();
	DELETEZ (rpThis);
}

/*
 * SetLocation.
 */
void INetURLHistory::SetLocation( const String &rLocation )
{
	DBG_ASSERT (m_pImpl, "SetLocation(): no Implementation");
	if ( m_pImpl )
	{
		INetURLObject aObj( rLocation, INET_PROT_FILE );
		aObj.insertName( UniString( RTL_CONSTASCII_USTRINGPARAM( INETHIST_DEF_FILENAME ) ) );
		m_aLocation = aObj.GetMainURL();
		m_pImpl->load( m_aLocation );
	}
}

/*
 * NormalizeUrl_Impl.
 */
void INetURLHistory::NormalizeUrl_Impl (INetURLObject &rUrl)
{
	switch (rUrl.GetProtocol())
	{
		case INET_PROT_FILE:
			if (!rUrl.IsCaseSensitive())
			{
				String aPath (rUrl.GetURLPath(INetURLObject::NO_DECODE));
				aPath.ToLowerAscii();
				rUrl.SetURLPath (aPath, INetURLObject::NOT_CANONIC);
			}
			break;

		case INET_PROT_FTP:
			if (!rUrl.HasPort())
				rUrl.SetPort (INETHIST_DEF_FTP_PORT);
			break;

		case INET_PROT_HTTP:
			if (!rUrl.HasPort())
				rUrl.SetPort (INETHIST_DEF_HTTP_PORT);
			if (!rUrl.HasURLPath())
				rUrl.SetURLPath ("/");
			break;

		case INET_PROT_HTTPS:
			if (!rUrl.HasPort())
				rUrl.SetPort (INETHIST_DEF_HTTPS_PORT);
			if (!rUrl.HasURLPath())
				rUrl.SetURLPath ("/");
			break;

		default:
			break;
	}
}

/*
 * PutUrl_Impl.
 */
void INetURLHistory::PutUrl_Impl (const INetURLObject &rUrl)
{
	DBG_ASSERT (m_pImpl, "PutUrl_Impl(): no Implementation");
	if (m_pImpl)
	{
		INetURLObject aHistUrl (rUrl);
		NormalizeUrl_Impl (aHistUrl);

		m_pImpl->putUrl (aHistUrl.GetMainURL(INetURLObject::NO_DECODE));
		Broadcast (INetURLHistoryHint (&rUrl));

		if (aHistUrl.HasMark())
		{
			aHistUrl.SetURL (aHistUrl.GetURLNoMark(INetURLObject::NO_DECODE),
							 INetURLObject::NOT_CANONIC);

			m_pImpl->putUrl (aHistUrl.GetMainURL(INetURLObject::NO_DECODE));
			Broadcast (INetURLHistoryHint (&aHistUrl));
		}
	}
}

/*
 * QueryUrl_Impl.
 */
BOOL INetURLHistory::QueryUrl_Impl (const INetURLObject &rUrl)
{
	DBG_ASSERT (m_pImpl, "QueryUrl_Impl(): no Implementation");
	if (m_pImpl)
	{
		INetURLObject aHistUrl (rUrl);
		NormalizeUrl_Impl (aHistUrl);

		return m_pImpl->queryUrl (aHistUrl.GetMainURL(
			                                   INetURLObject::NO_DECODE));
	}
	return FALSE;
}


