/*************************************************************************
 *
 *  $RCSfile: xmlmerge.cxx,v $
 *
 *  $Revision: 1.34 $
 *
 *  last change: $Author: nf $ $Date: 2001/10/18 10:11:34 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRUNTY OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRUNTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc..
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/
#include <stdio.h>
#include <tools/string.hxx>
#include <tools/fsys.hxx>

// local includes
#include "export.hxx"
#include "xmlmerge.hxx"
#include "tokens.h"
#include "utf8conv.hxx"

extern "C" { yyerror( char * ); }
extern "C" { YYWarning( char * ); }

// defines to parse command line
#define STATE_NON  		0x0001
#define STATE_INPUT		0x0002
#define STATE_OUTPUT	0x0003
#define STATE_PRJ		0x0004
#define STATE_ROOT		0x0005
#define STATE_MERGESRC	0x0006
#define STATE_ERRORLOG	0x0007
#define STATE_DONEFILE	0x0008
#define STATE_GENERATOR	0x0009
#define STATE_TYPE		0x000A
#define STATE_UTF8		0x000B
#define STATE_LANGUAGES	0x000C
#define STATE_ISOCODE99	0x000D

// set of global variables
BOOL bEnableExport;
BOOL bMergeMode;
BOOL bErrorLog;
BOOL bGenerator;
BOOL bDoneFile;
BOOL bUTF8;
USHORT nGeneratorType;
USHORT nInputFileType;
ByteString sPrj;
ByteString sPrjRoot;
ByteString sInputFileName;
ByteString sActFileName;
ByteString sOutputFile;
ByteString sMergeSrc;
ByteString sDoneFile;
XMLResParser *pParser = NULL;

#define XML_GEN_PROPERTY	0x0000
#define XML_GEN_XULDTD		0x0001

#define XML_GEN_XRB			0x0000
#define XML_GEN_XGF			0x0001

extern "C" {
// the whole interface to lexer is in this extern "C" section

/*****************************************************************************/
extern char *GetOutputFile( int argc, char* argv[])
/*****************************************************************************/
{
	bEnableExport = FALSE;
	bMergeMode = FALSE;
	bErrorLog = TRUE;
	bGenerator = FALSE;
	bUTF8 = TRUE;
	nGeneratorType = XML_GEN_PROPERTY;
	nInputFileType = XML_GEN_XRB;
	bDoneFile = FALSE;
	sPrj = "";
	sPrjRoot = "";
	sInputFileName = "";
	sActFileName = "";
	Export::sLanguages = "";

	USHORT nState = STATE_NON;
	BOOL bInput = FALSE;

	// parse command line
	for( int i = 1; i < argc; i++ ) {
		if ( ByteString( argv[ i ] ).ToUpperAscii() == "-I" ) {
			nState = STATE_INPUT; // next token specifies source file
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-O" ) {
			nState = STATE_OUTPUT; // next token specifies the dest file
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-P" ) {
			nState = STATE_PRJ; // next token specifies the cur. project
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-R" ) {
			nState = STATE_ROOT; // next token specifies path to project root
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-M" ) {
			nState = STATE_MERGESRC; // next token specifies the merge database
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-E" ) {
			nState = STATE_ERRORLOG;
			bErrorLog = FALSE;
		}
		else if ( ByteString( argv[ i ]).GetToken( 0, ':' ).ToUpperAscii() == "-G" ) {
			nState = STATE_GENERATOR;
			bGenerator = TRUE;
			if ( ByteString( argv[ i ]).GetTokenCount( ':' ) > 1 ) {
				if ( ByteString( argv[ i ]).GetToken( 1, ':' ).ToUpperAscii() == "DTD" )
					nGeneratorType = XML_GEN_XULDTD;
			}
		}
		else if ( ByteString( argv[ i ]).GetToken( 0, ':' ).ToUpperAscii() == "-T" ) {
			nState = STATE_TYPE;
			if ( ByteString( argv[ i ]).GetTokenCount( ':' ) > 1 ) {
				if ( ByteString( argv[ i ]).GetToken( 1, ':' ).ToUpperAscii() == "XGF" )
					nInputFileType = XML_GEN_XGF;
			}
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-D" ) {
			nState = STATE_DONEFILE;
			bDoneFile = TRUE;
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-UTF8" ) {
			nState = STATE_UTF8;
			bUTF8 = TRUE;
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-NOUTF8" ) {
			nState = STATE_UTF8;
			bUTF8 = FALSE;
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-L" ) {
			nState = STATE_LANGUAGES;
		}
		else if ( ByteString( argv[ i ] ).ToUpperAscii() == "-ISO99" ) {
			nState = STATE_ISOCODE99;
		}
		else {
			switch ( nState ) {
				case STATE_NON: {
					return NULL;	// no valid command line
				}
				break;
				case STATE_INPUT: {
					sInputFileName = argv[ i ];
					bInput = TRUE; // source file found
				}
				break;
				case STATE_OUTPUT: {
					sOutputFile = argv[ i ]; // the dest. file
				}
				break;
				case STATE_PRJ: {
					sPrj = ByteString( argv[ i ]);
//					sPrj.ToLowerAscii(); // the project
				}
				break;
				case STATE_ROOT: {
					sPrjRoot = ByteString( argv[ i ]); // path to project root
				}
				break;
				case STATE_MERGESRC: {
					sMergeSrc = ByteString( argv[ i ]);
					bMergeMode = TRUE; // activate merge mode, cause merge database found
				}
				case STATE_DONEFILE: {
					sDoneFile = ByteString( argv[ i ]);
				}
				break;
				case STATE_LANGUAGES: {
					Export::sLanguages = ByteString( argv[ i ]);
				}
				break;
				case STATE_ISOCODE99: {
					Export::sIsoCode99 = ByteString( argv[ i ]);
				}
				break;
			}
		}
	}

	if ( bInput ) {
		// command line is valid
		bEnableExport = TRUE;
		char *pReturn = new char[ sOutputFile.Len() + 1 ];
		strcpy( pReturn, sOutputFile.GetBuffer());
		return pReturn;
	}

	// command line is not valid
	return NULL;
}

/*****************************************************************************/
int IsXGF()
/*****************************************************************************/
{
	return nInputFileType == XML_GEN_XGF;
}

/*****************************************************************************/
int InitXmlExport( char *pOutput )
/*****************************************************************************/
{
	// instanciate Export
	ByteString sOutput( pOutput );

	if ( bGenerator ) {
		DirEntry aEntry( String( sInputFileName, RTL_TEXTENCODING_ASCII_US ));
		ByteString sBaseName( aEntry.GetBase(), RTL_TEXTENCODING_ASCII_US );
		switch ( nGeneratorType ) {
			case XML_GEN_PROPERTY:
				pParser = new XMLPropertyGenerator( sOutputFile, sBaseName );
			break;
			case XML_GEN_XULDTD:
				pParser = new XML_XULDtdGenerator( sOutputFile, sBaseName );
			break;				
		}
	}
	else if ( bMergeMode )
		pParser = new XMLResMerge( sMergeSrc, sOutputFile, bErrorLog );
  	else if ( sOutputFile.Len()) {
		pParser = new XMLResExport( sOutputFile, sPrj, sActFileName );
	}
	else
		pParser = new XMLResParser();

	pParser->SetFileType( nInputFileType );

	return 1;
}

/*****************************************************************************/
int EndXmlExport()
/*****************************************************************************/
{
	if ( bGenerator && bDoneFile && !pParser->GetError()) {
		DirEntry aEntry( String( sDoneFile, RTL_TEXTENCODING_ASCII_US ));
		SvFileStream aStream(
			aEntry.GetFull(), STREAM_STD_WRITE | STREAM_TRUNC );
		if ( !aStream.IsOpen()) {
			ByteString sError( "Unable to open done file: " );
			sError += ByteString(
				aEntry.GetFull(), RTL_TEXTENCODING_ASCII_US );
	 	}
		else
			aStream.Close();
	}

	delete pParser;
	return 1;
}

/*****************************************************************************/
extern FILE *GetXmlFile()
/*****************************************************************************/
{
	// look for valid filename
	if ( sInputFileName.Len()) {

		// able to open file?
		FILE *pFile = fopen( sInputFileName.GetBuffer(), "r" );
		if ( !pFile )
			fprintf( stderr, "Error: Could not open file %s\n",
				sInputFileName.GetBuffer());
		else {
			// this is a valid file which can be opened, so
			// create path to project root
			DirEntry aEntry( String( sInputFileName, RTL_TEXTENCODING_ASCII_US ));
			aEntry.ToAbs();
			ByteString sFullEntry( aEntry.GetFull(), RTL_TEXTENCODING_ASCII_US );
			aEntry += DirEntry( String( "..", RTL_TEXTENCODING_ASCII_US ));
			aEntry += DirEntry( sPrjRoot );
			ByteString sPrjEntry( aEntry.GetFull(), RTL_TEXTENCODING_ASCII_US );

			// create file name, beginnig with project root
			// (e.g.: source\ui\src\menue.src)
			sActFileName = sFullEntry.Copy( sPrjEntry.Len() + 1 );
//			sActFileName.ToLowerAscii();

			fprintf( stdout, "\nProcessing File %s ...\n", sInputFileName.GetBuffer());

			sActFileName.SearchAndReplaceAll( "/", "\\" );

			return pFile;
		}
	}
	// this means the file could not be opened
	return NULL;
}

/*****************************************************************************/
int WorkOnTokenSet( int nTyp, char *pTokenText )
/*****************************************************************************/
{
	pParser->Execute( nTyp, pTokenText );

	return 1;
}

/*****************************************************************************/
void Argument( char *pArg )
/*****************************************************************************/
{
	ByteString sArg( pArg );
	pParser->Argument( sArg );
}

/*****************************************************************************/
int SetError()
/*****************************************************************************/
{
	pParser->SetError();
	return 1;
}
}

extern "C" {
/*****************************************************************************/
int GetError()
/*****************************************************************************/
{
	return pParser->GetError();
}
}

//
// class XMLResParser
//

// defines for DEA-States
#define XMLRESSTATE_START					0x0000
#define XMLRESSTATE_RESSOURCEBUNDLE         0x0001
#define XMLRESSTATE_CONTEXT                 0x0002
#define XMLRESSTATE_KEY                     0x0004
#define XMLRESSTATE_TEXT                    0x0008

/*****************************************************************************/
XMLResParser::XMLResParser()
/*****************************************************************************/
				: nDEAState( XMLRESSTATE_START ),
				bError( FALSE ),
				nFileType( XML_GEN_XRB )
{
}

/*****************************************************************************/
XMLResParser::~XMLResParser()
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::Argument( const ByteString &rArg )
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::ConvertStringToDBFormat( ByteString &rString )
/*****************************************************************************/
{
	ByteString sResult;
	do {
		sResult = rString;
		rString.EraseLeadingChars( _LF );
		rString.EraseLeadingChars( ' ' );
		rString.EraseLeadingChars( '\t' );
		rString.EraseTrailingChars( ' ' );
		rString.EraseTrailingChars( '\t' );
	} while ( sResult != rString );

	rString.SearchAndReplaceAll( "\t", "\\t" );
}

/*****************************************************************************/
void XMLResParser::ConvertStringToXMLFormat( ByteString &rString )
/*****************************************************************************/
{
	rString.SearchAndReplaceAll( "\\t", "\t" );
}

/*****************************************************************************/
int XMLResParser::Execute( int nToken, char * pToken )
/*****************************************************************************/
{
	BOOL bError = FALSE;
	ByteString sToken( pToken );

	switch ( nDEAState ) {
		case XMLRESSTATE_START:
			switch ( nToken ) {
				case XML_RESSOURCEBUNDLE_START:
					nDEAState = XMLRESSTATE_RESSOURCEBUNDLE;
					sCurrentRessourceBundle=
						sToken.GetToken( 1, '\"' ).GetToken( 0, '\"' );
					if ( !sCurrentRessourceBundle.Len())
						sCurrentRessourceBundle=
							sToken.GetToken( 1, '\'' ).GetToken( 0, '\'' );
					Output( sToken );
				break;
				case XML_RESSOURCEBUNDLE_END:
				case XML_CONTEXT_START:
				case XML_CONTEXT_END:
				case XML_KEY_START:
				case XML_KEY_END:
				case XML_TEXT_START	:
				case XML_TEXT_END:
					Error( "SyntaxExrror: misplaced Tag" );
					Output( sToken );
				break;
				default:
					Output( sToken );
				break;
			}
		break;
		case XMLRESSTATE_RESSOURCEBUNDLE:
			switch ( nToken ) {
				case XML_RESSOURCEBUNDLE_END:
					nDEAState = XMLRESSTATE_START;
					EndOfRessourceBundle( sCurrentRessourceBundle );
					Output( sToken );
				break;
				case XML_CONTEXT_START:
					nDEAState = XMLRESSTATE_CONTEXT;
					sCurrentContext =
						sToken.GetToken( 1, '\"' ).GetToken( 0, '\"' );
					if ( !sCurrentContext.Len())
						sCurrentContext =
							sToken.GetToken( 1, '\'' ).GetToken( 0, '\'' );
					Output( sToken );
				break;
				case XML_RESSOURCEBUNDLE_START:
				case XML_CONTEXT_END:
				case XML_KEY_START:
				case XML_KEY_END:
				case XML_TEXT_START	:
				case XML_TEXT_END:
					Error( "SyntaxExrror: misplaced Tag" );
					Output( sToken );
				break;
				default:
					Output( sToken );
				break;
			}
		break;
		case XMLRESSTATE_CONTEXT:
			switch ( nToken ) {
				case XML_CONTEXT_END:
					nDEAState = XMLRESSTATE_RESSOURCEBUNDLE;
					Output( sToken );
				break;
				case XML_KEY_START:
					nDEAState = XMLRESSTATE_KEY;
					sCurrentKey =
						sToken.GetToken( 1, '\"' ).GetToken( 0, '\"' );
					if ( !sCurrentKey.Len())
						sCurrentKey =
							sToken.GetToken( 1, '\'' ).GetToken( 0, '\'' );
					Output( sToken );
				break;
				case XML_RESSOURCEBUNDLE_START:
				case XML_RESSOURCEBUNDLE_END:
				case XML_CONTEXT_START:
				case XML_KEY_END:
				case XML_TEXT_START	:
				case XML_TEXT_END:
					Error( "SyntaxExrror: misplaced Tag" );
					Output( sToken );
				break;
				default:
					Output( sToken );
				break;
			}
		break;
		case XMLRESSTATE_KEY:
			switch ( nToken ) {
				case XML_KEY_END:
					nDEAState = XMLRESSTATE_CONTEXT;
					EndOfKey(
						sCurrentRessourceBundle,
						sCurrentContext,
						sCurrentKey
					);
					Output( sToken );
				break;
				case XML_TEXT_START:
					nDEAState = XMLRESSTATE_TEXT;
					nCurrentLang = GetLang( sToken );
					Output( sToken );
				break;
				case XML_RESSOURCEBUNDLE_START:
				case XML_RESSOURCEBUNDLE_END:
				case XML_CONTEXT_START:
				case XML_CONTEXT_END:
				case XML_KEY_START:
				case XML_TEXT_END:
					Error( "SyntaxExrror: misplaced Tag" );
					Output( sToken );
				break;
				default:
					Output( sToken );
				break;
			}
		break;
		case XMLRESSTATE_TEXT:
			switch ( nToken ) {
				case XML_TEXT_END:
					nDEAState = XMLRESSTATE_KEY;

					if ( nCurrentLang != 0xFFFF )
						WorkOnText(
							sCurrentRessourceBundle,
							sCurrentContext,
							sCurrentKey,
							nCurrentLang,
							sCurrentText
						);
					Output( sCurrentText );
					sCurrentText = "";

					Output( sToken );
				break;
				case XML_TEXTCHAR:
					nDEAState = XMLRESSTATE_TEXT;
					sCurrentText += sToken;
				break;
				case XML_RESSOURCEBUNDLE_START:
				case XML_RESSOURCEBUNDLE_END:
				case XML_CONTEXT_START:
				case XML_CONTEXT_END:
				case XML_KEY_START:
				case XML_KEY_END:
				case XML_TEXT_START	:
					Error( "SyntaxExrror: misplaced Tag" );
					Output( sToken );
				break;
				default:
					Output( sToken );
				break;
			}
		break;
		default:
			Output( sToken );
		break;
	}

	return 1;
};

/*****************************************************************************/
USHORT XMLResParser::GetLang( const ByteString &rStartTextToken )
/*****************************************************************************/
{
	ByteString sLowerLang(
		rStartTextToken.GetToken( 1, '\"' ).GetToken( 0, '\"' ));
	if ( !sLowerLang.Len())
		sLowerLang =
			rStartTextToken.GetToken( 1, '\'' ).GetToken( 0, '\'' );
	ByteString sLang( sLowerLang );
	sLang.ToUpperAscii();

	USHORT nLang = Export::GetLangByIsoLang( sLang );

	if ( nLang != 0xFFFF )
		return nLang;

	ByteString sError( "Unknown language code: " );
	sError += sLowerLang;
	Error( sError );
	return 0xFFFF;
}

/*****************************************************************************/
ByteString XMLResParser::GetIsoLangByIndex( USHORT nIndex )
/*****************************************************************************/
{
	return Export::GetIsoLangByIndex( nIndex );
}

/*****************************************************************************/
ByteString XMLResParser::GetPropertyLangByIndex( USHORT nIndex )
/*****************************************************************************/
{
	switch ( nIndex ) {
		case ENGLISH_US_INDEX: return ENGLISH_US_PROPERTY;
		case PORTUGUESE_BRAZILIAN_INDEX: return PORTUGUESE_BRAZILIAN_PROPERTY;
		case CHINESE_SIMPLIFIED_INDEX: return CHINESE_SIMPLIFIED_PROPERTY;
		case CHINESE_TRADITIONAL_INDEX: return CHINESE_TRADITIONAL_PROPERTY;
		case EXTERN_INDEX: {
			ByteString sReturn( Export::sIsoCode99 );
			sReturn.SearchAndReplace( "-", "_" );
			return sReturn;
		}
	}
	return GetIsoLangByIndex( nIndex );
}

/*****************************************************************************/
void XMLResParser::WorkOnText(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
	const ByteString &sKey,	USHORT nLang, ByteString &rText	)
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::EndOfRessourceBundle(
	const ByteString &rRessourceBundle )
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::Output( const ByteString& rOutput )
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::EndOfKey( const ByteString &rRessourceBundle,
	const ByteString &rContext,	const ByteString &sKey )
/*****************************************************************************/
{
}

/*****************************************************************************/
void XMLResParser::Error( const ByteString &rError )
/*****************************************************************************/
{
	yyerror(( char * ) rError.GetBuffer());
}

//
// class XMLResOutputParser
//

/*****************************************************************************/
XMLResOutputParser::XMLResOutputParser ( const ByteString &rOutputFile )
/*****************************************************************************/
{
	pOutputStream =
		new SvFileStream(
			String( rOutputFile, RTL_TEXTENCODING_ASCII_US ),
			STREAM_STD_WRITE | STREAM_TRUNC
		);

	if ( !pOutputStream->IsOpen()) {
		ByteString sError( "Unable to open output file: " );
		sError += rOutputFile;
		Error( sError );
		delete pOutputStream;
		pOutputStream = NULL;
	}
}

/*****************************************************************************/
XMLResOutputParser::~XMLResOutputParser()
/*****************************************************************************/
{
	if ( pOutputStream ) {
		pOutputStream->Close();
		delete pOutputStream;
	}
}


//
// class XMLResExport
//

/*****************************************************************************/
XMLResExport::XMLResExport(
	const ByteString &rOutputFile, const ByteString &rProject,
	const ByteString &rFilePath )
/*****************************************************************************/
				: XMLResOutputParser( rOutputFile ),
				pResData( NULL ),
				sPrj( rProject ),
				sPath( rFilePath )
{
}

/*****************************************************************************/
XMLResExport::~XMLResExport()
/*****************************************************************************/
{
	delete pResData;
}


/*****************************************************************************/
void XMLResExport::WorkOnText(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
		const ByteString &sKey,	USHORT nLang, ByteString &rText )
/*****************************************************************************/
{
	if ( !pResData ) {
		ByteString sGroupId( rRessourceBundle );
		sGroupId += ".";
		sGroupId += rContext;

		ByteString sPlatform( "" );

		pResData = new ResData( sPlatform, sGroupId );
		pResData->sId = sKey;
	}

	if ( !bUTF8 )
		pResData->sText[ Export::GetLangIndex( nLang )] =
			UTF8Converter::ConvertFromUTF8( rText, Export::GetCharSet( nLang ));
	else 
		pResData->sText[ Export::GetLangIndex( nLang )] = rText;

	ConvertStringToDBFormat( pResData->sText[ Export::GetLangIndex( nLang )]);
}

/*****************************************************************************/
void XMLResExport::EndOfKey(
	const ByteString &rRessourceBundle, const ByteString &rContext,
	const ByteString &sKey )
/*****************************************************************************/
{
	if ( pOutputStream ) {

		char cSearch = 0x00;
		ByteString sSearch( cSearch );

		if ( pResData->sText[ GERMAN_INDEX ].Len() &&
		   	( pResData->sText[ ENGLISH_US_INDEX ].Len() ||
		   		pResData->sText[ ENGLISH_INDEX ].Len()))
		{
			Export::FillInFallbacks( pResData );

			ByteString sTimeStamp( Export::GetTimeStamp());

			for ( ULONG i = 0; i < LANGUAGES; i++ ) {
				if ( LANGUAGE_ALLOWED( i ) || ( i == COMMENT_INDEX )) {
					ByteString sAct = pResData->sText[ i ];
					if ( !sAct.Len() && i ) {
						if ( pResData->sText[ ENGLISH_US_INDEX ].Len())
							sAct = pResData->sText[ ENGLISH_US_INDEX ];
						else
							sAct = pResData->sText[ ENGLISH_INDEX ];
					}

					Export::UnquotHTML( sAct );
					sAct.EraseTrailingChars( 0x0A );

					ByteString sOutput( sPrj ); sOutput += "\t";
					sOutput += sPath;
					sOutput += "\t0\t";
					if ( nFileType == XML_GEN_XRB ) 
						sOutput += "XMLText\t";
					else
						sOutput += "XMLImage\t";
					sOutput += pResData->sGId; sOutput += "\t";
					sOutput += pResData->sId; sOutput += "\t\t\t0\t";
					sOutput += ByteString::CreateFromInt64( Export::LangId[ i ] ); sOutput += "\t";
					sOutput += sAct; sOutput += "\t\t\t\t";
					sOutput += sTimeStamp;

					sOutput.SearchAndReplaceAll( sSearch, "_" );

					pOutputStream->WriteLine( sOutput );
				}
			}
		}
	}
	delete pResData;
	pResData = NULL;
}

//
// class XMLResMerge
//

/*****************************************************************************/
XMLResMerge::XMLResMerge(
	const ByteString &rMergeSource, const ByteString &rOutputFile,
	BOOL bErrorLog )
/*****************************************************************************/
				: XMLResOutputParser( rOutputFile ),
				pMergeDataFile( NULL ),
				pResData( NULL )
{
	if ( rMergeSource.Len())
		pMergeDataFile = new MergeDataFile(
			rMergeSource, bErrorLog, RTL_TEXTENCODING_MS_1252, bUTF8 );
}

/*****************************************************************************/
XMLResMerge::~XMLResMerge()
/*****************************************************************************/
{
	delete pMergeDataFile;
	delete pResData;
}

/*****************************************************************************/
void XMLResMerge::WorkOnText(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
	const ByteString &sKey,	USHORT nLang, ByteString &rText )
/*****************************************************************************/
{
	if ( pMergeDataFile ) {
		if ( !pResData ) {
			ByteString sGroupId( rRessourceBundle );
			sGroupId += ".";
			sGroupId += rContext;

			ByteString sPlatform( "" );

			pResData = new ResData( sPlatform, sGroupId );
			pResData->sId = sKey;
			if ( nFileType == XML_GEN_XRB ) 
				pResData->sResTyp = "xmltext";
			else
				pResData->sResTyp = "xmlimage";
		}

		PFormEntrys *pEntrys = pMergeDataFile->GetPFormEntrys( pResData );
		if ( pEntrys ) {
			ByteString sContent;
			if (( nLang != GERMAN ) &&
				( nLang != ENGLISH ) &&
				( pEntrys->GetText(
					sContent, STRING_TYP_TEXT, Export::GetLangIndex( nLang ))) &&
				( sContent != "-" ) && ( sContent.Len()))
		
			{
				rText = UTF8Converter::ConvertToUTF8(
					sContent, Export::GetCharSet( nLang )) ;
				ConvertStringToXMLFormat( rText );
				Export::QuotHTML( rText );
			}
		}
	}
}

/*****************************************************************************/
void XMLResMerge::Output( const ByteString& rOutput )
/*****************************************************************************/
{
	if ( pOutputStream )
		pOutputStream->Write( rOutput.GetBuffer(), rOutput.Len());
}

/*****************************************************************************/
void XMLResMerge::EndOfKey(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
	const ByteString &sKey )
/*****************************************************************************/
{
	if ( pMergeDataFile && pResData ) {
		PFormEntrys *pEntrys = pMergeDataFile->GetPFormEntrys( pResData );
		if ( pEntrys ) {
			for ( ULONG nIndex = 0; nIndex < LANGUAGES; nIndex++ ) {
				ByteString sContent;
				if (( nIndex != GERMAN_INDEX ) &&
					( nIndex != ENGLISH_INDEX ) &&
					( LANGUAGE_ALLOWED( nIndex )) &&
					( pEntrys->GetText(
						sContent, STRING_TYP_TEXT, nIndex, TRUE )) &&
					( sContent != "-" ) && ( sContent.Len()))
				{
					ByteString sText = UTF8Converter::ConvertToUTF8(
						sContent, Export::GetCharSet( Export::LangId[ nIndex ]));
	
					ConvertStringToXMLFormat( sText );

					Export::QuotHTML( sText );

					ByteString sAdditionalLine( "\t<Text xml:lang=\"" );
					sAdditionalLine += GetIsoLangByIndex( nIndex );
					sAdditionalLine += "\">";
					sAdditionalLine += sText;
					sAdditionalLine += "</Text>\n\t";

					Output( sAdditionalLine );
				}
			}
		}
	}
	delete pResData;
	pResData = NULL;
}

//
// class XMLPropertyGenerator
//

/*****************************************************************************/
XMLPropertyGenerator::XMLPropertyGenerator(
	const ByteString &rOutputPath, const ByteString &rBaseName )
/*****************************************************************************/
				: sBaseName( rBaseName ),
				sOutputPath( rOutputPath )
{
	for ( ULONG i = 0; i < LANGUAGES; i++ )
		pOutputStream[ i ] = NULL;
}

/*****************************************************************************/
XMLPropertyGenerator::~XMLPropertyGenerator()
/*****************************************************************************/
{
	for ( ULONG i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[i ] && pOutputStream[ i ]->IsOpen())
			pOutputStream[ i ]->Close();
		delete pOutputStream[ i ];
	}
}

/*****************************************************************************/
void XMLPropertyGenerator::WorkOnText(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
	const ByteString &sKey,	USHORT nLang, ByteString &rText )
/*****************************************************************************/
{
	USHORT nIndex = Export::GetLangIndex( nLang );
	USHORT nHackIndex = nIndex;

	if ( getenv( "XML_LANG_HACK" )) {
		ByteString sHack( getenv( "XML_LANG_HACK" ));
		if ( sHack.ToUpperAscii() == "TRUE" ) {
			switch ( nIndex ) {
				case ENGLISH_INDEX:
					nHackIndex = ENGLISH_US_INDEX;
				break;
				case ENGLISH_US_INDEX:
					nHackIndex = ENGLISH_INDEX;
				break;
			}
		}
	}

	if ( nIndex ) { 
		if ( !pOutputStream[ nHackIndex ] ) {
			DirEntry aEntry( String( sOutputPath, RTL_TEXTENCODING_ASCII_US ));
			aEntry.MakeDir( TRUE );
			aEntry.MakeDir();
			for ( ULONG i = 0;
				i < rRessourceBundle.GetTokenCount( '.' ); i++ )
			{
				aEntry += DirEntry(
					String( rRessourceBundle.GetToken( i, '.' ),
						RTL_TEXTENCODING_ASCII_US ));
				aEntry.MakeDir();
			}

			ByteString sOutputName( sBaseName );
			sOutputName += "_";
			sOutputName += GetPropertyLangByIndex( nHackIndex );
			sOutputName += ".";
			sOutputName += "properties";

			aEntry += DirEntry( String( sOutputName, RTL_TEXTENCODING_ASCII_US ));

			pOutputStream[ nHackIndex ] =
				new SvFileStream( aEntry.GetFull(),
				 	STREAM_STD_WRITE | STREAM_TRUNC );

			if ( !pOutputStream[ nHackIndex ]->IsOpen()) {
				ByteString sError( "Error: Could not open file " );
				sError += ByteString( pOutputStream[ nIndex ]->
					GetFileName(), RTL_TEXTENCODING_ASCII_US );
				Error( sError );
			}
		}
		if ( pOutputStream[ nHackIndex ]->IsOpen()) {
			ByteString sIdentifier( rContext );
			sIdentifier += ".";
			sIdentifier += sKey;

			ByteString sText( rText );
			ReplaceArguments( sText );

			sIdentifier += "=";

			Export::UnquotHTML( sText );

			sIdentifier += sText;

			pOutputStream[ nHackIndex ]->WriteLine( sIdentifier );
		}
	}
}

/*****************************************************************************/
void XMLPropertyGenerator::EndOfRessourceBundle(
	const ByteString &rRessourceBundle )
/*****************************************************************************/
{
	ULONG i;
	ByteString sOutput;

	fprintf( stdout, "\nProperties generated:\n" );

	for ( i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[ i ] && pOutputStream[ i ]->IsOpen()) {
			pOutputStream[ i ]->Close();

			DirEntry aEntry( pOutputStream[ i ]->GetFileName());

			sOutput = ByteString( aEntry.GetName(), RTL_TEXTENCODING_ASCII_US );
			fprintf( stdout, "\t%s\n", sOutput.GetBuffer());
		}
	}

	String sSource;

	if ( pOutputStream[ ENGLISH_US_INDEX ] )
		sSource = pOutputStream[ ENGLISH_US_INDEX ]->GetFileName();
	else if ( pOutputStream[ ENGLISH_INDEX ] )
		sSource = pOutputStream[ ENGLISH_INDEX ]->GetFileName();

	if ( sSource.Len()) {
		fprintf( stdout, "\nFallback generated:\n" );

		String sBase = sSource.GetToken( 0, '_' );

		if ( sSource.GetTokenCount( '_' ) >= 2 ) {
			for ( ULONG i = 1; i < sSource.GetTokenCount( '_' ) - 1; i++ ) {
				sBase += String::CreateFromAscii( "_" );
				sBase += sSource.GetToken( i, '_' );
			}
		}

		String sDestination = sBase;
		
		sDestination += String::CreateFromAscii( ".properties" );

		DirEntry aSource( sSource );
		DirEntry aDestination( sDestination );

		aSource.CopyTo( aDestination, TRUE );

		sOutput = ByteString( aDestination.GetName(), RTL_TEXTENCODING_ASCII_US );
		fprintf( stdout, "\t%s\n", sOutput.GetBuffer());

		for ( i = 0; i < LANGUAGES; i++ ) {
			if ( i != COMMENT_INDEX && LANGUAGE_ALLOWED( i ) && !pOutputStream[ i ] ) {
				sDestination = sBase;
				sDestination += String::CreateFromAscii( "_" );
				sDestination += String( GetPropertyLangByIndex( i ), RTL_TEXTENCODING_ASCII_US );
				sDestination += String::CreateFromAscii( ".properties" );
				
				DirEntry aSource( sSource );
				DirEntry aDestination( sDestination );

				aSource.CopyTo( aDestination, TRUE );

				sOutput = ByteString( aDestination.GetName(), RTL_TEXTENCODING_ASCII_US );
				fprintf( stdout, "\t%s\n", sOutput.GetBuffer());
			}
		}
	}

	fprintf( stdout, "\n" );

	for ( i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[ i ] )
			delete pOutputStream[ i ];
		pOutputStream[ i ] = NULL;
	}
}

/*****************************************************************************/
void XMLPropertyGenerator::Argument( const ByteString &rArg )
/*****************************************************************************/
{
	aArgumentList.Insert( new ByteString( rArg ), LIST_APPEND );
}

/*****************************************************************************/
void XMLPropertyGenerator::ReplaceArguments( ByteString &rText )
/*****************************************************************************/
{
	while ( aArgumentList.Count()) {
		ByteString sArgument( *aArgumentList.GetObject(( ULONG ) 0 ));
		delete aArgumentList.Remove(( ULONG ) 0 );

		ByteString sExchange( "{" );
		if ( sArgument.Search( "\"" ) != STRING_NOTFOUND )
			sExchange += sArgument.GetToken( 1, '\"' );
		else
			sExchange += sArgument.GetToken( 1, '\'' );
		sExchange += "}";

		rText.SearchAndReplace( sArgument, sExchange );
	}
}

//
// class XML_XULDtdGenerator
//

/*****************************************************************************/
XML_XULDtdGenerator::XML_XULDtdGenerator(
	const ByteString &rOutputPath, const ByteString &rBaseName )
/*****************************************************************************/
				: sBaseName( rBaseName ),
				sOutputPath( rOutputPath )
{
	for ( ULONG i = 0; i < LANGUAGES; i++ )
		pOutputStream[ i ] = NULL;
}

/*****************************************************************************/
XML_XULDtdGenerator::~XML_XULDtdGenerator()
/*****************************************************************************/
{
	for ( ULONG i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[i ] && pOutputStream[ i ]->IsOpen())
			pOutputStream[ i ]->Close();
		delete pOutputStream[ i ];
	}
}

/*****************************************************************************/
void XML_XULDtdGenerator::WorkOnText(
	const ByteString &rRessourceBundle,	const ByteString &rContext,
	const ByteString &sKey,	USHORT nLang, ByteString &rText )
/*****************************************************************************/
{
	USHORT nIndex = Export::GetLangIndex( nLang );
	USHORT nHackIndex = nIndex;

	if ( getenv( "XML_LANG_HACK" )) {
		ByteString sHack( getenv( "XML_LANG_HACK" ));
		if ( sHack.ToUpperAscii() == "TRUE" ) {
			switch ( nIndex ) {
				case ENGLISH_INDEX:
					nHackIndex = ENGLISH_US_INDEX;
				break;
				case ENGLISH_US_INDEX:
					nHackIndex = ENGLISH_INDEX;
				break;
			}
		}
	}

	if ( nIndex ) { 
		if ( !pOutputStream[ nHackIndex ] ) {
			DirEntry aEntry( String( sOutputPath, RTL_TEXTENCODING_ASCII_US ));
			aEntry.MakeDir( TRUE );
			aEntry.MakeDir();
			aEntry += DirEntry( String( GetIsoLangByIndex( nHackIndex ), RTL_TEXTENCODING_ASCII_US ));
			aEntry.MakeDir();
			for ( ULONG i = 0;
				i < rRessourceBundle.GetTokenCount( '.' ); i++ )
			{
				aEntry += DirEntry(
					String( rRessourceBundle.GetToken( i, '.' ),
						RTL_TEXTENCODING_ASCII_US ));
				aEntry.MakeDir();
			}

			ByteString sOutputName( sBaseName );
			sOutputName += ".dtd";

			aEntry += DirEntry( String( sOutputName, RTL_TEXTENCODING_ASCII_US ));

			pOutputStream[ nHackIndex ] =
				new SvFileStream( aEntry.GetFull(),
				 	STREAM_STD_WRITE | STREAM_TRUNC );

			if ( !pOutputStream[ nHackIndex ]->IsOpen()) {
				ByteString sError( "Error: Could not open file " );
				sError += ByteString( pOutputStream[ nIndex ]->
					GetFileName(), RTL_TEXTENCODING_ASCII_US );
				Error( sError );
			}
		}
		if ( pOutputStream[ nHackIndex ]->IsOpen()) {
			ByteString sIdentifier = "<!ENTITY ";
			sIdentifier += rContext;
			sIdentifier += ".";
			sIdentifier += sKey;
			sIdentifier += " \"";

			ByteString sText( rText );
			ReplaceArguments( sText );

			sIdentifier += sText;
			sIdentifier += "\">";

			pOutputStream[ nHackIndex ]->WriteLine( sIdentifier );
		}
	}
}

/*****************************************************************************/
void XML_XULDtdGenerator::EndOfRessourceBundle(
	const ByteString &rRessourceBundle )
/*****************************************************************************/
{
	ULONG i;
	ByteString sOutput;

	fprintf( stdout, "\nDtd generated:\n" );

	for ( i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[ i ] && pOutputStream[ i ]->IsOpen()) {
			pOutputStream[ i ]->Close();

			DirEntry aEntry( pOutputStream[ i ]->GetFileName());

			sOutput = "..."; 
			sOutput += GetIsoLangByIndex( i );
			sOutput += "...";
			sOutput += ByteString( aEntry.GetName(), RTL_TEXTENCODING_ASCII_US );
			fprintf( stdout, "\t%s\n", sOutput.GetBuffer());
		}
	}

	String sSource;
	String sSearch;

	if ( pOutputStream[ ENGLISH_US_INDEX ] ) {
		sSource = pOutputStream[ ENGLISH_US_INDEX ]->GetFileName();
		sSearch = String( GetIsoLangByIndex( ENGLISH_US_INDEX ), RTL_TEXTENCODING_ASCII_US );
	}
	else if ( pOutputStream[ ENGLISH_INDEX ] ) {
		sSource = pOutputStream[ ENGLISH_INDEX ]->GetFileName();
		sSearch = String( GetIsoLangByIndex( ENGLISH_INDEX ), RTL_TEXTENCODING_ASCII_US );
	}

	if ( sSource.Len()) {
		fprintf( stdout, "\nFallback generated:\n" );

		String sDestinationBase = sSource;

		DirEntry aSource( sSource );

		for ( i = 0; i < LANGUAGES; i++ ) {
			if ( i != COMMENT_INDEX && LANGUAGE_ALLOWED( i ) && !pOutputStream[ i ] ) {

				String sReplace( GetIsoLangByIndex( i ), RTL_TEXTENCODING_ASCII_US );
				String sDestination = sDestinationBase;
				sDestination.SearchAndReplaceAll( sSearch, sReplace );

				DirEntry aDestination( sDestination );
				DirEntry aPath = aDestination.GetPath();
				aPath.MakeDir( TRUE );
				aPath.MakeDir();

				aSource.CopyTo( aDestination, TRUE );

				sOutput = "..."; 
				sOutput += GetIsoLangByIndex( i );
				sOutput += "...";
				sOutput += ByteString( aDestination.GetName(), RTL_TEXTENCODING_ASCII_US );
				fprintf( stdout, "\t%s\n", sOutput.GetBuffer());
			}
		}
	//this section generates a fallback in a en-DE directory, not XUL conform !!!
/*		String sReplace( String::CreateFromAscii( "en-DE" ));
		String sDestination = sDestinationBase;
		sDestination.SearchAndReplaceAll( sSearch, sReplace );

		DirEntry aDestination( sDestination );
		DirEntry aPath = aDestination.GetPath();
		aPath.MakeDir( TRUE );
		aPath.MakeDir();

		aSource.CopyTo( aDestination, TRUE );

		sOutput = "...en-DE..."; 
		sOutput += ByteString( aDestination.GetName(), RTL_TEXTENCODING_ASCII_US );
		fprintf( stdout, "\t%s\n", sOutput.GetBuffer());
*/
	} 

	fprintf( stdout, "\n" );

	for ( i = 0; i < LANGUAGES; i++ ) {
		if ( pOutputStream[ i ] )
			delete pOutputStream[ i ];
		pOutputStream[ i ] = NULL;
	}
}

/*****************************************************************************/
void XML_XULDtdGenerator::Argument( const ByteString &rArg )
/*****************************************************************************/
{
	aArgumentList.Insert( new ByteString( rArg ), LIST_APPEND );
}

/*****************************************************************************/
void XML_XULDtdGenerator::ReplaceArguments( ByteString &rText )
/*****************************************************************************/
{
	while ( aArgumentList.Count()) {
		ByteString sArgument( *aArgumentList.GetObject(( ULONG ) 0 ));
		delete aArgumentList.Remove(( ULONG ) 0 );

		ByteString sExchange( "{" );
		if ( sArgument.Search( "\"" ) != STRING_NOTFOUND )
			sExchange += sArgument.GetToken( 1, '\"' );
		else
			sExchange += sArgument.GetToken( 1, '\'' );
		sExchange += "}";

		rText.SearchAndReplace( sArgument, sExchange );
	}
}

