 /*************************************************************************
 *
 *  $RCSfile: rapofflinefilestream.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: sb $ $Date: 2001/10/01 14:53:46 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef INCLUDED_UCB_RAPOFFLINEFILESTREAM_HXX
#include "rapofflinefilestream.hxx"
#endif

#ifndef _COM_SUN_STAR_IO_IOEXCEPTION_HPP_
#include "com/sun/star/io/IOException.hpp"
#endif
#ifndef _COM_SUN_STAR_IO_XINPUTSTREAM_HPP_
#include "com/sun/star/io/XInputStream.hpp"
#endif
#ifndef _COM_SUN_STAR_IO_XMARKABLESTREAM_HPP_
#include "com/sun/star/io/XMarkableStream.hpp"
#endif
#ifndef _COM_SUN_STAR_IO_XOUTPUTSTREAM_HPP_
#include "com/sun/star/io/XOutputStream.hpp"
#endif
#ifndef _COM_SUN_STAR_IO_XSEEKABLE_HPP_
#include "com/sun/star/io/XSeekable.hpp"
#endif
#ifndef _COM_SUN_STAR_LANG_ILLEGALARGUMENTEXCEPTION_HPP_
#include "com/sun/star/lang/IllegalArgumentException.hpp"
#endif
#ifndef _COM_SUN_STAR_LANG_XTYPEPROVIDER_HPP_
#include "com/sun/star/lang/XTypeProvider.hpp"
#endif
#ifndef _COM_SUN_STAR_UNO_ANY_HXX_
#include "com/sun/star/uno/Any.hxx"
#endif
#ifndef _COM_SUN_STAR_UNO_REFERENCE_HXX_
#include "com/sun/star/uno/Reference.hxx"
#endif
#ifndef _COM_SUN_STAR_UNO_RUNTIMEEXCEPTION_HPP_
#include "com/sun/star/uno/RuntimeException.hpp"
#endif
#ifndef _COM_SUN_STAR_UNO_SEQUENCE_HXX_
#include "com/sun/star/uno/Sequence.hxx"
#endif
#ifndef _COM_SUN_STAR_UNO_TYPE_HXX_
#include "com/sun/star/uno/Type.hxx"
#endif
#ifndef _CPPUHELPER_QUERYINTERFACE_HXX_
#include "cppuhelper/queryinterface.hxx"
#endif
#ifndef _RTL_USTRING_H_
#include "rtl/ustring.h"
#endif
#ifndef _RTL_USTRING_HXX_
#include "rtl/ustring.hxx"
#endif
#ifndef _RTL_STRING_H_
#include "rtl/string.h"
#endif
#ifndef _OSL_DIAGNOSE_H_
#include "osl/diagnose.h"
#endif
#ifndef _OSL_FILE_HXX_
#include "osl/file.hxx"
#endif
#ifndef _OSL_MUTEX_HXX_
#include "osl/mutex.hxx"
#endif
#ifndef _SAL_TYPES_H_
#include "sal/types.h"
#endif

#include <algorithm>

using namespace com::sun;
using ucprmt::offline::FileStream;

sal_uInt64 FileStream::getPos() SAL_THROW((star::io::IOException))
{
    sal_uInt64 nPos;
    osl::FileBase::RC eError = m_aFile.getPos(nPos);
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::getPos():"
                                  " getPos error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);

    return nPos;
}

void FileStream::setPos(sal_uInt64 nPos) SAL_THROW((star::io::IOException))
{
    osl::FileBase::RC eError = m_aFile.setPos(Pos_Absolut, nPos);
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::setPos():"
                                  " setPos error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);
}

sal_uInt64 FileStream::getSize(sal_uInt64 * pPos)
    SAL_THROW((star::io::IOException))
{
    //TODO! Is there no easy way to get the file size?

    sal_uInt64 nPos = getPos();

    osl::FileBase::RC eError = m_aFile.setPos(Pos_End, 0);
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::getSize():"
                                  " setPos error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);

    sal_uInt64 nSize;
    eError = m_aFile.getPos(nSize);
    osl::FileBase::RC eError2 = m_aFile.setPos(Pos_Absolut, nPos);
    if (eError2 != osl::FileBase::E_None)
        close(true, true, false);
            // if the position cannot be reset, it seems best to stop all
            // further activities...
    if (eError == osl::FileBase::E_None)
        eError = eError2;
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::getSize():"
                                  " setPos error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);

    if (pPos)
        *pPos = nPos;
    return nSize;
}

FileStream::MarkIdentifiers::iterator FileStream::findMark(sal_Int32 nMark)
    SAL_THROW((star::lang::IllegalArgumentException))
{
    if (!m_xMarkPositions.get())
        throw star::lang::IllegalArgumentException(
                  rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                                    "ucprmt::offline::FileStream::findMark():"
                                        " bad mark")),
                  *this,
                  0);

    MarkIdentifiers::iterator aIt(m_xMarkIdentifiers->find(nMark));
    if (aIt == m_xMarkIdentifiers->end())
        throw star::lang::IllegalArgumentException(
                  rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                                    "ucprmt::offline::FileStream::findMark():"
                                        " bad mark")),
                  *this,
                  0);

    return aIt;
}

void FileStream::close(bool bInput, bool bOutput, bool bCheckErrors)
    SAL_THROW((star::io::IOException))
{
    // Close the file only if the status indeed drops from 'at lest input or
    // output open' to 'neither input nor output open:'
    if (m_bInputOpen && bInput && !(m_bOutputOpen && !bOutput)
        || m_bOutputOpen && bOutput && !(m_bInputOpen && !bInput))
    {
        osl::FileBase::RC eError = m_aFile.close();
        if (bCheckErrors && eError != osl::FileBase::E_None)
            throw star::io::IOException(
                      rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::close():"
                                  " close error #"))
                          + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                       eError)),
                      *this);
    }
    m_bInputOpen = m_bInputOpen && !bInput;
    m_bOutputOpen = m_bOutputOpen && !bOutput;
}

FileStream::FileStream(rtl::OUString const & rSystemPath,
                       sal_uInt32 nMode,
                       bool bVolatile):
    m_aFile(rSystemPath),
    m_nNextMarkIdentifier(0),
    m_bInputStream((nMode & OpenFlag_Read) != 0),
    m_bOutputStream((nMode & OpenFlag_Write) != 0),
    m_bMarkableStream(!bVolatile)
{
    OSL_ASSERT(m_bInputStream || m_bOutputStream);
    m_eOpenError = m_aFile.open(nMode);
    m_bInputOpen = m_bInputStream && m_eOpenError == osl::FileBase::E_None;
    m_bOutputOpen = m_bOutputStream && m_eOpenError == osl::FileBase::E_None;
}

FileStream::~FileStream()
{}

star::uno::Any SAL_CALL
FileStream::queryInterface(star::uno::Type const & rType)
    throw (star::uno::RuntimeException)
{
    star::uno::Any
        aRet(cppu::queryInterface(
                 rType,
                 static_cast< star::lang::XTypeProvider * >(this),
                 static_cast< star::io::XSeekable * >(this)));
    if (aRet.hasValue())
        return aRet;
    else if (rType
                 == getCppuType(
                        static_cast<
                            star::uno::Reference<
                                star::io::XInputStream > const * >(0)))
        if (m_bInputStream)
        {
            star::io::XInputStream const * p = this;
            return star::uno::Any(&p, rType);
        }
        else
            return star::uno::Any();
    else if (rType
                 == getCppuType(
                        static_cast<
                            star::uno::Reference<
                                star::io::XOutputStream > const * >(0)))
        if (m_bOutputStream)
        {
            star::io::XOutputStream const * p = this;
            return star::uno::Any(&p, rType);
        }
        else
            return star::uno::Any();
    else if (rType
                 == getCppuType(
                        static_cast<
                            star::uno::Reference<
                                star::io::XMarkableStream > const * >(0)))
        if (m_bMarkableStream)
        {
            star::io::XMarkableStream const * p = this;
            return star::uno::Any(&p, rType);
        }
        else
            return star::uno::Any();
    else
    {
        // MSVC++ needs it this way:
        typedef cppu::WeakImplHelper4< com::sun::star::io::XInputStream,
                                       com::sun::star::io::XOutputStream,
                                       com::sun::star::io::XSeekable,
                                       com::sun::star::io::XMarkableStream >
            Parent;
        return Parent::queryInterface(rType);
    }
}

sal_Int32 SAL_CALL
FileStream::readBytes(star::uno::Sequence< sal_Int8 > & rData,
                      sal_Int32 nBytesToRead)
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bInputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::readBytes():"
                              " not XInputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!m_bInputOpen)
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::readBytes():"
                              " not open")),
                  *this);

    // For efficiency reasons, always use rData as buffer (but callers of this
    // method have to be aware that rData can be modified even if this method
    // does not return successfully):
    if (nBytesToRead < 0)
        nBytesToRead = 0;
    if (nBytesToRead > rData.getLength())
        rData.realloc(nBytesToRead);
    sal_uInt64 nBytesRead;
    osl::FileBase::RC eError
        = m_aFile.read(rData.getArray(), nBytesToRead, nBytesRead);
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::readBytes():"
                                  " read error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);

    OSL_ASSERT(nBytesRead <= nBytesToRead);
    return static_cast< sal_Int32 >(nBytesRead);
        //TODO! rData.realloc(static_cast< sal_Int32 >(nBytesRead)) ?
}

sal_Int32 SAL_CALL
FileStream::readSomeBytes(star::uno::Sequence< sal_Int8 > & rData,
                          sal_Int32 nMaxBytesToRead)
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    return readBytes(rData, nMaxBytesToRead);
        //TODO! The semantics of osl::File::read() are unclear to me.  Maybe
        // it would be better to have a looping read() function that tries to
        // read nBytesToRead bytes (in the readBytes() case) or at least
        // std::min(1, nMaxBytesToRead) bytes (in the readSomeBytes() case).
        // What about the various errors returned by osl::File::read() then?
        // How to detect EOF?
}

void SAL_CALL FileStream::skipBytes(sal_Int32 nBytesToSkip)
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bInputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::skipBytes():"
                              " not XInputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!m_bInputOpen)
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::skipBytes():"
                              " not open")),
                  *this);

    osl::FileBase::RC eError
        = m_aFile.setPos(Pos_Current, sal_uInt64(nBytesToSkip));
        //TODO! the second argument to osl::File::setPos() would better be of
        // type sal_Int64 instead of sal_uInt64 in this case (negative
        // sal_Int32 values are converted to large sal_uInt64 values without
        // causing loss of information or undefined behaviour, but does
        // osl::File::setPos() interpret these as negative values?)...
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::skipBytes():"
                                  " setPos error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);
}

sal_Int32 SAL_CALL FileStream::available()
    throw (star::io::NotConnectedException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bInputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::available():"
                              " not XInputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!m_bInputOpen)
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::available():"
                              " not open")),
                  *this);

    sal_uInt64 nPos;
    sal_uInt64 nSize = getSize(&nPos);
    return static_cast< sal_Int32 >(
               std::min(static_cast< sal_uInt64 >(nSize >= nPos ?
                                                      nSize - nPos : 0),
                        static_cast< sal_uInt64 >(0x7FFFFFFF)));
}

void SAL_CALL FileStream::closeInput()
    throw (star::io::NotConnectedException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bInputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::closeInput():"
                              " not XInputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    close(true, false, true);
}

void SAL_CALL
FileStream::writeBytes(star::uno::Sequence< sal_Int8 > const & rData)
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bOutputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::writeBytes():"
                              " not XOutputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!m_bOutputOpen)
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::writeBytes():"
                              " not open")),
                  *this);

    sal_uInt64 nBytesToWrite = sal_uInt64(rData.getLength());
    sal_uInt64 nBytesWritten;
    osl::FileBase::RC eError = m_aFile.write(rData.getConstArray(),
                                             nBytesToWrite,
                                             nBytesWritten);
    if (eError != osl::FileBase::E_None)
        throw star::io::IOException(
                  rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::writeBytes():"
                                  " write error #"))
                      + rtl::OUString::valueOf(static_cast< sal_Int32 >(
                                                   eError)),
                  *this);
    if (nBytesWritten != nBytesToWrite)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::writeBytes():"
                              " wrote too little")),
                  *this);
}

void SAL_CALL FileStream::flush()
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bOutputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::flush():"
                              " not XOutputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!m_bOutputOpen)
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::flush(): not open")),
                  *this);

    //TODO! Does the osl::File implementation not support the concept of
    // flushing?
}

void SAL_CALL FileStream::closeOutput()
    throw (star::io::NotConnectedException,
           star::io::BufferSizeExceededException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    if (!m_bOutputStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::closeOutput():"
                              " not XOutputStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    close(false, true, true);
}

void SAL_CALL FileStream::seek(sal_Int64 nLocation)
    throw (star::lang::IllegalArgumentException,
           star::io::IOException,
           star::uno::RuntimeException)
{
    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::seek(): not open")),
                  *this);

    if (nLocation < 0)
        throw star::lang::IllegalArgumentException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::seek():"
                              " negative location")),
                  *this,
                  0);

    //TODO! throw IllegalArgumentException if nLocation is past end of stream

    setPos(sal_uInt64(nLocation));
}

sal_Int64 SAL_CALL FileStream::getPosition()
    throw (star::io::IOException, star::uno::RuntimeException)
{
    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::getPosition():"
                              " not open")),
                  *this);

    sal_uInt64 nPos = getPos();
    if (nPos > sal_uInt64(0x80000000) * sal_uInt64(0xFFFFFFFF)
                   + sal_uInt64(0x7FFFFFFF)) // nPos > 0x7FFFFFFFFFFFFFFF
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::getPosition():"
                              " too large")),
                  *this);
    return sal_Int64(nPos);
}

sal_Int64 SAL_CALL FileStream::getLength()
    throw (star::io::IOException, star::uno::RuntimeException)
{
    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::getLength():"
                              " not open")),
                  *this);

    sal_uInt64 nSize = getSize(0);
    if (nSize > sal_uInt64(0x80000000) * sal_uInt64(0xFFFFFFFF)
                    + sal_uInt64(0x7FFFFFFF)) // nSize > 0x7FFFFFFFFFFFFFFF
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::getLength():"
                              " too large")),
                  *this);
    return sal_Int64(nSize);
}

sal_Int32 SAL_CALL FileStream::createMark()
    throw (star::io::IOException, star::uno::RuntimeException)
{
    if (!m_bMarkableStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::createMark():"
                              " not XMarkableStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::createMark():"
                              " not open")),
                  *this);

    if (!m_xMarkPositions.get())
    {
        m_xMarkPositions.reset(new MarkPositions);
        m_xMarkIdentifiers.reset(new MarkIdentifiers);
    }

    // Find an unused identifier (TODO! looking at the XMarkableStream
    // documentation, is it ok to recycle mark identifiers?):
    sal_Int32 nIdentifier;
    MarkIdentifiers::iterator aEnd(m_xMarkIdentifiers->end());
    for (sal_Int32 nLoop = m_nNextMarkIdentifier;;)
    {
        nIdentifier = m_nNextMarkIdentifier;
        bool bFree = m_xMarkIdentifiers->find(nIdentifier) == aEnd;
        if (m_nNextMarkIdentifier == 0x7FFFFFFF)
            m_nNextMarkIdentifier = static_cast< sal_Int32 >(-0x7FFFFFFF) - 1;
        else
            ++m_nNextMarkIdentifier;
            // work around undefined behaviour of signed integer overflow...
        if (bFree)
            break;
        if (m_nNextMarkIdentifier == nLoop)
            throw star::io::IOException(
                      rtl::OUString(
                          RTL_CONSTASCII_USTRINGPARAM(
                              "ucprmt::offline::FileStream::createMark():"
                                  " too many marks")),
                      *this);
    }

    (*m_xMarkIdentifiers)[nIdentifier]
        = m_xMarkPositions->insert(MarkPositions::value_type(getPos(),
                                                             nIdentifier));
    return nIdentifier;
}

void SAL_CALL FileStream::deleteMark(sal_Int32 nMark)
    throw (star::io::IOException,
           star::lang::IllegalArgumentException,
           star::uno::RuntimeException)
{
    if (!m_bMarkableStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::deleteMark():"
                              " not XMarkableStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::deleteMark():"
                              " not open")),
                  *this);

    MarkIdentifiers::iterator aIdIt(findMark(nMark));
    sal_uInt64 nPos = aIdIt->second->first;

    // Remove all marks after this one:
    MarkPositions::iterator aPosIt(m_xMarkPositions->upper_bound(nPos));
    MarkPositions::iterator aPosEnd(m_xMarkPositions->end());
    for (MarkPositions::iterator aIt(aPosIt); aIt != aPosEnd; ++aPosIt)
        m_xMarkIdentifiers->erase(aIt->second);
    m_xMarkPositions->erase(aPosIt, aPosEnd);

    // Remove this mark:
    m_xMarkPositions->erase(aIdIt->second);
    m_xMarkIdentifiers->erase(aIdIt);
}

void SAL_CALL FileStream::jumpToMark(sal_Int32 nMark)
    throw (star::io::IOException,
           star::lang::IllegalArgumentException,
           star::uno::RuntimeException)
{
    if (!m_bMarkableStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::jumpToMark():"
                              " not XMarkableStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::jumpToMark():"
                              " not open")),
                  *this);

    setPos(findMark(nMark)->second->first);
}

void SAL_CALL FileStream::jumpToFurthest()
    throw (star::io::IOException, star::uno::RuntimeException)
{
    if (!m_bMarkableStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::jumpToFurthest():"
                              " not XMarkableStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::jumpToFurthest():"
                              " not open")),
                  *this);

    //TODO! What is the semantics of this method?
}

sal_Int32 SAL_CALL FileStream::offsetToMark(sal_Int32 nMark)
    throw (star::io::IOException,
           star::lang::IllegalArgumentException,
           star::uno::RuntimeException)
{
    if (!m_bMarkableStream)
        throw star::uno::RuntimeException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::offsetToMark():"
                              " not XMarkableStream")),
                  *this);

    osl::MutexGuard aGuard(m_aMutex);

    if (!(m_bInputOpen || m_bOutputOpen))
        throw star::io::IOException(
                  rtl::OUString(
                      RTL_CONSTASCII_USTRINGPARAM(
                          "ucprmt::offline::FileStream::offsetToMark():"
                              " not open")),
                  *this);

    // Calculate nCurrentPos - nMarkPos:
    sal_uInt64 nMarkPos = findMark(nMark)->second->first;
    sal_uInt64 nCurrentPos = getPos();
    if (nCurrentPos >= nMarkPos)
    {
        sal_uInt64 nOffset = nCurrentPos - nMarkPos;
        if (nOffset > 0x7FFFFFFF)
            throw star::io::IOException(
                rtl::OUString(
                    RTL_CONSTASCII_USTRINGPARAM(
                        "ucprmt::offline::FileStream::offsetToMark():"
                            " too large")),
                *this);
        return static_cast< sal_Int32 >(nOffset);
    }
    else
    {
        sal_uInt64 nNegOffset = nMarkPos - nCurrentPos;
        if (nNegOffset > 0x80000000)
            throw star::io::IOException(
                rtl::OUString(
                    RTL_CONSTASCII_USTRINGPARAM(
                        "ucprmt::offline::FileStream::offsetToMark():"
                            " too large")),
                *this);
        return nNegOffset == 0x80000000 ?
                   static_cast< sal_Int32 >(-0x7FFFFFFF) - 1 :
                   -static_cast< sal_Int32 >(nNegOffset);
    }
}
