/*************************************************************************
 *
 *  $RCSfile: devaudiosound.cxx,v $
 *
 *  $Revision: 1.3.2.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/10/31 20:27:44 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include <salsound.hxx>
#include <salimpsound.hxx>

#include <svapp.hxx>

#ifdef USE_DEVAUDIO

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>
#include <errno.h>

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif

#ifdef _USE_NAMESPACE
using namespace vcl_sal;
using namespace vos;
#endif

int									DevAudioSound::s_nDevice = -1;
DevAudioDataList					DevAudioSound::s_aDataList;
OMutex								DevAudioSound::s_aProtector;
BOOL								DevAudioSound::s_bPlaying = FALSE;
DevAudioSoundList					DevAudioSound::s_aSounds;

static const char* getAudioDevName()
{
    static const char* pDev = NULL;
    if( ! pDev )
    {
        pDev = getenv( "AUDIODEV" );
        if( ! pDev || ! *pDev )
            pDev = "/dev/audio";
    }
    return pDev;
}

BOOL DevAudioSound::open()
{
	OGuard aGuard( s_aProtector );
	if( s_nDevice == -1 )
	{
		s_nDevice = ::open( getAudioDevName(), O_WRONLY | O_NDELAY );
		if( s_nDevice == -1 )
		{
			SalDbgAssert( "ERROR: could not open %s, errno=%d\n", getAudioDevName(), errno );
			return FALSE;
		}
#ifdef DEBUG
		audio_device_t aDevice;
		if( ! ioctl( s_nDevice, AUDIO_GETDEV, &aDevice ) )
		  SalDbgAssert( "\nopened %s of type:\nName: %s\nVersion: %s\nConfig: %s\n\n",
			   getAudioDevName(), aDevice.name, aDevice.version, aDevice.config );
#endif
	}
	else
	{
		// sadly there is no documented reset control for /dev/audio
		// ioctl( s_nDevice, SNDCTL_DSP_RESET, 0 );
	}
	return TRUE;
}

void DevAudioSound::close()
{
	OGuard aGuard( s_aProtector );
	if( s_nDevice != -1 )
	{
	  ioctl( s_nDevice, AUDIO_DRAIN, NULL );
		::close( s_nDevice );
	}
	s_nDevice = -1;
}

BOOL DevAudioSound::startSound( DevAudioData* pData )
{
	if( pData && pData->m_pSound && pData->m_pSound->m_pBuffer &&
		s_nDevice >= 0 )
	{
		if( ! strncmp( pData->m_pSound->m_pBuffer, ".snd", 4 ) )
			return startAU( pData );
		else if( ! strncmp( pData->m_pSound->m_pBuffer, "RIFF", 4 ) )
			return startRIFF( pData );
	}
	return FALSE;
}

BOOL DevAudioSound::startAU( DevAudioData* pData )
{
	char* pBuffer = pData->m_pSound->m_pBuffer;
	ULONG nDataOffset	= readBELong( pBuffer + 4 );
	ULONG nDataLen		= readBELong( pBuffer + 8 );
	ULONG nEncoding		= readBELong( pBuffer + 12 );
	ULONG nSampleRate	= readBELong( pBuffer + 16 );
	ULONG nChannels		= readBELong( pBuffer + 20 );

	if( nDataLen == ~0 )
		nDataLen = pData->m_pSound->m_aStat.st_size - nDataOffset;

	pData->m_nDataLen	= nDataLen;
	pData->m_nStartPos	= nDataOffset;
	pData->m_nEndPos	= pData->m_nStartPos + pData->m_nDataLen;

	if( nChannels != 1 && nChannels != 2 )
	{
		SalDbgAssert( "%d Channels are not supported\n" );
		return FALSE;
	}

	audio_info_t aAudioInfo;
	AUDIO_INITINFO( &aAudioInfo );

	aAudioInfo.play.gain 			= 127;
	aAudioInfo.play.channels		= nChannels;
	aAudioInfo.play.sample_rate		= nSampleRate;
	switch( nEncoding )
	{
		case 1:
			aAudioInfo.play.encoding = AUDIO_ENCODING_ULAW;
			break;
		case 2:
			aAudioInfo.play.precision	= 8;
			aAudioInfo.play.encoding	= AUDIO_ENCODING_LINEAR;
			break;
		case 3:
			aAudioInfo.play.precision	= 16;
			aAudioInfo.play.encoding	= AUDIO_ENCODING_LINEAR;
			break;
		case 23:
			aAudioInfo.play.encoding	= AUDIO_ENCODING_ULAW;
			pData->m_pConverter = AudioConverter::createG721ToULawConverter();
			break;
		default:
			SalDbgAssert( "au format %d unsupported\n", nEncoding );
			return FALSE;
	};

	if( ioctl( s_nDevice, AUDIO_SETINFO, &aAudioInfo ) )
	{
		SalDbgAssert( "ERROR: ioctl AUDIO_SETINFO failed\n" );
		return FALSE;
	}

	SalDbgAssert( "playing %d data bytes at %d bytes/s in format %d on %d channels \n", pData->m_nDataLen, nSampleRate, nEncoding, nChannels );
	return TRUE;
}

BOOL DevAudioSound::startRIFF( DevAudioData* pData )
{
	int nPos = findChunk( pData, "fmt " );
	if( nPos == -1 )
		return FALSE;
	
	int nFormat		= readLEShort( pData->m_pSound->m_pBuffer + nPos + 8 );
	int nChannels	= readLEShort( pData->m_pSound->m_pBuffer + nPos + 10 );
	int nSampleRate	= readLEInt( pData->m_pSound->m_pBuffer + nPos + 12 );
	int nByteRate	= readLEInt( pData->m_pSound->m_pBuffer + nPos + 16 );
	int nAlign		= readLEShort( pData->m_pSound->m_pBuffer + nPos + 20 );
	SalDbgAssert( "format is tag = %x, channels = %d, samplesPerSec = %d, avgBytesPerSec = %d, blockAlign = %d\n", nFormat, nChannels, nSampleRate, nByteRate, nAlign );
	if( nChannels != 1 && nChannels != 2 )
	{
		SalDbgAssert( "%d Channels are not supported\n" );
		return FALSE;
	}
	
	int nBitsPerSample = 0;
	switch( nFormat )
	{
		default:
			SalDbgAssert( "unknown format\n" );
			return FALSE;
			break;
		case 1:
			nBitsPerSample = readLEShort( pData->m_pSound->m_pBuffer + nPos + 22 );
			break;
	}
	
	nPos = findChunk( pData, "data" );
	if( nPos == -1 )
	{
		SalDbgAssert( "ERROR: no \"data\" chunk found\n" );
		return FALSE;
	}
	
	pData->m_nDataLen = readLEInt( pData->m_pSound->m_pBuffer + nPos + 4 );
	pData->m_nStartPos = nPos+8;
	pData->m_nEndPos = pData->m_nStartPos + pData->m_nDataLen;
	
	audio_info_t aAudioInfo;
	AUDIO_INITINFO( &aAudioInfo );

	aAudioInfo.play.gain 			= 127;
	aAudioInfo.play.channels		= nChannels;
	aAudioInfo.play.sample_rate		= nSampleRate;
	switch( nBitsPerSample )
	{
		case 8:
#if defined SOLARIS && defined SPARC
			aAudioInfo.play.precision	= 16;
			aAudioInfo.play.encoding	= AUDIO_ENCODING_LINEAR;
			pData->m_pConverter = AudioConverter::createINT8ToUINT16Converter();
#else
			aAudioInfo.play.precision	= 8;
			aAudioInfo.play.encoding	= AUDIO_ENCODING_LINEAR;
#endif
			break;
		case 16:
			aAudioInfo.play.precision	= 16;
			aAudioInfo.play.encoding	= AUDIO_ENCODING_LINEAR;
#ifdef SPARC
			pData->m_pConverter = AudioConverter::createSwapINT16Converter();
#endif
			break;
		default:
			SalDbgAssert( "%d bits per sample is not usable\n", nBitsPerSample );
			return FALSE;
	};

	if( ioctl( s_nDevice, AUDIO_SETINFO, &aAudioInfo ) )
	{
		SalDbgAssert( "ERROR: ioctl AUDIO_SETINFO failed\n" );
		return FALSE;
	}
	
	SalDbgAssert( "playing %d data bytes at %d bytes/s in %d bits quality on %d channels \n", pData->m_nDataLen, nSampleRate, nBitsPerSample, nChannels );
	return TRUE;
}

int DevAudioSound::findChunk( DevAudioData* pData, char* pChunk )
{
	if( !pData || ! pData->m_pSound || ! pData->m_pSound->m_pBuffer )
		return -1;

	int nPos = 12;
	while( nPos < pData->m_pSound->m_aStat.st_size )
	{
		if( ! strncmp( pData->m_pSound->m_pBuffer+nPos, pChunk, 4 ) )
			return nPos;

		// get LSB int length
		nPos += readLEInt( pData->m_pSound->m_pBuffer+nPos+4 ) + 8;
	}
	return -1;
}

short int DevAudioSound::readLEShort( char* pBuf )
{
	unsigned char* pBuffer = (unsigned char*)pBuf;
	short int nRet = *pBuffer++;
	nRet |= *pBuffer << 8;

	return nRet;
}

ULONG DevAudioSound::readBELong( char* pBuf )
{
	unsigned char* pBuffer = (unsigned char*)pBuf;
	ULONG nRet	 = *pBuffer++ << 24;
	nRet		|= *pBuffer++ << 16;
	nRet		|= *pBuffer++ << 8;
	nRet		|= *pBuffer;

	return nRet;
}

int DevAudioSound::readLEInt( char* pBuf )
{
	unsigned char* pBuffer = (unsigned char*)pBuf;
	int nRet = *pBuffer++;
	nRet |= (int)(*pBuffer++) << 8;
	nRet |= (int)(*pBuffer++) << 16;
	nRet |= (int)(*pBuffer++) << 24;

	return nRet;
}

void DevAudioSound::run()
{
	SalDbgAssert( "DevAudioWorker::run\n" );
	s_bPlaying = TRUE;
	DevAudioData* pData;
    if( open() )
    {
        do
        {
            {
                OGuard aGuard( s_aProtector );
                pData = s_aDataList.Count() ? s_aDataList.GetObject( 0 ) : NULL;
            }
            if( ! pData )
                break;
            
            if( startSound( pData ) )
            {
                pData->m_nCurPos = pData->m_nStartPos;
                while( pData->m_nCurPos < pData->m_nEndPos )
                {
                    int nBytes = 4096;
                    
                    char* pBuffer = pData->m_pSound->m_pBuffer;
                    
                    if( pData->m_nCurPos + nBytes > pData->m_nEndPos )
                        nBytes = pData->m_nEndPos - pData->m_nCurPos;
                    
                    if( pData->m_pConverter )
                    {
                        int nBytesOut;
                        char* pConvBuffer = pData->m_pConverter->convert( pBuffer + pData->m_nCurPos, nBytes, nBytesOut );
                        write( s_nDevice, pConvBuffer, nBytesOut );
                        delete pConvBuffer;
                    }
                    else
                    {
                        nBytes = write( s_nDevice,
                                        pBuffer + pData->m_nCurPos, nBytes );
                    }
                    pData->m_nCurPos += nBytes;
                    usleep(50000);
                    {
                        OGuard aGuard( s_aProtector );
                        if( s_aDataList.GetObject( 0 ) != pData )
                        {
				  // 						ioctl( s_nDevice, SNDCTL_DSP_RESET, 0);
                            break;
                        }
                    }
                }
                if( pData->m_pSound->m_pSalSound->m_bLoop )
                    pData->m_pSound->play();
                else
                {
                    // /dev/audio does not seem to have a mechanism to request
                    // wether the device is actually playing. Therefore
                    // we need to synchronzie before signaling sound stop
                    ioctl( s_nDevice, AUDIO_DRAIN, NULL );
                    Application::PostUserEvent( LINK( pData->m_pSound, DevAudioSound, notifyStopHdl ), NULL );
                }
            }
            else
				Application::PostUserEvent( LINK( pData->m_pSound, DevAudioSound, notifyErrorHdl ), (void*)SOUNDERR_INVALID_FILE );

            {
                OGuard aGuard( s_aProtector );
                s_aDataList.Remove( pData );
                delete pData;
            }
        } while( pData );
        close();
    }
	s_bPlaying = FALSE;
}

void DevAudioSound::append( DevAudioSound* pSound )
{
	OGuard aGuard( s_aProtector );
	s_aDataList.Insert( new DevAudioData( pSound ), LIST_APPEND );
	if( ! s_bPlaying )
		pSound->create();
}

void DevAudioSound::remove( DevAudioSound* pSound )
{
	OGuard aGuard( s_aProtector );
	DevAudioData* pData;
	for( int i = s_aDataList.Count()-1; i >= 0; i-- )
	{
		pData = s_aDataList.GetObject( i );
		if( pData->m_pSound == pSound )
		{
			s_aDataList.Remove( i );
			if( i > 0 )
				delete pData;
		}
	}
}

DevAudioSound::DevAudioSound( ::SalSound* pSound ) :
		VSound( pSound )
{
	initBuffer();
	s_aSounds.Insert( this );
}

DevAudioSound::~DevAudioSound()
{
	stop();
	join();
	if( m_pBuffer )
		releaseBuffer();
	s_aSounds.Remove( this );
}

BOOL DevAudioSound::isValid()
{
	if( ! m_pBuffer )
		return FALSE;

    if( s_nDevice == -1 )
    {
        int nTryOpen = ::open( getAudioDevName(), O_WRONLY | O_NDELAY );
        if( nTryOpen == -1 )
            return FALSE;
        ::close( nTryOpen );
    }
	
	// check for valid file format
	if( ! strncmp( "RIFF", m_pBuffer, 4 ) )
	{
		DevAudioData aData( this );

		// check for data chunk
		if( findChunk( &aData, "data" ) == -1 )
			goto failed;
		int nPos = findChunk( &aData, "fmt " );
		if( nPos == -1 )
			goto failed;
		
		int nFormat		= readLEShort( m_pBuffer + nPos + 8 );
		int nChannels	= readLEShort( m_pBuffer + nPos + 10 );
		// check channels
		if( nChannels != 1 && nChannels != 2 )
			goto failed;
		// check formats
		// playable is MS-PCM only at now
		if( nFormat != 1 )
			goto failed;
		return TRUE;
	}
	else if( ! strncmp( ".snd", m_pBuffer, 4 ) )
	{
		ULONG nEncoding		= readBELong( m_pBuffer + 12 );
		ULONG nChannels		= readBELong( m_pBuffer + 20 );

		// check for playable encodings
		if( nEncoding != 1 && nEncoding != 2 && nEncoding != 3 && nEncoding != 23 )
			goto failed;
		// check channels
		if( nChannels != 1 && nChannels != 2 )
			goto failed;
		return TRUE;
	}
  failed:
	releaseBuffer();
	return FALSE;
}

void DevAudioSound::play()
{
	if( ! m_pBuffer )
		return;

	append( this );
	m_pSalSound->m_bPlaying	= TRUE;
	m_pSalSound->m_bPaused	= FALSE;
}

void DevAudioSound::stop()
{
	remove( this );
	m_pSalSound->m_bPlaying	= FALSE;
	m_pSalSound->m_bPaused	= FALSE;
}

void DevAudioSound::pause()
{
	remove( this );
	m_pSalSound->m_bPaused = TRUE;
}

void DevAudioSound::cont()
{
	play();
}

IMPL_LINK( DevAudioSound, notifyStopHdl, void*, pVoid )
{
	if( s_aSounds.GetPos( this ) != LIST_ENTRY_NOTFOUND )
	{
		m_pSalSound->changeStateStop();
	}
	return 0;
}

IMPL_LINK( DevAudioSound, notifyErrorHdl, void*, nError )
{
	if( s_aSounds.GetPos( this ) != LIST_ENTRY_NOTFOUND )
	{
		m_pSalSound->setError( (ULONG)nError );
	}
	return 0;
}

#endif
