/*************************************************************************
 *
 *  $RCSfile: reactor.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 15:18:17 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


#include <sal/config.h>

#include <osl/socket.h>

#include <vos/socket.hxx>
#include <vos/mutex.hxx>
#include <vos/thread.hxx>
#include <vos/diagnose.hxx>
#include <vos/macros.hxx>
#include <vos/reactor.hxx>

#include <list>

#ifdef _USE_NAMESPACE
using namespace vos;
#endif

/*========================================================================
 *
 * OReactor internals.
 *
 *======================================================================*/
typedef NAMESPACE_STD(list<OReactorCallbackContext*>) callback_list;

/*========================================================================
 *
 * OReactor implementation.
 *
 *======================================================================*/
VOS_IMPLEMENT_CLASSINFO(
	VOS_CLASSNAME(OReactor, vos),
	VOS_NAMESPACE(OReactor, vos),
	VOS_NAMESPACE(OThread, vos),
	0);

/*
 * OReactor.
 */
OReactor::OReactor (void)
	: OThread      (),
	  m_xSocket    (NULL),
	  m_SocketAddr (rtl::OUString::createFromAscii("127.0.0.1"),0L)
{
	m_xSocket = new NAMESPACE_VOS(ODatagramSocket);
	m_xSocket->enableNonBlockingMode();
	
	m_xSocket->setReuseAddr(1);
	if (!m_xSocket->bind(m_SocketAddr)) /* bind (INADDR_LOOPBACK) */
	{
		OSocket::TSocketError eErrCode = m_xSocket->getError();
		if (eErrCode == OSocket::E_AddrNotAvail)
		{
			// Observed under OS/2 (but not understood).
			m_SocketAddr.setAddr(rtl::OUString::createFromAscii("0.0.0.0"));
			m_xSocket->bind(m_SocketAddr); /* bind (INADDR_ANY) */
		}
	}
	m_xSocket->getLocalAddr(m_SocketAddr);

	registerCallback(m_xSocket.getBodyPtr(), REACTOR_EVENT_READ, NULL, NULL);

	m_Lock.acquire();
	create();
	yield();
	m_Lock.release();
}

/*
 * ~OReactor.
 */
OReactor::~OReactor (void)
{
	kill();

	deregisterCallback(m_xSocket.getBodyPtr(), 0);
	m_xSocket = NULL;

	callback_list::iterator it;

	for (it = m_Incoming.begin(); it != m_Incoming.end(); ++it)
		delete (*it);
	m_Incoming.clear();

	for (it = m_Outgoing.begin(); it != m_Outgoing.end(); ++it)
		delete (*it);
	m_Outgoing.clear();

	for (it = m_OutOfBand.begin(); it != m_OutOfBand.end(); ++it)
		delete (*it);
	m_OutOfBand.clear();
}

/*
 * registerCallback.
 */
void OReactor::registerCallback(
	OSocketRef xSocket, sal_uInt32 nEventMask,
	OReactorCallback *pfnCallback, void *pData)
{
	if (nEventMask == 0)
	{
		deregisterCallback(xSocket, nEventMask);
		return;
	}

	m_Lock.acquire();

	sal_Bool changed = sal_False;
	callback_list::iterator it;

	if (nEventMask & REACTOR_EVENT_READ)
	{
		for (it = m_Incoming.begin(); it != m_Incoming.end(); it++)
			if ((*it)->m_xSocket == xSocket) break;

		if (it == m_Incoming.end())
		{
			OReactorCallbackContext *pCtx = new OReactorCallbackContext;

			pCtx->m_xSocket     = xSocket;
			pCtx->m_nEventMask  = nEventMask;
			pCtx->m_pfnCallback = pfnCallback;
			pCtx->m_pData       = pData;

			m_Incoming.insert (m_Incoming.end(), pCtx);
			changed = sal_True;
		}
	}

	if (nEventMask & REACTOR_EVENT_WRITE)
	{
		for (it = m_Outgoing.begin(); it != m_Outgoing.end(); it++)
			if ((*it)->m_xSocket == xSocket) break;

		if (it == m_Outgoing.end())
		{
			OReactorCallbackContext *pCtx = new OReactorCallbackContext;

			pCtx->m_xSocket     = xSocket;
			pCtx->m_nEventMask  = nEventMask;
			pCtx->m_pfnCallback = pfnCallback;
			pCtx->m_pData       = pData;

			m_Outgoing.insert (m_Outgoing.end(), pCtx);
			changed = sal_True;
		}
	}

	if (nEventMask & REACTOR_EVENT_OOB)
	{
		for (it = m_OutOfBand.begin(); it != m_OutOfBand.end(); it++)
			if ((*it)->m_xSocket == xSocket) break;

		if (it == m_OutOfBand.end())
		{
			OReactorCallbackContext *pCtx = new OReactorCallbackContext;

			pCtx->m_xSocket     = xSocket;
			pCtx->m_nEventMask  = nEventMask;
			pCtx->m_pfnCallback = pfnCallback;
			pCtx->m_pData       = pData;

			m_OutOfBand.insert (m_OutOfBand.end(), pCtx);
			changed = sal_True;
		}
	}

	if (changed) unblock();
	m_Lock.release();
}

/*
 * deregisterCallback.
 */
void OReactor::deregisterCallback (
	OSocketRef xSocket, sal_uInt32 nEventMask)
{
	if (nEventMask == 0) nEventMask = (sal_uInt32)(~0);

	m_Lock.acquire();
	sal_Bool changed = sal_False;

	if (nEventMask & REACTOR_EVENT_READ)
	{
		callback_list::iterator first = m_Incoming.begin();
		callback_list::iterator last  = m_Incoming.end();

		while (first != last)
		{
			callback_list::iterator next = first;
			++next;

			OReactorCallbackContext *pCtx = (*first);
			if (pCtx->m_xSocket == xSocket)
			{
				m_Incoming.erase (first);
				delete pCtx;
				changed = sal_True;
			}
			first = next;
		}
	}

	if (nEventMask & REACTOR_EVENT_WRITE)
	{
		callback_list::iterator first = m_Outgoing.begin();
		callback_list::iterator last  = m_Outgoing.end();

		while (first != last)
		{
			callback_list::iterator next = first;
			++next;

			OReactorCallbackContext *pCtx = (*first);
			if (pCtx->m_xSocket == xSocket)
			{
				m_Outgoing.erase (first);
				delete pCtx;
				changed = sal_True;
			}
			first = next;
		}
	}

	if (nEventMask & REACTOR_EVENT_OOB)
	{
		callback_list::iterator first = m_OutOfBand.begin();
		callback_list::iterator last  = m_OutOfBand.end();

		while (first != last)
		{
			callback_list::iterator next = first;
			++next;

			OReactorCallbackContext *pCtx = (*first);
			if (pCtx->m_xSocket == xSocket)
			{
				m_OutOfBand.erase (first);
				delete pCtx;
				changed = sal_True;
			}
			first = next;
		}
	}

	if (changed) unblock();
	m_Lock.release();
}

/*
 * unblock.
 */
void OReactor::unblock (void)
{
	VOS_ASSERT (m_xSocket.isValid());
	if (m_xSocket.isValid())
	{
		// Return from blocking select().
		if (m_xSocket->sendTo(m_SocketAddr, "B", 1))
		{
			m_xSocket->getError();
        }
		yield();
	}
}

/*
 * terminate.
 */
void OReactor::terminate (void)
{
	m_Lock.acquire();
	unblock();
	OThread::terminate();
	m_Lock.release();
}

/*
 * kill.
 */
void OReactor::kill (void)
{
	terminate();
	join();
}

/*
 * run.
 */
void OReactor::run (void)
{
	oslSocketSet IncomingSet  = osl_createSocketSet();
	oslSocketSet OutgoingSet  = osl_createSocketSet();
	oslSocketSet OutOfBandSet = osl_createSocketSet();

	VOS_ASSERT (IncomingSet);
	VOS_ASSERT (OutgoingSet);
	VOS_ASSERT (OutOfBandSet);

	if ((IncomingSet  == (oslSocketSet)NULL) ||
		(OutgoingSet  == (oslSocketSet)NULL) ||
		(OutOfBandSet == (oslSocketSet)NULL)    )
	{
		osl_destroySocketSet (IncomingSet);
		osl_destroySocketSet (OutgoingSet);
		osl_destroySocketSet (OutOfBandSet);
		return;
	}

	while (schedule())
	{
		// Initialize.
		osl_clearSocketSet (IncomingSet);
		osl_clearSocketSet (OutgoingSet);
		osl_clearSocketSet (OutOfBandSet);

		callback_list::iterator first;
		callback_list::iterator last;
		OReactorCallbackContext *pCtx = NULL;

		// Process waiting sockets.
		m_Lock.acquire();

		// Incoming.
		first = m_Incoming.begin();
		last  = m_Incoming.end();

		while (first != last)
		{
			pCtx = (*first);
			if (pCtx->m_xSocket->isValid())
			{
				oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
				osl_addToSocketSet (IncomingSet, socket);
			}
			first++;
		}

		// Outgoing.
		first = m_Outgoing.begin();
		last  = m_Outgoing.end();

		while (first != last)
		{
			pCtx = (*first);
			if (pCtx->m_xSocket->isValid())
			{
				oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
				osl_addToSocketSet (OutgoingSet, socket);
			}
			first++;
		}

		// OutOfBand.
		first = m_OutOfBand.begin();
		last  = m_OutOfBand.end();

		while (first != last)
		{
			pCtx = (*first);
			if (pCtx->m_xSocket->isValid())
			{
				oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
				osl_addToSocketSet (OutOfBandSet, socket);
			}
			first++;
		}

		// Done.
		m_Lock.release();

		// Enter blocking select().
		sal_Int32 nReady = osl_demultiplexSocketEvents (
			IncomingSet, OutgoingSet, OutOfBandSet, NULL);

		if (nReady <= 0)
		{
			// Failure.
            oslSocketError eErrCode = osl_getLastSocketError(NULL);
			continue;
		}

		// Process ready sockets.
		m_Lock.acquire();

		// Incoming.
		first = m_Incoming.begin();
		last  = m_Incoming.end();

		while ((nReady > 0) && (first != last))
		{
			callback_list::iterator next = first;
			++next;

			pCtx = (*first);
			oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
			if (osl_isInSocketSet (IncomingSet, socket))
			{
				nReady--;
				if (socket == (oslSocket)*m_xSocket)
				{
					NAMESPACE_VOS(OSocketAddr) aFromAddr;
					sal_Char c = '\0';

					while (m_xSocket->recvFrom (&c, 1, &aFromAddr) > 0)
                        c = '\0';
				}
				else
				{
					m_Incoming.erase (first);

					if (pCtx->m_pfnCallback)
						(pCtx->m_pfnCallback) (pCtx->m_xSocket,
											   REACTOR_EVENT_READ,
											   pCtx->m_pData);
					delete pCtx;
				}
			}
			first = next;
		}

		// Outgoing.
		first = m_Outgoing.begin();
		last  = m_Outgoing.end();

		while ((nReady > 0) && (first != last))
		{
			callback_list::iterator next = first;
			++next;

			pCtx = (*first);
			oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
			if (osl_isInSocketSet (OutgoingSet, socket))
			{
				nReady--;
				m_Outgoing.erase (first);

				if (pCtx->m_pfnCallback)
					(pCtx->m_pfnCallback) (pCtx->m_xSocket,
										   REACTOR_EVENT_WRITE,
										   pCtx->m_pData);
				delete pCtx;
			}
			first = next;
		}

		// OutOfBand.
		first = m_OutOfBand.begin();
		last  = m_OutOfBand.end();

		while ((nReady > 0) && (first != last))
		{
			callback_list::iterator next = first;
			++next;

			pCtx = (*first);
			oslSocket socket = (oslSocket)(*(pCtx->m_xSocket));
			if (osl_isInSocketSet (OutOfBandSet, socket))
			{
				nReady--;
				m_OutOfBand.erase (first);

				if (pCtx->m_pfnCallback)
					(pCtx->m_pfnCallback) (pCtx->m_xSocket,
										   REACTOR_EVENT_OOB,
										   pCtx->m_pData);
				delete pCtx;
			}
			first = next;
		}

		// Done.
		m_Lock.release();
	}

	osl_destroySocketSet (IncomingSet);
	osl_destroySocketSet (OutgoingSet);
	osl_destroySocketSet (OutOfBandSet);
}

