/*************************************************************************
 *
 *  $RCSfile: proxy.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: hr $ $Date: 2000/10/25 10:43:01 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _SAL_TYPES_H_
#include <sal/types.h>
#endif
#ifndef _OSL_CONDITN_HXX_
#include <osl/conditn.hxx>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif
#ifndef _RTL_ALLOC_H_
#include <rtl/alloc.h>
#endif

#ifndef _VOS_EXECABL_HXX_
#include <vos/execabl.hxx>
#endif
#ifndef _VOS_PROCESS_HXX_
#include <vos/process.hxx>
#endif
#ifndef _VOS_SOCKET_HXX_
#include <vos/socket.hxx>
#endif
#ifndef _VOS_THRDSRV_HXX_
#include <vos/thrdsrv.hxx>
#endif

#ifndef _PROTOCOL_HXX_
#include "protocol.hxx"
#endif
#ifndef _UTILITY_HXX_
#include "utility.hxx"
#endif

#include <stdio.h>

using namespace com::sun::star::proxy;
using namespace osl;
using namespace rtl;

typedef NAMESPACE_VOS(OThreadingServer) executor_type;

/*========================================================================
 *
 * OProxyChannel.
 * (com::sun::star::proxy)
 *
 *======================================================================*/
class OProxyChannel : public NAMESPACE_VOS(OExecutable)
{
	typedef NAMESPACE_VOS(OStreamSocket) socket_type;

public:
	/** Parameter.
	 */
	struct parameter
	{
		/** Representation.
		 */
		sal_uInt32               m_nBandwidth;
		sal_uInt32               m_nLatency;
		sal_Int32                m_nTcpNoDelay;
		IProtocolHandlerFactory *m_pFactory;

		/** Construction.
		 */
		parameter (void)
			: m_nBandwidth  (0),
			  m_nLatency    (0),
			  m_nTcpNoDelay (0),
			  m_pFactory    (NULL)
		{}

		/** Operation.
		 */
		sal_Bool createInstance (
			const OString     &rName,
			IProtocolHandler *&rpHandler)
		{
			if (m_pFactory)
				rpHandler = m_pFactory->createInstance (rName);
			else
				rpHandler = NULL;
			return (!!rpHandler);
		}
		
	};

	/** Construction.
	 */
	OProxyChannel (
		const OString     &rName,
		const socket_type &rSrc,
		const socket_type &rDst,
		const parameter   &rParams);

	/** IExecutable.
	 */
	virtual sal_Bool SAL_CALL execute (void);
	virtual sal_Bool SAL_CALL unblock (void);

protected:
	/** Destruction.
	 */
	virtual ~OProxyChannel (void);

private:
	/** Representation.
	 */
	OString           m_aName;
	socket_type       m_aSrc;
	socket_type       m_aDst;
	parameter         m_aParams;

	Condition         m_aCondition;
	IProtocolHandler *m_pHandler;
	sal_uInt32        m_nData;
	sal_uInt32        m_nSize;

	/** Implementation.
	 */
	void analyzeInit   (void);
	void analyzeUpdate (const sal_uInt8 *pData, sal_uInt32 nSize);
	void analyzeFinal  (void);

	OTimeValue waitInit  (void);
	void       waitFinal (const OTimeValue& rInit, sal_uInt32 nBytes);

	sal_Bool send (
		const sal_uInt8 *pData, sal_uInt32 nSize);

	/** Not implemented.
	 */
	OProxyChannel (const OProxyChannel&);
	OProxyChannel& operator= (const OProxyChannel&);
};

/*
 * OProxyChannel.
 */
OProxyChannel::OProxyChannel (
	const OString     &rName,
	const socket_type &rSrc,
	const socket_type &rDst,
	const parameter   &rParams)
	: m_aName    (rName),
	  m_aSrc     (rSrc),
	  m_aDst     (rDst),
	  m_aParams  (rParams),
	  m_pHandler (NULL),
	  m_nData    (0),
	  m_nSize    (0)
{
	// Disable Nagle algorithm.
	m_aDst.setTcpNoDelay (m_aParams.m_nTcpNoDelay);

	// Create ProtocolHandler instance.
	m_aParams.createInstance (m_aName, m_pHandler);
}

/*
 * ~OProxyChannel.
 */
OProxyChannel::~OProxyChannel (void)
{
	delete m_pHandler;
}

/*
 * analyzeInit.
 */
void OProxyChannel::analyzeInit (void)
{
	if (m_pHandler)
		m_pHandler->init();
}

/*
 * analyzeUpdate.
 */
void OProxyChannel::analyzeUpdate (const sal_uInt8 *pData, sal_uInt32 nSize)
{
	if (m_pHandler)
		m_pHandler->update (pData, nSize);
}

/*
 * analyzeFinal.
 */
void OProxyChannel::analyzeFinal (void)
{
	if (m_pHandler)
		m_pHandler->final();
}

/*
 * waitInit.
 */
OTimeValue OProxyChannel::waitInit (void)
{
	if (m_aParams.m_nLatency)
	{
		OTimeValue tv (0, m_aParams.m_nLatency * 1000000);
		m_aCondition.wait (&tv);
	}
	return OTimeValue::getSystemTime();
}

/*
 * waitFinal.
 */
void OProxyChannel::waitFinal (const OTimeValue& rInit, sal_uInt32 nBytes)
{
	if (m_aParams.m_nBandwidth)
	{
		double nMillisec = double(nBytes) / double(m_aParams.m_nBandwidth);
		OTimeValue tv (0, nMillisec * 1000000);
		tv += rInit;

		OTimeValue now (OTimeValue::getSystemTime());
		if (tv > now)
		{
			tv -= now;
			m_aCondition.wait (&tv);
		}
	}
}

/*
 * send.
 */
sal_Bool OProxyChannel::send (const sal_uInt8 *pData, sal_uInt32 nSize)
{
	OTimeValue tv = waitInit();
	if (m_aDst.send (pData, nSize) > 0)
	{
		waitFinal (tv, nSize);

		m_nData += 1;
		m_nSize += nSize;

		fprintf (stdout, "%s (%d, %d): %d\n",
				 m_aName.pData->buffer, m_nData, nSize, m_nSize);
		fflush (stdout);
	}
	else if (m_aDst.isValid())
	{
		m_aDst.shutdown();
		osl_closeSocket (m_aDst);
		m_aDst.close();
	}
	return m_aDst.isValid();
}

/*
 * execute.
 */
sal_Bool SAL_CALL OProxyChannel::execute (void)
{

#define PROXYBUFSIZE 256*1024

	sal_uInt8 *pBuffer;
	sal_Int32 nRecv = 0;

	pBuffer = (sal_uInt8 *)rtl_allocateMemory(PROXYBUFSIZE);
	VOS_ASSERT(pBuffer);

	analyzeInit();
	while ((nRecv = m_aSrc.recv (pBuffer, PROXYBUFSIZE)) > 0)
	{
		analyzeUpdate (pBuffer, nRecv);
		if (!send (pBuffer, nRecv))
		{
			m_aSrc.shutdown();
			break;
		}
	}
	analyzeFinal();

	if (m_aSrc.isValid())
	{
		osl_closeSocket (m_aSrc);
		m_aSrc.close();
	}
	if (m_aDst.isValid())
	{
		osl_closeSocket (m_aDst);
		m_aDst.close();
	}
	rtl_freeMemory(pBuffer);
	return sal_False;
}

/*
 * unblock.
 */
sal_Bool SAL_CALL OProxyChannel::unblock (void)
{
	if (m_aSrc.isValid())
	{
		osl_closeSocket (m_aSrc);
		m_aSrc.close();
	}
	return sal_True;
}

/*========================================================================
 *
 * OProxyConnector.
 * (com::sun::star::proxy)
 *
 *======================================================================*/
class OProxyConnector : public NAMESPACE_VOS(OExecutable)
{
public:
	/** Parameter.
	 */
	struct parameter
	{
		/** Representation.
		 */
		NAMESPACE_VOS(OInetSocketAddr) m_aMyAddr;
		NAMESPACE_VOS(OInetSocketAddr) m_aToAddr;
		OProxyChannel::parameter       m_aChannelParams;

		/** Construction.
		 */
		parameter (void)
			: m_aMyAddr (OUString::createFromAscii("0.0.0.0")  , 9000),
			  m_aToAddr (OUString::createFromAscii("127.0.0.1"), 8125)
		{}

		/** Operation.
		 */
		void setMyAddr (const OUString &rHost)
		{
			m_aMyAddr.setAddr (rHost);
		}
		void setMyPort (sal_Int32 nPort)
		{
			m_aMyAddr.setPort (nPort);
		}

		void setToAddr (const OUString &rHost)
		{
			m_aToAddr.setAddr (rHost);
		}
		void setToPort (sal_Int32 nPort)
		{
			m_aToAddr.setPort (nPort);
		}

		void setBandwidth (sal_uInt32 nBandwidth)
		{
			m_aChannelParams.m_nBandwidth = nBandwidth;
		}
		void setLatency (sal_uInt32 nLatency)
		{
			m_aChannelParams.m_nLatency = nLatency;
		}
		void setTcpNoDelay (sal_Int32 nTcpNoDelay)
		{
			m_aChannelParams.m_nTcpNoDelay = ((nTcpNoDelay > 0) ? 1 : 0);
		}
		void setProtocol (const OUString &rProtocol)
		{
			OProtocolHandlerRegistry::queryFactory (
				rProtocol, m_aChannelParams.m_pFactory);
		}
	};

	/** Construction.
	 */
	OProxyConnector (
		executor_type   &rExecutor,
		const parameter &rParams);

	/** IExecutable.
	 */
	virtual sal_Bool SAL_CALL execute (void);
	virtual sal_Bool SAL_CALL unblock (void);

protected:
	/** Destruction.
	 */
	virtual ~OProxyConnector (void);

private:
	/** Representation.
	 */
	executor_type                 &m_rExecutor;
	parameter                      m_aParams;
	NAMESPACE_VOS(OAcceptorSocket) m_aSocket;

	/** inet_ntoa.
	 */
	OString inet_ntoa (
		const NAMESPACE_VOS(OInetSocketAddr) &rAddr) const;

	/** Not implemented.
	 */
	OProxyConnector (const OProxyConnector&);
	OProxyConnector& operator= (const OProxyConnector&);
};

/*
 * OProxyConnector.
 */
OProxyConnector::OProxyConnector (
	executor_type   &rExecutor,
	const parameter &rParams)
	: m_rExecutor (rExecutor),
	  m_aParams   (rParams)
{
	m_aSocket.setReuseAddr(1);
	if (m_aSocket.bind (m_aParams.m_aMyAddr) && m_aSocket.listen())
	{
		// Listening.
		m_aSocket.getLocalAddr (m_aParams.m_aMyAddr);

		OString aMyAddr (inet_ntoa (m_aParams.m_aMyAddr));
		OString aToAddr (inet_ntoa (m_aParams.m_aToAddr));
		
		fprintf (
			stderr, "Accepting connections at [%s:%d] for [%s:%d] ...\n",
			aMyAddr.pData->buffer, m_aParams.m_aMyAddr.getPort(),
			aToAddr.pData->buffer, m_aParams.m_aToAddr.getPort());
		fflush (stderr);
	}
	else
	{
		// Failure.
		m_aSocket.close();
	}
}

/*
 * ~OProxyConnector.
 */
OProxyConnector::~OProxyConnector (void)
{
}

/*
 * execute.
 */
sal_Bool SAL_CALL OProxyConnector::execute (void)
{
	NAMESPACE_VOS(OStreamSocket)   aSource;
	NAMESPACE_VOS(OInetSocketAddr) aFrom;

	while (m_aSocket.acceptConnection (aSource, aFrom) == osl_Socket_Ok)
	{
		NAMESPACE_VOS(OConnectorSocket) aTarget;

		OString aFromAddr (inet_ntoa (aFrom));
		fprintf (
			stderr, "Connect from [%s:%d] ...",
			aFromAddr.pData->buffer, aFrom.getPort());
		fflush (stderr);

		if (aTarget.connect (m_aParams.m_aToAddr) == osl_Socket_Ok)
		{
			fprintf (stderr, " done.\n");
			m_rExecutor.add (new OProxyChannel (
				"C->S", aSource, aTarget, m_aParams.m_aChannelParams));
			m_rExecutor.add (new OProxyChannel (
				"S->C", aTarget, aSource, m_aParams.m_aChannelParams));
		}
		else
		{
			fprintf (stderr, " failed.\n");
			aSource.shutdown();
			aSource.close();
		}
		fflush (stderr);
	}
	return sal_False;
}

/*
 * unblock.
 */
sal_Bool SAL_CALL OProxyConnector::unblock (void)
{
	m_aSocket.close();
	return sal_True;
}

/*
 * inet_ntoa.
 */
OString OProxyConnector::inet_ntoa (
	const NAMESPACE_VOS(OInetSocketAddr) &rAddr) const
{
	OUString aDottedAddr;
	rAddr.getDottedAddr (aDottedAddr);
	return OString (
		aDottedAddr.pData->buffer,
		aDottedAddr.pData->length,
		RTL_TEXTENCODING_ASCII_US);
}

/*========================================================================
 *
 * Command Line Options.
 *
 *======================================================================*/
static void getOptVal (
	const OUString &rArg,
	OUString       &rOpt,
	OUString       &rVal,
	sal_Unicode     ch)
{
	sal_Int32 k = rArg.indexOf (ch);
	if (k < 0)
	{
		rOpt = rArg;
		rVal = OUString();
	}
	else
	{
		rOpt = rArg.copy (0, k);
		rVal = rArg.copy (k + 1);
	}
}

/*========================================================================
 *
 * usage.
 *
 *======================================================================*/
inline void usage_intro (const sal_Char *option)
{
	printf ("\nUsage: proxy %s\n\n", option);
}

static void usage_help (void)
{
	usage_intro ("-help[=<option>]");
	printf ("\tDisplays [option specific] help text and then exits.\n");
}

static void usage_local (void)
{
	usage_intro ("-local=[host][:port]");
	printf ("  Specifies the local endpoint "
			"for accepting incoming connections.\n");
	printf ("  The default 0.0.0.0:9000 "
			"accepts connections on port 9000 from any interface.\n");
	printf ("  Specify host as 127.0.0.1 "
			"to accept local connections only.\n");
}

static void usage_remote (void)
{
	usage_intro ("-remote=[host][:port]");
	printf ("  Specifies the remote endpoint "
			"for establishing outgoing connections.\n");
	printf ("  The default 127.0.0.1:8125 "
			"establishes a local connection to port 8125.\n");
}

static void usage_bandwidth (void)
{
	usage_intro ("-bandwidth=<number>");
	printf ("  Reduces the bandwidth to <number> kilobytes per second.\n");
	printf ("  Default is 0, i.e. no reduction in bandwidth.\n");
}

static void usage_latency (void)
{
	usage_intro ("-latency=<number>");
	printf ("  Delays received packets "
			"by <number> milliseconds before sending.\n");
	printf ("  Default is 0, i.e. no additional latency.\n");
}

static void usage_tcpnodelay (void)
{
	usage_intro ("-tcpnodelay[=<1|0>]");
	printf ("  This socket option disables (1) "
			"or enables (0) the Nagle algorithm.\n\n");
	printf ("  The Nagle algorithm reduces the number "
			"of small packets sent by a host\n");
	printf ("  by buffering send data until a full-size "
			"packet can be sent.\n");
}

static void usage (void)
{
	usage_intro ("[options]");
	printf ("Options:\n");
	printf ("  -help[=<option>]       display this help text and exit\n");
	printf ("  -local=[host][:port]   local endpoint  [0.0.0.0:9000]\n");
	printf ("  -remote=[host][:port]  remote endpoint [127.0.0.1:8125]\n");
	printf ("  -bandwidth=<number>    approximate bandwidth in [kb/s]\n");
	printf ("  -latency=<number>      additional latency in [ms]\n");
	printf ("  -tcpnodelay[=<1|0>]    disable the Nagle algorithm\n");
}

/*========================================================================
 *
 * main.
 *
 *======================================================================*/
int SAL_CALL main (int, char**)
{
	// Parameters.
	OProxyConnector::parameter aParams;

	// Command line arguments.
	NAMESPACE_VOS(OStartupInfo) aStartup;
	sal_Int32 i, n = aStartup.getCommandArgCount();
	for (i = 0; i < n; i++)
	{
		OUString aArg;
		aStartup.getCommandArg (i, aArg);
		if (aArg.toChar() == '-')
		{
			OUString aOpt, aVal;
			getOptVal (aArg, aOpt, aVal, '=');

			// General.
			if (aOpt.compareToAscii("-h", 2) == 0)
			{
				// Usage.
				if (aVal.compareToAscii("help") == 0)
					usage_help();
				else if (aVal.compareToAscii("local") == 0)
					usage_local();
				else if (aVal.compareToAscii("remote") == 0)
					usage_remote();
				else if (aVal.compareToAscii("bandwidth") == 0)
					usage_bandwidth();
				else if (aVal.compareToAscii("latency") == 0)
					usage_latency();
				else if (aVal.compareToAscii("tcpnodelay") == 0)
					usage_tcpnodelay();
				else
					usage();
				exit(0);
			}

			// OProxyConnector.
			if (aOpt.compareToAscii("-local") == 0)
			{
				OUString aHost, aPort;
				getOptVal (aVal, aHost, aPort, ':');
				if (aHost.getLength() > 0)
				{
					// Override MyAddr.
					aParams.setMyAddr (aHost);
				}

				sal_Int32 nPort = aPort.toInt32();
				if (nPort > 0)
				{
					// Override MyPort.
					aParams.setMyPort (nPort);
				}
				continue;
			}
			if (aOpt.compareToAscii("-remote") == 0)
			{
				OUString aHost, aPort;
				getOptVal (aVal, aHost, aPort, ':');
				if (aHost.getLength() > 0)
				{
					// Override ToAddr.
					aParams.setToAddr (aHost);
				}

				sal_Int32 nPort = aPort.toInt32();
				if (nPort > 0)
				{
					// Override ToPort.
					aParams.setToPort (nPort);
				}
				continue;
			}

			// OProxyChannel.
			if (aOpt.compareToAscii("-bandwidth") == 0)
			{
				// Bandwidth [kb/s].
				aParams.setBandwidth (aVal.toInt32());
				continue;
			}
			if (aOpt.compareToAscii("-latency") == 0)
			{
				// Latency [ms].
				aParams.setLatency (aVal.toInt32());
				continue;
			}
			if (aOpt.compareToAscii("-tcpnodelay") == 0)
			{
				// TcpNoDelay.
				if (aVal.getLength())
					aParams.setTcpNoDelay (aVal.toInt32());
				else
					aParams.setTcpNoDelay (1);
				continue;
			}
			if (aOpt.compareToAscii("-proto") == 0)
			{
				// Protocol (raw | uno | iiop).
				aParams.setProtocol (aVal);
				continue;
			}
		}
	}

	// Execute.
	NAMESPACE_VOS(ORef)<executor_type> xExecutor (new executor_type());
	xExecutor->add (new OProxyConnector (*xExecutor, aParams));
	xExecutor->complete();

	// Done.
	return 0;
}

