#  Licensed under the Apache License, Version 2.0 (the "License"); you may
#  not use this file except in compliance with the License. You may obtain
#  a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#  License for the specific language governing permissions and limitations
#  under the License.

from oslo_log import versionutils
from oslo_policy import policy

from neutron.conf.policies import base


COLLECTION_PATH = '/trunks'
RESOURCE_PATH = '/trunks/{id}'

DEPRECATED_REASON = (
    "The trunks API now supports system scope and default roles.")


rules = [
    policy.DocumentedRuleDefault(
        name='create_trunk',
        check_str=base.PROJECT_MEMBER,
        scope_types=['project'],
        description='Create a trunk',
        operations=[
            {
                'method': 'POST',
                'path': COLLECTION_PATH,
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='create_trunk',
            check_str=base.RULE_ANY,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='get_trunk',
        check_str=base.PROJECT_READER,
        scope_types=['project'],
        description='Get a trunk',
        operations=[
            {
                'method': 'GET',
                'path': COLLECTION_PATH,
            },
            {
                'method': 'GET',
                'path': RESOURCE_PATH,
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='get_trunk',
            check_str=base.RULE_ADMIN_OR_OWNER,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='update_trunk',
        check_str=base.PROJECT_MEMBER,
        scope_types=['project'],
        description='Update a trunk',
        operations=[
            {
                'method': 'PUT',
                'path': RESOURCE_PATH,
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='update_trunk',
            check_str=base.RULE_ADMIN_OR_OWNER,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='delete_trunk',
        check_str=base.PROJECT_MEMBER,
        scope_types=['project'],
        description='Delete a trunk',
        operations=[
            {
                'method': 'DELETE',
                'path': RESOURCE_PATH,
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='delete_trunk',
            check_str=base.RULE_ADMIN_OR_OWNER,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='get_subports',
        check_str=base.PROJECT_READER,
        scope_types=['project'],
        description='List subports attached to a trunk',
        operations=[
            {
                'method': 'GET',
                'path': '/trunks/{id}/get_subports',
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='get_subports',
            check_str=base.RULE_ANY,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='add_subports',
        check_str=base.PROJECT_MEMBER,
        scope_types=['project'],
        description='Add subports to a trunk',
        operations=[
            {
                'method': 'PUT',
                'path': '/trunks/{id}/add_subports',
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='add_subports',
            check_str=base.RULE_ADMIN_OR_OWNER,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
    policy.DocumentedRuleDefault(
        name='remove_subports',
        check_str=base.PROJECT_MEMBER,
        scope_types=['project'],
        description='Delete subports from a trunk',
        operations=[
            {
                'method': 'PUT',
                'path': '/trunks/{id}/remove_subports',
            },
        ],
        deprecated_rule=policy.DeprecatedRule(
            name='remove_subports',
            check_str=base.RULE_ADMIN_OR_OWNER,
            deprecated_reason=DEPRECATED_REASON,
            deprecated_since=versionutils.deprecated.WALLABY)
    ),
]


def list_rules():
    return rules
