# This file is dual licensed under the terms of the Apache License, Version
# 2.0, and the BSD License. See the LICENSE file in the root of this repository
# for complete details.
from __future__ import absolute_import, division, print_function

import pytest

from packaging.utils import canonicalize_name, canonicalize_version


@pytest.mark.parametrize(
    ("name", "expected"),
    [
        ("foo", "foo"),
        ("Foo", "foo"),
        ("fOo", "foo"),
        ("foo.bar", "foo-bar"),
        ("Foo.Bar", "foo-bar"),
        ("Foo.....Bar", "foo-bar"),
        ("foo_bar", "foo-bar"),
        ("foo___bar", "foo-bar"),
        ("foo-bar", "foo-bar"),
        ("foo----bar", "foo-bar"),
    ],
)
def test_canonicalize_name(name, expected):
    assert canonicalize_name(name) == expected


@pytest.mark.parametrize(
    ("version", "expected"),
    [
        ('1.4.0', '1.4'),
        ('1.40.0', '1.40'),
        ('1.4.0.0.00.000.0000', '1.4'),
        ('1.0', '1'),
        ('1.0+abc', '1+abc'),
        ('1.0.dev0', '1.dev0'),
        ('1.0.post0', '1.post0'),
        ('1.0a0', '1a0'),
        ('1.0rc0', '1rc0'),
        ('100!0.0', '100!0'),
        ('1.0.1-test7', '1.0.1-test7'),  # LegacyVersion is unchanged
    ]
)
def test_canonicalize_version(version, expected):
    assert canonicalize_version(version) == expected
