# (c) 2018, NetApp Inc.
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
from __future__ import absolute_import, division, print_function

__metaclass__ = type

from ansible.modules.storage.netapp.netapp_e_facts import Facts
from units.modules.utils import AnsibleFailJson, ModuleTestCase, set_module_args

from units.compat import mock


class FactsTest(ModuleTestCase):
    REQUIRED_PARAMS = {
        'api_username': 'rw',
        'api_password': 'password',
        'api_url': 'http://localhost',
        'ssid': '1'
    }
    REQUEST_FUNC = 'ansible.modules.storage.netapp.netapp_e_facts.Facts.request'
    GET_CONTROLLERS_FUNC = 'ansible.modules.storage.netapp.netapp_e_facts.Facts.get_controllers'
    WORKLOAD_RESPONSE = [{"id": "4200000001000000000000000000000000000000", "name": "beegfs_metadata",
                          "workloadAttributes": [{"key": "profileId", "value": "ansible_workload_1"}]},
                         {"id": "4200000002000000000000000000000000000000", "name": "other_workload_1",
                          "workloadAttributes": [{"key": "profileId", "value": "Other_1"}]}]
    GRAPH_RESPONSE = {
        "sa": {"saData": {"storageArrayLabel": "ictm0718s01c1", "saId": {"worldWideName": "600A098000A4B28D000000005CF10481"}, "fwVersion": "08.42.30.05",
                          "chassisSerialNumber": "021633035190"},
               "featureParameters": {"cacheBlockSizes": [4096, 8192, 16384, 32768],
                                     "supportedSegSizes": [32768, 65536, 131072, 262144, 524288, 495616, 655360, 1982464]},
               "capabilities": ["autoCodeSync", "autoLunTransfer", "subLunsAllowed", "stagedDownload", "mixedDriveTypes", "bundleMigration", "raid6",
                                "performanceTier", "secureVolume", "protectionInformation", "ssdSupport", "driveSlotLimit", "flashReadCache",
                                "storagePoolsType2", "totalNumberOfArvmMirrorsPerArray", "totalNumberOfPitsPerArray", "totalNumberOfThinVolumesPerArray"],
               "premiumFeatures": [],
               "hostSpecificVals": [{"hostType": "FactoryDefault", "index": 0}, {"hostType": "W2KNETNCL", "index": 1}, {"hostPortType": "W2KNETCL", "index": 8},
                                    {"hostType": "LnxTPGSALUA_SF", "index": 27}, {"hostType": "LnxDHALUA", "index": 28}]}, "controller": [
            {"active": True, "quiesced": False, "status": "optimal", "controllerRef": "070000000000000000000001",
             "physicalLocation": {"trayRef": "0E00000000000000000000000000000000000000", "slot": 1,
                                  "locationParent": {"refType": "generic", "controllerRef": None, "symbolRef": "0000000000000000000000000000000000000000",
                                                     "typedReference": None}, "locationPosition": 1, "label": "A"}, "manufacturer": "NETAPP  ",
             "manufacturerDate": "1474675200", "appVersion": "08.42.30.05", "bootVersion": "08.42.30.05", "productID": "INF-01-00       ",
             "productRevLevel": "0842", "serialNumber": "021619039162    ", "boardID": "2806", "cacheMemorySize": 3328, "processorMemorySize": 1278,
             "hostInterfaces": [{"interfaceType": "iscsi", "fibre": None, "ib": None,
                                 "iscsi": {"channel": 1, "channelPortRef": "1F00010001010000000000000000000000000000", "tcpListenPort": 3260,
                                           "ipv4Enabled": True, "ipv4Data": {"ipv4Address": "0.0.0.0", "ipv4AddressConfigMethod": "configStatic",
                                                                             "ipv4OutboundPacketPriority": {"isEnabled": False, "value": 1},
                                                                             "ipv4VlanId": {"isEnabled": False, "value": 1},
                                                                             "ipv4AddressData": {"configState": "configured", "ipv4Address": "10.10.11.110",
                                                                                                 "ipv4SubnetMask": "255.255.255.0",
                                                                                                 "ipv4GatewayAddress": "0.0.0.0"}},
                                           "interfaceData": {"type": "ethernet", "ethernetData": {
                                               "partData": {"vendorName": "QLogic Corporation", "partNumber": "83xx", "revisionNumber": "5.5.31.511",
                                                            "serialNumber": "00a098a4b28f"}, "macAddress": "00A098A4B293", "fullDuplex": True,
                                               "maximumFramePayloadSize": 9000, "currentInterfaceSpeed": "speed10gig", "maximumInterfaceSpeed": "speed10gig",
                                               "linkStatus": "up", "supportedInterfaceSpeeds": ["speed1gig", "speed10gig"], "autoconfigSupport": False,
                                               "copperCableDiagnosticsSupport": False}, "infinibandData": None},
                                           "interfaceRef": "2201020000000000000000000000000000000000", "ipv6Enabled": True,
                                           "ipv6Data": {"ipv6LocalAddresses": [
                                               {"address": "FE8000000000000002A098FFFEA4B293",
                                                "addressState": {"addressType": "typeInterface", "interfaceAddressState": "configured",
                                                                 "routerAddressState": "__UNDEFINED"}}], "ipv6RoutableAddresses": [
                                               {"address": "00000000000000000000000000000000",
                                                "addressState": {"addressType": "typeInterface", "interfaceAddressState": "unconfigured",
                                                                 "routerAddressState": "__UNDEFINED"}},
                                               {"address": "00000000000000000000000000000000",
                                                "addressState": {"addressType": "typeInterface", "interfaceAddressState": "unconfigured",
                                                                 "routerAddressState": "__UNDEFINED"}}],
                                               "ipv6PortRouterAddress": {"address": "00000000000000000000000000000000",
                                                                         "addressState": {"addressType": "typeRouter", "interfaceAddressState": "__UNDEFINED",
                                                                                          "routerAddressState": "unknown"}},
                                               "ipv6AddressConfigMethod": "configStateless", "ipv6OutboundPacketPriority": {"isEnabled": False, "value": 1},
                                               "ipv6VlanId": {"isEnabled": False, "value": 1}, "ipv6HopLimit": 64, "ipv6NdReachableTime": 30000,
                                               "ipv6NdRetransmitTime": 1000, "ipv6NdStaleTimeout": 30000, "ipv6DuplicateAddressDetectionAttempts": 1},
                                           "physicalLocation": {"trayRef": "0000000000000000000000000000000000000000", "slot": 0,
                                                                "locationParent": {"refType": "generic", "controllerRef": None,
                                                                                   "symbolRef": "0000000000000000000000000000000000000000",
                                                                                   "typedReference": None}, "locationPosition": 0, "label": ""},
                                           "protectionInformationCapable": True, "isIPv6Capable": True, "oneWayMaxRate": "1230000000",
                                           "bidirectionalMaxRate": "2120000000", "iqn": "iqn.1992-08.com.netapp:2806.600a098000a4b28d000000005cf10481",
                                           "controllerId": "070000000000000000000001",
                                           "addressId": "iqn.1992-08.com.netapp:2806.600a098000a4b28d000000005cf10481",
                                           "niceAddressId": "iqn.1992-08.com.netapp:2806.600a098000a4b28d000000005cf10481",
                                           "interfaceId": "2201020000000000000000000000000000000000", "id": "2201020000000000000000000000000000000000"},
                                 "sas": None, "sata": None, "scsi": None}],
             "driveInterfaces": [
                 {"interfaceType": "sas", "fibre": None, "ib": None, "iscsi": None,
                  "sas": {"channel": 1, "currentInterfaceSpeed": "speed12gig", "maximumInterfaceSpeed": "speed12gig", "part": "LSISAS3008",
                          "revision": 172688896, "isDegraded": False,
                          "iocPort": {
                              "parent": {"type": "controller", "controller": "070000000000000000000001", "drive": None, "expander": None, "hostBoardRef": None},
                              "attachedDevice": {"channel": 1, "channelType": "driveside",
                                                 "sasAttachedDeviceData": {"type": "expander", "alternateController": None, "drive": None,
                                                                           "expander": "2000000000000000000000630001000000000000",
                                                                           "remoteHostPortAddress": None,
                                                                           "localController": None, "physicalLocation": None}}, "state": "optimal",
                              "miswireType": "None", "channelPortRef": "1F01000001010000000000000000000000000000",
                              "sasPhys": [{"phyIdentifier": 4, "isOperational": True}, {"phyIdentifier": 5, "isOperational": True},
                                          {"phyIdentifier": 6, "isOperational": True}, {"phyIdentifier": 7, "isOperational": True}],
                              "portTypeData": {"portType": "endDevice", "portIdentifier": "500A098A4B28D004", "routingType": "__UNDEFINED"},
                              "portMode": "internal",
                              "domainNumber": 1, "attachedChannelPortRef": "0000000000000000000000000000000000000000", "discoveryStatus": 0},
                          "interfaceRef": "2201000000000000000000000000000000000000",
                          "physicalLocation": {"trayRef": "0000000000000000000000000000000000000000", "slot": 0,
                                               "locationParent": {"refType": "generic", "controllerRef": None,
                                                                  "symbolRef": "0000000000000000000000000000000000000000", "typedReference": None},
                                               "locationPosition": 0, "label": ""}, "protectionInformationCapable": True, "oneWayMaxRate": "4400000000",
                          "bidirectionalMaxRate": "8400000000", "controllerId": None, "addressId": "500A098A4B28D004", "niceAddressId": "500A098A4B28D004",
                          "interfaceId": "2201000000000000000000000000000000000000", "basePortAddress": "500A098A4B28D00",
                          "id": "2201000000000000000000000000000000000000"}, "sata": None, "scsi": None}],
             "netInterfaces": [{"interfaceType": "ethernet",
                                "ethernet": {"interfaceName": "wan0", "channel": 1, "speed": 1000, "ip": 175178176, "alias": "ictm0718s01c1-a",
                                             "macAddr": "00A098A4B28D", "gatewayIp": 175177985, "subnetMask": -256, "bootpUsed": False, "rloginEnabled": True,
                                             "reserved1": "0000000000000000", "setupError": False, "reserved2": "",
                                             "interfaceRef": "2800070000000000000000000001000000000000", "linkStatus": "up", "ipv4Enabled": True,
                                             "ipv4Address": "10.113.1.192", "ipv4SubnetMask": "255.255.255.0", "ipv4AddressConfigMethod": "configStatic",
                                             "ipv6Enabled": False, "ipv6LocalAddress": {"address": "00000000000000000000000000000000",
                                                                                        "addressState": {"addressType": "typeInterface",
                                                                                                         "interfaceAddressState": "configured",
                                                                                                         "routerAddressState": "__UNDEFINED"}},
                                             "ipv6PortStaticRoutableAddress": {"address": "00000000000000000000000000000000",
                                                                               "addressState": {"addressType": "typeInterface",
                                                                                                "interfaceAddressState": "__UNDEFINED",
                                                                                                "routerAddressState": "__UNDEFINED"}},
                                             "ipv6PortRoutableAddresses": [], "ipv6AddressConfigMethod": "configStatic", "fullDuplex": True,
                                             "supportedSpeedSettings": ["speedAutoNegotiated", "speed10MbitHalfDuplex", "speed10MbitFullDuplex",
                                                                        "speed100MbitHalfDuplex", "speed100MbitFullDuplex", "speed1000MbitFullDuplex"],
                                             "configuredSpeedSetting": "speedAutoNegotiated", "currentSpeed": "speed1gig",
                                             "physicalLocation": {"trayRef": "0E00000000000000000000000000000000000000", "slot": 0,
                                                                  "locationParent": {"refType": "controller", "controllerRef": "070000000000000000000001",
                                                                                     "symbolRef": None, "typedReference": None}, "locationPosition": 1,
                                                                  "label": "P1"}, "ipv4GatewayAddress": "10.113.1.1",
                                             "controllerRef": "070000000000000000000001", "controllerSlot": 1,
                                             "dnsProperties": {
                                                 "acquisitionProperties": {"dnsAcquisitionType": "stat",
                                                                           "dnsServers": [
                                                                               {"addressType": "ipv4", "ipv4Address": "10.193.0.250", "ipv6Address": None},
                                                                               {"addressType": "ipv4", "ipv4Address": "10.192.0.250", "ipv6Address": None}]},
                                                 "dhcpAcquiredDnsServers": []},
                                             "ntpProperties": {
                                                 "acquisitionProperties": {"ntpAcquisitionType": "stat", "ntpServers": [
                                                     {"addrType": "ipvx", "domainName": None,
                                                      "ipvxAddress": {"addressType": "ipv4", "ipv4Address": "216.239.35.0", "ipv6Address": None}},
                                                     {"addrType": "ipvx", "domainName": None,
                                                      "ipvxAddress": {"addressType": "ipv4", "ipv4Address": "216.239.35.4", "ipv6Address": None}}]},
                                                 "dhcpAcquiredNtpServers": []},
                                             "id": "2800070000000000000000000001000000000000"}}],
             "inventory": [], "reserved1": "000000000000000000000000", "reserved2": "", "hostBoardID": "None", "physicalCacheMemorySize": 4864,
             "readyToRemove": False, "boardSubmodelID": "319", "submodelSupported": True, "oemPartNumber": "E2800A-8GB", "partNumber": "111-02829+C0   ",
             "rtrAttributes": {"cruType": "dedicated", "parentCru": None, "rtrAttributeData": {"hasReadyToRemoveIndicator": False, "readyToRemove": False}},
             "bootTime": "1563988406", "modelName": "2806",
             "networkSettings": {"ipv4DefaultRouterAddress": "10.113.1.1",
                                 "ipv6DefaultRouterAddress": {"address": "00000000000000000000000000000000",
                                                              "addressState": {"addressType": "typeInterface",
                                                                               "interfaceAddressState": "__UNDEFINED", "routerAddressState": "__UNDEFINED"}},
                                 "ipv6CandidateDefaultRouterAddresses": [],
                                 "remoteAccessEnabled": True,
                                 "dnsProperties": {"acquisitionProperties": {"dnsAcquisitionType": "stat",
                                                                             "dnsServers": [
                                                                                 {"addressType": "ipv4", "ipv4Address": "10.193.0.250", "ipv6Address": None},
                                                                                 {"addressType": "ipv4", "ipv4Address": "10.192.0.250", "ipv6Address": None}]},
                                                   "dhcpAcquiredDnsServers": []},
                                 "ntpProperties": {
                                     "acquisitionProperties": {
                                         "ntpAcquisitionType": "stat", "ntpServers": [
                                             {"addrType": "ipvx", "domainName": None,
                                              "ipvxAddress": {"addressType": "ipv4", "ipv4Address": "216.239.35.0", "ipv6Address": None}},
                                             {"addrType": "ipvx", "domainName": None,
                                              "ipvxAddress": {"addressType": "ipv4", "ipv4Address": "216.239.35.4", "ipv6Address": None}}]},
                                     "dhcpAcquiredNtpServers": []}},
             "repairPolicy": {"removalData": {"removalMethod": "__UNDEFINED", "rtrAttributes": None}, "replacementMethod": "__UNDEFINED"},
             "flashCacheMemorySize": 419430400, "ctrlIocDumpData": {"iocDumpNeedsRetrieved": False, "iocDumpTag": 0, "timeStamp": "0"},
             "locateInProgress": False, "hasTrayIdentityIndicator": False, "controllerErrorMode": "notInErrorMode",
             "codeVersions": [{"codeModule": "raid", "versionString": "08.42.30.05"}, {"codeModule": "hypervisor", "versionString": "08.42.30.05"},
                              {"codeModule": "management", "versionString": "11.42.0000.0026"}, {"codeModule": "iom", "versionString": "11.42.0G00.0001"},
                              {"codeModule": "bundle", "versionString": "08.42.30.05"}, {"codeModule": "bundleDisplay", "versionString": "11.40.3R2"}],
             "id": "070000000000000000000001"}],
        "drive": [{"offline": False, "hotSpare": False, "invalidDriveData": False, "available": True, "pfa": False,
                   "driveRef": "0100000050000396AC882ED10000000000000000", "status": "optimal", "cause": "None",
                   "interfaceType": {"driveType": "sas", "fibre": None,
                                     "sas": {"deviceName": "50000396AC882ED1",
                                             "drivePortAddresses": [{"channel": 2, "portIdentifier": "50000396AC882ED3"},
                                                                    {"channel": 1, "portIdentifier": "50000396AC882ED2"}]},
                                     "scsi": None},
                   "physicalLocation": {"trayRef": "0E00000000000000000000000000000000000000", "slot": 6,
                                        "locationParent": {"refType": "genericTyped", "controllerRef": None, "symbolRef": None,
                                                           "typedReference": {"componentType": "tray",
                                                                              "symbolRef": "0E00000000000000000000000000000000000000"}},
                                        "locationPosition": 6, "label": "5"}, "manufacturer": "TOSHIBA ",
                   "manufacturerDate": "1447200000", "productID": "PX04SVQ160      ", "serialNumber": "Y530A001T5MD", "softwareVersion": "MSB6", "blkSize": 512,
                   "usableCapacity": "1599784443904", "rawCapacity": "1600321314816", "worldWideName": "50000396AC882ED10000000000000000",
                   "currentVolumeGroupRef": "0000000000000000000000000000000000000000", "sparedForDriveRef": "0000000000000000000000000000000000000000",
                   "mirrorDrive": "0000000000000000000000000000000000000000", "nonRedundantAccess": False, "workingChannel": -1, "volumeGroupIndex": -1,
                   "currentSpeed": "speed12gig", "maxSpeed": "speed12gig", "uncertified": False, "hasDegradedChannel": False, "degradedChannels": [],
                   "phyDriveType": "sas", "spindleSpeed": 0, "rtrAttributes": {"cruType": "dedicated", "parentCru": None,
                                                                               "rtrAttributeData": {"hasReadyToRemoveIndicator": False,
                                                                                                    "readyToRemove": False}}, "reserved": "",
                   "phyDriveTypeData": {"phyDriveType": "sas", "sataDriveAttributes": None}, "pfaReason": "None", "bypassSource": [],
                   "repairPolicy": {"removalData": {"removalMethod": "self", "rtrAttributes": {"hasReadyToRemoveIndicator": False, "readyToRemove": False}},
                                    "replacementMethod": "self"}, "fdeCapable": True, "fdeEnabled": False, "fdeLocked": False,
                   "lockKeyID": "0000000000000000000000000000000000000000",
                   "ssdWearLife": {"averageEraseCountPercent": 18, "spareBlocksRemainingPercent": 91, "isWearLifeMonitoringSupported": True,
                                   "percentEnduranceUsed": 18}, "driveMediaType": "ssd", "fpgaVersion": "",
                   "protectionInformationCapabilities": {"protectionInformationCapable": True, "protectionType": "type2Protection"},
                   "protectionInformationCapable": False, "protectionType": "type0Protection", "interposerPresent": False,
                   "interposerRef": "0000000000000000000000000000000000000000", "currentCommandAgingTimeout": 6, "defaultCommandAgingTimeout": 6,
                   "driveTemperature": {"currentTemp": 25, "refTemp": 64}, "blkSizePhysical": 4096, "lowestAlignedLBA": "0", "removed": False,
                   "locateInProgress": False, "fipsCapable": False, "firmwareVersion": "MSB6", "lockKeyIDValue": None,
                   "id": "0100000050000396AC882ED10000000000000000"},
                  {"offline": False, "hotSpare": False, "invalidDriveData": False, "available": True, "pfa": False,
                   "driveRef": "0100000050000396AC882EDD0000000000000000", "status": "optimal", "cause": "None",
                   "interfaceType": {"driveType": "sas", "fibre": None,
                                     "sas": {"deviceName": "50000396AC882EDD",
                                             "drivePortAddresses": [{"channel": 2, "portIdentifier": "50000396AC882EDF"},
                                                                    {"channel": 1, "portIdentifier": "50000396AC882EDE"}]},
                                     "scsi": None},
                   "physicalLocation": {"trayRef": "0E00000000000000000000000000000000000000", "slot": 8,
                                        "locationParent": {"refType": "genericTyped", "controllerRef": None, "symbolRef": None,
                                                           "typedReference": {"componentType": "tray",
                                                                              "symbolRef": "0E00000000000000000000000000000000000000"}},
                                        "locationPosition": 8, "label": "7"}, "manufacturer": "TOSHIBA ",
                   "manufacturerDate": "1447200000", "productID": "PX04SVQ160      ", "serialNumber": "Y530A004T5MD", "softwareVersion": "MSB6", "blkSize": 512,
                   "usableCapacity": "1599784443904", "rawCapacity": "1600321314816", "worldWideName": "50000396AC882EDD0000000000000000",
                   "currentVolumeGroupRef": "0000000000000000000000000000000000000000", "sparedForDriveRef": "0000000000000000000000000000000000000000",
                   "mirrorDrive": "0000000000000000000000000000000000000000", "nonRedundantAccess": False, "workingChannel": -1, "volumeGroupIndex": -1,
                   "currentSpeed": "speed12gig", "maxSpeed": "speed12gig", "uncertified": False, "hasDegradedChannel": False, "degradedChannels": [],
                   "phyDriveType": "sas", "spindleSpeed": 0, "rtrAttributes": {"cruType": "dedicated", "parentCru": None,
                                                                               "rtrAttributeData": {"hasReadyToRemoveIndicator": False,
                                                                                                    "readyToRemove": False}}, "reserved": "",
                   "phyDriveTypeData": {"phyDriveType": "sas", "sataDriveAttributes": None}, "pfaReason": "None", "bypassSource": [],
                   "repairPolicy": {"removalData": {"removalMethod": "self", "rtrAttributes": {"hasReadyToRemoveIndicator": False, "readyToRemove": False}},
                                    "replacementMethod": "self"}, "fdeCapable": True, "fdeEnabled": False, "fdeLocked": False,
                   "lockKeyID": "0000000000000000000000000000000000000000",
                   "ssdWearLife": {"averageEraseCountPercent": 18, "spareBlocksRemainingPercent": 91, "isWearLifeMonitoringSupported": True,
                                   "percentEnduranceUsed": 18}, "driveMediaType": "ssd", "fpgaVersion": "",
                   "protectionInformationCapabilities": {"protectionInformationCapable": True, "protectionType": "type2Protection"},
                   "protectionInformationCapable": False, "protectionType": "type0Protection", "interposerPresent": False,
                   "interposerRef": "0000000000000000000000000000000000000000", "currentCommandAgingTimeout": 6, "defaultCommandAgingTimeout": 6,
                   "driveTemperature": {"currentTemp": 25, "refTemp": 64}, "blkSizePhysical": 4096, "lowestAlignedLBA": "0", "removed": False,
                   "locateInProgress": False, "fipsCapable": False, "firmwareVersion": "MSB6", "lockKeyIDValue": None,
                   "id": "0100000050000396AC882EDD0000000000000000"}],
        "volumeGroup": [
            {"sequenceNum": 1, "offline": False, "raidLevel": "raid6", "worldWideName": "600A098000A4B9D10000380A5D4AAC3C",
             "volumeGroupRef": "04000000600A098000A4B9D10000380A5D4AAC3C", "reserved1": "000000000000000000000000", "reserved2": "",
             "trayLossProtection": False, "label": "beegfs_storage_vg", "state": "complete", "spindleSpeedMatch": True, "spindleSpeed": 10500,
             "isInaccessible": False, "securityType": "capable", "drawerLossProtection": False, "protectionInformationCapable": False,
             "protectionInformationCapabilities": {"protectionInformationCapable": True, "protectionType": "type2Protection"},
             "volumeGroupData": {"type": "unknown", "diskPoolData": None},
             "usage": "standard", "driveBlockFormat": "allNative", "reservedSpaceAllocated": False, "securityLevel": "fde", "usedSpace": "1099511627776",
             "totalRaidedSpace": "9597654597632",
             "extents": [{"sectorOffset": "268435456", "rawCapacity": "8498142969856", "raidLevel": "raid6",
                          "volumeGroupRef": "04000000600A098000A4B9D10000380A5D4AAC3C", "freeExtentRef": "03000000600A098000A4B9D10000380A5D4AAC3C",
                          "reserved1": "000000000000000000000000", "reserved2": ""}],
             "largestFreeExtentSize": "8498142969856", "raidStatus": "optimal", "freeSpace": "8498142969856", "drivePhysicalType": "sas",
             "driveMediaType": "hdd", "normalizedSpindleSpeed": "spindleSpeed10k", "diskPool": False,
             "id": "04000000600A098000A4B9D10000380A5D4AAC3C", "name": "beegfs_storage_vg"}], "volume": [
            {"offline": False, "extremeProtection": False, "volumeHandle": 0, "raidLevel": "raid6", "sectorOffset": "0",
             "worldWideName": "600A098000A4B28D00003E435D4AAC54", "label": "beegfs_storage_01_1", "blkSize": 512, "capacity": "1099511627776",
             "reconPriority": 1, "segmentSize": 131072, "action": "None",
             "cache": {"cwob": False, "enterpriseCacheDump": False, "mirrorActive": True, "mirrorEnable": True, "readCacheActive": False,
                       "readCacheEnable": False, "writeCacheActive": True, "writeCacheEnable": True, "cacheFlushModifier": "flush10Sec",
                       "readAheadMultiplier": 1}, "mediaScan": {"enable": True, "parityValidationEnable": True},
             "volumeRef": "02000000600A098000A4B28D00003E435D4AAC54", "status": "optimal", "volumeGroupRef": "04000000600A098000A4B9D10000380A5D4AAC3C",
             "currentManager": "070000000000000000000001", "preferredManager": "070000000000000000000001",
             "perms": {"mapToLUN": True, "snapShot": True, "format": True, "reconfigure": True, "mirrorPrimary": True, "mirrorSecondary": True,
                       "copySource": True, "copyTarget": True, "readable": True, "writable": True, "rollback": True, "mirrorSync": True, "newImage": True,
                       "allowDVE": True, "allowDSS": True, "concatVolumeMember": False, "flashReadCache": True, "asyncMirrorPrimary": True,
                       "asyncMirrorSecondary": True, "pitGroup": True, "cacheParametersChangeable": True, "allowThinManualExpansion": False,
                       "allowThinGrowthParametersChange": False},
             "mgmtClientAttribute": 0, "dssPreallocEnabled": False, "dssMaxSegmentSize": 0, "preReadRedundancyCheckEnabled": False,
             "protectionInformationCapable": False, "protectionType": "type0Protection", "applicationTagOwned": True,
             "repairedBlockCount": 0, "extendedUniqueIdentifier": "", "cacheMirroringValidateProtectionInformation": False,
             "expectedProtectionInformationAppTag": 0, "volumeUse": "standardVolume", "volumeFull": False, "volumeCopyTarget": False, "volumeCopySource": False,
             "pitBaseVolume": False, "asyncMirrorTarget": False, "asyncMirrorSource": False, "remoteMirrorSource": False, "remoteMirrorTarget": False,
             "diskPool": False, "flashCached": False, "increasingBy": "0", "metadata": [], "dataAssurance": False, "objectType": "volume",
             "listOfMappings": [
                 {"lunMappingRef": "88000000A1010000000000000000000000000000", "lun": 1, "ssid": 0, "perms": 15,
                  "volumeRef": "02000000600A098000A4B28D00003E435D4AAC54", "type": "host", "mapRef": "84000000600A098000A4B28D00303D065D430118",
                  "id": "88000000A1010000000000000000000000000000"}],
             "mapped": True, "currentControllerId": "070000000000000000000001",
             "cacheSettings": {"cwob": False, "enterpriseCacheDump": False, "mirrorActive": True, "mirrorEnable": True, "readCacheActive": False,
                               "readCacheEnable": False, "writeCacheActive": True, "writeCacheEnable": True, "cacheFlushModifier": "flush10Sec",
                               "readAheadMultiplier": 1},
             "thinProvisioned": False, "preferredControllerId": "070000000000000000000001", "totalSizeInBytes": "1099511627776", "onlineVolumeCopy": False,
             "wwn": "600A098000A4B28D00003E435D4AAC54", "name": "beegfs_storage_01_1", "id": "02000000600A098000A4B28D00003E435D4AAC54"}],
        "storagePoolBundle": {"cluster": [], "host": [
            {"hostRef": "84000000600A098000A4B28D00303D005D430107", "clusterRef": "0000000000000000000000000000000000000000", "label": "test",
             "isSAControlled": False, "confirmLUNMappingCreation": False, "hostTypeIndex": 28, "protectionInformationCapableAccessMethod": True,
             "isLargeBlockFormatHost": False, "isLun0Restricted": False, "ports": [],
             "initiators": [
                 {"initiatorRef": "89000000600A098000A4B9D1003037005D4300F5",
                  "nodeName": {"ioInterfaceType": "iscsi", "iscsiNodeName": "iqn.iscsi_tests1", "remoteNodeWWN": None, "nvmeNodeName": None},
                  "alias": {"ioInterfaceType": "iscsi", "iscsiAlias": ""}, "label": "iscsi_test1",
                  "configuredAuthMethods": {"authMethodData": [{"authMethod": "None", "chapSecret": None}]},
                  "hostRef": "84000000600A098000A4B28D00303D005D430107", "initiatorInactive": False, "id": "89000000600A098000A4B9D1003037005D4300F5"}],
             "hostSidePorts": [{"type": "iscsi", "address": "iqn.iscsi_tests1", "label": "iscsi_test1"}],
             "id": "84000000600A098000A4B28D00303D005D430107", "name": "test"},
            {"hostRef": "84000000600A098000A4B9D1003037035D4300F8", "clusterRef": "0000000000000000000000000000000000000000", "label": "test2",
             "isSAControlled": True, "confirmLUNMappingCreation": False, "hostTypeIndex": 28, "protectionInformationCapableAccessMethod": True,
             "isLargeBlockFormatHost": False, "isLun0Restricted": False, "ports": [],
             "initiators": [
                 {"initiatorRef": "89000000600A098000A4B9D1003037075D4300F9",
                  "nodeName": {"ioInterfaceType": "iscsi", "iscsiNodeName": "iqn.iscsi_tests2", "remoteNodeWWN": None, "nvmeNodeName": None},
                  "alias": {"ioInterfaceType": "iscsi", "iscsiAlias": ""}, "label": "iscsi_test2",
                  "configuredAuthMethods": {"authMethodData": [{"authMethod": "None", "chapSecret": None}]},
                  "hostRef": "84000000600A098000A4B9D1003037035D4300F8", "initiatorInactive": False, "id": "89000000600A098000A4B9D1003037075D4300F9"}],
             "hostSidePorts": [{"type": "iscsi", "address": "iqn.iscsi_tests2", "label": "iscsi_test2"}],
             "id": "84000000600A098000A4B9D1003037035D4300F8", "name": "test2"},
            {"hostRef": "84000000600A098000A4B28D00303D065D430118", "clusterRef": "0000000000000000000000000000000000000000", "label": "beegfs_storage1",
             "isSAControlled": False, "confirmLUNMappingCreation": False, "hostTypeIndex": 28, "protectionInformationCapableAccessMethod": True,
             "isLargeBlockFormatHost": False, "isLun0Restricted": False, "ports": [],
             "initiators": [
                 {"initiatorRef": "89000000600A098000A4B28D00303CF55D4300E3",
                  "nodeName": {"ioInterfaceType": "iscsi", "iscsiNodeName": "iqn.1993-08.org.debian.beegfs-storage1:01:b0621126818", "remoteNodeWWN": None,
                               "nvmeNodeName": None}, "alias": {"ioInterfaceType": "iscsi", "iscsiAlias": ""}, "label": "beegfs_storage1_iscsi_0",
                  "configuredAuthMethods": {"authMethodData": [{"authMethod": "None", "chapSecret": None}]},
                  "hostRef": "84000000600A098000A4B28D00303D065D430118", "initiatorInactive": False, "id": "89000000600A098000A4B28D00303CF55D4300E3"}],
             "hostSidePorts": [{"type": "iscsi", "address": "iqn.1993-08.org.debian.beegfs-storage1:01:b0621126818", "label": "beegfs_storage1_iscsi_0"}],
             "id": "84000000600A098000A4B28D00303D065D430118", "name": "beegfs_storage1"},
            {"hostRef": "84000000600A098000A4B9D10030370B5D430109", "clusterRef": "0000000000000000000000000000000000000000", "label": "beegfs_metadata1",
             "isSAControlled": False, "confirmLUNMappingCreation": False, "hostTypeIndex": 28, "protectionInformationCapableAccessMethod": True,
             "isLargeBlockFormatHost": False, "isLun0Restricted": False, "ports": [],
             "initiators": [
                 {"initiatorRef": "89000000600A098000A4B28D00303CFC5D4300F7",
                  "nodeName": {"ioInterfaceType": "iscsi", "iscsiNodeName": "iqn.1993-08.org.debian.beegfs-metadata:01:69e4efdf30b8", "remoteNodeWWN": None,
                               "nvmeNodeName": None}, "alias": {"ioInterfaceType": "iscsi", "iscsiAlias": ""}, "label": "beegfs_metadata1_iscsi_0",
                  "configuredAuthMethods": {"authMethodData": [{"authMethod": "None", "chapSecret": None}]},
                  "hostRef": "84000000600A098000A4B9D10030370B5D430109", "initiatorInactive": False, "id": "89000000600A098000A4B28D00303CFC5D4300F7"}],
             "hostSidePorts": [{"type": "iscsi", "address": "iqn.1993-08.org.debian.beegfs-metadata:01:69e4efdf30b8", "label": "beegfs_metadata1_iscsi_0"}],
             "id": "84000000600A098000A4B9D10030370B5D430109", "name": "beegfs_metadata1"}], "lunMapping": [
            {"lunMappingRef": "8800000000000000000000000000000000000000", "lun": 7, "ssid": 16384, "perms": 15,
             "volumeRef": "21000000600A098000A4B28D000027EC5CF10481", "type": "all", "mapRef": "0000000000000000000000000000000000000000",
             "id": "8800000000000000000000000000000000000000"},
            {"lunMappingRef": "880000008B010000000000000000000000000000", "lun": 7, "ssid": 16384, "perms": 15,
             "volumeRef": "21000000600A098000A4B28D000027EC5CF10481", "type": "host", "mapRef": "84000000600A098000A4B28D00303D065D430118",
             "id": "880000008B010000000000000000000000000000"},
            {"lunMappingRef": "8800000090010000000000000000000000000000", "lun": 7, "ssid": 16384, "perms": 15,
             "volumeRef": "21000000600A098000A4B28D000027EC5CF10481", "type": "host", "mapRef": "84000000600A098000A4B9D10030370B5D430109",
             "id": "8800000090010000000000000000000000000000"},
            {"lunMappingRef": "8800000092010000000000000000000000000000", "lun": 7, "ssid": 16384, "perms": 15,
             "volumeRef": "21000000600A098000A4B28D000027EC5CF10481", "type": "host", "mapRef": "84000000600A098000A4B28D00303D005D430107",
             "id": "8800000092010000000000000000000000000000"}, {"lunMappingRef": "88000000A1010000000000000000000000000000", "lun": 1, "ssid": 0, "perms": 15,
                                                                 "volumeRef": "02000000600A098000A4B28D00003E435D4AAC54", "type": "host",
                                                                 "mapRef": "84000000600A098000A4B28D00303D065D430118",
                                                                 "id": "88000000A1010000000000000000000000000000"}]}, "highLevelVolBundle": {"pit": []}}

    EXPECTED_GET_ARRAY_FACTS = {'facts_from_proxy': False, 'netapp_controllers': [{'name': 'A', 'serial': '021619039162', 'status': 'optimal'}],
                                'netapp_disks': [
                                    {'available': True, 'firmware_version': 'MSB6', 'id': '0100000050000396AC882ED10000000000000000', 'media_type': 'ssd',
                                     'product_id': 'PX04SVQ160      ', 'serial_number': 'Y530A001T5MD', 'status': 'optimal',
                                     'tray_ref': '0E00000000000000000000000000000000000000', 'usable_bytes': '1599784443904'},
                                    {'available': True, 'firmware_version': 'MSB6', 'id': '0100000050000396AC882EDD0000000000000000', 'media_type': 'ssd',
                                     'product_id': 'PX04SVQ160      ', 'serial_number': 'Y530A004T5MD', 'status': 'optimal',
                                     'tray_ref': '0E00000000000000000000000000000000000000', 'usable_bytes': '1599784443904'}],
                                'netapp_driveside_interfaces': [{'controller': 'A', 'interface_speed': '12g', 'interface_type': 'sas'}],
                                'netapp_enabled_features': ['autoCodeSync', 'autoLunTransfer', 'bundleMigration', 'driveSlotLimit', 'flashReadCache',
                                                            'mixedDriveTypes', 'performanceTier', 'protectionInformation', 'raid6', 'secureVolume',
                                                            'ssdSupport', 'stagedDownload', 'storagePoolsType2', 'subLunsAllowed',
                                                            'totalNumberOfArvmMirrorsPerArray', 'totalNumberOfPitsPerArray',
                                                            'totalNumberOfThinVolumesPerArray'], 'netapp_host_groups': [],
                                'netapp_host_types': [{'index': 0, 'type': 'FactoryDefault'}, {'index': 1, 'type': 'W2KNETNCL'},
                                                      {'index': 27, 'type': 'LnxTPGSALUA_SF'}, {'index': 28, 'type': 'LnxDHALUA'}],
                                'netapp_hosts': [
                                    {'group_id': '0000000000000000000000000000000000000000', 'host_type_index': 28,
                                     'hosts_reference': '84000000600A098000A4B28D00303D005D430107',
                                     'id': '84000000600A098000A4B28D00303D005D430107', 'name': 'test',
                                     'posts': [{'address': 'iqn.iscsi_tests1', 'label': 'iscsi_test1', 'type': 'iscsi'}]},
                                    {'group_id': '0000000000000000000000000000000000000000', 'host_type_index': 28,
                                     'hosts_reference': '84000000600A098000A4B9D1003037035D4300F8',
                                     'id': '84000000600A098000A4B9D1003037035D4300F8', 'name': 'test2',
                                     'posts': [{'address': 'iqn.iscsi_tests2', 'label': 'iscsi_test2', 'type': 'iscsi'}]},
                                    {'group_id': '0000000000000000000000000000000000000000', 'host_type_index': 28,
                                     'hosts_reference': '84000000600A098000A4B28D00303D065D430118',
                                     'id': '84000000600A098000A4B28D00303D065D430118', 'name': 'beegfs_storage1',
                                     'posts': [{'address': 'iqn.1993-08.org.debian.beegfs-storage1:01:b0621126818', 'label': 'beegfs_storage1_iscsi_0',
                                                'type': 'iscsi'}]},
                                    {'group_id': '0000000000000000000000000000000000000000', 'host_type_index': 28,
                                     'hosts_reference': '84000000600A098000A4B9D10030370B5D430109',
                                     'id': '84000000600A098000A4B9D10030370B5D430109', 'name': 'beegfs_metadata1',
                                     'posts': [{'address': 'iqn.1993-08.org.debian.beegfs-metadata:01:69e4efdf30b8', 'label': 'beegfs_metadata1_iscsi_0',
                                                'type': 'iscsi'}]}],
                                'netapp_hostside_interfaces': [{'fc': [], 'ib': [],
                                                                'iscsi': [
                                                                    {'controller': 'A', 'current_interface_speed': '10g', 'ipv4_address': '10.10.11.110',
                                                                     'ipv4_enabled': True,
                                                                     'ipv4_gateway': '0.0.0.0', 'ipv4_subnet_mask': '255.255.255.0', 'ipv6_enabled': True,
                                                                     'iqn': 'iqn.1992-08.com.netapp:2806.600a098000a4b28d000000005cf10481', 'link_status': 'up',
                                                                     'mtu': 9000,
                                                                     'supported_interface_speeds': ['1g', '10g']}], 'sas': []}],
                                'netapp_management_interfaces': [
                                    {'alias': 'ictm0718s01c1-a', 'channel': 1, 'controller': 'A', 'dns_config_method': 'stat',
                                     'dns_servers': [{'addressType': 'ipv4', 'ipv4Address': '10.193.0.250', 'ipv6Address': None},
                                                     {'addressType': 'ipv4', 'ipv4Address': '10.192.0.250', 'ipv6Address': None}],
                                     'ipv4_address': '10.113.1.192',
                                     'ipv4_address_config_method': 'static', 'ipv4_enabled': True, 'ipv4_gateway': '10.113.1.1',
                                     'ipv4_subnet_mask': '255.255.255.0', 'ipv6_enabled': False, 'link_status': 'up',
                                     'mac_address': '00A098A4B28D', 'name': 'wan0', 'ntp_config_method': 'stat',
                                     'ntp_servers': [
                                         {'addrType': 'ipvx', 'domainName': None,
                                          'ipvxAddress': {'addressType': 'ipv4', 'ipv4Address': '216.239.35.0', 'ipv6Address': None}},
                                         {'addrType': 'ipvx', 'domainName': None,
                                          'ipvxAddress': {'addressType': 'ipv4', 'ipv4Address': '216.239.35.4', 'ipv6Address': None}}],
                                     'remote_ssh_access': True}],
                                'netapp_storage_array': {'cache_block_sizes': [4096, 8192, 16384, 32768], 'chassis_serial': '021633035190',
                                                         'firmware': '08.42.30.05', 'name': 'ictm0718s01c1',
                                                         'segment_sizes': [32768, 65536, 131072, 262144, 524288, 495616, 655360, 1982464],
                                                         'wwn': '600A098000A4B28D000000005CF10481'},
                                'netapp_storage_pools': [
                                    {'available_capacity': '8498142969856', 'id': '04000000600A098000A4B9D10000380A5D4AAC3C', 'name': 'beegfs_storage_vg',
                                     'total_capacity': '9597654597632', 'used_capacity': '1099511627776'}],
                                'netapp_volumes': [
                                    {'capacity': '1099511627776', 'id': '02000000600A098000A4B28D00003E435D4AAC54', 'is_thin_provisioned': False,
                                     'name': 'beegfs_storage_01_1', 'parent_storage_pool_id': '04000000600A098000A4B9D10000380A5D4AAC3C', 'workload': []}],
                                'netapp_volumes_by_initiators': {'beegfs_metadata1': [],
                                                                 'beegfs_storage1': [{
                                                                     'id': '02000000600A098000A4B28D00003E435D4AAC54',
                                                                     'meta_data': {},
                                                                     'name': 'beegfs_storage_01_1',
                                                                     'workload_name': '',
                                                                     'wwn': '600A098000A4B28D00003E435D4AAC54'}],
                                                                 'test': [], 'test2': []},
                                'netapp_workload_tags': [
                                    {'attributes': [{'key': 'profileId', 'value': 'ansible_workload_1'}], 'id': '4200000001000000000000000000000000000000',
                                     'name': 'beegfs_metadata'},
                                    {'attributes': [{'key': 'profileId', 'value': 'Other_1'}], 'id': '4200000002000000000000000000000000000000',
                                     'name': 'other_workload_1'}], 'snapshot_images': [], 'ssid': '1'}

    def _set_args(self, **kwargs):
        module_args = self.REQUIRED_PARAMS.copy()
        if kwargs is not None:
            module_args.update(kwargs)
        set_module_args(module_args)

    def test_get_controllers_pass(self):
        """Verify get_controllers returns the expected results."""
        self._set_args()
        facts = Facts()
        with mock.patch(self.REQUEST_FUNC, return_value=(200, ["070000000000000000000002", "070000000000000000000001"])):
            self.assertEqual(facts.get_controllers(), {"070000000000000000000001": "A", "070000000000000000000002": "B"})

    def test_get_controllers_fail(self):
        """Verify get_controllers throws the expected exceptions."""
        self._set_args()
        facts = Facts()
        with self.assertRaisesRegexp(AnsibleFailJson, "Failed to retrieve controller list!"):
            with mock.patch(self.REQUEST_FUNC, return_value=Exception()):
                facts.get_controllers()

    def test_get_array_facts_pass(self):
        """Verify get_array_facts method returns expected results."""
        self._set_args()
        facts = Facts()
        with mock.patch(self.GET_CONTROLLERS_FUNC, return_value={"070000000000000000000001": "A", "070000000000000000000002": "B"}):
            with mock.patch(self.REQUEST_FUNC, side_effect=[(200, self.GRAPH_RESPONSE), (200, self.WORKLOAD_RESPONSE)]):
                self.assertEquals(facts.get_array_facts(), self.EXPECTED_GET_ARRAY_FACTS)
