// -*- Mode: C++; tab-width: 2; -*-
// vi: set ts=2:
//

#include <BALL/FORMAT/commandlineParser.h>
#include <BALL/QSAR/registry.h>
#include <BALL/QSAR/configIO.h>
#include <fstream>
#include "version.h"

using namespace BALL::QSAR;
using namespace BALL;
using namespace std;

void startPrediction(PredictionConfiguration& conf, QSARData* q, String* data_filename);


void startPrediction(ifstream& in, QSARData* q, String* data_filename)
{
	PredictionConfiguration conf = ConfigIO::readPredictionConfiguration(&in);
	if(conf.done) return; // stop processing this section

	startPrediction(conf,q,data_filename);
}


void startPrediction(PredictionConfiguration& conf, QSARData* q, String* data_filename)
{
	bool created_data_object=0;
	if(q==NULL || data_filename==NULL || conf.data!=*data_filename)
	{
		if(q==NULL)
		{
			q = new QSARData;
			created_data_object=1;
		}
		q->readFromFile(conf.data);
		if(data_filename) *data_filename = conf.data;
	}

	Registry reg;
	Model* m;
	String model_type;
	ifstream model_input(conf.model.c_str()); // read model-abbreviation
	if(!model_input)
	{
		Log.error()<<"Error: Model-file '"<<conf.model<<"' does not exist!!"<<endl;
		return;
	}
	getline(model_input,model_type);
	getline(model_input,model_type);
	model_type = model_type.getField(0,"\t");
	model_input.close();

	RegistryEntry* entry = reg.getEntry(model_type);

	if(!entry->kernel)
	{
		m = (*entry->create)(*q);
	}
	else
	{
		// parameters irrelevant; will be overwritten by those read from file
		m = (*entry->createKernel1)(*q,1,1, -1);
	}

	m->readFromFile(conf.model.c_str()); // do NOT train again (done by ModelCreator) !!

	m->model_val->selectStat(conf.statistic);
	m->model_val->testInputData(1);	// calculate prediction quality
	m->model_val->setCVRes(m->model_val->getFitRes());
	m->model_val->saveToFile(conf.output);

	ofstream out(conf.output.c_str(),ios::app);
	out<<endl<<"[Predictions]"<<endl;
	int no_act = q->getNoResponseVariables();
	int no_cols = no_act;
	if(conf.print_expected)
	{
		no_cols*=2;
		out<<"# format: predition0, expectation0, ..."<<endl;
	}
	out<<"expected_values = "<<conf.print_expected<<endl;
	out<<"dimensions = "<<q->getNoSubstances()<<" "<<no_cols<<endl;

	for(unsigned int i=0;i<q->getNoSubstances();i++)
	{
		vector<double>* v = q->getSubstance(i); // get UNcentered descriptor-vector of test compound

		Eigen::VectorXd res = m->predict(*v,1); // transform val. data according to centering of training data
		delete v;

		vector<double>* exp = q->getActivity(i); // get UNcentered response value vector
		//for(int j=1; j<=res.getSize();j++) getSize is part of MATH/LinAlg and was replaced by rows() in Eigen
		for(int j=1; j<=res.rows();j++)
		{
			out<<res(j)<<"\t";
			if(conf.print_expected)
			{
				out<<(*exp)[j-1]<<"\t";
			}
		}
		delete exp;
		out<<endl;
	}

	if(created_data_object) delete q;
	delete m;
}


#ifndef EXT_MAIN
int main(int argc, char* argv[])
{
	CommandlineParser par("Predictor","predict activities with QSAR model", VERSION ,String(__DATE__), "QuEasy (QSAR)");
	par.registerParameter("i","input mod-file",INFILE,true);
	par.registerParameter("dat","data-file containing prediction data set",INFILE,true);
	par.registerParameter("o","output text file",OUTFILE,true);

	String man = "This tool predictes the response values of compounds in the given data-file using the specified QSAR model.\n\nInput of this tool is a model-file as generated by ModelCreator or FeatureSelector and a data-file generated by InputReader.\n\nOutput of this tool (as specified by '-o') is a text file containing the predicted and, if any, the expected response values in one column each.\nIf you would prefer to use molecule files (sdf,mol2,drf) for input and output, please use the tool MolPredictor instead of this one.";
	par.setToolManual(man);
	par.setSupportedFormats("i","mod");
	par.setSupportedFormats("dat","dat");
	par.setSupportedFormats("o","txt");
	par.parse(argc,argv);

	Registry reg;
	PredictionConfiguration conf;
	conf.model = par.get("i");
	conf.data = par.get("dat");
	conf.output = par.get("o");
	conf.print_expected = true;

	startPrediction(conf,0,0);
}
#endif
