//  This file is distributed as part of the bit-babbler package.
//  Copyright 2003 - 2016,  Ron <ron@debian.org>

#ifndef _BB_EXCEPTIONS_H
#define _BB_EXCEPTIONS_H

#include <string>
#include <cxxabi.h>
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <cstdarg>
#include <cerrno>


#define BB_NORETURN __attribute__ ((noreturn))


#if EM_PLATFORM_MSW

 // The format checking misfires badly in mingw-w64 4.9.2-21+15.4
 // and complains about format characters that are supported, so
 // just disable it there until that gets fixed.
 #define BB_PRINTF_FORMAT( fmt, arg1, ... )

#else

 #define BB_PRINTF_FORMAT( fmt, arg1, ... )                \
        __attribute__ ((format (__printf__,fmt,arg1), ##__VA_ARGS__))

#endif


namespace BitB
{

    static inline int Vasprintf( char **strp, const char *format, va_list arglist )
    { //{{{

      #if HAVE_VASPRINTF

        return vasprintf( strp, format, arglist );

      #else
        // We can implement this in alternative ways if needed
        #error "vasprintf is not supported"
      #endif

    } //}}}


    class Exception : public std::exception
    { //{{{
    private:

        std::string     m_msg;


    public:

        Exception() throw() {}

        Exception( const std::string &msg ) throw()
            : m_msg( msg )
        {}

        BB_PRINTF_FORMAT(2,3)
        Exception( const char *format, ... ) throw()
        {
            va_list     arglist;

            va_start( arglist, format );
            SetMessage( format, arglist );
            va_end( arglist );
        }

        // Default copy ctor and assignment oper ok.

        ~Exception() throw() {}


        void SetMessage( const std::string &msg ) throw()
        {
            m_msg = msg;
        }

        void SetMessage( const char *format, va_list args ) throw()
        {
            char    *msg = NULL;

            if( Vasprintf( &msg, format, args ) >= 0 )
            {
                m_msg = msg;
                free( msg );
            }
            else
                m_msg.append( " *** Error in BitB::Exception::SetMessage" );
        }

        BB_PRINTF_FORMAT(2,3)
        void SetMessage( const char *format, ... ) throw()
        {
            va_list     arglist;

            va_start( arglist, format );
            SetMessage( format, arglist );
            va_end( arglist );
        }

        const char *what() const throw()
        {
            return m_msg.empty() ? "Unspecified BitB::Exception" : m_msg.c_str();
        }

    }; //}}}


    class Error : public Exception
    { //{{{
    public:

        Error() throw() {}

        Error( const std::string &msg ) throw()
            : Exception( msg )
        {}

        BB_PRINTF_FORMAT(2,3)
        Error( const char *format, ... ) throw()
        {
            va_list     arglist;

            va_start( arglist, format );
            SetMessage( format, arglist );
            va_end( arglist );
        }

    }; //}}}


    class SystemError : public Error
    { //{{{
    private:

        int     m_errno;


    public:

        SystemError() throw()
            : m_errno( errno )
        {
            SetMessage( "System Error: %s", strerror( m_errno ) );
        }

        SystemError( const std::string &msg ) throw()
            : Error( msg + ": " + strerror(errno) )
            , m_errno( errno )
        {}

        BB_PRINTF_FORMAT(2,3)
        SystemError( const char *format, ... ) throw()
            : m_errno( errno )
        {
            va_list         arglist;
            std::string     msg( format );

            msg.append( ": " ).append( strerror(m_errno) );

            va_start( arglist, format );
            SetMessage( msg.c_str(), arglist );
            va_end( arglist );
        }

        BB_PRINTF_FORMAT(3,4)
        SystemError( int code, const char *format, ... ) throw()
            : m_errno( code )
        {
            va_list         arglist;
            std::string     msg( format );

            msg.append( ": " ).append( strerror(m_errno) );

            va_start( arglist, format );
            SetMessage( msg.c_str(), arglist );
            va_end( arglist );
        }


        int GetErrorCode() const { return m_errno; }

    }; //}}}

}

#endif  // _BB_EXCEPTIONS_H

// vi:sts=4:sw=4:et:foldmethod=marker
