# -*- coding: utf-8 -*-

# Bluemindo: A really simple but powerful audio player in Python/PyGTK.
# Copyright (C) 2007-2009  Erwan Briand

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3 of the License.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from os.path import join
from random import shuffle as random_shuffle
from ConfigParser import ConfigParser, NoSectionError
from gobject import (TYPE_STRING as gString, markup_escape_text)
from gtk.glade import XML as glade_XML
from gtk import (ListStore, TreeViewColumn, CellRendererText,
                 TREE_VIEW_COLUMN_FIXED, MenuItem)

from common.functions import Functions
from common.playlists import Playlists

from modules.explorer.editsong import EditSong

from modules.explorer import viewscommon
unescape = viewscommon.unescape

class Playlist(object):
    ref = None 
    ref2 = None 

    def __new__(cls, *args, **kws):
        # Singleton
        if cls.ref is None:
            cls.ref = super(Playlist, cls).__new__(cls, args, kws)
        return cls.ref

    def __init__(self, glade_file, config, userconf):
        glade_file.get_widget('vpaned1').show()
        glade_file.get_widget('scrolledwindow3').show()
        glade_file.get_widget('toolbar4').show()

        if Playlist.ref2 is None: 
            Playlist.ref2 = 42 
            self.launch(glade_file, config, userconf)

    def launch(self, glade_file, config, userconf):
        self.glade_file = glade_file
        self.config = config
        self.userconf = userconf
        self.extensions = self.config['__extensions']

        self.functions = Functions()
        self.playlists_management = Playlists()
        self.column_status = {0: True, 1: True, 2: True,
                              3: True, 4: True, 5: True,
                              6: True, 7: True
                             }

        # Show related widgets
        self.glade_file.get_widget('vpaned1').show()
        self.glade_file.get_widget('hbox6').hide()
        self.glade_file.get_widget('toolbar4').show()

        # Customize the TreeView
        playlist_win_tree = self.glade_file.get_widget('treeview2')
        playlist_win_tree.set_rules_hint(True)
        playlist_win_tree.connect('row-activated', self.load_song)

        # Right click context menu
        playlist_win_tree.connect('button-release-event', self.context_menu,
                                  playlist_win_tree.get_selection())

        # Create the TreeStore
        self.liststore = ListStore(gString, gString, gString, gString, gString,
                                   gString, gString, gString, gString)

        # Create the TreeModelFilter
        self.liststore_filter = self.liststore.filter_new(None)
        self.liststore_filter.set_visible_func(self.filter_visible_songs)

        # Set this model
        playlist_win_tree.set_model(self.liststore_filter)
        playlist_win_tree.set_headers_clickable(True)

        # Playlist buttons
        btn_clear = self.glade_file.get_widget('tool-clear')
        btn_shuffle = self.glade_file.get_widget('tool-shuffle')
        btn_repeat = self.glade_file.get_widget('tool-repeat')

        # Playlist filter
        combo = self.glade_file.get_widget('combobox1')
        combo.append_text(_('Title'))
        combo.append_text(_('Artist'))
        combo.append_text(_('Album'))
        combo.set_active(0)

        self.glade_file.get_widget('entry1').connect('key-release-event',
                                                      self.playlist_filter)
        self.glade_file.get_widget('toolbutton5').connect('clicked',
                                                           self.clearfilter)

        # Get repeat mode
        if self.config['repeat']:
            self.is_repeat = True
        else:
            self.is_repeat = False

        # Active or deactive repeat button
        if self.is_repeat:
            btn_repeat.set_active(True)
        else:
            btn_repeat.set_active(False)

        # GTK handlers
        btn_clear.connect('clicked', self.clear_playlist)
        btn_shuffle.connect('clicked', self.shuffle_playlist)
        btn_repeat.connect('clicked', self.repeat)

        # This function add a column to the playlist treeview
        def add_column(infos, show, column_name):
        
            # This function calculate the real width of a column
            def width_calculator(percent):
                return int(playlist_win_tree.get_allocation().width /
                           100. * percent)

            title = infos[0]
            expand = infos[1]
            width = width_calculator(infos[2])
            position = infos[3]

            column = TreeViewColumn()
            column.set_title(title)
            self.columns_dict[column_name] = column

            if expand:
                column.set_min_width(width)
                column.set_expand(True)
                column.set_sizing(TREE_VIEW_COLUMN_FIXED)
            else:
                column.set_min_width(width)

            render_text = CellRendererText()
            column.pack_start(render_text, expand=True)
            column.add_attribute(render_text, 'markup', position)
            column.set_clickable(True)
            column.connect('clicked', self.reorder_by_column, position)

            if not show:
                column.set_visible(False)

            playlist_win_tree.append_column(column)

        # Get playlist configuration
        columns_ = self.config['columns']
        show_columns = columns_.split(',')

        # Column settings
        # For each colum, a list contains: title, expand, width and position
        columns_infos = {'track': [_('#'), False, 4, 0],
                         'title': [_('Title'), True, 31, 1],
                         'artist': [_('Artist'), True, 20, 2],
                         'album': [_('Album'), True, 20, 3],
                         'genre': [_('Genre'), True, 14, 4],
                         'comment': [_('Comment'), True, 14, 5],
                         'year': [_('Year'), False, 8, 6],
                         'length': [_('Length'), False, 9, 7],
                         '__infos': ['', False, 0, 8]
                        }
        self.columns_list = ['track', 'title', 'artist', 'album', 'genre',
                             'comment', 'year', 'length', '__infos']
        self.columns_dict = {'track': None, 'title': None, 'artist': None,
                             'album': None, 'genre': None, 'comment': None,
                             'year': None, 'length': None, '__infos': None}

       # Add columns
        for column in self.columns_list:
            if column in show_columns:
                add_column(columns_infos[column], True, column)
            else:
                add_column(columns_infos[column], False, column)

    def update_columns(self, cols):
        """This function updates the hidden columns."""
        show = cols.split(',')

        for col in self.columns_list:
            if col in show:
                self.columns_dict[col].set_visible(True)
            else:
                self.columns_dict[col].set_visible(False)

    def filter_visible_songs(self, model, iter_):
        """This function filter visible songs."""
        if len(model) > 0:
            search = self.glade_file.get_widget('entry1').get_text()
            typ = self.glade_file.get_widget('combobox1').get_active()

            if typ:    
                searchin = model.get_value(iter_, typ + 1)
            else:
                searchin = model.get_value(iter_, 1)

            if search and searchin:
                if search.upper() in searchin.upper():
                    return True
            else:
                return True

    def playlist_filter(self, entry, event):
        """This function changes the filter."""
        self.liststore_filter.refilter()

    def clearfilter(self, widget):
        """This function deletes the filter."""
        self.glade_file.get_widget('entry1').set_text('')
        self.liststore_filter.refilter()

    def clear_playlist(self, widget=None):
        """This function clears all songs in the playlist."""
        self.liststore.clear()
        self.named_playlist = None

    def shuffle_playlist(self, widget):
        """This function shuffles all songs in the playlist."""
        n = 0
        for a in self.liststore_filter:
            n += 1

        index = range(0, n)
        random_shuffle(index)

        # Reorder the TreeModel
        if len(index) > 1:
            self.liststore.reorder(index)

    def repeat(self, widget):
        """This function turns on/off the repeat mode."""
        _repeat = widget.get_active()

        try:
            cfgparser = ConfigParser()
            cfgparser.read(self.config['__config-file'])
            cfgparser.set(self.config['__module-name'], 'repeat', _repeat)
            cfgparser.write(open(self.config['__config-file'], 'w'))
        except NoSectionError:
            pass

        self.config['repeat'] = _repeat
        self.is_repeat = _repeat

    def reorder_by_column(self, treeviewcolumn, columnid):
        """This function reorders the playlist by column."""
        def add_zero(userdata):
            if userdata.isdigit():
                if int(userdata) < 10:
                    return '0' + userdata

            return userdata

        # Create a list with all songs
        songs = []
        for sg in self.liststore:
            songs.append((sg[0], sg[1], sg[2], sg[3], sg[4],
                          sg[5], sg[6], sg[7], sg[8]))

        # Sort the playlist
        songs.sort(lambda x,y:cmp(
                   add_zero(self.functions.clear_html(x[columnid])),
                   add_zero(self.functions.clear_html(y[columnid]))
        ))

        if self.column_status[columnid]:
            songs.reverse()
            self.column_status[columnid] = False
        else:
            self.column_status[columnid] = True

        # Empty-ise playlist
        self.liststore.clear()

        # Re-fill the playlist
        for sg in songs:
            self.liststore.append(sg)

    def context_menu(self, widget, event, selection):
        """Generate the right click context menu on the playlist."""
        if event.button == 3:
            (mod, iter_) = selection.get_selected()

            if iter_:
                # A song is selected
                infos = eval(unescape(mod.get_value(iter_, 8)))
                filename = infos[8]

                def get_lyrics(widget):
                    """Show lyrics of the song."""
                    self.extensions.load_event('OnLyricsRequest', infos)

                def edit_song(widget):
                    """Edit song's datas."""
                    song = eval(unescape(unicode(mod[iter_][8])))
                    editsong = EditSong()
                    editsong.edit_song_datas(song)

                def remove_item(widget):
                    """Remove a song from the playlist."""
                    if (self.named_playlist is not None and self.named_playlist
                        not in ('s50', 's100', 'a10', 'r50', 'r100')):
                        self.playlists_management.remove_item_from_playlist(
                                                 filename, self.named_playlist)

                    del mod[iter_]

                def add_in_playlist(widget, playlist_name):
                    """This function adds an item to a playlist."""
                    self.playlists_management.add_item_to_playlist(
                                              filename, playlist_name)

                widgets = glade_XML(join(self.functions.datadir, 'glade',
                          'playlistmenu.glade'), 'menu1', domain='bluemindo')

                menu = widgets.get_widget('menu1')
                menu.attach_to_widget(widget, None)
                menu.popup(None, None, None, event.button, event.time)

                widgets.get_widget('menu-play').connect('activate',
                                                         self.load_song,
                                                         None, None)

                widgets.get_widget('menu-playlist-remove').connect('activate',
                                                                   remove_item)

                widgets.get_widget('menu-playlist-lyrics').connect('activate',
                                                                    get_lyrics)

                widgets.get_widget('menu-playlist-edit').connect('activate',
                                                                  edit_song)

                playlist_menu = widgets.get_widget('menu-playlist-add_menu')
                for plist in self.playlists_management.get_playlists_list():
                    menu_item = MenuItem(label=plist, use_underline=False)
                    playlist_menu.append(menu_item)

                    menu_item.connect('activate', add_in_playlist, plist)
                    del menu_item

                menu.show_all()

                # Hide song edit for the moment
                widgets.get_widget('menu-playlist-edit').hide()

    def song_already_in(self, song):
        """Return TRUE if the song is already in the playlist."""
        for sg in self.liststore:
            if eval(unescape(sg[8])) == song:
                return True

    def add_songs(self, songs, orderise=True, named_playlist=None):
        """This function add a list of songs in the playlist."""
        def add_zero(userdata):
            if userdata.isdigit():
                if int(userdata) < 10:
                    return '0' + userdata

            return userdata

        stbar = self.glade_file.get_widget('statusbar1')
        ordered_songs = []
        n_length = 0
        n_occurs = 0

        start = '<span size="small">'
        end = '</span>'

        for song in songs:
            if not self.song_already_in(song):
                _title = start + markup_escape_text(song[0]) + end
                _artist = start + markup_escape_text(song[1]) + end
                _album = start + markup_escape_text(song[2]) + end
                _comment = start + markup_escape_text(song[3]) + end
                _genre = start + markup_escape_text(song[4]) + end
                _year = start + markup_escape_text(str(song[5])) + end
                _track = start + add_zero(str(song[6])) + end
                _length = start + self.functions.human_length(song[7]) + end

                ordered_songs.append((_track, _title, _artist, _album,
                                      _genre, _comment, _year, _length,
                                      markup_escape_text(str(song))
                                    ))

                n_length += song[7]
                n_occurs += 1

        if orderise:
            ordered_songs.sort(lambda x,y:cmp(x[0], y[0]))
            ordered_songs.sort(lambda x,y:cmp(x[3].lower(), y[3].lower()))
            ordered_songs.sort(lambda x,y:cmp(x[2].lower(), y[2].lower()))

        for song in ordered_songs:
            self.liststore.append(song)

        if named_playlist is not None:
            self.named_playlist = named_playlist
        else:
            self.named_playlist = None

        if n_occurs > 0:
            stbar.push(0, _('Added %(nbsongs)u songs (%(totalength)s).') % {
                         'nbsongs': n_occurs,
                         'totalength': self.functions.human_length(n_length)})

    def load_song(self, widget, rowid, column):
        """Load a song."""
        tview = self.glade_file.get_widget('treeview2')
        (mod, iter_) = tview.get_selection().get_selected()

        # Test if a song is selected
        if tview.get_selection().get_selected()[1]:
            song = eval(unescape(mod.get_value(iter_, 8)))
            self.extensions.load_event('OnPlayNewSong', song)