//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/View/SampleDesigner/SampleForm.cpp
//! @brief     Implements class SampleForm
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#include "GUI/View/SampleDesigner/SampleForm.h"
#include "GUI/Application/ApplicationSettings.h"
#include "GUI/Model/Sample/SampleItem.h"
#include "GUI/Support/Util/ActionFactory.h"
#include "GUI/View/Numeric/DoubleSpinBox.h"
#include "GUI/View/SampleDesigner/CompoundForm.h"
#include "GUI/View/SampleDesigner/CoreAndShellForm.h"
#include "GUI/View/SampleDesigner/LayerForm.h"
#include "GUI/View/SampleDesigner/MesocrystalForm.h"
#include "GUI/View/SampleDesigner/ParticleForm.h"
#include "GUI/View/SampleDesigner/ParticleLayoutForm.h"
#include "GUI/View/Tool/GroupBoxCollapser.h"
#include <QBoxLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPushButton>
#include <QTextEdit>

namespace {

//! Widget with a button to add a layer (the "add layer" buttons shown between layers)
class AddLayerWidget : public QWidget {
public:
    AddLayerWidget(QWidget* parent, LayerItem* layer, SampleEditorController* ec)
        : QWidget(parent)
        , m_layer(layer)
    {
        auto* l = new QHBoxLayout(this);
        l->setContentsMargins(0, 0, 0, 0);
        auto* btn = new QPushButton("Add layer", this);
        l->addStretch();
        l->addWidget(btn);
        l->addStretch();
        connect(btn, &QPushButton::clicked, [=]() { ec->addLayerItem(layer); });
    }

    LayerItem* m_layer;
};

} // namespace

SampleForm::SampleForm(QWidget* parent, SampleItem* sampleItem, SampleEditorController* ec)
    : QWidget(parent)
    , m_sampleItem(sampleItem)
    , m_ec(ec)
    , m_useAngstrom(false)
    , m_useRadiant(false)
{
    setObjectName("MultiLayerForm"); // important for style sheet addressing
    setAttribute(Qt::WA_StyledBackground, true);

    m_layout = new QVBoxLayout(this);

    auto* props = new QGroupBox(this);
    props->setTitle("Sample");
    FormLayouter layouter(props, ec);
    layouter.setContentsMargins(6, 6, 0, 6);

    auto* nameEdit = new QLineEdit(props);
    layouter.addRow("Name:", nameEdit);
    nameEdit->setText(sampleItem->sampleName());
    nameEdit->setFixedWidth(585);
    connect(nameEdit, &QLineEdit::textEdited, ec, &SampleEditorController::setSampleName);

    auto* descriptionEdit = new QTextEdit(props);
    descriptionEdit->setFixedWidth(585);
    descriptionEdit->setMaximumHeight(100);
    descriptionEdit->setAcceptRichText(false);
    descriptionEdit->setTabChangesFocus(true);
    descriptionEdit->setPlainText(sampleItem->description());
    layouter.addRow("Description:", descriptionEdit);
    connect(descriptionEdit, &QTextEdit::textChanged, [this, descriptionEdit]() {
        m_ec->setSampleDescription(descriptionEdit->toPlainText());
    });

    layouter.addValue(sampleItem->crossCorrLength());

    // Processing external field is not implemented yet, so temporary disable it (see issue #654)
    // layouter.addVector(sampleItem->externalField(), false);

    auto* collapser = GroupBoxCollapser::installIntoGroupBox(props, false);
    collapser->setExpanded(sampleItem->isExpandInfo());
    connect(collapser, &GroupBoxCollapser::toggled, this,
            [sampleItem](bool b) { sampleItem->setExpandInfo(b); });

    auto* showInRealspaceAction = ActionFactory::createShowInRealspaceAction(
        this, "sample", [this] { m_ec->requestViewInRealspace(m_sampleItem); });

    collapser->addAction(showInRealspaceAction);

    m_layout->addWidget(props);

    for (auto* layer : sampleItem->layerItems()) {
        m_layout->addWidget(new AddLayerWidget(this, layer, m_ec));
        m_layout->addWidget(new LayerForm(this, layer, m_ec));
    }
    m_layout->addWidget(new AddLayerWidget(this, nullptr, m_ec));
    m_layout->setSizeConstraint(QLayout::SetMinimumSize);
    m_layout->addStretch(1);
}

void SampleForm::showInlineEditButtons(bool b)
{
    m_showInlineEditButtons = b;
    updateRowVisibilities();
}

void SampleForm::onLayerAdded(LayerItem* layerItem)
{
    const int rowInMultiLayer = m_sampleItem->layerItems().indexOf(layerItem);

    const int rowInLayout = rowInMultiLayer * 2 + 1;

    m_layout->insertWidget(rowInLayout, new LayerForm(this, layerItem, m_ec));

    // same row => button is above!
    m_layout->insertWidget(rowInLayout, new AddLayerWidget(this, layerItem, m_ec));

    updateRowVisibilities();
}

void SampleForm::onLayerMoved(LayerItem* layerItem)
{
    LayerForm* wl = nullptr;
    AddLayerWidget* al = nullptr;
    for (int index = 0; index < m_layout->count(); index++) {
        if (auto* w = dynamic_cast<AddLayerWidget*>(m_layout->itemAt(index)->widget()))
            if (w->m_layer == layerItem) {
                al = w;
                m_layout->takeAt(index);
                break;
            }
    }

    for (int index = 0; index < m_layout->count(); index++) {
        if (auto* w = dynamic_cast<LayerForm*>(m_layout->itemAt(index)->widget()))
            if (w->layerItem() == layerItem) {
                wl = w;
                m_layout->takeAt(index);
                break;
            }
    }

    const int rowInMultiLayer = m_sampleItem->layerItems().indexOf(layerItem);
    const int rowInLayout = rowInMultiLayer * 2 + 1;

    m_layout->insertWidget(rowInLayout, wl);

    // same row => button is above!
    m_layout->insertWidget(rowInLayout, al);

    updateRowVisibilities();
}

void SampleForm::onAboutToRemoveLayer(LayerItem* layerItem)
{
    LayerForm* layerForm = nullptr;
    AddLayerWidget* addLayerWidget = nullptr;
    for (auto* c : findChildren<QWidget*>()) {
        if (auto* w = dynamic_cast<AddLayerWidget*>(c))
            if (w->m_layer == layerItem)
                addLayerWidget = w;

        if (auto* w = dynamic_cast<LayerForm*>(c)) {
            if (w->layerItem() == layerItem)
                layerForm = w;
        }
    }

    if (layerForm) {
        // delete editors which are subscribed to SessionItems
        GUI::Util::Layout::clearLayout(layerForm->layout());
        layerForm->hide();
        layerForm->setParent(nullptr); // so it is not findable in update routines
        layerForm->deleteLater();      // delete later (this is the sender)
    }


    delete addLayerWidget;
}

void SampleForm::updateRowVisibilities()
{
    for (auto* c : findChildren<QWidget*>()) {
        if (auto* w = dynamic_cast<LayerForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
        if (auto* w = dynamic_cast<ParticleLayoutForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
        if (auto* w = dynamic_cast<ParticleForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
        if (auto* w = dynamic_cast<CompoundForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
        if (auto* w = dynamic_cast<CoreAndShellForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
        if (auto* w = dynamic_cast<MesocrystalForm*>(c))
            w->enableStructureEditing(m_showInlineEditButtons);
    }

    showAddLayerButtons(m_showInlineEditButtons);

    for (auto* c : findChildren<LayerForm*>())
        c->updateLayerPositionDependentElements();
}


void SampleForm::updateUnits()
{
    const auto set = [](DoubleSpinBox* spinbox, Unit valueUnit, Unit displayUnit) {
        if (spinbox->baseUnit() == valueUnit)
            spinbox->setDisplayUnit(displayUnit);
    };

    for (auto* editor : findChildren<DoubleSpinBox*>()) {
        if (m_useAngstrom) {
            set(editor, Unit::nanometer, Unit::angstrom);
            set(editor, Unit::angstrom, Unit::angstrom);
            set(editor, Unit::nanometer2, Unit::angstrom2);
            set(editor, Unit::angstrom2, Unit::angstrom2);
            set(editor, Unit::nanometerMinus2, Unit::angstromMinus2);
            set(editor, Unit::angstromMinus2, Unit::angstromMinus2);
        } else {
            set(editor, Unit::nanometer, Unit::nanometer);
            set(editor, Unit::angstrom, Unit::nanometer);
            set(editor, Unit::nanometer2, Unit::nanometer2);
            set(editor, Unit::angstrom2, Unit::nanometer2);
            set(editor, Unit::nanometerMinus2, Unit::nanometerMinus2);
            set(editor, Unit::angstromMinus2, Unit::nanometerMinus2);
        }

        if (m_useRadiant) {
            set(editor, Unit::degree, Unit::radiant);
            set(editor, Unit::radiant, Unit::radiant);
        } else {
            set(editor, Unit::degree, Unit::degree);
            set(editor, Unit::radiant, Unit::degree);
        }
    }
    for (auto* label : findChildren<QLabel*>())
        LayerEditorUtil::updateLabelUnit(label);
}

void SampleForm::ensureVisible(QWidget* /*w*/)
{
    // #baLayerEditor implement ensureVisible
}

void SampleForm::setUseAngstrom(bool angstrom)
{
    m_useAngstrom = angstrom;
    appSettings->setDefaultUnitIsAngstrom(angstrom);
    updateUnits();
}

bool SampleForm::useAngstrom() const
{
    return m_useAngstrom;
}

void SampleForm::setUseRadiant(bool radiant)
{
    m_useRadiant = radiant;
    updateUnits();
}

bool SampleForm::useRadiant() const
{
    return m_useRadiant;
}

void SampleForm::showAddLayerButtons(bool show)
{
    for (auto* c : findChildren<QWidget*>())
        if (dynamic_cast<AddLayerWidget*>(c))
            c->setVisible(show);
}

LayerForm* SampleForm::findNextLayerForm(QWidget* w)
{
    while (w != nullptr && dynamic_cast<LayerForm*>(w) == nullptr) {
        const auto index = m_layout->indexOf(w);
        if (index + 1 < m_layout->count())
            w = m_layout->itemAt(index + 1)->widget();
        else
            return nullptr;
    }

    return dynamic_cast<LayerForm*>(w);
}
