package client

import (
	"context"
	"fmt"
	"io"
	"net/url"
	"time"

	"github.com/moby/moby/client/internal/timestamp"
)

// ServiceLogsOptions holds parameters to filter logs with.
type ServiceLogsOptions struct {
	ShowStdout bool
	ShowStderr bool
	Since      string
	Until      string
	Timestamps bool
	Follow     bool
	Tail       string
	Details    bool
}

// ServiceLogsResult holds the result of a service logs operation.
// It implements [io.ReadCloser].
// It's up to the caller to close the stream.
type ServiceLogsResult interface {
	io.ReadCloser
}

// ServiceLogs returns the logs generated by a service in a [ServiceLogsResult].
// as an [io.ReadCloser]. Callers should close the stream.
//
// The underlying [io.ReadCloser] is automatically closed if the context is canceled,
func (cli *Client) ServiceLogs(ctx context.Context, serviceID string, options ServiceLogsOptions) (ServiceLogsResult, error) {
	// TODO(thaJeztah): this function needs documentation about the format of ths stream (similar to for container logs)
	// TODO(thaJeztah): migrate CLI utilities to the client where suitable; https://github.com/docker/cli/blob/v29.0.0-rc.1/cli/command/service/logs.go#L73-L348

	serviceID, err := trimID("service", serviceID)
	if err != nil {
		return nil, err
	}

	query := url.Values{}
	if options.ShowStdout {
		query.Set("stdout", "1")
	}

	if options.ShowStderr {
		query.Set("stderr", "1")
	}

	if options.Since != "" {
		ts, err := timestamp.GetTimestamp(options.Since, time.Now())
		if err != nil {
			return nil, fmt.Errorf(`invalid value for "since": %w`, err)
		}
		query.Set("since", ts)
	}

	if options.Timestamps {
		query.Set("timestamps", "1")
	}

	if options.Details {
		query.Set("details", "1")
	}

	if options.Follow {
		query.Set("follow", "1")
	}
	query.Set("tail", options.Tail)

	resp, err := cli.get(ctx, "/services/"+serviceID+"/logs", query, nil)
	if err != nil {
		return nil, err
	}
	return &serviceLogsResult{
		ReadCloser: newCancelReadCloser(ctx, resp.Body),
	}, nil
}

type serviceLogsResult struct {
	io.ReadCloser
}

var (
	_ io.ReadCloser     = (*serviceLogsResult)(nil)
	_ ServiceLogsResult = (*serviceLogsResult)(nil)
)
