@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-association-put: Update/Create an Association
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use Term::ReadKey;
use File::Basename;
use Cwd 'abs_path';

my %opt = (
	gateway  => undef,
	password => 0,
	version  => 0,
	help     => 0,
	manual   => 0,
	retry	 => 0,
	debug    => 0
);

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  gateway=s
	  password
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

my $sourceFile = $ARGV[0];

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No source-file specified!\n" )             if not($sourceFile);
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "Source-file is not a plain text file!\n" ) if not( -f $sourceFile && -T $sourceFile );

# Prompt for a new password if required
my $password = '';
if ( $opt{password} ) {
	print "New password: ";
	ReadMode('noecho');
	$password = <STDIN>;
	ReadMode('normal');
	chomp($password);
	print "\n";
}

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# If no Gateway is specified then use the default Gateway
$opt{gateway} = $controlChannel->getGatewayName()->result if ( not $opt{gateway} );

# Load/check configuration from file
no warnings qw(once);
unless ( $return = do( abs_path($sourceFile) ) ) {
	die "Couldn't process $sourceFile: $@\n" if $@;
	die "Couldn't process $sourceFile: $!\n" unless defined $return;
	die "Couldn't process $sourceFile\n"     unless $return;
}

# Update the association
$controlChannel->putAssociation(
	$token,
	$opt{gateway},
	SOAP::Data->name(
		"request" => \SOAP::Data->value(
			SOAP::Data->name( 'name'            => $name ),
			SOAP::Data->name( 'password'        => $password ),
			SOAP::Data->name( 'active'          => $active eq 'yes' ? 'true' : 'false' ),
			SOAP::Data->name( 'comment'         => $comment ),
			SOAP::Data->name( 'grantedUserList' => $grantedUserList ),
			SOAP::Data->name( 'data'            => $data ),
			SOAP::Data->name( 'directory'       => $directory ),
			SOAP::Data->name( 'hostName'        => $hostName ),
			SOAP::Data->name( 'protocol'        => $protocol ),
			SOAP::Data->name( 'login'           => $login )
		)
	)
);

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-association-put - Update/Create an Association

=head1 SYNOPSIS

B<ecaccess-association-put -version|-help|-manual>

B<ecaccess-association-put [-debug] [-gateway> I<name>B<] [-password]> I<source-file>

=head1 DESCRIPTION

Allow updating or creating an ECtrans Association. If the name exists then the Association is updated. If
it does not exists then a new Association is created. In order to get a template for a new Association please
use the following command:

B<ecaccess-association-get -template> I<association-name> I<target-file>

The I<target-file> for the Association Descriptive File does not have to be the name of the Association
itself. The name for the Association is specified in the I<association-name> parameter.

When creating a new Association the B<-password> option should be used to set the password (it can not be set
in the Association Descriptive File as this would not be secure).

=head1 ARGUMENTS

=over 8

=item I<source-file>

The name of the Association Descriptive File to upload.

=back

=head1 OPTIONS

=over 8

=item B<-gateway> I<name>

This is the name of the ECaccess Gateway where the Association should be installed.
It is by default the Gateway you are connected to. In order to get the name
of your current Gateway you can use the B<ecaccess-gateway-name> command. When
using the commands at ECMWF the default Gateway is always "ecaccess.ecmwf.int".

=item B<-password>

Allow prompting for a new password for the Association.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-association-put -password> I<./test>

Push the Association described in the I<./test> file of your current directory on your default Gateway. You will
be also prompted for a new password.

B<ecaccess-association-put -gateway> I<ecaccess.ecmwf.int> I<./test>

Push the Association described in the I<./test> file of your current directory on the I<ecaccess.ecmwf.int> Gateway.

=head1 SEE ALSO

B<ecaccess-association-delete>, B<ecaccess-association-get>, B<ecaccess-association-list>,
B<ecaccess-association-protocol> and B<ecaccess>.

=cut

__END__
:endofperl
