import random
import freeorion as fo
import util


# tuple of galaxy shapes to randomly choose from when shape is "random"
shapes = (fo.galaxyShape.spiral2,    fo.galaxyShape.spiral3,     fo.galaxyShape.spiral4,
          fo.galaxyShape.cluster,    fo.galaxyShape.elliptical,  fo.galaxyShape.ring,
          fo.galaxyShape.irregular1, fo.galaxyShape.irregular2)


class AdjacencyGrid:
    def __init__(self, universe_width):
        self.min_dist = fo.min_system_separation()
        self.cell_size = max(universe_width / 50, self.min_dist)
        self.width = int(universe_width / self.cell_size) + 1
        self.grid = [[[] for _ in range(self.width)] for _ in range(self.width)]
        print "Adjacency Grid: width", self.width, ", cell size", self.cell_size

    def insert_pos(self, pos):
        self.grid[int(pos.x / self.cell_size)][int(pos.y / self.cell_size)].append(pos)

    def too_close_to_other_positions(self, x, y):
        """
        Checks if the specified position is too close to the positions stored in the grid
        """
        cell_x = int(x / self.cell_size)
        cell_y = int(y / self.cell_size)
        upper_left_x = max(0, cell_x - 1)
        upper_left_y = max(0, cell_y - 1)
        lower_right_x = min(self.width - 1, cell_x + 1)
        lower_right_y = min(self.width - 1, cell_y + 1)
        return any(util.distance(pos.x, pos.y, x, y) < self.min_dist for cx in range(upper_left_x, lower_right_x + 1)
                   for cy in range(upper_left_y, lower_right_y + 1) for pos in self.grid[cx][cy])


def get_systems_within_jumps(origin_system, jumps):
    """
    Returns all systems within jumps jumps of system origin_system (including origin_system).
    If jumps is 0, return list that only contains system origin_system.
    If jumps is negative, return empty list.
    """
    if jumps < 0:
        return []
    matching_systems = [origin_system]
    next_origin_systems = [origin_system]
    while jumps > 0:
        origin_systems = list(next_origin_systems)
        next_origin_systems = []
        for system in origin_systems:
            neighbor_systems = [s for s in fo.sys_get_starlanes(system) if s not in matching_systems]
            next_origin_systems.extend(neighbor_systems)
            matching_systems.extend(neighbor_systems)
        jumps -= 1
    return matching_systems


def irregular2_galaxy_calc_positions(positions, size, width):
    """
    Calculate positions for the 'Python Test' galaxy shape
    """
    adjacency_grid = AdjacencyGrid(width)
    max_delta = max(min(float(fo.max_starlane_length()), width / 10.0), adjacency_grid.min_dist * 2.0)
    print "Irregular2 galaxy shape: max delta distance =", max_delta
    origin_x, origin_y = width / 2.0, width / 2.0
    prev_x, prev_y = origin_x, origin_y
    reset_to_origin = 0
    for n in range(size):
        attempts = 100
        found = False
        while (attempts > 0) and not found:
            attempts -= 1
            x = prev_x + random.uniform(-max_delta, max_delta)
            y = prev_y + random.uniform(-max_delta, max_delta)
            if util.distance(x, y, origin_x, origin_y) > width * 0.45:
                prev_x, prev_y = origin_x, origin_y
                reset_to_origin += 1
                continue
            found = not adjacency_grid.too_close_to_other_positions(x, y)
            if attempts % 10:
                prev_x, prev_y = x, y
        if found:
            pos = fo.SystemPosition(x, y)
            adjacency_grid.insert_pos(pos)
            positions.append(pos)
        prev_x, prev_y = x, y
    print "Reset to origin", reset_to_origin, "times"


def recalc_universe_width(positions):
    """
    Recalculates the universe width. This is done by shifting all positions by a delta so too much "extra space"
    beyond the uppermost, lowermost, leftmost and rightmost positions is cropped, and adjust the universe width
    accordingly.

    Returns the new universe width and the recalculated positions.
    """
    print "Recalculating universe width..."
    # first, get the uppermost, lowermost, leftmost and rightmost positions
    # (these are those with their x or y coordinate closest to or farthest away from the x or y axis)
    min_x = min(positions, key=lambda p: p.x).x
    max_x = max(positions, key=lambda p: p.x).x
    min_y = min(positions, key=lambda p: p.y).y
    max_y = max(positions, key=lambda p: p.y).y
    print "...the leftmost system position is at x coordinate", min_x
    print "...the uppermost system position is at y coordinate", min_y
    print "...the rightmost system position is at x coordinate", max_x
    print "...the lowermost system position is at y coordinate", max_y

    # calculate the actual universe width by determining the width and height of an rectangle that encompasses all
    # positions, and take the greater of the two as the new actual width for the universe
    # also add a constant value to the width so we have some small space around
    width = max_x - min_x
    height = max_y - min_y
    actual_width = max(width, height) + 20.0
    print "...recalculated universe width:", actual_width

    # shift all positions so the entire map is centered in a quadratic box of the width we just calculated
    # this box defines the extends of our universe
    delta_x = ((actual_width - width) / 2) - min_x
    delta_y = ((actual_width - height) / 2) - min_y
    print "...shifting all system positions by", delta_x, "/", delta_y
    new_positions = fo.SystemPositionVec()
    for position in positions:
        new_positions.append(fo.SystemPosition(position.x + delta_x, position.y + delta_y))

    print "...the leftmost system position is now at x coordinate", min(new_positions, key=lambda p: p.x).x
    print "...the uppermost system position is now at y coordinate", min(new_positions, key=lambda p: p.y).y
    print "...the rightmost system position is now at x coordinate", max(new_positions, key=lambda p: p.x).x
    print "...the lowermost system position is now at y coordinate", max(new_positions, key=lambda p: p.y).y

    return actual_width, new_positions


def calc_star_system_positions(shape, size):
    """
    Calculates list of positions (x, y) for a given galaxy shape,
    number of systems and width
    Uses universe generator helper functions provided by the API
    """

    # calculate typical width for universe based on number of systems
    width = fo.calc_typical_universe_width(size)
    if shape == fo.galaxyShape.irregular2:
        width *= 1.4
    if shape == fo.galaxyShape.elliptical:
        width *= 1.4
    print "Set universe width to", width
    fo.set_universe_width(width)

    positions = fo.SystemPositionVec()
    if shape == fo.galaxyShape.random:
        shape = random.choice(shapes)

    if shape == fo.galaxyShape.spiral2:
        fo.spiral_galaxy_calc_positions(positions, 2, size, width, width)
    elif shape == fo.galaxyShape.spiral3:
        fo.spiral_galaxy_calc_positions(positions, 3, size, width, width)
    elif shape == fo.galaxyShape.spiral4:
        fo.spiral_galaxy_calc_positions(positions, 4, size, width, width)
    elif shape == fo.galaxyShape.elliptical:
        fo.elliptical_galaxy_calc_positions(positions, size, width, width)
    elif shape == fo.galaxyShape.cluster:
        # Typically a galaxy with 100 systems should have ~5 clusters
        avg_clusters = size / 20
        if avg_clusters < 2:
            avg_clusters = 2
        # Add a bit of random variation (+/- 20%)
        clusters = random.randint((avg_clusters * 8) / 10, (avg_clusters * 12) / 10)
        if clusters >= 2:
            fo.cluster_galaxy_calc_positions(positions, clusters, size, width, width)
    elif shape == fo.galaxyShape.ring:
        fo.ring_galaxy_calc_positions(positions, size, width, width)
    elif shape == fo.galaxyShape.irregular2:
        irregular2_galaxy_calc_positions(positions, size, width)

    # Check if any positions have been calculated...
    if not positions:
        # ...if not, fall back on irregular1 shape
        fo.irregular_galaxy_positions(positions, size, width, width)

    # to avoid having too much "extra space" around the system positions of our galaxy map, recalculate the universe
    # width and shift all positions accordingly
    width, positions = recalc_universe_width(positions)
    print "Set universe width to", width
    fo.set_universe_width(width)

    return positions
