# Copyright 2010 Mackenzie Morgan <maco.m@ubuntu.com>
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

import sys, os 
from xml.dom.minidom import parse

from PyKDE4.kdecore import KLocale, KGlobal

from gally.Lesson import Lesson, Sign
from gally.Language import Language
from gally.Quiz import Quiz, Question, Option


class LessonParser():
    """ Parse .lang files defining language lessons and lesson signs.

        lesson.lang: list of lessons for a language
        signs.lang: list of signs in a lesson
    """
    def __init__(self, langList):
        self.langs = langList

    def parseLanguageFile(self, path):
        """ Return Language object.

            Parameter path is path to language in filesystem
            Parses lessons.lang for language
        """
        langFile = path+"/lessons.lang"
        #print langFile
        dom = parse(langFile)
        if len(dom.getElementsByTagName("language")) > 0:
            langName = self.getTranslatedText(dom, "language")
        else:
            langName = ""
        #print "Language: "+langName
        
        lessonNodes = dom.getElementsByTagName("lesson")
        #print "Number of Lessons: "+str(lessonNodes.length)

        lessons = []
        for node in lessonNodes:
            num = node.getAttribute('id')
            if len(node.getElementsByTagName("subject")) > 0:
                subject = self.getTranslatedText(node, "subject")
            else:
                subject = ""
            dir = os.path.join(path,node.getElementsByTagName("dir")[0].childNodes[0].nodeValue)
            lesson = Lesson(int(num), subject, dir)
            lessons.append(lesson)


        dom.unlink()
        directory = path
        langInfo = Language(langName, lessonNodes.length, lessons, directory)
        #print "Lessons:"
        #langInfo.printLessonList()
        return langInfo


    def parseLessonFile(self, path, num):
        """ Modify lesson object to include list of signs to be learned.


            Parameter path to lesson's directory in filesystem
            Parameter num is lesson number within language
            Parses signs.lang for lesson
        """
        langInfo = self.parseLanguageFile(path)
        lessons = langInfo.getLessonsList()
        less = lessons[num-1] # because arrays start at 0
        dir = less.dir
        lessonFile = os.path.join(dir,"signs.lang")
        #print lessonFile
        dom = parse(lessonFile)
        signList = []

        if len(dom.getElementsByTagName("intro")) > 0:
            less.intro = self.getTranslatedText(dom, "intro")
        else:
            less.intro = ""

        signs = dom.getElementsByTagName("lesson")
        if (signs):
            for sign in signs:
                if len(sign.getElementsByTagName("sign")) > 0:
                    signName = self.getTranslatedText(sign, "sign")
                else:
                    signName = ""

                displayType = sign.getAttribute("type")
                if (sign.getElementsByTagName("context")[0].hasChildNodes()):
                    hasContext = True
                    context = os.path.join(dir,sign.getElementsByTagName("context")[0].childNodes[0].nodeValue)
                else:
                    hasContext = False 
                    context = ""
                if (sign.getElementsByTagName("demo")[0].hasChildNodes()):
                    filename = sign.getElementsByTagName("demo")[0].childNodes[0].nodeValue
                    full_filename = os.path.join(dir,filename)
                else:
                    full_filename=""
                if len(sign.getElementsByTagName("text")) > 0:
                    text = self.getTranslatedText(sign, "text")
                else:
                    text = ""
                signList.append(Sign(signName, displayType, hasContext, context, full_filename, text))
    
            less.signs = signList

        quizSigns = dom.getElementsByTagName("quiz")
        if (quizSigns):
            questionList = []
            for sign in quizSigns:
                vid = sign.getElementsByTagName("vid")[0].childNodes[0].nodeValue
                options = []
                for i in range(4):
                    word = sign.getElementsByTagName("option")[i].childNodes[0].nodeValue
                    value = sign.getElementsByTagName("option")[i].getAttribute('value')
                    opt = Option(word,value)
                    options.append(opt)

                questionList.append(Question(vid,options))
            less.quiz = Quiz(questionList,len(questionList))

        dom.unlink()
        return less

    def getTranslatedText(self, element, thing):
        texts = element.getElementsByTagName(thing)
        transTexts = {}
        for text in texts:
            if text.hasChildNodes():
                if text.hasAttribute("lang"):
                    transTexts[text.getAttribute("lang")] = text.childNodes[0].nodeValue
                else:
                    transTexts["NONE"] = text.childNodes[0].nodeValue
        transText = None
        for lang in self.langs:
            if lang in transTexts:
                transText = transTexts[lang]
                break
        if transText is None:
            if "en" in transTexts:
                transText = transTexts["en"]
            elif "NONE" in transTexts:
                transText = transTexts["NONE"]
            else:
                transText = ""
        return transText
                        
