/*
** Copyright (c) 2018-2023 Valve Corporation
** Copyright (c) 2018-2023 LunarG, Inc.
** Copyright (c) 2023 Advanced Micro Devices, Inc.
**
** Permission is hereby granted, free of charge, to any person obtaining a
** copy of this software and associated documentation files (the "Software"),
** to deal in the Software without restriction, including without limitation
** the rights to use, copy, modify, merge, publish, distribute, sublicense,
** and/or sell copies of the Software, and to permit persons to whom the
** Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in
** all copies or substantial portions of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
** FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
** DEALINGS IN THE SOFTWARE.
*/

/*
** This file is generated from the Khronos Vulkan XML API Registry.
**
*/

#include "generated/generated_vulkan_api_call_encoders.h"

#include "encode/custom_vulkan_encoder_commands.h"
#include "encode/custom_vulkan_array_size_2d.h"
#include "encode/parameter_encoder.h"
#include "encode/struct_pointer_encoder.h"
#include "encode/vulkan_capture_manager.h"
#include "encode/vulkan_handle_wrapper_util.h"
#include "encode/vulkan_handle_wrappers.h"
#include "format/api_call_id.h"
#include "generated/generated_vulkan_command_buffer_util.h"
#include "generated/generated_vulkan_struct_handle_wrappers.h"
#include "util/defines.h"

#include "vulkan/vulkan.h"
#include "vk_video/vulkan_video_codec_h264std.h"
#include "vk_video/vulkan_video_codec_h264std_decode.h"
#include "vk_video/vulkan_video_codec_h264std_encode.h"
#include "vk_video/vulkan_video_codec_h265std.h"
#include "vk_video/vulkan_video_codec_h265std_decode.h"
#include "vk_video/vulkan_video_codec_h265std_encode.h"
#include "vk_video/vulkan_video_codecs_common.h"

GFXRECON_BEGIN_NAMESPACE(gfxrecon)
GFXRECON_BEGIN_NAMESPACE(encode)

VKAPI_ATTR VkResult VKAPI_CALL CreateInstance(
    const VkInstanceCreateInfo*                 pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkInstance*                                 pInstance)
{
    auto api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateInstance>::Dispatch(VulkanCaptureManager::Get(), pCreateInfo, pAllocator, pInstance);

    VkResult result = VulkanCaptureManager::OverrideCreateInstance(pCreateInfo, pAllocator, pInstance);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = VulkanCaptureManager::Get()->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateInstance);
    if (encoder)
    {
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<InstanceWrapper>(pInstance, omit_output_data);
        encoder->EncodeEnumValue(result);
        VulkanCaptureManager::Get()->EndCreateApiCallCapture<const void*, InstanceWrapper, VkInstanceCreateInfo>(result, nullptr, pInstance, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateInstance>::Dispatch(VulkanCaptureManager::Get(), result, pCreateInfo, pAllocator, pInstance);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyInstance(
    VkInstance                                  instance,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyInstance>::Dispatch(manager, instance, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyInstance);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<InstanceWrapper>(instance);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetInstanceTable(instance)->DestroyInstance(instance, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyInstance>::Dispatch(manager, instance, pAllocator);

    DestroyWrappedHandle<InstanceWrapper>(instance);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDevices(
    VkInstance                                  instance,
    uint32_t*                                   pPhysicalDeviceCount,
    VkPhysicalDevice*                           pPhysicalDevices)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDevices>::Dispatch(manager, instance, pPhysicalDeviceCount, pPhysicalDevices);

    VkResult result = GetInstanceTable(instance)->EnumeratePhysicalDevices(instance, pPhysicalDeviceCount, pPhysicalDevices);

    if (result >= 0)
    {
        CreateWrappedHandles<InstanceWrapper, NoParentWrapper, PhysicalDeviceWrapper>(instance, NoParentWrapper::kHandleValue, pPhysicalDevices, (pPhysicalDeviceCount != nullptr) ? (*pPhysicalDeviceCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkEnumeratePhysicalDevices);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeUInt32Ptr(pPhysicalDeviceCount, omit_output_data);
        encoder->EncodeHandleArray<PhysicalDeviceWrapper>(pPhysicalDevices, (pPhysicalDeviceCount != nullptr) ? (*pPhysicalDeviceCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndGroupCreateApiCallCapture<VkInstance, void*, PhysicalDeviceWrapper, void>(result, instance, nullptr, (pPhysicalDeviceCount != nullptr) ? (*pPhysicalDeviceCount) : 0, pPhysicalDevices, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDevices>::Dispatch(manager, result, instance, pPhysicalDeviceCount, pPhysicalDevices);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceFeatures*                   pFeatures)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures>::Dispatch(manager, physicalDevice, pFeatures);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFeatures(physicalDevice, pFeatures);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pFeatures);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures>::Dispatch(manager, physicalDevice, pFeatures);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkFormatProperties*                         pFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties>::Dispatch(manager, physicalDevice, format, pFormatProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFormatProperties(physicalDevice, format, pFormatProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        EncodeStructPtr(encoder, pFormatProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties>::Dispatch(manager, physicalDevice, format, pFormatProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkImageType                                 type,
    VkImageTiling                               tiling,
    VkImageUsageFlags                           usage,
    VkImageCreateFlags                          flags,
    VkImageFormatProperties*                    pImageFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties>::Dispatch(manager, physicalDevice, format, type, tiling, usage, flags, pImageFormatProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceImageFormatProperties(physicalDevice, format, type, tiling, usage, flags, pImageFormatProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        encoder->EncodeEnumValue(type);
        encoder->EncodeEnumValue(tiling);
        encoder->EncodeFlagsValue(usage);
        encoder->EncodeFlagsValue(flags);
        EncodeStructPtr(encoder, pImageFormatProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties>::Dispatch(manager, result, physicalDevice, format, type, tiling, usage, flags, pImageFormatProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceProperties*                 pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties>::Dispatch(manager, physicalDevice, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceProperties(physicalDevice, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties>::Dispatch(manager, physicalDevice, pProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pQueueFamilyPropertyCount,
    VkQueueFamilyProperties*                    pQueueFamilyProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    manager->OverrideGetPhysicalDeviceQueueFamilyProperties(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pQueueFamilyPropertyCount);
        EncodeStructArray(encoder, pQueueFamilyProperties, (pQueueFamilyPropertyCount != nullptr) ? (*pQueueFamilyPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceMemoryProperties*           pMemoryProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties>::Dispatch(manager, physicalDevice, pMemoryProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceMemoryProperties(physicalDevice, pMemoryProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pMemoryProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties>::Dispatch(manager, physicalDevice, pMemoryProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDevice(
    VkPhysicalDevice                            physicalDevice,
    const VkDeviceCreateInfo*                   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDevice*                                   pDevice)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDevice>::Dispatch(manager, physicalDevice, pCreateInfo, pAllocator, pDevice);

    VkResult result = manager->OverrideCreateDevice(physicalDevice, pCreateInfo, pAllocator, pDevice);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDevice);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DeviceWrapper>(pDevice, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkPhysicalDevice, DeviceWrapper, VkDeviceCreateInfo>(result, physicalDevice, pDevice, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDevice>::Dispatch(manager, result, physicalDevice, pCreateInfo, pAllocator, pDevice);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDevice(
    VkDevice                                    device,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDevice>::Dispatch(manager, device, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDevice);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DeviceWrapper>(device);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDevice(device, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDevice>::Dispatch(manager, device, pAllocator);

    DestroyWrappedHandle<DeviceWrapper>(device);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceQueue(
    VkDevice                                    device,
    uint32_t                                    queueFamilyIndex,
    uint32_t                                    queueIndex,
    VkQueue*                                    pQueue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceQueue>::Dispatch(manager, device, queueFamilyIndex, queueIndex, pQueue);

    GetDeviceTable(device)->GetDeviceQueue(device, queueFamilyIndex, queueIndex, pQueue);

    CreateWrappedHandle<DeviceWrapper, NoParentWrapper, QueueWrapper>(device, NoParentWrapper::kHandleValue, pQueue, VulkanCaptureManager::GetUniqueId);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceQueue);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeUInt32Value(queueIndex);
        encoder->EncodeHandlePtr<QueueWrapper>(pQueue);
        manager->EndCreateApiCallCapture<VkDevice, QueueWrapper, void>(VK_SUCCESS, device, pQueue, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceQueue>::Dispatch(manager, device, queueFamilyIndex, queueIndex, pQueue);
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit(
    VkQueue                                     queue,
    uint32_t                                    submitCount,
    const VkSubmitInfo*                         pSubmits,
    VkFence                                     fence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueSubmit>::Dispatch(manager, queue, submitCount, pSubmits, fence);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSubmitInfo* pSubmits_unwrapped = UnwrapStructArrayHandles(pSubmits, submitCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(queue)->QueueSubmit(queue, submitCount, pSubmits_unwrapped, fence);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueSubmit);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Value(submitCount);
        EncodeStructArray(encoder, pSubmits, submitCount);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueSubmit>::Dispatch(manager, result, queue, submitCount, pSubmits, fence);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueueWaitIdle(
    VkQueue                                     queue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueWaitIdle>::Dispatch(manager, queue);

    VkResult result = GetDeviceTable(queue)->QueueWaitIdle(queue);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueWaitIdle);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueWaitIdle>::Dispatch(manager, result, queue);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DeviceWaitIdle(
    VkDevice                                    device)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDeviceWaitIdle>::Dispatch(manager, device);

    VkResult result = GetDeviceTable(device)->DeviceWaitIdle(device);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDeviceWaitIdle);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDeviceWaitIdle>::Dispatch(manager, result, device);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateMemory(
    VkDevice                                    device,
    const VkMemoryAllocateInfo*                 pAllocateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDeviceMemory*                             pMemory)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAllocateMemory>::Dispatch(manager, device, pAllocateInfo, pAllocator, pMemory);

    VkResult result = manager->OverrideAllocateMemory(device, pAllocateInfo, pAllocator, pMemory);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkAllocateMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAllocateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DeviceMemoryWrapper>(pMemory, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DeviceMemoryWrapper, VkMemoryAllocateInfo>(result, device, pMemory, pAllocateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAllocateMemory>::Dispatch(manager, result, device, pAllocateInfo, pAllocator, pMemory);

    return result;
}

VKAPI_ATTR void VKAPI_CALL FreeMemory(
    VkDevice                                    device,
    VkDeviceMemory                              memory,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkFreeMemory>::Dispatch(manager, device, memory, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkFreeMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DeviceMemoryWrapper>(memory);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->FreeMemory(device, memory, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkFreeMemory>::Dispatch(manager, device, memory, pAllocator);

    DestroyWrappedHandle<DeviceMemoryWrapper>(memory);
}

VKAPI_ATTR VkResult VKAPI_CALL MapMemory(
    VkDevice                                    device,
    VkDeviceMemory                              memory,
    VkDeviceSize                                offset,
    VkDeviceSize                                size,
    VkMemoryMapFlags                            flags,
    void**                                      ppData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkMapMemory>::Dispatch(manager, device, memory, offset, size, flags, ppData);

    VkResult result = GetDeviceTable(device)->MapMemory(device, memory, offset, size, flags, ppData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkMapMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeVkDeviceSizeValue(size);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeVoidPtrPtr(ppData, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkMapMemory>::Dispatch(manager, result, device, memory, offset, size, flags, ppData);

    return result;
}

VKAPI_ATTR void VKAPI_CALL UnmapMemory(
    VkDevice                                    device,
    VkDeviceMemory                              memory)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkUnmapMemory>::Dispatch(manager, device, memory);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkUnmapMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->UnmapMemory(device, memory);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkUnmapMemory>::Dispatch(manager, device, memory);
}

VKAPI_ATTR VkResult VKAPI_CALL FlushMappedMemoryRanges(
    VkDevice                                    device,
    uint32_t                                    memoryRangeCount,
    const VkMappedMemoryRange*                  pMemoryRanges)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkFlushMappedMemoryRanges>::Dispatch(manager, device, memoryRangeCount, pMemoryRanges);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMappedMemoryRange* pMemoryRanges_unwrapped = UnwrapStructArrayHandles(pMemoryRanges, memoryRangeCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->FlushMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkFlushMappedMemoryRanges);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(memoryRangeCount);
        EncodeStructArray(encoder, pMemoryRanges, memoryRangeCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkFlushMappedMemoryRanges>::Dispatch(manager, result, device, memoryRangeCount, pMemoryRanges);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL InvalidateMappedMemoryRanges(
    VkDevice                                    device,
    uint32_t                                    memoryRangeCount,
    const VkMappedMemoryRange*                  pMemoryRanges)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkInvalidateMappedMemoryRanges>::Dispatch(manager, device, memoryRangeCount, pMemoryRanges);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMappedMemoryRange* pMemoryRanges_unwrapped = UnwrapStructArrayHandles(pMemoryRanges, memoryRangeCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->InvalidateMappedMemoryRanges(device, memoryRangeCount, pMemoryRanges_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkInvalidateMappedMemoryRanges);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(memoryRangeCount);
        EncodeStructArray(encoder, pMemoryRanges, memoryRangeCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkInvalidateMappedMemoryRanges>::Dispatch(manager, result, device, memoryRangeCount, pMemoryRanges);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDeviceMemoryCommitment(
    VkDevice                                    device,
    VkDeviceMemory                              memory,
    VkDeviceSize*                               pCommittedMemoryInBytes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryCommitment>::Dispatch(manager, device, memory, pCommittedMemoryInBytes);

    GetDeviceTable(device)->GetDeviceMemoryCommitment(device, memory, pCommittedMemoryInBytes);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceMemoryCommitment);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeVkDeviceSizePtr(pCommittedMemoryInBytes);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryCommitment>::Dispatch(manager, device, memory, pCommittedMemoryInBytes);
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory(
    VkDevice                                    device,
    VkBuffer                                    buffer,
    VkDeviceMemory                              memory,
    VkDeviceSize                                memoryOffset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindBufferMemory>::Dispatch(manager, device, buffer, memory, memoryOffset);

    VkResult result = GetDeviceTable(device)->BindBufferMemory(device, buffer, memory, memoryOffset);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindBufferMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeVkDeviceSizeValue(memoryOffset);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindBufferMemory>::Dispatch(manager, result, device, buffer, memory, memoryOffset);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory(
    VkDevice                                    device,
    VkImage                                     image,
    VkDeviceMemory                              memory,
    VkDeviceSize                                memoryOffset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindImageMemory>::Dispatch(manager, device, image, memory, memoryOffset);

    VkResult result = GetDeviceTable(device)->BindImageMemory(device, image, memory, memoryOffset);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindImageMemory);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeVkDeviceSizeValue(memoryOffset);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindImageMemory>::Dispatch(manager, result, device, image, memory, memoryOffset);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements(
    VkDevice                                    device,
    VkBuffer                                    buffer,
    VkMemoryRequirements*                       pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements>::Dispatch(manager, device, buffer, pMemoryRequirements);

    GetDeviceTable(device)->GetBufferMemoryRequirements(device, buffer, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements>::Dispatch(manager, device, buffer, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements(
    VkDevice                                    device,
    VkImage                                     image,
    VkMemoryRequirements*                       pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements>::Dispatch(manager, device, image, pMemoryRequirements);

    GetDeviceTable(device)->GetImageMemoryRequirements(device, image, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements>::Dispatch(manager, device, image, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements(
    VkDevice                                    device,
    VkImage                                     image,
    uint32_t*                                   pSparseMemoryRequirementCount,
    VkSparseImageMemoryRequirements*            pSparseMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements>::Dispatch(manager, device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    GetDeviceTable(device)->GetImageSparseMemoryRequirements(device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        encoder->EncodeUInt32Ptr(pSparseMemoryRequirementCount);
        EncodeStructArray(encoder, pSparseMemoryRequirements, (pSparseMemoryRequirementCount != nullptr) ? (*pSparseMemoryRequirementCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements>::Dispatch(manager, device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkImageType                                 type,
    VkSampleCountFlagBits                       samples,
    VkImageUsageFlags                           usage,
    VkImageTiling                               tiling,
    uint32_t*                                   pPropertyCount,
    VkSparseImageFormatProperties*              pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties>::Dispatch(manager, physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceSparseImageFormatProperties(physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        encoder->EncodeEnumValue(type);
        encoder->EncodeEnumValue(samples);
        encoder->EncodeFlagsValue(usage);
        encoder->EncodeEnumValue(tiling);
        encoder->EncodeUInt32Ptr(pPropertyCount);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties>::Dispatch(manager, physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL QueueBindSparse(
    VkQueue                                     queue,
    uint32_t                                    bindInfoCount,
    const VkBindSparseInfo*                     pBindInfo,
    VkFence                                     fence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueBindSparse>::Dispatch(manager, queue, bindInfoCount, pBindInfo, fence);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindSparseInfo* pBindInfo_unwrapped = UnwrapStructArrayHandles(pBindInfo, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(queue)->QueueBindSparse(queue, bindInfoCount, pBindInfo_unwrapped, fence);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueBindSparse);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfo, bindInfoCount);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueBindSparse>::Dispatch(manager, result, queue, bindInfoCount, pBindInfo, fence);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateFence(
    VkDevice                                    device,
    const VkFenceCreateInfo*                    pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkFence*                                    pFence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateFence>::Dispatch(manager, device, pCreateInfo, pAllocator, pFence);

    VkResult result = GetDeviceTable(device)->CreateFence(device, pCreateInfo, pAllocator, pFence);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, FenceWrapper>(device, NoParentWrapper::kHandleValue, pFence, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateFence);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<FenceWrapper>(pFence, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, FenceWrapper, VkFenceCreateInfo>(result, device, pFence, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateFence>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pFence);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyFence(
    VkDevice                                    device,
    VkFence                                     fence,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyFence>::Dispatch(manager, device, fence, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyFence);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<FenceWrapper>(fence);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyFence(device, fence, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyFence>::Dispatch(manager, device, fence, pAllocator);

    DestroyWrappedHandle<FenceWrapper>(fence);
}

VKAPI_ATTR VkResult VKAPI_CALL ResetFences(
    VkDevice                                    device,
    uint32_t                                    fenceCount,
    const VkFence*                              pFences)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetFences>::Dispatch(manager, device, fenceCount, pFences);

    VkResult result = GetDeviceTable(device)->ResetFences(device, fenceCount, pFences);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetFences);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(fenceCount);
        encoder->EncodeHandleArray<FenceWrapper>(pFences, fenceCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetFences>::Dispatch(manager, result, device, fenceCount, pFences);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceStatus(
    VkDevice                                    device,
    VkFence                                     fence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetFenceStatus>::Dispatch(manager, device, fence);

    VkResult result = GetDeviceTable(device)->GetFenceStatus(device, fence);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetFenceStatus);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetFenceStatus>::Dispatch(manager, result, device, fence);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitForFences(
    VkDevice                                    device,
    uint32_t                                    fenceCount,
    const VkFence*                              pFences,
    VkBool32                                    waitAll,
    uint64_t                                    timeout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkWaitForFences>::Dispatch(manager, device, fenceCount, pFences, waitAll, timeout);

    VkResult result = GetDeviceTable(device)->WaitForFences(device, fenceCount, pFences, waitAll, timeout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkWaitForFences);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(fenceCount);
        encoder->EncodeHandleArray<FenceWrapper>(pFences, fenceCount);
        encoder->EncodeVkBool32Value(waitAll);
        encoder->EncodeUInt64Value(timeout);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkWaitForFences>::Dispatch(manager, result, device, fenceCount, pFences, waitAll, timeout);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSemaphore(
    VkDevice                                    device,
    const VkSemaphoreCreateInfo*                pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSemaphore*                                pSemaphore)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSemaphore>::Dispatch(manager, device, pCreateInfo, pAllocator, pSemaphore);

    VkResult result = GetDeviceTable(device)->CreateSemaphore(device, pCreateInfo, pAllocator, pSemaphore);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, SemaphoreWrapper>(device, NoParentWrapper::kHandleValue, pSemaphore, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSemaphore);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SemaphoreWrapper>(pSemaphore, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, SemaphoreWrapper, VkSemaphoreCreateInfo>(result, device, pSemaphore, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSemaphore>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pSemaphore);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySemaphore(
    VkDevice                                    device,
    VkSemaphore                                 semaphore,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySemaphore>::Dispatch(manager, device, semaphore, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySemaphore);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SemaphoreWrapper>(semaphore);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SemaphoreWrapper>(semaphore);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroySemaphore(device, semaphore, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySemaphore>::Dispatch(manager, device, semaphore, pAllocator);

    DestroyWrappedHandle<SemaphoreWrapper>(semaphore);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateEvent(
    VkDevice                                    device,
    const VkEventCreateInfo*                    pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkEvent*                                    pEvent)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateEvent>::Dispatch(manager, device, pCreateInfo, pAllocator, pEvent);

    VkResult result = GetDeviceTable(device)->CreateEvent(device, pCreateInfo, pAllocator, pEvent);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, EventWrapper>(device, NoParentWrapper::kHandleValue, pEvent, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<EventWrapper>(pEvent, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, EventWrapper, VkEventCreateInfo>(result, device, pEvent, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateEvent>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pEvent);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyEvent(
    VkDevice                                    device,
    VkEvent                                     event,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyEvent>::Dispatch(manager, device, event, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<EventWrapper>(event);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<EventWrapper>(event);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyEvent(device, event, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyEvent>::Dispatch(manager, device, event, pAllocator);

    DestroyWrappedHandle<EventWrapper>(event);
}

VKAPI_ATTR VkResult VKAPI_CALL GetEventStatus(
    VkDevice                                    device,
    VkEvent                                     event)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetEventStatus>::Dispatch(manager, device, event);

    VkResult result = GetDeviceTable(device)->GetEventStatus(device, event);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetEventStatus);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetEventStatus>::Dispatch(manager, result, device, event);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SetEvent(
    VkDevice                                    device,
    VkEvent                                     event)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetEvent>::Dispatch(manager, device, event);

    VkResult result = GetDeviceTable(device)->SetEvent(device, event);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetEvent>::Dispatch(manager, result, device, event);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ResetEvent(
    VkDevice                                    device,
    VkEvent                                     event)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetEvent>::Dispatch(manager, device, event);

    VkResult result = GetDeviceTable(device)->ResetEvent(device, event);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetEvent>::Dispatch(manager, result, device, event);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateQueryPool(
    VkDevice                                    device,
    const VkQueryPoolCreateInfo*                pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkQueryPool*                                pQueryPool)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateQueryPool>::Dispatch(manager, device, pCreateInfo, pAllocator, pQueryPool);

    VkResult result = GetDeviceTable(device)->CreateQueryPool(device, pCreateInfo, pAllocator, pQueryPool);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, QueryPoolWrapper>(device, NoParentWrapper::kHandleValue, pQueryPool, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateQueryPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<QueryPoolWrapper>(pQueryPool, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, QueryPoolWrapper, VkQueryPoolCreateInfo>(result, device, pQueryPool, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateQueryPool>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pQueryPool);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyQueryPool(
    VkDevice                                    device,
    VkQueryPool                                 queryPool,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyQueryPool>::Dispatch(manager, device, queryPool, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyQueryPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<QueryPoolWrapper>(queryPool);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyQueryPool(device, queryPool, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyQueryPool>::Dispatch(manager, device, queryPool, pAllocator);

    DestroyWrappedHandle<QueryPoolWrapper>(queryPool);
}

VKAPI_ATTR VkResult VKAPI_CALL GetQueryPoolResults(
    VkDevice                                    device,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery,
    uint32_t                                    queryCount,
    size_t                                      dataSize,
    void*                                       pData,
    VkDeviceSize                                stride,
    VkQueryResultFlags                          flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetQueryPoolResults>::Dispatch(manager, device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags);

    VkResult result = GetDeviceTable(device)->GetQueryPoolResults(device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetQueryPoolResults);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        encoder->EncodeUInt32Value(queryCount);
        encoder->EncodeSizeTValue(dataSize);
        encoder->EncodeVoidArray(pData, dataSize, omit_output_data);
        encoder->EncodeVkDeviceSizeValue(stride);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetQueryPoolResults>::Dispatch(manager, result, device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateBuffer(
    VkDevice                                    device,
    const VkBufferCreateInfo*                   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkBuffer*                                   pBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateBuffer>::Dispatch(manager, device, pCreateInfo, pAllocator, pBuffer);

    VkResult result = manager->OverrideCreateBuffer(device, pCreateInfo, pAllocator, pBuffer);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<BufferWrapper>(pBuffer, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, BufferWrapper, VkBufferCreateInfo>(result, device, pBuffer, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateBuffer>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pBuffer);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyBuffer(
    VkDevice                                    device,
    VkBuffer                                    buffer,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyBuffer>::Dispatch(manager, device, buffer, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<BufferWrapper>(buffer);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyBuffer(device, buffer, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyBuffer>::Dispatch(manager, device, buffer, pAllocator);

    DestroyWrappedHandle<BufferWrapper>(buffer);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateBufferView(
    VkDevice                                    device,
    const VkBufferViewCreateInfo*               pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkBufferView*                               pView)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateBufferView>::Dispatch(manager, device, pCreateInfo, pAllocator, pView);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferViewCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateBufferView(device, pCreateInfo_unwrapped, pAllocator, pView);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, BufferViewWrapper>(device, NoParentWrapper::kHandleValue, pView, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateBufferView);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<BufferViewWrapper>(pView, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, BufferViewWrapper, VkBufferViewCreateInfo>(result, device, pView, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateBufferView>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pView);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyBufferView(
    VkDevice                                    device,
    VkBufferView                                bufferView,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyBufferView>::Dispatch(manager, device, bufferView, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyBufferView);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<BufferViewWrapper>(bufferView);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<BufferViewWrapper>(bufferView);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyBufferView(device, bufferView, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyBufferView>::Dispatch(manager, device, bufferView, pAllocator);

    DestroyWrappedHandle<BufferViewWrapper>(bufferView);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateImage(
    VkDevice                                    device,
    const VkImageCreateInfo*                    pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkImage*                                    pImage)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateImage>::Dispatch(manager, device, pCreateInfo, pAllocator, pImage);

    VkResult result = manager->OverrideCreateImage(device, pCreateInfo, pAllocator, pImage);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<ImageWrapper>(pImage, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, ImageWrapper, VkImageCreateInfo>(result, device, pImage, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateImage>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pImage);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyImage(
    VkDevice                                    device,
    VkImage                                     image,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyImage>::Dispatch(manager, device, image, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<ImageWrapper>(image);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyImage(device, image, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyImage>::Dispatch(manager, device, image, pAllocator);

    DestroyWrappedHandle<ImageWrapper>(image);
}

VKAPI_ATTR void VKAPI_CALL GetImageSubresourceLayout(
    VkDevice                                    device,
    VkImage                                     image,
    const VkImageSubresource*                   pSubresource,
    VkSubresourceLayout*                        pLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout>::Dispatch(manager, device, image, pSubresource, pLayout);

    GetDeviceTable(device)->GetImageSubresourceLayout(device, image, pSubresource, pLayout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSubresourceLayout);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pSubresource);
        EncodeStructPtr(encoder, pLayout);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout>::Dispatch(manager, device, image, pSubresource, pLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateImageView(
    VkDevice                                    device,
    const VkImageViewCreateInfo*                pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkImageView*                                pView)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateImageView>::Dispatch(manager, device, pCreateInfo, pAllocator, pView);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageViewCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateImageView(device, pCreateInfo_unwrapped, pAllocator, pView);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, ImageViewWrapper>(device, NoParentWrapper::kHandleValue, pView, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateImageView);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<ImageViewWrapper>(pView, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, ImageViewWrapper, VkImageViewCreateInfo>(result, device, pView, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateImageView>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pView);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyImageView(
    VkDevice                                    device,
    VkImageView                                 imageView,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyImageView>::Dispatch(manager, device, imageView, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyImageView);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageViewWrapper>(imageView);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<ImageViewWrapper>(imageView);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyImageView(device, imageView, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyImageView>::Dispatch(manager, device, imageView, pAllocator);

    DestroyWrappedHandle<ImageViewWrapper>(imageView);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateShaderModule(
    VkDevice                                    device,
    const VkShaderModuleCreateInfo*             pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkShaderModule*                             pShaderModule)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateShaderModule>::Dispatch(manager, device, pCreateInfo, pAllocator, pShaderModule);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkShaderModuleCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateShaderModule(device, pCreateInfo_unwrapped, pAllocator, pShaderModule);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, ShaderModuleWrapper>(device, NoParentWrapper::kHandleValue, pShaderModule, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateShaderModule);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<ShaderModuleWrapper>(pShaderModule, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, ShaderModuleWrapper, VkShaderModuleCreateInfo>(result, device, pShaderModule, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateShaderModule>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pShaderModule);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyShaderModule(
    VkDevice                                    device,
    VkShaderModule                              shaderModule,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyShaderModule>::Dispatch(manager, device, shaderModule, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyShaderModule);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ShaderModuleWrapper>(shaderModule);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<ShaderModuleWrapper>(shaderModule);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyShaderModule(device, shaderModule, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyShaderModule>::Dispatch(manager, device, shaderModule, pAllocator);

    DestroyWrappedHandle<ShaderModuleWrapper>(shaderModule);
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePipelineCache(
    VkDevice                                    device,
    const VkPipelineCacheCreateInfo*            pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkPipelineCache*                            pPipelineCache)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreatePipelineCache>::Dispatch(manager, device, pCreateInfo, pAllocator, pPipelineCache);

    VkResult result = GetDeviceTable(device)->CreatePipelineCache(device, pCreateInfo, pAllocator, pPipelineCache);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, PipelineCacheWrapper>(device, NoParentWrapper::kHandleValue, pPipelineCache, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreatePipelineCache);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<PipelineCacheWrapper>(pPipelineCache, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, PipelineCacheWrapper, VkPipelineCacheCreateInfo>(result, device, pPipelineCache, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreatePipelineCache>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pPipelineCache);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipelineCache(
    VkDevice                                    device,
    VkPipelineCache                             pipelineCache,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyPipelineCache>::Dispatch(manager, device, pipelineCache, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyPipelineCache);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineCacheWrapper>(pipelineCache);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<PipelineCacheWrapper>(pipelineCache);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyPipelineCache(device, pipelineCache, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyPipelineCache>::Dispatch(manager, device, pipelineCache, pAllocator);

    DestroyWrappedHandle<PipelineCacheWrapper>(pipelineCache);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineCacheData(
    VkDevice                                    device,
    VkPipelineCache                             pipelineCache,
    size_t*                                     pDataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPipelineCacheData>::Dispatch(manager, device, pipelineCache, pDataSize, pData);

    VkResult result = GetDeviceTable(device)->GetPipelineCacheData(device, pipelineCache, pDataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPipelineCacheData);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineCacheWrapper>(pipelineCache);
        encoder->EncodeSizeTPtr(pDataSize, omit_output_data);
        encoder->EncodeVoidArray(pData, (pDataSize != nullptr) ? (*pDataSize) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPipelineCacheData>::Dispatch(manager, result, device, pipelineCache, pDataSize, pData);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL MergePipelineCaches(
    VkDevice                                    device,
    VkPipelineCache                             dstCache,
    uint32_t                                    srcCacheCount,
    const VkPipelineCache*                      pSrcCaches)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkMergePipelineCaches>::Dispatch(manager, device, dstCache, srcCacheCount, pSrcCaches);

    VkResult result = GetDeviceTable(device)->MergePipelineCaches(device, dstCache, srcCacheCount, pSrcCaches);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkMergePipelineCaches);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineCacheWrapper>(dstCache);
        encoder->EncodeUInt32Value(srcCacheCount);
        encoder->EncodeHandleArray<PipelineCacheWrapper>(pSrcCaches, srcCacheCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkMergePipelineCaches>::Dispatch(manager, result, device, dstCache, srcCacheCount, pSrcCaches);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipeline(
    VkDevice                                    device,
    VkPipeline                                  pipeline,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyPipeline>::Dispatch(manager, device, pipeline, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyPipeline);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<PipelineWrapper>(pipeline);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyPipeline(device, pipeline, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyPipeline>::Dispatch(manager, device, pipeline, pAllocator);

    DestroyWrappedHandle<PipelineWrapper>(pipeline);
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePipelineLayout(
    VkDevice                                    device,
    const VkPipelineLayoutCreateInfo*           pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkPipelineLayout*                           pPipelineLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreatePipelineLayout>::Dispatch(manager, device, pCreateInfo, pAllocator, pPipelineLayout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPipelineLayoutCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreatePipelineLayout(device, pCreateInfo_unwrapped, pAllocator, pPipelineLayout);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, PipelineLayoutWrapper>(device, NoParentWrapper::kHandleValue, pPipelineLayout, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreatePipelineLayout);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<PipelineLayoutWrapper>(pPipelineLayout, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, PipelineLayoutWrapper, VkPipelineLayoutCreateInfo>(result, device, pPipelineLayout, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreatePipelineLayout>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pPipelineLayout);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPipelineLayout(
    VkDevice                                    device,
    VkPipelineLayout                            pipelineLayout,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyPipelineLayout>::Dispatch(manager, device, pipelineLayout, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyPipelineLayout);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineLayoutWrapper>(pipelineLayout);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<PipelineLayoutWrapper>(pipelineLayout);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyPipelineLayout(device, pipelineLayout, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyPipelineLayout>::Dispatch(manager, device, pipelineLayout, pAllocator);

    DestroyWrappedHandle<PipelineLayoutWrapper>(pipelineLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSampler(
    VkDevice                                    device,
    const VkSamplerCreateInfo*                  pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSampler*                                  pSampler)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSampler>::Dispatch(manager, device, pCreateInfo, pAllocator, pSampler);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSamplerCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateSampler(device, pCreateInfo_unwrapped, pAllocator, pSampler);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, SamplerWrapper>(device, NoParentWrapper::kHandleValue, pSampler, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSampler);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SamplerWrapper>(pSampler, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, SamplerWrapper, VkSamplerCreateInfo>(result, device, pSampler, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSampler>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pSampler);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySampler(
    VkDevice                                    device,
    VkSampler                                   sampler,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySampler>::Dispatch(manager, device, sampler, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySampler);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SamplerWrapper>(sampler);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SamplerWrapper>(sampler);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroySampler(device, sampler, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySampler>::Dispatch(manager, device, sampler, pAllocator);

    DestroyWrappedHandle<SamplerWrapper>(sampler);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorSetLayout(
    VkDevice                                    device,
    const VkDescriptorSetLayoutCreateInfo*      pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDescriptorSetLayout*                      pSetLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDescriptorSetLayout>::Dispatch(manager, device, pCreateInfo, pAllocator, pSetLayout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorSetLayoutCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateDescriptorSetLayout(device, pCreateInfo_unwrapped, pAllocator, pSetLayout);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, DescriptorSetLayoutWrapper>(device, NoParentWrapper::kHandleValue, pSetLayout, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDescriptorSetLayout);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DescriptorSetLayoutWrapper>(pSetLayout, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DescriptorSetLayoutWrapper, VkDescriptorSetLayoutCreateInfo>(result, device, pSetLayout, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDescriptorSetLayout>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pSetLayout);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorSetLayout(
    VkDevice                                    device,
    VkDescriptorSetLayout                       descriptorSetLayout,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDescriptorSetLayout>::Dispatch(manager, device, descriptorSetLayout, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDescriptorSetLayout);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorSetLayoutWrapper>(descriptorSetLayout);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DescriptorSetLayoutWrapper>(descriptorSetLayout);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDescriptorSetLayout(device, descriptorSetLayout, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDescriptorSetLayout>::Dispatch(manager, device, descriptorSetLayout, pAllocator);

    DestroyWrappedHandle<DescriptorSetLayoutWrapper>(descriptorSetLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorPool(
    VkDevice                                    device,
    const VkDescriptorPoolCreateInfo*           pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDescriptorPool*                           pDescriptorPool)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDescriptorPool>::Dispatch(manager, device, pCreateInfo, pAllocator, pDescriptorPool);

    VkResult result = GetDeviceTable(device)->CreateDescriptorPool(device, pCreateInfo, pAllocator, pDescriptorPool);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, DescriptorPoolWrapper>(device, NoParentWrapper::kHandleValue, pDescriptorPool, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDescriptorPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DescriptorPoolWrapper>(pDescriptorPool, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DescriptorPoolWrapper, VkDescriptorPoolCreateInfo>(result, device, pDescriptorPool, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDescriptorPool>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pDescriptorPool);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorPool(
    VkDevice                                    device,
    VkDescriptorPool                            descriptorPool,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDescriptorPool>::Dispatch(manager, device, descriptorPool, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDescriptorPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorPoolWrapper>(descriptorPool);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DescriptorPoolWrapper>(descriptorPool);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDescriptorPool(device, descriptorPool, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDescriptorPool>::Dispatch(manager, device, descriptorPool, pAllocator);

    DestroyWrappedHandle<DescriptorPoolWrapper>(descriptorPool);
}

VKAPI_ATTR VkResult VKAPI_CALL ResetDescriptorPool(
    VkDevice                                    device,
    VkDescriptorPool                            descriptorPool,
    VkDescriptorPoolResetFlags                  flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetDescriptorPool>::Dispatch(manager, device, descriptorPool, flags);

    ScopedDestroyLock exclusive_scoped_lock;
    VkResult result = GetDeviceTable(device)->ResetDescriptorPool(device, descriptorPool, flags);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetDescriptorPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorPoolWrapper>(descriptorPool);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetDescriptorPool>::Dispatch(manager, result, device, descriptorPool, flags);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateDescriptorSets(
    VkDevice                                    device,
    const VkDescriptorSetAllocateInfo*          pAllocateInfo,
    VkDescriptorSet*                            pDescriptorSets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAllocateDescriptorSets>::Dispatch(manager, device, pAllocateInfo, pDescriptorSets);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorSetAllocateInfo* pAllocateInfo_unwrapped = UnwrapStructPtrHandles(pAllocateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->AllocateDescriptorSets(device, pAllocateInfo_unwrapped, pDescriptorSets);

    if (result >= 0)
    {
        CreateWrappedHandles<DeviceWrapper, DescriptorPoolWrapper, DescriptorSetWrapper>(device, pAllocateInfo->descriptorPool, pDescriptorSets, pAllocateInfo->descriptorSetCount, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkAllocateDescriptorSets);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAllocateInfo);
        encoder->EncodeHandleArray<DescriptorSetWrapper>(pDescriptorSets, (pAllocateInfo != nullptr) ? (pAllocateInfo->descriptorSetCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndPoolCreateApiCallCapture<VkDevice, DescriptorSetWrapper, VkDescriptorSetAllocateInfo>(result, device, (pAllocateInfo != nullptr) ? (pAllocateInfo->descriptorSetCount) : 0, pDescriptorSets, pAllocateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAllocateDescriptorSets>::Dispatch(manager, result, device, pAllocateInfo, pDescriptorSets);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL FreeDescriptorSets(
    VkDevice                                    device,
    VkDescriptorPool                            descriptorPool,
    uint32_t                                    descriptorSetCount,
    const VkDescriptorSet*                      pDescriptorSets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkFreeDescriptorSets>::Dispatch(manager, device, descriptorPool, descriptorSetCount, pDescriptorSets);

    ScopedDestroyLock exclusive_scoped_lock;
    VkResult result = GetDeviceTable(device)->FreeDescriptorSets(device, descriptorPool, descriptorSetCount, pDescriptorSets);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkFreeDescriptorSets);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorPoolWrapper>(descriptorPool);
        encoder->EncodeUInt32Value(descriptorSetCount);
        encoder->EncodeHandleArray<DescriptorSetWrapper>(pDescriptorSets, descriptorSetCount);
        encoder->EncodeEnumValue(result);
        manager->EndDestroyApiCallCapture<DescriptorSetWrapper>(descriptorSetCount, pDescriptorSets);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkFreeDescriptorSets>::Dispatch(manager, result, device, descriptorPool, descriptorSetCount, pDescriptorSets);

    DestroyWrappedHandles<DescriptorSetWrapper>(pDescriptorSets, descriptorSetCount);

    return result;
}

VKAPI_ATTR void VKAPI_CALL UpdateDescriptorSets(
    VkDevice                                    device,
    uint32_t                                    descriptorWriteCount,
    const VkWriteDescriptorSet*                 pDescriptorWrites,
    uint32_t                                    descriptorCopyCount,
    const VkCopyDescriptorSet*                  pDescriptorCopies)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkUpdateDescriptorSets>::Dispatch(manager, device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount, pDescriptorCopies);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkUpdateDescriptorSets);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(descriptorWriteCount);
        EncodeStructArray(encoder, pDescriptorWrites, descriptorWriteCount);
        encoder->EncodeUInt32Value(descriptorCopyCount);
        EncodeStructArray(encoder, pDescriptorCopies, descriptorCopyCount);
        manager->EndApiCallCapture();
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkWriteDescriptorSet* pDescriptorWrites_unwrapped = UnwrapStructArrayHandles(pDescriptorWrites, descriptorWriteCount, handle_unwrap_memory);
    const VkCopyDescriptorSet* pDescriptorCopies_unwrapped = UnwrapStructArrayHandles(pDescriptorCopies, descriptorCopyCount, handle_unwrap_memory);

    GetDeviceTable(device)->UpdateDescriptorSets(device, descriptorWriteCount, pDescriptorWrites_unwrapped, descriptorCopyCount, pDescriptorCopies_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkUpdateDescriptorSets>::Dispatch(manager, device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount, pDescriptorCopies);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateFramebuffer(
    VkDevice                                    device,
    const VkFramebufferCreateInfo*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkFramebuffer*                              pFramebuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateFramebuffer>::Dispatch(manager, device, pCreateInfo, pAllocator, pFramebuffer);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkFramebufferCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateFramebuffer(device, pCreateInfo_unwrapped, pAllocator, pFramebuffer);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, FramebufferWrapper>(device, NoParentWrapper::kHandleValue, pFramebuffer, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateFramebuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<FramebufferWrapper>(pFramebuffer, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, FramebufferWrapper, VkFramebufferCreateInfo>(result, device, pFramebuffer, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateFramebuffer>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pFramebuffer);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyFramebuffer(
    VkDevice                                    device,
    VkFramebuffer                               framebuffer,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyFramebuffer>::Dispatch(manager, device, framebuffer, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyFramebuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<FramebufferWrapper>(framebuffer);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<FramebufferWrapper>(framebuffer);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyFramebuffer(device, framebuffer, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyFramebuffer>::Dispatch(manager, device, framebuffer, pAllocator);

    DestroyWrappedHandle<FramebufferWrapper>(framebuffer);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass(
    VkDevice                                    device,
    const VkRenderPassCreateInfo*               pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkRenderPass*                               pRenderPass)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateRenderPass>::Dispatch(manager, device, pCreateInfo, pAllocator, pRenderPass);

    VkResult result = GetDeviceTable(device)->CreateRenderPass(device, pCreateInfo, pAllocator, pRenderPass);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, RenderPassWrapper>(device, NoParentWrapper::kHandleValue, pRenderPass, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateRenderPass);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<RenderPassWrapper>(pRenderPass, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, RenderPassWrapper, VkRenderPassCreateInfo>(result, device, pRenderPass, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateRenderPass>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pRenderPass);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyRenderPass(
    VkDevice                                    device,
    VkRenderPass                                renderPass,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyRenderPass>::Dispatch(manager, device, renderPass, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyRenderPass);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<RenderPassWrapper>(renderPass);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<RenderPassWrapper>(renderPass);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyRenderPass(device, renderPass, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyRenderPass>::Dispatch(manager, device, renderPass, pAllocator);

    DestroyWrappedHandle<RenderPassWrapper>(renderPass);
}

VKAPI_ATTR void VKAPI_CALL GetRenderAreaGranularity(
    VkDevice                                    device,
    VkRenderPass                                renderPass,
    VkExtent2D*                                 pGranularity)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRenderAreaGranularity>::Dispatch(manager, device, renderPass, pGranularity);

    GetDeviceTable(device)->GetRenderAreaGranularity(device, renderPass, pGranularity);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetRenderAreaGranularity);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<RenderPassWrapper>(renderPass);
        EncodeStructPtr(encoder, pGranularity);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRenderAreaGranularity>::Dispatch(manager, device, renderPass, pGranularity);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateCommandPool(
    VkDevice                                    device,
    const VkCommandPoolCreateInfo*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkCommandPool*                              pCommandPool)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateCommandPool>::Dispatch(manager, device, pCreateInfo, pAllocator, pCommandPool);

    VkResult result = GetDeviceTable(device)->CreateCommandPool(device, pCreateInfo, pAllocator, pCommandPool);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, CommandPoolWrapper>(device, NoParentWrapper::kHandleValue, pCommandPool, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateCommandPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<CommandPoolWrapper>(pCommandPool, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, CommandPoolWrapper, VkCommandPoolCreateInfo>(result, device, pCommandPool, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateCommandPool>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pCommandPool);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyCommandPool(
    VkDevice                                    device,
    VkCommandPool                               commandPool,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyCommandPool>::Dispatch(manager, device, commandPool, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyCommandPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<CommandPoolWrapper>(commandPool);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<CommandPoolWrapper>(commandPool);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyCommandPool(device, commandPool, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyCommandPool>::Dispatch(manager, device, commandPool, pAllocator);

    DestroyWrappedHandle<CommandPoolWrapper>(commandPool);
}

VKAPI_ATTR VkResult VKAPI_CALL ResetCommandPool(
    VkDevice                                    device,
    VkCommandPool                               commandPool,
    VkCommandPoolResetFlags                     flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetCommandPool>::Dispatch(manager, device, commandPool, flags);

    VkResult result = GetDeviceTable(device)->ResetCommandPool(device, commandPool, flags);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetCommandPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<CommandPoolWrapper>(commandPool);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetCommandPool>::Dispatch(manager, result, device, commandPool, flags);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AllocateCommandBuffers(
    VkDevice                                    device,
    const VkCommandBufferAllocateInfo*          pAllocateInfo,
    VkCommandBuffer*                            pCommandBuffers)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAllocateCommandBuffers>::Dispatch(manager, device, pAllocateInfo, pCommandBuffers);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCommandBufferAllocateInfo* pAllocateInfo_unwrapped = UnwrapStructPtrHandles(pAllocateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->AllocateCommandBuffers(device, pAllocateInfo_unwrapped, pCommandBuffers);

    if (result >= 0)
    {
        CreateWrappedHandles<DeviceWrapper, CommandPoolWrapper, CommandBufferWrapper>(device, pAllocateInfo->commandPool, pCommandBuffers, pAllocateInfo->commandBufferCount, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkAllocateCommandBuffers);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAllocateInfo);
        encoder->EncodeHandleArray<CommandBufferWrapper>(pCommandBuffers, (pAllocateInfo != nullptr) ? (pAllocateInfo->commandBufferCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndPoolCreateApiCallCapture<VkDevice, CommandBufferWrapper, VkCommandBufferAllocateInfo>(result, device, (pAllocateInfo != nullptr) ? (pAllocateInfo->commandBufferCount) : 0, pCommandBuffers, pAllocateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAllocateCommandBuffers>::Dispatch(manager, result, device, pAllocateInfo, pCommandBuffers);

    return result;
}

VKAPI_ATTR void VKAPI_CALL FreeCommandBuffers(
    VkDevice                                    device,
    VkCommandPool                               commandPool,
    uint32_t                                    commandBufferCount,
    const VkCommandBuffer*                      pCommandBuffers)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkFreeCommandBuffers>::Dispatch(manager, device, commandPool, commandBufferCount, pCommandBuffers);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkFreeCommandBuffers);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<CommandPoolWrapper>(commandPool);
        encoder->EncodeUInt32Value(commandBufferCount);
        encoder->EncodeHandleArray<CommandBufferWrapper>(pCommandBuffers, commandBufferCount);
        manager->EndDestroyApiCallCapture<CommandBufferWrapper>(commandBufferCount, pCommandBuffers);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->FreeCommandBuffers(device, commandPool, commandBufferCount, pCommandBuffers);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkFreeCommandBuffers>::Dispatch(manager, device, commandPool, commandBufferCount, pCommandBuffers);

    DestroyWrappedHandles<CommandBufferWrapper>(pCommandBuffers, commandBufferCount);
}

VKAPI_ATTR VkResult VKAPI_CALL BeginCommandBuffer(
    VkCommandBuffer                             commandBuffer,
    const VkCommandBufferBeginInfo*             pBeginInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBeginCommandBuffer>::Dispatch(manager, commandBuffer, pBeginInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCommandBufferBeginInfo* pBeginInfo_unwrapped = UnwrapStructPtrHandles(pBeginInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(commandBuffer)->BeginCommandBuffer(commandBuffer, pBeginInfo_unwrapped);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkBeginCommandBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBeginInfo);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer, TrackBeginCommandBufferHandles, pBeginInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBeginCommandBuffer>::Dispatch(manager, result, commandBuffer, pBeginInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL EndCommandBuffer(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkEndCommandBuffer>::Dispatch(manager, commandBuffer);

    VkResult result = GetDeviceTable(commandBuffer)->EndCommandBuffer(commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkEndCommandBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkEndCommandBuffer>::Dispatch(manager, result, commandBuffer);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ResetCommandBuffer(
    VkCommandBuffer                             commandBuffer,
    VkCommandBufferResetFlags                   flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetCommandBuffer>::Dispatch(manager, commandBuffer, flags);

    VkResult result = GetDeviceTable(commandBuffer)->ResetCommandBuffer(commandBuffer, flags);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkResetCommandBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetCommandBuffer>::Dispatch(manager, result, commandBuffer, flags);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindPipeline(
    VkCommandBuffer                             commandBuffer,
    VkPipelineBindPoint                         pipelineBindPoint,
    VkPipeline                                  pipeline)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindPipeline>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindPipeline);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineBindPoint);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindPipelineHandles, pipeline);
    }

    GetDeviceTable(commandBuffer)->CmdBindPipeline(commandBuffer, pipelineBindPoint, pipeline);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindPipeline>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline);
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewport(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstViewport,
    uint32_t                                    viewportCount,
    const VkViewport*                           pViewports)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetViewport>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pViewports);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetViewport);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstViewport);
        encoder->EncodeUInt32Value(viewportCount);
        EncodeStructArray(encoder, pViewports, viewportCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetViewport(commandBuffer, firstViewport, viewportCount, pViewports);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetViewport>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pViewports);
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissor(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstScissor,
    uint32_t                                    scissorCount,
    const VkRect2D*                             pScissors)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetScissor>::Dispatch(manager, commandBuffer, firstScissor, scissorCount, pScissors);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetScissor);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstScissor);
        encoder->EncodeUInt32Value(scissorCount);
        EncodeStructArray(encoder, pScissors, scissorCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetScissor(commandBuffer, firstScissor, scissorCount, pScissors);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetScissor>::Dispatch(manager, commandBuffer, firstScissor, scissorCount, pScissors);
}

VKAPI_ATTR void VKAPI_CALL CmdSetLineWidth(
    VkCommandBuffer                             commandBuffer,
    float                                       lineWidth)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetLineWidth>::Dispatch(manager, commandBuffer, lineWidth);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetLineWidth);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFloatValue(lineWidth);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetLineWidth(commandBuffer, lineWidth);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetLineWidth>::Dispatch(manager, commandBuffer, lineWidth);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBias(
    VkCommandBuffer                             commandBuffer,
    float                                       depthBiasConstantFactor,
    float                                       depthBiasClamp,
    float                                       depthBiasSlopeFactor)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBias>::Dispatch(manager, commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBias);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFloatValue(depthBiasConstantFactor);
        encoder->EncodeFloatValue(depthBiasClamp);
        encoder->EncodeFloatValue(depthBiasSlopeFactor);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBias(commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBias>::Dispatch(manager, commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor);
}

VKAPI_ATTR void VKAPI_CALL CmdSetBlendConstants(
    VkCommandBuffer                             commandBuffer,
    const float                                 blendConstants[4])
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetBlendConstants>::Dispatch(manager, commandBuffer, blendConstants);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetBlendConstants);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFloatArray(blendConstants, 4);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetBlendConstants(commandBuffer, blendConstants);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetBlendConstants>::Dispatch(manager, commandBuffer, blendConstants);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBounds(
    VkCommandBuffer                             commandBuffer,
    float                                       minDepthBounds,
    float                                       maxDepthBounds)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBounds>::Dispatch(manager, commandBuffer, minDepthBounds, maxDepthBounds);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBounds);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFloatValue(minDepthBounds);
        encoder->EncodeFloatValue(maxDepthBounds);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBounds(commandBuffer, minDepthBounds, maxDepthBounds);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBounds>::Dispatch(manager, commandBuffer, minDepthBounds, maxDepthBounds);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilCompareMask(
    VkCommandBuffer                             commandBuffer,
    VkStencilFaceFlags                          faceMask,
    uint32_t                                    compareMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilCompareMask>::Dispatch(manager, commandBuffer, faceMask, compareMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilCompareMask);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(faceMask);
        encoder->EncodeUInt32Value(compareMask);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilCompareMask(commandBuffer, faceMask, compareMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilCompareMask>::Dispatch(manager, commandBuffer, faceMask, compareMask);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilWriteMask(
    VkCommandBuffer                             commandBuffer,
    VkStencilFaceFlags                          faceMask,
    uint32_t                                    writeMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilWriteMask>::Dispatch(manager, commandBuffer, faceMask, writeMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilWriteMask);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(faceMask);
        encoder->EncodeUInt32Value(writeMask);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilWriteMask(commandBuffer, faceMask, writeMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilWriteMask>::Dispatch(manager, commandBuffer, faceMask, writeMask);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilReference(
    VkCommandBuffer                             commandBuffer,
    VkStencilFaceFlags                          faceMask,
    uint32_t                                    reference)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilReference>::Dispatch(manager, commandBuffer, faceMask, reference);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilReference);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(faceMask);
        encoder->EncodeUInt32Value(reference);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilReference(commandBuffer, faceMask, reference);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilReference>::Dispatch(manager, commandBuffer, faceMask, reference);
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorSets(
    VkCommandBuffer                             commandBuffer,
    VkPipelineBindPoint                         pipelineBindPoint,
    VkPipelineLayout                            layout,
    uint32_t                                    firstSet,
    uint32_t                                    descriptorSetCount,
    const VkDescriptorSet*                      pDescriptorSets,
    uint32_t                                    dynamicOffsetCount,
    const uint32_t*                             pDynamicOffsets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorSets>::Dispatch(manager, commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets, dynamicOffsetCount, pDynamicOffsets);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindDescriptorSets);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineBindPoint);
        encoder->EncodeHandleValue<PipelineLayoutWrapper>(layout);
        encoder->EncodeUInt32Value(firstSet);
        encoder->EncodeUInt32Value(descriptorSetCount);
        encoder->EncodeHandleArray<DescriptorSetWrapper>(pDescriptorSets, descriptorSetCount);
        encoder->EncodeUInt32Value(dynamicOffsetCount);
        encoder->EncodeUInt32Array(pDynamicOffsets, dynamicOffsetCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindDescriptorSetsHandles, layout, descriptorSetCount, pDescriptorSets);
    }

    GetDeviceTable(commandBuffer)->CmdBindDescriptorSets(commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets, dynamicOffsetCount, pDynamicOffsets);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorSets>::Dispatch(manager, commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets, dynamicOffsetCount, pDynamicOffsets);
}

VKAPI_ATTR void VKAPI_CALL CmdBindIndexBuffer(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkIndexType                                 indexType)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindIndexBuffer>::Dispatch(manager, commandBuffer, buffer, offset, indexType);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindIndexBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeEnumValue(indexType);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindIndexBufferHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdBindIndexBuffer(commandBuffer, buffer, offset, indexType);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindIndexBuffer>::Dispatch(manager, commandBuffer, buffer, offset, indexType);
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstBinding,
    uint32_t                                    bindingCount,
    const VkBuffer*                             pBuffers,
    const VkDeviceSize*                         pOffsets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindVertexBuffers);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstBinding);
        encoder->EncodeUInt32Value(bindingCount);
        encoder->EncodeHandleArray<BufferWrapper>(pBuffers, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pOffsets, bindingCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindVertexBuffersHandles, bindingCount, pBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdBindVertexBuffers(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets);
}

VKAPI_ATTR void VKAPI_CALL CmdDraw(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    vertexCount,
    uint32_t                                    instanceCount,
    uint32_t                                    firstVertex,
    uint32_t                                    firstInstance)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDraw>::Dispatch(manager, commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDraw);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(vertexCount);
        encoder->EncodeUInt32Value(instanceCount);
        encoder->EncodeUInt32Value(firstVertex);
        encoder->EncodeUInt32Value(firstInstance);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDraw(commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDraw>::Dispatch(manager, commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexed(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    indexCount,
    uint32_t                                    instanceCount,
    uint32_t                                    firstIndex,
    int32_t                                     vertexOffset,
    uint32_t                                    firstInstance)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndexed>::Dispatch(manager, commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndexed);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(indexCount);
        encoder->EncodeUInt32Value(instanceCount);
        encoder->EncodeUInt32Value(firstIndex);
        encoder->EncodeInt32Value(vertexOffset);
        encoder->EncodeUInt32Value(firstInstance);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndexed(commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndexed>::Dispatch(manager, commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirect(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    uint32_t                                    drawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndirect>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndirect);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeUInt32Value(drawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndirectHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndirect(commandBuffer, buffer, offset, drawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndirect>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirect(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    uint32_t                                    drawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirect>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirect);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeUInt32Value(drawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndexedIndirectHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndexedIndirect(commandBuffer, buffer, offset, drawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirect>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDispatch(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    groupCountX,
    uint32_t                                    groupCountY,
    uint32_t                                    groupCountZ)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDispatch>::Dispatch(manager, commandBuffer, groupCountX, groupCountY, groupCountZ);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDispatch);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(groupCountX);
        encoder->EncodeUInt32Value(groupCountY);
        encoder->EncodeUInt32Value(groupCountZ);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDispatch(commandBuffer, groupCountX, groupCountY, groupCountZ);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDispatch>::Dispatch(manager, commandBuffer, groupCountX, groupCountY, groupCountZ);
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchIndirect(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDispatchIndirect>::Dispatch(manager, commandBuffer, buffer, offset);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDispatchIndirect);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDispatchIndirectHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdDispatchIndirect(commandBuffer, buffer, offset);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDispatchIndirect>::Dispatch(manager, commandBuffer, buffer, offset);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    srcBuffer,
    VkBuffer                                    dstBuffer,
    uint32_t                                    regionCount,
    const VkBufferCopy*                         pRegions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer>::Dispatch(manager, commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(srcBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBufferHandles, srcBuffer, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdCopyBuffer(commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer>::Dispatch(manager, commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     srcImage,
    VkImageLayout                               srcImageLayout,
    VkImage                                     dstImage,
    VkImageLayout                               dstImageLayout,
    uint32_t                                    regionCount,
    const VkImageCopy*                          pRegions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(srcImage);
        encoder->EncodeEnumValue(srcImageLayout);
        encoder->EncodeHandleValue<ImageWrapper>(dstImage);
        encoder->EncodeEnumValue(dstImageLayout);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImageHandles, srcImage, dstImage);
    }

    GetDeviceTable(commandBuffer)->CmdCopyImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     srcImage,
    VkImageLayout                               srcImageLayout,
    VkImage                                     dstImage,
    VkImageLayout                               dstImageLayout,
    uint32_t                                    regionCount,
    const VkImageBlit*                          pRegions,
    VkFilter                                    filter)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBlitImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBlitImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(srcImage);
        encoder->EncodeEnumValue(srcImageLayout);
        encoder->EncodeHandleValue<ImageWrapper>(dstImage);
        encoder->EncodeEnumValue(dstImageLayout);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        encoder->EncodeEnumValue(filter);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBlitImageHandles, srcImage, dstImage);
    }

    GetDeviceTable(commandBuffer)->CmdBlitImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBlitImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    srcBuffer,
    VkImage                                     dstImage,
    VkImageLayout                               dstImageLayout,
    uint32_t                                    regionCount,
    const VkBufferImageCopy*                    pRegions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage>::Dispatch(manager, commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBufferToImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(srcBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(dstImage);
        encoder->EncodeEnumValue(dstImageLayout);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBufferToImageHandles, srcBuffer, dstImage);
    }

    GetDeviceTable(commandBuffer)->CmdCopyBufferToImage(commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage>::Dispatch(manager, commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     srcImage,
    VkImageLayout                               srcImageLayout,
    VkBuffer                                    dstBuffer,
    uint32_t                                    regionCount,
    const VkBufferImageCopy*                    pRegions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(srcImage);
        encoder->EncodeEnumValue(srcImageLayout);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImageToBufferHandles, srcImage, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdCopyImageToBuffer(commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions);
}

VKAPI_ATTR void VKAPI_CALL CmdUpdateBuffer(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    dstBuffer,
    VkDeviceSize                                dstOffset,
    VkDeviceSize                                dataSize,
    const void*                                 pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdUpdateBuffer>::Dispatch(manager, commandBuffer, dstBuffer, dstOffset, dataSize, pData);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdUpdateBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeVkDeviceSizeValue(dstOffset);
        encoder->EncodeVkDeviceSizeValue(dataSize);
        encoder->EncodeVoidArray(pData, static_cast<size_t>(dataSize));
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdUpdateBufferHandles, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdUpdateBuffer(commandBuffer, dstBuffer, dstOffset, dataSize, pData);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdUpdateBuffer>::Dispatch(manager, commandBuffer, dstBuffer, dstOffset, dataSize, pData);
}

VKAPI_ATTR void VKAPI_CALL CmdFillBuffer(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    dstBuffer,
    VkDeviceSize                                dstOffset,
    VkDeviceSize                                size,
    uint32_t                                    data)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdFillBuffer>::Dispatch(manager, commandBuffer, dstBuffer, dstOffset, size, data);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdFillBuffer);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeVkDeviceSizeValue(dstOffset);
        encoder->EncodeVkDeviceSizeValue(size);
        encoder->EncodeUInt32Value(data);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdFillBufferHandles, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdFillBuffer(commandBuffer, dstBuffer, dstOffset, size, data);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdFillBuffer>::Dispatch(manager, commandBuffer, dstBuffer, dstOffset, size, data);
}

VKAPI_ATTR void VKAPI_CALL CmdClearColorImage(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     image,
    VkImageLayout                               imageLayout,
    const VkClearColorValue*                    pColor,
    uint32_t                                    rangeCount,
    const VkImageSubresourceRange*              pRanges)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdClearColorImage>::Dispatch(manager, commandBuffer, image, imageLayout, pColor, rangeCount, pRanges);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdClearColorImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        encoder->EncodeEnumValue(imageLayout);
        EncodeStructPtr(encoder, pColor);
        encoder->EncodeUInt32Value(rangeCount);
        EncodeStructArray(encoder, pRanges, rangeCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdClearColorImageHandles, image);
    }

    GetDeviceTable(commandBuffer)->CmdClearColorImage(commandBuffer, image, imageLayout, pColor, rangeCount, pRanges);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdClearColorImage>::Dispatch(manager, commandBuffer, image, imageLayout, pColor, rangeCount, pRanges);
}

VKAPI_ATTR void VKAPI_CALL CmdClearDepthStencilImage(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     image,
    VkImageLayout                               imageLayout,
    const VkClearDepthStencilValue*             pDepthStencil,
    uint32_t                                    rangeCount,
    const VkImageSubresourceRange*              pRanges)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdClearDepthStencilImage>::Dispatch(manager, commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdClearDepthStencilImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        encoder->EncodeEnumValue(imageLayout);
        EncodeStructPtr(encoder, pDepthStencil);
        encoder->EncodeUInt32Value(rangeCount);
        EncodeStructArray(encoder, pRanges, rangeCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdClearDepthStencilImageHandles, image);
    }

    GetDeviceTable(commandBuffer)->CmdClearDepthStencilImage(commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdClearDepthStencilImage>::Dispatch(manager, commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges);
}

VKAPI_ATTR void VKAPI_CALL CmdClearAttachments(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    attachmentCount,
    const VkClearAttachment*                    pAttachments,
    uint32_t                                    rectCount,
    const VkClearRect*                          pRects)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdClearAttachments>::Dispatch(manager, commandBuffer, attachmentCount, pAttachments, rectCount, pRects);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdClearAttachments);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(attachmentCount);
        EncodeStructArray(encoder, pAttachments, attachmentCount);
        encoder->EncodeUInt32Value(rectCount);
        EncodeStructArray(encoder, pRects, rectCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdClearAttachments(commandBuffer, attachmentCount, pAttachments, rectCount, pRects);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdClearAttachments>::Dispatch(manager, commandBuffer, attachmentCount, pAttachments, rectCount, pRects);
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage(
    VkCommandBuffer                             commandBuffer,
    VkImage                                     srcImage,
    VkImageLayout                               srcImageLayout,
    VkImage                                     dstImage,
    VkImageLayout                               dstImageLayout,
    uint32_t                                    regionCount,
    const VkImageResolve*                       pRegions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResolveImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResolveImage);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageWrapper>(srcImage);
        encoder->EncodeEnumValue(srcImageLayout);
        encoder->EncodeHandleValue<ImageWrapper>(dstImage);
        encoder->EncodeEnumValue(dstImageLayout);
        encoder->EncodeUInt32Value(regionCount);
        EncodeStructArray(encoder, pRegions, regionCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResolveImageHandles, srcImage, dstImage);
    }

    GetDeviceTable(commandBuffer)->CmdResolveImage(commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResolveImage>::Dispatch(manager, commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions);
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    VkPipelineStageFlags                        stageMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetEvent>::Dispatch(manager, commandBuffer, event, stageMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeFlagsValue(stageMask);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdSetEventHandles, event);
    }

    GetDeviceTable(commandBuffer)->CmdSetEvent(commandBuffer, event, stageMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetEvent>::Dispatch(manager, commandBuffer, event, stageMask);
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    VkPipelineStageFlags                        stageMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResetEvent>::Dispatch(manager, commandBuffer, event, stageMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResetEvent);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeFlagsValue(stageMask);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResetEventHandles, event);
    }

    GetDeviceTable(commandBuffer)->CmdResetEvent(commandBuffer, event, stageMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResetEvent>::Dispatch(manager, commandBuffer, event, stageMask);
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    eventCount,
    const VkEvent*                              pEvents,
    VkPipelineStageFlags                        srcStageMask,
    VkPipelineStageFlags                        dstStageMask,
    uint32_t                                    memoryBarrierCount,
    const VkMemoryBarrier*                      pMemoryBarriers,
    uint32_t                                    bufferMemoryBarrierCount,
    const VkBufferMemoryBarrier*                pBufferMemoryBarriers,
    uint32_t                                    imageMemoryBarrierCount,
    const VkImageMemoryBarrier*                 pImageMemoryBarriers)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWaitEvents>::Dispatch(manager, commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWaitEvents);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(eventCount);
        encoder->EncodeHandleArray<EventWrapper>(pEvents, eventCount);
        encoder->EncodeFlagsValue(srcStageMask);
        encoder->EncodeFlagsValue(dstStageMask);
        encoder->EncodeUInt32Value(memoryBarrierCount);
        EncodeStructArray(encoder, pMemoryBarriers, memoryBarrierCount);
        encoder->EncodeUInt32Value(bufferMemoryBarrierCount);
        EncodeStructArray(encoder, pBufferMemoryBarriers, bufferMemoryBarrierCount);
        encoder->EncodeUInt32Value(imageMemoryBarrierCount);
        EncodeStructArray(encoder, pImageMemoryBarriers, imageMemoryBarrierCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWaitEventsHandles, eventCount, pEvents, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferMemoryBarrier* pBufferMemoryBarriers_unwrapped = UnwrapStructArrayHandles(pBufferMemoryBarriers, bufferMemoryBarrierCount, handle_unwrap_memory);
    const VkImageMemoryBarrier* pImageMemoryBarriers_unwrapped = UnwrapStructArrayHandles(pImageMemoryBarriers, imageMemoryBarrierCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdWaitEvents(commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers_unwrapped, imageMemoryBarrierCount, pImageMemoryBarriers_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWaitEvents>::Dispatch(manager, commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlags                        srcStageMask,
    VkPipelineStageFlags                        dstStageMask,
    VkDependencyFlags                           dependencyFlags,
    uint32_t                                    memoryBarrierCount,
    const VkMemoryBarrier*                      pMemoryBarriers,
    uint32_t                                    bufferMemoryBarrierCount,
    const VkBufferMemoryBarrier*                pBufferMemoryBarriers,
    uint32_t                                    imageMemoryBarrierCount,
    const VkImageMemoryBarrier*                 pImageMemoryBarriers)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier>::Dispatch(manager, commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPipelineBarrier);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(srcStageMask);
        encoder->EncodeFlagsValue(dstStageMask);
        encoder->EncodeFlagsValue(dependencyFlags);
        encoder->EncodeUInt32Value(memoryBarrierCount);
        EncodeStructArray(encoder, pMemoryBarriers, memoryBarrierCount);
        encoder->EncodeUInt32Value(bufferMemoryBarrierCount);
        EncodeStructArray(encoder, pBufferMemoryBarriers, bufferMemoryBarrierCount);
        encoder->EncodeUInt32Value(imageMemoryBarrierCount);
        EncodeStructArray(encoder, pImageMemoryBarriers, imageMemoryBarrierCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPipelineBarrierHandles, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferMemoryBarrier* pBufferMemoryBarriers_unwrapped = UnwrapStructArrayHandles(pBufferMemoryBarriers, bufferMemoryBarrierCount, handle_unwrap_memory);
    const VkImageMemoryBarrier* pImageMemoryBarriers_unwrapped = UnwrapStructArrayHandles(pImageMemoryBarriers, imageMemoryBarrierCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPipelineBarrier(commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers_unwrapped, imageMemoryBarrierCount, pImageMemoryBarriers_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier>::Dispatch(manager, commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginQuery(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    query,
    VkQueryControlFlags                         flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginQuery>::Dispatch(manager, commandBuffer, queryPool, query, flags);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginQuery);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        encoder->EncodeFlagsValue(flags);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginQueryHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdBeginQuery(commandBuffer, queryPool, query, flags);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginQuery>::Dispatch(manager, commandBuffer, queryPool, query, flags);
}

VKAPI_ATTR void VKAPI_CALL CmdEndQuery(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    query)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndQuery>::Dispatch(manager, commandBuffer, queryPool, query);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndQuery);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdEndQueryHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdEndQuery(commandBuffer, queryPool, query);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndQuery>::Dispatch(manager, commandBuffer, queryPool, query);
}

VKAPI_ATTR void VKAPI_CALL CmdResetQueryPool(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery,
    uint32_t                                    queryCount)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResetQueryPool>::Dispatch(manager, commandBuffer, queryPool, firstQuery, queryCount);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResetQueryPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        encoder->EncodeUInt32Value(queryCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResetQueryPoolHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdResetQueryPool(commandBuffer, queryPool, firstQuery, queryCount);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResetQueryPool>::Dispatch(manager, commandBuffer, queryPool, firstQuery, queryCount);
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlagBits                     pipelineStage,
    VkQueryPool                                 queryPool,
    uint32_t                                    query)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp>::Dispatch(manager, commandBuffer, pipelineStage, queryPool, query);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteTimestamp);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineStage);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteTimestampHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdWriteTimestamp(commandBuffer, pipelineStage, queryPool, query);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp>::Dispatch(manager, commandBuffer, pipelineStage, queryPool, query);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyQueryPoolResults(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery,
    uint32_t                                    queryCount,
    VkBuffer                                    dstBuffer,
    VkDeviceSize                                dstOffset,
    VkDeviceSize                                stride,
    VkQueryResultFlags                          flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyQueryPoolResults>::Dispatch(manager, commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyQueryPoolResults);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        encoder->EncodeUInt32Value(queryCount);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeVkDeviceSizeValue(dstOffset);
        encoder->EncodeVkDeviceSizeValue(stride);
        encoder->EncodeFlagsValue(flags);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyQueryPoolResultsHandles, queryPool, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdCopyQueryPoolResults(commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyQueryPoolResults>::Dispatch(manager, commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags);
}

VKAPI_ATTR void VKAPI_CALL CmdPushConstants(
    VkCommandBuffer                             commandBuffer,
    VkPipelineLayout                            layout,
    VkShaderStageFlags                          stageFlags,
    uint32_t                                    offset,
    uint32_t                                    size,
    const void*                                 pValues)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPushConstants>::Dispatch(manager, commandBuffer, layout, stageFlags, offset, size, pValues);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPushConstants);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<PipelineLayoutWrapper>(layout);
        encoder->EncodeFlagsValue(stageFlags);
        encoder->EncodeUInt32Value(offset);
        encoder->EncodeUInt32Value(size);
        encoder->EncodeVoidArray(pValues, size);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPushConstantsHandles, layout);
    }

    GetDeviceTable(commandBuffer)->CmdPushConstants(commandBuffer, layout, stageFlags, offset, size, pValues);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPushConstants>::Dispatch(manager, commandBuffer, layout, stageFlags, offset, size, pValues);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass(
    VkCommandBuffer                             commandBuffer,
    const VkRenderPassBeginInfo*                pRenderPassBegin,
    VkSubpassContents                           contents)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass>::Dispatch(manager, commandBuffer, pRenderPassBegin, contents);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginRenderPass);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pRenderPassBegin);
        encoder->EncodeEnumValue(contents);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginRenderPassHandles, pRenderPassBegin);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkRenderPassBeginInfo* pRenderPassBegin_unwrapped = UnwrapStructPtrHandles(pRenderPassBegin, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginRenderPass(commandBuffer, pRenderPassBegin_unwrapped, contents);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass>::Dispatch(manager, commandBuffer, pRenderPassBegin, contents);
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass(
    VkCommandBuffer                             commandBuffer,
    VkSubpassContents                           contents)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdNextSubpass>::Dispatch(manager, commandBuffer, contents);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdNextSubpass);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(contents);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdNextSubpass(commandBuffer, contents);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdNextSubpass>::Dispatch(manager, commandBuffer, contents);
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndRenderPass);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndRenderPass(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL CmdExecuteCommands(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    commandBufferCount,
    const VkCommandBuffer*                      pCommandBuffers)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdExecuteCommands>::Dispatch(manager, commandBuffer, commandBufferCount, pCommandBuffers);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdExecuteCommands);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(commandBufferCount);
        encoder->EncodeHandleArray<CommandBufferWrapper>(pCommandBuffers, commandBufferCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdExecuteCommandsHandles, commandBufferCount, pCommandBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdExecuteCommands(commandBuffer, commandBufferCount, pCommandBuffers);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdExecuteCommands>::Dispatch(manager, commandBuffer, commandBufferCount, pCommandBuffers);
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory2(
    VkDevice                                    device,
    uint32_t                                    bindInfoCount,
    const VkBindBufferMemoryInfo*               pBindInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindBufferMemory2>::Dispatch(manager, device, bindInfoCount, pBindInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindBufferMemoryInfo* pBindInfos_unwrapped = UnwrapStructArrayHandles(pBindInfos, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindBufferMemory2(device, bindInfoCount, pBindInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindBufferMemory2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfos, bindInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindBufferMemory2>::Dispatch(manager, result, device, bindInfoCount, pBindInfos);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory2(
    VkDevice                                    device,
    uint32_t                                    bindInfoCount,
    const VkBindImageMemoryInfo*                pBindInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindImageMemory2>::Dispatch(manager, device, bindInfoCount, pBindInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindImageMemoryInfo* pBindInfos_unwrapped = UnwrapStructArrayHandles(pBindInfos, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindImageMemory2(device, bindInfoCount, pBindInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindImageMemory2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfos, bindInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindImageMemory2>::Dispatch(manager, result, device, bindInfoCount, pBindInfos);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDeviceGroupPeerMemoryFeatures(
    VkDevice                                    device,
    uint32_t                                    heapIndex,
    uint32_t                                    localDeviceIndex,
    uint32_t                                    remoteDeviceIndex,
    VkPeerMemoryFeatureFlags*                   pPeerMemoryFeatures)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeatures>::Dispatch(manager, device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);

    GetDeviceTable(device)->GetDeviceGroupPeerMemoryFeatures(device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeatures);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(heapIndex);
        encoder->EncodeUInt32Value(localDeviceIndex);
        encoder->EncodeUInt32Value(remoteDeviceIndex);
        encoder->EncodeFlagsPtr(pPeerMemoryFeatures);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeatures>::Dispatch(manager, device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDeviceMask(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    deviceMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDeviceMask>::Dispatch(manager, commandBuffer, deviceMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDeviceMask);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(deviceMask);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDeviceMask(commandBuffer, deviceMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDeviceMask>::Dispatch(manager, commandBuffer, deviceMask);
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchBase(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    baseGroupX,
    uint32_t                                    baseGroupY,
    uint32_t                                    baseGroupZ,
    uint32_t                                    groupCountX,
    uint32_t                                    groupCountY,
    uint32_t                                    groupCountZ)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDispatchBase>::Dispatch(manager, commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDispatchBase);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(baseGroupX);
        encoder->EncodeUInt32Value(baseGroupY);
        encoder->EncodeUInt32Value(baseGroupZ);
        encoder->EncodeUInt32Value(groupCountX);
        encoder->EncodeUInt32Value(groupCountY);
        encoder->EncodeUInt32Value(groupCountZ);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDispatchBase(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDispatchBase>::Dispatch(manager, commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceGroups(
    VkInstance                                  instance,
    uint32_t*                                   pPhysicalDeviceGroupCount,
    VkPhysicalDeviceGroupProperties*            pPhysicalDeviceGroupProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroups>::Dispatch(manager, instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    VkResult result = GetInstanceTable(instance)->EnumeratePhysicalDeviceGroups(instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<InstanceWrapper, NoParentWrapper, VkPhysicalDeviceGroupProperties>(instance, NoParentWrapper::kHandleValue, pPhysicalDeviceGroupProperties, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroups);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeUInt32Ptr(pPhysicalDeviceGroupCount, omit_output_data);
        EncodeStructArray(encoder, pPhysicalDeviceGroupProperties, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkInstance, PhysicalDeviceWrapper, VkPhysicalDeviceGroupProperties>(result, instance, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, pPhysicalDeviceGroupProperties, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroups>::Dispatch(manager, result, instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements2(
    VkDevice                                    device,
    const VkImageMemoryRequirementsInfo2*       pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetImageMemoryRequirements2(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements2(
    VkDevice                                    device,
    const VkBufferMemoryRequirementsInfo2*      pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetBufferMemoryRequirements2(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements2(
    VkDevice                                    device,
    const VkImageSparseMemoryRequirementsInfo2* pInfo,
    uint32_t*                                   pSparseMemoryRequirementCount,
    VkSparseImageMemoryRequirements2*           pSparseMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageSparseMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetImageSparseMemoryRequirements2(device, pInfo_unwrapped, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt32Ptr(pSparseMemoryRequirementCount);
        EncodeStructArray(encoder, pSparseMemoryRequirements, (pSparseMemoryRequirementCount != nullptr) ? (*pSparseMemoryRequirementCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures2(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceFeatures2*                  pFeatures)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2>::Dispatch(manager, physicalDevice, pFeatures);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFeatures2(physicalDevice, pFeatures);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pFeatures);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2>::Dispatch(manager, physicalDevice, pFeatures);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties2(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceProperties2*                pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2>::Dispatch(manager, physicalDevice, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceProperties2(physicalDevice, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2>::Dispatch(manager, physicalDevice, pProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties2(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkFormatProperties2*                        pFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2>::Dispatch(manager, physicalDevice, format, pFormatProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFormatProperties2(physicalDevice, format, pFormatProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        EncodeStructPtr(encoder, pFormatProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2>::Dispatch(manager, physicalDevice, format, pFormatProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties2(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceImageFormatInfo2*     pImageFormatInfo,
    VkImageFormatProperties2*                   pImageFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2>::Dispatch(manager, physicalDevice, pImageFormatInfo, pImageFormatProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceImageFormatProperties2(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pImageFormatInfo);
        EncodeStructPtr(encoder, pImageFormatProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2>::Dispatch(manager, result, physicalDevice, pImageFormatInfo, pImageFormatProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties2(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pQueueFamilyPropertyCount,
    VkQueueFamilyProperties2*                   pQueueFamilyProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    manager->OverrideGetPhysicalDeviceQueueFamilyProperties2(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pQueueFamilyPropertyCount);
        EncodeStructArray(encoder, pQueueFamilyProperties, (pQueueFamilyPropertyCount != nullptr) ? (*pQueueFamilyPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties2(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceMemoryProperties2*          pMemoryProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2>::Dispatch(manager, physicalDevice, pMemoryProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceMemoryProperties2(physicalDevice, pMemoryProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pMemoryProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2>::Dispatch(manager, physicalDevice, pMemoryProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties2(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo,
    uint32_t*                                   pPropertyCount,
    VkSparseImageFormatProperties2*             pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2>::Dispatch(manager, physicalDevice, pFormatInfo, pPropertyCount, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceSparseImageFormatProperties2(physicalDevice, pFormatInfo, pPropertyCount, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pFormatInfo);
        encoder->EncodeUInt32Ptr(pPropertyCount);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2>::Dispatch(manager, physicalDevice, pFormatInfo, pPropertyCount, pProperties);
}

VKAPI_ATTR void VKAPI_CALL TrimCommandPool(
    VkDevice                                    device,
    VkCommandPool                               commandPool,
    VkCommandPoolTrimFlags                      flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkTrimCommandPool>::Dispatch(manager, device, commandPool, flags);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkTrimCommandPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<CommandPoolWrapper>(commandPool);
        encoder->EncodeFlagsValue(flags);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->TrimCommandPool(device, commandPool, flags);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkTrimCommandPool>::Dispatch(manager, device, commandPool, flags);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceQueue2(
    VkDevice                                    device,
    const VkDeviceQueueInfo2*                   pQueueInfo,
    VkQueue*                                    pQueue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceQueue2>::Dispatch(manager, device, pQueueInfo, pQueue);

    GetDeviceTable(device)->GetDeviceQueue2(device, pQueueInfo, pQueue);

    CreateWrappedHandle<DeviceWrapper, NoParentWrapper, QueueWrapper>(device, NoParentWrapper::kHandleValue, pQueue, VulkanCaptureManager::GetUniqueId);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceQueue2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pQueueInfo);
        encoder->EncodeHandlePtr<QueueWrapper>(pQueue);
        manager->EndCreateApiCallCapture<VkDevice, QueueWrapper, void>(VK_SUCCESS, device, pQueue, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceQueue2>::Dispatch(manager, device, pQueueInfo, pQueue);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSamplerYcbcrConversion(
    VkDevice                                    device,
    const VkSamplerYcbcrConversionCreateInfo*   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSamplerYcbcrConversion*                   pYcbcrConversion)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversion>::Dispatch(manager, device, pCreateInfo, pAllocator, pYcbcrConversion);

    VkResult result = GetDeviceTable(device)->CreateSamplerYcbcrConversion(device, pCreateInfo, pAllocator, pYcbcrConversion);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, SamplerYcbcrConversionWrapper>(device, NoParentWrapper::kHandleValue, pYcbcrConversion, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversion);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SamplerYcbcrConversionWrapper>(pYcbcrConversion, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, SamplerYcbcrConversionWrapper, VkSamplerYcbcrConversionCreateInfo>(result, device, pYcbcrConversion, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversion>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pYcbcrConversion);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySamplerYcbcrConversion(
    VkDevice                                    device,
    VkSamplerYcbcrConversion                    ycbcrConversion,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversion>::Dispatch(manager, device, ycbcrConversion, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversion);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SamplerYcbcrConversionWrapper>(ycbcrConversion);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SamplerYcbcrConversionWrapper>(ycbcrConversion);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroySamplerYcbcrConversion(device, ycbcrConversion, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversion>::Dispatch(manager, device, ycbcrConversion, pAllocator);

    DestroyWrappedHandle<SamplerYcbcrConversionWrapper>(ycbcrConversion);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorUpdateTemplate(
    VkDevice                                    device,
    const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDescriptorUpdateTemplate*                 pDescriptorUpdateTemplate)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplate>::Dispatch(manager, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateDescriptorUpdateTemplate(device, pCreateInfo_unwrapped, pAllocator, pDescriptorUpdateTemplate);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, DescriptorUpdateTemplateWrapper>(device, NoParentWrapper::kHandleValue, pDescriptorUpdateTemplate, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplate);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DescriptorUpdateTemplateWrapper>(pDescriptorUpdateTemplate, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DescriptorUpdateTemplateWrapper, VkDescriptorUpdateTemplateCreateInfo>(result, device, pDescriptorUpdateTemplate, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplate>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorUpdateTemplate(
    VkDevice                                    device,
    VkDescriptorUpdateTemplate                  descriptorUpdateTemplate,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplate>::Dispatch(manager, device, descriptorUpdateTemplate, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplate);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDescriptorUpdateTemplate(device, descriptorUpdateTemplate, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplate>::Dispatch(manager, device, descriptorUpdateTemplate, pAllocator);

    DestroyWrappedHandle<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalBufferProperties(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalBufferInfo*   pExternalBufferInfo,
    VkExternalBufferProperties*                 pExternalBufferProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferProperties>::Dispatch(manager, physicalDevice, pExternalBufferInfo, pExternalBufferProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalBufferProperties(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalBufferInfo);
        EncodeStructPtr(encoder, pExternalBufferProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferProperties>::Dispatch(manager, physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalFenceProperties(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalFenceInfo*    pExternalFenceInfo,
    VkExternalFenceProperties*                  pExternalFenceProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFenceProperties>::Dispatch(manager, physicalDevice, pExternalFenceInfo, pExternalFenceProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalFenceProperties(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFenceProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalFenceInfo);
        EncodeStructPtr(encoder, pExternalFenceProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFenceProperties>::Dispatch(manager, physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalSemaphoreProperties(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo,
    VkExternalSemaphoreProperties*              pExternalSemaphoreProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphoreProperties>::Dispatch(manager, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalSemaphoreProperties(physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphoreProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalSemaphoreInfo);
        EncodeStructPtr(encoder, pExternalSemaphoreProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphoreProperties>::Dispatch(manager, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetLayoutSupport(
    VkDevice                                    device,
    const VkDescriptorSetLayoutCreateInfo*      pCreateInfo,
    VkDescriptorSetLayoutSupport*               pSupport)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupport>::Dispatch(manager, device, pCreateInfo, pSupport);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorSetLayoutCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDescriptorSetLayoutSupport(device, pCreateInfo_unwrapped, pSupport);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupport);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pSupport);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupport>::Dispatch(manager, device, pCreateInfo, pSupport);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCount(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCount>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndirectCount);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndirectCountHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCount>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCount(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCount>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCount);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndexedIndirectCountHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndexedIndirectCount(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCount>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass2(
    VkDevice                                    device,
    const VkRenderPassCreateInfo2*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkRenderPass*                               pRenderPass)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateRenderPass2>::Dispatch(manager, device, pCreateInfo, pAllocator, pRenderPass);

    VkResult result = GetDeviceTable(device)->CreateRenderPass2(device, pCreateInfo, pAllocator, pRenderPass);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, RenderPassWrapper>(device, NoParentWrapper::kHandleValue, pRenderPass, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateRenderPass2);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<RenderPassWrapper>(pRenderPass, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, RenderPassWrapper, VkRenderPassCreateInfo2>(result, device, pRenderPass, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateRenderPass2>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pRenderPass);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass2(
    VkCommandBuffer                             commandBuffer,
    const VkRenderPassBeginInfo*                pRenderPassBegin,
    const VkSubpassBeginInfo*                   pSubpassBeginInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass2>::Dispatch(manager, commandBuffer, pRenderPassBegin, pSubpassBeginInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginRenderPass2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pRenderPassBegin);
        EncodeStructPtr(encoder, pSubpassBeginInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginRenderPass2Handles, pRenderPassBegin);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkRenderPassBeginInfo* pRenderPassBegin_unwrapped = UnwrapStructPtrHandles(pRenderPassBegin, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginRenderPass2(commandBuffer, pRenderPassBegin_unwrapped, pSubpassBeginInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass2>::Dispatch(manager, commandBuffer, pRenderPassBegin, pSubpassBeginInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass2(
    VkCommandBuffer                             commandBuffer,
    const VkSubpassBeginInfo*                   pSubpassBeginInfo,
    const VkSubpassEndInfo*                     pSubpassEndInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdNextSubpass2>::Dispatch(manager, commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdNextSubpass2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSubpassBeginInfo);
        EncodeStructPtr(encoder, pSubpassEndInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdNextSubpass2(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdNextSubpass2>::Dispatch(manager, commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass2(
    VkCommandBuffer                             commandBuffer,
    const VkSubpassEndInfo*                     pSubpassEndInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass2>::Dispatch(manager, commandBuffer, pSubpassEndInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndRenderPass2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSubpassEndInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndRenderPass2(commandBuffer, pSubpassEndInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass2>::Dispatch(manager, commandBuffer, pSubpassEndInfo);
}

VKAPI_ATTR void VKAPI_CALL ResetQueryPool(
    VkDevice                                    device,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery,
    uint32_t                                    queryCount)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetQueryPool>::Dispatch(manager, device, queryPool, firstQuery, queryCount);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetQueryPool);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        encoder->EncodeUInt32Value(queryCount);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->ResetQueryPool(device, queryPool, firstQuery, queryCount);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetQueryPool>::Dispatch(manager, device, queryPool, firstQuery, queryCount);
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreCounterValue(
    VkDevice                                    device,
    VkSemaphore                                 semaphore,
    uint64_t*                                   pValue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSemaphoreCounterValue>::Dispatch(manager, device, semaphore, pValue);

    VkResult result = GetDeviceTable(device)->GetSemaphoreCounterValue(device, semaphore, pValue);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSemaphoreCounterValue);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SemaphoreWrapper>(semaphore);
        encoder->EncodeUInt64Ptr(pValue, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSemaphoreCounterValue>::Dispatch(manager, result, device, semaphore, pValue);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitSemaphores(
    VkDevice                                    device,
    const VkSemaphoreWaitInfo*                  pWaitInfo,
    uint64_t                                    timeout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkWaitSemaphores>::Dispatch(manager, device, pWaitInfo, timeout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreWaitInfo* pWaitInfo_unwrapped = UnwrapStructPtrHandles(pWaitInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->WaitSemaphores(device, pWaitInfo_unwrapped, timeout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkWaitSemaphores);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pWaitInfo);
        encoder->EncodeUInt64Value(timeout);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkWaitSemaphores>::Dispatch(manager, result, device, pWaitInfo, timeout);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SignalSemaphore(
    VkDevice                                    device,
    const VkSemaphoreSignalInfo*                pSignalInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSignalSemaphore>::Dispatch(manager, device, pSignalInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreSignalInfo* pSignalInfo_unwrapped = UnwrapStructPtrHandles(pSignalInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->SignalSemaphore(device, pSignalInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSignalSemaphore);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pSignalInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSignalSemaphore>::Dispatch(manager, result, device, pSignalInfo);

    return result;
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddress(
    VkDevice                                    device,
    const VkBufferDeviceAddressInfo*            pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddress>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferDeviceAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkDeviceAddress result = GetDeviceTable(device)->GetBufferDeviceAddress(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferDeviceAddress);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeVkDeviceAddressValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddress>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetBufferOpaqueCaptureAddress(
    VkDevice                                    device,
    const VkBufferDeviceAddressInfo*            pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddress>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferDeviceAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    uint64_t result = GetDeviceTable(device)->GetBufferOpaqueCaptureAddress(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddress);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt64Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddress>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetDeviceMemoryOpaqueCaptureAddress(
    VkDevice                                    device,
    const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddress>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    uint64_t result = GetDeviceTable(device)->GetDeviceMemoryOpaqueCaptureAddress(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddress);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt64Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddress>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceToolProperties(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pToolCount,
    VkPhysicalDeviceToolProperties*             pToolProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolProperties>::Dispatch(manager, physicalDevice, pToolCount, pToolProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceToolProperties(physicalDevice, pToolCount, pToolProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolProperties);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pToolCount, omit_output_data);
        EncodeStructArray(encoder, pToolProperties, (pToolCount != nullptr) ? (*pToolCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolProperties>::Dispatch(manager, result, physicalDevice, pToolCount, pToolProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePrivateDataSlot(
    VkDevice                                    device,
    const VkPrivateDataSlotCreateInfo*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkPrivateDataSlot*                          pPrivateDataSlot)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreatePrivateDataSlot>::Dispatch(manager, device, pCreateInfo, pAllocator, pPrivateDataSlot);

    VkResult result = GetDeviceTable(device)->CreatePrivateDataSlot(device, pCreateInfo, pAllocator, pPrivateDataSlot);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, PrivateDataSlotWrapper>(device, NoParentWrapper::kHandleValue, pPrivateDataSlot, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreatePrivateDataSlot);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<PrivateDataSlotWrapper>(pPrivateDataSlot, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, PrivateDataSlotWrapper, VkPrivateDataSlotCreateInfo>(result, device, pPrivateDataSlot, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreatePrivateDataSlot>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pPrivateDataSlot);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPrivateDataSlot(
    VkDevice                                    device,
    VkPrivateDataSlot                           privateDataSlot,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyPrivateDataSlot>::Dispatch(manager, device, privateDataSlot, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyPrivateDataSlot);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<PrivateDataSlotWrapper>(privateDataSlot);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyPrivateDataSlot(device, privateDataSlot, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyPrivateDataSlot>::Dispatch(manager, device, privateDataSlot, pAllocator);

    DestroyWrappedHandle<PrivateDataSlotWrapper>(privateDataSlot);
}

VKAPI_ATTR VkResult VKAPI_CALL SetPrivateData(
    VkDevice                                    device,
    VkObjectType                                objectType,
    uint64_t                                    objectHandle,
    VkPrivateDataSlot                           privateDataSlot,
    uint64_t                                    data)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetPrivateData>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, data);

    VkResult result = GetDeviceTable(device)->SetPrivateData(device, objectType, objectHandle, privateDataSlot, data);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetPrivateData);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(objectType);
        encoder->EncodeUInt64Value(GetWrappedId(objectHandle, objectType));
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        encoder->EncodeUInt64Value(data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetPrivateData>::Dispatch(manager, result, device, objectType, objectHandle, privateDataSlot, data);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPrivateData(
    VkDevice                                    device,
    VkObjectType                                objectType,
    uint64_t                                    objectHandle,
    VkPrivateDataSlot                           privateDataSlot,
    uint64_t*                                   pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPrivateData>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, pData);

    GetDeviceTable(device)->GetPrivateData(device, objectType, objectHandle, privateDataSlot, pData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPrivateData);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(objectType);
        encoder->EncodeUInt64Value(GetWrappedId(objectHandle, objectType));
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        encoder->EncodeUInt64Ptr(pData);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPrivateData>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, pData);
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent2(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    const VkDependencyInfo*                     pDependencyInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetEvent2>::Dispatch(manager, commandBuffer, event, pDependencyInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetEvent2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        EncodeStructPtr(encoder, pDependencyInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdSetEvent2Handles, event, pDependencyInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfo_unwrapped = UnwrapStructPtrHandles(pDependencyInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdSetEvent2(commandBuffer, event, pDependencyInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetEvent2>::Dispatch(manager, commandBuffer, event, pDependencyInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent2(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    VkPipelineStageFlags2                       stageMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResetEvent2>::Dispatch(manager, commandBuffer, event, stageMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResetEvent2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeFlags64Value(stageMask);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResetEvent2Handles, event);
    }

    GetDeviceTable(commandBuffer)->CmdResetEvent2(commandBuffer, event, stageMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResetEvent2>::Dispatch(manager, commandBuffer, event, stageMask);
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents2(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    eventCount,
    const VkEvent*                              pEvents,
    const VkDependencyInfo*                     pDependencyInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWaitEvents2>::Dispatch(manager, commandBuffer, eventCount, pEvents, pDependencyInfos);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWaitEvents2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(eventCount);
        encoder->EncodeHandleArray<EventWrapper>(pEvents, eventCount);
        EncodeStructArray(encoder, pDependencyInfos, eventCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWaitEvents2Handles, eventCount, pEvents, pDependencyInfos);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfos_unwrapped = UnwrapStructArrayHandles(pDependencyInfos, eventCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdWaitEvents2(commandBuffer, eventCount, pEvents, pDependencyInfos_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWaitEvents2>::Dispatch(manager, commandBuffer, eventCount, pEvents, pDependencyInfos);
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier2(
    VkCommandBuffer                             commandBuffer,
    const VkDependencyInfo*                     pDependencyInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier2>::Dispatch(manager, commandBuffer, pDependencyInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPipelineBarrier2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pDependencyInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPipelineBarrier2Handles, pDependencyInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfo_unwrapped = UnwrapStructPtrHandles(pDependencyInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPipelineBarrier2(commandBuffer, pDependencyInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier2>::Dispatch(manager, commandBuffer, pDependencyInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp2(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlags2                       stage,
    VkQueryPool                                 queryPool,
    uint32_t                                    query)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp2>::Dispatch(manager, commandBuffer, stage, queryPool, query);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteTimestamp2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlags64Value(stage);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteTimestamp2Handles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdWriteTimestamp2(commandBuffer, stage, queryPool, query);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp2>::Dispatch(manager, commandBuffer, stage, queryPool, query);
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit2(
    VkQueue                                     queue,
    uint32_t                                    submitCount,
    const VkSubmitInfo2*                        pSubmits,
    VkFence                                     fence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueSubmit2>::Dispatch(manager, queue, submitCount, pSubmits, fence);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSubmitInfo2* pSubmits_unwrapped = UnwrapStructArrayHandles(pSubmits, submitCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(queue)->QueueSubmit2(queue, submitCount, pSubmits_unwrapped, fence);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueSubmit2);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Value(submitCount);
        EncodeStructArray(encoder, pSubmits, submitCount);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueSubmit2>::Dispatch(manager, result, queue, submitCount, pSubmits, fence);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer2(
    VkCommandBuffer                             commandBuffer,
    const VkCopyBufferInfo2*                    pCopyBufferInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer2>::Dispatch(manager, commandBuffer, pCopyBufferInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBuffer2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyBufferInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBuffer2Handles, pCopyBufferInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyBufferInfo2* pCopyBufferInfo_unwrapped = UnwrapStructPtrHandles(pCopyBufferInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyBuffer2(commandBuffer, pCopyBufferInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer2>::Dispatch(manager, commandBuffer, pCopyBufferInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage2(
    VkCommandBuffer                             commandBuffer,
    const VkCopyImageInfo2*                     pCopyImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImage2>::Dispatch(manager, commandBuffer, pCopyImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImage2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImage2Handles, pCopyImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageInfo2* pCopyImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyImage2(commandBuffer, pCopyImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImage2>::Dispatch(manager, commandBuffer, pCopyImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage2(
    VkCommandBuffer                             commandBuffer,
    const VkCopyBufferToImageInfo2*             pCopyBufferToImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2>::Dispatch(manager, commandBuffer, pCopyBufferToImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyBufferToImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBufferToImage2Handles, pCopyBufferToImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyBufferToImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyBufferToImage2(commandBuffer, pCopyBufferToImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2>::Dispatch(manager, commandBuffer, pCopyBufferToImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer2(
    VkCommandBuffer                             commandBuffer,
    const VkCopyImageToBufferInfo2*             pCopyImageToBufferInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2>::Dispatch(manager, commandBuffer, pCopyImageToBufferInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyImageToBufferInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImageToBuffer2Handles, pCopyImageToBufferInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageToBufferInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyImageToBuffer2(commandBuffer, pCopyImageToBufferInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2>::Dispatch(manager, commandBuffer, pCopyImageToBufferInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage2(
    VkCommandBuffer                             commandBuffer,
    const VkBlitImageInfo2*                     pBlitImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBlitImage2>::Dispatch(manager, commandBuffer, pBlitImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBlitImage2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBlitImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBlitImage2Handles, pBlitImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBlitImageInfo2* pBlitImageInfo_unwrapped = UnwrapStructPtrHandles(pBlitImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBlitImage2(commandBuffer, pBlitImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBlitImage2>::Dispatch(manager, commandBuffer, pBlitImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage2(
    VkCommandBuffer                             commandBuffer,
    const VkResolveImageInfo2*                  pResolveImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResolveImage2>::Dispatch(manager, commandBuffer, pResolveImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResolveImage2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pResolveImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResolveImage2Handles, pResolveImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkResolveImageInfo2* pResolveImageInfo_unwrapped = UnwrapStructPtrHandles(pResolveImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdResolveImage2(commandBuffer, pResolveImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResolveImage2>::Dispatch(manager, commandBuffer, pResolveImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRendering(
    VkCommandBuffer                             commandBuffer,
    const VkRenderingInfo*                      pRenderingInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginRendering>::Dispatch(manager, commandBuffer, pRenderingInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginRendering);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pRenderingInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginRenderingHandles, pRenderingInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkRenderingInfo* pRenderingInfo_unwrapped = UnwrapStructPtrHandles(pRenderingInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginRendering(commandBuffer, pRenderingInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginRendering>::Dispatch(manager, commandBuffer, pRenderingInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndRendering(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndRendering>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndRendering);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndRendering(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndRendering>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL CmdSetCullMode(
    VkCommandBuffer                             commandBuffer,
    VkCullModeFlags                             cullMode)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetCullMode>::Dispatch(manager, commandBuffer, cullMode);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetCullMode);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(cullMode);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetCullMode(commandBuffer, cullMode);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetCullMode>::Dispatch(manager, commandBuffer, cullMode);
}

VKAPI_ATTR void VKAPI_CALL CmdSetFrontFace(
    VkCommandBuffer                             commandBuffer,
    VkFrontFace                                 frontFace)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetFrontFace>::Dispatch(manager, commandBuffer, frontFace);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetFrontFace);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(frontFace);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetFrontFace(commandBuffer, frontFace);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetFrontFace>::Dispatch(manager, commandBuffer, frontFace);
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveTopology(
    VkCommandBuffer                             commandBuffer,
    VkPrimitiveTopology                         primitiveTopology)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopology>::Dispatch(manager, commandBuffer, primitiveTopology);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopology);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(primitiveTopology);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetPrimitiveTopology(commandBuffer, primitiveTopology);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopology>::Dispatch(manager, commandBuffer, primitiveTopology);
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWithCount(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    viewportCount,
    const VkViewport*                           pViewports)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetViewportWithCount>::Dispatch(manager, commandBuffer, viewportCount, pViewports);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetViewportWithCount);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(viewportCount);
        EncodeStructArray(encoder, pViewports, viewportCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetViewportWithCount(commandBuffer, viewportCount, pViewports);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetViewportWithCount>::Dispatch(manager, commandBuffer, viewportCount, pViewports);
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissorWithCount(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    scissorCount,
    const VkRect2D*                             pScissors)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetScissorWithCount>::Dispatch(manager, commandBuffer, scissorCount, pScissors);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetScissorWithCount);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(scissorCount);
        EncodeStructArray(encoder, pScissors, scissorCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetScissorWithCount(commandBuffer, scissorCount, pScissors);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetScissorWithCount>::Dispatch(manager, commandBuffer, scissorCount, pScissors);
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers2(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstBinding,
    uint32_t                                    bindingCount,
    const VkBuffer*                             pBuffers,
    const VkDeviceSize*                         pOffsets,
    const VkDeviceSize*                         pSizes,
    const VkDeviceSize*                         pStrides)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstBinding);
        encoder->EncodeUInt32Value(bindingCount);
        encoder->EncodeHandleArray<BufferWrapper>(pBuffers, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pOffsets, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pSizes, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pStrides, bindingCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindVertexBuffers2Handles, bindingCount, pBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdBindVertexBuffers2(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthTestEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthTestEnable>::Dispatch(manager, commandBuffer, depthTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthTestEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthTestEnable(commandBuffer, depthTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthTestEnable>::Dispatch(manager, commandBuffer, depthTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthWriteEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthWriteEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnable>::Dispatch(manager, commandBuffer, depthWriteEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthWriteEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthWriteEnable(commandBuffer, depthWriteEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnable>::Dispatch(manager, commandBuffer, depthWriteEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthCompareOp(
    VkCommandBuffer                             commandBuffer,
    VkCompareOp                                 depthCompareOp)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthCompareOp>::Dispatch(manager, commandBuffer, depthCompareOp);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthCompareOp);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(depthCompareOp);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthCompareOp(commandBuffer, depthCompareOp);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthCompareOp>::Dispatch(manager, commandBuffer, depthCompareOp);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBoundsTestEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthBoundsTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnable>::Dispatch(manager, commandBuffer, depthBoundsTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthBoundsTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBoundsTestEnable(commandBuffer, depthBoundsTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnable>::Dispatch(manager, commandBuffer, depthBoundsTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilTestEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    stencilTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilTestEnable>::Dispatch(manager, commandBuffer, stencilTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilTestEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(stencilTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilTestEnable(commandBuffer, stencilTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilTestEnable>::Dispatch(manager, commandBuffer, stencilTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilOp(
    VkCommandBuffer                             commandBuffer,
    VkStencilFaceFlags                          faceMask,
    VkStencilOp                                 failOp,
    VkStencilOp                                 passOp,
    VkStencilOp                                 depthFailOp,
    VkCompareOp                                 compareOp)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilOp>::Dispatch(manager, commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilOp);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(faceMask);
        encoder->EncodeEnumValue(failOp);
        encoder->EncodeEnumValue(passOp);
        encoder->EncodeEnumValue(depthFailOp);
        encoder->EncodeEnumValue(compareOp);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilOp(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilOp>::Dispatch(manager, commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);
}

VKAPI_ATTR void VKAPI_CALL CmdSetRasterizerDiscardEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    rasterizerDiscardEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnable>::Dispatch(manager, commandBuffer, rasterizerDiscardEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(rasterizerDiscardEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetRasterizerDiscardEnable(commandBuffer, rasterizerDiscardEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnable>::Dispatch(manager, commandBuffer, rasterizerDiscardEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBiasEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthBiasEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnable>::Dispatch(manager, commandBuffer, depthBiasEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthBiasEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBiasEnable(commandBuffer, depthBiasEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnable>::Dispatch(manager, commandBuffer, depthBiasEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveRestartEnable(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    primitiveRestartEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnable>::Dispatch(manager, commandBuffer, primitiveRestartEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnable);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(primitiveRestartEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetPrimitiveRestartEnable(commandBuffer, primitiveRestartEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnable>::Dispatch(manager, commandBuffer, primitiveRestartEnable);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceBufferMemoryRequirements(
    VkDevice                                    device,
    const VkDeviceBufferMemoryRequirements*     pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirements>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    GetDeviceTable(device)->GetDeviceBufferMemoryRequirements(device, pInfo, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirements>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageMemoryRequirements(
    VkDevice                                    device,
    const VkDeviceImageMemoryRequirements*      pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirements>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceImageMemoryRequirements* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDeviceImageMemoryRequirements(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirements>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSparseMemoryRequirements(
    VkDevice                                    device,
    const VkDeviceImageMemoryRequirements*      pInfo,
    uint32_t*                                   pSparseMemoryRequirementCount,
    VkSparseImageMemoryRequirements2*           pSparseMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirements>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceImageMemoryRequirements* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDeviceImageSparseMemoryRequirements(device, pInfo_unwrapped, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirements);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt32Ptr(pSparseMemoryRequirementCount);
        EncodeStructArray(encoder, pSparseMemoryRequirements, (pSparseMemoryRequirementCount != nullptr) ? (*pSparseMemoryRequirementCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirements>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL DestroySurfaceKHR(
    VkInstance                                  instance,
    VkSurfaceKHR                                surface,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySurfaceKHR>::Dispatch(manager, instance, surface, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SurfaceKHRWrapper>(surface);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetInstanceTable(instance)->DestroySurfaceKHR(instance, surface, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySurfaceKHR>::Dispatch(manager, instance, surface, pAllocator);

    DestroyWrappedHandle<SurfaceKHRWrapper>(surface);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceSupportKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    VkSurfaceKHR                                surface,
    VkBool32*                                   pSupported)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceSupportKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex, surface, pSupported);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceSupportKHR(physicalDevice, queueFamilyIndex, surface, pSupported);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        encoder->EncodeVkBool32Ptr(pSupported, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceSupportKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, surface, pSupported);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilitiesKHR(
    VkPhysicalDevice                            physicalDevice,
    VkSurfaceKHR                                surface,
    VkSurfaceCapabilitiesKHR*                   pSurfaceCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilitiesKHR>::Dispatch(manager, physicalDevice, surface, pSurfaceCapabilities);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceCapabilitiesKHR(physicalDevice, surface, pSurfaceCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilitiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        EncodeStructPtr(encoder, pSurfaceCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilitiesKHR>::Dispatch(manager, result, physicalDevice, surface, pSurfaceCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceFormatsKHR(
    VkPhysicalDevice                            physicalDevice,
    VkSurfaceKHR                                surface,
    uint32_t*                                   pSurfaceFormatCount,
    VkSurfaceFormatKHR*                         pSurfaceFormats)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormatsKHR>::Dispatch(manager, physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceFormatsKHR(physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormatsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        encoder->EncodeUInt32Ptr(pSurfaceFormatCount, omit_output_data);
        EncodeStructArray(encoder, pSurfaceFormats, (pSurfaceFormatCount != nullptr) ? (*pSurfaceFormatCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormatsKHR>::Dispatch(manager, result, physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfacePresentModesKHR(
    VkPhysicalDevice                            physicalDevice,
    VkSurfaceKHR                                surface,
    uint32_t*                                   pPresentModeCount,
    VkPresentModeKHR*                           pPresentModes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModesKHR>::Dispatch(manager, physicalDevice, surface, pPresentModeCount, pPresentModes);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfacePresentModesKHR(physicalDevice, surface, pPresentModeCount, pPresentModes);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        encoder->EncodeUInt32Ptr(pPresentModeCount, omit_output_data);
        encoder->EncodeEnumArray(pPresentModes, (pPresentModeCount != nullptr) ? (*pPresentModeCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModesKHR>::Dispatch(manager, result, physicalDevice, surface, pPresentModeCount, pPresentModes);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSwapchainKHR(
    VkDevice                                    device,
    const VkSwapchainCreateInfoKHR*             pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSwapchainKHR*                             pSwapchain)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSwapchainKHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pSwapchain);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSwapchainCreateInfoKHR* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateSwapchainKHR(device, pCreateInfo_unwrapped, pAllocator, pSwapchain);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, SwapchainKHRWrapper>(device, NoParentWrapper::kHandleValue, pSwapchain, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSwapchainKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SwapchainKHRWrapper>(pSwapchain, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, SwapchainKHRWrapper, VkSwapchainCreateInfoKHR>(result, device, pSwapchain, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSwapchainKHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pSwapchain);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySwapchainKHR(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySwapchainKHR>::Dispatch(manager, device, swapchain, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySwapchainKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SwapchainKHRWrapper>(swapchain);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroySwapchainKHR(device, swapchain, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySwapchainKHR>::Dispatch(manager, device, swapchain, pAllocator);

    DestroyWrappedHandle<SwapchainKHRWrapper>(swapchain);
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainImagesKHR(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    uint32_t*                                   pSwapchainImageCount,
    VkImage*                                    pSwapchainImages)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSwapchainImagesKHR>::Dispatch(manager, device, swapchain, pSwapchainImageCount, pSwapchainImages);

    VkResult result = GetDeviceTable(device)->GetSwapchainImagesKHR(device, swapchain, pSwapchainImageCount, pSwapchainImages);

    if (result >= 0)
    {
        CreateWrappedHandles<DeviceWrapper, SwapchainKHRWrapper, ImageWrapper>(device, swapchain, pSwapchainImages, (pSwapchainImageCount != nullptr) ? (*pSwapchainImageCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetSwapchainImagesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeUInt32Ptr(pSwapchainImageCount, omit_output_data);
        encoder->EncodeHandleArray<ImageWrapper>(pSwapchainImages, (pSwapchainImageCount != nullptr) ? (*pSwapchainImageCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndGroupCreateApiCallCapture<VkDevice, VkSwapchainKHR, ImageWrapper, void>(result, device, swapchain, (pSwapchainImageCount != nullptr) ? (*pSwapchainImageCount) : 0, pSwapchainImages, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSwapchainImagesKHR>::Dispatch(manager, result, device, swapchain, pSwapchainImageCount, pSwapchainImages);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireNextImageKHR(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    uint64_t                                    timeout,
    VkSemaphore                                 semaphore,
    VkFence                                     fence,
    uint32_t*                                   pImageIndex)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireNextImageKHR>::Dispatch(manager, device, swapchain, timeout, semaphore, fence, pImageIndex);

    VkResult result = GetDeviceTable(device)->AcquireNextImageKHR(device, swapchain, timeout, semaphore, fence, pImageIndex);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireNextImageKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeUInt64Value(timeout);
        encoder->EncodeHandleValue<SemaphoreWrapper>(semaphore);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeUInt32Ptr(pImageIndex, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireNextImageKHR>::Dispatch(manager, result, device, swapchain, timeout, semaphore, fence, pImageIndex);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueuePresentKHR(
    VkQueue                                     queue,
    const VkPresentInfoKHR*                     pPresentInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueuePresentKHR>::Dispatch(manager, queue, pPresentInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPresentInfoKHR* pPresentInfo_unwrapped = UnwrapStructPtrHandles(pPresentInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(queue)->QueuePresentKHR(queue, pPresentInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueuePresentKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        EncodeStructPtr(encoder, pPresentInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueuePresentKHR>::Dispatch(manager, result, queue, pPresentInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceGroupPresentCapabilitiesKHR(
    VkDevice                                    device,
    VkDeviceGroupPresentCapabilitiesKHR*        pDeviceGroupPresentCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPresentCapabilitiesKHR>::Dispatch(manager, device, pDeviceGroupPresentCapabilities);

    VkResult result = GetDeviceTable(device)->GetDeviceGroupPresentCapabilitiesKHR(device, pDeviceGroupPresentCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceGroupPresentCapabilitiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pDeviceGroupPresentCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPresentCapabilitiesKHR>::Dispatch(manager, result, device, pDeviceGroupPresentCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceGroupSurfacePresentModesKHR(
    VkDevice                                    device,
    VkSurfaceKHR                                surface,
    VkDeviceGroupPresentModeFlagsKHR*           pModes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModesKHR>::Dispatch(manager, device, surface, pModes);

    VkResult result = GetDeviceTable(device)->GetDeviceGroupSurfacePresentModesKHR(device, surface, pModes);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        encoder->EncodeFlagsPtr(pModes, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModesKHR>::Dispatch(manager, result, device, surface, pModes);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDevicePresentRectanglesKHR(
    VkPhysicalDevice                            physicalDevice,
    VkSurfaceKHR                                surface,
    uint32_t*                                   pRectCount,
    VkRect2D*                                   pRects)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDevicePresentRectanglesKHR>::Dispatch(manager, physicalDevice, surface, pRectCount, pRects);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDevicePresentRectanglesKHR(physicalDevice, surface, pRectCount, pRects);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDevicePresentRectanglesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        encoder->EncodeUInt32Ptr(pRectCount, omit_output_data);
        EncodeStructArray(encoder, pRects, (pRectCount != nullptr) ? (*pRectCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDevicePresentRectanglesKHR>::Dispatch(manager, result, physicalDevice, surface, pRectCount, pRects);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireNextImage2KHR(
    VkDevice                                    device,
    const VkAcquireNextImageInfoKHR*            pAcquireInfo,
    uint32_t*                                   pImageIndex)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireNextImage2KHR>::Dispatch(manager, device, pAcquireInfo, pImageIndex);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkAcquireNextImageInfoKHR* pAcquireInfo_unwrapped = UnwrapStructPtrHandles(pAcquireInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->AcquireNextImage2KHR(device, pAcquireInfo_unwrapped, pImageIndex);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireNextImage2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAcquireInfo);
        encoder->EncodeUInt32Ptr(pImageIndex, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireNextImage2KHR>::Dispatch(manager, result, device, pAcquireInfo, pImageIndex);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkDisplayPropertiesKHR*                     pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPropertiesKHR>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceDisplayPropertiesKHR(physicalDevice, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, NoParentWrapper, VkDisplayPropertiesKHR>(physicalDevice, NoParentWrapper::kHandleValue, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, VkDisplayPropertiesKHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayPropertiesKHR* handle_struct)->DisplayKHRWrapper* { return GetWrapper<DisplayKHRWrapper>(handle_struct->display); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPropertiesKHR>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPlanePropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkDisplayPlanePropertiesKHR*                pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlanePropertiesKHR>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceDisplayPlanePropertiesKHR(physicalDevice, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, NoParentWrapper, VkDisplayPlanePropertiesKHR>(physicalDevice, NoParentWrapper::kHandleValue, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlanePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, VkDisplayPlanePropertiesKHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayPlanePropertiesKHR* handle_struct)->DisplayKHRWrapper* { return GetWrapper<DisplayKHRWrapper>(handle_struct->currentDisplay); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlanePropertiesKHR>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneSupportedDisplaysKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    planeIndex,
    uint32_t*                                   pDisplayCount,
    VkDisplayKHR*                               pDisplays)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneSupportedDisplaysKHR>::Dispatch(manager, physicalDevice, planeIndex, pDisplayCount, pDisplays);

    VkResult result = GetInstanceTable(physicalDevice)->GetDisplayPlaneSupportedDisplaysKHR(physicalDevice, planeIndex, pDisplayCount, pDisplays);

    if (result >= 0)
    {
        CreateWrappedHandles<PhysicalDeviceWrapper, NoParentWrapper, DisplayKHRWrapper>(physicalDevice, NoParentWrapper::kHandleValue, pDisplays, (pDisplayCount != nullptr) ? (*pDisplayCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDisplayPlaneSupportedDisplaysKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(planeIndex);
        encoder->EncodeUInt32Ptr(pDisplayCount, omit_output_data);
        encoder->EncodeHandleArray<DisplayKHRWrapper>(pDisplays, (pDisplayCount != nullptr) ? (*pDisplayCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndGroupCreateApiCallCapture<VkPhysicalDevice, void*, DisplayKHRWrapper, void>(result, physicalDevice, nullptr, (pDisplayCount != nullptr) ? (*pDisplayCount) : 0, pDisplays, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneSupportedDisplaysKHR>::Dispatch(manager, result, physicalDevice, planeIndex, pDisplayCount, pDisplays);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayModePropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayKHR                                display,
    uint32_t*                                   pPropertyCount,
    VkDisplayModePropertiesKHR*                 pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDisplayModePropertiesKHR>::Dispatch(manager, physicalDevice, display, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetDisplayModePropertiesKHR(physicalDevice, display, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, DisplayKHRWrapper, VkDisplayModePropertiesKHR>(physicalDevice, display, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDisplayModePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayModeKHRWrapper, VkDisplayModePropertiesKHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayModePropertiesKHR* handle_struct)->DisplayModeKHRWrapper* { return GetWrapper<DisplayModeKHRWrapper>(handle_struct->displayMode); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDisplayModePropertiesKHR>::Dispatch(manager, result, physicalDevice, display, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDisplayModeKHR(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayKHR                                display,
    const VkDisplayModeCreateInfoKHR*           pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDisplayModeKHR*                           pMode)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDisplayModeKHR>::Dispatch(manager, physicalDevice, display, pCreateInfo, pAllocator, pMode);

    VkResult result = GetInstanceTable(physicalDevice)->CreateDisplayModeKHR(physicalDevice, display, pCreateInfo, pAllocator, pMode);

    if (result >= 0)
    {
        CreateWrappedHandle<PhysicalDeviceWrapper, DisplayKHRWrapper, DisplayModeKHRWrapper>(physicalDevice, display, pMode, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDisplayModeKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DisplayModeKHRWrapper>(pMode, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkPhysicalDevice, DisplayModeKHRWrapper, VkDisplayModeCreateInfoKHR>(result, physicalDevice, pMode, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDisplayModeKHR>::Dispatch(manager, result, physicalDevice, display, pCreateInfo, pAllocator, pMode);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneCapabilitiesKHR(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayModeKHR                            mode,
    uint32_t                                    planeIndex,
    VkDisplayPlaneCapabilitiesKHR*              pCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilitiesKHR>::Dispatch(manager, physicalDevice, mode, planeIndex, pCapabilities);

    VkResult result = GetInstanceTable(physicalDevice)->GetDisplayPlaneCapabilitiesKHR(physicalDevice, mode, planeIndex, pCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilitiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayModeKHRWrapper>(mode);
        encoder->EncodeUInt32Value(planeIndex);
        EncodeStructPtr(encoder, pCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilitiesKHR>::Dispatch(manager, result, physicalDevice, mode, planeIndex, pCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDisplayPlaneSurfaceKHR(
    VkInstance                                  instance,
    const VkDisplaySurfaceCreateInfoKHR*        pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDisplayPlaneSurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDisplaySurfaceCreateInfoKHR* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetInstanceTable(instance)->CreateDisplayPlaneSurfaceKHR(instance, pCreateInfo_unwrapped, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDisplayPlaneSurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkDisplaySurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDisplayPlaneSurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSharedSwapchainsKHR(
    VkDevice                                    device,
    uint32_t                                    swapchainCount,
    const VkSwapchainCreateInfoKHR*             pCreateInfos,
    const VkAllocationCallbacks*                pAllocator,
    VkSwapchainKHR*                             pSwapchains)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSharedSwapchainsKHR>::Dispatch(manager, device, swapchainCount, pCreateInfos, pAllocator, pSwapchains);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSwapchainCreateInfoKHR* pCreateInfos_unwrapped = UnwrapStructArrayHandles(pCreateInfos, swapchainCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateSharedSwapchainsKHR(device, swapchainCount, pCreateInfos_unwrapped, pAllocator, pSwapchains);

    if (result >= 0)
    {
        CreateWrappedHandles<DeviceWrapper, NoParentWrapper, SwapchainKHRWrapper>(device, NoParentWrapper::kHandleValue, pSwapchains, swapchainCount, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSharedSwapchainsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(swapchainCount);
        EncodeStructArray(encoder, pCreateInfos, swapchainCount);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandleArray<SwapchainKHRWrapper>(pSwapchains, swapchainCount, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndGroupCreateApiCallCapture<VkDevice, void*, SwapchainKHRWrapper, VkSwapchainCreateInfoKHR>(result, device, nullptr, swapchainCount, pSwapchains, pCreateInfos);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSharedSwapchainsKHR>::Dispatch(manager, result, device, swapchainCount, pCreateInfos, pAllocator, pSwapchains);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateXlibSurfaceKHR(
    VkInstance                                  instance,
    const VkXlibSurfaceCreateInfoKHR*           pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateXlibSurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateXlibSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateXlibSurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkXlibSurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateXlibSurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceXlibPresentationSupportKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    Display*                                    dpy,
    VisualID                                    visualID)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceXlibPresentationSupportKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex, dpy, visualID);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceXlibPresentationSupportKHR(physicalDevice, queueFamilyIndex, dpy, visualID);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceXlibPresentationSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVoidPtr(dpy);
        encoder->EncodeSizeTValue(visualID);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceXlibPresentationSupportKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, dpy, visualID);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateXcbSurfaceKHR(
    VkInstance                                  instance,
    const VkXcbSurfaceCreateInfoKHR*            pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateXcbSurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateXcbSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateXcbSurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkXcbSurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateXcbSurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceXcbPresentationSupportKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    xcb_connection_t*                           connection,
    xcb_visualid_t                              visual_id)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceXcbPresentationSupportKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex, connection, visual_id);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceXcbPresentationSupportKHR(physicalDevice, queueFamilyIndex, connection, visual_id);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceXcbPresentationSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVoidPtr(connection);
        encoder->EncodeUInt32Value(visual_id);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceXcbPresentationSupportKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, connection, visual_id);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateWaylandSurfaceKHR(
    VkInstance                                  instance,
    const VkWaylandSurfaceCreateInfoKHR*        pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateWaylandSurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateWaylandSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateWaylandSurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkWaylandSurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateWaylandSurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceWaylandPresentationSupportKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    struct wl_display*                          display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceWaylandPresentationSupportKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex, display);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceWaylandPresentationSupportKHR(physicalDevice, queueFamilyIndex, display);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceWaylandPresentationSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVoidPtr(display);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceWaylandPresentationSupportKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateAndroidSurfaceKHR(
    VkInstance                                  instance,
    const VkAndroidSurfaceCreateInfoKHR*        pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateAndroidSurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateAndroidSurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateAndroidSurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkAndroidSurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateAndroidSurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateWin32SurfaceKHR(
    VkInstance                                  instance,
    const VkWin32SurfaceCreateInfoKHR*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateWin32SurfaceKHR>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateWin32SurfaceKHR(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateWin32SurfaceKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkWin32SurfaceCreateInfoKHR>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateWin32SurfaceKHR>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceWin32PresentationSupportKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceWin32PresentationSupportKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceWin32PresentationSupportKHR(physicalDevice, queueFamilyIndex);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceWin32PresentationSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceWin32PresentationSupportKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoCapabilitiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkVideoProfileInfoKHR*                pVideoProfile,
    VkVideoCapabilitiesKHR*                     pCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoCapabilitiesKHR>::Dispatch(manager, physicalDevice, pVideoProfile, pCapabilities);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceVideoCapabilitiesKHR(physicalDevice, pVideoProfile, pCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoCapabilitiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pVideoProfile);
        EncodeStructPtr(encoder, pCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoCapabilitiesKHR>::Dispatch(manager, result, physicalDevice, pVideoProfile, pCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoFormatPropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceVideoFormatInfoKHR*   pVideoFormatInfo,
    uint32_t*                                   pVideoFormatPropertyCount,
    VkVideoFormatPropertiesKHR*                 pVideoFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoFormatPropertiesKHR>::Dispatch(manager, physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceVideoFormatPropertiesKHR(physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoFormatPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pVideoFormatInfo);
        encoder->EncodeUInt32Ptr(pVideoFormatPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pVideoFormatProperties, (pVideoFormatPropertyCount != nullptr) ? (*pVideoFormatPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoFormatPropertiesKHR>::Dispatch(manager, result, physicalDevice, pVideoFormatInfo, pVideoFormatPropertyCount, pVideoFormatProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateVideoSessionKHR(
    VkDevice                                    device,
    const VkVideoSessionCreateInfoKHR*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkVideoSessionKHR*                          pVideoSession)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateVideoSessionKHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pVideoSession);

    VkResult result = GetDeviceTable(device)->CreateVideoSessionKHR(device, pCreateInfo, pAllocator, pVideoSession);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, VideoSessionKHRWrapper>(device, NoParentWrapper::kHandleValue, pVideoSession, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateVideoSessionKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<VideoSessionKHRWrapper>(pVideoSession, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, VideoSessionKHRWrapper, VkVideoSessionCreateInfoKHR>(result, device, pVideoSession, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateVideoSessionKHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pVideoSession);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyVideoSessionKHR(
    VkDevice                                    device,
    VkVideoSessionKHR                           videoSession,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyVideoSessionKHR>::Dispatch(manager, device, videoSession, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyVideoSessionKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<VideoSessionKHRWrapper>(videoSession);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<VideoSessionKHRWrapper>(videoSession);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyVideoSessionKHR(device, videoSession, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyVideoSessionKHR>::Dispatch(manager, device, videoSession, pAllocator);

    DestroyWrappedHandle<VideoSessionKHRWrapper>(videoSession);
}

VKAPI_ATTR VkResult VKAPI_CALL GetVideoSessionMemoryRequirementsKHR(
    VkDevice                                    device,
    VkVideoSessionKHR                           videoSession,
    uint32_t*                                   pMemoryRequirementsCount,
    VkVideoSessionMemoryRequirementsKHR*        pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetVideoSessionMemoryRequirementsKHR>::Dispatch(manager, device, videoSession, pMemoryRequirementsCount, pMemoryRequirements);

    VkResult result = GetDeviceTable(device)->GetVideoSessionMemoryRequirementsKHR(device, videoSession, pMemoryRequirementsCount, pMemoryRequirements);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetVideoSessionMemoryRequirementsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<VideoSessionKHRWrapper>(videoSession);
        encoder->EncodeUInt32Ptr(pMemoryRequirementsCount, omit_output_data);
        EncodeStructArray(encoder, pMemoryRequirements, (pMemoryRequirementsCount != nullptr) ? (*pMemoryRequirementsCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetVideoSessionMemoryRequirementsKHR>::Dispatch(manager, result, device, videoSession, pMemoryRequirementsCount, pMemoryRequirements);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindVideoSessionMemoryKHR(
    VkDevice                                    device,
    VkVideoSessionKHR                           videoSession,
    uint32_t                                    bindSessionMemoryInfoCount,
    const VkBindVideoSessionMemoryInfoKHR*      pBindSessionMemoryInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindVideoSessionMemoryKHR>::Dispatch(manager, device, videoSession, bindSessionMemoryInfoCount, pBindSessionMemoryInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindVideoSessionMemoryInfoKHR* pBindSessionMemoryInfos_unwrapped = UnwrapStructArrayHandles(pBindSessionMemoryInfos, bindSessionMemoryInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindVideoSessionMemoryKHR(device, videoSession, bindSessionMemoryInfoCount, pBindSessionMemoryInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindVideoSessionMemoryKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<VideoSessionKHRWrapper>(videoSession);
        encoder->EncodeUInt32Value(bindSessionMemoryInfoCount);
        EncodeStructArray(encoder, pBindSessionMemoryInfos, bindSessionMemoryInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindVideoSessionMemoryKHR>::Dispatch(manager, result, device, videoSession, bindSessionMemoryInfoCount, pBindSessionMemoryInfos);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateVideoSessionParametersKHR(
    VkDevice                                    device,
    const VkVideoSessionParametersCreateInfoKHR* pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkVideoSessionParametersKHR*                pVideoSessionParameters)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateVideoSessionParametersKHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pVideoSessionParameters);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkVideoSessionParametersCreateInfoKHR* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateVideoSessionParametersKHR(device, pCreateInfo_unwrapped, pAllocator, pVideoSessionParameters);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, VideoSessionParametersKHRWrapper>(device, NoParentWrapper::kHandleValue, pVideoSessionParameters, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateVideoSessionParametersKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<VideoSessionParametersKHRWrapper>(pVideoSessionParameters, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, VideoSessionParametersKHRWrapper, VkVideoSessionParametersCreateInfoKHR>(result, device, pVideoSessionParameters, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateVideoSessionParametersKHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pVideoSessionParameters);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL UpdateVideoSessionParametersKHR(
    VkDevice                                    device,
    VkVideoSessionParametersKHR                 videoSessionParameters,
    const VkVideoSessionParametersUpdateInfoKHR* pUpdateInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkUpdateVideoSessionParametersKHR>::Dispatch(manager, device, videoSessionParameters, pUpdateInfo);

    VkResult result = GetDeviceTable(device)->UpdateVideoSessionParametersKHR(device, videoSessionParameters, pUpdateInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkUpdateVideoSessionParametersKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<VideoSessionParametersKHRWrapper>(videoSessionParameters);
        EncodeStructPtr(encoder, pUpdateInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkUpdateVideoSessionParametersKHR>::Dispatch(manager, result, device, videoSessionParameters, pUpdateInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyVideoSessionParametersKHR(
    VkDevice                                    device,
    VkVideoSessionParametersKHR                 videoSessionParameters,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyVideoSessionParametersKHR>::Dispatch(manager, device, videoSessionParameters, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyVideoSessionParametersKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<VideoSessionParametersKHRWrapper>(videoSessionParameters);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<VideoSessionParametersKHRWrapper>(videoSessionParameters);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyVideoSessionParametersKHR(device, videoSessionParameters, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyVideoSessionParametersKHR>::Dispatch(manager, device, videoSessionParameters, pAllocator);

    DestroyWrappedHandle<VideoSessionParametersKHRWrapper>(videoSessionParameters);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginVideoCodingKHR(
    VkCommandBuffer                             commandBuffer,
    const VkVideoBeginCodingInfoKHR*            pBeginInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginVideoCodingKHR>::Dispatch(manager, commandBuffer, pBeginInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginVideoCodingKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBeginInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginVideoCodingKHRHandles, pBeginInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkVideoBeginCodingInfoKHR* pBeginInfo_unwrapped = UnwrapStructPtrHandles(pBeginInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginVideoCodingKHR(commandBuffer, pBeginInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginVideoCodingKHR>::Dispatch(manager, commandBuffer, pBeginInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndVideoCodingKHR(
    VkCommandBuffer                             commandBuffer,
    const VkVideoEndCodingInfoKHR*              pEndCodingInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndVideoCodingKHR>::Dispatch(manager, commandBuffer, pEndCodingInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndVideoCodingKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pEndCodingInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndVideoCodingKHR(commandBuffer, pEndCodingInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndVideoCodingKHR>::Dispatch(manager, commandBuffer, pEndCodingInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdControlVideoCodingKHR(
    VkCommandBuffer                             commandBuffer,
    const VkVideoCodingControlInfoKHR*          pCodingControlInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdControlVideoCodingKHR>::Dispatch(manager, commandBuffer, pCodingControlInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdControlVideoCodingKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCodingControlInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdControlVideoCodingKHR(commandBuffer, pCodingControlInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdControlVideoCodingKHR>::Dispatch(manager, commandBuffer, pCodingControlInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdDecodeVideoKHR(
    VkCommandBuffer                             commandBuffer,
    const VkVideoDecodeInfoKHR*                 pDecodeInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDecodeVideoKHR>::Dispatch(manager, commandBuffer, pDecodeInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDecodeVideoKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pDecodeInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDecodeVideoKHRHandles, pDecodeInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkVideoDecodeInfoKHR* pDecodeInfo_unwrapped = UnwrapStructPtrHandles(pDecodeInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdDecodeVideoKHR(commandBuffer, pDecodeInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDecodeVideoKHR>::Dispatch(manager, commandBuffer, pDecodeInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderingKHR(
    VkCommandBuffer                             commandBuffer,
    const VkRenderingInfo*                      pRenderingInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginRenderingKHR>::Dispatch(manager, commandBuffer, pRenderingInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginRenderingKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pRenderingInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginRenderingKHRHandles, pRenderingInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkRenderingInfo* pRenderingInfo_unwrapped = UnwrapStructPtrHandles(pRenderingInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginRenderingKHR(commandBuffer, pRenderingInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginRenderingKHR>::Dispatch(manager, commandBuffer, pRenderingInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderingKHR(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndRenderingKHR>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndRenderingKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndRenderingKHR(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndRenderingKHR>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFeatures2KHR(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceFeatures2*                  pFeatures)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2KHR>::Dispatch(manager, physicalDevice, pFeatures);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFeatures2KHR(physicalDevice, pFeatures);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pFeatures);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFeatures2KHR>::Dispatch(manager, physicalDevice, pFeatures);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceProperties2*                pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2KHR>::Dispatch(manager, physicalDevice, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceProperties2KHR(physicalDevice, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceProperties2KHR>::Dispatch(manager, physicalDevice, pProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceFormatProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkFormatProperties2*                        pFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2KHR>::Dispatch(manager, physicalDevice, format, pFormatProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceFormatProperties2KHR(physicalDevice, format, pFormatProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        EncodeStructPtr(encoder, pFormatProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFormatProperties2KHR>::Dispatch(manager, physicalDevice, format, pFormatProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceImageFormatProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceImageFormatInfo2*     pImageFormatInfo,
    VkImageFormatProperties2*                   pImageFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2KHR>::Dispatch(manager, physicalDevice, pImageFormatInfo, pImageFormatProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceImageFormatProperties2KHR(physicalDevice, pImageFormatInfo, pImageFormatProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pImageFormatInfo);
        EncodeStructPtr(encoder, pImageFormatProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceImageFormatProperties2KHR>::Dispatch(manager, result, physicalDevice, pImageFormatInfo, pImageFormatProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pQueueFamilyPropertyCount,
    VkQueueFamilyProperties2*                   pQueueFamilyProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2KHR>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    manager->OverrideGetPhysicalDeviceQueueFamilyProperties2KHR(physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pQueueFamilyPropertyCount);
        EncodeStructArray(encoder, pQueueFamilyProperties, (pQueueFamilyPropertyCount != nullptr) ? (*pQueueFamilyPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyProperties2KHR>::Dispatch(manager, physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMemoryProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    VkPhysicalDeviceMemoryProperties2*          pMemoryProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2KHR>::Dispatch(manager, physicalDevice, pMemoryProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceMemoryProperties2KHR(physicalDevice, pMemoryProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pMemoryProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMemoryProperties2KHR>::Dispatch(manager, physicalDevice, pMemoryProperties);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceSparseImageFormatProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo,
    uint32_t*                                   pPropertyCount,
    VkSparseImageFormatProperties2*             pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2KHR>::Dispatch(manager, physicalDevice, pFormatInfo, pPropertyCount, pProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceSparseImageFormatProperties2KHR(physicalDevice, pFormatInfo, pPropertyCount, pProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pFormatInfo);
        encoder->EncodeUInt32Ptr(pPropertyCount);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSparseImageFormatProperties2KHR>::Dispatch(manager, physicalDevice, pFormatInfo, pPropertyCount, pProperties);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceGroupPeerMemoryFeaturesKHR(
    VkDevice                                    device,
    uint32_t                                    heapIndex,
    uint32_t                                    localDeviceIndex,
    uint32_t                                    remoteDeviceIndex,
    VkPeerMemoryFeatureFlags*                   pPeerMemoryFeatures)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeaturesKHR>::Dispatch(manager, device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);

    GetDeviceTable(device)->GetDeviceGroupPeerMemoryFeaturesKHR(device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeaturesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(heapIndex);
        encoder->EncodeUInt32Value(localDeviceIndex);
        encoder->EncodeUInt32Value(remoteDeviceIndex);
        encoder->EncodeFlagsPtr(pPeerMemoryFeatures);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceGroupPeerMemoryFeaturesKHR>::Dispatch(manager, device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDeviceMaskKHR(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    deviceMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDeviceMaskKHR>::Dispatch(manager, commandBuffer, deviceMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDeviceMaskKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(deviceMask);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDeviceMaskKHR(commandBuffer, deviceMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDeviceMaskKHR>::Dispatch(manager, commandBuffer, deviceMask);
}

VKAPI_ATTR void VKAPI_CALL CmdDispatchBaseKHR(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    baseGroupX,
    uint32_t                                    baseGroupY,
    uint32_t                                    baseGroupZ,
    uint32_t                                    groupCountX,
    uint32_t                                    groupCountY,
    uint32_t                                    groupCountZ)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDispatchBaseKHR>::Dispatch(manager, commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDispatchBaseKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(baseGroupX);
        encoder->EncodeUInt32Value(baseGroupY);
        encoder->EncodeUInt32Value(baseGroupZ);
        encoder->EncodeUInt32Value(groupCountX);
        encoder->EncodeUInt32Value(groupCountY);
        encoder->EncodeUInt32Value(groupCountZ);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDispatchBaseKHR(commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDispatchBaseKHR>::Dispatch(manager, commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ);
}

VKAPI_ATTR void VKAPI_CALL TrimCommandPoolKHR(
    VkDevice                                    device,
    VkCommandPool                               commandPool,
    VkCommandPoolTrimFlags                      flags)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkTrimCommandPoolKHR>::Dispatch(manager, device, commandPool, flags);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkTrimCommandPoolKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<CommandPoolWrapper>(commandPool);
        encoder->EncodeFlagsValue(flags);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->TrimCommandPoolKHR(device, commandPool, flags);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkTrimCommandPoolKHR>::Dispatch(manager, device, commandPool, flags);
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceGroupsKHR(
    VkInstance                                  instance,
    uint32_t*                                   pPhysicalDeviceGroupCount,
    VkPhysicalDeviceGroupProperties*            pPhysicalDeviceGroupProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroupsKHR>::Dispatch(manager, instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    VkResult result = GetInstanceTable(instance)->EnumeratePhysicalDeviceGroupsKHR(instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<InstanceWrapper, NoParentWrapper, VkPhysicalDeviceGroupProperties>(instance, NoParentWrapper::kHandleValue, pPhysicalDeviceGroupProperties, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroupsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeUInt32Ptr(pPhysicalDeviceGroupCount, omit_output_data);
        EncodeStructArray(encoder, pPhysicalDeviceGroupProperties, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkInstance, PhysicalDeviceWrapper, VkPhysicalDeviceGroupProperties>(result, instance, (pPhysicalDeviceGroupCount != nullptr) ? (*pPhysicalDeviceGroupCount) : 0, pPhysicalDeviceGroupProperties, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceGroupsKHR>::Dispatch(manager, result, instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalBufferPropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalBufferInfo*   pExternalBufferInfo,
    VkExternalBufferProperties*                 pExternalBufferProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferPropertiesKHR>::Dispatch(manager, physicalDevice, pExternalBufferInfo, pExternalBufferProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalBufferPropertiesKHR(physicalDevice, pExternalBufferInfo, pExternalBufferProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalBufferInfo);
        EncodeStructPtr(encoder, pExternalBufferProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalBufferPropertiesKHR>::Dispatch(manager, physicalDevice, pExternalBufferInfo, pExternalBufferProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandleKHR(
    VkDevice                                    device,
    const VkMemoryGetWin32HandleInfoKHR*        pGetWin32HandleInfo,
    HANDLE*                                     pHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandleKHR>::Dispatch(manager, device, pGetWin32HandleInfo, pHandle);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryGetWin32HandleInfoKHR* pGetWin32HandleInfo_unwrapped = UnwrapStructPtrHandles(pGetWin32HandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetMemoryWin32HandleKHR(device, pGetWin32HandleInfo_unwrapped, pHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryWin32HandleKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetWin32HandleInfo);
        encoder->EncodeVoidPtrPtr(pHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandleKHR>::Dispatch(manager, result, device, pGetWin32HandleInfo, pHandle);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandlePropertiesKHR(
    VkDevice                                    device,
    VkExternalMemoryHandleTypeFlagBits          handleType,
    HANDLE                                      handle,
    VkMemoryWin32HandlePropertiesKHR*           pMemoryWin32HandleProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandlePropertiesKHR>::Dispatch(manager, device, handleType, handle, pMemoryWin32HandleProperties);

    VkResult result = GetDeviceTable(device)->GetMemoryWin32HandlePropertiesKHR(device, handleType, handle, pMemoryWin32HandleProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryWin32HandlePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(handleType);
        encoder->EncodeVoidPtr(handle);
        EncodeStructPtr(encoder, pMemoryWin32HandleProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandlePropertiesKHR>::Dispatch(manager, result, device, handleType, handle, pMemoryWin32HandleProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryFdKHR(
    VkDevice                                    device,
    const VkMemoryGetFdInfoKHR*                 pGetFdInfo,
    int*                                        pFd)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryFdKHR>::Dispatch(manager, device, pGetFdInfo, pFd);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryGetFdInfoKHR* pGetFdInfo_unwrapped = UnwrapStructPtrHandles(pGetFdInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetMemoryFdKHR(device, pGetFdInfo_unwrapped, pFd);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryFdKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetFdInfo);
        encoder->EncodeInt32Ptr(pFd, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryFdKHR>::Dispatch(manager, result, device, pGetFdInfo, pFd);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryFdPropertiesKHR(
    VkDevice                                    device,
    VkExternalMemoryHandleTypeFlagBits          handleType,
    int                                         fd,
    VkMemoryFdPropertiesKHR*                    pMemoryFdProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryFdPropertiesKHR>::Dispatch(manager, device, handleType, fd, pMemoryFdProperties);

    VkResult result = GetDeviceTable(device)->GetMemoryFdPropertiesKHR(device, handleType, fd, pMemoryFdProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryFdPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(handleType);
        encoder->EncodeInt32Value(fd);
        EncodeStructPtr(encoder, pMemoryFdProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryFdPropertiesKHR>::Dispatch(manager, result, device, handleType, fd, pMemoryFdProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalSemaphorePropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo,
    VkExternalSemaphoreProperties*              pExternalSemaphoreProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR>::Dispatch(manager, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalSemaphorePropertiesKHR(physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalSemaphoreInfo);
        EncodeStructPtr(encoder, pExternalSemaphoreProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalSemaphorePropertiesKHR>::Dispatch(manager, physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL ImportSemaphoreWin32HandleKHR(
    VkDevice                                    device,
    const VkImportSemaphoreWin32HandleInfoKHR*  pImportSemaphoreWin32HandleInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkImportSemaphoreWin32HandleKHR>::Dispatch(manager, device, pImportSemaphoreWin32HandleInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImportSemaphoreWin32HandleInfoKHR* pImportSemaphoreWin32HandleInfo_unwrapped = UnwrapStructPtrHandles(pImportSemaphoreWin32HandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ImportSemaphoreWin32HandleKHR(device, pImportSemaphoreWin32HandleInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkImportSemaphoreWin32HandleKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pImportSemaphoreWin32HandleInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkImportSemaphoreWin32HandleKHR>::Dispatch(manager, result, device, pImportSemaphoreWin32HandleInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreWin32HandleKHR(
    VkDevice                                    device,
    const VkSemaphoreGetWin32HandleInfoKHR*     pGetWin32HandleInfo,
    HANDLE*                                     pHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSemaphoreWin32HandleKHR>::Dispatch(manager, device, pGetWin32HandleInfo, pHandle);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreGetWin32HandleInfoKHR* pGetWin32HandleInfo_unwrapped = UnwrapStructPtrHandles(pGetWin32HandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetSemaphoreWin32HandleKHR(device, pGetWin32HandleInfo_unwrapped, pHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSemaphoreWin32HandleKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetWin32HandleInfo);
        encoder->EncodeVoidPtrPtr(pHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSemaphoreWin32HandleKHR>::Dispatch(manager, result, device, pGetWin32HandleInfo, pHandle);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ImportSemaphoreFdKHR(
    VkDevice                                    device,
    const VkImportSemaphoreFdInfoKHR*           pImportSemaphoreFdInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkImportSemaphoreFdKHR>::Dispatch(manager, device, pImportSemaphoreFdInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo_unwrapped = UnwrapStructPtrHandles(pImportSemaphoreFdInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ImportSemaphoreFdKHR(device, pImportSemaphoreFdInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkImportSemaphoreFdKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pImportSemaphoreFdInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkImportSemaphoreFdKHR>::Dispatch(manager, result, device, pImportSemaphoreFdInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreFdKHR(
    VkDevice                                    device,
    const VkSemaphoreGetFdInfoKHR*              pGetFdInfo,
    int*                                        pFd)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSemaphoreFdKHR>::Dispatch(manager, device, pGetFdInfo, pFd);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreGetFdInfoKHR* pGetFdInfo_unwrapped = UnwrapStructPtrHandles(pGetFdInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetSemaphoreFdKHR(device, pGetFdInfo_unwrapped, pFd);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSemaphoreFdKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetFdInfo);
        encoder->EncodeInt32Ptr(pFd, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSemaphoreFdKHR>::Dispatch(manager, result, device, pGetFdInfo, pFd);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSetKHR(
    VkCommandBuffer                             commandBuffer,
    VkPipelineBindPoint                         pipelineBindPoint,
    VkPipelineLayout                            layout,
    uint32_t                                    set,
    uint32_t                                    descriptorWriteCount,
    const VkWriteDescriptorSet*                 pDescriptorWrites)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSetKHR>::Dispatch(manager, commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPushDescriptorSetKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineBindPoint);
        encoder->EncodeHandleValue<PipelineLayoutWrapper>(layout);
        encoder->EncodeUInt32Value(set);
        encoder->EncodeUInt32Value(descriptorWriteCount);
        EncodeStructArray(encoder, pDescriptorWrites, descriptorWriteCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPushDescriptorSetKHRHandles, layout, descriptorWriteCount, pDescriptorWrites);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkWriteDescriptorSet* pDescriptorWrites_unwrapped = UnwrapStructArrayHandles(pDescriptorWrites, descriptorWriteCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPushDescriptorSetKHR(commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSetKHR>::Dispatch(manager, commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDescriptorUpdateTemplateKHR(
    VkDevice                                    device,
    const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDescriptorUpdateTemplate*                 pDescriptorUpdateTemplate)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplateKHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateDescriptorUpdateTemplateKHR(device, pCreateInfo_unwrapped, pAllocator, pDescriptorUpdateTemplate);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, DescriptorUpdateTemplateWrapper>(device, NoParentWrapper::kHandleValue, pDescriptorUpdateTemplate, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplateKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DescriptorUpdateTemplateWrapper>(pDescriptorUpdateTemplate, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DescriptorUpdateTemplateWrapper, VkDescriptorUpdateTemplateCreateInfo>(result, device, pDescriptorUpdateTemplate, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDescriptorUpdateTemplateKHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDescriptorUpdateTemplateKHR(
    VkDevice                                    device,
    VkDescriptorUpdateTemplate                  descriptorUpdateTemplate,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplateKHR>::Dispatch(manager, device, descriptorUpdateTemplate, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplateKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDescriptorUpdateTemplateKHR(device, descriptorUpdateTemplate, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDescriptorUpdateTemplateKHR>::Dispatch(manager, device, descriptorUpdateTemplate, pAllocator);

    DestroyWrappedHandle<DescriptorUpdateTemplateWrapper>(descriptorUpdateTemplate);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateRenderPass2KHR(
    VkDevice                                    device,
    const VkRenderPassCreateInfo2*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkRenderPass*                               pRenderPass)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateRenderPass2KHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pRenderPass);

    VkResult result = GetDeviceTable(device)->CreateRenderPass2KHR(device, pCreateInfo, pAllocator, pRenderPass);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, RenderPassWrapper>(device, NoParentWrapper::kHandleValue, pRenderPass, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateRenderPass2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<RenderPassWrapper>(pRenderPass, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, RenderPassWrapper, VkRenderPassCreateInfo2>(result, device, pRenderPass, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateRenderPass2KHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pRenderPass);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBeginRenderPass2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkRenderPassBeginInfo*                pRenderPassBegin,
    const VkSubpassBeginInfo*                   pSubpassBeginInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass2KHR>::Dispatch(manager, commandBuffer, pRenderPassBegin, pSubpassBeginInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginRenderPass2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pRenderPassBegin);
        EncodeStructPtr(encoder, pSubpassBeginInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginRenderPass2KHRHandles, pRenderPassBegin);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkRenderPassBeginInfo* pRenderPassBegin_unwrapped = UnwrapStructPtrHandles(pRenderPassBegin, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginRenderPass2KHR(commandBuffer, pRenderPassBegin_unwrapped, pSubpassBeginInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginRenderPass2KHR>::Dispatch(manager, commandBuffer, pRenderPassBegin, pSubpassBeginInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdNextSubpass2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkSubpassBeginInfo*                   pSubpassBeginInfo,
    const VkSubpassEndInfo*                     pSubpassEndInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdNextSubpass2KHR>::Dispatch(manager, commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdNextSubpass2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSubpassBeginInfo);
        EncodeStructPtr(encoder, pSubpassEndInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdNextSubpass2KHR(commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdNextSubpass2KHR>::Dispatch(manager, commandBuffer, pSubpassBeginInfo, pSubpassEndInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndRenderPass2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkSubpassEndInfo*                     pSubpassEndInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass2KHR>::Dispatch(manager, commandBuffer, pSubpassEndInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndRenderPass2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSubpassEndInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndRenderPass2KHR(commandBuffer, pSubpassEndInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndRenderPass2KHR>::Dispatch(manager, commandBuffer, pSubpassEndInfo);
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainStatusKHR(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSwapchainStatusKHR>::Dispatch(manager, device, swapchain);

    VkResult result = GetDeviceTable(device)->GetSwapchainStatusKHR(device, swapchain);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSwapchainStatusKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSwapchainStatusKHR>::Dispatch(manager, result, device, swapchain);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceExternalFencePropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceExternalFenceInfo*    pExternalFenceInfo,
    VkExternalFenceProperties*                  pExternalFenceProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFencePropertiesKHR>::Dispatch(manager, physicalDevice, pExternalFenceInfo, pExternalFenceProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalFencePropertiesKHR(physicalDevice, pExternalFenceInfo, pExternalFenceProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFencePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pExternalFenceInfo);
        EncodeStructPtr(encoder, pExternalFenceProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalFencePropertiesKHR>::Dispatch(manager, physicalDevice, pExternalFenceInfo, pExternalFenceProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL ImportFenceWin32HandleKHR(
    VkDevice                                    device,
    const VkImportFenceWin32HandleInfoKHR*      pImportFenceWin32HandleInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkImportFenceWin32HandleKHR>::Dispatch(manager, device, pImportFenceWin32HandleInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImportFenceWin32HandleInfoKHR* pImportFenceWin32HandleInfo_unwrapped = UnwrapStructPtrHandles(pImportFenceWin32HandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ImportFenceWin32HandleKHR(device, pImportFenceWin32HandleInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkImportFenceWin32HandleKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pImportFenceWin32HandleInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkImportFenceWin32HandleKHR>::Dispatch(manager, result, device, pImportFenceWin32HandleInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceWin32HandleKHR(
    VkDevice                                    device,
    const VkFenceGetWin32HandleInfoKHR*         pGetWin32HandleInfo,
    HANDLE*                                     pHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetFenceWin32HandleKHR>::Dispatch(manager, device, pGetWin32HandleInfo, pHandle);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkFenceGetWin32HandleInfoKHR* pGetWin32HandleInfo_unwrapped = UnwrapStructPtrHandles(pGetWin32HandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetFenceWin32HandleKHR(device, pGetWin32HandleInfo_unwrapped, pHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetFenceWin32HandleKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetWin32HandleInfo);
        encoder->EncodeVoidPtrPtr(pHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetFenceWin32HandleKHR>::Dispatch(manager, result, device, pGetWin32HandleInfo, pHandle);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ImportFenceFdKHR(
    VkDevice                                    device,
    const VkImportFenceFdInfoKHR*               pImportFenceFdInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkImportFenceFdKHR>::Dispatch(manager, device, pImportFenceFdInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImportFenceFdInfoKHR* pImportFenceFdInfo_unwrapped = UnwrapStructPtrHandles(pImportFenceFdInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ImportFenceFdKHR(device, pImportFenceFdInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkImportFenceFdKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pImportFenceFdInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkImportFenceFdKHR>::Dispatch(manager, result, device, pImportFenceFdInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetFenceFdKHR(
    VkDevice                                    device,
    const VkFenceGetFdInfoKHR*                  pGetFdInfo,
    int*                                        pFd)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetFenceFdKHR>::Dispatch(manager, device, pGetFdInfo, pFd);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkFenceGetFdInfoKHR* pGetFdInfo_unwrapped = UnwrapStructPtrHandles(pGetFdInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetFenceFdKHR(device, pGetFdInfo_unwrapped, pFd);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetFenceFdKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetFdInfo);
        encoder->EncodeInt32Ptr(pFd, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetFenceFdKHR>::Dispatch(manager, result, device, pGetFdInfo, pFd);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL EnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    uint32_t*                                   pCounterCount,
    VkPerformanceCounterKHR*                    pCounters,
    VkPerformanceCounterDescriptionKHR*         pCounterDescriptions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR>::Dispatch(manager, physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions);

    VkResult result = GetInstanceTable(physicalDevice)->EnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR(physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeUInt32Ptr(pCounterCount, omit_output_data);
        EncodeStructArray(encoder, pCounters, (pCounterCount != nullptr) ? (*pCounterCount) : 0, omit_output_data);
        EncodeStructArray(encoder, pCounterDescriptions, (pCounterCount != nullptr) ? (*pCounterCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkEnumeratePhysicalDeviceQueueFamilyPerformanceQueryCountersKHR>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, pCounterCount, pCounters, pCounterDescriptions);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkQueryPoolPerformanceCreateInfoKHR*  pPerformanceQueryCreateInfo,
    uint32_t*                                   pNumPasses)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR>::Dispatch(manager, physicalDevice, pPerformanceQueryCreateInfo, pNumPasses);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR(physicalDevice, pPerformanceQueryCreateInfo, pNumPasses);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pPerformanceQueryCreateInfo);
        encoder->EncodeUInt32Ptr(pNumPasses);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceQueueFamilyPerformanceQueryPassesKHR>::Dispatch(manager, physicalDevice, pPerformanceQueryCreateInfo, pNumPasses);
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireProfilingLockKHR(
    VkDevice                                    device,
    const VkAcquireProfilingLockInfoKHR*        pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireProfilingLockKHR>::Dispatch(manager, device, pInfo);

    VkResult result = GetDeviceTable(device)->AcquireProfilingLockKHR(device, pInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireProfilingLockKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireProfilingLockKHR>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL ReleaseProfilingLockKHR(
    VkDevice                                    device)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkReleaseProfilingLockKHR>::Dispatch(manager, device);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkReleaseProfilingLockKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->ReleaseProfilingLockKHR(device);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkReleaseProfilingLockKHR>::Dispatch(manager, device);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilities2KHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceSurfaceInfo2KHR*      pSurfaceInfo,
    VkSurfaceCapabilities2KHR*                  pSurfaceCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2KHR>::Dispatch(manager, physicalDevice, pSurfaceInfo, pSurfaceCapabilities);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo_unwrapped = UnwrapStructPtrHandles(pSurfaceInfo, handle_unwrap_memory);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceCapabilities2KHR(physicalDevice, pSurfaceInfo_unwrapped, pSurfaceCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pSurfaceInfo);
        EncodeStructPtr(encoder, pSurfaceCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2KHR>::Dispatch(manager, result, physicalDevice, pSurfaceInfo, pSurfaceCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceFormats2KHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceSurfaceInfo2KHR*      pSurfaceInfo,
    uint32_t*                                   pSurfaceFormatCount,
    VkSurfaceFormat2KHR*                        pSurfaceFormats)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormats2KHR>::Dispatch(manager, physicalDevice, pSurfaceInfo, pSurfaceFormatCount, pSurfaceFormats);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo_unwrapped = UnwrapStructPtrHandles(pSurfaceInfo, handle_unwrap_memory);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceFormats2KHR(physicalDevice, pSurfaceInfo_unwrapped, pSurfaceFormatCount, pSurfaceFormats);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormats2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pSurfaceInfo);
        encoder->EncodeUInt32Ptr(pSurfaceFormatCount, omit_output_data);
        EncodeStructArray(encoder, pSurfaceFormats, (pSurfaceFormatCount != nullptr) ? (*pSurfaceFormatCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceFormats2KHR>::Dispatch(manager, result, physicalDevice, pSurfaceInfo, pSurfaceFormatCount, pSurfaceFormats);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkDisplayProperties2KHR*                    pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayProperties2KHR>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceDisplayProperties2KHR(physicalDevice, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, NoParentWrapper, VkDisplayProperties2KHR>(physicalDevice, NoParentWrapper::kHandleValue, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, VkDisplayProperties2KHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayProperties2KHR* handle_struct)->DisplayKHRWrapper* { return GetWrapper<DisplayKHRWrapper>(handle_struct->displayProperties.display); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayProperties2KHR>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceDisplayPlaneProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkDisplayPlaneProperties2KHR*               pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlaneProperties2KHR>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceDisplayPlaneProperties2KHR(physicalDevice, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, NoParentWrapper, VkDisplayPlaneProperties2KHR>(physicalDevice, NoParentWrapper::kHandleValue, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlaneProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, VkDisplayPlaneProperties2KHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayPlaneProperties2KHR* handle_struct)->DisplayKHRWrapper* { return GetWrapper<DisplayKHRWrapper>(handle_struct->displayPlaneProperties.currentDisplay); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDisplayPlaneProperties2KHR>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayModeProperties2KHR(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayKHR                                display,
    uint32_t*                                   pPropertyCount,
    VkDisplayModeProperties2KHR*                pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDisplayModeProperties2KHR>::Dispatch(manager, physicalDevice, display, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetDisplayModeProperties2KHR(physicalDevice, display, pPropertyCount, pProperties);

    if (result >= 0)
    {
        CreateWrappedStructArrayHandles<PhysicalDeviceWrapper, DisplayKHRWrapper, VkDisplayModeProperties2KHR>(physicalDevice, display, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDisplayModeProperties2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndStructGroupCreateApiCallCapture<VkPhysicalDevice, DisplayModeKHRWrapper, VkDisplayModeProperties2KHR>(result, physicalDevice, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, pProperties, [](VkDisplayModeProperties2KHR* handle_struct)->DisplayModeKHRWrapper* { return GetWrapper<DisplayModeKHRWrapper>(handle_struct->displayModeProperties.displayMode); });
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDisplayModeProperties2KHR>::Dispatch(manager, result, physicalDevice, display, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDisplayPlaneCapabilities2KHR(
    VkPhysicalDevice                            physicalDevice,
    const VkDisplayPlaneInfo2KHR*               pDisplayPlaneInfo,
    VkDisplayPlaneCapabilities2KHR*             pCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilities2KHR>::Dispatch(manager, physicalDevice, pDisplayPlaneInfo, pCapabilities);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDisplayPlaneInfo2KHR* pDisplayPlaneInfo_unwrapped = UnwrapStructPtrHandles(pDisplayPlaneInfo, handle_unwrap_memory);

    VkResult result = GetInstanceTable(physicalDevice)->GetDisplayPlaneCapabilities2KHR(physicalDevice, pDisplayPlaneInfo_unwrapped, pCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilities2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pDisplayPlaneInfo);
        EncodeStructPtr(encoder, pCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDisplayPlaneCapabilities2KHR>::Dispatch(manager, result, physicalDevice, pDisplayPlaneInfo, pCapabilities);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetImageMemoryRequirements2KHR(
    VkDevice                                    device,
    const VkImageMemoryRequirementsInfo2*       pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetImageMemoryRequirements2KHR(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetBufferMemoryRequirements2KHR(
    VkDevice                                    device,
    const VkBufferMemoryRequirementsInfo2*      pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetBufferMemoryRequirements2KHR(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetImageSparseMemoryRequirements2KHR(
    VkDevice                                    device,
    const VkImageSparseMemoryRequirementsInfo2* pInfo,
    uint32_t*                                   pSparseMemoryRequirementCount,
    VkSparseImageMemoryRequirements2*           pSparseMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageSparseMemoryRequirementsInfo2* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetImageSparseMemoryRequirements2KHR(device, pInfo_unwrapped, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt32Ptr(pSparseMemoryRequirementCount);
        EncodeStructArray(encoder, pSparseMemoryRequirements, (pSparseMemoryRequirementCount != nullptr) ? (*pSparseMemoryRequirementCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSparseMemoryRequirements2KHR>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateSamplerYcbcrConversionKHR(
    VkDevice                                    device,
    const VkSamplerYcbcrConversionCreateInfo*   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSamplerYcbcrConversion*                   pYcbcrConversion)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversionKHR>::Dispatch(manager, device, pCreateInfo, pAllocator, pYcbcrConversion);

    VkResult result = GetDeviceTable(device)->CreateSamplerYcbcrConversionKHR(device, pCreateInfo, pAllocator, pYcbcrConversion);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, SamplerYcbcrConversionWrapper>(device, NoParentWrapper::kHandleValue, pYcbcrConversion, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversionKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SamplerYcbcrConversionWrapper>(pYcbcrConversion, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, SamplerYcbcrConversionWrapper, VkSamplerYcbcrConversionCreateInfo>(result, device, pYcbcrConversion, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateSamplerYcbcrConversionKHR>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pYcbcrConversion);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroySamplerYcbcrConversionKHR(
    VkDevice                                    device,
    VkSamplerYcbcrConversion                    ycbcrConversion,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversionKHR>::Dispatch(manager, device, ycbcrConversion, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversionKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SamplerYcbcrConversionWrapper>(ycbcrConversion);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<SamplerYcbcrConversionWrapper>(ycbcrConversion);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroySamplerYcbcrConversionKHR(device, ycbcrConversion, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroySamplerYcbcrConversionKHR>::Dispatch(manager, device, ycbcrConversion, pAllocator);

    DestroyWrappedHandle<SamplerYcbcrConversionWrapper>(ycbcrConversion);
}

VKAPI_ATTR VkResult VKAPI_CALL BindBufferMemory2KHR(
    VkDevice                                    device,
    uint32_t                                    bindInfoCount,
    const VkBindBufferMemoryInfo*               pBindInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindBufferMemory2KHR>::Dispatch(manager, device, bindInfoCount, pBindInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindBufferMemoryInfo* pBindInfos_unwrapped = UnwrapStructArrayHandles(pBindInfos, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindBufferMemory2KHR(device, bindInfoCount, pBindInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindBufferMemory2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfos, bindInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindBufferMemory2KHR>::Dispatch(manager, result, device, bindInfoCount, pBindInfos);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL BindImageMemory2KHR(
    VkDevice                                    device,
    uint32_t                                    bindInfoCount,
    const VkBindImageMemoryInfo*                pBindInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindImageMemory2KHR>::Dispatch(manager, device, bindInfoCount, pBindInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindImageMemoryInfo* pBindInfos_unwrapped = UnwrapStructArrayHandles(pBindInfos, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindImageMemory2KHR(device, bindInfoCount, pBindInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindImageMemory2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfos, bindInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindImageMemory2KHR>::Dispatch(manager, result, device, bindInfoCount, pBindInfos);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetLayoutSupportKHR(
    VkDevice                                    device,
    const VkDescriptorSetLayoutCreateInfo*      pCreateInfo,
    VkDescriptorSetLayoutSupport*               pSupport)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupportKHR>::Dispatch(manager, device, pCreateInfo, pSupport);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorSetLayoutCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDescriptorSetLayoutSupportKHR(device, pCreateInfo_unwrapped, pSupport);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupportKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pSupport);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutSupportKHR>::Dispatch(manager, device, pCreateInfo, pSupport);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCountKHR(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCountKHR>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndirectCountKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndirectCountKHRHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCountKHR>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCountKHR(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountKHR>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndexedIndirectCountKHRHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndexedIndirectCountKHR(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountKHR>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreCounterValueKHR(
    VkDevice                                    device,
    VkSemaphore                                 semaphore,
    uint64_t*                                   pValue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSemaphoreCounterValueKHR>::Dispatch(manager, device, semaphore, pValue);

    VkResult result = GetDeviceTable(device)->GetSemaphoreCounterValueKHR(device, semaphore, pValue);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSemaphoreCounterValueKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SemaphoreWrapper>(semaphore);
        encoder->EncodeUInt64Ptr(pValue, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSemaphoreCounterValueKHR>::Dispatch(manager, result, device, semaphore, pValue);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WaitSemaphoresKHR(
    VkDevice                                    device,
    const VkSemaphoreWaitInfo*                  pWaitInfo,
    uint64_t                                    timeout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkWaitSemaphoresKHR>::Dispatch(manager, device, pWaitInfo, timeout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreWaitInfo* pWaitInfo_unwrapped = UnwrapStructPtrHandles(pWaitInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->WaitSemaphoresKHR(device, pWaitInfo_unwrapped, timeout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkWaitSemaphoresKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pWaitInfo);
        encoder->EncodeUInt64Value(timeout);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkWaitSemaphoresKHR>::Dispatch(manager, result, device, pWaitInfo, timeout);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SignalSemaphoreKHR(
    VkDevice                                    device,
    const VkSemaphoreSignalInfo*                pSignalInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSignalSemaphoreKHR>::Dispatch(manager, device, pSignalInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreSignalInfo* pSignalInfo_unwrapped = UnwrapStructPtrHandles(pSignalInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->SignalSemaphoreKHR(device, pSignalInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSignalSemaphoreKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pSignalInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSignalSemaphoreKHR>::Dispatch(manager, result, device, pSignalInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceFragmentShadingRatesKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pFragmentShadingRateCount,
    VkPhysicalDeviceFragmentShadingRateKHR*     pFragmentShadingRates)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFragmentShadingRatesKHR>::Dispatch(manager, physicalDevice, pFragmentShadingRateCount, pFragmentShadingRates);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceFragmentShadingRatesKHR(physicalDevice, pFragmentShadingRateCount, pFragmentShadingRates);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceFragmentShadingRatesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pFragmentShadingRateCount, omit_output_data);
        EncodeStructArray(encoder, pFragmentShadingRates, (pFragmentShadingRateCount != nullptr) ? (*pFragmentShadingRateCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceFragmentShadingRatesKHR>::Dispatch(manager, result, physicalDevice, pFragmentShadingRateCount, pFragmentShadingRates);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetFragmentShadingRateKHR(
    VkCommandBuffer                             commandBuffer,
    const VkExtent2D*                           pFragmentSize,
    const VkFragmentShadingRateCombinerOpKHR    combinerOps[2])
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateKHR>::Dispatch(manager, commandBuffer, pFragmentSize, combinerOps);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pFragmentSize);
        encoder->EncodeEnumArray(combinerOps, 2);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetFragmentShadingRateKHR(commandBuffer, pFragmentSize, combinerOps);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateKHR>::Dispatch(manager, commandBuffer, pFragmentSize, combinerOps);
}

VKAPI_ATTR VkResult VKAPI_CALL WaitForPresentKHR(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    uint64_t                                    presentId,
    uint64_t                                    timeout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkWaitForPresentKHR>::Dispatch(manager, device, swapchain, presentId, timeout);

    VkResult result = GetDeviceTable(device)->WaitForPresentKHR(device, swapchain, presentId, timeout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkWaitForPresentKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeUInt64Value(presentId);
        encoder->EncodeUInt64Value(timeout);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkWaitForPresentKHR>::Dispatch(manager, result, device, swapchain, presentId, timeout);

    return result;
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddressKHR(
    VkDevice                                    device,
    const VkBufferDeviceAddressInfo*            pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddressKHR>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferDeviceAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkDeviceAddress result = GetDeviceTable(device)->GetBufferDeviceAddressKHR(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferDeviceAddressKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeVkDeviceAddressValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddressKHR>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetBufferOpaqueCaptureAddressKHR(
    VkDevice                                    device,
    const VkBufferDeviceAddressInfo*            pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddressKHR>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferDeviceAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    uint64_t result = GetDeviceTable(device)->GetBufferOpaqueCaptureAddressKHR(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddressKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt64Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferOpaqueCaptureAddressKHR>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR uint64_t VKAPI_CALL GetDeviceMemoryOpaqueCaptureAddressKHR(
    VkDevice                                    device,
    const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddressKHR>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceMemoryOpaqueCaptureAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    uint64_t result = GetDeviceTable(device)->GetDeviceMemoryOpaqueCaptureAddressKHR(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddressKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt64Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceMemoryOpaqueCaptureAddressKHR>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDeferredOperationKHR(
    VkDevice                                    device,
    const VkAllocationCallbacks*                pAllocator,
    VkDeferredOperationKHR*                     pDeferredOperation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDeferredOperationKHR>::Dispatch(manager, device, pAllocator, pDeferredOperation);

    VkResult result = GetDeviceTable(device)->CreateDeferredOperationKHR(device, pAllocator, pDeferredOperation);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, DeferredOperationKHRWrapper>(device, NoParentWrapper::kHandleValue, pDeferredOperation, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDeferredOperationKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DeferredOperationKHRWrapper>(pDeferredOperation, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, DeferredOperationKHRWrapper, void>(result, device, pDeferredOperation, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDeferredOperationKHR>::Dispatch(manager, result, device, pAllocator, pDeferredOperation);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDeferredOperationKHR(
    VkDevice                                    device,
    VkDeferredOperationKHR                      operation,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDeferredOperationKHR>::Dispatch(manager, device, operation, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDeferredOperationKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(operation);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DeferredOperationKHRWrapper>(operation);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyDeferredOperationKHR(device, operation, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDeferredOperationKHR>::Dispatch(manager, device, operation, pAllocator);

    DestroyWrappedHandle<DeferredOperationKHRWrapper>(operation);
}

VKAPI_ATTR uint32_t VKAPI_CALL GetDeferredOperationMaxConcurrencyKHR(
    VkDevice                                    device,
    VkDeferredOperationKHR                      operation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeferredOperationMaxConcurrencyKHR>::Dispatch(manager, device, operation);

    uint32_t result = GetDeviceTable(device)->GetDeferredOperationMaxConcurrencyKHR(device, operation);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeferredOperationMaxConcurrencyKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(operation);
        encoder->EncodeUInt32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeferredOperationMaxConcurrencyKHR>::Dispatch(manager, result, device, operation);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeferredOperationResultKHR(
    VkDevice                                    device,
    VkDeferredOperationKHR                      operation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeferredOperationResultKHR>::Dispatch(manager, device, operation);

    VkResult result = manager->OverrideGetDeferredOperationResultKHR(device, operation);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeferredOperationResultKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(operation);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeferredOperationResultKHR>::Dispatch(manager, result, device, operation);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DeferredOperationJoinKHR(
    VkDevice                                    device,
    VkDeferredOperationKHR                      operation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDeferredOperationJoinKHR>::Dispatch(manager, device, operation);

    VkResult result = manager->OverrideDeferredOperationJoinKHR(device, operation);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDeferredOperationJoinKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(operation);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDeferredOperationJoinKHR>::Dispatch(manager, result, device, operation);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutablePropertiesKHR(
    VkDevice                                    device,
    const VkPipelineInfoKHR*                    pPipelineInfo,
    uint32_t*                                   pExecutableCount,
    VkPipelineExecutablePropertiesKHR*          pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPipelineExecutablePropertiesKHR>::Dispatch(manager, device, pPipelineInfo, pExecutableCount, pProperties);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPipelineInfoKHR* pPipelineInfo_unwrapped = UnwrapStructPtrHandles(pPipelineInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetPipelineExecutablePropertiesKHR(device, pPipelineInfo_unwrapped, pExecutableCount, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPipelineExecutablePropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pPipelineInfo);
        encoder->EncodeUInt32Ptr(pExecutableCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pExecutableCount != nullptr) ? (*pExecutableCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPipelineExecutablePropertiesKHR>::Dispatch(manager, result, device, pPipelineInfo, pExecutableCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutableStatisticsKHR(
    VkDevice                                    device,
    const VkPipelineExecutableInfoKHR*          pExecutableInfo,
    uint32_t*                                   pStatisticCount,
    VkPipelineExecutableStatisticKHR*           pStatistics)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPipelineExecutableStatisticsKHR>::Dispatch(manager, device, pExecutableInfo, pStatisticCount, pStatistics);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPipelineExecutableInfoKHR* pExecutableInfo_unwrapped = UnwrapStructPtrHandles(pExecutableInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetPipelineExecutableStatisticsKHR(device, pExecutableInfo_unwrapped, pStatisticCount, pStatistics);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPipelineExecutableStatisticsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pExecutableInfo);
        encoder->EncodeUInt32Ptr(pStatisticCount, omit_output_data);
        EncodeStructArray(encoder, pStatistics, (pStatisticCount != nullptr) ? (*pStatisticCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPipelineExecutableStatisticsKHR>::Dispatch(manager, result, device, pExecutableInfo, pStatisticCount, pStatistics);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPipelineExecutableInternalRepresentationsKHR(
    VkDevice                                    device,
    const VkPipelineExecutableInfoKHR*          pExecutableInfo,
    uint32_t*                                   pInternalRepresentationCount,
    VkPipelineExecutableInternalRepresentationKHR* pInternalRepresentations)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPipelineExecutableInternalRepresentationsKHR>::Dispatch(manager, device, pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPipelineExecutableInfoKHR* pExecutableInfo_unwrapped = UnwrapStructPtrHandles(pExecutableInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetPipelineExecutableInternalRepresentationsKHR(device, pExecutableInfo_unwrapped, pInternalRepresentationCount, pInternalRepresentations);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPipelineExecutableInternalRepresentationsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pExecutableInfo);
        encoder->EncodeUInt32Ptr(pInternalRepresentationCount, omit_output_data);
        EncodeStructArray(encoder, pInternalRepresentations, (pInternalRepresentationCount != nullptr) ? (*pInternalRepresentationCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPipelineExecutableInternalRepresentationsKHR>::Dispatch(manager, result, device, pExecutableInfo, pInternalRepresentationCount, pInternalRepresentations);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL MapMemory2KHR(
    VkDevice                                    device,
    const VkMemoryMapInfoKHR*                   pMemoryMapInfo,
    void**                                      ppData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkMapMemory2KHR>::Dispatch(manager, device, pMemoryMapInfo, ppData);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryMapInfoKHR* pMemoryMapInfo_unwrapped = UnwrapStructPtrHandles(pMemoryMapInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->MapMemory2KHR(device, pMemoryMapInfo_unwrapped, ppData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkMapMemory2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pMemoryMapInfo);
        encoder->EncodeVoidPtrPtr(ppData, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkMapMemory2KHR>::Dispatch(manager, result, device, pMemoryMapInfo, ppData);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL UnmapMemory2KHR(
    VkDevice                                    device,
    const VkMemoryUnmapInfoKHR*                 pMemoryUnmapInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkUnmapMemory2KHR>::Dispatch(manager, device, pMemoryUnmapInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryUnmapInfoKHR* pMemoryUnmapInfo_unwrapped = UnwrapStructPtrHandles(pMemoryUnmapInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->UnmapMemory2KHR(device, pMemoryUnmapInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkUnmapMemory2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pMemoryUnmapInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkUnmapMemory2KHR>::Dispatch(manager, result, device, pMemoryUnmapInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceVideoEncodeQualityLevelInfoKHR* pQualityLevelInfo,
    VkVideoEncodeQualityLevelPropertiesKHR*     pQualityLevelProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR>::Dispatch(manager, physicalDevice, pQualityLevelInfo, pQualityLevelProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR(physicalDevice, pQualityLevelInfo, pQualityLevelProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pQualityLevelInfo);
        EncodeStructPtr(encoder, pQualityLevelProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceVideoEncodeQualityLevelPropertiesKHR>::Dispatch(manager, result, physicalDevice, pQualityLevelInfo, pQualityLevelProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetEncodedVideoSessionParametersKHR(
    VkDevice                                    device,
    const VkVideoEncodeSessionParametersGetInfoKHR* pVideoSessionParametersInfo,
    VkVideoEncodeSessionParametersFeedbackInfoKHR* pFeedbackInfo,
    size_t*                                     pDataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetEncodedVideoSessionParametersKHR>::Dispatch(manager, device, pVideoSessionParametersInfo, pFeedbackInfo, pDataSize, pData);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkVideoEncodeSessionParametersGetInfoKHR* pVideoSessionParametersInfo_unwrapped = UnwrapStructPtrHandles(pVideoSessionParametersInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetEncodedVideoSessionParametersKHR(device, pVideoSessionParametersInfo_unwrapped, pFeedbackInfo, pDataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetEncodedVideoSessionParametersKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pVideoSessionParametersInfo);
        EncodeStructPtr(encoder, pFeedbackInfo, omit_output_data);
        encoder->EncodeSizeTPtr(pDataSize, omit_output_data);
        encoder->EncodeVoidArray(pData, (pDataSize != nullptr) ? (*pDataSize) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetEncodedVideoSessionParametersKHR>::Dispatch(manager, result, device, pVideoSessionParametersInfo, pFeedbackInfo, pDataSize, pData);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdEncodeVideoKHR(
    VkCommandBuffer                             commandBuffer,
    const VkVideoEncodeInfoKHR*                 pEncodeInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEncodeVideoKHR>::Dispatch(manager, commandBuffer, pEncodeInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEncodeVideoKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pEncodeInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdEncodeVideoKHRHandles, pEncodeInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkVideoEncodeInfoKHR* pEncodeInfo_unwrapped = UnwrapStructPtrHandles(pEncodeInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdEncodeVideoKHR(commandBuffer, pEncodeInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEncodeVideoKHR>::Dispatch(manager, commandBuffer, pEncodeInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdSetEvent2KHR(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    const VkDependencyInfo*                     pDependencyInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetEvent2KHR>::Dispatch(manager, commandBuffer, event, pDependencyInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetEvent2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        EncodeStructPtr(encoder, pDependencyInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdSetEvent2KHRHandles, event, pDependencyInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfo_unwrapped = UnwrapStructPtrHandles(pDependencyInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdSetEvent2KHR(commandBuffer, event, pDependencyInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetEvent2KHR>::Dispatch(manager, commandBuffer, event, pDependencyInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdResetEvent2KHR(
    VkCommandBuffer                             commandBuffer,
    VkEvent                                     event,
    VkPipelineStageFlags2                       stageMask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResetEvent2KHR>::Dispatch(manager, commandBuffer, event, stageMask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResetEvent2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<EventWrapper>(event);
        encoder->EncodeFlags64Value(stageMask);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResetEvent2KHRHandles, event);
    }

    GetDeviceTable(commandBuffer)->CmdResetEvent2KHR(commandBuffer, event, stageMask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResetEvent2KHR>::Dispatch(manager, commandBuffer, event, stageMask);
}

VKAPI_ATTR void VKAPI_CALL CmdWaitEvents2KHR(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    eventCount,
    const VkEvent*                              pEvents,
    const VkDependencyInfo*                     pDependencyInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWaitEvents2KHR>::Dispatch(manager, commandBuffer, eventCount, pEvents, pDependencyInfos);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWaitEvents2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(eventCount);
        encoder->EncodeHandleArray<EventWrapper>(pEvents, eventCount);
        EncodeStructArray(encoder, pDependencyInfos, eventCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWaitEvents2KHRHandles, eventCount, pEvents, pDependencyInfos);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfos_unwrapped = UnwrapStructArrayHandles(pDependencyInfos, eventCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdWaitEvents2KHR(commandBuffer, eventCount, pEvents, pDependencyInfos_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWaitEvents2KHR>::Dispatch(manager, commandBuffer, eventCount, pEvents, pDependencyInfos);
}

VKAPI_ATTR void VKAPI_CALL CmdPipelineBarrier2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkDependencyInfo*                     pDependencyInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier2KHR>::Dispatch(manager, commandBuffer, pDependencyInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPipelineBarrier2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pDependencyInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPipelineBarrier2KHRHandles, pDependencyInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDependencyInfo* pDependencyInfo_unwrapped = UnwrapStructPtrHandles(pDependencyInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPipelineBarrier2KHR(commandBuffer, pDependencyInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPipelineBarrier2KHR>::Dispatch(manager, commandBuffer, pDependencyInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdWriteTimestamp2KHR(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlags2                       stage,
    VkQueryPool                                 queryPool,
    uint32_t                                    query)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp2KHR>::Dispatch(manager, commandBuffer, stage, queryPool, query);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteTimestamp2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlags64Value(stage);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteTimestamp2KHRHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdWriteTimestamp2KHR(commandBuffer, stage, queryPool, query);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteTimestamp2KHR>::Dispatch(manager, commandBuffer, stage, queryPool, query);
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSubmit2KHR(
    VkQueue                                     queue,
    uint32_t                                    submitCount,
    const VkSubmitInfo2*                        pSubmits,
    VkFence                                     fence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueSubmit2KHR>::Dispatch(manager, queue, submitCount, pSubmits, fence);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSubmitInfo2* pSubmits_unwrapped = UnwrapStructArrayHandles(pSubmits, submitCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(queue)->QueueSubmit2KHR(queue, submitCount, pSubmits_unwrapped, fence);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueSubmit2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Value(submitCount);
        EncodeStructArray(encoder, pSubmits, submitCount);
        encoder->EncodeHandleValue<FenceWrapper>(fence);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueSubmit2KHR>::Dispatch(manager, result, queue, submitCount, pSubmits, fence);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteBufferMarker2AMD(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlags2                       stage,
    VkBuffer                                    dstBuffer,
    VkDeviceSize                                dstOffset,
    uint32_t                                    marker)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteBufferMarker2AMD>::Dispatch(manager, commandBuffer, stage, dstBuffer, dstOffset, marker);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteBufferMarker2AMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlags64Value(stage);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeVkDeviceSizeValue(dstOffset);
        encoder->EncodeUInt32Value(marker);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteBufferMarker2AMDHandles, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdWriteBufferMarker2AMD(commandBuffer, stage, dstBuffer, dstOffset, marker);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteBufferMarker2AMD>::Dispatch(manager, commandBuffer, stage, dstBuffer, dstOffset, marker);
}

VKAPI_ATTR void VKAPI_CALL GetQueueCheckpointData2NV(
    VkQueue                                     queue,
    uint32_t*                                   pCheckpointDataCount,
    VkCheckpointData2NV*                        pCheckpointData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetQueueCheckpointData2NV>::Dispatch(manager, queue, pCheckpointDataCount, pCheckpointData);

    GetDeviceTable(queue)->GetQueueCheckpointData2NV(queue, pCheckpointDataCount, pCheckpointData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetQueueCheckpointData2NV);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Ptr(pCheckpointDataCount);
        EncodeStructArray(encoder, pCheckpointData, (pCheckpointDataCount != nullptr) ? (*pCheckpointDataCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetQueueCheckpointData2NV>::Dispatch(manager, queue, pCheckpointDataCount, pCheckpointData);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBuffer2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkCopyBufferInfo2*                    pCopyBufferInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer2KHR>::Dispatch(manager, commandBuffer, pCopyBufferInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBuffer2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyBufferInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBuffer2KHRHandles, pCopyBufferInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyBufferInfo2* pCopyBufferInfo_unwrapped = UnwrapStructPtrHandles(pCopyBufferInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyBuffer2KHR(commandBuffer, pCopyBufferInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBuffer2KHR>::Dispatch(manager, commandBuffer, pCopyBufferInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImage2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkCopyImageInfo2*                     pCopyImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImage2KHR>::Dispatch(manager, commandBuffer, pCopyImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImage2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImage2KHRHandles, pCopyImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageInfo2* pCopyImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyImage2KHR(commandBuffer, pCopyImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImage2KHR>::Dispatch(manager, commandBuffer, pCopyImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyBufferToImage2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkCopyBufferToImageInfo2*             pCopyBufferToImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2KHR>::Dispatch(manager, commandBuffer, pCopyBufferToImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyBufferToImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyBufferToImage2KHRHandles, pCopyBufferToImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyBufferToImageInfo2* pCopyBufferToImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyBufferToImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyBufferToImage2KHR(commandBuffer, pCopyBufferToImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyBufferToImage2KHR>::Dispatch(manager, commandBuffer, pCopyBufferToImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyImageToBuffer2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkCopyImageToBufferInfo2*             pCopyImageToBufferInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2KHR>::Dispatch(manager, commandBuffer, pCopyImageToBufferInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pCopyImageToBufferInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyImageToBuffer2KHRHandles, pCopyImageToBufferInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageToBufferInfo2* pCopyImageToBufferInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageToBufferInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyImageToBuffer2KHR(commandBuffer, pCopyImageToBufferInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyImageToBuffer2KHR>::Dispatch(manager, commandBuffer, pCopyImageToBufferInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBlitImage2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkBlitImageInfo2*                     pBlitImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBlitImage2KHR>::Dispatch(manager, commandBuffer, pBlitImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBlitImage2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBlitImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBlitImage2KHRHandles, pBlitImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBlitImageInfo2* pBlitImageInfo_unwrapped = UnwrapStructPtrHandles(pBlitImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBlitImage2KHR(commandBuffer, pBlitImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBlitImage2KHR>::Dispatch(manager, commandBuffer, pBlitImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdResolveImage2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkResolveImageInfo2*                  pResolveImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdResolveImage2KHR>::Dispatch(manager, commandBuffer, pResolveImageInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdResolveImage2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pResolveImageInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdResolveImage2KHRHandles, pResolveImageInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkResolveImageInfo2* pResolveImageInfo_unwrapped = UnwrapStructPtrHandles(pResolveImageInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdResolveImage2KHR(commandBuffer, pResolveImageInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdResolveImage2KHR>::Dispatch(manager, commandBuffer, pResolveImageInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdTraceRaysIndirect2KHR(
    VkCommandBuffer                             commandBuffer,
    VkDeviceAddress                             indirectDeviceAddress)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdTraceRaysIndirect2KHR>::Dispatch(manager, commandBuffer, indirectDeviceAddress);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdTraceRaysIndirect2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkDeviceAddressValue(indirectDeviceAddress);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdTraceRaysIndirect2KHR(commandBuffer, indirectDeviceAddress);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdTraceRaysIndirect2KHR>::Dispatch(manager, commandBuffer, indirectDeviceAddress);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceBufferMemoryRequirementsKHR(
    VkDevice                                    device,
    const VkDeviceBufferMemoryRequirements*     pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    GetDeviceTable(device)->GetDeviceBufferMemoryRequirementsKHR(device, pInfo, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirementsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceBufferMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageMemoryRequirementsKHR(
    VkDevice                                    device,
    const VkDeviceImageMemoryRequirements*      pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceImageMemoryRequirements* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDeviceImageMemoryRequirementsKHR(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirementsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceImageMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSparseMemoryRequirementsKHR(
    VkDevice                                    device,
    const VkDeviceImageMemoryRequirements*      pInfo,
    uint32_t*                                   pSparseMemoryRequirementCount,
    VkSparseImageMemoryRequirements2*           pSparseMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceImageMemoryRequirements* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDeviceImageSparseMemoryRequirementsKHR(device, pInfo_unwrapped, pSparseMemoryRequirementCount, pSparseMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirementsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt32Ptr(pSparseMemoryRequirementCount);
        EncodeStructArray(encoder, pSparseMemoryRequirements, (pSparseMemoryRequirementCount != nullptr) ? (*pSparseMemoryRequirementCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceImageSparseMemoryRequirementsKHR>::Dispatch(manager, device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL CmdBindIndexBuffer2KHR(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkDeviceSize                                size,
    VkIndexType                                 indexType)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindIndexBuffer2KHR>::Dispatch(manager, commandBuffer, buffer, offset, size, indexType);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindIndexBuffer2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeVkDeviceSizeValue(size);
        encoder->EncodeEnumValue(indexType);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindIndexBuffer2KHRHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdBindIndexBuffer2KHR(commandBuffer, buffer, offset, size, indexType);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindIndexBuffer2KHR>::Dispatch(manager, commandBuffer, buffer, offset, size, indexType);
}

VKAPI_ATTR void VKAPI_CALL GetRenderingAreaGranularityKHR(
    VkDevice                                    device,
    const VkRenderingAreaInfoKHR*               pRenderingAreaInfo,
    VkExtent2D*                                 pGranularity)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRenderingAreaGranularityKHR>::Dispatch(manager, device, pRenderingAreaInfo, pGranularity);

    GetDeviceTable(device)->GetRenderingAreaGranularityKHR(device, pRenderingAreaInfo, pGranularity);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetRenderingAreaGranularityKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pRenderingAreaInfo);
        EncodeStructPtr(encoder, pGranularity);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRenderingAreaGranularityKHR>::Dispatch(manager, device, pRenderingAreaInfo, pGranularity);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceImageSubresourceLayoutKHR(
    VkDevice                                    device,
    const VkDeviceImageSubresourceInfoKHR*      pInfo,
    VkSubresourceLayout2KHR*                    pLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceImageSubresourceLayoutKHR>::Dispatch(manager, device, pInfo, pLayout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDeviceImageSubresourceInfoKHR* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetDeviceImageSubresourceLayoutKHR(device, pInfo_unwrapped, pLayout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceImageSubresourceLayoutKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pLayout);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceImageSubresourceLayoutKHR>::Dispatch(manager, device, pInfo, pLayout);
}

VKAPI_ATTR void VKAPI_CALL GetImageSubresourceLayout2KHR(
    VkDevice                                    device,
    VkImage                                     image,
    const VkImageSubresource2KHR*               pSubresource,
    VkSubresourceLayout2KHR*                    pLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2KHR>::Dispatch(manager, device, image, pSubresource, pLayout);

    GetDeviceTable(device)->GetImageSubresourceLayout2KHR(device, image, pSubresource, pLayout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pSubresource);
        EncodeStructPtr(encoder, pLayout);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2KHR>::Dispatch(manager, device, image, pSubresource, pLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCooperativeMatrixPropertiesKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkCooperativeMatrixPropertiesKHR*           pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceCooperativeMatrixPropertiesKHR(physicalDevice, pPropertyCount, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesKHR>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCalibrateableTimeDomainsKHR(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pTimeDomainCount,
    VkTimeDomainKHR*                            pTimeDomains)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsKHR>::Dispatch(manager, physicalDevice, pTimeDomainCount, pTimeDomains);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceCalibrateableTimeDomainsKHR(physicalDevice, pTimeDomainCount, pTimeDomains);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pTimeDomainCount, omit_output_data);
        encoder->EncodeEnumArray(pTimeDomains, (pTimeDomainCount != nullptr) ? (*pTimeDomainCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsKHR>::Dispatch(manager, result, physicalDevice, pTimeDomainCount, pTimeDomains);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetCalibratedTimestampsKHR(
    VkDevice                                    device,
    uint32_t                                    timestampCount,
    const VkCalibratedTimestampInfoKHR*         pTimestampInfos,
    uint64_t*                                   pTimestamps,
    uint64_t*                                   pMaxDeviation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetCalibratedTimestampsKHR>::Dispatch(manager, device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);

    VkResult result = GetDeviceTable(device)->GetCalibratedTimestampsKHR(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetCalibratedTimestampsKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(timestampCount);
        EncodeStructArray(encoder, pTimestampInfos, timestampCount);
        encoder->EncodeUInt64Array(pTimestamps, timestampCount, omit_output_data);
        encoder->EncodeUInt64Ptr(pMaxDeviation, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetCalibratedTimestampsKHR>::Dispatch(manager, result, device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorSets2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkBindDescriptorSetsInfoKHR*          pBindDescriptorSetsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorSets2KHR>::Dispatch(manager, commandBuffer, pBindDescriptorSetsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindDescriptorSets2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBindDescriptorSetsInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindDescriptorSets2KHRHandles, pBindDescriptorSetsInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindDescriptorSetsInfoKHR* pBindDescriptorSetsInfo_unwrapped = UnwrapStructPtrHandles(pBindDescriptorSetsInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBindDescriptorSets2KHR(commandBuffer, pBindDescriptorSetsInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorSets2KHR>::Dispatch(manager, commandBuffer, pBindDescriptorSetsInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdPushConstants2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkPushConstantsInfoKHR*               pPushConstantsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPushConstants2KHR>::Dispatch(manager, commandBuffer, pPushConstantsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPushConstants2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pPushConstantsInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPushConstants2KHRHandles, pPushConstantsInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPushConstantsInfoKHR* pPushConstantsInfo_unwrapped = UnwrapStructPtrHandles(pPushConstantsInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPushConstants2KHR(commandBuffer, pPushConstantsInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPushConstants2KHR>::Dispatch(manager, commandBuffer, pPushConstantsInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSet2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkPushDescriptorSetInfoKHR*           pPushDescriptorSetInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSet2KHR>::Dispatch(manager, commandBuffer, pPushDescriptorSetInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPushDescriptorSet2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pPushDescriptorSetInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPushDescriptorSet2KHRHandles, pPushDescriptorSetInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPushDescriptorSetInfoKHR* pPushDescriptorSetInfo_unwrapped = UnwrapStructPtrHandles(pPushDescriptorSetInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPushDescriptorSet2KHR(commandBuffer, pPushDescriptorSetInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSet2KHR>::Dispatch(manager, commandBuffer, pPushDescriptorSetInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdPushDescriptorSetWithTemplate2KHR(
    VkCommandBuffer                             commandBuffer,
    const VkPushDescriptorSetWithTemplateInfoKHR* pPushDescriptorSetWithTemplateInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSetWithTemplate2KHR>::Dispatch(manager, commandBuffer, pPushDescriptorSetWithTemplateInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPushDescriptorSetWithTemplate2KHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pPushDescriptorSetWithTemplateInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPushDescriptorSetWithTemplate2KHRHandles, pPushDescriptorSetWithTemplateInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPushDescriptorSetWithTemplateInfoKHR* pPushDescriptorSetWithTemplateInfo_unwrapped = UnwrapStructPtrHandles(pPushDescriptorSetWithTemplateInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPushDescriptorSetWithTemplate2KHR(commandBuffer, pPushDescriptorSetWithTemplateInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPushDescriptorSetWithTemplate2KHR>::Dispatch(manager, commandBuffer, pPushDescriptorSetWithTemplateInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDescriptorBufferOffsets2EXT(
    VkCommandBuffer                             commandBuffer,
    const VkSetDescriptorBufferOffsetsInfoEXT*  pSetDescriptorBufferOffsetsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDescriptorBufferOffsets2EXT>::Dispatch(manager, commandBuffer, pSetDescriptorBufferOffsetsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDescriptorBufferOffsets2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSetDescriptorBufferOffsetsInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdSetDescriptorBufferOffsets2EXTHandles, pSetDescriptorBufferOffsetsInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSetDescriptorBufferOffsetsInfoEXT* pSetDescriptorBufferOffsetsInfo_unwrapped = UnwrapStructPtrHandles(pSetDescriptorBufferOffsetsInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdSetDescriptorBufferOffsets2EXT(commandBuffer, pSetDescriptorBufferOffsetsInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDescriptorBufferOffsets2EXT>::Dispatch(manager, commandBuffer, pSetDescriptorBufferOffsetsInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBindDescriptorBufferEmbeddedSamplers2EXT(
    VkCommandBuffer                             commandBuffer,
    const VkBindDescriptorBufferEmbeddedSamplersInfoEXT* pBindDescriptorBufferEmbeddedSamplersInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorBufferEmbeddedSamplers2EXT>::Dispatch(manager, commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindDescriptorBufferEmbeddedSamplers2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pBindDescriptorBufferEmbeddedSamplersInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindDescriptorBufferEmbeddedSamplers2EXTHandles, pBindDescriptorBufferEmbeddedSamplersInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindDescriptorBufferEmbeddedSamplersInfoEXT* pBindDescriptorBufferEmbeddedSamplersInfo_unwrapped = UnwrapStructPtrHandles(pBindDescriptorBufferEmbeddedSamplersInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBindDescriptorBufferEmbeddedSamplers2EXT(commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindDescriptorBufferEmbeddedSamplers2EXT>::Dispatch(manager, commandBuffer, pBindDescriptorBufferEmbeddedSamplersInfo);
}

VKAPI_ATTR void VKAPI_CALL FrameBoundaryANDROID(
    VkDevice                                    device,
    VkSemaphore                                 semaphore,
    VkImage                                     image)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkFrameBoundaryANDROID>::Dispatch(manager, device, semaphore, image);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkFrameBoundaryANDROID);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SemaphoreWrapper>(semaphore);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->FrameBoundaryANDROID(device, semaphore, image);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkFrameBoundaryANDROID>::Dispatch(manager, device, semaphore, image);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDebugReportCallbackEXT(
    VkInstance                                  instance,
    const VkDebugReportCallbackCreateInfoEXT*   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDebugReportCallbackEXT*                   pCallback)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDebugReportCallbackEXT>::Dispatch(manager, instance, pCreateInfo, pAllocator, pCallback);

    VkResult result = GetInstanceTable(instance)->CreateDebugReportCallbackEXT(instance, pCreateInfo, pAllocator, pCallback);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, DebugReportCallbackEXTWrapper>(instance, NoParentWrapper::kHandleValue, pCallback, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDebugReportCallbackEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DebugReportCallbackEXTWrapper>(pCallback, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, DebugReportCallbackEXTWrapper, VkDebugReportCallbackCreateInfoEXT>(result, instance, pCallback, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDebugReportCallbackEXT>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pCallback);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDebugReportCallbackEXT(
    VkInstance                                  instance,
    VkDebugReportCallbackEXT                    callback,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDebugReportCallbackEXT>::Dispatch(manager, instance, callback, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDebugReportCallbackEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeHandleValue<DebugReportCallbackEXTWrapper>(callback);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DebugReportCallbackEXTWrapper>(callback);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetInstanceTable(instance)->DestroyDebugReportCallbackEXT(instance, callback, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDebugReportCallbackEXT>::Dispatch(manager, instance, callback, pAllocator);

    DestroyWrappedHandle<DebugReportCallbackEXTWrapper>(callback);
}

VKAPI_ATTR void VKAPI_CALL DebugReportMessageEXT(
    VkInstance                                  instance,
    VkDebugReportFlagsEXT                       flags,
    VkDebugReportObjectTypeEXT                  objectType,
    uint64_t                                    object,
    size_t                                      location,
    int32_t                                     messageCode,
    const char*                                 pLayerPrefix,
    const char*                                 pMessage)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDebugReportMessageEXT>::Dispatch(manager, instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDebugReportMessageEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeEnumValue(objectType);
        encoder->EncodeUInt64Value(GetWrappedId(object, objectType));
        encoder->EncodeSizeTValue(location);
        encoder->EncodeInt32Value(messageCode);
        encoder->EncodeString(pLayerPrefix);
        encoder->EncodeString(pMessage);
        manager->EndApiCallCapture();
    }

    GetInstanceTable(instance)->DebugReportMessageEXT(instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDebugReportMessageEXT>::Dispatch(manager, instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage);
}

VKAPI_ATTR VkResult VKAPI_CALL DebugMarkerSetObjectTagEXT(
    VkDevice                                    device,
    const VkDebugMarkerObjectTagInfoEXT*        pTagInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDebugMarkerSetObjectTagEXT>::Dispatch(manager, device, pTagInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDebugMarkerObjectTagInfoEXT* pTagInfo_unwrapped = UnwrapStructPtrHandles(pTagInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->DebugMarkerSetObjectTagEXT(device, pTagInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDebugMarkerSetObjectTagEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pTagInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDebugMarkerSetObjectTagEXT>::Dispatch(manager, result, device, pTagInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DebugMarkerSetObjectNameEXT(
    VkDevice                                    device,
    const VkDebugMarkerObjectNameInfoEXT*       pNameInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDebugMarkerSetObjectNameEXT>::Dispatch(manager, device, pNameInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDebugMarkerObjectNameInfoEXT* pNameInfo_unwrapped = UnwrapStructPtrHandles(pNameInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->DebugMarkerSetObjectNameEXT(device, pNameInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDebugMarkerSetObjectNameEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pNameInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDebugMarkerSetObjectNameEXT>::Dispatch(manager, result, device, pNameInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerBeginEXT(
    VkCommandBuffer                             commandBuffer,
    const VkDebugMarkerMarkerInfoEXT*           pMarkerInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerBeginEXT>::Dispatch(manager, commandBuffer, pMarkerInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDebugMarkerBeginEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pMarkerInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDebugMarkerBeginEXT(commandBuffer, pMarkerInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerBeginEXT>::Dispatch(manager, commandBuffer, pMarkerInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerEndEXT(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerEndEXT>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDebugMarkerEndEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDebugMarkerEndEXT(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerEndEXT>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL CmdDebugMarkerInsertEXT(
    VkCommandBuffer                             commandBuffer,
    const VkDebugMarkerMarkerInfoEXT*           pMarkerInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerInsertEXT>::Dispatch(manager, commandBuffer, pMarkerInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDebugMarkerInsertEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pMarkerInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDebugMarkerInsertEXT(commandBuffer, pMarkerInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDebugMarkerInsertEXT>::Dispatch(manager, commandBuffer, pMarkerInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBindTransformFeedbackBuffersEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstBinding,
    uint32_t                                    bindingCount,
    const VkBuffer*                             pBuffers,
    const VkDeviceSize*                         pOffsets,
    const VkDeviceSize*                         pSizes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindTransformFeedbackBuffersEXT>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindTransformFeedbackBuffersEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstBinding);
        encoder->EncodeUInt32Value(bindingCount);
        encoder->EncodeHandleArray<BufferWrapper>(pBuffers, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pOffsets, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pSizes, bindingCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindTransformFeedbackBuffersEXTHandles, bindingCount, pBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdBindTransformFeedbackBuffersEXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindTransformFeedbackBuffersEXT>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginTransformFeedbackEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstCounterBuffer,
    uint32_t                                    counterBufferCount,
    const VkBuffer*                             pCounterBuffers,
    const VkDeviceSize*                         pCounterBufferOffsets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginTransformFeedbackEXT>::Dispatch(manager, commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginTransformFeedbackEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstCounterBuffer);
        encoder->EncodeUInt32Value(counterBufferCount);
        encoder->EncodeHandleArray<BufferWrapper>(pCounterBuffers, counterBufferCount);
        encoder->EncodeVkDeviceSizeArray(pCounterBufferOffsets, counterBufferCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginTransformFeedbackEXTHandles, counterBufferCount, pCounterBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdBeginTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginTransformFeedbackEXT>::Dispatch(manager, commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);
}

VKAPI_ATTR void VKAPI_CALL CmdEndTransformFeedbackEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstCounterBuffer,
    uint32_t                                    counterBufferCount,
    const VkBuffer*                             pCounterBuffers,
    const VkDeviceSize*                         pCounterBufferOffsets)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndTransformFeedbackEXT>::Dispatch(manager, commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndTransformFeedbackEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstCounterBuffer);
        encoder->EncodeUInt32Value(counterBufferCount);
        encoder->EncodeHandleArray<BufferWrapper>(pCounterBuffers, counterBufferCount);
        encoder->EncodeVkDeviceSizeArray(pCounterBufferOffsets, counterBufferCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdEndTransformFeedbackEXTHandles, counterBufferCount, pCounterBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdEndTransformFeedbackEXT(commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndTransformFeedbackEXT>::Dispatch(manager, commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginQueryIndexedEXT(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    query,
    VkQueryControlFlags                         flags,
    uint32_t                                    index)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginQueryIndexedEXT>::Dispatch(manager, commandBuffer, queryPool, query, flags, index);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginQueryIndexedEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeUInt32Value(index);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginQueryIndexedEXTHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdBeginQueryIndexedEXT(commandBuffer, queryPool, query, flags, index);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginQueryIndexedEXT>::Dispatch(manager, commandBuffer, queryPool, query, flags, index);
}

VKAPI_ATTR void VKAPI_CALL CmdEndQueryIndexedEXT(
    VkCommandBuffer                             commandBuffer,
    VkQueryPool                                 queryPool,
    uint32_t                                    query,
    uint32_t                                    index)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndQueryIndexedEXT>::Dispatch(manager, commandBuffer, queryPool, query, index);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndQueryIndexedEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(query);
        encoder->EncodeUInt32Value(index);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdEndQueryIndexedEXTHandles, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdEndQueryIndexedEXT(commandBuffer, queryPool, query, index);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndQueryIndexedEXT>::Dispatch(manager, commandBuffer, queryPool, query, index);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectByteCountEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    instanceCount,
    uint32_t                                    firstInstance,
    VkBuffer                                    counterBuffer,
    VkDeviceSize                                counterBufferOffset,
    uint32_t                                    counterOffset,
    uint32_t                                    vertexStride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectByteCountEXT>::Dispatch(manager, commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndirectByteCountEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(instanceCount);
        encoder->EncodeUInt32Value(firstInstance);
        encoder->EncodeHandleValue<BufferWrapper>(counterBuffer);
        encoder->EncodeVkDeviceSizeValue(counterBufferOffset);
        encoder->EncodeUInt32Value(counterOffset);
        encoder->EncodeUInt32Value(vertexStride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndirectByteCountEXTHandles, counterBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndirectByteCountEXT(commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectByteCountEXT>::Dispatch(manager, commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride);
}

VKAPI_ATTR uint32_t VKAPI_CALL GetImageViewHandleNVX(
    VkDevice                                    device,
    const VkImageViewHandleInfoNVX*             pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageViewHandleNVX>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImageViewHandleInfoNVX* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    uint32_t result = GetDeviceTable(device)->GetImageViewHandleNVX(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageViewHandleNVX);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeUInt32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageViewHandleNVX>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetImageViewAddressNVX(
    VkDevice                                    device,
    VkImageView                                 imageView,
    VkImageViewAddressPropertiesNVX*            pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageViewAddressNVX>::Dispatch(manager, device, imageView, pProperties);

    VkResult result = GetDeviceTable(device)->GetImageViewAddressNVX(device, imageView, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageViewAddressNVX);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageViewWrapper>(imageView);
        EncodeStructPtr(encoder, pProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageViewAddressNVX>::Dispatch(manager, result, device, imageView, pProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndirectCountAMD(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCountAMD>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndirectCountAMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndirectCountAMDHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndirectCountAMD>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawIndexedIndirectCountAMD(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountAMD>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountAMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawIndexedIndirectCountAMDHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawIndexedIndirectCountAMD(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawIndexedIndirectCountAMD>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR VkResult VKAPI_CALL GetShaderInfoAMD(
    VkDevice                                    device,
    VkPipeline                                  pipeline,
    VkShaderStageFlagBits                       shaderStage,
    VkShaderInfoTypeAMD                         infoType,
    size_t*                                     pInfoSize,
    void*                                       pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetShaderInfoAMD>::Dispatch(manager, device, pipeline, shaderStage, infoType, pInfoSize, pInfo);

    VkResult result = GetDeviceTable(device)->GetShaderInfoAMD(device, pipeline, shaderStage, infoType, pInfoSize, pInfo);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetShaderInfoAMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        encoder->EncodeEnumValue(shaderStage);
        encoder->EncodeEnumValue(infoType);
        encoder->EncodeSizeTPtr(pInfoSize, omit_output_data);
        encoder->EncodeVoidArray(pInfo, (pInfoSize != nullptr) ? (*pInfoSize) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetShaderInfoAMD>::Dispatch(manager, result, device, pipeline, shaderStage, infoType, pInfoSize, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateStreamDescriptorSurfaceGGP(
    VkInstance                                  instance,
    const VkStreamDescriptorSurfaceCreateInfoGGP* pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateStreamDescriptorSurfaceGGP>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateStreamDescriptorSurfaceGGP(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateStreamDescriptorSurfaceGGP);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkStreamDescriptorSurfaceCreateInfoGGP>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateStreamDescriptorSurfaceGGP>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceExternalImageFormatPropertiesNV(
    VkPhysicalDevice                            physicalDevice,
    VkFormat                                    format,
    VkImageType                                 type,
    VkImageTiling                               tiling,
    VkImageUsageFlags                           usage,
    VkImageCreateFlags                          flags,
    VkExternalMemoryHandleTypeFlagsNV           externalHandleType,
    VkExternalImageFormatPropertiesNV*          pExternalImageFormatProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalImageFormatPropertiesNV>::Dispatch(manager, physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceExternalImageFormatPropertiesNV(physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalImageFormatPropertiesNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(format);
        encoder->EncodeEnumValue(type);
        encoder->EncodeEnumValue(tiling);
        encoder->EncodeFlagsValue(usage);
        encoder->EncodeFlagsValue(flags);
        encoder->EncodeFlagsValue(externalHandleType);
        EncodeStructPtr(encoder, pExternalImageFormatProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceExternalImageFormatPropertiesNV>::Dispatch(manager, result, physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryWin32HandleNV(
    VkDevice                                    device,
    VkDeviceMemory                              memory,
    VkExternalMemoryHandleTypeFlagsNV           handleType,
    HANDLE*                                     pHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandleNV>::Dispatch(manager, device, memory, handleType, pHandle);

    VkResult result = GetDeviceTable(device)->GetMemoryWin32HandleNV(device, memory, handleType, pHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryWin32HandleNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeFlagsValue(handleType);
        encoder->EncodeVoidPtrPtr(pHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryWin32HandleNV>::Dispatch(manager, result, device, memory, handleType, pHandle);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateViSurfaceNN(
    VkInstance                                  instance,
    const VkViSurfaceCreateInfoNN*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateViSurfaceNN>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateViSurfaceNN(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateViSurfaceNN);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkViSurfaceCreateInfoNN>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateViSurfaceNN>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBeginConditionalRenderingEXT(
    VkCommandBuffer                             commandBuffer,
    const VkConditionalRenderingBeginInfoEXT*   pConditionalRenderingBegin)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginConditionalRenderingEXT>::Dispatch(manager, commandBuffer, pConditionalRenderingBegin);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginConditionalRenderingEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pConditionalRenderingBegin);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBeginConditionalRenderingEXTHandles, pConditionalRenderingBegin);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkConditionalRenderingBeginInfoEXT* pConditionalRenderingBegin_unwrapped = UnwrapStructPtrHandles(pConditionalRenderingBegin, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBeginConditionalRenderingEXT(commandBuffer, pConditionalRenderingBegin_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginConditionalRenderingEXT>::Dispatch(manager, commandBuffer, pConditionalRenderingBegin);
}

VKAPI_ATTR void VKAPI_CALL CmdEndConditionalRenderingEXT(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndConditionalRenderingEXT>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndConditionalRenderingEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndConditionalRenderingEXT(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndConditionalRenderingEXT>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWScalingNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstViewport,
    uint32_t                                    viewportCount,
    const VkViewportWScalingNV*                 pViewportWScalings)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetViewportWScalingNV>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pViewportWScalings);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetViewportWScalingNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstViewport);
        encoder->EncodeUInt32Value(viewportCount);
        EncodeStructArray(encoder, pViewportWScalings, viewportCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetViewportWScalingNV(commandBuffer, firstViewport, viewportCount, pViewportWScalings);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetViewportWScalingNV>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pViewportWScalings);
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseDisplayEXT(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayKHR                                display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkReleaseDisplayEXT>::Dispatch(manager, physicalDevice, display);

    VkResult result = GetInstanceTable(physicalDevice)->ReleaseDisplayEXT(physicalDevice, display);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkReleaseDisplayEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkReleaseDisplayEXT>::Dispatch(manager, result, physicalDevice, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireXlibDisplayEXT(
    VkPhysicalDevice                            physicalDevice,
    Display*                                    dpy,
    VkDisplayKHR                                display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireXlibDisplayEXT>::Dispatch(manager, physicalDevice, dpy, display);

    VkResult result = GetInstanceTable(physicalDevice)->AcquireXlibDisplayEXT(physicalDevice, dpy, display);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireXlibDisplayEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeVoidPtr(dpy);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireXlibDisplayEXT>::Dispatch(manager, result, physicalDevice, dpy, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRandROutputDisplayEXT(
    VkPhysicalDevice                            physicalDevice,
    Display*                                    dpy,
    RROutput                                    rrOutput,
    VkDisplayKHR*                               pDisplay)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRandROutputDisplayEXT>::Dispatch(manager, physicalDevice, dpy, rrOutput, pDisplay);

    VkResult result = GetInstanceTable(physicalDevice)->GetRandROutputDisplayEXT(physicalDevice, dpy, rrOutput, pDisplay);

    if (result >= 0)
    {
        CreateWrappedHandle<PhysicalDeviceWrapper, NoParentWrapper, DisplayKHRWrapper>(physicalDevice, NoParentWrapper::kHandleValue, pDisplay, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetRandROutputDisplayEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeVoidPtr(dpy);
        encoder->EncodeSizeTValue(rrOutput);
        encoder->EncodeHandlePtr<DisplayKHRWrapper>(pDisplay, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, void>(result, physicalDevice, pDisplay, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRandROutputDisplayEXT>::Dispatch(manager, result, physicalDevice, dpy, rrOutput, pDisplay);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfaceCapabilities2EXT(
    VkPhysicalDevice                            physicalDevice,
    VkSurfaceKHR                                surface,
    VkSurfaceCapabilities2EXT*                  pSurfaceCapabilities)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2EXT>::Dispatch(manager, physicalDevice, surface, pSurfaceCapabilities);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfaceCapabilities2EXT(physicalDevice, surface, pSurfaceCapabilities);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<SurfaceKHRWrapper>(surface);
        EncodeStructPtr(encoder, pSurfaceCapabilities, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfaceCapabilities2EXT>::Dispatch(manager, result, physicalDevice, surface, pSurfaceCapabilities);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL DisplayPowerControlEXT(
    VkDevice                                    device,
    VkDisplayKHR                                display,
    const VkDisplayPowerInfoEXT*                pDisplayPowerInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDisplayPowerControlEXT>::Dispatch(manager, device, display, pDisplayPowerInfo);

    VkResult result = GetDeviceTable(device)->DisplayPowerControlEXT(device, display, pDisplayPowerInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkDisplayPowerControlEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        EncodeStructPtr(encoder, pDisplayPowerInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDisplayPowerControlEXT>::Dispatch(manager, result, device, display, pDisplayPowerInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL RegisterDeviceEventEXT(
    VkDevice                                    device,
    const VkDeviceEventInfoEXT*                 pDeviceEventInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkFence*                                    pFence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkRegisterDeviceEventEXT>::Dispatch(manager, device, pDeviceEventInfo, pAllocator, pFence);

    VkResult result = GetDeviceTable(device)->RegisterDeviceEventEXT(device, pDeviceEventInfo, pAllocator, pFence);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, FenceWrapper>(device, NoParentWrapper::kHandleValue, pFence, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkRegisterDeviceEventEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pDeviceEventInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<FenceWrapper>(pFence, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, FenceWrapper, void>(result, device, pFence, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkRegisterDeviceEventEXT>::Dispatch(manager, result, device, pDeviceEventInfo, pAllocator, pFence);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL RegisterDisplayEventEXT(
    VkDevice                                    device,
    VkDisplayKHR                                display,
    const VkDisplayEventInfoEXT*                pDisplayEventInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkFence*                                    pFence)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkRegisterDisplayEventEXT>::Dispatch(manager, device, display, pDisplayEventInfo, pAllocator, pFence);

    VkResult result = GetDeviceTable(device)->RegisterDisplayEventEXT(device, display, pDisplayEventInfo, pAllocator, pFence);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, DisplayKHRWrapper, FenceWrapper>(device, display, pFence, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkRegisterDisplayEventEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        EncodeStructPtr(encoder, pDisplayEventInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<FenceWrapper>(pFence, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, FenceWrapper, void>(result, device, pFence, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkRegisterDisplayEventEXT>::Dispatch(manager, result, device, display, pDisplayEventInfo, pAllocator, pFence);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSwapchainCounterEXT(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    VkSurfaceCounterFlagBitsEXT                 counter,
    uint64_t*                                   pCounterValue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSwapchainCounterEXT>::Dispatch(manager, device, swapchain, counter, pCounterValue);

    VkResult result = GetDeviceTable(device)->GetSwapchainCounterEXT(device, swapchain, counter, pCounterValue);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSwapchainCounterEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeEnumValue(counter);
        encoder->EncodeUInt64Ptr(pCounterValue, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSwapchainCounterEXT>::Dispatch(manager, result, device, swapchain, counter, pCounterValue);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRefreshCycleDurationGOOGLE(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    VkRefreshCycleDurationGOOGLE*               pDisplayTimingProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRefreshCycleDurationGOOGLE>::Dispatch(manager, device, swapchain, pDisplayTimingProperties);

    VkResult result = GetDeviceTable(device)->GetRefreshCycleDurationGOOGLE(device, swapchain, pDisplayTimingProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetRefreshCycleDurationGOOGLE);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        EncodeStructPtr(encoder, pDisplayTimingProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRefreshCycleDurationGOOGLE>::Dispatch(manager, result, device, swapchain, pDisplayTimingProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPastPresentationTimingGOOGLE(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain,
    uint32_t*                                   pPresentationTimingCount,
    VkPastPresentationTimingGOOGLE*             pPresentationTimings)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPastPresentationTimingGOOGLE>::Dispatch(manager, device, swapchain, pPresentationTimingCount, pPresentationTimings);

    VkResult result = GetDeviceTable(device)->GetPastPresentationTimingGOOGLE(device, swapchain, pPresentationTimingCount, pPresentationTimings);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPastPresentationTimingGOOGLE);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeUInt32Ptr(pPresentationTimingCount, omit_output_data);
        EncodeStructArray(encoder, pPresentationTimings, (pPresentationTimingCount != nullptr) ? (*pPresentationTimingCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPastPresentationTimingGOOGLE>::Dispatch(manager, result, device, swapchain, pPresentationTimingCount, pPresentationTimings);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstDiscardRectangle,
    uint32_t                                    discardRectangleCount,
    const VkRect2D*                             pDiscardRectangles)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEXT>::Dispatch(manager, commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstDiscardRectangle);
        encoder->EncodeUInt32Value(discardRectangleCount);
        EncodeStructArray(encoder, pDiscardRectangles, discardRectangleCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDiscardRectangleEXT(commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEXT>::Dispatch(manager, commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    discardRectangleEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEnableEXT>::Dispatch(manager, commandBuffer, discardRectangleEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(discardRectangleEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDiscardRectangleEnableEXT(commandBuffer, discardRectangleEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleEnableEXT>::Dispatch(manager, commandBuffer, discardRectangleEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDiscardRectangleModeEXT(
    VkCommandBuffer                             commandBuffer,
    VkDiscardRectangleModeEXT                   discardRectangleMode)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleModeEXT>::Dispatch(manager, commandBuffer, discardRectangleMode);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleModeEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(discardRectangleMode);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDiscardRectangleModeEXT(commandBuffer, discardRectangleMode);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDiscardRectangleModeEXT>::Dispatch(manager, commandBuffer, discardRectangleMode);
}

VKAPI_ATTR void VKAPI_CALL SetHdrMetadataEXT(
    VkDevice                                    device,
    uint32_t                                    swapchainCount,
    const VkSwapchainKHR*                       pSwapchains,
    const VkHdrMetadataEXT*                     pMetadata)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetHdrMetadataEXT>::Dispatch(manager, device, swapchainCount, pSwapchains, pMetadata);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetHdrMetadataEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(swapchainCount);
        encoder->EncodeHandleArray<SwapchainKHRWrapper>(pSwapchains, swapchainCount);
        EncodeStructArray(encoder, pMetadata, swapchainCount);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->SetHdrMetadataEXT(device, swapchainCount, pSwapchains, pMetadata);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetHdrMetadataEXT>::Dispatch(manager, device, swapchainCount, pSwapchains, pMetadata);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateIOSSurfaceMVK(
    VkInstance                                  instance,
    const VkIOSSurfaceCreateInfoMVK*            pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateIOSSurfaceMVK>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateIOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateIOSSurfaceMVK);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkIOSSurfaceCreateInfoMVK>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateIOSSurfaceMVK>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateMacOSSurfaceMVK(
    VkInstance                                  instance,
    const VkMacOSSurfaceCreateInfoMVK*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateMacOSSurfaceMVK>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateMacOSSurfaceMVK(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateMacOSSurfaceMVK);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkMacOSSurfaceCreateInfoMVK>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateMacOSSurfaceMVK>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SetDebugUtilsObjectNameEXT(
    VkDevice                                    device,
    const VkDebugUtilsObjectNameInfoEXT*        pNameInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetDebugUtilsObjectNameEXT>::Dispatch(manager, device, pNameInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDebugUtilsObjectNameInfoEXT* pNameInfo_unwrapped = UnwrapStructPtrHandles(pNameInfo, handle_unwrap_memory);

    auto device_wrapper = GetWrapper<DeviceWrapper>(device);
    auto physical_device = device_wrapper->physical_device->handle;
    VkResult result = GetInstanceTable(physical_device)->SetDebugUtilsObjectNameEXT(device, pNameInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetDebugUtilsObjectNameEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pNameInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetDebugUtilsObjectNameEXT>::Dispatch(manager, result, device, pNameInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL SetDebugUtilsObjectTagEXT(
    VkDevice                                    device,
    const VkDebugUtilsObjectTagInfoEXT*         pTagInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetDebugUtilsObjectTagEXT>::Dispatch(manager, device, pTagInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDebugUtilsObjectTagInfoEXT* pTagInfo_unwrapped = UnwrapStructPtrHandles(pTagInfo, handle_unwrap_memory);

    auto device_wrapper = GetWrapper<DeviceWrapper>(device);
    auto physical_device = device_wrapper->physical_device->handle;
    VkResult result = GetInstanceTable(physical_device)->SetDebugUtilsObjectTagEXT(device, pTagInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetDebugUtilsObjectTagEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pTagInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetDebugUtilsObjectTagEXT>::Dispatch(manager, result, device, pTagInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL QueueBeginDebugUtilsLabelEXT(
    VkQueue                                     queue,
    const VkDebugUtilsLabelEXT*                 pLabelInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueBeginDebugUtilsLabelEXT>::Dispatch(manager, queue, pLabelInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueBeginDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        EncodeStructPtr(encoder, pLabelInfo);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(queue)->QueueBeginDebugUtilsLabelEXT(queue, pLabelInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueBeginDebugUtilsLabelEXT>::Dispatch(manager, queue, pLabelInfo);
}

VKAPI_ATTR void VKAPI_CALL QueueEndDebugUtilsLabelEXT(
    VkQueue                                     queue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueEndDebugUtilsLabelEXT>::Dispatch(manager, queue);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueEndDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(queue)->QueueEndDebugUtilsLabelEXT(queue);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueEndDebugUtilsLabelEXT>::Dispatch(manager, queue);
}

VKAPI_ATTR void VKAPI_CALL QueueInsertDebugUtilsLabelEXT(
    VkQueue                                     queue,
    const VkDebugUtilsLabelEXT*                 pLabelInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueInsertDebugUtilsLabelEXT>::Dispatch(manager, queue, pLabelInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueInsertDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        EncodeStructPtr(encoder, pLabelInfo);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(queue)->QueueInsertDebugUtilsLabelEXT(queue, pLabelInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueInsertDebugUtilsLabelEXT>::Dispatch(manager, queue, pLabelInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBeginDebugUtilsLabelEXT(
    VkCommandBuffer                             commandBuffer,
    const VkDebugUtilsLabelEXT*                 pLabelInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBeginDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer, pLabelInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBeginDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pLabelInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdBeginDebugUtilsLabelEXT(commandBuffer, pLabelInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBeginDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer, pLabelInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdEndDebugUtilsLabelEXT(
    VkCommandBuffer                             commandBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdEndDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdEndDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdEndDebugUtilsLabelEXT(commandBuffer);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdEndDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer);
}

VKAPI_ATTR void VKAPI_CALL CmdInsertDebugUtilsLabelEXT(
    VkCommandBuffer                             commandBuffer,
    const VkDebugUtilsLabelEXT*                 pLabelInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdInsertDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer, pLabelInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdInsertDebugUtilsLabelEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pLabelInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdInsertDebugUtilsLabelEXT(commandBuffer, pLabelInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdInsertDebugUtilsLabelEXT>::Dispatch(manager, commandBuffer, pLabelInfo);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDebugUtilsMessengerEXT(
    VkInstance                                  instance,
    const VkDebugUtilsMessengerCreateInfoEXT*   pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkDebugUtilsMessengerEXT*                   pMessenger)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDebugUtilsMessengerEXT>::Dispatch(manager, instance, pCreateInfo, pAllocator, pMessenger);

    VkResult result = GetInstanceTable(instance)->CreateDebugUtilsMessengerEXT(instance, pCreateInfo, pAllocator, pMessenger);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, DebugUtilsMessengerEXTWrapper>(instance, NoParentWrapper::kHandleValue, pMessenger, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDebugUtilsMessengerEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<DebugUtilsMessengerEXTWrapper>(pMessenger, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, DebugUtilsMessengerEXTWrapper, VkDebugUtilsMessengerCreateInfoEXT>(result, instance, pMessenger, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDebugUtilsMessengerEXT>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pMessenger);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyDebugUtilsMessengerEXT(
    VkInstance                                  instance,
    VkDebugUtilsMessengerEXT                    messenger,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyDebugUtilsMessengerEXT>::Dispatch(manager, instance, messenger, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyDebugUtilsMessengerEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeHandleValue<DebugUtilsMessengerEXTWrapper>(messenger);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<DebugUtilsMessengerEXTWrapper>(messenger);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetInstanceTable(instance)->DestroyDebugUtilsMessengerEXT(instance, messenger, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyDebugUtilsMessengerEXT>::Dispatch(manager, instance, messenger, pAllocator);

    DestroyWrappedHandle<DebugUtilsMessengerEXTWrapper>(messenger);
}

VKAPI_ATTR void VKAPI_CALL SubmitDebugUtilsMessageEXT(
    VkInstance                                  instance,
    VkDebugUtilsMessageSeverityFlagBitsEXT      messageSeverity,
    VkDebugUtilsMessageTypeFlagsEXT             messageTypes,
    const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSubmitDebugUtilsMessageEXT>::Dispatch(manager, instance, messageSeverity, messageTypes, pCallbackData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSubmitDebugUtilsMessageEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        encoder->EncodeEnumValue(messageSeverity);
        encoder->EncodeFlagsValue(messageTypes);
        EncodeStructPtr(encoder, pCallbackData);
        manager->EndApiCallCapture();
    }

    GetInstanceTable(instance)->SubmitDebugUtilsMessageEXT(instance, messageSeverity, messageTypes, pCallbackData);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSubmitDebugUtilsMessageEXT>::Dispatch(manager, instance, messageSeverity, messageTypes, pCallbackData);
}

VKAPI_ATTR VkResult VKAPI_CALL GetAndroidHardwareBufferPropertiesANDROID(
    VkDevice                                    device,
    const struct AHardwareBuffer*               buffer,
    VkAndroidHardwareBufferPropertiesANDROID*   pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetAndroidHardwareBufferPropertiesANDROID>::Dispatch(manager, device, buffer, pProperties);

    VkResult result = GetDeviceTable(device)->GetAndroidHardwareBufferPropertiesANDROID(device, buffer, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetAndroidHardwareBufferPropertiesANDROID);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeVoidPtr(buffer);
        EncodeStructPtr(encoder, pProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetAndroidHardwareBufferPropertiesANDROID>::Dispatch(manager, result, device, buffer, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryAndroidHardwareBufferANDROID(
    VkDevice                                    device,
    const VkMemoryGetAndroidHardwareBufferInfoANDROID* pInfo,
    struct AHardwareBuffer**                    pBuffer)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryAndroidHardwareBufferANDROID>::Dispatch(manager, device, pInfo, pBuffer);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryGetAndroidHardwareBufferInfoANDROID* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetMemoryAndroidHardwareBufferANDROID(device, pInfo_unwrapped, pBuffer);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryAndroidHardwareBufferANDROID);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeVoidPtrPtr(pBuffer, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryAndroidHardwareBufferANDROID>::Dispatch(manager, result, device, pInfo, pBuffer);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetSampleLocationsEXT(
    VkCommandBuffer                             commandBuffer,
    const VkSampleLocationsInfoEXT*             pSampleLocationsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetSampleLocationsEXT>::Dispatch(manager, commandBuffer, pSampleLocationsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetSampleLocationsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pSampleLocationsInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetSampleLocationsEXT(commandBuffer, pSampleLocationsInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetSampleLocationsEXT>::Dispatch(manager, commandBuffer, pSampleLocationsInfo);
}

VKAPI_ATTR void VKAPI_CALL GetPhysicalDeviceMultisamplePropertiesEXT(
    VkPhysicalDevice                            physicalDevice,
    VkSampleCountFlagBits                       samples,
    VkMultisamplePropertiesEXT*                 pMultisampleProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMultisamplePropertiesEXT>::Dispatch(manager, physicalDevice, samples, pMultisampleProperties);

    GetInstanceTable(physicalDevice)->GetPhysicalDeviceMultisamplePropertiesEXT(physicalDevice, samples, pMultisampleProperties);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceMultisamplePropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeEnumValue(samples);
        EncodeStructPtr(encoder, pMultisampleProperties);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceMultisamplePropertiesEXT>::Dispatch(manager, physicalDevice, samples, pMultisampleProperties);
}

VKAPI_ATTR VkResult VKAPI_CALL GetImageDrmFormatModifierPropertiesEXT(
    VkDevice                                    device,
    VkImage                                     image,
    VkImageDrmFormatModifierPropertiesEXT*      pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageDrmFormatModifierPropertiesEXT>::Dispatch(manager, device, image, pProperties);

    VkResult result = GetDeviceTable(device)->GetImageDrmFormatModifierPropertiesEXT(device, image, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageDrmFormatModifierPropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageDrmFormatModifierPropertiesEXT>::Dispatch(manager, result, device, image, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateValidationCacheEXT(
    VkDevice                                    device,
    const VkValidationCacheCreateInfoEXT*       pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkValidationCacheEXT*                       pValidationCache)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateValidationCacheEXT>::Dispatch(manager, device, pCreateInfo, pAllocator, pValidationCache);

    VkResult result = GetDeviceTable(device)->CreateValidationCacheEXT(device, pCreateInfo, pAllocator, pValidationCache);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, ValidationCacheEXTWrapper>(device, NoParentWrapper::kHandleValue, pValidationCache, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateValidationCacheEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<ValidationCacheEXTWrapper>(pValidationCache, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, ValidationCacheEXTWrapper, VkValidationCacheCreateInfoEXT>(result, device, pValidationCache, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateValidationCacheEXT>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pValidationCache);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyValidationCacheEXT(
    VkDevice                                    device,
    VkValidationCacheEXT                        validationCache,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyValidationCacheEXT>::Dispatch(manager, device, validationCache, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyValidationCacheEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ValidationCacheEXTWrapper>(validationCache);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<ValidationCacheEXTWrapper>(validationCache);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyValidationCacheEXT(device, validationCache, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyValidationCacheEXT>::Dispatch(manager, device, validationCache, pAllocator);

    DestroyWrappedHandle<ValidationCacheEXTWrapper>(validationCache);
}

VKAPI_ATTR VkResult VKAPI_CALL MergeValidationCachesEXT(
    VkDevice                                    device,
    VkValidationCacheEXT                        dstCache,
    uint32_t                                    srcCacheCount,
    const VkValidationCacheEXT*                 pSrcCaches)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkMergeValidationCachesEXT>::Dispatch(manager, device, dstCache, srcCacheCount, pSrcCaches);

    VkResult result = GetDeviceTable(device)->MergeValidationCachesEXT(device, dstCache, srcCacheCount, pSrcCaches);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkMergeValidationCachesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ValidationCacheEXTWrapper>(dstCache);
        encoder->EncodeUInt32Value(srcCacheCount);
        encoder->EncodeHandleArray<ValidationCacheEXTWrapper>(pSrcCaches, srcCacheCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkMergeValidationCachesEXT>::Dispatch(manager, result, device, dstCache, srcCacheCount, pSrcCaches);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetValidationCacheDataEXT(
    VkDevice                                    device,
    VkValidationCacheEXT                        validationCache,
    size_t*                                     pDataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetValidationCacheDataEXT>::Dispatch(manager, device, validationCache, pDataSize, pData);

    VkResult result = GetDeviceTable(device)->GetValidationCacheDataEXT(device, validationCache, pDataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetValidationCacheDataEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ValidationCacheEXTWrapper>(validationCache);
        encoder->EncodeSizeTPtr(pDataSize, omit_output_data);
        encoder->EncodeVoidArray(pData, (pDataSize != nullptr) ? (*pDataSize) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetValidationCacheDataEXT>::Dispatch(manager, result, device, validationCache, pDataSize, pData);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindShadingRateImageNV(
    VkCommandBuffer                             commandBuffer,
    VkImageView                                 imageView,
    VkImageLayout                               imageLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindShadingRateImageNV>::Dispatch(manager, commandBuffer, imageView, imageLayout);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindShadingRateImageNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageViewWrapper>(imageView);
        encoder->EncodeEnumValue(imageLayout);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindShadingRateImageNVHandles, imageView);
    }

    GetDeviceTable(commandBuffer)->CmdBindShadingRateImageNV(commandBuffer, imageView, imageLayout);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindShadingRateImageNV>::Dispatch(manager, commandBuffer, imageView, imageLayout);
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportShadingRatePaletteNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstViewport,
    uint32_t                                    viewportCount,
    const VkShadingRatePaletteNV*               pShadingRatePalettes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetViewportShadingRatePaletteNV>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pShadingRatePalettes);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetViewportShadingRatePaletteNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstViewport);
        encoder->EncodeUInt32Value(viewportCount);
        EncodeStructArray(encoder, pShadingRatePalettes, viewportCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetViewportShadingRatePaletteNV(commandBuffer, firstViewport, viewportCount, pShadingRatePalettes);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetViewportShadingRatePaletteNV>::Dispatch(manager, commandBuffer, firstViewport, viewportCount, pShadingRatePalettes);
}

VKAPI_ATTR void VKAPI_CALL CmdSetCoarseSampleOrderNV(
    VkCommandBuffer                             commandBuffer,
    VkCoarseSampleOrderTypeNV                   sampleOrderType,
    uint32_t                                    customSampleOrderCount,
    const VkCoarseSampleOrderCustomNV*          pCustomSampleOrders)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetCoarseSampleOrderNV>::Dispatch(manager, commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetCoarseSampleOrderNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(sampleOrderType);
        encoder->EncodeUInt32Value(customSampleOrderCount);
        EncodeStructArray(encoder, pCustomSampleOrders, customSampleOrderCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetCoarseSampleOrderNV(commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetCoarseSampleOrderNV>::Dispatch(manager, commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateAccelerationStructureNV(
    VkDevice                                    device,
    const VkAccelerationStructureCreateInfoNV*  pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkAccelerationStructureNV*                  pAccelerationStructure)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateAccelerationStructureNV>::Dispatch(manager, device, pCreateInfo, pAllocator, pAccelerationStructure);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkAccelerationStructureCreateInfoNV* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateAccelerationStructureNV(device, pCreateInfo_unwrapped, pAllocator, pAccelerationStructure);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, AccelerationStructureNVWrapper>(device, NoParentWrapper::kHandleValue, pAccelerationStructure, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateAccelerationStructureNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<AccelerationStructureNVWrapper>(pAccelerationStructure, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, AccelerationStructureNVWrapper, VkAccelerationStructureCreateInfoNV>(result, device, pAccelerationStructure, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateAccelerationStructureNV>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pAccelerationStructure);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyAccelerationStructureNV(
    VkDevice                                    device,
    VkAccelerationStructureNV                   accelerationStructure,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyAccelerationStructureNV>::Dispatch(manager, device, accelerationStructure, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyAccelerationStructureNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(accelerationStructure);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<AccelerationStructureNVWrapper>(accelerationStructure);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyAccelerationStructureNV(device, accelerationStructure, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyAccelerationStructureNV>::Dispatch(manager, device, accelerationStructure, pAllocator);

    DestroyWrappedHandle<AccelerationStructureNVWrapper>(accelerationStructure);
}

VKAPI_ATTR void VKAPI_CALL GetAccelerationStructureMemoryRequirementsNV(
    VkDevice                                    device,
    const VkAccelerationStructureMemoryRequirementsInfoNV* pInfo,
    VkMemoryRequirements2KHR*                   pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetAccelerationStructureMemoryRequirementsNV>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkAccelerationStructureMemoryRequirementsInfoNV* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetAccelerationStructureMemoryRequirementsNV(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetAccelerationStructureMemoryRequirementsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetAccelerationStructureMemoryRequirementsNV>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR VkResult VKAPI_CALL BindAccelerationStructureMemoryNV(
    VkDevice                                    device,
    uint32_t                                    bindInfoCount,
    const VkBindAccelerationStructureMemoryInfoNV* pBindInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBindAccelerationStructureMemoryNV>::Dispatch(manager, device, bindInfoCount, pBindInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBindAccelerationStructureMemoryInfoNV* pBindInfos_unwrapped = UnwrapStructArrayHandles(pBindInfos, bindInfoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BindAccelerationStructureMemoryNV(device, bindInfoCount, pBindInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBindAccelerationStructureMemoryNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(bindInfoCount);
        EncodeStructArray(encoder, pBindInfos, bindInfoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBindAccelerationStructureMemoryNV>::Dispatch(manager, result, device, bindInfoCount, pBindInfos);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBuildAccelerationStructureNV(
    VkCommandBuffer                             commandBuffer,
    const VkAccelerationStructureInfoNV*        pInfo,
    VkBuffer                                    instanceData,
    VkDeviceSize                                instanceOffset,
    VkBool32                                    update,
    VkAccelerationStructureNV                   dst,
    VkAccelerationStructureNV                   src,
    VkBuffer                                    scratch,
    VkDeviceSize                                scratchOffset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBuildAccelerationStructureNV>::Dispatch(manager, commandBuffer, pInfo, instanceData, instanceOffset, update, dst, src, scratch, scratchOffset);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBuildAccelerationStructureNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeHandleValue<BufferWrapper>(instanceData);
        encoder->EncodeVkDeviceSizeValue(instanceOffset);
        encoder->EncodeVkBool32Value(update);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(dst);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(src);
        encoder->EncodeHandleValue<BufferWrapper>(scratch);
        encoder->EncodeVkDeviceSizeValue(scratchOffset);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBuildAccelerationStructureNVHandles, pInfo, instanceData, dst, src, scratch);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkAccelerationStructureInfoNV* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBuildAccelerationStructureNV(commandBuffer, pInfo_unwrapped, instanceData, instanceOffset, update, dst, src, scratch, scratchOffset);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBuildAccelerationStructureNV>::Dispatch(manager, commandBuffer, pInfo, instanceData, instanceOffset, update, dst, src, scratch, scratchOffset);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyAccelerationStructureNV(
    VkCommandBuffer                             commandBuffer,
    VkAccelerationStructureNV                   dst,
    VkAccelerationStructureNV                   src,
    VkCopyAccelerationStructureModeKHR          mode)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyAccelerationStructureNV>::Dispatch(manager, commandBuffer, dst, src, mode);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyAccelerationStructureNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(dst);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(src);
        encoder->EncodeEnumValue(mode);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyAccelerationStructureNVHandles, dst, src);
    }

    GetDeviceTable(commandBuffer)->CmdCopyAccelerationStructureNV(commandBuffer, dst, src, mode);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyAccelerationStructureNV>::Dispatch(manager, commandBuffer, dst, src, mode);
}

VKAPI_ATTR void VKAPI_CALL CmdTraceRaysNV(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    raygenShaderBindingTableBuffer,
    VkDeviceSize                                raygenShaderBindingOffset,
    VkBuffer                                    missShaderBindingTableBuffer,
    VkDeviceSize                                missShaderBindingOffset,
    VkDeviceSize                                missShaderBindingStride,
    VkBuffer                                    hitShaderBindingTableBuffer,
    VkDeviceSize                                hitShaderBindingOffset,
    VkDeviceSize                                hitShaderBindingStride,
    VkBuffer                                    callableShaderBindingTableBuffer,
    VkDeviceSize                                callableShaderBindingOffset,
    VkDeviceSize                                callableShaderBindingStride,
    uint32_t                                    width,
    uint32_t                                    height,
    uint32_t                                    depth)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdTraceRaysNV>::Dispatch(manager, commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width, height, depth);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdTraceRaysNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(raygenShaderBindingTableBuffer);
        encoder->EncodeVkDeviceSizeValue(raygenShaderBindingOffset);
        encoder->EncodeHandleValue<BufferWrapper>(missShaderBindingTableBuffer);
        encoder->EncodeVkDeviceSizeValue(missShaderBindingOffset);
        encoder->EncodeVkDeviceSizeValue(missShaderBindingStride);
        encoder->EncodeHandleValue<BufferWrapper>(hitShaderBindingTableBuffer);
        encoder->EncodeVkDeviceSizeValue(hitShaderBindingOffset);
        encoder->EncodeVkDeviceSizeValue(hitShaderBindingStride);
        encoder->EncodeHandleValue<BufferWrapper>(callableShaderBindingTableBuffer);
        encoder->EncodeVkDeviceSizeValue(callableShaderBindingOffset);
        encoder->EncodeVkDeviceSizeValue(callableShaderBindingStride);
        encoder->EncodeUInt32Value(width);
        encoder->EncodeUInt32Value(height);
        encoder->EncodeUInt32Value(depth);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdTraceRaysNVHandles, raygenShaderBindingTableBuffer, missShaderBindingTableBuffer, hitShaderBindingTableBuffer, callableShaderBindingTableBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdTraceRaysNV(commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width, height, depth);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdTraceRaysNV>::Dispatch(manager, commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width, height, depth);
}

VKAPI_ATTR VkResult VKAPI_CALL GetRayTracingShaderGroupHandlesKHR(
    VkDevice                                    device,
    VkPipeline                                  pipeline,
    uint32_t                                    firstGroup,
    uint32_t                                    groupCount,
    size_t                                      dataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesKHR>::Dispatch(manager, device, pipeline, firstGroup, groupCount, dataSize, pData);

    VkResult result = GetDeviceTable(device)->GetRayTracingShaderGroupHandlesKHR(device, pipeline, firstGroup, groupCount, dataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesKHR);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        encoder->EncodeUInt32Value(firstGroup);
        encoder->EncodeUInt32Value(groupCount);
        encoder->EncodeSizeTValue(dataSize);
        encoder->EncodeVoidArray(pData, dataSize, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesKHR>::Dispatch(manager, result, device, pipeline, firstGroup, groupCount, dataSize, pData);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetRayTracingShaderGroupHandlesNV(
    VkDevice                                    device,
    VkPipeline                                  pipeline,
    uint32_t                                    firstGroup,
    uint32_t                                    groupCount,
    size_t                                      dataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesNV>::Dispatch(manager, device, pipeline, firstGroup, groupCount, dataSize, pData);

    VkResult result = GetDeviceTable(device)->GetRayTracingShaderGroupHandlesNV(device, pipeline, firstGroup, groupCount, dataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        encoder->EncodeUInt32Value(firstGroup);
        encoder->EncodeUInt32Value(groupCount);
        encoder->EncodeSizeTValue(dataSize);
        encoder->EncodeVoidArray(pData, dataSize, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetRayTracingShaderGroupHandlesNV>::Dispatch(manager, result, device, pipeline, firstGroup, groupCount, dataSize, pData);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetAccelerationStructureHandleNV(
    VkDevice                                    device,
    VkAccelerationStructureNV                   accelerationStructure,
    size_t                                      dataSize,
    void*                                       pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetAccelerationStructureHandleNV>::Dispatch(manager, device, accelerationStructure, dataSize, pData);

    VkResult result = GetDeviceTable(device)->GetAccelerationStructureHandleNV(device, accelerationStructure, dataSize, pData);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetAccelerationStructureHandleNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<AccelerationStructureNVWrapper>(accelerationStructure);
        encoder->EncodeSizeTValue(dataSize);
        encoder->EncodeVoidArray(pData, dataSize, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetAccelerationStructureHandleNV>::Dispatch(manager, result, device, accelerationStructure, dataSize, pData);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteAccelerationStructuresPropertiesNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    accelerationStructureCount,
    const VkAccelerationStructureNV*            pAccelerationStructures,
    VkQueryType                                 queryType,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteAccelerationStructuresPropertiesNV>::Dispatch(manager, commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteAccelerationStructuresPropertiesNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(accelerationStructureCount);
        encoder->EncodeHandleArray<AccelerationStructureNVWrapper>(pAccelerationStructures, accelerationStructureCount);
        encoder->EncodeEnumValue(queryType);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteAccelerationStructuresPropertiesNVHandles, accelerationStructureCount, pAccelerationStructures, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdWriteAccelerationStructuresPropertiesNV(commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteAccelerationStructuresPropertiesNV>::Dispatch(manager, commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery);
}

VKAPI_ATTR VkResult VKAPI_CALL CompileDeferredNV(
    VkDevice                                    device,
    VkPipeline                                  pipeline,
    uint32_t                                    shader)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCompileDeferredNV>::Dispatch(manager, device, pipeline, shader);

    VkResult result = GetDeviceTable(device)->CompileDeferredNV(device, pipeline, shader);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCompileDeferredNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        encoder->EncodeUInt32Value(shader);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCompileDeferredNV>::Dispatch(manager, result, device, pipeline, shader);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryHostPointerPropertiesEXT(
    VkDevice                                    device,
    VkExternalMemoryHandleTypeFlagBits          handleType,
    const void*                                 pHostPointer,
    VkMemoryHostPointerPropertiesEXT*           pMemoryHostPointerProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryHostPointerPropertiesEXT>::Dispatch(manager, device, handleType, pHostPointer, pMemoryHostPointerProperties);

    VkResult result = GetDeviceTable(device)->GetMemoryHostPointerPropertiesEXT(device, handleType, pHostPointer, pMemoryHostPointerProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryHostPointerPropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(handleType);
        encoder->EncodeVoidPtr(pHostPointer);
        EncodeStructPtr(encoder, pMemoryHostPointerProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryHostPointerPropertiesEXT>::Dispatch(manager, result, device, handleType, pHostPointer, pMemoryHostPointerProperties);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdWriteBufferMarkerAMD(
    VkCommandBuffer                             commandBuffer,
    VkPipelineStageFlagBits                     pipelineStage,
    VkBuffer                                    dstBuffer,
    VkDeviceSize                                dstOffset,
    uint32_t                                    marker)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteBufferMarkerAMD>::Dispatch(manager, commandBuffer, pipelineStage, dstBuffer, dstOffset, marker);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteBufferMarkerAMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineStage);
        encoder->EncodeHandleValue<BufferWrapper>(dstBuffer);
        encoder->EncodeVkDeviceSizeValue(dstOffset);
        encoder->EncodeUInt32Value(marker);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteBufferMarkerAMDHandles, dstBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdWriteBufferMarkerAMD(commandBuffer, pipelineStage, dstBuffer, dstOffset, marker);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteBufferMarkerAMD>::Dispatch(manager, commandBuffer, pipelineStage, dstBuffer, dstOffset, marker);
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCalibrateableTimeDomainsEXT(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pTimeDomainCount,
    VkTimeDomainKHR*                            pTimeDomains)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT>::Dispatch(manager, physicalDevice, pTimeDomainCount, pTimeDomains);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceCalibrateableTimeDomainsEXT(physicalDevice, pTimeDomainCount, pTimeDomains);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pTimeDomainCount, omit_output_data);
        encoder->EncodeEnumArray(pTimeDomains, (pTimeDomainCount != nullptr) ? (*pTimeDomainCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCalibrateableTimeDomainsEXT>::Dispatch(manager, result, physicalDevice, pTimeDomainCount, pTimeDomains);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetCalibratedTimestampsEXT(
    VkDevice                                    device,
    uint32_t                                    timestampCount,
    const VkCalibratedTimestampInfoKHR*         pTimestampInfos,
    uint64_t*                                   pTimestamps,
    uint64_t*                                   pMaxDeviation)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetCalibratedTimestampsEXT>::Dispatch(manager, device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);

    VkResult result = GetDeviceTable(device)->GetCalibratedTimestampsEXT(device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetCalibratedTimestampsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(timestampCount);
        EncodeStructArray(encoder, pTimestampInfos, timestampCount);
        encoder->EncodeUInt64Array(pTimestamps, timestampCount, omit_output_data);
        encoder->EncodeUInt64Ptr(pMaxDeviation, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetCalibratedTimestampsEXT>::Dispatch(manager, result, device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    taskCount,
    uint32_t                                    firstTask)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksNV>::Dispatch(manager, commandBuffer, taskCount, firstTask);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawMeshTasksNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(taskCount);
        encoder->EncodeUInt32Value(firstTask);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawMeshTasksNV(commandBuffer, taskCount, firstTask);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksNV>::Dispatch(manager, commandBuffer, taskCount, firstTask);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksIndirectNV(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    uint32_t                                    drawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectNV>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeUInt32Value(drawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawMeshTasksIndirectNVHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawMeshTasksIndirectNV(commandBuffer, buffer, offset, drawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectNV>::Dispatch(manager, commandBuffer, buffer, offset, drawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMeshTasksIndirectCountNV(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset,
    VkBuffer                                    countBuffer,
    VkDeviceSize                                countBufferOffset,
    uint32_t                                    maxDrawCount,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectCountNV>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectCountNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        encoder->EncodeHandleValue<BufferWrapper>(countBuffer);
        encoder->EncodeVkDeviceSizeValue(countBufferOffset);
        encoder->EncodeUInt32Value(maxDrawCount);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawMeshTasksIndirectCountNVHandles, buffer, countBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawMeshTasksIndirectCountNV(commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawMeshTasksIndirectCountNV>::Dispatch(manager, commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdSetExclusiveScissorEnableNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstExclusiveScissor,
    uint32_t                                    exclusiveScissorCount,
    const VkBool32*                             pExclusiveScissorEnables)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorEnableNV>::Dispatch(manager, commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissorEnables);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorEnableNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstExclusiveScissor);
        encoder->EncodeUInt32Value(exclusiveScissorCount);
        encoder->EncodeVkBool32Array(pExclusiveScissorEnables, exclusiveScissorCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetExclusiveScissorEnableNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissorEnables);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorEnableNV>::Dispatch(manager, commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissorEnables);
}

VKAPI_ATTR void VKAPI_CALL CmdSetExclusiveScissorNV(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstExclusiveScissor,
    uint32_t                                    exclusiveScissorCount,
    const VkRect2D*                             pExclusiveScissors)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorNV>::Dispatch(manager, commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstExclusiveScissor);
        encoder->EncodeUInt32Value(exclusiveScissorCount);
        EncodeStructArray(encoder, pExclusiveScissors, exclusiveScissorCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetExclusiveScissorNV(commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetExclusiveScissorNV>::Dispatch(manager, commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors);
}

VKAPI_ATTR void VKAPI_CALL CmdSetCheckpointNV(
    VkCommandBuffer                             commandBuffer,
    const void*                                 pCheckpointMarker)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetCheckpointNV>::Dispatch(manager, commandBuffer, pCheckpointMarker);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetCheckpointNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVoidPtr(pCheckpointMarker);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetCheckpointNV(commandBuffer, pCheckpointMarker);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetCheckpointNV>::Dispatch(manager, commandBuffer, pCheckpointMarker);
}

VKAPI_ATTR void VKAPI_CALL GetQueueCheckpointDataNV(
    VkQueue                                     queue,
    uint32_t*                                   pCheckpointDataCount,
    VkCheckpointDataNV*                         pCheckpointData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetQueueCheckpointDataNV>::Dispatch(manager, queue, pCheckpointDataCount, pCheckpointData);

    GetDeviceTable(queue)->GetQueueCheckpointDataNV(queue, pCheckpointDataCount, pCheckpointData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetQueueCheckpointDataNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeUInt32Ptr(pCheckpointDataCount);
        EncodeStructArray(encoder, pCheckpointData, (pCheckpointDataCount != nullptr) ? (*pCheckpointDataCount) : 0);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetQueueCheckpointDataNV>::Dispatch(manager, queue, pCheckpointDataCount, pCheckpointData);
}

VKAPI_ATTR VkResult VKAPI_CALL InitializePerformanceApiINTEL(
    VkDevice                                    device,
    const VkInitializePerformanceApiInfoINTEL*  pInitializeInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkInitializePerformanceApiINTEL>::Dispatch(manager, device, pInitializeInfo);

    VkResult result = GetDeviceTable(device)->InitializePerformanceApiINTEL(device, pInitializeInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkInitializePerformanceApiINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInitializeInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkInitializePerformanceApiINTEL>::Dispatch(manager, result, device, pInitializeInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL UninitializePerformanceApiINTEL(
    VkDevice                                    device)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkUninitializePerformanceApiINTEL>::Dispatch(manager, device);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkUninitializePerformanceApiINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->UninitializePerformanceApiINTEL(device);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkUninitializePerformanceApiINTEL>::Dispatch(manager, device);
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceMarkerINTEL(
    VkCommandBuffer                             commandBuffer,
    const VkPerformanceMarkerInfoINTEL*         pMarkerInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceMarkerINTEL>::Dispatch(manager, commandBuffer, pMarkerInfo);

    VkResult result = GetDeviceTable(commandBuffer)->CmdSetPerformanceMarkerINTEL(commandBuffer, pMarkerInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPerformanceMarkerINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pMarkerInfo);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceMarkerINTEL>::Dispatch(manager, result, commandBuffer, pMarkerInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceStreamMarkerINTEL(
    VkCommandBuffer                             commandBuffer,
    const VkPerformanceStreamMarkerInfoINTEL*   pMarkerInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceStreamMarkerINTEL>::Dispatch(manager, commandBuffer, pMarkerInfo);

    VkResult result = GetDeviceTable(commandBuffer)->CmdSetPerformanceStreamMarkerINTEL(commandBuffer, pMarkerInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPerformanceStreamMarkerINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pMarkerInfo);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceStreamMarkerINTEL>::Dispatch(manager, result, commandBuffer, pMarkerInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CmdSetPerformanceOverrideINTEL(
    VkCommandBuffer                             commandBuffer,
    const VkPerformanceOverrideInfoINTEL*       pOverrideInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceOverrideINTEL>::Dispatch(manager, commandBuffer, pOverrideInfo);

    VkResult result = GetDeviceTable(commandBuffer)->CmdSetPerformanceOverrideINTEL(commandBuffer, pOverrideInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPerformanceOverrideINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pOverrideInfo);
        encoder->EncodeEnumValue(result);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPerformanceOverrideINTEL>::Dispatch(manager, result, commandBuffer, pOverrideInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquirePerformanceConfigurationINTEL(
    VkDevice                                    device,
    const VkPerformanceConfigurationAcquireInfoINTEL* pAcquireInfo,
    VkPerformanceConfigurationINTEL*            pConfiguration)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquirePerformanceConfigurationINTEL>::Dispatch(manager, device, pAcquireInfo, pConfiguration);

    VkResult result = GetDeviceTable(device)->AcquirePerformanceConfigurationINTEL(device, pAcquireInfo, pConfiguration);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, PerformanceConfigurationINTELWrapper>(device, NoParentWrapper::kHandleValue, pConfiguration, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkAcquirePerformanceConfigurationINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pAcquireInfo);
        encoder->EncodeHandlePtr<PerformanceConfigurationINTELWrapper>(pConfiguration, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, PerformanceConfigurationINTELWrapper, void>(result, device, pConfiguration, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquirePerformanceConfigurationINTEL>::Dispatch(manager, result, device, pAcquireInfo, pConfiguration);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ReleasePerformanceConfigurationINTEL(
    VkDevice                                    device,
    VkPerformanceConfigurationINTEL             configuration)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkReleasePerformanceConfigurationINTEL>::Dispatch(manager, device, configuration);

    ScopedDestroyLock exclusive_scoped_lock;
    VkResult result = GetDeviceTable(device)->ReleasePerformanceConfigurationINTEL(device, configuration);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkReleasePerformanceConfigurationINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PerformanceConfigurationINTELWrapper>(configuration);
        encoder->EncodeEnumValue(result);
        manager->EndDestroyApiCallCapture<PerformanceConfigurationINTELWrapper>(configuration);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkReleasePerformanceConfigurationINTEL>::Dispatch(manager, result, device, configuration);

    DestroyWrappedHandle<PerformanceConfigurationINTELWrapper>(configuration);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL QueueSetPerformanceConfigurationINTEL(
    VkQueue                                     queue,
    VkPerformanceConfigurationINTEL             configuration)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkQueueSetPerformanceConfigurationINTEL>::Dispatch(manager, queue, configuration);

    VkResult result = GetDeviceTable(queue)->QueueSetPerformanceConfigurationINTEL(queue, configuration);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkQueueSetPerformanceConfigurationINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<QueueWrapper>(queue);
        encoder->EncodeHandleValue<PerformanceConfigurationINTELWrapper>(configuration);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkQueueSetPerformanceConfigurationINTEL>::Dispatch(manager, result, queue, configuration);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPerformanceParameterINTEL(
    VkDevice                                    device,
    VkPerformanceParameterTypeINTEL             parameter,
    VkPerformanceValueINTEL*                    pValue)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPerformanceParameterINTEL>::Dispatch(manager, device, parameter, pValue);

    VkResult result = GetDeviceTable(device)->GetPerformanceParameterINTEL(device, parameter, pValue);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPerformanceParameterINTEL);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(parameter);
        EncodeStructPtr(encoder, pValue, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPerformanceParameterINTEL>::Dispatch(manager, result, device, parameter, pValue);

    return result;
}

VKAPI_ATTR void VKAPI_CALL SetLocalDimmingAMD(
    VkDevice                                    device,
    VkSwapchainKHR                              swapChain,
    VkBool32                                    localDimmingEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetLocalDimmingAMD>::Dispatch(manager, device, swapChain, localDimmingEnable);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetLocalDimmingAMD);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapChain);
        encoder->EncodeVkBool32Value(localDimmingEnable);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->SetLocalDimmingAMD(device, swapChain, localDimmingEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetLocalDimmingAMD>::Dispatch(manager, device, swapChain, localDimmingEnable);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateImagePipeSurfaceFUCHSIA(
    VkInstance                                  instance,
    const VkImagePipeSurfaceCreateInfoFUCHSIA*  pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateImagePipeSurfaceFUCHSIA>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateImagePipeSurfaceFUCHSIA(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateImagePipeSurfaceFUCHSIA);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkImagePipeSurfaceCreateInfoFUCHSIA>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateImagePipeSurfaceFUCHSIA>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateMetalSurfaceEXT(
    VkInstance                                  instance,
    const VkMetalSurfaceCreateInfoEXT*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateMetalSurfaceEXT>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateMetalSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateMetalSurfaceEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkMetalSurfaceCreateInfoEXT>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateMetalSurfaceEXT>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetBufferDeviceAddressEXT(
    VkDevice                                    device,
    const VkBufferDeviceAddressInfo*            pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddressEXT>::Dispatch(manager, device, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkBufferDeviceAddressInfo* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkDeviceAddress result = GetDeviceTable(device)->GetBufferDeviceAddressEXT(device, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetBufferDeviceAddressEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeVkDeviceAddressValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetBufferDeviceAddressEXT>::Dispatch(manager, result, device, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceToolPropertiesEXT(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pToolCount,
    VkPhysicalDeviceToolProperties*             pToolProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolPropertiesEXT>::Dispatch(manager, physicalDevice, pToolCount, pToolProperties);

    VkResult result = manager->OverrideGetPhysicalDeviceToolPropertiesEXT(physicalDevice, pToolCount, pToolProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolPropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pToolCount, omit_output_data);
        EncodeStructArray(encoder, pToolProperties, (pToolCount != nullptr) ? (*pToolCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceToolPropertiesEXT>::Dispatch(manager, result, physicalDevice, pToolCount, pToolProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceCooperativeMatrixPropertiesNV(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pPropertyCount,
    VkCooperativeMatrixPropertiesNV*            pProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV>::Dispatch(manager, physicalDevice, pPropertyCount, pProperties);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceCooperativeMatrixPropertiesNV(physicalDevice, pPropertyCount, pProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pPropertyCount, omit_output_data);
        EncodeStructArray(encoder, pProperties, (pPropertyCount != nullptr) ? (*pPropertyCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceCooperativeMatrixPropertiesNV>::Dispatch(manager, result, physicalDevice, pPropertyCount, pProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(
    VkPhysicalDevice                            physicalDevice,
    uint32_t*                                   pCombinationCount,
    VkFramebufferMixedSamplesCombinationNV*     pCombinations)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV>::Dispatch(manager, physicalDevice, pCombinationCount, pCombinations);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV(physicalDevice, pCombinationCount, pCombinations);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Ptr(pCombinationCount, omit_output_data);
        EncodeStructArray(encoder, pCombinations, (pCombinationCount != nullptr) ? (*pCombinationCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSupportedFramebufferMixedSamplesCombinationsNV>::Dispatch(manager, result, physicalDevice, pCombinationCount, pCombinations);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetPhysicalDeviceSurfacePresentModes2EXT(
    VkPhysicalDevice                            physicalDevice,
    const VkPhysicalDeviceSurfaceInfo2KHR*      pSurfaceInfo,
    uint32_t*                                   pPresentModeCount,
    VkPresentModeKHR*                           pPresentModes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModes2EXT>::Dispatch(manager, physicalDevice, pSurfaceInfo, pPresentModeCount, pPresentModes);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo_unwrapped = UnwrapStructPtrHandles(pSurfaceInfo, handle_unwrap_memory);

    VkResult result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceSurfacePresentModes2EXT(physicalDevice, pSurfaceInfo_unwrapped, pPresentModeCount, pPresentModes);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModes2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        EncodeStructPtr(encoder, pSurfaceInfo);
        encoder->EncodeUInt32Ptr(pPresentModeCount, omit_output_data);
        encoder->EncodeEnumArray(pPresentModes, (pPresentModeCount != nullptr) ? (*pPresentModeCount) : 0, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceSurfacePresentModes2EXT>::Dispatch(manager, result, physicalDevice, pSurfaceInfo, pPresentModeCount, pPresentModes);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireFullScreenExclusiveModeEXT(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireFullScreenExclusiveModeEXT>::Dispatch(manager, device, swapchain);

    VkResult result = GetDeviceTable(device)->AcquireFullScreenExclusiveModeEXT(device, swapchain);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireFullScreenExclusiveModeEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireFullScreenExclusiveModeEXT>::Dispatch(manager, result, device, swapchain);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseFullScreenExclusiveModeEXT(
    VkDevice                                    device,
    VkSwapchainKHR                              swapchain)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkReleaseFullScreenExclusiveModeEXT>::Dispatch(manager, device, swapchain);

    VkResult result = GetDeviceTable(device)->ReleaseFullScreenExclusiveModeEXT(device, swapchain);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkReleaseFullScreenExclusiveModeEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<SwapchainKHRWrapper>(swapchain);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkReleaseFullScreenExclusiveModeEXT>::Dispatch(manager, result, device, swapchain);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceGroupSurfacePresentModes2EXT(
    VkDevice                                    device,
    const VkPhysicalDeviceSurfaceInfo2KHR*      pSurfaceInfo,
    VkDeviceGroupPresentModeFlagsKHR*           pModes)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModes2EXT>::Dispatch(manager, device, pSurfaceInfo, pModes);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo_unwrapped = UnwrapStructPtrHandles(pSurfaceInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetDeviceGroupSurfacePresentModes2EXT(device, pSurfaceInfo_unwrapped, pModes);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModes2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pSurfaceInfo);
        encoder->EncodeFlagsPtr(pModes, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceGroupSurfacePresentModes2EXT>::Dispatch(manager, result, device, pSurfaceInfo, pModes);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateHeadlessSurfaceEXT(
    VkInstance                                  instance,
    const VkHeadlessSurfaceCreateInfoEXT*       pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateHeadlessSurfaceEXT>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateHeadlessSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateHeadlessSurfaceEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkHeadlessSurfaceCreateInfoEXT>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateHeadlessSurfaceEXT>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetLineStippleEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    lineStippleFactor,
    uint16_t                                    lineStipplePattern)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetLineStippleEXT>::Dispatch(manager, commandBuffer, lineStippleFactor, lineStipplePattern);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetLineStippleEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(lineStippleFactor);
        encoder->EncodeUInt16Value(lineStipplePattern);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetLineStippleEXT(commandBuffer, lineStippleFactor, lineStipplePattern);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetLineStippleEXT>::Dispatch(manager, commandBuffer, lineStippleFactor, lineStipplePattern);
}

VKAPI_ATTR void VKAPI_CALL ResetQueryPoolEXT(
    VkDevice                                    device,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery,
    uint32_t                                    queryCount)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkResetQueryPoolEXT>::Dispatch(manager, device, queryPool, firstQuery, queryCount);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkResetQueryPoolEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        encoder->EncodeUInt32Value(queryCount);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->ResetQueryPoolEXT(device, queryPool, firstQuery, queryCount);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkResetQueryPoolEXT>::Dispatch(manager, device, queryPool, firstQuery, queryCount);
}

VKAPI_ATTR void VKAPI_CALL CmdSetCullModeEXT(
    VkCommandBuffer                             commandBuffer,
    VkCullModeFlags                             cullMode)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetCullModeEXT>::Dispatch(manager, commandBuffer, cullMode);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetCullModeEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(cullMode);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetCullModeEXT(commandBuffer, cullMode);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetCullModeEXT>::Dispatch(manager, commandBuffer, cullMode);
}

VKAPI_ATTR void VKAPI_CALL CmdSetFrontFaceEXT(
    VkCommandBuffer                             commandBuffer,
    VkFrontFace                                 frontFace)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetFrontFaceEXT>::Dispatch(manager, commandBuffer, frontFace);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetFrontFaceEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(frontFace);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetFrontFaceEXT(commandBuffer, frontFace);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetFrontFaceEXT>::Dispatch(manager, commandBuffer, frontFace);
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveTopologyEXT(
    VkCommandBuffer                             commandBuffer,
    VkPrimitiveTopology                         primitiveTopology)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopologyEXT>::Dispatch(manager, commandBuffer, primitiveTopology);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopologyEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(primitiveTopology);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetPrimitiveTopologyEXT(commandBuffer, primitiveTopology);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveTopologyEXT>::Dispatch(manager, commandBuffer, primitiveTopology);
}

VKAPI_ATTR void VKAPI_CALL CmdSetViewportWithCountEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    viewportCount,
    const VkViewport*                           pViewports)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetViewportWithCountEXT>::Dispatch(manager, commandBuffer, viewportCount, pViewports);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetViewportWithCountEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(viewportCount);
        EncodeStructArray(encoder, pViewports, viewportCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetViewportWithCountEXT(commandBuffer, viewportCount, pViewports);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetViewportWithCountEXT>::Dispatch(manager, commandBuffer, viewportCount, pViewports);
}

VKAPI_ATTR void VKAPI_CALL CmdSetScissorWithCountEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    scissorCount,
    const VkRect2D*                             pScissors)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetScissorWithCountEXT>::Dispatch(manager, commandBuffer, scissorCount, pScissors);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetScissorWithCountEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(scissorCount);
        EncodeStructArray(encoder, pScissors, scissorCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetScissorWithCountEXT(commandBuffer, scissorCount, pScissors);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetScissorWithCountEXT>::Dispatch(manager, commandBuffer, scissorCount, pScissors);
}

VKAPI_ATTR void VKAPI_CALL CmdBindVertexBuffers2EXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    firstBinding,
    uint32_t                                    bindingCount,
    const VkBuffer*                             pBuffers,
    const VkDeviceSize*                         pOffsets,
    const VkDeviceSize*                         pSizes,
    const VkDeviceSize*                         pStrides)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2EXT>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(firstBinding);
        encoder->EncodeUInt32Value(bindingCount);
        encoder->EncodeHandleArray<BufferWrapper>(pBuffers, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pOffsets, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pSizes, bindingCount);
        encoder->EncodeVkDeviceSizeArray(pStrides, bindingCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindVertexBuffers2EXTHandles, bindingCount, pBuffers);
    }

    GetDeviceTable(commandBuffer)->CmdBindVertexBuffers2EXT(commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindVertexBuffers2EXT>::Dispatch(manager, commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes, pStrides);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthTestEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthTestEnableEXT>::Dispatch(manager, commandBuffer, depthTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthTestEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthTestEnableEXT(commandBuffer, depthTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthTestEnableEXT>::Dispatch(manager, commandBuffer, depthTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthWriteEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthWriteEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnableEXT>::Dispatch(manager, commandBuffer, depthWriteEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthWriteEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthWriteEnableEXT(commandBuffer, depthWriteEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthWriteEnableEXT>::Dispatch(manager, commandBuffer, depthWriteEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthCompareOpEXT(
    VkCommandBuffer                             commandBuffer,
    VkCompareOp                                 depthCompareOp)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthCompareOpEXT>::Dispatch(manager, commandBuffer, depthCompareOp);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthCompareOpEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(depthCompareOp);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthCompareOpEXT(commandBuffer, depthCompareOp);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthCompareOpEXT>::Dispatch(manager, commandBuffer, depthCompareOp);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBoundsTestEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthBoundsTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnableEXT>::Dispatch(manager, commandBuffer, depthBoundsTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthBoundsTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBoundsTestEnableEXT(commandBuffer, depthBoundsTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBoundsTestEnableEXT>::Dispatch(manager, commandBuffer, depthBoundsTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilTestEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    stencilTestEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilTestEnableEXT>::Dispatch(manager, commandBuffer, stencilTestEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilTestEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(stencilTestEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilTestEnableEXT(commandBuffer, stencilTestEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilTestEnableEXT>::Dispatch(manager, commandBuffer, stencilTestEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetStencilOpEXT(
    VkCommandBuffer                             commandBuffer,
    VkStencilFaceFlags                          faceMask,
    VkStencilOp                                 failOp,
    VkStencilOp                                 passOp,
    VkStencilOp                                 depthFailOp,
    VkCompareOp                                 compareOp)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetStencilOpEXT>::Dispatch(manager, commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetStencilOpEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeFlagsValue(faceMask);
        encoder->EncodeEnumValue(failOp);
        encoder->EncodeEnumValue(passOp);
        encoder->EncodeEnumValue(depthFailOp);
        encoder->EncodeEnumValue(compareOp);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetStencilOpEXT(commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetStencilOpEXT>::Dispatch(manager, commandBuffer, faceMask, failOp, passOp, depthFailOp, compareOp);
}

VKAPI_ATTR VkResult VKAPI_CALL CopyMemoryToImageEXT(
    VkDevice                                    device,
    const VkCopyMemoryToImageInfoEXT*           pCopyMemoryToImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyMemoryToImageEXT>::Dispatch(manager, device, pCopyMemoryToImageInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMemoryToImageInfoEXT* pCopyMemoryToImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyMemoryToImageInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyMemoryToImageEXT(device, pCopyMemoryToImageInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyMemoryToImageEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCopyMemoryToImageInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyMemoryToImageEXT>::Dispatch(manager, result, device, pCopyMemoryToImageInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CopyImageToMemoryEXT(
    VkDevice                                    device,
    const VkCopyImageToMemoryInfoEXT*           pCopyImageToMemoryInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyImageToMemoryEXT>::Dispatch(manager, device, pCopyImageToMemoryInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageToMemoryInfoEXT* pCopyImageToMemoryInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageToMemoryInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyImageToMemoryEXT(device, pCopyImageToMemoryInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyImageToMemoryEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCopyImageToMemoryInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyImageToMemoryEXT>::Dispatch(manager, result, device, pCopyImageToMemoryInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CopyImageToImageEXT(
    VkDevice                                    device,
    const VkCopyImageToImageInfoEXT*            pCopyImageToImageInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyImageToImageEXT>::Dispatch(manager, device, pCopyImageToImageInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyImageToImageInfoEXT* pCopyImageToImageInfo_unwrapped = UnwrapStructPtrHandles(pCopyImageToImageInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyImageToImageEXT(device, pCopyImageToImageInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyImageToImageEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCopyImageToImageInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyImageToImageEXT>::Dispatch(manager, result, device, pCopyImageToImageInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL TransitionImageLayoutEXT(
    VkDevice                                    device,
    uint32_t                                    transitionCount,
    const VkHostImageLayoutTransitionInfoEXT*   pTransitions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkTransitionImageLayoutEXT>::Dispatch(manager, device, transitionCount, pTransitions);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkHostImageLayoutTransitionInfoEXT* pTransitions_unwrapped = UnwrapStructArrayHandles(pTransitions, transitionCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->TransitionImageLayoutEXT(device, transitionCount, pTransitions_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkTransitionImageLayoutEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(transitionCount);
        EncodeStructArray(encoder, pTransitions, transitionCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkTransitionImageLayoutEXT>::Dispatch(manager, result, device, transitionCount, pTransitions);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetImageSubresourceLayout2EXT(
    VkDevice                                    device,
    VkImage                                     image,
    const VkImageSubresource2KHR*               pSubresource,
    VkSubresourceLayout2KHR*                    pLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2EXT>::Dispatch(manager, device, image, pSubresource, pLayout);

    GetDeviceTable(device)->GetImageSubresourceLayout2EXT(device, image, pSubresource, pLayout);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<ImageWrapper>(image);
        EncodeStructPtr(encoder, pSubresource);
        EncodeStructPtr(encoder, pLayout);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetImageSubresourceLayout2EXT>::Dispatch(manager, device, image, pSubresource, pLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL ReleaseSwapchainImagesEXT(
    VkDevice                                    device,
    const VkReleaseSwapchainImagesInfoEXT*      pReleaseInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkReleaseSwapchainImagesEXT>::Dispatch(manager, device, pReleaseInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkReleaseSwapchainImagesInfoEXT* pReleaseInfo_unwrapped = UnwrapStructPtrHandles(pReleaseInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ReleaseSwapchainImagesEXT(device, pReleaseInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkReleaseSwapchainImagesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pReleaseInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkReleaseSwapchainImagesEXT>::Dispatch(manager, result, device, pReleaseInfo);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetGeneratedCommandsMemoryRequirementsNV(
    VkDevice                                    device,
    const VkGeneratedCommandsMemoryRequirementsInfoNV* pInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetGeneratedCommandsMemoryRequirementsNV>::Dispatch(manager, device, pInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkGeneratedCommandsMemoryRequirementsInfoNV* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetGeneratedCommandsMemoryRequirementsNV(device, pInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetGeneratedCommandsMemoryRequirementsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetGeneratedCommandsMemoryRequirementsNV>::Dispatch(manager, device, pInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL CmdPreprocessGeneratedCommandsNV(
    VkCommandBuffer                             commandBuffer,
    const VkGeneratedCommandsInfoNV*            pGeneratedCommandsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdPreprocessGeneratedCommandsNV>::Dispatch(manager, commandBuffer, pGeneratedCommandsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdPreprocessGeneratedCommandsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pGeneratedCommandsInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdPreprocessGeneratedCommandsNVHandles, pGeneratedCommandsInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkGeneratedCommandsInfoNV* pGeneratedCommandsInfo_unwrapped = UnwrapStructPtrHandles(pGeneratedCommandsInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdPreprocessGeneratedCommandsNV(commandBuffer, pGeneratedCommandsInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdPreprocessGeneratedCommandsNV>::Dispatch(manager, commandBuffer, pGeneratedCommandsInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdExecuteGeneratedCommandsNV(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    isPreprocessed,
    const VkGeneratedCommandsInfoNV*            pGeneratedCommandsInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdExecuteGeneratedCommandsNV>::Dispatch(manager, commandBuffer, isPreprocessed, pGeneratedCommandsInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdExecuteGeneratedCommandsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(isPreprocessed);
        EncodeStructPtr(encoder, pGeneratedCommandsInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdExecuteGeneratedCommandsNVHandles, pGeneratedCommandsInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkGeneratedCommandsInfoNV* pGeneratedCommandsInfo_unwrapped = UnwrapStructPtrHandles(pGeneratedCommandsInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdExecuteGeneratedCommandsNV(commandBuffer, isPreprocessed, pGeneratedCommandsInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdExecuteGeneratedCommandsNV>::Dispatch(manager, commandBuffer, isPreprocessed, pGeneratedCommandsInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdBindPipelineShaderGroupNV(
    VkCommandBuffer                             commandBuffer,
    VkPipelineBindPoint                         pipelineBindPoint,
    VkPipeline                                  pipeline,
    uint32_t                                    groupIndex)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindPipelineShaderGroupNV>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline, groupIndex);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindPipelineShaderGroupNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineBindPoint);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        encoder->EncodeUInt32Value(groupIndex);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindPipelineShaderGroupNVHandles, pipeline);
    }

    GetDeviceTable(commandBuffer)->CmdBindPipelineShaderGroupNV(commandBuffer, pipelineBindPoint, pipeline, groupIndex);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindPipelineShaderGroupNV>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline, groupIndex);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateIndirectCommandsLayoutNV(
    VkDevice                                    device,
    const VkIndirectCommandsLayoutCreateInfoNV* pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkIndirectCommandsLayoutNV*                 pIndirectCommandsLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateIndirectCommandsLayoutNV>::Dispatch(manager, device, pCreateInfo, pAllocator, pIndirectCommandsLayout);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkIndirectCommandsLayoutCreateInfoNV* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateIndirectCommandsLayoutNV(device, pCreateInfo_unwrapped, pAllocator, pIndirectCommandsLayout);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, IndirectCommandsLayoutNVWrapper>(device, NoParentWrapper::kHandleValue, pIndirectCommandsLayout, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateIndirectCommandsLayoutNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<IndirectCommandsLayoutNVWrapper>(pIndirectCommandsLayout, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, IndirectCommandsLayoutNVWrapper, VkIndirectCommandsLayoutCreateInfoNV>(result, device, pIndirectCommandsLayout, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateIndirectCommandsLayoutNV>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pIndirectCommandsLayout);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyIndirectCommandsLayoutNV(
    VkDevice                                    device,
    VkIndirectCommandsLayoutNV                  indirectCommandsLayout,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyIndirectCommandsLayoutNV>::Dispatch(manager, device, indirectCommandsLayout, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyIndirectCommandsLayoutNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<IndirectCommandsLayoutNVWrapper>(indirectCommandsLayout);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<IndirectCommandsLayoutNVWrapper>(indirectCommandsLayout);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyIndirectCommandsLayoutNV(device, indirectCommandsLayout, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyIndirectCommandsLayoutNV>::Dispatch(manager, device, indirectCommandsLayout, pAllocator);

    DestroyWrappedHandle<IndirectCommandsLayoutNVWrapper>(indirectCommandsLayout);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBias2EXT(
    VkCommandBuffer                             commandBuffer,
    const VkDepthBiasInfoEXT*                   pDepthBiasInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBias2EXT>::Dispatch(manager, commandBuffer, pDepthBiasInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBias2EXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pDepthBiasInfo);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBias2EXT(commandBuffer, pDepthBiasInfo);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBias2EXT>::Dispatch(manager, commandBuffer, pDepthBiasInfo);
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireDrmDisplayEXT(
    VkPhysicalDevice                            physicalDevice,
    int32_t                                     drmFd,
    VkDisplayKHR                                display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireDrmDisplayEXT>::Dispatch(manager, physicalDevice, drmFd, display);

    VkResult result = GetInstanceTable(physicalDevice)->AcquireDrmDisplayEXT(physicalDevice, drmFd, display);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireDrmDisplayEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeInt32Value(drmFd);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireDrmDisplayEXT>::Dispatch(manager, result, physicalDevice, drmFd, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetDrmDisplayEXT(
    VkPhysicalDevice                            physicalDevice,
    int32_t                                     drmFd,
    uint32_t                                    connectorId,
    VkDisplayKHR*                               display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDrmDisplayEXT>::Dispatch(manager, physicalDevice, drmFd, connectorId, display);

    VkResult result = GetInstanceTable(physicalDevice)->GetDrmDisplayEXT(physicalDevice, drmFd, connectorId, display);

    if (result >= 0)
    {
        CreateWrappedHandle<PhysicalDeviceWrapper, NoParentWrapper, DisplayKHRWrapper>(physicalDevice, NoParentWrapper::kHandleValue, display, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetDrmDisplayEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeInt32Value(drmFd);
        encoder->EncodeUInt32Value(connectorId);
        encoder->EncodeHandlePtr<DisplayKHRWrapper>(display, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, void>(result, physicalDevice, display, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDrmDisplayEXT>::Dispatch(manager, result, physicalDevice, drmFd, connectorId, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreatePrivateDataSlotEXT(
    VkDevice                                    device,
    const VkPrivateDataSlotCreateInfo*          pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkPrivateDataSlot*                          pPrivateDataSlot)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreatePrivateDataSlotEXT>::Dispatch(manager, device, pCreateInfo, pAllocator, pPrivateDataSlot);

    VkResult result = GetDeviceTable(device)->CreatePrivateDataSlotEXT(device, pCreateInfo, pAllocator, pPrivateDataSlot);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, PrivateDataSlotWrapper>(device, NoParentWrapper::kHandleValue, pPrivateDataSlot, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreatePrivateDataSlotEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<PrivateDataSlotWrapper>(pPrivateDataSlot, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, PrivateDataSlotWrapper, VkPrivateDataSlotCreateInfo>(result, device, pPrivateDataSlot, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreatePrivateDataSlotEXT>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pPrivateDataSlot);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyPrivateDataSlotEXT(
    VkDevice                                    device,
    VkPrivateDataSlot                           privateDataSlot,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyPrivateDataSlotEXT>::Dispatch(manager, device, privateDataSlot, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyPrivateDataSlotEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<PrivateDataSlotWrapper>(privateDataSlot);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyPrivateDataSlotEXT(device, privateDataSlot, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyPrivateDataSlotEXT>::Dispatch(manager, device, privateDataSlot, pAllocator);

    DestroyWrappedHandle<PrivateDataSlotWrapper>(privateDataSlot);
}

VKAPI_ATTR VkResult VKAPI_CALL SetPrivateDataEXT(
    VkDevice                                    device,
    VkObjectType                                objectType,
    uint64_t                                    objectHandle,
    VkPrivateDataSlot                           privateDataSlot,
    uint64_t                                    data)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetPrivateDataEXT>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, data);

    VkResult result = GetDeviceTable(device)->SetPrivateDataEXT(device, objectType, objectHandle, privateDataSlot, data);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetPrivateDataEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(objectType);
        encoder->EncodeUInt64Value(GetWrappedId(objectHandle, objectType));
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        encoder->EncodeUInt64Value(data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetPrivateDataEXT>::Dispatch(manager, result, device, objectType, objectHandle, privateDataSlot, data);

    return result;
}

VKAPI_ATTR void VKAPI_CALL GetPrivateDataEXT(
    VkDevice                                    device,
    VkObjectType                                objectType,
    uint64_t                                    objectHandle,
    VkPrivateDataSlot                           privateDataSlot,
    uint64_t*                                   pData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPrivateDataEXT>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, pData);

    GetDeviceTable(device)->GetPrivateDataEXT(device, objectType, objectHandle, privateDataSlot, pData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPrivateDataEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(objectType);
        encoder->EncodeUInt64Value(GetWrappedId(objectHandle, objectType));
        encoder->EncodeHandleValue<PrivateDataSlotWrapper>(privateDataSlot);
        encoder->EncodeUInt64Ptr(pData);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPrivateDataEXT>::Dispatch(manager, device, objectType, objectHandle, privateDataSlot, pData);
}

VKAPI_ATTR void VKAPI_CALL CmdSetFragmentShadingRateEnumNV(
    VkCommandBuffer                             commandBuffer,
    VkFragmentShadingRateNV                     shadingRate,
    const VkFragmentShadingRateCombinerOpKHR    combinerOps[2])
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateEnumNV>::Dispatch(manager, commandBuffer, shadingRate, combinerOps);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateEnumNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(shadingRate);
        encoder->EncodeEnumArray(combinerOps, 2);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetFragmentShadingRateEnumNV(commandBuffer, shadingRate, combinerOps);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetFragmentShadingRateEnumNV>::Dispatch(manager, commandBuffer, shadingRate, combinerOps);
}

VKAPI_ATTR VkResult VKAPI_CALL GetDeviceFaultInfoEXT(
    VkDevice                                    device,
    VkDeviceFaultCountsEXT*                     pFaultCounts,
    VkDeviceFaultInfoEXT*                       pFaultInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceFaultInfoEXT>::Dispatch(manager, device, pFaultCounts, pFaultInfo);

    VkResult result = GetDeviceTable(device)->GetDeviceFaultInfoEXT(device, pFaultCounts, pFaultInfo);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceFaultInfoEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pFaultCounts, omit_output_data);
        EncodeStructPtr(encoder, pFaultInfo, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceFaultInfoEXT>::Dispatch(manager, result, device, pFaultCounts, pFaultInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL AcquireWinrtDisplayNV(
    VkPhysicalDevice                            physicalDevice,
    VkDisplayKHR                                display)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkAcquireWinrtDisplayNV>::Dispatch(manager, physicalDevice, display);

    VkResult result = GetInstanceTable(physicalDevice)->AcquireWinrtDisplayNV(physicalDevice, display);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkAcquireWinrtDisplayNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeHandleValue<DisplayKHRWrapper>(display);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkAcquireWinrtDisplayNV>::Dispatch(manager, result, physicalDevice, display);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetWinrtDisplayNV(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    deviceRelativeId,
    VkDisplayKHR*                               pDisplay)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetWinrtDisplayNV>::Dispatch(manager, physicalDevice, deviceRelativeId, pDisplay);

    VkResult result = GetInstanceTable(physicalDevice)->GetWinrtDisplayNV(physicalDevice, deviceRelativeId, pDisplay);

    if (result >= 0)
    {
        CreateWrappedHandle<PhysicalDeviceWrapper, NoParentWrapper, DisplayKHRWrapper>(physicalDevice, NoParentWrapper::kHandleValue, pDisplay, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkGetWinrtDisplayNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(deviceRelativeId);
        encoder->EncodeHandlePtr<DisplayKHRWrapper>(pDisplay, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkPhysicalDevice, DisplayKHRWrapper, void>(result, physicalDevice, pDisplay, nullptr);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetWinrtDisplayNV>::Dispatch(manager, result, physicalDevice, deviceRelativeId, pDisplay);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CreateDirectFBSurfaceEXT(
    VkInstance                                  instance,
    const VkDirectFBSurfaceCreateInfoEXT*       pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateDirectFBSurfaceEXT>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateDirectFBSurfaceEXT(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateDirectFBSurfaceEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkDirectFBSurfaceCreateInfoEXT>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateDirectFBSurfaceEXT>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceDirectFBPresentationSupportEXT(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    IDirectFB*                                  dfb)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDirectFBPresentationSupportEXT>::Dispatch(manager, physicalDevice, queueFamilyIndex, dfb);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceDirectFBPresentationSupportEXT(physicalDevice, queueFamilyIndex, dfb);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceDirectFBPresentationSupportEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVoidPtr(dfb);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceDirectFBPresentationSupportEXT>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, dfb);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetVertexInputEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    vertexBindingDescriptionCount,
    const VkVertexInputBindingDescription2EXT*  pVertexBindingDescriptions,
    uint32_t                                    vertexAttributeDescriptionCount,
    const VkVertexInputAttributeDescription2EXT* pVertexAttributeDescriptions)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetVertexInputEXT>::Dispatch(manager, commandBuffer, vertexBindingDescriptionCount, pVertexBindingDescriptions, vertexAttributeDescriptionCount, pVertexAttributeDescriptions);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetVertexInputEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(vertexBindingDescriptionCount);
        EncodeStructArray(encoder, pVertexBindingDescriptions, vertexBindingDescriptionCount);
        encoder->EncodeUInt32Value(vertexAttributeDescriptionCount);
        EncodeStructArray(encoder, pVertexAttributeDescriptions, vertexAttributeDescriptionCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetVertexInputEXT(commandBuffer, vertexBindingDescriptionCount, pVertexBindingDescriptions, vertexAttributeDescriptionCount, pVertexAttributeDescriptions);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetVertexInputEXT>::Dispatch(manager, commandBuffer, vertexBindingDescriptionCount, pVertexBindingDescriptions, vertexAttributeDescriptionCount, pVertexAttributeDescriptions);
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryZirconHandleFUCHSIA(
    VkDevice                                    device,
    const VkMemoryGetZirconHandleInfoFUCHSIA*   pGetZirconHandleInfo,
    zx_handle_t*                                pZirconHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryZirconHandleFUCHSIA>::Dispatch(manager, device, pGetZirconHandleInfo, pZirconHandle);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryGetZirconHandleInfoFUCHSIA* pGetZirconHandleInfo_unwrapped = UnwrapStructPtrHandles(pGetZirconHandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetMemoryZirconHandleFUCHSIA(device, pGetZirconHandleInfo_unwrapped, pZirconHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryZirconHandleFUCHSIA);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetZirconHandleInfo);
        encoder->EncodeUInt32Ptr(pZirconHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryZirconHandleFUCHSIA>::Dispatch(manager, result, device, pGetZirconHandleInfo, pZirconHandle);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryZirconHandlePropertiesFUCHSIA(
    VkDevice                                    device,
    VkExternalMemoryHandleTypeFlagBits          handleType,
    zx_handle_t                                 zirconHandle,
    VkMemoryZirconHandlePropertiesFUCHSIA*      pMemoryZirconHandleProperties)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryZirconHandlePropertiesFUCHSIA>::Dispatch(manager, device, handleType, zirconHandle, pMemoryZirconHandleProperties);

    VkResult result = GetDeviceTable(device)->GetMemoryZirconHandlePropertiesFUCHSIA(device, handleType, zirconHandle, pMemoryZirconHandleProperties);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryZirconHandlePropertiesFUCHSIA);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(handleType);
        encoder->EncodeUInt32Value(zirconHandle);
        EncodeStructPtr(encoder, pMemoryZirconHandleProperties, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryZirconHandlePropertiesFUCHSIA>::Dispatch(manager, result, device, handleType, zirconHandle, pMemoryZirconHandleProperties);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL ImportSemaphoreZirconHandleFUCHSIA(
    VkDevice                                    device,
    const VkImportSemaphoreZirconHandleInfoFUCHSIA* pImportSemaphoreZirconHandleInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkImportSemaphoreZirconHandleFUCHSIA>::Dispatch(manager, device, pImportSemaphoreZirconHandleInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkImportSemaphoreZirconHandleInfoFUCHSIA* pImportSemaphoreZirconHandleInfo_unwrapped = UnwrapStructPtrHandles(pImportSemaphoreZirconHandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->ImportSemaphoreZirconHandleFUCHSIA(device, pImportSemaphoreZirconHandleInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkImportSemaphoreZirconHandleFUCHSIA);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pImportSemaphoreZirconHandleInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkImportSemaphoreZirconHandleFUCHSIA>::Dispatch(manager, result, device, pImportSemaphoreZirconHandleInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL GetSemaphoreZirconHandleFUCHSIA(
    VkDevice                                    device,
    const VkSemaphoreGetZirconHandleInfoFUCHSIA* pGetZirconHandleInfo,
    zx_handle_t*                                pZirconHandle)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetSemaphoreZirconHandleFUCHSIA>::Dispatch(manager, device, pGetZirconHandleInfo, pZirconHandle);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkSemaphoreGetZirconHandleInfoFUCHSIA* pGetZirconHandleInfo_unwrapped = UnwrapStructPtrHandles(pGetZirconHandleInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetSemaphoreZirconHandleFUCHSIA(device, pGetZirconHandleInfo_unwrapped, pZirconHandle);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetSemaphoreZirconHandleFUCHSIA);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pGetZirconHandleInfo);
        encoder->EncodeUInt32Ptr(pZirconHandle, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetSemaphoreZirconHandleFUCHSIA>::Dispatch(manager, result, device, pGetZirconHandleInfo, pZirconHandle);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdBindInvocationMaskHUAWEI(
    VkCommandBuffer                             commandBuffer,
    VkImageView                                 imageView,
    VkImageLayout                               imageLayout)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBindInvocationMaskHUAWEI>::Dispatch(manager, commandBuffer, imageView, imageLayout);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBindInvocationMaskHUAWEI);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<ImageViewWrapper>(imageView);
        encoder->EncodeEnumValue(imageLayout);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBindInvocationMaskHUAWEIHandles, imageView);
    }

    GetDeviceTable(commandBuffer)->CmdBindInvocationMaskHUAWEI(commandBuffer, imageView, imageLayout);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBindInvocationMaskHUAWEI>::Dispatch(manager, commandBuffer, imageView, imageLayout);
}

VKAPI_ATTR VkResult VKAPI_CALL GetMemoryRemoteAddressNV(
    VkDevice                                    device,
    const VkMemoryGetRemoteAddressInfoNV*       pMemoryGetRemoteAddressInfo,
    VkRemoteAddressNV*                          pAddress)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMemoryRemoteAddressNV>::Dispatch(manager, device, pMemoryGetRemoteAddressInfo, pAddress);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMemoryGetRemoteAddressInfoNV* pMemoryGetRemoteAddressInfo_unwrapped = UnwrapStructPtrHandles(pMemoryGetRemoteAddressInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->GetMemoryRemoteAddressNV(device, pMemoryGetRemoteAddressInfo_unwrapped, pAddress);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMemoryRemoteAddressNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pMemoryGetRemoteAddressInfo);
        encoder->EncodeVoidPtrPtr(pAddress, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMemoryRemoteAddressNV>::Dispatch(manager, result, device, pMemoryGetRemoteAddressInfo, pAddress);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdSetPatchControlPointsEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    patchControlPoints)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPatchControlPointsEXT>::Dispatch(manager, commandBuffer, patchControlPoints);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPatchControlPointsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(patchControlPoints);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetPatchControlPointsEXT(commandBuffer, patchControlPoints);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPatchControlPointsEXT>::Dispatch(manager, commandBuffer, patchControlPoints);
}

VKAPI_ATTR void VKAPI_CALL CmdSetRasterizerDiscardEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    rasterizerDiscardEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnableEXT>::Dispatch(manager, commandBuffer, rasterizerDiscardEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(rasterizerDiscardEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetRasterizerDiscardEnableEXT(commandBuffer, rasterizerDiscardEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetRasterizerDiscardEnableEXT>::Dispatch(manager, commandBuffer, rasterizerDiscardEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetDepthBiasEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    depthBiasEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnableEXT>::Dispatch(manager, commandBuffer, depthBiasEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(depthBiasEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetDepthBiasEnableEXT(commandBuffer, depthBiasEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetDepthBiasEnableEXT>::Dispatch(manager, commandBuffer, depthBiasEnable);
}

VKAPI_ATTR void VKAPI_CALL CmdSetLogicOpEXT(
    VkCommandBuffer                             commandBuffer,
    VkLogicOp                                   logicOp)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetLogicOpEXT>::Dispatch(manager, commandBuffer, logicOp);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetLogicOpEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(logicOp);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetLogicOpEXT(commandBuffer, logicOp);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetLogicOpEXT>::Dispatch(manager, commandBuffer, logicOp);
}

VKAPI_ATTR void VKAPI_CALL CmdSetPrimitiveRestartEnableEXT(
    VkCommandBuffer                             commandBuffer,
    VkBool32                                    primitiveRestartEnable)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnableEXT>::Dispatch(manager, commandBuffer, primitiveRestartEnable);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeVkBool32Value(primitiveRestartEnable);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetPrimitiveRestartEnableEXT(commandBuffer, primitiveRestartEnable);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetPrimitiveRestartEnableEXT>::Dispatch(manager, commandBuffer, primitiveRestartEnable);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateScreenSurfaceQNX(
    VkInstance                                  instance,
    const VkScreenSurfaceCreateInfoQNX*         pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkSurfaceKHR*                               pSurface)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateScreenSurfaceQNX>::Dispatch(manager, instance, pCreateInfo, pAllocator, pSurface);

    VkResult result = GetInstanceTable(instance)->CreateScreenSurfaceQNX(instance, pCreateInfo, pAllocator, pSurface);

    if (result >= 0)
    {
        CreateWrappedHandle<InstanceWrapper, NoParentWrapper, SurfaceKHRWrapper>(instance, NoParentWrapper::kHandleValue, pSurface, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateScreenSurfaceQNX);
    if (encoder)
    {
        encoder->EncodeHandleValue<InstanceWrapper>(instance);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<SurfaceKHRWrapper>(pSurface, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkInstance, SurfaceKHRWrapper, VkScreenSurfaceCreateInfoQNX>(result, instance, pSurface, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateScreenSurfaceQNX>::Dispatch(manager, result, instance, pCreateInfo, pAllocator, pSurface);

    return result;
}

VKAPI_ATTR VkBool32 VKAPI_CALL GetPhysicalDeviceScreenPresentationSupportQNX(
    VkPhysicalDevice                            physicalDevice,
    uint32_t                                    queueFamilyIndex,
    struct _screen_window*                      window)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceScreenPresentationSupportQNX>::Dispatch(manager, physicalDevice, queueFamilyIndex, window);

    VkBool32 result = GetInstanceTable(physicalDevice)->GetPhysicalDeviceScreenPresentationSupportQNX(physicalDevice, queueFamilyIndex, window);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPhysicalDeviceScreenPresentationSupportQNX);
    if (encoder)
    {
        encoder->EncodeHandleValue<PhysicalDeviceWrapper>(physicalDevice);
        encoder->EncodeUInt32Value(queueFamilyIndex);
        encoder->EncodeVoidPtr(window);
        encoder->EncodeVkBool32Value(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPhysicalDeviceScreenPresentationSupportQNX>::Dispatch(manager, result, physicalDevice, queueFamilyIndex, window);

    return result;
}

VKAPI_ATTR void                                    VKAPI_CALL CmdSetColorWriteEnableEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    attachmentCount,
    const VkBool32*                             pColorWriteEnables)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdSetColorWriteEnableEXT>::Dispatch(manager, commandBuffer, attachmentCount, pColorWriteEnables);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdSetColorWriteEnableEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(attachmentCount);
        encoder->EncodeVkBool32Array(pColorWriteEnables, attachmentCount);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdSetColorWriteEnableEXT(commandBuffer, attachmentCount, pColorWriteEnables);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdSetColorWriteEnableEXT>::Dispatch(manager, commandBuffer, attachmentCount, pColorWriteEnables);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMultiEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    drawCount,
    const VkMultiDrawInfoEXT*                   pVertexInfo,
    uint32_t                                    instanceCount,
    uint32_t                                    firstInstance,
    uint32_t                                    stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawMultiEXT>::Dispatch(manager, commandBuffer, drawCount, pVertexInfo, instanceCount, firstInstance, stride);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawMultiEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(drawCount);
        EncodeStructArray(encoder, pVertexInfo, drawCount);
        encoder->EncodeUInt32Value(instanceCount);
        encoder->EncodeUInt32Value(firstInstance);
        encoder->EncodeUInt32Value(stride);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawMultiEXT(commandBuffer, drawCount, pVertexInfo, instanceCount, firstInstance, stride);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawMultiEXT>::Dispatch(manager, commandBuffer, drawCount, pVertexInfo, instanceCount, firstInstance, stride);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawMultiIndexedEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    drawCount,
    const VkMultiDrawIndexedInfoEXT*            pIndexInfo,
    uint32_t                                    instanceCount,
    uint32_t                                    firstInstance,
    uint32_t                                    stride,
    const int32_t*                              pVertexOffset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawMultiIndexedEXT>::Dispatch(manager, commandBuffer, drawCount, pIndexInfo, instanceCount, firstInstance, stride, pVertexOffset);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawMultiIndexedEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(drawCount);
        EncodeStructArray(encoder, pIndexInfo, drawCount);
        encoder->EncodeUInt32Value(instanceCount);
        encoder->EncodeUInt32Value(firstInstance);
        encoder->EncodeUInt32Value(stride);
        encoder->EncodeInt32Ptr(pVertexOffset);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawMultiIndexedEXT(commandBuffer, drawCount, pIndexInfo, instanceCount, firstInstance, stride, pVertexOffset);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawMultiIndexedEXT>::Dispatch(manager, commandBuffer, drawCount, pIndexInfo, instanceCount, firstInstance, stride, pVertexOffset);
}

VKAPI_ATTR VkResult VKAPI_CALL CreateMicromapEXT(
    VkDevice                                    device,
    const VkMicromapCreateInfoEXT*              pCreateInfo,
    const VkAllocationCallbacks*                pAllocator,
    VkMicromapEXT*                              pMicromap)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCreateMicromapEXT>::Dispatch(manager, device, pCreateInfo, pAllocator, pMicromap);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMicromapCreateInfoEXT* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CreateMicromapEXT(device, pCreateInfo_unwrapped, pAllocator, pMicromap);

    if (result >= 0)
    {
        CreateWrappedHandle<DeviceWrapper, NoParentWrapper, MicromapEXTWrapper>(device, NoParentWrapper::kHandleValue, pMicromap, VulkanCaptureManager::GetUniqueId);
    }
    else
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCreateMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pAllocator);
        encoder->EncodeHandlePtr<MicromapEXTWrapper>(pMicromap, omit_output_data);
        encoder->EncodeEnumValue(result);
        manager->EndCreateApiCallCapture<VkDevice, MicromapEXTWrapper, VkMicromapCreateInfoEXT>(result, device, pMicromap, pCreateInfo);
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCreateMicromapEXT>::Dispatch(manager, result, device, pCreateInfo, pAllocator, pMicromap);

    return result;
}

VKAPI_ATTR void VKAPI_CALL DestroyMicromapEXT(
    VkDevice                                    device,
    VkMicromapEXT                               micromap,
    const VkAllocationCallbacks*                pAllocator)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkDestroyMicromapEXT>::Dispatch(manager, device, micromap, pAllocator);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkDestroyMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<MicromapEXTWrapper>(micromap);
        EncodeStructPtr(encoder, pAllocator);
        manager->EndDestroyApiCallCapture<MicromapEXTWrapper>(micromap);
    }

    ScopedDestroyLock exclusive_scoped_lock;
    GetDeviceTable(device)->DestroyMicromapEXT(device, micromap, pAllocator);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkDestroyMicromapEXT>::Dispatch(manager, device, micromap, pAllocator);

    DestroyWrappedHandle<MicromapEXTWrapper>(micromap);
}

VKAPI_ATTR void VKAPI_CALL CmdBuildMicromapsEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    infoCount,
    const VkMicromapBuildInfoEXT*               pInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdBuildMicromapsEXT>::Dispatch(manager, commandBuffer, infoCount, pInfos);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdBuildMicromapsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(infoCount);
        EncodeStructArray(encoder, pInfos, infoCount);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdBuildMicromapsEXTHandles, infoCount, pInfos);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMicromapBuildInfoEXT* pInfos_unwrapped = UnwrapStructArrayHandles(pInfos, infoCount, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdBuildMicromapsEXT(commandBuffer, infoCount, pInfos_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdBuildMicromapsEXT>::Dispatch(manager, commandBuffer, infoCount, pInfos);
}

VKAPI_ATTR VkResult VKAPI_CALL BuildMicromapsEXT(
    VkDevice                                    device,
    VkDeferredOperationKHR                      deferredOperation,
    uint32_t                                    infoCount,
    const VkMicromapBuildInfoEXT*               pInfos)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkBuildMicromapsEXT>::Dispatch(manager, device, deferredOperation, infoCount, pInfos);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMicromapBuildInfoEXT* pInfos_unwrapped = UnwrapStructArrayHandles(pInfos, infoCount, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->BuildMicromapsEXT(device, deferredOperation, infoCount, pInfos_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkBuildMicromapsEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(deferredOperation);
        encoder->EncodeUInt32Value(infoCount);
        EncodeStructArray(encoder, pInfos, infoCount);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkBuildMicromapsEXT>::Dispatch(manager, result, device, deferredOperation, infoCount, pInfos);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CopyMicromapEXT(
    VkDevice                                    device,
    VkDeferredOperationKHR                      deferredOperation,
    const VkCopyMicromapInfoEXT*                pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyMicromapEXT>::Dispatch(manager, device, deferredOperation, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMicromapInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyMicromapEXT(device, deferredOperation, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(deferredOperation);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyMicromapEXT>::Dispatch(manager, result, device, deferredOperation, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CopyMicromapToMemoryEXT(
    VkDevice                                    device,
    VkDeferredOperationKHR                      deferredOperation,
    const VkCopyMicromapToMemoryInfoEXT*        pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyMicromapToMemoryEXT>::Dispatch(manager, device, deferredOperation, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMicromapToMemoryInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyMicromapToMemoryEXT(device, deferredOperation, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyMicromapToMemoryEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(deferredOperation);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyMicromapToMemoryEXT>::Dispatch(manager, result, device, deferredOperation, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL CopyMemoryToMicromapEXT(
    VkDevice                                    device,
    VkDeferredOperationKHR                      deferredOperation,
    const VkCopyMemoryToMicromapInfoEXT*        pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCopyMemoryToMicromapEXT>::Dispatch(manager, device, deferredOperation, pInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMemoryToMicromapInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    VkResult result = GetDeviceTable(device)->CopyMemoryToMicromapEXT(device, deferredOperation, pInfo_unwrapped);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkCopyMemoryToMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeferredOperationKHRWrapper>(deferredOperation);
        EncodeStructPtr(encoder, pInfo);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCopyMemoryToMicromapEXT>::Dispatch(manager, result, device, deferredOperation, pInfo);

    return result;
}

VKAPI_ATTR VkResult VKAPI_CALL WriteMicromapsPropertiesEXT(
    VkDevice                                    device,
    uint32_t                                    micromapCount,
    const VkMicromapEXT*                        pMicromaps,
    VkQueryType                                 queryType,
    size_t                                      dataSize,
    void*                                       pData,
    size_t                                      stride)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    bool omit_output_data = false;

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkWriteMicromapsPropertiesEXT>::Dispatch(manager, device, micromapCount, pMicromaps, queryType, dataSize, pData, stride);

    VkResult result = GetDeviceTable(device)->WriteMicromapsPropertiesEXT(device, micromapCount, pMicromaps, queryType, dataSize, pData, stride);
    if (result < 0)
    {
        omit_output_data = true;
    }

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkWriteMicromapsPropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeUInt32Value(micromapCount);
        encoder->EncodeHandleArray<MicromapEXTWrapper>(pMicromaps, micromapCount);
        encoder->EncodeEnumValue(queryType);
        encoder->EncodeSizeTValue(dataSize);
        encoder->EncodeVoidArray(pData, dataSize, omit_output_data);
        encoder->EncodeSizeTValue(stride);
        encoder->EncodeEnumValue(result);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkWriteMicromapsPropertiesEXT>::Dispatch(manager, result, device, micromapCount, pMicromaps, queryType, dataSize, pData, stride);

    return result;
}

VKAPI_ATTR void VKAPI_CALL CmdCopyMicromapEXT(
    VkCommandBuffer                             commandBuffer,
    const VkCopyMicromapInfoEXT*                pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyMicromapEXT>::Dispatch(manager, commandBuffer, pInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyMicromapEXTHandles, pInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMicromapInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyMicromapEXT(commandBuffer, pInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyMicromapEXT>::Dispatch(manager, commandBuffer, pInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyMicromapToMemoryEXT(
    VkCommandBuffer                             commandBuffer,
    const VkCopyMicromapToMemoryInfoEXT*        pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyMicromapToMemoryEXT>::Dispatch(manager, commandBuffer, pInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyMicromapToMemoryEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyMicromapToMemoryEXTHandles, pInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMicromapToMemoryInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyMicromapToMemoryEXT(commandBuffer, pInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyMicromapToMemoryEXT>::Dispatch(manager, commandBuffer, pInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdCopyMemoryToMicromapEXT(
    VkCommandBuffer                             commandBuffer,
    const VkCopyMemoryToMicromapInfoEXT*        pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdCopyMemoryToMicromapEXT>::Dispatch(manager, commandBuffer, pInfo);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdCopyMemoryToMicromapEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        EncodeStructPtr(encoder, pInfo);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdCopyMemoryToMicromapEXTHandles, pInfo);
    }

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkCopyMemoryToMicromapInfoEXT* pInfo_unwrapped = UnwrapStructPtrHandles(pInfo, handle_unwrap_memory);

    GetDeviceTable(commandBuffer)->CmdCopyMemoryToMicromapEXT(commandBuffer, pInfo_unwrapped);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdCopyMemoryToMicromapEXT>::Dispatch(manager, commandBuffer, pInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdWriteMicromapsPropertiesEXT(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    micromapCount,
    const VkMicromapEXT*                        pMicromaps,
    VkQueryType                                 queryType,
    VkQueryPool                                 queryPool,
    uint32_t                                    firstQuery)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdWriteMicromapsPropertiesEXT>::Dispatch(manager, commandBuffer, micromapCount, pMicromaps, queryType, queryPool, firstQuery);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdWriteMicromapsPropertiesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(micromapCount);
        encoder->EncodeHandleArray<MicromapEXTWrapper>(pMicromaps, micromapCount);
        encoder->EncodeEnumValue(queryType);
        encoder->EncodeHandleValue<QueryPoolWrapper>(queryPool);
        encoder->EncodeUInt32Value(firstQuery);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdWriteMicromapsPropertiesEXTHandles, micromapCount, pMicromaps, queryPool);
    }

    GetDeviceTable(commandBuffer)->CmdWriteMicromapsPropertiesEXT(commandBuffer, micromapCount, pMicromaps, queryType, queryPool, firstQuery);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdWriteMicromapsPropertiesEXT>::Dispatch(manager, commandBuffer, micromapCount, pMicromaps, queryType, queryPool, firstQuery);
}

VKAPI_ATTR void VKAPI_CALL GetDeviceMicromapCompatibilityEXT(
    VkDevice                                    device,
    const VkMicromapVersionInfoEXT*             pVersionInfo,
    VkAccelerationStructureCompatibilityKHR*    pCompatibility)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDeviceMicromapCompatibilityEXT>::Dispatch(manager, device, pVersionInfo, pCompatibility);

    GetDeviceTable(device)->GetDeviceMicromapCompatibilityEXT(device, pVersionInfo, pCompatibility);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDeviceMicromapCompatibilityEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pVersionInfo);
        encoder->EncodeEnumPtr(pCompatibility);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDeviceMicromapCompatibilityEXT>::Dispatch(manager, device, pVersionInfo, pCompatibility);
}

VKAPI_ATTR void VKAPI_CALL GetMicromapBuildSizesEXT(
    VkDevice                                    device,
    VkAccelerationStructureBuildTypeKHR         buildType,
    const VkMicromapBuildInfoEXT*               pBuildInfo,
    VkMicromapBuildSizesInfoEXT*                pSizeInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetMicromapBuildSizesEXT>::Dispatch(manager, device, buildType, pBuildInfo, pSizeInfo);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkMicromapBuildInfoEXT* pBuildInfo_unwrapped = UnwrapStructPtrHandles(pBuildInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetMicromapBuildSizesEXT(device, buildType, pBuildInfo_unwrapped, pSizeInfo);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetMicromapBuildSizesEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeEnumValue(buildType);
        EncodeStructPtr(encoder, pBuildInfo);
        EncodeStructPtr(encoder, pSizeInfo);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetMicromapBuildSizesEXT>::Dispatch(manager, device, buildType, pBuildInfo, pSizeInfo);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawClusterHUAWEI(
    VkCommandBuffer                             commandBuffer,
    uint32_t                                    groupCountX,
    uint32_t                                    groupCountY,
    uint32_t                                    groupCountZ)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawClusterHUAWEI>::Dispatch(manager, commandBuffer, groupCountX, groupCountY, groupCountZ);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawClusterHUAWEI);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeUInt32Value(groupCountX);
        encoder->EncodeUInt32Value(groupCountY);
        encoder->EncodeUInt32Value(groupCountZ);
        manager->EndCommandApiCallCapture(commandBuffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawClusterHUAWEI(commandBuffer, groupCountX, groupCountY, groupCountZ);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawClusterHUAWEI>::Dispatch(manager, commandBuffer, groupCountX, groupCountY, groupCountZ);
}

VKAPI_ATTR void VKAPI_CALL CmdDrawClusterIndirectHUAWEI(
    VkCommandBuffer                             commandBuffer,
    VkBuffer                                    buffer,
    VkDeviceSize                                offset)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdDrawClusterIndirectHUAWEI>::Dispatch(manager, commandBuffer, buffer, offset);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdDrawClusterIndirectHUAWEI);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeHandleValue<BufferWrapper>(buffer);
        encoder->EncodeVkDeviceSizeValue(offset);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdDrawClusterIndirectHUAWEIHandles, buffer);
    }

    GetDeviceTable(commandBuffer)->CmdDrawClusterIndirectHUAWEI(commandBuffer, buffer, offset);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdDrawClusterIndirectHUAWEI>::Dispatch(manager, commandBuffer, buffer, offset);
}

VKAPI_ATTR void VKAPI_CALL SetDeviceMemoryPriorityEXT(
    VkDevice                                    device,
    VkDeviceMemory                              memory,
    float                                       priority)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkSetDeviceMemoryPriorityEXT>::Dispatch(manager, device, memory, priority);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkSetDeviceMemoryPriorityEXT);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DeviceMemoryWrapper>(memory);
        encoder->EncodeFloatValue(priority);
        manager->EndApiCallCapture();
    }

    GetDeviceTable(device)->SetDeviceMemoryPriorityEXT(device, memory, priority);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkSetDeviceMemoryPriorityEXT>::Dispatch(manager, device, memory, priority);
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetLayoutHostMappingInfoVALVE(
    VkDevice                                    device,
    const VkDescriptorSetBindingReferenceVALVE* pBindingReference,
    VkDescriptorSetLayoutHostMappingInfoVALVE*  pHostMapping)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutHostMappingInfoVALVE>::Dispatch(manager, device, pBindingReference, pHostMapping);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkDescriptorSetBindingReferenceVALVE* pBindingReference_unwrapped = UnwrapStructPtrHandles(pBindingReference, handle_unwrap_memory);

    GetDeviceTable(device)->GetDescriptorSetLayoutHostMappingInfoVALVE(device, pBindingReference_unwrapped, pHostMapping);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutHostMappingInfoVALVE);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pBindingReference);
        EncodeStructPtr(encoder, pHostMapping);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDescriptorSetLayoutHostMappingInfoVALVE>::Dispatch(manager, device, pBindingReference, pHostMapping);
}

VKAPI_ATTR void VKAPI_CALL GetDescriptorSetHostMappingVALVE(
    VkDevice                                    device,
    VkDescriptorSet                             descriptorSet,
    void**                                      ppData)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetDescriptorSetHostMappingVALVE>::Dispatch(manager, device, descriptorSet, ppData);

    GetDeviceTable(device)->GetDescriptorSetHostMappingVALVE(device, descriptorSet, ppData);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetDescriptorSetHostMappingVALVE);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        encoder->EncodeHandleValue<DescriptorSetWrapper>(descriptorSet);
        encoder->EncodeVoidPtrPtr(ppData);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetDescriptorSetHostMappingVALVE>::Dispatch(manager, device, descriptorSet, ppData);
}

VKAPI_ATTR void VKAPI_CALL GetPipelineIndirectMemoryRequirementsNV(
    VkDevice                                    device,
    const VkComputePipelineCreateInfo*          pCreateInfo,
    VkMemoryRequirements2*                      pMemoryRequirements)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkGetPipelineIndirectMemoryRequirementsNV>::Dispatch(manager, device, pCreateInfo, pMemoryRequirements);

    auto handle_unwrap_memory = manager->GetHandleUnwrapMemory();
    const VkComputePipelineCreateInfo* pCreateInfo_unwrapped = UnwrapStructPtrHandles(pCreateInfo, handle_unwrap_memory);

    GetDeviceTable(device)->GetPipelineIndirectMemoryRequirementsNV(device, pCreateInfo_unwrapped, pMemoryRequirements);

    auto encoder = manager->BeginApiCallCapture(format::ApiCallId::ApiCall_vkGetPipelineIndirectMemoryRequirementsNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<DeviceWrapper>(device);
        EncodeStructPtr(encoder, pCreateInfo);
        EncodeStructPtr(encoder, pMemoryRequirements);
        manager->EndApiCallCapture();
    }

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkGetPipelineIndirectMemoryRequirementsNV>::Dispatch(manager, device, pCreateInfo, pMemoryRequirements);
}

VKAPI_ATTR void VKAPI_CALL CmdUpdatePipelineIndirectBufferNV(
    VkCommandBuffer                             commandBuffer,
    VkPipelineBindPoint                         pipelineBindPoint,
    VkPipeline                                  pipeline)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
    if (force_command_serialization)
    {
        exclusive_api_call_lock = VulkanCaptureManager::AcquireExclusiveApiCallLock();
    }
    else
    {
        shared_api_call_lock = VulkanCaptureManager::AcquireSharedApiCallLock();
    }

    CustomEncoderPreCall<format::ApiCallId::ApiCall_vkCmdUpdatePipelineIndirectBufferNV>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline);

    auto encoder = manager->BeginTrackedApiCallCapture(format::ApiCallId::ApiCall_vkCmdUpdatePipelineIndirectBufferNV);
    if (encoder)
    {
        encoder->EncodeHandleValue<CommandBufferWrapper>(commandBuffer);
        encoder->EncodeEnumValue(pipelineBindPoint);
        encoder->EncodeHandleValue<PipelineWrapper>(pipeline);
        manager->EndCommandApiCallCapture(commandBuffer, TrackCmdUpdatePipelineIndirectBufferNVHandles, pipeline);
    }

    GetDeviceTable(commandBuffer)->CmdUpdatePipelineIndirectBufferNV(commandBuffer, pipelineBindPoint, pipeline);

    CustomEncoderPostCall<format::ApiCallId::ApiCall_vkCmdUpdatePipelineIndirectBufferNV>::Dispatch(manager, commandBuffer, pipelineBindPoint, pipeline);
}

VKAPI_ATTR VkDeviceAddress VKAPI_CALL GetPipelineIndirectDeviceAddressNV(
    VkDevice                                    device,
    const VkPipelineIndirectDeviceAddressInfoNV* pInfo)
{
    VulkanCaptureManager* manager = VulkanCaptureManager::Get();
    GFXRECON_ASSERT(manager != nullptr);
    auto force_command_serialization = manager->GetForceCommandSerialization();
    std::shared_lock<CaptureManager::ApiCallMutexT> shared_api_call_lock;
    std::unique_lock<CaptureManager::ApiCallMutexT> exclusive_api_call_lock;
