/*
  chord.cpp 
  To convert midinotes to chord symbols, or chord symbols to midi.
  Author: Robert Vogel

  This program is free software; you can redistribute it and/or modify
  it under the terms of version 2 of the GNU General Public License
  as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License (version 2) for more details.

  You should have received a copy of the GNU General Public License
 (version2)
  along with this program; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
*/

/* Alsa sequencer functions  by Matthias Nagorni 
   Chord functions by Robert Vogel 
   
   Note that there are two independent routines here:
	one converts a series of midi notes to a chord symbol
		For speed this uses a lot of switches.
	two converts the chord symbol to midi notes.
		using a c++ container.
	
*/
#include <stdio.h>
#include <stdlib.h>
#include <sched.h>
#include <unistd.h>
#include <sys/mman.h>
#include <alsa/asoundlib.h>
#include <FL/fl_message.H>
#include <iostream>
#include <map>
#include <vector>
#include <string>
using namespace std;

// ***********************************************************
#ifndef _RLV_CHORD_H
    #include "Chord.h"
#endif
// *******************
int laa1;
int laa3;
int laa5;
int laa7;

extern CHORD zing;
CHORD seperate;
int keynumber1;
int runningline;

// Print the number and letter equivalent of a numeric midi note.
void
Midi2Note(int MidiNote)
{
	int Octave = MidiNote/12;
	int NoteMod = MidiNote%12;
	std::string Notesym;
	switch(NoteMod)
		{
		case 0:
			Notesym = "C";
			break;
		case 1:
			Notesym = "C#";
			break;
		case 2:
			Notesym = "D";
			break;
		case 3:
			Notesym = "Eb";
			break;
		case 4:
			Notesym = "E";
			break;
		case 5:
			Notesym = "F";
			break;
		case 6:
			Notesym = "F#";
			break;
		case 7:
			Notesym = "G";
			break;
		case 8:
			Notesym = "G#";
			break;
		case 9:
			Notesym = "A";
			break;
		case 10:
			Notesym = "Bb";
			break;
		case 11:
			Notesym = "B";
			break;
		}
		cout << "*" << MidiNote << "("  <<  Notesym << Octave << ")";		
}
/*
   std::vector<int> MajorScale; // (MScale,MScale+7);
   std::vector<int> MinorScale; // (7,0,2,3,5,7,9,11);
   std::vector<int> DominantScale;
   std::vector<int> DiminishedScale;
   std::vector<int> AugmentedScale;
   std::vector<int> CurrentScale;
*/

void
CHORD::LoadScales()
	{
	int incr;
	int majorscale[7] = {0,2,4,5,7,9,11};
	std::vector<int> testvector;
	testvector.clear();
	for (incr=0;incr<7;incr++)
		{
		testvector.push_back(majorscale[incr]);
		}
	  MajorScale.clear();
	  MajorScale.push_back(0);
	  MajorScale.push_back(2);
	  MajorScale.push_back(4);
	  MajorScale.push_back(5);
	  MajorScale.push_back(7);
	  MajorScale.push_back(9);
	  MajorScale.push_back(11);
	  MinorScale.clear();
	  MinorScale.push_back(0);
	  MinorScale.push_back(2);
	  MinorScale.push_back(3);
	  MinorScale.push_back(5);
	  MinorScale.push_back(7);
	  MinorScale.push_back(9);
	  MinorScale.push_back(11);

	  DominantScale.clear();
	  DominantScale.push_back(0);
	  DominantScale.push_back(2);
	  DominantScale.push_back(3);
	  DominantScale.push_back(5);
	  DominantScale.push_back(7);
	  DominantScale.push_back(9);
	  DominantScale.push_back(11);

	  DiminishedScale.clear();
	  DiminishedScale.push_back(0);
	  DiminishedScale.push_back(2);
	  DiminishedScale.push_back(3);
	  DiminishedScale.push_back(5);
	  DiminishedScale.push_back(7);
	  DiminishedScale.push_back(9);
	  DiminishedScale.push_back(11);

	  AugmentedScale.clear();
	  AugmentedScale.push_back(0);
	  AugmentedScale.push_back(2);
	  AugmentedScale.push_back(4);
	  AugmentedScale.push_back(5);
	  AugmentedScale.push_back(7);
	  AugmentedScale.push_back(9);
	  AugmentedScale.push_back(11);

	CurrentHarmony.clear();
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(4);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(3);
	CurrentHarmony.push_back(4);
	CurrentHarmony.push_back(4);
	CurrentHarmony.push_back(4);
	MajorHarmony.clear();
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(4);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(3);
	MajorHarmony.push_back(4);
	MajorHarmony.push_back(4);
	MajorHarmony.push_back(4);
	MinorHarmony.clear();
	MinorHarmony.push_back(3);
	MinorHarmony.push_back(3);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(3);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(3);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(4);
	MinorHarmony.push_back(3);
	MinorHarmony.push_back(4);
	int dominantharmony[12] = {3,3,4,4,4,3,3,3,3,4,3,3};
	DominantHarmony.clear();
	for (incr=0;incr<12;incr++)
		{
		DominantHarmony.push_back(dominantharmony[incr]);
		}
	}
// **************************************************************************
// **************************************************************************
// Figure out what scale is appropriate for this chord.
// 	0 - Major
//	1 - Minor
//	2 - Dominant
//	3 - Diminished
//	4 - augmented
//	
// **************************************************************************
// **************************************************************************
int
CHORD::FindScale(std::string ChordType)
	{
	scale = 0;
	CurrentHarmony = MajorHarmony;
	if (ChordType.at(0) == 'M') {scale = 0; CurrentScale = MinorScale; CurrentHarmony = MajorHarmony;} 
	if (ChordType.at(0) == 'm') {scale = 1; CurrentScale = MinorScale; CurrentHarmony = MinorHarmony;} 
	if (ChordType.at(0) == '1') {scale = 2; CurrentScale = MinorScale; CurrentHarmony = DominantHarmony;}
	if (ChordType.at(0) == '7') {scale = 2; CurrentScale = DominantScale; CurrentHarmony = DominantHarmony;}
	if (ChordType.at(0) == '9') {scale = 2; CurrentScale = DominantScale; CurrentHarmony = DominantHarmony;}
	if (ChordType.at(0) == 'd') {scale = 3; CurrentScale = DiminishedScale;}
	if (ChordType.at(0) == '+') {scale = 4; CurrentScale = AugmentedScale;}
	return scale;
	}

typedef multimap <string,vector<int> > MMAP_CHORD;

MMAP_CHORD midi2chordtable;	

// **************************************************************************
// **************************************************************************
// load the Chord Table from a file. See Constructor.
// You can modify the file to agree with notation differences.
// Enter your symbol and then the intervals that form the chord.
// See the table for examples.
// **************************************************************************
// **************************************************************************
void
CHORD::loadChordtable(char *filename)
{

   FILE *fs;
   char temp[128];
   int i,k;
   int chordtablecount = 0;
   char Chrdtype[24];
   int a,b,c,d,e,f,g,h;
   a=b=c=d=e=f=g=h=0;
   bzero(temp,sizeof(temp));
   cout << " load Chord Table " << filename << endl;

  if ((fs = fopen (filename, "r")) != NULL)
   {
    while (!feof(fs)) {
         bzero(temp,sizeof(temp));
         fgets(temp, sizeof temp, fs);
         sscanf(temp,"%s %d,%d,%d,%d,%d,%d,%d,%d",&Chrdtype,&a,&b,&c,&d,&e,&f,&g,&h);
//	 cout << "loading Chordtable: " << Chrdtype << " vector: " << a << "," << b << "," << c << "," << d << "," << e << "," << f << "," << g << "," << h << endl;
		  MidiVector.clear();
		  MidiVector.push_back(a);
		  MidiVector.push_back(b);
		  MidiVector.push_back(c);
		  MidiVector.push_back(d);
		  MidiVector.push_back(e);
		  MidiVector.push_back(f);
		  MidiVector.push_back(g);
		  MidiVector.push_back(h);
         midi2chordtable.insert (MMAP_CHORD::value_type (Chrdtype,MidiVector));
	 chordtablecount++;
   	}
   }
   else
   {
   cout << " error opening chordtable file ...  " << endl;
   cout << " you may need to move the chordtable to a different directory " << endl;
   fl_message("Can't Find the chordtable file. It should be a hidden file (preceded with a dot) in your home directory.");
// ************************************************************************
// uncomment next lines to generate the chord table.
// Note that there is not any standardization of chord symbols,
// so if you like slightly different notation, you need to add
// to the table. This procedure, if you uncomment it, goes through
// all permutations of intervals writing the ones it recognizes as chords.
// A lot of combinations are not recognized, but you could add them. 
// ************************************************************************
/*
  zing.RinitChordtable = 0;
  std::vector<int> genintervals;
  size_t found;
  int tablecount =0;
  int testtotal =0;
  string Nullstring ("null");
// two intervals at a time.
  int i = 12;
        for (int j = 1; j < 12 ; ++j)
                {
                for (int k = 1; k < 12; ++k)
                        {
                        testtotal = (j+k);
                        if (testtotal > 11) break;
                        genintervals.clear();
                        genintervals.push_back(i);
                        genintervals.push_back(i + j);
                        genintervals.push_back(i + j + k);
                        std:string firstChordsym = zing.Midi2Chordsym(genintervals);
                        tablecount++;
                        }
                }  // j
// three at a time.
        cout << " **************** three at a time ******************** " << endl;
        for (int j = 1; j < 12 ; ++j)
                {
                for (int k = 1; k < 12; ++k)
                        {
                        for (int l = 1; l < 12; ++l)
                        {
                        testtotal = (j+k+l);
                        if (testtotal > 11) break;
                        genintervals.clear();
                        genintervals.push_back(i);
                        genintervals.push_back(i + j);
                        genintervals.push_back(i + j + k);
                        genintervals.push_back(i + j + k +l);
                        string secondChordsym = zing.Midi2Chordsym(genintervals);
                        tablecount++;
                        }
                        }
                }  // j
// four at a time
        cout << " **************** four at a time ******************** " << endl;
        for (int j = 1; j < 12 ; ++j)
                {
                for (int k = 1; k < 12; ++k)
                        {
                        for (int l = 1; l < 12; ++l)
                        {
                                for (int m = 1; m < 12; ++m)
                                {
                        testtotal = (j+k+l+m);
                        if (testtotal > 11) break;
                        genintervals.clear();
                        genintervals.push_back(i);
                        genintervals.push_back(i + j);
                        genintervals.push_back(i + j + k);
                        genintervals.push_back(i + j + k +l);
                        genintervals.push_back(i + j + k + l + m);
                        string thirdChordsym = zing.Midi2Chordsym(genintervals);
                        tablecount++;
                            }
                        }
                     }    //k
                }  // j
// five at a time
        cout << " **************** five at a time ******************** " << endl;
        for (int j = 1; j < 12 ; ++j)
                {
                for (int k = 1; k < 12; ++k)
                        {
                        for (int l = 1; l < 12; ++l)
                        {
                                for (int m = 1; m < 12; ++m)
                                {
                                for (int n = 1; n < 12; ++n)
                                {
                        testtotal = (j+k+l+m+n);
                        if (testtotal > 11) break;
                        genintervals.clear();
                        genintervals.push_back(i);
                        genintervals.push_back(i + j);
                        genintervals.push_back(i + j + k);
                        genintervals.push_back(i + j + k +l);
                        genintervals.push_back(i + j + k + l + m);
                        genintervals.push_back(i + j + k + l + m+ n);
                        string FourthChordsym = zing.Midi2Chordsym(genintervals);
                        tablecount++;
                                }
                            }
                        }
                     }    //k
                }  // j
        cout << " **************** six at a time ******************** " << endl;
        for (int j = 1; j < 12 ; ++j)
                {
                for (int k = 1; k < 12; ++k)
                        {
                        for (int l = 1; l < 12; ++l)
                        {
                                for (int m = 1; m < 12; ++m)
                                {
                                for (int n = 1; n < 12; ++n)
                                {
                                for (int vo = 1; vo < 12; ++vo)
                                {
                        testtotal = (j+k+l+m+n+vo);
                        if (testtotal > 11) break;
                        genintervals.clear();
                        genintervals.push_back(i);
                        genintervals.push_back(i + j);
                        genintervals.push_back(i + j + k);
                        genintervals.push_back(i + j + k +l);
                        genintervals.push_back(i + j + k + l + m);
                        genintervals.push_back(i + j + k + l + m + n);
                        genintervals.push_back(i + j + k + l + m + n + vo);
                        string FifthChordsym = zing.Midi2Chordsym(genintervals);
                        if (FifthChordsym != "null")
                                {
                                cout << tablecount << " six@X " << i << j << k << l << m << n << vo << endl;
                                }
                        cout << ".";
                        tablecount++;
                                }
                                }
                            }
                        }
                     }    //k
                }  // j
// **************************************************************************************
// Note: there are a lot more numbers generated than there are recognizable chords.
// Print some totals.
// **************************************************************************************
        cout << " number generated ============> " << tablecount << endl;
        cout << " load count is ===============> " << loadcount << endl;
*/

   }
   fclose(fs);
   cout << " loaded Chord Table... count:  " << chordtablecount << endl;	
};

// This could add to the table. 
void
CHORD::SetChordTable(string CurrentChordsym, std::vector<int> recintervals)
{
        midi2chordtable.insert (MMAP_CHORD::value_type (CurrentChordsym,recintervals));

}

// Initialize so something is there.
CHORD::CHORD()
        {
        CurrentChordsym = "C7";
	 iroot =0;
	 char filename[128];
	sprintf (filename, "%s%s", getenv ("HOME"), "/.ChordTable");
	loadChordtable(filename);
	LoadScales();
};
	
CHORD::~CHORD()
	{
	};
void
CHORD::SetCurrentChordsym(std::string Chordsym)
{
        CurrentChordsym = Chordsym;
}

std::string
CHORD::GetCurrentChordsym()
{
        return CurrentChordsym;
}

// It works, but it should be better.
std::vector<int>
CHORD::GetCurrentChordNotes()
{
int ik;
vector<int> NoteVector;
NoteVector = seperate.Chordsym2Midi(CurrentChordsym, 4);
// for (ik=0;ik<NoteVector.size(); ++ik) {
//                cout << "  ChordNotes: " << ik << ": " << NoteVector.at(ik) << endl;
//		}
return NoteVector;
}
// ****************************************************************************************
// Display the content of the chord table.
// ****************************************************************************************
void
CHORD::DumpChordTable()
{
        int dumpcount = 0;
         MMAP_CHORD::const_iterator dumptable;
         for ( dumptable = midi2chordtable.begin() ;
                dumptable != midi2chordtable.end() ;
                ++dumptable )
                {
                dumpcount++;
                cout << dumptable->first << " ";
                       for( vector<int>::const_iterator elementcount = dumptable->second.begin();
                        elementcount != dumptable->second.end(); ++elementcount )
                                {
                                cout << " " << *elementcount;
                                }
                cout << endl;
                }
}


// ******************************************************************************
// Given a chord symbol and a mode, produce a vector of midi notes.
// Mode corresponds to desired number of intervals produced. (TBD)
// ******************************************************************************
std::vector<int> CHORD::Chordsym2Midi(std::string Chordsym, int ChordMode)
{
int i;
// 
// Don't bother to look this up if its already done.
// if (CurrentChordsym == Chordsym) {return MidiVector;}
chordtype = "M";
scale = 0;
root.assign(Chordsym,0,1);
if (Chordsym.length() > 1)
	{
	if (( Chordsym.at(1) == 'b')  || (Chordsym.at(1) == '#'))
	        {
		root.assign(Chordsym,0,2);
		chordtype.assign(Chordsym.begin()+2,Chordsym.end());
		}
	else
	        {
	        root.assign(Chordsym,0,1);
		if (Chordsym.length() > 0)
			{
				chordtype.assign(Chordsym.begin()+1,Chordsym.end());
			}
	        }
	scale = FindScale(chordtype);
	}
// 	cout << "rlv Chordsym2Midi: " << Chordsym <<  " root: " << root << " type: " << chordtype << endl;
// note: set the root integer to 12, if it hasn't changed after these conditions...then it's an error.     
	iroot = 12;
	if (root == "C")
		{
		iroot =0;
		}
	if ((root == "C#") || (root == "Db"))
		{
		iroot =1;
		}
	if (root == "D")
		{
		iroot =2;
		}
	if ((root == "Eb") || (root == "D#"))
		{
		iroot =3;
		}
	if (root == "E")
		{
		iroot =4;
		}
	if (root == "F")
		{
		iroot =5;
		}
	if ((root == "F#") || (root == "Gb"))
		{
		iroot =6;
		}
	if (root == "G")
		{
		iroot =7;
		}
	if ((root == "G#") || (root == "Ab"))
		{
		iroot =8;
		}
	if (root == "A")
		{
		iroot =9;
		}
	if ((root == "Bb") || (root == "A#"))
		{
		iroot =10;
		}
	if (root == "B")
		{
		iroot =11;
		}
// if not found, return the vector with root 12, indicating reject 		
	if (iroot ==12)
		{
			MidiVector.clear();
       			MidiVector.push_back(iroot);
			return MidiVector;
		}
// Push back root integer.
	MMAP_CHORD::const_iterator ifound;
	string LookupString;
// By preceding the chordtype with a 1, the result will be root position. Other inversions are in the table using higher numbers.
	LookupString = "1" + chordtype;
	rootno = 1;
	ifound = midi2chordtable.find(LookupString);
	if (ifound == midi2chordtable.end())
		{
		rootno = 2;
		LookupString = "2" + chordtype;
		ifound = midi2chordtable.find(LookupString);
		}
	if (ifound == midi2chordtable.end())
		{
		rootno = 3;
		LookupString = "3" + chordtype;
		ifound = midi2chordtable.find(LookupString);
		}
	if (ifound == midi2chordtable.end())
		{
		rootno = 4;
		LookupString = "4" + chordtype;
		ifound = midi2chordtable.find(LookupString);
		}
	if (ifound == midi2chordtable.end())
		{
		rootno = 5;
		LookupString = "5" + chordtype;
		ifound = midi2chordtable.find(LookupString);
		}
	if (ifound == midi2chordtable.end())
		{
		rootno = 6;
		LookupString = "6" + chordtype;
		ifound = midi2chordtable.find(LookupString);
		}
	if (ifound != midi2chordtable.end())
		{
		SetCurrentChordsym(Chordsym);
		size_t nNumPairsInMap = midi2chordtable.count (LookupString);
		// a number of instances might be found though. More complete versions will have more intervals.
// 		cout << "  ========Chordsym2midi found it =============== " << Chordsym << " count: " << nNumPairsInMap << endl;
		int lastintervalsum = 0;
		for( size_t numberfound = 0 ;
			numberfound < nNumPairsInMap;
			++numberfound )
			{
			int intervalcount = 0;
// delete this ?
// sum of the intervals all the same ...?
			MidiVector.clear();
       			MidiVector.push_back(iroot);
			int intervalsum = 0;
			for( vector<int>::const_iterator intervalcounter = ifound->second.begin();
			intervalcounter != ifound->second.end(); ++intervalcounter )
				{				
				int whichone = *intervalcounter;
				if (whichone != 0)
					{
					 ++intervalcount;
					intervalsum = intervalsum + whichone;
					}
	       			MidiVector.push_back(whichone);
				}
			ifound++;
			}
		}	
	else
		{
		cout << "  === Chordsym2midi can't find it... but you could add it to the table. ==== " << Chordsym << endl;
		MidiVector.clear();
		iroot = 12;
       		MidiVector.push_back(iroot);
//		string CantFindIt = "Cannot Find " + Chordsym + "in the table ";
//		fl_message("Can't Find it.");
		}
return MidiVector;
}

// ******************************************************************************************************
// given midi input, figure out what kind of chord this is.
// input is evaluated to mod 12 and then the signature is evaluated.
// Works for most chords, although there is some ambiguity.
// For example E-Bb-D could be an Em9, but in context might be a C7. 
// *******************************************************************************************************
std::string CHORD::Midi2Chordsym(vector<int> midinotes)
{
std::string Chordsym;
	int numbernotes;
	int keynote[128];
	int keynumber;
	int modnote;
	int modwheel[12];
	int modwheelseq[12];
	int diff[12];
	int firstnote;
	int lastnote;
	int j;
	string Notesym;
	scale = 0;
	rootno =1;
	for (j=0; j<12; j++) {
		modwheel[j] = 0;
		modwheelseq[j] = 0;
		diff[j] = 0;
		}
	int k = 0;
        for (j=0; j<midinotes.size(); j++)
                {
                if (midinotes[j]!=0)
			{
			Midi2Note(midinotes[j]);
	                 k++;
        	         modnote = midinotes[j]%12;
    		             if (modwheel[modnote] == 0)
                	        {
                      	        modwheel[modnote] = midinotes[j];
           	                modwheelseq[modnote] = k;
                	        }
            		    if (k == 1)
               		         {
             		           lastnote = modnote;
              		          firstnote = modnote;
                	        }
           		     else
                   	     {
                     		   if (lastnote > modnote)
                     	           {
                	                diff[k] = ((modnote+12) -lastnote);
                 	               }
                	        else
                    	            {
                	                diff[k] = modnote - lastnote;
                 	               }
                   	     }
        	  lastnote = modnote;
           	     }
     		   numbernotes = k;
		}
	cout << endl;
	int aa;
	int z;
        k = firstnote;
	z = 0;
        aa = 1;
        for (j=0; j<12; j++) diff[j] = 0;
        for (j=0; j<12; j++)
                {
                if (aa == 1)
                        {
                        lastnote = k;
                        firstnote = k;
                        aa++;
                        }
                else
                        {
                                if (modwheelseq[k] != 0)
                                        {
                                        modnote = k;
                                        if (lastnote > modnote)
                                                {
                                                diff[aa] = ((modnote+12) -lastnote);
                                                }
                                        else
                                                {
                                                diff[aa] = modnote - lastnote;
                                                }
// 	       	                        printf("%s%i%s%i%s%i%s%i%s","rlv l2: ",aa," mw ",modwheel[z]," seq ",modwheelseq[z]," diff: ",diff[aa],"\n");
                                        lastnote = modnote;
                                        aa++;
					z++;
                                        }
                        }
                k++;
                if (k > 11) k = 0;
                }

// *******************************************************************************************************************
// Given intervals between the midi input. Determine the chord.
// Turns out there are a lot of combinations, most are not named. I might not have 
// found them all or even got them all right. Updates welcome.
// *******************************************************************************************************************
	int a, b, c, d, e, f, g, h, i, l;
	int ic;
	int jc;
	int kc;
	int root;
	int ii;
	int nointervals;
	nointervals = aa-2; 
//	printf("%s%i%s","rlv CHORD Midi2Chordsym... number intervals : ",nointervals,"\n");
	a =0; b =0; c=0; d=0; e=0; f=0; g=0; h=0; i=0; l = 0;
	for(ii=0; ii < nointervals+1; ii++)
	{
//	printf("%s%i%s%i%s","rlv CHORD interval : ",diff[ii]," ii: ",ii,"\n");
	if (ii ==1) a = diff[ii+1];
	if (ii ==2) b = diff[ii+1];
	if (ii ==3) c = diff[ii+1];
	if (ii ==4) d = diff[ii+1];
	if (ii ==5) e = diff[ii+1];
	if (ii ==6) f = diff[ii+1];
	}
//     ***********************************************************************************************************
//     ***********************************************************************************************************
//     ***********************************************************************************************************
	root = 1;
	scale = 0;
	Chordsym = " ";
	if ( nointervals < 2 ) return Chordsym;
// lookup depending on how many intervals have been passed. 
 	switch(nointervals)
	{
		case 1:			// one intervals: could use this for easy play. Someday.
		{
			Chordsym = "null";
			break;
		}
		case 2:			// two intervals: this is a triad.
		{
			Chordsym = "null";
			switch(a){
				case 1: {switch(b){
						case 3: {Chordsym = "mM7"; root=2; break;}
						case 4: {Chordsym = "M7"; root=2; break;}
						break;
						}
					break;
					}
				case 2: {switch(b){
						case 2: {Chordsym = "9"; break;}
						case 3: {Chordsym = "m7"; root=2; break;}
						case 4: {Chordsym = "7"; root=2; break;}
						case 5: {Chordsym = "sus"; root=3; break;}
						case 6: {Chordsym = "(b5)"; root=3; break;}
						case 7: {Chordsym = "m11"; root=3; break;}
						break;
						}
					break;
					}
				case 3: {switch(b){
						case 2: {Chordsym = "m11"; break;}
						case 3: {Chordsym = "dim"; break;}
						case 4: {Chordsym = "m"; break;}
						case 5: {Chordsym = "M"; root=3; break;}
						case 6: {Chordsym = "dim"; root=3; break;}
						case 7: {Chordsym = "m7"; break;}
						case 8: {Chordsym = "mM7"; break;}
						break;
						}
					break;
					}
				case 4: {switch(b){
						case 2: {Chordsym = "M(b5)"; break;}
						case 3: {Chordsym = "M"; break;}
						case 4: {Chordsym = "+"; break;}
						case 5: {Chordsym = "m"; root=3; break;}
						case 6: {Chordsym = "7"; break;}
						case 7: {Chordsym = "Maj7"; break;}
						break;
						}
					break;
					}
				case 5: {switch(b){
						case 2: {Chordsym = "sus4"; break;}
						case 3: {Chordsym = "m"; root=2; break;}
						case 4: {Chordsym = "M"; root=2; break;}
						case 5: {Chordsym = "M"; root=3; break;}
						break;
						}
					break;
					}
				case 6: {switch(b){
						case 2: {Chordsym = "7"; root=3; break;}
						case 3: {Chordsym = "dim"; root=2; break;}
						case 4: {Chordsym = "(b5)"; root=2; break;}
						break;
						}
					break;
					}
				case 7: {switch(b){
						case 1: {Chordsym = "Maj7"; root=3; break;}
						case 2: {Chordsym = "m7"; root=3; break;}
						case 3: {Chordsym = "m711"; root=2; break;}
						break;
						}
					break;
					}
				break;
				}       // end switch 2a.
		break;    // end of two intervals.     
		}
		case 3:			// three intervals: there are four distinct notes. 
		{	
			Chordsym = "null";
			switch(a){
				case 1: {switch(b){
						case 2: {switch(c){	
								case 2: {Chordsym = "mb6b9"; break;}
								}
							break;
							}	
						case 3: {switch(c){	
								case 2: {Chordsym = "?"; root =4; break;}
								case 3: {Chordsym = "dim(Maj7)"; root =2; break;}
								case 4: {Chordsym = "m(Maj7)"; root=2; break;}
								case 6: {Chordsym = "7b9"; break;}
								}
							break;
							}	
						case 4: {switch(c){	
								case 2: {Chordsym = "susb9"; break;}
								case 3: {Chordsym = "Maj7"; root =2; break;}
								case 4: {Chordsym = "+(Maj7)"; break;}
								}
							break;
							}	
						break;
						}
					break;
					}
				case 2: {switch(b){
						case 1: {switch(c){	
								case 3: {Chordsym = "m7"; root =3; break;}
								case 4: {Chordsym = "madd9"; break;}
								}
							break;
							}	
						case 2: {switch(c){	
								case 3: {Chordsym = "9"; break;}
								case 4: {Chordsym = "+7"; root =3; break;}
								case 6: {Chordsym = "7/9"; break;}
								}
							break;
							}	
						case 3: {switch(c){	
								case 2: {Chordsym = "7(Sus4)"; root =4; break;}
								case 3: {Chordsym = "m7(b5)"; root =2; break;}
								case 4: {Chordsym = "m7"; root=2; break;}
								}
							break;
							}	
						case 4: {switch(c){	
								case 2: {Chordsym = "7b5"; root =4; break;}
								case 3: {Chordsym = "7"; root =2; break;}
								case 4: {Chordsym = "+7"; root =2; break;}
								}
							break;
							}	
						case 5: {switch(c){	
								case 2: {Chordsym = "7(Sus4)"; root =2; break;}
								}
							break;
							}	
						break;
						}
					break;
					}
				case 3: {switch(b){
						case 1: {switch(c){	
								case 3: {Chordsym = "m7"; root =3; break;}
								case 4: {Chordsym = "+(Maj7)"; root =4; break;}
								}
							break;
							}	
						case 2: {switch(c){	
								case 3: {Chordsym = "m7"; root =3; break;}
								case 4: {Chordsym = "7"; root =3; break;}
								case 5: {Chordsym = "7(Sus4)"; root =3; break;}   // fourths
								}
							break;
							}	
						case 3: {switch(c){	
								case 2: {Chordsym = "7"; root=4; break;}
								case 3: {Chordsym = "dim7"; break;}
								case 4: {Chordsym = "m7(b5)"; break;}
								case 5: {Chordsym = "m(Maj7)(b5)"; break;}
								}
							break;
							}	
						case 4: {switch(c){	
								case 1: {Chordsym = "Maj7"; root=4; break;}
								case 2: {Chordsym = "m6"; break;}
								case 3: {Chordsym = "m7"; break;}
								case 4: {Chordsym = "m(Maj7)"; break;}
								case 5: {Chordsym = "m"; break;}
								}
							break;
							}	
						case 5: {switch(c){	
								case 2: {Chordsym = "m7#5"; break;}
								case 3: {Chordsym = "m#5M7"; break;}
								case 4: {Chordsym = " "; root=3; break;}
								}
							break;
							}
						case 6: {switch(c){	
								case 2: {Chordsym = "7b9"; root=2; break;}
								}
							break;
							}
						break;
						}
					break;
					}
				case 4: {switch(b){
						case 1: {switch(c){	
								case 3: {Chordsym = "m(Maj7)"; root =3; break;}
								case 4: {Chordsym = "Maj7"; root =3; break;}
								}
							break;
							}	
						case 2: {switch(c){	
								case 2: {Chordsym = "+7"; root=4; break;}
								case 3: {Chordsym = "m6"; root=4; break;}
								case 4: {Chordsym = "7-5"; root=3; break;}
								case 5: {Chordsym = "M7b5"; break;}
								}
							break;
							}	
						case 3: {switch(c){	
								case 2: {Chordsym = "6"; break;}
								case 3: {Chordsym = "7"; break;}
								case 4: {Chordsym = "Maj7"; break;}
								case 5: {Chordsym = ""; break;}
								}
							break;
							}	
						case 4: {switch(c){	
								case 1: {Chordsym = "mMaj7"; root=4; break;}
								case 2: {Chordsym = "+7"; break;}
								case 3: {Chordsym = "+M7"; break;}
								}
							break;
							}	
						case 5: {switch(c){	
								case 1: {Chordsym = "13"; break;}
								case 3: {Chordsym = "m"; break;}
								}
							break;
							}	
						break;
						}
					break;
					}
				case 5: {switch(b){
						case 1: {switch(c){	
								case 3: {Chordsym = "m(Maj7)"; root =3; break;}
								case 4: {Chordsym = "Maj7"; root =3; break;}
								}
							break;
							}	
						case 2: {switch(c){	
								case 3: {Chordsym = "7sus4"; break;}
								}
							break;
							}	
						case 3: {switch(c){	
								case 4: {Chordsym = "m"; root =2; break;}
								}
							break;
							}	
						case 4: {switch(c){	
								case 2: {Chordsym = "+7"; break;}
								case 3: {Chordsym = ""; root=2; break;}
								}
							break;
							}	
						break;
						}
					break;
					}
				break;
				}
                        break;
                }       // end of three intervals.
                case 4:                 // there are four intervals:
                {
                        Chordsym = "null";
			switch(a){
				case 1: {switch(b){
						case 2: {
							if (c == 2 & d == 3) {Chordsym = "Maj9"; root=2; break;}
							break;
							}	
						case 3: {
							if (c == 2 & d ==1) Chordsym = "7b9(13)";
							if (c == 3 & d == 2) Chordsym = "7b9b13";
							if (c == 3 & d == 3) Chordsym = "7(b9)";
							if (c == 3 & d == 4) Chordsym = "M7b9";
							if (c == 4 & d == 2) Chordsym = "7b9b13";
							if (c == 5 & d == 1) Chordsym = "(b9)13";
							break;
							}	
						case 4: {
							if (c == 2 & d == 3) {Chordsym = "7susb9"; break;}
							break;
							}	
						break;
						}
					break;
					}
				case 2: {switch(b){
						case 1: {
							if (c == 2 & d == 5 ) Chordsym="m911";
							if (c == 1 & d == 4 ) Chordsym="m911";
							if (c == 3 & d == 4 ) Chordsym="m9b5";
							if (c == 4 & d == 2) Chordsym = "m6/9";
							if (c == 4 & d == 3) Chordsym = "m7/9";
							if (c == 4 & d == 4) Chordsym = "mM9";
							if (c == 5 & d == 2) Chordsym = "m9#5";
							break;
							}
						case 2: {
							if (c == 1 & d == 4) {Chordsym="m9"; root =2; break;}
							if (c == 2 & d == 3) {Chordsym="9"; root=2; break;}
							if (c == 2 & d == 4) Chordsym="9-5";
							if (c == 2 & d == 5) Chordsym="M9b5";
							if (c == 3 & d == 2) Chordsym="69";
							if (c == 3 & d == 3) Chordsym="79";
							if (c == 3 & d == 4) Chordsym="Maj7/9";
							if (c == 4 & d == 2) Chordsym="9b13";
							if (c == 5 & d == 1) Chordsym="13";
							break;
							}
						case 3: {
							if (c == 2 & d == 3) {Chordsym="9sus4"; break;}
							if (c == 3 & d == 2) {Chordsym="9";root = 5; break;}
							break;
							}
						break;
						}
					break;
					}
				case 3: {switch(b){
						case 1: {
							if (c == 3 & d == 3) Chordsym="7#9";
							if (c == 4 & d == 2) Chordsym="+7#9";
							break;
							}	
						case 2: {
							if (c ==2 & d == 2) {Chordsym = "9"; root =3;break;}
							if (c == 3 & d == 2)
								{
								 root = 5;
								 Chordsym = "4ths";
								 break;
								}
							if (c ==2 & d == 1) {Chordsym = "m9"; root =3;break;}
							if (c ==2 & d == 3) Chordsym = "m7(13)";
							break;
							}	
						case 3: {
							if (c == 2 & d == 2) {Chordsym="9"; root=4; break;}
							break;
							}	
						case 4: {
							if (c == 1 & d == 2) {Chordsym="Maj9"; root=4; break;}
							if (c == 1 & d == 3) Chordsym="mM7b6";
							break;
							}	
						break;
						}
					break;
					}
				case 4: {switch(b){
						case 1: {
							if (c == 2 & d == 2) {Chordsym="Maj9"; root =3; break;}
							break;
							}	
						case 2: {
							if (c == 1 & d == 4) Chordsym="M7#11";
							if (c == 3 & d == 1) Chordsym="13b5";
							break;
							}	
						case 3: {
							if (c == 1 & d == 3) Chordsym="M7b6";
							if (c == 1 & d == 2) Chordsym="7b13";
							if (c == 2 & d == 1) Chordsym="13";
							if (c == 2 & d == 2) {Chordsym="m9"; root=4; break;}
							break;
							}	
						break;
						}
					break;
					}
				break;
				}
                        break;
                }       // end of four intervals.
                case 5:                 // five intervals:
                {
                        Chordsym = "null";
			switch(a){
				case 1: {switch(b){
						case 1: {
							if (c == 2 & d == 5 & e ==1) Chordsym = "7b913";
							break;
							}
						case 2: {
							if (c == 2 & d == 5 & e ==1) Chordsym = "xb913";
							break;
							}
						case 3: {
							if (c == 1 & d == 2 & e == 3) Chordsym="11-9";
							if (c == 2 & d == 3 & e == 1) Chordsym="7b9#1113";
							if (c == 2 & d == 1 & e == 3) Chordsym="7b9#11";
							if (c == 3 & d == 1 & e ==2) Chordsym = "7b9(13)";
							if (c == 3 & d == 2 & e ==1) Chordsym = "7b9(13)";
							break;
							}
						case 4: {
							if (c == 2 & d == 1 & e ==2) Chordsym = "7sus4b9b13";
							break;
							}
						break;
					   	}
					break;
					}
				case 2: {switch(b){                              // retest.
						case 1: {
							if (c == 2 & d == 2 & e ==3) Chordsym = "m7911";
							if (c == 2 & d == 2 & e ==2) Chordsym = "maj13";
							if (c == 2 & d == 2 & e ==1) Chordsym = "m13";
							if (c == 2 & d == 2 & e ==4) Chordsym = "m7911";
							if (c == 2 & d == 1 & e ==4) Chordsym = "m11b5";
							if (c == 2 & d == 3 & e ==2) Chordsym = "m11#5";
							break;
							}
						case 2: {
							if (c == 1 & d == 2 & e ==3) Chordsym = "11";
							if (c == 1 & d == 2 & e ==4) Chordsym = "Maj11";
							if (c == 2 & d == 1 & e ==2) Chordsym = "69#11";
							if (c == 2 & d == 1 & e ==3) Chordsym = "79#11";
							if (c == 2 & d == 1 & e ==4) Chordsym = "Maj79#11";
							if (c == 2 & d == 2 & e ==2) Chordsym = "9#5#11";
							if (c == 2 & d == 3 & e ==2) 
								{
								 root = 6;
								 Chordsym = "4ths";
								 break;
								}
							if (c == 3 & d == 2 & e ==1) Chordsym = "9/13";
							if (c == 3 & d == 2 & e ==2) Chordsym = "Maj6/9";
							if (c == 3 & d == 1 & e ==2) Chordsym = "9b13";
							if (c == 2 & d == 5 & e ==1) Chordsym = "xb913";
							break;
							}
						case 3: {
							if (c == 1 & d == 2 & e == 3) Chordsym="11-9";
							if (c == 2 & d == 2 & e == 1) Chordsym="13sus4";
							if (c == 3 & d == 1 & e ==2) Chordsym = "7b9(13)";
							if (c == 3 & d == 2 & e ==1) Chordsym = "7b9(13)";
							break;
							}
						case 4: {
							if (c == 1 & d == 2 & e == 1) Chordsym="9#1113";
							break;
							}
						break;
					   	}
					break;
					}
				case 3: {switch(b){
						case 1: {
							if (c == 3 & d == 2 & e ==1) Chordsym = "#913";
							break;
							}
						case 2: {
							break;
							}
						case 3: {
							break;
							}
						break;
					   	}
					break;
					}
				break;
				}
                        break;
                }       // end of five intervals.
                case 6:                 // six intervals:
                {
                        Chordsym = "null";
			switch(a){
				case 1: {switch(b){
						case 3: {
							if (c == 1 & d == 2 & e == 2 & f == 1) Chordsym="13-9";
							if (c == 1 & d == 2 & e == 3 & f == 1) Chordsym="13-9";
							if (c == 2 & d == 1 & e == 2 & f == 1) Chordsym="7b9#1113";
							break;
							}
						break;
					   	}
					break;
					}
				case 2: {switch(b){
						case 1: {
							if (c == 2 & d == 2 & e == 2 & f == 1) Chordsym="m13";
							break;
							}
						case 2: {
							if (c == 2 & d == 1 & e == 1 & f == 2) Chordsym="9#11b13";
							if (c == 2 & d == 1 & e == 2 & f == 1) Chordsym="7/9/#11/13";
							if (c == 2 & d == 1 & e == 2 & f == 2) Chordsym="M13#11";
							if (c == 1 & d == 2 & e == 3 & f == 1) Chordsym="uhhhh";
							if (c == 1 & d == 2 & e == 2 & f == 2) Chordsym="Maj13";
							if (c == 1 & d == 2 & e == 2 & f == 1) Chordsym="13";
							break;
							}
						case 3: {
							if (c == 2 & d == 2 & e == 1 & f == 2) Chordsym="7/9/#11/13";
							break;
							}
						break;
					   	}
					break;
					}
				case 3: {switch(b){
						case 1: {
							if (c == 2 & d == 1 & e == 2 & f == 1) Chordsym="7#9#1113";
							break;
							}
						break;
					   	}
					break;
					}
				break;
				}
                        break;
                }       // end of six intervals.
                case 7:                 // seven intervals:
                {
                        Chordsym = "null";
                        break;
                }       // end of seven intervals.
                case 8:                 // eight intervals:
                {
                        Chordsym = "null";
                        break;
                }       // end of eight intervals.
	}		// ********************** end of interval switches.

// ***************************************************************************************
	rootno = root;
	for (jc = 0; jc<12; jc++)
		{
		if (root == modwheelseq[jc])
			{
			ic = modwheel[jc];
			break;
			}
		}

	kc = ic%12;
	iroot = kc;
	keynumber1 = kc;
	laa1 = keynumber1;
	laa3 = keynumber1 + a;
	laa5 = keynumber1 + b;
	laa7 = keynumber1 + c;
	switch(kc)
		{
		case 0:
			Notesym = "C";
			break;
		case 1:
			Notesym = "C#";
			break;
		case 2:
			Notesym = "D";
			break;
		case 3:
			Notesym = "Eb";
			break;
		case 4:
			Notesym = "E";
			break;
		case 5:
			Notesym = "F";
			break;
		case 6:
			Notesym = "F#";
			break;
		case 7:
			Notesym = "G";
			break;
		case 8:
			Notesym = "G#";
			break;
		case 9:
			Notesym = "A";
			break;
		case 10:
			Notesym = "Bb";
			break;
		case 11:
			Notesym = "B";
			break;
		}
// 		cout << "tablechord ====> " <<  Chordsym << "  root: " << Notesym << ",("  << nointervals  << ")";		
// 		cout << a << "," << b << "," << c << "," << d << "," << e << "," << f << "," << g << "," << h << endl;		
		if (Chordsym != "null")
		{
		string strroot= "?";
// Determine which of the input notes is the root...  
		switch(root)
			{
			case 1: {strroot = "1"; break;}
			case 2: {strroot = "2"; break;}
			case 3: {strroot = "3"; break;}
			case 4: {strroot = "4"; break;}
			case 5: {strroot = "5"; break;}
			case 6: {strroot = "6"; break;}
			case 7: {strroot = "7"; break;}
			}	 
		string TableChordsym = strroot + Chordsym;
//		cout << Chordsym << "," << strroot << ",("  << nointervals  << ")";		
//		cout << a << "," << b << "," << c << "," << d << "," << e << "," << f << "," << g << "," << h << endl;		
		  MidiVector.clear();
		  MidiVector.push_back(iroot);
		  MidiVector.push_back(a);
		  MidiVector.push_back(b);
		  MidiVector.push_back(c);
		  MidiVector.push_back(d);
		  MidiVector.push_back(e);
		  MidiVector.push_back(f);
		  MidiVector.push_back(g);
// *************************************************************************************************
// This next switch can be used to initialize the chord table, but it is better just to
// continue using and updating the .ChordTable file which is edited to include more varied
// chord symbols. 
// *************************************************************************************************
		if (RinitChordtable == 0)
			{
        	        midi2chordtable.insert (MMAP_CHORD::value_type (TableChordsym,MidiVector));
			}
// *************************************************************************************************
// *************************************************************************************************
// *************************************************************************************************
		string ReturnChordsym = Notesym + Chordsym;
		scale = FindScale(Chordsym);
//		cout << " returning " << ReturnChordsym << " scale " << scale << endl;
		return ReturnChordsym;
		}
		else
		{
		return "";
		}
}
