/*
 *  This file is part of GNUDoQ, Copyright (C) 2005-2006 Luc Vo Van
 *
 *   GNUDoQ is free software; you can redistribute it and/or modify it
 *   under the terms of the GNU General Public License as published by the
 *   Free Software Foundation; either version 2, or (at your option) any
 *   later version.
 *
 *   GNUDoQ is distributed in the hope that it will be useful, but WITHOUT
 *   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *   for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with GNUDoQ; see the file COPYING.  If not, write to
 *   the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *   Boston, MA 02111-1307, USA.
 *
 *   Class          :   Number widget (implementation)
 *   Author         :   Luc Vo Van
 *   Original Date  :   September 2005
 *
 **/

#include <QtGui>
#include <QPen>
#include <QPainterPath>
#include <QPolygon>

#include "GNUDoQBoxWidget.H"

// Colors are arranged so that zones that are close together don't
// have similar color schemes.
const QColor
  GNUDoQBoxWidget::groupColor[] = {
  QColor::fromRgb (255, 191, 191),
  QColor::fromRgb (191, 255, 234),
  QColor::fromRgb (255, 191, 234),
  QColor::fromRgb (255, 234, 191),
  QColor::fromRgb (191, 234, 255),
  QColor::fromRgb (191, 255, 191),
  QColor::fromRgb (234, 191, 255),
  QColor::fromRgb (234, 255, 191),
  QColor::fromRgb (191, 191, 255)
};


GNUDoQBoxWidget::GNUDoQBoxWidget (int iGroup, QWidget * parent):
QWidget (parent)
{
  m_iValue = 0;
  m_iGroup = iGroup;
  m_bHovered = false;
  m_bEditable = false;
  m_bErroneous = false;
  m_bCpusolved = false;

  // Sets the font to be highly visible (bold, large..)
  QFont fFont = font ();
  fFont.setWeight (QFont::Black);
  fFont.setPointSize (fFont.pointSize () + 16);
  setFont (fFont);
  setFocusPolicy (Qt::StrongFocus);
}


void
GNUDoQBoxWidget::setErroneous (const bool bErroneous)
{
  m_bErroneous = bErroneous;
  repaint ();
}

void
GNUDoQBoxWidget::setEditable (const bool bEditable)
{
  m_bEditable = bEditable;
  repaint ();
}

void
GNUDoQBoxWidget::setValue (const int value)
{
  m_iValue = value;
  emit valueChanged ();
  repaint ();
}

void
GNUDoQBoxWidget::setCpusolved (const bool bCpusolved)
{
  m_bCpusolved = bCpusolved;
}

void
GNUDoQBoxWidget::keyPressEvent (QKeyEvent * event)
{
  if (!m_bEditable)
    return;

  bool ok;
  int iTmp = event->text ().toInt (&ok);
  // If the key is an integer, use its value for the Box
  if (ok)
    {
      setValue (iTmp);
      return;
    }

  // If we press backspace or delete, we reset the box
  if (event->key () == Qt::Key_Backspace
      || event->key () == Qt::Key_Delete || event->key () == Qt::Key_Space)
    setValue (0);
}

void
GNUDoQBoxWidget::mousePressEvent (QMouseEvent * event)
{
  if (!m_bEditable)
    return;

  // Left click increases the value, right click decreases (loops
  // if the limit is reached)
  // middle click clears the box (sets the value to 0)
  switch (event->button ())
    {
    case Qt::LeftButton:
      if (value () == 9)
	setValue (0);
      else
	setValue (value () + 1);
      break;

    case Qt::RightButton:
      if (value () == 0)
	setValue (9);
      else
	setValue (value () - 1);
      break;

    case Qt::MidButton:
      setValue (0);
      break;

    default:
      break;
    }
}

void
GNUDoQBoxWidget::wheelEvent (QWheelEvent * event)
{
  if (!m_bEditable)
    return;

  // Increases or decreases the value if the wheel is rolled
  // up or down.
  if (event->delta () > 0)
    {
      if (value () == 9)
	setValue (0);
      else
	setValue (value () + 1);
    }
  else
    {
      if (value () == 0)
	setValue (9);
      else
	setValue (value () - 1);
    }
}

void
GNUDoQBoxWidget::paintEvent (QPaintEvent *)
{
  QPainter painter (this);
  QLinearGradient gradient (0, 0, width (), height ());
  //QRadialGradient gradient (50, 50, 50, 50, 50);
  QColor cBackgroundColor, cBackgroundColor2;

  cBackgroundColor2 = Qt::gray;
  if (m_bErroneous)
    cBackgroundColor = Qt::red;
  else if (m_bCpusolved)
    {
      cBackgroundColor = QColor::fromRgb (0, 255, 0);
      cBackgroundColor2 = QColor::fromRgb (0, 128, 0);
    }
  else
    {
      if (m_bEditable)
	if (m_bHovered)
	  cBackgroundColor = groupColor[m_iGroup].dark (150);
	else
	  cBackgroundColor = groupColor[m_iGroup].dark (300);
      else
	cBackgroundColor = groupColor[m_iGroup];
    }

  gradient.setColorAt (0.0, cBackgroundColor);
  gradient.setColorAt (1.0, cBackgroundColor2);
  painter.setBrush (gradient);
  painter.drawRect (0, 0, width (), height ());

  // Draws the two corner triangles when the box is hovered
  // and editable in order to highlight its focus
  if (m_bEditable && m_bHovered)
    {	    
      QPainterPath ppPath;
      QBrush brush (QColor::fromRgb (255, 255, 255, 120));
      QPolygonF pPolygon1, pPolygon2;
      int iPolygonSize = (int) width () / 4;
      pPolygon1 << QPointF (width (), height ())
	<< QPointF (width () - iPolygonSize, height ())
	<< QPointF (width (), height () - iPolygonSize);
      pPolygon2 << QPointF (0, 0)
	<< QPointF (iPolygonSize, 0) << QPointF (0, iPolygonSize);
      ppPath.addPolygon (pPolygon1);
      ppPath.addPolygon (pPolygon2);
      painter.fillPath (ppPath, brush);
    }

  // Draws the text if ncessary
  if (value () > 0)
    {
      QString sToPrint = QString::number (m_iValue);
      QFontMetrics metrics (font ());
      int x = (width () - metrics.width (sToPrint)) / 2;
      int y = (height () + metrics.ascent () - metrics.descent ()) / 2;

      // Changes the color of the displayed text
      if (m_bEditable || m_bCpusolved)
	painter.setPen (Qt::white);
      else
	painter.setPen (foregroundRole ());

      painter.drawText (x, y, QString (sToPrint));
    }

  painter.end ();
}

void
GNUDoQBoxWidget::enterEvent (QEvent *)
{
  m_bHovered = true;
  setFocus ();
  repaint ();
}

void
GNUDoQBoxWidget::leaveEvent (QEvent *)
{
  m_bHovered = false;
  clearFocus ();
  repaint ();
}
