
import os, sys, logging
import node as nodemodule

import constants
from configuration import config

l = logging.getLogger(__name__)

try:
    from reportlab.pdfgen import canvas as canvasmodule
    from reportlab.lib.units import cm
    from reportlab.lib.pagesizes import letter, A4
    from reportlab.lib import colors
    l.debug("using installed reportlab")
except:
    l.debug("Using copy of reportlab")
    sys.path.append(os.path.join(config["kayalidir"],"copy"))
    from reportlab.pdfgen import canvas as canvasmodule
    from reportlab.lib.units import cm
    from reportlab.lib.pagesizes import letter, A4
    from reportlab.lib import colors

from pdf.afm import AFM

import logging
l = logging.getLogger(__name__)

replacements = {}
replacements[constants.INTEGRALSIGN] = unichr(242)
replacements[u'\u03C0'] = unichr(112) #PI
replacements[u'\u03B1'] = unichr(97)  #ALPHA
replacements[u'\u03B2'] = unichr(98)  #BETA
replacements[u'\u03B3'] = unichr(103) # GAMMA
replacements[u'\u03B4'] = unichr(100) # DELTA
replacements[u'\u03B5'] = unichr(101) #EPSILON
replacements[u'\u03A3'] = unichr(83) # capital SIGMA
#replacements["%ZETA"] = u'\u03B6'
replacements[u'\u03B7'] = unichr(103) #ETA
#replacements["%THETA"] = u'\u03B8'
#replacements["%IOTA"] = u'\u03B9'
#replacements["%KAPPA"] = u'\u03BA'
#replacements["%LAMBDA"] = u'\u03BB'
#replacements["%MU"] = u'\u03BC'
#replacements["%NU"] = u'\u03BD'
#replacements["%XI"] = u'\u03BE'

replacements[u'\u221E']= chr(165) #INF


def getScaledBracketWidth(spacewidth, h):
    return spacewidth*3

def getIntegralWidth(originalWidth, h):
    return originalWidth * h**(0.5/3.0)

def drawNodesToPdf(fileName, nodes, initialFontSize, verticalGap):
    saveFactory = nodemodule.fontfactory
    nodemodule.fontfactory = FontFactory()
    document = Document(fileName)
    w = document.width
    h = document.height
    y = verticalGap
    for node in nodes:
        node.setTextSize() # resets font
        try:
            subNodes = node.layoutNode(0,0,w, config["pdffont"], initialFontSize)
        except:
            l.exception("laying out node")
        try:
            for node in subNodes:
                if node.h + verticalGap > h:
                    l.error("node too big for PDF page")
                    continue
                if y + node.h > h:
                    # this node will go onto the next page
                    y = verticalGap
                    document.showPage()
                y += node.h - node.operatorcentreline
                painter = document.getPainter()
                l.debug("set text position to %s %s" % ((w - node.w)/2.0,y))
                node.setTextPosition((w - node.w)/2.0,y)
                node.drawNode(painter)
                painter.end()
                y += node.operatorcentreline + verticalGap
        except:
            l.exception("creating Pdf")
        y += verticalGap * 2
    document.finish()

    nodemodule.fontfactory = saveFactory
    l.debug("node fontfactory is %s" % nodemodule.fontfactory)


class Document:
    def __init__(self, fileName):
        self.canvas = canvasmodule.Canvas(fileName, pagesize=A4)
        self.width, self.height = A4
        self.canvas.setFont(config["pdffont"],config["pdffontsize"])
        self.canvas.setLineWidth(config["pdflinewidth"])

    def showPage(self):
        self.canvas.showPage()
        self.canvas.setFont(config["pdffont"],config["pdffontsize"])
        self.canvas.setLineWidth(config["pdflinewidth"])

    def getPainter(self):
        painter = Painter(self.canvas, self.width, self.height)
        return painter

    def finish(self):
        self.canvas.showPage()
        self.canvas.save()
        

class FontFactory:

    def makeFont(self):
        return Font()

    def getFontMetrics(self, font):
        return FontMetrics(font)


class Painter:
    def __init__(self, canvas, width, height):
        self.canvas, self.width, self.height = canvas, width, height
        
    def setFont(self, font):
        self.font = font
        #l.debug("font is %s family is %s size %s" % (font, font.family, font.pointSize))
        self.canvas.setFont(font.family(), font.pointSize())

    def setFontFamily(self, fontFamily):
        self.font.setFamily(fontFamily)
        #l.debug("font is %s family is %s size %s" % (font, font.family, font.pointSize))
        self.canvas.setFont(self.font.family(), self.font.pointSize())

    def save(self):
        self.stateSaveFontSize = self.font.pointSize()
        self.stateSaveFontFamily = self.font.family()
        self.canvas.saveState()

    def restore(self):
        self.canvas.restoreState()
        self.font.setPointSize( self.stateSaveFontSize)
        self.font.setFamily(self.stateSaveFontFamily)
        self.canvas.setFont(self.stateSaveFontFamily, self.stateSaveFontSize)

    def setPenColour(self, colour):
        if colour == constants.blue:
            colour = colors.blue
        if colour == constants.black:
            colour = colors.black
        if colour == constants.white:
            colour = colors.white
        if colour == constants.red:
            colour = colors.red
            
        self.canvas.setFillColor(colour)

    def drawText(self, x, y, text):
        saveFontFamily = self.font.family()
        if text in replacements:
            pass
            #text = replacements[text]
            #self.setFontFamily("Symbol")
        try:
            self.canvas.drawString(x,self.height - y, text)
        except Exception,ex:
            l.exception("drawing string %s (%s)",repr(text),ex)
        self.setFontFamily(saveFontFamily)

    def drawRect(self, x, y, w, h):
        self.canvas.rect(x,self.height - y - h,w,h)

    def drawLine(self, x1, y1, x2, y2):
        self.canvas.line(x1, self.height - y1, x2, self.height - y2)

    def scale(self, factorX, factorY):
        self.canvas.scale(factorX, factorY)

    def translate(self, x, y):
        self.canvas.translate(x,y)

    def drawCharacterToFit(self, metrics, ch, x, y, w, h):
        #ch = "T"
        #l.debug("draw into %4.2f %4.2f %4.2f %4.2f" % (x,y,w,h))
        #self.drawRect(x,y,w,h)
        chSize = metrics.charBoundingRect(ch)
        #l.debug("chsize is " + repr(chSize))
        saveFontFamily = self.font.family()
        self.save()
        try:
            if ch in replacements:
                #ch = replacements[ch]
                #c = ord(ch)
                #ch = chr(c)
                self.setFontFamily("Symbol")

            xScale = float(w)/chSize.width()
            yScale = float(h)/chSize.height()

            # add on the height
            self.translate(0,self.height)
            # flip y
            #self.scale(1,-1)
            self.scale(xScale, yScale)
            #l.debug("scale by %2.2f %2.2f" % (xScale, yScale))
            self.translate(-x + x/xScale,y - (y+h)/yScale)
            #l.debug("translate by %d %d" % (- x + x/xScale,- y + y/yScale))
            #l.debug("draw at %d %d with page height %d" % (x+chSize.x(), y-chSize.y()+chSize.height(),self.height))
            #ch = unichr(ord(ch))
            self.canvas.drawString(x-chSize.x(), -(y+chSize.y()), ch)
        except Exception, ex:
            l.exception("drawing to fit (pdf) %s (%s)",repr(ch),ex)
        self.restore()
        self.setFontFamily(saveFontFamily)

    def end(self):
        pass

class Font:
    def __init__(self):
        self.__bold = False
        self.__italic = False
        self.__pointSize = 8
        self.__family = config["pdffont"]

    def setBold(self, bold):
        self.__bold = bold

    def setItalic(self, italic):
        self.__italic = italic

    def setPointSize(self, size):
        self.__pointSize = size

    def pointSize(self):
        return self.__pointSize

    def setFamily(self, family):
        self.__family = family

    def family(self):
        return self.__family

    def __repr__(self):
        return ("<Font(pdf) %2.2f %s >" % (self.__pointSize, self.__family))
    
class FontMetrics:
    fh = file(os.path.join(config["kayalidir"],'pdf',config["pdffont"] + '.afm'))
    afmTimes = AFM(fh)
    fh = file(os.path.join(config["kayalidir"],'pdf','Symbol.afm'))
    afmSymbol = AFM(fh)
    # TODO: italic and bold etc

    def __init__(self, font):
        self.size = font.pointSize()

    def boundingRect(self,text):
        afm = FontMetrics.afmTimes
        # TODO: check for symbol font etc
        #l.debug("text is " + text + " " + text.__class__.__name__)
        for i in range(len(text)):
            c = text[i]
            if c in replacements:
                # TODO: mixed greek and latin
                #l.debug("replace with " + repr(replacements[c]))
                #text = text[:i+1] + replacements[c] + text[i+2:]
                afm = FontMetrics.afmSymbol
                text = replacements[c]
        try:
            w, h = afm.string_width_height(text)
            return Rect(0,0,w*self.size/1000.0,h*self.size/1000.0)
        except Exception, ex:
            l.exception("getting rect for %s (%s)", text, ex)

    def charBoundingRect(self,char):
        #l.debug("char is " + repr(char))
        afm = FontMetrics.afmTimes
        if char in replacements:
            char = replacements[char]
            afm = FontMetrics.afmSymbol
        if len(char) > 1:
            raise Exception("invalid char " + repr(char))
        try:
            x1,y1,x2,y2 = afm.get_bbox_char(char)
        except Exception, ex:
            l.exception("getting char bound rect for %d %x (%s) ",char,char,ex)
        # we want x,y,w,h
        return Rect(x1*self.size/1000.0,y1*self.size/1000.0,(x2-x1)*self.size/1000.0,(y2-y1)*self.size/1000.0)

class Rect:
    def __init__(self, x,y,w,h):
        self.__x = x
        self.__y = y
        self.__w = w
        self.__h = h

    def __repr__(self):
        return "<Rect (pdf) %s %s %s %s >" % (self.__x, self.__y, self.__w, self.__h)
    
    def width(self):
        return self.__w

    def height(self):
        return self.__h

    def x(self):
        return self.__x

    def y(self):
        return self.__y
