#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <sys/types.h>
#if defined (_WIN32) || defined (_WIN64) || defined _MSC_VER
#include <io.h>
#else
#include <unistd.h>
#endif
#include <fcntl.h>
#include "netcdf.h"
#include "netcdf_mem.h"

#undef MEM
#undef DISKLESS
#define USEINT

#undef DUMP
#undef DEBUG

#ifdef MEM
static int mem = 1;
#else
static int mem = 0;
#endif

#ifdef DISKLESS
#define MODE NC_DISKLESS
static int diskless = 1;
#else
#define MODE 0
static int diskless = 0;
#endif

#define PATH "tst_diskless5.nc"

#ifdef USEINT
#define T int
#define NC_GET_VARA nc_get_vara_int
static const char* type = "int";
#else
#define T float
#define NC_GET_VARA nc_get_vara_float
static const char* type = "float";
#endif

#define    HRAPY  200
#define    HRAPX  333

static T amountofprecip[HRAPY*HRAPX];

#ifdef DEBUG
static void fill();
#endif

static size_t value_count[] = {HRAPY, HRAPX};
static size_t start[] = {0, 0};

int
main(int argc, const char* argv[])
{
    int status;
    int ncid;
    int rh_id;
    T rh_vals[HRAPY*HRAPX];
    int i;
#ifdef MEM
    int fd;
    char buffer[1000000];
    size_t count;
    off_t off;
#endif

    fprintf(stderr,"PATH: %s\n",PATH);
    fprintf(stderr,"FLAGS: MEM=%d DISKLESS=%d TYPE=%s\n",mem,diskless,type);

#ifdef DEBUG
    fill();
#endif

#ifdef MEM
    fd = open(PATH,O_RDONLY);
    if(fd < 0) {
	fprintf(stderr,"could not open foo.nc\n");
	assert(0);
    }
    off = lseek(fd,0,SEEK_END);
    lseek(fd,0,SEEK_SET);
    fprintf(stderr,"off=%lld\n",off);
    count = (size_t)read(fd, buffer, sizeof(buffer));
    fprintf(stderr,"count=%u |rh_vals|=%u\n",count,sizeof(rh_vals));
    close(fd);
#endif

#ifdef MEM
    status = nc_open_mem(PATH, MODE, count, buffer, &ncid);
#else
    status = nc_open(PATH, MODE, &ncid);
#endif
    if(status != NC_NOERR) {
        fprintf(stderr,"%s\n",nc_strerror(status));
	assert(0);
    }
    status = nc_inq_varid(ncid, "amountofprecip", &rh_id);
    if(status != NC_NOERR) {
        fprintf(stderr,"%s\n", nc_strerror(status));
        assert(0);
    }
    status = NC_GET_VARA(ncid, rh_id, start, value_count, rh_vals);
    if(status != NC_NOERR) {
        fprintf(stderr,"%s", nc_strerror(status));
        assert(0);
    }
    nc_close(ncid);
    fprintf(stderr,"|amountofprecip|=%d\n",HRAPY*HRAPX);
    for(i=0;i<HRAPY*HRAPX;i++) {
	if(amountofprecip[i] != rh_vals[i]) {
	    fprintf(stderr,"mismatch: %u\n",i);
	    exit(1);
	}
    }
    return 0;
}

#ifdef DEBUG
static void fill()
{
    int i,j;
    for(i=0;i<(HRAPY*HRAPX);i++) {
	amountofprecip[i] = (T)i;
    }
#ifdef DUMP
    for(i=0;i<HRAPY;i++) {
      for(j=0;j<HRAPX;j++) {
#ifdef USEINT
	printf(" %d",(i*HRAPX)+j);
#else
	float f = (float)(i*HRAPX)+j;
	printf(" %f",f);
#endif
      }
      printf("\n");
    }
#endif
}

#else
static T amountofprecip[HRAPY*HRAPX] = {
  -47, 2, -36, -48, 78, -23, -99, -48, -4, 90, -47, 80, -20, 35, -63, -45,
    52, -40, 58, -52, 75, 56, -80, -26, 72, 18, -28, -12, -79, -23, 34, 24,
    5, -11, 73, -8, 40, -69, 46, -20, -6, 23, -36, 96, 34, 42, 27, 91, 72,
    -26, 38, 47, -22, 41, -77, 31, -61, 36, -10, 68, 71, 24, 25, 44, -25, 49,
    -7, -20, 97, 0, -13, -21, 58, 56, -50, 75, 35, 53, 5, 96, -5, -60, 36,
    75, -72, 19, -58, -72, -75, 16, 22, 73, 34, 92, 56, -50, -6, 25, 79, -63,
    -61, -40, -37, -99, 53, 49, -57, 5, -17, -100, 68, 0, -87, -88, 38, 77,
    -83, -81, 48, -73, 82, 92, -77, 63, -8, 99, -83, 27, 64, 17, 73, 79, -66,
    45, 71, -53, 40, 86, -84, -8, 20, -94, -100, 60, -87, 92, -73, -32, -74,
    66, 47, -32, -82, -14, -21, 84, -100, -41, -7, -19, -50, 57, -76, -95,
    85, -16, 35, 72, -11, -45, 40, 53, -64, 61, 17, 66, 44, 82, 45, -37, 9,
    66, 89, -13, -1, -32, -78, 62, 10, 80, 93, -65, -57, -92, 78, 32, 4, -51,
    66, -5, 36, 82, 33, -100, -34, 28, 74, 32, 84, -6, -67, -39, -61, 52, 53,
    60, -80, -89, 60, -22, 58, 72, 20, -28, 19, 2, -89, -40, -86, 60, -5, 68,
    26, 14, 86, -33, 64, -67, -26, -50, -7, -65, 8, -40, -19, 10, -93, -33,
    -60, -30, -85, 22, 25, 43, -59, 82, -45, 43, -49, -65, -31, 50, 21, -87,
    -41, -72, 8, 28, 75, 56, 28, -72, -32, -84, 76, -42, -78, -16, -94, -16,
    57, 73, -22, 54, 32, -100, -87, 39, -41, -1, 87, -51, -95, -88, -28, 25,
    30, -6, -83, -60, -16, -27, 7, -46, 29, 83, 2, 57, -78, -26, -96, 64, 40,
    -60, 11, -78, -41, -82, -44, -82, 48, 85, 52, 36, 81, -33, -20, -62, -67,
    9, -53, -96, 52,
  -98, 7, -49, -59, 89, 86, -68, -77, -44, 69, -79, 85, 55, 35, 21, 44, -33,
    83, 28, -96, 79, 31, 34, -76, 9, 61, -60, 12, 94, -5, 13, -85, -88, -28,
    -81, 61, -13, 5, 58, -44, 22, -14, -20, -47, -44, 98, -16, 8, -18, -22,
    -63, -15, -85, -67, -61, 66, 2, 4, -12, -26, -1, 29, 25, 21, -52, -55,
    91, 42, -11, -36, 26, -60, 10, 19, -93, 71, 7, 49, 69, -58, -77, 72, 46,
    -61, -90, -43, 78, -52, 95, 89, 90, -97, -96, 31, -79, 42, 53, -54, 4,
    -32, -56, 9, 68, 19, 40, -17, -26, 88, -21, 66, -20, -23, -22, -84, -69,
    12, -80, -60, -88, -82, 30, 34, 72, 46, -3, 96, 48, 91, -67, 95, 53, 92,
    -40, -22, 35, -27, 99, -78, -14, 21, 39, 52, -6, -22, -93, 87, -79, -74,
    0, -50, -5, -63, 50, -31, 78, -66, 61, 31, 80, 99, -72, 10, 56, 99, 84,
    -26, -76, 22, 53, -87, -88, -98, -39, -57, -43, -97, -93, -22, 52, -88,
    0, -39, -95, -12, -1, 20, 90, -78, -42, -70, 15, 89, -84, 39, -59, 82,
    -16, -35, -95, -23, -49, -90, 86, -10, 39, -7, -83, 18, 7, -31, -9, -47,
    -13, -48, -80, 33, 69, -1, 89, 81, -36, -28, -74, -97, -87, -61, 55, 77,
    28, -86, 4, 77, 35, 44, 98, 64, 60, 74, 15, -67, 58, 95, 58, -53, -15,
    76, -89, 62, 4, -80, 16, -7, 17, -96, -78, 32, -91, -29, -49, -59, -98,
    89, -1, -32, 24, 67, 59, -47, -49, 36, -57, -55, -80, -26, 7, -9, 41,
    -12, -79, 60, -60, 46, -67, 2, -94, 84, 36, 44, -28, 31, -46, 13, 30, 0,
    -2, -27, -59, 18, -21, -42, 18, -46, -33, 37, -77, 17, -86, 37, -41, 53,
    -17, -98, -1, -88, 4, 46, 42, -80, -61, -34, -60, 35, -94, 29, 16, -82,
    -47, 40, 21, 80, -82, -44, 65,
  -46, 4, 35, 21, -10, -49, -77, 56, -96, 11, -43, -11, 87, 13, -43, 8, 14,
    11, -21, 82, 66, -12, -89, -2, 86, 2, -29, 51, 98, -7, -42, -30, -45,
    -48, 74, -13, -30, 78, -14, -46, 7, 49, 72, -16, -65, 53, -72, -7, 25,
    82, 83, 59, 95, -53, 75, 59, -81, -21, 13, 67, 76, 18, -70, 87, 32, 40,
    -77, -46, 29, -93, -71, 59, 53, -13, -52, 83, -76, -24, -34, -86, 93,
    -54, 69, 71, 53, -77, -17, -20, 95, 38, -83, -71, -44, -71, 31, -32, -11,
    -57, -93, 0, 54, -8, 92, 79, 84, 84, 6, 18, -86, -80, 74, -91, 98, 81,
    77, -91, -50, -78, 6, 39, -17, 50, -43, 14, -72, -33, -50, -60, 74, -63,
    -19, -75, -91, -24, 99, 50, 1, -82, 1, 42, -26, 20, 74, -78, 3, -7, -67,
    14, -13, 65, -67, -33, -61, -81, -4, 0, 28, 81, 12, -76, -13, 53, -80,
    16, 54, 37, -18, -99, 22, -49, 58, -20, 37, -30, 58, -38, 23, 98, -27,
    53, 48, -2, 62, -91, -5, -77, -23, -60, -61, -55, 67, 8, 9, 47, -2, -66,
    -27, -91, -16, 35, -62, -18, -71, -24, 5, 20, -6, -65, 84, -70, 72, -47,
    64, 70, 30, -24, 60, 5, -59, 35, -100, -87, -24, -45, 36, 45, -84, -73,
    41, 33, 50, 23, 75, 87, 99, -26, -37, -26, -73, -58, -64, 87, 32, -96,
    86, -23, -82, 38, -52, -15, -47, 15, 99, 83, -64, 43, 98, -57, -77, -8,
    -76, 8, 79, -72, 99, 94, -8, 84, 8, 31, -55, -23, 52, -14, 62, 74, -97,
    48, -67, 67, -76, -1, -38, -10, 87, 58, 42, 26, -13, 13, -59, 15, 69,
    -66, -10, 79, 16, 86, -13, 41, -65, 46, 57, 30, 48, -100, -4, -5, -24,
    89, -71, -31, -36, -60, -7, 49, 2, 18, 89, -6, 52, -76, 24, -64, -49,
    -93, -92, 81, -4, -64, -14, -79, 10,
  1, -3, -38, 45, 20, 49, -67, -7, 19, 82, -65, 96, -1, 58, 34, -8, 90, -79,
    22, 65, 29, -84, 21, -33, 79, -30, -74, 60, 41, -49, 61, -8, -76, 51,
    -76, -54, -24, -97, -30, 21, 87, 20, 29, -94, 79, -16, -70, -27, 95, 60,
    -47, -83, -7, 74, -65, -39, 38, -43, 94, -79, 35, -58, 36, -39, 73, -89,
    85, -45, -88, 27, 77, -60, 29, 8, 74, 74, -17, 97, -84, 72, 96, 63, 41,
    -64, 42, 8, 76, -40, 80, -96, 73, 68, 87, -3, -25, 32, 83, -44, -29, -42,
    25, -44, 48, -15, 74, -62, -89, -47, -61, 49, 35, 76, 49, -80, 75, -64,
    43, 13, 63, 32, 88, -37, -69, 90, -31, 24, -7, 39, -100, 39, -95, -65,
    -21, 8, -54, 80, -88, 72, -11, -24, 27, 69, 89, 91, -94, 15, -95, 96, 37,
    69, 26, -20, -3, 93, -67, 84, 40, 48, -78, 10, -95, 31, -8, -33, -71, 23,
    -52, -59, 60, -91, -21, 73, -14, -92, 22, -100, -45, 95, 49, -19, 12,
    -41, 36, 46, -64, -51, -100, 81, 8, -79, -8, 55, 33, -49, -42, 64, -31,
    -68, 64, 43, -60, 27, -66, -29, -3, -100, 91, -100, 33, -92, -48, -81,
    -15, -35, -8, 34, -87, -32, 33, -77, 47, -80, 3, 84, 60, -6, -18, 50,
    -79, -89, 16, 94, 44, -4, 80, 6, 57, 71, -69, -26, -7, -53, -59, 91, 93,
    92, 54, 91, 2, 18, -48, 13, 7, -92, -62, 59, -49, -1, -51, 82, 36, -15,
    -1, 98, 8, 74, 98, -84, -51, -86, 40, -74, 53, -42, 0, 63, -67, 13, 32,
    -64, -74, 41, 8, -90, 64, 18, -78, -78, 72, -36, 63, 0, -3, -83, -40, 0,
    -7, -93, -61, 1, 2, 94, 77, 96, 35, 28, 18, -3, 58, -49, -81, 90, 89,
    -23, 6, -33, -80, 86, -51, -11, 51, 12, -9, -54, 86, -33, 43, 41, -52,
    76, -25, -14, -18,
  43, -95, 27, -41, -28, 46, 10, -88, 4, 52, -72, -7, -67, 94, 0, -5, 85, 82,
    8, 66, 80, -54, 8, 18, -77, 62, 93, 86, -35, 86, 6, 69, -30, 31, 70, 8,
    23, -77, -14, -51, 71, -6, 56, 86, -53, 56, 74, -48, 94, -31, 17, -25,
    46, -12, 27, 8, 95, -21, 72, -57, 32, -13, -23, -49, -71, 61, 96, -11,
    12, -78, -24, -24, 59, -66, 28, 63, -40, -76, -95, 99, 44, 23, -20, 3, 1,
    -74, 76, -25, -52, 26, -31, 82, -85, 14, 66, -42, -87, -16, 6, -68, 34,
    -22, -3, -75, 90, 49, -40, -83, 91, -48, 13, 95, -34, 8, 75, -35, 74,
    -71, 6, 59, 19, 22, -97, 24, -23, -39, 6, -11, 83, -9, -33, 48, -32, -56,
    -41, 87, -64, -51, -72, -54, -46, -83, 19, -44, -96, -39, -74, 1, -38,
    40, -46, 70, -27, -83, 59, 43, -93, -5, 6, -23, -72, -89, 65, -33, 75,
    -1, 45, 22, 39, -86, -97, 41, 0, 20, 14, 75, -30, -36, 7, -28, 97, 19,
    -90, -70, -52, 99, 55, -87, -69, -61, -70, -20, 31, -5, -80, 40, -10, 2,
    -82, -78, 94, -73, -71, -75, 92, 83, 21, 36, 54, -72, 9, 1, -99, -49, 19,
    -79, 62, -42, -25, -19, 78, 20, 82, 89, -3, -9, 85, -99, -68, 91, 27, 31,
    -97, -45, -75, 80, -89, -37, -97, -37, 25, 6, 49, 51, -74, 87, 91, -3,
    -90, 63, 2, -39, -72, 27, 55, 84, 70, 4, 39, -88, -16, 61, 77, -30, 22,
    -75, 40, 63, 71, 62, -37, -99, 2, -66, 58, -20, -12, 85, 37, 73, -16, 31,
    7, -71, 65, -93, 41, -66, -48, -5, 27, -19, -26, 82, 55, 54, 60, 75, 44,
    -14, -89, 31, 46, -84, 92, -90, 80, 55, 56, -92, 89, 38, -11, 47, 87,
    -98, -69, 1, -57, -87, -76, 41, -22, 69, -100, -40, 46, -31, 14, 16, 24,
    43, 77,
  13, -11, -88, -34, -19, 8, -55, 56, 35, -10, 71, 35, 93, 89, 62, 93, -92,
    -13, -43, 7, -78, 29, -48, 28, -69, -74, -74, 98, 4, -34, -90, 26, 9,
    -82, -69, -92, 19, -70, -55, 42, 51, 99, 15, 19, -21, 89, 56, 58, 17,
    -51, 9, 1, -91, -30, -21, 13, 95, -38, 38, 40, 54, -32, 90, -19, 50, -71,
    -33, -37, 28, 68, 20, -10, 66, 3, -29, 42, -5, 80, -7, 42, 86, 77, 67,
    43, -83, 9, -54, 12, -80, -87, -44, -1, 99, -38, 18, -7, 20, 10, -94,
    -47, 63, -86, -24, -37, 73, 95, -28, 73, -9, -86, 97, -43, 93, 65, -91,
    -16, 45, -27, -78, 43, 6, 4, 59, 6, 15, -75, 96, -27, -75, -88, -99, -74,
    -43, 57, 7, -47, -19, 83, -78, 2, -69, 21, -5, 57, 94, -35, 81, 56, 76,
    -99, -25, -26, 42, 43, -49, -28, -47, -33, 14, 25, 45, 15, 52, -78, 49,
    -11, 68, 53, -42, -66, -64, 27, -57, 8, -83, -80, 64, 28, 34, 33, 55, 23,
    -65, 58, -52, 30, 8, -85, 17, 8, 17, -32, 63, 70, 77, 29, 88, 98, 37,
    -44, 18, 39, 20, -98, -27, 35, 75, 59, 92, 98, 97, 20, -8, -62, -85, -39,
    -24, 97, -74, 63, 17, -74, -27, 74, -18, -56, -41, -96, -63, -16, 75,
    -42, 59, -5, 39, 59, 23, -36, 50, -46, 58, -25, -12, 3, -35, 30, -46, 45,
    10, 82, -93, -20, -85, -98, 80, -58, 0, 56, 81, 39, 16, -58, -5, -71, 37,
    -68, -42, -85, -87, 4, -10, 77, -17, 14, -38, -72, -22, 73, -63, 44, -19,
    27, -77, -98, -19, -86, 48, 71, 5, -66, -21, 59, -66, -8, -88, 46, -13,
    -1, -39, 9, 67, 92, 32, -81, -53, -35, 50, -35, -53, -68, -72, 63, -5,
    -68, 23, -16, -3, 36, 34, -45, -68, 43, 33, -61, 47, -1, 96, 86, -6, 69,
    28, 60, -65,
  -78, 3, 9, 97, 63, -76, -56, -100, -28, 40, 5, 14, -49, 13, -62, -71, 68,
    50, 61, 67, -97, 32, 2, 50, 1, 41, 12, -53, 17, -50, -94, -13, -36, -1,
    -41, -82, -70, -28, -72, 99, 82, -32, 48, -20, -17, 77, 31, -1, 90, -79,
    -48, -83, 65, 80, 83, 77, -28, 22, 3, -18, 69, 31, -14, 10, -49, -13,
    -16, 98, -58, -59, -53, -7, 52, 45, 87, -62, 28, -62, -3, 1, 8, 46, -84,
    56, 88, -38, -77, -73, -57, -64, 33, 83, 21, -33, 11, -52, 52, -39, -74,
    -42, -38, -65, 72, 19, -18, 26, -55, -6, 0, 94, 98, 56, -4, 98, 7, -45,
    26, -91, 91, 53, 51, 63, -25, -37, 43, -41, 46, -47, 31, 13, -47, 1, 97,
    -34, 45, 49, 81, 67, 72, 52, -57, 74, 99, 5, -78, -22, -79, -79, 30, -98,
    -11, -74, 68, 44, -76, 13, 5, -37, 4, -47, 73, -87, -44, 10, 21, -92,
    -26, 55, 44, -52, -2, -31, -28, -40, 42, -92, 62, 4, 36, 1, 63, -68, 3,
    -6, -10, 62, -9, -99, -91, -81, -92, -56, 71, -65, 72, 86, -37, 59, -36,
    13, 48, -31, -44, 11, -67, 70, -50, 50, -56, 52, 12, -67, -3, 9, -41, -8,
    24, 50, 10, 75, 36, 2, -25, -59, 33, -53, 90, -95, 92, 78, -28, 37, 20,
    -58, 46, 65, -100, -59, 73, -20, -4, -7, 2, -67, 72, 25, 52, -55, 40,
    -51, 75, -80, -22, 49, -79, 14, 78, -19, -27, 91, 17, -25, 41, 34, -11,
    -94, -94, 37, -98, -61, -59, -30, 8, 61, -15, 29, -65, -54, 9, -14, -38,
    62, 49, -27, -35, 96, 93, 17, 70, -59, 63, -19, -21, 75, -50, -5, -44,
    18, -48, 30, -3, 67, 7, -31, -84, 97, 73, -41, 40, -62, -41, -94, 95,
    -53, 80, -75, 2, 77, -21, -52, 41, 13, -40, 97, -76, -39, -18, 93, -97,
    43, 94, -98, -92,
  -56, 36, 82, -94, 53, 2, 66, -63, -72, 7, -39, 72, 69, -44, 12, -9, -66,
    -20, -81, -94, -44, 8, -61, -59, 89, 58, 57, -43, -57, 71, -66, 45, -81,
    47, -97, 24, 52, -14, 60, -12, 33, -71, 91, 45, -35, 2, 84, 62, -93, 26,
    -99, 3, 6, 9, -50, 76, -37, -19, 83, -78, -56, -64, -93, 34, 46, -17,
    -12, 73, -22, 41, -55, 25, 75, -24, -53, 88, 39, 58, -92, -83, -30, 81,
    -20, 64, 97, -54, -30, 15, -72, -49, 15, 5, 19, 3, 45, 7, -62, 76, 59,
    75, -97, 56, 38, 14, -28, -83, 82, 27, -96, 80, -33, -12, -5, 57, -47,
    56, 40, -72, 15, 73, -46, -42, -42, -23, 57, 44, 91, -68, -9, -50, 56,
    94, 29, 2, 4, -46, 76, -22, 93, -38, 98, 29, -85, 88, -32, -92, -83, 72,
    6, -44, -59, 11, -60, 89, -6, 35, 40, -9, 71, -30, 82, -30, -74, 89, 59,
    53, 90, 54, 25, -51, -76, 55, -92, -8, -52, 86, -43, -67, 50, -12, 12,
    36, -83, -12, 52, -49, 32, 92, 19, -50, 7, 70, -64, 0, 89, 71, 9, 35,
    -92, -87, -90, -33, 54, -26, -44, 38, 6, -2, 92, -15, -9, 4, 91, 35, -86,
    11, 48, 95, 31, 44, -5, -97, 19, 92, 25, 46, 29, -42, -51, -1, 89, -12,
    15, 18, -11, 77, 12, -97, -21, -49, -50, -53, 17, -13, 93, 29, -67, -71,
    18, 85, -28, -74, -18, 86, 12, -100, -99, 29, 17, 46, 58, 77, 9, -89, 37,
    1, -60, 79, 72, -18, -74, -76, -71, 89, -94, 91, -100, 8, 29, -32, 61,
    -66, -19, -35, -62, -7, -36, 90, 96, 55, 6, 76, -78, 70, -80, 25, -93,
    50, -9, 74, 39, 17, -92, 67, -57, -21, 40, 79, 18, -58, 92, 20, 3, -23,
    -64, -89, 35, 46, 53, -75, -2, -34, -27, -86, 27, 85, 67, 45, 14, -94,
    54, -32, 58,
  79, 74, 33, 3, 35, -11, -20, 61, -41, 35, -3, 81, -62, -9, 24, -44, 50, 68,
    -33, 42, 30, 47, 82, -8, -7, 12, -95, 69, 85, -17, -65, 77, 51, -25, -49,
    -88, -30, -7, 52, -63, 43, 34, 64, 37, 62, 50, 25, 84, 91, 53, 41, 80,
    15, 78, 86, -45, 4, 14, 69, 98, -6, 61, 22, -73, -23, 93, 86, -1, 99, -7,
    -20, 87, -26, 77, 27, 51, 29, -93, -91, 13, 66, 63, 12, 4, 25, 24, 69,
    -36, -10, 96, 85, -73, 59, 45, -22, 46, -100, 53, 39, -95, -7, 29, -38,
    92, -35, 46, 98, -93, 11, 41, -22, -55, 42, -59, -15, -51, 85, -12, 91,
    -5, 94, 78, 79, 59, 45, -41, -51, 37, 9, 10, -89, -61, 6, 94, 63, -31,
    65, 73, 74, -2, 93, 98, -55, 21, 8, 92, 30, 22, 2, -37, 0, -67, 78, 75,
    -2, -87, 10, -86, 2, -19, 43, -62, 5, -68, -54, -57, -12, -95, 94, -72,
    -27, 61, 3, 46, -7, 39, 58, 50, 58, -15, 14, -39, 99, 12, 70, -47, 98,
    -17, 75, -36, 77, 84, -63, -21, -85, 87, -59, 76, -32, 51, -48, 37, 79,
    93, 36, 1, -44, -87, 60, 34, -26, 70, -53, 72, -92, -93, 1, -77, -93,
    -48, -32, 53, 20, -63, -62, -39, 76, -100, -12, 94, -64, 99, 98, -91,
    -85, 49, -76, 58, 53, -5, -36, -41, 4, -61, 24, 85, -63, 72, 41, -84, 10,
    19, 15, 63, 94, 16, -80, -22, 84, -99, -61, 51, -37, 21, 27, -67, -18,
    94, 64, -2, 76, -87, -88, 21, 53, 58, -10, 16, -74, 82, -42, -42, -76,
    -14, 85, -82, -1, 34, -10, 64, -41, 87, -6, 84, 77, 85, -47, 27, 36, 89,
    13, -83, -37, 12, 10, 47, 43, -75, -41, 73, -14, -2, -10, 86, 93, 1, -17,
    66, -60, 99, 73, 24, 65, 2, -77, -82, -57, 3, 24, -35, -79, 9, 3,
  2, 47, 22, -45, 77, -45, -81, -69, 63, -7, 11, 4, 41, 14, -57, -17, 47, 98,
    13, 91, 63, -8, 41, -47, -28, -45, -88, 44, -63, -19, 30, -16, -41, 94,
    -55, -91, -93, -27, -85, -49, 65, -38, 38, -17, 18, 78, -81, -16, 32, 9,
    -76, -57, 80, -11, 72, -71, 87, -95, 85, -8, 85, -9, 49, 1, 1, -10, -27,
    14, -47, 44, 50, -10, -66, 7, 63, 12, -99, 27, 15, -20, 72, 65, 71, 71,
    17, 95, 47, -98, -8, -13, 0, 15, -29, -90, 17, -60, 56, 19, 37, -66, -34,
    -58, 81, -75, -47, 13, 95, 95, 22, -25, -44, 54, -72, 66, 46, 94, -6,
    -75, -61, 36, -81, -50, 59, -40, -32, 27, 90, 18, -48, -73, 81, -46, -91,
    57, 89, -46, -18, 59, -58, -99, 33, -41, 23, 68, -22, 17, 93, -57, 3,
    -74, 27, 46, 71, -54, -30, -88, -82, -73, -87, -29, 92, 14, -30, -63,
    -54, -15, -1, -55, 34, 40, 77, 18, -86, -97, 56, -43, -72, -42, -63, -54,
    -9, 54, 63, 34, -75, -84, -3, 60, -88, 28, -29, 56, 0, -38, 71, -81, -68,
    38, 74, -44, 94, 26, -53, 33, 90, 72, 20, 10, 48, 97, -48, 55, 11, -27,
    -92, -73, 24, 42, -12, 37, -89, -71, 60, 92, 68, 36, 86, 70, 63, -88, 0,
    -9, 21, -28, 56, 65, 33, -58, 45, -54, -83, 22, -3, 41, -28, -89, 36,
    -80, -75, 3, -21, -98, -51, -73, -82, -26, -86, -26, -55, -16, -64, -84,
    -55, -73, 31, -29, -36, 15, 39, -27, -79, 9, 8, 21, -77, 6, 42, -1, -70,
    -95, 73, -100, 46, -34, 36, -66, -5, -39, -45, -52, -56, -96, 91, -34,
    69, 24, -89, 0, -11, 91, -41, -86, -8, -44, -10, -24, -10, 43, -30, -29,
    -76, -19, 61, 76, 29, 83, -28, -15, 66, 8, -26, -74, -57, -21, 32, 61,
    -49, -57, -90, 50, 54, -33, -68,
  -13, -4, 35, 3, -17, 74, -74, -56, -77, 21, -5, 72, -43, -29, -54, 22, -47,
    5, 92, -12, 13, 68, 15, -57, -34, 20, 70, -36, 63, 94, 12, 14, 12, 27,
    -33, -85, -72, 11, 99, -84, -85, -23, 82, 63, -80, 72, -4, 19, 60, -44,
    92, 75, -55, -84, -70, -29, -94, 21, 40, 71, -81, 88, 66, 58, -40, 81,
    -15, 32, 82, -56, -17, 44, 62, 18, -29, 72, -70, -79, 77, 22, 95, 59, 80,
    -73, -58, -35, 60, 89, -82, -47, -18, 45, -61, -97, -14, -24, 13, 43, 66,
    -44, -18, -48, -7, 26, -56, 66, 34, -62, 23, 44, 69, -85, -5, -57, 93,
    -54, -26, -92, -17, 54, -79, 52, -48, 74, -38, 89, 23, 58, 93, 4, -31,
    69, 5, 82, -42, 9, 57, 60, -27, -22, -66, -26, 8, -44, -88, 68, -60, 26,
    88, 37, -90, -80, -38, 2, 41, -11, -40, -16, 1, -13, -27, -19, 36, 78,
    -47, 80, -94, -68, 23, 2, 97, -62, 49, -49, -14, -92, 63, -64, -55, -71,
    40, -84, 79, -1, -76, 47, -96, -36, 15, 47, 30, -17, 22, -85, 96, 69,
    -40, 67, -9, -46, 15, 38, -86, 56, -55, -76, 94, -62, -9, -25, 46, 80,
    -1, -97, -68, 0, -21, -24, 58, -28, -74, -90, -8, -16, 9, 36, -90, 91,
    64, -70, -31, -28, -11, -62, 35, 99, 15, 25, -48, 92, 97, 13, -58, 38, 8,
    -59, 38, 35, -43, 73, 39, -80, 41, 19, 12, 97, 44, -75, 58, 40, -91, -80,
    -85, 14, 97, 11, 13, -6, 25, 39, -10, 14, 47, -34, -49, 12, 49, 26, 10,
    11, -78, -7, -92, 47, 93, -27, 57, -41, -22, -78, -30, 4, -30, 3, -70,
    14, 75, -76, -31, -22, -88, 65, -61, -31, -95, -96, 59, -14, -28, 2, 81,
    96, -6, -13, 19, -90, 39, -51, -33, -91, 95, -34, 79, 31, -96, -60, 71,
    25, 3, 27, 45, 30, 58,
  -88, -41, -8, -60, 48, 88, 51, -57, -82, -37, 74, -51, 72, 6, -82, -34,
    -39, -93, 26, -94, -98, -67, 45, -36, -68, -57, 10, -100, -40, 38, -88,
    -1, -89, 59, -98, 35, -2, -60, -91, 32, -57, -30, 17, -18, 87, 6, -19,
    -54, 14, 33, -55, 54, 70, 35, -53, 41, -64, 24, -46, -32, -69, 70, 0, 90,
    19, 45, 28, -68, -47, -49, 83, 38, -79, 0, 99, -17, -83, 71, -11, -17,
    -93, 14, -33, -41, -39, 80, -78, -70, -22, 49, -68, -74, -28, 66, -75,
    96, 6, 21, -9, 21, 33, -69, 40, -47, 54, 13, 74, -84, -57, -48, -23, -13,
    23, -90, 80, -29, -78, 97, 96, -57, -58, 89, 52, -13, 0, -57, -48, -45,
    -88, 1, 74, -86, -91, -95, -85, -34, 75, -52, -20, -57, 64, 49, -29, 88,
    -45, 25, -26, 82, 77, -33, 58, 10, 30, -41, 62, 80, -8, 95, 7, -21, -25,
    -48, 78, -2, -72, -98, -53, 51, 25, -11, 63, -24, -87, 75, 17, 30, 59,
    30, -53, -34, 91, 54, -78, -43, 93, -40, -44, 5, 99, 67, 8, -36, 39, 87,
    -28, 22, -66, -99, -32, 85, -94, -80, -8, 17, -7, -81, -69, 34, 94, 86,
    -7, 62, -71, 53, 96, -35, 28, 21, -34, 51, -39, -5, 68, 17, -95, 34, 43,
    43, -41, -20, 33, -2, 59, -83, -55, 62, -80, -48, -83, -53, 11, -36, 69,
    1, 30, -93, -54, -73, 15, 20, 81, -30, 55, -94, -96, 12, -93, 68, -100,
    -18, -73, 71, -82, -33, 88, 49, 87, -2, -7, 19, -23, -97, 46, 37, -15,
    -49, 9, 19, 16, 74, 65, -7, 24, -16, 88, -20, 86, -37, -88, -63, 36, -37,
    -22, -20, 21, 84, 8, -1, -23, -77, 91, -90, 2, 35, -52, -66, 13, -40, 38,
    -29, -46, 80, 22, 49, -33, 16, 85, -15, -82, 31, -88, 86, 11, -30, -90,
    -51, -19, -10, -18, -24, -52, -68, -92,
  -8, 56, 89, -65, 6, -49, -39, 93, -46, 15, 77, -27, 47, 53, 45, -39, 37,
    -92, -47, -2, -81, 84, 44, 37, 49, -94, 39, -18, -44, -23, 39, -3, -100,
    69, 48, -2, 44, 25, -59, -44, -93, 25, 91, 25, 13, -54, 9, 63, 8, 86, 76,
    19, -45, 25, 78, 46, -50, -67, 66, -28, -99, -33, 1, 99, -43, 94, 35,
    -53, 78, 66, -13, 9, 4, 63, -27, 35, -43, -31, 39, -60, -97, 66, -3, 53,
    18, -17, -40, -42, -57, 84, 89, 7, -71, -53, -17, -73, 70, 90, 32, 62, 5,
    -11, -93, 87, -5, 3, 66, -2, -1, -41, -43, 68, 50, -3, 34, -56, 35, -51,
    24, 88, -53, 19, 37, 48, 93, 26, -4, -1, 99, 37, -57, -39, -22, 4, -34,
    95, 56, 26, 66, 48, -66, 88, 65, 19, -65, 53, -33, 60, -99, -41, -10, 92,
    -51, -18, 19, -79, -28, 85, 27, -47, -74, -84, -30, 68, 73, -62, -15, 84,
    72, 49, 37, -54, -14, -91, 32, 97, 55, -6, -41, 68, -75, -65, -63, -15,
    -22, 65, 28, -99, -62, -98, 42, -99, 74, -98, -99, -66, -20, 68, 95, -63,
    -24, 44, -40, 89, 56, -73, 10, 58, 0, -73, -6, -38, 99, -38, -19, 43, 99,
    53, 41, -79, 71, 17, 46, 4, -16, -80, 20, -50, -33, -23, 11, 88, -13, 2,
    -81, 55, 53, 43, -37, -99, -53, 82, 26, 98, 65, 7, -92, -3, 5, -30, 59,
    -23, 44, -56, 58, 69, -14, -13, -9, -94, -83, -13, -64, 71, -29, 72, 95,
    39, 58, 64, -81, -93, -33, -69, 95, 88, 26, 49, -90, -13, -72, -42, -22,
    83, -65, -27, -51, 41, 5, 94, -19, -86, 18, 28, 42, 93, -92, -24, 42, 30,
    99, -19, -64, 84, 23, -22, 37, 90, -89, -43, -27, 13, -8, -87, 72, 4,
    -18, 25, -38, 84, -15, 92, -24, 96, -31, 59, -26, -88, 27, 82, 10, 10, 37,
  29, 97, 59, 6, 9, 40, -85, -48, -10, -80, -66, -21, 24, -49, 25, -49, -39,
    9, -56, -39, 56, -70, -46, 65, -63, 80, 77, 6, 28, 40, -57, 4, 32, 6, 94,
    -18, -85, 94, -9, 75, 21, 24, -8, -23, 74, -92, 34, 92, -19, 84, -94, 49,
    33, 29, -9, -9, 94, -8, -2, -68, -40, -14, -14, -20, 57, 41, 4, 33, -66,
    57, 74, -82, 95, -42, 7, -69, -87, -13, -69, 68, -95, -35, 66, 43, -62,
    22, -86, -80, -84, -86, -92, 36, 57, -59, 20, -45, 7, -95, -12, -2, 7, 6,
    -73, -72, 6, -95, -78, 80, 96, -74, -44, 56, -28, 20, 54, -13, -22, -16,
    -63, 69, 11, 76, -39, -78, 61, 60, -42, 49, -39, -95, 60, 85, 76, -89,
    56, 46, -21, 71, -13, 0, 45, 82, 88, 57, -26, -14, -67, -55, -77, -97,
    -50, 12, -78, 27, 60, -50, 78, 72, 93, -70, -61, -2, 81, 62, -87, 75,
    -72, 25, -5, -21, -100, 55, -13, -11, 87, -20, 43, -33, 38, 31, 1, 21,
    -98, 47, 61, 15, 77, -18, 49, -30, 97, 48, -11, 37, -74, 45, -77, 29, 25,
    60, 52, -61, 55, -11, -40, 39, 32, -28, -36, 51, 56, 42, 13, -82, -82,
    -100, -71, 5, -37, 45, 9, -47, 23, -29, -42, 62, 93, -100, -77, 43, -97,
    -86, 25, -80, 73, -74, -99, 3, -74, 96, -43, -69, 58, -51, 41, 78, 3,
    -86, 47, -21, -8, -3, -64, -99, 54, 22, -59, -22, 4, 40, -5, -70, 35,
    -39, -63, -54, 29, 44, 99, 65, 59, -43, -6, -71, -45, 42, 21, -30, -49,
    -11, 77, 74, -19, -43, 53, 8, -35, 87, -59, 40, 67, 21, -23, -1, 18, -52,
    -75, -87, -11, 11, -74, 21, 78, -97, -90, -73, 36, 65, 95, -76, 48, 65,
    48, 41, 95, -36, 83, 15, 45, -81, 58, -32, 0, 92, 68, -26, -91, 96, -89,
    12, -76, 29, -6,
  68, -49, -49, -42, -56, 98, -85, -7, 24, -27, -15, -45, -38, -34, -79, -71,
    96, -61, -48, -90, 11, -8, 15, -45, -34, -18, -17, 30, 75, -73, 88, -94,
    68, 58, 16, 37, -35, 65, 5, -61, -2, 44, -61, -61, -18, 37, -69, -58,
    -100, -55, -46, -40, -96, -54, -65, -65, -93, 55, 28, -81, 14, 43, 18,
    72, 74, -24, 53, -21, -55, 45, -97, -43, -95, -79, -19, -50, 62, -67, 43,
    -92, -18, -97, 22, 56, -17, -20, 18, -77, -79, 92, 0, 74, 16, -2, 61, -3,
    94, 47, 66, -59, 26, -96, -20, -60, 80, 41, 70, -59, -67, -27, 14, -92,
    -39, -85, 90, -95, 80, 27, 42, 77, 27, -79, 11, 4, -100, 17, 74, -57, 37,
    70, -29, -83, 6, -16, 44, 2, 10, -80, -79, -52, -29, 28, 75, -68, -96,
    -4, 86, -34, 18, -76, -21, -58, -59, -1, 54, 38, 11, 34, -11, -82, 66,
    -59, -59, -69, 76, 19, -21, -38, -96, -14, -46, 43, -44, -27, -69, -6,
    -56, 77, 45, 90, -46, 4, -36, 76, 40, -88, -40, -65, 39, 98, 12, 59, 50,
    -83, 14, -18, 89, -33, 74, 81, 96, 37, -90, 50, -24, -54, 34, 88, -97,
    -18, -96, -42, -7, 10, 12, -9, -44, 33, -52, 85, -59, -34, 20, 74, 53,
    44, 63, -99, -3, -24, 36, -62, -9, -26, 91, -54, 42, -37, 98, 42, -48,
    -17, -73, 45, 25, 47, 79, 81, 24, 26, 36, -66, 51, -89, 17, -62, -54, 61,
    23, -85, -25, 65, -18, 27, -17, 4, -27, -99, 36, 82, -26, -87, -73, 32,
    88, 25, 46, -48, -2, 11, -41, 61, 9, -91, -29, -91, -91, -99, 23, -70,
    17, -64, -60, -98, -52, -2, -2, 33, 73, -64, 30, 77, -70, 44, 44, -51,
    93, 64, 23, 93, -98, -52, -67, -2, -91, 94, 41, -76, -6, -16, 27, -14,
    -1, -79, 4, -61, -10, 95, -77, 65, 44, -48, 48,
  -47, 40, 8, -59, -7, 76, 59, 80, -60, -41, -67, 79, 40, -59, 58, -84, 95,
    56, 21, 72, 31, 65, 48, 74, 96, -52, 43, 74, 29, -87, 76, 46, -39, 81,
    90, -46, 67, 64, 63, -98, 71, -83, 6, 47, -15, 63, 30, 17, -5, 7, -83,
    44, 73, 57, -91, 68, 92, -16, 65, -83, 35, -99, 50, 44, 11, -8, 58, 2,
    77, -90, 56, -12, -6, 87, 4, 29, -83, 30, 3, 89, -25, 96, -25, 5, -98,
    -25, 64, 67, -3, -18, 3, -8, -83, -15, -91, 86, 50, -95, -50, 71, -90,
    -32, -80, 60, -21, 42, -73, -13, -81, -9, 92, 85, 19, -31, -79, 25, -18,
    -60, 93, 79, -95, 70, -71, 78, 77, 8, 37, -92, 74, 81, -39, 31, -94, -9,
    -33, 2, 96, 40, -3, -1, -16, 35, 89, 37, -76, -78, 59, 15, -84, -95, -54,
    72, -5, -57, 76, -94, 87, -25, 45, -70, -73, 78, 62, 72, -46, -78, -1,
    -93, -65, -13, 40, 69, 90, -6, 22, -52, -58, -87, -24, -66, -42, -29, -7,
    -100, -29, 19, -11, -49, -33, 98, 80, -87, -89, -24, -29, -96, 93, 14,
    -8, -35, 38, 53, 44, 15, 56, -81, 65, -87, 52, 25, 79, -34, 18, -92, -50,
    63, -53, -2, 45, 16, -52, -28, 86, 62, 72, -52, -40, -79, -27, -63, -76,
    -63, -10, -72, -52, 72, -16, -35, -49, 8, -44, -22, 64, -78, -50, 5, 69,
    29, 61, 93, -52, 44, -95, -77, 72, -1, 68, 13, -97, -97, 21, -91, 28,
    -18, -95, -54, 86, 40, -41, 34, -58, 57, 83, 97, -38, -53, 89, -90, 57,
    -96, 64, -78, 49, 71, 58, 54, -52, 53, 7, 98, 38, -51, -72, 37, 35, 58,
    -83, 71, -4, -83, 19, 96, -33, 53, -52, 73, -9, -31, 56, 35, -74, -69,
    -76, -78, -45, 88, 36, -18, 39, -65, 2, -18, -31, 80, -3, -40, 12, 65,
    12, 51, -49, -94, -63,
  17, 94, 9, 92, 31, 90, 62, 98, -52, 5, -70, -44, 40, 6, -2, 26, 27, 0, -73,
    -82, -36, -76, 25, -45, -44, -10, 59, -39, -94, -11, -30, 99, -83, 26,
    -59, -21, -74, 23, 45, -33, -73, -62, -8, -30, 9, 92, -75, 11, 93, -16,
    -16, 5, 14, -36, 91, -28, 19, -57, -4, -24, -19, -34, -11, -85, 63, 48,
    10, -74, 49, -37, -29, -52, -79, 11, -86, 14, 32, -58, -32, -76, 15, 20,
    92, -2, -32, -23, -95, -99, 46, -38, -66, -4, 8, -94, -45, 45, -37, 25,
    -2, 58, 23, -76, 89, -100, -67, 77, -76, 0, 22, 69, 36, -20, -93, -95,
    53, -44, -15, -100, 58, -93, -72, 23, 32, -61, 18, 30, -22, 62, -3, -15,
    -30, -27, -60, -63, -15, 38, 10, -61, -6, -98, -27, 90, -6, -1, -100, 64,
    -9, -72, 75, 27, -65, -61, 90, -91, 15, -100, -9, 30, 88, 17, -71, 74,
    -44, -10, 86, -67, 5, 5, 41, 27, 52, 51, -93, -28, 6, -40, 3, 21, -57,
    17, 0, 46, -27, 6, -87, 41, 85, 59, 22, 60, -77, -41, -55, 27, 19, 48,
    -26, 9, -53, -70, 52, 52, 97, -26, 38, 10, 18, -62, 28, 36, 74, 35, 8,
    -94, -37, 16, 79, -35, 76, -9, 62, -62, -31, -90, 28, -36, 91, 40, 94,
    91, -49, -34, -6, -9, -3, 61, 2, -91, -28, -70, 46, 13, -50, -96, 52,
    -30, -56, -53, 75, -20, -31, -12, 77, -91, 11, -7, 36, -65, -55, 30, -47,
    -68, -7, 55, 76, 9, 61, -94, -16, -89, 42, 29, -100, 58, 82, -60, -90,
    -46, 53, 40, 18, 63, 99, -28, -14, -90, -79, 95, -44, 58, 3, 61, 46, -9,
    13, -87, -14, -49, -9, 0, -94, 53, 34, 39, 19, -20, 59, -87, -43, -87,
    28, 62, -79, 99, 92, -55, 50, -87, 74, 29, -90, 63, -66, 16, -90, 60, 7,
    -28, 30, 81, -57, -32, 74,
  65, 98, -63, -38, 53, -60, -25, -14, 14, 80, 16, -34, 81, 73, 77, 50, -79,
    -90, 54, 85, -16, -67, 91, 54, -37, -40, -61, -74, -28, -78, -14, 48, -3,
    3, 63, -75, 85, -52, 77, -21, -77, 49, -73, -57, -65, 48, -83, -10, -86,
    67, -83, -55, 84, 22, -32, -64, -81, -4, 60, 89, -21, 49, -89, -6, -31,
    -64, 70, 33, -43, -38, -44, 65, 95, 20, -60, 78, 25, 88, -15, -74, 59,
    -72, -83, 91, 51, 71, -96, 8, 53, -25, 81, 54, 5, 64, 79, -67, 95, -10,
    -69, 97, 71, 94, -87, -24, 16, 99, -13, -67, -21, 93, -28, 21, -25, 0,
    78, -67, 14, 79, 28, 3, 13, -33, -99, 85, 91, -87, -15, -22, -15, -1,
    -95, -66, -70, -59, 93, 47, -5, 52, -58, -47, 94, -47, 37, 58, 88, -57,
    -54, -51, 33, -11, 74, 3, 73, 2, 71, 40, -36, 91, 43, 39, -69, 37, -74,
    -57, 95, 10, -25, 55, 40, -52, -7, -73, -76, 61, 44, -3, 73, -65, -86,
    34, 96, -18, -67, -51, 56, 80, 0, -72, -67, -49, 19, -61, 61, -44, -81,
    -88, -29, -73, -25, -8, -4, -77, -83, 47, -71, -12, 11, -50, -99, 77,
    -96, -64, 55, 37, -96, 54, 94, -15, 90, 2, 74, -99, -63, -57, 2, -33,
    -60, -25, 49, -39, 75, -20, 65, -66, 51, -33, -26, 1, -8, -95, -34, 40,
    -67, -2, 69, 40, 50, 58, 54, -7, 92, -55, 6, 51, 83, -46, 87, -81, 42,
    30, -39, 62, -22, -57, -27, -19, -75, 81, -12, 46, 58, -49, 23, -55, 77,
    67, -42, -40, 79, 74, 62, 77, -47, 82, 35, 28, 30, 77, 20, 85, -32, -47,
    -63, -76, -94, 10, 93, 54, -11, -70, 7, 55, -31, 38, 11, -36, 68, -83,
    -33, -61, -33, -95, -39, 42, 60, -27, -11, -4, 43, 30, 93, -45, -60, 12,
    27, 93, -30, 92, 47, 17, -29, -13, 0,
  -83, -12, -18, 85, 87, 58, 46, -1, 38, -16, 58, 93, -30, 96, 22, -44, 83,
    62, 65, 28, -31, -79, 51, 62, -73, -53, 31, 27, -44, 7, -18, -9, 3, -95,
    11, 59, -15, 53, 51, 64, 52, -85, -58, -95, 85, -28, 17, 81, 77, -78, 22,
    78, -98, 57, -82, 25, -50, 50, 94, 15, -62, -76, 26, -55, -100, -15, 7,
    54, -60, -17, 8, 16, 99, 48, -31, -40, -49, 94, 85, 44, 0, 3, -23, 18,
    -22, -41, -81, -36, 49, 60, -3, -41, 36, 4, 83, -21, 67, 15, -29, -44,
    -72, -64, -19, -47, 45, -78, 26, -41, 43, 57, 2, -13, 84, -9, -36, -71,
    -91, -95, 30, 14, 42, 46, 47, 46, -40, 67, -24, -46, 54, 54, -8, -97,
    -31, -52, 89, -6, 9, -13, -20, 93, 23, -67, 72, -65, 50, -74, -51, -68,
    68, -84, 48, -21, -94, -68, -17, -95, 83, -21, 25, -77, -73, 6, -4, -59,
    -79, -41, -71, -96, -42, 38, -54, -35, -32, -31, -21, 68, -58, 48, 2, 92,
    -10, -32, -10, 19, -22, 4, 20, 30, 94, -31, 29, -91, -81, -78, 90, -68,
    74, -17, -32, 68, -29, 79, -58, -38, -80, -31, 73, -91, 11, 65, 48, -19,
    -93, 36, 48, -60, -63, 60, 61, 50, -49, -43, -78, -48, 78, 7, 55, -32,
    79, -87, 42, -54, -88, -52, 51, 32, 31, 56, 92, 61, -25, 6, 69, -74, 96,
    3, -12, -37, 17, 74, 73, 80, -32, 82, 38, -44, -61, 31, 0, -8, -13, -14,
    -39, 86, -7, -69, -29, 82, 50, 34, 19, 17, 73, 10, 61, 82, -58, -12, -4,
    -53, -76, -9, -33, -11, 84, 72, -29, -13, -45, -70, -15, 47, -52, 84,
    -39, 18, 51, 70, -80, 3, 36, -71, -2, 64, 94, -77, 83, 39, -75, -69, -95,
    15, -87, -53, 98, 11, 70, -34, -37, -64, -69, 97, -93, -54, -41, 99, -18,
    -22, 58, -56, -21, -57, -19,
  84, 79, -12, -29, 77, -7, -48, -76, -72, 3, 43, 69, 60, 2, 90, -8, -94, 67,
    -14, -32, -26, -6, 37, -70, -46, -62, -60, 83, -69, -96, -51, 55, -39,
    79, -82, -50, -14, -74, -10, 34, 95, -19, 15, -35, -34, 47, -95, 10, -61,
    -7, -92, -94, -45, -100, -52, -52, 41, 48, 49, -84, 10, -78, -95, -37,
    -100, -48, 12, -15, 47, 1, -75, -33, 69, -27, 60, -59, 39, -68, -22, 73,
    89, 99, 6, -68, -61, 61, 22, -70, 48, 61, 13, -24, -75, 72, -11, -42, -2,
    -39, 3, 90, 93, -24, -24, -78, 74, -1, 69, -10, -97, 67, -9, -25, -86,
    67, 64, -91, -89, 92, -53, -70, -90, -29, -6, 23, 48, 59, -17, 72, -49,
    -35, 44, 34, 43, -79, 22, -77, 63, -53, -32, 3, -59, -1, 86, 65, 86, 14,
    -72, 10, 44, 24, -69, -86, 24, 37, 31, 22, 52, 97, 34, -43, 80, -86, -33,
    -89, 29, -10, -63, 61, 16, -43, 53, 89, -47, 15, -81, -20, 83, -81, -81,
    -83, -50, 88, 88, 43, -13, 60, 80, -97, -2, -11, 71, -75, 42, 29, -30,
    27, -35, 23, -16, 31, 22, 63, -29, 25, 77, 59, -93, -38, 23, -84, 7, 72,
    -57, -89, -4, 4, 8, -92, 42, 93, 88, 98, -9, 60, 14, -25, -81, -81, 40,
    83, -4, -32, -61, 70, -34, -54, 10, -75, -6, 60, 74, -79, -71, 2, 65,
    -55, 47, -48, -32, 97, 27, 82, 52, -47, -26, -85, -62, 65, 4, 13, 1, 58,
    -42, -70, 73, -51, 84, 1, -9, -3, 77, -61, 54, 2, 81, -60, 7, -21, 27,
    53, -11, -14, -96, 54, 12, -78, -72, -76, 17, 7, -18, 41, -20, 92, 59,
    -24, -39, -11, 34, -32, 26, 94, -16, 96, 18, 76, 74, 73, 6, -21, 71, -56,
    -40, 41, -79, 78, -5, 15, 10, -100, 1, 36, -94, -29, 55, -79, 35, 55,
    -61, -5, -60, 5, 64,
  -24, -17, -78, 90, 48, 49, 95, -60, 98, 99, 43, 42, 34, 90, 35, -17, -8,
    82, 48, 80, -27, -66, -9, 60, -60, 59, 38, -53, -83, 1, 69, -18, -20, 73,
    74, 70, -23, 96, 47, 77, 86, -100, 26, -54, -48, -56, 32, -90, -90, -9,
    69, 55, 26, 5, -85, 6, -92, 9, 30, -92, -53, 37, 41, -17, 67, -68, -95,
    60, 96, 33, -6, -43, 72, -10, -7, 68, 41, -12, -57, 63, 0, 62, -74, 66,
    59, -12, -46, -25, -69, -32, 82, -52, -37, -12, 61, 24, 72, 13, -25, -68,
    77, 55, 68, 24, -83, 71, 55, 90, 1, -26, 88, -2, 87, 75, 24, 44, -8, 72,
    -100, 57, 59, -20, 57, 83, 34, -88, -2, -48, -15, -28, 54, 31, -59, 71,
    91, 66, 31, 38, 92, 22, 83, 23, 87, -92, 39, -96, -94, 78, 24, -10, -79,
    -66, 42, -83, -93, 4, 87, -8, 47, 24, 47, 78, 36, 8, -97, -32, -30, -78,
    -22, 3, 73, -72, 67, 14, -21, 65, -41, -49, 97, 69, 44, 68, -65, -85, 62,
    -88, -94, 20, -82, 19, -59, -57, 13, 11, -88, 28, 7, -50, -87, -96, -39,
    38, 36, 38, 39, -44, 37, 80, -81, 59, 82, 82, -48, 78, -16, -32, 23, -96,
    -2, 77, -46, -53, -100, -63, -59, 7, 23, -97, -95, -29, -25, -53, -29, 0,
    60, -29, 84, 45, -82, 59, 73, -95, 31, 17, -23, -40, -92, 76, 18, 86, 24,
    19, -65, 40, -45, -62, -58, -27, -54, -45, 89, -5, -39, -73, 58, -29,
    -31, 17, 32, 23, 5, -95, -65, -6, 35, 48, 85, 3, -48, -92, 3, 76, 86,
    -27, 68, -92, -77, -26, 31, 23, 68, -73, -58, -47, 35, -89, -75, -19, 32,
    -69, 18, 25, 20, 52, -89, -82, -11, 65, -2, -8, 26, 75, 98, 48, 71, -30,
    1, -20, 68, -87, 56, 86, -28, -87, -51, 10, -26, -90, 69, 42, -70, 89, 31,
  1, 45, 97, -29, -78, -76, -72, 36, -53, -44, 16, -100, -93, -63, -5, -41,
    -53, -54, -17, -90, 51, 72, 26, 10, 33, -30, 38, 61, -16, -92, -37, -67,
    8, 75, 67, 2, 16, 25, -26, -88, 14, -15, -78, 97, -98, 61, -63, -25, 77,
    76, -71, -57, 65, 31, 15, 79, 44, -89, 25, -15, -57, 29, 68, -67, 93,
    -47, 61, 76, 52, 15, -95, -29, 19, 32, -57, 74, 7, -96, 5, -37, -23, -49,
    79, -81, 92, -82, 52, -29, 26, -38, -26, 78, 66, -87, -40, 85, 11, 83,
    89, -46, 27, 24, 18, -26, -48, -45, 43, 63, -44, -35, 48, -67, -49, -96,
    63, -26, 8, -84, -47, -84, -26, 22, -45, -8, -89, -92, -5, 73, 64, -9,
    -60, -100, 76, 55, -40, -12, -9, 13, -24, 96, 49, 91, 98, 36, 15, -10,
    38, 79, 79, -20, 98, 2, -39, 14, -3, 93, -97, -42, -15, -22, 97, 98, 8,
    29, -25, 49, 12, -5, 82, -55, -14, -29, -42, 90, -48, 11, 93, 24, 42, 30,
    -52, -47, -21, 25, 68, 69, 37, -68, 91, 1, -85, -80, 20, -7, -85, 52,
    -71, 2, 88, -9, 37, -94, 52, -5, -68, 41, -57, 38, 0, -16, -33, -32, -71,
    52, -98, -80, -51, 43, 32, 81, -54, -13, 89, 38, 99, 38, 33, -37, 10,
    -30, -85, -6, 96, -19, 66, -54, 1, -32, -3, 13, -68, 54, -39, -94, -7,
    -88, 93, -3, -96, 11, 1, 30, -96, 83, 81, -83, 78, 31, -33, 19, -70, -56,
    89, 77, 31, 24, -40, 21, -61, 51, 50, 24, 27, 35, -47, 31, 91, 14, 59,
    -92, -47, -4, -4, 63, 55, -64, 42, -56, -46, 38, 8, -12, 37, -50, -89,
    -37, 38, 33, -32, 90, 84, 63, 3, -47, -19, 89, 51, -41, -80, 10, -34,
    -46, -91, 1, -10, 37, -3, -57, 78, 86, 55, -37, 17, 92, 52, -1, -50, -57,
    -71, 95, 65, 38, 27,
  -29, 51, -86, 83, 24, -63, 18, -20, 8, -46, 27, -82, 58, -17, -67, -88,
    -41, 89, 91, 57, 64, 62, -57, 31, -51, -87, 27, -98, -67, 44, 57, 84, 37,
    -58, 3, 34, 40, 53, 64, -71, -35, -66, 42, -81, -44, -20, 49, 76, 28,
    -13, 32, -93, -15, -79, 49, 53, -75, 72, 29, -62, -41, 70, 19, -43, -9,
    -11, 3, 97, 31, 62, 37, -9, 60, 15, 73, -85, -48, -58, -97, 91, -97, -61,
    -92, -68, -39, 51, -30, 88, 45, 3, 17, -9, 9, -48, 80, 51, 88, -34, -43,
    -42, 65, -89, 30, 39, 84, 70, 89, -25, -32, 57, 33, -58, -88, -27, 59,
    -84, -99, 15, 13, -5, 90, 80, 78, 76, 23, -15, -4, -25, -5, 11, 8, 55,
    68, -85, -91, 68, 88, -19, -22, 61, -95, -42, -52, 34, 20, 35, -20, 24,
    26, -95, -71, 89, -54, -96, -59, -43, -12, -3, -10, -10, -37, -29, 91, 6,
    76, -58, -31, 85, -36, -21, 2, -79, 62, 30, -55, 2, -29, -14, 65, 53, 84,
    -67, 55, 57, 72, -30, 41, 81, -31, -52, -14, 80, -68, 63, 43, 86, -85,
    16, 63, -29, -91, -55, 32, -26, -32, -76, -73, -28, -57, 0, -85, -22, 43,
    94, 62, -55, -95, 36, 96, 94, -62, -92, -44, -16, -9, -7, -62, -11, -53,
    30, -7, -82, 88, -37, -21, 0, -60, 14, 88, -64, -47, -70, -64, 76, 31,
    36, -42, -58, 9, 24, -9, 12, -71, -37, 34, -25, 79, 10, 48, -95, 41, -14,
    -77, -60, -74, 59, 20, 69, -1, 41, 97, -43, 21, -73, 49, 9, -55, -71,
    -50, 97, -2, -22, -73, -92, -83, 86, -50, 73, 0, -6, -57, 30, 61, 38, -1,
    -88, -95, -88, 55, 54, -44, 3, -55, -5, -73, 60, -90, 64, 28, -23, -90,
    -80, 93, -16, 60, 82, -84, -86, -60, 44, 90, 96, 57, -100, 76, -83, 34,
    37, -54, -45, -52, 25, 7,
  51, -86, -88, 48, -91, -61, -60, 52, 76, 0, 72, 52, 0, -8, -36, 43, 48,
    -58, -41, 40, 41, 54, 15, 39, -64, 84, -60, 9, 22, -26, 89, 20, 64, -88,
    63, -17, 38, -58, 85, 82, 98, -9, -32, 18, -67, 40, -77, 66, 3, 34, -89,
    -84, -30, 51, -95, -34, -42, 64, 48, 62, -89, -2, 29, 11, 94, -96, 93,
    67, -81, -47, -90, -21, -90, 71, -78, -24, -21, -10, -11, -67, 80, 59,
    -44, -26, -66, 7, -71, -19, 59, 62, 81, 41, -15, 44, 42, 95, 69, -11, 98,
    -78, 19, 39, 44, -55, 5, -82, -3, -28, 30, 5, 32, 18, 24, -50, 39, 12,
    72, 20, 48, 85, 91, 74, 25, -71, 55, -12, -16, 39, 40, -82, 72, -34, -73,
    39, 88, -15, 83, 54, -79, -48, -51, 50, 57, 32, -90, 43, 18, 42, -20,
    -38, -7, 96, 12, -67, 1, -46, 13, -14, -69, -46, 97, 24, 90, -47, -30,
    -57, -1, 47, 74, -40, 38, 43, -77, -77, -96, 71, -62, -93, -18, -48, -34,
    -78, -93, -14, 39, -70, -26, -76, 91, -56, -60, 89, -91, -23, -47, 28,
    89, 69, 97, -83, -47, 62, 45, 43, 47, 68, 5, 0, 6, 76, 57, -35, 80, 31,
    36, -93, -94, 66, 5, -17, -22, -7, 72, 43, 13, -4, -18, 96, -66, -60,
    -36, -34, 59, 20, -81, 74, 3, -5, 29, -15, -21, 78, 89, -45, 18, -64, 5,
    -35, 88, -28, -32, -88, -90, 35, -86, -82, -4, -79, 55, -57, 70, -71, 79,
    -48, -64, 35, 90, 49, -94, -83, -51, -56, -3, 94, -89, -13, 18, 40, -47,
    -28, -3, -12, -24, -70, -83, -75, -11, -28, 70, -3, -49, 1, 87, -77, 42,
    -30, 83, 46, -11, 6, -44, -77, 51, 40, 82, 51, 46, 53, 96, 74, -10, -48,
    -15, -33, -80, -79, 21, 26, -31, 23, -46, 17, 28, -58, 8, 35, -2, 99,
    -21, -38, -74, 86, -89,
  -70, -10, 18, -2, -78, 91, -65, 96, -34, 55, -51, -58, 15, 61, -20, 87,
    -69, -74, -17, -35, 67, 60, -82, 94, 2, 50, 62, -70, -56, 61, 64, -41,
    64, 24, 85, -15, -76, -27, -70, 20, -29, -38, -1, 30, 61, -4, 67, -25,
    -39, -12, 58, -79, -79, -58, -7, -69, -78, -64, 78, -11, -20, 55, 42, 28,
    32, 49, 75, -94, 58, 55, 98, -99, 9, 38, -89, 21, -7, -72, 68, -29, -6,
    -61, -95, -41, 43, -60, 5, -79, -90, 14, -95, -56, 66, -90, 29, -96, -18,
    49, 56, -50, 57, -49, -58, 21, -68, 92, 32, -34, 18, -29, 83, -83, 63,
    -61, 71, -57, 31, -81, 42, -52, -34, -72, 59, 34, -9, 25, 92, 60, 87,
    -25, -44, 82, -63, 62, -97, -49, 40, -13, -5, -75, -29, 82, -42, -37, 76,
    83, 73, -12, -92, 24, 70, -53, 11, 12, -6, -29, 40, 80, -56, 35, 11, 92,
    -25, -20, -23, -23, -1, 25, -4, 22, -84, -44, 84, 51, -47, -51, 27, 78,
    -5, 49, -65, 91, 6, -62, -7, 86, -63, -45, -99, 65, -44, -54, 69, 19,
    -100, -42, 19, 41, -17, 66, -57, -95, 52, 59, 51, -100, -5, -92, -32,
    -96, 11, -17, -92, -59, -50, -69, -74, 71, -26, 90, 97, 58, 73, 8, -20,
    -87, -25, 64, -84, -34, -46, 54, 97, 98, 11, -92, -18, 73, 44, -100, 16,
    -79, 76, 9, 52, -60, 21, -62, 45, -49, -89, 71, 96, 84, 36, 30, -19, -11,
    -88, 87, -24, 10, 79, 50, -39, -66, 63, -80, 16, 14, -28, -20, -54, -40,
    31, 9, 26, 36, -73, -42, 79, -5, 76, -8, -89, 32, -60, 4, 28, 26, -85,
    -94, 72, -69, 26, -68, 30, -95, -72, -38, 49, -22, 65, -4, -30, 32, 28,
    51, 46, 45, -48, -23, -22, 16, -98, -35, 94, 20, -7, -95, -19, 7, -22,
    30, -75, -51, -52, 16, -60, -25, -37, 15, -97,
  77, 9, 14, -31, 69, 63, 32, -39, -87, 88, -25, -41, 91, 44, 85, 90, 69,
    -23, 57, 33, -71, -50, 42, -8, 3, -75, -53, 39, 73, 90, 45, 35, -46, 55,
    26, -43, -42, -20, -10, 8, 36, 53, 69, -48, -58, -21, -24, -99, 12, -53,
    -33, 92, 92, -98, -86, -67, 48, -18, 33, -96, 24, -94, -73, 43, -21, -40,
    -37, -77, 14, 28, 14, -91, 64, 71, 57, 79, -15, 39, -16, -62, -89, 75,
    49, 25, 54, -39, -63, -64, -38, 23, 33, -46, 25, 4, -76, 21, 8, 67, 19,
    30, 45, -100, 12, -68, 9, 35, -36, -3, 12, -62, 91, -55, -46, -10, -8,
    22, 83, -20, -98, -74, 25, -60, -54, 85, -91, -89, -11, -52, -65, -5,
    -14, -20, 2, 92, 11, -17, -48, 22, -82, 89, 28, 15, -24, 77, -16, -61,
    84, -14, 33, -26, 55, 15, 85, 57, -44, 86, 31, -55, -58, 97, 6, -88, 5,
    31, -58, 67, 31, -96, 99, 65, 78, -72, -3, 9, -1, 31, -70, -78, 36, -82,
    -50, -42, 41, -14, -51, -39, -23, -65, 70, -17, 31, -12, 32, 72, -48, -3,
    -73, 19, -23, 72, 35, -78, -74, -97, -5, 90, -70, -21, -47, 65, 79, -20,
    82, -93, -11, -29, 81, -3, 40, 64, 10, 24, -5, -42, -15, 82, 46, -21, 41,
    -65, -89, -66, 81, -75, 57, -99, 51, -46, 4, 26, -47, -71, -44, -14, 51,
    3, -42, -62, 82, 3, -77, 48, 40, 36, 66, -34, -85, 23, -59, 99, 29, -87,
    -70, 95, -42, -60, -53, 80, 36, 74, -45, 46, 93, 67, -13, 77, -4, 77,
    -92, -48, -1, 45, -87, 24, 83, 34, -76, -61, 68, -84, 74, 7, -53, -20,
    -77, 55, 15, 31, -8, -38, 85, 21, -63, -20, 88, -85, 38, -28, 0, 95, -40,
    -6, -18, 65, -80, 55, -28, -22, -7, 75, 93, -77, -58, 69, -51, -26, -25,
    39, 11, 93, -51, 29, -56,
  -24, 64, -93, -14, -41, 64, -59, -90, 17, -81, 5, 81, 76, 24, -92, -75, 33,
    -12, -58, -65, -17, -46, 79, 87, 56, -28, 18, 24, 60, -21, 49, -13, 7,
    -21, -98, -56, -71, 59, -10, -24, -30, -29, -22, 42, -36, 29, 50, 14, 76,
    96, -92, 85, 16, -64, -81, 23, -1, 88, -14, -84, 47, -22, 89, -51, -81,
    -51, -36, 8, 27, 69, -94, 81, -94, -49, 82, -49, 41, 89, -13, -17, 38,
    -54, 83, -21, -54, 66, -32, 39, -30, 24, -60, -88, 2, 84, -100, -9, -10,
    -29, -92, -46, -25, -44, -16, 44, -1, 56, 10, -54, 30, -43, 30, 22, 46,
    -46, -89, -66, -15, -37, -7, -5, -26, 50, -72, 21, -35, 24, -45, 85, -16,
    72, -58, -18, -59, 0, -35, 7, 43, 38, 56, -94, -91, 77, 56, 72, 53, -83,
    -58, 54, -12, 42, -65, 70, 10, 73, -21, 47, -83, 7, -50, 23, 23, 76, -79,
    33, -60, -40, -34, -63, 33, -50, -84, 53, 58, 8, -95, -80, 25, 39, -65,
    -9, 17, 9, 47, -60, -16, 71, -62, -90, -37, 19, 51, 16, -81, -88, 5, -23,
    -94, 51, -2, -11, -30, 15, -12, 20, 31, -56, 30, 18, 91, -77, -49, -47,
    14, 95, 8, 43, -29, 61, -90, 16, 82, 29, -11, 80, 89, 36, -64, -10, 71,
    -11, -99, 61, -85, 70, -39, -63, 77, 54, 8, 75, -46, 27, 90, 77, 92, -89,
    -32, -32, 19, 48, 22, 89, 81, -31, -18, 34, 21, -50, -68, 50, -98, -46,
    53, -90, -59, 4, 47, -41, -15, 44, 63, -66, -63, 52, -66, -48, -91, 75,
    2, -86, 96, 45, 68, -25, -37, -17, 19, 74, 18, -28, 34, -65, -76, 63,
    -58, 12, 95, -87, -61, -84, -92, 5, 34, -15, -43, -78, -20, -51, -12, 93,
    -71, -53, 78, -82, 88, -10, 62, -22, 99, 29, -63, -35, -82, 17, 72, 59,
    -16, -1, -27, 19, 61, 86, -58,
  10, 52, -70, 71, 84, 71, -29, -99, -2, -80, -44, 76, -70, 49, -65, 19, 46,
    4, -55, 64, -36, 38, -94, 55, 44, 91, 98, -28, -26, 89, -58, -16, 56,
    -10, 55, 21, -17, 34, -34, -66, 3, 54, 1, 17, 20, -30, -92, 72, -91, -36,
    -46, -57, -8, -5, 36, -63, -73, -91, 50, -41, 49, -73, 47, 4, 16, -96, 4,
    -67, 78, -59, -42, -8, -46, -8, -10, -7, 56, 98, 12, -72, -40, 85, -48,
    24, 26, -28, 99, -92, 20, 33, 91, -41, 19, 81, 26, 45, -35, -79, -12,
    -24, -56, 71, 56, 87, -3, 85, 54, -83, 50, -34, 65, 65, 31, -57, 4, 78,
    32, -15, -54, -26, -36, 21, -98, 45, 68, 99, 41, -50, -19, 75, 51, 68,
    20, 31, 2, 1, 90, -29, -48, 34, 89, 61, -40, 81, -24, 86, 3, 68, 48, -86,
    53, -81, 79, 46, 36, 30, -23, 10, -76, 42, -22, 35, 86, 25, -67, 75, 47,
    -37, 45, 75, 30, 29, 93, 14, -8, 13, -44, -23, 39, 84, -58, 68, 69, 67,
    33, 67, 22, -24, 64, -3, 57, -71, -21, 31, -8, 96, -37, -73, -64, -36,
    51, -4, 88, 88, -56, 47, -58, -44, -75, -10, -37, 79, -97, 5, 9, 16, -41,
    17, 83, 80, -70, 94, -41, -1, 0, -32, 42, -45, -37, 81, 72, -22, -77,
    -14, 81, 69, -19, 13, -29, -38, -52, 50, 77, -29, -60, 89, -8, 65, 50,
    62, -93, -17, -23, -91, 85, 51, -63, -95, 52, 87, -70, -55, -50, -55, 6,
    -23, -52, -9, -80, 92, 13, -83, 59, 60, 98, -26, 48, 1, -73, 56, -66, 22,
    27, -92, -35, -85, 53, 63, -22, 67, 41, 55, -77, 28, 19, -30, 38, -96,
    44, -71, -61, -3, -90, 14, 41, 84, 99, -47, 41, 30, -61, -42, -20, -60,
    30, 6, -58, 45, -1, 80, 72, 15, -43, -55, 95, -75, 99, -92, 75, -58, 41,
    15, 39,
  -52, 24, 59, -73, 48, 3, 96, 84, -83, 41, 81, 14, 3, 31, -72, -51, 64, -26,
    -12, -10, 65, 76, 99, -25, 6, 57, 9, 23, -69, -66, 14, 56, 2, -18, -65,
    -7, 42, 58, 47, 94, -54, 61, 46, 92, -53, -25, 23, 32, -47, 71, 90, -61,
    -54, 35, 94, 3, -81, 85, -60, -18, 8, -44, 66, -50, 5, 0, 42, -50, -62,
    9, 98, 52, -68, -66, -33, -36, 49, 90, 53, -68, 83, 98, -50, 88, -93, 38,
    -41, -3, -70, -93, -11, 80, -65, -72, -21, -23, 56, -78, -72, 26, 30, 94,
    86, 32, 4, 39, 5, -62, 1, -28, 73, 73, -2, 40, 82, 35, -85, -34, -49,
    -84, -68, 69, 96, 80, 69, 71, 70, 20, -15, 69, 1, -78, 50, 50, 24, -100,
    81, -38, 55, 75, 78, 67, 22, -9, -48, 18, 23, 50, -76, -47, 21, 46, 70,
    -35, 21, -47, 78, 70, -3, 84, 86, 43, 10, 60, -44, 17, -12, -33, 77, 36,
    41, 84, 42, 9, 0, 25, -99, 92, -88, 28, 32, 96, 25, -44, -96, -10, 17,
    72, 94, 33, -87, -39, -19, 44, -49, -8, -65, 25, -75, -82, -56, 65, 27,
    97, -92, -70, 19, -57, 29, 19, -87, -46, 69, 66, -41, 46, -72, -17, -26,
    21, -25, -45, -98, 59, 70, -40, 92, -48, -36, 99, -14, 56, 36, -15, -40,
    -89, -71, -41, 6, 33, 71, -100, -20, -86, 86, 0, -58, 8, 78, 75, -86,
    -88, 63, -90, 76, 0, -20, -98, 35, -34, -35, 95, -50, 90, -22, -47, -11,
    -44, 48, -69, 84, -20, -3, -99, 6, 74, -66, 44, -70, 97, -85, 45, 70, 17,
    -83, -8, 13, -81, -95, -76, -20, -77, 12, 97, -50, -54, 53, 91, 92, 21,
    -5, 50, -91, 79, 95, -41, 26, -42, 64, -70, -56, 59, 39, 4, 49, -77, 73,
    -41, -75, 73, -48, -4, 92, -8, 29, 92, 31, -48, 16, -79, -22, 54, 33,
  67, 50, 91, -53, 52, -79, -73, 51, -83, 58, -2, 24, -5, 65, 51, 46, 58, 79,
    65, -82, 46, 95, 64, -26, -21, 5, 87, -31, 3, -19, 9, 75, -12, 81, -14,
    18, -94, -12, 72, -14, 5, 47, 85, -97, 31, 46, -79, 52, 88, -17, -14, 47,
    62, -44, -18, 94, 98, 42, 96, -46, -28, -31, 49, 54, -43, -52, 64, 48,
    -26, -88, -43, -48, 92, 91, 76, -32, -67, 6, -73, -88, -38, 18, 40, -27,
    97, 1, -4, -37, -83, 84, -27, -95, 61, -80, -97, -58, 20, -48, -16, -44,
    -97, -34, 45, 35, 54, -87, 90, -27, 2, -89, -91, 40, -74, 19, 13, 25,
    -49, -45, 15, -19, -54, 1, 32, 15, 66, -11, -52, -15, -95, 92, 81, -12,
    9, -45, -78, 31, -64, -9, 2, 45, -83, 46, -85, 57, 39, 21, 31, 94, -75,
    -24, 3, 13, -16, 48, 16, -68, 28, -67, -7, 8, -42, -14, 40, -38, 70, 68,
    51, -17, 83, 90, 32, 70, -84, 76, 35, -20, -93, 94, 15, 52, 4, -4, 40,
    -14, 46, 49, 96, -21, -14, -20, -53, -48, -54, 27, -18, 22, 29, 64, 63,
    45, 19, 57, 6, -43, 91, 40, -63, -4, 52, -39, 96, -7, 43, -49, -12, -69,
    -5, -52, -17, -74, 57, 4, 47, -92, -28, 90, 49, 29, -72, 78, 22, -65,
    -14, 81, 63, 93, -94, 88, -87, 58, 42, -6, 87, -16, -61, -31, -62, 24,
    -9, 54, -10, -72, -77, -37, 59, 34, 65, -27, -50, 44, 53, 63, -31, -24,
    -53, 27, -89, 36, 63, 86, -94, 74, 60, 73, 66, 9, -11, -16, -40, -10, 66,
    12, 73, 88, -73, 86, 73, 35, -32, -56, 0, -36, -89, -60, 50, 36, 38, -56,
    -20, 48, 23, -65, -18, 55, -15, 42, -100, -85, 90, 12, 80, -16, -14, -68,
    -31, -98, 3, -20, 90, -65, 77, -41, -57, 85, 98, 91, 83, 93, 1, 89, -16,
    13, 33,
  -73, 72, 59, -25, 18, -43, -2, -19, 50, -100, -76, 18, -44, -23, -6, 17,
    -91, 30, -76, 11, -39, -25, -76, 73, 91, -85, -10, 8, 0, -79, -13, 32,
    94, -53, 24, -4, 21, 67, -95, 31, 15, 90, 96, -50, -53, -47, 71, 8, -33,
    -21, -3, -8, 63, 19, -18, 10, -88, 94, 67, -83, -88, 98, -29, -80, -95,
    -22, -74, 31, 6, -23, 29, -5, -58, 24, 72, 81, 34, 66, 33, -62, -23, -85,
    78, 52, -72, 2, 93, -51, 12, 17, -89, -67, 81, 45, 65, -91, 19, -97, -60,
    -72, -97, -12, -64, 99, -5, -66, -56, -50, -99, -16, -7, -77, 15, 65,
    -27, -15, -36, 63, -34, 36, -27, 10, 32, -94, 33, -34, -80, 47, -97, -79,
    23, 31, 6, 60, 32, -24, 14, 29, -23, 86, 21, -90, 39, -31, -85, -10, 67,
    -37, 7, 52, 21, -55, 94, -72, 31, -57, -1, 32, -22, 45, 93, -72, -24,
    -59, 15, 44, -46, 38, -91, -8, 25, -87, -10, 19, -24, -41, 66, -99, -58,
    38, -83, -93, -39, -96, -1, -36, 42, 99, -50, -29, -54, 77, -91, -80, 74,
    -8, -26, 8, 88, 79, -34, -30, 37, 68, 19, -28, 88, -15, 48, -61, -30, 73,
    89, -10, -9, -46, 1, -34, -93, -53, -45, 99, -48, 68, 29, 40, 44, -50,
    -78, -84, 52, -36, 26, 21, -96, -2, -18, -46, 84, 62, 24, 73, -48, 93,
    58, 79, 94, 3, 78, -90, -94, 50, -67, -33, 85, -95, -49, -67, 7, -52, 87,
    46, -87, -33, 78, 53, -99, -69, -82, 41, 97, 38, -79, 81, 62, 0, 33, -64,
    -66, -98, 74, -37, 57, -63, 98, 73, 16, 12, -34, 93, -95, -83, 13, 58,
    -86, -81, 25, 5, 24, 21, 93, 86, 25, 51, 4, 79, -42, -28, -35, 76, 73,
    -79, 49, 15, -54, -47, -19, 71, -95, 84, -3, 83, 57, 66, 76, -72, -44, 9,
    19, -40, -8, 10, 29,
  15, 39, -32, -30, 11, -7, -35, 97, 93, 97, 58, -78, 28, 79, -43, 42, 53,
    -92, -34, 68, -54, -26, 48, 25, -3, 51, 12, 71, 46, -4, -67, -46, 83,
    -13, 28, 5, 32, 1, 39, -66, 50, -61, 18, 50, -96, -82, -4, -56, -14, -97,
    -80, 14, -27, -75, -62, -81, -31, 32, 22, 89, -7, -85, 60, 90, 29, 99,
    -52, -11, 80, 84, -6, -96, -53, -87, 26, 72, 6, -100, 70, -29, 39, 70,
    -2, 23, 23, -33, -48, 23, 66, -53, 30, 56, 98, 23, -24, 76, -57, 33, -45,
    -74, -52, 33, 27, 19, -13, -90, 4, 83, 43, 35, 12, 62, 24, -17, -60, 82,
    49, -95, -99, 71, 0, 43, 52, 50, 27, -82, 4, 20, 19, -7, -15, 11, -81,
    15, 79, 10, 25, 36, -33, -47, 29, -75, -36, -23, -9, 11, 57, 7, 52, -21,
    -89, -57, -61, -46, -96, 16, -43, 84, 18, 64, 84, -89, -25, -64, 1, 26,
    -78, 19, -52, -1, -63, -35, 16, 59, 15, -62, -64, 10, -72, -51, 90, 5,
    -34, -61, -47, -47, 92, -73, -12, 19, 97, -49, 47, -68, -72, -34, -41,
    14, 42, 14, -29, -32, -90, 21, 85, 44, 60, -16, 89, 0, 61, 41, 25, -37,
    0, -52, 63, -60, -65, 53, -29, 68, 80, 70, 80, 65, -14, -12, -83, -9, 74,
    -19, -20, -40, -100, 18, 44, -28, -41, 27, 23, 72, 34, -77, 16, -58, -93,
    70, -6, 40, 3, -35, -43, 23, 24, -27, 30, 21, 71, -94, 49, 88, 48, 72,
    31, -30, 63, -79, -3, 12, -42, -90, -62, 12, 37, 92, -4, 25, -54, -50,
    -86, -14, -53, 27, 20, 7, 39, 60, 56, 12, 91, 8, -64, -37, -36, -68, -55,
    76, -28, -32, -29, 1, -90, 30, -59, -49, 88, -12, 67, -53, 2, -1, -47,
    -87, 47, 80, 17, 71, 61, 55, 90, 46, -19, 51, 48, -74, 33, -19, -25, 54,
    41, 49, 32,
  -95, -60, 80, 54, -36, -21, 37, 89, -10, 84, 57, -11, 98, -81, -43, -25,
    -11, -94, -57, 71, -9, 44, 64, 4, -83, 16, 6, -83, 97, 51, -9, 96, -53,
    18, 4, 70, -2, -11, -8, -87, 22, 38, -9, -9, -20, 30, -13, 93, -56, 16,
    93, -25, -55, -23, -38, 3, 30, 39, 43, 38, 13, -39, -92, 64, 81, -52,
    -30, -8, -1, -3, 27, 19, 22, -41, -1, -41, 13, -84, -80, 64, -71, 73, 15,
    -73, -14, 14, -40, -60, 56, -9, 46, -40, -20, -78, 29, -2, -1, 86, -80,
    21, -68, -84, -15, 88, -85, -86, 83, -43, 54, -8, -1, 74, 12, 17, 27, 41,
    -17, 66, 20, 8, 64, 27, 14, -40, 6, 62, -71, -3, 1, -5, -91, -43, -81,
    46, 83, 10, 37, -12, 36, 6, -3, -1, 59, 89, -17, 35, -22, -19, -53, -92,
    40, -58, -11, 28, 74, 50, 89, 39, 30, 42, -38, -52, -32, -44, -36, -89,
    23, -11, -71, 38, 90, 65, 63, -3, -86, 78, -31, -82, -41, -74, 29, 5, 22,
    1, -93, -86, 15, -77, -68, 48, -14, -63, 95, 11, -5, 50, 64, 62, -1, 36,
    -2, 60, 32, 56, -25, -27, 36, -79, 33, -25, -90, -40, 32, 98, -21, -81,
    -37, 67, -30, 84, 44, 23, 71, 56, 45, -53, 80, -89, -51, -83, -14, -51,
    -44, 26, 78, -71, -25, -18, 66, -37, 48, 2, -59, 78, -63, -89, -19, -49,
    -45, 39, 31, 99, 12, -59, 34, 12, -31, 67, 51, -45, -3, 25, -7, -78, 47,
    72, -100, 74, 28, 31, -23, -9, -12, -27, 71, 25, -82, 48, 51, -54, 85,
    81, 75, 45, 68, 14, -34, 7, -13, -4, -44, 18, -26, 20, 6, -4, -47, -91,
    -15, -94, -70, -5, 77, 97, 19, -49, -40, 70, 46, 31, 31, 33, 31, 60, 93,
    -79, 66, 91, -7, 82, 75, -13, -76, 8, 15, 28, 88, -6, 35, -36, -52, -75, -8,
  69, -25, -82, 50, -85, 99, 61, 30, -45, -18, 63, 24, 71, -71, -28, -61, 44,
    -92, -100, 47, -33, 15, 9, -99, -20, -36, -36, -87, -67, -17, 83, -100,
    -13, 9, -88, -84, -89, -78, 52, 52, -43, 64, 10, -2, 40, 82, 63, 24, 79,
    40, -20, -61, -74, 68, 91, -77, -7, -71, 42, -49, -36, -97, -85, 92, 69,
    71, -26, 67, -58, 65, -1, 50, -90, -48, -30, -89, -94, 69, -14, 75, 40,
    39, 14, -67, -39, -89, -54, 17, 26, -52, -12, 68, 19, -81, -94, -41, 2,
    -94, -26, 54, -20, -63, -47, 30, -64, 28, 19, 11, 81, -75, 26, 47, 43, 8,
    45, -52, -1, -95, 21, 96, -91, -60, -20, 27, 90, -17, -71, -16, -18, 5,
    77, -81, 99, -70, -13, 51, 16, -90, -99, 2, -68, -53, 27, -80, -97, -4,
    8, -58, 90, -56, 49, 21, -61, -66, 29, -100, 18, 27, -94, -44, -94, 62,
    47, 15, -20, 18, 48, -21, 81, 27, 59, 86, 13, 43, -30, 89, -52, -98, 20,
    -35, -53, -5, -87, 50, -16, -13, -8, -78, 33, 85, -75, 1, 1, 64, -80,
    -10, -15, -44, -66, -45, -37, 40, -43, -72, -46, 29, 56, 54, 76, -36, -7,
    -50, 84, -31, -76, 79, 34, -52, 93, -97, 64, -88, -74, -99, 19, 74, 38,
    41, -3, 22, -56, -94, 73, -77, -39, 72, 23, -57, -86, -3, 43, -6, 8, 42,
    -68, -45, 32, -90, -45, 2, -88, 67, 72, 7, -46, 69, 5, -46, 66, -8, 56,
    83, 15, 36, 69, -88, 45, 30, 17, 2, -94, -54, 62, 63, -64, 3, -99, -9,
    -24, -90, 71, 59, -95, -43, -83, 54, 48, -14, -80, 11, 35, 85, -100, 50,
    -15, -67, 87, 54, -83, -35, -40, 28, -7, 2, -23, 31, -35, 51, -41, 39,
    80, -32, -61, 96, -20, -52, -94, -14, -40, -76, -59, 75, -15, -80, -28,
    20, 54, -85, -2, -67, -13, -82, -32,
  -33, 90, -14, 75, -78, 45, 44, 37, 31, 67, -28, -29, -45, -95, -100, -2,
    -55, 97, -70, -5, -63, -97, 10, -35, 30, 19, 98, -47, 76, 92, -87, -63,
    -22, 58, -97, 49, -93, 16, 14, 6, 1, 71, 42, -30, 70, -37, 27, 8, 26,
    -76, 41, 85, -27, 20, -30, -96, -74, 23, -25, 78, -12, 21, 37, 36, 35,
    61, 76, 99, 69, 61, 41, 77, 18, -17, 54, -70, 73, -33, -45, 31, -100,
    -90, 30, -13, -69, -41, -85, -7, -43, -9, 72, -75, 50, 53, -41, -89, -95,
    7, -81, 56, 30, 63, -21, -4, 6, 12, -65, 52, 25, -41, -97, 41, 44, -14,
    77, -64, 22, 77, 57, 47, 47, -34, -19, -65, -12, 82, 31, -48, -52, -27,
    86, 37, -44, 53, 61, 64, 7, 62, -33, -9, 0, 37, 38, -78, 96, 9, -60, 3,
    -71, -58, -2, -55, -49, 79, -100, -1, -23, 13, 49, 48, -15, 26, 28, 6,
    -30, -57, 21, -64, -86, 7, -29, -2, -75, -38, 53, -34, -50, -83, 63, 42,
    81, -97, -98, 43, 10, 36, -90, -22, 46, 9, -15, -64, -10, 3, -64, -86,
    -80, -77, -86, 76, -88, 41, -22, -28, 63, 44, 89, 40, -57, 60, -45, 43,
    76, 62, -11, 31, 0, -54, 38, -81, -69, 55, -66, 91, -47, -67, -60, -11,
    27, -88, 43, 40, -88, 66, -30, -58, 96, 56, 18, 10, 8, 6, -34, -21, -66,
    -11, 14, -82, -83, -39, 32, 27, 5, 95, 70, 73, -19, 42, -25, -21, -21,
    39, 98, -76, 54, -69, -78, -47, 2, 94, 93, 23, -60, 49, -34, -7, -5, 80,
    -75, -15, 37, -15, 39, -93, -15, -6, 83, 79, 19, -86, 72, 55, -9, 81, 53,
    -32, 75, 63, 21, -63, -27, -90, -11, 65, 87, -65, -95, 89, 70, 24, -77,
    -14, -64, -73, 52, -59, 40, -99, 82, 59, 18, 74, 39, 50, 74, -98, 72,
    -24, -23, -11, 78, 72, 73,
  -62, 17, 43, -54, -11, -34, 57, -45, -74, -81, 81, -67, 49, 94, -19, -42,
    -64, -33, 58, 37, 10, -75, 77, 38, 88, -53, -48, 26, 86, -94, 13, 67,
    -71, 46, -33, 85, 9, -59, 81, -73, -98, -23, 45, 78, 17, 49, 57, -54, -7,
    28, 83, -10, 15, 77, 63, -25, -73, -74, 7, -60, -30, 59, 11, -55, 31, 51,
    -51, 77, 73, -60, -10, 74, 60, 88, 73, -39, -74, -30, -15, -90, 72, 40,
    -41, -20, 82, -88, -85, -47, -31, 26, 77, 6, -41, 36, -2, -26, 77, 71,
    -90, -67, -98, -89, 48, -57, -90, 73, -100, 42, 35, 94, -40, -86, -34,
    38, -88, 61, 63, 74, -72, -100, 80, 31, -73, 55, -2, 9, 48, 52, 16, 80,
    49, 26, -27, -31, 38, -90, -47, -19, -7, 4, -14, 51, -45, 98, -61, -15,
    -11, 72, 69, -74, 11, 53, -35, -23, -20, 63, 1, -56, 35, 36, 89, 5, -30,
    89, -81, 82, -78, 88, 94, 75, -71, -94, 61, -42, -98, -26, 0, -6, 7, -22,
    69, 33, 72, -47, 96, 71, -8, 43, 82, 19, 92, 31, 85, -49, -4, -8, 97,
    -51, -100, 74, -48, -50, 42, 22, -58, 72, -33, -10, 3, 73, 2, -74, -77,
    90, -25, -97, -60, 63, 47, 1, -17, -89, -41, 84, -47, -79, 80, 67, 37,
    -1, -4, -26, 21, 90, -33, 38, -73, -3, -66, -23, 90, 26, 39, -34, 93, -2,
    -21, -86, -74, -13, -59, 8, 81, 88, 83, 44, 57, 85, -69, 61, 63, 80, -36,
    -52, -64, -67, 60, -96, -33, -1, 70, 97, -93, 79, -64, 10, 43, -6, 83,
    51, -80, 33, 68, -73, -30, 72, -76, -5, 96, 17, -68, 81, -69, -12, -52,
    93, 6, -54, 62, -90, -58, 13, 73, -8, 64, 93, -68, -71, 45, -89, 17, 82,
    87, 40, 63, -44, -92, -74, -58, -72, -33, 90, 45, 52, -19, 67, -80, 61,
    78, 92, -12, 31, -28,
  48, -83, -12, -7, 88, -22, 53, -67, 88, 52, -31, -86, -60, -100, -6, 55,
    -76, 74, -55, 89, 80, -53, -34, -96, 48, 73, 26, -88, -84, 81, -72, -54,
    29, 47, 60, -24, 62, -84, 26, 1, 71, -84, 36, -68, -43, -53, 89, -6, -81,
    -45, -17, -5, 66, 20, 66, 31, -56, 29, -60, 24, 99, 43, -78, 34, 28, -97,
    13, -63, -42, 28, -40, 0, -47, 92, -28, 64, 96, 80, -15, -30, -77, -59,
    30, 64, -36, 87, -31, 81, 37, 92, 18, 56, -21, -60, 99, 78, -34, 65, -30,
    -62, 46, -44, 62, -8, 96, 21, 31, 7, -43, 38, -13, 2, 88, 36, 12, -71,
    93, -98, -21, -19, -85, 9, -91, 93, 73, 42, -91, -60, 88, 95, -43, 68,
    42, -35, 70, 29, 57, 87, 12, 62, -72, 71, -88, 43, 63, -54, -27, 36, -76,
    -9, -94, 83, -23, -85, -15, -71, -40, -14, 46, 70, 27, 23, -55, 52, 29,
    38, -97, -62, -28, 75, -67, -18, -27, -39, 19, -95, 53, -26, -41, -59,
    63, -38, 77, -78, -54, -66, -18, 11, -21, 90, -97, -73, -35, -27, -11,
    92, -22, 27, 3, 40, -73, -99, 76, 33, -41, -44, 83, -13, 11, 68, -45,
    -84, 61, 67, 31, -24, -62, 48, -57, 37, -98, 63, -10, 49, 61, 32, -37,
    -88, 39, 93, -77, 57, 30, -52, -10, -68, -34, -87, 94, -47, 58, 86, -14,
    -27, 87, 1, -47, -99, -98, 76, -33, -56, 22, -80, -73, 75, 63, -21, -12,
    20, -79, -1, -46, -5, 73, 34, -30, 79, 34, -78, 76, 62, -37, -77, 61,
    -44, -3, -85, -69, 62, -98, -36, 34, -29, -12, 86, -47, -84, -86, 90, 36,
    -24, 97, 72, 49, 10, 5, 89, -48, 26, 7, -40, 8, 14, -85, -93, 50, -47,
    28, 13, -4, 9, -5, 72, 62, 25, -24, -73, -79, 68, -13, -13, -12, -55, 87,
    47, -61, 34, -52, -20, -26, -36, 35,
  86, -99, -54, 77, 5, 35, -56, -33, -38, -41, -63, 23, 87, 12, 83, -12, -1,
    40, 40, -57, 80, 13, 59, -100, -24, -79, -17, -35, 63, -27, -42, -7, -77,
    -60, 64, -95, -42, 64, -2, 38, 68, 18, 99, 55, 70, 23, 94, -61, -87, 31,
    1, 26, 68, -42, -79, -9, -82, -43, -46, 67, 11, 6, -87, -49, 56, -54, 88,
    93, -7, -13, 77, 56, 35, 49, -46, 51, -37, -13, 96, -6, -28, -8, -57, 99,
    -3, 14, 20, -8, -72, -33, 26, 45, 7, -75, 38, 66, -59, 39, -61, 46, -91,
    93, -12, -10, 43, -57, -43, 80, 9, 5, -28, -65, 55, 24, -59, -32, -76,
    -42, -88, -5, -11, 8, 55, 20, -53, 22, 34, 61, 60, -55, -90, -43, -80,
    93, 87, -59, 62, 38, -36, 89, 22, -71, -21, -86, 40, 18, 71, 79, -65, 69,
    16, -47, -4, -44, 3, -57, -25, -22, -82, -28, -61, -19, -92, -27, 83, 24,
    -71, 39, 4, 77, -72, 73, -69, -47, -51, 78, 97, 19, -39, 62, 33, -18,
    -48, 31, -93, 29, 45, 21, 94, -88, 61, 35, 6, 78, 71, 26, 45, 87, -76, 3,
    -49, 61, 53, 68, -11, -52, -49, -58, 21, -73, -73, -31, 16, 28, 71, -57,
    -92, -78, 47, -96, -48, -88, 65, 30, 31, -82, -15, -93, -64, 65, 41, -22,
    -52, 87, -58, -67, -53, -71, -80, 87, 73, 46, 1, 29, 4, 36, -44, 2, 97,
    -50, -70, -95, -62, -69, 38, -50, 28, -40, -98, 58, 35, -90, 95, -91, -4,
    23, -74, -52, 17, 66, 91, 98, 39, -84, 64, 3, 23, -74, -93, 23, 7, -37,
    25, 36, 93, -25, 21, -35, 3, 28, 6, -40, -54, 0, -69, -74, -100, -49, 91,
    26, -16, 2, 20, -90, -65, 38, -49, 78, -30, -11, -14, 74, -61, -2, 68,
    -99, -16, -18, -80, -74, -11, 67, -95, -43, 59, 2, 45, -11, -31, 4, -24,
    39, 96,
  87, -79, -98, -18, 57, -46, 46, -13, -17, -28, 13, 15, 66, -31, -77, 80,
    -38, 82, -84, -63, 79, 24, -49, -96, -39, 14, -22, 55, 40, 58, -18, -3,
    -32, -54, -46, -91, -88, -7, -16, 17, -72, 20, -31, 87, 4, 36, 83, -28,
    -43, 49, -38, 37, 13, -31, -29, -96, -96, 62, -85, 24, 0, 71, -51, -59,
    63, -23, 66, 7, -66, 2, -5, 3, 30, -57, 89, 99, 76, -24, 82, -46, 22,
    -52, -39, 2, -44, -67, -60, 18, -33, -41, -14, 66, -37, -60, -67, 3, 70,
    -14, -1, 53, 35, -2, 17, 16, -97, -3, -47, -64, -56, 81, 74, -13, 44, 24,
    -11, -34, -57, 59, -12, 97, 85, 10, 1, -19, -47, 56, 62, -29, -59, -97,
    14, -95, -63, -19, -67, -79, 56, -59, 29, -73, 83, -49, -83, 31, 64, 63,
    -88, 83, 41, -77, 3, -33, -31, 99, 92, 99, -30, 81, -98, -39, -90, -56,
    23, 75, 14, -79, -93, -36, 25, 2, 23, -49, -95, 46, -61, -90, 67, -66, 6,
    90, -92, 69, -51, 64, -14, -75, 95, 92, 63, 54, -1, 22, 80, 1, 94, -1,
    50, -67, 6, 35, 33, -25, 43, 2, -90, -51, -62, 65, 92, -48, -70, 26, 74,
    -90, -38, 3, -11, 22, 47, 50, -50, -45, 36, 74, 96, 63, -88, 17, -99,
    -84, 1, -96, 41, -12, -63, 16, 92, -80, 91, -88, 65, -73, -55, 48, 54,
    38, -79, -86, 91, -82, -50, 77, -9, 18, -77, 65, -27, 5, -73, 83, -45,
    82, -18, -63, 80, -52, -29, 95, 66, 29, -60, 1, 77, 83, 11, -8, 98, 64,
    -11, -8, 9, -90, -35, 86, 72, 17, 25, -19, -32, -36, -41, 82, 9, 56, 68,
    5, -61, -34, -21, 6, 92, -38, 30, -72, 30, 71, -70, 24, 13, 40, -8, -24,
    48, -25, 53, 96, -73, 11, 87, -66, 92, 83, -31, 19, 18, -78, -72, -35,
    27, -83, 38, 92, -66,
  75, -70, -92, 64, -32, -59, -79, -63, -100, 18, -12, -34, 78, 62, 76, 16,
    -27, 99, 8, -54, -57, 55, -34, 21, 43, -54, 12, -44, -62, -82, 39, 85,
    -1, -88, 45, -38, 42, -47, 82, 55, -51, 42, -81, -11, -8, -94, -37, 92,
    19, 94, -63, 93, -16, -9, -74, -30, -89, -8, 8, -72, 40, -43, 53, -99,
    10, 25, -14, 46, -43, 56, -77, 27, -86, 37, 36, 34, 10, 96, 82, 76, 89,
    -61, -82, -66, 91, -77, 33, 26, -40, -52, 46, -69, 53, 73, -49, -98, 23,
    82, -33, -57, 38, -35, 25, 8, 61, -45, 40, 61, 50, -79, -18, -24, -64,
    -76, 44, -73, 94, -20, 21, 77, -14, -83, 4, -24, 38, -23, -33, 21, -79,
    43, -19, -90, -49, 58, 87, -68, -74, -73, 96, 29, -76, -2, 50, -11, 85,
    70, -17, -79, 60, 95, -47, 75, -23, 92, -81, 8, 96, -62, 6, 55, 91, 87,
    -2, 62, 72, 39, -89, -49, 33, -60, -27, -45, 56, 73, -85, 22, 98, 95, 25,
    -18, 81, -98, 86, -82, 32, -83, -47, -37, -33, -42, -50, 84, -79, 9, -9,
    79, -66, -50, 81, -85, -83, -49, -28, -33, 40, 78, 70, -98, -74, 7, 94,
    -34, 15, 42, 89, 5, 32, 34, 65, -27, 79, 90, 32, 92, 3, -18, 28, -70,
    -29, -71, -21, -75, 31, -10, -13, 66, -66, 7, 6, -62, 99, -45, 47, 66,
    -57, -100, -13, -67, 43, -84, 60, -52, -89, -31, 92, 16, -64, 96, -86,
    13, 79, 58, 59, -20, -17, -15, 20, 88, -77, 30, -58, -11, -95, 41, 28,
    -42, -96, 81, -23, -56, -96, 3, 91, 91, -11, 17, 16, 1, -29, -67, 44, 15,
    70, 30, -4, 89, 99, 34, 46, -42, -24, 17, 28, 29, 23, -71, -26, -53, -27,
    -40, 24, -3, -72, 34, -12, -5, 58, -71, 75, 62, -67, 4, 92, -60, -62, 78,
    35, 6, -51, 11, 83, -48, -76,
  -59, -71, -2, 4, -75, 21, -26, -33, 64, 4, 59, -76, -37, 2, 62, -51, -17,
    30, -64, 98, -12, 7, 93, -38, -64, 71, 34, 95, 41, -21, -4, 43, 71, 38,
    -9, -48, 92, -25, 5, 71, -38, 14, -83, 18, 66, 3, 1, 61, 50, 74, -46,
    -50, -5, -25, 47, -94, 76, -99, 57, -17, -76, -7, -28, -59, 72, -9, -54,
    -73, 95, -53, 5, -89, 88, -12, -74, 41, 30, 22, -98, -62, 59, 17, -32,
    -73, -95, -17, -94, 81, -37, -14, 3, -29, 43, -88, -78, 96, -48, 75, -16,
    -83, 64, 8, 16, 26, -94, -98, -36, -20, -93, -40, 84, 16, 12, -92, -29,
    31, -35, -57, 94, -35, -11, 52, 28, -56, -36, 32, 38, -27, 28, 99, -5,
    -51, 38, -39, -75, -72, 28, 0, -37, -20, -65, -45, 56, 66, 85, -10, 43,
    78, -70, -60, 86, -15, 35, -32, -18, -88, -38, -84, 19, 56, -19, 54, 54,
    32, -49, -38, -51, 58, -21, 97, -41, -86, -42, -30, 40, -17, -96, 31,
    -17, -65, 61, -77, -1, 81, -33, -15, 86, 68, 24, 90, -61, -99, 38, 97,
    -6, 98, 91, 44, -31, 76, -14, 91, 96, 37, -85, 48, 9, -43, -62, -40, -34,
    -43, 61, 41, -13, 13, 5, 10, -26, -59, -48, 12, -54, -88, 2, 0, -27, -89,
    -23, -83, 7, -41, 75, 29, -57, -8, -47, -29, -55, -20, 26, -82, -23, 21,
    -44, 30, -88, 95, 82, 26, -85, 29, 40, -97, -38, -59, 88, -49, 53, 21,
    -5, -47, -95, -27, -98, 27, 43, 9, 15, 41, 12, 64, 73, 95, -75, 8, -28,
    -38, -98, 58, -54, -70, 94, 52, -6, -71, 58, 70, -49, 62, 36, 78, 73, 26,
    43, -64, 52, 37, -14, 16, -4, -75, 37, 8, -51, 70, 29, 7, -17, 10, -93,
    72, -60, 56, -58, 42, 91, -22, 5, -28, 5, -36, -51, -92, 67, -56, -68, 7,
    -36, -91, -15, -12, 54,
  -96, -80, 5, -18, 11, -77, -72, -31, -5, 26, -91, -92, 11, -57, -77, 65,
    32, 29, 69, 68, 29, 96, -41, -73, -26, 95, -34, 91, 72, 5, 89, 13, 86,
    91, -52, 74, 72, -96, -97, -29, -27, 45, -55, -11, -86, -83, -25, 32,
    -38, 13, -92, -1, -98, -11, -29, 85, 99, 69, -88, 42, 81, -88, -56, 3,
    -65, -68, 38, -60, -68, 1, -37, 82, -96, 78, 33, 60, -50, -16, 5, -40,
    -90, -36, -39, -23, 5, -8, -19, -79, -74, -1, -68, 57, 32, -12, 74, -59,
    -20, -73, 91, 82, -65, 0, 48, 56, 92, -37, -65, 21, 25, 90, -17, 5, -38,
    63, -49, 4, 11, -20, 59, -51, -94, 65, 65, 94, -17, 19, -16, -6, -88, 50,
    -27, 51, -31, 87, -99, 33, 4, 50, -80, 30, 4, 34, 8, -49, 82, 19, -68,
    66, -73, -69, 6, 99, -82, 43, -92, 30, -5, 87, 15, -69, 18, 95, -59, 71,
    -20, 60, -90, -35, -61, 53, -58, 46, 76, 28, -33, 73, 9, -90, -3, -39,
    70, -48, 76, 80, -9, 1, -5, -68, -18, 86, 38, -43, -84, 35, -94, 91, 34,
    -85, -31, -19, 35, -75, 97, -2, 32, -43, 37, -75, -93, -56, 92, -70, 28,
    -63, 4, -11, 76, 42, -8, 46, 62, -24, -70, -43, -78, -97, 32, 87, 35, 3,
    -82, -8, -44, 73, 30, 97, -10, -51, 2, 60, -10, -33, 87, -32, -98, -20,
    -79, -66, -54, -97, 53, 78, -27, -47, -52, -97, -90, -74, -73, 43, 66,
    80, 87, -53, 45, 17, 11, -57, 2, 11, 8, 50, -79, -73, 45, -75, -14, 89,
    60, 75, 77, -55, -69, 32, 69, -8, -40, -44, -4, 91, -60, -57, -7, 72, 98,
    47, 25, -76, -71, -60, -12, 10, -57, -85, 96, 96, 90, -24, -75, -8, 1,
    26, -98, 38, -33, 54, -71, -92, -35, 83, -89, -37, -8, -15, -24, 98, -3,
    -43, -74, -20, -96, 49, 97,
  86, -65, -60, -75, -98, -75, -48, 43, -81, 49, -52, 25, -72, -11, 67, -80,
    -22, -32, -5, 2, -91, -11, 86, 41, -14, 73, -7, 55, 9, -1, 35, -80, -25,
    -75, 0, -75, -87, -85, 13, 16, -75, -76, -10, 66, -47, 93, -91, 54, -38,
    -62, -26, 23, -20, -18, -67, 76, -5, 34, 77, 88, 19, 86, -72, 15, -66,
    -98, -48, -72, -71, -69, -7, 65, 46, 79, 0, -100, -37, -86, 1, -58, 91,
    77, 73, 38, -32, 58, 96, -3, -10, 15, 38, -73, 24, -32, 42, 22, 63, -27,
    -2, -16, -98, -24, 85, -58, 85, -54, -43, -98, -54, -33, 46, 84, 71, 39,
    -81, -33, 76, -87, -58, 56, -30, -89, 11, 82, 75, -52, -27, 63, 90, 37,
    -92, 42, -40, 89, -13, 24, 64, 7, 33, -7, -54, -56, 35, -4, 59, 68, 85,
    3, 57, -83, 49, -53, 95, 18, -53, 35, 64, -56, 28, -48, -16, 46, -40, 51,
    16, 50, -16, -67, 59, 3, 91, -81, -86, 57, -34, -7, 37, -69, -1, -77, 28,
    53, 63, -93, 16, 70, 16, -87, 10, -39, 46, 73, 26, -97, -35, -71, -87,
    -17, 7, -4, 31, 72, -43, -44, -65, -85, -14, 61, 19, -70, -90, 77, 89,
    -14, -25, 8, -62, 13, 91, -68, -83, 13, -29, -97, -47, 6, 66, -7, -97,
    -50, -25, -21, 38, 1, 76, 48, 79, 84, -7, -89, 18, 0, -20, -95, -89, -7,
    78, -83, -73, 32, 62, -44, 85, 93, 52, 10, 91, 50, 90, -5, 51, 78, 63,
    72, -51, -49, -80, -18, -86, 32, -45, 34, -53, -72, 2, -95, 86, 18, -72,
    -54, 75, -27, -7, 21, 69, 54, 34, 20, -6, 65, 69, -96, -23, 8, -67, 26,
    -13, -16, 95, -93, -8, 56, 41, -35, 40, 64, 12, 33, -24, -69, -28, 7,
    -82, 0, 82, -65, -90, -77, 27, -69, 7, 79, -4, 60, 20, -42, -61, 16, 59,
    -85, 69, 34, 24,
  -8, 7, -86, -37, 92, -37, 57, -38, 92, -57, 61, -100, -37, 99, 18, -19, 1,
    31, 14, 56, -1, -51, 5, -44, 23, -50, 17, 45, 37, -9, 11, -15, 78, -61,
    -72, 63, 21, -41, -86, 79, 20, -61, -58, -13, -41, 59, -60, -70, -85,
    -16, 1, 63, -46, -12, 4, 12, -2, 30, -18, 35, 23, 18, 87, -95, -34, -82,
    -4, -88, 47, -24, -7, -48, -58, 13, -95, -6, -57, -57, 22, 24, 33, -67,
    71, -48, -98, -14, -18, 10, 52, -9, -54, -14, 52, 78, -76, 38, 73, 80,
    43, -22, 77, 66, 7, -13, -19, 1, 31, -64, 4, -11, -19, -17, 28, 76, -81,
    -5, 18, 85, -16, -22, 12, -76, -92, -7, -16, -42, 62, -58, 90, 19, 51,
    36, -91, 74, -83, 90, 35, 8, 9, 80, -45, -23, -52, -39, 76, -12, -70,
    -52, -21, 32, -79, 19, -85, 43, 36, -29, 31, -82, -87, 92, 80, 88, -80,
    28, 42, -81, 62, 10, 77, -1, -7, -37, -26, -18, 34, -88, -20, -76, -100,
    -14, -26, -26, 25, -54, 63, 28, 40, -38, 65, 70, 8, 18, -29, -5, -40,
    -91, -27, 94, -36, 79, 79, 22, 58, 34, 25, -30, -92, 74, -44, 79, -1,
    -52, -58, 41, 97, 68, -20, -39, -12, 54, -86, 64, -7, -46, 53, -71, -95,
    -60, -100, -34, -62, 51, 85, 90, 65, 36, 63, 98, 40, 96, -10, 45, -22,
    66, -55, -87, 3, -30, 77, 67, 80, -39, -97, 11, 7, -46, 9, 54, -44, -61,
    -83, 65, -96, -84, -74, 6, -71, -43, -3, -98, 18, -68, 96, 70, 27, 98,
    56, -72, -11, -38, 52, -40, -94, -46, 40, -2, 16, 40, 45, -88, -76, 25,
    -50, 31, -44, -67, 94, -47, -25, -21, 6, 86, 72, 9, -50, 36, 48, 85, 8,
    43, -44, 73, -7, 62, -92, 86, 26, -87, 12, 4, 48, -88, 40, 36, -15, 86,
    -24, 65, 2, -88, 79, -63, -83,
  25, -52, -47, -14, 83, -96, 1, 47, -32, 80, 68, -18, -71, 27, 84, 69, -39,
    -35, 94, -68, -93, -5, 28, 0, 60, -25, 11, 30, 57, 25, -84, 57, 59, 19,
    -6, -6, -82, -98, 98, -35, 68, -83, -99, 2, -38, 15, 67, 86, 87, -42,
    -38, -84, 8, 9, 30, -73, -17, 93, -32, -65, 51, -8, -8, 66, 74, -32, 92,
    -6, 9, -19, 24, 37, 6, 50, -46, -80, 19, -79, -63, 11, -41, -30, -96, -5,
    -81, -16, -53, -83, 78, 2, 53, -87, -41, 45, 55, -16, 9, 53, -99, -3,
    -63, 42, -95, -19, -25, 29, 57, -88, 6, 45, -91, 51, 46, -23, -71, -97,
    46, -35, -51, -71, -19, 12, -61, 92, -70, 58, 64, -85, 28, -34, 96, -26,
    -31, -54, -54, 81, -75, 93, 93, -10, -83, 85, -10, 58, 27, -28, 24, -81,
    -10, 47, -85, 68, -46, -15, 12, 70, -82, -78, -17, 59, 4, 43, -100, 29,
    -57, 50, 31, 25, 10, 9, 92, -51, 66, 12, 45, 54, 82, 28, 51, -20, 85, 46,
    -4, -62, 73, 68, 17, 50, -96, 87, -39, 70, 75, -42, 51, -85, 55, 5, -81,
    16, 39, -68, 92, -96, 71, 11, 44, -95, -98, 17, -90, -70, 26, -90, -40,
    -2, 0, 89, 27, -62, -79, -19, -49, 59, -69, 90, -98, -82, -82, -24, 54,
    -82, -90, -69, -89, -72, 15, 22, 86, -70, 33, -67, 53, -70, -27, 97, -97,
    81, -38, 96, -71, -75, -59, 74, 32, 26, -40, 34, 29, -69, -28, -23, 56,
    28, 32, 4, 56, -99, 45, -58, -72, -17, -2, 41, 25, 0, 79, 73, 67, -35,
    -82, 24, 41, -72, 23, 85, -52, 53, 50, 41, 99, -81, 66, -86, -47, 65, 83,
    0, 49, -66, 62, -23, 34, 17, 54, 71, -15, 4, -18, 36, 55, 84, -19, -26,
    75, -38, 1, -87, 5, 51, -1, -71, 91, -43, 25, -67, -48, -72, 87, 41, -51,
    14, 60,
  74, -54, -15, 16, 61, -8, -54, 92, 55, 7, 40, 84, 5, -44, 18, -49, 12, -65,
    -36, -82, 99, 92, -52, 7, 96, -62, -61, -60, -52, -45, -13, 47, 17, -84,
    92, 18, -84, -67, -70, -10, -23, -82, 47, 39, 38, 88, -10, 18, -12, -96,
    -63, 9, 74, 58, -43, 42, 71, -56, 11, -100, 61, -65, 43, -36, 28, 81, -5,
    -86, 33, -46, -78, 98, 46, 24, 3, 75, 75, 17, -8, -92, 23, 96, 20, 51,
    64, 39, -5, -88, 53, 92, 50, -25, -82, 50, -56, -16, -9, 66, 45, 39, -71,
    -34, 32, 2, -83, 5, 90, 91, -55, -74, 67, 55, 4, 81, -5, -41, -90, -91,
    38, -58, 8, 56, 72, -51, -12, 96, -17, -50, 1, 97, -17, 67, -1, 43, -65,
    35, 43, 48, -98, -100, -18, 10, 62, 48, 80, -93, -52, -52, 42, 45, 55,
    -18, 25, -50, -69, -24, -90, 55, -91, 87, 67, -46, 55, -28, -39, 23, 96,
    39, 79, 77, 95, -26, 96, -75, -40, 85, -2, 45, 99, -41, 81, 9, 8, 83, 96,
    -24, 1, 79, -34, -66, -64, -4, 11, 22, 5, 60, -78, 31, 78, -31, -84, -9,
    56, -71, 49, 55, -57, 92, 43, 61, -27, 72, -5, 54, -24, 34, 19, 44, 86,
    19, 44, 56, -28, 2, 47, -61, -30, 19, -90, 78, 58, 52, -25, 13, -97, 18,
    55, 56, 26, -78, 6, 81, 40, -55, 54, -20, 13, 90, 46, -67, 12, 80, 71,
    32, 12, -49, 66, 89, 83, -51, -52, -40, -39, 24, 89, -35, 7, 83, 95, -47,
    -75, 54, 1, -29, 69, 89, -25, 57, 94, 85, -54, -25, 82, 75, 50, 23, -61,
    -84, -79, 10, 13, -49, -68, -38, -49, -22, -100, -63, 99, -90, -97, 99,
    44, 39, -63, -99, -58, -89, 77, -57, -70, -17, 47, 24, -90, -8, -94, 23,
    46, 70, -70, -37, 77, 54, 74, -31, -2, 26, 13, -8, -23, 58, 11,
  -69, 49, -64, -31, 18, -78, 6, 54, 23, 33, -83, -77, 36, 10, 46, 51, 69,
    -2, -77, 79, -85, 6, -33, 4, 41, 86, 31, 9, -20, 70, 37, -3, -52, -53,
    97, -69, -12, -46, 63, 18, -64, 4, 97, -99, -81, 26, 99, 23, -49, 84,
    -70, -68, 7, 72, -68, -100, 99, 67, 4, 95, -65, -78, 61, 48, -51, -21,
    -33, 93, -59, 91, -16, -75, -29, 98, -3, -34, -13, 61, 64, -75, 88, 33,
    46, -31, 6, 15, 37, 97, 69, -42, 13, -11, -65, -5, 80, 41, 54, -44, 92,
    45, 5, 71, -77, 83, 54, 43, -22, 75, 50, -19, -52, 69, 47, -73, 39, -18,
    69, 63, -26, 58, -20, -47, 1, 54, -35, 77, -91, -23, -67, 78, -4, 37,
    -57, -32, -44, 4, 45, 1, -21, -94, -29, -94, -5, 6, 32, 48, -80, -55, -9,
    41, -10, -67, -87, 10, 43, 62, 6, -32, -42, -75, -73, -31, -5, -64, -94,
    76, -77, 45, 5, 32, -94, 75, -76, -56, 44, -37, 52, 89, -70, 23, 77, -8,
    -68, 78, 71, -100, -6, -17, -6, 39, 83, -29, 84, -99, -13, -59, 34, 34,
    20, -42, 71, -64, 84, -94, 22, 85, 44, -54, 41, 38, -9, -49, -74, 28,
    -83, 37, -68, -28, -38, -60, 42, 39, -58, -12, -80, -44, -47, 8, -94, 16,
    -53, -81, -63, 81, 25, 9, -90, 68, 69, -53, -75, 90, -16, -98, -98, -21,
    94, 24, -60, 67, -92, 98, -50, -41, -45, -70, 74, 85, -37, 32, 26, 84,
    96, 27, -55, -85, -18, 13, -17, 66, -46, 21, 81, -19, 81, 38, 67, -20,
    10, 43, -98, -71, -61, 97, 84, -36, 46, -77, 20, 59, -72, 67, 81, 54, 85,
    -40, -84, 50, -53, 12, -52, -83, 40, -17, 48, -88, 52, -76, 50, 89, 60,
    -83, 58, -22, 13, -94, 88, -12, -88, 86, -79, 54, 46, 77, -37, 85, 91,
    26, 70, 8, -2, -20, 83,
  0, -24, 8, 44, 99, -5, -40, 4, -51, -93, 97, 24, 31, -23, -13, 12, -82, 91,
    25, -6, -32, -40, -4, -72, 62, -8, -45, 76, -92, -42, 86, 92, -67, -55,
    -61, 76, -33, 86, -5, -36, 85, -99, -34, -84, -50, 38, -99, -31, 94, -75,
    85, -79, 64, 7, -53, -94, 5, 17, 57, -6, -49, -58, 69, 47, -90, 51, -97,
    45, -15, 0, -33, -6, -41, 57, 87, 56, 72, 18, 34, -92, -83, -52, 48, 6,
    -70, 32, -71, 82, 56, -1, 60, 20, 79, 60, 78, 65, -4, 70, -47, 4, -19,
    44, -6, 28, -41, 13, 24, -5, -89, -49, -73, 71, 64, -96, -82, -31, -75,
    15, 68, 65, -7, -99, -21, 30, -15, -97, 94, -27, 30, 43, 99, -24, -26,
    -32, -60, 53, 73, -34, 70, -30, -70, -14, 53, -66, 45, -27, 84, 34, 60,
    -26, -58, 34, -18, -53, -79, 54, -99, 29, 71, -73, -43, 84, 33, -16, 96,
    54, -52, 35, -17, -16, -90, -66, -72, 11, 71, 16, -92, 17, -28, 61, -81,
    -54, 82, 73, 40, -78, -87, 66, -38, 33, -66, 60, 0, 42, -9, -60, 13, 97,
    13, -2, 68, -6, 58, -15, 23, 6, -52, -79, 72, 44, 26, -27, 78, 29, -47,
    -26, -78, -6, 44, -48, -53, -3, -3, 33, 17, 81, 45, 87, -39, 61, 44, 11,
    22, 5, 11, 68, 11, -55, -30, -84, -90, -32, -75, -49, 38, 99, -94, 5, 78,
    -38, 25, -68, -46, 82, 19, 71, -70, 40, 13, 21, 19, -37, 19, 12, 87, -9,
    56, -85, 2, -4, -95, 13, 21, 34, 62, -25, -1, 20, 57, 62, -53, -43, 55,
    71, 75, 29, -37, 29, -79, -40, -68, -95, -23, 53, -13, 50, 47, -34, 26,
    23, -17, 46, -99, 10, 6, -99, 77, -26, -26, 21, -67, 22, -78, 93, 14,
    -30, 78, 40, -28, 78, -34, -40, 27, 65, 26, 65, -55, 71, 71, 99, -73,
    -13, 5,
  -41, -74, 59, -27, 2, -34, -5, -42, 96, -80, -11, -20, 43, -93, -10, -33,
    -27, 14, -73, 86, 72, 78, 13, 83, 31, 43, 3, 0, 30, 53, 92, 16, -68, -68,
    -67, -54, -81, -18, -97, -24, -65, 41, 75, -40, 93, 69, -94, -9, 6, 23,
    -78, -3, 14, 9, -10, -2, -66, 35, -58, -28, 88, 43, 5, -52, 21, 20, -8,
    -32, -29, -52, 11, 75, -24, 91, 55, -54, -71, 70, 77, -31, -46, -56, -40,
    89, -91, -9, 73, 57, 63, 53, 96, -55, -46, -37, 47, -68, -26, 92, 81,
    -49, 85, 38, 93, 11, 3, 75, -77, -57, -45, -19, 91, -61, -46, 23, -47,
    -86, 89, -69, 10, 31, 49, 27, 24, 77, -48, 22, 82, 26, -95, -37, -44, 36,
    14, 43, 36, 58, 35, -25, -15, 19, 54, 96, -51, 76, -71, 24, 37, 23, 89,
    -80, -64, 67, -21, 93, 92, -9, 45, -32, 51, -52, -30, -67, -92, -59, -50,
    67, -29, 15, -31, -39, -25, 16, -88, 12, 77, 37, 78, -92, -100, 16, 83,
    4, -33, 88, -97, 6, 61, 12, 64, 0, -63, 35, -97, -50, 93, 65, 20, -56,
    87, 84, 59, 82, 26, 46, -51, -67, 26, -54, 71, -68, -78, 50, -82, 16, 9,
    -35, -56, -12, -78, 69, -2, -8, -29, 39, 15, -48, 63, 81, 23, 43, -37,
    -99, -70, -28, 40, 80, -4, 49, -98, -10, 32, -16, -87, 46, 72, 94, -95,
    77, -55, 48, -4, 31, 79, -77, 2, 56, 4, 6, -17, 26, -42, 49, -57, 59,
    -28, 4, 89, -48, -2, 81, -10, -74, -13, -75, 72, -4, -32, 67, -3, 8, 35,
    53, 48, -59, 10, 40, 0, 76, 13, 4, 48, 32, 9, 21, 79, 39, -57, -48, 32,
    -59, 20, -87, 73, 6, 16, 90, -82, 90, 85, -91, 24, -36, -75, 31, 57, -24,
    -25, -66, 23, -10, 22, 58, 13, 62, 0, -1, 84, 21, -57, -13, 26, -57, 12,
  26, 15, 96, -17, 78, 27, -37, -96, 0, 95, -67, 48, 71, 65, -43, 39, -99,
    28, -2, 83, 13, -19, -63, 17, 83, 47, 76, 21, 17, -73, -2, -19, 38, -2,
    -89, -46, -80, -54, 67, -3, 69, 44, -96, 56, -15, -84, -49, -13, 22, 63,
    58, -37, -97, -86, -89, 87, -33, 95, -51, -67, -79, -75, 5, 90, -70, 54,
    -97, 80, 35, 11, 48, -49, -99, -63, -93, 41, -56, -63, -19, -7, -26, -8,
    -65, 99, -15, -30, 40, -41, -92, -58, 22, 23, 23, -90, 48, 69, 71, 1, 11,
    -68, -75, -33, -16, 59, -89, -55, 43, 19, -42, -100, 0, -43, -26, -17,
    -37, 52, -90, 12, -95, 54, 17, -80, -52, -2, -19, 72, -5, 80, 93, -89,
    89, -90, -39, -51, 0, 40, -87, 77, -12, 59, -19, 56, 81, 64, 87, -5, -3,
    62, 98, -31, -37, 54, -3, -29, -72, -9, 12, 80, 91, 46, -26, -75, 13, 98,
    2, -16, 44, -54, -7, 89, -48, -31, 51, 1, -90, -69, 8, -70, -34, -8, 59,
    -60, -15, -64, 74, 12, -80, 67, -14, 82, -85, -38, -69, 0, -23, -42, -51,
    50, -52, -17, -30, -58, -60, 41, 40, 29, -65, -35, -71, 82, -27, 5, 14,
    8, 19, -25, 84, 40, -1, 60, 10, 12, 46, -98, -55, 42, -71, -88, -52, -29,
    23, 59, -26, 62, 13, 56, -75, -53, -49, 73, 88, 9, 52, 10, -43, -21, -71,
    94, -58, 79, -81, 3, -85, -2, -6, 66, 82, -32, -68, -79, 2, 29, -36,
    -100, -17, -33, -98, -22, -44, -1, 54, -22, -75, -30, -99, 75, -57, -61,
    48, -8, -59, -71, 51, 48, 36, 37, 62, -37, -51, -9, -26, -29, 54, -19,
    -13, 66, -46, 48, 80, 40, -79, 98, 14, 71, 82, 46, -60, -8, 87, -69, 90,
    -59, -17, -4, 0, 26, 96, -14, -25, 27, -15, 88, -59, 46, -90, -95, -4,
    -38, -99, -34, -90, 22, -26,
  -62, 88, 95, 70, -30, -68, -78, 27, -43, -85, -43, -97, -64, -49, -96, -36,
    -52, -93, 89, 70, -48, -62, 8, 21, 71, 67, -48, -56, 79, 82, -83, 29, 65,
    30, 64, -7, -16, -59, 79, -31, -95, 41, 66, -36, 11, 85, -91, -97, -82,
    55, 56, -21, -49, 37, 89, 99, 26, -76, 0, 58, -67, -100, -18, 28, -9, 41,
    -57, 51, 60, -87, 55, -98, 14, 43, 72, -83, -56, -67, 64, -24, 64, -80,
    61, -80, -22, 13, -38, -11, 51, 3, 66, 94, 26, -31, -4, -59, -23, 61,
    -62, -36, 19, -78, -61, 76, -33, -7, 67, 36, -43, -59, -35, -56, 34, -19,
    -35, -29, 38, -42, 82, -61, -7, -28, 91, 84, 87, 80, -54, 11, 63, 85,
    -31, -85, 22, -88, -66, -10, -27, -25, -48, -1, -78, -79, 62, -1, -73,
    14, -43, -32, 53, -89, 78, -64, -64, -33, 10, 20, 51, 22, -58, -91, 85,
    -37, -31, 86, -51, -95, 80, 91, 25, -88, 32, 46, 7, 99, -32, -76, -57,
    -57, 7, 79, -35, -36, 60, -26, 82, -18, 3, 54, 91, -51, 69, -31, -97, 41,
    89, -53, -88, 72, 48, 99, 35, 14, 22, 26, -32, -46, 17, 94, 13, 44, 16,
    89, 63, -29, -40, -34, -2, 25, -33, 46, 46, 14, -94, -35, 60, -17, 80,
    -56, -77, -62, 1, 52, 22, -74, -18, 18, -97, -59, 80, -29, -53, -92, 24,
    81, 47, 19, 29, -12, -74, 63, 27, 12, -45, 80, -4, -67, -42, 19, 7, 23,
    13, 15, 30, 94, 8, 81, 96, 78, 6, -67, 74, 47, 79, 50, 82, 84, -34, 80,
    -5, 47, -22, 9, -86, 34, -52, -54, -72, -2, 2, 2, 81, 79, 27, 78, 83, 60,
    -38, -90, -18, -89, 29, -70, 86, 26, -48, 32, 31, -64, 55, -9, -52, 92,
    67, -34, 26, 99, 24, 29, -51, 82, 36, 6, -50, 69, -70, 81, 40, -62, -91,
    71, -35, -66, 39,
  -91, 19, 75, 12, -63, -30, 71, -20, -35, -57, 55, 56, -98, -6, 59, -91,
    -35, 8, -46, 39, 98, 99, 30, 78, -56, -82, -48, 83, 94, 89, 12, 67, 58,
    78, 76, -87, 13, -58, 76, 86, 53, 30, 79, 60, 62, 92, 73, 89, -44, 95,
    -49, 84, -2, 94, 14, 86, -94, -47, -71, -39, 8, 87, 16, -67, 64, -73,
    -49, -72, -35, 21, -16, 23, -41, -31, 95, -34, -86, -87, -84, -86, 48,
    -41, -48, 17, -74, 69, 6, 57, -28, 77, -97, -52, -55, 58, -19, -50, -35,
    17, -64, 53, 51, 17, 34, 3, 13, 35, 49, 37, -53, 90, -2, -59, -77, -19,
    -78, -63, 81, -26, 1, 39, 73, -73, -22, -53, -45, -71, -1, -92, 3, -89,
    -56, -51, 31, 81, 68, 33, 79, 72, 58, 23, -71, 65, 42, 44, 58, -61, 98,
    8, 49, 57, 78, 70, -23, 41, -36, -48, 97, -55, -6, -25, -23, 7, 71, 84,
    -18, -90, 21, -67, 31, 20, 73, -77, 81, -76, -39, -15, 10, -31, -26, -97,
    -92, 49, -72, -96, 15, 89, -16, -12, 95, 7, -73, 80, 31, 8, -43, -79,
    -71, -53, 75, 80, -24, 87, 21, 32, -42, 88, 81, -61, 62, -69, -31, -62,
    91, 9, 67, 21, 57, -40, 62, 86, -20, -8, 30, -86, -100, -44, 64, 75, 5,
    -84, -37, 79, -63, -94, 91, -71, -97, 17, 82, 29, -20, -43, 50, 90, -100,
    96, -30, -92, 46, -79, 29, 26, 98, -81, -23, 79, 36, 11, -53, -100, -18,
    54, 56, 34, 0, 58, 19, -30, 46, 35, -84, 18, 44, -23, 32, -76, -69, 55,
    -66, 26, 46, -27, 99, -37, -27, 91, -27, 70, -10, 36, 85, 37, -95, 89,
    -53, -81, 97, -59, 68, 56, 67, 92, -77, -47, -5, 82, -77, 38, 34, -95,
    -54, -61, 49, 99, -63, -33, -21, 85, 73, -17, 54, 14, -50, 41, -31, -4,
    45, -17, -71, 26, 3, -13, 95,
  -23, 70, -40, -6, 38, -75, 97, 36, 41, -29, 1, 89, 13, 77, -31, 68, 68,
    -32, 24, -9, 17, -25, -8, 34, 12, -41, -72, 49, 40, -87, 10, 27, 1, 40,
    76, 5, -39, 42, -11, -92, 59, 25, -93, 42, -36, -36, 76, 28, -65, -99, 1,
    -54, -32, 68, -26, -82, -44, -99, 80, 44, 88, 86, -7, -70, 43, -32, 62,
    61, 74, -8, -49, 36, -23, 83, 44, 63, 71, 52, -37, -28, -28, -100, 82,
    -10, 2, 6, 33, 49, 3, 63, -2, -40, -48, 16, 92, -57, 5, -69, 75, -72, 94,
    -42, 20, -25, 79, -66, 63, 78, 78, 79, -60, -80, 37, 98, -55, 65, -52,
    88, -74, 25, 78, -6, -96, -40, -25, -72, 84, -42, -62, 73, -51, 5, -14,
    5, -13, -91, -51, -63, 63, 28, -74, -44, -74, 84, 93, -9, 67, -60, -4,
    -5, 41, -62, -89, -22, 79, -97, 23, 18, -90, -66, 16, -79, -95, -59, -59,
    -1, -96, 84, 50, -19, -18, 40, -91, -64, 45, 54, 62, 99, -41, 64, 78, 29,
    -46, 7, 31, 19, -97, -68, 78, -13, -40, 22, 95, -2, -11, -78, -48, 21,
    59, -50, -73, 14, -89, -18, 81, 49, 33, -38, -51, -50, -69, 73, 16, 43,
    -10, 6, 82, -20, -72, -14, 38, 6, -45, 62, -6, 0, -73, 1, 4, 40, 95, 4,
    -73, 58, 67, -69, 31, 30, -3, -47, 11, 36, -6, 44, 61, -41, 23, -44, 8,
    14, -87, -11, -74, 7, -76, -14, 18, 11, -7, -2, -96, -45, 82, -70, 26,
    18, -3, -31, 28, 15, -29, -85, -87, -2, -12, -47, -66, 79, 81, 67, -8,
    86, 19, -70, 54, -63, -46, -75, 40, -66, -9, -79, 55, -7, -60, -90, 77,
    2, 86, -28, 19, -13, 0, 7, 81, 5, -93, -92, -79, 67, -43, 2, -58, -24,
    97, 2, -54, 32, -66, -30, 89, -84, 0, 35, 61, 48, 85, 44, 31, -58, -59,
    98, 94,
  38, 49, 49, 86, -40, 55, 44, 59, 12, -18, -79, 54, -78, -92, 56, -7, 51,
    -97, 40, 66, 77, -58, 34, -28, -13, -4, -79, -80, -11, 95, -74, 41, -96,
    -75, 60, 28, 54, -10, -75, -90, -18, 43, 86, -17, -81, -82, 23, -60, -43,
    -87, -65, -82, 34, -63, 20, -94, 24, 63, -61, 96, -46, -30, -97, 40, 66,
    17, 77, 21, -44, 45, 60, -28, 15, -71, -25, 31, 55, 21, 10, 58, -27, 84,
    -70, -61, -100, -62, 9, 97, 77, -83, -94, -67, 68, -52, 14, -44, -10,
    -81, 56, -86, -76, 30, 64, 31, -39, 77, -49, 96, -44, -1, -14, -63, -8,
    23, 29, -77, 31, -8, 7, -71, -4, 46, -88, 84, -13, -21, 26, -78, -51,
    -21, -6, 41, -59, -10, 15, 25, 64, 95, -2, -29, -59, 48, 0, -40, 23, -18,
    -70, -24, 35, -12, 0, 55, -43, 36, -2, -51, -60, 91, -84, -76, -97, 95,
    84, 98, -16, -56, 58, 50, -36, -18, -39, -4, -30, 17, 97, -62, -20, 92,
    -18, 17, 15, -61, 57, -98, 78, 95, -60, 36, 23, 2, -78, -7, 96, -96, 69,
    -75, -20, -58, 20, -100, -2, -72, -40, -26, 83, -41, 1, 76, -94, 18, 69,
    -85, -5, -56, 30, -33, -77, 11, 1, -48, -82, 83, 27, -38, -61, 93, -36,
    36, 84, -81, 37, -55, 94, -41, -55, -38, 56, -19, 99, -13, -98, -18, -81,
    -78, 82, -17, -79, -74, 89, -22, -35, -92, 25, 78, -63, 38, 92, 21, -65,
    -59, -32, 0, 79, -67, -31, -6, -74, 37, -10, -66, 75, 54, 18, 69, -4,
    -71, 28, 36, 15, 63, -95, 99, 67, 57, -52, 67, -11, -61, -39, 50, -34,
    -43, -82, 19, 74, -38, 69, -43, -37, 67, 32, -65, 55, -19, 82, -92, 35,
    -64, 59, -30, -94, 72, 34, -76, -81, -82, 91, 63, -65, 83, 99, 79, -40,
    -88, 67, -74, -9, 14, 81, 19, 32, -98, -33,
  26, -66, -78, -43, -29, 42, 83, 45, -48, -92, 29, -28, 14, 3, -59, -78,
    -83, -48, -47, 35, 45, 52, -85, 37, 2, -91, 33, -19, 11, -21, -45, -48,
    -38, -12, 60, 15, -74, 9, -28, -49, -31, 67, 19, 75, 16, 77, 43, -86, 71,
    -15, -45, 16, -48, -32, -28, 32, 40, -52, 46, 60, -12, 72, -46, 87, -54,
    26, 47, -36, 33, 79, -55, 50, -54, -21, -23, 61, -32, -34, -45, -25, 56,
    28, -56, -2, -15, -92, 89, 69, -25, 15, -74, 59, -85, 84, 26, -29, -40,
    75, -85, -61, 81, 43, 95, 44, 68, -63, 30, -94, 63, 36, 29, -77, -12, 9,
    -90, 18, -99, -15, 41, -68, 83, -56, 21, -96, 81, -31, 78, -27, 8, 38,
    -78, -76, -8, -45, -46, 54, -79, 22, 89, 45, -89, 94, 19, -21, 30, -46,
    -67, 2, -93, 82, 20, 27, -65, -13, 54, -36, -13, 87, 46, -50, 72, -63,
    73, -77, -46, -94, -36, 30, -30, -27, 86, 16, 21, 4, 25, -55, 89, 52, 69,
    -20, -55, -10, -62, 10, 90, 19, -98, -16, -28, 68, 72, 48, -61, 40, 90,
    -38, -38, -82, -39, -40, 34, -19, 27, 76, -78, -28, 37, 77, -99, -20,
    -53, -7, -61, 59, 79, -36, -1, 32, -21, 21, -18, 3, -12, 10, 63, -74,
    -18, -40, 68, 73, 87, 82, 41, -27, -22, 60, -66, 81, -82, 53, -90, 99,
    -97, -16, -19, 24, -12, -99, 59, -29, 42, -14, 59, 67, -54, 69, -35, 6,
    -99, -79, 42, -6, 71, 85, 84, -25, -26, -98, 12, -98, -84, 39, 0, 54,
    -67, 4, -50, -59, -56, -3, 34, -64, -24, 63, 91, 57, 60, 44, -88, 23, 4,
    -60, 54, -52, -83, 88, -59, -79, 8, 48, 80, -41, -86, -38, 18, 54, 85,
    -31, 57, -45, -15, 61, -41, -61, -65, -2, -38, -53, -13, -75, -26, -98,
    81, 79, -83, -14, -82, -24, -20, 52, 39, 66, 26,
  52, 52, -91, 6, -52, -31, 31, -65, -98, 7, -29, 36, 29, -34, -52, 29, -89,
    43, 24, -97, -66, -73, 12, 96, 34, 2, 66, -52, -78, -88, -19, -24, 64,
    -35, -22, 31, 23, -18, 31, -38, 35, -8, 90, -75, 75, -61, -61, -23, 53,
    13, -49, 24, 30, 88, -82, -67, 78, 19, -82, 34, 46, 34, -72, -35, -18,
    -20, 16, 56, 24, -85, 1, -79, 13, 42, -52, -95, -42, -35, 23, 10, -34,
    87, -5, 12, -78, 12, -83, 41, 81, -27, 48, 84, -13, 87, -56, 22, 90, -13,
    -9, 35, -3, -45, -100, 8, 44, 48, -78, 85, 50, -44, 70, 74, 84, -12, -55,
    31, 59, 17, 41, 5, 54, -1, 20, -20, 81, 42, 65, -2, -11, -30, -23, -97,
    60, -11, -83, -53, -74, 55, 91, 38, -73, 48, 21, 73, 62, 55, 60, 53, -54,
    -53, -44, -21, -30, 63, 31, 57, 92, 51, -38, -77, 65, 9, -78, -25, 40,
    -67, -5, -22, 2, -28, 73, -29, 33, 46, 24, -36, -58, 88, 24, -84, 73,
    -90, 81, -96, 90, -2, 94, 47, 84, 0, -46, 6, -27, 58, 41, -73, 44, 63,
    -35, 4, -87, 71, -53, 56, -53, -69, -10, -61, 57, 15, -59, 13, -15, -21,
    45, -78, -11, 2, -70, 18, -20, 20, 19, -91, 88, -55, -35, -26, 78, 14,
    42, -36, -94, -87, 1, -13, -49, 75, 45, -29, 70, 34, -12, 77, -19, -78,
    -95, 98, -29, 73, -16, -21, -53, -73, -43, -92, 88, -75, 64, -87, 74,
    -94, -5, 99, -76, 91, -92, 35, 87, -34, 12, -67, 95, -21, 77, -6, 87,
    -65, -77, 76, 89, -80, -69, 67, 52, -89, 30, 37, -23, -100, -35, 32, 97,
    -17, -95, 78, 25, 71, 24, 97, 28, -42, -8, -87, -30, -25, 89, 34, 3, -84,
    -84, -85, -51, 62, 72, 28, 4, 83, -26, 96, 19, -95, 50, 81, -27, 31, 59,
    -7, -98, 75, -52, 50, 89,
  38, 56, 32, -99, -68, 75, 3, 32, -76, 12, 47, 75, -14, 7, -22, 84, 85, 33,
    19, 10, -96, -52, -2, -66, 9, -10, -66, -77, 7, -58, -72, -66, -21, -63,
    46, 65, 28, 34, 80, 19, 24, -88, 50, 20, -3, -38, 73, -24, -27, -2, 72,
    -3, 15, 79, 79, -21, -52, 33, -56, -87, 97, -59, 67, -56, -86, 7, 35, 93,
    -19, -81, -29, -76, -75, -94, 97, -8, 24, 79, -35, 4, 63, 43, -84, -2,
    -72, -31, 8, 30, -38, -36, -6, -47, 50, 8, 95, 29, -17, -23, 94, -29,
    -31, -96, 89, -46, -87, -28, -98, -9, -83, 1, -50, -47, -52, 40, -81,
    -61, -27, -10, 31, -73, -57, -100, -80, 85, 35, -37, -33, -67, -13, -25,
    82, -57, -18, -23, 19, -71, 16, 96, -83, 79, -2, -89, 5, 89, -73, 5, -37,
    57, 42, -56, -35, -87, -81, -57, 94, 85, -14, 2, -41, -60, 13, -10, -61,
    2, -17, 6, -94, 91, -88, 10, 24, 86, -97, 24, -60, 81, 48, -42, 87, 75,
    -88, 84, 77, 72, -60, -56, -5, 90, -39, -26, -4, 45, -78, -50, 8, -35,
    -30, 88, 19, -14, 70, 14, -20, 54, 24, 85, -73, 62, 94, -90, 58, 43, 19,
    -75, -46, 95, -57, -61, 30, -29, -36, 21, -51, 23, 22, -72, -85, -4, 39,
    -13, -40, -46, 31, -36, -35, -76, 4, -22, -27, -26, 11, -18, 70, -75, 95,
    -30, -48, -40, -83, 83, 34, -44, 55, 28, -52, -17, 43, 8, -25, 15, 50,
    -96, -99, 74, -81, -50, -85, 93, -73, 62, -47, 32, -50, 76, 67, 47, 42,
    19, -88, 4, 10, -18, -89, 23, 94, 66, -31, -33, -24, -50, -73, 27, -97,
    34, 13, -48, 17, -74, -22, 66, -72, -92, -3, -12, -21, 35, 9, -100, 97,
    -78, 12, 40, -46, -2, 47, 43, 64, 93, -66, 57, 14, -31, 80, 1, -22, -100,
    -23, -86, -11, 7, 80, 62, 99,
  -49, -47, 40, 74, 17, -71, -26, 32, -6, 25, 26, -88, 5, 3, -61, -81, 24,
    96, 77, 69, 46, -21, -35, -76, 80, 27, -8, 39, 93, 27, 79, -37, 97, -70,
    -50, -25, 90, 3, -28, -14, 89, 82, -59, -27, 57, -55, 58, 81, -70, 98,
    -87, 53, 55, 26, -42, 28, 95, 68, 50, 78, 73, -69, 27, 89, 78, 34, 14,
    -20, 66, 33, -69, 67, -71, 86, 46, 83, 53, -32, -68, -69, -89, 63, 82,
    -83, 90, -11, -76, -51, 9, 11, -59, -73, 80, -39, -45, 39, 98, 62, 75,
    41, -42, 27, -58, -25, -63, -14, 99, 99, -38, -63, 88, 0, 36, 24, 5, 40,
    -87, -63, 52, 3, 27, -10, -49, 94, 72, 10, 75, 51, -54, -31, 86, 90, -89,
    83, -69, 8, 47, -22, -98, -62, -53, 91, 73, -31, -73, -59, -46, -7, 25,
    56, -8, 24, -71, -91, -60, -13, -5, 62, 76, 35, -34, 59, 88, 42, -94,
    -92, -38, 19, 96, -22, -80, -27, 61, 29, 13, 70, -63, -1, 10, -52, 48,
    -45, 4, -93, 98, 71, -48, -40, -89, -68, 11, -19, -89, -25, 83, -59, 93,
    85, 91, 22, -24, 90, -91, -96, -85, 30, 19, 56, -41, -31, -86, -26, 77,
    54, -51, 4, 31, -10, -72, -73, -13, 96, 50, 20, -25, 14, 75, 94, -50,
    -52, 26, -93, -64, -11, -73, -96, 89, -30, -14, -4, -12, -67, 95, 76, 94,
    -98, -90, 21, 77, 7, -50, -32, 54, 86, 30, -12, -80, -89, -79, 75, 60,
    28, 39, -1, 86, -56, 56, 41, 6, 19, -33, -18, 77, 21, -7, -33, -27, 15,
    -2, -61, 40, 75, 49, 69, -26, -82, -36, 30, 69, 66, -90, 72, -96, -72,
    55, -60, 57, 19, -1, -24, -26, -95, 40, 0, 72, 65, 61, -25, 12, 71, -11,
    1, 28, -99, 40, 9, -12, 73, -78, 75, 41, -29, -51, -60, 12, 67, 49, -82,
    -32, -59, -84, -91, 49,
  -55, 95, -57, 8, -76, 40, -52, 24, 37, -75, -99, -48, 34, 45, 48, 39, -33,
    40, 6, -83, -14, 40, 43, -95, -68, 49, -98, -38, -78, -65, 25, -79, 63,
    19, 7, -25, -90, -8, -84, 22, 32, -62, -38, 38, -67, 93, 70, -71, -72,
    -80, -89, 56, 54, -17, -96, -7, -28, -83, 63, -62, -6, 8, 43, -90, 18,
    -4, 52, 3, -56, -44, 72, -100, 55, 2, 4, 2, -58, -15, 58, -51, -45, 9,
    -47, 10, -43, -38, -69, 7, -20, 22, -92, -79, -50, 36, -65, -61, 26, 60,
    -80, -86, -3, -98, -92, 83, 84, -85, -76, -78, -14, -23, -67, 31, 6, 36,
    -21, 4, 88, -54, 89, 16, -49, 40, -76, 54, -85, -43, 27, -15, -24, -78,
    -65, 2, -37, -36, -73, -8, 65, 85, -9, -70, -99, 11, -93, 90, -82, 0, -6,
    -16, -21, -67, 25, -25, 83, -35, -2, 93, 82, -80, -68, 2, -11, -66, -12,
    -8, 34, -37, 24, -78, 40, -88, 38, 47, 79, -67, -14, -43, -67, 48, 8,
    -99, -19, 26, 27, 45, 34, -63, 4, -14, 18, -13, -52, -40, 93, 67, -59,
    -67, 85, -1, -20, -38, 36, -67, -44, -35, -65, -73, 16, -93, -70, -45,
    62, 62, -39, 40, 25, 3, 67, 17, -98, -67, -34, -13, -89, -2, 71, 97, -91,
    -66, 0, -17, 42, 14, 44, -3, -64, -56, -85, -100, -2, -59, -29, 95, 9,
    -67, -72, 88, -61, 85, 4, 43, 92, -90, -92, -39, 82, 29, 53, 98, 69, -33,
    -100, -58, -45, -40, 43, 56, 71, -90, -15, 51, 91, -61, 50, -20, 24, 84,
    41, 16, -79, -57, 38, -75, 99, 61, -2, 73, 86, 7, -91, 52, 2, -50, 20,
    98, -12, 45, 3, -41, 61, -77, 24, -16, -98, 59, -22, 42, -84, 8, 30, 3,
    -90, 1, 56, -51, 72, -98, 6, 64, -44, 84, 44, -69, -61, 18, 5, 96, -25,
    80, 32, 97, 45, 2, 5,
  -49, 89, -29, -40, 61, 81, 81, 35, -38, 9, -45, 12, -88, 48, 93, -22, -5,
    60, -78, -50, -2, -18, 65, 45, 62, 37, -84, 1, 87, -43, -8, 22, -79, -31,
    84, 87, -77, -9, -27, -12, 83, -36, -35, 10, -18, 75, 31, 94, -1, -96,
    -28, 58, -97, -23, -73, 77, -20, 10, -60, 63, 64, -58, 36, -62, -44, 11,
    76, 5, 55, -68, -99, -56, -96, -7, -85, 62, 33, 34, -5, 98, -73, -27, 18,
    61, -99, 53, 20, 28, 99, 90, 33, -62, 48, 51, 84, 0, 18, 51, -46, -65,
    29, -65, -80, 6, -64, 35, 28, -34, 52, -33, 49, 82, 77, -25, 58, -42,
    -57, -37, -33, -48, -70, -13, 56, -33, -24, -67, -31, 64, -53, -86, 56,
    49, -89, -4, 83, -48, 56, 76, -12, 48, 39, 14, -64, 94, 88, 55, -35, 54,
    53, -48, 22, -4, 12, 69, -36, 32, 31, -26, -98, -81, -24, -79, 68, 0, 99,
    0, -42, -6, -74, 97, 18, -92, -5, -11, -64, 63, 56, -26, -75, -11, -38,
    14, 70, 55, 85, 56, 8, -59, 50, 65, 96, 14, 66, 17, -77, 98, 92, -15,
    -97, 9, 92, 8, 92, -41, 42, -84, 92, 65, 42, 69, -55, -13, 33, 66, -83,
    96, 61, -87, -79, 1, -80, -95, 47, 15, 99, 49, -41, 29, 50, -50, 97, 86,
    -49, 78, 46, 55, 57, -94, -32, -96, -97, -90, -53, -4, -1, 11, 93, 48,
    33, 42, 9, 81, -1, 70, 86, -53, -68, -96, 81, -81, -66, -29, -57, -98, 7,
    64, 12, 48, 29, 97, -3, 75, -41, -31, -81, 85, 75, 29, 88, 68, -35, 94,
    27, -65, -37, 55, 58, 24, -51, -61, 82, 72, -91, 74, -15, -45, 94, 49,
    -24, -42, -25, -42, 86, 56, 77, 72, 92, -6, -89, 58, -52, 80, -33, 25,
    -78, 75, 51, 75, -56, -75, -83, 15, 87, -54, 84, 23, -20, -81, -41, -22,
    20, 82, 11,
  -51, 35, 43, 66, 39, -72, 10, 57, -99, 37, -73, 83, -6, -65, 58, -39, -99,
    -45, -42, 87, -83, -93, 77, -26, -21, -39, 27, -64, -31, 46, 50, 70, -46,
    -43, -96, -36, -51, 85, -5, 93, 69, -75, -20, 9, 39, 1, 80, 19, 51, -3,
    75, 2, 76, -47, 67, 51, 36, -61, 20, 15, -74, 38, 0, 20, 60, 38, 95, -18,
    26, -19, 68, -25, -99, 42, 64, -94, 48, -53, -93, 84, -49, -75, 42, 89,
    -41, 5, 6, -8, 89, -74, 71, 43, -41, -15, 3, -75, -82, 3, -69, -37, -67,
    -20, 71, -89, -100, 11, -95, -36, 70, 87, 90, -40, 99, 71, -14, 93, -80,
    -19, 7, -17, -17, -45, -25, 43, 72, 96, 64, 81, 52, -43, 81, 26, 10, 29,
    -28, -31, -5, -98, 84, 31, 7, -45, -50, -96, 40, 74, 42, 26, -66, -90,
    -35, -67, -16, -91, -30, 30, -72, 94, -86, -14, 32, -65, -82, -13, -88,
    22, 76, 37, -27, -78, 86, 33, -51, 74, 68, 94, -82, 29, -78, -2, 83, -78,
    -46, 59, 91, -15, 57, -39, -21, -3, 27, -9, 39, -90, -82, -86, -28, 98,
    -58, -99, -51, -56, 57, -55, -50, -10, -90, 98, -94, 32, -100, -26, 26,
    -2, 77, -86, -33, 84, -12, 42, -82, 20, -31, -99, 48, 85, -87, -3, -64,
    -83, -22, 24, 90, 21, 77, 8, 30, -19, 23, 34, 18, -17, -32, 83, -67, -74,
    43, 78, 81, -46, -7, 12, -35, 55, -54, 56, -84, 72, -9, 77, 97, 36, -9,
    -97, -35, 80, -82, -48, 90, -92, -88, 70, -23, -84, -71, 55, -7, 59, -53,
    56, 38, 29, 61, -27, -24, -61, -15, 52, -97, -87, 90, -19, 32, 83, -5,
    99, -62, 21, -25, 71, -2, 48, -37, 17, -48, -53, -73, 11, 45, 12, -89,
    20, 26, 45, -36, 46, -67, -100, -3, 24, -77, -2, -84, -19, -13, -91, -36,
    -9, -74, -31, 54, -74, 10,
  66, -21, -38, 52, 34, -38, 79, -15, -2, 70, -45, -38, 11, 23, -11, 40, -22,
    -65, 83, -67, 50, -42, 21, -85, 71, 75, 12, 53, 83, -72, -39, 44, -61,
    29, 44, 66, -54, 6, -96, -31, -40, -43, 58, -94, 34, 8, -87, 23, 28, -29,
    64, 46, 75, 87, -58, 98, 54, -30, -39, 1, 32, 66, -38, -70, -45, 20, 68,
    -3, 37, -64, -63, 80, -22, 85, 91, 20, 5, -79, -80, 83, 55, -73, -95, 31,
    -28, -23, -46, 59, 24, 64, -94, 1, -72, 1, -67, 36, -34, 47, 34, -95, -4,
    -60, 22, -10, -92, 66, -11, -22, 16, -17, 25, -72, 39, 91, -96, -37, 45,
    54, -73, 50, -32, 40, -26, 43, -40, 31, -71, 86, -66, -56, -80, 36, 92,
    92, -56, 57, -25, 85, 94, -58, 27, -70, 81, 61, 82, -52, -64, 93, -17,
    66, 18, 88, -62, -19, 53, -58, -11, -25, -98, -84, -75, 87, 93, 55, 33,
    -7, -79, -12, -30, -84, -46, 94, -10, -52, -77, -94, -45, -25, -55, -84,
    -25, 15, 44, 53, 32, -18, -55, 30, -71, 28, -7, -72, 14, -83, 39, 48,
    -74, 8, -85, 99, 39, 38, -62, 49, 95, 82, -71, -24, -50, -46, 14, 66, 69,
    83, -100, -92, 17, 85, -53, 97, 46, 55, -62, 18, 53, 38, 39, 18, 70, 13,
    -52, -65, -64, -16, 44, -100, 92, 96, 72, 43, 1, 8, -30, -51, 72, -18,
    29, -37, -8, -21, 58, -18, -69, -45, -6, -6, -11, -12, 81, 5, 35, 74,
    -80, -77, -37, -29, 85, -22, -31, -19, -21, -93, 9, 75, 35, -11, -55, 68,
    -8, -99, 51, -25, 49, 45, -71, 50, 24, -36, -85, 54, -6, 2, -21, 14, -74,
    90, -46, 85, -13, -17, -72, 33, -11, 60, 60, -48, 28, 66, -60, -28, -74,
    66, 4, 72, 67, -71, -13, -20, -60, 79, 85, -54, 11, -26, -55, 74, 12, 76,
    -26, -49, 80, 95, 68,
  -16, -58, 83, 56, 22, 85, -36, 93, 28, 58, -15, -19, -6, -78, -44, 62, -3,
    73, 86, -38, -65, -28, 93, 88, 87, 98, -70, 22, 59, -12, -96, -60, -88,
    24, -18, -10, 61, -66, -81, 1, -91, -4, 86, 97, 28, 63, -73, -99, 63,
    -19, -2, -52, -66, -23, -38, 0, -46, -80, -26, 15, -84, -53, -51, 99, 45,
    94, 88, -27, 42, -32, -89, 86, 47, -29, -99, 90, 94, -99, -15, 71, -91,
    81, -22, -40, 16, -20, -40, 32, -43, -40, -32, 99, -49, -64, 94, 63, -32,
    29, 26, 93, 58, -33, 28, 20, -39, -45, -64, 38, -23, -45, 63, 57, 5, 34,
    36, 65, 84, 92, 58, 89, 93, -23, -65, -2, 29, 10, -51, -22, -40, 59, 53,
    -69, 97, -37, 35, -53, 59, 92, 69, 86, 82, -72, -12, 17, 29, 67, 95, -76,
    -96, 82, 66, 40, 63, 14, -95, -68, 69, 72, 14, -70, -56, 51, -38, -4, 63,
    -24, -35, -77, -1, 67, -26, 32, 59, 25, 87, 85, -6, 35, 24, 44, 90, 76,
    72, 69, 66, -12, -93, -29, -56, 14, -52, 64, 14, 75, 0, -50, -49, 78,
    -100, 99, 94, 89, -47, -59, 56, 2, -42, 8, -3, 72, 70, 41, -3, 5, -34,
    43, 13, -33, -4, -43, -23, 25, 53, -48, -6, -23, 81, 12, -57, -18, -66,
    -1, 49, -19, 90, 40, 76, -46, -11, 20, -99, -47, 65, -40, 53, -67, -52,
    -11, 35, -3, 48, 74, 81, -11, -63, 77, -80, -83, -85, -20, -3, -9, 74, 6,
    -75, -21, 94, -39, 18, -7, -84, -77, -29, 97, -3, 2, 42, 23, 66, 63, 82,
    -35, -5, 67, -24, -78, -1, -72, 79, -94, -53, 64, -70, -57, -22, 79, -35,
    67, -92, 10, 33, 86, -13, 41, -70, 32, -88, -9, 25, 40, -12, 67, -64,
    -32, -59, 60, -76, -3, 96, -26, -39, 39, -90, 72, -100, -20, -5, 79, 93,
    -1, -30, 12, -98,
  -99, -14, -53, 26, -47, -34, 24, 98, -52, -89, -45, 53, 23, 89, -53, 65,
    -69, -48, -26, 93, 65, 79, 65, -97, 25, -26, -64, -16, -32, 29, -8, 89,
    -32, 39, 28, 60, -87, -88, 72, -60, -50, -46, -39, -4, 39, 72, 33, -35,
    50, -58, -38, 23, 70, 35, 30, -38, -49, 85, 58, -56, -38, -29, -11, -32,
    -88, -30, -73, 27, 23, 54, 92, 76, 41, -86, -12, -39, 87, 74, -4, -72,
    85, -29, -15, -30, 24, 83, -18, -54, -35, 43, -89, -26, 39, -60, 59, 40,
    22, 77, -27, -11, 70, 26, -2, -40, 89, 94, -20, 25, 66, 29, 66, 6, 73,
    55, 78, 4, 22, -42, -15, -29, 72, 4, -60, 62, 4, -4, 47, -95, -82, -14,
    1, 52, -82, -4, 14, -22, 48, -28, 87, -83, -34, -65, 36, -8, -11, 24, 15,
    60, 59, 69, 67, -7, 33, -28, 62, 25, 97, -92, -65, 59, -59, -79, 69, -51,
    31, -5, -58, 89, -18, 14, -83, 19, -46, -7, 28, 82, 0, 46, -86, -77, 56,
    12, 54, 21, -90, -47, -16, -66, 95, 44, 32, 4, 73, 19, 43, 93, -61, 94,
    35, -55, -98, -93, -8, 59, -87, -69, 47, 32, 28, 19, 28, 67, 41, 58, -83,
    -67, -64, -55, 56, -85, -42, -86, 35, 14, -57, 93, -11, 21, -1, -87, 44,
    -6, 71, 16, -46, -17, -62, 16, -21, 66, -14, 45, -73, -68, 84, -50, 82,
    20, 5, -57, 61, -32, 46, 38, 83, -51, 77, -19, -20, 19, -87, 77, -91, 85,
    -58, -95, -40, 76, -96, 9, 40, -70, 43, -33, -84, -49, -42, -41, -79, 89,
    6, 0, 29, -43, -50, 46, -40, -55, -58, -54, -80, 29, -23, -54, -87, -22,
    -78, -33, -55, -74, 27, 77, -5, -38, 55, 26, -33, -98, -57, -94, -56, 54,
    91, -33, -98, -45, 74, -73, 43, 61, -73, -69, -47, -91, -60, 78, 74, 61,
    67, 13, -64, 96, -8,
  -51, -67, -13, 31, -49, -28, 4, 14, 88, -30, 29, 16, 91, 50, 66, 83, 40,
    -65, 35, 90, 73, -69, -47, 47, 33, 93, 97, 53, 43, 60, 48, -83, 25, -46,
    1, -75, -62, -78, 63, -35, -1, 49, -99, -7, 28, 18, -18, 96, -29, -58,
    -80, -67, -56, 47, 10, 75, -97, 66, 14, 81, -22, -88, -65, -38, 22, -42,
    48, -37, -77, -58, -90, 82, 73, 99, -65, -90, 26, -55, 71, -85, -12, 32,
    39, 91, 83, -39, -73, -18, -39, 93, 60, -34, -43, 20, -86, -50, 40, 34,
    77, -59, -72, -95, -67, -25, -3, 49, 41, 41, -49, -77, 24, 17, 13, 23,
    -62, 9, -8, -54, -66, -31, -32, -50, -8, 46, -71, 77, 96, -92, 0, 32,
    -47, -86, 11, 22, -78, 59, -51, 21, 6, -83, -33, -61, 79, 83, -40, -8,
    62, 56, 35, -46, -15, 90, 43, 34, -85, -33, -27, 26, 3, 94, 85, -48, 73,
    29, -100, 93, -50, -3, -51, 73, 4, 48, -51, -64, 34, -31, 46, 77, -13,
    61, -18, -14, 38, 67, 5, 40, -14, -10, 12, -62, -44, -45, 41, 51, 80, 30,
    -28, -33, 49, -46, -99, 84, -89, -65, -24, 75, 2, 71, 32, 97, 0, -54,
    -40, 13, -49, -93, -13, 52, -100, 0, 9, 98, 19, -63, -63, 36, -55, -55,
    -76, 30, 13, 62, -51, 65, 17, 75, -53, 65, -83, -6, -14, -75, 86, -64,
    13, 76, -68, 19, 14, -95, 50, 33, 76, -92, 10, 39, 0, 44, -22, 28, -36,
    -48, 93, 62, -55, 66, -68, 26, -58, 46, 98, 96, 5, 48, -98, 13, 77, -22,
    -65, 8, -6, -93, -53, 59, 5, 38, 17, 98, -40, -36, -66, -100, -65, -74,
    96, 89, 48, 55, 93, -83, -77, -95, -91, 96, 98, 48, 37, -15, 80, 86, 14,
    92, -78, 61, 4, -64, 97, 89, -86, -32, 26, -53, 19, 97, 61, -13, -2, 64,
    -82, 98, 82, 57, 66,
  -59, -93, -97, -84, 45, -36, -46, 98, 74, -19, 43, 9, 2, 3, 37, 99, 88,
    -81, 3, -82, -70, 45, -38, -26, -87, 2, 56, -92, 29, 35, -3, -80, 17, 68,
    -68, -13, 71, 10, -53, -84, 16, -68, -76, 65, 93, -13, 34, 94, 46, 9,
    -90, -85, 54, -43, -57, -26, 52, -84, -17, 29, 78, 35, -6, -29, -48, -70,
    -4, 68, 42, 22, -54, 30, -86, 46, 60, -46, -8, -25, 60, -83, 9, 37, -26,
    75, 97, 94, -7, -95, -93, 69, 56, -2, -77, 24, 0, -18, 45, -1, -50, -15,
    -93, -54, -83, 21, -17, -42, -21, -41, -39, -54, 74, -17, 78, -63, -77,
    -64, -82, 19, 62, 14, 39, -67, -43, -7, -26, -13, -6, -24, 15, -90, -14,
    66, 9, 11, 8, 41, -54, 3, -68, -81, -81, 57, 99, -40, 34, 88, -66, 20,
    30, 32, 22, -64, 41, 27, 14, -35, 53, 1, -78, -49, -52, -2, 86, 14, 86,
    -24, 33, 69, 52, 35, -35, -5, 0, 17, -3, -72, 47, 97, 35, 6, 32, -91,
    -89, 63, -25, 45, 27, 8, 84, -45, 89, 25, -56, -40, -53, 31, 30, 2, -40,
    51, 59, 69, 70, -59, 30, 31, -30, -74, -58, 16, 94, -79, -90, -74, 89,
    -22, -38, 11, 97, 21, 14, -17, -67, 60, 96, 44, -41, -19, -51, -69, -66,
    9, 39, 20, -23, 53, 2, 46, 17, -42, -14, 85, 44, 69, -73, -1, 80, 97, 99,
    53, -17, -46, -60, 70, -19, 69, 98, -53, -28, -83, -52, -26, -17, -30,
    -96, -24, 26, 92, -20, 58, 17, 42, 60, 27, -26, -50, -21, 9, 97, -7, -82,
    -61, -11, 8, -43, -10, -91, 16, 21, -53, -81, 32, 15, -49, -30, -47, -25,
    -68, -70, -10, -86, -16, -40, -33, -81, -47, 17, 33, 46, -38, 34, -42,
    10, -92, -47, -71, -7, -15, -60, 61, -83, 8, 51, 37, -24, 26, -44, -5,
    29, 77, 61, -71, -64,
  78, 3, -55, -40, -68, -10, -55, 59, -62, 58, -8, 38, -56, -89, 99, -18,
    -82, -71, 98, 38, 23, 20, 47, 28, -23, -61, -21, -78, 4, -92, 74, -13,
    72, 53, 85, -52, -28, -29, 70, 39, -23, -83, -76, 28, 23, -100, -95, 98,
    -59, 4, -77, 54, 89, 16, -70, 13, 15, -71, 32, 67, 36, 44, -85, -51, -47,
    -57, 37, 66, 33, -83, 12, 3, 59, 84, 98, -79, 3, -91, 10, -50, -32, 14,
    63, 21, 65, -100, 73, -26, -14, 58, -47, -81, -54, -74, -72, -92, -83,
    -22, -97, -33, -95, 38, 49, -66, -23, -96, -67, -46, 38, 90, 83, -61,
    -63, 48, 74, -93, -29, 58, -70, 0, -12, 36, 59, 72, -78, -21, 27, -22, 1,
    -20, -97, 12, -30, -51, 40, 20, -79, -42, 6, 80, 94, -8, -30, -47, -65,
    -27, 69, 43, 3, 71, 73, 22, 42, 80, 47, -42, 54, 38, -14, -47, 37, -3,
    -59, -37, 38, 72, -44, 33, -62, -33, 62, -11, -16, -47, 41, 6, 49, -96,
    -29, -44, -29, 92, 79, -6, -33, 83, -79, 75, 10, -68, -85, -86, 68, 79,
    63, -33, 27, -79, 50, -50, -62, 40, -5, 65, 29, -87, -20, -5, -62, -4,
    -78, -97, 5, -20, 63, -77, 37, 61, -11, -19, -67, 1, 45, 40, 88, 94, -74,
    39, 84, 0, 58, -79, 50, 20, -88, -32, 61, 1, -42, 99, -82, 81, -44, 44,
    72, 4, 99, 53, 42, -20, -88, 26, 23, 22, -72, 72, -55, -74, -14, 6, -39,
    -5, 49, 68, 43, 6, 87, -23, 58, 38, -41, -56, -11, 97, 87, -94, -37, -58,
    -19, -67, -38, -85, 2, -37, -3, 48, 21, 77, -69, -98, -87, -84, 42, 16,
    -94, -2, -80, -36, -87, -4, 69, 98, -4, 23, -75, 24, -72, 68, 67, 49, 54,
    -72, -25, -99, -95, -66, -96, 70, -17, 26, -83, -31, -9, -34, -9, -15,
    -83, -18, 76, 58, 78, -6, -71,
  58, -10, 10, 55, 30, 1, 38, 44, -23, -94, -49, -83, 18, -78, 88, 53, 56,
    -25, -52, -11, -69, 83, -81, -1, -35, 35, -97, -96, 19, -30, 24, -74, 83,
    79, 82, 23, 26, 40, 55, 2, -82, 80, 50, 18, -26, -35, -47, 14, -63, -64,
    -74, 66, 97, 74, 70, 16, -14, -23, -85, -96, -66, 59, 58, 23, -94, -6,
    75, 56, 42, -13, 97, 67, 15, -90, -59, -34, 33, 94, -29, 3, -70, 42, 58,
    -37, -34, -62, -60, -16, -31, 19, 58, 56, -8, -47, 29, 43, -26, 26, -65,
    56, -48, -76, 75, -79, -50, 75, 71, -9, -16, -9, -40, -44, -100, 19, -71,
    -73, 25, -68, 40, -100, -86, 76, 45, 65, -4, -11, 91, 16, 12, 21, 52,
    -49, -38, 70, -2, -65, 34, 2, 19, -21, 46, 9, -67, 63, 44, 99, -58, 18,
    79, 87, -2, -43, -33, -62, -9, -79, -32, 17, -56, 82, -8, -88, 65, -94,
    -87, -58, -76, 95, -96, -24, -20, -57, 30, 80, 17, -6, -11, 71, -50, 75,
    -34, -28, 43, -5, 27, -15, -97, -32, 53, -68, 97, 44, 51, 47, 6, -19, -1,
    -43, 65, -89, -91, 21, -63, -59, 31, -9, 26, -79, -74, 43, 30, -38, 32,
    86, -55, -48, -48, 42, 17, 70, 18, -40, 11, 74, -20, -35, -37, -64, 38,
    -53, 28, -76, -39, 83, 50, 25, 59, -2, 44, -40, -81, -90, -17, 40, -45,
    87, 12, -60, -31, -81, -76, 96, 79, -69, 58, 73, 12, 17, -34, 67, -70,
    85, 25, -87, -86, 59, 13, -68, -5, -53, -60, 31, -5, -51, -7, -61, -76,
    91, -12, -42, -42, -16, 23, 50, -17, 66, -55, 72, 56, 84, 80, -19, 73,
    -79, -97, 57, 44, -96, -65, 83, -31, 40, 74, 98, -44, -42, -89, 64, -50,
    49, -28, -26, -17, -23, 57, -58, 89, 51, -57, 62, 97, 31, -12, -15, -18,
    -73, -91, -70, 75, -65, -11, 72, 29,
  36, -94, -100, 67, -24, -92, -40, 15, -79, -45, -72, 99, 98, 93, -39, 40,
    34, -55, 40, -38, 54, -46, -70, -86, -32, -72, 60, 25, 66, -97, 24, -41,
    8, 94, -9, -48, -34, 55, -46, 22, -92, -37, 80, 20, -43, 18, -78, -37,
    78, -43, -13, 74, -72, 71, 72, -6, -21, -30, 75, 64, -73, -42, 40, -50,
    -11, -90, 91, 92, -19, -16, 51, -20, -61, 87, 61, 24, -93, 28, -9, -32,
    -13, -10, 51, 14, 68, 88, 33, 42, -79, 4, -25, 35, -78, 25, -64, -45,
    -96, 14, 1, -73, 91, -19, -9, 42, 0, 56, -53, -78, -35, -96, 82, -92,
    -17, 62, -86, -80, -41, -98, -24, -49, 43, -15, -67, 92, -10, -50, 74,
    -16, 27, 12, -25, -3, 42, 0, -78, -82, -74, 12, -38, -74, -2, 23, -41,
    -20, 6, 85, -49, -89, 71, -3, -49, -95, 23, -53, 61, 29, 4, 33, -41, 51,
    -8, -22, -72, -77, -54, 81, -87, -81, 98, 40, -21, -55, 86, 23, -99, 80,
    59, -5, -58, -18, -77, -14, 5, -74, -94, 57, -67, -6, -86, -32, -39, -68,
    -8, 3, -68, 71, -49, 59, -100, 2, 68, -53, 55, -69, 88, 25, -14, -59, 6,
    -15, 18, -99, -10, -17, 44, -39, -63, 25, -65, 31, -75, -26, 73, 52, 60,
    72, 85, 25, -40, -84, 79, -7, -33, -50, 9, 60, 83, 80, -83, 90, 10, 42,
    -67, -40, -59, -54, -96, -76, 74, 16, -99, 34, -63, -51, 11, -51, -2, 19,
    32, 15, -47, 25, 86, 98, 20, 72, -6, -17, 67, 4, 35, -77, -16, -54, -98,
    -63, -16, 92, -52, -51, 90, -28, -62, -23, 50, -54, 16, -6, 40, 29, -10,
    86, -24, 14, 94, 99, -84, -43, -35, 13, -49, -98, 50, 27, -28, -13, 36,
    2, -7, 79, 59, -49, -17, 10, 98, -5, -87, 15, -24, 8, 25, 92, -50, 42,
    -88, -70, -21, 14, -76, 61, -61, -10, 22,
  16, -82, -59, -14, -3, 83, -70, -35, -54, 97, -86, 34, 47, 11, -50, -52,
    -71, 80, 57, 58, 29, 7, -95, 79, -63, 35, 42, 24, -59, -51, 54, -61, 21,
    -33, 20, -77, 25, -7, -30, 70, -81, 92, 70, 29, -12, -89, 53, 50, -46,
    57, 37, 71, 64, -9, -74, 69, -24, -11, 46, -90, -14, -21, 45, 40, 61,
    -32, 2, 96, -91, -79, 26, -23, -56, 70, 69, -91, -61, -18, -92, 69, 90,
    -55, 28, -57, 59, -54, -37, 46, 36, 91, -30, 78, 76, 0, -81, 81, -7, 87,
    -91, -1, -13, 81, -60, -50, -34, 76, 90, 55, 72, -21, 28, -14, 60, 20,
    97, -98, -3, 9, 9, -46, -46, -91, -100, -39, -82, 15, 46, -73, -66, 72,
    64, 73, 14, -30, 20, -1, 90, 18, -72, -51, 9, 21, -81, -91, 6, -28, 9, 6,
    21, -75, 42, 65, 79, -24, 4, 12, -12, 65, -100, -83, 55, 97, 88, 31, 97,
    -85, 57, 61, -57, -75, -26, 2, -62, 31, 3, 92, 54, 37, -60, -54, -13, 46,
    -91, 53, 21, 95, -34, 52, -38, -28, 97, 8, -51, -95, 79, 26, 26, 36, 1,
    -69, 83, -36, -12, -22, -94, -95, 97, 8, -52, -4, -18, 78, 50, -55, 16,
    42, -96, 87, 88, -98, -70, 64, 57, 90, -26, -80, 20, -49, -34, 56, 77,
    17, -7, 95, 29, -43, -99, 40, 56, -77, -4, 98, -90, 27, -27, -13, 76,
    -97, 22, 39, 45, -3, 77, -10, 12, -20, 10, -85, 48, 22, -1, -55, -21,
    -39, 15, -28, -49, 73, 72, 66, -99, 3, -63, 2, -32, -94, 1, 25, -62, -90,
    39, -85, 85, -37, -93, 74, 21, -86, -7, 26, -17, 44, 49, 8, 59, 19, -21,
    50, 31, 91, 55, -55, 13, 66, -88, 95, 52, -12, 13, -17, -68, -3, 98, 21,
    -70, -7, 28, -85, 37, -100, 46, 93, 93, -30, 80, 19, 54, 36, -2, -80, 3,
    65, 1,
  -7, -61, 52, 80, -85, -56, -77, -91, -22, -37, -87, -58, -57, -43, 42, 7,
    -74, 18, -5, 10, -59, -18, 52, -22, -53, -99, -90, 27, -84, -6, 93, 25,
    26, -46, 6, -54, -85, 50, 9, 46, -25, -95, -48, 40, -6, 49, 58, -22, 78,
    -25, 11, -41, -26, 68, -59, 29, 51, 1, -59, 9, 67, -58, -29, 8, 13, 76,
    78, -81, 82, 93, 86, -10, 62, -88, 19, 76, 77, -8, 0, -63, 76, -70, -12,
    38, -100, 81, -88, -72, 20, -95, 66, 92, -78, -37, 23, 68, -85, 4, -4,
    71, -95, -84, 49, 55, 71, 41, 50, -51, -51, -32, 57, 40, 10, -83, -73,
    -59, -11, 23, -58, 21, -8, -13, -68, 85, 66, -64, 74, -59, -72, -39, 74,
    -43, 88, -3, -49, -11, -42, -12, -43, -63, -78, 70, -34, 17, -9, -10, -5,
    98, 31, -91, -49, 66, -39, 5, 9, 22, -58, 33, 67, 67, -5, 0, 8, 60, -96,
    -76, -40, 85, 96, 84, 89, 82, -56, -85, 38, -17, -58, -98, -95, -62, -86,
    72, 4, -97, -25, -9, 48, -93, 35, 46, -25, 77, -29, -6, -24, -5, -93,
    -88, -100, -63, -36, -36, -10, -99, 70, 86, -91, -70, 20, 6, -20, 51,
    -51, 22, -92, -59, 67, 22, -77, 28, -55, 0, 60, -17, 35, -44, -9, 46, 89,
    23, 46, 76, 73, -42, 87, -43, 80, 4, 99, -80, -76, 63, -10, 31, 62, 9,
    -73, -79, 61, -98, -58, -29, 64, -49, -51, -60, -86, 33, -28, 35, -78,
    41, 87, -42, -85, 34, -45, -97, 47, -43, 10, -20, 0, 16, 15, 41, -65, 8,
    -74, -7, -53, 76, 34, 83, -45, -23, 87, -33, 40, -71, -31, 56, 57, -49,
    49, -6, 7, -77, -50, -93, -10, -80, 38, -40, -80, -40, -79, -59, 6, 71,
    24, -33, -50, 84, -13, 56, 56, 16, 31, -33, 96, -48, 53, 83, -49, -72,
    -57, -36, -82, 22, -78, -10, 58,
  -36, -7, 66, -38, -31, -53, -3, 57, 55, -95, 75, -63, -42, 52, -5, -57,
    -100, -85, 35, 48, 61, 29, -29, -80, -35, -30, -71, 46, 8, 40, -24, 68,
    90, 5, -68, -85, 34, -96, -39, -98, 54, -71, 79, 74, 88, -73, -50, -85,
    32, -48, 48, 6, 68, -87, -29, -80, 97, 95, 76, 83, -92, -23, 74, 64, -13,
    64, 30, -41, 98, 38, -2, -18, -14, 57, -32, -34, 62, -55, 64, 33, 82, 39,
    -94, -12, -58, 27, 10, -66, -87, 41, -68, 9, -90, -14, 37, -71, -74, -78,
    31, 92, -16, 82, 94, 55, 48, 92, -12, -26, -97, 35, -75, 0, 32, -52, -48,
    -61, 95, -19, 67, -61, 50, 61, -66, 89, -54, -6, -55, -2, 1, -46, -91,
    -7, 77, -14, 31, 89, -9, -52, 18, 99, 24, 65, -8, -12, -15, 98, 80, -5,
    -48, 58, 76, -7, -32, -2, -14, -97, -71, 3, -78, -81, -71, -38, -91, -89,
    -19, -14, -20, 28, -82, 66, 24, -44, -17, 35, 93, 57, -14, 42, -78, -2,
    80, 48, -62, 81, -72, -8, -46, -99, -9, -3, -29, 76, -87, 22, -35, -81,
    -33, 33, 45, 99, -23, -47, 20, -44, 36, -27, -1, 74, -69, 50, 55, -5,
    -91, 86, -29, 26, -42, -60, 51, -16, -5, 47, -10, -47, -56, -8, 29, 1,
    -71, 50, -70, 57, -64, -97, 21, -9, -75, -76, 22, 72, 25, 41, -25, -19,
    22, -13, -75, -80, -38, -13, 88, -23, -24, 45, -11, 25, -75, 40, 7, 60,
    18, -53, -6, 18, -99, -21, 67, 22, 20, 55, -37, 76, -98, 17, -13, 51, 86,
    83, 58, -27, 41, 6, -1, -86, -9, 34, -95, -4, 72, -79, 81, 90, -39, -92,
    -10, 63, 37, -22, -47, -80, -60, -2, -81, -75, -1, -21, -6, -28, -14,
    -99, 34, 13, 86, -81, -75, -61, -7, -38, -87, -51, -18, 42, 77, -80, 97,
    -69, 99, 24, -59, 2, -7, 46, -67,
  62, -29, 3, -27, 58, 58, 46, -67, -33, -10, -40, -16, 46, -27, 2, 71, -81,
    -53, -34, 5, -84, -68, 51, 4, 8, 81, -11, -32, -47, 96, 50, -80, -89,
    -60, -48, -88, -42, 17, -67, 15, 14, 92, -42, -2, -35, -78, 11, 92, 90,
    61, -95, -25, 51, 28, 33, 50, 73, 12, -51, 45, -51, 48, -5, -57, -4, 61,
    51, -71, -24, -36, -76, -65, -1, -55, -35, 91, -11, 41, -33, -16, 25,
    -48, -73, 23, 96, 61, -65, 80, -52, 43, 63, 14, -92, -94, -43, 15, 50,
    -28, 46, 77, -57, 31, 81, -5, 82, -46, 92, -25, 67, -37, -11, 31, 4, 79,
    -55, -99, -69, 93, -87, 79, 81, 8, -80, -63, -36, 25, -42, 62, -96, -88,
    -25, 21, 4, -78, 52, 53, 82, 4, 51, -53, -23, -95, 27, -79, -99, -96, 2,
    89, 93, 24, -5, -19, 80, -51, 29, 20, -76, -31, 6, 30, 12, -85, -91, 58,
    -89, -99, -6, 4, 0, -56, 16, -70, 37, 93, -80, 79, 57, 16, -5, -7, 20,
    14, -56, -10, -89, -74, 86, -75, -61, -76, -17, 33, 13, -66, -3, -3, 38,
    -4, 67, -55, -69, 60, -64, 54, -35, 61, 44, 25, 81, -67, 87, 64, -67, 79,
    -7, 19, 64, 82, 74, 28, -89, 52, 38, 47, -13, 80, -61, 14, 47, -70, -98,
    42, -19, -49, -84, 56, -74, -99, -11, 98, -53, -55, -52, -35, -79, 6, 39,
    -50, -35, 70, -30, -46, -28, 46, 18, 59, 8, -16, -20, -77, 82, -80, 60,
    43, 25, -75, 39, -19, 71, -72, 65, -21, -20, -77, -24, -66, -25, -22, 24,
    71, 1, -97, -20, -91, 34, -92, -33, -30, 32, 93, 65, 67, -36, 82, 35,
    -51, -77, -39, 42, 85, -2, 50, -1, -93, 32, -98, -99, 17, -34, 69, -45,
    -47, -4, -22, 80, 70, -13, 93, 45, 83, 49, 45, -29, -22, -80, 91, -51,
    60, 52, -80, 57, 24, -73,
  97, -15, 32, 76, -25, 67, -27, -83, -14, -40, 95, 15, -92, -48, 41, -19,
    87, -55, -2, 99, 92, -54, 53, 36, -73, 28, 25, 6, 85, -69, 58, -88, 84,
    38, 50, -96, 60, 66, -93, -66, -66, 99, -52, 56, 62, -63, -79, -96, -72,
    3, 58, 65, 76, -9, -8, 71, -82, 84, -37, 39, -31, -36, -83, 45, -62, -86,
    69, -39, -93, -1, -39, -29, 98, 3, 79, 15, 11, -18, 94, 49, -79, 47, -92,
    78, -67, -8, 99, 8, -17, -9, -32, 49, 49, 67, -3, 45, -61, 71, 51, 37,
    96, 56, 86, -53, -89, 49, 37, 61, -47, 52, -63, 46, 59, -81, 86, -10,
    -18, -35, 52, -31, 81, -82, 80, -45, 92, -4, 2, -79, -61, 21, 75, -91, 1,
    -4, -23, -42, 57, -33, -18, 30, -62, -20, 80, 26, -61, 57, -90, 22, -12,
    -78, 22, -19, 22, -77, 39, -1, -73, -90, -33, 0, 43, -81, 40, -9, 93, 3,
    61, -76, -75, -11, -15, 81, -61, -35, -84, -25, -90, -100, 39, -49, 13,
    37, 37, -5, 65, -62, -11, 55, 28, -95, 84, 57, -98, -77, -61, 43, 57, 80,
    -34, -72, -92, -41, -84, -77, 41, 8, -58, -68, 83, -77, -28, -74, 92,
    -39, -94, -17, -54, 25, -1, -41, 69, -30, 24, 60, -97, -28, -82, -83, 98,
    -83, -86, -99, -30, -18, -86, -35, -21, -49, -21, -98, -80, -94, 12, 59,
    -71, 55, -54, -97, 0, -35, 7, -66, 18, -36, -86, 3, 43, -87, -71, 39, 92,
    80, -5, 32, -85, -53, -81, -72, -96, 79, -42, 45, 95, -39, -82, -67, -66,
    -64, -23, 11, 92, 94, 87, 33, -42, -61, 79, -54, 15, -69, -47, -53, -40,
    78, -97, -48, -12, -49, -75, 1, 99, 28, -26, 20, -64, 44, -84, 7, 13, 29,
    40, 26, 5, -77, 0, 30, -47, 52, 33, -67, 57, -93, 64, -2, -91, 99, -62,
    45, -94, 81, -9, -11, -35,
  98, -9, -78, -20, 25, -20, 94, -5, 85, -65, 10, 46, 97, 12, -12, -90, 67,
    49, 39, -44, 73, 20, -58, -34, 30, -75, 2, -98, 24, -56, -38, -28, -38,
    -26, 75, -56, 37, 36, 7, 32, 36, 18, 92, -84, -67, 27, -99, -38, 61, -16,
    -77, 65, -80, -75, -23, 15, -13, -73, -70, 70, 2, -98, 24, 23, 34, 7, 29,
    27, 13, 92, 61, 73, -79, -7, -23, -79, -15, -18, 71, -24, 96, 86, -25,
    94, -51, 95, -84, -54, -12, 52, -10, 44, 32, -86, -8, -29, -100, -72,
    -100, -71, -20, -56, -9, 76, -98, 63, -7, -56, 55, -6, 51, -88, -100, 87,
    -89, 64, -96, 85, -18, 86, 0, -97, -1, 85, 48, -17, 60, -9, -62, 47, -22,
    42, -10, 27, -62, 98, -78, 32, 26, -14, -8, 53, 84, -19, -85, -46, 31,
    29, 84, 69, -7, -98, 61, 54, -16, 15, 22, 29, 71, 65, 96, -44, -90, 51,
    -14, 78, 61, -94, -22, -88, -81, -15, 80, 6, 9, -29, 60, 94, -13, 79, 7,
    -54, -24, -71, 33, 45, -73, 75, -91, -87, 7, 23, 54, 31, -62, 57, -9, 87,
    -7, -51, 98, 4, 32, 79, -25, 39, 63, -100, -93, -54, -90, 17, 19, 33,
    -53, -46, 50, 57, -53, -20, -43, -30, -15, -10, -64, 20, -35, 37, 7, 71,
    86, -85, 34, 70, 19, 85, 57, 34, -34, -69, 30, -88, 51, 30, 52, 88, -94,
    14, -25, -42, 86, 21, -74, 56, 98, -68, 14, -80, -88, 45, 79, 83, -30,
    -87, 27, 10, 96, -25, -19, -18, -67, 62, -98, -60, -73, 26, 41, 8, 50,
    -81, -49, -89, 26, 33, 70, 16, 21, -27, -36, -76, -99, 41, -72, 41, 13,
    -3, -53, 37, -71, 68, -37, -99, -79, 69, 54, -13, -36, -80, -23, 69, 49,
    -46, 44, 19, 94, 85, -94, 8, -95, 7, 87, 54, 96, 36, -4, 17, -7, -87, 88,
    4, -34, 33, 48,
  -41, 96, -88, -48, -23, 36, 12, 91, 15, -16, 22, -18, 17, -23, 30, -73,
    -42, -94, -20, -33, -33, 89, -45, -26, 29, 7, -64, 43, -36, -59, -38,
    -47, 11, 35, -34, 48, 30, 54, 52, 8, 36, -89, -43, -35, 24, -28, 74, 49,
    -70, -71, 8, 82, -26, -95, 68, 9, -60, 63, 20, 98, -55, -76, -9, 85, -72,
    72, -96, 11, -14, 54, 61, -7, 31, -5, 54, 88, -86, 15, -62, 95, 12, 56,
    -48, 25, -71, -14, -51, -4, -40, 2, 88, 90, 90, -47, -21, -1, -71, -9,
    34, 91, -22, 65, -19, -92, 37, 50, 42, -22, -52, 49, 61, -8, 22, 17, -56,
    31, 20, 29, 6, 69, -13, 80, -26, 78, 75, 20, 69, -19, 66, 41, -8, -21,
    34, -30, 46, 56, -96, -1, -71, -80, -67, 51, -25, -45, 81, 47, 31, -62,
    88, 27, -86, 56, 83, -97, -14, 65, 94, -82, 23, 20, -26, -83, -89, 68,
    66, 35, 7, -24, -71, 13, 37, 18, 6, -59, 61, -90, 30, -94, -88, 40, 91,
    -77, 89, -88, -69, -40, -8, -91, -69, 86, 41, 27, -25, 44, 25, 56, -97,
    -81, 0, -62, -58, 70, 10, -7, -49, 2, -69, 17, -91, -95, 54, -72, -80,
    21, 9, -20, -13, 3, -39, 28, 24, 91, -26, 20, -100, 73, -57, 50, 88, -10,
    -13, 99, 41, 81, 73, -37, 92, -2, -11, -75, 2, 99, 42, -6, 66, 29, 49,
    -25, 45, 65, 79, 17, -13, -16, -4, 24, 50, -33, -36, 24, 68, -14, -38, 6,
    -56, 53, 42, -99, -38, 25, -95, -71, 81, 96, -12, -69, 10, -85, -86, -19,
    -76, 75, -69, -86, -90, -98, 53, -79, 97, 19, -53, -44, -70, 35, -76, -2,
    76, 0, -71, 33, 66, 86, 77, 93, 41, 58, -9, -24, -28, -4, -87, 10, -42,
    15, 36, 43, -37, -10, -1, 38, -43, -87, 33, -23, 99, -56, 59, 26, 24, 67,
    49, 79, 81,
  92, -97, 27, 62, 52, -89, 51, 25, -60, 75, -3, 71, 30, -89, 55, -34, 29,
    68, -50, 80, 30, -46, -81, -46, 70, 32, 14, 47, -23, -51, -97, -77, -11,
    15, -15, -48, -87, -13, 83, 71, -76, -58, 15, 16, -81, 35, -6, 39, 60,
    -90, -5, -63, 67, -72, 92, -26, -49, -12, -97, -80, 58, -18, -62, -13,
    -41, -55, -48, 96, -80, -10, 11, 52, 81, 19, 32, 51, -83, 62, -21, -89,
    14, -73, 46, 46, 25, 41, -40, -44, -62, 87, 67, 51, -58, -18, 55, -33,
    -48, 89, 82, -52, -81, -36, 99, 55, 62, -45, -73, -47, 98, -79, 32, -2,
    62, -65, -95, 55, 25, -58, -18, 38, 89, 33, 39, 88, -10, 86, 4, -2, 55,
    90, -87, -22, -17, -15, 24, 99, -63, -88, -36, 96, 89, -83, -34, -75, 82,
    -80, -7, 75, -82, 73, -9, 12, 49, 32, 14, 13, 76, -10, -1, 68, -77, 96,
    -63, 25, -82, -35, 17, 5, -86, 18, 62, 83, 34, -85, -51, -75, -82, -24,
    -27, -64, -74, 18, -88, -100, 81, 48, -30, -5, 33, 40, 48, 95, 0, 13, 4,
    -16, -40, -31, 56, 57, 55, -91, -25, 80, 76, -34, -64, -67, -77, -60, 71,
    0, 59, -37, -9, -15, 35, 92, -45, 2, 74, 70, -42, -58, -87, 88, 57, 48,
    14, 29, -94, -92, 43, 25, 29, -97, 40, 63, -75, -81, 51, 0, 17, 15, 24,
    -95, -14, 68, 9, 64, -28, 27, -19, -23, -43, 99, -77, -92, 77, 65, 35,
    -55, 33, 31, 26, -92, -36, -97, -62, -49, 16, -94, -41, -42, -21, -62,
    94, 27, 43, 52, 2, 45, 90, -97, 85, -80, 18, 59, 92, 47, 26, -51, -93,
    55, 16, -29, 18, -56, -7, -95, -79, -72, 47, -11, 81, -96, -49, -42, -23,
    -9, -75, -13, 35, 30, 40, -95, 58, 11, 43, -17, 83, -92, -42, 83, 71,
    -45, -85, -41, 49, -10, 64, 13, -2,
  -57, -38, 70, -39, 23, 92, 57, 61, -70, -98, 82, 98, 12, 58, -31, 8, -26,
    7, 51, 71, 29, -8, -10, 63, 74, 31, -95, 25, 9, -64, -40, -62, 62, -55,
    22, -21, -40, 53, -54, -34, -36, -91, 58, -69, -96, 25, 91, 6, -58, -80,
    -66, 39, 25, -38, -76, 79, -16, 26, -34, -1, -96, 49, 54, -57, -13, -40,
    -70, 37, -20, -64, -95, 68, 9, -18, -25, 58, -44, -27, -43, -19, 76, -71,
    62, -77, 69, 60, 93, -61, -28, 82, 90, 16, -43, -67, 24, -16, 11, -89,
    11, 84, -38, 91, 20, 76, -15, -1, 56, -19, 21, 18, 60, 40, 92, -89, 55,
    -62, -88, 45, 37, -41, 56, 13, 65, 33, -47, -56, -51, -35, 66, -88, 92,
    35, -18, 17, 75, 61, 98, 66, -83, -53, 98, -45, -69, 66, 44, 87, 80, -50,
    96, -3, 69, 94, 92, -13, -30, -82, -70, -88, -23, -66, 81, 70, -91, -43,
    -51, -6, -66, -26, -81, -64, 79, 63, 38, -19, 48, 25, 72, -50, -52, 44,
    -64, 53, -79, -2, 68, 94, 67, -5, 52, 20, 58, -69, 79, -34, 44, -79, -36,
    -24, -55, -96, 78, 56, -78, -62, 56, -67, 90, 13, -20, 15, -63, 38, -74,
    -15, 64, 8, -20, -19, 5, -89, -10, -31, -65, 40, 2, 87, -52, 26, -19,
    -31, 81, -99, 51, 41, -23, -87, 59, -89, -30, 33, -92, 98, -49, 92, 47,
    90, 44, 95, -20, -37, 27, -34, -70, 0, -83, 75, 13, 90, 60, -15, -100,
    40, 47, 60, 73, 69, 84, 8, -43, 49, 49, 64, -18, -11, -43, 50, -88, -16,
    -71, -23, 69, -22, -55, 70, -42, 6, -77, -80, 47, 84, -88, -51, -42, 97,
    11, 55, 9, 35, -59, -31, 47, 7, -69, -88, 10, -94, 64, -66, -28, -49, 55,
    27, 57, 33, -39, -21, 17, 22, 73, 18, 42, 87, 52, -74, -14, -81, 65, 21,
    -15, -100, 62, 44, -69,
  22, -59, 8, -46, 24, 10, -71, -64, -63, -1, 94, -82, 14, 10, -17, 25, -60,
    45, 58, 31, 67, -54, -81, -73, -44, 43, 59, 9, 85, -89, -7, -88, 23, 65,
    -75, -97, -40, -76, 38, 89, 8, 46, -46, -35, -78, -94, 31, -18, 86, 20,
    -99, 31, 10, -15, -78, -63, 92, -15, -71, 35, -79, -30, 94, 12, 99, -89,
    -79, 40, 90, -89, -75, 31, 92, -16, 49, -70, -30, 9, 75, -7, 17, -46,
    -59, -96, 3, -55, 48, 16, 85, 72, 68, -21, -84, -14, 31, 43, -30, 39,
    -82, -36, 95, -17, -46, 29, 89, -99, 18, -52, 40, 61, -9, -36, -27, 0,
    15, 93, 75, 20, -35, 49, -25, -37, 11, 74, 11, 18, -46, 46, 17, 98, 91,
    70, -32, -41, -96, -56, 13, 5, -16, 24, 50, -76, -60, 75, 85, -78, -62,
    98, 39, 13, 7, 72, -54, 27, 30, -92, -98, 82, 57, 25, 86, -14, 12, -56,
    -10, -30, -53, -38, 25, -49, 35, 46, 97, 19, 45, -30, -80, -72, -82, -40,
    31, -48, 39, -31, -100, -47, -94, 74, -8, -49, -39, -7, -78, -44, -71,
    -10, 35, -92, 33, -84, -81, -62, -65, -79, -39, 67, 62, -4, 96, 94, -64,
    -54, -100, -49, 15, 1, -13, -96, -38, -83, -15, 76, 61, 54, 10, 98, -31,
    31, -64, -74, -72, 75, -100, 20, -41, -33, 62, 67, 0, -30, -22, 35, 65,
    -95, 81, 56, -68, -96, -77, 43, 88, -4, 46, 26, -23, -7, -21, -16, -22,
    -46, 91, 33, -88, 31, -37, 92, -14, 3, 80, 34, -86, -12, -92, 73, -18,
    -4, -89, 18, 97, 3, 21, 3, -63, -11, 0, 63, -73, 70, -18, 78, 33, 26, 63,
    -23, -37, 82, -52, -53, 60, -59, -17, -22, -14, 44, -86, 10, -96, -98,
    64, 66, 16, -58, -96, -93, 89, 29, 71, -52, -36, -74, 25, -15, -16, -93,
    -24, -97, -94, -100, 58, -38, 46, -60, -47,
  32, -18, 80, -98, 32, 43, 56, -32, 33, 10, -25, -28, -46, 34, 87, 76, -31,
    -51, -94, 65, 84, -94, -19, -64, 41, 27, 68, -83, 20, 10, 30, 12, -55,
    98, 52, -74, 56, -90, 23, -52, -14, 37, 65, 49, -89, -56, 2, -50, 51,
    -55, -41, 84, 9, -88, 2, -43, 48, 80, -47, 9, 42, -83, -89, 40, -92, 54,
    -16, 57, -10, -67, 42, -48, 12, -90, -46, 9, 62, 42, -41, -83, 92, 60,
    -46, -81, 78, -32, -86, 28, -55, -84, 81, -27, -91, -38, -31, 90, -33,
    59, -45, -28, 37, -57, 75, 30, 85, -15, 25, 74, 74, -68, 55, 42, -51, 74,
    95, -42, -25, 90, 9, 99, 40, -62, -50, -10, 97, -52, -70, -48, -55, -79,
    19, 39, -53, -64, 35, 8, 36, 34, 40, 59, -85, 68, -7, 74, 97, 17, -73,
    50, -37, -7, -75, -67, 81, 70, 29, 28, 95, 69, 16, 18, 5, 55, -37, -46,
    -21, 61, 25, 62, 77, 49, 72, 62, -2, -15, -50, -77, -17, 24, -60, 50,
    -13, 23, 83, 4, 74, -56, 96, -1, -72, -66, -73, 85, -58, -83, -24, -80,
    -17, -66, 32, -82, 15, -46, 40, 62, 48, -57, 80, -46, -2, 47, -42, -98,
    14, -17, -42, 22, -83, -6, -35, 95, -68, 19, 43, -21, -100, -29, 17, -12,
    48, 97, 7, 3, 29, 70, 7, -62, -33, -79, 80, 79, -41, -87, 48, 12, 7, -22,
    8, -29, -11, 87, -56, 75, -98, 79, -17, -10, -10, -71, 79, 50, -57, -13,
    68, 78, 61, -28, -31, -62, 7, 65, -96, -78, -64, 27, 21, 27, 19, 15, -73,
    80, 2, 22, -38, -90, -74, -57, -7, 40, -94, -65, -82, -6, -46, -8, 74,
    94, -27, -84, -23, -10, 91, -85, 82, -33, 51, 54, -61, 19, 31, -61, 47,
    -20, 44, 33, -89, -40, -53, 86, -2, -86, -53, 78, 22, -97, -85, -8, 94,
    -38, -41, 32, 34, -20, 78,
  -85, -22, 61, 79, -25, -15, -83, -46, -1, -53, 22, -62, -66, -66, 98, 26,
    83, 56, -92, -91, -39, -19, 21, 87, 58, -43, -97, -33, -49, -92, 72, 11,
    61, -5, 77, 62, 24, -1, -57, 57, -45, 85, -77, 45, -52, 56, 44, -82, -81,
    -73, 6, 94, -35, 56, -49, -64, 45, 69, 53, 40, 36, -31, 26, -84, 68, 10,
    27, 15, 56, -14, 54, -94, 57, -74, 98, -71, 62, 5, 50, 66, -48, -2, 50,
    79, 31, -2, -18, -80, 18, 19, 51, -61, -25, -39, -5, -29, -7, -92, -11,
    -76, -62, -70, 15, -93, 7, 7, 62, -58, 67, -10, -26, -52, -66, 64, 83,
    35, -16, 37, -16, -95, 4, -26, 14, -16, -81, 40, 81, 47, -76, -70, 34,
    79, -12, 50, -71, 65, 35, 27, -86, 91, -79, -8, 88, 41, 54, 5, 30, -42,
    89, 50, 65, -57, -49, 20, -26, 8, -61, -69, -27, -38, 16, -98, 79, 66,
    12, 94, -94, 67, 10, 53, -44, -37, -80, -90, -94, 16, 72, 38, -68, 92,
    95, -78, -52, 15, -85, 27, -58, -98, -51, -36, 18, -33, -76, 98, -94,
    -30, -63, 9, 9, -6, 83, -69, -66, -45, 10, 86, -39, -38, 70, -61, 62,
    -82, -23, 86, -16, 75, 50, -1, 52, -87, 62, -95, -56, 36, -42, -43, 77,
    44, -65, 95, 28, -63, 98, -88, -46, 89, 59, 90, -61, 13, 14, 27, 87, 46,
    -5, 15, -25, 84, 0, 1, -29, -64, 42, -74, 81, 13, -65, -55, 53, -90, 40,
    50, -48, 68, -63, 65, 25, 5, 84, -32, -68, 29, -65, 95, 69, 30, -23, -50,
    23, -2, 28, -6, -56, 10, -26, -79, 25, 3, -40, 42, -66, -25, -28, -41,
    47, -30, 53, 99, 61, -57, -23, -6, 32, 24, 76, 17, -5, -99, 45, 64, 56,
    84, 23, -47, -54, 3, -41, -53, 78, 34, -38, -19, -14, 64, -1, -20, 23,
    -37, 43, 41, 11, -99, 94,
  -12, 37, -100, 28, -69, 28, -80, -93, 65, -52, 77, 46, -82, 56, -76, -15,
    -60, -14, 92, -41, 48, -47, -52, 31, 35, -24, 21, -33, 57, -27, 66, 54,
    38, 37, 12, -96, 37, 90, -19, 73, -18, 61, -98, -45, 12, 63, -94, 83,
    -73, 59, -32, -53, -15, -76, -91, 21, -81, -45, 33, 32, -74, -83, 59, 90,
    54, 15, 39, 93, 40, 10, -33, -13, 63, 94, -24, -29, 75, 58, -64, 58, 68,
    -40, 22, 89, 50, 75, -2, 66, -96, -96, -78, 9, 65, 4, 84, -4, -10, -32,
    -6, -38, -20, 91, 26, 32, 80, 74, -90, 48, 59, -74, 63, -99, 64, -73,
    -17, -88, 33, 56, -91, 71, -96, -97, -20, -31, 0, -78, 86, 36, -90, -42,
    -2, -9, -89, 25, -89, -68, -10, 60, 84, -48, 12, 56, -14, -21, -69, 48,
    4, -42, -12, -66, 69, -53, 76, 18, 67, -4, -7, -55, -77, 6, 98, 4, 9, 18,
    -27, -21, -97, 56, 64, -73, -79, -72, 49, 9, -15, 88, 44, -44, 29, -37,
    64, 61, 36, 46, -87, 10, 18, 72, -25, -68, 61, 66, -86, 41, -20, -26,
    -13, -24, 25, -54, 49, -98, -97, -87, -49, 97, 96, -58, 36, -76, 16, -44,
    -98, 29, -61, 5, 6, 19, -76, -25, -30, 19, -5, 43, 98, -8, 46, 98, -53,
    -52, 96, -65, 15, 37, -84, 12, 1, -95, 55, -82, -100, -86, 27, 79, 30,
    -20, -32, -37, -70, 26, 74, -79, 70, -63, 62, -65, 52, 18, -41, -73, -8,
    -96, -7, -52, -96, -46, 98, 38, -28, -26, -81, 54, -15, -69, 34, -32, 23,
    -3, -100, -48, 32, 97, -86, 4, -3, 68, 43, -51, -44, -70, 5, 40, 59, 62,
    -80, 21, 49, -34, -91, -75, 55, -7, 62, 43, 13, 43, 23, -31, 45, 43, 14,
    54, 70, -35, -85, 71, -42, 73, 21, 85, -66, 33, 81, 90, -24, -71, 28, -1,
    -7, 2, -43, 37, -29,
  31, 81, 97, 7, -97, 23, -41, -50, -77, 22, 86, 15, 93, 96, -87, -33, -6, 0,
    18, 9, 51, -77, -48, 19, -53, 8, -32, 2, -77, -69, 8, -9, 1, 97, 21, 88,
    30, 47, 24, -28, -70, 63, -93, 43, -88, 55, 69, 95, -18, -17, 88, 75, -1,
    81, -4, -13, -55, 24, 27, 13, -79, -69, -92, -66, -18, 87, 80, -33, 97,
    -68, 45, 75, -74, -32, -62, -3, -18, 72, 55, -91, -81, 28, 82, -24, 60,
    34, -57, 70, -4, 70, 8, -25, -1, 75, 96, 96, 36, 74, 79, 40, 35, -68,
    -81, -49, 81, 57, -83, 19, -100, 77, 2, 47, 84, -29, -91, -90, 47, -10,
    71, -8, -65, -84, -38, 24, 2, 52, -92, 37, 16, 37, 13, -41, -26, -77, 58,
    -32, 75, 89, -12, -17, 41, 92, 1, 36, -25, 53, -7, 69, -57, 19, -89, -77,
    -9, -2, -83, 71, -49, -47, -60, -77, 30, 75, 57, -51, -24, 11, -54, -32,
    16, -39, -40, -28, 35, 6, -10, -38, -19, -55, -43, -50, 56, 43, 78, 38,
    10, -15, -38, 26, 51, -32, -86, 83, -18, 87, 83, 62, -15, 60, -70, -98,
    -44, -81, -53, -30, -8, -55, 81, -43, 32, 15, -86, 26, 54, 53, -66, -31,
    26, -91, -23, 9, 2, 38, 69, 28, -40, 83, 68, -91, -35, -7, -22, -38, 44,
    -82, -13, 41, 61, 47, 58, -26, -81, -78, -41, -98, -13, 35, 12, -94, -14,
    -45, 33, -31, -67, -21, -94, 73, 12, -59, -47, -5, -94, -75, 64, 56, -8,
    28, 21, 93, 57, 92, 37, 77, 50, 95, 58, -25, -61, -39, -45, 88, -42, -37,
    78, 31, 51, -64, -59, 95, -70, -91, -12, 46, -31, -12, 12, -43, -14, 46,
    12, 2, 63, 7, 38, 21, -12, -11, -65, -96, 26, 37, -13, -66, -14, 69, -3,
    -68, 75, 77, -77, 99, 49, -21, -26, -46, 92, -77, 45, -95, 29, -6, 51,
    -17, -52,
  11, 94, -31, -64, -39, -33, 45, -48, -22, -6, -94, -27, 96, 48, -7, 55, 11,
    -37, -82, -66, -24, -17, 74, -53, 53, -7, -30, 15, 16, -16, -25, 96, 4,
    -3, 57, -9, -6, -3, 32, 88, -62, -7, 29, 57, 9, -90, -42, 68, 87, -73,
    56, 14, -73, -73, -45, 45, 61, 89, -70, 55, 80, 1, -7, -78, -28, 7, 83,
    -63, 34, 12, -10, 51, -79, -75, -31, -92, 81, 18, 28, -86, 61, -19, 62,
    -74, 43, 95, 99, 60, 68, 85, 48, -93, 48, -79, 13, -27, 57, 24, -77, -83,
    22, -84, 35, -11, -59, 63, 7, -27, -31, 69, 19, -4, 86, 52, -62, 3, 29,
    -75, -42, -85, -95, -48, -9, 86, 40, -59, -52, 19, -67, -18, -77, 0, 99,
    62, 16, 54, 38, -64, -93, -72, 1, 87, 99, 5, -99, 24, 77, 29, -64, -53,
    83, 72, -29, -78, -4, 60, -42, 99, -25, 50, -27, -43, 83, -97, -94, 27,
    52, -69, 43, 67, -73, -78, -98, 30, 35, 85, 27, 89, -29, -95, -47, -32,
    -26, -59, -15, -60, -58, 90, 93, -39, 83, -64, 12, 28, 99, -62, 97, -45,
    -80, 89, 55, 59, 99, -74, -64, -4, -4, -1, -39, 80, 23, 18, -25, 66, -10,
    -59, 56, -5, 48, 28, -11, 47, 76, 51, 71, 84, -51, -52, 44, 16, 96, -80,
    87, 66, -84, -64, -41, -43, -87, -16, 12, 53, 91, 89, -98, 92, 70, 19,
    -7, 38, -93, -68, 39, 42, -44, 92, 87, 91, -3, 85, -95, -88, -55, -19, 0,
    26, 8, 86, 73, 31, 97, -31, 91, 80, -85, -63, -71, -78, -26, 69, -20,
    -92, 39, -8, 50, 60, -38, -46, 83, -62, -76, 31, 16, 38, 69, -27, -16,
    68, 58, -80, -77, -46, 91, -13, 10, 94, 87, -89, 4, -8, 84, -60, -97,
    -48, -5, -59, -65, 79, -39, 66, 95, -13, 92, -1, -60, -36, 20, 33, 21,
    -89, -54, -90, 25,
  78, -72, 46, -58, -70, 39, -89, 16, 1, -50, -77, -66, 27, -73, 30, 85, 87,
    52, -27, -2, 85, -21, -23, -3, 91, 4, -42, 63, -3, -14, 48, -10, -43, 81,
    77, 95, 43, 32, -37, 75, -31, 40, -44, 31, 81, 70, -28, 53, 98, 66, -10,
    24, -15, 6, -64, -51, -3, -31, 67, -13, -51, 95, 23, 7, 83, -54, 37, -61,
    -48, -52, -11, 24, 65, 83, 86, 2, 82, -50, 62, 11, -43, -55, 33, 34, -6,
    0, 88, 17, 68, 89, 39, -95, -74, -66, 13, 8, -23, 92, 53, 72, 10, -49,
    59, -80, -48, -11, 36, 44, -41, -81, 11, 42, -30, -76, -86, -14, 75, -9,
    46, -90, 13, 28, 23, 40, 42, -17, 42, -61, 5, -59, -68, -31, -69, -99,
    -78, 42, 67, 13, 51, -15, 26, 41, 21, 12, 61, -39, 60, 52, 90, -66, 28,
    -97, 38, -73, 74, -5, -33, 18, 93, -79, -42, 44, -8, -59, -55, 59, -67,
    -41, -91, -86, -71, -98, -86, -32, -62, 72, -47, 84, 11, -98, 46, -57,
    32, -21, -2, 84, -2, -95, -87, -81, -37, -37, 52, -95, 24, -41, 29, -97,
    78, 57, 2, 87, -87, 35, -45, 44, 68, -20, -30, -66, 70, -59, -22, -84,
    -32, 30, 7, 89, -93, -8, 87, 83, -3, 54, -63, 29, -84, -31, -46, -63,
    -19, -48, 26, 58, -16, 58, 91, -99, -58, 91, -12, -80, 58, 14, -87, 93,
    -56, -28, -2, -58, 51, -44, -16, -79, -40, 9, 43, 38, -41, -60, -48, 12,
    -92, 0, 90, -49, -87, -52, -77, 10, -24, 41, -91, -46, -77, -47, -21, 59,
    -49, 59, -14, -8, 27, 79, -2, -96, 57, -86, 11, -84, -45, -56, 27, -16,
    -21, -42, 47, 25, 44, 87, 22, 53, -95, 78, 78, -72, 16, 55, 84, -22, 92,
    58, 18, 71, 76, -40, 10, -77, 28, -31, -48, -37, 71, 92, 76, -97, 97,
    -100, 76, -54, -89, -13, 77,
  64, 76, -57, -18, -5, -57, 94, 55, -38, 30, 96, 21, 91, -73, 16, -2, -33,
    -63, -22, -84, -28, 15, -16, 44, -40, -8, 47, 40, -16, -75, 29, 55, 31,
    -66, 96, 61, 8, 29, -34, -59, -15, 69, 87, -13, 35, 72, 84, -20, -69,
    -31, 43, 87, -52, 6, -75, -22, 26, -47, -45, 78, -53, -4, 67, 89, -81,
    -40, 38, -49, 13, -6, 70, 11, -23, -78, -4, -60, 74, 30, 17, -41, -95,
    -80, 10, -72, 53, 76, -66, -64, -86, 97, -92, -73, 3, -69, -23, 68, -72,
    -46, -54, 96, 86, 60, -30, 87, 16, -20, -82, 89, -30, -51, 55, -58, 46,
    -1, 57, -100, -11, -91, -29, -57, -23, -55, -66, 97, 51, -50, -95, -66,
    -9, -23, -93, 13, 77, -12, -57, 23, -100, -25, -60, -98, -49, 53, -14,
    47, 95, 44, 59, -49, -28, -16, 43, 46, -70, -34, -99, -86, -18, -51, 4,
    46, 64, 7, -100, -66, -17, -78, 76, -98, -76, -31, 4, -54, -79, 97, 56,
    -28, 87, -98, -88, 16, 15, 30, 8, -51, -91, 4, -7, -43, 63, -83, 44, -97,
    21, 72, -72, -99, -20, 67, -15, -71, -98, -24, -35, -1, 58, 20, -48, -46,
    -49, 8, 68, 76, -100, 95, -7, 84, -34, -7, -82, 43, 18, 82, -67, -91, 76,
    -64, 61, 89, -62, 16, -71, 24, 54, 46, -44, 96, -83, -34, 5, 65, -65, 74,
    -17, -31, 24, 5, -95, -65, -74, -48, 32, -90, 45, 81, -18, -47, -91, 74,
    68, -78, 92, -12, -87, -82, -94, -70, 58, -98, -22, -35, 71, 48, -43,
    -55, 91, -72, -80, 41, -35, 65, 22, -29, -51, -98, -45, -21, -7, -67, -2,
    5, 27, 5, -86, 14, -59, -17, 61, -16, -71, 82, -96, 52, -43, -3, -87, 86,
    31, 3, -40, 53, -66, -88, 68, 18, 37, 38, -36, 66, 57, 94, 21, -83, 52,
    -35, 1, 75, 73, 49, 74, 82, 52, 90, -59,
  22, -23, 89, -11, -35, 59, -17, 71, -88, -27, 74, 38, 99, -74, 63, -39, 93,
    87, -32, -81, 48, -68, 88, -86, -25, 41, 47, 73, -50, 98, -31, 60, 53,
    84, -72, -46, -42, 74, -45, 60, -15, 63, -40, -42, 53, 54, -12, -22, 38,
    -26, 90, 27, -55, -9, 43, -45, 51, 25, -32, -2, 41, -56, -16, -32, 36,
    -32, -99, 29, -12, -57, 64, 34, -35, -66, 80, 49, 43, 62, -25, -50, 21,
    42, 71, -39, -84, 80, -13, -74, -97, -6, -14, -11, -84, 92, 37, -79, 7,
    -21, 47, -61, -92, -68, -46, 60, -16, 3, 25, 24, -55, -62, -18, 72, -17,
    -69, -3, -60, -82, -9, -24, -15, -44, -88, 19, -55, -23, -98, 18, -81,
    -35, 68, -97, 71, -34, 23, -48, -48, 27, -23, 70, -72, 89, -17, 94, -79,
    -27, -22, 51, 1, 54, -66, -87, 3, 20, 73, 97, 17, 19, 64, -47, -13, -55,
    -96, 46, -67, 49, -7, -77, -14, -83, 37, 70, 41, 32, -77, 41, -34, 38,
    34, -9, -22, -93, -67, 49, -40, -94, 44, -37, -33, 18, -89, -91, 50, 54,
    -30, -85, -59, -44, 13, 21, 44, 43, 30, 9, -74, -18, 40, 11, -41, -71,
    52, 56, 0, -71, -93, -50, -54, -49, 53, 58, -67, -83, 49, -21, 98, 14,
    -34, -13, -58, -39, -71, 22, -77, 88, 91, -91, 97, -13, 36, 34, -9, -36,
    81, 18, 24, 19, 36, 1, -24, -44, -70, 68, -87, -53, 8, -41, -1, 6, -67,
    89, -40, -27, 23, -68, -65, -39, -26, 5, 56, 59, 61, 12, 75, -73, -44,
    94, -76, 13, 63, -91, 94, 34, -16, -88, -3, -63, -23, -94, 28, 33, -27,
    -57, 8, 61, -16, 45, -53, -100, -76, 38, -58, 98, -23, 0, -10, -61, -36,
    63, -29, -51, 31, 78, -77, 6, 94, 30, 50, 90, 14, -38, 42, 66, 96, -47,
    11, 31, 1, 84, -17, -90, 18, 76, 87, 42,
  72, -82, 40, 88, 46, 53, 83, 97, -92, 39, -47, -34, 43, -86, -10, 89, -1,
    98, -12, -19, -31, -36, 91, 3, -6, -58, -18, -96, 5, -34, 7, -87, -27,
    -5, 70, 0, 71, -63, -68, -4, -51, 56, 78, 62, 79, 31, 96, 89, 99, -94,
    88, 21, 59, 1, -95, 57, 48, 21, 56, -43, -42, -92, 30, -68, -53, -4, -87,
    -15, -38, -99, -22, -49, 0, -50, -40, 2, 92, -16, -81, 83, 46, -66, -35,
    -11, -69, 96, -75, 24, 44, 37, -79, 54, 30, -24, 12, -30, -65, 70, -50,
    54, -1, -60, 73, -64, 3, 68, -50, -5, -81, 52, -18, -91, -91, 68, 47,
    -93, 16, -22, -81, 16, 12, 81, -85, 40, -1, 99, 58, -90, 81, 63, 45, 6,
    -2, -99, -30, 55, -76, 35, -30, -97, 98, 13, 44, 32, 53, -3, -85, -20,
    -95, 92, -19, 26, 81, -25, 54, -86, 62, 14, 91, -44, -4, -69, -76, 71,
    92, 25, 55, -35, 70, -100, -13, -47, -7, -16, 46, -28, -64, -37, -61, 21,
    -21, 54, -86, 14, 13, 0, -85, 65, 53, 55, 35, 54, -72, -91, 37, 10, 94,
    19, -50, -19, -7, 64, -96, -88, -56, 44, -87, 1, 28, 37, 58, 99, 33, 9,
    70, -87, -81, 89, 46, -47, 22, 44, 13, 80, -44, -44, -70, 83, 97, 20, 81,
    -79, -59, 62, -38, -69, 13, -21, 4, 64, -95, -6, 63, -94, 27, 68, 49,
    -11, -92, -76, -43, 59, -37, -62, -7, 48, -20, -51, 42, 4, 14, 95, -12,
    3, 27, 87, -76, 82, 60, -63, 92, -29, 88, 60, 60, 70, 30, 94, -12, 88,
    -16, -30, -82, -48, 75, 74, -48, -76, -80, -33, -1, -12, -15, -45, -59,
    66, -38, -100, 73, -39, 89, -45, 41, 71, 39, -17, -26, 90, 42, 12, 82,
    -97, -68, -6, -59, -30, 65, -19, 68, 28, -33, 91, 45, 21, -32, -96, 80,
    -25, 22, -94, 57, 64, 64,
  30, 52, -7, -47, 20, 16, 28, -17, -83, 32, -15, 7, -52, -86, -99, 11, 52,
    -52, -16, 72, -4, 7, -25, -80, -100, 87, -56, 84, 97, 92, -33, -4, 26,
    -35, -63, 19, 46, 79, 67, 36, 60, 26, -19, -50, 81, -96, -65, -56, -42,
    -84, -48, 90, -83, -58, 69, -32, 59, 96, 40, 95, -84, -78, -100, 61, -54,
    69, -67, -13, 60, 26, -49, -97, -27, 64, 33, 50, -16, 30, -24, -50, 94,
    -75, 90, -25, 55, 84, 86, 59, -69, -95, -63, 95, -81, -61, 5, 7, -93,
    -40, 74, -10, 36, -32, 82, -6, -67, 78, 6, -59, 36, -65, 49, 60, -61, 23,
    -64, -78, 47, -67, -45, -8, -80, 68, 27, -15, 90, -21, 15, -78, 56, -12,
    3, -41, 67, -38, -93, -30, 10, 39, -68, 51, -94, -19, -36, -72, 22, -93,
    -31, -62, 41, -12, -94, -32, -37, 15, 49, 51, 35, 20, -39, 8, -51, -34,
    91, 52, -21, -5, 96, -70, -39, 22, -19, -30, 64, 91, 77, 88, -88, -49,
    84, 72, 89, -91, -68, 25, -48, -19, 45, 53, 15, -90, 69, 21, 11, -3, -26,
    -71, 77, 44, 5, -72, 24, 23, 90, 65, 93, -62, -40, -85, -5, 30, 49, 36,
    61, 47, -99, -100, 40, -57, -83, -93, -98, 40, -57, 58, 79, 38, -4, 55,
    89, -22, 66, 63, 78, -92, -60, 2, -57, 23, -96, -35, -93, -43, -90, -80,
    -28, 75, -35, -20, 21, 95, 43, 84, -62, -58, 98, 29, -97, -26, 78, 41,
    -18, -100, -71, 42, -50, -55, -87, -40, 83, -58, -54, 6, -83, 57, -39,
    -72, -78, 12, -24, 15, -95, 55, -55, 18, 31, -57, 74, 96, -36, -81, -98,
    78, 48, -18, 72, -72, -97, -12, 34, -56, -98, 24, 98, -95, -19, -89, -38,
    81, -11, 44, -22, 34, 80, 36, -88, 50, 86, 95, 0, -44, 14, 83, 73, 29,
    -56, 58, 88, 9, 23, 39, -18, -91, -10,
  -28, -24, 22, -57, 37, 75, 36, -4, -17, -18, 99, -11, -99, 91, 25, -99, 93,
    28, -18, 61, -62, -9, -49, 87, -62, 60, -85, 2, -91, 69, -22, -40, 17,
    -6, -54, -33, -38, 58, 81, -16, 23, -74, 2, 63, 35, 23, -54, 41, 49, -33,
    -6, 94, 44, 92, -57, -83, -34, 85, -73, 41, -24, 90, 5, 18, 61, -44, 52,
    87, 32, -72, -67, 31, 72, 1, -66, 5, 52, -48, 14, -91, -43, -14, 16, 58,
    -47, -97, -62, 63, -83, -54, 41, -54, 59, 60, -51, 23, 43, 83, -96, -40,
    28, -28, -31, 41, 50, 20, -20, -60, -51, -2, -75, -16, -6, -12, 6, -77,
    -28, -12, -98, 49, 86, -77, 88, -98, -96, 5, 30, -89, -79, -49, -19, 1,
    -43, -22, -6, -96, -74, -4, -61, -44, -18, 91, 38, -34, -44, -2, 6, -38,
    -7, 82, -62, 96, 23, -80, 49, -34, -18, -16, -90, 53, 20, 41, 47, -47,
    -93, 30, 1, -37, 34, 65, 41, 20, -6, -83, -2, 25, 45, 97, 61, 42, 52, 75,
    70, -70, -36, 69, -16, -10, -5, -1, -36, 62, -45, 87, -42, -44, -95, -60,
    -89, 60, 97, -93, -100, -43, 19, 47, 43, 19, 70, 52, -10, -9, 58, 19,
    -10, -1, -33, 95, 29, 41, -93, 41, 50, -48, -12, -5, -47, 77, -75, -18,
    -44, 58, -68, -77, 72, 66, -60, -52, -62, -6, 10, 51, 73, 55, 94, -90,
    40, -54, 58, 2, 22, 84, 58, 44, -57, -58, 65, -62, -21, 47, -95, 28, 64,
    -25, -18, -63, 84, -31, 96, -39, -5, 1, 0, 12, -37, 71, 96, -17, -59,
    -19, 53, 43, 34, 58, -10, -78, 53, 27, 49, 44, 42, 98, 59, -75, -11, 62,
    -22, -51, -39, 39, 9, -95, -49, 17, -25, 73, 28, -67, -43, -95, -24, -16,
    5, -53, -5, -96, -99, -54, -69, 60, 55, 56, -43, -28, 95, -47, -10, -68,
    -50, -85, -51, -88, -18,
  -78, 92, 68, 75, 68, -72, 47, 38, -31, -35, 7, -34, -55, -95, -84, -99,
    -99, 72, 45, 66, 87, -98, -53, -33, 11, 60, -94, 3, 50, 57, -42, -1, -42,
    -54, 19, -49, 28, -24, -27, -91, -32, -58, -84, 10, 80, 81, 5, 56, 0, 27,
    34, -32, 23, -90, 29, -40, 75, -79, 71, -16, 20, 35, 5, -43, -34, -36,
    -45, 97, 90, 58, -13, 46, -24, -40, -27, -62, -73, -25, -61, -27, 89, 58,
    -23, 38, 59, -36, -65, -47, -76, -39, -66, 25, 82, 82, 79, 29, -13, -27,
    57, -7, -30, -21, 26, 85, -22, -84, -29, -87, 44, 72, -23, 16, -86, -89,
    97, -57, -69, -46, 46, -25, 71, 30, -6, 72, 68, -47, -53, -83, -89, -22,
    27, 97, 66, 11, -79, 18, 60, 16, 69, -64, 15, 86, -61, -21, 17, 2, -97,
    33, -79, -94, 68, -26, 40, 98, 92, -2, -85, 91, -44, -64, 13, 80, -18,
    -21, -87, -33, -45, -42, 12, -61, 6, -69, 37, -43, -56, -74, -59, -2,
    -26, -45, -77, 61, 74, 62, 27, -6, -43, -91, -59, 49, -6, 11, 29, -85,
    -23, 13, 84, -68, -88, -92, -38, -75, 60, -23, 7, -24, 30, 6, 62, 85,
    -77, 44, -81, 35, -1, 33, -2, -78, 21, -74, 88, -90, 70, -100, 75, -69,
    -6, 95, -19, -57, -85, 48, -30, -90, -17, -11, 26, 79, -31, -91, 49, -24,
    24, 82, -98, 64, 28, 36, -24, -94, -51, -38, 58, 24, -29, 18, -21, -25,
    27, 4, -87, 9, -59, 31, -31, -86, -74, -14, -53, 74, 74, -97, 56, -37,
    -33, 92, 23, 4, 92, -59, 27, -52, 88, 19, 52, 58, 88, -84, 66, -74, -64,
    82, 45, -10, -44, -18, -79, 92, 11, -71, -18, 51, -82, -94, -66, 14, -30,
    -5, -59, -44, 3, 32, 87, -44, -98, 40, 82, 26, 55, 98, 23, 1, 97, -57,
    -63, -24, 28, 49, 7, -96, 26, 22, 64,
  -95, 35, -39, 82, -50, 14, 15, 3, 99, 83, 3, 57, 60, 98, -96, -60, -63, 87,
    24, -80, -9, -90, 83, -53, -56, -6, -57, 77, -96, 68, -80, 94, -79, -17,
    -37, -5, -20, -4, -38, -44, 15, -39, 87, 47, 1, -40, -17, -63, -21, 98,
    -15, 49, 97, 29, 85, -62, -1, -38, -39, 1, 44, -6, -63, 2, 72, -1, 14,
    91, -72, 31, -55, -28, -24, -82, -35, -58, -3, 99, 41, 57, -1, -60, 6,
    81, -2, 19, -90, 95, 89, 40, -38, 61, -38, -1, -27, -96, 49, -34, -72,
    46, -82, -73, -80, -9, 71, 84, 93, -33, 46, -84, -46, -67, 25, 86, 6, 99,
    -1, 8, -86, 99, 24, 41, -67, 69, 8, -56, 37, -23, -46, 15, 56, -100, 27,
    -76, -56, 77, -4, -82, -15, -59, 62, -39, 19, -34, -77, 28, -3, -99, 56,
    -53, -32, 84, -25, 75, -82, -55, 3, -64, 70, 87, -76, 46, 92, 29, 75, 64,
    -96, 80, 56, -88, -95, -6, -85, 99, -31, 20, -10, 61, -5, 83, 16, -7,
    -17, -28, -35, -33, 52, -41, -65, -1, -36, 55, -97, -44, 94, -46, -31,
    41, -85, 47, -3, -48, 37, 45, 47, 7, 67, 35, -14, -30, -36, 13, 94, 77,
    -31, -58, -37, 37, 13, 66, -17, -48, 36, -70, 80, -19, -32, 68, 54, 44,
    -63, -45, 0, 59, -5, -78, 1, -84, 71, 97, 78, 9, -86, -43, 32, -48, -15,
    83, 77, -54, 20, -70, -93, 83, 56, 75, 13, -86, -45, -87, 20, 2, 29, -50,
    28, -92, 20, -50, -46, 92, 41, 56, -19, 23, 91, 5, 11, 34, 89, -66, 6,
    23, 82, 63, -67, -40, -16, -18, -37, 18, 27, -45, 31, -30, 45, 21, 97,
    75, 11, 63, 5, -80, -62, 96, 4, -69, -69, -66, 44, -55, -5, -17, 70, 11,
    70, -19, 81, 6, -9, -75, -77, 37, 57, -31, 33, 25, -69, -20, -88, -40,
    37, 16, 27,
  52, 43, -67, 68, 85, -40, -32, 13, 39, -32, -54, 63, -30, -11, -65, 40,
    -93, -22, 24, -56, 57, -31, -19, -13, 78, -24, -49, 63, 93, -52, -46,
    -19, 62, -60, -79, -54, -46, 45, -59, -73, 82, -46, 9, -58, 25, 51, -74,
    62, 6, 86, -7, 40, 26, -88, -70, 3, 61, -3, 97, 27, -26, -21, -8, -69,
    -30, -45, -42, -46, 8, -88, -34, 4, -43, 9, -38, -23, 34, 50, 10, -46,
    23, 45, -44, -86, 8, -16, -74, -16, 81, -13, 16, -93, 82, -52, 28, -61,
    16, 88, -24, 26, 29, 37, -73, -65, 84, -19, -67, 6, 62, -46, 91, -48, 55,
    -27, 4, -2, 82, -19, 4, 37, 98, 62, -30, -61, -79, 75, -34, 29, 34, -74,
    61, -15, 82, -70, 1, -26, -27, 17, -39, -75, 98, 29, -14, 90, 59, -4, 28,
    77, -2, 88, -21, -70, 6, -67, -17, -92, -77, -62, 29, -14, 17, 24, 67,
    -72, 12, -6, -40, -75, 44, 70, -37, -18, -31, 48, -76, 66, 23, -99, -84,
    -81, 9, -25, -56, 42, 31, 24, 42, 8, 59, -22, -83, 53, 73, 77, -62, 55,
    -58, 23, 1, -33, 90, 69, 45, 26, -99, -47, -48, -11, -33, 52, 16, -89,
    23, -66, -37, -29, 44, 98, -22, -73, -75, 50, -82, -22, 14, 60, -100, 40,
    48, 24, 16, -91, -84, -11, -87, 54, 35, 12, -42, -67, 74, -3, -43, -90,
    30, 93, -14, -32, 37, 39, 39, -100, 44, -96, 48, 98, -19, 79, 83, 28,
    -78, 24, -48, -37, -20, 30, -38, 74, 44, 7, 8, -77, -63, 25, -49, -15,
    83, 32, 63, 68, 83, -17, 33, 28, 34, -82, 62, 0, 60, -3, 39, -81, -10,
    64, -37, 56, -86, -39, -31, -35, 40, -32, 69, -68, -19, -14, 50, -62,
    -60, 74, -72, 17, 73, 22, -26, 70, -97, -76, -65, 73, -94, 54, 29, 42,
    -93, 76, -85, -46, -67, 72, 18, -30, -28,
  72, -33, 59, -86, 89, -100, 38, -84, -62, -48, 86, -73, 15, 17, 22, -32,
    35, 53, -93, -87, -58, -37, 23, -9, 12, -84, -30, 75, -77, -39, -99, -1,
    -97, -42, 98, 92, 85, 18, -23, -4, 36, 76, -80, -16, -28, -85, 81, 70,
    -99, 89, 4, -23, 14, -48, -30, 81, -82, 92, 69, -53, -99, -3, -3, 27,
    -87, 29, 79, -26, -79, -12, 93, 98, 13, 28, -36, 62, 47, 1, -28, 11, -99,
    53, -2, -24, 38, -14, 1, -46, -84, -31, -47, -69, -68, -19, 28, -88, -46,
    68, -24, -13, 84, 54, 46, 82, -59, 49, -15, -36, -11, 95, 49, -86, -53,
    17, 88, -92, 26, 0, -45, -57, 60, -87, -12, 9, 17, -74, 11, -3, -97, -13,
    8, -63, -2, 67, -39, 7, -28, 48, 58, 21, -70, 14, -41, 12, 86, 92, -41,
    -26, -56, -7, 98, 9, 1, 6, 42, 34, -39, -72, -93, 12, -34, 64, 44, 83,
    -51, 63, 48, -48, 4, -18, 85, -70, 5, 93, -2, 8, -25, -97, -38, -8, -85,
    39, 74, -67, 49, -22, 6, 70, -56, -75, 22, 48, 80, -19, 61, -77, 4, -65,
    64, -64, 2, -51, 30, -69, -25, 29, 55, 74, -34, -19, -62, 20, 80, 11,
    -30, 16, -50, 10, 64, 99, 27, -90, 39, 64, 18, -85, -27, 89, -27, 41,
    -37, 71, 3, 9, 98, -79, 1, 59, -39, 51, -64, 43, -93, 66, 39, 19, -43,
    15, 75, 23, 81, 24, 18, -30, -49, 54, 3, 58, -87, 30, 19, 89, -31, -37,
    -32, 14, -74, 37, 25, -59, -27, -76, 15, 3, -51, -24, -75, 17, -72, 14,
    -46, -61, 68, -21, -57, -53, 29, -87, -2, 3, -22, 91, -4, -25, -31, -86,
    -15, -91, 56, 59, -13, 11, -23, -87, -3, -36, 13, 62, -88, -31, -93, 70,
    -85, 67, 52, 49, -69, 51, -68, 82, 34, 28, 99, 38, -9, 53, 35, 30, -67,
    1, -33, 52, 72,
  66, 49, -43, 8, -85, -2, 10, 98, -32, 44, -96, 11, 79, 70, -97, -3, -37,
    -10, -98, 14, 39, 55, 26, 22, -60, 78, 62, -7, -68, 65, -75, -81, -10,
    50, 98, -7, 96, 75, -85, -48, -55, 78, -98, -6, 94, -83, 8, 82, 74, -10,
    74, -8, -90, 54, -61, 79, 83, -66, -38, 99, 73, -44, 76, -13, 20, 26,
    -59, -51, 31, 90, 82, 34, -46, 58, 51, 48, -80, 37, 97, 54, 94, -91, -20,
    75, -68, 92, 16, -29, 13, 12, -2, -66, -44, -38, 98, 9, -63, 64, -30, 25,
    96, -69, 60, -38, -12, 53, 31, -27, 29, 49, 69, -53, -19, 19, -69, 16,
    -28, -35, 95, -30, 31, 94, 28, -41, 24, 92, -32, -7, 36, 66, -85, 49,
    -36, -87, -41, -83, 6, -19, 58, -58, 82, 23, 96, -87, 23, -64, -6, -84,
    -92, 62, 45, -64, 14, 27, -48, 43, -90, -12, -11, -20, -51, 84, -43, -71,
    -37, -99, -11, 93, -58, 42, 40, -51, 54, -39, -35, -65, -35, -96, 67,
    -18, -72, -17, 42, -42, 82, 81, -16, -50, 58, 63, 49, 4, 24, -16, 0, 53,
    -92, -54, -54, 36, -90, -68, -7, 34, 48, -69, -83, 10, 43, 36, 73, -79,
    -85, -76, 20, -29, 54, 41, -84, 62, 75, 11, -21, -41, 72, -47, 47, -16,
    -58, 61, -8, 99, -46, -44, 28, -10, 57, -25, 2, 67, 80, 19, -61, 5, -54,
    23, 91, -65, 40, 8, 70, 90, -48, 80, 5, -6, 30, 33, -57, 98, 8, -9, -26,
    57, -91, -11, 52, -57, -99, -64, 38, 78, 58, -16, 57, 38, -40, -44, 60,
    -41, -25, 41, -51, 24, -59, -58, 5, -31, -100, -61, 33, 27, -18, -36,
    -26, 84, -31, 21, -6, 47, -47, -90, -69, -47, 79, 56, -49, -64, 7, 37,
    92, -48, -12, 44, -6, -17, -25, -39, -77, -19, -77, 59, -44, 97, -6, 65,
    -12, 9, -59, 16, -34, 33, -39,
  65, -81, -17, -99, 20, 6, -86, 67, -17, 3, 95, 57, -57, 88, -66, 99, 34,
    -40, -45, 67, 9, -67, -59, -3, -47, 31, -73, -83, 30, -10, 9, 65, -77,
    40, -44, 2, 72, -96, 88, 64, -17, 69, 80, 62, -30, 1, 50, -85, 8, -24,
    15, -85, -26, 48, 74, 45, 45, 60, 3, -62, 21, -45, 93, -53, -53, 77, -62,
    17, 63, -57, 40, 20, -6, -12, 95, 85, 58, 90, 76, 88, 60, -100, 24, 49,
    84, -87, -57, -76, -42, -30, -45, -64, -55, 27, 0, -71, 46, 83, 15, -18,
    98, 64, -20, 67, -89, 35, 5, -14, 33, 52, -10, 59, -55, 46, -29, -91, -9,
    -78, 86, -55, 12, -35, 46, 65, 57, -39, -90, 52, 20, 51, -40, -59, 72,
    72, -98, -40, 25, 96, -32, 87, 25, 31, -7, -44, 24, -74, -32, 24, 58,
    -20, -88, 0, -34, -82, -73, 33, 0, 95, -48, -12, -55, -49, 68, -98, -72,
    -42, -21, -87, -31, 61, -80, -80, 16, -66, -66, 36, -76, -94, -81, -6,
    -95, -18, 13, 40, -2, 97, -36, 64, 96, -83, 67, 74, -70, -33, -48, -73,
    -78, 30, 16, 28, -67, 86, -62, -57, 9, 5, -64, 55, 32, 84, -53, 15, -45,
    19, -76, -10, -80, 32, 75, -79, 26, -30, -79, -58, -63, 67, 22, -59, -24,
    -27, 11, -61, 47, -49, 92, -83, 48, 30, 13, 97, -40, -5, -73, -49, 52,
    64, 85, -5, 19, 51, -4, 63, -11, 88, 88, 80, 34, 88, 3, -60, 95, 41, 17,
    6, -27, -100, 24, -72, -1, 58, 51, -83, -41, -14, 20, -49, -29, 63, -91,
    3, 68, 70, -30, 97, -97, 44, 51, -53, -71, -75, -63, 2, 92, 46, -91, -16,
    37, 51, 84, -24, -27, -70, -86, 75, -1, -32, -97, 64, 79, -74, 84, -69,
    60, 14, -24, 35, -82, 4, 66, -43, 3, -1, -71, -92, -60, 25, -62, -36, 81,
    -3, -85, 9, 9,
  -15, 54, -24, 98, -27, 76, -67, -58, 19, 36, -83, -17, 60, 96, 7, -78, 30,
    -16, -12, -6, 47, 52, 67, 25, -65, 47, -91, 54, 18, -20, 60, -18, -19, 6,
    -53, 19, 15, 85, 14, -34, -78, -67, -89, -37, 53, -94, 75, 52, 48, -86,
    42, 97, -32, 21, 12, 28, 4, 90, 7, -35, -56, -89, -80, -72, 50, -54, 91,
    -17, 11, -21, 12, 62, 4, -57, 97, -96, 5, 16, -21, -3, 24, 55, -31, -10,
    -61, -68, 95, 31, 61, 4, 26, -52, 49, 26, -57, -44, -57, 25, -47, 11, 51,
    -93, 51, 97, -81, 37, -43, 3, -91, 8, 44, 93, -78, -66, -59, 93, -79,
    -19, 80, -4, -36, -86, -7, -41, -18, 9, -6, -8, 90, -41, 88, -64, 42, 37,
    13, 0, 32, 23, -3, -87, -74, -42, -25, 91, 12, 90, 59, 55, -77, -18,
    -100, 63, 96, 45, 39, -8, -92, -90, -39, 95, -47, -34, -18, -26, -83,
    -70, 35, -71, -30, 51, 66, -6, -55, -8, 52, 10, 69, 99, -45, 96, -27, 49,
    55, -17, 99, 52, 94, -92, 81, 30, 90, 36, 70, -59, -88, 18, 84, -15, -87,
    -46, -78, 71, 31, 93, -5, -9, -73, 85, 5, -48, 78, -29, -52, 77, 74, 83,
    79, -97, -60, 58, -34, 71, -53, 85, 72, 93, -30, -24, 41, -66, 34, -76,
    -95, -20, -8, 90, -82, 19, -3, -11, -28, -34, 93, 40, -56, -76, 98, 94,
    -45, -45, 17, 30, -7, -35, -33, 17, -100, 57, -84, 55, -26, -31, -42, 42,
    24, 25, -24, -17, 16, 89, -64, 94, 56, -38, -2, -54, -67, 80, 24, -46,
    -99, -96, 10, -48, 89, -70, 84, 63, 46, 7, 7, 49, 33, -3, -65, 95, -76,
    52, -10, -18, 48, 35, 10, 6, -86, 92, 72, -88, -77, -6, -75, 33, 10, -56,
    -45, -82, -45, 28, 48, 12, 82, -26, 25, 81, 27, 10, -3, -37, -42, -37,
    -25, 93, 98,
  -63, 0, 88, -36, 32, 23, -56, -92, -66, 0, -45, -4, 43, 86, -87, 74, 2, 2,
    -68, 62, 64, -19, 22, -38, -4, -70, -96, 72, 55, 43, -79, -17, -69, -48,
    -66, -46, 15, -3, 9, 95, -91, -40, 85, 32, 14, -83, -2, -29, -52, -13,
    37, 53, -52, -97, 60, 79, -39, -43, 1, -93, 71, -25, -79, 82, -55, 66,
    -70, 66, -91, -100, -59, -81, -57, -99, -58, -85, -81, 83, -90, -75, -58,
    -77, -58, -58, 0, -56, 68, 22, -64, 27, 6, 54, 96, 28, 36, 63, -69, 45,
    -56, -53, -46, -3, -49, 65, 37, -64, 81, -18, -78, 37, -43, -73, 2, 51,
    53, -74, 37, -82, 4, -64, -76, 80, 55, -91, -90, -10, 40, 59, -96, -94,
    -60, -53, -76, 5, -51, -70, -56, -76, -16, 93, -49, -63, -61, 83, 20, 70,
    3, 30, 0, 48, -44, 6, 19, -94, 84, -26, 41, 41, 6, 20, -19, 21, 85, -76,
    38, -75, -3, -82, 67, -67, -2, 43, -40, -9, -87, 62, -100, -4, 32, -52,
    -42, -3, -80, 39, -38, -63, 21, 60, 4, -19, -34, -78, -82, 93, -86, 90,
    -32, -12, 14, -56, 62, 32, 36, 78, 31, -50, 40, 96, -69, -37, -71, 59,
    -17, 39, -7, 26, -3, -50, 96, -76, 92, 43, 38, 75, 20, 82, -63, -29, -57,
    41, -43, -61, 85, -6, -79, 67, -60, 57, 7, 23, -11, 29, -79, -30, -41,
    -87, 77, -91, -80, -41, -49, -34, 99, -79, 57, 16, 57, -32, 56, -84, 69,
    -91, 21, -18, -24, 50, 59, 4, 83, 39, 64, -27, -80, 89, -28, 83, 60, -50,
    46, 29, 85, -39, -98, 66, 53, 0, -63, 77, -34, 88, 20, 56, -44, -35, -41,
    52, 68, 50, -1, 68, -66, -52, -2, 99, -78, 78, 33, 43, 80, 50, 56, 48,
    -70, -98, -67, -89, 53, -14, -42, -91, -54, -22, 8, -65, 91, 89, -35,
    -75, -37, -31, 20, -59, 75,
  42, -52, 99, -17, -56, -27, 74, 0, 98, 85, 18, 63, -6, -96, 73, -80, -37,
    12, 82, 74, -39, 15, 75, 61, -92, 67, 35, 66, -67, 81, -67, 95, 3, -17,
    95, 47, -82, -98, -83, 55, -86, -94, -76, -86, -71, -16, -1, -24, -57,
    -34, 19, 29, -13, 21, 88, -74, 34, 94, -67, -50, 45, 66, -11, -2, 70,
    -51, -19, 60, 74, 20, 33, 62, 44, -26, 42, -65, 34, -81, -46, -82, 2,
    -86, -2, 62, 90, -27, 39, 23, -100, -67, -63, -35, 26, 4, -80, 63, -87,
    48, -95, 31, 18, -70, -46, 40, 73, -98, 1, -96, -10, 35, 78, -54, 36, 87,
    -55, 97, -48, -46, -36, 57, 77, -34, -18, 76, -45, 76, 13, 50, 20, -8,
    12, 67, 27, 73, -5, -100, -49, -1, 26, 89, 58, 46, -48, -35, -83, -73,
    51, 37, -22, -62, -82, 92, 11, 74, 14, -51, -63, 91, -67, 53, 7, 29, -95,
    -14, 36, 79, 10, -94, -64, 80, 83, -14, 6, -55, -81, 51, 74, -63, 31, 24,
    19, -6, 33, 2, 62, 61, 15, 45, 76, -76, -67, 71, -7, -54, -44, 8, 51,
    -53, 80, 38, 91, -6, -43, 20, -65, 50, -47, -12, -35, -32, -76, 33, 13,
    -13, 37, -65, -86, -15, 12, -94, 90, -73, -74, -97, 71, 80, -78, 96, -23,
    -31, -52, -75, 99, 71, -11, -99, -77, 80, -76, -87, -69, 56, -96, 89,
    -38, 94, -48, -24, -18, -82, -34, -45, -64, -50, 93, -98, 22, -69, -2,
    -84, 82, 20, 41, 15, -75, -100, -86, -6, 99, 54, -5, -18, -2, 12, -8,
    -10, -32, -66, 56, 32, -69, -21, -53, -55, -19, 81, -75, 81, -42, -54,
    -16, -100, -61, -16, 30, -35, 25, -16, -5, -75, -51, 32, 99, 62, 42, 63,
    93, -73, -71, -59, -25, 59, -31, 35, -58, -85, -24, 11, -58, -82, -58,
    -50, -69, -69, 8, 93, -64, -6, 1, 50, 99, -1, -24,
  -71, 76, 96, 37, 20, 8, -67, -47, -89, 62, 52, -52, 78, 86, -62, 60, 46,
    44, -68, -44, -34, 28, 30, -30, -2, -1, 14, -64, -68, -63, -88, -86, -95,
    40, -70, -55, 5, -56, -50, -61, 88, 57, 45, -73, -3, -45, 19, 5, -73, -6,
    -78, -82, -37, 2, 61, -44, -67, 26, -27, 94, 3, -34, -10, 71, -46, 19,
    68, -35, 46, -22, 61, 70, 68, 99, -99, 60, -44, 92, 48, 98, -68, -10, 97,
    1, 12, -94, -82, 49, -98, -84, -1, 14, -7, 23, -65, -67, 63, -40, -48,
    35, -68, -7, -87, -14, 61, -53, -62, -27, 35, -16, 87, 79, -11, 74, 74,
    66, -89, -62, -71, -96, -66, 24, -7, -84, -94, -75, 70, -38, 10, -20, 89,
    32, -3, 59, -73, 15, -2, 36, -32, -4, -10, -38, -48, -70, -41, 67, -73,
    -20, -37, 70, 12, -49, 93, 88, -96, 96, 23, -96, -23, -84, 6, -64, 38,
    58, 20, 89, -85, -88, -41, 1, -90, -87, -54, 24, -82, 23, -21, -85, 20,
    -13, -81, -72, 28, 94, 16, -34, 66, 90, 44, -42, -95, 37, 74, 62, -23,
    -10, -47, -20, -80, -45, 10, 61, 28, -70, -5, 43, -68, -74, 21, 86, 26,
    -90, -64, -33, -31, -81, -77, -38, -98, 16, -100, 70, -99, 63, -20, 21,
    70, -42, 28, -24, 59, -83, -88, 53, 12, 43, -6, -8, -22, 58, 3, -83, 27,
    73, 13, -75, 44, -60, -66, -54, -27, 35, -55, 8, 61, 16, -7, -53, -98,
    -65, 72, 23, -80, 21, -58, 13, 48, -57, -80, 87, -21, 0, 31, 5, -11, 31,
    -86, 14, -32, -87, -29, 45, -33, -28, 7, -38, -79, -27, -83, -67, -44,
    96, 11, -12, -100, -97, 8, -83, 35, -12, -94, -23, -77, 65, -25, -91, -5,
    -1, -8, 48, 37, 49, 44, 8, 71, 68, -62, 91, 30, 15, 19, -42, -36, 41, 81,
    87, 60, -1, -38, 0, -71, -66, 41,
  -31, -100, 12, 35, 71, 19, -1, 73, -73, -72, 66, -25, -86, -76, -58, -81,
    18, -25, 82, 29, -13, -47, 11, 30, -29, -84, -58, 2, -68, 56, -14, 84,
    36, -61, -1, -69, -99, -88, -53, -67, 23, -84, 87, -1, -84, 42, 54, 65,
    66, 11, 8, -43, -97, 61, 42, -59, -27, -9, 48, 75, -40, 26, -61, 26, 73,
    34, -21, -18, -54, -34, 79, -49, -94, -57, -91, -67, -71, -45, -21, 17,
    -56, -54, -8, 67, 30, 25, -45, 10, -67, -48, 56, 28, -53, -8, -41, -62,
    84, -17, -87, 73, 74, 82, 70, -2, 57, -84, 79, -51, 9, -54, 0, -9, 37,
    76, -26, 14, -43, 96, 67, -15, -17, 34, 89, -89, 75, -91, 84, 62, -64,
    -35, 40, 67, -38, 75, 34, -91, -42, -2, -25, 26, -16, 97, 53, -7, 38,
    -16, -52, -18, -84, -75, -4, -20, -6, 19, -29, 63, -53, -81, 11, -97, 43,
    83, 77, -100, -80, 93, 98, 99, 55, 17, -11, 54, -18, -95, -80, 98, 20,
    92, 79, 34, 51, -19, 54, -24, 78, 95, -48, -40, 24, 31, -59, 47, -29,
    -11, 70, -6, -98, -86, 99, -5, -94, -15, -82, -28, -71, -67, 72, -52,
    -83, -46, -76, -91, 94, -1, -44, -44, 99, 89, -59, 18, 6, 51, -51, 71,
    -99, -55, -65, 67, -48, -67, 46, -4, 92, 61, -99, 83, -13, 90, -72, 62,
    -77, -32, -74, -83, 15, 3, 86, -7, -73, -79, -76, -28, -15, 64, 13, -65,
    94, 65, -20, -41, -83, -79, 80, 56, -58, 46, -77, -32, -76, -38, 82, -5,
    -95, -21, 68, -41, 10, -55, -80, -82, -45, 62, -21, -32, 40, 85, 9, -38,
    2, 19, 19, 79, -44, -73, -12, 45, -49, 89, 48, 36, -78, 70, 22, -55, -3,
    -44, -65, 93, -80, -74, 55, -96, 7, -69, -45, 56, -69, 34, 75, -52, 37,
    91, -5, -78, 12, -82, 51, 3, -34, 56, -60, -6, 9,
  66, 43, 5, -62, -63, -18, -20, 74, -99, 88, -18, -35, -42, -60, 94, 49,
    -11, 55, -62, 94, -86, 30, 45, -63, -81, 17, 51, -72, -13, 27, -25, -51,
    69, 8, -62, 95, -50, -82, -92, 54, -61, -80, -19, 32, -88, -66, 8, 44,
    65, 10, 54, 53, -90, -94, 74, 35, 3, -96, -30, 11, -25, -69, -30, 43, 34,
    18, 71, -78, -60, 70, 52, 45, -36, 86, -11, 63, 80, -52, 18, -17, 47, 69,
    63, -99, 17, 16, 49, -87, -91, -74, 98, -51, -29, 30, 34, 33, 63, 31, 30,
    54, 87, -57, 44, 77, 75, -16, 26, -74, -23, -69, -59, -87, -54, 14, 34,
    -6, -71, 63, -63, 16, 67, -27, 18, 29, -30, 3, -48, -48, -39, 85, 78, 26,
    98, 37, -5, 59, -18, 17, -28, -36, -62, -40, 14, 17, -35, -33, 46, -60,
    9, 75, 32, 39, 30, -68, 2, 28, -60, -94, 89, 45, -86, -98, -63, -22, -35,
    74, 27, 40, 44, 42, 19, -48, -68, -4, 42, 18, -28, -55, -38, -75, 54, 2,
    -69, 69, 2, 60, 43, -76, 41, -9, -28, 50, -52, 51, -26, 37, 27, -42, 7,
    70, -96, -36, -15, -12, -31, -90, 37, -94, -42, -69, -45, 86, -55, 56,
    -80, -13, -20, -84, 84, -30, -51, -80, -29, -18, -34, -38, -74, 90, 43,
    -8, 62, 60, 22, 33, 57, 47, -38, 42, -60, -62, -42, -50, 39, 36, -70, 93,
    78, -3, 74, 30, 59, -1, -69, -87, 44, -14, -49, -64, -27, -23, -85, -43,
    86, -39, 11, -35, 40, 49, -32, 54, 63, 12, -60, -21, 45, -63, -37, 70, 1,
    61, -45, 69, 8, -36, -50, 88, -4, 88, 90, -59, -79, 90, -96, 90, 21, -4,
    99, 59, -99, 61, 9, -52, 17, -17, -58, 48, 71, 80, 33, 45, 14, -35, 81,
    -92, -59, 5, -93, -56, 6, 51, 1, 6, 96, -7, -43, -47, -85, -99, 59, 79,
    83, -97, -53,
  69, -75, 39, 78, -10, -65, -37, 58, 58, 14, 86, 69, -84, -48, -33, 99, 60,
    11, -35, -33, -18, 94, 68, -91, 75, 85, -15, 9, -20, -15, 40, -20, -41,
    47, -99, 68, -35, -15, -74, -31, -44, 9, -20, 45, -91, 53, -29, 23, -34,
    70, 33, 15, 37, -90, -85, 33, 58, 9, 43, -78, -31, 35, -68, 82, 84, 18,
    46, -91, -31, 94, 79, -8, 94, -52, -30, -11, 47, -5, -61, 99, 48, 22, 48,
    -34, 99, -30, 56, 45, 89, -39, 75, 85, 7, -3, 16, 55, 29, 30, -40, 24,
    -100, -65, 6, 55, 64, -62, 37, 45, -79, -63, 73, 73, -77, 28, -4, -42,
    71, -59, -44, 25, -67, 99, 61, 8, -50, -71, 13, 20, -32, 47, -14, 1, -20,
    72, 59, 84, -35, 66, -40, 83, -86, 16, 66, 12, 70, -50, -95, -3, 41, -48,
    -15, -48, -96, -9, 3, 74, -89, -23, 98, -81, 66, 51, 44, -96, -3, 55, 46,
    -38, -42, -37, 74, -87, 25, -40, 41, -42, 79, -58, -17, -50, -74, 92,
    -64, 34, 90, -65, 86, 85, 4, 71, -86, -37, -16, -46, -28, 30, 76, -93,
    -18, 34, 50, 50, 8, -48, 82, -99, -91, -43, -61, 74, -54, -71, 20, -60,
    -61, 27, -8, 85, 25, 40, -77, 95, -91, 35, 33, 77, -49, -78, -71, -5, 11,
    28, 43, 28, 61, 67, -72, -67, -12, -30, -16, 62, -98, -39, 87, -53, 84,
    13, 25, -25, -37, -96, 94, -51, 50, -73, 87, -27, -43, -72, -88, 35, 94,
    72, -39, -87, -35, -97, 5, 58, -89, 11, 18, 34, 35, 30, -44, 15, 88, 64,
    -64, 86, -90, -98, -10, -39, -13, -2, -82, 59, -37, -71, -22, -63, 13,
    -75, 96, 60, -37, 72, -34, -96, -3, -18, 44, -20, -11, -61, 55, -4, -44,
    -65, 20, -49, -82, 77, -31, -70, 90, -70, -90, -92, -88, 2, 21, 10, -47,
    -60, -36, 80, -51, -92, -95,
  -59, -81, 10, -18, 75, -8, -44, -56, -45, -14, -26, 40, -62, -16, -12, 60,
    66, 23, 68, 93, -47, 13, 19, 58, -17, 35, -46, 38, -52, 95, -56, 49, -60,
    -23, -70, 3, -53, 19, -45, 96, 3, -24, 91, -72, -3, 12, -63, -57, 1, -59,
    49, 25, 99, 35, 23, -91, 90, 60, 85, -3, 68, 80, -29, 75, 78, 55, -36,
    -24, 84, -13, -87, -36, 70, -97, -27, 32, -57, -87, 37, -48, 45, -85,
    -15, -59, -38, -13, -5, 73, 60, -21, 35, -51, -24, -19, 74, 28, -37, -90,
    27, 13, 59, 0, -74, 36, 85, -42, -62, -94, 76, -7, -38, -52, 15, -27, 35,
    69, -13, -25, 82, 32, -54, -21, 78, -12, 78, -98, -39, 37, -83, 18, -21,
    -27, 74, 31, -52, 53, 99, 88, 39, -13, -71, -74, 62, 2, 98, 44, 22, -12,
    39, -58, 36, -15, 46, 10, -97, -80, 42, 88, -14, 22, 21, -42, -100, -83,
    10, 97, 95, 75, 43, -89, -47, -81, 54, -51, -85, 78, 87, -39, -61, -40,
    -22, -55, 10, 11, 13, 59, 77, 65, -29, -75, -58, -50, 25, -26, -46, 34,
    96, 89, -71, 97, 26, -6, 4, 27, -91, 22, 97, 0, 89, 3, -14, -25, 88, -66,
    -87, -78, -70, -75, 45, -6, -88, -16, -67, 43, 28, -71, 27, 40, -92, -65,
    2, -40, -67, 31, 53, 42, -49, 89, 61, 1, 90, 79, 84, 17, 78, -19, -13,
    25, -49, -58, 85, 32, 29, -49, -70, -11, -67, -23, 93, 51, 0, -15, 80,
    74, 67, 69, 44, 25, -95, 28, 62, 64, -52, 86, -39, -95, -26, -88, 20,
    -45, -88, -40, -60, -87, -88, 96, 59, -6, -55, 9, 99, 69, -28, -81, 32,
    36, -11, -86, -87, -84, -45, -85, -77, -48, -89, 66, 62, 83, 16, -17, -7,
    2, -68, -93, 69, 69, -92, -86, 76, -41, 0, 88, -94, -68, -47, -34, -45,
    32, 74, -65, 36, -20, -13,
  35, -20, -35, -44, -62, 44, 65, -42, -48, 89, 36, -9, -14, 2, 52, -75, -84,
    88, 18, -52, 63, 46, 41, -59, -40, 58, 65, -59, 60, -88, -30, 8, 6, 76,
    46, 45, -66, 10, 21, -50, -28, 66, -67, 74, -58, 37, -46, -26, -63, 38,
    -36, 59, 34, 56, -28, -91, -65, -50, 46, -54, 53, -62, 45, -24, 12, 3,
    -58, -25, 3, -86, -78, -9, -9, -16, 64, -28, 96, 55, -68, -78, -88, -95,
    48, 40, -69, -63, -12, -6, -38, 64, 97, -51, 11, 7, -55, 83, -40, 69, 54,
    28, 58, -29, -82, 29, -41, 11, -67, 88, -37, -91, 49, 27, 80, -20, -76,
    64, -9, -20, -45, -64, 97, -18, -49, -48, 7, 83, 68, -50, -86, -82, 4,
    -33, -49, -27, -14, 79, -43, -69, 10, -35, 76, 36, -37, -19, -37, -73,
    -52, 15, 64, 97, 69, 25, -80, 93, -69, -87, 62, 50, 2, 20, -29, 2, 71,
    -16, -50, 44, -6, 87, -87, -25, -63, 63, 81, -30, -11, -91, -71, 14, 83,
    -8, -3, 11, 53, 84, -93, 43, -86, 64, 89, 9, -19, 4, 9, -22, -52, 0, -28,
    -38, 92, -42, -2, -20, 85, 37, 30, 46, -67, 65, -2, -67, -7, 30, -61, 65,
    -65, -81, 70, 17, 47, 64, 90, -58, -77, 2, 57, 56, 65, -35, -55, 54, -31,
    14, -41, -48, -28, 10, 43, -80, 44, 28, 19, -68, 79, -42, -21, -62, 37,
    -28, 93, -11, 87, 92, -71, 60, 35, -71, -84, -21, 71, -99, -84, -92, -77,
    -41, 85, 47, -73, 61, 39, -62, -69, 89, 23, 22, -90, 43, -83, 91, 67, 7,
    -87, 61, -3, 5, 77, 60, 38, -55, -22, -17, 36, 90, -19, 51, -80, 72, 63,
    -92, -3, 71, 20, 97, 21, 34, 97, -60, -74, 36, 63, 48, 39, 83, -57, -4,
    -25, 62, 91, 48, 60, 95, -48, -25, 63, -37, 66, 35, 93, 40, 62, -94, -39,
    23, -65,
  -51, -23, 75, -3, -35, 65, -75, 69, -34, -40, -59, 88, 45, -76, 57, 23, 0,
    87, 53, -33, 79, -62, -49, -35, 68, -4, -30, -23, -61, -89, -47, 13, 36,
    -60, 5, 66, -89, 27, 89, -89, -3, -24, -18, -80, 46, -68, -66, -28, -54,
    -32, 3, -16, 99, -11, -43, 98, -43, 81, 13, 68, 76, -5, -53, 32, -85, 19,
    -16, 15, -66, -33, -26, 90, -91, -25, -9, 67, -100, 80, 13, -45, -75, 92,
    87, 88, 97, -22, 41, -57, -37, -94, 80, 58, 58, 68, -55, -78, 1, 51, 86,
    -47, -32, -96, -49, -66, -91, 25, -83, -2, -88, 28, -5, 62, 14, -42, -3,
    32, -22, 82, 98, -72, 85, 39, -88, 63, -42, -59, -96, 1, 70, -72, 54,
    -54, 59, 50, -82, -57, -73, 25, -66, -90, -75, -90, 54, 96, -4, 73, -81,
    -70, 28, 84, -18, -31, -13, -64, 10, 19, -39, 13, -87, 54, 7, -4, 80,
    -35, -54, 45, 26, -22, -86, 99, -29, 10, -44, 71, -88, 45, 72, 97, -20,
    -22, 83, 5, 3, 41, 84, -45, -20, 4, 7, 46, -21, -44, 99, -42, 51, -72,
    71, -58, -41, -9, 8, -84, -62, 68, -31, -54, -86, -49, 42, -47, -95, -81,
    -66, 24, -68, -98, -81, 47, -53, -16, 71, -98, 96, 59, -22, 14, 91, -18,
    -50, 73, -58, -32, 45, -36, -59, -81, -36, 41, -84, -98, 19, -46, -24,
    -43, 42, 63, -56, -64, 79, 15, -20, -57, 44, 78, 34, -28, -32, -47, -86,
    -89, -51, -14, -72, 34, -71, -17, 90, -71, 46, -88, 49, 15, -76, 93, 33,
    88, 68, -55, 4, -19, 29, -9, 79, 52, 88, -55, -77, -13, 57, 90, -40, 35,
    -70, -93, 23, 69, -1, 3, -21, 65, 44, -89, 5, 25, -61, -95, -97, 65, -80,
    58, -60, 43, 52, 70, 77, -18, 43, -23, -11, 15, 82, 70, 43, 28, 48, 20,
    -8, -35, 35, -46, -95, -67, -73,
  93, 11, -86, 76, -48, -66, -68, 18, -67, 27, -51, -68, -83, -38, -99, 84,
    -65, -27, 41, -56, -12, 57, 87, -64, -39, -57, -34, -22, 6, -94, 98, -14,
    -67, -9, 20, 67, -77, -86, 14, 40, -61, 34, 48, -51, -63, 51, -41, -64,
    -11, -10, -32, 28, 33, -58, -75, 57, 49, 13, 70, -3, 37, -94, 81, -52,
    32, -32, 28, 70, 87, 75, -65, -67, -14, -80, -67, -80, 92, -20, -66, 27,
    18, 67, 99, 62, -50, -76, -78, -48, 89, -68, 58, 1, 1, 74, 90, 99, -70,
    -43, 49, -99, -86, -37, 81, 95, 98, 59, -80, 34, -38, -75, 99, 42, -10,
    40, 84, -63, 48, -37, 86, 87, 69, -27, 32, -61, 5, -3, 18, -33, 97, 31,
    74, 97, 78, -29, -92, -75, -50, -28, 92, -12, -43, 38, 1, 50, 86, 94,
    -32, 27, 3, 65, -5, 33, 17, 29, -42, -45, 76, -74, -32, 54, 18, 90, -20,
    51, -71, 6, -26, 36, -76, 98, -100, 11, 48, -23, -78, -64, 38, 37, 53,
    77, -57, -83, 13, -27, -14, 2, 74, -94, 35, -27, 13, 46, -37, -13, 79,
    24, 38, 18, 64, 2, -69, -23, 87, -70, -72, 74, 20, -20, -71, -21, 18,
    -30, -8, 9, 22, 4, 51, -45, 48, -16, 61, 68, -25, 15, -35, 74, -16, -3,
    13, 28, 30, -57, 97, 2, -71, 18, 2, -45, 41, 26, 52, 30, -16, -71, 29,
    54, -45, -58, -10, 25, 91, -44, -50, -1, 80, -46, 11, 16, -35, -17, 0,
    25, -81, 21, -39, 16, -86, 4, -69, -81, -32, 96, -10, 88, 21, -8, -35,
    -25, -20, -58, 25, -9, -62, 37, -27, 63, -85, -3, 11, -31, 46, -29, 71,
    88, 86, 97, 47, -32, 30, 22, -48, 97, -92, 26, 87, -16, -88, -92, 1, -2,
    -17, -25, 56, -86, -18, -37, 24, -38, -36, -46, -5, 73, 82, -35, -96, 90,
    -11, 46, -69, 43, 26, -49, 86,
  98, -67, 61, -9, -61, 15, 27, -25, -80, -73, -21, 43, -40, 31, -69, -62,
    -63, 3, 19, 0, 64, 29, -80, -87, -37, 94, 61, 41, -14, -63, 99, -5, 40,
    79, -54, -84, 35, -16, -35, -86, -2, -13, -51, -3, -66, 20, 47, 13, -28,
    -60, 73, 89, 39, 68, 55, 85, 21, 16, -37, 43, 19, 96, 30, -5, 9, 90, -90,
    -24, 74, -33, -58, -95, -34, -80, 79, -67, -97, 96, -87, 68, -79, -33,
    -60, -48, -28, -51, 59, 70, -68, 93, 94, 87, 84, -74, 6, 93, -28, -69,
    -14, 63, -92, 97, -57, 59, -39, -76, -38, 80, -48, -20, -57, -84, 14, 95,
    2, -19, 80, 50, 2, -50, 6, -36, 92, 84, 99, 39, 33, 7, 36, 38, -82, 87,
    -84, 19, -4, -32, 58, -41, 25, -56, -14, -88, -81, -59, -98, 26, 18, 28,
    -24, 96, -54, -43, 44, -35, -64, -56, 60, 70, -49, 10, 56, 70, -95, 37,
    -20, 22, 28, 4, 4, 89, 2, 28, 88, 66, 12, -36, 60, -66, -90, -45, 93,
    -62, -25, 7, -36, -42, 6, -13, -12, 61, 13, -93, 60, 32, 25, 37, -43,
    -26, -31, -35, 78, 94, -21, -95, 64, -91, -54, 98, -97, 94, 85, 12, -35,
    64, 68, 10, -4, -50, 95, -82, -5, -90, 60, 39, -35, -14, 84, -96, -90,
    -66, 90, 19, 75, -22, -60, -70, 28, -93, 57, -34, -36, -70, 26, -27, -30,
    -30, 18, -61, -17, 16, -65, 40, -98, -26, -93, 21, -60, -76, -36, 5, 96,
    30, -90, 41, 85, -9, -13, 87, 96, 5, 75, 18, 38, 21, 82, -58, -96, -34,
    -60, -22, -20, -13, -28, 79, -85, 56, 67, -4, -69, -43, 29, -24, 54, -70,
    -6, -74, -82, -57, 66, -79, 88, 74, -44, 0, -97, -19, -90, -30, -66, -44,
    68, 60, -79, 49, -23, -38, 21, 90, -36, 37, -87, -76, 28, 74, 20, 9, 89,
    -100, 63, 97, -95, 23, 76,
  34, -47, 27, -58, -91, 82, 37, 53, 21, -99, 23, -11, -75, 70, 26, -93, -82,
    -62, 83, 78, -60, 65, -17, 12, 57, -1, -75, 34, -15, 90, -24, -78, 68,
    -29, -4, 39, -61, 44, -35, 17, 27, -12, -10, 85, -46, -23, -10, -80, -40,
    -22, 96, 54, -71, -73, 17, 32, -27, 20, 2, -28, 29, 59, 41, 86, 4, -37,
    -29, -87, 15, 14, -15, 46, 62, 85, 39, 65, 25, 87, -42, -30, -64, 11, 33,
    50, 47, -1, -25, -42, -37, 29, -53, -95, 81, 22, -90, 55, -12, 36, 70,
    -1, -36, -96, -100, -3, -72, 17, -74, -56, -69, -60, 73, -79, 36, 6, 43,
    -44, 45, -37, 61, -38, 63, 83, -51, 19, -37, 77, 42, -21, -19, -96, -96,
    -32, -3, -76, -78, 73, 71, -53, -11, -34, -54, -10, -79, -28, -65, -57,
    -41, -20, 78, 66, -23, -49, -13, -74, -8, -68, -65, 59, -14, 94, 35, -20,
    52, 60, -69, -16, 50, 40, 47, -44, 45, -90, 39, 0, 48, 97, -68, -80, -50,
    17, -36, 47, -35, 19, 64, 48, -19, -49, -89, 46, 54, -75, -25, 57, -17,
    35, -44, -8, 90, -54, -29, 65, -95, -29, -42, 20, -49, -51, -29, 56, 52,
    89, 36, 89, 17, -71, 66, 10, -27, -59, 64, 23, 76, 40, -27, -73, 49, 93,
    13, -27, -45, 88, -34, 62, -85, 89, -17, -60, 78, 39, -16, 57, 47, 11,
    -55, -81, 8, 51, -82, 61, 10, -48, -85, -38, -28, -75, -33, 39, -69, -51,
    -57, -56, 66, -33, 58, 87, 6, 45, 28, 72, 98, 56, 10, 56, -29, -81, 66,
    25, 15, 56, 27, -76, -66, -31, -29, 69, -73, 18, -99, 64, 55, 19, -55,
    44, 4, 18, 32, 57, -5, 64, 85, 82, -98, -89, 47, -10, -67, -62, -45, -19,
    -3, -89, -26, 9, 54, 85, 86, -86, -89, 25, -38, -33, -2, -93, 25, -34,
    85, 99, -15, -9, 80, 69, -13,
  -41, 88, 15, -42, -7, -7, -37, -25, 68, -55, 29, -35, 51, -56, 96, 27, -4,
    54, 12, 40, 13, -64, -58, 49, -71, -9, -35, -92, -62, -62, -63, 78, -38,
    -75, 16, -38, 93, -44, 0, -31, 78, 80, -2, -29, -96, 27, -30, 40, -6,
    -64, 31, -76, -98, 76, 75, 76, 23, 42, -60, -63, -21, 63, -98, -96, 43,
    45, 91, -78, 83, 1, -97, 54, -30, -26, -35, 12, 25, 0, 18, 91, 39, 6,
    -100, -82, -50, -27, -92, -74, -92, -91, -37, -90, 43, 52, 5, -54, -72,
    73, 29, -91, -68, -51, 75, -72, 33, 68, -72, -47, -69, 39, 32, 55, 49,
    -76, -3, -100, 88, 7, -35, 72, -82, 5, -97, -19, 3, -90, -10, 72, -17,
    87, 24, 9, -75, 20, 4, 44, -4, 3, 47, -72, 14, -45, -100, -74, 51, 17,
    28, -41, -11, -41, -36, 37, -97, -56, 45, -64, -35, 40, -45, 76, -52,
    -19, 48, -38, -31, -54, 15, 8, 5, 91, 80, -38, 8, 17, -17, 74, 7, -46,
    -30, -35, -10, -34, 85, 1, -58, -62, 73, -11, 26, -3, -49, -49, -55, -8,
    31, 86, 86, 32, -72, 36, -52, -44, 12, -50, 87, -13, 20, 57, 98, 63, 9,
    -36, -12, -87, -2, -41, -53, -56, -28, 6, -88, -16, 98, 74, 77, -23, 36,
    71, 47, 57, -64, 32, 89, 3, -63, 34, 81, 84, 35, -15, -30, -30, -84, -24,
    -97, -82, -66, -93, -64, 72, 96, -46, -17, 53, 92, 99, 92, 35, -31, -65,
    26, 33, 75, -12, -26, -91, -69, 18, -4, 27, -54, 81, -61, -57, 58, 15,
    -19, -7, -5, -87, -63, 43, 34, -64, 15, -1, 2, -74, -39, 2, 91, -40, -55,
    -44, 65, 50, 2, -27, -65, -18, -92, 38, -27, -15, -16, -83, 97, 65, 38,
    -80, 45, -35, -81, 28, 10, -92, 69, 95, -50, 92, -29, -51, -40, 94, -38,
    -4, -65, 16, 61, -91, 83, 8, 16,
  -60, -100, -91, 81, -59, 98, -91, -53, -72, -14, -22, 83, 48, 83, -52, 12,
    99, -60, 96, 46, -7, 9, 72, -25, 94, 2, -48, -86, 48, 86, 80, -31, -22,
    -26, 75, -53, 45, -49, 92, -70, 1, -21, 92, 87, 95, -33, 17, -68, -39,
    84, 83, -45, -17, -19, -2, -38, 24, 55, 8, 71, 86, -46, -36, 91, 46, -9,
    89, -32, -31, -54, -14, -56, 28, -50, 9, -2, -13, 44, -52, 66, 40, -72,
    69, -74, 89, 83, -84, -41, 91, -91, 79, -17, 8, -22, 19, 12, 74, -39, 23,
    18, -83, -99, -56, 61, 52, -21, 33, 81, 27, 60, 10, -67, -51, 70, 61,
    -76, 77, 9, 50, -81, 18, -44, -28, -9, 44, 93, 73, -92, -42, -20, 11, 55,
    71, -6, -52, 0, -71, -96, -97, -80, -32, -24, 5, 83, -63, -2, -88, 80,
    76, 77, -49, -86, -86, 48, 34, -42, -58, -54, 11, 91, 20, 43, -54, 27,
    -12, -89, -85, -100, -78, -37, -24, -34, 45, 7, 96, 33, 42, 21, -86, -99,
    26, 19, -17, 79, 60, 6, -53, -53, 8, 96, -35, -11, -32, 93, -71, -11, 87,
    8, 91, 19, 6, -50, -70, -92, -52, 40, 47, 32, 58, 57, 66, 54, -16, 86,
    68, -8, 85, 98, 42, -8, -17, 44, -5, 5, 31, 13, -72, 86, 86, 78, -18, 59,
    9, -84, -83, -75, -23, 96, 14, -8, -6, 63, 87, -93, 24, -90, -65, -93,
    19, -94, -1, -76, 63, -41, -94, 34, -49, -69, 22, 53, -68, 8, -70, -71,
    -15, -56, -1, -35, 56, -57, -21, -96, -44, -12, 9, 24, 26, -17, -21, 6,
    -56, -3, -72, 18, 35, 4, -4, 53, 2, -33, -49, 0, 76, 42, -46, 10, 77, 89,
    18, 23, 76, -64, 91, 50, -78, -35, 22, -27, -84, -92, 36, -61, 98, 40,
    -3, -35, 29, 49, 63, -1, -63, 92, 77, 98, 96, -37, 0, -38, -6, -55, -45,
    60, 71,
  -78, -77, -42, -60, 77, -97, -70, 51, 67, 5, -79, 9, 3, 46, 6, 71, -28, 34,
    -45, -66, 57, 27, -21, 7, 58, -76, -20, 40, -75, -14, 20, -51, 49, 55,
    -31, 79, -7, -17, 71, -81, 98, -3, 96, 80, -29, -15, -87, -71, 91, 31,
    -33, 19, 67, -18, -18, -27, 34, -94, 68, -33, 29, -68, -58, 7, -31, 1,
    63, -2, 58, -100, 91, -82, -48, -75, 96, 67, -47, -87, 79, 45, -34, 53,
    74, 34, 40, -95, 99, 46, -88, 87, 75, 54, 55, 95, -89, -21, 52, -12, 34,
    -18, -64, -40, -26, 18, -20, 34, -25, -53, 76, -46, -80, -28, 3, 98, 98,
    -12, -71, 5, 36, 5, -35, -81, 23, -76, -75, -42, -45, 69, 34, -75, 93, 2,
    -89, -15, 91, 43, 9, -18, 39, 64, 22, 64, 62, -9, -38, -24, -5, -16, -82,
    3, -49, 40, -7, 97, -74, 85, 98, 5, -10, -23, -97, -27, -78, 75, 62, 57,
    -81, -35, -22, -58, 31, 68, -47, -88, 78, 69, -4, -28, -40, 91, 3, 86,
    -30, -7, -83, -54, -84, 85, 74, 59, 68, -22, 70, -77, 83, -1, 21, -14,
    -57, 84, -67, 87, -16, 58, -13, 7, -100, -74, -95, 28, -67, 15, 22, 83,
    82, 44, 44, -68, 49, -82, -71, 57, 1, 13, -87, 74, 70, 82, 11, -71, 7,
    -52, -23, -80, -75, -24, -77, -94, 98, 40, -18, 19, 4, 30, 69, -56, 82,
    71, 85, 81, 24, -23, 21, -44, 84, -74, 13, -19, 9, -44, -14, 78, -10,
    -27, 15, 36, 96, 17, -69, 3, -74, 27, -38, 34, -42, -44, -93, 93, 81, 48,
    96, -30, 54, 14, 99, -40, 21, -77, -79, -81, -94, 13, 51, -67, 67, -78,
    20, 55, 96, -21, 65, 70, 22, -92, -74, 0, -70, 47, -55, -51, 89, -96, 34,
    49, 73, -56, -96, -4, -24, 30, -9, -98, 64, -80, -63, -67, -62, -40, -30,
    -92, -11, 51, -13,
  8, -54, 78, 91, 90, -15, -52, -23, 92, -75, -53, -94, 73, 55, 3, 28, -35,
    97, -71, -60, -68, -2, 6, 52, 39, -90, 12, -50, -67, -62, 43, -14, -24,
    61, -64, 96, -47, 4, 89, -81, 8, 86, 37, -99, -65, 73, -18, -81, -89,
    -62, -32, -88, 1, -97, 63, -100, -20, 8, -8, 65, 38, -96, 19, -60, -89,
    26, -8, -38, -25, -65, -33, 61, 83, 58, 66, -82, 18, 88, -61, 84, 90, 81,
    -39, -79, 95, 87, 67, 13, 13, 32, 20, 66, -94, 64, 36, 9, -11, 43, -93,
    58, 78, -81, -21, 63, -71, 67, -49, -23, 88, -41, -52, -84, -34, 97, 51,
    69, 11, 25, 68, -19, 24, 45, 23, 3, -66, -65, 30, 94, 21, 21, -74, -87,
    82, 16, -34, -3, -63, 66, 23, 87, -40, 85, 75, 39, -7, -51, 93, 72, 3,
    -46, -57, 40, -14, 27, -59, -27, 14, -41, -40, -66, -62, -53, 72, -25,
    -19, -98, 69, 96, -85, 50, -93, 38, 77, -50, 81, -85, 26, -48, 31, -83,
    90, 75, -42, 39, 49, 17, 29, 66, -87, 95, 22, -15, 86, 12, -71, 82, -79,
    86, -28, -90, -11, 60, -22, -93, -19, 50, 94, -6, 89, 78, -79, 96, -17,
    -21, -27, 30, 88, 56, -100, 8, 17, -50, 23, -1, -86, -5, 69, -3, -46, 89,
    -23, -29, -98, 49, -36, -100, -88, 65, -77, -96, -13, -67, -70, -51, 31,
    87, -99, -84, -66, 29, -48, -76, -97, -83, -26, 40, -63, -61, -23, -53,
    90, 80, 78, -36, -55, 70, -63, 38, -14, 62, -58, -61, 1, 81, -40, -9,
    -31, 33, 95, 8, 82, -3, 36, -21, 80, -25, 73, 80, 21, -80, 28, 46, 41,
    -13, -30, 21, -18, 45, -5, -73, -72, 24, -42, 11, -86, 92, 98, -65, 70,
    75, -85, 96, 3, 3, -11, -49, -99, -23, 78, -79, -13, 19, 7, -8, 23, -60,
    -90, 75, -5, -52, 24, 16, 41,
  76, 75, -99, 37, -21, -72, -38, -29, 19, 90, 8, -81, -44, -4, -97, -8, 79,
    67, 73, -55, -79, -5, 85, 63, 95, -30, -19, -47, -57, 27, -84, 69, -26,
    -40, -3, 88, -37, -78, 84, -38, 96, -5, 94, 3, -2, 59, 54, 87, -89, 93,
    44, 86, 39, 58, -39, -79, -67, -39, -98, 36, -93, -10, -84, -91, -88,
    -67, -7, -90, -79, 77, -15, 19, 57, 82, -20, 79, -94, 23, 76, -24, 30,
    -59, 33, 70, -44, 51, 70, 72, -37, 54, -73, 13, -31, -3, -32, 94, -24,
    11, -13, -4, 2, 63, -77, 63, 7, 58, 91, -35, 91, 29, -26, 93, -17, -49,
    -23, 0, 97, 51, 93, -35, -29, -59, 77, -65, 56, 48, -45, -97, 65, -34, 4,
    -40, 7, -84, -73, -92, 27, -87, 49, 30, -28, 66, 15, -55, 87, -49, -79,
    50, -61, -10, -45, -98, 23, 14, -44, 5, 62, -70, 55, -10, -12, -11, -18,
    -22, -76, 31, -34, -57, 55, 65, -95, 48, 74, -86, -15, -33, 49, -72, 44,
    17, 55, 34, 23, 45, 53, 45, -56, 74, 38, 94, -21, -37, -34, -51, 87, -93,
    8, -27, -91, 26, 23, -66, 7, 4, -56, -66, 35, 2, -85, 84, -56, -67, -69,
    -34, -72, -85, 74, -15, 13, -31, -45, -6, -83, -72, -30, -100, -59, 58,
    -38, 76, 91, 21, -51, -86, 20, -66, -47, -87, -29, 6, -76, -7, -7, -47,
    6, -42, -100, 63, -86, -78, -42, 71, 17, -90, 67, -57, -14, -76, -49, 86,
    -88, 16, 0, -41, -50, 47, -51, -37, -95, 19, 18, 63, 46, -73, -20, -69,
    -19, -50, 79, 42, 80, 75, 45, -36, -54, 93, -94, 97, -100, -85, -94, -62,
    -33, 17, 21, 82, 70, 14, -56, 88, -68, -32, -79, -10, -86, 90, 44, 16,
    -79, -9, -66, -85, 74, -61, -5, -81, -4, 81, 57, -28, -47, 80, -99, -81,
    -55, 68, 41, 32, -55, -6, 43, -45, 56,
  -86, 23, 37, -16, -99, 66, -45, 36, 93, 64, 95, -71, -84, -43, -21, 6, -85,
    -11, -29, -41, 17, -92, -94, -37, -80, -10, 79, -24, 93, -45, -88, 94,
    37, 28, 52, 68, 31, 18, 18, 18, 67, -27, 81, 87, 80, -32, 42, -39, 9,
    -14, 81, -6, -56, -91, -4, -11, -19, 84, 65, 65, 97, 91, 17, -53, -94,
    -11, 69, 19, -90, -17, 77, 77, -86, 87, -59, 62, -51, 90, -87, 45, 12,
    -72, -73, 41, -26, -80, 47, 80, 34, 3, -33, -54, -28, 5, 29, 36, -56, 7,
    -72, -18, 98, -48, -11, 49, -16, -85, 46, 65, 0, -66, 88, 39, 38, 98, 5,
    16, 85, 0, -97, 36, -45, -96, 28, -60, 9, 68, 42, 65, 13, 58, 77, -15,
    21, 28, 7, 68, 55, 17, -82, -86, -38, 49, 56, 47, -31, 9, -17, -46, 7,
    -11, 15, 86, -9, 60, -1, -73, 87, 76, 6, -70, -77, -85, -53, -48, -91,
    -72, 21, 76, -9, -92, 15, -98, -48, 54, -53, 59, -1, -98, -88, -30, 24,
    9, -25, -9, -66, 14, 53, -93, 94, 37, 74, -5, 65, -92, -91, 52, -72, -53,
    81, 86, -54, 69, -56, -43, 66, 44, 8, -15, -49, -67, -51, 81, 55, -80,
    14, 2, 17, -82, -5, 9, 52, -2, 63, -48, 13, 81, 61, -14, -66, -60, 10,
    69, 50, 6, 58, 8, 8, -25, -46, 68, -34, 73, -58, 30, -30, 35, -100, 13,
    2, -61, -74, 5, -94, 18, 23, 41, -9, 12, -41, -52, -11, -24, 82, -82, 6,
    -56, 33, 43, 59, 25, -21, -63, 20, 75, -70, 90, 35, 75, 18, -3, -40, -20,
    59, -26, -15, -51, 28, 68, -5, -94, 47, -55, -37, -98, -15, -16, -11,
    -68, 61, 31, 18, 27, -62, -66, -29, -51, -64, -60, 38, -5, -94, 13, 49,
    24, -21, 71, -84, -34, -63, -29, -50, 53, -9, 86, 83, -34, -53, -68, 7,
    3, 39, -7, 7,
  -46, -41, -53, -10, 19, 10, -82, -72, -92, 95, -27, 34, -28, -11, -45, 93,
    -72, 97, 61, -49, -88, 31, -37, -95, 56, -9, -21, -33, -8, -38, 8, 14,
    -85, 97, -94, -43, 86, -96, -97, 23, -42, -99, -21, 30, -66, -84, -25,
    52, -94, -95, -67, -90, 53, -58, 10, -94, -44, -1, 60, 64, 31, -15, -70,
    -93, 5, -4, -28, 0, -54, -44, -34, -80, 37, 55, 83, -10, -87, 94, 62,
    -32, 32, -31, -58, -32, -13, 77, 0, 28, 5, -16, -1, 79, 43, 16, 99, -99,
    45, 48, 78, 20, -93, 71, -60, -31, -42, -43, 36, 41, -10, -16, 3, 85,
    -27, -38, 27, -26, 5, -37, 10, -58, 82, -10, -72, 36, 95, -84, -63, -80,
    -6, -67, -78, 70, -46, -32, -48, 55, -96, 74, 74, 3, 43, -27, 94, -74,
    13, -38, 92, 88, 59, -98, 11, -76, 25, -22, 54, -57, 42, -38, -58, -97,
    -61, 47, -40, 26, -95, 32, 91, -1, 20, -75, -12, -41, -6, -35, 22, 42,
    -48, -16, 47, -63, 75, 56, 98, 27, -79, -1, 67, -90, 12, 82, 40, 55, -12,
    55, 51, -93, 89, -94, 17, 15, -63, 57, -83, -47, -57, -65, 37, 49, -15,
    7, -12, -79, 63, -63, -95, 11, 57, 69, -76, -13, -89, -5, -14, -53, -78,
    -93, -25, -80, 4, 79, 74, 75, 49, 24, 10, 4, 59, -35, 57, -92, -3, -76,
    4, -43, 63, -62, 28, -45, -6, -42, 46, 81, -99, 32, -59, 85, 12, 8, -91,
    -37, 69, 89, -72, -78, 73, -95, 11, -44, -23, 81, -78, -97, -32, 37, 60,
    -62, -54, -43, -93, 77, 1, 15, -15, -15, -87, -4, -92, 60, 40, 10, 8,
    -27, -8, -3, -57, 31, -23, 0, -50, -92, -23, 58, 31, 5, -58, 98, 69, -68,
    22, -54, 85, -17, 2, 20, -22, 18, -46, 84, 76, 35, 8, -30, -24, 74, 82,
    -38, -36, -79, 30, 6, 94, -33, 22,
  -86, 28, -62, -53, -31, 5, -84, 86, -49, -25, 82, 6, -69, -21, -36, 46,
    -50, 81, 5, -23, 16, 85, 85, 79, -99, 87, 29, 2, 61, 98, 58, -90, -21,
    -82, 30, -26, -24, 78, 52, -56, -69, 76, -14, -75, 56, -96, -74, -81,
    -16, -66, 73, -20, 89, -40, -10, 85, 46, -85, 81, 6, 19, 65, -12, -91,
    10, -42, 54, -73, -64, -28, -90, 48, 82, -12, 92, 66, -41, 64, -71, -49,
    43, -61, 37, 74, -13, -57, -100, 75, 65, 3, -9, -47, -56, -59, -22, 46,
    -54, -65, -17, 66, 85, 38, -31, 94, -37, -21, 2, 5, -99, 35, 22, -95,
    -79, -15, -73, -59, 61, -36, 69, -59, -10, -43, 15, -47, 3, -6, 9, 80,
    17, 85, 27, -56, 16, 8, -87, -65, -36, 78, 84, 75, -10, -18, 5, -27, -14,
    6, 19, -8, -62, -59, 10, -38, 15, 43, -67, -74, -42, 4, 73, 0, 33, 30,
    -25, -44, -89, -47, 14, -100, 96, 74, -93, 69, -33, -85, 95, -28, -48,
    -57, 28, 31, 81, 94, -17, -50, 80, -63, -12, -30, -31, -64, -67, 66, -3,
    30, 75, -27, 24, -94, -28, -82, 54, 9, 35, -83, 39, -91, -76, -6, -77,
    -48, -85, -67, 67, 89, 58, 51, -19, 67, 19, -27, -35, -31, -56, -91, 93,
    -32, 0, -68, 58, 37, -100, 34, 70, -20, 45, -85, 8, -55, -8, 32, 13, 79,
    32, -31, -75, 65, -6, -8, 12, 61, -75, -72, -2, -92, 3, 60, -94, -13, 93,
    -4, -34, -58, -25, -44, -86, -94, 12, 0, -35, 14, 77, 79, -36, 19, -84,
    -71, -18, 78, 0, 9, 62, 67, -9, 4, -33, -3, 23, -93, 30, -60, -82, 12,
    34, -1, 61, 34, -6, -100, -89, 6, 36, -77, 88, -57, -43, 16, -19, 86, 70,
    -49, -89, -76, -45, -46, -40, -51, 54, -10, -80, 20, -40, 26, 17, -34,
    -93, -13, 6, 50, 75, -39, -93, 20, 36,
  8, -51, -75, 29, 67, -55, -23, 30, -48, 31, -73, 93, 74, -62, 87, -94, 24,
    58, -100, 61, 65, 89, -7, -44, 18, 19, 62, 80, -84, 39, 25, -76, 41, -98,
    -69, 95, 26, -73, -94, -72, 72, 7, 34, 4, 84, -76, -23, -17, -93, 71,
    -31, 37, -75, 12, 60, -51, 17, 50, 58, 63, -48, 61, -84, 50, -86, 11,
    -26, 68, -2, -40, -93, -58, 75, -1, 27, 8, -65, -55, -51, 12, -92, -26,
    94, -56, 48, -56, 80, -79, 6, 39, 72, 54, 46, 9, -96, -57, -33, 5, -92,
    -1, 68, 16, -39, -16, 26, 22, -82, 27, -94, -24, -7, -44, 1, -72, -21,
    38, 22, 27, -40, 11, 85, 12, -58, 28, 86, -80, 55, -68, 64, -64, 68, -69,
    98, 69, -41, -66, -81, -85, 43, 5, -93, 3, 57, -69, 18, -60, -93, -95,
    -73, 51, 17, 3, -69, 42, -14, 76, -85, 55, -78, -57, -68, 9, 46, 19, 42,
    -95, -56, -51, -14, 19, -60, 2, -48, 99, -97, -74, 18, -89, -60, 2, -63,
    -18, 91, 97, 94, 16, 9, -7, 11, 35, 91, 26, -17, -87, 43, -5, -25, -85,
    46, -76, -4, -13, -59, -45, -11, 58, -88, -88, -27, 66, -71, 70, 32, 64,
    -10, 11, 32, -46, 74, -54, 68, -50, 83, 56, 28, -91, 38, 37, 72, 47, 64,
    25, -37, 66, -94, -44, 62, -79, -69, 0, 26, 1, 9, 38, 32, 62, -65, 72,
    -3, 53, -60, -33, -98, 24, -70, 24, 97, -83, -94, -33, -36, -65, 95, -11,
    -19, -98, -32, 46, 97, 18, 2, 37, 54, -48, 40, 52, 79, 35, 49, 46, -77,
    45, 10, 26, -64, -56, -1, 36, 23, -76, -41, 15, -85, 95, -48, -19, -19,
    -81, -82, 23, 13, 23, -3, -71, -44, -61, -75, -68, -18, 20, 70, -90, -95,
    -19, 44, -52, 32, 8, 21, 70, 36, 99, 12, -12, -28, 81, -20, 24, -47, 72,
    75, -12, 38,
  -24, -26, 0, -30, -92, 39, 79, -54, 86, 84, 75, -11, -44, 64, 40, -87, -75,
    10, -28, 96, 5, 14, -78, 17, -78, 59, -43, 99, 47, -82, -86, -60, -65,
    -63, -50, -58, 15, -27, 92, 86, -92, 47, -60, -70, -25, -91, -15, 21,
    -95, 58, 76, 42, 68, 80, 91, -52, -92, 21, 37, 41, 6, 87, 7, -74, 61, 4,
    -36, -5, -47, 6, 32, 66, -21, 93, 59, -72, 41, -1, -6, -18, -35, -50,
    -87, -93, -67, -30, 77, 56, -80, -78, -51, -4, -28, -66, -71, -37, 80,
    -18, 6, 21, 86, -84, -21, 63, -7, 10, 13, -65, 59, -96, 84, 12, -87, -41,
    68, 9, -30, -95, 60, 75, -70, -19, -89, -26, 22, 39, 63, 59, 69, -21, 56,
    87, 94, -11, -33, 62, 90, -83, 91, 13, 57, -28, -12, -47, 64, 29, 4, -25,
    -38, 93, -8, 25, 61, 85, -16, 98, -23, -11, -33, 18, -22, -59, 29, -97,
    -17, 48, 97, 91, -8, 8, 38, -98, 26, 32, 88, -58, 73, -4, 59, 88, -57,
    -10, 39, -65, -66, -82, 70, 1, -53, 1, -42, -4, -59, 37, -89, 38, 80,
    -90, 57, 64, -48, -83, -91, 87, 6, 54, 96, -60, -50, -69, 53, 73, 86,
    -29, -73, 77, 36, 59, -22, -13, 71, -67, 61, 38, 58, 6, -67, -26, 65,
    -83, -39, -3, 80, 86, -57, -33, 82, -93, 49, 76, -89, -40, -23, -32, -40,
    43, 6, -12, -55, 86, 34, -24, 31, 47, -92, 73, 68, 96, 41, -97, -17, 94,
    89, -94, 18, -58, 17, 29, 56, -19, -61, -80, 66, -63, -47, -37, 80, 69,
    -63, -9, -72, -30, 30, -65, 70, -47, 37, 33, -99, 68, -60, 41, -26, -8,
    -67, 11, 71, 80, 89, -95, 25, 36, 54, -84, 30, 12, -70, -82, -60, 98, 15,
    -57, 42, 95, -34, -52, 92, -7, -97, -84, 5, 46, -14, -75, -72, 66, -32,
    43, 89, 23, -29, 95, 51,
  23, 13, 78, 57, 46, 47, -14, 31, -49, -4, 1, 33, 97, 19, -12, 72, 99, 0,
    61, 88, 95, -79, 46, 40, 6, -27, 56, -19, -74, -45, -8, 89, -76, -39,
    -71, -50, -56, 67, -30, -3, -3, -82, 73, 49, 97, -56, 58, -39, -5, 87,
    -35, -73, 79, -82, -96, 35, 26, 25, 99, -3, -82, 31, -98, 30, -86, 36,
    -51, 59, -80, 24, 18, 64, 78, -52, 45, 17, 56, -57, -77, -5, -100, 27,
    21, -35, -58, 24, -19, 10, -99, 30, 27, -48, 90, -33, -84, 48, -8, -64,
    51, -55, -7, -70, 0, -96, 1, 30, -76, 65, 28, 9, -2, 17, 15, -3, -84, 15,
    -68, 39, -27, 66, 50, -84, 31, -58, -3, 10, 29, 28, 39, 26, -34, -94,
    -99, -93, 11, 52, -100, 28, -1, 5, -53, 73, -21, -49, -12, 17, -33, -99,
    -52, -76, -90, 78, 5, -53, 75, 63, -59, -6, -84, -33, -81, 16, 93, -61,
    -10, 82, 27, 7, 68, -69, 72, -71, 16, 61, 27, -54, 96, 91, 5, 99, -78,
    -28, 72, -61, 47, 7, -75, 88, -11, -43, 72, -78, -82, -62, 40, -92, -37,
    90, -39, -91, 72, -86, 26, -78, -71, 40, -1, 96, 31, -73, 58, -38, -25,
    48, 46, 36, -48, -93, 63, 10, -25, -28, -67, 84, 79, 52, -72, 54, -81,
    -97, -59, 87, -61, 8, -93, 55, 52, -70, -91, 83, 6, 69, 14, -87, 35, -92,
    -50, -57, 1, -71, 87, 0, 78, -92, 1, -1, -3, 91, 46, 30, 36, -8, 37, 93,
    82, 53, 20, -68, 41, 28, -44, 7, 61, 41, -58, -74, 96, 84, 83, 53, -95,
    -38, 91, -88, 46, 34, -57, -11, 89, -60, 44, -61, 29, 20, 78, -79, 52,
    32, 95, 14, 57, 81, -38, -86, -55, 24, 75, -88, 23, -69, -80, 10, -23, 7,
    -38, 8, 89, -4, -79, 4, 57, 29, 1, 76, -27, -31, 2, 55, 59, -99, 90, -86, 21,
  71, -31, -16, -23, 42, -98, 70, -75, 7, 7, -50, -21, -78, 40, -1, -56, 70,
    43, -33, 67, -10, -22, -17, -62, 72, 82, -24, 41, 45, -41, 82, 51, -19,
    -25, -68, -73, 78, -2, 71, -34, 31, -27, -7, -27, 55, -46, 64, -91, -9,
    1, 36, 25, 8, 13, 13, -73, -7, 29, 21, -82, 73, -69, 48, -79, -79, -54,
    -17, -22, -32, 38, 70, 87, 80, 91, -34, -43, -17, -56, -58, -27, 39, 16,
    -58, -97, 53, 10, 6, 86, -41, 40, 79, -95, 82, 27, -37, 9, -47, 66, 72,
    7, -14, -34, 31, 66, -55, 68, -10, 53, -78, 16, -11, 48, -32, -49, -46,
    97, 35, 56, -52, 5, 51, 74, -34, -66, 31, -18, 79, 52, -8, -26, 29, -90,
    -63, -53, 54, 51, 63, -100, -74, -23, -82, -1, -3, -71, -17, -86, 73, 34,
    -7, 79, -61, -95, -17, -61, 70, -87, -15, -38, 4, 64, -19, 0, 89, 55, 10,
    -23, 64, 69, -58, -5, -14, 84, -82, 2, 35, 41, 13, 19, 3, -20, -4, 50,
    65, -32, -79, 88, -60, -27, -21, -49, -95, 31, 4, -53, 80, 21, 48, -69,
    29, 86, -94, 36, 78, -61, -56, 51, 69, 82, 49, 75, -38, -26, -44, -35,
    65, 5, -48, 20, 24, 9, 63, -99, -62, -7, 92, 51, 70, -65, 64, 93, -14,
    -29, -9, -87, -12, -34, -77, 70, 54, 51, 26, -20, -60, -46, 30, -2, -98,
    -88, -45, -85, 70, -50, -47, -9, 48, 9, 98, 64, -26, 96, 75, 57, 57, -6,
    -89, -65, -38, -5, 17, -46, 68, 35, 99, -71, -69, 70, 92, -78, 58, -5,
    89, -27, -11, 30, 28, 35, 77, -87, 97, 21, 52, -75, 58, 33, 54, -81, -44,
    38, 9, 24, 20, -16, -72, -47, -87, -50, -87, -3, 42, 7, 11, 62, -67, 2,
    83, -92, 57, -25, -31, 6, 22, -98, -47, -74, 11, 94, 72, -70, 11, 50,
    -30, -46, 76,
  -42, 4, 21, 94, -13, 30, 21, 7, -76, -70, -35, 73, -55, -65, 90, -84, 65,
    54, -99, -78, -91, -56, 63, -78, -97, -56, 90, 43, -39, -51, -98, -56,
    32, -84, 99, 95, -100, -9, -5, -24, 10, 97, -14, 63, 91, -73, -100, 59,
    53, 6, 51, 71, 26, 52, 47, -96, 32, -30, -98, 92, -67, 45, -38, 29, -67,
    79, -59, -61, 66, -27, 12, -32, -71, 18, -9, -61, -34, -1, 18, -23, -22,
    7, -57, 12, -69, -14, -45, 56, -80, -77, -14, 37, -68, 40, -67, -45, -23,
    23, 99, -31, -78, 47, -46, 21, 88, 41, -14, -57, 54, 47, -10, 49, 44,
    -65, -40, 7, -61, -64, 35, 65, 88, 78, -98, 11, -85, -57, 60, -24, 92,
    -8, -96, -84, -92, -8, -56, -66, -9, 36, -34, 96, 91, -33, 53, 23, -40,
    -42, -19, -97, -77, 62, 89, 1, 93, -16, 40, 92, -34, 22, -33, 26, -2, 4,
    -23, -5, 68, -33, 28, -96, -11, -66, 70, -35, -91, 19, -32, 93, -78, 80,
    -5, 51, -75, 82, -89, -70, -25, -5, -14, 51, -50, 10, 19, -3, 18, 4, -43,
    69, -56, 61, -48, -89, 71, 48, -47, -5, -84, 27, 4, -95, 3, -82, 32, -75,
    38, -36, 89, -12, -39, -36, -8, 1, 41, 53, -63, 7, -65, 86, -1, 16, 16,
    13, 4, -76, 8, 60, -14, 56, -88, 81, -63, 21, 67, 83, -10, -19, -18, 4,
    20, -17, 80, -53, 44, -54, -2, -42, -28, -8, -42, 85, 89, 93, -3, 33,
    -82, 79, 32, 22, 11, 75, 14, 29, 1, 45, 81, -54, 23, 72, -91, -73, 70,
    -79, -22, -45, 48, -61, -45, -31, -86, -41, -100, 55, -54, 3, -4, 86, 7,
    14, -28, -18, 69, -34, 90, -89, -33, -1, 10, 86, 99, 55, 98, 95, -56, 61,
    -80, 20, 55, 13, -79, 31, 85, 67, -47, -36, 85, -89, 30, 35, -53, 57, 24,
    6, 14, 67, -16,
  -60, -55, -83, -48, 20, -65, 68, -16, 82, -4, -72, -62, -75, -85, 11, -69,
    12, 76, 45, 51, -23, -52, 48, -83, -1, 9, -16, -34, 68, -50, 3, 87, -12,
    -71, -73, -88, 4, -90, 77, -89, 82, 96, 28, -41, -33, 47, -32, 27, 39,
    -28, 21, -97, -51, -11, -33, 79, -49, -94, -87, -19, -66, -62, 78, 73,
    79, -45, 88, -48, 90, -14, 46, -1, -98, -15, -22, -4, 91, 26, -56, 78,
    -96, 78, 71, -83, -17, 68, -35, -97, -98, 24, -28, -88, -51, 82, 35, 47,
    38, -66, 53, -82, -34, -5, -53, 99, -28, 28, -63, -65, 7, -22, -60, -60,
    -28, -98, 45, -46, -27, 38, -53, -2, 19, -96, -31, -27, -61, -58, -70,
    -42, 91, 14, 40, -96, -54, 18, 63, -89, 91, -18, -27, -8, -80, -77, 46,
    -13, -69, -93, -87, 93, -9, -94, -28, 33, 21, -53, 26, 57, -7, 90, -11,
    25, 54, 17, 69, 81, 56, -82, 90, -84, -4, -91, -84, -83, -96, 84, -45,
    94, -86, 26, 8, -88, -98, 83, -6, 28, 93, 99, -64, 19, -18, -97, 48, 9,
    86, -71, -2, 77, 38, -92, -91, -63, 23, 1, -13, -70, 2, 25, 84, 39, 13,
    47, -74, 49, 5, -23, 50, 1, -66, 29, -48, -23, -86, 87, -60, -19, -32,
    -10, -81, 68, -94, -68, 40, -100, -62, 34, 45, 6, -73, 28, -86, -22, -30,
    -48, -8, -24, 60, 98, -69, 1, -29, 99, 13, -59, -93, -24, -41, -80, -47,
    50, 78, -36, 92, 88, -85, -58, 40, -23, -48, -66, -82, -86, 25, 58, -47,
    58, 16, 1, 98, 27, 42, 65, -77, -73, -49, 92, 94, 31, -4, 49, -83, -31,
    16, 3, 45, 15, -58, -52, 3, 79, 49, 89, 58, 46, 56, 66, 87, -85, -35,
    -21, 30, -97, -37, -96, 99, -6, 73, 80, 74, -24, -27, 13, -27, -99, -90,
    92, 22, -47, -39, 0, -50, -29, 0, -59, 69,
  3, 86, -9, -67, 36, 69, 63, -31, 83, -100, -28, -3, -3, -29, 16, -92, -85,
    61, -45, 64, 87, -49, -93, 4, -20, 4, -16, 96, 92, -9, -34, 11, -69, 75,
    -20, 8, 55, 27, 15, 22, 53, -19, 25, -82, 92, 88, 18, 86, -58, 74, 99,
    -39, 22, -1, 12, 37, 66, -61, 16, 58, -95, 91, 53, -39, -97, 20, -4, -26,
    31, 53, -30, -87, 79, 19, 95, -75, 85, -61, -89, 85, 79, 10, 21, -87,
    -91, 48, -28, 43, -32, -68, 58, -48, 61, 24, -51, 79, 70, 74, -91, 82,
    -8, 21, -56, -14, 5, -22, -94, 89, 92, -11, 63, -1, 48, 27, 63, -58, -27,
    1, -66, -77, -60, 58, -52, 49, -79, 98, -22, -29, -91, 68, 72, 30, 63,
    -44, 34, 48, -70, -50, -35, 73, 16, 6, 16, -61, 66, -41, 44, -78, 92,
    -95, 53, 96, -18, -22, 15, -62, 44, 89, 57, 5, 6, -60, -50, -16, -21, 94,
    -40, -100, -94, -80, -74, -21, -100, -100, 97, -76, 18, 57, 50, 24, -52,
    89, 68, -43, 93, 40, -73, 1, -3, -35, 50, -32, 85, -4, -45, 50, -37, 42,
    80, 80, 78, -56, -89, 77, -58, 15, 85, 63, 98, -84, 80, 68, -61, 17, -17,
    -7, -39, -8, -20, 12, 31, 41, 4, -88, -16, 73, 67, -25, 38, -5, -38, -69,
    52, -64, 29, -85, 27, 93, -67, 38, 69, -5, -52, -81, 21, 37, 98, -53,
    -27, 92, 7, 52, 9, 56, 42, 71, 76, 52, 82, 57, 2, -24, -55, -3, -85, 40,
    64, 67, -28, -70, -63, 9, -49, 96, -62, 61, -83, -71, -31, -46, 82, 47,
    -16, 5, -67, 12, -85, 0, 57, 0, 72, -52, -9, -94, 49, -83, -55, -53, -65,
    38, 16, 77, -21, 27, 65, -89, -79, -76, -60, -20, 27, 9, 63, 97, -28, 3,
    95, 0, -38, -71, 80, 32, -45, 79, 20, 72, 77, -86, -32, -15, -57, 24, 79,
  -100, -18, 9, -54, 8, 40, 33, -85, -54, -8, 19, 92, -40, -36, 29, -3, 12,
    54, -65, 62, 1, 6, -37, -63, 54, 10, -67, 48, -15, -80, -34, 90, 10, -21,
    98, 32, 34, 94, 15, 87, -36, -1, 36, -46, 78, -25, -2, -9, 79, -18, -36,
    7, 88, 49, -42, -52, 49, 45, 63, 9, -10, 99, -86, 97, 61, -10, 34, -87,
    -50, -9, -66, 51, -21, -63, 53, -57, 79, 6, 54, -91, -3, -26, -95, -47,
    -71, 14, 29, -56, 28, -21, 35, 40, -27, 48, -13, 69, -95, -51, -33, -42,
    1, -38, 4, 14, 29, -27, 81, -11, -18, -21, 91, -59, 94, -97, -64, -5,
    -60, -87, -65, -29, 30, -68, -91, 19, -19, 35, 25, 45, 14, 59, 76, 56,
    -78, 86, -91, 27, -38, -72, 93, 56, 47, 40, -34, 45, 6, -84, -21, -71,
    -9, 97, 24, -67, 67, -85, 66, -88, 32, -66, 8, 42, -50, -79, 40, -42,
    -67, -58, -29, 32, -8, 90, 30, -28, 68, -97, 17, -50, 2, -26, 61, 92,
    -25, -19, -62, 11, -3, -20, 13, -16, 46, 17, -84, 14, 15, -25, -97, -61,
    4, 38, 73, -53, -40, 44, -41, 62, -54, 91, -38, -83, 26, -77, 68, 58,
    -69, 43, -59, -89, -78, 11, 81, 59, -3, -49, 3, -9, 92, -91, -63, -42,
    83, 38, 70, 62, 45, 20, 46, -27, 93, -76, -14, 70, 2, 72, 42, 15, 6, -92,
    7, -28, -15, -52, -91, 13, 81, 13, -24, 66, -82, -13, 26, -54, 38, -57,
    -8, 75, 10, 37, -36, -83, 93, -92, 91, 15, 36, -90, 92, -82, 9, -88, 55,
    23, -92, -9, -2, 11, 61, 49, -96, -55, 11, 89, -49, 79, 3, 15, 13, 15,
    96, 10, -33, -15, 91, 76, 99, -45, -31, 25, 5, -85, -31, -18, 87, -87,
    -27, -63, -4, -17, 25, 22, -37, -85, -5, 37, -54, -3, 5, 0, -45, -85,
    -51, 57, -85, 57, -45,
  -96, 11, 49, 15, -52, 71, 58, 19, -77, 50, 46, 59, 42, 12, 68, 44, 69, 4,
    31, -57, 33, -14, -73, 99, 14, 36, 35, 93, -45, 38, 12, 25, 90, 63, 2,
    44, 82, -87, 98, -71, 85, 7, -39, 90, 53, -44, -95, 89, 78, -95, -51, 60,
    68, -28, 23, -62, -77, -28, -16, -26, 63, -54, 30, -20, -2, -84, -87, 66,
    27, 94, 80, 88, 21, -59, 82, -51, -19, 2, -87, -51, -92, -42, 55, -21,
    93, -49, -22, 43, 20, -94, 47, 31, -91, -58, 57, -77, 18, 91, 79, 19, 84,
    -14, 86, 45, -71, -14, -45, -55, 39, 18, 19, 24, 45, -7, -23, -97, -19,
    98, -3, -65, -18, 89, -51, -81, 9, -71, -7, 19, 2, 56, -53, 90, -50, -52,
    31, 23, 19, 8, 6, 98, -38, 93, -17, -36, -15, -46, 29, 87, 36, -11, -46,
    -30, -18, 43, -33, 46, 40, 65, 47, 97, 17, -77, 14, 44, -95, -3, -20,
    -15, 11, 42, 25, -13, -12, 91, 42, 59, 1, -92, -4, 31, 53, -33, 85, 8,
    -92, -2, -52, -41, 98, 69, 62, 95, -61, 30, -71, 4, -76, -37, 89, -54,
    -78, 85, 96, -66, -72, -2, 3, -42, -16, 46, -2, 52, 33, -6, 58, -59, -89,
    -54, 39, -93, 42, 37, -91, 49, -19, -23, 65, -11, -6, 78, 2, -10, -68,
    53, -91, 13, 10, 93, -45, -63, 51, 73, -81, -11, 28, 57, -7, -52, 49,
    -39, 58, -16, 89, 31, -45, -69, -92, -25, 63, 61, 26, 15, 41, -80, -51,
    69, 34, -47, 62, 15, -23, -37, -89, 49, -24, -92, 83, 55, 61, 20, -40, 7,
    -78, -86, -53, 3, -50, -87, 21, -28, -79, -54, 68, 6, 37, -30, 81, -43,
    75, 99, 69, 34, -84, 87, 72, -79, -40, 69, -52, -77, 51, -74, -55, 42,
    -33, 27, -68, -53, -94, 57, 98, -78, -61, -56, 54, -5, -47, 93, 39, 78,
    47, -27, 40,
  34, 60, -72, -69, 41, -34, -35, 77, 84, 39, 16, 42, 88, -6, -94, -65, 74,
    88, -59, 64, 72, 56, 30, 1, 33, 99, -44, 37, -62, 53, 2, 93, -58, 82, 9,
    -21, -32, 76, -82, -67, 69, -75, -64, 91, -74, -41, -99, 0, -65, -25,
    -16, 65, 89, 99, -65, -28, 8, -16, 80, -92, 98, 69, 71, 78, -62, 98, 32,
    -41, 99, -20, -7, 50, 44, -19, -63, -48, 24, -10, 81, -71, -99, 19, -42,
    -94, 90, 69, -57, 7, 17, 91, -7, 88, 21, 20, 84, 53, -3, -34, 51, -4, 68,
    -37, 43, 81, 40, -4, 6, -83, -12, -57, 25, -17, 20, 90, 14, -78, 78, -87,
    12, -88, -16, 37, 30, 54, -1, -29, -31, 53, -17, 47, -81, 82, -59, -17,
    -85, -76, 30, -72, 36, -92, -94, -35, -3, -70, -39, -49, -50, 31, 4, -87,
    -53, 73, 98, -4, -91, 99, -13, 12, 22, -53, 35, 84, 57, 32, -62, -51, 72,
    -30, 91, -59, 4, -46, -97, 7, 19, -98, -78, -82, 81, -38, -76, 51, 86,
    -63, -96, 96, -55, -32, -74, 47, 68, 91, 43, -96, -12, -59, -64, -67, 86,
    78, -35, 93, -4, 18, 78, 80, 81, 30, 48, -3, -33, -92, 52, -41, -38, 40,
    4, -2, -27, -42, -95, -83, 30, -55, 50, 9, -74, 77, -45, 95, -62, 68, -6,
    46, -93, 64, -20, -96, 39, 61, -50, 38, -37, 48, 42, -83, 30, 48, 65, 73,
    -36, 48, -67, 41, 69, -59, -75, 77, -95, -41, 66, 57, -7, 28, 97, 62,
    -27, 48, -11, -87, -52, -60, 98, 8, 91, -49, 93, -77, 89, -55, 62, 9,
    -16, 47, -93, -10, 27, 49, -28, 99, -97, -93, 16, 43, 54, -80, -51, 41,
    9, 41, -56, 71, 8, -28, 49, 0, 18, -93, -46, 53, -90, -22, -18, -61, 57,
    9, -8, 80, 75, 37, 36, 24, -93, 57, -15, 63, -54, 4, -85, -80, 5, 12, -19,
  -5, 3, -16, 99, -11, 24, 30, 87, -84, -31, -39, 26, 4, -74, 16, 20, -65,
    -43, -27, 94, 45, 26, 54, -16, 58, -43, 92, -69, -19, 99, -35, -95, 98,
    6, 20, 74, 88, 2, 8, 55, 13, 17, 34, -95, -59, 75, 52, 92, 90, 6, 88, 94,
    35, -49, -85, -17, -9, -27, -47, 44, -41, -17, -83, 19, -14, -87, -21,
    -5, -37, -18, 90, 88, 70, -68, 33, 28, -84, 76, 4, 98, -49, 29, -11, -3,
    -76, 72, -4, 37, -40, 43, -23, 7, -7, 68, -7, 24, 1, 98, 22, -80, 7, -92,
    38, 22, 77, -42, -40, -53, 94, 50, -52, -82, -49, -89, -77, -67, -92, 28,
    11, -93, 54, -90, 80, 32, -63, -66, -47, -2, -83, -44, -39, 71, 75, 76,
    3, -47, 24, 6, -61, 42, -21, 75, 47, -87, -27, -33, 99, 94, -44, 76, -44,
    13, 29, 69, 2, 74, 44, -63, -97, -64, -69, 48, -69, 74, -44, 74, -65,
    -85, 26, -75, 27, -52, -6, 33, 82, 81, 42, -96, 36, -49, -93, 73, -33,
    69, 17, -77, 48, 10, -32, -55, 85, 50, 7, -4, 89, -90, 40, -40, -78, 15,
    -100, -54, -6, -33, -95, 8, 97, 53, 66, 60, 33, 13, -90, 20, 1, -22, 18,
    64, -17, -33, 38, -38, 5, 30, 51, 76, 91, -71, -59, -19, -25, 34, -35,
    -99, -83, -60, 85, -59, -87, 19, -57, 78, -52, -1, -57, -63, -89, -99,
    -7, 78, -47, 54, 54, 30, 87, -45, -12, 6, 55, 78, 94, 69, -52, -43, -66,
    -9, 72, -17, -54, 70, -11, 41, -52, -53, -3, -58, 95, 86, -93, -33, 30,
    -97, 28, 70, -35, -2, -24, -39, -79, 18, -77, 15, 93, -36, -87, 90, 85,
    85, -53, -80, 84, 99, 43, -31, -6, -88, 59, 86, 68, 45, -20, -51, 3, -28,
    30, -26, 50, -31, 91, 25, 20, 86, 35, 74, 88, 8, -37, 58, -34, -93, -82,
    -95, -68,
  36, -59, 52, 43, 25, 52, 0, -21, -6, -6, -34, 56, -53, 39, -56, 54, -85,
    -84, -54, 10, -75, 54, 74, -39, 73, -72, -20, -59, -80, 19, 16, -69, 1,
    -83, 91, -98, -57, -81, 40, 41, -89, -27, 16, -22, 24, -17, 62, 27, -40,
    82, -11, 69, 35, -8, -97, 14, 30, 16, -79, -71, 99, 42, 36, 22, 6, 52,
    -6, 81, -15, -96, 21, 46, -17, 95, -19, -48, -95, 3, -57, 4, 65, 16, -2,
    46, -99, -49, -40, 45, 63, -83, 53, 85, 18, -47, 37, 39, -11, 77, -79,
    -40, 21, 35, 56, 11, 34, -11, 71, -54, -69, -84, 75, 92, -58, -4, 93,
    -51, -100, -12, -5, -21, 24, 47, -70, 15, -72, -81, -20, -79, -66, 56,
    -54, 92, 39, 66, -2, 47, -75, 56, 96, 29, 14, 73, -49, 57, -63, 32, 69,
    65, -48, -3, 50, 88, 33, -61, -24, -81, -70, 45, 60, 77, 40, -73, -2, 96,
    4, 92, 14, -51, 35, 41, 86, 24, 78, -59, 51, -4, -55, -59, 62, 43, 44,
    52, 3, -17, -4, -88, 41, -87, -28, 85, -10, 10, -49, -66, 45, 80, 47,
    -99, -40, 17, 42, 66, -93, 57, 25, 92, -61, -34, -32, -60, -76, 33, 43,
    -61, 14, 99, 77, 14, -86, -98, 41, 72, -19, 30, 18, -48, 10, 27, 98, 8,
    -3, 3, 59, -36, -94, 51, -2, 58, -6, 43, 55, -6, 65, 21, -54, 11, 82, 30,
    -22, -75, -88, 55, -25, -60, 59, 99, 22, -25, 25, 73, -80, -21, 5, 96,
    -1, 98, -86, 50, -10, -8, -9, -25, -39, 68, -41, 25, -16, -93, 13, -92,
    -99, 5, -18, -86, -89, -25, 30, 4, 72, -62, -42, -20, 82, -79, -43, -5,
    -100, 49, 54, 10, -27, -92, -83, -60, -4, 17, -14, -73, -3, -81, -85, 82,
    -42, -76, -96, -100, 6, 74, 59, 33, 24, -36, 19, 88, 87, 84, 42, -13,
    -57, 87, 3, 46, -93,
  -66, -94, -56, -53, 31, 2, 63, -32, 94, 76, 90, 32, -72, -91, -7, -39, -8,
    76, 24, 33, 33, 1, -4, 33, 61, -2, 85, -56, 49, 78, -58, -32, -37, 39,
    -32, 58, -93, -82, 72, 45, 11, 35, -86, -28, -29, 54, -83, -95, 33, -100,
    -71, -40, -72, -36, 65, 78, -46, -4, -15, -89, 27, 14, 33, 20, -26, 43,
    31, 32, 8, -27, 75, 68, -1, 79, 28, 90, 2, 88, 52, 15, 12, 67, -80, -29,
    25, 9, -33, -70, -99, -4, 20, 54, 14, 5, -58, -6, 51, 40, -1, -43, -11,
    71, 22, -31, 12, -83, 66, -34, -26, -35, -75, 15, 24, 19, -3, 3, -3, 58,
    42, -10, -57, 82, 60, -46, 81, -44, -36, -21, 57, -20, 82, -38, 15, 87,
    83, 53, -7, -16, 75, -60, 47, 91, 94, -57, 87, -36, -96, -89, 10, 60,
    -36, 86, 85, 1, -35, 63, -27, 93, 81, 91, -30, 92, 75, 64, -29, 57, -74,
    16, -25, 86, -82, 56, 26, -97, -9, -87, -33, 0, -2, 0, -95, 94, -17,
    -100, 56, -18, -47, -96, 91, -69, -60, 66, 42, -76, 95, 80, -3, 24, -61,
    -9, -19, -42, 29, 52, -1, -10, -55, 4, -9, 29, -50, -21, 26, -85, -48,
    71, 51, -30, -62, 65, -41, 89, 20, -27, -92, -73, 90, -67, 20, -46, 42,
    31, -95, -43, -12, 21, -35, -34, 20, -5, -35, 33, -7, -42, 56, 19, -38,
    19, 68, -54, -46, 30, -52, 68, 7, -17, -46, 11, 40, -93, 4, -18, -73,
    -48, 71, -2, 86, 1, 38, -49, -86, 22, 67, -91, 45, 79, 49, -95, -33, 0,
    -1, -80, -71, -89, 12, -3, -24, -54, 92, -58, 55, -95, 85, 85, 86, -94,
    -92, 4, 73, -95, -85, 72, 56, 70, -77, 93, 93, -58, 66, 47, -45, 77, -83,
    -50, -73, -59, -68, -84, 4, -62, -81, 71, -85, 52, 3, 54, -56, 34, 25,
    75, -57, 79, -82,
  4, -70, 44, 71, -22, -56, -9, 87, 39, -66, -25, -29, -41, 79, -18, 29, 41,
    80, -78, 9, -90, 37, -66, 55, 40, -45, 65, 84, 52, -90, -95, -91, 79,
    -51, 23, 63, 57, 96, -90, -56, 12, 22, -65, 4, -89, -30, -72, -80, 77,
    54, 7, 67, 30, -31, 69, 59, -33, 29, -14, -8, -93, -67, -95, -59, -51,
    35, 76, -15, -60, -24, 2, 47, -40, -1, 71, -34, 99, 48, -79, 37, 19, -64,
    -58, -8, -39, -21, -78, 85, 56, 1, -91, -29, -77, 3, -11, -97, -58, -29,
    78, 36, 72, -3, -35, -33, 32, 79, -36, 45, 63, 16, 45, -91, 67, 87, -48,
    -68, 22, 27, -18, 67, -27, 67, -24, 13, 56, 29, 84, 14, -67, 3, 36, 6,
    -44, 93, 47, 6, 58, -94, -35, -38, 35, -43, 67, 1, -18, 55, -26, 62, -43,
    -11, -13, 91, -26, 2, 77, -3, -54, -44, -47, 28, -23, 13, -62, 8, 75,
    -54, 17, 74, 24, 65, 9, -81, 85, -6, -99, 64, -88, -7, -5, 78, 43, -63,
    -76, 36, 91, 62, 24, 98, 1, 30, -64, 65, 64, 19, -88, 63, -61, 12, -94,
    46, -49, 89, 18, 98, -94, -56, 46, 8, -47, 44, -97, -96, 69, 88, -65, -3,
    -90, -63, -22, 83, 14, -96, 11, 68, -33, 62, 44, -61, 82, -99, -47, 5,
    -95, 82, 68, 9, -59, -71, -30, 70, -16, 71, 76, -100, 52, 62, -34, -8,
    31, 1, -75, -100, -60, 89, 70, -30, -29, 14, 2, -38, -43, 73, -6, 73, 13,
    -72, 35, -92, 56, 32, -33, -85, -14, -65, -11, -7, 30, 32, 80, -19, 14,
    -75, 40, 17, -38, 7, -78, 26, 61, 89, 74, 43, 80, 71, -50, -10, 75, -94,
    49, -12, -26, 64, -33, 11, 8, -38, -28, 41, -88, -35, 51, -57, 61, 61,
    47, -87, -2, -36, 12, 65, -30, -17, 23, -49, -35, 5, -92, 2, 23, 72, 47,
    82, 4,
  -8, -90, -16, 71, -74, -12, 98, -89, 69, 58, -98, 52, -13, -17, -53, 63,
    98, 82, -90, -17, 71, -85, -92, 91, 46, 40, -22, 3, -63, 66, -33, 18, 29,
    11, -71, 92, -69, -93, 72, 97, -72, -71, 99, 25, -38, -88, -75, 46, 63,
    87, -67, -94, -92, 94, 29, 76, 2, 60, -11, 60, -1, 31, -79, -86, -26, 66,
    96, 11, 41, 38, 11, 97, 29, 85, -24, 95, -52, 29, 96, 3, -89, 88, -86,
    -79, 58, 88, 86, 63, 37, -33, 46, -87, -97, 49, 10, -62, 48, -76, 64,
    -80, 78, 63, -38, -22, -3, 28, -75, -2, -72, 71, -90, -22, 23, 64, -100,
    -53, -8, -34, 41, 21, 22, 17, -62, -84, 75, 56, 36, 23, -22, 73, 94, 62,
    -3, -10, -89, 93, 71, -44, 47, 84, 35, 48, -40, -88, -37, -48, 3, 61,
    -55, 86, -28, -63, 58, -97, 27, 61, -81, 94, 18, -6, 94, -49, -45, -82,
    -75, 48, -77, -8, -7, 87, -43, 62, -23, 47, 72, 16, -96, -58, 71, 13, 79,
    39, 37, 82, 72, -50, 24, 83, -42, 53, 81, -95, 95, -33, 72, 58, 42, 98,
    -22, 1, 45, 65, 67, 26, 64, -10, 92, 77, -49, 11, 98, 57, 53, 88, 92,
    -13, 12, 55, 8, -55, -8, 31, 44, -77, 38, 1, 91, -39, -37, 7, 19, 16, 25,
    -52, 77, -2, -81, -83, 94, 79, 20, 46, -3, 61, 6, -47, 89, -69, -71, 48,
    -64, -35, 40, -93, -99, -85, 36, -84, -96, 79, 15, 65, 9, -77, 52, 30,
    99, 29, -20, -38, -48, -9, 73, -95, 14, -24, 56, 53, 29, -15, -5, 1, 82,
    53, -21, -65, 5, -39, -50, 66, 31, -68, -45, 43, -66, -37, 83, 92, -41,
    83, -2, 54, -47, -5, -36, 65, -15, 96, -82, -27, -62, -82, 74, 92, 88,
    89, 39, 17, 38, -92, 99, -80, 18, -37, 48, -73, 9, -76, 85, 20, -64, -74, 5,
  -61, -2, 25, -96, 61, 34, -83, 4, 95, -7, -78, -60, -21, -58, -1, -4, -100,
    -85, -81, 14, -42, 58, -80, -61, 21, -15, 66, 71, 2, 8, -52, 99, 16, 6,
    -1, -90, -50, 28, 83, -1, -38, 67, -58, -100, 73, -95, -25, -72, 16, -44,
    73, 72, -19, -80, 47, 45, 1, -78, -10, 72, -85, 60, -15, -55, 41, -100,
    70, 81, 97, -71, -41, -97, 84, -64, -89, -39, 90, -59, -29, 17, 55, -87,
    69, -65, 95, -99, 15, -23, -56, -14, 17, -26, -18, 16, -22, -11, -40, 84,
    -73, -50, -44, 81, -66, 63, 88, 14, -36, -51, -14, 42, 53, 74, -98, 29,
    -29, 77, -7, 35, 11, 1, -3, -2, 37, 48, 37, -99, -6, 28, 27, -94, 59,
    -40, -50, 61, -45, -43, 46, -56, -63, 58, 10, -87, -19, -25, -51, 55,
    -28, 50, 26, 38, 23, -83, 20, -60, 65, -47, 85, -24, 89, -93, -49, -27,
    61, 80, -9, 29, -36, 89, -68, -90, 46, 95, -66, 70, -93, 44, -99, -94,
    19, -59, 13, -50, -16, 22, -43, -31, 81, -43, 7, 25, 24, 11, -81, -91,
    -89, -96, 72, -76, 9, -69, -98, -51, -30, -16, 61, 72, 5, 50, -84, -30,
    97, -73, 35, 27, -54, -14, 65, 20, -39, -47, 5, -35, -21, -60, -80, 35,
    99, -54, 70, -40, 60, -13, -78, -75, -19, 80, 73, -4, 22, 7, -98, -81, 9,
    44, -91, -67, -11, 86, 1, 6, -7, 23, 91, -82, -68, 98, 1, 21, 6, 40, -79,
    -11, -45, -94, 21, -72, 71, 0, -70, 9, -77, -8, -93, -42, -74, -95, -25,
    -51, -79, -79, -94, -60, 13, 83, -86, -4, -93, -4, 64, -1, 23, -63, -49,
    46, 94, -77, 50, -18, 36, 90, 63, -48, -8, -86, -96, -13, -84, 48, 93,
    66, -1, 1, 45, 5, 73, -57, -12, -43, -43, 24, 11, -34, -65, -1, -46, -92,
    -6, 81, -90, 55, 30, 42, -32,
  29, -91, 82, 85, 50, -68, 35, 57, -2, 55, -39, -32, 18, 89, 69, -78, 7, 20,
    -77, -5, 42, 28, -17, -90, -53, 16, 43, -7, 53, 81, 10, -66, -36, 25,
    -79, 9, -13, 85, 60, -83, 59, -37, -56, -54, -79, -3, 15, -8, 86, -36, 3,
    2, 14, -88, -41, -31, -12, -69, -32, 73, -19, -61, 36, -77, -71, 25, 10,
    17, 65, 21, 48, 6, 41, -78, -12, 35, -36, 13, -26, -29, 7, -97, 85, -70,
    -90, 22, -79, -3, -7, -80, -21, -24, -77, -26, 97, 62, -83, 5, -100, 98,
    -63, -41, 5, 96, -73, 96, -96, 87, -14, 86, -58, -14, -95, 54, -63, -3,
    -88, -78, 35, 7, 13, 70, -74, 82, 82, 6, 11, 55, 23, -13, 81, -39, -8,
    -42, -9, -87, -67, -39, 44, 29, -66, 57, 12, 47, 10, 9, -35, 92, -15,
    -24, -53, -99, 5, 87, 86, -40, 91, -70, -26, 10, -83, 12, -87, 29, 36,
    36, -89, 55, -77, -2, -17, 95, 63, -60, -21, 93, -43, -8, -73, -51, 66,
    -80, -5, -30, -2, -47, -80, 35, 98, -40, -96, -54, 88, -70, -79, 1, -11,
    -63, -77, 49, 52, 19, -16, -89, -10, 43, 64, -58, 60, -94, -72, -92, -61,
    -10, -18, 60, -98, -52, -59, 87, 19, 31, -53, -58, 0, 87, 97, -93, 72,
    -32, 46, 75, -15, -60, -19, -67, 41, -75, 86, 12, 5, 61, 49, -88, 62, 59,
    -7, 48, 81, 32, -41, -80, 95, 77, -73, -63, -55, 83, -75, 18, 66, 19, 98,
    61, 38, -44, -22, -35, 10, -35, -50, 20, -90, -15, -62, -22, 47, 73, -88,
    -75, 42, 9, 48, -95, 24, 53, -83, 23, -92, -97, 49, -91, 56, 96, -69,
    -69, 97, 74, -89, 42, 5, 69, -83, 98, 57, 0, -33, 33, 51, 89, -48, -91,
    49, -80, -3, -87, -94, 74, 35, 51, -6, -23, 95, 79, 12, 88, 80, 10, 93,
    -74, 25, 59, -17,
  -4, -12, 60, -51, 97, -40, 45, -56, 55, -55, 50, 80, -93, 38, 91, -25, -29,
    -45, -51, -85, -34, 92, -36, 66, 16, -59, 73, 48, 73, -78, 74, 23, -39,
    -13, -30, 68, -59, 25, -42, 98, -6, 19, -72, -17, -72, 18, -56, -46, 44,
    33, 46, 22, 83, 52, 52, -57, 85, 76, 46, 79, 37, 39, 60, -37, 37, -32,
    -48, -21, 49, -36, 78, 62, -80, 35, 75, 49, -68, 96, -52, -16, 41, -26,
    -39, 96, 58, 10, 66, -84, -22, 74, -82, -31, -26, 64, 34, -8, -85, -51,
    -10, 20, 14, 9, -24, 45, 14, 39, -7, 59, 0, 9, 25, -39, 25, -32, 21, -60,
    -83, 89, 27, 15, 47, -73, 80, 89, 55, 84, 81, -22, 93, -5, -74, 3, -72,
    52, 59, 18, 66, 88, 36, 62, 79, -70, -86, -48, -44, 88, 20, 44, 50, 68,
    75, -51, -60, -35, -19, 7, 47, -28, -69, 43, -96, -48, -97, -95, -96,
    -15, -83, 34, -95, 34, -90, -67, -59, -17, 82, -74, 81, -88, -45, -66,
    99, 73, -66, -56, 70, -65, -80, -93, 31, 80, 57, 78, -68, 55, 27, -82,
    -64, -15, -57, 84, 88, 53, 74, 50, -48, 70, -24, -57, 24, 70, 25, 17,
    -23, -92, 64, 20, 88, 60, 28, -53, -26, -62, -94, -95, -95, 72, 29, 73,
    -32, 99, -44, 56, 2, 98, -100, -33, 15, 4, -28, -94, 20, -92, -83, -38,
    62, 76, 76, -23, -43, 22, -31, 58, -57, 6, 40, -13, -44, 7, 55, -28, 63,
    -1, 32, -68, 80, 50, -22, -96, 84, 59, 52, -2, 65, 82, 48, -30, 91, -48,
    12, -67, -40, -6, 62, 57, -79, -98, 56, -49, 0, 43, 89, -33, -53, -13,
    61, -46, 60, -13, -66, 82, -49, 46, 36, 93, 86, -26, -22, 86, 57, -17,
    -82, -73, 27, -10, -70, 82, 7, -23, 35, -1, -25, 1, 73, -10, 85, 57, -24,
    13, 32, 63, 38, -28, 14,
  -84, -75, -19, -31, -34, 42, -53, -67, 36, -37, 49, -65, -13, 54, 74, 21,
    -25, 36, -25, -34, -96, -97, 44, 56, -53, 71, -56, 47, 96, 1, 15, -26,
    -88, -20, -49, -83, 57, 44, -25, 14, 62, -90, -8, -73, -8, -53, 9, -73,
    -20, -16, -91, -86, -2, 53, -35, -5, -99, -20, 19, 33, -55, 25, 6, -71,
    71, -49, -43, 67, -25, 34, 23, -6, 89, -3, 18, 79, -94, -97, -4, 70, 73,
    -45, 52, 34, 58, 43, 17, 74, 0, 23, 33, -91, 55, 55, 53, -10, -52, -97,
    18, -49, 67, -44, -36, 10, 95, -64, -56, 91, -25, 20, 58, -11, -36, -66,
    -1, -80, -59, 11, 84, -72, 85, 29, -79, 63, -97, -63, -88, -23, 39, -64,
    -48, 76, -85, 37, -53, 50, 96, 56, 42, 11, 81, -79, 1, 29, 59, 23, 43,
    -65, 27, -94, -24, 57, -93, -84, -42, -30, -11, -46, 56, -11, -1, -98,
    90, -2, 98, -91, 4, -23, 62, 39, 29, -38, -5, -56, -63, 17, 96, 77, 18,
    40, -44, -6, 81, 7, 91, -29, -14, -81, -9, 4, -14, 76, 73, -44, -90, -56,
    -69, 89, -16, -34, -73, -38, 8, 69, 37, -63, -7, -82, 51, -25, 86, 50,
    11, 0, -41, 59, -21, -89, -32, -6, -51, -33, 88, -13, 63, 74, -80, -32,
    13, 90, -82, 26, 28, -64, 67, -89, 26, -83, 78, 27, -65, 30, 39, -91,
    -13, -54, 3, -44, -79, 90, 3, 94, 78, -72, -93, 82, -47, 43, -98, 29,
    -29, 34, -64, 99, 46, -52, 24, 50, 78, -15, 74, -31, -46, 12, -2, -63,
    13, 94, 71, 99, 65, -43, 66, -96, -15, -98, -96, -39, -97, -78, -15, -3,
    59, 8, -67, 70, -54, -11, -21, 35, 53, -30, 96, 17, -5, -40, 90, 82, 31,
    -77, -84, -12, -86, 77, 99, -66, 70, 58, 50, -74, 77, -52, 88, -34, 12,
    -96, 12, 64, 44, 84, -50, -65, -99,
  34, -54, -2, 47, 78, -76, -34, -67, -2, -48, -11, 11, -55, -86, -67, 78,
    -4, 91, -57, -75, 80, 10, -61, -30, 3, 72, -33, 11, 30, 72, 7, 87, -93,
    35, -95, -30, -87, 45, 59, 96, -7, 93, -56, 79, 71, 25, -34, 81, -21, 95,
    41, 44, 69, -46, -61, 24, -26, -30, -68, 23, 9, -4, -24, 48, -51, -89,
    -4, -49, 84, -24, -43, 32, 72, 89, -46, 23, -42, -13, -17, 14, 77, -7,
    -9, 83, 21, 17, -87, -100, 86, -37, 11, 54, 71, 37, -37, -71, -7, -22,
    -55, 17, -34, 6, 69, -15, -42, -30, -11, -69, 44, 63, 52, -66, 87, 93,
    94, 23, 89, -69, -66, 61, 28, -59, -100, -73, -92, -36, 96, -22, -3, 41,
    10, -43, 46, 67, -9, 26, 76, -93, -91, 60, 57, 4, -36, 51, 10, 73, -15,
    96, 51, -95, 17, -17, -4, 87, -72, -21, -73, -8, 32, 42, -13, 65, -41,
    92, -57, 86, 27, 70, 83, 65, 72, 84, 98, 72, -26, 95, -75, -32, -23, -63,
    -12, -83, -18, 27, -39, 64, -48, 1, -46, -83, -74, -23, -96, 97, -21,
    -88, -6, 81, 19, 36, -17, -46, -6, -89, 71, 1, -89, 65, 68, 87, 68, 96,
    -11, 92, -50, 35, 2, 47, -79, -58, -5, 99, -36, 96, -58, 86, 55, -80, 85,
    7, 58, -38, -41, 68, -80, -86, 31, -42, -63, 46, 18, -87, 26, 36, 41, 91,
    -39, -63, 71, -39, -99, 28, -38, -53, -21, 79, 5, 61, -19, -45, 23, -18,
    -79, 92, 56, 52, 97, 62, 44, -68, -42, -12, 87, -42, 58, -25, -23, -83,
    86, 94, -41, -81, -51, 69, -5, -100, -4, -78, -6, -85, 98, -9, 96, 38,
    -29, -58, -4, -92, 74, -24, -51, -30, -15, -74, -35, 7, 15, -70, -12,
    -63, -24, -11, -64, -64, 3, -34, -41, 17, -2, -34, 53, 16, 85, -26, -23,
    94, -23, -47, -71, 59, -27, -3, 98,
  55, -58, -8, 84, 24, 46, -73, 92, 3, 10, 65, 25, -32, 93, -63, 35, 7, 89,
    49, 1, -35, 87, -41, -7, 93, 17, 30, 89, -97, 40, -44, 11, 0, 32, 86, 80,
    -39, -75, 85, 3, -77, -19, 97, 90, 11, 67, -25, -35, -94, 30, 88, 24, 18,
    -67, -73, -6, 29, -36, -10, 28, -76, 10, 2, -92, -31, -54, -91, 51, -66,
    34, 16, -18, 64, 8, -78, 41, 65, 38, 48, 32, -46, -42, -31, -33, -1, -15,
    66, 49, -48, 21, -63, 31, -67, -42, -64, -15, -69, 64, 23, -93, 98, 75,
    58, 17, -78, 80, 83, 14, 0, 77, -2, 65, -28, 98, -13, 72, 0, 80, -53, 90,
    -94, -72, 62, -49, -51, -93, 82, 29, -84, 31, -48, 63, -59, -86, -93,
    -57, 14, -80, -27, 49, 98, 12, -63, -7, -95, 74, 23, -53, 44, -71, -1,
    -80, -35, -35, -2, -78, -53, 70, 0, -72, -38, 64, -86, 70, -27, 58, -86,
    -15, 52, 35, 87, 99, 71, -84, 89, -82, 38, 77, -62, 57, 31, 52, 55, 37,
    -2, 81, 63, -2, 4, -17, -31, -46, 13, 13, 55, -69, -97, -3, 41, 88, 54,
    -3, 64, -88, -52, -80, -11, -73, 43, 64, -95, -19, -60, -68, 20, -83, 53,
    66, 90, 1, -53, -34, 96, 38, 64, -35, 0, 11, -78, -80, 34, -82, 51, -15,
    -75, -39, -18, 89, -83, -79, -71, -55, 30, -98, 55, -65, 55, -39, 61,
    -61, 23, -73, -37, -61, 69, -98, 11, -50, 18, 88, 79, 26, -34, 36, -24,
    15, -19, -47, 50, 50, -44, -91, 39, -50, 84, -22, -34, 10, 78, -86, 43,
    30, -81, -76, -94, 45, -99, -88, -84, 54, 39, -49, 51, -90, 71, 34, -99,
    -35, -9, -23, 14, -51, -92, 33, -9, 72, 86, -29, 65, 86, 82, 15, -15, 35,
    25, 86, -84, 75, -28, 47, -14, -33, -9, 77, -99, 95, 85, 51, 5, -93, 5,
    -50, 70,
  52, -43, 99, -68, 42, 43, -6, 29, 53, -85, -99, -18, 91, 80, -54, 27, -76,
    2, -30, -70, -79, 68, -68, 54, 21, -46, 13, 23, 7, 9, 61, -20, -33, -94,
    -22, -15, 42, -50, -36, 20, -17, -56, 38, 88, 68, -66, -86, 61, -44, 78,
    46, -99, -10, 57, -90, -66, -40, 7, 85, -57, 93, 48, 60, -31, -52, 33,
    -20, 15, 83, 28, 7, 3, -8, -22, 10, 76, 20, -73, 73, -82, -5, -94, 98,
    -57, -38, 20, -98, -13, -32, 23, 49, -2, -82, 70, 95, -41, 99, -9, -50,
    24, 5, -48, -45, 82, -68, -35, 10, 12, -83, 87, 96, -32, 60, 53, 42, 59,
    -38, 62, -1, -83, -32, -28, -100, -79, 6, 4, -34, 83, 61, -33, 90, 26,
    18, 17, 32, -76, 22, 14, -72, 97, 99, -62, -59, -81, -8, -58, 92, 59, 2,
    -86, -56, -70, -4, -69, -60, 95, -23, 4, 10, -51, -12, -23, 97, -79, -44,
    -36, -84, 60, -34, -58, -62, 1, 54, 26, 60, -58, -20, -69, -60, 18, -20,
    9, 85, -82, -27, -39, -56, 67, 26, 13, 16, -75, -26, -92, 15, -74, 31,
    -93, 90, -24, 35, 66, 8, -35, 21, 78, -5, 28, 31, -39, -82, 77, 78, -59,
    -50, 64, 59, 16, 63, 53, -19, 95, 13, -84, -73, -75, 13, 13, -66, -78,
    -49, -72, 51, 98, -12, -99, -43, 83, 52, -74, -87, -14, 71, 45, 28, -39,
    48, -55, -50, -76, 87, -23, -39, 86, -54, 16, -48, 15, 93, 2, -81, 20,
    47, -67, 76, 85, -54, 65, 19, -74, 6, 26, -40, 73, -80, -90, 84, -22,
    -30, 52, 58, -31, -10, 78, 89, -14, 99, -47, -96, -18, -87, -56, 19, -27,
    36, -74, -16, 98, -89, 78, -53, 43, 54, 41, -46, 46, 84, -61, -54, 35,
    73, 72, -21, -36, -43, -57, 24, -66, -44, 87, 44, -30, -70, -81, -77,
    -70, -30, -89, 58, 71, -53, -36, -8,
  -1, 6, -75, 17, -39, 90, -2, 64, -19, -20, -15, -55, 49, 26, -71, -12, -65,
    93, -61, -89, 18, 60, 63, 91, 3, -68, -87, 44, -34, -55, -21, -83, -91,
    21, -25, 91, 64, -31, 84, -74, 80, -22, -70, -55, 59, 15, -57, 3, -14,
    32, 78, 95, 79, 53, 62, -35, 91, 10, 69, -76, 89, -44, -31, -12, 29, 43,
    -47, -32, -11, 78, -57, -8, 95, -98, 91, -2, 8, -41, -86, -1, 41, 74, 22,
    -81, 70, -96, 57, -45, -67, -44, -68, 32, -85, -5, 57, 63, -60, 74, -93,
    -1, -8, -97, -99, -29, -2, 55, 61, 97, 91, -82, 77, 67, -36, 67, -72, 30,
    -47, -18, 24, -69, -70, -71, -82, -83, -59, -19, 57, -23, 62, 81, -11,
    -33, 36, -54, -14, 58, 95, -96, -78, 25, -42, 33, -19, 5, -85, 1, 57,
    -69, -82, 80, 69, 89, -87, -37, -37, -1, -83, 13, -20, -17, -9, -66, 11,
    -3, -31, -53, 69, 43, 44, -99, -72, 77, -10, -20, -12, -91, -9, 63, -88,
    -16, 34, 70, 81, 21, -84, -52, -47, -94, 21, 59, -91, -37, 96, 24, 48,
    47, -65, -58, -44, 43, -42, -10, 32, -74, -78, -97, -100, -87, 96, 9,
    -19, -56, -74, 85, 78, -81, 4, -17, 86, -11, 57, 40, 8, -18, 75, 13, 93,
    85, 61, 32, 31, 21, -25, 65, -87, -43, 94, 44, 8, -61, -88, 98, -5, -63,
    37, 80, 79, -22, 77, 88, 19, -96, 44, -22, 6, 16, 64, -40, -81, 46, -64,
    -95, -5, -48, -42, -20, 6, -79, -67, 99, 50, 62, 22, 18, -61, -8, -65,
    56, 19, 63, -9, 88, 46, -44, -71, 37, -45, -4, 63, -58, -61, -55, 55, 92,
    90, -83, -21, -47, 53, 17, 91, 17, 21, -43, -3, -30, -38, -30, -43, 98,
    -78, 44, 64, 12, 52, -12, 28, 88, -92, 48, -56, -44, 75, 69, 44, 35, -98,
    18, -63, 91, -49, -87, -91,
  14, 12, -61, 12, -2, -51, -31, 86, 86, -94, -67, 90, -3, -24, 47, 87, 68,
    31, -89, -39, 3, 86, -76, -28, -35, -90, 19, -38, 26, -69, 65, -44, 85,
    27, -37, -38, -78, 48, -94, -20, 0, -54, -22, 92, 26, 60, 43, -7, 68, 35,
    39, -45, 68, 2, 75, 63, 77, -32, 2, 20, -80, -44, -96, -46, -14, -29,
    -54, -84, 44, -21, -54, 63, 8, 79, 47, -81, 36, -54, 56, 78, 10, -86, 8,
    23, 80, 57, -25, -82, 6, -64, -98, -35, 21, 77, -17, -39, -64, -87, 81,
    -28, -24, 83, -18, 94, 13, -22, -70, 84, -53, -88, 32, -62, -63, -24, 14,
    -100, -57, -5, -33, -15, 88, -24, -68, -23, -16, -38, -7, -32, -48, 56,
    18, -26, 27, 1, -98, -94, 66, -39, -19, 96, -19, -66, 70, 48, -86, 28,
    -54, 48, -38, -95, 54, 35, 52, -44, 63, 49, 65, 5, -2, 83, -84, 9, -41,
    -91, -9, -75, -79, 47, -49, -68, -54, 47, -69, -8, 42, -25, 58, 2, -63,
    -62, 14, 68, 21, -45, -10, 66, -76, 27, -77, 21, 87, -23, -47, 40, 83,
    -50, -27, -4, -89, 9, -65, 83, 13, 4, 56, -60, -38, -30, -6, 56, 96, 90,
    -59, -51, 88, 41, 41, -61, -96, 95, 59, -99, -43, -96, 55, -99, 80, 69,
    97, 64, 28, 96, 94, -68, 25, 98, 74, 12, 49, -98, 43, -29, 22, 72, 70,
    -2, -94, -76, -99, 82, -7, 21, -32, -64, 53, 26, -29, -4, 85, 13, 81,
    -19, 59, -19, -35, 25, -77, -33, -69, 49, 11, -7, 91, -52, 96, 30, 84,
    -44, 26, -9, 95, -31, 18, 20, -86, 92, -14, -55, 82, 88, -32, 25, -49,
    93, -59, -71, 47, 51, -89, -63, 60, 28, -50, -76, -32, -20, 39, 89, 24,
    -41, 53, -27, 56, 23, -44, -97, -4, -99, -40, 83, -38, -13, -45, 37, 79,
    -14, -48, -81, -50, 0, 98, 26, -94,
  14, 67, 44, -40, 41, -98, -50, -68, -7, 66, -43, 50, -41, -21, -68, -37,
    28, 67, -37, -71, -54, -15, -26, -5, 8, 20, -25, 90, -78, 82, 57, 57, 21,
    71, 36, 85, 22, 20, -75, -64, -29, 92, -44, 20, 6, -37, -32, -99, -13,
    -96, 81, -39, 51, -20, 38, -75, 1, 38, -89, 55, 87, 0, 73, -35, -38, -25,
    -9, 48, -64, -25, -20, -96, -8, 29, -91, 24, -86, 96, 98, 21, 70, 37,
    -28, 39, 33, 27, 59, 52, 13, 84, -33, -70, 3, 27, -4, -37, -54, 85, -7,
    48, 74, -92, -50, 86, -44, 64, -44, -43, -33, -10, 31, -13, -76, -65,
    -98, -74, 89, 27, -96, -32, 53, 68, 36, 91, -94, 25, -56, -100, 60, -36,
    74, -16, -68, 39, -12, 27, 83, -53, -92, -21, 30, -75, 41, -46, 33, -22,
    -3, 37, 34, 39, 48, 25, 72, -67, -85, -23, 96, 85, -54, 30, 95, 16, -9,
    -89, -52, 73, 15, 42, -72, 32, 55, -3, -71, -29, 54, 69, 90, 24, -7, 75,
    -94, -27, 61, 15, 18, -86, 12, 20, -25, 69, 90, 68, -90, 62, 79, 42, 74,
    56, 15, 32, -28, 14, 69, -17, -8, -55, 14, 67, 26, -90, 92, 68, -9, -50,
    -88, 68, -13, -50, -47, -51, 29, -69, 66, -95, 85, -58, 44, -21, -38,
    -37, 93, -57, -31, -48, -68, 14, 62, 91, 29, -5, -87, -70, -92, -45, -26,
    -11, -48, -31, 58, 79, 30, 13, -82, -33, 44, 74, 90, -96, 13, 80, -32,
    -18, 40, 5, 88, 55, 8, 3, -12, -57, -77, 88, -29, -92, 86, 5, 93, -54,
    29, 49, -28, 42, -58, -2, 97, 73, 50, 58, -90, 31, -8, 42, -71, 99, -98,
    92, -83, -24, -21, 75, 93, -31, 50, -100, -38, 4, -18, 35, 77, 46, -53,
    16, 47, -28, 82, -36, 50, -63, 39, -6, 81, -81, -33, -54, 2, -50, 88, 95,
    77, -36, -99, 38, -26,
  -44, -78, -83, -79, 42, -34, -10, -83, 29, 27, 82, 15, -81, 92, 4, 82, -58,
    -61, -6, -77, -56, 44, 26, 11, 19, -3, -35, 5, -5, 62, -35, 23, 80, -41,
    30, 85, -8, 5, -79, -68, 85, 22, -83, -19, 54, 10, 12, 76, -39, -86, 83,
    61, -83, 44, -100, 81, -60, -87, -23, 84, -71, -8, 50, -8, 36, -17, -62,
    -39, 95, 93, 91, 76, 15, -26, -5, 16, 80, -22, 66, 68, 49, -64, 18, 4,
    -15, 44, 3, 75, 68, -10, 91, 2, 3, 44, 97, 62, 91, -79, -85, 23, 43, -12,
    -93, -22, -20, 92, 67, -77, -20, -100, -47, 83, -71, 0, 34, 4, 37, -45,
    -53, -83, 82, -96, -59, 57, 49, 49, 86, 91, -36, 92, -24, 95, -85, 63,
    92, -36, 84, -60, -42, 5, -61, -61, 88, 84, 83, 27, 26, 46, 9, 96, -71,
    -65, 61, -92, -23, 83, 22, 45, -35, -47, -58, 69, 0, -100, -86, -18, -94,
    -97, 94, 66, -43, 57, -1, -78, 95, 35, 51, 88, 92, -42, 47, -99, -95,
    -41, -25, -69, 83, 15, 60, -23, -96, 25, -24, 8, 81, 91, 54, 4, -49, -26,
    -76, 34, 10, -22, 16, 96, -97, 87, 62, 11, 67, 23, 62, -65, -42, 47, 20,
    -32, 0, -40, 39, -84, 36, -97, 42, 91, -16, -86, -97, 83, -75, -40, 34,
    69, 6, -32, 23, -100, 25, -78, -52, 7, -85, -36, -87, -39, 13, 51, 87,
    24, 20, -88, 0, 83, 52, -68, 78, 92, -87, -54, -45, 6, 40, 2, 25, 5, -55,
    -64, 75, 37, 73, -6, 65, 31, -60, -20, 74, 85, -49, -80, -40, -96, 85,
    88, -61, 13, -92, 62, -52, -34, -40, 85, 76, -36, -28, 58, -73, -70, -39,
    -84, 82, 98, -21, 72, 56, -98, -54, -100, -22, -40, 22, 83, 95, 98, -46,
    55, -1, -12, -12, -35, 28, 61, -4, -25, -100, -27, -24, 48, 28, -95, 81,
    41, 40,
  -57, 49, 13, 15, 74, 72, 35, 72, 11, -59, -22, -49, 84, -13, 18, -64, -98,
    -53, 20, 93, -49, 58, -8, 3, 57, 85, 52, 77, 85, -20, 7, 13, -47, 1, -94,
    -29, 8, 87, -25, 70, 48, 44, -10, -53, -24, 77, -63, 27, 57, -77, 29, 99,
    -32, -80, 88, -94, 6, -13, -2, 19, 78, -78, 8, -87, 9, -25, 15, 36, 8,
    -82, 13, -62, -70, -98, 13, 48, 13, 7, -13, -95, -8, -63, -4, 28, -95,
    -31, -16, -25, 19, 86, 50, 67, 23, -56, -47, -54, -11, -28, 17, 13, -9,
    61, -36, 75, 81, 65, -6, 98, -16, 43, -28, 12, -2, -56, 93, -52, 91, 6,
    86, 9, 33, 45, -15, 53, 30, 60, -38, -90, 58, -87, 40, -28, 48, 24, -8,
    52, 67, -72, 99, -86, -26, 2, -99, 12, 6, -69, 0, 84, -97, -10, -51, 38,
    58, 84, -65, 28, -92, -32, 70, 19, 27, 57, 80, -19, -69, 7, 72, -1, 38,
    27, 39, 58, 78, -25, -28, -85, 41, -86, 33, 60, 32, 75, -43, 85, 70, -52,
    -91, -94, -24, -72, -68, -59, -58, 98, -75, 53, -92, -93, -72, -53, -28,
    30, 41, 68, -61, -20, 20, -54, -43, -76, -1, -87, 72, 44, 20, 1, 14, -26,
    -84, -8, -59, 23, -16, 40, 85, -87, -34, 26, 50, -87, 69, 71, 82, -100,
    36, -75, -1, 16, -95, -77, -68, -46, -18, -41, -10, 54, -77, 37, 92, -8,
    99, 70, 18, -92, -69, 14, -60, -67, 98, 99, -18, -50, -83, -8, -80, -59,
    -53, 30, 26, 94, -41, -86, -26, 18, 50, 63, -27, -67, -68, 68, -38, 48,
    59, 31, -72, 35, -25, 29, -35, -22, 44, 59, -37, 82, 63, 4, 1, 8, -14, 4,
    -70, -65, -66, -41, 54, 60, -93, -50, -81, -28, 22, 9, -99, 76, -47, 48,
    70, 8, -63, 98, -85, 69, 10, 92, -4, 46, 39, 23, 64, -83, -24, -23, 47,
  -15, -14, 87, 92, -28, -9, -76, 99, -50, -38, 41, 66, -93, -63, -38, -67,
    -9, -29, 72, -48, 52, 72, 19, 58, 67, 85, -3, 79, -14, 30, 5, -90, 22,
    48, -84, 32, 38, 14, 36, 21, 68, 32, 78, -87, -63, 3, 50, -84, 78, 58, 7,
    -74, -14, 77, -37, 61, 14, 50, 61, -32, 68, -62, -84, 55, 95, -45, 72,
    -52, 47, -51, 7, 13, 29, -55, -4, 84, 76, -8, 80, -13, 1, -56, -19, -42,
    -82, 1, -63, -39, -92, 9, 65, 65, 64, 84, -86, -97, 78, 42, 9, -51, 26,
    -31, 56, 37, -31, 13, 11, -43, 61, -35, 57, -72, -84, 6, 86, 75, -59,
    -56, 0, 73, 57, 67, 16, 73, -41, 80, -33, 79, -20, 69, 57, 42, -12, 68,
    82, -92, 74, 95, -76, 64, -21, -47, 73, 13, -14, 66, 49, 1, -13, 85, -90,
    7, -98, 78, -77, -68, -62, 6, 51, -96, 17, 54, -85, 27, 27, -30, 34, -36,
    78, 96, -83, 62, -22, -23, 34, -82, -92, 82, 41, 82, 77, 6, -77, 82, -81,
    27, -32, -13, -98, -36, 26, 10, 2, 52, 48, -94, 28, -51, 54, -43, 29, 2,
    -33, 39, -37, -59, -64, 54, -73, 56, 84, 82, 82, 88, -75, 4, -58, 66, 46,
    -36, 24, 93, 69, -5, -86, -4, -44, 87, 70, -10, 35, 60, 47, -1, -8, -23,
    88, 1, -44, -30, 36, -100, -38, 14, 39, 99, 73, -21, -75, -90, -43, 19,
    -88, -85, 36, -99, -47, -29, 14, -36, 81, -53, 54, -92, -6, -37, -95,
    -43, 99, -63, 9, -32, 78, 80, 25, -2, -41, 20, 40, 89, -4, 87, 44, 2, 36,
    22, 44, 94, -54, 89, -5, 77, -50, -21, -73, -40, 73, -8, 85, -6, -28, 42,
    -32, -33, -62, -100, -48, 78, 74, -44, 35, 46, -2, -70, -74, 22, 7, 48,
    4, 26, -94, 24, 71, 49, -56, -93, -62, -70, -26, -72, -29, 47, -28,
  -83, -91, -45, 28, -84, -46, 69, 98, -3, 58, 29, 17, -99, 19, 83, 20, -13,
    -99, -9, 88, -70, -61, -54, -33, -57, 34, -64, -14, 62, -39, 28, -92,
    -86, 33, 97, 97, -24, 3, 4, 37, 26, -63, -48, 66, -88, 90, -90, 27, 2,
    77, -45, -67, 74, -46, -50, -98, 63, -93, -5, -97, 37, 39, 55, -21, 57,
    83, 38, 17, -82, -27, -61, 77, -2, 72, -23, 85, -30, -7, -81, -72, 57,
    -97, -9, 80, 92, -95, -14, -13, 40, 19, 6, 8, 98, -11, 3, 96, 88, -1, 16,
    26, -57, 94, -47, 69, 5, 20, -9, 31, -50, 99, -30, 57, -97, -20, 54, 71,
    4, 1, 81, -10, 71, 54, 38, 26, -12, 62, 42, -94, -80, 60, 47, 38, 98, 69,
    -46, 72, 28, -61, -31, -52, -11, 22, 2, -75, 61, 68, 90, 77, -24, 53,
    -12, -42, 26, -2, -97, -3, -12, -19, -99, -56, -28, -70, 21, 45, -100,
    95, 7, 78, 25, -74, 1, -15, -73, 33, 57, 98, 95, -85, 96, -86, -72, -90,
    -69, -85, -80, 68, 86, 48, -39, -57, -77, 14, 1, -49, 12, 47, -47, 52,
    -22, 16, 9, 86, 31, -59, 56, -64, 65, -39, -45, 88, 27, 54, -79, -53,
    -92, -26, 76, 66, 78, -89, -93, -33, 49, 97, 84, 67, -23, 88, 45, 1, 20,
    37, -97, 71, -3, -99, -48, -15, -96, -23, -20, 80, -60, -49, 60, -77,
    -36, -66, -17, 28, 94, -66, -66, -66, 82, 91, -13, 88, -54, 70, -11, -11,
    85, 15, 40, 87, 3, 86, -44, -36, 8, -10, 13, -76, 98, -48, -89, 91, 68,
    -83, 67, 9, -53, -70, 93, -40, 32, -74, 25, -17, -68, 42, 83, -21, 93,
    39, -80, 33, 18, 54, 96, -15, 66, -6, 95, 87, -94, 24, 58, 13, 70, 29,
    40, 50, 93, -2, -1, -69, 95, 86, -35, 65, -72, 72, 4, -92, -97, 9, 16,
    15, 48, 55, -48,
  -32, -15, -20, -63, -90, -26, 68, -15, 52, -62, -73, 61, -13, 31, 93, -87,
    -86, 2, 99, 72, -1, -32, -17, 46, 98, 50, -57, -99, 73, 23, 97, 60, 12,
    -54, 43, 64, -15, -52, -26, -93, -70, 21, 68, -96, 17, 74, -26, -25, -44,
    89, -1, -92, 74, -71, -69, -25, -73, -83, 18, -52, -37, -63, -80, -30,
    95, 11, 41, 63, 72, 71, 14, -65, -5, -6, -100, 2, 98, -74, 87, 65, 97,
    93, 28, -72, -28, -4, -91, 7, 1, 44, 20, 74, -71, -87, -7, 49, -67, 42,
    38, -33, 20, 4, 27, 6, 53, 82, -9, -8, 90, 74, 64, 40, -44, 20, -61, -75,
    -88, 71, -11, -75, -46, 65, -36, 71, -51, -8, 66, 96, -62, 67, -72, -51,
    -82, 29, -39, 95, 53, 7, -84, -31, -3, 4, -94, 99, -64, 26, -98, 22, 97,
    60, -97, 85, -75, 41, -99, -67, -3, -26, 32, -8, 88, -32, 13, -35, -90,
    -79, -21, -62, 5, 8, -34, -6, 51, 9, 27, -60, 99, 44, 6, -39, -87, 84,
    20, 12, 77, -6, -12, -85, 91, 44, -24, -62, 6, 22, 41, 49, 2, 4, 61, -29,
    80, -88, -81, 48, -45, 87, 41, 35, -71, 20, -16, 96, 87, 23, -91, -11,
    -8, -49, 13, 55, 52, 5, -53, -21, 24, -24, -6, -63, -80, -45, 10, -32,
    -55, 15, -94, 34, 7, 27, 21, -82, 58, 89, 78, 10, 56, -47, -93, -20, -31,
    -69, -15, -3, -39, -13, -62, -52, 13, 99, 81, -41, 61, 18, -59, -98, -12,
    78, -13, -61, 0, 39, 75, -50, -87, 26, 51, -51, -63, -36, 35, 1, 88, -78,
    -20, 12, 51, 97, -34, -97, 98, -82, -84, -42, 43, 51, 92, -91, 86, -86,
    47, 21, 73, 59, -74, 36, 25, 45, 89, 88, 60, 8, -36, -2, 35, 67, 48, -4,
    -75, 53, 76, 50, 42, 72, -38, -34, 27, 60, 28, 22, 23, -42, 80, -67, -67,
  -56, -80, -97, -7, -61, -98, 48, -42, 79, 43, -99, 37, 87, -55, -58, -14,
    56, 4, -56, -38, 92, -80, -45, -95, 8, 57, 59, 28, 14, -37, -17, 7, 6,
    -74, 99, -65, 64, 29, -100, -84, -35, -78, 57, 69, 46, 11, 97, 89, -87,
    -51, 56, -79, 71, 56, -80, 77, -77, 82, -50, -26, 46, 31, -62, 82, 2, 1,
    70, 26, 43, 40, 48, 93, 89, 90, -76, 14, -19, -63, 95, -17, -94, -5, -72,
    -78, -96, -36, -86, -57, -75, -28, -52, -28, -80, 82, 33, -90, -23, -35,
    8, 47, 65, 89, 89, 32, -20, 32, -97, -82, -51, -73, 27, -58, 20, 60, 23,
    -65, -21, -92, 56, 95, -51, 88, 76, -59, -77, -63, -27, -7, 94, -22, 27,
    -7, 9, 41, -54, 3, -39, 90, -72, 27, -17, -90, 69, -43, -66, 58, -5, 63,
    38, 59, 47, -8, 77, -26, -82, 71, 12, 41, -34, -21, 43, -79, 84, 82, -24,
    -54, 62, 27, -47, 76, -35, -81, -78, -65, 50, -5, 58, -16, -30, 92, -47,
    75, -40, -42, -97, -90, -8, 13, -10, -79, 80, -15, -29, 2, 77, 49, -62,
    1, -2, -32, -19, -98, 82, 39, -47, 7, -35, -56, -38, 61, -98, -18, -29,
    28, 46, 98, 92, 54, -3, -9, -56, 39, 61, 92, 47, -2, 24, -55, 94, 46,
    -31, 81, -51, -78, 48, 80, 36, 98, 99, -52, 74, -3, -16, -98, -16, -94,
    72, -58, 67, 18, 93, 26, 91, -2, 36, 70, 66, -67, 86, -12, -51, 47, -63,
    27, -37, -20, 72, 88, -60, -48, -16, 36, -85, -89, 13, -99, 50, -47, -90,
    -96, -11, 24, 20, 75, 51, 33, 72, -18, -53, 64, 61, -28, -30, -32, -52,
    90, -51, -56, 69, -69, 37, -10, 42, 61, 74, -45, 95, 50, 41, 82, 52, -67,
    -49, 85, -97, 22, 30, -99, -61, 80, 90, 9, -8, 19, 96, -31, -90, 58, -52,
    -68, 34, -42, -93,
  36, 4, 1, -99, 17, -66, 74, 36, -81, -86, 2, 29, 99, 77, 75, 46, -77, -79,
    79, -91, 61, -54, 5, -6, -96, 74, 74, 93, -62, 19, 21, 93, -14, -89, 23,
    -42, 54, 22, 28, -18, 32, -75, -85, -66, -19, 26, -60, -59, 19, -44, -78,
    35, -5, -24, 53, -19, 80, 24, -64, 70, 39, 76, -98, 16, 22, 34, 22, -35,
    -8, 86, -81, 45, -98, -5, -99, -70, 71, -55, -20, 86, 4, 81, -7, -22,
    -80, -72, -62, 78, 57, -56, -26, 10, -3, 61, -16, 38, 85, -32, -96, 16,
    48, 30, 11, -50, 58, 42, 12, 44, 36, -48, 63, 38, 77, -62, -31, 69, 2,
    13, -97, 9, 25, 48, 19, 80, -22, 90, 25, -79, 85, 76, -66, -52, -1, -40,
    -82, -58, 85, 28, -80, 71, -83, -7, 25, 27, -46, -43, 76, -97, -60, -49,
    32, 16, -95, 26, 3, -96, 23, 85, -12, -51, -71, 28, -8, -15, -79, -40,
    -6, 31, -55, -56, -1, 76, -43, 7, 94, -73, 35, 6, -74, -48, -15, -82,
    -52, 28, 7, -4, 60, -38, 94, -94, -89, 21, -88, 91, 83, -40, -48, -74,
    -76, 43, 53, 58, 59, 41, 96, -38, -41, -22, 67, -74, -62, 45, 93, -23,
    -93, 50, 57, 99, -99, 99, 7, -96, 21, 89, -20, -30, -53, 17, -48, 87, -8,
    -45, 40, 67, 17, -83, -84, -66, -28, 83, -74, -6, 23, 0, 70, 45, -61,
    -96, 78, 44, -86, -28, -84, -33, -26, 90, -97, 9, -60, -89, 10, -4, -12,
    -92, -32, -17, -55, 59, 79, -56, -77, 15, -50, 62, 57, 68, -31, -12, -6,
    -88, 18, -21, -77, 22, -50, 16, -27, -76, 6, 14, -31, -63, -15, 33, 82,
    -32, 67, -20, -55, 19, -26, 4, -44, 43, 24, -93, 89, 41, -80, -75, 78,
    62, -7, 37, 54, 53, 66, -48, 26, 45, 19, 97, -67, -77, -19, 92, 16, 31,
    88, 70, -83, -67, -3,
  -60, 5, 44, 85, 17, 53, 54, 1, -2, -92, 86, -95, -37, -65, -5, 75, 98, -39,
    87, -10, 4, -44, -96, -67, 23, 24, -80, -60, -84, -47, 7, -42, -76, 73,
    -84, -25, -82, -44, -38, -72, 25, 4, 63, 92, 51, -99, -68, -30, -50, -59,
    99, -64, 44, 66, -65, -33, -25, 56, 23, -59, -37, 21, 41, 63, -17, 87,
    71, 30, 19, -93, -73, 90, -47, -30, -52, 95, 19, -12, 9, -56, 66, 68, -8,
    78, 61, 85, -38, 7, 12, 23, -9, 75, -87, 1, 11, 80, -19, 20, 53, -23, -4,
    73, -7, -54, -73, -93, -70, -57, -75, 10, 45, 11, -47, -43, -43, -79,
    -39, 79, 91, 10, 21, -70, 12, -43, 63, 89, -81, -76, -92, 90, -16, 13,
    88, -90, -82, 71, -66, 0, 26, 91, 46, 34, 67, -61, 16, -40, -72, 78, 80,
    60, -13, -22, 10, 8, -90, 35, -31, -47, -25, -15, 32, -19, -61, 50, 36,
    -20, 89, 79, 29, -26, -41, -59, -87, -36, 17, -100, -70, -64, 21, -91,
    -82, -56, 32, 91, -76, -61, -44, 76, -19, -68, -94, 32, -44, -14, 75,
    -77, -50, -66, -12, 32, 33, 77, -90, 59, 49, -37, 10, -45, -30, 6, 80,
    65, -32, 69, 98, 60, -100, 3, -30, -63, 44, -51, -93, 0, 44, -6, -74, 10,
    -63, 30, 51, -99, -19, -12, -90, -31, 32, 44, -17, -82, 47, -17, 81, 25,
    49, 74, 14, 93, 71, -91, -30, -16, -62, 29, 1, 24, -37, -69, -76, -79,
    -81, 17, -100, -37, -95, -10, -31, 53, 37, -76, -17, -77, -28, 67, 30,
    43, -52, -34, -47, 37, -82, 12, 55, 87, -27, 95, 87, 9, 33, 50, -21, 71,
    -8, 1, -97, 41, 34, 40, -45, -3, 14, -9, 67, -36, 19, -32, 25, -69, -67,
    -63, 43, 79, -23, 24, 32, -39, 20, -88, -6, 99, 9, -34, -97, 36, -43,
    -74, 51, -1, -94, -95, -31, -94, 62,
  53, 2, -42, -28, 50, 29, 47, 24, 33, -35, 9, 18, -43, -14, 20, 39, -90, 22,
    9, -69, -52, -5, 25, 96, -9, -83, -10, 51, -85, -11, 38, -43, 24, 64, -6,
    -57, 69, 63, 63, 11, 35, -40, 85, -65, 32, -22, 6, 80, 51, -80, 99, -7,
    -13, 48, 75, -75, 80, 44, 39, 34, 54, 63, 70, 59, 79, -5, -50, 2, -23,
    -89, -8, 67, -36, 98, 54, 16, -61, -69, -42, -80, 70, -31, 80, -71, 21,
    -7, 51, 0, 2, -36, -13, 27, 61, -54, -7, 35, -98, 85, 59, 98, 56, -9, 89,
    -88, -81, -71, -78, 45, 13, 95, 95, 40, 14, -97, 50, -60, 99, 56, 91, 53,
    22, 37, 59, 19, -70, 94, 69, 12, -53, 40, 21, 57, -92, 75, 93, -52, -41,
    2, -10, -97, 1, 96, 31, 61, 23, -3, -83, 46, -29, -46, 22, 21, -42, 93,
    94, 65, 82, -39, -79, -60, -54, 25, -6, -92, -56, -32, -80, -92, 17, 1,
    -49, -7, -1, 4, 17, 67, 38, -4, 69, 50, -99, -99, -79, 34, -23, 23, -29,
    75, 45, 26, -53, -71, 54, 64, 48, 43, -86, -88, -50, -15, 23, -64, -89,
    85, -31, 5, 97, 2, -37, -46, -99, 44, -10, 38, -10, -85, 36, -100, 66,
    -83, 31, 4, 88, -8, -75, -24, -79, 94, 78, 32, -93, 25, -55, 44, -7, 35,
    6, 28, -56, -22, -10, 52, -81, -30, -70, 77, 24, 97, -36, -18, 90, 59, 7,
    21, 47, 30, -91, -13, 95, -4, 95, 96, -43, -46, -76, 87, -31, 67, 30, 41,
    -23, 42, -63, 7, 18, 45, 27, -76, 84, -59, 43, 92, 60, 90, 0, -49, -3,
    73, 44, 2, 36, -83, -74, -8, -14, -99, 57, -66, 70, -49, 16, -61, -65,
    37, -21, 88, -89, 2, -86, -96, 98, 10, 32, -71, 23, 10, -44, 34, 5, -78,
    35, -9, 83, -78, -98, -77, 9, 91, -85, 75, -91, -8, 69,
  60, 69, -84, -76, 77, 41, -3, -8, -61, 66, 1, -100, -39, 76, 38, -22, -33,
    -87, -79, -26, 1, 96, -71, -29, 65, 59, 98, -74, 27, 8, 66, -19, -69, 75,
    -32, 37, -57, -29, -40, -56, -4, 0, 66, -40, -2, 18, -52, 23, -97, 79,
    -95, 58, -59, 44, -61, -6, -8, -43, 7, -14, -30, 53, -38, -22, -77, -87,
    93, -7, 35, -60, -48, -9, -29, 4, 22, -82, 30, 55, 77, 90, 72, -91, 77,
    93, -74, -10, 86, 51, -15, 74, 34, -8, -86, -9, 7, -3, 10, -9, -99, -35,
    54, -45, 33, 63, 59, 75, 68, 17, -44, -67, 91, -54, 99, 48, 17, -66, -58,
    -81, -45, -43, 63, -51, 10, -76, -67, -39, -36, -82, 33, -6, -84, 58, 40,
    -34, 6, 40, -29, -95, 41, 53, -15, -54, 50, 71, 96, 20, -20, -27, 64,
    -32, 26, -96, 39, -10, 52, 10, -50, -31, -77, 7, -45, -92, -81, 25, 87,
    -25, -34, 70, -30, 38, -7, 42, 34, -31, -39, -27, -29, -48, 60, 0, 75,
    97, 75, -57, 40, 64, 46, -74, 25, -94, 83, -54, -24, -89, -50, -69, 3,
    10, 58, -30, -39, -65, -94, -96, -54, -34, 19, 65, 15, 46, -19, 7, -92,
    -8, 99, 37, -22, -37, -71, 71, -11, 0, 12, 50, -14, -3, 13, 30, -43, 64,
    -33, 5, -45, -72, -10, -95, 24, 25, 20, -46, 78, -58, 54, -91, -52, 36,
    6, -30, 47, 44, 10, -18, 67, 26, 19, -80, -88, -84, -53, 57, 63, 38, -3,
    83, -7, -87, -41, -60, -54, 84, 26, 34, 97, -11, -53, 37, 12, -30, 43,
    20, 37, -76, 56, -41, 4, -49, -94, 12, -91, 19, -72, -85, -82, -19, 11,
    60, 31, 82, 86, -29, 62, -38, -5, -53, 94, 44, -9, -70, -27, 22, -54,
    -65, -46, -6, -21, -70, -61, -2, -17, -91, 21, 95, 74, 92, -51, -58, -90,
    20, -62, 37, 8, 73, -93,
  -49, -29, -68, 64, -49, 95, 77, 37, -89, 61, -62, 21, -28, -8, -79, -62,
    -26, 8, 92, 6, -88, 66, 86, 36, -34, 84, 57, 92, 9, -71, -3, 30, -99,
    -70, -30, -47, 60, -64, -62, -95, -23, -17, 20, 16, -74, 28, -91, -69,
    -19, 72, -65, -73, 71, 78, -22, -73, 48, 95, -17, 7, -54, 5, -28, 6, -75,
    -31, -53, -7, 36, -98, -52, 10, 19, 25, 37, -71, -54, 86, 90, -78, -27,
    -43, 86, -10, 94, -4, -12, 20, -89, -81, 10, -45, 61, 53, -62, 96, 97,
    -84, 8, 64, -63, 60, 52, 63, 20, 19, 28, -56, 24, 17, 71, -64, 73, -99,
    38, -53, -71, -95, 55, -57, 61, -32, 85, -91, -70, -69, -61, -79, 90, 30,
    -87, -42, -98, -24, -31, 84, 36, 84, 99, -48, 65, 70, 76, -58, 3, 59, 82,
    -28, -88, -94, -1, -28, 47, 88, -77, 14, -97, 52, 80, 24, -9, -59, 88,
    68, 25, -51, -63, -49, 14, -68, 85, -13, -84, 5, 95, -17, -81, -74, 1,
    -55, 32, 98, 32, -77, -84, 91, -65, -64, -83, -74, 19, -9, -45, -94, -28,
    -38, -88, -68, -55, -15, -16, 18, 52, -53, 69, -69, -81, -71, -97, -20,
    -6, -2, -88, -11, 94, -11, 46, 91, 59, -5, -4, 94, 18, 81, 34, -30, -28,
    -2, 12, 67, 40, -75, 39, 72, 13, -10, 28, -38, -76, 75, 91, 41, 92, -23,
    -93, 39, -3, -93, -56, 90, -55, 53, -100, 20, 57, 24, 37, 57, 18, -1,
    -46, -11, 8, -3, 77, 68, -27, 1, -74, -28, 71, -16, 85, -20, 4, -100, 16,
    56, 63, -97, 48, -94, -11, 82, -60, -39, -45, -71, -58, -31, 19, -78,
    -11, -51, -95, 20, 2, 6, -99, -58, -38, 12, -53, 84, 12, -36, -69, -27,
    -3, -21, 81, -85, 70, 4, 24, -55, 5, 87, -21, -33, 3, -44, -25, 12, -1,
    -100, 62, 17, -92, 46, -74, 44, -50,
  -94, 46, 15, 45, 81, -44, 3, -46, 89, -59, -21, -12, 93, 70, 90, -13, -5,
    52, 40, -93, 81, 47, -58, -26, 52, 15, -35, 93, -53, -21, 45, -8, -60,
    69, -74, 75, -30, 1, 49, 19, -29, 98, -13, -40, 13, 57, -62, 97, -14,
    -33, -78, -12, -31, -20, -51, -41, 79, 67, -38, -60, -28, 12, -86, -36,
    -100, -99, 51, 44, 85, 78, 80, -5, -42, -52, -85, -2, -43, -27, -34, -98,
    25, -58, -90, 10, 32, 25, -72, -65, 38, 66, 99, 32, 44, 79, 63, -67, -9,
    -81, -87, 83, 74, 92, 26, 91, -21, 81, 79, -65, 5, 25, -64, 81, 28, -17,
    93, -47, -95, 89, -9, 70, -92, -50, -98, -9, -99, -97, -65, 14, 97, 23,
    24, 82, -67, -19, 26, 80, 44, 62, -75, 44, -89, 50, 51, 77, 4, -4, -99,
    -49, -9, -90, 43, -38, 52, 68, 63, -84, -67, 0, -66, 75, -64, 25, 39, 52,
    -75, -100, -94, 60, 13, 35, 84, -6, -68, 84, -18, -84, 32, -75, -61, -8,
    87, -14, -67, 74, 74, -93, -63, -77, -97, 85, -76, -76, 25, 84, -24, -23,
    -53, 67, 49, 95, -95, 46, -40, 93, -12, 68, 2, 61, 85, -27, 46, 3, 77,
    -23, -93, 81, -99, -73, -35, -99, 39, -15, 46, 6, 93, -5, 16, 3, -92, 52,
    93, 70, -44, 71, 23, -23, 70, 48, -32, 35, 85, 55, -52, -39, 1, 9, -69,
    -81, 8, -72, 59, -47, 72, 44, -99, 84, 33, 99, 74, 96, 64, 4, 44, 33, 84,
    93, -59, -92, 32, 56, -46, -23, -66, -75, -7, 38, -96, -60, 47, -29, -62,
    56, 66, -78, -81, -15, -12, 96, 31, 73, 82, 30, 39, 46, 75, 49, -47, -58,
    -35, -19, -43, -69, 53, 18, -12, 26, 31, 9, -44, 37, -2, -53, 13, 66,
    -17, -18, 14, 36, 73, 94, 36, 36, -11, -62, 25, -92, 20, 17, -7, 63, 49,
    72, 23,
  13, -36, 22, -76, 41, -83, 1, -65, 64, 62, -62, -84, 7, 56, 76, 0, 39, 51,
    93, -31, -23, -66, 55, 99, -30, 38, -99, 12, -5, 31, -25, 22, 6, 43, 18,
    -32, -2, -72, -61, 37, -10, 36, 12, 46, -31, 77, 6, -99, -31, -91, -97,
    94, -78, 87, 86, 54, 1, 50, 52, -93, -18, 5, 2, -50, 6, -57, -72, 63, 78,
    19, -68, 32, 13, 96, -63, -76, -68, 42, -2, -36, -25, 37, -23, -24, -52,
    -63, 9, -3, 32, -15, -30, 82, -10, -36, -62, 19, 57, -46, 40, 79, 7, 43,
    64, -34, -20, -61, 66, -27, -63, -67, -32, 70, -77, -75, -33, 25, -76,
    -48, -73, 60, 65, 36, 62, 49, -44, -62, -50, 70, 70, -6, -9, 3, 98, 32,
    -50, 96, -40, 83, -17, -85, -90, -3, -28, -74, 29, -83, 65, -5, 83, 61,
    17, -21, 84, -91, -27, 43, -72, 14, -62, -22, 38, 80, -60, -11, -68, 56,
    -20, -45, -32, 22, -99, -1, 20, -9, 49, 99, -9, -100, 67, -89, 45, -43,
    -52, 89, -21, 79, 27, 55, 57, -37, 29, 35, 69, -77, 32, 9, -30, -21, -23,
    79, -38, -45, 20, 65, -68, 11, 49, -11, -26, -5, -18, -97, 88, 34, -63,
    4, 79, 70, 59, 50, 86, 39, 68, -95, 63, 11, 47, 37, -28, -53, 31, -71,
    -62, 88, 28, 90, 38, 17, 80, 33, 38, -64, -36, 34, 12, 54, -52, -7, -54,
    79, -17, -53, 87, 70, -85, 56, 57, -83, 45, -46, 45, -51, -40, 34, 51,
    -92, 96, -11, -51, 38, 72, -64, -76, -24, 24, -3, 16, -80, 45, -3, -60,
    5, 34, -69, 56, -29, -22, -57, 55, -18, -21, -48, 31, -96, -45, -6, 34,
    -19, 97, 83, -15, 76, -6, -74, -14, -66, 93, 11, 57, -67, -30, 24, 57,
    -17, -32, -98, 62, -25, -34, -94, 57, -91, -72, -57, -80, 59, -43, 97,
    27, -64, -36, 1, 2,
  17, -75, 21, 65, -37, -99, -21, 17, -11, 61, 88, -35, -79, 14, -27, -18,
    -91, 91, 26, 48, 5, -58, 45, 78, 52, -27, -90, -69, -56, -68, -35, -90,
    59, -59, -75, -43, -81, -38, -33, -20, -49, -17, -3, 24, 57, -85, -59,
    93, -87, -56, 89, 49, -79, 63, -56, -83, -96, 50, 25, 73, 27, -46, 70,
    55, -83, -98, 86, -48, 69, -67, -62, 26, -97, 73, -22, 63, 76, -75, 2,
    -72, 32, 51, -57, -39, 20, 44, -92, 84, 34, 55, -100, -68, -6, 68, 31,
    -7, -86, 28, -84, 47, -79, 54, -43, 69, 58, -58, 87, -10, -12, 14, -29,
    37, 24, 68, -93, -35, 75, -67, -4, 74, -89, 29, 56, -51, -1, -37, -90,
    -91, -75, 14, 24, -30, -73, 65, -1, 53, 9, -16, 68, 40, -88, 91, -80, 71,
    -59, -27, 64, 7, -2, -33, 79, 51, -79, 13, 85, 8, 18, 2, 79, 50, -8, 4,
    -91, -62, 43, 4, 40, -83, 2, 78, -38, -53, -35, 5, 55, -5, 78, -60, -34,
    -4, 22, -70, 43, -66, 89, -73, -45, 74, 17, -2, -42, 19, -55, 33, 59,
    -33, -31, -8, -92, -100, -60, 90, 0, -48, -100, 87, -28, 95, -32, -57,
    -76, 76, -95, -22, 16, -86, 23, 26, 45, 4, -3, 57, -48, -17, 19, -34,
    -95, -83, 40, -40, 13, 74, -62, -88, -84, 67, -80, 40, 0, -9, 55, -62,
    69, -39, 74, -41, -27, -58, 69, 47, 28, 79, -18, 40, 68, -57, 69, 56,
    -54, 51, -85, -45, -76, 34, 65, -56, 35, -89, -66, 95, -86, 55, -92, -94,
    -12, -20, -99, 51, -76, 57, 59, -87, 74, 7, -77, -24, -81, -77, -41, -19,
    47, 18, -89, 11, 73, 59, 77, -82, 99, -97, 21, -96, 43, 97, -83, 93, 22,
    -25, 21, -27, -26, 11, -58, -8, 49, -73, -89, -81, 13, -39, 30, 15, 73,
    68, 98, -83, 45, -82, -81, 64, -26, -63, 89,
  -15, -98, -90, -73, -78, 57, -99, 21, 21, 39, -42, -2, 60, 74, 4, -13, 90,
    -79, -89, 86, -5, -94, -75, 48, 74, 11, -32, 62, -37, 91, -66, -41, 48,
    67, 95, -89, -43, 9, 77, -63, -26, 42, -83, 7, 75, 19, -33, 63, 56, 67,
    -15, -88, 66, 47, 55, -63, -48, 75, -28, 84, 50, 85, -28, -15, 10, 39,
    -100, 98, -92, -11, 6, 44, 16, 81, -67, -1, 43, -73, 3, 73, 49, 33, -53,
    3, 83, 20, 1, -39, -73, -86, -48, 40, 67, -33, 48, -91, 1, 44, -76, -92,
    -41, 80, -10, -21, -57, 51, -78, -72, 16, 80, 80, -71, -21, -85, 77, 7,
    -28, 32, 17, -56, -88, 26, 37, 61, 53, -60, -59, -39, -90, 85, -38, -84,
    24, -17, -87, 31, 15, -74, -51, -52, -62, 71, -88, -14, 71, -23, 25, -61,
    -92, -25, -17, -81, -34, 63, 87, -10, 56, -35, -51, 9, 59, 42, 46, -2,
    -98, 0, 57, 8, 63, -29, 81, 0, 54, 77, -35, -30, -56, -9, 36, 88, -36,
    84, -58, 63, 36, 76, -26, 33, 21, -87, 95, -19, -38, -37, -4, -82, 46,
    -67, 58, 12, -33, 49, 62, -95, -55, -90, 69, 20, -73, 80, 15, 14, -8, 46,
    -53, -93, 89, -3, -91, -83, -10, -52, 22, 17, 59, 90, 38, -67, -85, 0, 8,
    -1, 13, -67, 81, 13, -61, 73, 64, -42, -19, -84, 12, 18, 47, 33, -19, 3,
    -36, 87, -32, 93, -96, 91, 63, 51, -91, 4, -98, -77, -2, -48, 21, 24,
    -88, 20, 91, -77, -42, 58, 89, 79, 79, 58, -12, 55, 36, 7, 13, 85, 90,
    56, -84, -71, 38, 5, -14, -29, -87, 13, -12, -49, -18, -93, 87, -23, 88,
    13, 24, 31, -83, -86, -31, -7, 75, 64, 72, 82, -36, -83, 54, 50, -73,
    -43, -11, 26, -60, 36, -76, 12, -62, -44, 60, -25, 27, -64, 73, -74, -67,
    92, 32, 82, 32,
  53, 69, -42, -17, 56, 36, 87, -14, 1, -36, 83, 75, 71, 81, -92, -59, 61,
    -45, -23, -68, -26, 29, 45, 70, -35, 29, 52, 86, 63, -55, -51, -8, 88,
    -30, -6, -24, -12, 95, -6, -77, 85, 82, 46, -69, -24, -90, 65, -73, -47,
    -99, 58, 67, 15, -21, 93, 22, 28, -20, -29, -48, -42, -98, -82, 74, 3,
    -42, 6, 12, -44, -61, -68, 49, 2, -17, -78, -33, -26, -67, 74, 90, -68,
    82, -89, 40, 15, -94, 58, -96, 12, 7, -76, 24, -26, -43, -51, -62, -78,
    63, 2, 26, -21, -100, -21, -61, 60, 5, -20, -16, -78, -95, 59, -36, 71,
    -41, 95, 33, -10, 87, 69, 69, -88, -11, 97, -39, -69, -35, -78, -18, 8,
    49, 71, -10, -55, 77, -94, -77, -80, 39, 28, 28, -76, -64, 51, -81, 92,
    -73, 73, 51, -78, -33, 12, -56, 14, 53, -50, -67, 91, 86, 63, 9, 0, -22,
    25, -100, 81, -81, 57, 92, 5, 22, 44, 84, -84, -32, 91, 76, -65, -45,
    -80, 79, 36, 95, -87, -34, 63, -62, 57, -76, -85, -73, 87, -99, -1, 13,
    -54, -87, -24, -55, -34, -95, 13, -67, -85, -69, -84, 30, -29, 41, -85,
    -41, -69, 62, -42, 40, -82, -55, -41, -90, 22, -57, 46, 61, 54, 73, 31,
    57, -43, 45, 48, 34, -42, -47, -91, -82, 46, 68, -18, -19, 78, 16, -63,
    -27, 71, -76, -55, -54, 31, 87, -99, -68, 48, 4, 42, 28, 14, 87, 73, -38,
    30, 5, 67, 52, 71, -53, 40, -95, -52, 19, 68, 39, -56, 59, 19, 98, 47,
    12, 90, -28, 23, 55, 34, 40, 86, -68, 3, 63, 73, 77, -16, 67, -36, -37,
    13, 26, 4, -93, 1, 14, -61, -78, -71, 94, -86, 46, -91, -93, 14, -9, -42,
    83, -98, -90, 0, -90, 27, 51, -81, -21, 98, -99, -18, -40, 58, 64, 80,
    90, -85, 92, -66, 8, 54, 9, 15,
  10, -59, 86, -3, -21, 13, 92, 31, -59, -26, 69, 82, 93, -16, -32, 2, -4,
    -59, 86, -1, 10, -32, -51, 88, 17, -20, -62, -26, 74, 36, 99, 18, -90,
    25, 90, 2, 86, 16, -77, -34, -88, -3, 10, -96, 37, 7, 97, -17, 98, 38,
    31, -85, 37, -63, 54, -95, -63, -83, 90, -16, 71, -39, -29, 35, 45, 36,
    -13, -73, -2, -94, -5, 76, -16, -39, 70, 41, -80, 41, 24, -68, -58, 10,
    98, 1, -72, 10, 82, 30, -54, -82, -87, 79, 71, 6, -74, 72, -28, -16, -10,
    -33, -86, 90, -1, 32, 22, 83, -98, 27, -69, 47, -91, 62, 46, 87, -55,
    -77, -19, 20, 78, 73, -71, 75, 19, -93, -96, 98, 63, -41, 34, -97, 20,
    -39, 5, -69, -73, -56, 94, -93, -78, 58, 66, 20, -16, -71, -78, 88, -81,
    1, 69, 97, 53, -4, 24, -82, 11, -63, -58, 0, 49, 24, 50, -91, 8, 6, -79,
    32, 72, -3, 62, 92, 35, -48, -87, -83, 90, 95, -36, -80, -7, 12, -85,
    -27, 32, -70, -82, 4, 28, 65, 66, 66, 99, 5, -11, -20, -66, 56, 72, -28,
    -72, -48, -72, -9, 44, 18, -28, 24, -84, -38, -14, -32, 90, 45, -7, -5,
    -72, -46, 93, -55, -43, -15, 14, -99, -88, -29, -18, 63, 38, 82, -95, 50,
    80, -41, 28, -40, 44, 72, 17, 48, -87, 61, 32, 75, -34, -65, -75, 38, 92,
    38, -42, 88, 48, -59, -60, -39, 71, 13, 75, 25, 31, -50, 7, -51, 60, -93,
    -60, 19, -33, 38, 43, -98, -70, 72, 56, -13, -71, -54, 68, 24, -46, 85,
    47, 58, -30, 19, -92, 17, -72, -53, 44, 33, -36, -60, -15, -10, 44, 5,
    19, 54, -12, -15, 35, 67, -9, 63, 60, 37, 46, -2, 63, 96, 83, -57, -58,
    -68, -85, 76, 87, -93, -51, 18, 97, 82, -50, -13, 4, 96, 99, -87, 50, 29,
    94, -31, -44,
  -16, -34, 55, 0, 99, 79, 18, -98, 97, 8, 6, -74, -21, 17, 6, 28, -27, 54,
    36, 85, -43, 18, 73, 94, 52, -14, 54, -21, 15, -92, -90, 13, -74, -9, 6,
    -21, -37, -75, 5, -5, -81, -85, -73, -10, -72, 90, -29, 68, 87, -13, -45,
    -14, -64, 4, -37, 42, -87, 97, 11, 49, 86, 40, -37, 7, 10, 11, -16, -85,
    -54, 57, -13, -25, -17, 97, -56, -88, 98, -13, 42, -38, 27, -29, -87,
    -34, 58, -49, 25, -32, 96, 56, -20, 6, -6, 85, -83, -36, -51, -80, -39,
    53, -51, 11, -91, 40, 21, -22, 8, -69, 26, 26, -17, -30, -40, 76, -15,
    -43, -55, 22, -67, -86, -77, 3, 74, -66, -64, -24, 54, -30, 12, 23, 20,
    -16, -15, -77, 4, 2, 54, -57, 24, -94, -100, -7, -60, -41, 7, 69, -55, 8,
    -32, 2, -46, 99, -51, 40, 85, -16, 92, -18, -46, -42, 72, 95, -64, -53,
    23, -44, 71, 86, 30, -86, 75, -2, -67, -97, -65, 76, 8, -92, -59, -100,
    -42, 34, -36, -39, 99, -33, -71, 16, -8, -15, -88, 12, -31, -26, 91, -59,
    69, 65, 34, -94, 14, 0, 12, 75, -7, 1, -35, 21, 8, -59, -29, -77, -1, 56,
    -63, -80, 31, 72, -83, 63, -42, 76, 4, 10, 33, 30, 20, -49, -8, -52,
    -100, -43, 75, -21, 98, 62, -65, 60, 5, 54, 8, 89, 78, 60, -51, -4, -24,
    16, 2, -9, 87, 45, -54, -95, -10, -9, -8, -96, 70, 93, -62, -67, -16,
    -72, -69, 56, -43, -63, 1, 46, -94, -33, -4, -46, -33, -50, 15, 67, -79,
    81, 94, -67, 38, 71, 0, 31, 66, 53, -90, 31, -81, -65, 81, -64, 7, -25,
    55, 87, 79, -87, 19, -22, 22, 36, 21, -38, -61, -58, -70, 96, -93, 26,
    62, 16, 4, 53, 39, -17, 35, 32, -42, 38, 9, -65, 6, -28, 54, -64, -23,
    -94, 8, 19, 76,
  17, -97, 2, -97, 2, 71, 86, -25, 71, 97, -67, -77, -27, -63, 59, -81, 25,
    -59, -43, -38, 65, 98, 10, -63, -58, 66, 63, -88, 94, 59, 24, -29, -58,
    -23, 74, 20, -44, 22, -92, -57, -10, -55, 36, 29, -9, 3, -41, 68, 82, 79,
    -71, 80, -65, 5, -32, 73, -90, 50, 26, -61, 90, 22, -39, 45, -64, 64,
    -80, -15, 85, -96, -42, 66, 50, -80, 38, 79, -40, 70, 44, -12, -78, -16,
    45, 6, -55, 55, -56, -97, -89, -43, 42, 13, -32, -83, -61, -82, 71, 48,
    5, -78, -35, -96, 13, -39, -41, -6, -68, 38, -90, 90, -86, 14, -55, -97,
    -87, -4, 19, 68, -65, 72, 64, 21, -51, 25, -93, -35, -51, -41, -35, -74,
    11, -5, 51, -52, -96, 26, -80, 21, -97, 0, 3, 13, -53, -53, 25, -63, -41,
    -38, 79, -3, 47, 10, -26, 15, -83, 72, 90, -25, 11, -9, -82, -17, 71,
    -74, -49, -67, -7, 41, 6, -21, 40, -12, 95, 79, -62, 41, 58, -19, -55,
    -87, 28, 9, -24, -93, 18, -85, -92, 50, 55, 33, -74, -51, -25, 21, 95,
    86, -66, 79, -32, -53, 95, -24, 64, 99, 50, 45, -68, -63, 12, -66, 29,
    -11, 37, -70, -3, -62, -23, 48, 77, -17, -98, 72, -76, 47, 45, -16, 30,
    12, 80, 66, -54, 89, -56, -24, 9, -52, 10, -17, 66, 53, -71, -65, 53,
    -85, -31, 10, -1, 40, 65, -58, 8, 42, -66, 36, 93, 37, -47, 74, -16, 96,
    77, -36, -80, -79, 18, -33, 89, -88, 35, 93, -53, 43, 39, 30, 4, 32, -86,
    -5, -65, 8, -77, -6, 32, 30, -18, 44, -65, -47, 21, 3, 76, 8, -75, -45,
    5, -54, -44, 56, 85, 11, -40, 58, -61, -78, -57, -99, -28, -88, -21, -35,
    79, 86, 29, -14, 71, 21, 46, 58, -51, -76, 85, -44, 53, 52, 31, -25, 81,
    -70, -71, 86, 71, 25, -3,
  88, 65, 32, -33, -83, 93, 94, -46, -23, 62, 77, -77, 42, 0, -76, -20, 59,
    -7, 59, -10, 88, -94, -69, -8, -34, 52, 36, -67, -62, -65, -83, 52, 53,
    56, 83, 30, -8, -83, 88, 28, -98, 41, 17, 29, 7, -52, -80, -85, -13, 28,
    -29, 81, 62, 10, -24, -44, 60, -82, 1, -91, -79, 35, -59, 66, 69, -48,
    42, 22, 71, -21, 7, 9, 23, -58, 80, 29, -17, -54, 2, -25, -15, -31, -55,
    -50, 77, 94, 15, 92, 15, -69, -60, -2, -79, -47, -47, 45, -31, 7, -39,
    41, -3, -30, -75, -59, 18, -31, 44, 90, -79, -43, -30, -51, -80, -14,
    -40, 45, -75, 99, 5, -98, 52, 88, -43, 26, -37, 47, -18, 72, -15, 68, 24,
    96, -57, -60, -9, 61, -35, 97, 74, -38, -25, 81, -57, 46, -26, -6, 72, 2,
    -57, 16, -74, 71, 47, -48, 85, -63, -21, -28, 82, 0, -76, -76, -7, 25,
    -24, -83, 37, -87, -37, -53, 65, -50, -88, 59, -96, -42, 7, 18, 18, 18,
    -99, -53, 16, 29, -89, 75, 72, -20, 25, -78, -5, 58, -48, 63, -21, -84,
    68, 20, -11, -74, 93, 88, -78, 61, -78, -74, -43, 68, -100, -2, 93, 67,
    4, -60, -52, -12, 83, -25, 22, 87, -17, 98, -58, -54, 94, -90, -61, -96,
    -14, -66, -20, -32, 21, 78, 53, -100, 22, 1, -86, -64, -95, -44, -38, 75,
    -87, -89, -72, 70, -95, 64, 11, -82, -10, -42, 85, 32, -49, -29, -2, -61,
    -38, 20, 4, -7, 56, -68, 95, 22, -9, 39, 43, 16, -63, 45, -18, -30, 8,
    13, 12, 65, -5, -87, -95, 46, 33, 54, 39, -62, -95, 31, 67, -17, -16,
    -49, -59, 27, 44, 23, 85, -11, -32, 95, -21, 66, -84, -75, 91, 61, -93,
    -80, 36, 28, -32, -35, -69, -93, 8, 18, 37, 77, 34, 82, -42, 75, 56, -46,
    -37, -90, -33, 35, -25, -46, -66,
  72, -93, 33, 65, -13, 6, -44, -88, 21, -33, 5, 21, 14, 20, -47, 75, -37,
    94, -94, -27, -37, 74, 9, 94, 56, 91, -39, 0, 81, 3, -82, -96, -93, -32,
    -100, 97, 50, 93, -9, 13, -37, -88, 39, -31, -100, -38, 98, -57, -82, 88,
    -11, 96, 21, -90, -89, -8, -24, -43, 3, 71, 1, -32, -40, -78, -52, 42,
    -84, 18, 41, 13, 59, -86, 17, 2, 64, -50, -89, 41, 71, 23, 84, 38, 90,
    -43, -100, 9, 26, -24, -36, 55, 98, -90, -35, 12, 33, 20, 63, -16, 89,
    -9, 68, 90, -28, -68, -47, 78, -44, 83, -12, -12, 51, -29, 33, -53, 8, 4,
    -91, -58, 9, -35, 80, -2, 20, -16, -51, 73, 75, 90, -58, 93, 37, -75, 92,
    44, -56, 47, 88, -3, -83, 45, 30, 41, 20, -14, 97, -3, -32, -97, -85,
    -88, 50, 2, 4, 64, -55, 86, 1, -53, -89, 55, -29, 5, -76, 6, -25, -10,
    -70, -77, 30, -19, 1, -20, 75, 16, 11, 41, -21, 72, -99, 33, -64, 45,
    -95, 82, -81, 55, -65, -60, -86, -30, 65, 5, 95, -67, -60, -52, -79, 39,
    -4, -91, 54, 75, -87, 8, -12, 15, 10, 56, -86, -97, -23, -68, -8, -77,
    65, -93, 30, -50, -64, -16, 69, 12, -32, -50, 95, -34, 25, 55, -85, -98,
    69, 88, -72, 65, -38, -38, 19, 67, -10, -80, 52, 41, -22, -67, -59, -98,
    -86, 62, -54, -81, 82, 73, -28, -11, -55, -2, -37, -3, 46, -66, 2, 16,
    -64, -72, -76, 4, 44, -98, 5, -26, -42, -86, -99, 62, 32, -93, -38, -46,
    -67, 38, 43, 49, -55, 40, -86, 40, -46, 49, -29, 34, -85, 62, -34, 60,
    53, -33, 70, -50, 24, -48, 36, -71, 78, -39, 64, 28, 93, 84, -66, -31,
    25, 85, -50, 38, -82, 14, 68, -94, -47, -61, 85, 86, -6, 87, 68, 73, 92,
    92, 62, 86, 25, 56, 55,
  -33, -45, -63, 80, 95, 79, -37, 7, 80, 68, -95, -3, 90, 94, -50, 53, -63,
    -66, 53, -61, 6, 38, -94, 55, -57, -72, -72, -66, 30, 5, -42, -47, 18,
    86, 3, 62, -93, -8, -74, 19, 2, 28, 76, 96, 26, 82, -74, -50, -39, 71,
    19, -63, -63, 71, 74, 12, -50, 79, -70, -95, -35, 10, -6, 92, -7, 38, 51,
    -36, 5, -40, -42, -73, -84, 18, 84, -49, 24, 65, -92, 77, 72, 14, 22, 78,
    -85, 26, -59, -93, 35, 65, 5, 52, 89, -34, -48, 35, -37, -1, -48, -18,
    81, -8, -31, -10, 26, 87, -2, -91, 47, -40, 30, 48, -82, -22, 50, 90, 25,
    81, 23, -80, 29, -25, 44, -92, -25, 3, -50, -61, 87, 20, -84, -94, -55,
    -34, -85, 14, 90, 72, 98, 14, 88, -42, 91, 59, -81, -39, -52, 34, -9, 97,
    2, 21, 75, 63, -26, -4, -70, 82, -41, 40, 38, -13, 52, -1, 17, -35, 42,
    5, 42, -1, 57, 14, -36, 68, 68, -99, 27, 11, -62, 84, 86, 60, -55, -62,
    12, 79, 90, -79, -32, 30, -97, 79, 78, 50, 89, 74, -7, 25, 29, -97, 27,
    43, -23, 33, 89, -84, -93, 91, -51, -96, -25, 8, -71, 26, 54, 53, 94,
    -51, 13, -1, 70, -21, -11, 77, -66, -63, 12, -87, -3, 76, 56, 80, -23,
    -83, -99, 33, -74, -12, 34, -14, -59, 77, -43, 46, -45, 87, -68, -16,
    -71, 60, -99, 12, -11, 36, 8, 51, 47, 18, -94, 35, -94, -5, -67, -33, 11,
    35, 40, -67, -52, 28, -34, 80, 86, 28, -88, -72, 34, -31, -13, -20, 38,
    30, -80, 37, -37, 28, -44, -73, 31, 2, -28, -66, 59, -83, 57, 55, -59,
    25, 91, -26, 40, -58, 18, -75, 22, -1, -2, -22, -72, 14, 62, -25, -80,
    -22, -54, 51, 50, 63, -37, -8, -67, 81, 93, -18, 83, 59, -26, -74, 44,
    59, 40, -40, 39,
  -97, -8, -25, -21, 12, -63, 93, -21, 72, -78, -25, -49, -70, -76, 84, -4,
    -4, 2, 9, 90, -68, 22, 54, 33, 89, 73, 46, 8, 55, -65, -44, 86, -99, -44,
    25, 23, -29, -8, -5, 82, -50, -3, 42, -69, -83, -89, 92, 13, -80, 52, 97,
    -10, -16, 50, -73, 36, -66, -36, 74, 38, -65, 84, 82, 8, -65, 36, 76,
    -80, 3, 80, -41, 84, 3, -53, 81, 26, 51, 96, -75, 45, 19, -94, -28, -99,
    46, -99, -13, 96, 98, 52, -4, -33, 80, -35, -44, 40, 26, -64, -88, -39,
    45, 49, -91, 43, -27, -39, 2, -7, -62, 51, -27, -33, 62, -69, -55, -49,
    83, -71, 74, -9, 27, -17, 52, -17, 7, -19, -84, -32, -43, 66, 49, -56,
    12, 42, 43, 75, -3, 75, 87, 13, 44, 18, -59, 75, 68, 55, -7, -30, -46,
    -13, 24, -48, 55, 89, -69, 8, 85, 56, 63, -58, 47, 12, -91, 62, -42, -99,
    -29, 14, 78, 46, 13, 11, -69, -9, -22, 38, -81, 4, 31, -16, 34, 58, 72,
    -60, 46, -92, -40, -46, -24, 45, 65, 32, 7, -32, -64, 53, -82, 92, -22,
    -52, 12, 53, 47, -69, 16, 67, 5, -89, 80, -36, 34, -64, -3, -58, -11, 65,
    70, -33, -36, 25, -90, 68, 47, -64, 31, -1, -99, 27, 86, -22, -99, 43,
    50, -67, 73, -40, 61, 27, -96, -18, 61, 23, 6, 68, 57, 96, 55, -15, -25,
    -43, 24, 40, -92, 50, -18, 36, -9, -14, 88, 3, -97, -52, -9, -49, 39, 86,
    63, 88, -100, 46, 70, 90, -56, 5, 5, 92, -55, 62, 68, -19, 61, -16, 6,
    33, -28, -35, 12, 99, -98, 0, -45, 61, 63, 84, 56, -26, 42, 37, -68, -12,
    -62, -14, -58, -90, 88, 5, -71, 60, -53, -86, -29, 41, -51, 77, 66, -31,
    22, -31, -14, 97, -73, 63, -30, -28, -23, -56, 43, 30, -34, 61, -32, -16,
    -57,
  88, -97, -8, 96, 95, -7, -55, 98, 1, -17, 73, -85, -97, -21, 66, 2, 68, 61,
    30, 15, 90, -78, 98, 36, 19, 98, 76, -18, -13, 75, 81, 5, 86, -51, -2,
    -80, 48, -90, 54, -71, 57, 43, -95, -13, -82, -87, -93, -44, 55, 66, 65,
    -57, 36, 49, 97, 73, 39, -31, 59, 97, 31, 60, -100, 43, 70, 69, 80, -40,
    62, 95, 62, 61, 42, -7, -9, -67, -11, -36, 8, -48, -13, -93, 35, 16, 74,
    1, -90, -68, -14, -20, -17, 66, -57, -62, -70, 91, -77, -97, -5, 34, -84,
    94, -77, 77, -37, 13, -54, -3, -20, -77, -61, 98, -29, 46, 43, -30, -60,
    -48, 1, -73, 10, 5, -38, -88, -86, -96, 79, 67, -26, 19, -100, -97, 32,
    -39, -24, 7, 78, -53, -38, -10, 86, -71, -2, 47, 0, -7, 76, 36, -62, -67,
    36, 24, -21, -8, -39, -53, -6, -88, -52, -8, -61, 26, -35, -11, -61, 39,
    39, -49, 29, 46, 92, 92, -29, 9, -20, -18, 85, 29, 9, -11, -83, 31, 34,
    -60, -10, 53, -36, -52, 70, -21, 39, -95, -16, 47, 84, -10, 35, 22, -63,
    76, 48, -98, 26, -26, -77, 52, -15, 60, 48, 47, 83, -53, -32, -83, -19,
    39, -6, -11, -91, -48, 42, 75, -18, -58, -56, 93, -64, 62, -19, -44, 55,
    74, 20, 8, 32, -21, -94, 32, 99, -77, 59, 99, 6, -69, -35, -89, 71, 93,
    -74, -87, 1, -52, -9, 62, 15, 57, 42, -60, -62, -23, 87, -86, -40, 77, 3,
    3, 28, 70, -86, -82, 2, 81, -1, -89, -18, -1, -93, -11, 55, -78, 84, 37,
    -83, 10, 27, -53, -40, -58, -25, 13, 91, 21, 21, 19, 23, 50, 81, 18, 51,
    96, 6, 14, -10, 30, 25, 67, 32, -48, -91, -9, 50, -44, -10, 60, -18, -5,
    -72, -68, 62, 83, 19, 46, 70, 90, 49, 59, 12, -68, -19, -3, -27, -32, 29,
  -40, 25, 56, -90, 69, -98, 64, 12, -71, -5, -80, 52, -44, -23, -90, 75,
    -14, -40, 93, 53, 41, -15, -84, -14, 24, -65, -42, 22, -12, 42, -42, 72,
    30, 81, -98, 36, 57, 94, -80, -13, -35, 8, 84, 57, -64, 54, -91, -46, 78,
    24, -6, 37, 55, 92, 97, 78, -7, -13, 88, 43, -1, 6, 32, 61, -40, -34, 91,
    -30, 82, 41, 15, -81, -99, -86, -20, -27, 47, 25, 86, 87, 0, -88, 88,
    -72, -9, -15, -57, 25, 65, 66, -30, 69, -81, -73, -92, 22, 35, 62, 11,
    36, -35, -69, 6, 54, 20, 99, 65, 20, 12, -6, 95, -54, 59, 2, -45, -29,
    -4, 87, 73, -87, 4, 68, 46, -93, -74, -8, 54, 41, 24, 78, 41, -55, -55,
    38, 30, -97, -73, 85, -65, -9, -27, 63, -85, -58, -62, -60, 91, 84, 31,
    72, 46, 65, -45, 48, -86, 53, 47, -37, 88, -5, 97, -62, 31, -38, 62, 64,
    -1, -77, 99, 19, -71, 25, 45, -69, -51, -92, -55, 65, 5, 93, -44, 98,
    -40, -72, 54, 97, 80, -20, 50, 14, 35, 83, 67, -12, -77, -29, 59, 2, -32,
    4, 10, 10, 92, -54, -43, 66, 33, 36, -8, 60, -49, 13, 21, 1, -89, -39,
    31, 49, 82, -31, 39, 42, -4, -61, 94, -52, -24, -97, 91, 87, -16, -89,
    -98, -54, -9, 88, -94, 10, -18, -28, -6, 43, 91, -54, 40, -16, 30, 98,
    52, -65, 43, -6, 26, 40, 88, -41, 66, -57, 48, 9, 91, -37, 42, -12, 93,
    21, -73, 4, 74, -57, 43, -85, -25, -31, 78, 69, -50, 10, -68, -6, -40,
    -26, 19, -50, 82, -87, -27, 55, 1, -81, -69, -84, 10, 3, -67, 44, 85, 47,
    -63, 55, 34, -9, 35, -5, 18, -1, -93, -5, 73, 0, 15, 95, -37, -27, -41,
    -57, 54, -84, -55, 56, -25, -66, 14, 3, 15, 97, -87, 58, -65, -72, 23,
    -21, 97,
  -29, -76, 35, -21, -78, -68, -4, -97, -26, -70, 54, 34, 5, -32, 43, 40,
    -59, -42, 66, 22, -36, 59, 94, -33, -6, -41, -73, -34, 10, -59, 95, 85,
    54, -100, 84, -90, 44, -66, -42, -28, -18, 10, 96, 56, -34, -36, -13,
    -81, -11, 48, 47, 37, 73, 83, -68, 27, -8, -19, 97, 92, 96, -65, 64, 1,
    -81, 20, 33, -6, -19, 38, -81, 91, 82, 98, -42, -30, 43, 42, -87, -58,
    -75, 89, 61, -24, -84, 83, 23, 20, 5, 12, 84, -93, 24, -25, -47, 67, 38,
    -6, 87, 40, 12, -56, -50, -27, 2, -87, -57, -3, -42, 59, -70, -72, -96,
    -75, 53, 81, -36, 44, 82, -24, -51, -14, 81, 58, -31, 27, 55, 40, 42, 39,
    23, -15, 62, 2, -3, -59, -76, -95, 21, 18, 84, -31, 3, -4, -1, -80, 94,
    79, 17, -94, -39, 59, -72, -40, 48, -94, -58, 98, -86, -11, 33, 88, -15,
    99, 75, -93, -88, 43, 70, -30, -42, 52, -79, 71, -98, 36, -37, -83, -98,
    69, 57, 12, 74, -73, -21, 48, 66, -75, 99, 61, 59, 98, 39, -25, -71, 26,
    -8, 35, -65, 0, -69, -44, 21, -95, 32, -19, 42, -100, -22, -77, 24, -39,
    57, -38, -33, 3, -46, 74, 44, 50, -45, -53, 49, 48, 91, 54, 95, 43, -57,
    98, -48, 84, -70, 40, -78, -6, 77, -59, -52, 69, -54, 84, -20, -37, -1,
    -100, 25, 27, 49, 43, -9, 23, -86, -14, 39, -32, 98, 54, -75, 95, -43,
    94, -38, -47, -83, 83, -7, 45, 27, 43, -24, 12, 21, 69, 75, 94, 15, -49,
    -93, 32, 4, 35, 19, 86, -1, 59, 43, -43, 13, -1, -93, 12, -52, -48, -39,
    81, -100, -55, 38, -49, -86, -89, 80, -39, 11, -75, -31, -72, -61, -71,
    -32, -12, 86, 91, -50, -87, 54, 18, -56, -28, -31, 3, 18, 67, -30, -51,
    -14, 0, -84, 33, 21, -99, 98,
  12, -96, 74, -40, 91, 13, -88, -78, 31, 99, 9, 39, -68, -6, -7, 72, -32,
    -22, 74, 68, 14, 48, 30, -61, -18, 34, -16, 29, -19, -12, -36, -93, -95,
    76, 60, -73, -21, -63, 76, 39, -37, -52, 62, -34, 15, -43, 72, -1, -83,
    -94, -66, -52, 7, 21, 24, -44, 70, -62, 57, -60, -38, -52, 35, 65, -33,
    -87, -18, -50, -22, -5, -30, 3, -33, -14, -53, -98, 15, 1, -98, -100,
    -100, -70, -39, -36, 66, -85, 24, -43, -5, -42, -1, 74, -90, -86, -71,
    -55, 16, -64, 21, 66, 34, -43, -98, -66, -92, -46, 24, -79, 88, 51, 73,
    -59, 47, -58, -97, -64, 92, -52, -93, -22, 18, 97, 19, 71, -85, -69, 35,
    -58, -57, -64, -21, -59, -98, 57, 41, 94, 8, 59, -85, 0, 67, -18, 61,
    -17, -44, 59, -75, -58, 10, -67, -37, 98, 22, 0, 57, -79, -57, -93, 96,
    33, 69, 12, 5, -20, -70, 17, -52, -63, -60, 77, 19, -42, 76, -2, -49, 18,
    85, 25, -69, 62, 62, 97, -80, 27, -34, -64, -96, 32, 41, -9, -76, -74,
    46, 51, -69, -52, -73, -56, 70, -67, 50, -86, -8, 57, -12, 27, 47, 15, 0,
    93, 62, -72, -80, -95, -62, -55, -92, -47, 58, -92, -76, 87, 73, 76, 23,
    21, 2, -20, 46, 97, 64, -59, 25, 37, 19, -37, -86, 15, -51, 19, 85, -8,
    -23, 43, -13, -97, -62, 64, 70, -97, -17, -82, 32, 59, -19, -56, 91, -22,
    -24, 3, -96, 36, 82, -100, 34, 7, 85, 78, -60, 97, -92, -99, -74, -86,
    49, 80, 6, 65, -40, -1, -3, -38, -60, -25, 31, 28, -71, 85, -80, -32, 97,
    -59, 47, 64, 62, 80, -19, 66, 22, -73, 72, -35, 51, 38, -88, 75, 97, -19,
    49, 51, 5, -11, 18, 99, 76, -42, 13, -84, -57, 39, 4, 71, 70, -22, 0, 84,
    -76, -100, -85, -10, 37, -75, -32,
  -37, 70, 8, -88, -17, 48, 36, 82, 70, 95, 95, -27, -51, -28, 52, -37, 83,
    54, -55, -48, -42, 6, -75, 13, 2, 29, -55, 95, 27, -66, 38, 3, -97, 40,
    90, 80, 31, -52, -52, -12, -41, -79, 14, 21, 11, -11, -21, 48, -7, -17,
    79, -10, -17, -12, -43, 31, -75, -10, 55, -61, 8, 95, 7, -78, -20, -77,
    -5, 66, 73, -2, 10, 33, -66, 95, 69, -77, -17, -18, -4, -7, 53, -58, 64,
    -98, 9, 57, -38, 70, 75, -95, -72, 88, -76, -15, -17, -40, -95, -51, -30,
    -84, -48, 49, -78, 67, 4, -82, -36, 98, -36, 18, 53, -49, 26, 15, -4,
    -82, -55, -43, -99, -9, -16, -47, -17, 38, -14, 90, 1, 87, -65, 50, -44,
    -46, -15, -14, 76, 7, 42, 7, 23, 72, 10, -83, -55, 18, 30, -63, 78, 5,
    -55, -94, -40, 73, -64, 9, -100, 2, -51, -35, -87, 9, 68, -14, 96, -52,
    76, -45, -33, -70, 47, -53, 61, -55, 46, 51, -65, 74, -9, -85, -16, 12,
    19, 68, -26, -90, -10, 3, -97, -49, -67, 91, 96, 31, -8, -53, 39, 69,
    -34, -9, -56, 52, -81, 38, 75, -55, -69, 2, 72, -55, 39, 94, 70, -89,
    -80, -54, 33, -20, -80, 10, -21, 43, 44, -58, 28, 54, 48, 85, 36, 81,
    -17, 94, -58, 36, -74, -76, -69, -50, -100, -54, -51, 39, 7, -77, -23,
    -54, 79, 85, -85, -96, -21, 47, 40, -32, 32, -7, -69, -51, -46, 39, 87,
    -79, -49, -19, 30, -11, 31, 2, -64, -17, 44, 44, -45, 59, 91, 31, 51,
    -63, -48, 38, 70, 89, 90, 66, 19, 26, -43, 80, -46, -69, 71, 78, -38,
    -45, 94, 95, 43, 48, -74, -6, 23, 62, -50, -90, -19, 74, -81, 87, -81,
    69, -74, -19, -100, 85, -24, 32, -20, 22, -56, -87, 40, 55, -36, -33,
    -20, -30, 4, -18, 4, -71, -88, 42, 80, -46, -48,
  7, -17, -100, -54, 20, -86, 25, -83, -83, 44, -50, -49, -18, -45, 15, 81,
    21, -23, 44, -16, -39, 21, 39, -74, 53, 34, 57, -94, 19, -66, -60, 81,
    -82, 7, 84, -91, -82, 29, -77, -11, 9, 48, -24, 13, -10, 56, 63, 44, 13,
    48, 55, 92, 23, 67, 88, -77, 73, 5, 99, 50, -90, 79, -12, -23, 18, -72,
    -52, 36, -90, -52, -23, -23, 62, 55, 15, -64, -52, -92, -34, -17, 29, 45,
    58, 17, 26, 57, 30, -70, -68, -81, 90, -70, 8, 15, 43, -30, 99, -37, -86,
    -65, -4, -24, -29, -31, 53, 39, 86, 92, -39, -42, -84, -58, -6, -64, -88,
    16, 77, 51, -18, 17, -68, 12, -26, 51, 53, -98, -49, 34, -53, -82, -57,
    29, 77, -8, 35, -92, 35, 99, 96, 6, -91, 88, 64, 24, -52, -82, 15, -77,
    -82, 70, 7, 18, -15, -31, 75, -92, 91, 22, -32, -5, 58, -21, -34, 36,
    -76, -61, -34, 71, -89, -5, 78, -1, -5, -17, 8, -13, 56, 92, -81, 12, 52,
    -47, 95, -13, 91, 15, 8, -52, -20, 8, -37, 11, -63, -62, 9, -65, -95, 34,
    -80, -2, -53, 82, -21, -15, 35, -36, -22, 45, -79, 37, -55, 0, -58, 3,
    68, -36, 58, 94, 16, -24, 67, -99, 39, 69, 65, -34, 98, 44, -87, 32, 24,
    -9, 81, -66, -42, 63, -11, -99, 68, 59, -96, 43, 58, 48, -32, 79, -73,
    -17, 40, -16, 50, 27, 78, 93, -14, 28, -82, 75, 62, 47, -44, 27, 72, -74,
    -78, -65, -29, 64, -51, 61, -91, -69, 3, -62, 82, -67, -65, 17, -91, -76,
    57, 68, 97, -38, 69, 9, 73, 19, 88, 65, 16, -75, -21, 42, 77, -47, 71,
    14, -45, -71, -14, 3, 50, -84, -29, 66, 26, 19, -70, -3, -1, 62, 36, 7,
    91, -71, -29, 75, 80, -4, 96, -79, 84, 53, -85, -84, -19, -20, 63, 22,
    80, 38, -73,
  84, -42, 26, -32, 20, 80, 41, 67, 10, -66, 98, 15, 92, -57, 69, -28, -87,
    -37, 65, -53, -15, -73, -88, 97, -34, -52, -60, -72, -62, 61, -37, -15,
    -61, 46, -39, -47, -31, 11, -38, -81, -5, 82, -34, -27, 25, 38, -59, -50,
    32, 53, 81, -50, 1, 80, 82, 11, 80, 19, 41, -22, 78, -41, -49, -35, 7,
    -92, -1, -55, -18, 34, 45, -71, 44, 35, 72, 0, 72, 76, -22, -29, 41, 96,
    -25, -24, -10, -69, 40, 65, 93, 80, 40, -53, 7, -32, 34, -1, 85, -91, 7,
    40, 22, 31, -93, 44, 82, -78, 18, 23, -84, -60, -23, -32, 67, -98, -79,
    -54, -70, 49, -70, 29, -33, -42, -25, -37, -41, 30, 88, -62, 53, 0, -17,
    22, -36, -65, 30, 45, -20, 41, 38, -34, 45, 87, -3, 28, 89, 51, 57, 27,
    -67, 49, 60, -98, 5, 61, 8, 73, -46, 15, 18, 32, -92, 23, -67, -78, -89,
    -68, 45, -20, -86, -81, 76, 30, -97, 12, -100, -78, 87, 5, -75, 52, -49,
    81, 61, 31, 48, 90, 58, -59, -1, 85, 68, -17, -83, 40, -94, -28, 98, 93,
    -16, -28, -77, -29, -39, -22, -21, -45, 55, -9, 68, -9, -83, 78, -57, 34,
    27, -55, -31, 6, -77, -100, 13, 81, 56, 54, -31, 9, -13, -88, 73, -100,
    28, 12, 60, -62, 27, -53, 19, -86, -30, -78, 72, 49, 29, -96, -12, -20,
    -82, -52, -39, -50, -7, -90, -26, -52, 90, -12, 22, -41, -98, 4, 41, -5,
    -57, -10, -27, 79, 24, -7, 7, -12, -20, 89, 27, -93, 73, -10, -41, 42,
    -6, -75, -73, -35, -42, -26, 7, 11, 33, -54, 69, -73, -30, 4, -14, -37,
    72, 35, 76, -68, -18, 10, 44, 2, 25, -48, -91, 92, -20, 61, -11, 55, -20,
    4, 54, -44, -39, -14, -40, -63, -83, 48, 68, -64, 57, -36, 10, 3, -26,
    90, 24, 40, 7, 4, 58,
  -100, -26, 7, -64, 84, -83, 1, -76, 90, 48, -21, -85, -61, -25, 62, 74, 48,
    24, -18, 96, 2, -20, -63, 33, 42, -26, -39, 62, -62, -64, 63, 39, 63,
    -47, 72, -58, -1, -90, -56, -7, -99, -40, -82, 24, -58, 5, 93, 64, -5,
    94, 90, 90, -19, -46, 89, -90, 92, 33, 6, -12, -23, 50, 30, -77, -33, 43,
    38, 23, -11, -30, -94, -9, -10, -93, 88, 21, 84, -87, -93, 26, 33, -99,
    -34, 55, 15, -60, -36, 62, -51, -12, -93, -37, 53, -31, 35, 44, -82, 57,
    4, -93, 40, -53, -47, 85, -16, 26, 53, -9, -94, 42, -16, -53, 93, -50,
    -89, -21, -45, 83, 91, -12, -27, -75, 52, -15, 82, 33, 6, -43, -50, 44,
    35, 24, -22, 59, 58, -72, -58, 16, -14, 39, -6, -83, -20, -65, -89, 49,
    49, 68, -92, -30, 33, -54, -11, -9, 59, 7, 89, -18, -79, 26, -71, 4, 87,
    -46, -64, -77, -55, -96, -50, 47, 2, 17, -7, 46, 47, -76, -94, -74, 65,
    45, -98, 91, 0, 13, 33, 76, -100, -34, -78, 95, 78, 20, 74, -35, 27, 71,
    16, 95, 92, 3, 17, 11, 58, 99, 13, 22, -35, 5, 37, 90, 75, -16, 13, -31,
    -60, 64, 53, -72, 47, 11, -30, 85, 36, 51, -62, 8, 55, -74, -61, 17, 0,
    34, -42, -66, 53, -99, 25, -55, -10, -64, 39, 3, 61, -28, 18, 69, 72, 21,
    -59, -5, 52, -52, -97, -15, -41, 63, -45, -20, 43, -2, 49, -28, -33, -87,
    30, 5, -41, -74, -71, -21, -63, 19, -74, 95, -66, 86, -2, -53, 55, 47,
    91, 89, 81, 90, -96, -37, 40, -29, -37, -36, 97, 66, -8, -17, 79, 63,
    -51, 30, 31, 35, 91, 81, -87, -30, 17, -54, 58, -35, 32, 70, 51, -27,
    -67, 58, -52, 31, 33, 97, 84, 1, -80, -37, -7, 20, -4, -1, -25, -42,
    -100, 31, 36, 40, -94,
  9, -79, 71, -45, 84, 32, 15, -83, -16, 29, 80, 15, 57, -20, 26, 35, -61,
    -57, 91, 44, -44, -11, 65, -69, 70, -68, 21, -79, 92, 99, 58, -93, 99,
    41, -87, 69, -85, 77, -42, 95, 35, 56, 24, 61, -11, -74, -28, 8, 21, -50,
    -18, 91, -84, -52, 90, 49, 23, 64, 82, 28, -33, 65, -22, 73, 17, 57, -92,
    -86, -95, -96, -65, -74, 93, 64, 0, 73, -85, 98, -24, -53, -17, -99, -14,
    -83, 60, -47, 29, -32, 43, 73, -62, 3, -85, 30, 49, -54, 39, 28, -33, 16,
    -57, -8, 46, 67, 25, 26, 95, -52, 64, -65, -16, 43, 1, -33, -39, -47,
    -95, 6, -96, 25, 87, 97, -3, -90, -31, -34, 14, 96, -96, -17, 49, 8, -80,
    -5, -87, -86, 33, -60, -38, 11, -75, 24, -24, 78, -41, 9, 54, -5, -99,
    24, -80, 71, -33, -88, -54, -40, 33, -99, 54, -4, -89, -53, -28, 25, -17,
    21, 94, 75, -100, -93, 41, -26, -91, 6, 20, -67, -82, -29, -67, 70, -64,
    -36, -56, 25, -58, -32, 84, -45, -89, 92, -53, 87, -68, -1, 61, 55, -19,
    51, -5, -77, 17, -91, -3, 1, -82, -98, -34, -30, 95, -8, -46, -40, -99,
    80, 71, -23, 96, -6, -12, 51, 25, 86, -57, 85, -56, 60, 16, -36, 35, -68,
    -98, 44, 12, 79, -49, -78, -98, -59, -83, -7, -79, 13, 16, 99, 43, -2,
    39, -94, -83, 19, -87, 62, -10, 54, 89, 0, -31, 80, 90, -94, -46, 14,
    -15, -56, -64, 0, -17, -26, 94, 85, -66, -44, 73, -3, 5, 76, -89, -67,
    35, 56, -9, 98, 40, -44, 44, 89, -71, 75, 18, -46, 96, -11, -35, 90, 45,
    -32, 89, -71, 80, 33, -76, -67, -3, -65, -45, 33, 5, -25, -37, -37, 95,
    -100, -88, 72, -8, -59, -97, -8, 17, -5, 25, 15, -9, 47, -10, 55, -75,
    -43, 48, 48, -31, 40, 45,
  6, 86, -79, -52, 43, 57, 29, 68, -53, 9, 20, -39, 71, 36, 42, -30, 80, -52,
    -60, 4, 43, 36, 45, -79, -58, 69, -11, 78, 31, 96, 56, -60, 18, 22, 73,
    97, -66, 28, 48, 68, -19, 70, -56, 82, 54, 91, 2, 24, -69, -49, 23, 23,
    -29, -80, -8, -27, 5, -54, -6, -88, -57, -73, 87, -29, 62, -56, 32, -34,
    -7, -47, -49, -15, 52, 69, 40, -73, 42, -92, -10, -15, -22, -32, 5, 93,
    13, 40, -23, 40, -82, -49, -53, -72, -75, 5, -56, 84, -67, -8, -93, -10,
    -78, 4, -81, -73, -2, -39, 29, -69, -60, 26, -66, -65, -32, -80, -62, 30,
    -5, -93, 7, -41, 75, 58, 94, 84, -39, 42, -96, 84, 45, -31, 24, -11, 5,
    55, 75, -54, 25, 63, 85, 51, -34, 58, -44, 48, -67, -59, 81, -44, 22,
    -71, -28, -58, 83, 17, 14, 90, 49, 6, 38, -43, 75, -20, -67, 66, -43,
    -23, 45, 96, -45, -84, -54, -68, 19, 45, 28, 36, 61, -99, 68, 24, -8,
    -34, -77, 56, -34, -88, 73, -79, 26, -43, -88, 43, -20, -55, 51, -24, 36,
    3, -36, 18, -41, -52, -81, 52, 22, 38, 66, -64, -44, -33, -46, 64, 3,
    -32, 83, -83, 89, 15, 13, -74, 50, -96, 81, 40, -6, 15, -84, -86, 60,
    -55, -41, -71, -91, 4, 18, 78, 87, -73, -90, 77, 57, 17, -1, -88, -37,
    -94, -44, -85, 55, -56, -53, 94, 79, -84, 56, -29, 35, 2, 50, -6, -63,
    -90, -5, 71, -1, -82, 46, -26, -39, -59, 75, -34, -23, -62, 83, 77, -35,
    41, -35, -92, 96, 60, 44, -14, -37, -67, -71, -17, -50, 64, -66, 78, 5,
    41, -29, -85, 89, -56, -90, -22, -28, -6, -18, 78, 78, 17, -64, -35, 94,
    7, -54, 53, 31, 28, -52, 73, -100, -88, 99, 1, -11, -95, 9, 92, 52, 33,
    10, -84, -42, 33, -94, 20, 83,
  -42, -97, -86, -70, 85, -59, -45, -79, 15, 94, 28, -3, -63, -68, -94, -54,
    -58, 74, 6, -43, 56, 63, 49, 24, -63, -87, 97, -8, 26, 23, -73, -38,
    -100, -14, 58, 26, -10, -38, -66, 95, 11, 67, -11, -30, -55, -17, 32, 38,
    -41, 83, -70, 87, -94, -38, 19, -60, -13, -94, 3, -6, 74, 0, -88, -85,
    71, 57, -28, -99, 12, -47, -72, -11, 54, -32, 59, 2, 4, -22, -87, 99,
    -65, -36, 93, -49, -97, -79, 15, -89, -49, -50, -48, 45, 75, 1, -36, -92,
    66, 12, 53, 28, -75, -27, 53, 64, -70, -57, -35, -68, -18, -32, -6, -23,
    -68, 79, -44, 45, 11, -91, -3, -85, 45, 53, -61, -41, 73, 17, 52, 80, 44,
    -44, 54, 90, -33, 40, -41, -26, 91, 70, 63, 83, 71, -91, -60, 16, -98,
    -3, 22, 53, 60, 76, 4, 39, -7, 67, 51, -83, -57, -75, 41, -76, 64, -67,
    -1, 81, -37, 94, -4, 41, -30, -19, 44, -55, 62, -75, -40, 61, -33, -6,
    -5, -67, 34, -64, 25, -39, -93, 3, -5, -49, 58, 54, -94, 18, -76, -36,
    -34, 37, -25, -77, 82, -70, -19, 87, 66, -24, 28, 96, -19, 97, 39, 96,
    -47, -45, -38, 16, 5, -83, -88, -65, -5, -12, 90, 86, -60, -61, 86, 87,
    -62, 18, -95, -31, -75, 27, -79, -60, -31, 50, 49, 30, -58, -12, -66,
    -79, 16, 19, 14, -12, 93, 74, -20, -57, -44, -46, 33, -12, 0, -68, -16,
    -63, 59, 12, 47, 44, 43, 44, 45, 38, 18, -35, -55, -2, -15, -65, -91,
    -71, 29, -93, -82, -28, 46, -20, 8, 1, -37, -18, -25, -26, -83, 47, 18,
    -97, -93, 7, 49, -78, -6, -87, -90, 61, -26, 57, -77, 10, -65, -43, 81,
    -96, -13, 65, -29, -85, 15, -97, 36, 79, 22, -2, -92, 0, -84, 49, -32,
    -35, -67, 58, -34, -21, 13, 80, 84, 14, 44, -4, -72,
  -60, -89, 49, -46, 72, 13, -59, 14, 42, -7, -16, -36, -81, -51, -50, -18,
    13, 91, 62, 33, -12, -36, 98, -20, -46, -51, 7, 10, 58, 25, 2, 39, 77,
    -43, 50, -80, -9, 17, -36, 78, 23, 84, -46, -24, 40, -17, -13, 84, 48,
    51, -97, -18, 20, -59, 10, -50, 63, 53, -80, -7, 34, -33, -39, -26, -41,
    -96, 58, 93, 52, -76, 78, 47, -61, 67, -86, 45, -85, 52, -6, 63, 95, 85,
    -46, -11, -68, 73, 91, -84, -70, 67, 43, -73, -98, -89, -67, 54, 22, -43,
    52, -37, -81, 20, 13, -58, 30, -84, -98, 88, 21, -92, -49, -64, 93, -91,
    20, -66, 32, 11, 4, -24, 89, -61, 13, 74, -51, -31, -34, -66, 28, -31,
    -79, -52, -51, 32, 14, -47, 6, 93, 40, -29, 81, -85, -3, -1, -31, 42,
    -28, -7, -37, -18, 4, -50, -5, -46, -38, -36, 46, -45, -6, -10, -94, 5,
    -7, -27, -64, -23, 93, 19, 98, 55, 99, -96, 41, -41, -14, 93, 91, -23,
    52, 94, -62, 30, -96, 88, 58, 15, 17, 35, 85, -21, 51, 96, -16, 26, 22,
    -52, -10, -11, -72, 43, -64, 72, -65, 45, 16, -75, 53, 52, 5, 23, 88, 31,
    60, 74, -27, 44, 99, -79, 6, 90, -19, -77, 23, 81, 28, 73, 26, 45, 85,
    -86, 95, 24, 68, 57, 26, -88, 73, 32, 0, -19, 85, -27, 54, 82, 31, 36,
    89, 32, 53, 81, 43, 78, 5, 6, -34, 1, 92, 62, 58, 53, 24, -8, 8, -24, 24,
    23, 72, 14, -35, 7, -68, -1, 36, 65, -40, -30, -16, 48, 30, 30, -97, 77,
    -60, -29, 41, -42, -8, -36, -98, -73, 35, -84, -42, -46, 13, 65, 8, 93,
    -4, -28, -80, 0, -36, -70, -30, 50, -96, -3, -78, -92, 98, 70, -20, -84,
    17, 77, 23, -33, 78, -41, -26, -18, 78, 6, 51, 81, 49, -62, -48, -13,
    -68, 13, 36,
  1, -23, 34, 9, -54, 21, 49, 76, -99, 5, -4, 25, -44, 19, -79, 35, 26, -13,
    -58, -81, -63, 22, 99, 91, 44, -64, -63, -86, 24, 17, 81, -81, -23, -10,
    84, -33, 26, -47, -96, 81, 30, -72, 93, -81, 47, 11, -99, 92, 38, 49,
    -64, 60, -3, -59, -56, -8, 71, -60, 80, 0, 48, -37, 62, -92, 35, 99, 77,
    41, 86, 61, 26, 89, 0, -35, -63, 21, -11, -78, -54, 60, 57, -56, 35, 73,
    59, 3, 20, -1, -93, -69, 31, -93, 99, -23, -52, -77, -85, -5, 75, -23,
    44, -31, 69, 43, 49, 3, -77, -94, 81, 63, -81, 34, -11, 44, -29, -82, 59,
    86, 83, -97, 87, 44, 65, 87, 99, 26, -31, -7, 53, 85, 23, 67, 69, -67,
    23, 20, -42, 81, -73, -26, 11, 83, 80, 83, 18, -90, -13, -10, 16, -25,
    -95, 30, 71, 43, 49, 2, 36, -17, 80, 9, 50, -34, 5, -63, -47, 73, 86, 39,
    27, -8, 35, -11, 49, -48, -56, -31, 35, -45, 19, -90, 93, 64, 74, 55, 45,
    -93, 38, 79, 51, 63, -38, 79, -5, -77, -8, -94, -20, -83, 35, -68, -79,
    19, -78, 47, -66, 31, -40, 15, -64, -68, -100, -75, 6, -28, 58, 33, 3, 7,
    -67, -63, 23, 98, 12, -74, -36, -63, 88, 72, -71, -64, -34, -16, -94, 82,
    20, -64, -69, 59, 4, -89, 67, -94, 64, 89, 65, 88, 59, 74, -68, -45, -47,
    19, -86, -11, 43, 92, -80, 62, -38, 56, -22, 67, 49, 29, 66, -20, 80, 41,
    -96, -25, 79, 69, 58, 49, 8, -92, -77, 58, 50, 47, -56, -43, 80, 24, -98,
    86, -73, -40, -92, -87, -80, 31, 7, 19, -28, -91, 63, 40, 61, -50, -73,
    -5, -49, 25, 4, 64, 29, 62, -1, -86, 96, 2, 76, -9, -50, -37, 30, 32,
    -45, -25, 73, 94, 24, -79, -46, -65, 34, 22, 96, 99, 39, -67, -17,
  -39, -61, 9, -29, -26, -95, 68, 29, -39, -77, 89, -37, 87, -64, 60, 11, 96,
    -42, 30, 80, 65, 0, -56, 48, 77, 78, -100, -6, -75, 1, -14, -6, -11, -51,
    80, -52, -33, -54, 23, -18, 51, -20, 73, -96, -48, -83, -73, -51, 63, 85,
    -6, -4, 80, 3, 52, 50, 15, -17, 54, -94, 54, 37, 84, -76, 40, -17, 76,
    -81, 45, 4, 19, -47, 87, 53, 43, -47, 39, -68, -7, 48, -95, 7, 50, 9, 4,
    -94, 62, 44, -91, 16, -69, 18, -23, 6, 47, 74, 90, -36, -32, 35, -14,
    -81, -98, -90, -34, 26, -42, -48, 4, 2, 80, 85, 21, -44, 27, 1, 48, -66,
    24, 18, 30, -79, -80, 2, -30, -56, -14, 60, -58, 77, 92, 65, 50, -21,
    -35, 36, 20, 15, 95, -40, 83, -97, -37, -81, 15, -46, -23, 73, -35, 23,
    -55, 4, -36, -71, 19, 71, -52, 61, 15, -10, 45, 42, 64, 3, -63, 83, 61,
    55, -42, -99, -28, -42, -17, 74, -29, -60, -1, -77, 31, -50, 82, -86, 4,
    -92, -34, -87, -35, -70, -76, 75, 65, -48, 30, -22, -45, 67, -63, -29,
    -10, 21, 4, -58, -85, 15, 32, -65, -80, 32, -33, 56, 46, -77, 41, 40,
    -89, -80, 12, -55, 22, -10, -28, 22, 32, 13, -72, 76, 5, 41, 89, 30, -86,
    74, 79, 15, 31, 24, 17, 68, -15, 12, 66, 44, 51, -6, -19, 25, 70, 34,
    -13, -4, 49, -90, -17, -56, 58, -73, -8, -93, 63, 45, -95, -45, -91, 49,
    -70, -44, -72, -82, -40, -68, -90, 98, 69, -22, 9, -38, -61, 11, 7, -73,
    18, -62, -42, -98, 41, -31, -50, 52, 28, 56, 16, 26, -62, -14, 32, -83,
    13, 58, -19, 79, -1, 12, -100, -23, -30, 50, -95, -56, 25, -11, 5, 11,
    -46, -93, -39, 64, 28, 7, 70, -24, 21, 1, 55, 91, -37, -55, -69, 46, 87,
    81, -77, 85, -59,
  -40, -87, -4, -95, -57, 30, 4, -14, -45, 59, 54, 9, 89, -71, 53, 69, -66,
    -60, -90, -42, 16, 40, 64, -74, -2, -16, 80, 11, -97, -88, 83, 84, -57,
    90, -57, 35, 87, 81, 1, -71, -19, 86, 83, 52, 56, -77, 30, 86, -77, -31,
    -92, 68, 79, 7, 12, -81, 65, 11, 59, 29, -25, 35, 70, 98, -38, 33, -68,
    11, 72, 94, 15, 65, 18, 25, -51, -85, 80, -46, 27, 22, 47, 10, -3, 78,
    96, 66, 46, -48, -30, -40, -95, -94, -28, 25, -4, 46, 58, -29, -14, 19,
    -52, 93, 61, -76, -59, -83, -71, 42, 31, -77, -64, -57, 48, 13, -19, 90,
    83, -15, 4, -96, -27, -33, -55, -99, 40, -31, 65, 49, -75, -88, 70, 38,
    80, 35, -73, 8, 96, -59, 1, -42, -58, 66, 69, 35, -47, -99, -41, -92, 46,
    90, -63, 8, -54, 14, -13, 81, -29, 54, -80, 47, 34, -46, 64, -64, 39, 36,
    -49, -33, 60, -64, -50, 97, 97, -35, 36, 2, 39, -13, 45, -3, 22, 9, 19,
    -100, 58, 22, -25, 89, 86, -20, 51, 19, -13, -63, 68, -36, 83, 52, -35,
    64, -28, 27, -74, 11, -33, -9, 85, -27, -24, -42, 94, -29, 34, 42, -2,
    -5, 57, 30, 1, 73, -32, 0, 77, 39, 25, -32, 70, 31, 62, -54, 56, -13, 59,
    94, 85, 91, 85, 90, 60, -79, -83, -14, -22, -50, 12, -8, -79, -54, -59,
    68, -1, -75, 59, 23, 45, 29, -33, -51, -83, -37, 37, -9, -86, 74, 74,
    -58, 28, -86, 58, -45, 3, -53, -28, 54, 55, 23, -23, -62, 99, -66, -90,
    35, 88, 85, 47, -22, 79, 33, -34, -35, 2, 91, -51, -44, -50, 96, 28, 79,
    -19, -97, 28, -46, 40, 87, 17, -32, -22, 37, 64, 68, -99, -78, -82, -86,
    18, -93, -14, -92, -17, 84, -23, 12, 76, 82, -94, 30, 80, -84, 11, -81,
    28, -79, 83,
  -39, -91, 50, 26, 9, -86, 25, -20, -21, -20, 78, -9, -69, -14, 7, -76, 29,
    74, -50, 89, 89, 49, 1, -85, 50, -61, -46, 57, 53, -79, -83, 76, -11, 61,
    -14, -23, -33, 69, 60, -91, -20, 4, -18, 36, 29, 2, 99, 90, -72, -75,
    -32, 64, 55, 16, 84, -100, -84, -43, -91, 75, 14, -49, 58, 62, 83, -73,
    -53, 74, 5, 67, -100, -49, 78, 37, 61, -85, 72, -81, -97, -98, -16, -75,
    53, 12, -70, -6, -65, 94, 12, -29, 26, 52, -41, -93, 63, 30, -95, -43,
    -10, -5, 10, -27, 56, 52, -5, -79, -11, 34, -53, 29, -10, -63, 4, 81,
    -11, 83, -30, 0, 42, -25, 64, 25, 54, 60, 71, -68, -39, 74, -100, 46, 12,
    26, 57, 2, -43, -75, 0, 30, -53, 25, 33, 80, 48, 31, -18, 81, 65, -48,
    65, -84, -3, 70, 61, 89, -20, 37, -18, -88, 53, -94, 50, 91, 63, -86,
    -29, 29, 78, -24, -16, -28, 47, -74, -42, 47, 61, 8, -90, 56, 11, 54,
    -70, 51, 89, 9, 10, -92, -42, -5, 29, -51, -96, 7, 40, 67, 76, 53, 36,
    21, -56, -78, 99, 84, 93, -3, -62, -87, 17, -8, 9, 12, -60, -62, 17, -88,
    -18, -96, 51, -72, 73, -19, -1, 53, -16, -28, -39, 52, 80, 52, 26, -17,
    6, 24, -81, -26, 62, 75, 75, 24, -97, 86, -46, -9, 74, -35, 45, 97, 92,
    99, -28, 90, -45, 41, -47, 38, -58, -37, 66, -89, -94, 69, 44, -67, 33,
    82, -68, 52, -71, 43, -24, 41, 19, 12, 52, -99, 53, 85, -5, 78, 53, -10,
    87, 92, 8, 5, -7, -53, -10, 89, -88, -92, -26, -16, 60, -97, -73, 77, 40,
    85, 15, -49, -68, 74, -83, 30, 14, -23, -70, -47, -62, 58, 9, -81, -96,
    35, 4, 54, 51, -4, -84, -43, -51, 32, -8, -54, -28, 65, -8, -7, -26, -33,
    -44, -19, 95,
  -99, 73, 30, 23, 42, -36, -67, -23, 73, 54, 77, 23, 70, 29, -63, -57, 57,
    1, 37, 11, -39, -48, -65, -69, 79, -52, 31, 71, -16, -2, 53, 82, 27, -11,
    36, 76, -83, 27, -47, 40, 34, -45, 36, -1, 7, 8, -52, -85, 74, 92, 51,
    81, 92, -20, -11, 97, 43, -77, 34, -28, -71, 68, -93, 48, 92, 26, 51, 71,
    -75, 9, -20, 71, -7, 4, -86, 14, -54, 77, 36, 68, -81, -3, -68, 84, -29,
    -9, -38, 74, -76, -90, 87, 0, -24, 22, 67, -87, -31, -92, -43, 51, -1,
    -36, -3, 51, 94, 77, 16, -80, 14, 42, 57, 80, -87, -99, -46, 66, 22, 15,
    -92, 42, -58, 34, -35, -55, -30, 12, -49, -74, 76, -86, 62, -73, -67, 57,
    -98, -8, 38, 84, 51, -80, 2, 12, -87, -33, 76, 19, 64, -60, -13, -11, 1,
    -3, -28, -91, -5, 76, 45, -90, 5, 63, -19, 94, -4, -13, -69, -73, -37,
    -84, -47, -58, -93, -87, -43, 25, -17, -81, 64, -69, -15, 35, 80, 46, 55,
    -3, -66, -91, 38, 76, -16, 17, 31, -32, -87, -22, -53, -12, -82, 93, 92,
    56, 68, 29, -39, -90, 30, 52, -96, 59, 66, -37, 87, -37, -8, -49, 12, 38,
    -36, 40, -27, 48, 87, -85, -8, -68, -16, 48, 84, 90, 2, 0, 69, -24, 95,
    46, 11, 1, 74, -21, 3, 63, 71, 42, -70, 95, 78, 72, -1, 37, -20, -79,
    -99, -21, 2, 54, 28, 32, -81, 54, 39, 14, 40, -90, 27, 13, 77, 18, -63,
    16, 40, -85, 80, -22, 52, 37, -78, 4, 39, -46, 62, -4, -39, -92, 26, 29,
    74, 52, -60, 22, -6, 27, 9, -36, 29, 44, -45, -100, 20, -12, 27, 2, -77,
    64, -97, -78, -82, -33, 81, -53, 30, 39, -49, -76, -65, 76, -45, 72, -97,
    -75, 12, -9, -23, -25, -95, 82, 74, 12, -16, -50, -73, -63, 20, -41, -54,
  56, 42, -31, -38, 28, -97, -97, -65, -72, 24, -60, 68, -94, 51, -11, 87,
    79, -52, 25, 38, -84, 66, 38, 53, -13, -41, -92, 50, -97, -6, -7, -50,
    -4, -27, -92, 52, 69, -45, 23, 94, -52, 52, -22, 7, -12, -15, -16, 54,
    -12, 31, -89, -76, -89, -70, -7, -56, -97, -56, -97, 13, -99, 10, 2, -18,
    31, 87, -100, 41, -83, 38, 12, 96, 40, 20, -44, -76, 70, 15, -45, -73,
    64, -60, -29, 30, 2, 89, 90, -90, 59, -40, -82, 88, 59, -27, -22, 88, 74,
    80, -90, -91, 95, 56, -28, -73, -28, -37, 40, -88, 96, -33, -100, -20,
    83, 82, 91, 87, -37, -41, 63, 68, 69, -85, 27, -75, -21, -89, 60, -67,
    42, 2, 43, 38, 46, 96, 49, 5, 41, -66, 92, -26, -21, 2, -41, 64, -21, 80,
    21, -50, 18, -44, 38, 11, -93, -55, 10, -63, 90, 76, -74, -49, -39, 40,
    -16, -89, -42, 90, 66, -43, 93, 59, 87, 68, 26, 35, -58, 48, -78, 45, 83,
    33, -35, -94, 51, 93, 72, 17, 20, -98, 13, -61, -74, 3, -27, -70, 73, 73,
    -58, -71, -14, -94, 47, 86, -68, 81, -89, 12, 8, -39, 46, 85, 23, 71, 88,
    -75, -45, 86, 16, 33, -5, -13, -74, -43, 58, -10, -38, 5, 64, -33, 25,
    54, -62, -3, 27, 87, -54, -94, 46, -48, 82, 33, -89, -12, 10, -38, -13,
    37, -48, 85, -35, 30, -69, 96, -61, 22, 12, 59, -96, -69, -35, 24, 11,
    -61, 80, 11, -54, 74, 66, 39, 44, 68, 64, -71, -40, -21, -31, 55, 73, 77,
    -68, 19, 80, -76, -21, 20, 21, -49, 17, 12, 88, -68, 68, -67, -83, 52, 1,
    -88, -46, 39, -87, 24, -3, -93, 57, 52, 52, -65, -26, 14, 35, 95, 83,
    -84, 2, 18, 62, -50, 71, 0, 59, -63, -97, -74, 44, 34, 5, -9, 70, 58,
    -84, -6, 26, -5, -19,
  -78, -6, 64, 22, -51, 78, 31, -37, 67, -67, 52, 5, 43, 19, -59, 2, -14, 87,
    -80, -70, 7, -53, 84, -67, 45, -15, -67, 19, -19, 86, 32, 77, 68, -35,
    -6, 60, 31, 92, -3, -86, 65, 76, 22, -70, 96, -48, 24, 34, 25, 76, 65, 0,
    -95, 30, 95, 32, 32, 96, -28, 52, -97, 88, 10, -71, 6, 55, -8, 3, 32,
    -50, -41, 11, 42, -70, -42, 51, 95, 29, 4, -89, 25, -15, -94, 8, 74, 91,
    -93, 4, -34, 5, 27, -11, -20, 6, -14, -6, -8, -23, 41, -43, -85, 6, -97,
    50, -17, -42, 5, 44, -64, -53, 58, -5, -32, -90, -81, -96, 83, 74, 86,
    25, 55, 67, -85, -84, 74, 56, 89, -51, -78, 29, -48, -22, 24, 61, 33, 1,
    -36, -88, 40, -89, -76, 32, 93, 96, 49, -45, -51, -22, 12, 56, 9, 74,
    -97, 4, 95, 27, -81, 74, 37, -66, -7, 42, -45, -80, 67, -44, -55, -92,
    -97, 39, 83, 13, 92, 24, 87, 86, -36, 17, 79, 51, 6, -13, 74, -36, -31,
    -65, 25, 64, 16, 70, -17, 14, -34, -29, 54, 22, -70, -16, 70, 31, -46,
    -7, -100, 16, 63, -70, 53, 79, -50, -73, 24, 12, -38, 2, -66, -59, 59,
    82, 88, 36, -32, -41, 93, -94, 17, -88, 12, 78, 66, 12, 11, -95, -28,
    -87, -81, 34, -53, 39, -96, -60, -61, -20, 30, -27, 36, 45, 80, -63, 12,
    -80, -75, -83, 60, -54, -57, -2, 14, -100, 70, 23, -6, 27, 80, -57, 9,
    -90, 8, 48, -82, 91, -2, 99, -2, -48, 88, 99, 3, 92, -49, 33, -68, -86,
    -10, 4, -45, -70, -7, 47, 34, -26, -74, -40, -15, 92, 53, -26, -34, 73,
    -91, 74, 93, -81, -9, -35, 51, 82, -94, 77, 65, 36, 65, -58, -46, 51, 61,
    -80, 35, -8, 1, -63, -89, -25, 21, -4, -49, 75, -95, 31, 9, -58, 0, 57,
    -48,
  11, -16, -62, -66, 98, -69, -31, -82, 38, -46, 98, 85, 12, -82, 8, 74, 78,
    -11, -39, 54, 79, -40, -97, -72, -17, -52, 99, 77, 45, 98, -48, -97, -44,
    32, 68, -76, -68, -51, -94, -74, 40, 32, -94, 93, 98, -9, -26, 94, -46,
    31, -20, -93, 62, -8, 50, -98, 29, -48, -34, -24, 60, 15, 29, -35, 69,
    26, 88, -86, -93, 39, -56, 59, 26, -47, -15, -73, -33, -8, -8, 63, 98,
    -65, 57, -39, -35, 47, -14, 22, -6, 78, -90, -39, 37, -22, 43, -1, 89,
    98, 65, -62, 60, -40, -79, -49, 1, 6, -52, 2, -88, -46, 34, 55, 93, 50,
    5, -26, 97, 41, 41, -87, -5, 92, -80, -98, 81, -57, -90, -18, 37, 45,
    -64, -11, 4, -10, -55, -12, -18, -24, 0, 96, -29, 10, 32, 82, -86, 28,
    -40, -78, 88, 65, 16, -59, -41, -60, -73, 71, 77, 9, -85, -28, 98, -85,
    -63, -26, -45, -13, 34, 5, 92, 89, -86, -22, 13, -2, -5, 57, 85, -100,
    97, 87, 50, 45, -100, -22, -22, 27, -18, -3, 97, -60, 77, 23, 18, 23,
    -84, -59, 82, -85, -39, 21, 87, 34, -32, 83, -14, -61, -44, -15, 96, 99,
    -21, 61, 51, 20, 3, 24, -20, -77, 69, -4, -36, -17, 12, -81, 65, 49, -66,
    -30, 94, 57, -6, 8, 18, -66, -44, 89, 2, -25, 2, -62, -56, 99, -73, 56,
    -87, 88, 40, -54, 74, -84, 28, 12, 41, 63, 83, 7, -84, 6, 81, 75, 79,
    -97, 31, -81, 46, 8, 64, -13, -70, 10, 21, -70, 1, -49, -97, 61, 57, -83,
    96, 94, -35, -7, -38, 40, 77, -70, -65, 8, 49, -69, -70, 20, 74, -73,
    -38, 66, -13, -7, 44, 50, -20, 29, 7, -33, -59, 95, 74, 45, -67, -81,
    -29, 16, 28, -47, 40, 42, 65, -47, -87, 23, 64, -15, -26, -74, -79, 28,
    -97, 53, 20, 58, 10, 94, -30,
  -63, 8, 92, -45, 81, -26, -11, -51, 17, -79, -4, 90, 56, 9, -35, 28, -83,
    22, -13, 77, 63, -2, -92, -3, 86, -87, 16, -90, -82, -78, -7, 21, 92,
    -18, 3, 19, -69, 72, -65, 3, -41, -96, -58, -35, -67, 68, -97, -73, 7,
    -88, -75, -73, 18, -66, 32, -50, 15, -49, 8, -48, 89, -36, -22, -60, -6,
    95, -48, -6, -52, 65, -41, -17, -45, -4, -98, -75, 16, -89, -72, 16, 42,
    -97, -21, -33, 38, 37, -10, 26, 82, -21, 17, 74, -43, -62, 7, -74, -61,
    -65, -15, -60, 97, 46, -19, 32, 65, 46, 85, -67, 93, 26, 63, 13, 74, 58,
    76, -100, -46, -36, -87, 57, -34, -39, -35, 87, 34, 21, -32, 24, 75, -43,
    -67, 86, 43, -56, -70, -66, 14, -33, -8, -97, 55, -78, -82, -85, 42, -69,
    -22, -11, -80, 95, 23, -86, -96, -65, -67, 89, 50, 27, 93, -40, 94, 85,
    -22, -14, 18, -81, -42, 20, 36, 18, -7, 63, 7, -61, -80, 47, 66, -21,
    -44, 60, 54, -7, 0, -78, -65, -99, 86, -37, -26, 25, -27, 58, 23, 10,
    -59, 4, 49, -61, 95, 9, -31, 8, 71, -9, -1, 12, 29, 80, 27, 62, -98, -88,
    -83, -18, -93, -80, -72, 67, 75, 15, 42, 64, -64, 71, 4, -60, 5, -12, 96,
    64, -21, -78, 65, -85, -55, -78, -60, -81, 16, -10, -43, -34, -49, -57,
    -2, 87, 60, 93, -30, -55, -52, 82, -44, -43, -56, -5, -68, -100, -6, -69,
    90, -49, -75, -77, -31, -12, -58, 3, 78, 28, -50, -5, 18, -65, -67, 85,
    -47, 23, 63, -26, 16, -35, 32, -15, -40, -72, -2, -7, 13, -78, -83, -45,
    -12, 32, -17, -73, 90, 50, 79, 40, 72, -74, -58, 52, -4, 61, 86, -37,
    -31, 20, 89, -94, -85, 46, -18, -81, 68, -18, -65, 81, -64, -49, -12, 61,
    95, 39, -5, -36, -9, 92, 13, -16, -45,
  -24, 13, -12, 6, -74, -24, -97, -11, -96, 23, 75, 33, 29, 36, -35, 2, -95,
    27, 59, 2, -22, 14, 3, 2, 4, -53, 33, -64, 79, -74, 64, 88, -91, 11, -87,
    -47, 24, -50, 44, -81, 41, 25, -36, -11, 3, 80, 41, 79, -44, -10, 56,
    -72, -5, -66, -65, 83, -61, -29, -45, 94, -75, 84, -38, -59, 28, 39, -4,
    -5, 7, 92, -96, 84, 91, -26, 41, -6, -62, 44, -35, -58, 89, -8, -59, 53,
    12, 41, -27, 36, -55, -92, 58, -18, -69, -69, 60, -4, 23, -16, 89, -69,
    -81, -56, -38, 73, -55, 28, -52, 70, -37, -77, 94, 24, 82, -73, 26, 59,
    -97, -31, -38, 25, -23, 56, 43, 18, 62, -38, 13, -43, -5, 7, 89, -76,
    -11, 43, 54, 90, -60, -57, 77, 30, -39, -78, 55, -10, -49, -78, 96, 11,
    -84, 44, 95, -64, -76, 9, -79, -59, -62, 74, 15, 57, -14, 14, -60, -63,
    -47, -43, 16, -32, 78, -41, -25, -57, 22, -46, 51, 12, 33, 64, -28, 56,
    -81, -41, 75, -18, 1, 70, -43, 66, 89, -71, 79, 51, -93, -93, -62, -46,
    -23, -99, -51, 30, 75, -36, 54, 9, 39, -18, 68, 98, 46, 82, -74, -62, 97,
    -79, -82, 35, 27, -76, -55, 84, -34, 79, -40, -71, 90, 52, -12, -98, -48,
    16, 79, 52, 90, 34, -9, 21, -77, 17, -49, 96, 78, -65, 46, -60, 43, 61,
    18, 71, 39, -16, 99, 75, -88, 82, -5, 12, 64, -97, -14, -3, 34, 9, -65,
    66, 29, -69, -83, 4, -93, 27, -66, 31, 53, -35, 90, -73, 14, 26, 24, 93,
    -97, -94, 2, 72, 9, -12, 35, 23, 1, 89, 49, -58, 47, 17, -9, -47, 51, 50,
    -82, 72, -5, -12, 15, -99, 56, 75, -64, 48, 59, -75, 14, 48, 3, 88, 55,
    30, -50, -99, -14, 25, -64, -54, 86, 47, 11, 9, -35, 80, -73, -8, 91, 31, 48,
  -94, -74, -70, 49, -75, 37, 25, -38, -27, -84, -64, -41, -31, 55, -97, -22,
    -19, -99, 55, -10, -7, -54, 89, 69, -76, 88, -81, 66, -35, -47, 25, -23,
    -9, -57, -17, 58, -85, 88, -98, -18, 72, 28, 92, -65, -14, -85, 62, -52,
    7, 76, 3, 77, -15, -66, 92, -70, -48, -71, -5, -39, -81, -21, 23, -67,
    -22, -71, 36, -46, 44, -62, -84, 61, -87, 86, -36, 93, -45, -37, -61, -3,
    -64, -79, 20, -4, 96, -4, 14, 60, -18, -27, 45, -76, 11, -62, -15, -10,
    90, 32, -66, -87, -73, 14, 74, 24, 52, 47, -35, -39, 94, -33, 37, -60,
    -30, -77, -99, 4, 68, -13, -16, -100, -26, 34, 82, 14, 87, -15, 58, 67,
    -7, -94, 68, 73, -79, 13, -72, -15, -19, 49, 85, -46, -90, -54, -43, -9,
    86, -49, 97, 64, -91, -89, -31, -63, -60, -43, -74, 44, -35, -73, 57,
    -14, -64, 79, 71, -57, -99, -54, 47, 11, 84, 90, -61, 33, 93, -100, 84,
    34, 69, -91, 53, 94, 98, 96, 3, -21, 56, -65, -26, -90, 29, -90, 24, -10,
    -15, 76, -38, -32, 9, -27, -85, -33, -33, 47, 10, -28, 63, -46, -87, -17,
    -46, -65, 9, 22, -53, 16, 95, 76, 10, 99, 54, -25, 90, -53, 99, -57, 70,
    -54, 41, 49, -95, -83, 0, 13, -35, -26, 24, 29, 48, 93, -78, -45, 9, 6,
    -39, -31, 51, 80, -6, -57, 25, 65, 43, 5, 12, 40, 22, 54, -99, -53, -8,
    -73, -40, -35, -6, 48, -43, 90, 34, 0, 35, -86, 33, 42, 79, 44, 61, -89,
    -40, -84, 73, 33, -20, -13, 11, 65, 15, 82, -95, -84, -82, 76, -38, -45,
    -14, 15, 78, 66, 29, 38, -67, -14, -92, -97, 41, 52, -62, -75, 37, -48,
    61, -48, 97, -96, 24, 9, 62, -13, -62, -31, 8, -18, 55, -82, -14, -22,
    -11, 20, -59, -52, 96, 69, 90, -16, -26,
  -36, 7, 51, 96, -97, -18, 22, -73, -81, 65, 25, -57, -20, 40, 42, -22, 12,
    -88, -32, -58, 6, -3, -82, -61, -85, -66, -80, -80, -42, -100, 29, 23,
    76, 35, -90, 87, 98, 55, -54, -88, -61, -8, -68, -4, -100, 11, 69, 37,
    -79, 67, -33, 11, -31, 24, -31, -81, 65, -23, 33, 21, 20, 2, -3, -3, -11,
    7, 46, 77, -26, -94, 43, -61, 40, 79, 86, 74, 34, 92, -13, -58, -53, 20,
    -72, 33, -22, -50, -70, 71, 94, -99, -88, -100, -10, -5, 87, 86, 13, 43,
    -72, -79, -12, 16, -42, -49, 33, 93, -7, -87, -60, 27, 54, 86, 13, 46,
    -78, 59, -98, -54, -16, 64, 85, -10, -25, -30, 68, -26, -82, -89, 54,
    -10, 64, 0, -16, 5, 25, 44, 91, 62, 50, 60, -91, -60, -22, 27, 82, 95,
    90, -91, -21, 41, -74, 46, 44, 28, 96, -70, -4, -25, -66, -74, 94, 24,
    80, -69, -15, 79, -76, 39, 69, 64, 29, -85, 48, 51, -59, 7, 32, -66, 19,
    98, 54, -45, 75, -11, -7, 5, -6, -66, -39, -50, -29, -44, -58, -27, -65,
    54, -100, 53, -7, 97, -95, 47, -95, -43, -57, -32, -42, -57, 67, -11, 90,
    71, 96, 37, -100, -51, -21, 78, -9, 46, -13, 86, -9, -80, -54, -94, -86,
    98, 52, -12, -86, -31, 63, 73, 68, 89, -73, 72, 2, -44, 10, -54, 84, -5,
    -30, -59, 42, -60, 95, 30, -28, -19, 3, -33, -60, 63, -13, -37, 88, 20,
    66, 18, 63, 95, -8, 29, -38, -92, 69, -17, 10, 78, 47, -21, -53, 58, 99,
    5, -55, 77, -66, -47, -50, -13, 35, -80, -74, -15, 45, 11, 18, 64, -96,
    -26, 45, -41, 49, 40, -5, 63, 75, -8, 96, -85, -67, -92, 66, 6, 77, -9,
    17, -40, 84, 84, 6, -58, -60, 77, 30, -15, 68, 0, -44, -32, 53, 33, 97,
    82, -34, 82, 29, -9, -98,
  -2, -34, -74, -81, -20, 71, 14, -41, -20, -76, -59, 1, -63, 41, -23, -18,
    -71, -10, 70, -19, -45, -60, 42, -64, -24, -50, -74, -42, -33, 81, -86,
    -100, 92, -63, 70, -11, -86, 25, 45, 6, -97, -63, 82, 14, 68, -56, -44,
    98, 11, 25, -60, -91, 0, 95, 74, -47, 7, 61, 81, 4, 68, -9, 57, -5, 43,
    -12, -3, -88, -41, 90, 5, -94, 4, -63, -37, 16, 41, 52, 46, -78, 20, 41,
    -44, -76, 28, 18, -17, 31, 77, -72, -38, 13, 88, 55, 74, 30, -18, -71,
    70, 70, -68, -14, 78, 97, -38, -27, 46, 43, 21, -82, 35, -75, 42, -79,
    -11, -36, 46, -4, 70, 64, 28, 63, -15, 96, 87, 63, -98, -98, -79, -13,
    -23, -15, 91, 18, 87, -52, -62, 54, -54, -61, 87, -20, 50, 8, 33, -49,
    -4, 88, -23, 74, 34, 25, -74, 18, 40, 22, -81, -69, 40, -95, 58, -47,
    -71, 47, 12, -78, -73, 39, -82, -22, -71, 88, 84, -72, 55, 86, -20, -50,
    -60, -65, -47, -2, -31, 85, 70, -86, -89, 16, 12, -67, 77, -78, 29, -34,
    -10, 66, -62, 38, -99, 87, -80, -41, 92, -1, 77, 78, -87, 80, 56, -64,
    -56, -14, 82, -68, 31, -88, 31, -91, -40, -47, -54, -24, 33, -8, -97, 36,
    -92, 89, 90, 48, 91, 43, -100, 49, -66, -3, -27, 27, 69, -51, 29, -44,
    -76, -21, 21, 87, -66, 3, -15, -9, -32, 84, 8, 22, 94, -78, -95, 12, 16,
    -99, -13, 15, 57, -40, 86, 62, -100, -46, -24, 79, 63, -71, 68, -48, -85,
    84, -45, -91, 74, 80, -98, 83, -44, 6, -56, 23, 58, -85, 21, 46, 20, 90,
    35, 24, -62, -99, -75, 27, 93, -91, 74, 15, -44, -74, 50, -9, 78, 28,
    -81, 50, -80, -43, -61, -57, -27, -92, 42, 80, -66, 59, 41, 99, 63, 91,
    66, -40, -49, 6, 77, 23, 57, 48, -70,
  0, -17, 5, -44, 76, 42, -73, 23, 93, 61, -10, 32, -88, 72, -96, -28, 0,
    -37, 26, -72, 32, 57, 15, -99, 5, -13, -16, 61, 84, -33, 8, -21, 4, -90,
    -19, 4, -65, -88, -40, 81, 21, 28, 14, -84, -19, -41, 22, -58, 70, 10,
    -15, -100, -55, -91, -43, -41, -54, -58, -14, -84, -11, 20, -9, -24, 77,
    76, 10, -42, -72, -96, 48, 74, 29, -56, -36, -55, -70, 4, -34, -32, -95,
    -59, -33, 0, -62, 21, 88, -88, 64, -4, -31, -85, 40, 89, -52, -6, 13, 20,
    -84, 0, -16, 73, 74, -90, -40, 75, -20, -89, 96, -94, 40, -69, -6, -33,
    16, 10, -76, -75, 87, 2, -56, 23, 61, -23, -70, 4, -48, 8, 72, 59, -39,
    -56, -72, 50, -8, 15, -72, 93, 63, 24, 56, 51, -31, -53, 10, -85, -63,
    18, 87, 78, 59, -54, -11, 26, 27, 29, -41, -39, -26, 4, 46, -30, -5, 70,
    51, 30, -40, 89, 52, 69, -33, -50, 53, -32, -12, -38, -83, -70, 72, 43,
    18, 90, 65, -16, -67, -37, 71, -30, -24, -3, -72, 2, 0, -22, -44, 22,
    -56, 34, -63, -76, 92, 36, -99, 54, -10, 36, 74, -1, -94, -87, 56, 94,
    -50, 65, 67, 20, -31, 67, -37, 11, 54, -41, 59, -12, 75, 46, -50, -43,
    -79, -98, -87, 22, -32, -37, 48, -31, 71, 7, -45, -61, -48, 48, -33, -71,
    -66, 99, -47, 81, -36, -4, 23, -89, -89, -9, -67, 28, 19, -43, 44, 13, 6,
    -85, 40, 15, -86, 92, 51, 66, 61, 37, 53, -55, 80, 40, -32, 84, -39, -46,
    -85, 26, -92, 87, 40, 76, 23, 68, -54, 11, -83, -46, 30, 36, -68, -85,
    -22, -33, 82, -92, 47, -66, -98, -78, 17, 37, -46, -51, 6, 56, -55, 59,
    95, -23, 54, -21, -57, 77, 54, -35, -16, 11, 88, -5, 63, -24, 44, -64,
    17, -44, -39, 42, -51, 62, 68,
  61, 92, 91, 16, -6, 64, 17, 46, 83, -86, 82, -33, 65, -33, 35, -25, -55,
    -85, -95, -91, -70, -39, -71, 12, 97, 40, -45, 41, 42, -67, 68, 46, 61,
    -83, 92, -65, -69, -3, -19, 82, -40, -43, -32, 32, 76, -41, 91, 86, 12,
    23, -28, -23, 50, 46, -20, 11, -1, 38, -74, -46, -55, 50, 35, -71, -79,
    80, -46, -32, 85, -93, -88, 17, -30, 92, 91, -62, 65, 86, 61, -41, 14,
    84, 8, 64, -51, -63, -97, -26, -17, 42, -3, 85, -1, 57, -31, 54, 6, 74,
    -67, 63, -70, -12, 60, 9, 43, -67, 55, -82, -25, 33, -99, -24, -41, -20,
    26, 68, -7, 54, 75, -62, -54, -1, 76, -43, -41, -94, 22, -61, -95, 2,
    -72, 19, 61, 55, -85, -3, 34, -62, -73, -31, 43, -19, 87, 4, 53, 71, -76,
    -37, 92, -43, -55, -79, -94, -17, -33, 11, 67, -75, -75, -76, 35, 26,
    -36, -32, -62, 86, 50, -62, -90, 32, -80, 18, 10, -99, -20, 26, 43, 31,
    1, -19, -1, 10, -73, 78, -34, -80, 83, 55, -53, 99, -57, 45, 17, 3, -16,
    -60, 49, 33, 99, 17, -95, -10, -68, 49, 67, -47, 22, 56, -74, 31, -86,
    -94, 84, 15, -93, 89, -90, -93, -89, -21, -58, 65, 57, 25, 38, -58, 37,
    -82, -70, 41, 51, -64, -68, 7, -81, 3, -66, 83, -27, 20, 22, -68, -38,
    -14, 79, 34, -82, -30, 82, -76, -82, 78, -89, 36, -52, 27, 23, -64, -46,
    41, 16, 2, 86, 39, 56, -67, 77, 63, -89, -92, -21, 12, -51, -78, 59, -16,
    -81, 96, 0, 92, 85, -78, 36, -24, -35, 62, -62, 19, -89, -70, 26, 2, 68,
    -89, 17, -49, -31, 86, 87, -73, 59, 25, 94, -28, 58, -83, 53, 80, 99,
    -14, 16, -19, -9, 99, 75, -99, -98, -38, 92, -89, -40, -28, -48, -66, 23,
    -83, -97, -61, -59, 75, 57, 10, -55,
  -95, -6, 82, 62, -12, -86, -47, 64, -20, -66, 44, -82, -8, 80, 51, -94, 90,
    -84, -16, 88, 67, 13, -52, 93, -53, 98, 18, 0, 63, -19, 18, 50, 38, 75,
    50, 16, -51, 45, 14, -74, -32, -89, -20, 94, -28, -46, -74, 67, -82, -27,
    10, 69, 93, -66, 58, 82, -34, -68, 60, -86, 47, 64, 38, 38, -32, 18, 44,
    50, -65, 59, 72, 90, -24, -67, -94, 24, 39, -61, -100, 96, -11, -43, -63,
    74, -81, -19, -67, -54, -74, -76, -33, 16, 75, -2, 81, 68, -48, -73, -72,
    -26, -28, -63, 6, 20, 78, 70, -3, -66, 33, 56, -22, 87, -37, 94, -41, 10,
    -51, -33, -70, -83, -44, -82, 91, -21, -36, 24, 96, 40, -83, -18, -59,
    75, -13, 92, -65, -65, -1, -27, -33, 18, -61, -46, -16, -75, -90, 6, 74,
    -38, 66, -7, -24, 90, -12, 11, 90, -18, -37, 94, 3, 67, -11, 40, -15, 53,
    -62, -85, 90, 47, 19, 29, -42, -33, -93, -65, 16, 65, -68, -8, 98, -66,
    28, -95, 93, 15, -12, -34, -96, -15, -68, -97, 48, 97, -93, -85, -60,
    -23, 19, 21, -95, 42, 91, 15, -98, 51, 32, -19, 71, 78, -77, 27, -30,
    -53, 92, -67, -29, -53, -49, -10, 25, 12, 0, -80, -98, -28, 86, 38, 9,
    81, 36, -3, -19, -4, -72, -10, 96, 32, 8, 2, 42, -68, 9, 69, 44, -46, -5,
    71, 18, 68, -76, 98, 40, -18, -76, -19, 1, -75, -84, -99, 58, 10, -80,
    79, -49, 97, 53, 36, -70, -52, -34, -90, 35, 93, 40, 54, 86, -47, -29,
    -95, -39, -11, -49, -47, 97, 47, 46, 30, -48, 45, 85, -73, 39, 57, -71,
    97, 86, -28, 88, 12, -59, -93, -52, -26, 44, 79, 74, 55, 34, 63, 79, 48,
    26, -8, -34, 26, -26, 46, -29, 91, -82, -13, 17, -27, 27, 17, -64, -87,
    -87, -9, 92, 42, 87, -58, 7,
  89, -71, 63, -24, 23, -7, -60, 45, -87, 58, 45, -44, -26, -29, -80, -92,
    -91, 56, -95, 8, 9, -76, -18, -61, -88, -75, 20, -30, -46, 84, 95, 95,
    57, -17, 23, 70, -20, 16, 11, 79, -60, 24, -75, -87, -89, -91, 10, -91,
    -28, -35, 56, 65, -69, 46, -4, -2, 78, 84, 12, -28, 87, -93, 58, -60, 28,
    -30, -69, -10, 75, 71, 41, -30, -83, -78, 96, 55, 36, -50, 51, 45, 38,
    16, 40, -47, 8, 94, 33, -53, 5, -60, -42, -11, -98, 48, 36, -70, 40, -44,
    -37, 10, 2, 12, -2, 80, -69, -67, -65, -100, 6, 16, -36, -99, 70, 21,
    -13, 1, 37, 92, 60, 51, 46, 17, 17, -46, -47, -94, 71, 27, -72, -91, -13,
    -13, 73, 82, 55, -13, 75, -52, -88, 70, -35, 20, -44, 71, 74, -62, -21,
    70, -66, 18, 3, -50, -69, 21, 82, -85, -30, -7, 99, -88, -33, 73, 70,
    -100, 67, 57, 39, 95, -40, 65, 0, 56, -97, -54, 61, 23, -43, -15, 59, 72,
    -30, 14, -16, -20, -94, -38, 77, 73, -10, 94, 94, 58, -40, -69, -44, 98,
    -79, -92, 34, -47, -66, -48, -61, -94, -63, -27, 36, 18, -19, -41, 24,
    -62, 81, 42, -43, 25, 44, -20, 72, -86, -23, 1, 0, 66, -98, -96, 84, -82,
    -2, -42, -78, 29, -17, -18, -58, 60, 40, -42, 97, 26, 10, 24, -11, -69,
    -6, 67, 34, -97, -82, 3, -56, -13, 12, -63, -85, -86, 86, -60, -22, -52,
    89, 88, 99, 96, -82, 87, -40, 98, 91, 58, -50, 84, 73, -17, 91, -90, 40,
    -52, -30, -27, 46, -90, 89, -47, -1, -19, 54, -45, -57, -98, 62, -12, 71,
    -48, 84, -64, 40, 25, -34, 32, -83, 78, 19, -34, 61, 8, 65, 11, -73, 64,
    34, -9, -37, 45, 62, -68, -9, -8, 23, 50, 84, 35, -25, -77, 6, 63, 7, 47,
    39, 92, 30, -48, -82,
  7, -36, 98, -5, -90, 46, 3, -76, 53, 96, 26, -8, -30, 77, 29, 77, 48, 31,
    67, -68, 17, 15, -47, -57, -2, 10, -32, -30, 66, 97, 31, 6, 9, -89, -100,
    -3, -29, 93, 27, -81, -74, 85, -70, 95, 46, 41, -99, -34, 87, -93, 75,
    -32, -44, 3, -43, -15, 55, 36, 4, 49, 69, 93, -37, -100, -37, -29, 38,
    89, -71, 78, -87, -90, -47, 94, -87, -91, -99, -42, -91, -10, -25, -8,
    -94, -7, 72, 90, -67, -3, 38, -84, -93, 40, -22, -14, 57, 62, 55, 82, 40,
    48, 47, -87, -4, -35, 36, -71, 71, 73, 11, -58, 48, -16, -85, 37, 74, 26,
    -58, -88, 1, -30, 99, 17, 40, -59, 36, -72, -36, 10, 28, 34, -62, -73,
    20, -47, -19, 54, -97, 79, 68, -66, -69, -10, -11, -89, 12, -54, -100,
    -65, -62, 89, -18, -99, -72, -50, 87, 75, -45, 95, 50, -14, -70, -21,
    -92, -76, 79, -99, -7, -4, 23, -82, -85, 93, -53, -14, 14, -57, 56, 31,
    57, 75, -48, 90, 84, 12, 72, -67, -64, 39, 49, 57, -93, -16, 85, 30, -87,
    -12, 11, 51, 94, -47, -50, -95, 84, 0, -30, 11, -73, 64, 35, 24, -77,
    -64, 52, 11, 26, -11, 54, 24, -98, 94, 21, -58, -48, -99, 12, 25, -93,
    -70, -39, -84, -97, 66, -5, -84, -68, -79, 11, -97, -6, 21, 84, -81, -22,
    80, -39, -83, -72, 15, 55, -17, -7, -49, 89, -57, 84, -18, 26, -5, 52,
    -95, 95, -87, 73, -68, 92, 45, 12, 59, -14, 52, 94, 9, 52, -68, -61, -28,
    -72, 17, 28, 61, 46, 70, 61, -8, -73, -77, -89, 8, -27, -14, -60, 97,
    -70, 94, 5, -3, -25, -43, -65, -28, -76, 45, 99, 74, -93, -20, -96, 89,
    30, 81, 47, -39, -25, 17, -81, 83, 77, 21, -22, -7, 34, 3, -87, -74, 7,
    51, 40, 48, -62, 47, 69, 32, 92,
  -35, -64, -76, 76, 23, 76, 24, 80, -78, -57, 49, -62, 6, 45, 82, -34, -76,
    -36, -30, -19, -62, -88, -65, 84, 14, -8, 70, -4, -66, -59, 22, 88, -1,
    73, 93, 82, 86, -12, 50, -71, -88, 53, -36, -89, 13, -98, 93, 40, -43,
    -71, -92, -18, 94, -85, -94, -31, 66, 26, -36, 9, -9, -22, 2, 83, 77,
    -17, -69, -45, -30, -76, -58, -40, 79, 58, 58, -60, 71, 18, -70, -51, 44,
    -91, 38, -88, 0, -16, 5, 36, -83, -79, -3, -3, -25, 71, -65, 42, -83, 99,
    -40, 43, -31, 81, 78, -35, 59, -69, -6, -70, -40, -7, 12, -68, 7, 3, 97,
    38, -90, -51, -52, -27, 78, -17, 48, 89, 99, 39, -78, 43, -25, -58, 29,
    37, -23, -96, -65, -41, 79, -24, 60, 45, -34, -52, -46, 20, 65, 22, 46,
    38, -61, -53, 67, 52, 0, -56, -88, -25, -42, 59, -15, -32, -78, -98, -60,
    -96, 77, 53, 21, -23, -11, 61, 13, -74, -30, -14, 50, 35, -14, 42, 91,
    59, 66, -39, 7, -36, 79, -19, 5, -1, -91, 79, 47, 73, -53, 43, -13, 74,
    2, -57, 3, 53, -25, 16, -12, -15, 78, 96, 28, 45, 64, -12, 74, 60, -31,
    53, -2, -46, -78, 77, -77, 1, 94, -59, 57, -68, -85, 71, -59, 85, 44, 1,
    90, 50, -38, 16, 68, 72, -2, -9, -83, -78, -82, -45, -1, -29, 73, 60,
    -56, -16, 98, 9, -59, 58, -37, -80, 11, -99, -100, -62, -32, -40, 53,
    -15, 30, -3, -97, -35, 13, -3, 44, -3, -82, 21, 88, 29, -73, -28, 11,
    -49, -65, 20, -15, 49, -11, -92, -64, -73, 76, 62, -38, -37, 4, 41, 0,
    59, 6, 1, -80, -68, 56, 1, 3, -99, -68, -68, -66, 85, 75, 66, -30, -19,
    38, 17, 17, 68, 26, 26, -99, 39, 75, 43, -56, 0, -93, 33, 1, -7, 31, -87,
    -69, 79, 67, 43, -60,
  -5, 72, 63, -100, 84, -91, -70, -37, -89, -81, -28, -2, -56, -49, -46, -97,
    -44, -5, -37, 35, 25, 41, -16, 63, -100, -36, -83, 26, 24, -18, -42, -41,
    -18, -57, -12, 17, 99, -40, 66, -67, 63, 87, 67, 12, -34, -38, 91, -55,
    78, -5, -32, 45, 2, 97, 86, -53, -93, 87, 83, 38, -13, 12, 80, 38, 55,
    90, 3, 51, 57, -75, 17, -36, 39, -15, -36, -90, 57, -12, 92, 94, -33, 82,
    75, -79, -31, -39, 12, -30, -64, -81, 53, 36, -8, 47, 18, 88, 64, 73, -7,
    -61, -6, -64, 68, -97, 94, 64, -16, -88, 57, -11, -19, 45, 63, -52, -63,
    -30, 70, 98, -100, 99, -43, 42, -65, 57, 38, -80, 5, 14, 1, -80, -74, 22,
    -69, -31, 11, -21, -49, -48, -65, -63, 35, 10, -89, 10, 84, 4, -29, 77,
    -66, 87, -80, -21, -40, 18, -54, -97, 97, -65, 88, -14, -76, -81, 60,
    -18, -17, 53, 30, 81, -74, -28, -9, -43, -29, -87, -92, -47, -60, 47, 80,
    55, 81, -74, -35, 74, -39, 55, -6, -90, -76, 14, 8, 68, -42, 93, 90, 35,
    -35, -6, -54, -70, -73, 11, 30, -72, -89, 85, 55, 96, -73, -93, -9, -45,
    6, 2, -45, 85, 62, 19, -53, 5, -77, 87, 10, 35, -57, 71, 38, -93, -59,
    -71, -88, -18, 60, -84, 5, 20, 7, 62, -78, 35, -38, 93, 53, 69, -35, 88,
    -3, 77, 34, 71, 46, -70, -75, -67, 25, -67, 69, 31, 34, 18, -43, -55,
    -40, 5, 48, 64, -35, 25, -72, 57, -57, -20, 76, -80, -17, -50, 52, -27,
    -61, -18, -7, 17, 75, -77, -56, 11, 66, 48, 31, 71, 90, 3, -3, 89, 6,
    -58, -87, -75, -77, 26, -16, -49, 77, -7, -53, 47, 26, -80, -22, 34, -48,
    15, 3, 8, 15, 39, -63, 4, 41, -74, -94, -56, 78, -76, -37, -25, -58, -96,
    -90, 92, 36, 26, -63,
  60, -95, 48, -76, -45, 4, 81, 97, -40, -92, 83, 59, -99, -50, 61, -14, 11,
    92, 56, 85, 43, 75, 16, -59, -93, 58, 59, 95, 73, 68, -44, -59, 2, 90,
    70, -100, 33, -4, 37, 57, -94, -5, -35, 58, -3, 91, -74, 7, -27, 77, -28,
    43, 49, -56, -94, -13, 25, 16, -88, 40, -21, 28, 43, 66, 59, -4, -77, 82,
    -90, -100, 57, -8, 81, -14, 92, -63, -75, 13, 8, -59, -26, -10, -6, -59,
    -70, -62, 94, 10, 57, -35, -31, 59, -32, -29, -59, 87, 61, 4, 14, -53,
    -97, -63, 4, -53, -33, -44, -21, 80, -51, -31, -85, 71, -40, 12, 84, 78,
    -35, -81, 74, -96, -88, 59, 95, -8, 73, 97, 45, 27, 55, -23, -80, 10,
    -30, 30, 65, 82, 35, 47, -100, 28, 65, 84, 62, 99, -35, 5, 15, -44, -55,
    -56, -52, 32, 36, -31, 38, 76, -1, 98, -5, 71, -29, 73, -77, -44, -23,
    80, -26, -47, -3, 26, -76, 44, -50, 88, 31, -66, -7, -14, -91, 10, -92,
    31, -70, 43, -57, 98, 70, 91, 9, 28, 92, -48, -5, -64, 11, 85, -64, 61,
    22, 53, -63, 42, 69, 56, -12, -66, 23, -22, 0, 49, -41, 89, 81, 94, -33,
    65, -12, 98, -93, -68, -97, -94, -78, 50, -99, -57, 98, 88, -70, -64, 73,
    62, 38, -89, 3, 97, -64, -92, -30, 55, -25, 53, 5, -7, 24, 56, -85, -24,
    66, 71, -93, 6, 48, 23, -45, -49, -88, 33, -21, 52, -89, 49, -22, 49,
    -24, 65, 47, -45, 68, -36, -75, 22, 96, -91, 24, -75, 8, 17, -51, -34,
    -83, 3, 61, 54, 98, -42, -74, 85, 47, 9, -45, 51, -29, -90, -21, -39,
    -53, -40, -94, -39, -84, 98, -5, -48, -15, -30, 7, -64, -63, 21, -87,
    -46, 1, -72, -39, 29, -24, -97, 24, -60, 68, -22, 33, -8, -99, -7, 42,
    -73, 88, 8, -25, -35, 22,
  -22, 11, -55, 75, 55, -65, -90, 49, -43, -100, 7, -27, 6, 1, 39, -80, -58,
    -96, -100, 46, 28, -96, 45, -10, 99, 84, 10, -91, 22, 28, -69, 7, -64,
    15, 58, 71, -57, 69, 82, -7, 44, -14, -96, -100, 47, -68, -46, 96, 16,
    38, -71, -42, -47, -5, 25, -99, 87, 14, 36, 60, -47, 0, -20, -70, -83,
    16, 99, -97, -30, 71, 61, 63, 37, -40, -10, -24, -90, -84, 64, -92, -6,
    75, 89, 96, -89, -78, 33, -56, -78, 80, -64, 26, -59, 54, -58, -9, -29,
    -46, 29, 33, 73, -13, 24, -80, 54, 18, 51, -34, -75, -100, 62, -1, -98,
    17, -28, -18, 50, -86, -43, 97, 77, 13, 59, -66, -51, -45, 63, 75, 61,
    -64, -82, -74, 26, -49, -87, -80, -13, 93, 27, -53, 83, 8, -96, 81, 49,
    10, 22, -88, -16, 46, -27, -7, 85, -3, -6, 79, -82, -86, 93, 64, 18, -68,
    73, -22, -52, 54, -63, 72, -75, 26, -32, -72, -46, -64, 57, -31, -23,
    -87, 73, 88, 99, -26, 68, 70, 12, -76, -27, -57, 65, -49, 25, 10, -45,
    -42, 9, -35, 49, -14, -54, 47, -81, -10, 10, -45, 36, -73, -32, -22, 64,
    60, 53, -39, -90, -89, -11, -51, -69, -83, 53, 4, -1, 89, 51, 43, 75,
    -12, 34, -15, -39, 74, 20, -74, -80, 11, -68, -4, 21, 56, 93, -33, -85,
    5, 35, 72, -10, -40, -35, 0, 2, 68, -8, 64, 10, -47, 16, -9, -77, -4,
    -90, -81, -92, 23, 40, 44, -72, -52, 54, -51, 27, -83, 73, 72, -73, -58,
    -68, -9, 54, -24, 67, -21, -27, 34, -100, 71, -81, 49, -56, -21, 26, 29,
    -22, -37, 51, -71, 26, -32, -29, 50, -10, 48, 26, 21, 54, 11, -72, -11,
    61, -34, -11, -70, 60, -2, 40, -36, 69, -66, 33, -74, -40, 27, -85, 88,
    14, 29, -29, 18, 33, 26, -55, -79, -64, 32, 22,
  71, 57, -76, -50, -29, -68, 79, -9, -3, -26, 0, -21, -16, -98, -15, -28,
    -100, -55, -66, -41, -88, -27, -31, -20, 23, -58, -4, -6, 61, -25, 85,
    -16, -31, 66, -7, 78, -80, -88, 98, 13, -6, 30, -78, 2, 45, -64, -48,
    -84, -94, -63, 94, 76, 13, 45, 98, 61, 7, -85, 93, 21, 84, 98, 88, -74,
    77, -5, 9, -70, -70, 8, -71, -94, 45, 9, -41, 83, 22, -39, -72, -28, -11,
    -79, 77, 65, 39, -41, -21, 89, 0, -46, 82, 55, -14, 26, -63, 92, 54, -68,
    -98, 90, 29, 47, 1, -23, 25, 10, -62, 67, -52, 46, 20, -22, -7, 27, 79,
    42, -94, 89, 37, 93, 98, 21, -35, 37, 48, -1, 85, 80, -10, 21, -83, -73,
    -18, -56, -63, -22, 18, -91, 35, 96, -54, 35, -23, 85, 46, 57, -55, 30,
    73, 31, 62, 80, -1, -56, -64, -32, -54, -1, -41, -94, -40, 19, 62, 35,
    91, -21, -1, 31, 79, -39, 9, -66, 64, -41, 40, 22, -37, -4, -56, 46, 46,
    -29, 64, 38, 1, -3, -2, 45, 87, -8, -10, 46, 51, 64, 77, 98, -73, -13,
    47, 9, 24, -78, 18, 18, -73, 85, 60, 22, 91, -67, 96, 8, -97, 31, 23,
    -42, 28, -52, -46, -81, -25, 68, 54, 7, -10, 13, -70, -37, 21, 12, -31,
    99, 26, 79, 34, 29, -82, -2, 34, -70, -68, -78, -5, -54, 46, 55, 41, -19,
    69, -47, 80, -94, -46, -56, -33, -82, 26, -7, 74, -7, 93, 29, -61, -87,
    -90, 7, -2, -64, -97, -76, 85, 11, 52, -1, -29, 56, 80, 4, -76, 82, 68,
    33, -85, 71, -67, -53, -78, -12, -52, 98, -83, -25, -51, -88, 47, 39, 64,
    -89, 86, 77, 85, -67, 49, -85, -39, -15, -18, 32, 4, -96, -59, -34, 97,
    84, 65, 81, 94, -29, -49, -16, -35, 72, -56, 39, 85, 93, -26, -46, -36,
    73, -82, -95, -6
};
#endif
