/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#include "gesturearea.h"

#include <QApplication>
#include <QDesktopWidget>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QGraphicsWidget>

#include "device.h"
#include "geissingleton.h"
#include "gesture.h"
#include "gestureevent.h"

GestureArea::GestureArea(QDeclarativeItem* parent, Primitive primitive)
  : QDeclarativeItem(parent),
    primitive_(primitive),
    window_id_(0),
    subscription_(parent),
    enabled_(true) {
  connect(GeisSingleton::Instance(), SIGNAL(Initialized()),
          SLOT(GeisInitialized()));
  connect(&subscription_, SIGNAL(devicesChanged()),
          SLOT(SubscriptionChanged()));
  connect(&subscription_, SIGNAL(_UNITY_globalGestureChanged()),
          SLOT(SubscriptionChanged()));
  connect(subscription_.touches(), SIGNAL(startChanged()),
          SLOT(SubscriptionChanged()));
}

QVariant GestureArea::itemChange(GraphicsItemChange change,
                                 const QVariant& value) {
  if (change == QGraphicsItem::ItemSceneHasChanged) {
    if (subscription_._UNITY_globalGesture())
      window_id_ = QApplication::desktop()->winId();
    else
      window_id_ = scene()->views().first()->winId();

    if (window_id_) {
      if (GeisSingleton::Instance()->initialized())
        BuildSubscription(&subscription_);
    } else {
      qCritical("Failed to determine window ID of GestureArea");
    }
  }
  return QDeclarativeItem::itemChange(change, value);
}

void GestureArea::GeisInitialized() {
  if (window_id_)
    BuildSubscription(&subscription_);
}

#define ADD_FILTER_TERM(filter, facility, term ...) \
  if (geis_filter_add_term((filter), (facility), \
                           ##term) != GEIS_STATUS_SUCCESS) { \
    qCritical("Failed to add term to filter"); \
    throw; \
  }

void GestureArea::SubscriptionChanged() {
  GeisSingleton::Instance()->UnregisterGestureArea(this);
  subscription_.DeleteSubscription();

  if (window_id_) {
    if (subscription_._UNITY_globalGesture())
      window_id_ = QApplication::desktop()->winId();
    else
      window_id_ = scene()->views().first()->window()->effectiveWinId();

    if (window_id_) {
      BuildSubscription(&subscription_);
      GeisSingleton::Instance()->RegisterGestureArea(this);
    }
  }
}

void GestureArea::BuildSubscription(Gesture* gesture) {
  GeisStatus status;
  GeisSubscriptionFlags flags;
  GeisSubscription subscription;

  /* Return if already built */
  if (gesture->subscription())
    return;

  Geis geis = GeisSingleton::Instance()->geis();
  GeisFilter filter = geis_filter_new(geis, "GestureArea subscription filter");
  if (!filter) {
    qCritical("Failed to create GeisFilter for subscription");
    return;
  }

  ADD_FILTER_TERM(filter,
                  GEIS_FILTER_REGION,
                  GEIS_REGION_ATTRIBUTE_WINDOWID,
                  GEIS_FILTER_OP_EQ,
                  window_id_,
                  NULL);

  switch (gesture->devices()) {
    case Gesture::TouchScreens:
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_TRUE,
                      NULL);
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_FALSE,
                      NULL);
      break;

    case Gesture::TouchPads:
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_FALSE,
                      NULL);
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_FALSE,
                      NULL);
      break;

    case Gesture::Independents:
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_FALSE,
                      NULL);
      ADD_FILTER_TERM(filter,
                      GEIS_FILTER_DEVICE,
                      GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH,
                      GEIS_FILTER_OP_EQ,
                      GEIS_TRUE,
                      NULL);
      break;

    default: /* Gesture::All */
      break;
  }

  const char* primitive;
  switch (primitive_) {
    case kDrag:
      primitive = GEIS_GESTURE_DRAG;
      break;

    case kPinch:
      primitive = GEIS_GESTURE_PINCH;
      break;

    case kRotate:
      primitive = GEIS_GESTURE_ROTATE;
      break;

    case kTap:
      primitive = GEIS_GESTURE_TAP;
      break;

    default:
      qCritical("Bad gesture type when building subscription (%d)", primitive_);
      goto error;
  }

  ADD_FILTER_TERM(filter,
                  GEIS_FILTER_CLASS,
                  GEIS_CLASS_ATTRIBUTE_NAME,
                  GEIS_FILTER_OP_EQ,
                  primitive,
                  NULL);
  ADD_FILTER_TERM(filter,
                  GEIS_FILTER_CLASS,
                  GEIS_GESTURE_ATTRIBUTE_TOUCHES,
                  GEIS_FILTER_OP_EQ,
                  gesture->touches()->start(),
                  NULL);

  flags =
    subscription_._UNITY_globalGesture() ? (GeisSubscriptionFlags)(
      GEIS_SUBSCRIPTION_GRAB
      |
      GEIS_SUBSCRIPTION_CONT)
    : GEIS_SUBSCRIPTION_NONE;
  subscription = geis_subscription_new(geis, "GestureArea subscription", flags);
  if (!subscription) {
    qCritical("Failed to create GeisSubscription");
    goto error;
  }

  status = geis_subscription_add_filter(subscription, filter);
  if (status != GEIS_STATUS_SUCCESS) {
    qCritical("Failed to add filter to subscription");
    geis_subscription_delete(subscription);
    goto error;
  }

  subscription_.set_subscription(subscription);

  if (!subscription_.EnableSubscription())
    goto error;

  GeisSingleton::Instance()->RegisterGestureArea(this);

 error:
  geis_filter_delete(filter);
}

void GestureArea::set_enabled(bool new_value) {
  if (enabled_ != new_value) {
    if (new_value == true) {
      if (!subscription_.EnableSubscription())
        return;
    } else if (!subscription_.DisableSubscription()) {
      return;
    }

    enabled_ = new_value;
    emit enabledChanged();
  }
}

bool GestureArea::IsGestureEventHandled(GestureEvent* event) {
  float x;
  float y;
  if (primitive_ == kDrag || primitive_ == kTap) {
    x = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_POSITION_X).toFloat();
    y = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_POSITION_Y).toFloat();
  } else {
    x = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_CENTROID_X).toFloat();
    y = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_CENTROID_Y).toFloat();
  }

  QPointF point(x, y);

  if (!subscription_._UNITY_globalGesture() &&
    event->device()->device_type() == Device::TouchScreen) {
    point = scene()->views().first()->mapFromGlobal(point.toPoint());
    point = mapFromScene(point);
  }

  centroid_.set(point);

  x = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_FOCUS_X).toFloat();
  y = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_FOCUS_Y).toFloat();
  if (subscription_._UNITY_globalGesture()) {
    focus_ = QPointF(x, y);
  } else {
    point = scene()->views().first()->mapFromGlobal(QPoint(x, y));
    focus_ = mapFromScene(point);
  }
  emit focusChanged();

  return centroid_.receivers(SIGNAL(initialChanged())) > 0 ||
         centroid_.receivers(SIGNAL(currentChanged())) > 0 ||
         receivers(SIGNAL(focusChanged())) > 0;
}

void GestureArea::HandleGestureUpdateEvent(bool end,
                                                GestureEvent* event) {
  float x;
  float y;
  if (primitive_ == kDrag) {
    x = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_POSITION_X).toFloat();
    y = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_POSITION_Y).toFloat();
  } else {
    x = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_CENTROID_X).toFloat();
    y = event->attributes().value(GEIS_GESTURE_ATTRIBUTE_CENTROID_Y).toFloat();
  }

  centroid_.update(QPointF(x, y));
}

GestureArea::~GestureArea() {
  GeisSingleton::Instance()->UnregisterGestureArea(this);
}
