/*
 *  Copyright (c) 2007-2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>

// OpenCTL Headers
#include <OpenCTL/Module.h>
#include <OpenCTL/ModulesManager.h>
#include <OpenCTL/Version.h>
#include <OpenCTL/compiler/Compiler.h>
#include <GTLCore/CompilationMessages.h>

void printVersion()
{
  std::cout << OpenCTL::LibraryShortName() << " - " << OpenCTL::LibraryName() << " - " << OpenCTL::LibraryVersionString() << std::endl;
  std::cout << OpenCTL::LibraryCopyright() << std::endl;
  std::cout << "CTL Version : " << OpenCTL::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : ctlc [option] fileName.ctl" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -S --asm-source         print the assembly source code generated" << std::endl;
  std::cout << "  -C --c-source           print the source code as C" << std::endl;
  std::cout << "  -L --module-dir         add a location where to find modules" << std::endl;
  std::cout << "     --no-std-lib         no standard library" << std::endl; // TODO: only show this options when tests are enabled
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  bool showAssembly = false;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-S","--asm-source")) {
      showAssembly = true;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc - 1 )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        OpenCTL::ModulesManager::instance()->addDirectory(argv[ai]);
      }
    } else if( argv[ai] == GTLCore::String("--no-std-lib") ) {
      OpenCTL::Compiler::disableStandardLibrary();
    } else {
      if( ai != argc - 1)
      {
        std::cerr << "Invalid command line parameters." << std::endl;
      } else {
        fileName = argv[ai];
      }
    }
  }
  if( fileName == "")
  {
    printHelp();
  } else {
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    std::string str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    OpenCTL::Module p;
    p.setSource( fileName, source );
    p.compile();
    if(not p.isCompiled())
    {
      std::cout << "Error: " << std::endl << p.compilationMessages().toString() << std::endl;
      return EXIT_FAILURE;
    }
    if( showAssembly )
    {
      std::cout << p.asmSourceCode();
    }
  }
  return EXIT_SUCCESS;
}
