/*
 *  Copyright (c) 2008, 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "LibrariesManager.h"

#include <map>

// LLVM
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#else
#include <llvm/Support/Path.h>
#endif

#include "GTLCore/Debug.h"
#include "Library.h"

#include <GTLCore/Macros_p.h>

using namespace GTLFragment;

struct LibrariesManager::Private {
  std::map<GTLCore::String, Library*> libraries;
  std::list<GTLCore::String> directories;
  AbstractLibrariesManager* gaisuManager;
  AbstractLibrariesManager* shivaManager;
  AbstractLibrariesManager* rijnManager;
};

STATIC_POINTER(LibrariesManager, s_instance);

LibrariesManager::LibrariesManager() : d(new Private)
{
  addDirectory( "." );
  addDirectory( _GTLFRAGMENT_STD_LIB_SRC_DIR_ );
  addDirectory( _GTLFRAGMENT_LIBRARIES_DIR_ );
  d->gaisuManager = this;
  d->shivaManager = 0;
  d->rijnManager = 0;
}

LibrariesManager::~LibrariesManager()
{
  for(std::map<GTLCore::String, Library*>::iterator it = d->libraries.begin();
      it != d->libraries.end(); ++it)
  {
    delete it->second;
  }
  delete d;
}

void LibrariesManager::addDirectory(const GTLCore::String& directory)
{
  d->directories.push_back( directory );
}

Library* LibrariesManager::library(const GTLCore::String& name, int _channelsNb)
{
  std::map<GTLCore::String, Library*>::iterator it = d->libraries.find(name);
  if( it == d->libraries.end())
  {
    return 0;
  }
  return it->second;
}


Library* LibrariesManager::loadLibrary( const GTLCore::String& name, int _channelsNb)
{
  Library* l = library( name, _channelsNb );
  if(not l )
  {
    GTLCore::String sourceName = name + ".fragment";
    for( std::list<GTLCore::String>::iterator it = d->directories.begin();
        it != d->directories.end(); ++it )
    {
      llvm::sys::Path path( (const std::string&)*it );
      path.appendComponent( (const std::string&)sourceName);
      GTL_DEBUG("try " << path.c_str() );
      if(path.exists() and path.canRead())
      {
        l = new Library( Library::FRAGMENT_LIBRARY, _channelsNb );
        registerLibrary( name, l );
        l->loadFromFile( path.c_str());
        break;
      }
    }
  }
  return l;
}

Library* LibrariesManager::loadLibrary( const GTLCore::String& filename, int _channelsNb, Library::Type _type)
{
  switch(_type) {
    case Library::FRAGMENT_LIBRARY:
      // Will be handled after the switch
      break;
    case Library::SHIVA_KERNEL:
    case Library::SHIVA_LIBRARY:
    {
      GTL_ASSERT(d->shivaManager);
      Library* l = d->shivaManager->loadLibrary(filename, _channelsNb);
      if(l) return l;
    }
      break;
    case Library::RIJN_SKETCH:
    case Library::RIJN_LIBRARY:
    {
      GTL_ASSERT(d->rijnManager);
      Library* l = d->rijnManager->loadLibrary(filename, _channelsNb);
      if(l) return l;
    }
      break;
  }
  return loadLibrary(filename, _channelsNb);
}

void LibrariesManager::registerLibrary(const GTLCore::String& name, Library* l)
{
  d->libraries[ name ] = l;
}

void LibrariesManager::registerLibrariesManager(AbstractLibrariesManager* manager)
{
  switch(manager->libraryType())
  {
    case Library::SHIVA_LIBRARY:
      GTL_ASSERT(d->shivaManager == 0);
      d->shivaManager = manager;
      break;
    case Library::RIJN_LIBRARY:
      GTL_ASSERT(d->rijnManager == 0);
      d->rijnManager = manager;
      break;
    default:
      GTL_ABORT("Unsupported type");
  }
}

LibrariesManager* LibrariesManager::instance()
{
  if( not s_instance )
  {
    s_instance = new LibrariesManager;
  }
  return s_instance;
}

Library::Type LibrariesManager::libraryType() const
{
  return Library::FRAGMENT_LIBRARY;
}
