///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file Base.h
 * \brief This header file includes the standard system headers used by the basic classes.
 *
 * This header file is included by every .cpp file in the project to facilitate
 * the generation of precompiled headers. This feature is mainly used with the
 * Microsoft Visual C++ compiler.
 */

#ifndef __OVITO_BASE_H
#define __OVITO_BASE_H

#if WIN32 && !defined(OVITO_MONOLITHIC_BUILD)
  #define DLL_IMPORT_DECL	__declspec(dllimport)
  #define DLL_EXPORT_DECL	__declspec(dllexport)
#else
  #define DLL_IMPORT_DECL
  #define DLL_EXPORT_DECL
#endif

/******************************************************************************
* Define the DLL export macros.
******************************************************************************/
#ifdef MAKING_MODULE_BASE
  #define BASE_DLLEXPORT DLL_EXPORT_DECL
#else
  #define BASE_DLLEXPORT DLL_IMPORT_DECL
#endif

namespace Base {

/******************************************************************************
* Define the number type used for numerical computations.
*
* The CORE uses FloatType as the default value type for its computations.
******************************************************************************/
#ifndef USE_DOUBLE_PRECISION_FP

	// Use 32-bit float as default floating point type.
	typedef float FloatType;
	#define FLOATTYPE_FLOAT		// This tells the program that we're using 32-bit floating point.
	/// A small epsilon value for the FloatType.
	#define FLOATTYPE_EPSILON	1e-6f

#else

	// Use 64-bit double as default floating point type.
	typedef double FloatType;
	#define FLOATTYPE_DOUBLE	// This tells the program that we're using 64-bit floating point.
	/// A small epsilon value for the FloatType.
	#define FLOATTYPE_EPSILON	1e-9

#endif

/// Computes the square of a number.
template<typename T> inline T square(const T& f) { return f*f; }

/// The maximum value for floating point variables.
#define FLOATTYPE_MAX	(numeric_limits<FloatType>::max())
#define FLOATTYPE_PI	((FloatType)3.14159265358979323846)

}; // End of namespace Base

/// Include the standard system header like STL, BOOST library etc.
/******************************************************************************
* Standard Template Library (STL)
******************************************************************************/

// Microsoft VC 8 specifc
#ifndef _CRT_SECURE_NO_DEPRECATE
	#define _CRT_SECURE_NO_DEPRECATE 1	// Tell the compiler to stop complaining about using localtime etc...
#endif
#ifndef _SCL_SECURE_NO_DEPRECATE
	#define _SCL_SECURE_NO_DEPRECATE 1	// Stop complaining about std::copy etc...
#endif
#ifndef _CRT_NONSTDC_NO_DEPRECATE
	#define _CRT_NONSTDC_NO_DEPRECATE 1	// To stop complaining about old POSIX names.
#endif

#include <string>
#include <sstream>
#include <fstream>
#include <map>
#include <deque>
#include <stack>
#include <exception>
#include <iostream>
#include <ostream>
#include <istream>
#include <ios>
#include <limits>
#include <queue>
#include <locale>
#include <vector>
#include <complex>
#include <algorithm>
#include <numeric>
#include <set>
#include <cstdlib>
#include <valarray>

// use the namespace of the STL
using namespace std;

/******************************************************************************
* C Run-Time Library
******************************************************************************/
#include <math.h>

/******************************************************************************
* Boost Library
******************************************************************************/
#include <boost/static_assert.hpp>
#include <boost/utility.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/scoped_array.hpp>
#include <boost/intrusive_ptr.hpp>
#include <boost/get_pointer.hpp>
#include <boost/cstdint.hpp>
#include <boost/array.hpp>
#include <boost/dynamic_bitset.hpp>
#include <boost/type_traits.hpp>
using namespace boost;


/******************************************************************************
* QT Library
******************************************************************************/
#include <QtGlobal>
#include <QChar>
#include <QString>
#include <QVector>
#include <QDataStream>
#include <QIODevice>
#include <QFile>
#include <QMessageBox>
#include <QDebug>
#include <QApplication>
#include <QDateTime>
#include <QDomDocument>
#include <QDomElement>
#include <QDockWidget>
#include <QMetaType>
#include <QRectF>
#include <QVariant>
#include <QtConcurrentMap>
#include <QtConcurrentFilter>

#ifdef max
#undef max
#endif

#ifdef min
#undef min
#endif


/******************************************************************************
* Include the debugging stuff.
******************************************************************************/
#include <base/utilities/Debugging.h>
#include <base/utilities/Logger.h>
#include <base/io/SaveStream.h>
#include <base/io/LoadStream.h>

// Make all classes of the Base namespace available by default.
namespace Base {};
using namespace Base;

#endif // __OVITO_BASE_H
