///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/data/importexport/ImportExportManager.h>
#include <core/plugins/PluginManager.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ImporterExporter, RefTarget)

///////////////////////////// SINGLETON CLASS METHODS ///////////////////////////////

/// The singleton instance of this class.
ImportExportManager* ImportExportManager::_singletonInstance = NULL;

/******************************************************************************
* Initializes the manager.
******************************************************************************/
ImportExportManager::ImportExportManager()
{
	// Scan for installed import/export plugins.
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(ImporterExporter))) {

        // Extract properties of the plugin class from manifest metadata.
		QDomElement descriptionElement = clazz->getMetaData("Import-Export-Plugin-Description");
		if(descriptionElement.isNull())
			continue;

		ImportExportDescriptor descr;
		descr._pluginClass = clazz;
		descr._fileFilter = descriptionElement.attribute("File-Filter");
		descr._fileFilterDescription = descriptionElement.attribute("File-Filter-Description");

		if(descriptionElement.attribute("Import") == "true")
			_importers.push_back(descr);
		if(descriptionElement.attribute("Export") == "true")
			_exporters.push_back(descr);
	}
}

/******************************************************************************
* Tries to detect the format of the given file.
******************************************************************************/
ImporterExporter::SmartPtr ImportExportManager::autodetectFileFormat(const QString& file)
{
	Q_FOREACH(const ImportExportDescriptor& importerType, importers()) {
		try {
			ImporterExporter::SmartPtr importer = importerType.createService();
			if(importer && importer->checkFileFormat(file))
				return importer;
		}
		catch(const Exception& ex) {
			ex.showError();
		}
	}
	return NULL;
}

};
