///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/viewport/OpenGLInterface.h>

namespace Core {

/******************************************************************************
* Error handling function.
* Returns a human-readable string that describes the OpenGL error with
* the given code.
******************************************************************************/
const char* openglErrorString(GLenum errorCode)
{
	switch(errorCode) {
	case GL_NO_ERROR: return "GL_NO_ERROR - No error has been recorded.";
	case GL_INVALID_ENUM: return "GL_INVALID_ENUM - An unacceptable value is specified for an enumerated argument.";
	case GL_INVALID_VALUE: return "GL_INVALID_VALUE - A numeric argument is out of range.";
	case GL_INVALID_OPERATION: return "GL_INVALID_OPERATION - The specified operation is not allowed in the current state.";
	case GL_STACK_OVERFLOW: return "GL_STACK_OVERFLOW - This command would cause a stack overflow.";
	case GL_STACK_UNDERFLOW: return "GL_STACK_UNDERFLOW - This command would cause a stack underflow.";
	case GL_OUT_OF_MEMORY: return "GL_OUT_OF_MEMORY - There is not enough memory left to execute the command.";
	case GL_TABLE_TOO_LARGE: return "GL_TABLE_TOO_LARGE - The specified table exceeds the implementation's maximum supported table size.";
	default: return "Unkown OpenGL error code.";
	}
}

/******************************************************************************
* Checks if a single OpenGL extension is available.
******************************************************************************/
bool OpenGLExtensions::isExtensionSupported(const char* szTargetExtension)
{
	const unsigned char *pszExtensions = NULL;
	const unsigned char *pszStart;
	unsigned char *pszWhere, *pszTerminator;

	// Extension names should not have spaces
	pszWhere = (unsigned char *)strchr(szTargetExtension, ' ');
	if(pszWhere || *szTargetExtension == '\0')
		return false;

	// Get extensions list
	pszExtensions = glGetString(GL_EXTENSIONS);

	// Search The Extensions String For An Exact Copy
	pszStart = pszExtensions;
	for(;;) {
		pszWhere = (unsigned char *)strstr((const char *)pszStart, szTargetExtension);
		if(!pszWhere)
			break;
		pszTerminator = pszWhere + strlen(szTargetExtension);
		if(pszWhere == pszStart || *(pszWhere - 1) == ' ')
			if(*pszTerminator == ' ' || *pszTerminator == '\0')
				return true;
		pszStart = pszTerminator;
	}
	return false;
}

/******************************************************************************
* Returns the address of an OpenGL extension procedure.
******************************************************************************/
void* OpenGLExtensions::GetProcAddress(const char* procedureName)
{
#if defined(Q_WS_WIN)	
	return wglGetProcAddress(procedureName);
#elif defined(Q_WS_X11)	
	return (void*)glXGetProcAddress((const GLubyte*)procedureName);
#else
	OVITO_ASSERT_MSG(QGLContext::currentContext() != NULL, "OpenGLExtensions::GetProcAddress", "No GL context is active.");
	return QGLContext::currentContext()->getProcAddress(procedureName);
#endif	
}

/******************************************************************************
* Detects the available OpenGL extensions.
******************************************************************************/
void OpenGLExtensions::detectOpenGLExtensions()
{	
	_driverVendor = (const char*)glGetString(GL_VENDOR);
	_rendererName = (const char*)glGetString(GL_RENDERER);
	
	glNewBufferRegion = (PFNGLNEWBUFFERREGIONPROC)GetProcAddress("glNewBufferRegion");
	glDeleteBufferRegion = (PFNGLDELETEBUFFERREGIONPROC)GetProcAddress("glDeleteBufferRegion");
	glBufferRegionEnabled = (PFNGLBUFFERREGIONENABLEDPROC)GetProcAddress("glBufferRegionEnabled");
	glReadBufferRegion = (PFNGLREADBUFFERREGIONPROC)GetProcAddress("glReadBufferRegion");
	glDrawBufferRegion = (PFNGLDRAWBUFFERREGIONPROC)GetProcAddress("glDrawBufferRegion");
	_hasBufferRegions = isExtensionSupported("GL_KTX_buffer_region") && (glNewBufferRegion != NULL);

	glLockArrays = (PFNGLLOCKARRAYSEXTPROC)GetProcAddress("glLockArraysEXT");
	glUnlockArrays = (PFNGLUNLOCKARRAYSEXTPROC)GetProcAddress("glUnlockArraysEXT");
	_hasCompiledVertexArrays = isExtensionSupported("GL_EXT_compiled_vertex_array") && (glLockArrays != NULL);

	glAddSwapHintRectWIN = (PFNGLADDSWAPHINTRECTWINPROC)GetProcAddress("glAddSwapHintRectWIN");
	_hasSwapHint = isExtensionSupported("GL_WIN_swap_hint") && (glAddSwapHintRectWIN != NULL);

	glPointParameterfARB  = (PFNGLPOINTPARAMETERFARBPROC)GetProcAddress("glPointParameterfARB");
	glPointParameterfvARB = (PFNGLPOINTPARAMETERFVARBPROC)GetProcAddress("glPointParameterfvARB");
	_hasPointParameters = isExtensionSupported("GL_ARB_point_parameters") && (glPointParameterfARB != NULL);

	glGenBuffersARB = (PFNGLGENBUFFERSARBPROC)GetProcAddress("glGenBuffersARB");
	glBindBufferARB = (PFNGLBINDBUFFERARBPROC)GetProcAddress("glBindBufferARB");
	glBufferDataARB = (PFNGLBUFFERDATAARBPROC)GetProcAddress("glBufferDataARB");
	glDeleteBuffersARB  = (PFNGLDELETEBUFFERSARBPROC)GetProcAddress("glDeleteBuffersARB");
	glMapBufferARB = (PFNGLMAPBUFFERARBPROC)GetProcAddress("glMapBufferARB");
	glUnmapBufferARB = (PFNGLUNMAPBUFFERARBPROC)GetProcAddress("glUnmapBufferARB");
	glGetBufferParameterivARB = (PFNGLGETBUFFERPARAMETERIVARBPROC)GetProcAddress("glGetBufferParameterivARB");
	glGetBufferPointervARB = (PFNGLGETBUFFERPOINTERVARBPROC)GetProcAddress("glGetBufferPointervARB");
	_hasVertexBufferObjects = isExtensionSupported("GL_ARB_vertex_buffer_object") && (glGenBuffersARB != NULL);

	glIsRenderbufferEXT = (PFNGLISRENDERBUFFEREXTPROC)GetProcAddress("glIsRenderbufferEXT");
	glBindRenderbufferEXT = (PFNGLBINDRENDERBUFFEREXTPROC)GetProcAddress("glBindRenderbufferEXT");
	glDeleteRenderbuffersEXT = (PFNGLDELETERENDERBUFFERSEXTPROC)GetProcAddress("glDeleteRenderbuffersEXT");
	glGenRenderbuffersEXT = (PFNGLGENRENDERBUFFERSEXTPROC)GetProcAddress("glGenRenderbuffersEXT");
	glRenderbufferStorageEXT = (PFNGLRENDERBUFFERSTORAGEEXTPROC)GetProcAddress("glRenderbufferStorageEXT");
	glGetRenderbufferParameterivEXT = (PFNGLGETRENDERBUFFERPARAMETERIVEXTPROC)GetProcAddress("glGetRenderbufferParameterivEXT");
	glIsFramebufferEXT = (PFNGLISFRAMEBUFFEREXTPROC)GetProcAddress("glIsFramebufferEXT");
	glBindFramebufferEXT = (PFNGLBINDFRAMEBUFFEREXTPROC)GetProcAddress("glBindFramebufferEXT");
	glDeleteFramebuffersEXT = (PFNGLDELETEFRAMEBUFFERSEXTPROC)GetProcAddress("glDeleteFramebuffersEXT");
	glGenFramebuffersEXT = (PFNGLGENFRAMEBUFFERSEXTPROC)GetProcAddress("glGenFramebuffersEXT");
	glCheckFramebufferStatusEXT = (PFNGLCHECKFRAMEBUFFERSTATUSEXTPROC)GetProcAddress("glCheckFramebufferStatusEXT");
	glFramebufferTexture1DEXT = (PFNGLFRAMEBUFFERTEXTURE1DEXTPROC)GetProcAddress("glFramebufferTexture1DEXT");
	glFramebufferTexture2DEXT = (PFNGLFRAMEBUFFERTEXTURE2DEXTPROC)GetProcAddress("glFramebufferTexture2DEXT");
	glFramebufferTexture3DEXT = (PFNGLFRAMEBUFFERTEXTURE3DEXTPROC)GetProcAddress("glFramebufferTexture3DEXT");
	glFramebufferRenderbufferEXT = (PFNGLFRAMEBUFFERRENDERBUFFEREXTPROC)GetProcAddress("glFramebufferRenderbufferEXT");
	glGetFramebufferAttachmentParameterivEXT = (PFNGLGETFRAMEBUFFERATTACHMENTPARAMETERIVEXTPROC)GetProcAddress("glGetFramebufferAttachmentParameterivEXT");
	glGenerateMipmapEXT = (PFNGLGENERATEMIPMAPEXTPROC)GetProcAddress("glGenerateMipmapEXT");
	_hasFrameBuffer = isExtensionSupported("GL_EXT_framebuffer_object") && (glIsRenderbufferEXT != NULL);

	glCreateShaderObjectARB = (PFNGLCREATESHADEROBJECTARBPROC)GetProcAddress("glCreateShaderObjectARB");
	glShaderSourceARB = (PFNGLSHADERSOURCEARBPROC)GetProcAddress("glShaderSourceARB");
	glCompileShaderARB = (PFNGLCOMPILESHADERARBPROC)GetProcAddress("glCompileShaderARB");
	glDeleteObjectARB = (PFNGLDELETEOBJECTARBPROC)GetProcAddress("glDeleteObjectARB");
	glCreateProgramObjectARB = (PFNGLCREATEPROGRAMOBJECTARBPROC)GetProcAddress("glCreateProgramObjectARB");
	glAttachObjectARB = (PFNGLATTACHOBJECTARBPROC)GetProcAddress("glAttachObjectARB");
	glLinkProgramARB = (PFNGLLINKPROGRAMARBPROC)GetProcAddress("glLinkProgramARB");
	glUseProgramObjectARB = (PFNGLUSEPROGRAMOBJECTARBPROC)GetProcAddress("glUseProgramObjectARB");
	glGetInfoLogARB = (PFNGLGETINFOLOGARBPROC)GetProcAddress("glGetInfoLogARB");
	glGetHandleARB = (PFNGLGETHANDLEARBPROC)GetProcAddress("glGetHandleARB");
	glGetObjectParameterfvARB = (PFNGLGETOBJECTPARAMETERFVARBPROC)GetProcAddress("glGetObjectParameterfvARB");
	glGetObjectParameterivARB = (PFNGLGETOBJECTPARAMETERIVARBPROC)GetProcAddress("glGetObjectParameterivARB");
	glGetUniformLocationARB = (PFNGLGETUNIFORMLOCATIONARBPROC)GetProcAddress("glGetUniformLocationARB");
	glUniform1fARB = (PFNGLUNIFORM1FARBPROC)GetProcAddress("glUniform1fARB");
	glUniform2fARB = (PFNGLUNIFORM2FARBPROC)GetProcAddress("glUniform2fARB");
	glUniform3fARB = (PFNGLUNIFORM3FARBPROC)GetProcAddress("glUniform3fARB");
	glUniform4fARB = (PFNGLUNIFORM4FARBPROC)GetProcAddress("glUniform4fARB");
	glUniform1iARB = (PFNGLUNIFORM1IARBPROC)GetProcAddress("glUniform1iARB");
	glUniform2iARB = (PFNGLUNIFORM2IARBPROC)GetProcAddress("glUniform2iARB");
	glUniform3iARB = (PFNGLUNIFORM3IARBPROC)GetProcAddress("glUniform3iARB");
	glUniform4iARB = (PFNGLUNIFORM4IARBPROC)GetProcAddress("glUniform4iARB");
	glUniform1fvARB = (PFNGLUNIFORM1FVARBPROC)GetProcAddress("glUniform1fvARB");
	glUniform2fvARB = (PFNGLUNIFORM2FVARBPROC)GetProcAddress("glUniform2fvARB");
	glUniform3fvARB = (PFNGLUNIFORM3FVARBPROC)GetProcAddress("glUniform3fvARB");
	glUniform4fvARB = (PFNGLUNIFORM4FVARBPROC)GetProcAddress("glUniform4fvARB");
	glUniform1ivARB = (PFNGLUNIFORM1IVARBPROC)GetProcAddress("glUniform1ivARB");
	glUniform2ivARB = (PFNGLUNIFORM2IVARBPROC)GetProcAddress("glUniform2ivARB");
	glUniform3ivARB = (PFNGLUNIFORM3IVARBPROC)GetProcAddress("glUniform3ivARB");
	glUniform4ivARB = (PFNGLUNIFORM4IVARBPROC)GetProcAddress("glUniform4ivARB");
	glUniformMatrix2fvARB = (PFNGLUNIFORMMATRIX2FVARBPROC)GetProcAddress("glUniformMatrix2fvARB");
	glUniformMatrix3fvARB = (PFNGLUNIFORMMATRIX2FVARBPROC)GetProcAddress("glUniformMatrix2fvARB");
	glUniformMatrix4fvARB = (PFNGLUNIFORMMATRIX2FVARBPROC)GetProcAddress("glUniformMatrix2fvARB");
	_hasShaderObjects = isExtensionSupported("GL_ARB_shader_objects") && (glCreateShaderObjectARB != NULL);

	glEnableVertexAttribArrayARB = (PFNGLENABLEVERTEXATTRIBARRAYARBPROC)GetProcAddress("glEnableVertexAttribArrayARB");
	glDisableVertexAttribArrayARB = (PFNGLDISABLEVERTEXATTRIBARRAYARBPROC)GetProcAddress("glDisableVertexAttribArrayARB");
	glBindAttribLocationARB = (PFNGLBINDATTRIBLOCATIONARBPROC)GetProcAddress("glBindAttribLocationARB");
	glGetAttribLocationARB = (PFNGLGETATTRIBLOCATIONARBPROC)GetProcAddress("glGetAttribLocationARB");
	glVertexAttribPointerARB = (PFNGLVERTEXATTRIBPOINTERARBPROC)GetProcAddress("glVertexAttribPointerARB");
	glGetActiveAttribARB = (PFNGLGETACTIVEATTRIBARBPROC)GetProcAddress("glGetActiveAttribARB");
	glVertexAttrib1fARB = (PFNGLVERTEXATTRIB1FARBPROC)GetProcAddress("glVertexAttrib1fARB");
	glVertexAttrib1sARB = (PFNGLVERTEXATTRIB1SARBPROC)GetProcAddress("glVertexAttrib1sARB");
	glVertexAttrib1dARB = (PFNGLVERTEXATTRIB1DARBPROC)GetProcAddress("glVertexAttrib1dARB");
	glVertexAttrib2fARB = (PFNGLVERTEXATTRIB2FARBPROC)GetProcAddress("glVertexAttrib2fARB");
	glVertexAttrib2sARB = (PFNGLVERTEXATTRIB2SARBPROC)GetProcAddress("glVertexAttrib2sARB");
	glVertexAttrib2dARB = (PFNGLVERTEXATTRIB2DARBPROC)GetProcAddress("glVertexAttrib2dARB");
	glVertexAttrib3fARB = (PFNGLVERTEXATTRIB3FARBPROC)GetProcAddress("glVertexAttrib3fARB");
	glVertexAttrib3sARB = (PFNGLVERTEXATTRIB3SARBPROC)GetProcAddress("glVertexAttrib3sARB");
	glVertexAttrib3dARB = (PFNGLVERTEXATTRIB3DARBPROC)GetProcAddress("glVertexAttrib3dARB");
	glVertexAttrib4fARB = (PFNGLVERTEXATTRIB4FARBPROC)GetProcAddress("glVertexAttrib4fARB");
	glVertexAttrib4sARB = (PFNGLVERTEXATTRIB4SARBPROC)GetProcAddress("glVertexAttrib4sARB");
	glVertexAttrib4dARB = (PFNGLVERTEXATTRIB4DARBPROC)GetProcAddress("glVertexAttrib4dARB");
	glVertexAttrib4NubARB = (PFNGLVERTEXATTRIB4NUBARBPROC)GetProcAddress("glVertexAttrib4NubARB");
	glVertexAttrib1fvARB = (PFNGLVERTEXATTRIB1FVARBPROC)GetProcAddress("glVertexAttrib1fvARB");
	glVertexAttrib2fvARB = (PFNGLVERTEXATTRIB2FVARBPROC)GetProcAddress("glVertexAttrib2fvARB");
	glVertexAttrib3fvARB = (PFNGLVERTEXATTRIB3FVARBPROC)GetProcAddress("glVertexAttrib3fvARB");
	glVertexAttrib4fvARB = (PFNGLVERTEXATTRIB4FVARBPROC)GetProcAddress("glVertexAttrib4fvARB");
	_hasVertexShader = isExtensionSupported("GL_ARB_vertex_shader") && (glEnableVertexAttribArrayARB != NULL);
	
	_hasFragmentShader = isExtensionSupported("GL_ARB_fragment_shader");

	glFogCoordfEXT = (PFNGLFOGCOORDFEXTPROC)GetProcAddress("glFogCoordfEXT");
	glFogCoorddEXT = (PFNGLFOGCOORDDEXTPROC)GetProcAddress("glFogCoorddEXT");
	glFogCoordfvEXT = (PFNGLFOGCOORDFVEXTPROC)GetProcAddress("glFogCoorddvEXT");
	glFogCoorddvEXT = (PFNGLFOGCOORDDVEXTPROC)GetProcAddress("glFogCoorddvEXT");
	glFogCoordPointerEXT = (PFNGLFOGCOORDPOINTEREXTPROC)GetProcAddress("glFogCoordPointerEXT");
	_hasFogCoord = isExtensionSupported("GL_EXT_fog_coord") && (glFogCoordfEXT != NULL);
}

};
