/*	Messenger

PIRL CVS ID: Messenger.java,v 1.65 2012/04/16 06:11:36 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Messenger;

import	PIRL.PVL.PVL_Exception;
import	PIRL.Utilities.Host;

import	java.nio.channels.SocketChannel;
import	java.nio.channels.AsynchronousCloseException;
import	java.nio.ByteBuffer;
import	java.io.IOException;
import	java.io.EOFException;
import	java.io.InterruptedIOException;
import	java.io.PrintWriter;
import	java.io.StringWriter;
import	java.net.InetAddress;
import	java.net.InetSocketAddress;
import	java.net.SocketAddress;
import	java.net.Socket;
import	java.net.SocketException;
import	java.net.SocketTimeoutException;
import	java.net.ConnectException;
import	java.text.SimpleDateFormat;
import	java.util.Date;
import	java.util.Vector;


/**	A <i>Messenger</i> sends and receives <i>Message</i>s over a
	communication channel and routes received messages to their
	destination.
<p>
<h4>Client Communication
</h4><p>
	A Messenger has a client communication channel to which Messages may
	be sent and from which they may be received. Messages may be received
	synchronously or asynchronously. A message that is received
	synchronously is returned directly to the caller. A timeout for
	synchronous message receipt may be specified to prevent indefinate
	blocking if no message arrives. Because only one communication
	channel is used for receiving messages synchronous receiving of
	messages can not be done while messages are being received
	asynchronously. Messages are received asynchronously on a separate
	message reader thread which waits indefinately until a message
	arrives and then dispatches the message based on its address routing
	information.
<p>
	Sending and receiving of messages is thread safe. A Message will be
	completely sent before another will be allowed to be sent. A Message
	will be completely received and delivered before another will be
	received. A Message may be sent by one thread while a Message is
	being received and delivered by another thread. While a Message is
	being sent or received and delivered another thread trying to do the
	same operation will be blocked until the current operation completes.
	Message level synchronization ensures that Messages will not collide
	and be corrupted during transmission.
<p>
	The client communication channel may be closed at any time at either
	end of the channel. When the channel is closed an explanation may be
	provided which is then delivered in a Message to the Messenger
	Employer. This offers the employer an opportunity to take appropriate
	action. Once the client communication channel is closed in can not
	be reopened. To reopen communication with the client a new Messenger
	must be constructed.
<p>
<h4>Message Routing
</h4><p>
	Messages received asynchronously are examined for routing
	information. Each Messenger has a unique address which is always
	added to a Message's route-from list when it is sent. On receipt if
	the last entry of a Message's route-to list is the address of the
	receiving Messenger, or the route-to list is empty, the Message is
	delivered to the Messenger employer. If the address matches that of a
	forwarding Messenger it is used to send the Message. If the address
	is unknown the Message is sent back with an appropriate undeliverable
	notation.
<p>
<h4>Message Content
</h4><p>
	In addition to routing information contained in the Message label,
	the Message body may have any content that can be represented
	using Parameter Value Language (PVL) formatting.
<p>
	The input stream that is read is assumed to have one character per
	byte. This is because a Message is based on PVL formatted syntax
	using US-ASCII encoded text. The {@link Message#Packet_Label(int)
	message start synchronization sequence}, which is critical for
	identifying the location of a message in the communication channel
	byte stream, must be composed of the required ASCII characters if it
	is to be recognized (this is done automatically by the Message
	class). Multi-byte message body characters and binary data may be
	transmitted in ASCII transcoded form such as standard HTML character
	references for multi-byte characters and hexadecimal value character
	pairs for binary bytes, but it is the responsibility of the
	application using the delivered Message to interpret transcoded
	sequences.
<p>
<p>
<h4>Message Statistics
</h4><p>
	A count of the total number of Messages successfully sent and
	received is maintained. Also a count of Messages dropped because they
	could not be sent or received for any reason is maintained. If a
	Message can not be sent or received due to a problem with the client
	communication channel the channel is closed and the Messenger
	employer is notified with a Done Message.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.65
	@see		Message
*/
public class Messenger
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Messenger (1.65 2012/04/16 06:11:36)";


//	Mesenger identification:

/**	The system time, in milliseconds since the start of the system epoch,
	when this Messanger was constructed.
*/
private long
	Start_Time					= System.currentTimeMillis ();

/**	The {@link #DATE_FORMAT} pattern.
<p>
	The date format is:
<p>
	<i>d MMMMM yyyy kk</i><b>:</b><i>mm</i><b>:</b><i>ss</i><b>.</b><i>SSS Z</i>
<p>
	where <i>d</i> is the day number in the month (one or two digits),
	<i>MMMMM</i> is the full month name in the year, <i>yyyy</i> is the full
	year number, <i>kk</i> is the hour in the 24 hour day (two digits),
	<i>mm</i> is the minute in the hour (two digits), <i>ss</i> is the
	second in the minute (two digits), <i>SSS</i> is the millisecond in
	the second (three digits), and <i>Z</i> is the hours-minutes offset
	of the local time zone from the reference UTC zone.
*/
public static final String
	DATE_FORMATTING				= "d MMMMM yyyy kk:mm:ss.SSS Z";

/**	Date formatter used to represent the Messenger {@link #Start_Time()
	start time}.
<p>
	@see	#DATE_FORMATTING
*/
public static final SimpleDateFormat
	DATE_FORMAT					= new SimpleDateFormat (DATE_FORMATTING);

//	The address of this Messenger.
private String
	Address;

//	The client's canonical hostname.
private String
	Client_Hostname				= null;

/**	The delimiter character separating the sections of a Messenger
	{@link #Address() address}.
*/
public static final char
	ADDRESS_SEGMENT_DELIMITER	= ':';

/**	An optional Message containing parameters used to {@link
	#Identity(Message) identify} the Messenger by employer applications.
<p>
	The content of the message is determined by the application.
*/
protected Message
	Identity					= null;

/**	The name given to an {@link #Identity(Message) Identity Message}.
*/
public static final String
	IDENTITY_PARAMETER_NAME		= Message.IDENTITY_ACTION;

/**	The name of the identity parameter that provides the name of the
	Messenger being identified.
*/
public static final String
	NAME_PARAMETER_NAME			= Message.NAME_PARAMETER_NAME;

/**	{@link #Identity(Message) Identity} Message parameter set to
	contain the Messenger Address.
*/
public static final String
	ADDRESS_PARAMETER_NAME		= "Address";

/**	Message {@link #ACTION_PARAMETER_NAME} value sent in response to
	receiving an {@link #ADDRESS_PARAMETER_NAME} action Message for
	{@link #Deliver(Message) delivery}.
*/
public static final String
	MESSENGER_ADDRESS_PARAMETER_NAME	= "Messenger_Address";


/**	Message label parameter whose value is the message {@link
	Message#Route_To() route-to list}.
<p>
	@see	Message#Packet_Label(int)
*/
public static final String
	ROUTE_TO_PARAMETER_NAME		= Message.ROUTE_TO_PARAMETER_NAME;

/**	Message label parameter whose value is the message {@link
	Message#Route_From() route-from list}.
<p>
	@see	Message#Packet_Label(int)
*/
public static final String
	ROUTE_FROM_PARAMETER_NAME	= Message.ROUTE_FROM_PARAMETER_NAME;


/**	Message parameter specifying the action associated with a Message.
*/
public static final String
	ACTION_PARAMETER_NAME		= Message.ACTION_PARAMETER_NAME;


/**	The {@link #Employer() employer} to whom Messages addressed to this
	Messenger are to be {@link Deliver(Message) delivered}.
*/
private Message_Delivered_Listener
	Employer					= null;

//	Clients:

/**	The message communication SocketChannel to which messages will be
	sent and from which they will be received.
*/
private SocketChannel
	Client_Channel;

/**	A list of Messengers to whom Messages may be forwarded.
*/
private Vector<Messenger>
	Forwarding_Messengers		= new Vector<Messenger> ();

/**	The name of the {@link #Forwarding_Messengers() forwarding
	Messengers} identities parameter group.
*/
public static final String
	FORWARDING_MESSENGERS_PARAMETER_NAME
		= "Forwarding_Messengers";

//	Incoming message information:

//	Global ThreadGroup for Messengers.
private static ThreadGroup
	Message_Listeners			= new ThreadGroup ("Messengers");

private Thread
	Message_Listener			= null;

/**	Flag indicating whether {@link #Listening_for_Messages()}
	should continue.
*/
private volatile boolean
	Listening_for_Messages		= false;

/**	Maximum number of digits in the message label length value sequence.
<p>
	The message start synchronization sequence includes the length of the
	message label section as a decimal integer value. The text
	representation of the label length value has a maximum number of
	digits for a reasonable value. Seven digits should be more than
	sufficient for the largest reasonable label length values.
<p>
	@see	#DEFAULT_BUFFER_INCREMENT
*/
public static final int
	MAX_LABEL_LENGTH_DIGITS		= 7;

/**	The minimum size of the read buffer.
<p>
	The minimum read buffer size must allow for the Message start
	synchronization sequence:
<p>
<ol>
<li>{@link Message#START_MARKER}
<li>{@link Message#START_MARKER_DELIMITER}
<li>Up to {@link #MAX_LABEL_LENGTH_DIGITS}
<li>{@link Message#START_MARKER_END}
</ol>
<p>
	@see	#DEFAULT_BUFFER_INCREMENT
*/
public static final int
	MINIMUM_BUFFER_SIZE			= Message.START_MARKER.length ()
									+ MAX_LABEL_LENGTH_DIGITS + 2;
/**	The read buffer size increment.
<p>
	When additional read buffer capacity is needed for storing an
	incoming message the buffer size is increased by a multiple of the
	size increment to at least the required capacity. It follows that the
	initital read buffer size is the size increment.
<p>
	A reasonable buffer size increment will not be too small so that
	buffer expansion reallocation and data content copying will not need
	to happen too often; yet not so large that an excessive buffer size
	will result. It must not be smaller than the {@link
	#MINIMUM_BUFFER_SIZE}. The read buffer size will be increased as
	needed, but never decreased unless {@link #Shrink_Buffer()} is
	called.
*/
public static final int
	DEFAULT_BUFFER_INCREMENT	= 1024;
private static int
	Default_Buffer_Increment	= DEFAULT_BUFFER_INCREMENT;
private volatile int
	Buffer_Increment			= Default_Buffer_Increment;

/**	The buffer used to store message content that is received.
<p>
	It might (only might) be more efficient to use a direct buffer which
	may optimize data transfers during socket input. However, for
	receive timeouts to be effective input must be done via the socket's
	InputStream which can not use a direct buffer, rather than via the
	SocketChannel itself; a temporary byte array must be allocated to
	hold the input which is then copied into the direct buffer. If a
	non-direct buffer is used its backing byte array can be provided to
	the InputStream for better efficiency.
<p>
	<b.N.B.</b>: The buffer is used as the read synchronization lock.
	This ensures that a Messenger will only read and deliver one
	message at a time so no other thread can cause message corruption.
*/
private ByteBuffer
	Read_Buffer					= ByteBuffer.allocate (Buffer_Increment);

//	Message transmit and receive statistics.
private long
	Messages_Received			= 0,
	Message_Bytes_Received		= 0,
	Messages_Received_Dropped	= 0,
	Messages_Received_Corrupted	= 0,
	Messages_Unforwardable		= 0,
	Messages_Sent				= 0,
	Message_Bytes_Sent			= 0,
	Messages_Sent_Dropped		= 0,
	Messages_Sent_Corrupted		= 0;

private int
	SEQUENTIAL_CORRUPTED_MESSAGES_LIMIT	= 5;

/**	Bounced message due to unknown route-to address Message {@link
	#ACTION_PARAMETER_NAME} value.
*/
public static final String
	UNDELIVERABLE_ACTION			= "Undeliverable";

/**	Group of parameters containing the bounced message in an {@link
	#UNDELIVERABLE_ACTION} Message.
*/
public static final String
	ORIGINAL_MESSAGE_PARAMETER_NAME	= "Original_Message";

/**	A parameter containing an explanation description.
*/
public static final String
	EXPLANATION_PARAMETER_NAME		= "Explanation";

/**	A parameter containing an exception description.
*/
public static final String
	EXCEPTION_PARAMETER_NAME		= "Exception";


/**	The amount of time, in seconds, to wait for a server
	{@link #Connect(SocketAddress) connection}.
*/
public static final int
	DEFAULT_CONNECT_TIMEOUT		= 10;
private static int
	Connect_Timeout				= DEFAULT_CONNECT_TIMEOUT;

/**	The maximum number of repeated server reconnect tries.
*/
public static final int
	MAX_RECONNECT_ATTEMPTS		= 5;

/**	The amount of time, in seconds, to wait for closing a socket to complete.
*/
public static final int
	DEFAULT_CLOSE_TIME			= 2;

private Exception
	Error_Condition				= null;

private boolean
	Done_Delivered				= false;

private static String
	NL							= System.getProperty ("line.separator");


// Debug control.
private static final int
	DEBUG_OFF				= 0,
	DEBUG_CONSTRUCTOR		= 1 << 1,
	DEBUG_IDENTITY			= 1 << 2,
	DEBUG_LISTENING			= 1 << 3,
	DEBUG_SEND				= 1 << 4,
	DEBUG_RECEIVE			= 1 << 5,
	DEBUG_MESSAGE_START		= 1 << 6,
	DEBUG_READ				= 1 << 7,
	DEBUG_BUFFER			= 1 << 8,
	DEBUG_ROUTING			= 1 << 9,
	DEBUG_FORWARDING		= 1 << 10,
	DEBUG_CONNECT			= 1 << 11,
	DEBUG_CONFIGURE			= 1 << 12,
	DEBUG_ALL				= -1,

	DEBUG					= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Messenger on a client's communication channel
	connection.
<p>
	The {@link #Address() address} and {@link #Identity() identity}
	are intialized.
<p>
	@param	employer	The Message_Delivered_Listener to which messages
		will be delivered that are not automatically forwared. If null,
		messages can only be {@link #Receive(int) received}
		synchronously until an {@link
		#Employer(Message_Delivered_Listener) employer} is assigned.
	@param	client_channel	The SocketChannel to which messages will
		be sent and from which they will be received.
	@throws	IllegalArgumentException	If the client is null.
*/
public Messenger
	(
	Message_Delivered_Listener	employer,
	SocketChannel				client_channel
	)
	throws IllegalArgumentException
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">>> Messenger Constructor:" + NL
		+"    Client_Channel - " + client_channel);

if ((Client_Channel = client_channel) == null)
	throw new IllegalArgumentException (ID + NL
		+ "Unable to construct a Messenger with a null client SocketChannel.");

Address = Host.IP_ADDRESS
	+ ADDRESS_SEGMENT_DELIMITER
	+ Port ()
	+ ADDRESS_SEGMENT_DELIMITER
	+ Start_Time;
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Address - " + Address);

if (employer != null)
	Employer (employer);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("    Employer - " + employer);

Identity (null);
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		("<< Messenger Constructor: Address " + Address);
}

/**	Construct a Messenger on a client's connection address.
<p>
	The client address is used to {@link #Connect(SocketAddress)
	connect} to the client's communication channel.
<p>
	The {@link #Address() address} and {@link #Identity() identity}
	are intialized.
<p>
	@param	employer	The Message_Delivered_Listener to which messages
		will be delivered that are not automatically forwared. If null,
		messages can only be {@link #Receive(int) received}
		synchronously until an {@link
		#Employer(Message_Delivered_Listener) employer} is assigned.
	@param	client_address	An InetSocketAddress to be used in
		establishing a communication channel with the client.
	@throws	IllegalArgumentException	If either the employer
		or server is null.
	@throws	IOException	If there was a problem connecting to the
		server.
	@see	#Messenger(Message_Delivered_Listener, SocketChannel)
*/
public Messenger
	(
	Message_Delivered_Listener	employer,
	InetSocketAddress			client_address
	)
	throws IllegalArgumentException, IOException
{this (employer, Connect (client_address));}

/**	Construct a Messenger to a client on a specified host and port.
<p>
	The hostname and port number define the address of the client
	where the communication channel connection is to be made.
<p>
	The {@link #Address() address} and {@link #Identity() identity}
	are intialized.
<p>
	@param	employer	The Message_Delivered_Listener to which messages
		will be delivered that are not automatically forwared. If null,
		messages can only be {@link #Receive(int) received}
		synchronously until an {@link
		#Employer(Message_Delivered_Listener) employer} is assigned.
	@param	host	The name or IP address of the host system where the
		server is running.
	@param	port	The port number on the server host system to use
		when establishing a connection.
	@throws	IllegalArgumentException	If either the employer
		or hostname is null or the port number is invalid.
	@throws	IOException	If there was a problem connecting to the
		server.
	@see	#Messenger(Message_Delivered_Listener, InetSocketAddress)
*/
public Messenger
	(
	Message_Delivered_Listener	employer,
	String						host,
	int							port
	)
	throws IllegalArgumentException, IOException
{this (employer, new InetSocketAddress (host, port));}

/**	Construct a Messenger to a client on a specified host and port.
<p>
	The hostname and port number define the address of the client
	where the communication channel connection is to be made.
<p>
	The {@link #Address() address} and {@link #Identity() identity}
	are intialized.
<p>
	<b>N.B.</b>: Messages can only be {@link #Receive(int) received}
	synchronously until an {@link #Employer(Message_Delivered_Listener)
	employer} is assigned.
<p>
	@param	host	The name or IP address of the host system where the
		server is running.
	@param	port	The port number on the server host system to use
		when establishing a connection.
	@throws	IllegalArgumentException	If either the employer
		or hostname is null or the port number is invalid.
	@throws	IOException	If there was a problem connecting to the
		server.
	@see	#Messenger(Message_Delivered_Listener, String, int)
*/
public Messenger
	(
	String						host,
	int							port
	)
	throws IllegalArgumentException, IOException
{this (null, host, port);}


private Messenger ()
{}

/*==============================================================================
	Accessors
*/
/*------------------------------------------------------------------------------
	Identification
*/
/**	Get the address of this Messenger.
<p>
	The format of the address is:
<p>
	<i>IP_ADDRESS</i><b>:</b><i>Port</i><b>:</b><i>Start_Time</i>
<p>
	where <i>IP_ADDRESS</i> is the {@link Host#IP_Address() Internet
	Protocol address} of the host system, <i>{@link #Port() Port}</i> is
	the local port number used to communicate Messages, and <i>{@link
	#Start_Time() Start_Time}</i> is the time when this Messgener was
	constructed. This address is used to uniquely identify this Messenger
	when Messages are {@link #Send(Message) sent} and {@link
	#Receive(int) received}.
<p>
	@return	The address String for this Messenger.
	@see	Message#Route_To()
	@see	Message#Route_From()
*/
public String Address ()
{return Address;}

/**	Get the time when this Messenger was constructed.
<p>
	The {@link System#currentTimeMillis() time} is obtained from the host
	system as the difference, measured in milliseconds, between the
	Messenger construction time and midnight, January 1, 1970 UTC.
*/
public long Start_Time ()
{return Start_Time;}

/**	Get the Message describing the identity of this Messenger.
<p>
	<b>N.B.</b>: A reference to, not a copy of, the identity Message
	is returned. This is done because the identity Message is not
	owned by the Messenger; it is controlled by the application.
<p>
	@return	A Message containing parameters the identify this Messenger.
		This will be null if no {#Identity(Message) identity} has been
		assigned.
*/
public Message Identity ()
{return Identity;}

/**	Set the identity of this Messenger.
<p>
	The specified Message is copied, unless it is null in which case
	an empty message is provided.
<p>
	The identity Message may contain any parameters used by the
	employing application to identify this Messenger. The identity
	Message will be given the Name {@link #IDENTITY_PARAMETER_NAME} and
	it will be guaranteed to contain an {@link #ADDRESS_PARAMETER_NAME}
	with the {@link #Address() address} of this Messenger as its value.
	If the identity Message contains an {@link #ACTION_PARAMETER_NAME}
	parameter it is removed.
<p>
	@param	identity	A Message containing parameters that describe
		the identity of this Messenger.
	@return	This Messenger.	
*/
public Messenger Identity
	(
	Message	identity
	)
{
if (identity == null)
	identity = new Message ();
else
	{
	try {identity = new Message (identity);}
	catch (PVL_Exception exception) {/* Shouldn't happen */}
	}
identity
	.Remove (ACTION_PARAMETER_NAME);
identity
	.Set (ADDRESS_PARAMETER_NAME, Address)
	.Name (IDENTITY_PARAMETER_NAME);
Identity = identity;
if ((DEBUG & DEBUG_IDENTITY) != 0)
	System.out.println
		(">-< Messenger.Identity:" + NL
		+ identity);
return this;
}

/**	Get a description of this Messenger.
<p>
	@return	A String describing this Messenger.
*/
public String toString ()
{
String
	string = ID + NL
	+"    " + Host.FULL_HOSTNAME + ':' + Port ()
		+ ' ' + DATE_FORMAT.format (new Date (Start_Time))
		+ " (" + Address + ')' + NL
	+"    Client_Channel: " + Client_Channel;
return string;
}

/*------------------------------------------------------------------------------
	Communication Channel
*/
/**	Get the amount of time, in seconds, to wait for the communication
	channel to connect.
<p>
	@return	The connection timeout, in seconds.
	@see	#Connect_Timeout(int)
*/
public static int Connect_Timeout ()
{return Connect_Timeout;}

/**	Set the amount of time, in seconds, to wait for the communication
	channel to connect.
<p>
	@param	timeout	The connection timeout, in seconds. If less than one
		the {@link #DEFAULT_CONNECT_TIMEOUT} will be used.
	@see	#Connect(SocketAddress)
*/
public static void Connect_Timeout
	(
	int		timeout
	)
{
if (timeout < 1)
	timeout = DEFAULT_CONNECT_TIMEOUT;
Connect_Timeout = timeout;
}

/**	Get the connonical, fully qualified hostname for the remote host to
	which the Messenger communication channel is connected.
<p>
	@return	The remote hostname String, or null if the Messenger is not
		connected. This may, of course, be the same as the local {@link
		Host#FULL_HOSTNAME}.
*/
public String Client_Hostname ()
{
if (Client_Hostname == null)
	{
	InetAddress
		address = Client_Channel.socket ().getInetAddress ();
	if (address != null)
		Client_Hostname = address.getCanonicalHostName ();
	}
return Client_Hostname;
}

/**	Get the remote port number of the Message communication channel.
<p>
	<b>N.B.</b>: The availability of a remote port number does not
	imply that the communication channel {@link #Is_Connected() is
	connected}.
<p>
	@return	The port number on the remote host system (which, of course,
		may be the same as the local host system) used for Message
		communication. This will be 0 if the Messenger is not yet connected.
*/
public int Client_Port ()
{return Client_Channel.socket ().getPort ();}

/**	Test if the Messenger's communication channel is connected to
	the client.
<p>
	A Messenger is connected if the {@link #Client_Channel()
	client communication channel} is open and connected.
<p>
	@return	true if the client communication channel is connected;
		false otherwise.
*/
public boolean Is_Connected ()
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">-< Messenger.Is_Connected: " +
			 (Client_Channel.isOpen () &&
			  Client_Channel.isConnected ()));
return
	Client_Channel.isOpen () &&
	Client_Channel.isConnected ();
}

/**	Get the client communication channel.
<p>
	@return	The SocketChannel used for communication with the
		Messenger's client.
	@see	#Is_Connected()
*/
public SocketChannel Client_Channel ()
{return Client_Channel;}

/**	Get the local port number used to communicate Messages.
<p>
	The port number is determined by the communication channel used to
	construct the Messenger. All Messages {@link #Receive(int) received}
	will be read from this port and all Messages {@link #Send(Message)
	sent} will be written to this port.
<p>
	@return	The communication local port number.
*/
public int Port ()
{return Client_Channel.socket ().getLocalPort ();}

/*------------------------------------------------------------------------------
	Messages
*/
/**	Get the Messenger employer.
<p>
	Messages recieved are {@link #Deliver(Message) delivered} to the
	employer if they are addressed to this Messanger or have no address.
<p>
	@return	A Message_Delivered_Listener employer object.
	@see	Message_Delivered_Listener
	@see	#Employer(Message_Delivered_Listener)
*/
public Message_Delivered_Listener Employer ()
{return Employer;}

/**	Set the Messenger employer.
<p>
	Messages recieved are {@link #Deliver(Message) delivered} to the
	employer if they are addressed to this Messanger or have no address.
<p>
	@param	employer	A Message_Delivered_Listener employer object.
	@return	This Messenger.
	@throws	IllegalArgumentException	If the employer is null.
	@see	Message_Delivered_Listener
*/
public Messenger Employer
	(
	Message_Delivered_Listener	employer
	)
{
if (employer == null)
	throw new IllegalArgumentException (ID + NL
		+ "The Message_Delivered_Listener employer can not be null.");
synchronized (Address) {Employer = employer;}
return this;
}

/**	Add a Messenger to the Message forwarding list.
<p>
	When a message is asynchronously received it is {@link
	#Route_Packet(Message, ByteBuffer) routed} based on its route-to
	address list. The list of forwarding Messengers provide routing
	possibilities in addition to delivery to the employer of this
	Messenger.
<p>
	@param	messenger	A Messenger. If null nothing is done.
	@return	This Messenger.
	@see	#Remove_Forwarding(Messenger)
	@see	#Forwarding_Messenger(String)
*/
public Messenger Add_Forwarding
	(
	Messenger	messenger
	)
{
if ((DEBUG & DEBUG_FORWARDING) != 0)
	System.out.println
		(">-< Messenger.Add_Forward:" + NL
		+ messenger
		+"    From -" + NL
		+ this.toString ());
if (messenger != null)
	{
	synchronized (Forwarding_Messengers)
		{
		if (! Forwarding_Messengers.contains (messenger))
			Forwarding_Messengers.add (messenger);
		}
	}
return this;
}

/**	Remove a Messenger from the Message forwarding list.
<p>
	@param	messenger	A Messenger.
	@return	true if the Messenger was found in the list and removed;
		false otherwise.
	@see	#Add_Forwarding(Messenger)
*/
public boolean Remove_Forwarding
	(
	Messenger	messenger
	)
{
if ((DEBUG & DEBUG_FORWARDING) != 0)
	System.out.println
		(">-< Messenger.Remove_Forward:" + NL
		+ messenger
		+"    From -" + NL
		+ this.toString ());
synchronized (Forwarding_Messengers)
	{return Forwarding_Messengers.remove (messenger);}
}

/**	Get the Messenger associated with a forwarding address.
<p>
	@param	address	A Messenger (@link #Address() address}. If null
		null is returned.
	@return	A Messenger, or null if the address does not match that of
		any Messenger in the {@link #Add_Forwarding(Messenger) forward
		Messenger list}.
	@see	#Add_Forwarding(Messenger)
*/
public Messenger Forwarding_Messenger
	(
	String	address
	)
{
if (address != null)
	{
	Messenger
		messenger = null;
	synchronized (Forwarding_Messengers)
		{
		int
			index = -1,
			total = Forwarding_Messengers.size ();
		while (++index < total)
			{
			messenger = Forwarding_Messengers.get (index);
			if (messenger.Address.equals (address))
				return messenger;
			}
		}
	}
return null;
}

/**	Get the identities of all Messengers in the forwarding list.
<p>
	@return	A {@link #FORWARDING_MESSENGERS_PARAMETER_NAME} Message
		containing the identities of all the Messengers in the
		forwarding list. This will be null if the forwarding list
		is empty. 
*/
public Message Forwarding_Messengers ()
{
Message
	message = null,
	identity;
synchronized (Forwarding_Messengers)
	{
	if (! Forwarding_Messengers.isEmpty ())
		{
		message = new Message ();
		message.Name (FORWARDING_MESSENGERS_PARAMETER_NAME);
		int
			index = -1,
			total = Forwarding_Messengers.size ();
		while (++index < total)
			{
			try
				{
				identity = new Message
					(Forwarding_Messengers.get (index).Identity ());
				identity.Name (identity.Get (NAME_PARAMETER_NAME));
				message.Add (identity);
				}
			catch (PVL_Exception exception) {}
			}
		}
	}
return message;
}

/**	Get the number of Messages successfully {@link #Send(Message) sent}.
<p>
	@return	The number of Messages successfully sent.
*/
public long Messages_Sent ()
{return Messages_Sent;}

/**	Get the number of message bytes {@link #Send(Message) sent}.
<p>
	Message bytes include the message transmission {@link
	Message#Packet_Label(int) label} and the message body content.
<p>
	@return	The total number of message packet bytes sent.
*/
public long Message_Bytes_Sent ()
{return Message_Bytes_Sent;}

/**	Get the number of Messages that could not be {@link #Send(Message)
	sent}.
<p>
	@return	The number of Messages that could not be sent because the
		output was shutdown.
*/
public long Messages_Sent_Dropped ()
{return Messages_Sent_Dropped;}

/**	Get the number of Messages to be {@link #Send(Message) sent} that
	had content for which a PVL representation could not be generated.
<p>
	@return	The number of Messages that could not be sent because the
		content was corrupted.
*/
public long Messages_Sent_Corrupted ()
{return Messages_Sent_Corrupted;}

/**	Get the number of Messages successfully {@link #Read_Packet(int) received}.
<p>
	@return	The number of Messages successfully received.
*/
public long Messages_Received ()
{return Messages_Received;}

/**	Get the number of message bytes {@link #Read_Packet(int) received}.
<p>
	Message bytes include the message transmission {@link
	Message#Packet_Label(int) label} and the message body content.
<p>
	@return	The total number of message packet bytes received.
*/
public long Message_Bytes_Received ()
{return Message_Bytes_Received;}

/**	Get the number of Messages that were dropped becasue they could not
	be completely {@link #Read_Packet(int) received}.
<p>
	@return	The number of Messages that were dropped due to input
		shutdown before the message was completely received.
*/
public long Messages_Received_Dropped ()
{return Messages_Received_Dropped;}

/**	Get the number of Messages received that were corrupted in transmission.
<p>
	@return	The number of Messages that were received but their PVL content
		could not be correctly parsed.
*/
public long Messages_Received_Corrupted ()
{return Messages_Received_Corrupted;}

/**	Get the number of Messages that could not be {@link
	#Forward(String, Message, ByteBuffer) forwarded}.
<p>
	@return	The number of Messages that could not be forwarded due to
		an IOException.
*/
public long Messages_Unforwardable ()
{return Messages_Unforwardable;}

/**	Get the most recent error exception.
<p>
	@return	The Exception that most recently occured during while
		attempting to {@link #Send(Message) send} or {@link #Receive(int)
		receive} a message. This will be null if no error condition has
		occurred since the Messenger was constructed or the last
		{@link #Reset_Error_Condition() reset} of the error condition.
*/
public Exception Error_Condition ()
{return Error_Condition;}

/**	Clear the most recent error condition.
<p>
	@return	This Messenger.
	@see	#Error_Condition()
*/
public Messenger Reset_Error_Condition ()
{
Error_Condition = null;
return this;
}

/*------------------------------------------------------------------------------
	Read Buffer
*/
/**	Set the default size to use for the read buffer increment.
<p>
	This value is only used when a Messenger is constructed and
	when a specified {@link #Buffer_Increment(int) read buffer
	increment} is to be set to the default value.
<p>
	@param	amount	The size, in bytes, of the default read buffer increment.
	@return	The previous read buffer increment amount.
	@see	#Buffer_Increment(int)
*/
public static int Default_Buffer_Increment
	(
	int		amount
	)
{
int
	previous_amount = Default_Buffer_Increment;
if (amount <= 0)
	amount = DEFAULT_BUFFER_INCREMENT;
if (amount < MINIMUM_BUFFER_SIZE)
	amount = MINIMUM_BUFFER_SIZE;
Default_Buffer_Increment = amount;
return previous_amount;
}

/**	Get the current default read buffer increment amount.
<p>
	@return	The current default read buffer increment amount.
*/
public static int Default_Buffer_Increment ()
{return Default_Buffer_Increment;}

/**	Set the read buffer increment amount.
<p>
	The read buffer holds the message packet for a message that
	has been received until the message is delivered or forwarded.
<p>
	While a message packet is being read from the communication channel
	if the read buffer is found to have insufficient space for the
	message packet content being read the buffer capacity is increased by
	to the amount of space required rounded up to the next buffer
	increment amount.
<p>
	If a {@link #Shrink_Buffer() buffer shrink} is pending it will
	remain pending after the new buffer increment is set.
<p>
	@param	amount	The read buffer increment amount. This will not be
		allowed to be less than the {@link #MINIMUM_BUFFER_SIZE}. If the
		specified amount is less than or equal to zero the {@link
		#Default_Buffer_Increment() default buffer increment} is used.
*/
public Messenger Buffer_Increment
	(
	int		amount
	)
{
if (amount <= 0)
	amount = Default_Buffer_Increment;
if (amount < MINIMUM_BUFFER_SIZE)
	amount = MINIMUM_BUFFER_SIZE;
if (Buffer_Increment < 0)
	//	Shrink buffer flag.
	amount = -amount;
Buffer_Increment = amount;
return this;
}

/**	Get the read buffer increment amount.
<p>
	@return	The read buffer increment amount.
*/
public int Buffer_Increment ()
{
int
	amount = Buffer_Increment;
if (amount < 0)
	amount = -amount;
return amount;
}

/**	Get the capacity of the read buffer.
<p>
	@return The capacity of the read buffer.
	@see	#Shrink_Buffer()
*/
public int Buffer_Capacity ()
{synchronized (Read_Buffer) {return Read_Buffer.capacity ();}}

/**	Shrink the read buffer size.
<p>
	When the next Message is read the size of the read buffer will be
	reduced to the minimum capacity to hold the Message. This capacity
	will be the Message label and content size rounded up to the next
	multiple of the {@link #Buffer_Increment(int) buffer size increment}.
	The buffer shrink will not be done again until this method is called
	again.
<p>
	<b>N.B.</b>: In general, the read buffer should be allowed to
	expand to the maximum size for messages to be read after which
	point the buffer will no longer need to be reallocated (a
	relatively expensive operation). However, if an extraordinarily
	large message has been read it might be beneficial to shrink the
	buffer to free the excess storage.
<p>
	@return	This Messenger.
	@see	#Buffer_Capacity()
*/
public Messenger Shrink_Buffer ()
{
if (Buffer_Increment > 0)
	Buffer_Increment = -Buffer_Increment;
return this;
}

/*==============================================================================
	Asynchrous Receiver
*/
/**	Test if the Messenger thread is running and marked to continue {@link
	#Listen_for_Messages() listening for messages}.
<p>
	To only test if the Messenger thread is running use {@link
	Thread#isAlive()}.
<p>
	@return	If true the Messenger thread is running and marked to
		continue listening for messages. If false the Messenger thread
		will stop listening for messages after the current Message is
		handled, or the next Message is handled if the Messenger thread
		is waiting for an incoming message.
	@see	#Stop_Listening_for_Messages()
*/
public boolean Listening_for_Messages ()
{return Message_Listener != null;}

/**	Start the Messenger thread listening for messages.
<p>
	Asynchronous message receipt can only occur when this Messenger
	has an {@link #Employer(Message_Delivered_Listener) employer}
	to whom the messages will be delivered.
<p>
	If the message listener thread is not running and the Messenger
	is {@link #Is_Connected() connected} it will be started.
<p>
	@return	The message listenter Thread. This will be null only if the
		Messenger is not connected.
	@throws	IllegalStateException	If the Messenger was constructed
		without an employer.
	@see	#Listening_for_Messages()
	@see	#Stop_Listening_for_Messages()
*/
public synchronized Thread Listen_for_Messages ()
{
if ((DEBUG & DEBUG_LISTENING) != 0)
	System.out.println
		(">>> Messenger.Listen_for_Messages");
if (Employer == null)
	throw new IllegalStateException (toString () + NL
		+ "Can't listen for messages without an employer that will receive them.");

Thread
	message_listener = Message_Listener;
if (message_listener == null &&
	Is_Connected ())
	{
	if ((DEBUG & DEBUG_LISTENING) != 0)
		System.out.println
			("    Starting a Message_Reader thread.");
	Message_Listener = message_listener = new Message_Reader ();
	Message_Listener.start ();
	}
if ((DEBUG & DEBUG_LISTENING) != 0)
	System.out.println
		("<<< Messenger.Listen_for_Messages");
return message_listener;
}

/**	Flag the Messenger thread to stop {@link #Listen_for_Messages()
	listening for messages}.
<p>
	The Messenger thread is flagged to stop listening for messages after
	the current Message is handled, or the next Message is handled if the
	Messenger thread is currently waiting for an incoming message.
<p>
	<b>N.B.</b>: The Messenger thread, if it is running, will not
	stop immediately. To wait for the Messenger thread to stop:
<p>
<code><pre>
    messenger.Stop_Listening_for_Messages ();
    while (messenger.isAlive ())
        {
        try {messenger.join ())
        catch (InterruptedException e) {}
        }
</pre></code>
<p>
	@return	true if the Messenger has stopped listing for messages;
		false if the Messenger has not yet stopped.
*/
public synchronized boolean Stop_Listening_for_Messages ()
{
if ((DEBUG & DEBUG_LISTENING) != 0)
	System.out.println
		(">-< Messenger.Stop_Listening_for_Messages: "
			+ (Message_Listener == null));
Listening_for_Messages = false;
return Message_Listener == null;
}

/**	Get the group of all message listener threads.
<p>
	When a Messenger begins to asynchronously {@link
	#Listen_for_Messages() listen for messages} it runs a Thread which
	excecutes a loop that waits for a message packet to be {@link
	#Read_Packet(int) read} and then {@link #Route_Packet(Message,
	ByteBuffer) routes} the packet for delivery. The message listener
	thread will only stop on its own if a) after handling a message that
	has been read, the thread has been {@link
	#Stop_Listening_for_Messages() flagged to stop}; b) the communication
	input channel has been shutdown; or c) an unexpected exception
	occurs.
<p>
	Each message listener thread is placed in a ThreadGroup named
	"Messengers" that is used by all Messengers. The name of each message
	listener thread in the group is the Messenger's {@link #Address()
	address}.
<p>
	@return	The ThreadGroup containing all the
		message listener threads in use by Messengers.
*/
public static ThreadGroup Message_Listeners ()
{return Message_Listeners;}


//	Message listener thread.
private class Message_Reader
	extends Thread
{
public Message_Reader ()
{super (Message_Listeners, "Messenger " + Address);}

public void run ()
{
Receive_Messages ();
Message_Listener = null;
}
}	//	End of Message_Reader class.

/**	Run the Messenger {@link #Listen_for_Messages() listening for
	messages} thread.
<p>
	This method will not return until the thread stops. The thread
	will stop if, after handling a message that has been read, the
	thread has been {@link #Stop_Listening_for_Messages() flagged
	to stop}. It will also stop if the communication input channel
	has been shutdown, or if an unexpected exception occurs.
*/
private void Receive_Messages ()
{
if (! Is_Connected ())
	{
	if ((DEBUG & DEBUG_LISTENING) != 0)
		System.out.println
			("    The client channel is closed.");
	return;
	}

String
	report = null;
Exception
	exception = null;
int
	sequential_corrupted_messages = 0;

Listening_for_Messages = true;
while (Listening_for_Messages &&
		! Client_Channel.socket ().isInputShutdown ())
	{
	if ((DEBUG & DEBUG_LISTENING) != 0)
		System.out.println
			("==> Messenger: Listening for message ...");
	synchronized (Read_Buffer)
		{
		try
			{
			Message
				packet = Read_Packet (0);
			if ((DEBUG & DEBUG_LISTENING) != 0)
				System.out.println
					("==> Messenger: Routing packet ...");
			Route_Packet (packet, Read_Buffer.slice ());
			}
		catch (PVL_Exception except)
			{
			if ((DEBUG & DEBUG_LISTENING) != 0)
				System.out.println
					("!!! Messenger: Receive failed -" + NL
					+ except);
			if (++sequential_corrupted_messages
					< SEQUENTIAL_CORRUPTED_MESSAGES_LIMIT)
				continue;
			report = String.valueOf (sequential_corrupted_messages)
				+ " sequential corrupted messages while listening for messages.";
			exception = except;
			break;
			}
		catch (IOException except)
			{
			if ((DEBUG & DEBUG_LISTENING) != 0)
				System.out.println
					("!!! Messenger: Receive failed -" + NL
					+ except);
			if (! (except instanceof EOFException))
				report = "Receive failed while listening for messages";
			exception = except;
			break;
			}
		catch (Exception except)
			{
			if ((DEBUG & DEBUG_LISTENING) != 0)
				{
				System.out.println
					("!!! Messenger: Unexpected exception -" + NL
					+ exception);
				exception.printStackTrace (System.out);
				}
			report = "Unexpected exception while listening for messages.";
			exception = except;
			break;
			}
		}
	sequential_corrupted_messages = 0;
	}
if ((DEBUG & DEBUG_LISTENING) != 0)
	System.out.println
		("==> Messenger: Stop listening for messages");
if (Listening_for_Messages)
	Done (report, exception);

Listening_for_Messages = false;
}

/*==============================================================================
	I/O
*/
/*------------------------------------------------------------------------------
	Send
*/
/**	Send a Message over the communication channel.
<p>
	The message label routing information is {@link
	#Update_Send_Routing(Message) updated}. Then the Message is converted
	to a {@link Message#Packet() message packet} which is {@link
	#Send_Packet(Message, ByteBuffer) sent} to the output channel. After
	successfully sending the message the count of (@link #Messages_Sent()
	messages sent} is incremented and the count of {@link
	#Message_Bytes_Sent() message bytes sent} is updated.
<p>
	If the message can not be sent because the content can not be
	represented by PVL syntax the count of {@link
	#Messages_Sent_Corrupted() corrupted send messages } is incremented
	and the {@link #Error_Condition() error condition} is noted.
<p>
	@param	message	The Message to be sent. If null or empty, nothing
		is sent.
	@throws	EOFException	If the output communication channel was found
		to be shutdown before attempting to send the message.
	@throws	IOException	If there was a problem while writing the message
		to the communication channel.
	@throws	PVL_Exception	If the message content contains parameters
		that can not be represented by the Message PVL syntax.
*/
public void Send
	(
	Message	message
	)
	throws IOException, PVL_Exception
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> Messenger.Send:" + NL
		+ message.Routing () + NL
		+ message);
if (message == null ||
	message.List_Size () == 0)
	{
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("<<< Messenger.Send: No message");
	return;
	}

Update_Send_Routing (message);

//	Send the message content.
try {Send_Packet (null, message.Packet ());}
catch (PVL_Exception exception)
	{
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("!!! Message is invalid -" + NL
			+ exception.getMessage ());
	++Messages_Sent_Corrupted;
	Error_Condition = exception;
	throw exception;
	}
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< Messenger.Send");
}

/**	Send the message packet.
<p>
	If the content is null or has nothing remaining, nothing is done.
<p>
	If the output channel has been shutdown the count of {@link
	#Messages_Sent_Dropped() send messages dropped} is incremented, the
	{@link #Error_Condition() error condition} is set to an EOFException,
	and the Messenger is put in the {@link #Done(String, Exception) done}
	state.
<p>
	If the message label is not null its routing information is {@link
	#Update_Send_Routing(Message) updated} and then converted to a {@link
	Message#Packet_Label(int) packet label String} which is wrapped in a
	ByteBuffer that is added to a message content buffers array. If the
	message label is null the packet content contains the entire message,
	including the message label, as provided by the {@link Message#Packet()
	message packet} generator.
<p>
	The content ByteBuffer is added to a message message content buffers
	array and the sequence of buffers is written to the output channel.
	This is done as a single operation, rather than writing the message
	label separately from the message body content, to ensure that the
	message will not be inadvertently interleaved with another message
	being written to the same ouput channel at the same time. The count
	of {@link #Messages_Sent() messages sent} is incremented and the
	count of {@link #Message_Bytes_Sent() message bytes sent} is updated.
<p>
	If the message packet can not be transmitted the count of {@link
	#Messages_Sent_Dropped() send messages dropped} is incremented, the
	{@link #Error_Condition() error condition} is set to the exception
	that occured, and the Messenger is put in the {@link #Done(String,
	Exception) done} state.
<p>
	<b>N.B.</b>: The message routing address lists are only updated if
	a message label is provided. Otherwise the content buffer is
	presumed to already contain the correct routing information.
<p>
	@param	message_label	A Message containing the message start
		synchronization sequence and the routing parameters. This may be
		null if the content contains the entire message including the
		message label.
	@param	content	A ByteBuffer containing the message packet content.
		If null or empty nothing will be done.
	@throws	EOFException	If the output communication channel was found
		to be shutdown before attempting to send the message.
	@throws	IOException	If there was a problem while writing the message
		to the communication channel.
*/
protected void Send_Packet
	(
	Message		message_label,
	ByteBuffer	content
	)
	throws IOException
{
if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		(">>> Messenger.Send_Packet:");
if (content == null ||
	! content.hasRemaining ())
	{
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("<<< Messenger.Send_Packet: No message");
	return;
	}

if (Client_Channel.socket ().isOutputShutdown ())
	{
	EOFException
		exception = new EOFException (toString () + NL
			+ ((message_label == null) ? "" : (message_label.Routing () + NL))
			+ "Send can not be done because output is shutdown.");
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("!!! Message could not be sent -" + NL
			+ exception.getMessage ());
	++Messages_Sent_Dropped;
	Error_Condition = exception;
	Done ("Unable to send message.", Error_Condition);
	throw exception;
	}

//	Data buffer(s) that will be transmitted.
ByteBuffer[]
	content_buffers;
if (message_label != null)
	{
	Update_Send_Routing (message_label);

	//	Get the Message label in PVL form.
	String
		label = message_label.Packet_Label (content.remaining ());
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.print
			("    Message label:" + NL
			+ label);
	//	Message label content.
	content_buffers = new ByteBuffer[2];
	content_buffers[0] = ByteBuffer.wrap (label.getBytes ());
	content_buffers[1] = content;
	}
else
	{
	//	Message body content.
	content_buffers = new ByteBuffer[1];
	content_buffers[0] = content;
	}

if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("    Message content:" + NL
		+ Message.Content_String (content));
try
	{
	//	Transmit the message content.
	Message_Bytes_Sent += Client_Channel.write (content_buffers);
	++Messages_Sent;
	}
catch (IOException exception)
	{
	exception = new IOException (toString () + NL
		+ ((message_label == null) ? "" : (message_label.Routing () + NL))
		+ "Unable to send message."
		+ exception);
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("!!! Message could not be sent -" + NL
			+ exception);
	++Messages_Sent_Dropped;
	Error_Condition = exception;
	Done ("Unable to send message.", Error_Condition);
	throw exception;
	}
catch (Exception exception)
	{
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("!!! Message could not be sent -" + NL
			+ exception);
	++Messages_Sent_Dropped;
	Error_Condition = new IOException (toString () + NL
		+ exception);
	Error_Condition.initCause (exception);
	Done ("Unable to send message.", Error_Condition);
	throw (IOException)Error_Condition;
	}

if ((DEBUG & DEBUG_SEND) != 0)
	System.out.println
		("<<< Messenger.Send_Packet");
}

/**	Update the message send routing information of a Message.
<p>
	If the message is not null and {@link Message#To() the last
	address on its route-to list} is the same as this Messenger's {@link
	#Address() address} it is (@link Message#Pop_To() removed from the
	route-to list} and added to the {@link Message#From(String)
	route-from} list.
<p>
	<b>N.B.</b>: This method is only intended for use when sending a
	message. Message {@link #Deliver(Message) delivery} and synchronous
	{@link #Receive(int) receive} is effectively "sending" the message
	to the Messenger {@link #Employer() employer}.
<p>
	@param	message	The Message to have its routing information updated.
		If null, nothing is done.
*/
protected void Update_Send_Routing
	(
	Message	message
	)
{
if (message != null)
	{
	if (Address.equals (message.To ()))
		//	Remove this Messenger's address from the To list.
		message.Pop_To ();
	//	Add this Messenger's address to the From list.
	message.From (Address);
	}
}

/*------------------------------------------------------------------------------
	Receive
*/
/**	Receive a message synchronously.
<p>
	If this Messenger is currently asynchronously listening for messages
	an IllegalStateException will be thrown. The Messenger must first be
	{@link #Stop_Listening_for_Messages() flagged to stop listening} and
	have actually stopped.
<p>
	A {@link #Read_Packet(int) message packet is read} from the input
	channel. Its routing information is {@link
	#Update_Send_Routing(Message) updated} and the packet content is
	used to set the {@link Message#Content(ByteBuffer) message content}.
<p>
	@param	timeout	The maximum amount of time, in seconds, to wait for
		message arrival. If less than or equal to zero the wait will be
		indefinate.
	@return	The Message that was received. This will not be null.
	@throws	IllegalStateException	If this Messenger is currently
		asynchronously listening for messages.
	@throws	IOException	If a message could not be read. There are
		various possible forms of this exception:
		<p>
		<dl>
		<dt>SocketTimeoutException
			<dd>The timeout expired.
		<dt>EOFException
			<dd>The input channel was shutdown or closed.
		<dt>Other
			<dd>Read error on the input channel.
		</dl>
	@throws	PVL_Exception	If the message content can not be parsed or
		is not valid.
*/
public Message Receive
	(
	int		timeout
	)
	throws
		IllegalStateException,
		//	This exception is passed through from the Read method.
		IOException,
		//	This exception may be passed through from Read_Packet.
		PVL_Exception
{
if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println
		(">>> Messenger.Receive: " + timeout);
if (timeout < 0)
	timeout = 0;

if (Message_Listener != null)
	throw new IllegalStateException (toString () + NL
		+ "Can not synchronously receive a message" + NL
		+ "while asynchronously listening for messages.");

Message
	message = null;
PVL_Exception
	except = null;

/*	Get an exclusive lock on the Read_Buffer while it is being used.

	The Read_Buffer is reused so it is important that only one thread
	will access it for reading a message at any time.
*/
try
	{
	synchronized (Read_Buffer)
		{
		//	Read the Message packet.
		message = Read_Packet (timeout);
		Update_Send_Routing (message);

		//	Set the message content from the Read_Buffer.
		try {message.Content (Read_Buffer);}
		catch (PVL_Exception exception)
			{
			except = new PVL_Exception (toString () + NL
				+ "The message content received is invalid."
				+ exception.getMessage ());
			if ((DEBUG & DEBUG_RECEIVE) != 0)
				System.out.println
					("!!! Message is invalid -" + NL
					+ except.getMessage ());
			++Messages_Received_Corrupted;
			Error_Condition = except;
			}
		}
	}
catch (PVL_Exception exception)
	{
	//	From Read_Packet.
	Done ("Receive message failed.", exception);
	throw exception;
	}
catch (IOException exception)
	{
	Done ("Receive message failed.", exception);
	throw exception;
	}
if (except != null)
	{
	Done ("Receive message failed.", except);
	throw except;
	}

if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println
		("    Message -" + NL
		+ message + NL
		+"<<< Messenger.Receive");
return message;
}

/**	Receive a message synchronously.
<p>
	This is the same as {@link #Receive(int) receiving a message} with no
	timeout.
<p>
	@return	The Message that was received. This will not be null.
	@throws	IllegalStateException	If this Messenger is currently
		asynchronously listening for messages.
	@throws	IOException	If a message could not be read.
	@throws	PVL_Exception	If the message content can not be parsed or
		is not valid.
*/
public Message Receive ()
	throws
		IllegalStateException,
		IOException,
		PVL_Exception
{return Receive (0);}

/**	Read a Message packet from the communication channel.
<p>
	A Message packet is a tentative Message: A packet Message contains
	only the label section of the transmitted message; the message body
	content remains unparsed in the read buffer. The label section
	must start with the message start synchronization sequence:
<p>
<ol>
<li>{@link Message#START_MARKER}
<li>{@link Message#START_MARKER_DELIMITER}
<li>Up to {@link #MAX_LABEL_LENGTH_DIGITS}
<li>{@link Message#START_MARKER_END}
</ol>
<p>
	The input {@link #Client_Channel() communication channel} is read one
	byte/character at a time until the entire message start
	synchnronization sequence has been recognized. Any unexpected
	character value will cause the scan for the message start
	synchronization sequence to start again from the beginning. The
	digits in the start synchronization sequence represent the size of
	the message label section that follows. This many bytes are read from
	the input channel and used to construct a {@link
	Message#Message(ByteBuffer) message}. This message label has its
	{@link Message#Route_To(Value) route-to list} and {@link
	Message#Route_From(Value) route-from list} set from the {@link
	#ROUTE_TO_PARAMETER_NAME} Value and {@link #ROUTE_FROM_PARAMETER_NAME}
	Value, respectively, if they are present. The {@link
	Message#Content_Amount() content amount} of the message label is
	obtained and this many bytes are read from the input channel. The
	message is invalid if the content amount can not be obtained from
	the message label.
<p>
	Though the content section of the message will have been read, it
	remains unparsed in the Read_Buffer with its position set to the
	beginning of the content data and its limit set to the end of
	message data. If the message packet is to be forwarded there is no
	need to parse the message content section; otherwise the Receive
	method will replace the Message with the content section to generate
	the Message to be delivered.
<p>
	<b>N.B.</b>: If the label section can not be parsed the count of
	{@link #Messages_Received_Corrupted() corrupted messages} will be
	incremented and then it will not be possible to read the content
	section because its size is unknown. In this case message
	synchronization will creep over the remaining unread message to
	locate the beginning of the next message.
<p>
	The count of {@link #Messages_Received() messages received} is
	incremented when an entire Message has been received. The count
	of {@link #Message_Bytes_Received() message bytes received} is
	updated as they are read. Thus the number of bytes received
	includes {@link #Messages_Received_Corrupted() corrupted messages}.
<p>
	<b>N.B.</b>: It is the responsibility of the user to call the {@link
	#Done(String, Exception) Done} method if an exception is thrown. Done
	is not called here because this method is expected to be called
	within a block synchronized on the Read_Buffer and the Done method
	should not be called within this sychnronized block since the {@link
	#Deliver(Message) delivery} of the {@link Message#Done() Done
	Message} could result in an Employer's delivery Thread calling Done
	again which would result in a deadlock.
<p>
	@param	timeout	The amount of time (seconds) to wait for the
		arrival of the message before a timeout exception will occur.
	@return	A Messsage with message label section parameters with the
		To and From data members having been set from these parameters.
	@throws	IOException	If there was a problem reading the message.
	@throws	PVL_Exception	If the label section could not be parsed as
		valid PVL syntax. 
*/
protected Message Read_Packet
	(
	//		Passed on to the Read method.
	int		timeout
	)
	throws
		//	This exception is passed through from the Read method.
		IOException,
		//	This exception is passed through from the Message constructor.
		PVL_Exception
{
if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println
		(">>> Messenger.Read_Packet: " + timeout);
Message
	packet = null;

//	Read the message label section.
Ensure_Read_Buffer_Capacity (Read_Message_Start (timeout));
Read (timeout);

//	Reset the Read_Buffer for parsing of the label.
Read_Buffer.flip ();
try
	{
	//	Construct the label Message.
	packet = new Message (Read_Buffer);

	//	Apply the routing values from the label.
	packet.Route_To   (packet.Value_of (ROUTE_TO_PARAMETER_NAME));
	packet.Route_From (packet.Value_of (ROUTE_FROM_PARAMETER_NAME));

	//	Move the Read_Buffer position forward to end-of-data.
	int
		content_position = Read_Buffer.limit ();
	Read_Buffer.position (content_position);

	//	Read the message content section.
	Ensure_Read_Buffer_Capacity (packet.Content_Amount ());
	if ((DEBUG & DEBUG_RECEIVE) != 0)
		System.out.println
			("    Content amount = " + Read_Buffer.remaining ());
	Read (timeout);

	//	Reset the Read_Buffer position to the beginning of the content section.
	Read_Buffer.position (content_position);
	}
catch (PVL_Exception exception)
	{
	exception = new PVL_Exception (toString () + NL
		+ "Invalid message packet label." + NL
		+ exception.getMessage ());
	if ((DEBUG & DEBUG_RECEIVE) != 0)
		System.out.println
			("!!! Message is invalid -" + NL
			+ exception.getMessage ());
	++Messages_Received_Corrupted;
	Error_Condition = exception;
	throw exception;
	}

if ((DEBUG & DEBUG_RECEIVE) != 0)
	System.out.println
		("    Message label -" + NL
		+ packet + NL
		+"<<< Messenger.Read_Packet");
++Messages_Received;
return packet;
}

/**	Read a message start synchronization sequence.
<p>
	See the {@link Read_Packet(int)} method for a description of the
	message start synchronization sequence.
<p>
	It is assumed that the input stream being read is unreliable. To find
	a message start synchronization sequence in the input stream one byte
	at a time is {@link Read(int)} into the Read_Buffer - the buffer
	limit is one more than the buffer position - and tested for being the
	expected character. If any byte read fails to provide the expected
	character the search for the synchronization sequence begins again
	with the last byte read.
<p>
	@param	timeout	The maximum time, in seconds, to wait for a
		byte to be read.
	@return	The label size value obtained from the message start
		synchronization sequence
	@throws	IOException	If there was a problem reading a byte.
*/
private int Read_Message_Start
	(
	int		timeout
	)
	//	This exception is passed through from the Read method.
	throws IOException
{
if ((DEBUG & DEBUG_MESSAGE_START) != 0)
	System.out.println
		(">>> Messenger.Read_Message_Start: " + timeout);
int
	index;
char
	character;
String
	message_length;

//	Prime the pump.
Read_Buffer
	.position (0)
	.limit (1);
Read (timeout);

Read_until_Message:
while (true)
	{
	//	Read until the Message START_MARKER has been read.
	index = 0;
	while (true)
		{
		if ((DEBUG & DEBUG_MESSAGE_START) != 0)
			System.out.println
				("    Messenger.Read_Message_Start: "
					+ "expecting '"
					+ Message.START_MARKER.charAt (index) + "'");
		if ((char)Read_Buffer.get (index)
				== Message.START_MARKER.charAt (index))
			{
			if (++index == Message.START_MARKER.length ())
				//	START_MARKER has been read.
				break;

			//	Read the next byte.
			Read_Buffer.limit (index + 1);
			Read (timeout);
			}
		else
			{
			//	Start again.
			Read_Buffer.put (0, Read_Buffer.get (index));
			Read_Buffer
				.position (1)
				.limit (1);
			index = 0;
			}
		}

	//	Check for the parameter name delimiter.
	Read_Buffer.limit (Read_Buffer.limit () + 1);
	Read (timeout);
	if ((DEBUG & DEBUG_MESSAGE_START) != 0)
		System.out.println
			("    Messenger.Read_Message_Start: "
				+ "expecting '" + Message.START_MARKER_DELIMITER + "'");
	if ((char)Read_Buffer.get (Read_Buffer.position () - 1)
			!= Message.START_MARKER_DELIMITER)
		{
		//	Start all over again.
		Read_Buffer.put (0, Read_Buffer.get (Read_Buffer.position () - 1));
		Read_Buffer
			.position (1)
			.limit (1);
		continue;
		}

	//	Read digits until the statement end delimiter is seen.
	Read_Buffer.limit (Read_Buffer.limit () + 1);
	Read (timeout);
	message_length = "";
	while (true)
		{
		character = (char)Read_Buffer.get (Read_Buffer.position () - 1);
		if ((DEBUG & DEBUG_MESSAGE_START) != 0)
			System.out.println
				("    Messenger.Read_Message_Start: expecting a digit or '"
					+ Message.START_MARKER_END + "'");
		if (character == Message.START_MARKER_END)
			//	End of message start synchronization sequence.
			break Read_until_Message;

		if (character < '0' ||
			character > '9' ||
			message_length.length () == MAX_LABEL_LENGTH_DIGITS)
			{
			//	Start all over again.
			Read_Buffer.put (0, (byte)character);
			Read_Buffer
				.position (1)
				.limit (1);
			continue Read_until_Message;
			}
		message_length += character;

		//	Read the next byte.
		Read_Buffer.limit (Read_Buffer.limit () + 1);
		Read (timeout);
		}
	}
if ((DEBUG & DEBUG_MESSAGE_START) != 0)
	System.out.println
		("<<< Messenger.Read_Message_Start: " + message_length);
return Integer.parseInt (message_length);
}

/**	Read into the Read_Buffer.
<p>
	The Read_Buffer position and limit determine the number of bytes to
	read. This method only returns after successfully reading all
	required bytes; otherwise an IOException is thrown.
<p>
	@param	timeout	The maximum time, in seconds, to wait for a
		read to complete.
	@throws	IOException	If there is a problem with the Client_Channel.
		This includes a SocketTimeoutException that might occur if the
		Client_Channel socket had been set to timeout. It also includes
		an EOFException if socket input was shutdown or an EOF
		condition was encountered while attempting to read from it.
*/
private void Read
	(
	int		timeout
	)
	throws IOException
{
if (! Read_Buffer.hasRemaining ())
	//	Only enter if data is to be read.
	return;
if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		(">>> Messenger.Read: " + timeout + NL
		+ Read_Buffer);

if (Client_Channel.socket ().isInputShutdown ())
	{
	EOFException
		exception = new EOFException (toString () + NL
			+ "Read can not be done because input is shutdown.");
	if ((DEBUG & DEBUG_SEND) != 0)
		System.out.println
			("!!! Read can not be done because input is shutdown -" + NL
			+ exception.getMessage ());
	if (Read_Buffer.position () > 0)
		++Messages_Received_Dropped;
	Error_Condition = exception;
	throw exception;
	}

IOException
	IO_exception = null;
//	DEBUG
int
	start_position = Read_Buffer.position (),
	amount;
try
	{
	if (timeout != 0)
		{
		byte[]
			byte_array;
		int
			offset;
		if (Read_Buffer.hasArray ())
			{
			byte_array = Read_Buffer.array ();
			offset = Read_Buffer.arrayOffset () + Read_Buffer.position ();
			}
		else
			{
			byte_array = new byte[Read_Buffer.remaining ()];
			offset = 0;
			}
		Client_Channel.socket ().setSoTimeout (timeout * 1000);
		while ((amount = Client_Channel.socket ().getInputStream ().read
					(byte_array, offset, Read_Buffer.remaining ())) >= 0)
			{
			if ((DEBUG & DEBUG_READ) != 0)
				{
				System.out.print
					("    Messenger.Read: read " + amount + " -" + NL
					+ '>');
				for (int
						index = offset;
						index < (offset + amount);
						index++)
					System.out.print ((char)byte_array[index]);
				System.out.println ("<");
				}
			if (amount > 0)
				{
				Message_Bytes_Received += amount;
				if (Read_Buffer.hasArray ())
					Read_Buffer.position (Read_Buffer.position () + amount);
				else
					Read_Buffer.put (byte_array, offset, amount);
				if (! Read_Buffer.hasRemaining ())
					{
					if ((DEBUG & DEBUG_READ) != 0)
						System.out.println
							("<<< Messenger.Read");
					return;
					}
				}
			}
		}
	else
		{
		while (! Client_Channel.socket ().isInputShutdown () &&
				(amount = Client_Channel.read (Read_Buffer)) >= 0)
			{
			if ((DEBUG & DEBUG_READ) != 0)
				{
				int
					end_position = Read_Buffer.position ();
				System.out.print
					("    Messenger.Read: read " + amount + " -" + NL + '>');
				Read_Buffer.position (start_position);
				while (Read_Buffer.position () < end_position)
					System.out.print ((char)Read_Buffer.get ());
				System.out.println ("<");
				start_position = end_position;
				}
			Message_Bytes_Received += amount;
			if (! Read_Buffer.hasRemaining ())
				{
				if ((DEBUG & DEBUG_READ) != 0)
					System.out.println
						("<<< Messenger.Read");
				return;
				}

			//	Try again after a brief pause.
			try {Message_Listener.sleep (250);}
			catch (InterruptedException exception) {}
			}
		}
	}
catch (IOException exception)
	{
	if ((DEBUG & DEBUG_READ) != 0)
		System.out.println
			("    Messenger.Read: IOException -" + NL
			+ exception);
	/*	Hack!

		When the socket connection is reset by the peer
		this is equivalent to an EOF condition.
	*/
	if (exception.getMessage () != null &&
		exception.getMessage ().indexOf ("Connection reset by peer") < 0)
		IO_exception = exception;
	}
catch (Exception exception)
	{
	if ((DEBUG & DEBUG_READ) != 0)
		System.out.println
			("    Messenger.Read: Exception -" + NL
			+ exception);
	IO_exception = (IOException)new IOException (toString () + NL
		+ exception)
		.initCause (exception);
	}
String
	explanation = "Read did not complete." + NL;
if (IO_exception == null)
	//	Either input was shutdown or an EOF was encountered.
	IO_exception = new EOFException (toString () + NL
		+ "End of input encountered.");
else
if (IO_exception instanceof AsynchronousCloseException)
	{
	explanation += "The communication channel was closed.";
	IO_exception = new EOFException (toString () + NL
		+ explanation + NL
		+ IO_exception);
	}
else
if (IO_exception instanceof SocketTimeoutException)
	{
	explanation += "Timeout after " + timeout + " second"
					+ ((timeout == 1) ? "" : "s");
	IO_exception = new SocketTimeoutException (toString () + NL
		+ explanation);
	}
else
	{
	explanation += IO_exception.toString ();
	IO_exception = (IOException)new IOException (toString () + NL
		+ explanation)
		.initCause (IO_exception);
	}
Error_Condition = IO_exception;

if ((DEBUG & DEBUG_READ) != 0)
	System.out.println
		("    Messenger.Read: " + IO_exception);
throw IO_exception;
}

/**	Ensure that the Read_Buffer has sufficient storage space.
<p>
	If the space between the current Read_Buffer position and its
	capacity is less than the amount required a new buffer is allocated
	that will provide the required space rounded up to the next
	Buffer_Increment.
<p>
	The Read_Buffer limit, regardless of whether a new buffer was
	allocated or not, is set to the current Read_Buffer position plus the
	amount.
<p>
	@param	amount	The amount of storage space beyond the current
		position that is required.
*/
private void Ensure_Read_Buffer_Capacity
	(
	int		amount
	)
{
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		(">>> Messenger.Ensure_Read_Buffer_Capacity: " + amount + NL
		+"    " + Read_Buffer + NL
		+"    Available = "
			+ (Read_Buffer.capacity () - Read_Buffer.position ()));
if (amount <= 0)
	throw new IllegalArgumentException (toString () + NL
		+ "Invalid Ensure_Read_Buffer_Capacity amount: " + amount);

//	Required Read_Buffer limit.
int
	limit = Read_Buffer.position () + amount,
	increment = Buffer_Increment;
if (limit > Read_Buffer.capacity () ||
	//	Buffer shrink flag.
	increment < 0)
	{
	if (increment < 0)
		//	Reset the buffer size increment.
		Buffer_Increment = increment = -increment;

	//	Round up to the next buffer size increment.
	amount = increment
		* ((limit / increment)
			+ (((limit % increment) == 0) ? 0 : 1));
	if ((DEBUG & DEBUG_BUFFER) != 0)
		System.out.println
			("    Allocated = " + amount);
	ByteBuffer
		buffer = ByteBuffer.allocate (amount);

	//	Copy over the existing data.
	Read_Buffer.flip ();
	buffer.put (Read_Buffer);

	//	Free the old buffer and assign the new one.
	Read_Buffer = null;
	System.gc ();
	Read_Buffer = buffer;
	}
Read_Buffer.limit (limit);
if ((DEBUG & DEBUG_BUFFER) != 0)
	System.out.println
		("    " + Read_Buffer + NL
		+"<<< Messenger.Ensure_Read_Buffer_Capacity");
}

/*------------------------------------------------------------------------------
	Routing
*/
/**	Route a message packet to a recipient.
<p>
	The packet's {@link Message#To() next route-to address} examined. If
	there is an address (the route-to list was not empty) and the address
	matches the {@link #Address() address of this Messenger} then the
	{@link Message#To() next address on the packet's route-to list}, if
	there is one, is compared with the addresses of the {@link
	#Forwarding_Messenger(String) forwarding Messengers list}. If the
	address is for a forwarding Messenger it is used to {@link
	#Send_Packet(Message, ByteBuffer) send the packet}.
<p>
	If the address removed from the route-to list is not for this
	Messenger (an empty route-to list is implicitly addressed to this
	Messenger) or a forwarding address is found that does not match the
	address of the forwarding Messenger, the message is bounced back to
	this Messenger's client. The bounce message contains an {@link
	#UNDELIVERABLE_ACTION} {@link Message#Action() action parameter}, an
	{@link #EXPLANATION_PARAMETER_NAME}, {@link
	#ROUTE_TO_PARAMETER_NAME} and {@link #ROUTE_FROM_PARAMETER_NAME} 
	parameters {@link Message#Set(String, Object) set} to the address
	routing of the undeliverable Message, and the message that was
	received as an {@link #ORIGINAL_MESSAGE_PARAMETER_NAME}. If message
	forwarding failed an attempt will also be made to bounce the message
	back to this Messenger's client; an {@link
	#EXCEPTION_PARAMETER_NAME} is included in this case that provides
	the detail message from the exception that occurred.
<p>
	If the packet is neither forwarded nor bounced the Message {@link
	Message#Content(ByteBuffer) contains} is {@link #Deliver(Message)
	delivered} to this Messenger's {@link #Employer() employer}.
<p>
	If the content could not be parsed as valid PVL the {@link
	#Messages_Received_Corrupted() corrupted messages count} is incremented
	and the {@link #Error_Condition() error condition} is noted.
<p>
	@param	message	A received Message label. If null nothing is done.
	@param	content	A ByteBuffer holding the message content.
	@throws	PVL_Exception	If the content could not be parsed for
		Message delivery.
*/
protected void Route_Packet
	(
	Message		message,
	ByteBuffer	content
	)
	throws PVL_Exception
{
if (message == null)
	return;
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		(">>> Messenger.Route_Packet:" + NL
		+ message.Routing () + NL
		+ message);

//	Check the address on the end of the To list.
String
	address = message.To ();
while (address != null &&
	   address.equals (Address))
	{
	/*	The Message is addressed to this Messenger.

		The message will either be delivered to the employer or forwarded
		to another Messenger, but in either case the address for this
		Messenger is to be removed from the end of the To list and the
		next address examined.
	*/
	message.Pop_To ();
	address = message.To ();
	}

if (address != null)
	{
	//	The address is for another Messenger.
	Forward (address, message, content);
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("<<< Message.Route_Packet");
	return;
	}

//	Set the message content and deliver.
try {Deliver (message.Content (content));}
catch (PVL_Exception exception)
	{
	exception = new PVL_Exception (toString () + NL
		+ message.Routing () + NL
		+ "Unable to route message." + NL
		+ exception.getMessage ());
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("!!! Message is invalid -" + NL
			+ exception.getMessage ());
	++Messages_Received_Corrupted;
	Error_Condition = exception;
	throw exception;
	}
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		("    Packet message content -" + NL
		+ message + NL
		+"<<< Messenger.Route_Packet");
}

/**	Forward a message packet.
<p>
	The address of this Messenger is added to the route-from list and
	the message packet is sent using the forwarding Messenger. If an
	IOException occurs message forwarding is disabled, the {@link
	#Messages_Unforwardable() unforwardable messages counter} is
	incremented and the {@link #Error_Condition() error condition} is
	set to the exception that occurred; the exception is not thrown
	since this will be done by the forwarding Messenger.
<p>
	@param	message_label	The packet label Message.
	@param	content	A ByteBuffer containing the message body content.
*/
private void Forward
	(
	String		address,
	Message		message_label,
	ByteBuffer	content
	)
{
if ((DEBUG & DEBUG_FORWARDING) != 0)
	System.out.println
		(">>> Message.Forward: to " + address + NL
		+ message_label.Routing () + NL
		+ message_label);
Messenger
	messenger = Forwarding_Messenger (address);
if (messenger == null)
	{
	//	Addressee unknown.
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("!!! Addressee unknown");
	//	Bounce.
	try {Send (Message
		.Action (UNDELIVERABLE_ACTION)
		.Set (EXPLANATION_PARAMETER_NAME, "Addressee unknown")
		.Set (ROUTE_TO_PARAMETER_NAME, message_label.Route_To ())
		.Set (ROUTE_FROM_PARAMETER_NAME, message_label.Route_From ())
		.Add (ORIGINAL_MESSAGE_PARAMETER_NAME, new Message (content))
		.Reply_To (message_label));}
	catch (Exception except)
		{/* Junk mail */}
	++Messages_Unforwardable;
	if ((DEBUG & DEBUG_FORWARDING) != 0)
		System.out.println
			("    Messages_Unforwardable = " + Messages_Unforwardable + NL
			+"<<< Message.Forward");
	return;
	}

//	Put this Messenger's address on the From list.
message_label.From (Address);

//	Forward the packet for routing.
try {messenger.Send_Packet (message_label, content);}
catch (IOException exception)
	{
	/*	DO NOT THROW THIS EXCEPTION!

		The Forwarding_Messenger will take care of its exception.
		It is sufficient that this Messenger removes the
		forwarding messenger that is no longer output capable.
	*/
	if ((DEBUG & DEBUG_FORWARDING)
		!= 0)
		System.out.println
			("!!! Message could not be forwarded -" + NL
			+ exception.getMessage ());

	//	Bounce.
	try {Send (Message
			.Action (UNDELIVERABLE_ACTION)
			.Set (EXPLANATION_PARAMETER_NAME, "Message could not be forwarded to "
				+ messenger.Address ())
			.Set (EXCEPTION_PARAMETER_NAME, exception.toString ())
			.Set (ROUTE_TO_PARAMETER_NAME, message_label.Route_To ())
			.Set (ROUTE_FROM_PARAMETER_NAME, message_label.Route_From ())
			.Add (ORIGINAL_MESSAGE_PARAMETER_NAME, new Message (content))
			.Reply_To (message_label));}
	catch (Exception except)
		{/* Junk mail */}

	Remove_Forwarding (messenger);
	++Messages_Unforwardable;
	if ((DEBUG & DEBUG_FORWARDING) != 0)
		System.out.println
			("    Messages_Unforwardable = " + Messages_Unforwardable);
	Error_Condition = exception;
	}
if ((DEBUG & DEBUG_FORWARDING) != 0)
	System.out.println
		("<<< Message.Forward");
}

/**	Deliver a Message locally.
<p>
	If the Message contains an {@link #ADDRESS_PARAMETER_NAME} {@link
	#ACTION_PARAMETER_NAME} value it is changed to {@link
	#MESSENGER_ADDRESS_PARAMETER_NAME} and sent as a {@link
	Message#Reply_To(Message) reply to} the source of the Message. In
	effect this special Message is delivered to the Messenger which
	simply replies so the sender can it can obtain the address routing
	information for this Messenger.
<p>
	If the Messenger has an {@link #Employer() employer} all non-Address
	Messages are delivered to it. The {@link #Address() address} of this
	Messenger is {@link Message#From(String) added to the Message's
	route-from list} and the {@link Message#Route_To() route-to list} is
	emptied. The  employer's {@link
	Message_Delivered_Listener#Message_Delivered Message_Delivered
	method} is invoked with a {@link Message_Delivered_Event} containing
	the Message obtained by this Messenger. <b>N.B.</b>: The delivery of
	Message is synchronized on the employer to prevent it changing while
	the during delivery; do not attempt changing the employer in the
	thread that receives the Message delivery or a deadlock will occur.
<p>
	<b>N.B.</b>: If the Messenger has been constructed without an
	employer, and one has not been assigned, then obviously no Message
	can be delivered.
<p>
	@param	message	The Message to be delivered to the employer. If null,
		nothing is done.
*/
protected void Deliver
	(
	Message	message
	)
{
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		(">>> Messenger.Deliver:" + NL
		+ message);
if (message == null)
	{
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("    No Message" + NL
			+"<<< Messenger.Deliver");
	return;
	}

if (ADDRESS_PARAMETER_NAME.equals (message.Action ()))
	{
	try
		{
		Send (message
		.Set (ACTION_PARAMETER_NAME, MESSENGER_ADDRESS_PARAMETER_NAME)
		.Reply_To (message));
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("    " + MESSENGER_ADDRESS_PARAMETER_NAME + " reply sent");
		}
	catch (Exception exception) {/* Lost Message */}
	if ((DEBUG & DEBUG_ROUTING) != 0)
		System.out.println
			("<<< Messenger.Deliver");
	return;
	}

synchronized (Address)	//	Prevent employer change during delivery.
	{
	if (Employer == null)
		{
		if ((DEBUG & DEBUG_ROUTING) != 0)
			System.out.println
				("    No Employer" + NL
				+"<<< Messenger.Deliver");
		return;
		}

	//	Clear the To list.
	message.Clear_Route_To ();

	//	Add the address of this Messenger to the From list.
	message.From (Address);

	//	Deliver the message to the employer.
	Employer.Message_Delivered
		(new Message_Delivered_Event (this, message));
	}
if ((DEBUG & DEBUG_ROUTING) != 0)
	System.out.println
		("<<< Messenger.Deliver");
}

/**	Force the Messenger to close its client communication channel.
<p>
	Input and output for the socket of the {@link #Client_Channel()
	client communication channel} is shutdown if it not already done.
	The channel is then closed if it was open.
<p>
	<b>N.B.</b>: No {@link Message#Done() Done} Message notification is
	delivered to the {@link #Employer() employer}.
<p>
	@return	true if the Messenger was not done when the method was called
		(this call caused the communication channel I/O to be shutdown
		and/or closed); false if the communication channel had already
		been shutdown and closed for any reason.
	@see	#Done(String, Exception)
*/
public boolean Done ()
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> Messenger.Done" + NL
		+ this.toString ());
Listening_for_Messages = false;

boolean
	done = false;
synchronized (Client_Channel)
	{
	if (! Client_Channel.socket ().isInputShutdown ())
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Shutting down input");
		try {Client_Channel.socket ().shutdownInput ();}
		catch (IOException exception) {/* Nothing for it */}
		done = true;
		}
	if (! Client_Channel.socket ().isOutputShutdown ())
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Shutting down output");
		try {Client_Channel.socket ().shutdownOutput ();}
		catch (IOException exception) {/* Nothing for it */}
		done = true;
		}
	if (Client_Channel.isOpen ())
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Closing the Client_Channel connection");
		try {Client_Channel.close ();}
		catch (IOException exception) {/* Sigh */}
		done = true;
		}
	}
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(this.toString ()
		+"<<< Messenger.Done: " + done);

return done;
}

/**	Deliver a {@link Message#Done() Done} Message to the {@link
	#Employer() employer} and close its communication channel.
<p>
	<b>N.B.</b>: The Done Message is delivered before the request to
	close the communication channel occurs; the communicaiton channel may
	already have been closed by an external event, however.
<p>
	<b>N.B.</b>: Only one Done Message will ever be delivered by each
	Messenger object.
<p>
	@param	explanation	If non-null an {@link #EXPLANATION_PARAMETER_NAME}
		is included in the Done Message with the explanation text as its
		value.
	@param	exception	If non-null an {@link #EXCEPTION_PARAMETER_NAME}
		is included in the Done Message that provides the exception detail
		message as its value.
	@see	#Done()
	@see	#Is_Connected()
*/
public void Done
	(
	String		explanation,
	Exception	exception
	)
{
if (Done_Delivered)
	return;
Message
	message = Message.Done ();
if (exception != null)
	{
	if (exception instanceof IOException)
		{
		String
			report = exception.toString ();
		Throwable
			cause = exception.getCause ();
		while (cause != null)
			{
			report += NL + cause;
			cause = cause.getCause ();
			}
		message.Set (EXCEPTION_PARAMETER_NAME, report);
		}
	else
		{
		StringWriter
			report = new StringWriter ();
		exception.printStackTrace (new PrintWriter (report, true));
		message.Set (EXCEPTION_PARAMETER_NAME, report.toString ());
		}
	}
if (explanation == null ||
	explanation.length () == 0)
	explanation = "Communication with the client is done.";
message.Set (EXPLANATION_PARAMETER_NAME, explanation);

//	Prevent recursive Done operations from the message recipient.
Done_Delivered = true;
Deliver (message);

//	Shutdown client I/O.
Done ();
}

/**	Force the Messenger to close its communication channel and
	deliver a {@link Message#Done() Done} Message.
<p>
	@param	explanation	If non-null an {@link #EXPLANATION_PARAMETER_NAME}
		is included in the Done Message with the explanation text as its
		value.
	@see	#Done(String, Exception)
*/
public void Done
	(
	String	explanation
	)
{Done (explanation, null);}

/*==============================================================================
	Helper Functions
*/
/**	Connect to a SocketAddress.
<p>
	A new SocketChannel is opened and its socket set to SO_REUSEADDR mode
	(which allows the socket to be bound to the address port even if it
	is currently in a closed connection timeout state). An attempt is
	then made using the socket to connect to the specified address with a
	connection timeout of {@link #Connect_Timeout()} seconds. If
	the connection fails for any reason the SocketChannel is closed and
	the connection attempt is repeated. After {@link
	#MAX_RECONNECT_ATTEMPTS} a failure to connect will throw the last
	exception that cause the connection to fail.
<p>
	@param	address	The SocketAddress to which to connect.
	@return	A connected SocketChannel.
	@throws	IllegalArgumentException	If the address is null.
	@throws	IOException	If there was a problem establishing the
		connection.
*/
public static SocketChannel Connect
	(
	SocketAddress	address
	)
	throws IOException, IllegalArgumentException
{
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		(">>> Messenger.Connect");
if (address == null)
	throw new IllegalArgumentException (Messenger.ID + NL
		+"Unable to connect to a null address.");
SocketChannel
	channel = null;
int
	attempt = 0;
while (attempt++ < MAX_RECONNECT_ATTEMPTS)
	{
	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    Attempt " + attempt);
	//	Close the current socket.
	if (channel != null)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Closing the current connection");
		try {channel.close ();}
		catch (IOException exception) {}
		channel = null;
		}

	if ((DEBUG & DEBUG_CONNECT) != 0)
		System.out.println
			("    ServerChannel.");
	channel = SocketChannel.open ();
	try {Configure_Socket (channel.socket ());}
	catch (SocketException exception)
		{
		throw new SocketException (ID + NL
			+ "Unable to configure the communication socket." + NL
			+ exception);
		}
	try {
		channel.socket ().connect (address, Connect_Timeout * 1000);
		break;
		}
	catch (SocketTimeoutException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    Socket connect timeout.");
		String
			message = exception.getMessage ();
		if (message == null)
			message = exception.toString ();
		throw new SocketTimeoutException (Messenger.ID + NL
			+ "Unable to connect to the server at " + address + NL
			+ "after waiting " + Connect_Timeout + " seconds." + NL
			+ message);
		}
	catch (IOException exception)
		{
		if ((DEBUG & DEBUG_CONNECT) != 0)
			System.out.println
				("    IOException: "+ exception);
		if (attempt == MAX_RECONNECT_ATTEMPTS)
			{
			String
				message = exception.getMessage ();
			if (message == null)
				message = exception.toString ();
			message = Messenger.ID + NL
				+ "Unable to connect to the server at " + address + NL
				+ message;
			if (exception instanceof ConnectException)
				throw new ConnectException (message);
			else
				throw new IOException (message);
			}
		}
	}
if ((DEBUG & DEBUG_CONNECT) != 0)
	System.out.println
		("<<< Messenger.Connect");
return channel;
}

/**	Configures a Socket.
<p>
	The socket options are set to:
<ul>
<li>Performance preferences with short connection time having top priority,
	followed by low latency priority over bandwidth.
<p>
	These priorities are a hint that may have no effect on the specific
	socket implementation being used by the system.

<li><code>SO_REUSEADDR</code> enabled.
<p>
	Allows a socket to be bound even though a previous connection is in a
	timeout state.

<li><code>SO_KEEPALIVE</code> enabled.
<p>
	Provide a persistent socket connection even if there is no
	application communication for an extended time.

<li><code>SO_LINGER</code> enabled with a linger time of {@link
	#DEFAULT_CLOSE_TIME} seconds.
<p>
	When the socket is closed and unsent data is queued for transmission
	the close will block until the data has been trasmitted or the
	linger time has been reached.
</ul>
<b>N.B.</b>: The socket must be configured before it is conneected for
the socket options to be effective.
<p>
	@param	socket	The socket to be configured.
*/
public static void Configure_Socket
	(
	Socket	socket
	)
	throws IOException
{
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		(">>> Messenger.Configure_Socket:" + NL
		+"    Performance preferences - "
			+ "connect time 2, latency 1, bandwith 0" + NL
		+"    Reuse address - "
			+ socket.getReuseAddress () + " -> true" + NL
		+"    Keep alive - "
			+ socket.getKeepAlive () + " -> true" + NL
		+"    Linger - "
			+ socket.getSoLinger () + " -> true, "
			+ DEFAULT_CLOSE_TIME + NL
		+"    Input buffer - " + socket.getReceiveBufferSize () + NL
		+"    Output buffer - " + socket.getSendBufferSize () + NL
		+"    No delay - " + socket.getTcpNoDelay () + NL
		+"    Traffic class - " + socket.getTrafficClass ()
		);
socket.setPerformancePreferences (2, 1, 0);
socket.setReuseAddress (true);
socket.setKeepAlive (true);
socket.setSoLinger (true, DEFAULT_CLOSE_TIME);
if ((DEBUG & DEBUG_CONFIGURE) != 0)
	System.out.println
		("<<< Messenger.Configure_Socket");
}

}
