///////////////////////////////////////////////////////////////
//  Copyright Christopher Kormanyos 2002 - 2011.
//  Copyright 2011 John Maddock. Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_
//
// This work is based on an earlier work:
// "Algorithm 910: A Portable C++ Multiple-Precision System for Special-Function Calculations",
// in ACM TOMS, {VOL 37, ISSUE 4, (February 2011)} (C) ACM, 2011. http://doi.acm.org/10.1145/1916461.1916469

#ifdef _MSC_VER
#  define _SCL_SECURE_NO_WARNINGS
#endif

#include <boost/detail/lightweight_test.hpp>
#include <boost/array.hpp>
#include "test.hpp"

#if !defined(TEST_MPF_50) && !defined(TEST_MPF) && !defined(TEST_BACKEND) && !defined(TEST_MPZ) && !defined(TEST_CPP_DEC_FLOAT) && !defined(TEST_MPFR) && !defined(TEST_MPFR_50) && !defined(TEST_MPQ)
#  define TEST_MPF_50
//#  define TEST_MPF
#  define TEST_BACKEND
#  define TEST_CPP_DEC_FLOAT
#  define TEST_MPFR_50

#ifdef _MSC_VER
#pragma message("CAUTION!!: No backend type specified so testing everything.... this will take some time!!")
#endif
#ifdef __GNUC__
#pragma warning "CAUTION!!: No backend type specified so testing everything.... this will take some time!!"
#endif

#endif

#if defined(TEST_MPF_50)
#include <boost/multiprecision/gmp.hpp>
#endif
#if defined(TEST_MPFR_50)
#include <boost/multiprecision/mpfr.hpp>
#endif
#ifdef TEST_BACKEND
#include <boost/multiprecision/concepts/mp_number_archetypes.hpp>
#endif
#ifdef TEST_CPP_DEC_FLOAT
#include <boost/multiprecision/cpp_dec_float.hpp>
#endif

template <class T>
void test()
{
   std::cout << "Testing type: " << typeid(T).name() << std::endl;
   static const boost::array<boost::array<const char*, 3>, 500> data =
   {{
      {{ "9.85291419463953934609889984130859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "8.41670862661703722551465034484863281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.66781019918879108354226292713799369912745098619466567318041467361236175866573114777000418440761829061824953361563557580935951901405814080757565313129904800947783606850513184296612628910656615168927761935861462770335134108408073175949783551491164949747086136599355880600420490275164555515512645668845e4202" }},
      {{ "1.65291435510970359246130101382732391357421875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.93652894786225093781695250072516500949859619140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.14852190138491736562279120519668390081141778746743826681517181158751264571350304943298501284647564489716027476004029804110782430201351750715968215799574772058637701195873519618784612525011353325208230847515250406858209757415407239793739704829281917272032288615255140733409015874140869835291148547238" }},
      {{ "2.95673953882856110908505797851830720901489257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.30915509717990971694234758615493774414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "9.16137091634529046709182958669115589757019722367818747705232600776240821857227718551873067599347313641575272826830417174591636547915061517391300131065931465784194067711359196329143298499263726513131614582697830908424187934868152998838921216586737465291051289708299364409624364999366486746226077050764e249" }},
      {{ "6.08581487679821513288314083212071636808104813098907470703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "7.65778516457987279864028096199035644531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.27056224622668410282266686919173718609676411389459024877282902746284033954589536288634973757568176464096389580016576960309487231057438455595655805167665817497028295657665023126747065443986739792258870082022889306909488139458069146748438042701949551909468708242176394539901613782225328586614222105675e-3229" }},
      {{ "2.72264614486854572561469467473216354846954345703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.86954434555903360043460637029966164845973253250122070312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99366682241962146368589809577120011723198308450433185299366732500677873637460155970547267099719914327909694930518383012498797681761846242526638144261306834523078708261122471803832490341687603813336927181436827534082878781252668704774445437548691251936499670077735883150461534069270846997649139259014e-1" }},
      {{ "1.15866069758449596882599053060403093695640563964843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.91253262757665787319183436920866370201110839843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.46868982488456565751167443881314840659743514173491612010770748877346398177489343237604160804413931931718223750314831998907846386274396179552445359933868165880372381338724806279781831953896898649659594755834925976964179275873971118494605322798850925802574178528777014182338769959249415096934378557388e-1" }},
      {{ "4.72825938314130529761314392089843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.22893132739781236750786774791777133941650390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.47245157077269889091085696140795527282280334677569169487973419233796250189843816560261965111196061138868370890204338231535198008079068304184762140493830719955323588998399173073261634891767823614143691840392792367527470000847416054709559319143615840412839768396194451923527439592939735228732869467965e69" }},
      {{ "1.29556572255334176588803529739379882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.81521083299418428680382930906489491462707519531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.91729899451736210436889587963352494969987265352294254442114149825297350300933403119961677078224257818567229354512108221647637427508041941647518586565006416113534560289731038664091154857692367872604020422463457596747145816854139701847749519264660326361171185139298113322096978245544144264234481020815e7" }},
      {{ "1.74305909145137593441177159547805786132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.55727459709495003048473904527781996876001358032226562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.12324923143821767715038683964417533829057341785931080679952341815408539775130405061465897612861320390765996927155081774925717154208894105154455487856335006846587865128269477586014136390788140643457017602926503892493119037260205017744346004538379559837051387829449807650507431889923695496342461088404" }},
      {{ "5.94723096329786358182900585234165191650390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.47960584914021491681479503199625469278544187545776367187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00101355895664265703253905800820834352012946978175213925289823900262226487475877307647753894219794796436589570196875943062842928843129407133278574235943906905626144071246066853301987332156463703818534415183682692042219695734377740067313312425523058170350486981674302932951851973390826740834796342180" }},
      {{ "1.68908811949766600121487086783034214931831229478120803833007812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "7.31583500510190560817136429250240325927734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.06485337562137503941405427253471002412719863672407294570575625371322654317127374055512455786988091092329605000808871766177184698431732837248355728401235010135527651826220232715652024742668291949645619992685612328535253252695622896976648954823150769316159100729259724133493553742448044299816541989455e-423" }},
      {{ "4.72470471679805708838453814912838879536138847470283508300781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.93150097545414496380544733256101608276367187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.36613880262643802742756290736395850668317512466586871167321608424137867570108690626258507320043808302565319397131444616166042735793688809613906106252620462557968555895414582976985576875374557668514391264049547180886976991031038592004458693177738810104222966751846844587127740341157319952915743398285e-103" }},
      {{ "5.42042332941097132481544917936844285577535629272460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.51174849138805500459170616522897034883499145507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.77171703127754502681310204125715941650801721154976863870699221072355335125966565182666925044598746890911836338141033359039531496488403405876863230608143768785806156629534985369811294999635059473513624761915818310578096052710097713441725036078711101258417410421951107958524860020565804089659046743943e-1" }},
      {{ "1.03814929656497190180175493878778070211410522460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.02775539432865528274851918411059159552678465843200683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99908769642175788386137778958539300137602918424410284952085874015336769726750047569734816766197409755257562872230833189101650544474710424482558254479079319860825174137124403380959665595507587367668848404286717760343524270837960434045414179621713212480893218438355518348995643697164109731552178914851e-1" }},
      {{ "3.82635641630229656584560871124267578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.04696645144304778529686927868169732391834259033203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02545504718938094372102791727911930176506035001600018053644429991055918696319033730794852043021666923875675436795843500904336366408589219201909169612259476814126333181168932741070039124737361896932592045285182764869916877379718622220890978571031406624988992549594836404906978120992311725853941493659" }},
      {{ "1.33672486906479148771609288814943283796310424804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.20701621203230615719803608953952789306640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.82608741435940656086722594561546646099634760918110959565977761208896418216166536561903222226214939613599640463597249764023513875896636280664458251642404729724932181291845871312202787897768372355186920789243514016890287945724482698867187289280837997988218972650355222505584789955736058952829050123955e-11" }},
      {{ "7.88803854908014727698173373937606811523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.46278561425722486433187441434711217880249023437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.72840141730863663391876877521764355960382145920572562742914162972208384723706307292705275033115082012368548682595775882643927894139786449383322664316773126131611883398659050539189893838100041408464889082404113404032943316931275847874983317572290683797208861989532362102600980996636385789799495020849e4" }},
      {{ "6.02908445011699427595885936170816421508789062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.73159824509963122807221225230023264884948730468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.06403097260563050081855800637166449495962164712719263330810086375883439970326530663445685346228862337912240907582664457942174627433334293660419014781254710335542481599007896808711916825623070383641755045318205803207383228319360990374222920104547227487223738392731388028498147565624523546346417715655" }},
      {{ "5.21882079270016063787807070184499025344848632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.00797343550563134328967862529680132865905761718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "9.36552325485277806964373331313098838333278330423455390715861973295876349888773267892681165876113176797510574847645252615081953879524429911159719101193921387688552351634752458416166446355778230365755243829643708299147847893065939954861919388676503519612640610354931769525373202729617723199328630093636e-1" }},
      {{ "6.21172003041995424155174987390637397766113281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "8.60919435020819194006019142717534009534574579447507858276367187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000157241706535645559625081132898066253320364132991765749046567536915342813631920409642832019004224458248512335711478395060579077427182783938451114029622700384959716532968625901664719629811957905865676402174328291417935322613778738659214041802729741480722967793087943757739011593472415850478420639" }},
      {{ "8.93348943116835901567807098899720585905015468597412109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.15322293979125788609962910413742065429687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.91021563983953636397848520840523223624643742835964326188016961937573012460753870407293657436708430593126320103796954182972032017026192445957161066771807675200884423939605981304732630092257312925408813941089831976477439943114677798839623996214029155361569909503935179401196449493080818512569086682097e-1267" }},
      {{ "5.55105519395504379644989967346191406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "2.20561294990389011871911861817352473735809326171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.11282177288045201072507504088709286277978249266318510332909847928909226338145185595245293917086320817076726850666066198249106037631163063795831697507293260413136608221812252367908447186422344164553977681855896096040438700902356349275031836676568495994537852124264298684105398360497193752327260914137e1" }},
      {{ "2.64139377644600310476863391784263512818142771720886230468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "7.78537344498752810831310178230069141136482357978820800781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99179632299559171314926709030763809736309139005374052895886931807671532458904052556141342139885810036321222015645710865315806166952058325774560966552187527103940189373824279065702873017246336399071137612269219545181070886990906061668084063703411460250321467244169857420435253409159030432098292621067e-1" }},
      {{ "1.04207364225271594477817416191101074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "9.49096324860676521240065994788892567157745361328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00881927761618177965011845644574187803759898330801723609976606642566910129111150781463974721119043760111173993259232795682413324213317011846206782754096089290427628991647487342082014619699610677423897602146820644840945077548746946762791624180147374361784328110343084702444523109612274014821745626982" }},
      {{ "3.31519320566263809429985442989163857419043779373168945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "5.69130350190431280252290058285780105507001280784606933593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99544126695324189746630737456946723361046661345027805407888971023026631339988056392642825478722792254481831582440796961181158845762815786131168245934078509438401554967954057934563178474479427510608187860234090448388699710858468168174057416883471915845617321874136073232354461932587707630236256378020e-1" }},
      {{ "1.88337177923367888807137937590141518739983439445495605468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.32540268863411711208755150437355041503906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.75229630554212330195665922771148699451829771147200356624805857657236782065140481467338634818457146889580617522654929450001033313425457042558128784684181381335537144806647956114444609625124141647264003246426211941949785732284342909061255617831919446477154897012961652269654460432165747451828975060062e-1612" }},
      {{ "3.95459290525709050712066527921706438064575195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.67791671378427392183514677004296800078009255230426788330078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99997515679857005761440503886095927815406391244413835291624899797364615653895949902813000744851723242602512239539568754651988070279277456656779917196036522934849059282355864096704432345340316634144618927930200708582994637711792811242475638942101629267185320507818308058050356385241284795438747530570e-1" }},
      {{ "4.18765706320910169324633898213505744934082031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.16291806714740175721090054139494895935058593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.21448401185251170746394861880464242649255630391558005989017991140563422704674077045117022763467561974222030061746655113958910240758539727055748249625892285904547489203480154655618632992990390332430221914382681227157928848406067945057161854616129785541408035327136685741399621377579757754012545382969e1" }},
      {{ "1.28834086985551640736069600734481355175375938415527343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.50705923255961032891647732867568265646696090698242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.76932835372685523370171735287063726416767225734834615849134547109660778793799398169080725972094573408844765674179657286895933198717192952354877012591779343412877647689917686841315165602494901218010719779271020694334581849406783713948998204010537229621398578990055098138781426451663477493080600789821e-1" }},
      {{ "2.09720612044679288743509459891356527805328369140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.58040082173553630973117378744063898921012878417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.37769288971832154015429547694831033608424622845142547611771987181607855769209591840464089347717993026475816138178319620101164703046235086673631394869476080594839931563796671833715053698127488579638544943331524253087769152813629459325853986427109398088052746438762088357925061124187186100435982047033e-1" }},
      {{ "2.86348409095251425380765364359092473023338243365287780761718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.66147340815298694138846258283592760562896728515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.19957610695947876947018322120150143406142221469373757924519795345150006977773403730313023369079301615734601162566867121616810596830759956044509310651071485457850172811241087990683705383118703983140409701647142571470333867908682196499347683237058377349085352136214854181401946343510795949868835711429e-1" }},
      {{ "1.43297873345568077638745307922363281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.41681240009364082165985165850941029930254444479942321777343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00016822776212044529668140515271000711944889104836686684000447554242511520504646895662976301872271981369559245931879055379226726537073846553544561947637792091890791964244339592472180091648456672557297620691698497967178088487062658421331764320606009571740080394934111049676474128724114098218934644280" }},
      {{ "3.60098653144944158555063040694221854209899902343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.03376339210932499668160744477063417434692382812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.25275538744154336055615419863736730746930928726962752118772852620054361808569416481449220736645770958820155429279907622983889720952174836443567087662689941130263300184594808723215176724531461369529244321399979413684112346655770810116945662086924599874483949305102682382993398648882273436739947930631e-1" }},
      {{ "5.29540201171499924810703863187200113316066563129425048828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "4.56136148696097143417915731333778239786624908447265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.56082033932816309081231125453071826327896674917466727856568147854623107132514590973229074834141556000908164346345063684708226886531162257152516511661460364047957606893852284251293056880030888612998267350454535860387161852393199580592248818623376237183284153405740445167284591706260229351268548628924e-1" }},
      {{ "1.44301800298425852986383688403293490409851074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.14308392953894662376346236243307430413551628589630126953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000261963485792416853927136569198682383793838454319729684734973539039182323306199235318171814360855994012568229358281529883676852371448365509202038288834881570383584403906835594533516647281713968932211752518665238323196577224701747937440185305151190908616249351775361542696295892178890063882257116" }},
      {{ "6.22803809629584627372000937839402467943727970123291015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "3.25094187232484732597015408828156068921089172363281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.86658469541403998034943500870243069476823132486007751990824694615782893644279173776512596334159142603325455113617967643461939139102971978114976150505918018758878971981207590718138194009315889095960072548673588879789986017926322494375432269097377546556386620403909103214427884440798206932883636297352e-1" }},
      {{ "9.85553619558515492826700210571289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.19375835372379413001908687874674797058105468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.69327012293708811121144971175263370083910101169553278058859848877086709080419562447585355719348842415619132355236627485649548952876411849102018675375354238273102815474095986357261970461229247085634624604985094330242448976955585994530902647027597520370418448678147789523361182103563378206595110636366e6" }},
      {{ "3.58793922164308502070761974778179137501865625381469726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "6.91828844531360287462007363501470535993576049804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.77635990889125108309690852819112724262223736693646969131631611067612444282401471024845889526676870050656945639578159737127915466435929600368572355785708484463627825178182883323088006398240693308602881027224033181200011330121428663774743869527395379215026638702157803494202819893449358544841576526785e-1" }},
      {{ "4.96146848614257760345935821533203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "5.36919182658175953015700088144512847065925598144531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.57915370211935194723362359441601671775858730114448762856923579370262681814508708674839371719677055754578289843591415190008400693094242195743768201495967775525358591358166609121796030602701836084062498854380733868656008519376928763241674707747769169741392588580100945268376807756367217092609019332523" }},
      {{ "4.24495621335822494657116776828331694559892639517784118652343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "2.41273133041338017079624478355981409549713134765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.05650148888948869811331120941963319541414375806822687983830138205938420862200064078576598469877831249406211581526927963122105281131255004573845222711684347603254852806653766035434125383812586900316395999300527269121533046942253679078218107723668449508163238533047093749711627477323541829484653131292e-2" }},
      {{ "4.48640427032238338142633438110351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.57545512531989570200255390552968037809478119015693664550781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00002050314697607659431018359659361183713303212753613120902527904331498111603138394334757418054814218670389608976827498549914166555768752494414195237102560486571709642923543196007200071132086884670852209514113439317507321835737223329599950365222037362534859303381396326717532288233329927808913598785" }},
      {{ "6.49749570902381674386560916900634765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.67233186866778105184039304731413722038269042968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.37780724079840208711014533649991541109470796504084314835156759344214993302863952807171583660778803863957191115559320339273255056520571556246744533838811694006166732957726755945662570959533321672076547914911635593118101608294069731654269467619949055336628326621660012876406202543146880009002104821553e6" }},
      {{ "6.09674367112420512171411246526986360549926757812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.57044841435982965565515412365016345574986189603805541992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99995606813099499267634389464438683850041060806908350269346321155672893761130577244819124593817984249660934044363638948086248241297434338993626965911693249278796597285537898160766381999495115463474497883772569241592760426493272254012575287314033420480717621539078495596903806489817283498263020744184e-1" }},
      {{ "7.57503455130639213876975190942175686359405517578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.25659927896368083096931278497265793703263625502586364746093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99967576340235983529748322865757677186118554170023614050204881657758768441183672920424294568610952334609747551908015545068832481235294176033402023954402315837530793201944548984799957607361928025651266738940647431602585783131780321746876553060340085541214977820888348264644079307051095179685852079945e-1" }},
      {{ "7.23094922542264449683013793901409371756017208099365234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.59957635240683257102367598179171181982383131980895996093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98843860017333489092110505899347130551976013147438321970353479326257793437223848104429707854561661175450570835038515201752789245106905767296900600611401551754756721658819816241278926439311942572630487380703760937536160330693843537976099522621025621041857521661138411993182809798954282325807746896512e-1" }},
      {{ "2.22347866336470469832420349121093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "2.67574314969844628819828358246013522148132324218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.45597650920700334896440864288586774552136744768533970151074149218272851829624764505239431335468131476208480181217427109189696065749387968903474055278714181692867425657254225967806378137696550459591101296481511902742572646395696145586249150843548288850238158312747067673295059110101209836587291678717e1" }},
      {{ "1.61969754204121987178899289006039907690137624740600585937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "4.98433649539913870007978857756825163960456848144531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.77061455335439039778043359172467746025200146785630273407892949860314031306268447375871357416922980298986141833641561954857127992214135487677696097641048847861017455164355255934271546462888696419881919311556223441720520985938735112636787538160901409982382930452852563601752376961873325250302513132303e-1" }},
      {{ "1.10785802837759378425275258450710680335760116577148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.18652221207551884308087353225857896177330985665321350097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99946575396258084502018839446238846584110705812867877643924485585779113197512036317728340100285410504878437697437320927763146715271389949541429089452281046460116468251613835907291530307473134875676645637754443536935426771393775580503285493873005924313301673550096584687732186248324962291699197441007e-1" }},
      {{ "4.00900856150933880290426758463695477985311299562454223632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.51532437768501453796091832382231245901493821293115615844726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99981169291887424157759932262067246066753044936086128253970315838989839817149914187581608688565806547274970455207936904774502513857569845689591879527047150474914557088218340646463900082270385606124499095033110795779754466447281328092278318076549095302554108789079991538595665731579745334006426859960e-1" }},
      {{ "1.19021350263737038868705475636033952469006180763244628906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.68290288542928294646117137745022773742675781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "9.04852417540620009944566041453719467211390230149498516011819535440728246929538100576552383464173953438153608848998091884909329517100819960017904887302473398843345408874822800579053111675353115760245493436345787702548498655314405766561270926511701674435330529881522800583133171192665005622617374333010e-67" }},
      {{ "3.24276265467208501549166432820214822640991769731044769287109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "4.45408954542178808083674468321078165899962186813354492187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.94386254687686946963322383552185492075977337531973213930838015838274376322044860137122810561067818353529641305366066211688682157075369267213623551618566522232634253178245524174116891287095569137130501326353904791232655125453161983083517758472031555434930112729665552915867163490039363051308443099856e-1" }},
      {{ "6.15810616566200508259498747065663337707519531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.16615056291079111083774932922096922993659973144531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.84767212442093730437806115126470445228033445461780019945996712538430276008229376762120777978183512697217805509135802054380845495119030204478634136438561790634012659775242132306345870420149133737299397362388964674004273429551607587049902912378874846355087545159480302396357641147726629312485985544367e-1" }},
      {{ "4.98639800075090606696903705596923828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.77610720170316863455745731714685575752810109406709671020507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001921243238237193183827489311170506446717993399056537282384027062526108337893488664380391849139457202726524784919352515463921911589077429652463880766791641171380378763553978844617450998857121720620679684506344842701809891676387117033562056914974016308650056416678855410514380179891286420315463385" }},
      {{ "1.39829478131476389535237103700637817382812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.43499530583138337064275091847775911446660757064819335937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00024882796344591385461239880845260384691776851307067376321212831043330119768926261152601978474535337925904198175128520020096443925386965960197623129024285407478015708065567539370247164906581380771634920314388644089046167410745320267558557473972499677590220735439622489041126850128421653983634543138" }},
      {{ "5.53823542724402595460730935883475467562675476074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.89513503025480504506072865211763200932182371616363525390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99887300842861835164017418113429559930804468019069798605482088141983665217815479503166341320807290829070352700865762573462007338287440579346738382811838185553843038382146930720875234411175792603385681200743384496310942148612050283753172718455515566946341846284557394785250850393323918755270713010620e-1" }},
      {{ "1.41121361373833846300840377807617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "9.98754187978802221437035768758505582809448242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.39052029796505087174799036572060834290339669936231503015871931355721722463161747207749930806516332386763841930651054884331973674982367750538370231486178766251016302345689298357523496706436693633349722362506875465430935461687316908940273568043577086452494403668755303085058972545116855832701160169233e5" }},
      {{ "2.89668134268413328424353336831842398169101215898990631103515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.15865345460021584855780929501634091138839721679687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.28205556117411408113652874564375329933423424935399606070975188165013549346043442272013469493594176280229509402715241333520310808546934629428072668916547032154136142048364995378685732366063433675907163954131129548865718748066808278449447269502355535592014434886553141414017802588692926998785625006881e-1" }},
      {{ "6.80370452266789652639999985694885253906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.55585667546695538554502036276971921324729919433593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02281231404359612151527353897929180348478744328820333025332541070807332704656155540376559985312945856024293947964716753864592850899748540597581561259020876390111605898100116321942106603048757052249826147692127068934907088939143183343054183800841834132202006495897859489577613318023621649632711626369" }},
      {{ "4.62504113144313121035178237860918670776300132274627685546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.52163427846739888679716834474220377160236239433288574218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98482342652643471998471847918276729621115291527655407478347392440646810723671728307137559792389469491363165496772939188998042052370839020230988405367662374022902953797332354477989367493245470167443057076807839721016298819277997067517609396627866901281229526562169916899579412733467953808761281274902e-1" }},
      {{ "4.50822932159575771038362290710210800170898437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.06839470475542444583005874392256373539566993713378906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00407725957483558818239220892424948692876561612077691060007144800995480352451729509031292920021843587131856132257906499443064130817064638818765366250131236399994854411601609832821870319397908684888708832193783247348840517389795509100198832759307607589651546707827898198086014636944156848700425178047" }},
      {{ "2.97403319008989266980513832550059305503964424133300781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.86474774881857552344399664434604346752166748046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.05563040619657475424023632617553458144221260647471757349470769600263284068856263065983919485584722356671096360406091314847188039821446085937501687228128412834520625095444394738723676139779119999762553228944695777339743845083698817002881418226410778165865331625883646946752393289471811888730325566680e-1" }},
      {{ "2.38325080784211650097859092056751251220703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.66302465811868707690202029425563523545861244201660156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00914434741735633383477956146627482536171366958054332840551181784200344603582400731522959980543848217816660950109678863255860871237640121402147335369863635255623422637578873942180607947824451388135749116333642343887949089848641558138185462358132315292011403950110303162349396320495083970574820485984" }},
      {{ "1.71423190201370800878066802397370338439941406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.22776453453089651191021403064951300621032714843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.41715373496456388539324810858409600194977863833255435782643057881295200486872481370727134110261522844531904021514609023835218872201273620577170073750362820523072032985530745444647038907617563505404059737966630343043214665700476103118304785807103469358517871674775523305518419709923690374745267779192" }},
      {{ "2.24202183750390773639082908630371093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.88606137138923326407313751928995770867913961410522460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00356204426464651318381353366766656275190463831503522239513652128295092897680949875433153708401966246976000272332294389517143726474038495934684353460128449357504408039179941027444904612222505017810534008770086553370099204575810832664720970035523251219105643922854038752127247330051903230606763080754" }},
      {{ "6.68254924276967532392745852121151983737945556640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.37453434835288135218434035778045654296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.23884289615444976971058885660437787480627352137562427420631247881049437291942148842029267007686370097695430731343288589752635360722257432905786816202418979845289181427172830309982121527477449228884669859681103900842455539708329644639171581286590973017189366587396069226010533102477610964537380565038e-6" }},
      {{ "7.08735989603501259637141629355028271675109863281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.93170577274373727050260640680789947509765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.22994581178880428430588361219833883540463488675474104323353907511764504539625188516288826354093370529125373667749527512470360034116957923611728143647053700596011110669882224710437210488847861184277118067955181730553186100384807049379684590737156995709498402541295775367296456895644229479101255193586e-8" }},
      {{ "2.32014942323958166525699198246002197265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "4.44810310131192876426098337105941027402877807617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.41156739924941071144222113991630683194620198146427652318978812945951687592217439817665385168004660248141475796965481088709894832854582569992290051260681221672250427616673943995167609515610097891251991676291118804434690243303234426750763059520585143335380969303297852505455054177640936866538646738978" }},
      {{ "5.53282998141260031843557953834533691406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.84121602625912601070012897253036499023437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.58373770974866315239700400474362989477383658235694014601763955308120813900734144800418886820772189374330395186134971653113939250757648994415649668842051891937875262317539557069792723676293862948593761228728392700591159530165592758768030540795599964339740660316188438257430600649159884119157424191581e1437" }},
      {{ "1.85718777739859418943524360656738281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "5.22760764621501818278162532216413183050462976098060607910156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000634214788464577855189457084479231510812185767194037441581727975561469887723538341241488476848396809344125019310357146729535769441778199276860378224675705997700022744347653159656289098028240954603207536444605258988369343538458427994444681579339406242882282677673848923548267602318784029338545299" }},
      {{ "2.26509465497761368624196620658040046691894531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.03762510179397637877229954028734937310218811035156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.03290580019042823129797947218377400320349309579887379648069219653044521496312022139633395975447973076157177628011296170468228189168841156259900536323684747126636993787966799478738521263932697630770091216225209592893137197359045523997671979624680066252774430678393930591603773849047796785558663064178" }},
      {{ "3.00476295219746433740048985328030539676547050476074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "8.02409308153823985776398330926895141601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.14136997855533491337740509094908136487500777275671775674006769752964040924959289897651192452522290539206601428332553008348132654735739834724305690164751538343877568881013821199733121651153278576916744941958595422652696891642421407236449964345890318648778746090208231400439749942969896805715062234843e-203" }},
      {{ "9.70798896667321969289332628250122070312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "4.44245980458260469880826804001117125153541564941406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.04162803810086699019811359451605453492327199203637588333443087016790236547076001612983024940969264195708947484110311219661303765376669460085704437741732903824466154100734799424439147057676949660911617676769474501254927589549167841876728171722481679869607753266592648983928705022178056753619502736504" }},
      {{ "4.66389224589813756668976729713449458358809351921081542968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "8.41598683219974930125317769125103950500488281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.53382042139791789590459891074340066702060120383660058981707324170432197791546635984529280576877432127625617337120709804544743161881268886942872653531280982977576133511801284796672543413476367564502976743587063883733930418435751755822485352709171649159924126024328941448624940766411726855902924670549e-37" }},
      {{ "3.59808415633625783280891141657775733619928359985351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.89106070923696201191432919586077332496643066406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.11957324795765157901703029938063916662894000731995996196921601505519523032651152137148283098242267387419838627536448762986353433569901640648734154364900058234967711292575255307936095926242393050569604396410567616604312846123878721000606629431359673384574670669906697802557686478096291498804903738631e-1" }},
      {{ "8.29546112810515936034083672723227209644392132759094238281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.58324466499312066503080131951719522476196289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.86659658288740565122000959721270284304946720479579271651550103638161205367495371607443353257219806398111103943515458202894485596325631197266478165244814107159447017816960534609382148164169335790367935277372018345236031109996214855557877416239874683021494253068006756484498147004540748744410389570461e-11" }},
      {{ "8.63334593891555979183749514049850404262542724609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.53080323158965114771490334533154964447021484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.35233729973545869852697007160756749812975339497681040282999017995032382608550729201580422775220111291799102420635253476809267746237352183450521514570985348430714476644398637263228584608977432119909919809882847191196859131282077491115028311403034792745466352775458252803189617423439193439791933123479e-2" }},
      {{ "4.96802530016260956990947761369170621037483215332031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.02519683572433439877613636781461536884307861328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.79130174131074712614824707606461473196424333567411409743609116414751816130489331857231684597817414740294362613028844768413127008896909984819847802879905875872491406317289520166024819440680050252863090346899246519659384416734528764367429297819383807550500609868490474715994290375146313368129823880737e-1" }},
      {{ "1.24702033554851823282660916447639465332031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.71070595755497723185201142115374750574119389057159423828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00004316789477874277620263489613820769653924396429221925198065527832144152145855155330213971456389989525794479500052652310504867952437063466242776820964127214900733034755428153463014611408545155223418498794841462940237893200540509678171081868776875097700422657888013910264387140255547155547751726595" }},
      {{ "8.14130989781463318757936953318221640074625611305236816406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "5.47674360353472167162180994637310504913330078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.63199095965818056903859668356569903606584619352293451010588563431807635376986054905897873784109387681218165026966672415089273447045333707684197569751029547891913778442699838628510300753325723159251992398807517774034918042863197611983714480066975243968618918488213538490759809651966026386013669113012e-3" }},
      {{ "9.48316047562641761814461460744496434926986694335937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.87318705342295180847622670938790179206989705562591552734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99986164907695049446699147999970503354260553000953839878275159076458261755428730763560216631851642537174133195419034054495238529976374508256949441111167727864128507860277033299044051865971449902820004978297018651282279654117307535005136847959876138249613275046667240315151303933181873427450473536424e-1" }},
      {{ "6.72439622398628665533859560810014954768121242523193359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "7.25371945127648842799317208118736743927001953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.74287216286274945633266572714083322962101975717027023442493643122162512912802620909159732325958549010615659395950026917825109006470507725724883444926417334797999132724229966674370457592343778072797597317430071335665732334697884217471229705559230148495486170835726019893708870900510074940623722192258e-24" }},
      {{ "2.30442309265876887366175651550292968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "5.10077615004599603520940576117936871014535427093505859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00006298512189780533426621327292231284630240394286315128868176668180396156679505765231684868863847590833570938788465357619602169740491681317033999860409913654411707265663879954547321897801703570651803262843624273611588702980486012373865440788314427239276089286730821085936329475836740228791493079509" }},
      {{ "5.67608898683707252530439291149377822875976562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.07683021912563162914011627435684204101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.67068196412266823466861212116241893055807436102731155140415238019317223182255528301301262470626159261154640191153443230288813541880287940859066915763205210708548641176944414442344483827370453800486623574896474317715521007693847769491979549043149389934860292588058888100428936865217794693086764880012e458" }},
      {{ "2.81337246606456581503152847290039062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "6.76344844070871521052312758115476754028350114822387695312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00084899099059737772913859365556118027581054407285954758329055990529529377511066455820790791091871502993434775634942750452313898053395589443975739777882392931281046995883876066279980856281434476038018812014179817821871586805042023183706725376177160577170143175806815414662475357051404586743878413781" }},
      {{ "3.34983523837648674431610729129715764429420232772827148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "3.20852082105692326674528658259077928960323333740234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.74653990782509243550836183201075823369159155519692846798314436930630346572690178245424014813711682670551067058906680367646181984756645025448347553573937829174199564122446362713993201587207768412845129431576839987405545810458778769778400165994012026984577717831659943902405397254874487260228557294263e-1" }},
      {{ "1.17365003664101777758332900702953338623046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.78053489600925995972602322581224143505096435546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.05893503437891990856263354837986270829833882333366791751057714437977103925035624856886729661425365010268503431934637684230477655839134293793513971115798026793973457123299079643692631311303731259185411381265912635677151301389868547814580071368806352687212447229499485791883769033890017890187866384697" }},
      {{ "6.35473191480731591582298278808593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "8.15332646258649447190691716969013214111328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.39785593586175436998336047723152633390674636830187065022507274459265394076960202352145842758484227495355755965560336623550772098175134115092702375767668586879200691996181864401350579269221993542974245130770033239773182375358583651870979382999043883377805409110023459275245515025837447884289450716490e473" }},
      {{ "1.20953010742292040959000587463378906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.81528620999166889760090271010994911193847656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.36836082188852137389175804388074236213762814888145199618983165421342520318145120542719392963440700570926350816427711317556249274733873103430368341047528787950522843632555485443202700694068291761260828423023146529348899979040923182831897371799138328552240635790031739904574931369069808189107688786563" }},
      {{ "5.02826864423656406943852203994538285769522190093994140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.49148497614115050149963259684682270744815468788146972656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99886724184991506266822294352662914256716265702852450641368490227402118731911016182228523518485801671330942930187800995820804252420406598640376923122551567922827288661396415930336442260706727249870079649992447995120206397679683465259479974096061438201134430443691684817587750638183840953282722156209e-1" }},
      {{ "1.77623605706485581789272343300467582594137638807296752929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "7.16881559843790408872998831668610364431515336036682128906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99216151571584206550599285631558723279674057106881165894202136106399680298454020164284514110193651012658901118782805914666290560642145072003861919887611842296547047036059520963486973019219177746829500434614825844519776793639037139709040279058532565912742721673028532604681375807832044191447365386410e-1" }},
      {{ "5.44470752993507289829722139984369277954101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "7.89424994358623844748022030382017533156613353639841079711914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000315551756552929413349733123431754994533967419962917545562731445729245526256276165331606875912022797791137395975129631042701829381104269734643785979371368397149024305719491289967486350643357572991392648348784068298353851347894101762457227438358667841811859357203913787202416700523836489423208926" }},
      {{ "3.47952865539182312204502522945404052734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.70647055379554400678099312926860875450074672698974609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01003641035516199003729871729172100472722683758579980406598003587323081240911466133797745466321375213116846128724578817241014105529656120259049841543565831847323023771528373536119517374544673647922841984354734790648175557634969225996223997613913923117328699599574593399806697328078695200917124733876" }},
      {{ "4.89683247131728754952817439516365993767976760864257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.96528625490362656162875165666292787136626429855823516845703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99989546369402488423594127072150976413227845609208465040224219805251052465526350038203035493184013561718362361567469358800995276604754108514280603768744650567983276404672018483263585666869718723877857972643772304671740644914409892158811218430582592069739462405196179208350796857344173822141379782399e-1" }},
      {{ "1.46846575134083542901741736841358942911028861999511718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.13961295777929194628086406737565994262695312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.25155431004176641664360564432608271749851863826540763464893400702786658931519008803678187122457752047812131114041410952546777155616546380522319343918668230249104290662019435519965989945104554644065424460926764956490748710036580200886547368376656130754697653950693222962392418484839778727280176131769e-7" }},
      {{ "4.92258328635518438431972754187881946563720703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.16701285749932963387021089829431730322539806365966796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99846424401823072211356827414998110501335375349159811161820098061831954942111055007312660415928636319873898347702590642507419097876493991759663677610295879480797600223758201686500279349159643821096880668703139199785910150322740521690695804135686666464835791331742866153491003010427701847514162106732e-1" }},
      {{ "2.08891842024441984904115088284015655517578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.37451123552204679256760755379218608140945434570312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.51853459491101499462028153198307960464036529321862426769066486207166171296494170498179583135034855145454026518293079000045567402865046530341818676848400774327944796661350087157800975247279219703171839848135584471839442165788564723636737356977396947188532339857132058951624239309379955278406455872013" }},
      {{ "6.59693951898326247373915975913405418395996093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "8.58390504285117244415159731829589873086661100387573242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00016195756688330761491474273615136423552556614577036197755742996689336001512360595235890000441982158893267294826116899101351409093661584157256387065246323209052057870928088601548025462625595893993027794606583868972654751596093207938303600634384394989635869479864226583469214082919105133232859648980" }},
      {{ "4.72190302282203920185565948486328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.36872614795644276242114756314549595117568969726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.36273289377674003549552956596428296006868194528902491172931330043916572933784595495824585583342079403134944318241053700848989680616524963231553615168899405338957973320487490472555612135391925458093520067531657179731576065041675903036737870428047782297012296101361469066116145701281560378942697624754" }},
      {{ "4.82034977757478931038870317848932245397008955478668212890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.20114163951150231847420091213507475913502275943756103515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98806768192609411674337267672424056960083232464271360229113542018960233572696008296128150338304172594110837781004501757127999427554595909914289685513812993265024374119615672800606621652602006481141937435665444874495693840170640030665928027259502446424921700891997624978473239252852217099622859393453e-1" }},
      {{ "9.54197957425642506734675407642498612403869628906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.34304142023469919031697372702183201909065246582031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99984326565139803772938990236401058117431661922045573846220380640349699832779123098058540753008622857674232899802697126593866411818203678224123515319186907350469596960547397261912465753250671926347546302198056587430826863955515669743258039245240360599418185587449465990312391796474083173534248199413e-1" }},
      {{ "6.40960699263738433728576637804508209228515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.00816732049831557560537476092576980590820312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.44157504422123306075307680246115279780269010737951173392922320903011026254261224810748434749033392788723902179648854339741924679540874536758451693815806813776859218948113244848659597589114467756894234559650445795659560796369660440795499707193716216403592941616722628011042403021580672267269382135734e182" }},
      {{ "1.22311589009840382402671821182593703269958496093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.10469349157712311976231589483177231159061193466186523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00004238976442526402787977257335501538186147626008780569692886964327726967250839556988250363044075945205489441982444641281650300897861533023502296756146804366735904879346020295258735438264424281840920414740589412010520817925020516062766522452589181209496869654998063536725728358771467633669371863258" }},
      {{ "6.39291118430208007339388132095336914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.28391975830675164615968242287635803222656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.69505481296349945509651870173004703548467117826619443587513918127052944692060859604078331125022112009253571107992368448403956917792261351644585277417570509975725723863004283151424512684410008159602519982839123505049564570077423370661622862250195728714873045142974264003278199036815752137105673556433e4" }},
      {{ "9.83190653233646116859745234251022338867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.16849063361153184814611449837684631347656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.25770980590636267172342833096299211879468172398651157530921609762966193936659345808702337762714296668889853763786019297633422123145533613492472992798929637639386760407400078511978818609469799050619387136981965857651597610392961103496543359101531210986834447630570812748220097342632073167798940478519e830" }},
      {{ "5.47317500559057634745840914547443389892578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.25569401906009264768826483305019792169332504272460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.05154283217149739996322394073132732110285973177101672144476243536792403202554641841303821681674178605013975392578098758039213562899690576014988268448654777051869934823936115258309781471880324840875908371004233345043851515355481534420805940488435409518654838065662552503310797307929345306698760361427" }},
      {{ "6.49123836382549113110940197657328099012374877929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.85020261717664669109950636993744410574436187744140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.50660946670306311914365279482095005173354041210571854577128244379583503141132158648667793122314902465794942399303782457347308120094047863238127874892080844189140303644281120192942849585627941033052970332875969503978268193734299765207776684899260119431935484734340685682552324678913617232469193989216e-1" }},
      {{ "2.24692705476482810621891017532902878883760422468185424804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.52982695718321792810456827282905578613281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.90763013958916250310703922628537395346144248614383500040096138375431039104002049367585390738671354254176978486242291556216285278163895739264649656370403930009408842065500683071987767757954595210907449882232438368368764844630512751091553559176805208777768953085505825846870225770460080538105432353809e-17" }},
      {{ "1.03776311938838771311566233634948730468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.67758461032731062689826728728803573176264762878417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.03459301282359622046275858130909175858605738103500377882461906122638713902728757596922241637837677972497613038491667511030599129584893938038824302224512570610802632172627579650514542361582967805126424931338365938119495726516045340017819117717607391845493489250022446019685368890472485823905828290701" }},
      {{ "5.44463531752032992017120704986155033111572265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.08467925444702359527582302689552307128906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.09056011709343327076534816446309237286970058275834540830372719698670511928227959632093780315386474610345825780828956917908117330975173563693122316666283875774232598613235043753932040992980072681882395580571647274880789017536786697004043527146342091119566202577901703268215449281872604307112909919335e-264" }},
      {{ "4.32141091390930114357615821063518524169921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.79962553627032728620349644188536331057548522949218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.07012654071632775332556853428357456635166481137753650401618414453228087751942869456669180112375507267782856652222749904765639096915852769878278766680206962666525967341796532563009425557062229126958039051450556007688280632610398536337196273409572525834486954342631282999563123455233960154445776600968" }},
      {{ "8.85195395209329438657164246251340955495834350585937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.32014972604017088997352402657270431518554687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.71305713463862717236321613646893663926389070764619904563621824742856237889605485601780242185666715109642520925666586793175066523791156636682971412268154170045092095602942426637119803498838588967096817482914325839210531169438704998222940708453881877588305201012231120626503012557840854056737490425114e-25" }},
      {{ "5.11800358239917845513955452219079234055243432521820068359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.53194625802088957300384208792820572853088378906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.78675898815841251401603750577856756795950441245444722873452268896478608982683458941926503132911522781757532432483294128491039729829837206496101278313685758061174666050134299787052246310316361371623036791254936884792176693736621135978412809784428098496835040268753660201880757384305210529998116341854e-1" }},
      {{ "2.32741466149031985682915646407309395726770162582397460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "5.38640739832896395418437851532189597492106258869171142578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99549497162550438458304414161333920701781715268855145062033883882900750111819551249273252859663946257284929221986573479403421187776431235203160677411657816941834327558315579342010850796902138914924928940560282841455433144115320935123004884062392612303227832917659972351800421646517607120651251528968e-1" }},
      {{ "2.88820917341887019574642181396484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "3.72679079374876875974109680100809782743453979492187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.04797427152199505899776613380207153019163891696040089708565511495563600355422445163531206665239877024907641710076156338867860992891601417074486301976244681251883409289427117509266966735400170413169626794460003914944960446253186507533452481416547111722260834345331494363350198444447590272333886931493" }},
      {{ "9.11769074488846854364965111017227172851562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.34287283617854882322717458009719848632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.61412399451200792430940441597911814328017657658519351363425803166936981320098511499093754150499459010245435176020174953699548406222107608392288974835149178057255386825625780361600243592099112823611022187187648569904297395013151014275208089987059198137284635106672418842688183845527005332642735224166e1877" }},
      {{ "2.99902079785064756833889987319707870483398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.68497592645624978853480246954177346196956932544708251953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001063692844088210456002419855074011148005470079145471634788112628295771788190807123424233524759763995207569368229563514564551401717941193555837507092120724642714186373549513618453183103215293207086254916107790393722326161402579919794484826396299503609272588338890824693538523484869461094064640920" }},
      {{ "7.72854459904651157557964324951171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "4.48978348521208037863461370875484135467559099197387695312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00610574384563796400321310101756992783528974702739913146260646865660943239706104030906315422031586966995536370399679280585793557349882649609015460449277306668457058524687099432479479161198093426061867258598346229732606275929747805949435342506588647203249401335413541148238532118555631296171613732580" }},
      {{ "9.29770372877305817382875829935073852539062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.35261872461758123759034688760038989130407571792602539062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00037864179461996490975401602514900577512361928909810135170262283211442500106334885446423595314421485481346967556782507370070595069946883492382064486384013379235733594798086474310298773717536986583185147038897974675666723220012717870637206407697176312759927758943687773764377471241445265488722941855" }},
      {{ "1.08593927041971596736402716487646102905273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.54205328509148672372353061632566095795482397079467773437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00119232709184950657347355900529943667776581936561333029642844322931651851922987396688070423513516427393233449332830940932893802869470044377136541346608056612546712420915411514501773616973022311857494372432610405714927103499401270099666054931681878021711299630772502630342850129330691538468637467235" }},
      {{ "6.08955210127030499279499053955078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "3.93997641600216752522101160138845443725585937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.18201219777662049133348544929685038145294798461233898669466368271726917449741209374562376908881805268589020897308399041338055306698995074087125055703556791502100679603522120055594862139867723186128435543540272128120797808148788656196331354569711892744404054200303870971480027638854018077003767037387e22" }},
      {{ "4.17002058199641822966441395692527294158935546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.27638775324770278722132099691322082435362972319126129150390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000325050709508111108865304342178215954684414490544574201155545551225580135488259163273316494247157849444894518301320733204512003009459708444645141220074915996395890189435364804366091830247042355490183040883445209891891864651638351487974364904897430627412415757188029347485931227191708441175016047" }},
      {{ "7.67550081598081896869234519442670716671273112297058105468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.09358860541564251178670019726268947124481201171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.33361058888703312937877266409714100834748582910004102998779521040115944440359724219966898465717158230452195981198500825979353396690326287544928081118745655488492150353523103453131904616559184005975246310430624687681784477200062656846973772093687820868635241069192671298954998139357716654945366635020e-2" }},
      {{ "9.11928851435624994337558746337890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "7.77150273145831363419411275117454351857304573059082031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.01072215443565297495212875757828431150173416786594077990011283253611684077171961857977658555772993227256701225998130851047810587496804030395055038866276482446366864045774522304245866613460132889356389785400787106849829332460233953000396006079677394660376277801825929937944079178448460996666434091599" }},
      {{ "5.41420122645918127091135829687118530273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.89522592882503256809384462044931751734111458063125610351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00007565289778675652984140979321399062757791371437603651125252179594586538728496280877795475718254343845999376499222253067347916841829323365254311918387151598991472092918535912843066893311920458448486904240767256448921472261918689220109607027673999114234655407258002618927723100761498362332050377153" }},
      {{ "8.00879921885989665142346893844660371541976928710937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.65125145484174709054059349000453948974609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.51542106881981491731489464208722559455080283729538316329978070693260585378415308395373558536702307332302792317627310748549719761801156867236663991267318247059438589418271824484275577584765219427298744283723772247523591169183694405713456066811755471408195734266771535080425449850856084868376311119689e-106" }},
      {{ "1.09156997362350409730424871668219566345214843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.10878062686337663933500152779743075370788574218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01957066747628902940966302510092960488528925253540221172663471275503400990771789589916604068584334436432589666410671717466013196522730609847023637947764642131621149159485396625786863911618319465245064605684158683676154002669412578226202367577007287146331656090836085181386754790207143710478024828107" }},
      {{ "5.27981737437816178726279758848249912261962890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.41970852096179633861083857482299208641052246093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.43686253241293502543031164235680564280493113934425702290586259672655229919415634458697404687915141160074152506872313697917935113169212018685900336128740210227144508079482606408290248099236632255964024354639101204012497824568760884660053718968331056137812934970479251589590682389747704367113989950745" }},
      {{ "2.97632424685030514410755131393671035766601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "9.12458361916200374253094196319580078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.69742603562041824153926447786929280716116286355510132564473072728789802885515779796188477871744149730979853211785348379587523589298115735526488355106077778856284664084128422453984824424076757383964141778964343798745285559893404942252682634909106936299245602255752362338032501973165524906979409622054e1344" }},
      {{ "8.12277007629636675119400024414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "4.92673741168140821855558897368609905242919921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.30484996489149406381687564563779425178418692009670007244171321324888859123491778857160269626310259110411696125879945417306210294026918965107428093516365217530286177050990880067424193779915234192484015646474192080157981645949288654940510168136275269591421630149209078542082633232345605476593203338075e29" }},
      {{ "6.75063843580995381898901541717350482940673828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.02748489106749621435632491284195566549897193908691406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00196404947416174365030420006016203648060565955934806957206111637856432455172895201651727892958586584412299635092294986305134023877029915464080887124975616815060819276759381700802994334432976763795102196206567937500407359464356440088544622676160266170981561234435498384331071158546292310523236254753" }},
      {{ "3.50174444164054002612829208374023437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "7.80282755451220007281154522615906898863613605499267578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00819798888901128629632317940565742032581044787310510298394051237610722547925779466083057852335930819436022531750188258730486171674268778076637531078630844955462136795806889121317307097195981029957098770479755079052235805500969330960373439119072852284817639069793570751583988153638659438922106591664" }},
      {{ "1.26564950853483537684951443225145339965820312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.55352677703511933278261725188684749809908680617809295654296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000394312365539989563625486331626137964273811834798695005780013495760538177217100805072673596190622159320905648378255454000218262505715454699578920413495174693604468418384665870032433868756106543653906692751058614669494849726609491390952522788637204327400613561970408136313418894648226391505731573" }},
      {{ "2.15745882397261681817646206127392360940575599670410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.37905683348865193416088459343882277607917785644531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.64117851649613850769514625497977880278916078343037571542452910699608639199334716242087257033679813405503457913766626678856765747940796988239184619492044398845847330875248279377691256718144764352787323595310916665202454075432835782224634142882556904024349917556799442289975480861859420905292552431968e-1" }},
      {{ "1.04145372167880889691105039673857390880584716796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.37802182325172850596572970971465110778808593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.88526852493738132239113873166539203263138672058127955706951262195153797473038707601101237196070886256644090185553908945280688151557324839521372121066434881841798255953853902832020964805296391060444306684398308798244411237112095640393380398474592654522034844007881900667824504043884214270458892007901e-9" }},
      {{ "3.07012551115110277066833077697083353996276855468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.68879194147396471947786267264746129512786865234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.27959056811329343392664000245431816256252125501370190635108304735068414726869009253945094404747463601079257760616120868020903017002340732110943618640264478113302579667971697324950504699622151612181100300886867598913337113652973737056690812163582367071363756081204514554722838161602803979904270771466e-1" }},
      {{ "3.87855552747419096704106777906417846679687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.68123953003118842275398492347449064254760742187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.14414942990692656452341784125767590353064260907248467847740571016135049928321189261126610028098851785915087403250818786340815261171401002033061136458109120291157079359913768807250954139969052510045722307049663169894534742927788300920701107994896596920105547258658675585904959114220585301112702429345" }},
      {{ "7.26004581866997872330315999533922877162694931030273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.06413410745037140259228181093931198120117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.21086963011262781557962916039162488992600789192717542131434452153601149170623279185807445011140592247466432290015786700658875253434281122713288249913633617033214856219833669907164464033302015146556043425711342744639683355773156696931162767677537366527811567871143970446954952970890842842464186617910e-1" }},
      {{ "4.11288422352877880427968193544074892997741699218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.51736412939929010690320865251123905181884765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.24565288563751822331299972603179488550349468370067144446751898852644002616709216207408151506565589257442807820798232055968358812640290558093447292295273379034573533601586226697268290915360691120466719143962043078879323957769794563236709847407432551615445562560818926972156677354022231138620495943375e-4" }},
      {{ "3.83413786726422607898712158203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "5.00384576840691153165607829578220844268798828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "8.70586881100241909365086281828106615321437784717557599878404325717613404517722842063862151116432421295729131945352592123500947115034261116251456345076603582494211858526398683989440066160175940760838173788310914276938532934671434380694237696835717084352874743706871899256563045584501545107582617433574e27" }},
      {{ "1.45287638810286156079709662947152537526562809944152832031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.12685751933480298196330049442792642366839572787284851074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99976308406201382296511195677454028213966050559386949061001176100458674306535766658075727649858861936143578824990999294178518894376297296437371242014351633015801209221765213206140971726863151455918302666401442029878834264175629251102478211077868646609228275473662440136138981614575523961652841689231e-1" }},
      {{ "1.06060778784217379921983592794276773929595947265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.66538555043448722320365007476539176423102617263793945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99991775850777769594805464682102392095766388724998006545282885743063065509840288039071353911521871815954556155694843177720505424196812072094678443706799908575637539974472545844157811962504294140635183658965312242832638263593804028788156476482121148602286677141129409544764834011834242854688050032534e-1" }},
      {{ "5.88086377090030459208191970787993341218680143356323242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.28950414823927278495929726886970456689596176147460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.97772228399557115917897595553123200059617745525398631036747902716221825268917059130877148214691312502174434355597848183024102187672671321353149180399633699090375168044589993636218162674313714376215322688591337150653523844732601246528148296899452660176207080310884559549441104143585753370139662688312e-1" }},
      {{ "1.82088437442942030486392468446865677833557128906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.18170819309718790101129681779923430440248921513557434082031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000130754723378331003328108523077660890480086159377400421770916129060299296337133805493313328717117334983973342138421022906982702012383037259191961910456959250943670084798790257349266900179703918444391347033677458675178576414635996649806132816647233201959427396166295868642002149628537093670669003" }},
      {{ "1.81790864213692415773948596324771642684936523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "4.11112317505943991591266239993274211883544921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.16716616089260899158634927691311476940264689424270830038830803482761816779192664467113169986590716888439242048069862364270415910055880132349242712046577768880635878273030333655678293456983297487516817843334720432032181949191565525929129951842591644277698817197191950932369312197425519306770280944742e1" }},
      {{ "3.11682213641562033444643020629882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.94098588905476396315741283160605235025286674499511718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.03790338780611240821364394096187689009151442337723127625883997000112224823851501188946812750320424873330954741819494803462494193607811995436113951374495986635400485180461374008937550605117989616259238001766980179462154007176099462553721961527948038831707776585902099492828100272597482214965326918683" }},
      {{ "1.39087864126595343872150323250025394372642040252685546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.30102723808126441440435883123427629470825195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.14022877396721831842223346046335424470246823884137406233750859143164782910052273120452909159294424816461360814736670369100321052363875210113861850087708357854957657313512898387152990311841659387594066789695419982631901574195401313785030478101180804642923588217360853722001240454059693137169086797197e-1" }},
      {{ "2.70245297975474845853308125676051076879957690834999084472656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "2.27192418381639527069637551903724670410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.66297652134236153231549630311782646457435806521553869102045806271290140313907532572617601088189845177879703715370369231877915346831429034626669722739550299919960838955152823532091259662956246982857242736145549200749997723311019597549168518852317671248226020048798842889139150504890040937192852486477e-1266" }},
      {{ "1.24917144681230411151773296296596527099609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.15140916555233019380466430447995662689208984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.38260599513773427745927397841453042338248242456988517458147773019568704658853879515960004076402277283883489795069346406439823839016514254977654510204974986928186171239215823810556312920621087546486393810430469767985081698483192169492133075210737954383435371131440185644678524564816434215238704654516e24" }},
      {{ "3.11836231019214392290450632572174072265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.07745884054014173791813391289906576275825500488281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.18191130436089764539875276066824383600066278177797976268794231640866698789531115667856789828761726794355968881132857750694814866048339710833375629295256557712732385065863487440162509225900966765667937759713986800012768351783498265785505802770202831716066805035224331887004845557896771449008224340669" }},
      {{ "1.49079206164313831095569184981286525726318359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "9.42373605503094040614087134599685668945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.27012834121533023666785182346229437241289754499051958039863912420351581912036946871768101343922573098899177777951053704877386501476998681242293961635818871027535060307630949907397555951449420495765047830618356183657814571149049220337636474144629773918060565875711712572574729922979876342390263324772e1105" }},
      {{ "6.66620618596326164947279968586713039258029311895370483398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.21273515382789454086776004260173067450523376464843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "8.77204862888219369744788573660770287170392769001738688402155865708513288003188496251740638940642992963657767448328425353472870578497253649660611575851476001092271122636729433601460150801906619434951209083981408041494135365768470816162930615684520542260356335965614720676272646217323394466315267451500e-1" }},
      {{ "1.38078986432311534571226729894988238811492919921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.93562727297856326913461089134216308593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.09060418138444387424424059062331178711979043905828333720448765913252697584689444873629369083510366476460104106313375773568706388071930121668811178870306890980290075045537520987441428335324653171796426529164093528427568372203611999068229147036397950044150945977202609787933363721956655499688121770967e-511" }},
      {{ "2.16771050432941410690546035766601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.11416271213217389808580737309284813818521797657012939453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00085620226622470905732844417905757963463091991631468675045664887604696934739150514381777470211904644648980514762573227291787791600677422794047913591269830095072415283470532919592193733010388187339828883232881043401853930198928232094973120867346831075681954343729371567448461485192448773328157967492" }},
      {{ "8.66129276411148731409928025470890133874490857124328613281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.63230476162026535980054120500426506623625755310058593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.75677926316087201992937434104172212635183402081845367050961907386982900528845357336855433020027701890844425355924313158894887243517734179729721860852203216211626732274220956786288208809210677612308603888585509873354794360927290410971415768734554863757722931375046183502833124862440220040682623894401e-1" }},
      {{ "3.35057580340428966085564166377253059181384742259979248046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.16747650803829827736990409903228282928466796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.69762726818915994102636669640248641695258916516178220951615332541358403169898089796085534108216181605452622185779736827460245748311614902401628390881069582094010945399117381574868399439034359201699565625835678615787458358138834274021214751543009200113868243044350923559610553355961511493945076046223e-15" }},
      {{ "6.53381930771882064867828177057162974961102008819580078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.88128239916411388321648701094090938568115234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.44239576906295172295161064728901353975092629051590219076800377900713837403284136783431262977433666893756151770251819781412876948522321366628456614260642901179077056688683539196520237460889892611407142177331136348438247424975172992629120895139251739138531630904163246094893508737465346231066931419052e-25" }},
      {{ "9.58824111169796669855713844299316406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "4.76775888223340871263644658029079437255859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.29079715270199960942963556473318847100392035825806894633874834481890517355918139252275167622132083673533042352315135079499274198112245747602316235108479674713674202663546893395644195003527204518806401402778881498713093151993690595544754136969841959488665417573929913679381495568123074940059218587082e237" }},
      {{ "9.80462544807249748030075764404500660020858049392700195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.92380061818621106439763934758957475423812866210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.69367957215417366534795032414092044927898150798664469523819577887485052498975506353682229088031835479896261249192998498541269057857290594440231095317424708287026045610877468508170913712129807960043309223866462243177903185979844369471346406937023243553333556543631914530281862882346915332974736594434e-1" }},
      {{ "4.22715824828159014714401564560830593109130859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.73237473383745509636355563998222351074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.85142181625610748727512967615608623642319089823180852116567338775155480625397140714648406228861247464456116191851444571000366236244452154069748412545940070600491678946250810802096421175871317523376451483012224450957973158915531091130907648484702974041033736125577265886371493322366154555363836169078e108" }},
      {{ "6.96647127037580693453833191597368568181991577148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.57526947921747250802582129836082458496093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.21776552625627356590336877317582904052748969026951170943346144361121751863809186842688158978930311760773848599440472082708351294025960471667864592532594991999006914698692274398613842119283225752843571850040859177332114176284555447226936581391382855150937157990431287391269415649433263516860035782138e-414" }},
      {{ "8.87031657211726951572927646338939666748046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.34070958469201997331765596754848957061767578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.30679204065692425620221532321916410120280977665891018935099692677903142836365793713310074103183167526509810778546217793360039724544105606790669360053805611269103519872268834740224972933178750433774433912507891375768945267742879488012985036171398185645729556921004734943776799979369402749984084043452e26" }},
      {{ "1.38894800193875870772397540520159964216873049736022949218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "8.28834370250225660470277944114059209823608398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.35222056499059860815119534190744388049132560559707635179974362835024292102863415735162094842301916959003527949486946878329851243593307281685479130665725517616797588026104563060365221513904192277451172225341242860418605162130720263571624685346061029050164764398879401328940031001593736913742594656342e-4" }},
      {{ "1.09816947397275884975090254336294037784682586789131164550781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.58183945899629990446480509724835883389459922909736633300781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99936261439304212269593793274631621066678586897953594510473482514688267255174393243589871913940083113695300948318689338051222055912287552167309726448706279958229666015743107592418824002439360290161725288760084420735569321637244499982204153705684126108823317986930400592132185032791247608619277879575e-1" }},
      {{ "2.31096996516865660353801104065496474504470825195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.05549814372709782617221208056434988975524902343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.07259003265165794231974650932471010325197411080114721904301457157547097334313567798534807882346741414784333355642286915364891740931795753032224983168581398136163731017251242792429565442336378918505638415347606482959367519064678452374991852143987534832456623394400259546622175052847902597251732867983e-1" }},
      {{ "8.49670416441904163828979790196171961724758148193359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "5.06998079666728116224327926886417117202654480934143066406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99758288631873273226282145885741739757443924805027385096536700548341955223394388330738555808413815209906793614975662061712919602993250011794563507497959691981853673962928960543216580974445772540799559436678586781978389593065105455927885069982188267742598791601651598840088278986439446428992363728811e-1" }},
      {{ "1.57311385387133050244301557540893554687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.28071775496995989116096048565651699391310103237628936767578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00003170338187821543987107336746662729109295981087069294808638827112958850528655867141095917778641007871411889107729782401763128246612168904257048743190385169980528080025389670873111442583914650963121405799362630066123464254273948242629742051351814124726982856264616340030151043214382406894418159834" }},
      {{ "2.39202138047558889866195386275649070739746093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.49118181918548543762881308794021606445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.27339014094927962826314328575939814304054781018385168289190485168514741352754753149678911930165646861374735328175462043586928267957865268734389479790773849220996848074070389175534751472812216991107745875007249116250242613067032470443930894251972922681126465003519414999032823440924453088711434009253e757" }},
      {{ "1.64634850170019781216979026794433593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.58052312996124176558421225990969105623662471771240234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01546348252503362826107105198333600916123885616351929759095090370675683807853397698537671019348523544289454424258903395372242729481623263158900841743559291226322042608326721672299094222138476822715326186152957264351608036142453170432188932360709929281850323115131827188070509946087921486355921396837" }},
      {{ "1.29984771123733180422585298252613483782624825835227966308593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.48673306853607149757845462545446935109794139862060546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99382895797564405052408771210317692764125583490813721738650031785719409851796983570805688722349822979681595897622086712994137349075757656980372150545265412692073900115244181782170626388308672797472118433651575906690160786295103594530704832482562910548180219532761382129979082146445301288761331520852e-1" }},
      {{ "3.34499410762214122172508723451755940914154052734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.70362841707174993644002825021743774414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.40485852822901975442109716898610918193900542520961776863305069906497895793489542547982159745108266280524490077140951661552295048439266934614559790866489193897427949324691009938697269495988925070011169411841665487299534902136008055662912091113639828368592572760704050198717640286295766255031666140310e-272" }},
      {{ "4.58769266685160914676089305430650711059570312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.94008884841098259987912655333275324665009975433349609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00265878013514519296809865295705868217393246867225559245606227172893223522667583594724300322356488286875380629025637162385262938162022625886814529504008804417980773476428819488610083450291142913844379562656183905757559036430981251862699509205398028210683848825112302095809741897956564300820095580866" }},
      {{ "1.83659034529005538161072763614356517791748046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "4.28717719818835121259326115250587463378906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.53726930767980134628351414040002163694004937538301878071492381306637265403593248974390687238745936146996356402451151949754817939176416474547668969917407094657482938942615901000965138448816198787619804034435155352704149778316444976717855180409986287616695020212770570197624985971189801426264668362739e113" }},
      {{ "8.69656197110674742973418505087579433165956288576126098632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.53470532471532833396210548926319461315870285034179687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.36245700894899388555856775460955394048641742342223378203581109445790321544636857502869379407622931929904915591813410010543421311751441003345169506628319334514393326009523930245673734273160199057057608984977895920935149203153332718973184503752228376603699406900321208889195409621350940486281346111426e-1" }},
      {{ "4.77866412699338716230101908877259120345115661621093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.16847487332184418740155251725809648633003234863281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.00939095440309291336847802488312895385004430053772751642212203497067924870860269568166321148643247526572505972984054880543549485653717067216843154655077419087852705718322510631592242331848706588637404925586365121042986696317386544833327481602868949385627978856713298735650783856235081147982296346628e-1" }},
      {{ "1.54963799196492800547275692224502563476562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.11204960957075971172969275357900187373161315917968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.16782897318171377990330689886402903355059628193684114132825684051875757607797689933491076299185969963004000061078683006351157287011440524301691431503915118755080894616168235485089522630543579349054155603272650786560006738889360398191602249883148592535791585897969899944427735543027279681332101510149" }},
      {{ "2.47000679471161216497421264648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.52499082730789443929086957041363348253071308135986328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01911645804528060285295007365522901933244353571254108413776362642912835636609229733378877024828901273387523484327765744744502840827542422973162779826478327376782490303695679704814420712110588844007646988152318154340574171527693080085147287526207828218450010980129801914952299876654249129807403026665" }},
      {{ "3.03723549186705435616595583070420616422779858112335205078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "4.51110323265247643576003611087799072265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.62158625261315621796784260963880908165199570383258880752909601743905724906716730454255511124675934144867399204404978430279501754463613692043192608111413253272638040513583589571469911855611383025946828382761211165356334768347578605666178294674917101334371471550226399719868797818976469869780054171828e-204" }},
      {{ "1.79856395286417303225334762828424572944641113281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.32059115576763366967701779364219305534788873046636581420898437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000077517217823836935248077849892691299847903424265238524662417641027601052147271170002075034613200815926835872160958521395863660559755851166354980024967551112970300127247095245969854003499125350435566412954660499861774431261483707515396163088724936940160199016889915664585852452649068941190636190" }},
      {{ "4.41355937354146021789347287267446517944335937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.42641018144159374969603959470987319946289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.74548555036519520791276201322845305843766995125882533366758793048355043544777974793130718057932582612619627280143295972890756755586060864163343519807569602024123450673829021330095007485505005460893008699595178496760510808546265795771516348135206769564722897763853591915237047669941031110007831969575e-13" }},
      {{ "5.69171991829923172190319746732711791992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.98043102542145899036540868110023438930511474609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.35629700595091638395832213480888201727086153567756638562745744247364577078732772888391063571800696941049290536498639573875131887930844892410621248852158649248884548273739508794634953342193256645662324249539648039244590505104994015484168576899587920875428234320711411322714373060714140422162212909132e1" }},
      {{ "5.76954356011381364402468285845770878950133919715881347656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.25111872914564514758239965885877609252929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.94463304445870155806025859667737584217029774659090091223865132706141204011793568374081383686497430278169952431756767677964164234313821914580067310902314296078439102264644178954457549643252195734243656729528702645014237867821529512634908260975861471735012585206572278726071364924835281322609605715766e-3" }},
      {{ "1.05123768089208714283672857492391017331101465970277786254882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.80326412933219444312271662056446075439453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "9.01096635654174895730364890004274506998670967372315294988820585742497849923309225303695080415166238118944235696360277258545630995713146996451284285658251097270549397299546887024456736142173128664145649440632706184379221313274424739352233838868194748496986549948050292922466160812395403732938083994855e-1079" }},
      {{ "1.28163238607435392424349629436619579792022705078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.16524102155253899582021404057741165161132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.07879719574926016494490747415323778012199705593394831943581938296777477488312425323506097154699459156568032482136467971947677672762383992541647262353219851774496336332221673599562229050781431336707003943210680119687255794953033119424826346843688505841414943672869216307940977860809662475253945553141e-104" }},
      {{ "7.87202251453049029805697500705718994140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.39760572205021471603458849131129682064056396484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.53207555977032989055018208938771579303277505887799197079806149426805566503858217166434829573392590992745607709024458374528640995879305116834709101892580164791296787623750765678369128997876794757719509455355831607218534572758406336319858307548393197701741434159035106328532033366541994641153056109739" }},
      {{ "1.85413190872573295564507134258747100830078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.69406846692120716113549860892817378044128417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.95656735368615067592522859512955039643574891198504549379998394284320814651945584264960315505975008311532240586619468218955713277820799989277348072854744621874725681637220043518336051308494199906438625737004480910828551371454866506648373987268921256736551339157313351536645840751017335445520822528269e1" }},
      {{ "3.99959450842199415454558675264706835150718688964843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.86709578350118121647938096430152654647827148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.05340685717906270754605609530501932449443307474519553599524954021671059012176997891604734383935653924096296837478833892286774229478680952581431502458132810484806788952807655375243507984165689501659897282979828575230799553474262829100370309746639962529485962944150334131789310986002849666502428526895e-1" }},
      {{ "1.84674139572777286713578526189394324319437146186828613281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "2.25571659299317051061706251857685856521129608154296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.23722070622293990830799081708429380180065314114356604329745073190591014971022286949735867992102290979570786194665670044964971913376468986554402193499702945966888241927569660743521069322430532256408138686743620372490125687831732257527068587575688326301178711197108267406944507908002982504230542257301e-1" }},
      {{ "3.58473911190903163515031337738037109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "7.60245302222975851691444404423236846923828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.78031144785764416938612223102409855040299356306284612439958299013225825622202999580381834230141933121841779061836895061770649393695477680433223418771879979265527918578121214616513115300315506936777567650603403724811526574832706041635294702581230395655527148395841553708668562384577527922279281988840e346" }},
      {{ "3.71269374692093090395417220861418172717094421386718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.32704198530596029974130967143963744092616252601146697998046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.99998245586875139489901271639028340713779059130027015276766418411283294006537627833255021572845938748168207820412482122290518617722205438714506223415635135045747382101925106285222804464701875561904552995223695133038388901912278020591663993121292169965721308487953050363811103359158918910384771899011e-1" }},
      {{ "1.70108553436059684038639261416392400860786437988281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.16607504683781455865482223899221025931183248758316040039062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99978954129792622644650032269094574319631975777438486754903659150962464082118122945419410503323172656817200054190506282421527117453082829189458035945776425795407971906137471812197621061779135748565507246203908206485915975691199432826508885650310574999400116127525787480206412253878273855862615753183e-1" }},
      {{ "2.09225836667166789993643760681152343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "6.85669374483343502246768252916808705776929855346679687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00843563284699754528712961596450017035453860350588570410059871012011253379813933654529429045301434484839481489237559065322374683637748571974104032389976926103159187339564349254244671809494052575105796625285810628759847430784136996879048594191371809811836063684074535025913318716848524590469577859424" }},
      {{ "1.21714727542868383913798879802925512194633483886718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.04024274981674075615800667549137870082631707191467285156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99851738180285989619013743122680620229360476738383672375457223596142024538032582334141363701685216531955627726353087504478019029760906182307020379318817894422274668875967247480704625723584657033632048027924110460864147683431850155444025526123837240535234902995892523890236573378473187948012001689440e-1" }},
      {{ "5.07736820771065972511537633948819348006509244441986083984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.97089812664133678943301219987915828824043273925781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.61496103531524292145269969904333508188364139708137329208759077302004651060803860997069231846146525416244529356233531058823752839075298907756560474430217361656059248345442657322781131236619023757967452575027788503647455091630937116588678330308533380860226144011143233073363114955647590755520397783133e-1" }},
      {{ "1.05107095024027863416904438054189085960388183593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.03411347779367872590228216722607612609863281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.75435618440696236423847758785039217191601787351549036022831398969378880644344551767328463505949691251600586393797511528994909897140379260738080763330267660481025359227717949542759648019166102411005065795694130304853332521480946507674204615568889856249883941530835820527992056685661760535222848429821" }},
      {{ "6.04991801259759207250177315273731437628157436847686767578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "8.18837713372669895761646330356597900390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.65759593884760895858132938458504295554865358684199156580525629386190771627069601339728387260476959873950047671147626130674714736942807728977752816985626622779230687686049014608462755984076183185977629942323400832151270339085974963175219245006492664764899739861651980200882059489983917901390447959952e-3455" }},
      {{ "6.51744748140962656179908663034439086914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.07710216336688000492681493991398156140348874032497406005859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000328979143507415685697811442096253363291297955720636385496027164054688294193762566482130745852122932064218776073663482596744861678935376393042841878114830402198126626873661961854660156517253426476105318520395463096676472118157780746408448524800033381676594338617857965937768629354079942055143731" }},
      {{ "4.63884491638627549647821979306172579526901245117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.58892435788971937427049851976335048675537109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.33715260820791317269987064747412391800309849955602033217225409481400578667858265909262657939940738177907804353137088637745857746470695800199690029742037073210058760217313184489359114691758908649867287570270285365266943814797806817843515342646282514312200010453327035336204860831498248439822619093280e-38" }},
      {{ "1.49058285795790606154814783579354298126418143510818481445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.70317518258870839037433242424413037952035665512084960937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98108922622050827087546461298822670842973092479525322691759952475540660272868395330655428828449033553950251437644999243615808527145214868010990354809843842150796169597464947831412602935689456012651060606674408759590837715937317107067113406559618224088301687347090716482594166112826991011166298362788e-1" }},
      {{ "7.50315265219419380571252986555919051170349121093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.06222517749687384025492065120488405227661132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.40175419689520964937445314051307565108961445693694723005043311860973648366068520257018285775532813235354539177271941463790001972938524250513944002301467776271129144936307452041012641877576914077850978415809262774052418002232753061586235460151777159128199018488311311211799819289180487350061980646390e-1" }},
      {{ "4.26847036788872134273342684451790773891843855381011962890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.25843044022171965845742491740821833445806987583637237548828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99967215285852515192073395540137950482554540502668958238310575910483321879481952698928091493628164541705483350832773369892703138436296385686888980010809861693571807457349075609182274750398269557656870778473684116326560067055000556274923680535552295628380182973673412404308833575096574012516864815095e-1" }},
      {{ "2.62667296496937296979012899100780487060546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.98943432793326180699211569447015790501609444618225097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000067498249093796554190906335150484537007905333805853084798522029363004660855846859261868482180160316566139837296027351379815963628562093199672797915615284584800543052123602981811247334355647141931097257473080865200634384690696676990980451941954802180365065774462909125574737179287520936788897430" }},
      {{ "6.70966669446914210084287333302199840545654296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.16310384563290541287683410587305843364447355270385742187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001744257181843369125556914164986065767377720999640888705793438430385027106235014113094055738186706162642114104415940235112867181288621050157885040444050793876376258047873939583048138039901310315757778797936384132441774545886210926930901727852204521859059543099653693673834002390277355781829097039" }},
      {{ "7.41203371899830177426338195800781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.08045243497875006967630373555522282913443632423877716064453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001460353450390109753320201848917579634289514937879484407258848004890744776169944597998175250117503923620359242409184317540187082543653993640048666001395047165769964336795615132413269257911431079896523695358479509374941248440346987424514128254145411471716437568325046246372300034330801593929796699" }},
      {{ "1.69159255637993624077353160828351974487304687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.18026386927718329733494329047971405088901519775390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01763302348245649415440222014057755125288511751974439494118531238495212343084716356590900135150471489029857630623786718849006602487868725615880115082270216521029606377055101314827010471252021762724425333126829904634616800319752383603387644619737656819562990847546393907303039685961266007323499371276" }},
      {{ "4.71460529249768889314964326331391930580139160156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.65188827816496541345259174704551696777343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.57039862156640741370099717884062397571736028728524050580037613318788846117662035095375103738860527021001943141141148597120492387251375648252287053831107964249223144395667696166786054472499439767489130687268705183565392558321079668035341534647161063188715235627145576677691259556611945549071489064974e-120" }},
      {{ "6.64982363517832908428317750804126262664794921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.04124859899293669557307495665554597508162260055541992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99916721594674747376600569188805403610749202879145587384797447934876298553391460440217614450811626491475153827239769211949770987067539929764774811195211928780393852025574440532735397207179531706592529705918782911567387089221968997852275611755107175151890770337885129722749288992344581262036121822096e-1" }},
      {{ "1.16893441744004085194319486618041992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "8.15678606859390242789231706410646438598632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.51388885705521092082676462955313303770964042418254877907252507964234420916080103781270109330079123766686738994000772870583969928581773756873588657991143050640555136731739608625215812170605441235916882843841833089297543893622016574335981753027769495373907938555110085958780955474425308783814139862813e33" }},
      {{ "9.76400625826371292015348757331594242714345455169677734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.86442986632989526896153620327822864055633544921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.74623224798347601232565592691979711045580692330641567994756260200814740060669606546590062265074708542429821154943403107394779336685935160305728814542562226163058396994725320974264154815235226634053969860486205541938719842224052325659161610602552591765080618174794519808159266445853952938084827117419e-1" }},
      {{ "1.87868482843651634084380930289626121520996093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "7.16929872732812327740248292684555053710937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.12028119371494031430335044645564708556145176065304647173808454044697019950167941050283479818252925468559303066000892193796628006878431527857608384081735638513924751307741786259324802053021768713537078672257166368405154064404708746524318561350343547073960189134086007775921344634528456932868446419630e91" }},
      {{ "6.10956934891920536756515502929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "8.61245994668030334651120938360691070556640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.65700792854128829481158381754556960783201446467603874396477753649899987199726050381764154587841579338454205142269277734568001743334867145801385330344031718733980559420806485323497734470145803167224651985614888601581616073332671350983075972830731246525865989689804085186895425350117377876021236432000e41" }},
      {{ "5.12940713132187653400251292623579502105712890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.28564435019894255014284711879923861488350667059421539306640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000373701282973747520930655159796229096921541589463952802693848825441766023480118817224366828254246104620872707491850843628745742807472462411859435943238658855518813616022964453910564185237673973654468864102738810104394810487407665604967406614750596030541491730666890517480494969397423025082726828" }},
      {{ "5.96919527975153832953774529768153911390982102602720260620117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.57180582106150705794678917870399459388863760977983474731445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99977473944483761872828387581066184030968843746303338716635967128051482967926885604187120298468309076374477372898167140591203610593108721399453839682426096478159671434202750765691646383078954275710747248498138805135910075991839124150638583040898450493059066421550145802666193053679045930003027645237e-1" }},
      {{ "3.68606214248608399230988652561791241168975830078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.49260037966346908013329231046384393266635015606880187988281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99985103565493113484475922183999881089207789606742350698567516671823393925981239943259423556444582299741676725651226703123182719333272364174176377120721866858898501279006570635020422790273788959353152995392898746327367860857986758594342210926743928260938298323338805513021194833182527623776519608700e-1" }},
      {{ "2.75108139709647048221086151897907257080078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "4.50370430827189238698338158428668975830078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.22207684654158090072166193557643641149481975652647701708530399277363664695064351610680468504557225898194931419933867113331640493326162327026706800979963281990313857552256767561246682453632232432467037080629483164984556210425989798928121976072588554960434306958147259366170108308967342505596481568498e19" }},
      {{ "1.81162455673292015490005724132061004638671875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.55260000441009051353087599967039977855165489017963409423828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000739442221122260724005033635770725844741914207974197964181739354726511926916714698090584379202017604302885037125163720681613028683344989131816590462063741786918631523895090976438946284620031268598318287495015673164204411596879025777677083658398098322196424671094954029116891354161564494388093193" }},
      {{ "8.96493500879970817152297968277707695960998535156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.56875157541190674237441271543502807617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.18186472758714615747935532618938695169587522725634994323764763361322541875338494379271005378981383706775962465651100098009942121578720737172235160080858098630748438403135919432305006360154000115695658966205609995229485314131439314904570801339124033312271333615588491334684665719513845178970476180503e-41" }},
      {{ "1.26612879808979481716102455379768798593431711196899414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.90879340262861951771355961682274937629699707031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.56902896481771879143877379682638527882901861190229819789042792546979238367054331337767498254741605069110970147845996240165269976456347023401814558426675219095881612725781553874494895757386564444732770552998856125193687086542254594118439921159929235553297916239134941949770234940282427408194492643353e-1" }},
      {{ "1.05179746109396965946490565269044736851355992257595062255859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "6.59374100718207543231352829593561182264238595962524414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99092782662380324729163640655313848714756101648128347661760755472309885143398472302956067855710190800993042892779094559399021527010697359107101541167758299014549008387517426503427862359734988159027154498990787998696863714462719474369182220777439814743092111459144669908819231194636468040015004769487e-1" }},
      {{ "2.26677931802150922635519236791878938674926757812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.01284500605095928449372877366840839385986328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.04799524981909743425238268974274974124170089470250115095198088233403741098790024678291238778471336000887693628348108623078030852501843027636021269414992355439834109447548416843301419510567977485938290387496561912410461657784176868621026947292962832450377901755820567805673466030070278228671530695195e1" }},
      {{ "1.47645383833588006439185846829786896705627441406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "8.34770445962486519420053809881210327148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.81864953170833366001029957686457080479745389229343183507996187270256012032168347753440691289295163599099444007484206082939466510290612710007371495952373847856242772883552123786185299979765598455416935895627334117098932505280856391911876837712397381549095944110724452719067485663398985307194410072346e141" }},
      {{ "6.83690460562575097469562024343758821487426757812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.83530767198103986204138493576465407386422157287597656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.98542687144721504190991783229244229904437024428948393594123834827455412276977464755147963505927618995184986289919531376928027005710396398910486083058794955925781790662773364003988254537015097017811287414007455159323257827799204950099119284414629154584347927772687791013118617040392415483793228615528e-1" }},
      {{ "1.00205353905587213336758622972411103546619415283203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.21472319667746432969579473137855529785156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "9.08953646324319429602818744612054042909432561532126770140572825180403651664408870454180981154606920328617475099247353850589948028434533205453640961164641456021776839621300753911422835934782850338468313312210488559936609343534193099022841479411388023651172446221865345603921529386692401675022261632417e-1265" }},
      {{ "5.15232802831238245744316373020410537719726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.60510257181055615571807493324740789830684661865234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01431289853951002532506377614168990948051280337886607436878494527645645147723859636940385995419356184615086203041458176935646525562258868511947236909026624331932959131571326296466151873473074426347666272446566573176007728397599624815497760954111152173140369195400219514948443921717606923360226206339" }},
      {{ "1.50327863210486480966210365295410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "3.25928705204024481645319610834121704101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.21705224525603686559675227491858310594380597690754424568785114801297385692314088399732454361300955075549296567427255869777962816305497530312793581969492733775461667703068826501146498955373099052386164214730524062952781381010426318671419140310052915928431939434595652675036589780009141539593357125642e1687" }},
      {{ "2.52103950481096059522273522546242929820436984300613403320312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.04572569600166157249149234687024545564781874418258666992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99677562816744661723525153627124442052964231793875730324917063167910023632186323828596851920875156111103787347156093646859907405739307639598484356543171131155872282222202848608366838457688692164646235280817749009555765844448651859985081286265963272205351701235959915053427147246220456364075290757788e-1" }},
      {{ "6.69572868673203203115917858667671680450439453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.16763271218019875231225457312689286482054740190505981445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001172763619213033390863469162514923787137097731257478379858496528955583730797624990721270466292542630978166836841730894536953419182253636303821927335171827673847960943549331513055995583023583227615832251837950408037083523512010882821232754743304332778547391577736073428707081283889265580574098969" }},
      {{ "3.90611154895943855080986395478248596191406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.09392819734250876240189143118186620995402336120605468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01257436238989364436740053785780748984877719711041249303794530677616720891994615335093220833940115128861191606642103642393403535957041953141054150817528925853538686389534782202437853737346681413564667163289005149907512827692491679645604032151505662903966426036149340526908388743360309704155643902314" }},
      {{ "1.85738968908478855155408382415771484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "7.34900449082321379629112456655093410518020391464233398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00007223974053029723271350039259412141636069806425386754509098025549121050837680692710943206684686658573776447604596802766176331791002163659401284496154417281082333498905606330772551902793976098371405853098920109852460783688567109865594724989116482969799710457089092798143235841330467446343910216945" }},
      {{ "1.76070615176634455565363168716430664062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "4.74889370868937987779645482078194618225097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.45316765019878240189602403680427223818770335223288608146347951094991305163036799974448576646914163732716464031107846083674367407256925355345653200850687798959392811940414372053909694215614805160729642095127896071955737496902022738055554549005006428357198315347861922794020063710653902271293794797134e20" }},
      {{ "8.59257176966675461152189718916361016454175114631652832031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.23640678060561270967809832654893398284912109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.75626791055079856222412335079669380796397627500002019826921615591957724636638189051537678939438100683331885757581588353701345439775420172078880839292692181866988352748585676000159933320197094086021294353349620582341779917757035411891306338495037492064154895428084541796948375617939605049956389592963e-4" }},
      {{ "1.18373366744053128780857914170709932477620895951986312866210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.67326277601629289126013849786644982486905064433813095092773437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99977165516365318634154013098062676051482936340600676508115018286371174310194281664961388300484701624347701999911368618575116158865905431420415090042447533719781066713262060337264603666364847020671540664275635449461141562032657119830142469510330575908700332922190579477083626027776803818572001718803e-1" }},
      {{ "2.79297046837543444519624813437985721975564956665039062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.97135089362066378421145529387104033958166837692260742187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98254178388410952430693242810343707992963411946313883778028696826023938537768767830025754426281554000607392763511414140886618834915773502093937010404359418681811029932841236322654028803608283276794373781000755359892344545988755782370138251235941283542297935477464591913844965660089628462450619495872e-1" }},
      {{ "3.38113855762533148663351312279701232910156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.09301666014966266347874235265180686837993562221527099609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00010890478594524709067739908022368917088132812449962876399165997211946438304504899291487317819472340328330375585022333633583894340550549675803507677712058170618442005704437605073252304959372965881786681685588681179433355904579852317919540925268445449423495866437480095339165546614783853750408452453" }},
      {{ "2.06425930083115716229258396197110414505004882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.22320068732874709382940636714920401573181152343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.56994381329744106645164401249639056164129850194015191524774235721349310749300345652229762055800940290782810800288250321003967654939060262541583665184611869025848444849240326580936044484294913651408991979990620894378054306407392449503735235963890733618686812461995281192108663576402590588669671991241" }},
      {{ "1.34087695405307183449394869967363774776458740234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.48273418616638036349186791085230652242898941040039062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.97025233069613645577452155667707700431529405119046003459330154763708616924139333978711947517286910376855122315920165412886432318792826535847675162918439845483335638190760733366507881163660357422897573743052858674336878106847554903297760383077263299654131223242076136517464708165444087161451001347315e-1" }},
      {{ "1.13795829837500036774144973605871200561523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.03500201236379312419617904073021463773329742252826690673828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000963457189792748049744783509653303667431326364869818483704842948351380610875652979437988682746278703117572921123081696661945181740330111308085604188704279357002138744426600265673193487995475006875655840336689027158444662492494587307695412667171074545903613666004574014189607190377465209019809696" }},
      {{ "1.43733943515224175686145757779854648106265813112258911132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.32214836393443135875713778659701347351074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.69028929542482647910462333162857910188985784876407542387662931701295222184216133407420608557785163630976401541135533394749225188544894532723554319205860877128881077879568220494222775086397987960274421371074192662130785663013334009285908262792624635778476059436229937151198221859566242280319019160525e-12" }},
      {{ "1.40044272789335991546977311372756958007812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.53744104780148727051880541694117709994316101074218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01763558978681810389405573088195138451898513946030420888756116873548356901540068272802595421797514887115185086583712361722645481445016465906573909064890341678510048829617311165144452645678278211232399208153808746476971992206798944801345917584439323122735213774632862994806449009371980892407850308879" }},
      {{ "3.33770615636067580211943095491733402013778686523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.16340691059281198249664157629013061523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.74932879103617263287699269441230578657539752277509100498121992886363492927534654879230172186557771810338441534937330485904491247273189306610489852019997049552119104082607943340924820476508529648819831797793990936006114678588935819382061714015089476930073162215255275726210879447826384500950556498380e-77" }},
      {{ "4.28996264268685627030208706855773925781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "5.55843482735151467322154417161073070019483566284179687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.04758858160148749573207693865097611258682532053580644741708378607626299571741454781489071163331961478583318303704887429698138745438016282037356465726871543404600493318018423949357845164509841052287336032095968547544265578409278441961772371247405527391823186341452132702680834735192488255543508774968" }},
      {{ "1.28110344596063102651228859940601978451013565063476562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.43418931353877687762121695413952693343162536621093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.61015280162581259611603041281312219714750672075712379519625520748837900282815093599469657102449329452585522609992906027038154076071639645146113938200595344162570525215006263974571317023675884323716943933613404725430584249433688012398115002318410911938391274174242791011414117839535968615208469771466e-1" }},
      {{ "5.16672517946984571901580807207210455089807510375976562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.80381185032818629470231092115639626172196585685014724731445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99990502044552423124036840709042864832582112050272720529614683113869481837430803897286965523406403026476324601866776220157395635334324411401951420003880457537783083750584239021034205680737146324172685990190142433894632845319189846259349630351570776916287281796804645673831673367339618359511691534945e-1" }},
      {{ "1.27899907316783210262656211853027343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "5.22536024090327483326845481315103825181722640991210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00616341899333551359328361392752162344577264467244472586370749414301840554961173829121346442368844768868155278219296270614015234651325697529392935793355916715136924941286676982861345063267744784667777743656327428634674404454023683239192093835032100591051537658643852513812450455406749057692109035777" }},
      {{ "7.11730850188353797420859336853027343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "9.60788079495505627572460305430013249861076474189758300781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00107405241322382059953913596876333435454811871669131165966566428674069591664017854016649566117750922070455790150785398980453750449040916943909541516125417685189264145037334413362116738563574347465747605135946915752491452923545893529316360034715492483804067411471432361687478500733144787639791290510" }},
      {{ "7.25343826567665018956176936626434326171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.29954091776851328177144750952720642089843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "7.04970290589109038711395049954355776776342381310886782453562200726034000844667284742622588171082170563883520775218082400856123793786048177341878726458251943347876583569178437736653743701492064380769184875657546923031668880530296652280260428229230807500261114316496980898463695380987245993698126306466e943" }},
      {{ "1.05425305597257192857796326279640197753906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "9.41211561689230674643269480839080642908811569213867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00221940552517387456904940935153668768677803438018028587024892341085297549561710698873774378670329922261139925263482590967131460066291169951663366614190003024630959832086281000186573153943388176622129526851339850140935108594752581778544126368342198113455082270613501744317697800189856112116044092035" }},
      {{ "2.06616686835977376654471804329205042449757456779479980468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.69516133890853029630996218202199088409543037414550781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.85720096393058346372084885893839606320554663169456270644249004310463500725457211657627998908535169742871675934174757999843761965159654941281377745862394591058408452525449223020548106433276286869407305925521861569746775217097838170535894284575048909646530082294345563168457886536304399266956794807072e-1" }},
      {{ "3.64402451110348510332492821817140793427824974060058593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "3.20455824353290882888778412507235771045088768005371093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.74947852438644348963594070779458486239729374924703758041706549021145269914102751629698949629550481353884433142728878758835488685283460683679637491539837448325034596988851330419929610091798061933767303224300754233414005420110971299300889653323675957445130038468323617573447274322484402694978713617040e-1" }},
      {{ "9.29240918436359104014016924155683341268741060048341751098632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.10041539819032330171921785222366452217102050781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.30375096076042540689175588676000039497925847630595558133397707131285132566953633275113891277016515974090275087097707081223643727147456085023753621384884096242056199349318008502416132788061805089253485591443085259511013177192516510953942353610198267062490629651968866334656113567343023162640866499116e-7" }},
      {{ "4.70374294317859518955657338601383798959432169795036315917968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.29316103610866321105277165770530700683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.15122965827666394521150958097124834553922749706241413675626621322918765539817839286128014659593143570942315119838489669350005299860384297452971974330427527438060831168176929743817135707477251203472467806027668891154074309800997369457141925072379579611854152032663703570089179850821080592817602489379e-689" }},
      {{ "7.90111168678090834346927628928369813365861773490905761718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "2.50142935632569063386654306668788194656372070312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.72554973148662694166759157160166169303284736393781157514352684466477860400705537599392645875431865905681445772161441577785663278326674184262006684080907675404076284363331862467589217294540329968481020258307213288969704999028809892457240851077890832023610976308448700744934847296072460694151132193050e-13" }},
      {{ "2.19470224538514946388545467925723642110824584960937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.55402946608165493103115295525640249252319335937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.15537644035230762511257501642891052274838428807882763659106583075479589103986905326616108567723817554261320425463998387402938492727716723367633134645452074625478383896044161060182209094299460842378984988424538062120485203568234956869861031978112571715328186447068917826940102237728367749115256994036e-2" }},
      {{ "9.59375752767912236013830851177885961078573018312454223632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "4.70600447818681459466461092233657836914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.34074971005352694977724271428304704379730503069979006543824421982543154474277701862893673330086806692863058917258547615336829675716810145864841475235135109391399496144355613620559781473891279309689200469472005874869702401500915898827498117462493418903426151104912981808366378491461635507492430973906e-2833" }},
      {{ "2.20730165021212201281741727143526077270507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.06815138837059212754271309009368451370391994714736938476562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00003305295163452281794976529673484400203041012190269638476184821369788032078572490628055955783906515975070132315405678121749687518303529356301281556543290455691541229667405975677836660331331546866945589821129205572181796808319706436826468956587053252695087454167269382499127173288396374118362076812" }},
      {{ "1.85780789842078593210317194461822509765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.30823413709633314283564686775207519531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.59033859659617126254628613087390514142443448071790104857905386498498208916927300817260750485505223182216712213534955208892328370772179606978553650153083374335774046910050899642665243305280141884413167811281876091816572586748238812963621916941228650345907682195313189327181720960131248296641667634950e427" }},
      {{ "1.28002693604745983830747735332522552198497578501701354980468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "8.28203848010336862728308915393427014350891113281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.86706798756547273034023259748332998697114154360428093064398895642136945343512222443801509462788398508116083101485126653725181025691107909977090006260480711914768000641955268595050193784354949171562687163857371073429271907026305099383798868693635018067191669439164014101281294813574568021616736489720e-5" }},
      {{ "2.00223379702332143770249217595846857875585556030273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.62365481395694466471013583941385149955749511718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.05235460797399961706826012566932005845504749468248264540170967631534913675307979194267567883423064423939335134422258880838785383291061227375243059330869736175934549938928557322841085452000578723759590054565129213090997527802918663825422738502966602188020267614510519712792292721325579607440137453800e-1" }},
      {{ "1.46975278087147759168741567581406570752733387053012847900390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "5.86458989513311337860024252677249023690819740295410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.92154549068701501199366439766982295808371482320630816324433379463381296202639580495789257638722760912167367897485745357403740992265341544963450873213458100774396760614353635868165234585261876587876465994053070011062897736386668244554072898747895651881702386350638799402464850863950102149208151366166e-1" }},
      {{ "1.69869769328567166155607992550358176231384277343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.09451664353347934838245336663931084331125020980834960937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00005799594795372584357977768556944656253178249587480331145989344864034282666379944361611058467140259982713730781408566683105999320504528967383185252941819780496432948156779669264240460517640856647134097150209519411486934323027426354907715934579141058953345456267906766103552413735697517517849566921" }},
      {{ "3.15183975929332882515154778957366943359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.02384029084149653920920997052235179580748081207275390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00828189958217090945616829239889322422890752450091752768664294850744665096849748731214361917701488824218313674215416049035608435896320044761285630475680013499593324921963112908846599652343142683769107300940314664453753090942709855630290654440762603604322610449245678270429752911593206005276375357381" }},
      {{ "3.55087476619042874403930909465998411178588867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.98956931787756730045657604932785034179687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "9.32018908151418417859876732852718049962350652342105850490583184901223805273843023839695354873994074279947349253543933474250055215956442663452932535132575000870445343325403706967004656960749057879238157659107219147411710899538754744145348815414311088777452109665676822468399972746650501593917415911950e43" }},
      {{ "1.26775488796815989189781248569488525390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.00502285798295979898919760842090909136459231376647949218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00071834700218288240776940900136375146363527699090847426981092022769405513480212911323057036382071349683106853880233005923304991341200308955960878969600681793506999384869035120625501911973134511693383019876699301926591009688967553141397068289179934328294897849913598948019612426308123680588484240861" }},
      {{ "8.15220048105761634360533207654953002929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.75025311305504219921980180174614361021667718887329101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00038554010810039170759525901230037332621377704113376782308969416671995519944631372963790226541075839243378526551356024850278824940678165140037685990966568856425831851096416371162933528342527414028064433797747326021549595783932074419350022816349807211737168691836224208314746714465025589975731455428" }},
      {{ "2.65027968976974648285249713808298110961914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.00316549124626845923557993955910205841064453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.76325996094662828651112161237788953415887952374670102134300653855113287125254830377423982732034899162480399742930099350206403828614662703257591959653574878666923823464038672614887656376458325931158457664475267311806542914278433869759036943522883194636568041344283954333960707999807543121267715345128e4" }},
      {{ "3.65182751295248817768879234790802001953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.67556451265639907433069311082363128662109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.92218138915478317609234268428449378609898504811886050861657038624663939529613919692032879990877291035109701590929770282532279774612429349026418707716959296859399988519535881353467118924067126621065164445440389423581971051504964356883522778929468170495170856030672143818408689238481847643503940756672e59" }},
      {{ "3.06064234847405454518348699721741468238178640604019165039062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "2.20328027949478477687250688177300617098808288574218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.55974238261708442675674596584944622380379860666888634356645875331625278958028151760326676909826129760886331850336797218123658397234427228603106790759796963957806080469839015421053939928057033567772138680185158581228274260396763090865346625773393191666291620910582819221973998350912562080963747333068e-1" }},
      {{ "2.04040793254407122731208801269531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.86284726793166655056666058953851461410522460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.58821325525680163465105847555152044190569200388116750112228338962598195492329497459385020613020256287552717930952139248724578086838832538138376726496296560875204247514233017557960719338882649967806582509446295835645176351140001109236587197833843059204085625918013701386357232953416141714287393580849e15" }},
      {{ "3.02716069476693064643768593668937683105468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.04039959726672404105585201250505633652210235595703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.03510986044313446578771317219744132510998690381352607017246273580247626695630046904160285453167383058665284858123740052884532347981837364373948718495188424769656549948930405672217230120189497151355681193015659186537151773454794215525893522934864014682466868462268390720016762737891468184834184088104" }},
      {{ "4.47552570179789768189948517829179763793945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.42494522046174552087904885411262512207031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.11929646387532568976168597828914488086907276576025509760458985730962224619833220482536328707584889009590311478709195034015122056954215000823555978179815671349373291996702460104546989380293121918158085134301955237814810326108959984625471199876172816491246653847194071812034535823558646590118729075911e73" }},
      {{ "7.89920944916936605295632034540176391601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.04372395393338365465751849114894866943359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.67759107384668464385278885844815795751161179367906982405379998416580527424480738041217566954239691080320039512890036456504748798096287871747642118148076943027438116523332662829876900574449886819108021138850402281937549768122792675121441618853423616832699349260369305453263152392638804725787763327554e302" }},
      {{ "2.54801173789754968612708552200274425558745861053466796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "5.15927661277336413458982633528648875653743743896484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.58205367007949169809279999461100360912902896399937272021332641770690450812544971532999275755132753113048328902348279538087441845456071781043743985319802898282971020600433571724744632298362925010469603804610543751544460798372376992115798540860563907814359172789945116001784572399034814979737998689779e-1" }},
      {{ "1.89928750500089372508227825164794921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "5.41078495521190028227920265635475516319274902343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.06563424507714690167714042623337538555278963530996591844954713364193688715643758076458567982650781640747670966188519005031306885869355129425908043628325745338231076914378682618756362764508582923824741998110533664940173863055464267380496758389944654061554957122449256611900453680459384297799878623896e2" }},
      {{ "2.91020477415979972324322488930192776024341583251953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.10728022839575198851824744394889421528205275535583496093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99890103192413085441074667861315652249320196657611339570482857071941674209639709864703891665811884603908304481549592842627180817065779143546842079140388383757073035382383336415119892457712538117493661547669850819233137331852231641783994865993482070326235283184181541427573679758103030801595943776744e-1" }},
      {{ "3.14872001537003598059527575969696044921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.43125111933481261039524667921796208247542381286621093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02802334347014804115203210525086340106221839607209085819487247743041255274944512485633547164774253058201682853956553615818500164452268935135547407521495805328007119673698391733155607980906430692803512122247717040507439060960157575458288088753689807084410223837216910150868125550872659514690258206478" }},
      {{ "7.57561563668409548074483828017378073127474635839462280273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "5.50776874093542589960748045996297150850296020507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.22360322691615293009344442522576915365414410103567546362459956848539698481031529793394942148327475172323264893580159766513287240943753652400314434832413344966940876720610075372213401767379175382193046077868122711719538872402153088637192089166340900530573304907112834730373371836607562222448960151481e-1" }},
      {{ "2.62745508969974561619409314516815356910228729248046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.54116524398034698606352321803569793701171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.65627927121210139869453772647172972010233396005753285235163657937204725700240182301143436917519389589272478600896823141333467649759723515447005854045233682059842082604402664004516169590843725236432591088329366055949159027684989904142732333954835416376354569871287255391194720780011944013876616447352e-244" }},
      {{ "4.80817610372865829049260355532169342041015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.83678291961692252698412630707025527954101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.41928564331678515981471338829811408659000181192686008443981749115702769340722053174798292070290561882896008338642826088125297911559295342634515669888076588557229523820060662923011009955171952757056636448724507715649589034168433634032429475749149003931613573665241664071010924406015748971161354696738e64" }},
      {{ "3.20537873291325013269670307636260986328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "9.12063633247133534681996835935535727912792935967445373535156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000736273879463173771441699203291296246037525214803383466819484064993717806192280443913585501602480713195981402804796752564583674811227808842648414027111557663920686825861938638104671353044668858964832448138218436863850713355695690765716750462185486284733337309942715764828383697583139810455212638" }},
      {{ "3.61912176346828890621809549088538915384560823440551757812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.52277102882139392207860861816470787744037806987762451171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99641674908429772727741716066692522684337586976236285526703123104244360091941899419901468194625324926731894302874406101537005325420303987745335631550541398216024526757166654981325659490102941309976706722307972204026369136295841549401634445497494285259254938938876164197916350437289297642860805084296e-1" }},
      {{ "5.47727150864758538517662600497715175151824951171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.54622621028518670346224572753612847009208053350448608398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99989699815221047195237797415460745692681672883768835967449812852073485700626675461871958759788593113792388910307095890907937872017129700062714402629139690850899327106043179965950904385626167111246937006767223675683130961170323257583625795510235931710927527056466356951973276658601609644129515473818e-1" }},
      {{ "3.15637438341709639644250273704528808593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "4.72407848923734441370925196679309010505676269531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.49825547286670319702380922050187415294618434375837006137892453203122331928047047306492172079078511814726958266726273319242885369558258257128888847286538356045014816657847217085941522774109065130033471139571270092091883178037417387674630454737070888538218688730956097543661382768897165490297639608102e1" }},
      {{ "3.43503294936064165085554122924804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.27250150145350989297488020213222625898197293281555175781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00162336779090582285668555617540990956375149416793870895822616882369079038963730293631724525621870732184904017056404166569755879690536542415147024291651047693012257723344966042105272046767388935297934745863474716190137901174215179659999117641456704254362366255369408392878120515423933846401645588741" }},
      {{ "1.05074245129533483122941106557846069335937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.44512744072382729528143674446738486949470825493335723876953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00001005416673874970641874885661095503816717227379033676876612730865335989893449582345456543451081149107465171659175604293715962189257810117721711365871006508720993220186972724366817623975260748578680835030711473739357905906032894783455531711583331359705157662673931729626022752940417931688341732378" }},
      {{ "8.18657910086106118275478538137690520670730620622634887695312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "5.69535439208307230074534999175739358179271221160888671875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.93351224746448996937113594504910401882994345207771800323049779691276470148213924723483955246436042747455629177564845224233210837232201105509371449087604601761437328836060853287056217115691403748052298316839138971392756481564106383080155540424716692745736592178985310629391299501759133150627938918074e-1" }},
      {{ "6.94796213109996792234301921098449383862316608428955078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.01032189649519636986951809376478195190429687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.23006490342279217571806140077665886981099414044595093418295957860943720284086984665743578056568297229139299500245761284073691802977046208070538242215508979634862568996291341587581554469156573033924350089621252836592701785348565193758908337598409016657851371246098654329684058361927838809573986193758e-127" }},
      {{ "1.10592794589664117665961384773254394531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.25260821850983816716507135424762964248657226562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "9.77248676688621920044634297401786442645385941311525110002800741928600473067163385046961591455772161080604895545524410628330763921136911631698147157554500689676133210576592147935248743333122279013651774018515939625427205237675732186795164784795174668611013924030393724917154265078516887725307445272803" }},
      {{ "1.25391642141006403698803850499388090611319057643413543701171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "7.33579604520485872853896580636501312255859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.68392795308248572652843135364969720382138943730339914636303299072150410868121378543408470390954010868594468316840003936924097765196715356527524435528061810352989069642830612407823386666008449194429503821925534277567874494596141278120854458833913072506701454712189004957264318701887957792196559450829e-5" }},
      {{ "1.43124739542791035162849766493309289216995239257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.20942048850905869983307638904079794883728027343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.99052723424150661886931649071103918452644460542298398165637531920915224385808227208964029010419121462212498905692527594903577506198009266794868206489817939747878141799591393368960481067737790483228614343187263432824490432996787531286395390254093241449602142598445964130129975312810018929887788198701e-1" }},
      {{ "3.43751575420593244447609038161317585036158561706542968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.51206687051705870317164226435124874114990234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.57663597564430277353448388417312985615626273703902768815220625590828356254147933705830400685411299593103143556639123246778758775852967467852603984265071153374722320323459885871194691907711626373620710017973655475923376123554952810313797438555025227910644691182176423627488188876492434022067149277036e-38" }},
      {{ "2.17472395590978127746101256079214181227143853902816772460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.91971825175046888034557923674583435058593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.50169709235325052632986882192448754882997090613648358141414144591089112598426944965520244801106015239482741658118850562377407270011579313225688643981938806077750575133252661340927273680933994486522267294226372481412223764992202845720067257372057887189374997685336348698935086839444322249450808054944e-1362" }},
      {{ "5.61856815780757930039405328992430099788180086761713027954101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.29876093753134198849341629511400242336094379425048828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.81481816351315594384727007893186507859436644672331091167617928065788693663352781510746621160037607384076966766971591864898782687655354341611390803359295084395315487119152362491516158659729010541073082819881189326119491997777097209681555255784910126963441436697354364542857530552913114970532656768045e-1" }},
      {{ "2.69252008622562698292313143610954284667968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.15330514334751887961516558789298869669437408447265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.06666279730568497198113724274678005155752498157046477782004178363192374164474721840040331312046115275560496984877594971350062701246211629447728806246316863840490103633689163217579393790965785864180257773316151663393300617519826348688668218517932804149442470238145344903593621930871763066878050548745" }},
      {{ "3.16092590727489834989683004096150398254394531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.48064997212848559193787423282628878951072692871093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.12772520348616005115304677929962056919100702599904589876944000234727659228749834981600423100421009690523247572724398560756474248595294433046382239886297029833469577515642633205214954504439619606667497183510745512915018311469959758235733226013623764492488909973197069967553629996698907871574676780407" }},
      {{ "1.11417522478442774288426297957244059944059699773788452148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.07515622779491204497048784105572849512100219726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "9.37919294955190238371406645550683331410621638917073656116684574199081255090910984115004225393137968138900076398279876623105976571541314203892497339063273916638860131717065142379547629145053297558185567155580289078804833133323155623898592227362901992618266649001082738287327625009340169715002810953003e-2" }},
      {{ "1.54818678360652516035056436294325976632535457611083984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "5.62060471994897832819643168633660934574436396360397338867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99950690212005897959136097299917160482075781519044088208212676575453006587694459400083447882136665068803314657182010357110987268971672741796887858729353234727108572424227132082469362733286450718291677866426969142840900462763239110268260519434037322501413834614893396154343681850902784948541773957467e-1" }},
      {{ "2.28067710108071286176367742370985070010647177696228027343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.22516467629786623624865704584863124182447791099548339843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99645746030268443121474912265456530387059240825775542503945117534294341852612112033263023994402497137577664903179947031417093395690539107266054577344864123860512144985801819409282408928976825661093887120497669159146596001826170808186934642139325551337232119779982282458342982953928951761806573163414e-1" }},
      {{ "2.20739083812885155566618777811527252197265625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.63635428348883159158696359858353730487579014152288436889648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000883141260854728883702175584015782702915499431300231183450474406119123947650012070542489465023521252692179702983050986429423066780728766911719282820886407479657520041375764406817735028802434658714254220988205387649994799335160542734586791578044041036401530550092162441750788372567864862832251789" }},
      {{ "2.11115639194152252336444952618421666556969285011291503906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.91611178791150109645968768745660781860351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.03851749220927707096105838017357917405034340382528308320810037826391303266735158295669708446462859180309945589089175400795292926848603356140832744578280150360981737396616217244331933441226205091493168502347016694353942358220609837333188158011389094568440922575728826317171755408132992139505746448351e-181" }},
      {{ "2.38633916347037029481725767254829406738281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.47350156205279176901967730373144149780273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.50492562190358454748346649100342667975057580840217621906828483604765156733043048058740688210262627421835218492853562454601684989325985386603026971417507963657922417960947826930994787046883004260577743222590669257927836671693436821127356903028204248621426323317746936948905014560414890048500518981483e58" }},
      {{ "4.16496490135696059015080194853908324148505926132202148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.07526112190653293509967625141143798828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.95708970459274514353705613356165057031575229424640773311770967490560575262481854947938662630918869775857057855329254545872812061129308665775656748897011265399787487814796202063476887242748791769330372489092700441649278984973352735989854637536188114328280137258357850310518955976767163293227505459553e-135" }},
      {{ "5.60807874972006062050389196826927218353375792503356933593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.70244506350627981296208535155756180756725370883941650390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99735500279497094553640363695299701240525058593455541261394700469821111611056791299453732067954961427582921898800842237144105286658409494042658058247679724238096009700396852983653186891371272750369926281385138909360300419604531575348226930671075101518323161673992501032516753200240811209353821343888e-1" }},
      {{ "5.00853676861336616008002486921668605646118521690368652343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "4.00645550345715264484169892966747283935546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.42868005311869594653291738037482789532744928595938052618021862026535850449639612018073258671047578104461913132613998217526422325806836712765473733283115693708797988431341752008408657974271055896785613419004497755473106128411938210880970972796033569804407750025332012993610194266901383355037197031359e-213" }},
      {{ "5.35323167813788168132305145263671875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "3.62778514750403147104407253209501504898071289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.05603230256580909333512613256181086069213372778044080861620436376260245170503316441303581724403388552683668422287401763124771418367914789920284987220255387646685601968642931765033670724000770308127910918065433865509022691461489073952236533017504964013454255542118759406756885123664390687006240095440e20" }},
      {{ "3.22976764266951800463889554748675436712801456451416015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.89050305023335795340244658291339874267578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.01732691727954660011725367716319147371899471036670702622303272300897410300692262400537048572027256759129565470020582895602617924506137374758437886475976576037363655165839084170315722377939735400391371695808916328159137254265306017499308291350722410467152148965233006027461767679652965168830636324127e-849" }},
      {{ "1.09048941423813448636792600154876708984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "4.65254774353689448673776496434584259986877441406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.58980988503590960516637371738014660727349100194484396379725478994088734700803526572947540745308063381145664972855458934861597057007568531182327714420486870282357356343021375166932444789575430439279485545025094851301007569967666444995213425647227644871269194317170084391746473828083433195180432304317e1" }},
      {{ "2.44743479435021299650543369352817535400390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.82995112549842228050200887423670792486518621444702148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00266010667859458204491495940561046268243246346579041029260554228176172232635197593622622290922199036969298596977304010218474771392718302157017833152484401953670784025973565870850532277945591072432075029519001358625355897190571211275599240174542923282378185576537237233948130696870807231732661924935" }},
      {{ "1.51488749663457274436950683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.02354275702668705760531309145733303012093529105186462402343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000985234516448019885473896561732145157443536520866911511897918118736186775473199587722487758817093449012349451559620294147840372481518705271731392645893056248771116553702829600792931868987911801336627246125357506584399976395776801325943965572049289654345210028541577738692060413098304903948601150" }},
      {{ "2.67331385687573711007480881107767345383763313293457031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.37764930215238656785459170350804924964904785156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.47571438862309533743624874136133366541454712127562663177399504286662183346894197246880043192629661657584457838166576178504996182224295507635623850476949534360416194135273172780271343507597813231256012656614579448387095352094197763495149376101171772035626900389244626161777607049448871411562278802652e-2" }},
      {{ "1.01808534444340903490910932305268943309783935546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.72211847836940634692837193142622709274291992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.82611384964605354323505795669056920339301763307089989948479130915805719954810517867370884785746904749334148062069679398152101779177523464996450923425943206891878509254410570738031307822715129842441609577961369156460374327926946469260053825787159767563643637983753386908235973617270978178126680100594e-1" }},
      {{ "1.00731276656660706514401226741028949618339538574218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.18228984435489814508457584452116861939430236816406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.88175560346904726742930639398687086396036352224271818500936901685374885901648764892385495076288151387225190990418969616688304224966648010000551417499596854632379457426863173605879611399961688292945890600044144991480340984880181190619092493192482621545694847376381337620163394916517599662918605010835e-1" }},
      {{ "9.87804687799645529753395090111212084593717008829116821289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "8.48295760044814414868596941232681274414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.06922896545426989764243576324057040501874322165012440675766666713649225255160119756183042369835830558457195704471707778294936068250116736974227601059618844500705991834404101497665868066256287785153513125539305523985206318973305432078184236844170690028706643163133079135912179653756316354134757605909e-5095" }},
      {{ "4.41825747491548099787905812263488769531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "7.79827462222631995203414589923340827226638793945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.34369652134197483793288072833308732535658609465509415675669768410932243039179437886058214288267645007975345260332874670014492568423533076540783205684921434471069515293441857790677592200203989418352589897685209311617733829289352336172155226574531882652742067029068005584363117523423189997997484568014" }},
      {{ "1.52968448456620720321552653331309556961059570312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.63460821261542546913893281246288324837223626673221588134765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99996930989216600191785450269008898634303181250066938686025190496179003638914112281154344493820093586185759774048968741774280050131382009162499732926045936023057461223555387483027406180587409400576483490119308402153097647327397891029914779646839934539999749442740222968027612552511977099477561837845e-1" }},
      {{ "1.54590681969825345731806010007858276367187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "5.80711944432144022698594199027866125106811523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.53179160234960851931903141810390484338949709737193408192930930410912781308171727424842710235568081208824229115859793069399719735722193263870374052840504012604354910230724690998586625239385836126384899034476217672957422789468436133936022499041037971795682906772214055897416212584774512114169944288303" }},
      {{ "1.06149875426708788185223286681946319731650874018669128417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "7.82539004163816664938480016644462011754512786865234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.14272969225465289903926091127128136862591595250320697667906487225773277795209213908946435749298160698748843670986774816760975159538180372090105486573778335998950914987159909756145213525174507361115992516851061441846585401483368166991251261491681595295047033029078105940961946982855869715287402727788e-1" }},
      {{ "1.70942859622804244762450665007236239034682512283325195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.45014151369056887470876393564367390354163944721221923828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99874220008245006855542326558513156408919554643242887993818629193137080289456989925446796371679686209603602304321606023129900225876158421172053263787015124593801332165057745276555467837973732533352070833078756655605565367486807321355073518225701436176826535604183999704696382097767183866284922545237e-1" }},
      {{ "6.89165129121799887451649269110021123196929693222045898437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.03300713262715859741547319572418928146362304687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "8.04313106664538570098839747789552148131626561809170196746678563650616413426296764086879429552165570381706960290576103965119389999281879292911019435355504974691551612891938410889406352256775463548603589074490341705913775216055301816238813657180725380061984870607389371696477377291283392196645786369100e-3" }},
      {{ "4.27946433404347626492381095886230468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "6.70089369397394089659936167180376287433318793773651123046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00007146201085226752984788738357625307865767014027631803309640949680577376696724418376175136252913773532105485594068985474036691750060444806287575074228772867146937453568465581121950409526897590340390497454647087617789150538474509489560078795836109491866150406233412810302502603589419003168433407796" }},
      {{ "5.40162932367991470775092643918924295576289296150207519531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "8.23362153160343446245406084926798939704895019531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.06423742747299574139652654980590788059822860773454214351500321781043318590458743790589432652553137712034307825892416077869655718345307052092676953436188880309769835515354046500269913785918625489459139881331406660934866445646807527177908193314853725313276753369974838311556362819425625446719126859340e-4" }},
      {{ "6.98480482930233392835361883044242858886718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.82423550256237437978967808049901577760465443134307861328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00020487356414235503163915728935712973913169358041240146118411664231468676920896629650395007895636986611826729763726166983962518312776000911959157822099221405201493001705233813919293314432879950393142137203274049451871519519207057550355386892818843802770417350657588091700213982000763521441126819299" }},
      {{ "1.01174486687951502972282469272613525390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.80096038819214743220697982906131073832511901855468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.21386628363111930567875501391228231832773971221469956618940994372919354601446271218977478634940587702534597215097311823723977361204234536761437991482874205192547888855528921579052205936359371957155208181288324241439984156409711039174371275245144079425493549183603159162251977135775037801563222969321" }},
      {{ "4.58527642677075686350463001872412860393524169921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.20426803030317546711058795949611521791666746139526367187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98276019906690498641979219227000354726256121309260540509550375256349346398739094196651394582546860191649946783658447448964727247382391275461932541094336719168645988397711787831199214881092736264632640564846015317413042211277375865471754751928875230345189863352333606975129830522964976988522205012182e-1" }},
      {{ "7.60517875978827806449089621310122311115264892578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.47867011136062753706710282131098210811614990234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.91022318181765750138709868330053023824369950588567785742389327678334961200987486582852651651749970246817747258718976307116965890616698349626417133054222982034680094324870404036435780314465221090580409266996951521555820905007542674615310907131989836681135988524628981589839950536317534105789412104573e-1" }},
      {{ "8.74441787592109604670184003216704127225966658443212509155273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.16721737539795736814107751655456013395451009273529052734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99872128426369981492114383609632051952305521589536821086005990494116094292173923739983506376030713210025133335713606540746004193037337306885070775500980783374129579854971973151969528024543815880504327167057351706595072153954661171897742253459693134310482723707663819212838281958152922341896338434956e-1" }},
      {{ "1.91138738236028468236327171325683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.41870493681337528357744304230436682701110839843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.61393719165089070084284302313399424679983675134163509852591590498735859530953368982681984829992497504960254661066913076235623948356021050016415986014420596315994017260378613566207333104954408470522228787617243604128057260092817202582188202107977088110567991555379686212833460754614025473450009263444" }},
      {{ "3.95935805094525858294218778610229492187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "9.66342798237554857082587389527361665386706590652465820312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00080082312808327988582209522378397852005304579172107085124537382297137225033557828343923139785908077173538670582197264674005612125502043965115623737220659926592486850580770115661984729610019606048664824477525628872443957605574420631353398611478369946179434030057548832782863477926163435959118485879" }},
      {{ "9.52749482248151265129099130035683629103004932403564453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "7.36644237252481737243670067982748150825500488281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.99043290744436080750367189311776962864514898582236218043546095145053426015403970275978470316955018068268507145854768132371105755976745328665490014149700831691054362640281890694866429566348288921720874015164835895723114083232661342953621394611916380955923119767515256749281015210664840084139488728470e-1" }},
      {{ "2.48422895630792811018892896068877007564879022538661956787109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.55993124989414910785967549600172787904739379882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.91195173359063670429359403048505978785219394151399979472770021116863735850753993687072273831783588580026886709426140199445066463954822741869299289627867228462101496659941594784834735871174181621098355725481647468654367330416914560339262972917390166297854884128515599346578674725441419457079115897184e-1" }},
      {{ "4.94911189808764902409166097640991210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.00486277593047912404244925710372626781463623046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.45161540440045450223953783783894469144373503171768310299316311211396701039652476264548018632004490940551121650475174226555299061745541449779441170912616150358121670452871969103360706271231784736206842228875578338918267487091360709713844820843854370488212897734417544669039756642667411365232861750622" }},
      {{ "5.86830898821921187469285996485268697142601013183593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.15714500475533861589827466787028242833912372589111328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98538706799650092346821769070662036641738082636479009372539733284347573229317564170016552698886694627816395005478215973507005929122925461962335439146679105934195547032542346569150966461707254588652749552262131982196536881226818665443124490222705261425197752741794319954946333545210933092733146260794e-1" }},
      {{ "7.83501026710203586844727396965026855468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.38788040794881645245695267476548906415700912475585937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.03084303369653734759444079184018763110189145271757350290747789171120440011979438908956047437954061198621891331998729346005011749200857634828740000563446280260144290233886419978479827799434112209614567540357757105949708290107985650241181647155069956615108211465282807906696708079090952998142539773200" }},
      {{ "7.24040157384670202134202554589137434959411621093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.21397503061749550967718391802918631583452224731445312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.98962718742601158545306104847042012373529894592940838744141601234699871358047935296869274717524339141323620050266212255881930805755665793284647269071772742030623661802389270250239007055699061760303889999391509361560603832488400525248124730452423718749922363976939807153833465852271516354356322016570e-1" }},
      {{ "2.37737491856019333980043484189081937074661254882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.16670398708927525177614370477385818958282470703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.49589428217805567802447379970866851398217501608629580197067814528656191899243702632356211673360505654178344944705028006574948563210523891802606219873940153904076904718577079606890006710252221715434732369471615966209956056482424699085238332051063676105524207191799310422842401999054890360978090765282e-1" }},
      {{ "3.01647874000722708842658903449773788452148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.92136460953186087863286957144737243652343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.67051217459346602496477010560417251705712700254048639078839264060325224717762633346441800378240849611262512392424682983226473649608304764571881449403455973384432668551726106109236708017847884189936294979084049530151497877322166267026808199785689058092334141134841250691148642129989881676444005953065e28" }},
      {{ "2.63540555826397102784364179228759894613176584243774414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.22820432867716089475607077474705874919891357421875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.06662853666315384518454771688957273896640018166337060934230580664851581626370131366235045252515485872195142490483479220526746937459904335480861408290098145258297721984393013560504944686230699800648442282165588666276252945193146150539010418085408269547841393316042913883904363908615604194055289620194e-2" }},
      {{ "2.26046063075346914672536513535305857658386230468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.36270389219972226646504509517399128526449203491210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.49669691356986597021713215820093379236513850713669538194655152931335203737791678558428932820631859896635045077055245394317491407809178976243739664980597540207413635355482092381212167318467695663002155241642388129246468412919845420697165678187848016391630764936016535264450983179524692369001196616367e-1" }},
      {{ "7.44369864652441037833341397345066070556640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.92138014336919438207473831425886601209640502929687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.14010675669123183496921521116811427733135459824631474903189627969592690975861278168724266785431509343760131532834105760648359541103464859507378780843287835638142981311936553660415847409900494706845746029735940277531884780112929935900839635096070137371899495277565868989524061668633390669303733877713e-1" }},
      {{ "2.58707365843879599776045097314636223018169403076171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.94714110472695892894989810883998870849609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.94285469588168562425042075510651649105332456874498432065780910860149873485515925490595348642217083245456881656394030092730262652761300876175697759058061470631341983801941103052127203347591729319027059625352617883993869442078699806792081819748879905962225386908139580991345038356748418452094703929970e-310" }},
      {{ "5.98084615325454649337189039215445518493652343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.54141705573080567892238490479428492108127102255821228027343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000134883014846657688131845485191611848674206835683154280061306464539283951337684813473177431132125670490032207511582839375165363167348982592154407605611359695341748966890300250815981718309647127029989133813099179248600056821756358657145575917892530126627067999467459186401635829661375435200555943" }},
      {{ "6.95026071869872530406553323700791224837303161621093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "7.31225239711935908609730105922608345281332731246948242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99946829922765932774347186827261089119719545372876939189761602548364480617029835514052044796339343249676137993597737045638155557799285710458997819509839368231591299591459306098956962406996519241338464726435617191918273092870466980318418139849371272285377732672340181070589663937610591450209282798517e-1" }},
      {{ "2.24073444824900158689473528283997438848018646240234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.72011372485644930879419844949040907522430643439292907714843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99978273147710365618565727465750018434266860028552780634511158239820481200127637360576082082285493761515451879661983436902590331857062933298024957408602175417434650700857480478630047039957552639826605254005583176460691229891227955796802817287827361621819443402008175782397464338976239485836367013797e-1" }},
      {{ "1.02696336152198410170133247021340139326639473438262939453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "2.11791001950321867638993289517657103715464472770690917968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.98056858099165538580460452287541225753365554206313892868475688098072267264713973803329319240088967254425048329384938932204820454762680502689447633583472338302047491675207245782389326788006117820440333632657775030341985234164850042741621548603372080092391251779877542034296864156086357253603371919108e-1" }},
      {{ "8.67037052863078860909240574983414262533187866210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.52133920341677693691129213959811750100925564765930175781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99277952718489198651035604747543743540115676651768417733688848831717723612326298765655588890670608573536137096893609251918929861039853578444897886108139617036413401117695629638378951997111811960041138307910848973294872108144596558607179052633028242671568000242144929604511881481654385461936722413631e-1" }},
      {{ "2.52535597921406917976128170266747474670410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.67545906951556844433071091771125793457031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "8.98282264364143312494924261974141234695472294090992155155562018911995536275433125782292778674260525238398016330461123980463532873852775335243553615531641077025771609462191107211129184970987736565747180479720790198029098762244705019025351383419378700872375798060833692002857187730789356240125924455777e234" }},
      {{ "2.61640599122827916289679706096649169921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.04737226836765842108434299007058143615722656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.25310610377006493866619672909110501466370730517133791833746014957247005356303121141646462034804208859942897670276612700251850886627624409691966399085604959689485003780186276442904234165384933316935452289326908453230024435075360037297385764282630607352032826861327915100978947458904222670931759368924e35" }},
      {{ "2.82002594439443032836578595379251055419445037841796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.28903360394438468768812944276191956305410712957382202148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99981126663120663978315445418195178347518855090095327320928958000002380082757914082726230053423120527749961157170801907113926659877545130473348116514066094356797962151622116993547396704816898570832090901788549840104993042307779674567777216024811187461718618588120530554820575895739988590245022632269e-1" }},
      {{ "1.05597237376428054744792461860924959182739257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.74456803510535429779793048510327935218811035156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.05450439582280470237419243848514333965208787934652393730167951372955494178606431812200728304139567804628971230701860498398849175230634941866518705534394558725303044357497118657164159909944927143807152603979711611331658711835035945317940163428557910521645089962892594811777021284923659424445609434015" }},
      {{ "1.92238907809509029789296619128435850143432617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.08349612860076613971363479471676782850408926606178283691406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99982133131096723846335477747571205458555360592772378989843831769539959961315480174186773210831188200236203659966289556452394505676789152783853641019895730138295352465651299769502538786481056285110475730395967855975688507419189456451422446526295567348482093213471867865722148473015661603656725132323e-1" }},
      {{ "1.04631933860085252672433853149414062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e6", "7.47628658058347578263536092890717554837465286254882812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.10918690854877106790971565776367526216180983995819057803660864422406143847654780992569076690586240873186597255294638451033694659817944978005382660886521706741399259776875547778479274738700590433170032223482808245749786294093246294856718197278358866518098957741521257199802713807166809120449267805013" }},
      {{ "6.36259548895734333359697210141803225269541144371032714843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.22809772788395861198296188376843929290771484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.52598782338450286286135450458954573598472802663327374811636800203700927786701774385033841837545251692421774984587119423266677067201879737268514092791650544542213882071172025510515697098553107823303332247548034598510867132891098061561608401432719775239528846321974734634143406918807796517003534735158e-64" }},
      {{ "2.13062266921418196562697244811346308779320679605007171630859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "2.38534614031968601466360269114375114440917968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.19098455118066498520013706754490677591984461696830181149035955535619874262474066374087254218809585126187088590005269981305550987677753928730521021427525334082765124171365549957770314171493849134109885501003714622689267145862269434956103582260650590467252136804225198053437675495933201621005291059680e-136" }},
      {{ "1.40290401591987802021321840584278106689453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.13722689335705595681247359607368707656860351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.76476316747050528850692000246200519522360470716902181350858525854503943786512297020242024222896565430817144786078487939900580658865175044648134728561580288938753166434593435107710997858417174150763109875175878816362192204521588377137231081824131646740939989148534654273494711671253618986609286323224e2" }},
      {{ "4.37201757312766858376562595367431640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "5.50898368829070932406466454267501831054687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.50016375413563777673770541325000240347503109832508653488647336857427144256430628344920480322962866090482161278435394130795573385145704869750655237320185554045574687208211132875092396633526114578443250901360527371315672281676871350085960576744972765455104647558784495896122434499365424664074824652372e2556" }},
      {{ "2.82657718261440410630058739371861520339734852313995361328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.01065835722450432997924357758634528181573841720819473266601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99989414562934374733570123984537478193847375990023521202992404597564318809152441993727777418090556514104681985844118458823465596529403699968559686663737740106923387493303105200565503166558779713221587331443491180025140468746214432126679720477339697928392369627582889524402889778083449486287091857918e-1" }},
      {{ "1.82131588065436497009264016355700732674449682235717773437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "4.66518199450072917899845492684107739478349685668945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.40207373373989750079079577094631199620167306803671706440022399491116271902378455342284164919251165911009699545436187010204381502830971843813249510091499154307328869929569251004649662529059388775244320275509563520973233815719214546857449027774185623555318990471004404713606927167122651605783870374832e-1" }},
      {{ "1.46755915364810964092612266540527343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "7.89043697836551700675045140087604522705078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.64484232464543513497888195163429809804134777698713994865506287396508104602871734408178720046618390944324170991778876210342419644694461221793882529660281950244593440074231456401876196827229747637594809904497759425267031152839358279700896103069795279548051852526590187125738047135851503597639320208298e407" }},
      {{ "8.30433287676949054002761840820312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.52637571399967619800719376144115813076496124267578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.03503345548731106338710812130725055986503423501253010837510324365889248445099107670056377581266593900575457325985155519568071726317821696776190852210129272438899726884098324950090040468317646989831449979292183766681690581501814228892321721246601040936202160676525600474705308542633568308584316650080" }},
      {{ "8.81076109807957895100116729736328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "7.33372790631923177073758779442869126796722412109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.72892191263152309368418398483858745615162148917317277888912958706672096823639413941816735201743610855503568372158123380568416223942437204395543055191166267974357638190306465483077702922759360128463612732981531386413914170095452050612671264384868877433570747466951378322325020744405691770060335633788" }},
      {{ "7.53729022560753492143703624606132507324218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.44997674051331170439263495985926510911667719483375549316406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000626746844722828444925305995099577313805301524404103686494009081743532235207019307339151083010194102029079910572707461595043287395038108777455456317947638812268175715959797135803525856300281211049078122806988241700281453385528141559419083370029575853474851945010647429584718991721854559935746944" }},
      {{ "9.29196125311857485939981415867805480957031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.87364075941672677400040569750672148074954748153686523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00130310607123242618493657884019907082179077698498276221909900016213891825242765534255421206798520599892641330780794108614272121210817595736511473642012182311222406630247676600611837624670300497417793849558413334183751618128852678982366003027694097779362865347129226028454653385495003448455051752341" }},
      {{ "1.43727353532569850358413532376289367675781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.29608176142756548188117449171841144561767578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "4.54783993491495810650470280449947427623348347028981499281421336796280567779408506141109574684107695839213640033321254129078108321583824176685378348816016878194065833488610016432724082665467588150668096361892027109844713789484964974482948538692616206482165899790288506924374902752205806598101379280087e2" }},
      {{ "2.70759640224985531631318735890090465545654296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "7.30536731556413743149391848419327288866043090820312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.07547877767428083419096142604570911700485352561609679581466039671852713698452759329634869067401784752327384511745999840407965753203664746748858043080448574928987130431330974864221970333531951352649388383982665219131905566066113253186390249711325528920747650049981618690921655600270254173866304929129" }},
      {{ "4.86911459659213416298687016059432153269881382584571838378906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "4.64415295518658677309531412902288138866424560546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.41012673791647055159490710611797984019709597197766274319967811180940457128047417078813260208344264234412901043623761003849931025905704896535929048868960146432780155662155674231586473433894529344683868489738341572805069281662400777414260520026876565570790884943500995948410148791343831009292038245400e-3" }},
      {{ "7.84854690434243218444265721700503490865230560302734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.46294874434749200275973635143600404262542724609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.31540724563669307320322491667157973956200661651434919571593303846115863894103669513415833366999982869469042192760040827577397824232012793869428364179080549290195624467934132481201575233560543620306283174662613044511585064238810137330925608875093333350849757044686376810419383361338035393598016327803e-1" }},
      {{ "3.72120262782922945916652679443359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.94908530795382262112980242818593978881835937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.12206584334504219728846621264973553250302703946785171831645645721830894010675262018545815052840050758849380777393422282576480781495344500008748097426459462508969278927821357065182790433439566919608820582252667225317461885443234193297904549706413625058399088149505555110903375380342716569207812657836e18" }},
      {{ "2.37172835747338208930989900125041458522900938987731933593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "6.40757764503378617060302957497697207145392894744873046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99465320336479719185464362393617871174305860472162937262848024567759736497677820188393809529906684006391677099787283285239332531375600773760223206100883246634220752238455329498878968166316454533321577818353913376078773928516840980584615525022113173652501922620733886925072518116022140332545581471584e-1" }},
      {{ "3.53605103337158652720972895622253417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.84582476913116806827019900083541870117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.83532969791885885135620378353794410619553188299174712267165316998085251325072529407513466403156971688912550231810700054288873685181499788137652636209514628912399831379317778239395084315052983947372535292000124065735878003968456173427606734312841500343489942261612441073694733682134245803236657004841e725" }},
      {{ "8.31048089824369333293314215937996891625516582280397415161132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "4.15539351718839655713821912286221049726009368896484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.43482075927723057531497663442495255886217925614713738709933378320005101165593393456330280592636199704470243210249220699112677280162135519833702952111992891673206729843009842099688221317242192847449764385413153153282253151503752131156497992327652425308884952829049613144312048128169217759285595530627e-1" }},
      {{ "1.14666020001788820081856101751327514648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "3.34127651548888934485148638486862182617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.74479451169065114507177142090694649425184490446641781570460948781576584635840991601731927655494758023459925015377386019778946251375096661299820054396659345784163916943122911769409035509491324432205386045800229510005976694475685559497072577311014446199955236237444975977941360792089045512126998703937e1022" }},
      {{ "1.59284463946466757988673634827136993408203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.37368596141571408519865826747263781726360321044921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.12790541394790276732931574594592716168427963234746835691403175587826006982488095536822241070760245600803352183137207993347315747261870544496930500337717553335079860285134922718636682581363247953489931749833309595523465412068635376914524529775388476847978990731073661981262435307448746968442209991247" }},
      {{ "1.01037463548088457532259099025395698845386505126953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "3.83350084750183555115654598921537399291992187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.46827959515471898774763024079995706113716782750728469166998382738205474672874505408339070961454993286360763402251395607969974201963555866889613983970569781895238062738547839409015372756811222082668539050479266568911339662981230077013588037391296361362960017408451588351832195207411208108858157498757e-115" }},
      {{ "1.66695565004290529032004997134208679199218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.71804015666434395370743004605174064636230468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.16219309185108283690353246115834277672387666007297244563945515894098174972467389980428784832971321037184548414733298149740082669071577857913503517445753714679820984984835536300423998074680014379718142053804437180632038614386541542054507333387434066222641647278889877313111291185910035634611028423893e1" }},
      {{ "1.28464602296116354409605264663696289062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "2.32457638429370376798033248633146286010742187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.24945391385140670639469572823424256394544549042741674612450176003078448450232834405677926416541827735839125915448795224664876170676948629996836691342006517593947756961464160491430339434651171508096026536520981556090838358776179831242252299597098299062165399439487062435923683313278394249407345431587e95" }},
      {{ "1.04339649065651030425752598951305571972625330090522766113281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.57348570830582303869960014708340167999267578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.13059686908080605126564483508274490473647130063580548021629986277623406548383789633827569651948273474979014022501027829260141472868965231700274633764760142846471542740886183449424861759202613481741450387488820213753956013540876031520080117102079251759436887925675362188501650887210343405664270909838e-79" }},
      {{ "5.98527292754746476077798433834686875343322753906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.92842618905841398438405498438896756852045655250549316406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99969570850103897852254602475534171540024098089888531491644885537022647868647455296214377097624078476427734613966180360314627711193584708132475074727707481960045949327090840155765239976195485323033627665295521041657350135494380406033384926325020170964497589858453537791112107981125863527921983931294e-1" }},
      {{ "1.20170821629471988138682103652854493702761828899383544921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.83573171858579975435077358270063996315002441406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.28605001646404799122123246583375017811401439409264242217956456924187147921073178601773640157007541107990624712528914809901097933653148466740296742087373566926721281981068789281547799916665066119657434350735735616423796448248252402292977828214490771898387468010213868209841336736006316556313401379158e-10" }},
      {{ "1.79027858130276217707432806491851806640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.03754584982387711988849332556128501892089843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.63044664995037750836987820374386953646551425036186940515842751543311970070428773125366390621088048509415536569966741084563251083225715464442813762547476523458275835451374563942157742199280762833831415410370539590912680755089996665637460577287883121257390441169299493248423283996313182295602073608455e33" }},
      {{ "7.71075177335318884264125927074928767979145050048828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.64182056353383813984692096710205078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.43972399050369154988520798910605088410360743083682368257478281007222478749780386934407011016164443028717189065642983648339942330570417348717676304330188520673656075843370307967688074309785894752170432378373350782687762813282030251151752809306457995236444306078641839344649086096494148698382826630683e-559" }},
      {{ "1.22585961723861284554004669189453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.01372669446815572924860959780346547631779685616493225097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00011878102928302779143036441761880708224532433147368168056931216968198406137164127175122694955536795144107173837990207593653143943257391177552069467355860858316175819319731322452047752960070552385154812027281420138955970684262089648831370958477699568344849874652831370849460274952408206982772881433" }},
      {{ "5.82595149644963838930777910718461498618125915527343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.85058768638723995477498446282638155935273971408605575561523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.99998621052671820010209082288722125377109323080442152434584440919377662337625060557131747020218413297035621822998191924210038600199853200536989949496907248450515440309961142400248274163420688419892365590086355187405743025995286488853302886303794885670413483577122364857720736276931131470386157699818e-1" }},
      {{ "9.01125395544224438992841896833851933479309082031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.21806070291085433154876227490603923797607421875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.38368157061396129371090614663532617206257134398324144105974743560347038187851945959992361814602150755480311739774866352614225040566667708793196627370259194113771020229764238603211377558658426209396490633521450588911944454397547213176293597692411663128824894095155297793732492904564527595779976674988e-1" }},
      {{ "4.40965439472245090968272052123211324214935302734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.02453990431700336927178796031512320041656494140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "9.19534075097294910062427312364961024864527797483767863624171403928319226867976521096128692940847528103007338680691678979870596291215052686966371603169090630798213135736273205172476019108353742027081191692977102801637543740660392501627319952653856947188066651067307484842658131091249553304134437847486e-1" }},
      {{ "1.93717944386320728433759086328791454434394836425781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.08574263216941169218898721737787127494812011718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.56194064144518235667018224449588679976819296337783156090494451036523029483949448332763991177580492553296509263765798810431727918219796862387760870580577134565230417714219352767669331762365217747400655499790736729474440290141888015988952364748907315410294417222286566628219298383793200309754712308589e-1" }},
      {{ "3.66479892681643832474946975708007812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "7.71145011595037033202970633283257484436035156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.92304236247268349096312573893392045307545699411902317264880124837981164329935084560551082257778103184407170649203234984204658334295355705120603551098245781624244872037069412927213696722873101986921044921101077821551960590243440000669839151585453297946266135519928259442687204452169702512081165606273e19" }},
      {{ "1.32708985351551091298460960388183593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "6.56531833894015816668887630669360078172758221626281738281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00077473925602097877583679698815018000110220247989590351745049914772258652489667683543311902850183669682134777670514222872443151397469262756195899010288195180434965887160900063533882057343026426147190468334739066751949659071732450855977621834587152702675615140712006936143437739732339126497413250346" }},
      {{ "3.52775045264896220942318905144929885864257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.20207441294177685975341773882973939180374145507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.19164668712694362712251058826210564533792000244902438405631163228898011483361552530774835675890492313952004188178746361986328169935055931569379688204574777223286241370824460769287348812931304022631646073013552253933784290828875633916987679684918077103703055564642008994164871305721849760321403415697" }},
      {{ "5.26721150567696895450353622436523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.02594021105445992042559666401757567655295133590698242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00252610964243996116042029581448234570507107619387893463104238190901401098561255564385433004137657583567661490188312636640964083001255112695679226528406143723662756570159800308402030438165398677929717369572388554610405821138849352686065896479954227526563362385890018562652478441542017212666618852941" }},
      {{ "2.85382232656423795447153679560869932174682617187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.21622084331017621394721572869457304477691650390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.13603045090195516704596534175318401579207918841598587146324536013529784552205952203801294482696233870522877138037982641276177183524409958004888324787356863626406890600708780844863240402886258989044381114275612277702992295412653204859730899096318397139691123133990243168861782693287239971524166152533" }},
      {{ "3.93381694633284512141457867073768284171819686889648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02640519319720624480396509170532226562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.39863675434390989723351064011214037513398262547140577736762821607782111942948883337093856726075063500818067573716198405459606323946322650536233861891286348318106161892821504014313663700097828008365436512713907896457061009150442887680100088324705807016079909429412724343222986472982422270499720872159e-3" }},
      {{ "1.90723973215819387405645102262496948242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "5.12980954046070617650260325959266083373222500085830688476562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00003874831722429751496737922917004435479965257235816453591525868492670813319281541168275796018004391940278219820598993631669244673032352035813333397881915928790424205764915135090114218540888704442481908590391442502493189810976009897523082978954781479402200004590037271385433077485827559023806478611" }},
      {{ "2.08352376339207012279075570404529571533203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.77328372265651944869168515905855088021780829876661300659179687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000308249415661310193935943862153702303875909245307529450912006821815453371252563642757366042171762708256721399309832054533722706035892679765533583739735266608823632317266441068592608135365434098808509810551746075126687287986572543900322800829577692516676548344312574211742425657072976921014511642" }},
      {{ "9.71724513689399464055895805358886718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.53600001609863927917037917936227131576742976903915405273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00014104020491665960642725547509974756116986496973147973056122194845971531406444606607453003866992922578566310810305178136382912507500272971783788119308610688740112803511843108270710764262658648334376396166426766278251188688321297859103006142315969456950763638190401996565523343208463418705191958244" }},
      {{ "1.39802629394119207304099383737394646232132799923419952392578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "3.51936644048749802671238740003900602459907531738281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "6.22241870897130620142553713444278896319414844301127628403550478555392596780554049192378201563747768162180873879073883732945738916895322915439456198565943172406197196260279982663959045275534018807122490359500452521705604768928620538202140216274727339329295923594705918674314862791990908520063913560002e-1" }},
      {{ "1.82765007732204644297357987170471460558474063873291015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.84318950269757237947487737983465194702148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.44507161334831975914976330209249902713146345349963201960249788428960802366282255229864854849664361912073985037641790912699757539023105844864929631975643935921225809946412808629805438733741027621604921419911409131275740749522537202228860884159853851509071131465735369871648912668689687943767400944117e-133" }},
      {{ "2.49872242289261521364096552133560180664062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.70475708243581758338791587448213249444961547851562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.16348084392128872602170011561254935010976642009637548196335495165701186585536416597718806964461847864028617635977334329848073696152520245124091137376311572172062522206250814543168673719325221226847866492589752767830775399781911226358430095279351842770613889036173222279808875420192058143269430231674" }},
      {{ "9.73728711278181032710149489517448273545596748590469360351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "6.49131637906343872401748740230686962604522705078125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.07165047632947683093273589058329846367047057355046748020009102510780065766705586588667687646255828190502946698129852184038457736930348397388946238687402139213354445146888181529201139114650892414118512134805367250310852437034078342906915160983796227481872425495472151960083249503055095175373933146792e-1" }},
      {{ "4.41558345670632190066440259101909759920090436935424804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "4.02580267030248754346610784438098562532104551792144775390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99968900351231822950433739312428495072856038776486153703336951900488129027691896751286362525813536862927182360156159008954621569331770038862723809677808006540779343891491372231781933232571078612298233285052166498857617406809139173150092488738015228536389093793575323664246229233166651636689627818170e-1" }},
      {{ "1.69439965482008625397014611735357902944087982177734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.70724294373208412833875513570092152804136276245117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.82874963902734961145255850699575978876746826231267390332055952052622176060893943203741788304684214781121351119738947541263489494899175563981247061980860747331222509426652450764505228926170316851952179492921780437496906593243321687842665376169899786546461700589626075813241454757704572169462564574837e-1" }},
      {{ "1.03376346181700355373322963714599609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.28989017209520627493812838793019182048738002777099609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01199452382152846374481951076631164582584661608322894290555407864227990207508639820617740715279797178581146570337346215682655494757376897027306747771093102874076785395996716253897920655609144243672317898938847395985100637001354712944101351050038536687079382428227717980685264632621156939159222226580" }},
      {{ "3.11283305234331397759461879104492254555225372314453125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.01353477046427009958251552745878143468871712684631347656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99986074588519454560640521673442935641990554362956049271045530697088039943872329332770296441408515271037076808977915428133116228156704302455915102356256867263136007644093137679299480159178566422686424390921886424606676712277018147908640199369062110318937373042728100659267805929813035245614797996696e-1" }},
      {{ "5.28149186098080924978148686932399868965148925781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "9.16657975145258205884601920843124389648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "7.28853014056156731737794697397317103994217559431541111572375406126519183769444357872853226505354592130101907012445487770770781129901235873025553849331106253792412404560306141938477978026852566898693906386886543485615404988699045603447639500571979436388663669239796393481965967462640474331680310932353e-255" }},
      {{ "2.22250599362956213767006374837365001440048217773437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "4.64204894849468473694287240505218505859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.32843078440077439221637641999286892206805235377727400014747144482178933184733036820827184969102097810334387052048590277206436733653974464975608297721942370810634826788083279979408768576378602999295622590369226553818034587114239802440932660296228167167346743576531361276267309223102513376997022773121e-304" }},
      {{ "2.64401597631773445755243301391601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.48515468216955248692556779133155941963195800781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.22595558957063988125039996231413212457799685089809815878697787456948885300739760428233740293311835672030485009491199517382307607053064251483492857230753137256332086797523013500249415836024187602707789116720162609479623122987122977570646771495383250532455162474303561808394969890452286012410188771582e1" }},
      {{ "1.09582116879168154355284059420228004455566406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.12062729806059380166516348253935575485229492187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "3.40731230607113577867418131365007953849097556433032366687124266265609534681297061468250410931194577479128955229993084028648709681104213544161468193428670594705798621530529153971465865443890890717329532471903443507290510241292018220973611927868861647858691705199882686680328023542042496509604517393163" }},
      {{ "2.87650329232355375097540672868490219116210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.29636994307592203767853789031505584716796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.22908880787878338117870181426636174855444573322349359087228723914632875946311491878034980145023441888073390972797385266938998183475968716718276000295270333868881283934630328911650991709812223171518435502540229986453603655260898220094106548795285001784222894626512051690586979041086530161063766088286e48" }},
      {{ "5.43192335581155216501364479952940200746525079011917114257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "7.05490815336499605694119130472330425618565641343593597412109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.99991447218133598413920881391105179265568552579320617946001886143605253717825160221002770761833078341617996929941439664383856515269642825892255570319730959451798678974258583863414985420861854967974576953145352608585972242996479479728931155364675877878808323939895254383920003458645959690646207544957e-1" }},
      {{ "4.24450620737656849484897847446518426295369863510131835937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "2.42617657776718897988388334852061234414577484130859375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.28295177297287815001307142572162153750668927480049029937221930255174310738192949066940496577252702629998657734311395912708887847618250337367575924375521236807234588840327753338387239581722861162855798729933237542178055913633273793008757291901196363208416962469352000552707123078294608807586649015685e-1" }},
      {{ "1.30733273631510917012210626353407860733568668365478515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "5.28858724063975630660591753517962843034183606505393981933593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99964885633207072968580726334787422174518544602461721143383625224899368056665861649907581709049484723249000855383620986446956783244965088674964206494776757278305799759524067667215395257158003498553463221431113016348494389058271382782227321845047125985119505752219731349818937084514014330356212827292e-1" }},
      {{ "5.05693319148356598344284940083070978289470076560974121093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "3.71295988486159203589842547899024793878197669982910156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.63937121881680868303835544307084885674388908256423717510775350630463810161245157254316013064889945722190309457613724784742119883122943867724213022530021363225255978426044946346577578461808416905152263622045415849140604671011655772958990313726815312540544848648362428160986497782665450731113127323529e-1" }},
      {{ "4.84423969203509041108191013336181640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.98413458911828048669701018980049411766231060028076171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00430738669558002031403659131744896728312485313631409231096866927631637993851247415735380222493024677897434245013356617271278547792035688620287749522586547267897264556329014249359221662180556496497208175354676304526200770138492165863545577626683932270268515447259703077574615595169323328662157853185" }},
      {{ "3.70784803656541983585270827461499720811843872070312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.52501708992864104530440272355917841196060180664062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.33573197351168666887962631201029384882638323647836131662987813863879776215233605661322116050262224430353678691583537529582375506359810080440891529163886160078411813398386404062767305381544566179454141780145518165423921596307439563187082576279320332050551415851387221311994490392675793780232895364275e-1" }},
      {{ "9.71655514772415699553675949573516845703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.59166384187756904111665789969265460968017578125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.35377653844658353569031679679735299370922232948810544763027212743048187579990164187893014793934410534408779634667801500692064187926463087757507629287841868368096408503740782370080191376158895711024144836742418575636184986247750729343991462010937031402061673932392367262371858670977341473615463130359e1" }},
      {{ "2.19264585936548201061668805778026580810546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "6.96729316179001942387419288316152687912108376622200012207031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00003755636032301823537943813016356575902489314188145066916221494580392625081472816947024292161874513732270899683702071327371665141574567958235034154797657204764921080200596842458370044167411926333595258330617813605034939486089552138840272033589961004832483480269032835319807553392086565465392533307" }},
      {{ "3.07913249304984926180139837015303783118724822998046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "5.05463804011641570923529798164963722229003906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.01796919571840276681288861157787148204747502682382876961304668410493347723226232435621763407251840958497596600197906374696494268910490419389807332560043086656936398322936521311486726743408171670806339766715147358441693728848925652781609216889389399280179459317545232154188799226490332505068768929180e-13" }},
      {{ "2.72793625008602248271927237510681152343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.45107876494454290927649253717390820384025573730468750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.21356364414706225993709314217781803611734556534595777181951409142649823132949473277169982486320740015805393683502787636048469072138860871755431959907851351864995912275227191432009279405103766593019141570051245200073079914866268809278471723253071227068595740039873872137154853304817823898235004618070" }},
      {{ "7.23608622250063859837609925307333469390869140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "9.31713845716947162145515903830528259277343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.20546015073919711264373148066122758661068546232444031541084969085534241548251886930568813690809866385000095984750691172127616802373754602142656508934912781801862005537086317665378844206989588374435568687107987862008034665185792976330120721552229886514045101504253350965371418298181204870517230728955e80" }},
      {{ "2.57072303536429899395443499088287353515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.69790274993758974475532852466130862012505531311035156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02140974620888944930305959435799702728843289682449376214925750857783066006891649041158724227299321084257799199351924992147805951706584524451302681812312203333848599297059927350897068391576429287666840858314350106940048605718179391844174140189607359928083054157216235380992076831905704302805940691484" }},
      {{ "6.27071378772734664380550384521484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "9.77213705199215154541952932376602802833076566457748413085937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00013045496457016358902387902369248820481792505684604793782357003458740283012026953971512527832247937686018956926918760579827820618149654974789601261789353438501933209142889274324701479760186259188286221595702693452369288516143444427753506877231697874469255361407264955531595418134590240336559943863" }},
      {{ "1.22109376852135853575020973948994651436805725097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.11543635047828820815521422282756702770711854100227355957031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99995551589729160336804918676630911396340574056049766519068518427735361038006808700598379055737548489964622176597729664945550621782605173209704608243388212974829930345797712949588621847644770093271093043600103623962707878170666514346008821208975619634087411931268385031450753049330763535153583731711e-1" }},
      {{ "1.52900990354010689331598626949926256202161312103271484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.07910792856703210418345406651496887207031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.09437152552687759915144210042422837004776738858769798785362828201158918075365450163702134968526881164477730298716543350719085306748967317190021537573795780001155496528795641888775374745803795044275135632851107711558035622505495812638847592959079891240357223230602055612544312975892640303394964825816e-1149" }},
      {{ "3.33574880350019498109759297221899032592773437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.22409910922928020227118395268917083740234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "7.53927322890034742368017335227292720005783821228941858003370629012694382557138392281089806659775813675293365836774015539525201899124038456550795641920581579865504101847353382339574548922971435920120403286201070202878099356466592276232925556211910029587973272230882852354333961947656943741233580547242e33" }},
      {{ "2.93886093222884170472752884961664676666259765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.83152139210962529736886894582426066335756331682205200195312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "9.99965326837122789351065801513643019379691746234431978717545032882563543159129029000520238088911519893688798595680083260598269929340163335960195938697476035130411906261845669886681524297055544969116053569875568029319976030656002722062158836225698022385486930605555120535915405105091329811282147529892e-1" }},
      {{ "1.05363900341240332636516541242599487304687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "4.51733416318811920042719521006802096962928771972656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.36944656525417845173858897538026176879580880507614615066496128471220543538723767528991906038383446820702565624708556443997649629495841259671527437077527127938789704415913380536996498286764720566639200198900869760387706475703380674649201567096875423075710829719565752777057995176317511826228555593581" }},
      {{ "7.04420993592731428016920358636809851304860785603523254394531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.83514384888353143665540301299188286066055297851562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.42999851704960956730922360017793140547106959683768718211070193339741510527923658661871542385815054985958418322575081567325471819065963185600217971978825670497003073742499873659741487280818814513977323524960437583297927902838999554519027556228317521167236540123085211060151247184818776639943050911394e-2" }},
      {{ "1.36453344822830520570278167724609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "3.64161706602405751964646185570018133148550987243652343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.04399795038635921897857958256316651977445152567453727222518284598769789391999595233661749274120961391885225992798836625746470159354169324537076101684572935560321844745400772917299981122731280255675880379891143456808614768985468915540339803333726839718219814716863124296189939533443325484535601906033" }},
      {{ "2.96565824250737205147743225097656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.51611044873741196425953603466041386127471923828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "7.47522068392867838732450708572135930567101005550555468267204809080882851453167258980507378379243415338266777301093835073449659828508711197010120745211201439735694186362327003324753635700543968787946652069972776477402704033680932298818937199049452926851734709666219333409709557278942834445272506685923" }},
      {{ "3.49571885007329983636736869812011718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "4.05030442875070617603228129155468195676803588867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00424637003637001346519445137047903985527298734444205998841513031143713650570988838061609722582515075163633719319500702266902153041674931227041528924269656845803454259997891649918404783522081199789884938037449889035577760958358376931686006190221056227385652828889051574541455553647177469994824093880" }},
      {{ "3.48704350707848789170384407043457031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.94470451369696548105103772741131251677870750427246093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00348622623956426654419790365515385008204532308719120294458056691444071691627567637464093877906643804041743573926720499607391861016395825332249134843376717727039444569339034648222083969616342813466476442089528251404075917497087087133158228565604726619218594367232783173627524996497743616209681950686" }},
      {{ "1.56866354052483725212141507654450833797454833984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.92382444394148101329822253546808497048914432525634765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.99458548775941655985641049562781270043468152203580071617312952591624765716886645437516907451083025631663787001723138256606416990037410952933773901687094742927210507959414283830075225958048767485241813607677989510714884949211624158456814965576504829085206941026015695802504061055800475236320588403791e-1" }},
      {{ "3.02555496695998127343329997529508545994758605957031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.43476200926107111399687710218131542205810546875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.07162439112148019358772224877510168973399161717373536875684406796915754460308189843582561832996734392043166689610197386058626738901780367190433337506557349349095623450517180395264699151519926104197461968535179024894054340573408793813826035189319052898390671701445304365679647915192850527101155372317e-12" }},
      {{ "1.13301057369715999811887741088867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.63391498739915612750661644270167016657069325447082519531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00152645922689671305334533143447522531808487470636607472600090062282095427316474560067944838274073843864628772883486893974636361201579261003896147984190781801313303984627253458134496161410984325183430174864002871499528387099093919980953756534711057395376377803880481128200568335684903772454493560544" }},
      {{ "2.00289080215499270707368850708007812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "8.63919332962040243728552013635635375976562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.92699766759603617897105296135509926340232009672802063778499835631207758035200425882438232730073541633716557508264570905405926283429555595877466543065569320408150504538307810614249593452939955155496934950121411097262064539453295031460705244148452908856502571794692573823194011760644101511653087442625e3716" }},
      {{ "1.32507839540064445080225219442127126967534422874450683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.51887862883488423904054798185825347900390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.03647722584318600030967144335207527953463651392633257231597431308252840118498779108498324584815641015064709374407448575779770810144009458027431983385495055750280990347720737478088901276233575808268895728220588756650828610378844764861577519796740508015409537773234457412772265295624580311112182954835e-589" }},
      {{ "3.57603431414909023189352410554420202970504760742187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.25925527701659852259155614184749083506176248192787170410156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99982482017295343923291140868824543861889706024267738325920940955163642168596277387382324596268313941030962899361898533933991225186157467591500415565798572470081783136017764613428509329263882432996891459823007909353566621634663513526224023265569627278594547455316883257611223218894909706813593928661e-1" }},
      {{ "6.37895215316995745524764060974121093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.33606864421000417781498770253278962627518922090530395507812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00011705660623425091805720617137318115223118739797033368829524572193511821979599497313069101487866374165387923697805554918674050522785359139471078208862147954503878603840716656103170817064574728416607854591674468869938491395161398773239010400175784333016262331344116781579064223044329496394945074676" }},
      {{ "7.38104343240080229548993884236551821231842041015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.78526096140715813593603800146780713475891388952732086181640625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99992740925564330652806516420746938988010148239633275154076421571630454719695668630175026322867662102076692387029189869547293172340508820534949790931051258505036554857756821592275590340005422029182367955076901146714768419205676755504470790282646601620852828276821380880075367804416790635127471984294e-1" }},
      {{ "2.24606666462972934823483228683471679687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.56224853001515953110356349498033523559570312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.01973918494886267810001894567464669163247766970377570638787946610322435627850446066260492108701967988500602281478606685233882133420407449502588207373547970434356681888963673609480451591224779378429762251379481389292664486026843607177275521955621777327928762357765933587555111806380884809909558908126e155" }},
      {{ "9.00800182710125019180935623808181844651699066162109375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.64432725765355242097953691882139537483453750610351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.93259995118943779913113450886803213157300236441759177808360979990670369813799397673229074691051859756881735508284831070803950318509897851789286938821324800564473083981421925321740139076048924264684734691750703913517330543410997193207195299362629750503027996358069377923999788650279477370768077765657e-1" }},
      {{ "9.41820034138939809054136276245117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "7.77419300288664260104552761188045906237675808370113372802734375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "1.00000890381075384949081595209139696411199492416870417459364082687864113657489926105641478106668362778068330268736600393352152923274282322515415108939987073582719187381060734722668227714154297709452559911296248629890116220184188866083605360898785001127921221360852853375996265543887873427220880943937" }},
      {{ "3.73692731999517979007974233240929606836289167404174804687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.54122149350758519403825630433857440948486328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.92632813942166735469252865298175146289395053013193471532530902730842579927583105349449489218906918310069151271847219223504300355763614013961247019308729802033635205970100220700085021558381217246935232153985446384879850169504304149149649141113162707552300166988589633450196697040655835196191118606631e-25" }},
      {{ "1.36207381223418886456721921263124386314302682876586914062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "2.62468618182947011518990620970726013183593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.23349312978912488484797519539255018773821880984944037182332190114599517402131412348320914129812507474815113624208263309735483442308554370722059051755362465566371079854655405434396337539491832247301779055162543337811300162664812836135792078898777055591459214033387458633919871456241380657515870584905e-1015" }},
      {{ "4.99439352691324893385171890258789062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "3.82669140887688854490988887846469879150390625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.25800762575108176129137940894321369236803059535403474287787709049193999989553876351149791249607106743078893765328229169808063556877194151113548299740103775991789375636892691914854010648529483996498522632062568652379746122433944483375027665116468292026500341138320088041194003298397510386237462266236e179" }},
      {{ "6.48539296127664783853106200695037841796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.19081769830710604285428644288913346827030181884765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01763841928646063179945311430619776672100366204545937731495139936077193534063748395270846081233643612207997993464082794803985613863394615181696471224069637239381759785944556677466232970035407360811317566511537057177507901702757204703329800142152099723806486139061978862458791045685416916469263617024" }},
      {{ "1.26185358741126574244617586373351514339447021484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "4.18707874563769861381290482427175447810441255569458007812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.97208954505633083962348848418154736485944843062550245347612203304547965531131555025237286475690843458460809325029245321285039436520603179307117618768371875438665250167675789128497710216048233584274468422162333906486838232593986287174975031846178589748114260487176217087771250070897293526267935699686e-1" }},
      {{ "4.35488942342848749831318855285644531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "4.98678835531215099763358011841773986816406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.28936120581079817550967747499324146341809782859497306106568788729215792440868081012592482836115918565108733027645928062811796100015424546044055914594061783213133483941275372186699846142681689537346156745852091720729474664315028859269079769914368203114147150204153490030106741528606576208240181832379e2313" }},
      {{ "9.52079140369027854262640175875276327133178710937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.08107909706786978198955750940513098612427711486816406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.99946912731466247972527890006754510895764819973853843000662583785640845446566388520861934833991540886888853432324685563515222422612660717906957938307798158191679485565611171204322088502967506340693425660800080176478497228519680488062210430869030836704176100634086409155754383655120084467598988079204e-1" }},
      {{ "1.04090684013030454480031039565801620483398437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.93673921485533583108917810022830963134765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.11210338662691592819619664979726597449100163292683513479170961681889868927958630114518483290408310841588879531996056535646660865328863109763975044243074863201932805945130071457158971557983161341534770079019284360491414192422030250649176833692020314068650401152520803465262499102820988511235853988648e197" }},
      {{ "1.97149801027129978336915883119218051433563232421875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "5.08739590531112617632913952547824010252952575683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.91773157361409781701715493663763866515090466561690289420891165770114164193580683686846646130438934752909209919079027125210567061374829822112866779201464446464651026659338435230202216580437945262368984290532688736997482187636274012593325315635831427955296981578931251003363149110123293975764983649974e-1" }},
      {{ "2.79971511541740710526937618851661682128906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.78721875202228321077857486898210481740534305572509765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00326623969490901566461359485954800182836515927200276988639325398663724684484176518830065683294723102506600723636309590040090118084917994679575687454329398419185701984750026134555302778342008308134679599723649132979721129657717437355628051914400727321111879908949789141417261266151891576137055521720" }},
      {{ "6.47180277522192426431502099148929119110107421875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "3.54396046637904049703138298355042934417724609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "6.11393159288848940315388484390073955173736785149894558116032886653088516351969818186256818732041463123159800744120590173108691581334056179094211775690837141959813271450679799058611736101514220405786918675156348944553549256908731766892776136986705072991002327432436231317606705791658862062772822668961e-5" }},
      {{ "4.68478493824730685446411371231079101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "8.08503458490309867556788958609104156494140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.97762155395492793734284548790084346203097684458052431296538227291993735975496781864258983550151454075639982388971649044469779839674940147551147838593024539580668837295332526092577626791361518961708496544368414553255175123291725694241746786254191457634438577633532650001615818789185478534006606741407e296" }},
      {{ "1.54391459551929816074178347662382293492555618286132812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "9.30291502045438312151759419066365808248519897460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.61942064374907006758472323953419672912600288860094391695046538452658427001421356899386802781741233841276963853691331000772213773371503948038327239870459585412114416103967368390201886001946927622637207918880459451317082626408706757400699845882572764664471363683650469561537447695573432201521718527705e-1" }},
      {{ "6.48003875925739691865601344034075736999511718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.26121082452632900412770045051047418382950127124786376953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00006094496769763278820872383817879036100529479018908630432836412716347076273911182394224909321200839758060396148281450949863621182268008379068612869939290370399199699187329868380547441347229000962054100848350577521278040241049817698210922858722233509600098748764436396006044192563867258556456726002" }},
      {{ "1.02255146268133501052943756803870201110839843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "5.89932691098755263015007699323177803307771682739257812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00013156917767091552599892235129254853484398955770015153844565825349128084659104946293310766324345122756751921898192510410177121820174080396618577123202206891707775599129806431636068338328182197155856712217515198962725685962829886150997851437588418591804207516238583892908865355284748545102405714042" }},
      {{ "2.81888129121324254811042919754981994628906250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "4.07352602363188209810829221169115044176578521728515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02324693289074461806065314601130588099455748535483252525100029761881094483032981058215632709470546862166291555062468221229877852679809354657976457141823806639541535849041446001105725715917762334478759524935613974359759936992996718201195710259035985781666122257084622777055529931477121573215140197696" }},
      {{ "2.07185404527926539820087414867089137260336428880691528320312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.35292300705978618680092040449380874633789062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "8.48850440195174823298359540121818302112603372945421421598869878434979555804675505735326375593302973038814914197710751833892726619856756218734694899102028024355384042325664532283875944124268902285546831042825514027899847395159917919088321233227951525026080119430650746383889735094027434636578283016912e-26" }},
      {{ "4.31935286049695476781606906513388821622356772422790527343750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.67446027602891164636454701675205569699755869805812835693359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99973122517169758868298611393816315674468708382437784046013092517874934197350186005397961789565235843871109955511656116413964407693637441952651815512224774438246717669200447114406421198392169132912944363201879586414171509537947822165122722681222879324770937027844102716000158168894762456660415032672e-1" }},
      {{ "5.60728659565255868472196526530737514804059173911809921264648437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "6.38030853130188620203101790284705430167377926409244537353515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.99990816207746297627000228090032149940299388165914304992280548917955938225443662692605137525666097388080408339273626056667955459490552137971306327684500228508064672496432612203803093818358636762048072433804065084080756285538210846111127149390742779965190790248566867146590543170273767758612958658971e-1" }},
      {{ "4.94365872790929563630746290670003872946836054325103759765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "6.08107286586160853403271175920963287353515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.41889634080305711655291544430541860422410170311666966191109605478799772032330347327796646352426019590688459329571978188218768772574725991603534645626444629132445446486278236532522655245444099755490849273965885852698636839519891473821647868673766453693048753250768728056307822453273785810806442880557e-262" }},
      {{ "4.57863041241061873733997344970703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.27996139284958445614392985589802265167236328125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.85252911535679891073378835901479886089967576299789401385670015780551662533990060451545346245206661914640102653229509656129118798311517873586006356564037704026911300448200277564257378774264510533411693099418541902845247573044373703549276907520323049567611007543654341177174512295005876331502328496351e72" }},
      {{ "2.08423524510303792567356140352785587310791015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.35070210371325316600632504560053348541259765625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.69650994818884808551478851790421398330849110943521821889862054443832715742450095948994740658527477301306542925058870703929885411081233767845292798598894899925369153906803097693018622562446710881324699679389099964263723003154543262693904642747226514772235244787350437914151428906711221635213885798789" }},
      {{ "7.03507355339941797822689295571763068437576293945312500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.66736311094788106856867671012878417968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "7.68909935340978860422757035820741960584342288164122209394216788923406587897458745269054408633969378408300698489680800368145152233150125878615072546588959342193206959268161227822392820692595901101440058139059566058188384350410226233241151749648406111214064140648607944967440645591239738930953785109368e-1000" }},
      {{ "4.44777561798017704859375953674316406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "4.37314005329937316446375916711986064910888671875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.12314549862845492717877595964768115546325186339715764453604920761210996544623734651949041265076846799054130789000520895961692269440989421981634150513113011053879079524507516981642085862682964932699084859566571367826867471772759871670601934760070910316789800074922999274803369631048113774053097073929e20" }},
      {{ "4.78949184649386006640270352363586425781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "1.79183351174304107189527712762355804443359375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.79637132177375382162263956935980813855595791264327736518378404210637984016731507340572257630052502643907564502937056932274446805895054500000116906646009851736492937513833076602489625307279653461589071484812740516746714953908730424736460829219077432204380475346934742691432944334479643432469277088306e659" }},
      {{ "6.22325830215437599690631031990051269531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "2.25225952793516384642380501190928043797612190246582031250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01987069469372759270453628912642728954230513534833089604954661995171725801656380902963421144934906141234552439696120736550210382743489449329697576830407056888118360526079463661066722214220114165138840831176499526985462507289995825151030297233511088604059436353741508088143154130210563278515547311543" }},
      {{ "9.12400440701637786139466257395724824164062738418579101562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "2.34400553054916376341521022652614192338660359382629394531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "9.97821977486950583166084719161557689663568325790003522306021158417755185644936647878316368235185593329253132043532554677812935724747863173935255764658313396824835230126055857216828346612262350441572667853382236006603976105644273478945602579828808105896550462847104108659412377316272488653010471184196e-1" }},
      {{ "1.33795627811531119277788093313574790954589843750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.48183812852515298791582915027831290899484883993864059448242187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000384349299045776862801360403732363545774843774701655200272080627739941386437175886379606902727479191090187794979610761904053274682140855025508483613322969568848488035529156236560186742600966591613965074184599845133474883618868818832890976311295886503114503738779326119140373893527826165039256344" }},
      {{ "7.06939041310026368591934442520141601562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "7.52070113177227312917239032685756683349609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.16601617096668251300410633861451144300904853213668949893514704576723535839139585890152639279492333481781656085682222252609689619088940700517823681224990153082715763605916833922234577262387169239662906567674441665483003247288718595609022710170750805174093013421029787059183231600348513197446175716344e289" }},
      {{ "1.46809474819978857951241479895543307065963745117187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "6.55150258058036227826192998691112734377384185791015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.87508829078291473496350060709599928627355203095913409392108636898471105714839831302147923025254259824329731513966469918955762532776229610929533035120642195606951641982085398951090528368050382553009140996960228822308531093162518295270230167711636888797942089811261515649759631850124542086583338696866e-1" }},
      {{ "2.13403424835757107302924850955605506896972656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "6.74881959702026967828913939229096285998821258544921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.02087023232278898673025082821180943357336233310089987611387411667231521549575677239762600896132782138671932320359043850059141855317509884880628897347559802452573752440941620827448739200875424446592178873469737214134752340014718109265906882735838254335143242217171430272779729156476847140495477447567" }},
      {{ "7.95233844752690766100045038911048322916030883789062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "2.84851678481346848315070019452832639217376708984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.52320441273088937482705455163451804728010853035570734024869989669472225400000323970014547943529070869582874785761555559916107563529628329435884632549334664560080647778917243975126257892837384804686623525333256233581024540617818741704180357848449262419278792476592443919338009722004675820379508226554e-1" }},
      {{ "4.86469155616132400629658683044453937327489256858825683593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "5.22877434444638661270587931539921555668115615844726562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.49398596693121402232511551773946216238940594887656584112333721649128338309222255715733045830942439483758414668436262063880698192055748811167639524982228502529443333539688201437948685066022343441584667892362197265423697997073104183476242310932155094648160217817878533689247152065556146288540676206268e-1" }},
      {{ "9.97061023610679164619341463549062609672546386718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "5.88064355210892131253785919398069381713867187500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.31339250305317586137103280435203540286641209041963686984564977317675267872546421630007855518104235103746292194256810045376453068069833982650717834727768128915903290190711155087191155109113681049655966851631090461344965018352997694452739029417934674832819743056725077543693812469593253205921308221308e-59" }},
      {{ "9.25067228877758662708430392740410752594470977783203125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "6.58041095700439004900772488326765596866607666015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "7.34794411703227640768834559037146678081237450497151808641284692857257405073921261445846307144839876248119821368257643009964194784066475970698686506948363255089072439573285741549161564675949811535766568038690509602693098394199576228057991570285009737564027696021280582562413585412543950911326853262100e-1" }},
      {{ "1.27144511159262219734955579042434692382812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e3", "6.18836319449928007330186119361314922571182250976562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.55634262177085310455272777009125596103461766420555774353254576944288801886851836915203973701385795773437574206488784269131348804974222211728133676956373754063980028710578189114584286886146802067657659439377822750677886486205610448378645450751753499248735236114037268909383185086028320044219137753422" }},
      {{ "1.42205871841051475712447427213191986083984375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.49175822061007323782177991233766078948974609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.31499443769464065089935605375459683035642863575091614297392875457512297919442473147075015935528487370816930373009973708163470195938036050657595355692444453780282744702410807810566482106290630446087507470829783752836539009991223629616484834464198297022741851935670638283977884632514115384409382379451e5" }},
      {{ "1.79830846471683798881713300943374633789062500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "5.15740741801222002740900940409574104705825448036193847656250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.00026780930725147708425182905646433452420215464014520968261578887024273841780021979940791064143678705875846798937060716374571767664452465951073901977576420551043883735274454852404885141087100496560513919957329201186421887438051030917217421635429456043545707951122601188323754670287140035060211268236" }},
      {{ "1.41617493747205998033678042702376842498779296875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.06659473408631241819755075539433164522051811218261718750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.00214037426842670551611147350147052752193345171802567422360034056719592273707967867550926051192650203531759829732673928643729296057128383914827597014066465351929167430158631190618779931772000386360976972135737607634338444086468691888473748375182457345550820744493891034677687786078596412978684852640" }},
      {{ "3.03620445696295947202170140144517063163220882415771484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-4", "1.77755559529355344761114565699244849383831024169921875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.65907453809699509460147409829901675941527854788885630333979265214696163822247505595008528776776735251807610239492515526114770917079309768787319965003601255566849812163333057688474056980168539317656182802355025321396669769362336327872219770789870937515083196007462468888621524361865732646086424336453e-1" }},
      {{ "1.88150796984490007162094116210937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.96075222779542400530772283673286437988281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.33268783679377583213014321394305562443708551652324893198731298378098553656636607906131354129802125769805603337990709647117721910862833606977379369954861887407787985331605230823250624512728814254652735899657072127956911865541366861599183456838203731048557393074731179271410020592240238165188774584943e838" }},
      {{ "8.89727582886910006720881938235834240913391113281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "1.04127996077174773958962106235048850066959857940673828125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "9.99878344302358244722248144111222437762654865779811772935718213218215174416593116585765566781129897695750076434329093650632173287168263229734935010362448649517261601361617830618010680773234135778887817284697256494192909633866400288905649938277337678717548098982509457693289514879908537721749577461800e-1" }},
      {{ "1.83083205221824460440238624414632795378565788269042968750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "8.51305815268083279079291969537734985351562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "5.08759465190948202575969824985581134108808966132826957045953621966485495100532942367792704772987024490520063202480279983957515910756497430137889290866515761789926914423548570454774810411630371091100113544891279153106437138536746735808516608107152208446229082973737250416694137957514739571766896873306e-404" }},
      {{ "3.34665340825903966375139131361748923154664225876331329345703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "3.78685988588032159896101802587509155273437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "3.52212078787830927194297643296251781134291925779877466674871728836105722487573580131785145484495276918545575844013070305855330551951353365366550132397674036846629030890995284036782751533644901503503905390007269189142085867922573306793968734446410118933272999821289312420310855129959801669881803865858e-2074" }},
      {{ "5.33927735030905274151661471515595280834531877189874649047851562500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "6.20766189867572767122939902950662371949874795973300933837890625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "9.99991034310800110010707678195555035460840436574922962233018600570397473321611757701145927433415065972844383465094874922585959664097551124468322387734536918015102243687653110114625528443011692213994065730522520240541725522418247328328568974843758308408263849244150678651666777627782107985602066092068e-1" }},
      {{ "2.19994195074324068173154955729842185974121093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.06707480325778616969678580517211230471730232238769531250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.00952545862365855685219042115608145190455235613268820530406120489310419092421063376518661644406509494089174786391890344365290616699827245940846567812410266752069452669091303167176618868615804832415636793311353407444727299687139263635273616648724056070625009496090575286667611858626054037393426920940" }},
      {{ "9.49426131626506853745939285005306373932398855686187744140625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "3.33658593189578134285966370953246951103210449218750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-1", "2.10953513637051167906362488703903898283324995313357160919327331939686921714903739192273840983847338772103005796728222224591772689334463276313863779738778665022741134644128703431090827473653846081611135627094959784765932765625500258444912393296945910143335619574776807349923037475968356403544804516261e-2" }},
      {{ "1.58204381201744226359551248606294393539428710937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "1.69967088875234341601404286503651519524282775819301605224609375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "1.00000077966919198536118506000604021848342587459233214812502825319036185927111842640720441778472762148840769009480654434988580516960635247026640481971290957649218368276357248615028690535272593446475039055115505472342603119011692356413204456935489635677140077206818440909527541341514742179392759484948" }},
      {{ "7.89159836081502775182242026552614788670325651764869689941406250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-7", "4.81696106146368038025684654712677001953125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "1.94147359401180853490618926342983857830809890106752620847948172427377774972592621604834694930262822082783398795004418901297871494637794539318032606545630879764294273346094400196886499691750495365869112152459387957968585556903274684614177896724684391146995388586545846815713970966406314356957705636188e-2940" }},
      {{ "1.28909708384213387034833431243896484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "6.93039877413820920581954965200566221028566360473632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.08496647090160629262865176257751429047729744058064486558757367783712447981236338124011887293782800275409466363098994983981904989007840757365461380428739706364354348918768376442904733609028311603151255847131247523710764847675936168535507206845244124708518371354311381017836653040028826211975497871572" }},
      {{ "1.88979339146257488413205294364161090925335884094238281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.50746279790036936674368917010724544525146484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "8.76918423991144038304021445193438173201942067819279237648900596695575024236585474624026460405210909663314261583560910578839787952809598740673148266220449752385049510413309645447866767727463403580780664879153239139760411591518829772808581212853680494395035365090735562853954119264286528501466090236373e-42" }},
      {{ "8.41517212362860789198748534545302391052246093750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "3.64690954096749919699504971504211425781250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.36385641555849335879047779054573450503531219143280186506059811952516602814802026798922317458022305556248077861115239238036534162971730930403784483407045473794360984711531633071954017599185535725837014265828161774595330813847033913388549400091383297967089159291617295023155265969142242902513558326176e3" }},
      {{ "7.79453915372945402850746177136898040771484375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.37570185974411316465193522162735462188720703125000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "1.06022076559461302576262215724290203547920286581395364873074659478022575993646868223512587189711310741927159850526321306597320278985726000202501743927791980296512242522543006053260060771089791556404973296770985910119125940674256012428243689156025466514102435785983398909290409351186888763744330929933e26" }},
      {{ "1.53996830801468109712004661560058593750000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "1.55753945580631575351659989792096894234418869018554687500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "1.20447431374301175187511662239775886280442123057569989101048717965641664163683965122530380365781626549245733448495292659434928101251174655724999866929500920292760069066534315397658186379331270528348085084759405281722204998903474609597521346801285181467380964556686904383585877178604246579963496131145" }},
      {{ "1.58292085642406949773430824279785156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e5", "2.39839658341223938009534322191029787063598632812500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.95369184630899349835322600435195771430552393938403392920497743251505007693757163085390208500076347439614593692001276141903681987497427879862172599262551403341684739246319058993775975522937386429303489479759146923528741632009619169067978673180741577543638362298805900307916877489621750277562246056785e12" }},
      {{ "1.34007962789787852670997381210327148437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e4", "1.93078487352859630432622140006060362793505191802978515625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-3", "1.01851774851557016267678896947673493426952146980838508956199625705868791978344101642179134412654468177801448310465798049263061832294449615311708734175434429497593168411288076223275335865898358255932555661831601275564405349791579099870920484601878548075981946266790952650050188664408833668093366337128" }},
      {{ "6.57565090460510646153124980628490447998046875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "2.95349579102368409166956553235650062561035156250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "4.92883670959819008737323351550818895693909971573197862669756297233542900030950953480711710193053129535412775508169382249001427538228339649766961919628309117712145393012207284297044617755253176225355208718450804357511545954540599873761726532747435308616642768909463909723766874135347528684022648860560e53" }},
      {{ "7.73401481707858096115160151384770870208740234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000", "2.49148120277696079938323237001895904541015625000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e2", "2.21022119284238444262099118632640694751857489911985882302778583637174384391251394588072250813144879442061174245455228341031066580984347413608919792407019123272438031870871777573323381609534176617167457318208979616222032754767955294734636001910121462806120247278470491361024670390806812831433737112379e221" }},
      {{ "1.10978898863228390113400134708854238851927220821380615234375000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-5", "1.32219377403916382718307431787252426147460937500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e1", "3.07947867569658321194458380597181231565685445537647073748124994390202466909582021155440785847623999110729024670271166064504654713592331835440877924887265726281088902886492996205474623955394661696347608605548560131153946508949187941058346142181719469523416895370937975489878692955467549849494731965625e-66" }},
      {{ "6.01388538493091151515557157836155965924263000488281250000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "2.60022931107700908527451122420615092778461985290050506591796875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-6", "9.99992690524283092045041727666257712294170472055655983655789156306985603524706361035867762480294454684826579379760817381252880474747063748101481751389649370586878847928039436815138481244723824252557644191693486953238362877605577315573573222917218930477592061427550313462645068276263623663297863327092e-1" }},
      {{ "2.15534801432384071073755649194936268031597137451171875000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "4.05812814671632215990371150837745517492294311523437500000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000e-2", "8.55799445393602464144976083125961944702079982479924287374752573999342183140040918969698718557087349720031284500593972794455976395816766982409031253828819168807248785638632014800512514311905184872890032777260194266023572565329278809193542262172193713145497824126566400120337339261082686915643720103934e-1" }},
   }};

   unsigned max_err = 0;
   for(unsigned k = 0; k < data.size(); k++)
   {
      T val = pow(T(data[k][0]), T(data[k][1]));
      T e = relative_error(val, T(data[k][2]));
      unsigned err = e.template convert_to<unsigned>();
      if(err > max_err)
      {
         max_err = err;
      }
   }
   std::cout << "Max error was: " << max_err << std::endl;
}


int main()
{
#ifdef TEST_BACKEND
   test<boost::multiprecision::number<boost::multiprecision::concepts::number_backend_float_architype> >();
#endif
#ifdef TEST_MPF_50
   test<boost::multiprecision::mpf_float_50>();
   test<boost::multiprecision::mpf_float_100>();
#endif
#ifdef TEST_MPFR_50
   test<boost::multiprecision::mpfr_float_50>();
   test<boost::multiprecision::mpfr_float_100>();
#endif
#ifdef TEST_CPP_DEC_FLOAT
   test<boost::multiprecision::cpp_dec_float_50>();
   test<boost::multiprecision::cpp_dec_float_100>();
#ifndef SLOW_COMPLER
   // Some "peculiar" digit counts which stress our code:
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<65> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<64> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<63> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<62> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<61, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<60, long long> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<59, long long, std::allocator<void> > > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<58, long long, std::allocator<void> > > >();
   // Check low multiprecision digit counts.
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<9> > >();
   test<boost::multiprecision::number<boost::multiprecision::cpp_dec_float<18> > >();
#endif
#endif
   return boost::report_errors();
}



