/***************************************************************************
 *   Copyright (C) 2000-2008 by Johan Maes                                 *
 *   on4qz@telenet.be                                                      *
 *   http://users.telenet.be/on4qz                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "global.h"
#include <QMutex>
#include "soundcard.h"
#include <sys/soundcard.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <qtimer.h>
#include "wavio.h"
#include "dispatchevents.h"
#include "dispatcher.h"
#include <QApplication>



#define MODE AFMT_S16_LE

#define DEBUGSOUNDCARD

QMutex sndMutex;


/**
create an instance of the soundcard interface
\param[in] samplingrate nominal sampling rate (i.e. 8000, 11025, ...)
*/ 

soundcard::soundcard(int samplingrate)
{
	status=SND_IDLE;
	dspInfo.speed=samplingrate;
#ifdef DEBUGSOUNDCARD
 qDebug("soundcard: created %d",samplingrate);
 logfile.add("soundcard: created %d",samplingrate);
#endif
	audioBufferLen=AUDIOBUFFERSIZE; //just give it a default
	wave=NULL;
	sndSema=NULL;
	ioDst=SNDOUT;
	ioSrc=SNDIN;

}

/**
deletes instance of the soundcard \n
stop() is called before the delete of the instance
\sa stop
*/ 
soundcard::~soundcard()
{
#ifdef DEBUGSOUNDCARD
   	logfile.add("soundcard: deleted=%ux",(unsigned int)this);
#endif
	stop();
	delete sndSema;
}

/**
	Starts a new thread.

 This function starts a new thread. To stop the thread: call \a stop() first.
 \n The soundcard can be in different states:
 \li \c IDLE:          thread just goes to sleep for 500ms
 \li \c SND_READ       read data,if no data available goes to sleep for one fragment
 \li \c SND_WRITE      write data,if bufers are full,  goes to sleep for 4 fragments
 \li \c SND_READWRITE  not implemented yet
 \li \c SND_DELAYED_STOP waits for the buffers to empty
 \li \c SND_STOP         complete stops all activity on the soundcard and switches to idle
*/
void soundcard::run()
{
	abortRun=FALSE;
	flushed=FALSE;
	logfile.add("soundcard: thread start");
	while(!abortRun)
		{
			switch(status)
				{
					case SND_IDLE:
						msleep(500);
					break;
					case SND_READ:
						if (!read())
							{
								msleep(((audioBufferLen/2)*1000)/dspInfo.speed); // goto sleep for one fragment
							}
					break;
					case SND_WRITE:

						if(!write())
							{
								if(info.fragstotal>6)
									{
								 		msleep(((audioBufferLen/2)*4*1000)/dspInfo.speed); // goto sleep for 4 fragments
									}
								else
									{
										msleep(((audioBufferLen/4)*1000)/dspInfo.speed); // goto sleep for half a fragment
									}
							}
					break;
					case SND_READWRITE:
					break;
					case SND_DELAYED_STOP:
						msleep(waitEnd()*1000/dspInfo.speed);
						status=SND_STOP;
					break;
					case SND_STOP:
						stop();
						status=SND_IDLE;
					break;
				}
		}
	stop();
	soundcardStoppedEvent* ce = new soundcardStoppedEvent( "soundcard stopped running" );
	QApplication::postEvent(dispatchPtr, ce );  // Qt will delete it when done
}

void soundcard::abort()
{
 	abortRun=TRUE;
	logfile.add("soundcard: abort set");
	if(!isRunning()) stop();
}

void soundcard::setSrc(edataSrc s)
{
	ioSrc=s;
}
void soundcard::setDst(edataDst d)
{
	ioDst=d;
}


/**
	sets/changes the sampligrate of the  soundcrd
\param samplingrate nominal sampling rate (i.e. 8000, 11025, ...)
*/ 
void soundcard::setSamplingrate(int samplingrate)
{
 	dspInfo.speed=samplingrate;
}

/**
	stops all soundcard activity and closes the interface
*/ 
void  soundcard::stop()
{
	QString errorString;
	logfile.add("soundcard stop function");
  if (status!=SND_IDLE)
    {      
      if(close(dspInfo.fdDSP)!=0)
				{
					errorString=strerror(errno);
      		errorString+="Unable to close sounddevice\n";
					logfile.add("soundcard: %s",errorString.data());
				}
#ifdef DEBUGSOUNDCARD
    	logfile.add("soundcard: stopped");
#endif
			status=SND_IDLE;
		}
	if(((ioDst==SNDFILEOUT) ||(ioSrc==SNDINTOFILE))&&(wave!=0))
		{
			wave->write(&outputBuffer[readOutputIndex],0);
		}
}

/**
	function to be called before stopping the soundcard interface while transmitting
	\return The number of samples in the buffers still to be transmitted
*/
int soundcard::waitEnd()
{
  audio_buf_info info;
	int t;
  ioctl(dspInfo.fdDSP, SNDCTL_DSP_GETOSPACE, &info);

	ioctl(dspInfo.fdDSP, SNDCTL_DSP_GETODELAY, &t);
#ifdef DEBUGSOUNDCARD
	logfile.add("total %d,frag %d",info.fragstotal,info.fragments);
#endif
  return((info.fragstotal-info.fragments)*audioBufferLen/sizeof(int));
}

bool  soundcard::setParam(int format, int fragsize,int channels,QString &errorString)
{
  uint fs;
  int temp;
  dspInfo.channels=channels;
  dspInfo.format=format;
  for (fs=0;fragsize>1;fs++)
    {
      fragsize>>=1;
    }

  dspInfo.fragSize=0xffff0000+fs;  // encoded as MMMMSSSS where SSSS is the power of 2
    
  // set number of channels
  if (ioctl (dspInfo.fdDSP, SNDCTL_DSP_CHANNELS,&(dspInfo.channels)))
    {
      errorString="Error on ioctl SNDCTL_DSP_CHANNELS";
      return FALSE;
    }
  /* set the format */
  int reqFormat=dspInfo.format;
  if (ioctl (dspInfo.fdDSP, SNDCTL_DSP_SETFMT, &(dspInfo.format)))
    {
      errorString="Error on ioctl SNDCTL_DSP_SETFMT";
      return FALSE;
    }
  
  if (dspInfo.format !=reqFormat )
    {
      errorString="Requested Format not supported";
      return FALSE;
    }
  
  /* set the sampling rate */
  if (ioctl (dspInfo.fdDSP, SNDCTL_DSP_SPEED, &(dspInfo.speed)))
    {
      errorString="Error on ioctl SNDCTL_DSP_SPEED";
      return FALSE;
    }
  ioctl(dspInfo.fdDSP,SOUND_PCM_READ_RATE,&temp)	;

  if (ioctl (dspInfo.fdDSP, SNDCTL_DSP_SETFRAGMENT,&(dspInfo.fragSize)))
    {
#ifdef DEBUGSOUNDCARD
			logfile.add("soundcard: setfragment failed");
#endif
      // no action -we'll do our best...
    }
  // and read it back
  if(ioctl(dspInfo.fdDSP,SNDCTL_DSP_GETBLKSIZE,&(dspInfo.fragSize)))
    {
      errorString="Error on reading fragsize";
      return FALSE;
    }
  audioBufferLen=dspInfo.fragSize;
	status=SND_READ;
  return TRUE;
}



/**
	starts to receive from soundcard or file
	\param s name of the device to be opened (e.g. /dev/dsp)
	\param errorString verbose error message
	\return TRUE if successful, FALSE otherwise
	\sa stop, startTransmit, startFullDuplex
*/

bool soundcard::startReceive(const char *s,QString &errorString)
{
	int t;
	localReadIndex=0;
	if (status!=SND_IDLE)
		{
			errorString="sound start receive: NOT SND_IDLE";
			return FALSE;
		}
	writeInputIndex=readInputIndex=0;
	logfile.add("soundcard: startReceive=%x",(unsigned int)this);
#ifdef DEBUGSOUNDCARD
    	
			qDebug("soundcard starting");
#endif
  if ((dspInfo.fdDSP = open (s, O_RDONLY)) == -1)
    {
			errorString=strerror(errno);
      errorString+="Cannot open sounddevice\n";

      return (FALSE);
    }

	qDebug("soundcard started");
logfile.add("soundcard: startReceive=%x",(unsigned int)this);
  if(setParam(MODE, AUDIOBUFFERSIZE,1,errorString)==FALSE)
    { 
      close(dspInfo.fdDSP);
      return FALSE;
    }
	logfile.add("soundcard: dummy read start");
// dummy read to get things started
	t=::read(dspInfo.fdDSP,(char *)&inputBuffer[writeInputIndex],audioBufferLen);
	if (t<0) errorString=strerror(errno);
	logfile.add("soundcard: dummy read stop %s",errorString.toLatin1().data());
  status=SND_READ;
  return TRUE;
}

/**
	starts to transmit to soundcard or file
	\param s name of the device to be opened (e.g. /dev/dsp)
	\param errorString verbose error message
	\return TRUE if successful, FALSE otherwise
	\sa stop, startReceive, startFullDuplex
*/

bool soundcard::startTransmit(const char *s,QString &errorString)
{
	#ifdef DEBUGSOUNDCARD
    	logfile.add("soundcard: startTransmit=%x",(unsigned int)this);
#endif
	if (sndSema!=NULL)
		{
#ifdef DEBUGSOUNDCARD
			logfile.add("deleting sema");
#endif
			delete sndSema;
			sndSema=NULL;
		}

	if (status!=SND_IDLE)
		{
			errorString="sound start transmit: NOT SND_IDLE";
			logfile.add("soundcard: sound start transmit: NOT SND_IDLE");
			return FALSE;
		}
	readOutputIndex=writeOutputIndex=wIndex=0;
	int count=0;
	while(1)
		{
  		if ((dspInfo.fdDSP = open (s, O_WRONLY,0)) == -1)
    		{
					errorString=strerror(errno);
      		errorString+=" Cannot open sounddevice";
					logfile.add("soundcard: %s",errorString.toLatin1().data());
					if(count++>10) return FALSE;
					msleep(10);
      	}
			else
				{
					break;
				}
		}
  if(setParam(MODE, AUDIOBUFFERSIZE,1,errorString)==FALSE)
    {
      close(dspInfo.fdDSP);
			logfile.add("soundcard: unable to set param");
      return FALSE;
    }
	logfile.add("creating new sema");
	numberOfBuffers=(OUTPUTBUFFERLEN/(audioBufferLen/2))-1;
	sndSema=new QSemaphore(numberOfBuffers);
#ifdef DEBUGSOUNDCARD
	logfile.add("sematotal %d",numberOfBuffers);
#endif
  status=SND_WRITE;
  return TRUE;
}


/**
	starts to receive and transmit in full duplex (if the soundcard supports it)
	\param s name of the device to be opened (e.g. /dev/dsp)
	\param errorString verbose error message
	\return TRUE if successful, FALSE otherwise
	\sa stop, startReceive, startTransmit
*/

bool soundcard::startFullDuplex(const char *s,QString &errorString)
{
	if (sndSema!=NULL) delete sndSema;
	
  errorString="";
	if (status!=SND_IDLE)
		{
			errorString="sound start full duplex: NOT SND_IDLE";
			return FALSE;
		}
	writeInputIndex=readInputIndex=wIndex=0;
	readOutputIndex=writeOutputIndex=0;
  if ((dspInfo.fdDSP = open (s, O_RDWR,0)) == -1)
    {
      errorString="Cannot open sounddevice";
      return FALSE;
    }
	if (ioctl(dspInfo.fdDSP, SNDCTL_DSP_SETDUPLEX, 0))
		{
			errorString="Cannot set to duplex";
			return FALSE;
		}
	if (ioctl(dspInfo.fdDSP, SNDCTL_DSP_GETCAPS, &dspInfo.caps))
		{
			errorString="Cannot get caps";
			return FALSE;
		}
		
	if ((dspInfo.caps&DSP_CAP_DUPLEX) != DSP_CAP_DUPLEX)  /* we have to have a full duplex audio */
		{
			errorString="Cannot set to fullduplex";
			return FALSE;
		}
  if(setParam(MODE , AUDIOBUFFERSIZE,1,errorString)==FALSE)
    {
      close(dspInfo.fdDSP);
      return FALSE;
    }
	// dummy read to get things started
	::read(dspInfo.fdDSP,(char *)&inputBuffer[writeInputIndex],audioBufferLen);
	sndSema=new QSemaphore((OUTPUTBUFFERLEN/(audioBufferLen/2))-1);
#ifdef DEBUGSOUNDCARD
	logfile.add("sematotal %d",sndSema->available());
#endif
  status=SND_READWRITE;
  return TRUE;
}


/**
	retrieves data from the souncard

*/
bool soundcard::read()
{
	int ll,len;
	audio_buf_info info;
	if(ioSrc==SNDFILEIN) 
		{
			return FALSE; // data is read from file in getData function
		}
	else
		{
			ioctl(dspInfo.fdDSP, SNDCTL_DSP_GETISPACE, &info);
  		if(info.fragments==0)
				{
					return FALSE;
				}
			fragmentLatency=info.fragments;
			sndMutex.lock();
			len = ::read(dspInfo.fdDSP,(char *)&inputBuffer[writeInputIndex],audioBufferLen);
			if(len!=audioBufferLen) logfile.add("sound receive buffer read error");
			if(ioSrc==SNDINTOFILE)
				{
					ll=wave->write(&inputBuffer[writeInputIndex],audioBufferLen/sizeof(short int));
					if(ll<0)
						{
				 			logfile.add("soundcard: error writing to file");
				 			return FALSE;
						}
				}
			writeInputIndex=(writeInputIndex+len/sizeof(short int))&INPUTINDEXMASK;
			sndMutex.unlock();
			if(writeInputIndex==localReadIndex) logfile.add("soundcard: overrun error");
			
#ifdef DEBUGSOUNDCARD
 			logfile.add("soundcard: latency %d %d",fragmentLatency,len);
#endif
		}
	return TRUE;
}


/** 
	Sends a complete buffer to the audio interface.

	

	\return TRUE if successful, FALSE if busy
	
	
*/
bool soundcard::write()
{
	int len,ll;

	ioctl(dspInfo.fdDSP, SNDCTL_DSP_GETOSPACE, &info);
	if (info.fragstotal>6)
		{		
 			if(info.fragments<=(info.fragstotal/2))
    		{
     			return FALSE;
    		}
		}
	else if(info.fragments==0)
			{
				return FALSE;
			}

	if (sndSema->available()<numberOfBuffers)
		{
			
			len = ::write(dspInfo.fdDSP,(char *)&outputBuffer[readOutputIndex],audioBufferLen);
#ifdef DEBUGSOUNDCARD
			logfile.add("frag=%d totfrag=%d, readOutputIndex=%d, len=%d",info.fragments,info.fragstotal,readOutputIndex,len);
#endif
			if(ioDst==SNDFILEOUT)
				{
					ll=wave->write(&outputBuffer[readOutputIndex],audioBufferLen/sizeof(short int));
					if(ll<0)
						{
				 			logfile.add("soundcard: error writing to file");
				 			return FALSE;
						}
				}
			readOutputIndex=(readOutputIndex+len/sizeof(short int))&OUTPUTINDEXMASK;
#ifdef DEBUGSOUNDCARD
			sndSema->release(1);
			logfile.add("soundcard: sema available=%d",sndSema->available());
#endif
		}
	else
		{
			if (flushed) status=SND_DELAYED_STOP;
		}
  return TRUE;
}

/**
	\param  ri  readIndex from caller
	\return the number of available samples.
*/

uint soundcard::inputDataAvailable(uint ri)
{
	uint l;
	sndMutex.lock();
	l=((writeInputIndex-ri)&INPUTINDEXMASK);
	sndMutex.unlock();
	return l;
}

/**
	\return the number of samples that can be stored
*/


void soundcard::writeSample(short int sample)
{
	outputBuffer[writeOutputIndex+wIndex++]=sample;
	if(wIndex==(audioBufferLen/sizeof(short int)))
		{
			sndSema->tryAcquire(1);;
#ifdef DEBUGSOUNDCARD
			logfile.add("soundcard: sema available in writesample: %d, writeOutputIndex=%d",sndSema->available(),writeOutputIndex);
#endif
			wIndex=0;
			writeOutputIndex=(writeOutputIndex+audioBufferLen/sizeof(short int))&OUTPUTINDEXMASK;
		}
}

void soundcard::flush()
{
	sndSema->tryAcquire(1);
	while(wIndex<audioBufferLen/sizeof(short int))
		{
			outputBuffer[writeOutputIndex+wIndex++]=0;
		}
	wIndex=0;
	writeOutputIndex=(writeOutputIndex+audioBufferLen/sizeof(short int))&OUTPUTINDEXMASK;
	;
	flushed=TRUE;
}

uint soundcard::outputDataFree()
{
	uint l;
	sndMutex.lock();
	l=((writeOutputIndex-readOutputIndex)&OUTPUTINDEXMASK);
	sndMutex.unlock();
	return l;
}

/**
	transfer data from the soundcard inputbuffer to buf
	\param[in] buf target buffer (must be at least be able to hold len samples)
	\param[in,out] ri	the index into the souncard inputBuffer
	\param[in,out] len the number of samples to read, on return contains the number of samples read
	\return FALSE if end of file is reached, TRUE otherwise
*/

bool soundcard::getData(short int *buf,uint &ri,uint len, uint &samplesRead)
{
	int ll;
	uint l,i;
	
	if(ioSrc==SNDFILEIN)
		{
			if(endOfFile)
				{
					logfile.add("soundcard: read past endOfFile");
					return FALSE;
				}
			ll=wave->read(buf,len);
			if(ll<0)
				{
				 endOfFile=TRUE;
				logfile.add("soundcard: endOfFile");
				 return FALSE;
				}
			samplesRead=ll;
			msleep(20); //slowdown reading from file
		}
	else
		{
			sndMutex.lock();
			l=((writeInputIndex-ri)&INPUTINDEXMASK);
			sndMutex.unlock();
			if(l<len) samplesRead=l;
			else samplesRead=len;
			for(i=0;i<samplesRead;i++)
				{
					buf[i]=inputBuffer[ri++];
					ri&=INPUTINDEXMASK;
					localReadIndex=ri;
				}
		}
	return TRUE;
}
/**
	Set wave file pointer

	The caller is reponsible for opening the file in the appropriate mode before starting the soundcard.
If any changes to the input or output routing is done, then the soundcard should be stopped first by
calling stop();

	\param wv pointer to wavefile
*/

void soundcard::setWaveFile(wavIO *wv)
{
	wave=wv;
	endOfFile=FALSE;
}


