\name{rLGCP}
\alias{rLGCP}
\title{Simulate Log-Gaussian Cox Process}
\description{
  Generate a random point pattern, a realisation of the
  log-Gaussian Cox process.
}
\usage{
 rLGCP(model="exponential", mu = 0, param = NULL, ..., win=NULL)
}
\arguments{
  \item{model}{
    character string: the name of a covariance model for
    the Gaussian random field, as recognised by the function
    \code{\link[RandomFields]{GaussRF}} in the
    \pkg{RandomFields} package. 
  }
  \item{mu}{
    mean function of the Gaussian random field. Either a
    single number, a \code{function(x,y, ...)} or a pixel
    image (object of class \code{"im"}).
  }
  \item{param}{
    Numeric vector of parameters for the covariance,
    as understood by the function 
    \code{\link[RandomFields]{GaussRF}} in the
    \pkg{RandomFields} package.
  }
  \item{\ldots}{
    Further arguments passed to the function
    \code{\link[RandomFields]{GaussRF}} in the
    \pkg{RandomFields} package.
  }
  \item{win}{
    Window in which to simulate the pattern.
    An object of class \code{"owin"}.
  }
}
\value{
  The simulated point pattern (an object of class \code{"ppp"}).

  Additionally, the simulated intensity function is
  returned as an attribute \code{"Lambda"}.
}
\details{
  This function generates a realisation of a log-Gaussian Cox
  process (LGCP). This is a Cox point process in which
  the logarithm of the random intensity is a Gaussian random
  field with mean function \eqn{\mu} and covariance function
  \eqn{c(r)}. Conditional on the random intensity, the point process
  is a Poisson process with this intensity.

  The arguments \code{model} and \code{param} specify the covariance 
  function of the Gaussian random field, in the format expected by the
  \pkg{RandomFields} package. See 
  \code{\link[RandomFields]{GaussRF}} or
  \code{\link[RandomFields:CovarianceFct]{Covariance}}
  for information about this format. A list of all implemented
  models is available by typing \code{PrintModelList()}. 
  
  This algorithm uses the function \code{\link[RandomFields]{GaussRF}} in the
  \pkg{RandomFields} package to generate values of
  a Gaussian random field, with the specified mean function \code{mu}
  and the covariance specified by the arguments \code{model} and
  \code{param}, on the points of a regular grid. The exponential
  of this random field is taken as the intensity of a Poisson point
  process, and a realisation of the Poisson process is then generated by the 
  function \code{\link{rpoispp}} in the \pkg{spatstat} package.
  
  If the simulation window \code{win} is missing or \code{NULL},
  then it defaults to 
  \code{as.owin(mu)} if \code{mu} is a pixel image,
  and it defaults to the unit square otherwise.
  
  The LGCP model can be fitted to data using \code{\link{kppm}}.
}
\seealso{
\code{\link{rpoispp}},
\code{\link{rMatClust}},
\code{\link{rGaussPoisson}},
\code{\link{rNeymanScott}},
\code{\link{lgcp.estK}},
\code{\link{kppm}}
}
\references{
  Moller, J., Syversveen, A. and Waagepetersen, R. (1998)
  Log Gaussian Cox Processes.
  \emph{Scandinavian Journal of Statistics} \bold{25}, 451--482.
}
\examples{
  if(require(RandomFields)) {
    # homogeneous LGCP with exponential covariance function
    X <- rLGCP("exp", 3, c(0, variance=0.2, nugget=0, scale=.1 ))

    # inhomogeneous LGCP with Gaussian covariance function
    m <- as.im(function(x, y){5 - 1.5 * (x - 0.5)^2 + 2 * (y - 0.5)^2}, W=owin())
    X <- rLGCP("gauss", m, c(0, variance=0.15, nugget = 0, scale =0.5))
    plot(attr(X, "Lambda"))
    points(X)

    # inhomogeneous LGCP with Matern covariance function
    X <- rLGCP("matern", function(x, y){ 1 - 0.4 * x},
             c(0, variance=2, nugget=0, scale=0.7, a = 0.5),
             win = owin(c(0, 10), c(0, 10)))
    plot(X)
  } else message("Simulation requires the RandomFields package")
}
\author{Abdollah Jalilian and Rasmus Waagepetersen.
  Modified by Adrian Baddeley
  \email{Adrian.Baddeley@uwa.edu.au}
  \url{http://www.maths.uwa.edu.au/~adrian/}
  and Rolf Turner
  \email{r.turner@auckland.ac.nz}
}
\keyword{spatial}
\keyword{datagen}

