% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visNetwork.R
\name{visNetwork}
\alias{visNetwork}
\title{Network visualization}
\usage{
visNetwork(
  nodes = NULL,
  edges = NULL,
  dot = NULL,
  gephi = NULL,
  width = NULL,
  height = NULL,
  main = NULL,
  submain = NULL,
  footer = NULL,
  background = "rgba(0, 0, 0, 0)",
  ...
)
}
\arguments{
\item{nodes}{: data.frame or a list with nodes informations. Needed at least column "id". See  \link[visNetwork]{visNodes} 
\itemize{
 \item "id" id of the node, needed in edges information
 \item "label" label of the node
 \item "group" group of the node. Groups can be configure with  \link[visNetwork]{visGroups}
 \item "value" size of the node
 \item "title" tooltip of the node
 \item ...
}}

\item{edges}{: data.frame or a list  with edges informations. Needed at least columns "from" and "to". See  \link[visNetwork]{visEdges}
\itemize{
 \item "from" node id of begin of the edge
 \item "to" node id of end of the edge
 \item "label" label of the edge
 \item "value" size of the node
 \item "title" tooltip of the node
 \item ...
}}

\item{dot}{: Character DOT language.}

\item{gephi}{: Json export gephi path file.}

\item{width}{: Width (optional, defaults to automatic sizing)}

\item{height}{: Height (optional, defaults to automatic sizing)}

\item{main}{: For add a title. Character or a named list.
\itemize{
 \item "text" Character. Title.
 \item "style" Optional. Character. HTML style of title. Default to 'font-family:Georgia, Times New Roman, Times, serif;font-weight:bold;font-size:20px;text-align:center;'.
}}

\item{submain}{: For add a subtitle. Character or a named list.
\itemize{
 \item "text" Character. Subtitle.
 \item "style" Optional. Character. HTML style of submain. Default to 'font-family:Georgia, Times New Roman, Times, serif;font-size:12px;text-align:center;'.
}}

\item{footer}{: For add a footer. Character or a named list.
\itemize{
 \item "text" Character. footer.
 \item "style" Optional. Character. HTML style of footer. Default to 'font-family:Georgia, Times New Roman, Times, serif;font-size:12px;text-align:center;'.
}}

\item{background}{: Background color. Default to 'rgba(0, 0, 0, 0)' (transparent). Can be a valid color name ("red"), a HEX value ("#ff0000") or rgb/rgba ("rgb(255,0,0)")}

\item{...}{: Don't use.}
}
\description{
Network visualization using vis.js library. For full documentation, have a look at  \link[visNetwork]{visDocumentation}.
}
\examples{

# minimal example
nodes <- data.frame(id = 1:3)
edges <- data.frame(from = c(1,2), to = c(1,3))

visNetwork(nodes, edges)

\dontrun{
# add a title
visNetwork(nodes, edges, main = "visNetwork minimal example")
visNetwork(nodes, edges, main = list(text = "visNetwork minimal example",
 style = "font-family:Comic Sans MS;color:#ff0000;font-size:15px;text-align:center;"))
 
# and subtitle and footer
visNetwork(nodes, edges, main = "visNetwork minimal example",
 submain = "For add a subtitle", footer = "Fig.1 minimal example")
 
# change background color
visNetwork(nodes, edges, background = "black")

# customization adding more variables (see visNodes and visEdges)
nodes <- data.frame(id = 1:10, 
                    label = paste("Node", 1:10),                                 # labels
                    group = c("GrA", "GrB"),                                     # groups 
                    value = 1:10,                                                # size 
                    shape = c("square", "triangle", "box", "circle", "dot", "star",
                              "ellipse", "database", "text", "diamond"),         # shape
                    title = paste0("<p><b>", 1:10,"</b><br>Node !</p>"),         # tooltip
                    color = c("darkred", "grey", "orange", "darkblue", "purple"),# color
                    shadow = c(FALSE, TRUE, FALSE, TRUE, TRUE))                  # shadow

edges <- data.frame(from = sample(1:10,8), to = sample(1:10, 8),
                    label = paste("Edge", 1:8),                                 # labels
                    length = c(100,500),                                        # length
                    arrows = c("to", "from", "middle", "middle;to"),            # arrows
                    dashes = c(TRUE, FALSE),                                    # dashes
                    title = paste("Edge", 1:8),                                 # tooltip
                    smooth = c(FALSE, TRUE),                                    # smooth
                    shadow = c(FALSE, TRUE, FALSE, TRUE))                       # shadow

visNetwork(nodes, edges) 

# use more complex configuration : 
# when it's a list, you can use data.frame with specific notation like this
nodes <- data.frame(id = 1:3, color.background = c("red", "blue", "green"), 
 color.highlight.background = c("red", NA, "red"), shadow.size = c(5, 10, 15))
edges <- data.frame(from = c(1,2), to = c(1,3),
 label = LETTERS[1:2], font.color =c ("red", "blue"), font.size = c(10,20))

visNetwork(nodes, edges)

# highlight nearest
nodes <- data.frame(id = 1:15, label = paste("Label", 1:15),
 group = sample(LETTERS[1:3], 15, replace = TRUE))

edges <- data.frame(from = trunc(runif(15)*(15-1))+1,
 to = trunc(runif(15)*(15-1))+1)
 
visNetwork(nodes, edges) \%>\% visOptions(highlightNearest = TRUE)

# try an id node selection 
visNetwork(nodes, edges) \%>\% 
 visOptions(highlightNearest = TRUE, nodesIdSelection = TRUE)
 
# or add a selection on another column
visNetwork(nodes, edges) \%>\% 
 visOptions(selectedBy = "group")

nodes$sel <- sample(c("sel1", "sel2"), nrow(nodes), replace = TRUE)
visNetwork(nodes, edges) \%>\% 
 visOptions(selectedBy = "sel")
   
# add legend
visNetwork(nodes, edges) \%>\% visLegend()
    
# directed network
visNetwork(nodes, edges) \%>\% 
 visEdges(arrows = 'from', scaling = list(min = 2, max = 2))

# custom navigation
visNetwork(nodes, edges) \%>\%
 visInteraction(navigationButtons = TRUE)

# data Manipulation
visNetwork(nodes, edges) \%>\% visOptions(manipulation = TRUE)

# Hierarchical Layout
visNetwork(nodes, edges) \%>\% visHierarchicalLayout()

# freeze network
visNetwork(nodes, edges) \%>\%
 visInteraction(dragNodes = FALSE, dragView = FALSE, zoomView = FALSE)

# use fontAwesome icons using groups or nodes options 
# font-awesome is not part of dependencies. use addFontAwesome() if needed

nodes <- data.frame(id = 1:3, group = c("B", "A", "B"))
edges <- data.frame(from = c(1,2), to = c(2,3))

visNetwork(nodes, edges) \%>\%
  visGroups(groupname = "A", shape = "icon", icon = list(code = "f0c0", size = 75)) \%>\%
  visGroups(groupname = "B", shape = "icon", icon = list(code = "f007", color = "red")) \%>\%
  addFontAwesome()

nodes <- data.frame(id = 1:3)
edges <- data.frame(from = c(1,2), to = c(1,3))

visNetwork(nodes, edges) \%>\%
  visNodes(shape = "icon", icon = list( face ='FontAwesome', code = "f0c0")) \%>\%
  addFontAwesome()

# Save a network

network <- visNetwork(nodes, edges) \%>\% 
 visOptions(highlightNearest = TRUE, nodesIdSelection = TRUE,
 manipulation = TRUE) \%>\% visLegend()
 
network \%>\% visSave(file = "network.html")
# same as
visSave(network, file = "network.html")


# Export as png/jpeg (shiny or browser only)
visNetwork(nodes, edges) \%>\% 
 visExport()


# DOT language
visNetwork(dot = 'dinetwork {1 -> 1 -> 2; 2 -> 3; 2 -- 4; 2 -> 1 }')

# gephi json file

visNetwork(gephi = 'WorldCup2014.json') \%>\% visPhysics(stabilization = FALSE,   barnesHut = list(
    gravitationalConstant = -10000,
    springConstant = 0.002,
    springLength = 150
  ))
}

}
\references{
See online documentation \url{https://datastorm-open.github.io/visNetwork/}
}
\seealso{
\link[visNetwork]{visNodes} for nodes options,  \link[visNetwork]{visEdges} for edges options,  \link[visNetwork]{visGroups} for groups options, 
\link[visNetwork]{visLegend} for adding legend,  \link[visNetwork]{visOptions} for custom option,  \link[visNetwork]{visLayout} &  \link[visNetwork]{visHierarchicalLayout} for layout, 
\link[visNetwork]{visPhysics} for control physics,  \link[visNetwork]{visInteraction} for interaction,  \link[visNetwork]{visNetworkProxy} for play with network using shiny,
\link[visNetwork]{visTree} to visualize CART rpart tree,  \link[visNetwork]{visNetworkEditor} to edit your network, 
\link[visNetwork]{visDocumentation},  \link[visNetwork]{visEvents},  \link[visNetwork]{visConfigure} ...
}
